/**************************************************************************
 * 
 * Class:  Overlay implementation
 * Author: Mark Roseman
 * 
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 05/28/92 MR        initial version
 *
 **************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <gk/overlay.h>
#include <gk/conference.h>
#include <stdio.h>
#include <InterViews/event.h>
#include <InterViews/alloctbl.h>
#include <InterViews/hit.h>
#include <InterViews/geometry.h>

/**************************************************************************
 * 
 * Constructor.
 *
 **************************************************************************/

Overlay::Overlay(ActiveHandler* g, class Style* s, class Conference* c) : 
     ActiveHandler(g,s), conf_(c), inner_(g), am_inside_(false), 
     next_in_line_(nil), pressed_(false), recorded_time_(false), 
     have_entered_(false), processing_event_(false)
{ 
  Handler::ref( (Handler*)this);
}


/**************************************************************************
 * 
 * Event methods that by default call do nothing.
 *
 **************************************************************************/
 
void Overlay::press(const Event&) {  }
void Overlay::move(const Event&){ } 
void Overlay::drag(const Event&) {  }
void Overlay::release(const Event&) {  }
void Overlay::keystroke(const Event&) {  }
void Overlay::double_click(const Event&) {  }
void Overlay::enter() {   }
void Overlay::leave() {   }


/**************************************************************************
 * 
 * Handler event method, which just breaks things down by event type.
 * The last chunk of code calls 'event' on the handler "inside" the Overlay,
 * if any.  The 'processing_event_' flag is used by 'pick' -- after we've
 * processed our event, pick should return the inner handler, not us.
 * We also try to maintain the state of any grabbers.
 *
 **************************************************************************/

boolean Overlay::event(Event& e) {
  switch (e.type()) {
  case Event::down:
    down(e);
    break;
  case Event::motion:
    motion(e);
    break;
  case Event::up:
    up(e);
    break;
  case Event::key:
    keystroke(e);
    break;
  default:
    /* ignore */
    break;
  }

  processing_event_ = true;
  if( next_in_line_ != nil ) {
    if ( e.is_grabbing(this)) {
      e.ungrab(this);
      next_in_line_->event(e);
      e.grab(this);
    } else 
      next_in_line_->event(e);
  }
  processing_event_ = false;
  return true;
}


/**************************************************************************
 * 
 * handler for down events
 *
 **************************************************************************/

void Overlay::down(Event& e) { 
  if (!pressed_) {
    pressed_ = true;
    button_ = e.pointer_button();
    press(e);
  }
}

#define PTINRECT ((e.pointer_x() > allocation().left()) && (e.pointer_x() < allocation().right()) && (e.pointer_y() > allocation().bottom()) && (e.pointer_y() < allocation().top()))

/**************************************************************************
 * 
 * handler for motion events.  here we translate motion into drag/move or
 * enter and leave (the latter by grabbing the event when we enter, and
 * checking to see if we move outside the glyph's allocation).  
 *
 * things get a bit weird because we sometimes start receiving events before 
 * we've physically entered the glyph's allocation.  not sure why.  in any
 * case, the 'have_entered_' flag indicates when we start getting events,
 * and the 'am_inside_' flag marks when we're physically in the glyph (which
 * is when we do enter/leave).
 *
 **************************************************************************/

void Overlay::motion(Event& e) { 
  if (pressed_)
    drag(e);
  else {
    if (e.handler() == this) {
      if (have_entered_ == false) {
	have_entered_ = true;
	e.grab(this);
      } else {
	if(am_inside_ && !(PTINRECT)) {
	  leave();
	  am_inside_ = false;
	} else if (!am_inside_ && (PTINRECT)) {
	  enter();
	  am_inside_ = true;
	} else if (am_inside_)
	  move(e);
      }
    } else if (have_entered_) {
      have_entered_ = false;
      if(am_inside_) {
	am_inside_ = false;
	leave();
      }
      e.ungrab(this);
      Handler::ref((Handler*)this);
    }
  }
}


/**************************************************************************
 * 
 * handler for up.  threshold for double-click is hard-coded as opposed to 
 * being looked up in the style (as per InputHandler).  oh well.
 *
 **************************************************************************/

void Overlay::up(Event& e) { 
  if (pressed_ && e.pointer_button() == button_) {
    pressed_ = false;
    if (recorded_time_ && e.time() - click_time_ < 250)   // threshold
      double_click(e);
    click_time_ = e.time();
    recorded_time_ = true;
    release(e);
  }
}



/**************************************************************************
 * 
 * wrapper to get conference
 *
 **************************************************************************/

Conference* Overlay::conference() { return conf_; }


/**************************************************************************
 * 
 * pick routine.  call pick on the 'inside' glyph.  if anything is hit 
 * there, we intercept the event.  however, if we're in the middle of
 * processing our event when pick is called, we should pass through the
 * 'inside' glyph (see 'event', above).
 *
 **************************************************************************/

void Overlay::pick(Canvas* c, const Allocation& a, int depth, Hit &h) {
  inner_->pick(c,a,depth,h);
  if(h.any() && !processing_event_) {
    next_in_line_ = h.handler();
    h.target(depth, this, 0, this);
  }
  
}

/**************************************************************************
 * 
 * Wrappers for standard glyph methods.  Call the corresponding monoglyph
 * methods, not the InputHandler methods.
 *
 **************************************************************************/

void Overlay::draw( Canvas* c, const Allocation& a) const {
  MonoGlyph::draw(c,a);
}

void Overlay::allocate(Canvas* c, const Allocation& a, Extension& ext) {
  alloc_ = a;
  MonoGlyph::allocate(c,a,ext);
}

const Allocation& Overlay::allocation() const { 
  return alloc_;
}

void Overlay::request(Requisition& req) const {
  MonoGlyph::request(req);
}






