/**************************************************************************
 * 
 * very simple voting application glyph
 *
 **************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include "voteglyph.h"
#include <InterViews/background.h>
#include <InterViews/window.h>
#include <gk/conference.h>
#include <IV-look/kit.h>
#include <IV-look/field.h>
#include <InterViews/layout.h>
#include <InterViews/action.h>
#include <stdio.h>
#include <gk-ui/DialogMgr.h>
#include <InterViews/patch.h>
#include <gk/straction.h>
#include <gk/msgsender.h>
#include <OS/table.h>
#include <gk/groupkit.h>
#include <gk/reader.h>
#include <gk/infoconn.h>
#include <gk-ui/noeditfield.h>

declareActionCallback(VoteGlyph);
implementActionCallback(VoteGlyph);


/**************************************************************************
 * 
 * a glyph that keeps track of number of yes and no votes
 *
 **************************************************************************/

class QuestionerGlyph : public MonoGlyph {
public:
  QuestionerGlyph(Style* style, VoteGlyph* voter, char *question, int id);
  int responses;
  int yes;
  int no;
  void dismiss();
  void yes_vote();
  void no_vote();
  VoteGlyph* voter_;
  NoEditField* resp_;
  NoEditField* total_;
  NoEditField* yes_;
  NoEditField* no_;
  Patch* patch_;
  Window* win_;
};


/**************************************************************************
 * 
 * a glyph that allows users to answer yes/no questions
 *
 **************************************************************************/

class RespondGlyph : public MonoGlyph {
public:
  RespondGlyph(Style* style,VoteGlyph* voter,char *question, int id, int num);
  void yes();
  void no();
  VoteGlyph* voter_;
  int id_;
  int num_;
  Window* win_;
};


declareTable( PollStations, int, QuestionerGlyph*);
implementTable( PollStations, int, QuestionerGlyph*);

declareStrActionCallback(VoteGlyph);
implementStrActionCallback(VoteGlyph);


/**************************************************************************
 * 
 * the main vote glyph is just a button allowing users to create a new
 * vote
 *
 **************************************************************************/

VoteGlyph::VoteGlyph(Style* style, Conference* conf) : 
          ConferenceGlyph(nil, style, conf), style_(style), qc_(0) 
{
  WidgetKit& kit = *WidgetKit::instance();
  LayoutKit& layout = *LayoutKit::instance();
  dlgmgr_ = new DialogManager();
  body(
    kit.inset_frame(
      layout.vbox(
	kit.push_button("New Vote", new ActionCallback(VoteGlyph)
			(this, &VoteGlyph::vote))
      )
    )
  );
  conf->connections()->callbacks()->insert( NEWVOTE, 
					   new StrActionCallback(VoteGlyph)
					   (this, &VoteGlyph::new_vote));
  conf->connections()->callbacks()->insert( CASTVOTE, 
					   new StrActionCallback(VoteGlyph)
					   (this, &VoteGlyph::cast_vote));
  myvotes_ = new PollStations(20);
}


/**************************************************************************
 * 
 * create a new vote.  pop up a dialog box to get the question.  when done
 * create a questionerglyph to record the results, and broadcast the 
 * question to everyone else.
 *
 **************************************************************************/

void VoteGlyph::vote() {
  char *s;
  char t[200];
  if ((s= (char *)dlgmgr_->ask( win_, "Question", 
			       "Is the meaning of life 42?")) != nil) {
    QuestionerGlyph* gl = new QuestionerGlyph( style_, this, s, qc_ );
    Window* w = new TopLevelWindow( gl );
    gl->win_ = w;
    myvotes_->insert( qc_, gl );
    w->map();
    sprintf(t, "%d:%d:%s", conference()->localID(), qc_++, s);
    conference()->connections()->toAll( new StrMsgSender( NEWVOTE, t ));
  }
}

/**************************************************************************
 * 
 * someone else started up a vote, so pop up a respondglyph to answer
 * the question
 *
 **************************************************************************/

void VoteGlyph::new_vote(char *s) { 
  /* id:num:question */
  int id, num;
  char question[500];
  
  sscanf(s, "%d:%d:%[^:]", &id, &num, question);
  RespondGlyph* gl = new RespondGlyph( style_, this, question, id, num);
  Window* w = new TopLevelWindow( gl );
  gl->win_ = w;
  w->map();
}

/**************************************************************************
 * 
 * a vote has been cast.  if the vote is for one of the question's we've
 * got open, tell the questionerglyph about it.
 *
 **************************************************************************/

void VoteGlyph::cast_vote(char *s) { 
  /* id:num:yes/no */
  int num;
  char resp;
  QuestionerGlyph* gl;

  sscanf(s, "%d:%c", &num, &resp);
  if ( myvotes_->find( gl, num ))
    if ( resp=='y')
      gl->yes_vote();
    else
      gl->no_vote();
  else
    printf("couldn't find vote %d\n", num);
}



declareActionCallback(QuestionerGlyph);
implementActionCallback(QuestionerGlyph);

/**************************************************************************
 * 
 * construct a questionerglyph which records the results of a vote we
 * initiated.  consists of the numbers of yes and no votes, as well as
 * the total number of votes (out of the total possible).
 *
 **************************************************************************/

QuestionerGlyph::QuestionerGlyph(Style* style, VoteGlyph* voter, 
				 char *question, int id) : 
			  MonoGlyph(nil), voter_(voter) 
{
  char s[10];
  yes = no = responses = 0;
  WidgetKit& kit = *WidgetKit::instance();
  LayoutKit& layout = *LayoutKit::instance();
  resp_ = new NoEditField( " 0", &kit );
  sprintf(s, "%2d", voter->conference()->connections()->list()->count() );
  total_ = new NoEditField( s, &kit );
  yes_ = new NoEditField(" 0", &kit);
  no_ = new NoEditField(" 0", &kit);
  patch_ = new Patch(
    new Background(
      kit.inset_frame(
        layout.margin(				  
	  layout.vbox(
            kit.label(question),
	    layout.vglue(4.0),
	    layout.hbox(
	      kit.label("Responses:      "),
	      resp_,
	      kit.label(" of "),
	      total_
	    ),
	    layout.vglue(4.0),
	    layout.hbox(
	      kit.label("Yes:            "),
	      yes_
	    ),
	    layout.vglue(4.0),
	    layout.hbox(
	      kit.label("No:             "),
	      no_
	    ),
	    layout.vglue(4.0),
	    layout.hbox(
	      layout.hglue(),
	      kit.push_button("Dismiss", new ActionCallback(QuestionerGlyph)
			      (this, &QuestionerGlyph::dismiss))
	    )
	  ),
	  10.0
	)
      ),
      kit.background()
    )
  );
  body(patch_);
}


/**************************************************************************
 * 
 * get rid of the questionerglyph
 *
 **************************************************************************/

void QuestionerGlyph::dismiss() {
  delete win_;
}


/**************************************************************************
 * 
 * a yes vote has been received
 *
 **************************************************************************/

void QuestionerGlyph::yes_vote() {
  char s[10];
  yes++; responses++;
  sprintf(s,"%d", yes);
  yes_->field(s);
  sprintf(s,"%d", responses);
  resp_->field(s);
  patch_->redraw();
}


/**************************************************************************
 * 
 * a no vote has been received
 *
 **************************************************************************/

void QuestionerGlyph::no_vote() {
  char s[10];
  no++; responses++;
  sprintf(s,"%d", no);
  no_->field(s);
  sprintf(s,"%d", responses);
  resp_->field(s);
  patch_->redraw();
}


declareActionCallback(RespondGlyph);
implementActionCallback(RespondGlyph);

/**************************************************************************
 * 
 * a glyph allowing the user to respond to a yes / no question
 *
 **************************************************************************/

RespondGlyph::RespondGlyph(Style* style, VoteGlyph* voter, char *question, 
			   int id, int num) : 
		MonoGlyph(nil), voter_(voter), id_(id), num_(num) 
{
  WidgetKit& kit = *WidgetKit::instance();
  LayoutKit& layout = *LayoutKit::instance();
  body( 
    new Background(
      kit.inset_frame(
	layout.margin(
	  layout.vbox(
	    layout.vglue(),
	    kit.label(question),
	    layout.vglue(),
	    layout.hbox(
	      layout.hglue(),
	      kit.push_button("Yes", new ActionCallback(RespondGlyph)
			      (this, &RespondGlyph::yes)),
	      layout.hglue(),
	      kit.push_button("No", new ActionCallback(RespondGlyph)
			      (this, &RespondGlyph::no)),
	      layout.hglue()
	    ),
	    layout.vglue()
	  )
	  , 10.0
        )
      ),
      kit.background()
    )
  );
}

/**************************************************************************
 * 
 * the user selected yes - transmit to vote originator
 *
 **************************************************************************/

void RespondGlyph::yes() { 
  char s[20];
  sprintf(s, "%d:y", num_);
  voter_->conference()->connections()->sendTo( id_, 
					      new StrMsgSender( CASTVOTE, s ));
  delete win_;
}

/**************************************************************************
 * 
 * the user selected no - transmit to vote originator
 *
 **************************************************************************/
void RespondGlyph::no() { 
  char s[20];
  sprintf(s, "%d:n", num_);
  voter_->conference()->connections()->sendTo( id_, 
					      new StrMsgSender( CASTVOTE, s ));
  delete win_;
}

