/****************************************************************************
 *
 * Classes: MasterRegClient implementation
 * Author:  Mark Roseman
 *
 * Do an "Master" registrar client, suitable for use by a user in a strictly
 * facilitated conference.
 *
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 10/13/92 MR        initial version
 *
 ****************************************************************************/

/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <gk-reg/coordinator.h>
#include <InterViews/style.h>
#include <InterViews/geometry.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <IV-look/kit.h>
#include <InterViews/layout.h>
#include <gk-ui/strbrowser.h>
#include <IV-look/dialogs.h>
#include <InterViews/window.h>
#include "masterrc.h"
#include <gk/groupsession.h>
#include <gk-ui/labscrlst.h>
#include <InterViews/display.h>
#include <InterViews/patch.h>
#include <OS/table.h>
#include <OS/string.h>
#include <IV-look/field.h>
#include <InterViews/dialog.h>
#include <gk-ui/tabular.h>
#include "intaction.h"
#include <InterViews/action.h>
#include <InterViews/stencil.h>
#include <InterViews/bitmap.h>
#include <IV-look/choice.h>

#include "boot.bit"


/****************************************************************************
 * 
 * the display for a facilitated conference
 *
 ****************************************************************************/

class MasterDisplay : public MonoGlyph {
public:
  MasterDisplay(MasterRegClient*, WidgetKit*);
protected:
  friend class MasterRegClient;
  Glyph* menus();
  void quit_cb();
  void new_prog_cb(int,int);
  void boot_cb(int,int);
  void join_leave_cb(int,int);
  void add_user( AttributeList* );
  void add_conference( AttributeList* );
  WidgetKit* kit_;
  MasterRegClient* rc_;
  Tabular* tbl_;
  Stencil* boot_;
  Patch* patch_;
  PolyGlyph* inner_box_;
  PolyGlyph* outer_box_;
  class IntTable* users_;
  class IntTable* confs_;
};


declareTable(IntTable,int,int);
implementTable(IntTable,int,int);

declareActionCallback(MasterDisplay)
implementActionCallback(MasterDisplay)
declareIntActionCallback(MasterDisplay)
implementIntActionCallback(MasterDisplay)


/****************************************************************************
 * 
 * create the interface for the facilitated conference.  essentially its
 * a table where each row represents one of the users and each column
 * represents one of the conferences.  a check box allows the facilitator
 * to add or remove the indicated user from the indicated conference.  a
 * "boot" button for each user allows the facilitator to remove the user.
 * menus allow the creation of new conferences.
 *
 ****************************************************************************/

MasterDisplay::MasterDisplay(MasterRegClient* rc, WidgetKit* kit) : 
        MonoGlyph(nil), rc_(rc), kit_(kit) 
{
  users_ = new IntTable(20);
  confs_ = new IntTable(20);
  LayoutKit& layout = *LayoutKit::instance();
  Bitmap* bm = new Bitmap( boot_bits, boot_height, boot_width );
  boot_ = new Stencil ( bm, kit->foreground());
  tbl_ = new Tabular(2,1);
  tbl_->replace( 1, 0, kit->label("User") );

  inner_box_ = 
    layout.vbox(
      layout.hbox(layout.hglue(),kit->fancy_label(rc_->name_),layout.hglue()),
      layout.vglue(10.0),
      kit->outset_frame(tbl_)
    );

  outer_box_ = 
    layout.vbox(
      menus(),
      kit->inset_frame(
	layout.margin( 
	  inner_box_
	  , 10.0 
	)
      )
    );

  patch_ = new Patch( outer_box_ );
  body( patch_ );
}


/****************************************************************************
 * 
 * create the menus
 *
 ****************************************************************************/

Glyph* MasterDisplay::menus() {
  Menu* the_menu = kit_->menubar();
  
  /*
   * file menu with a quit item
   */

  MenuItem* file_ = kit_->menu_item("File");
  Menu* file_menu = kit_->pulldown();
  file_->menu(file_menu);
  MenuItem* quit_item = kit_->menu_item("Quit");
  quit_item->action( new ActionCallback(MasterDisplay)
		    (this, &MasterDisplay::quit_cb) );
  file_menu->append_item( quit_item );
  the_menu->append_item(file_);

  /*
   * new menu for creating conferences - one item per possible conference,
   *     determined from Xresources
   */

  MenuItem* new_ = kit_->menu_item("New");
  Menu* new_menu = kit_->pulldown();
  new_->menu(new_menu);

  long types;
  if (kit_->style()->find_attribute("conferenceTypes", types)) {
    for ( int i = 1; i <= types; i++ ) {
      String desc;
      char att[50];
      sprintf(att, "conf%d-desc", i);
      if ( kit_->style()->find_attribute( att, desc )) {
	MenuItem* mi = kit_->menu_item(desc);
	mi->action( new IntActionCallback(MasterDisplay)
		   (this, &MasterDisplay::new_prog_cb, i, 0));
	new_menu->append_item(mi);
      }
    }
  }

  the_menu->append_item(new_);
  return the_menu;
}



/****************************************************************************
 * 
 *  a new user has been found for the facilitated conference - add them 
 *      to the display
 *
 ****************************************************************************/

void MasterDisplay::add_user(AttributeList* al) {
  char user_name[80], user_num[80], fconf[80], confnum[80];
  int col;

  /*
   * insert a new row in the table
   */

  tbl_->insert_row( tbl_->rows() );


  /*
   * add a boot button in column 0
   */

  TelltaleState* t = new TelltaleState(TelltaleState::is_enabled);
  ChoiceItem* ch = new ChoiceItem(t, kit_->outset_frame(boot_),
				  kit_->inset_frame(boot_));
  Button* boot_button = new Button( ch, kit_->style(), t, nil );
  tbl_->replace( 0, tbl_->rows()-1, boot_button);

  /*
   * put the user's name in column 1 
   */

  al->find_attribute( "username", user_name );
  tbl_->replace( 1, tbl_->rows()-1, kit_->label( user_name ));

  /*
   * keep track of which line the user is on in the table for later
   */

  al->find_attribute( "usernum", user_num );
  users_->insert( atoi(user_num), tbl_->rows()-1 );

  /*
   * for every active conference, put in a check box for the user so the
   *    facilitator can have them join or leave the conference
   */

  for (TableIterator(AttrLstTbl) i(*rc_->conference_tbl_);i.more();i.next()) {
    if( (i.cur_value()->find_attribute("facil_conf", fconf)) && 
           (atoi(fconf)==rc_->confnum_)) {
      i.cur_value()->find_attribute("confnum", confnum);
      confs_->find( col, atoi(confnum)); 
      tbl_->replace( col, tbl_->rows()-1, 
		    kit_->check_box(" ", new IntActionCallback(MasterDisplay)
				    (this, &MasterDisplay::join_leave_cb,
				     atoi(confnum), atoi(user_num))));
    }
  }
  
  /*
   * resize the window to fit the new size of the table.  we have to modify
   *   the boxes so they recalculate their sizes rather than using cached
   *   results from earlier.  then ask the canvas to calculate its new size,
   *   and force the window to resize to the canvas.  a bit of a hack because
   *   there's no general mechanism in InterViews for doing this.
   */

  Requisition req;
  inner_box_->modified(2);  outer_box_->modified(1);
  body()->request(req);
  patch_->canvas()->size( req.x_requirement().natural(), 
			 req.y_requirement().natural() );
  patch_->canvas()->window()->resize();
}


/****************************************************************************
 * 
 * add a new conference to the display
 *
 ****************************************************************************/

void MasterDisplay::add_conference(AttributeList* al) {
  char type[80], confnum[80];

  /*
   * add a new column to the table, and associate the conference id with
   *    that column number for later
   */

  tbl_->insert_col ( tbl_->cols() );
  al->find_attribute("type", type);
  al->find_attribute("confnum", confnum);
  confs_->insert( atoi(confnum), tbl_->cols()-1);

  /*
   * put the type of the conference in row 0
   */

  tbl_->replace( tbl_->cols()-1, 0, kit_->label(type) );

  /*
   * for each user add a check box so the facilitator can make them join
   *    or leave the conference
   */

  for (TableIterator(IntTable) i(*users_); i.more(); i.next()) {
    tbl_->replace( tbl_->cols()-1, i.cur_value(), 
		  kit_->check_box(" ", new IntActionCallback(MasterDisplay)
				  (this, &MasterDisplay::join_leave_cb, 
				   atoi(confnum), i.cur_key())));
  }
  
  /*
   * resize the window to fit the new size of the table.  we have to modify
   *   the boxes so they recalculate their sizes rather than using cached
   *   results from earlier.  then ask the canvas to calculate its new size,
   *   and force the window to resize to the canvas.  a bit of a hack because
   *   there's no general mechanism in InterViews for doing this.
   */

  Requisition req;
  inner_box_->modified(2);  outer_box_->modified(1);
  body()->request(req);
  patch_->canvas()->size( req.x_requirement().natural(), 
			 req.y_requirement().natural() );
  patch_->canvas()->window()->resize();
}


/****************************************************************************
 * 
 * tear down the facilitated conference
 *
 ****************************************************************************/

void MasterDisplay::quit_cb() { 
  fprintf(stderr, "display::quit_cb\n");
}


/****************************************************************************
 * 
 * the facilitator used the "new" menu to create a new conference.
 *   ask the registrar (via the registrar client) to create a new conference,
 *   looking up the type in the Xdefaults.  the name will be the same as
 *   that of the main facilitated conference.
 *
 ****************************************************************************/

void MasterDisplay::new_prog_cb(int which,int) { 
  char s[80], att[80];
  String desc;
  AttributeList al;
  al.attribute("name", rc_->name_ );
  sprintf(att, "conf%d-desc", which );
  if (kit_->style()->find_attribute(att, desc)) {
    al.attribute("type", desc.string() );
    sprintf(s,"%d", rc_->confnum_);
    al.attribute("facil_conf", s );
    rc_->callNewConference(&al);
  }
}


/****************************************************************************
 * 
 * delete a user from the facilitated conference
 *
 ****************************************************************************/

void MasterDisplay::boot_cb(int,int) { 

}


/****************************************************************************
 * 
 * the facilitator wants to connect or disconnect a user from one of the
 *    conferences
 *
 ****************************************************************************/

void MasterDisplay::join_leave_cb(int confnum, int usernum) {
  AttrListTable* usrs;
  AttrListTable* fc_usrs;
  AttributeList* user;
  boolean found;
  int foundusernum;
  char host1[80], host2[80], port1[80], port2[80];

  /*
   * get host,port from facil-conf and scan through list of the indicated
   * conference to see if the user is there or not - if so delete them,
   * if not, join them up
   */

  found = false;
  if((rc_->users_tbl_->find( usrs, confnum)) && 
     (rc_->users_tbl_->find( fc_usrs, rc_->confnum_)) &&
     (fc_usrs->find( user, usernum ))) 
    
    for ( TableIterator(AttrLstTbl) i(*usrs); i.more(); i.next()) {
      user->find_attribute( "host", host1 );
      user->find_attribute( "port", port1 );
      i.cur_value()->find_attribute("host", host2 );
      i.cur_value()->find_attribute("port", port2 );
      if ( (strcmp(host1,host2)==0) && (atoi(port1)==atoi(port2))) {
	found = true;
	foundusernum = i.cur_key();
      }
    }
  
  if (found) 
    rc_->callLeaveConference(confnum, foundusernum);
  else {
    char s[80];
    sprintf(s, "%d", confnum);
    user->attribute( "confnum", s );
    rc_->callJoinConference( user );
  }
}	 



declareFieldEditorCallback(MasterRegClient);
implementFieldEditorCallback(MasterRegClient);

/****************************************************************************
 *
 * Constructor.  Create a dialog box prompting for the name of a new
 *   facilitated conference.
 *
 ****************************************************************************/


MasterRegClient::MasterRegClient(const char* host, int port, Coordinator* c) : 
                  RegistrarClient(host,port,c) 
{
  name_[0] = 0;
  disp_ = 0;
  confnum_ = -1;
  kit_ = WidgetKit::instance();
  LayoutKit& layout = *LayoutKit::instance();

  FieldEditor* fld = DialogKit::instance()->field_editor( 
	 String("Conference name"), kit_->style(), 
	 new FieldEditorCallback(MasterRegClient)
		     (this, &MasterRegClient::nameok, nil ));
  dlg_ = new Dialog( 
   kit_->inset_frame( layout.margin(
    layout.vbox(
      kit_->label("New Facilitated Conference:"),
      layout.vspace(5.0),
      fld
    ), 10.0)), kit_->style()
  ); 
  dlg_->append_input_handler( fld );
  dlg_->next_focus();
  dlg_->post_at( Session::instance()->default_display()->width() / 2.0,
	      Session::instance()->default_display()->height() / 2.0);
}




/****************************************************************************
 * 
 * we've got the name for the new facilitated conference, so ask the
 *    registrar to create it and nuke the dialog box
 *
 ****************************************************************************/


void MasterRegClient::nameok(FieldEditor* fld) {
  char cport[10];
  AttributeList* al = new AttributeList();
  al->attribute("name", fld->text()->string());
  al->attribute("type", "FacilitatedConference" );
  al->attribute("facilhost", GroupSession::host_name());
  sprintf(cport, "%d", _lPort);
  al->attribute("facilport", cport );
  callNewConference(al);
  dlg_->dismiss(true);
  strcpy( name_, fld->text()->string());
}


/****************************************************************************
 * 
 * we found a new conference - we're interested in when the facilitated
 *   conference gets created or when a conference managed by the facilitated
 *   conference gets created
 *
 ****************************************************************************/

void MasterRegClient::foundNewConference(AttributeList* al) {
  char conf[80], name[80], type[80], fconf[80];
  al->find_attribute("confnum", conf);
  conference_tbl_->insert( atoi(conf), al );
  users_tbl_->insert( atoi(conf), new AttrListTable(30));

  /*
   * check for our facilitated conference being created, and if we get it
   *    then create the new display for the conference
   */

  if( (disp_==0) && (al->find_attribute("name", name)) 
       && (strcmp(name, name_)==0) && (al->find_attribute("type", type)) 
       && (strcmp(type, "FacilitatedConference")==0)) 
  {
    conference_tbl_->insert( atoi(conf), al);
    users_tbl_->insert( atoi(conf), new AttrListTable(30));
    confnum_ = atoi(conf);
    disp_ = new MasterDisplay( this, kit_ );
    Window* w = new ApplicationWindow( disp_ );
    w->map();
  } 

  /*
   * if the new conference is one we asked to be created then create it and
   *    also join to it
   */

  else if ( (al->find_attribute("facil_conf", fconf)) && 
	   (atoi(fconf)==confnum_)) {
    conference_tbl_->insert( atoi(conf), al);
    users_tbl_->insert( atoi(conf), new AttrListTable(30));
    coord_->createConference(al);
    disp_->add_conference( al );
    callJoinConference( atoi(conf));
  }
}


/****************************************************************************
 *
 * found a deleted conference
 *
 ****************************************************************************/

void MasterRegClient::foundDeletedConference(int conf) {
  conference_tbl_->remove(conf);
  users_tbl_->remove(conf);
  coord_->deleteConference(conf);
}


/****************************************************************************
 *
 * found a new user - add them and tell the display
 *
 ****************************************************************************/

void MasterRegClient::foundNewUser(AttributeList* al) {
  char conf[80], usernum[80], hostname[80], port[10];
  AttrListTable* usrs;
  al->find_attribute("confnum", conf);
  al->find_attribute("usernum", usernum);
  al->find_attribute("host", hostname);
  al->find_attribute("port", port);
  if( users_tbl_->find( usrs, atoi(conf))) {
    usrs->insert( atoi(usernum), al);
  } else
    fprintf(stderr, "in foundnewuser, could not add\n");
  if( atoi(conf) == confnum_ )
    disp_->add_user( al );
}


/****************************************************************************
 *
 * found a deleted user - if its for one of our facilitated conferences
 *   tell the display so the check box can be turned off (NOT IMPLEMENTED)
 *
 ****************************************************************************/

void MasterRegClient::foundDeletedUser(int conf, int user) {
  AttrListTable* users;
  users_tbl_->find( users, conf );
  users->remove( user );
  coord_->deleteUser(conf,user);
}



/****************************************************************************
 * 
 * a local conference has been deleted - tell the registrar to leave.  we
 *    also delete all the people attached to this conference and the 
 *    conference itself (NOT IMPLEMENTED)
 *
 ****************************************************************************/

void MasterRegClient::userLeft(int conf_id, int /* user_id */) {
  char port[80], host[80], usernum[80];
  AttrListTable* usrs;
  users_tbl_->find( usrs, conf_id );
  for ( TableIterator(AttrLstTbl) i(*usrs); i.more(); i.next()) {
    i.cur_value()->find_attribute("host", host);
    i.cur_value()->find_attribute("port", port);
    if ( (atoi(port)==lPort()) && (strcmp(host, GroupSession::host_name())==0)) {
      i.cur_value()->find_attribute("usernum", usernum);
      callLeaveConference( conf_id, atoi(usernum));
    }
  }
}


