package Pod::Elemental::Transformer::WikiDoc;

use Moose;
with 'Pod::Elemental::Transformer';
# ABSTRACT: a transformer to replace "wikidoc" data regions with Pod5 elements

our $VERSION = '0.093004';

#pod =head1 SYNOPSIS
#pod
#pod   my $document = Pod::Elemental->read_string( $string );
#pod   Pod::Elemental::Transformer::Pod5->new->transform_node( $document );
#pod
#pod   Pod::Elemental::Transformer::WikiDoc->new->transform_node( $document );
#pod
#pod ...and if you had a section like this:
#pod
#pod   =begin wikidoc
#pod
#pod   == Look, a header!
#pod
#pod   * Foo
#pod   * Bar
#pod   * Baz
#pod
#pod   =end wikidoc
#pod
#pod ...you now have something more like this:
#pod
#pod   =head2 Look, a header!
#pod
#pod   =over 4
#pod   
#pod   =item Foo
#pod
#pod   =item Bar
#pod
#pod   =item Baz
#pod
#pod   =back
#pod
#pod For complete documentation on this dialect, see L<Pod::WikiDoc>.
#pod
#pod Possibly the most succinct, powerful use case is this:
#pod
#pod   =head2 some_method
#pod
#pod   This B<public> method takes the following arguments:
#pod
#pod   =for wikidoc
#pod   * name
#pod   * rank
#pod   * serial_number
#pod
#pod   =cut
#pod
#pod =cut

use namespace::autoclean;

use Pod::Elemental::Types qw(FormatName);
use Pod::WikiDoc;

#pod =attr format_name
#pod
#pod This attribute indicates the format name of regions to be transformed from
#pod WikiDoc.  By default, the transformer will look for regions of the format
#pod "wikidoc."
#pod
#pod =cut

has format_name => (
  is  => 'ro',
  isa => FormatName,
  default => 'wikidoc',
);

#pod =method transform_node
#pod
#pod =cut

sub transform_node {
  my ($self, $node) = @_;
  my $children = $node->children;

  # Note to future hackers. Don't try replacing this with a simple range operator.
  # $children gets modified on some loops!
  PASS: for (my $i = 0; $i < scalar @{ $children }; $i++) {
    my $para = $children->[$i];
    next unless $para->isa('Pod::Elemental::Element::Pod5::Region')
         and    ! $para->is_pod
         and    $para->format_name eq $self->format_name;

    confess "wikidoc transformer expects wikidoc region to contain 1 Data para"
      unless scalar @{$para->children} == 1
      and    $para->children->[0]->isa('Pod::Elemental::Element::Pod5::Data');

    my $text    = $para->children->[0]->content;
    my $parser  = Pod::WikiDoc->new;
    my $new_pod = $parser->convert($para->as_pod_string);

    $new_pod =~ s/^# Generated by Pod::WikiDoc version \S+$//sgm;

    my $new_doc = Pod::Elemental->read_string($new_pod);
    Pod::Elemental::Transformer::Pod5->transform_node($new_doc);
    shift @{ $new_doc->children }
      while
      $new_doc->children->[0]->isa('Pod::Elemental::Element::Pod5::Nonpod');

    splice @$children, $i, 1, @{ $new_doc->children };
  }

  return $node;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Pod::Elemental::Transformer::WikiDoc - a transformer to replace "wikidoc" data regions with Pod5 elements

=head1 VERSION

version 0.093004

=head1 SYNOPSIS

  my $document = Pod::Elemental->read_string( $string );
  Pod::Elemental::Transformer::Pod5->new->transform_node( $document );

  Pod::Elemental::Transformer::WikiDoc->new->transform_node( $document );

...and if you had a section like this:

  =begin wikidoc

  == Look, a header!

  * Foo
  * Bar
  * Baz

  =end wikidoc

...you now have something more like this:

  =head2 Look, a header!

  =over 4
  
  =item Foo

  =item Bar

  =item Baz

  =back

For complete documentation on this dialect, see L<Pod::WikiDoc>.

Possibly the most succinct, powerful use case is this:

  =head2 some_method

  This B<public> method takes the following arguments:

  =for wikidoc
  * name
  * rank
  * serial_number

  =cut

=head1 ATTRIBUTES

=head2 format_name

This attribute indicates the format name of regions to be transformed from
WikiDoc.  By default, the transformer will look for regions of the format
"wikidoc."

=head1 METHODS

=head2 transform_node

=for :stopwords cpan testmatrix url annocpan anno bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Bugs / Feature Requests

Please report any bugs or feature requests through the issue tracker
at L<https://github.com/dagolden/Pod-Elemental-Transformer-WikiDoc/issues>.
You will be notified automatically of any progress on your issue.

=head2 Source Code

This is open source software.  The code repository is available for
public review and contribution under the terms of the license.

L<https://github.com/dagolden/Pod-Elemental-Transformer-WikiDoc>

  git clone https://github.com/dagolden/Pod-Elemental-Transformer-WikiDoc.git

=head1 AUTHOR

Ricardo SIGNES <rjbs@cpan.org>

=head1 CONTRIBUTORS

=for stopwords David Golden Kent Fredric

=over 4

=item *

David Golden <dagolden@cpan.org>

=item *

Kent Fredric <kentfredric@gmail.com>

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Ricardo SIGNES.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
