/*
 * Electric(tm) VLSI Design System
 *
 * File: conlingtt.c
 * Linear inequality constraint system: graphics-to-text conversion
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */
#include "global.h"
#include "conlin.h"

extern AIDENTRY *us_aid;
WINDOWPART *cli_curwindow;

/* prototypes for local routines */
void cli_addequivcon(ARCINST*);
INTBIG cli_findtextconn(ARCINST*);
char *cli_makearcline(ARCINST*);
char *cli_makeportline(PORTPROTO*);
INTSML cli_uniquelayer(ARCINST*);
INTSML cli_addcomponentinfo(NODEINST*);
void cli_addcomponentdata(COMPONENT*);
char *cli_componentname(NODEINST*);
void cli_addcomponentline(char*, WINDOWPART*);
void cli_addattr(ATTR*);
void cli_addnodeprotoname(NODEPROTO*);

/************************** IDENTIFY EQUIVALENT TEXT **************************/

/*
 * routine to highlight the text that is equivalent to the graphic object "geom"
 */
void cli_highlightequivalent(GEOM *geom)
{
	REGISTER NODEPROTO *par;
	REGISTER ARCINST *ai;
	REGISTER NODEINST *ni;
	REGISTER INTBIG i, cindex, total;
	REGISTER char *str, *compname;
	REGISTER COMPONENTDEC *dec;
	REGISTER COMPONENT *compo;

	par = geomparent(geom);
	if (cli_curfacet != par) return;

	/* handle pointing to graphics */
	if (geom->entrytype == OBJARCINST)
	{
		ai = geom->entryaddr.ai;
		cindex = cli_findtextconn(ai);
		if (cindex != -1) (void)askaid(us_aid, "edit-highlightline", (INTBIG)cli_curwindow, cindex);
		return;
	}

	/* highlight the equivalent node */
	ni = geom->entryaddr.ni;
	compname = cli_componentname(ni);
	total = askaid(us_aid, "edit-totallines", (INTBIG)cli_curwindow);
	for(i=0; i<total; i++)
	{
		str = (char *)askaid(us_aid, "edit-getline", (INTBIG)cli_curwindow, i);
		if (str == NOSTRING) break;
		if (cli_linetype(str) != LINEDECL) continue;
		dec = cli_parsecomp(str, 0);
		if (dec == NOCOMPONENTDEC) continue;
		if (ni->proto != getnodeproto(dec->protoname))
		{
			cli_deletecomponentdec(dec);
			continue;
		}

		/* search for the changed node */
		for(compo = dec->firstcomponent; compo != NOCOMPONENT; compo = compo->nextcomponent)
			if (namesame(compname, compo->name) == 0) break;
		if (compo != NOCOMPONENT) (void)askaid(us_aid, "edit-highlightline", (INTBIG)cli_curwindow, i);
		cli_deletecomponentdec(dec);
		return;
	}
}

/**************************** CONVERT ENTIRE FACET ****************************/

/*
 * routine to fill the text window with the description of graphics facet "np".
 * If "np" is NONODEPROTO, turn off the text system.
 */
void cli_maketextfacet(NODEPROTO *np)
{
	char *name;
	REGISTER char *pt;
	REGISTER NODEINST *ni, *oni;
	REGISTER PORTPROTO *pp;
	REGISTER ARCINST *ai;
	REGISTER INTBIG first, lensofar, len, i, trunclen;
	INTSML chars, lines;

	/* initialize the textual equivalent window */
	cli_curfacet = np;
	if (cli_texton == 0)
	{
		(void)askaid(us_aid, "edit-describe", (INTBIG)&name);
		(void)initinfstr();
		(void)addstringtoinfstr(name);
		(void)addstringtoinfstr(" Editor equivalence for facet ");
		(void)addstringtoinfstr(np->cell->cellname);
		(void)allocstring(&name, returninfstr(), el_tempcluster);
		cli_curwindow = cli_makeeditorwindow(name, &chars, &lines);
		efree(name);
		if (cli_curwindow == NOWINDOWPART) return;
	}
	if (np == NONODEPROTO) return;

	/* convert all components into declarations */
	(void)askaid(us_aid, "edit-suspendgraphics", (INTBIG)cli_curwindow);

	/* erase the window */
	len = askaid(us_aid, "edit-totallines", (INTBIG)cli_curwindow);
	for(i=0; i<len; i++)
		(void)askaid(us_aid, "edit-deleteline", (INTBIG)cli_curwindow, i);

	/* load the facet */
	trunclen = chars / 4 * 3;
	cli_textlines = 0;
	(void)initinfstr();
	(void)addstringtoinfstr("BEGINFACET ");
	(void)addstringtoinfstr(np->cell->cellname);
	(void)addstringtoinfstr(";");
	(void)askaid(us_aid, "edit-addline", (INTBIG)cli_curwindow, 0, (INTBIG)returninfstr());
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		ni->temp1 = 0;
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		if (ni->temp1 != 0) continue;

		/* gather all of the nodes of this type */
		first = 0;
		lensofar = 0;
		for(oni = np->firstnodeinst; oni != NONODEINST; oni = oni->nextnodeinst)
		{
			if (oni->proto != ni->proto) continue;

			/* try to keep the lines down to a reasonable length */
			if (lensofar > trunclen)
			{
				(void)addtoinfstr(' ');
				cli_addnodeprotoname(ni->proto);
				(void)addtoinfstr(';');
				cli_addcomponentline(returninfstr(), cli_curwindow);
				first = 0;
				lensofar = 0;
			}
			if (first == 0)
			{
				(void)initinfstr();
				(void)addstringtoinfstr("declare ");
				lensofar = 8;
				first++;
			} else
			{
				(void)addstringtoinfstr(", ");
				lensofar += 2;
			}
			pt = cli_componentname(oni);
			lensofar += strlen(pt);
			(void)addstringtoinfstr(pt);
			oni->temp1 = 1;

			/* add in size/rotation/location if not default */
			lensofar += cli_addcomponentinfo(oni);
		}
		if (first != 0)
		{
			(void)addtoinfstr(' ');
			cli_addnodeprotoname(ni->proto);
			(void)addtoinfstr(';');
			cli_addcomponentline(returninfstr(), cli_curwindow);
		}
	}

	/* now generate the "export" declarations */
	for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		cli_addcomponentline(cli_makeportline(pp), cli_curwindow);

	/* now convert the constaints */
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		cli_addequivcon(ai);
	cli_replaceendfacet("ENDFACET;", cli_curwindow);
	(void)askaid(us_aid, "edit-resumegraphics", (INTBIG)cli_curwindow);
}

/***************************** GRAPHIC CHANGES *****************************/

/*
 * routine to update text when a new graphic node "ni" is created
 */
void cli_eq_newnode(NODEINST *ni)
{
	REGISTER COMPONENTDEC *dec;
	REGISTER COMPONENT *compo;
	REGISTER char *str;
	REGISTER INTBIG i, total;

	/* ignore changes generated by constraint results */
	if (cli_ownchanges != 0) return;

	/* ignore changes made to a facet not being equated */
	if (ni->parent != cli_curfacet) return;

	/* look for a declaration of this type of node */
	total = askaid(us_aid, "edit-totallines", (INTBIG)cli_curwindow);
	for(i=0; i<total; i++)
	{
		str = (char *)askaid(us_aid, "edit-getline", (INTBIG)cli_curwindow, i);
		if (str == NOSTRING) break;
		if (cli_linetype(str) != LINEDECL) continue;
		dec = cli_parsecomp(str, 0);
		if (dec == NOCOMPONENTDEC) continue;
		if (ni->proto != getnodeproto(dec->protoname))
		{
			cli_deletecomponentdec(dec);
			continue;
		}

		/* found the declaration, rebuild with existing components */
		(void)initinfstr();
		(void)addstringtoinfstr("declare ");
		for(compo = dec->firstcomponent; compo != NOCOMPONENT; compo = compo->nextcomponent)
		{
			(void)addstringtoinfstr(compo->name);
			cli_addcomponentdata(compo);
			(void)addstringtoinfstr(", ");
		}

		/* add in the new component */
		(void)addstringtoinfstr(cli_componentname(ni));
		(void)cli_addcomponentinfo(ni);
		(void)addtoinfstr(' ');

		/* finish the declaration and replace the line */
		(void)addstringtoinfstr(dec->protoname);
		(void)addtoinfstr(';');
		(void)askaid(us_aid, "edit-replaceline", (INTBIG)cli_curwindow, i, (INTBIG)returninfstr());
		cli_deletecomponentdec(dec);
		return;
	}

	/* build a simple declaration for this node */
	(void)initinfstr();
	(void)addstringtoinfstr("declare ");
	(void)addstringtoinfstr(cli_componentname(ni));
	(void)cli_addcomponentinfo(ni);
	(void)addtoinfstr(' ');
	cli_addnodeprotoname(ni->proto);
	(void)addtoinfstr(';');
	cli_addcomponentline(returninfstr(), cli_curwindow);
}

/*
 * routine to update text when graphic node "ni" is deleted
 */
void cli_eq_killnode(NODEINST *ni)
{
	REGISTER char *str, *compname;
	REGISTER INTBIG first, i, total;
	REGISTER COMPONENTDEC *dec;
	REGISTER COMPONENT *compo, *ocomp;

	/* see if node is in graphics facet that has equivalent text */
	if (cli_ownchanges != 0) return;

	/* ignore changes made to a facet not being equated */
	if (ni->parent != cli_curfacet) return;

	/* look for a declaration of this type of node */
	compname = cli_componentname(ni);
	total = askaid(us_aid, "edit-totallines", (INTBIG)cli_curwindow);
	for(i=0; i<total; i++)
	{
		str = (char *)askaid(us_aid, "edit-getline", (INTBIG)cli_curwindow, i);
		if (str == NOSTRING) break;
		if (cli_linetype(str) != LINEDECL) continue;
		dec = cli_parsecomp(str, 0);
		if (dec == NOCOMPONENTDEC) continue;
		if (ni->proto != getnodeproto(dec->protoname))
		{
			cli_deletecomponentdec(dec);
			continue;
		}

		/* search for the deleted node */
		for(compo = dec->firstcomponent; compo != NOCOMPONENT; compo = compo->nextcomponent)
			if (namesame(compname, compo->name) == 0) break;
		if (compo == NOCOMPONENT)
		{
			cli_deletecomponentdec(dec);
			continue;
		}

		/* delete the line if there are only two keywords */
		if (dec->count == 1 && namesame(dec->firstcomponent->name, compname) == 0)
		{
			/* line had only one keyword: delete the line */
			(void)askaid(us_aid, "edit-deleteline", (INTBIG)cli_curwindow, i);
			cli_textlines--;
			cli_deletecomponentdec(dec);
			return;
		}

		/* rebuild line without the deleted component */
		(void)initinfstr();
		(void)addstringtoinfstr("declare ");
		first = 0;
		for(ocomp = dec->firstcomponent; ocomp != NOCOMPONENT; ocomp = ocomp->nextcomponent)
		{
			if (compo == ocomp) continue;
			if (first != 0) (void)addstringtoinfstr(", ");
			first++;
			(void)addstringtoinfstr(ocomp->name);
			cli_addcomponentdata(ocomp);
		}
		(void)addtoinfstr(' ');
		(void)addstringtoinfstr(dec->protoname);
		(void)addtoinfstr(';');
		(void)askaid(us_aid, "edit-replaceline", (INTBIG)cli_curwindow, i, (INTBIG)returninfstr());
		cli_deletecomponentdec(dec);
		return;
	}
	ttyputmsg("Warning: cannot find node %s in text", compname);
}

/* handle the change of node "ni" */
void cli_eq_modnode(NODEINST *ni)
{
	/* see if node is in graphics facet that has equivalent text */
	if (cli_ownchanges != 0) return;
	if (ni->parent == cli_curfacet) cli_changeequivcomp(ni);
}

/* handle the creation of new arc "ai" */
void cli_eq_newarc(ARCINST *ai)
{
	REGISTER INTSML i, j;
	REGISTER NODEINST *ni;
	REGISTER PORTARCINST *pi;

	/* see if arc is in graphics facet that has equivalent text */
	if (cli_ownchanges != 0) return;
	if (ai->parent != cli_curfacet) return;

	cli_addequivcon(ai);

	/* see if end nodes now have one connection */
	for(i=0; i<2; i++)
	{
		ni = ai->end[i].nodeinst;
		j = 0;
		for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst) j++;
		if (j == 1) cli_changeequivcomp(ni);
	}
}

/* handle the deletion of arc "ai" */
void cli_eq_killarc(ARCINST *ai)
{
	REGISTER NODEINST *ni;
	REGISTER INTSML i;

	/* see if arc is in graphics facet that has equivalent text */
	if (cli_ownchanges != 0) return;
	if (ai->parent != cli_curfacet) return;

	cli_deleteequivcon(ai);

	/* see if formerly connecting nodes need redefinition */
	for(i=0; i<2; i++)
	{
		ni = ai->end[i].nodeinst;
		if (ni->firstportarcinst == NOPORTARCINST)
			cli_changeequivcomp(ni);
	}
}

/*
 * routine to handle the change of graphic arc "ai" by making an equivalent
 * change to the text
 */
void cli_eq_modarc(ARCINST *ai)
{
	REGISTER char *str;
	REGISTER INTBIG cindex;

	/* see if arc is in graphics facet that has equivalent text */
	if (cli_ownchanges != 0) return;

	/* ignore changes made to a facet not being equated */
	if (ai->parent != cli_curfacet) return;

	/* get the text node that describes this arc */
	cindex = cli_findtextconn(ai);
	if (cindex == -1)
	{
		ttyputerr("Cannot find text equivalent for arc %s", describearcinst(ai));
		return;
	}

	/* build the constraint message */
	str = cli_makearcline(ai);

	/* change the line */
	(void)askaid(us_aid, "edit-replaceline", (INTBIG)cli_curwindow, cindex, (INTBIG)str);
}

/* handle the creation of port "pp" */
void cli_eq_newport(PORTPROTO *pp)
{
	/* see if port is in graphics facet that has equivalent text */
	if (cli_ownchanges != 0) return;
	if (pp->parent != cli_curfacet) return;
	cli_addcomponentline(cli_makeportline(pp), cli_curwindow);
}

/* handle the deletion of port "pp" */
void cli_eq_killport(PORTPROTO *pp)
{
	REGISTER INTBIG i, total;
	REGISTER char *str;
	REGISTER EXPORT *e;

	/* see if port is in graphics facet that has equivalent text */
	if (cli_ownchanges != 0) return;
	if (pp->parent != cli_curfacet) return;

	/* find the export line with this port */
	total = askaid(us_aid, "edit-totallines", (INTBIG)cli_curwindow);
	for(i=0; i<total; i++)
	{
		/* get a line of text */
		str = (char *)askaid(us_aid, "edit-getline", (INTBIG)cli_curwindow, i);
		if (str == NOSTRING) break;
		if (cli_linetype(str) != LINEEXPORT) continue;

		/* see if export line is this one */
		e = cli_parseexport(str, 0);
		if (e == NOEXPORT) continue;
		if (namesame(pp->protoname, e->portname) != 0)
		{
			cli_deleteexport(e);
			continue;
		}

		/* export line found: delete it */
		(void)askaid(us_aid, "edit-deleteline", (INTBIG)cli_curwindow, i);
		cli_textlines--;
		cli_deleteexport(e);
		break;
	}
}

/* handle the modification of port "pp" */
void cli_eq_modport(PORTPROTO *pp)
{
	REGISTER char *str;
	REGISTER EXPORT *e;
	REGISTER INTBIG i, total;

	/* see if port is in graphics facet that has equivalent text */
	if (cli_ownchanges != 0) return;
	if (pp->parent != cli_curfacet) return;

	/* find the export line with this port */
	total = askaid(us_aid, "edit-totallines", (INTBIG)cli_curwindow);
	for(i=0; i<total; i++)
	{
		/* get a line of text */
		str = (char *)askaid(us_aid, "edit-getline", (INTBIG)cli_curwindow, i);
		if (str == NOSTRING) break;
		if (cli_linetype(str) != LINEEXPORT) continue;

		/* see if export line is this one */
		e = cli_parseexport(str, 0);
		if (e == NOEXPORT) continue;
		if (namesame(pp->protoname, e->portname) != 0)
		{
			cli_deleteexport(e);
			continue;
		}

		/* export line found: rewrite it */
		(void)askaid(us_aid, "edit-replaceline", (INTBIG)cli_curwindow, i,
			(INTBIG)cli_makeportline(pp));
		cli_deleteexport(e);
		break;
	}
}

/* handle the creation of variable type "type", name "key", and value "addr" */
void cli_eq_newvar(INTBIG addr, INTBIG type, INTBIG key)
{
	REGISTER NODEINST *ni;
	char line[50];
	REGISTER VARIABLE *var;

	if (cli_ownchanges != 0) return;

	if (type == VNODEINST)
	{
		/* node added variable: see if this is in a graphics facet */
		ni = (NODEINST *)addr;
		if (ni->parent != cli_curfacet) return;

		/* name changes are handled specially */
		if (key == el_node_name)
		{
			/* construct the original name */
			(void)sprintf(line, "NODE%ld", (INTBIG)ni);

			/* get the new node name */
			var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
			if (var == NOVARIABLE) return;
			cli_replacename(line, (char *)var->addr);
			return;
		}

		/* update the declaration */
		cli_changeequivcomp(ni);
		return;
	}

	if (type == VPORTPROTO)
	{
		cli_eq_modport((PORTPROTO *)addr);
		return;
	}

	if (type == VARCINST)
	{
		cli_eq_modarc((ARCINST *)addr);
		return;
	}
}

/* handle the deletion of variable type "type", name "key", and value "addr" */
void cli_eq_killvar(INTBIG addr, INTBIG type, INTBIG key)
{
	char line[50];
	REGISTER VARIABLE *var;
	REGISTER NODEINST *ni;

	if (cli_ownchanges != 0) return;
	if (type == VNODEINST)
	{
		/* node variable removed: see if this is in a graphics facet */
		ni = (NODEINST *)addr;
		if (ni->parent != cli_curfacet) return;

		if (key == el_node_name)
		{
			/* get the former name */
			var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
			if (var == NOVARIABLE) return;

			/* construct the new node name */
			(void)sprintf(line, "NODE%ld", (INTBIG)ni);
			cli_replacename((char *)var->addr, line);
			return;
		}

		/* update the declaration */
		cli_changeequivcomp(ni);
		return;
	}

	if (type == VPORTPROTO)
	{
		cli_eq_modport((PORTPROTO *)addr);
		return;
	}

	if (type == VARCINST)
	{
		cli_eq_modarc((ARCINST *)addr);
		return;
	}
}

void cli_eq_solve(void)
{
	cli_ownchanges = 0;
}

/***************************** HELPER ROUTINES *****************************/

/*
 * routine to handle the addition of the text associated with new layout arc "ai"
 */
void cli_addequivcon(ARCINST *ai)
{
	REGISTER INTBIG i, total;

	total = askaid(us_aid, "edit-totallines", (INTBIG)cli_curwindow);

	/* search for the last line with a valid connection declaration */
	for(i = total-1; i >= 0; i--)
		if (cli_linetype((char *)askaid(us_aid, "edit-getline",
			(INTBIG)cli_curwindow, i)) == LINECONN) break;
	if (i < 0)
	{
		/* no "connect" lines, take last line anywhere */
		i = total-1;
	}
	(void)askaid(us_aid, "edit-addline", (INTBIG)cli_curwindow, i+1, (INTBIG)cli_makearcline(ai));
	cli_textlines++;
}

/*
 * routine to handle the deletion of the text associated with layout arc "ai"
 */
void cli_deleteequivcon(ARCINST *ai)
{
	REGISTER INTBIG cindex;

	/* get the text node that describes this arc */
	cindex = cli_findtextconn(ai);
	if (cindex != -1)
	{
		(void)askaid(us_aid, "edit-deleteline", (INTBIG)cli_curwindow, cindex);
		cli_textlines--;
	}
}

/*
 * routine to update the text information about of node "ni"
 */
void cli_changeequivcomp(NODEINST *ni)
{
	REGISTER char *str, *compname;
	REGISTER INTBIG i, total;
	REGISTER COMPONENTDEC *dec;
	REGISTER COMPONENT *compo, *ocomp;

	/* look for a declaration of this type of node */
	compname = cli_componentname(ni);
	total = askaid(us_aid, "edit-totallines", (INTBIG)cli_curwindow);
	for(i=0; i<total; i++)
	{
		str = (char *)askaid(us_aid, "edit-getline", (INTBIG)cli_curwindow, i);
		if (str == NOSTRING) break;
		if (cli_linetype(str) != LINEDECL) continue;
		dec = cli_parsecomp(str, 0);
		if (dec == NOCOMPONENTDEC) continue;
		if (ni->proto != getnodeproto(dec->protoname))
		{
			cli_deletecomponentdec(dec);
			continue;
		}

		/* search for the changed node */
		for(compo = dec->firstcomponent; compo != NOCOMPONENT; compo = compo->nextcomponent)
			if (namesame(compname, compo->name) == 0) break;
		if (compo == NOCOMPONENT)
		{
			cli_deletecomponentdec(dec);
			continue;
		}

		(void)initinfstr();
		(void)addstringtoinfstr("declare ");
		for(ocomp = dec->firstcomponent; ocomp != NOCOMPONENT; ocomp = ocomp->nextcomponent)
		{
			if (ocomp != dec->firstcomponent) (void)addstringtoinfstr(", ");
			if (compo == ocomp)
			{
				(void)addstringtoinfstr(cli_componentname(ni));
				(void)cli_addcomponentinfo(ni);
			} else
			{
				(void)addstringtoinfstr(ocomp->name);
				cli_addcomponentdata(ocomp);
			}
		}
		(void)addtoinfstr(' ');
		(void)addstringtoinfstr(dec->protoname);
		(void)addtoinfstr(';');
		(void)askaid(us_aid, "edit-replaceline", (INTBIG)cli_curwindow, i, (INTBIG)returninfstr());
		cli_deletecomponentdec(dec);
		return;
	}
	ttyputmsg("Warning: cannot find node %s in text", compname);
}

/*
 * routine to replace the name of graphics node "oni" (which used to be called
 * "oldname") with the name "newname"
 */
void cli_replacename(char *oldname, char *newname)
{
	REGISTER char *str;
	REGISTER INTBIG i, rewrite, total;
	char line[50];
	REGISTER EXPORT *e;
	REGISTER COMPONENTDEC *dcl;
	REGISTER COMPONENT *compo, *ocomp;
	REGISTER CONNECTION *dec;
	REGISTER CONS *cons;
	REGISTER ATTR *a;

	/* look through all statements for the old name */
	total = askaid(us_aid, "edit-totallines", (INTBIG)cli_curwindow);
	for(i=0; i<total; i++)
	{
		str = (char *)askaid(us_aid, "edit-getline", (INTBIG)cli_curwindow, i);
		if (str == NOSTRING) break;

		/* search declaration statement */
		if (cli_linetype(str) == LINEDECL)
		{
			dcl = cli_parsecomp(str, 0);
			if (dcl == NOCOMPONENTDEC) continue;
			for(compo = dcl->firstcomponent; compo != NOCOMPONENT; compo = compo->nextcomponent)
				if (namesame(oldname, compo->name) == 0) break;
			if (compo == NOCOMPONENT)
			{
				cli_deletecomponentdec(dcl);
				continue;
			}

			/* rewrite the component declaration line */
			(void)initinfstr();
			(void)addstringtoinfstr("declare ");
			for(ocomp = dcl->firstcomponent; ocomp != NOCOMPONENT; ocomp = ocomp->nextcomponent)
			{
				if (ocomp != dcl->firstcomponent) (void)addstringtoinfstr(", ");
				if (ocomp == compo) (void)addstringtoinfstr(newname); else
					(void)addstringtoinfstr(ocomp->name);
				cli_addcomponentdata(ocomp);
			}
			(void)addtoinfstr(' ');
			(void)addstringtoinfstr(dcl->protoname);
			(void)addtoinfstr(';');
			(void)askaid(us_aid, "edit-replaceline", (INTBIG)cli_curwindow, i, (INTBIG)returninfstr());
			cli_deletecomponentdec(dcl);
			continue;
		}

		/* search export statement */
		if (cli_linetype(str) == LINEEXPORT)
		{
			e = cli_parseexport(str, 0);
			if (e == NOEXPORT) continue;
			if (namesame(e->component, oldname) != 0)
			{
				cli_deleteexport(e);
				continue;
			}

			/* rewrite the export line */
			(void)initinfstr();
			(void)addstringtoinfstr("export ");
			if ((e->flag&(EXPCHAR|EXPATTR)) != 0)
			{
				if ((e->flag&EXPCHAR) != 0)
				{
					(void)addstringtoinfstr("[type=");
					switch (e->bits&STATEBITS)
					{
						case INPORT :
							(void)addstringtoinfstr("input"); break;
						case OUTPORT:
							(void)addstringtoinfstr("output"); break;
						case BIDIRPORT:
							(void)addstringtoinfstr("bidirectional"); break;
						case PWRPORT:
							(void)addstringtoinfstr("power"); break;
						case GNDPORT:
							(void)addstringtoinfstr("ground"); break;
						case CLKPORT:
							(void)addstringtoinfstr("clock"); break;
						case C1PORT:
							(void)addstringtoinfstr("clock1"); break;
						case C2PORT:
							(void)addstringtoinfstr("clock2"); break;
						case C3PORT:
							(void)addstringtoinfstr("clock3"); break;
						case C4PORT:
							(void)addstringtoinfstr("clock4"); break;
						case C5PORT:
							(void)addstringtoinfstr("clock5"); break;
						case C6PORT:
							(void)addstringtoinfstr("clock6"); break;
						case REFOUTPORT:
							(void)addstringtoinfstr("refout"); break;
						case REFINPORT:
							(void)addstringtoinfstr("refin"); break;
					}
				}
				if ((e->flag&EXPATTR) != 0)
				{
					if ((e->flag&EXPCHAR) == 0) (void)addtoinfstr('['); else
						(void)addtoinfstr(',');
					for(a = e->firstattr; a != NOATTR; a = a->nextattr)
					{
						if (a != e->firstattr) (void)addtoinfstr(',');
						cli_addattr(a);
					}
				}
				(void)addstringtoinfstr("] ");
			}
			(void)addstringtoinfstr(e->portname);
			(void)addstringtoinfstr(" is ");
			(void)addstringtoinfstr(newname);
			if (e->subport != 0)
			{
				(void)addtoinfstr(':');
				(void)addstringtoinfstr(e->subport);
			}
			(void)addtoinfstr(';');
			(void)askaid(us_aid, "edit-replaceline", (INTBIG)cli_curwindow, i, (INTBIG)returninfstr());
			cli_deleteexport(e);
			continue;
		}

		/* search connection statement */
		if (cli_linetype(str) == LINECONN)
		{
			dec = cli_parseconn(str, 0);
			if (dec == NOCONNECTION) continue;

			/* see if the connection line needs to be rewriten */
			rewrite = 0;
			if ((dec->flag&END1VALID) != 0 && namesame(dec->end1, oldname) == 0)
			{
				(void)reallocstring(&dec->end1, newname, el_tempcluster);
				rewrite++;
			}
			if ((dec->flag&END2VALID) != 0 && namesame(dec->end2, oldname) == 0)
			{
				(void)reallocstring(&dec->end2, newname, el_tempcluster);
				rewrite++;
			}
			if (rewrite == 0)
			{
				cli_deleteconnection(dec);
				continue;
			}

			/* rewrite the connetion line */
			(void)initinfstr();
			(void)addstringtoinfstr("connect ");
			if ((dec->flag&(LAYERVALID|WIDTHVALID|ATTRVALID)) != 0)
			{
				if ((dec->flag&LAYERVALID) != 0)
				{
					(void)addstringtoinfstr("[layer=");
					(void)addstringtoinfstr(dec->layer);
				}
				if ((dec->flag&WIDTHVALID) != 0)
				{
					if ((dec->flag&LAYERVALID) != 0) (void)addtoinfstr(','); else
						(void)addtoinfstr('[');
					(void)sprintf(line, "width=%s", latoa(dec->width));
					(void)addstringtoinfstr(line);
				}
				if ((dec->flag&ATTRVALID) != 0)
				{
					if ((dec->flag&(LAYERVALID|WIDTHVALID)) != 0) (void)addtoinfstr(','); else
						(void)addtoinfstr('[');
					for(a = dec->firstattr; a != NOATTR; a = a->nextattr)
					{
						if (a != dec->firstattr) (void)addtoinfstr(',');
						cli_addattr(a);
					}
				}
				(void)addstringtoinfstr("] ");
			}

			(void)addstringtoinfstr(dec->end1);
			if ((dec->flag&PORT1VALID) != 0)
			{
				(void)addtoinfstr(':');
				(void)addstringtoinfstr(dec->port1);
			}

			for(cons = dec->firstcons; cons != NOCONS; cons = cons->nextcons)
			{
				(void)addtoinfstr(' ');
				(void)addstringtoinfstr(cons->direction);
				(void)addtoinfstr(' ');
				(void)addstringtoinfstr(frtoa(cons->amount));
				if (cons->flag == 1) (void)addstringtoinfstr(" or more");
				if (cons->flag == -1) (void)addstringtoinfstr(" or less");
			}

			/* decide whether to print the arc offset */
			if ((dec->flag&OFFSETVALID) != 0)
			{
				(void)sprintf(line, " [%s,%s]", latoa(dec->xoff), latoa(dec->yoff));
				(void)addstringtoinfstr(line);
			}
			(void)addstringtoinfstr(" to ");
			(void)addstringtoinfstr(dec->end2);
			if ((dec->flag&PORT2VALID) != 0)
			{
				(void)addtoinfstr(':');
				(void)addstringtoinfstr(dec->port2);
			}
			(void)addstringtoinfstr(";");
			(void)askaid(us_aid, "edit-replaceline", (INTBIG)cli_curwindow, i, (INTBIG)returninfstr());
			cli_deleteconnection(dec);
		}
	}
}

/***************************** UTILITIES *****************************/

/*
 * routine to find the text line associated with arc "ai".  Returns the line
 * index (-1 if it cannot be found).
 */
INTBIG cli_findtextconn(ARCINST *ai)
{
	REGISTER INTBIG i, total;
	REGISTER char *str;
	REGISTER CONNECTION *dec;

	total = askaid(us_aid, "edit-totallines", (INTBIG)cli_curwindow);
	for(i=0; i<total; i++)
	{
		/* get a valid connection line */
		str = (char *)askaid(us_aid, "edit-getline", (INTBIG)cli_curwindow, i);
		if (str == NOSTRING) break;
		if (cli_linetype(str) != LINECONN) continue;
		dec = cli_parseconn(str, 1);
		if (dec == NOCONNECTION) continue;

		/* must have both ends specified */
		if ((dec->flag&(END1VALID|END2VALID)) != (END1VALID|END2VALID))
		{
			cli_deleteconnection(dec);
			continue;
		}

		/* end nodes must be correct */
		if (namesame(dec->end1, cli_componentname(ai->end[0].nodeinst)) != 0 ||
			namesame(dec->end2, cli_componentname(ai->end[1].nodeinst)) != 0)
		{
			cli_deleteconnection(dec);
			continue;
		}

		/* specified ports must be correct */
		if ((dec->flag&PORT1VALID) != 0 &&
			namesame(dec->port1, ai->end[0].portarcinst->proto->protoname) != 0)
		{
			cli_deleteconnection(dec);
			continue;
		}
		if ((dec->flag&PORT2VALID) != 0 &&
			namesame(dec->port2, ai->end[1].portarcinst->proto->protoname) != 0)
		{
			cli_deleteconnection(dec);
			continue;
		}

		/* line of text found */
		cli_deleteconnection(dec);
		return(i);
	}
	return(-1);
}

/*
 * routine to convert an arc into a line of text with all constraints
 */
char *cli_makearcline(ARCINST *ai)
{
	char line[50];
	REGISTER VARIABLE *var;
	REGISTER INTSML i, len, doit, added;
	REGISTER INTBIG conx, cony, xcons, ycons;
	REGISTER LINCON *conptr;

	(void)initinfstr();
	(void)addstringtoinfstr("connect ");
	added = 0;
	if (cli_uniquelayer(ai) == 0)
	{
		(void)addstringtoinfstr("[layer=");
		(void)addstringtoinfstr(describearcproto(ai->proto));
		added++;
	}
	if (ai->width != ai->proto->nominalwidth)
	{
		if (added == 0) (void)addtoinfstr('['); else (void)addtoinfstr(',');
		(void)addstringtoinfstr("width=");
		(void)addstringtoinfstr(latoa(ai->width));
		added++;
	}
	if (ai->numvar != 0) for(i=0; i<ai->numvar; i++)
	{
		var = &ai->firstvar[i];
		if ((var->type&VISARRAY) != 0) continue;
		if (added == 0) (void)addtoinfstr('['); else (void)addtoinfstr(',');
		(void)addstringtoinfstr((char *)var->key);
		(void)addtoinfstr('=');
		(void)addstringtoinfstr(describevariable(var, -1, 1));
		added++;
	}
	if (added != 0) (void)addstringtoinfstr("] ");
	(void)addstringtoinfstr(cli_componentname(ai->end[0].nodeinst));
	if (cli_uniqueport(ai->end[0].nodeinst, ai->end[0].portarcinst->proto) == 0)
	{
		(void)addtoinfstr(':');
		(void)addstringtoinfstr(ai->end[0].portarcinst->proto->protoname);
	}
	var = getvalkey((INTBIG)ai, VARCINST, VINTEGER|VISARRAY, cli_properties);

	/* initialize the X constraint to be different from the actual arc size */
	xcons = ycons = 0;
	conx = cony = 0;
	if (var != NOVARIABLE)
	{
		len = getlength(var) / LINCONSIZE;
		for(i=0; i<len; i++)
		{
			conptr = &((LINCON *)var->addr)[i];

			switch (conptr->variable)
			{
				case CLLEFT:
					xcons++;
					conx = -conptr->value * ai->proto->tech->deflambda / WHOLE;
					(void)sprintf(line, " left %s", latoa(-conx));
					break;
				case CLRIGHT:
					xcons++;
					conx = conptr->value * ai->proto->tech->deflambda / WHOLE;
					(void)sprintf(line, " right %s", latoa(conx));
					break;
				case CLDOWN:
					ycons++;
					cony = -conptr->value * ai->proto->tech->deflambda / WHOLE;
					(void)sprintf(line, " down %s", latoa(-cony));
					break;
				case CLUP:
					ycons++;
					cony = conptr->value * ai->proto->tech->deflambda / WHOLE;
					(void)sprintf(line, " up %s", latoa(cony));
					break;
			}
			(void)addstringtoinfstr(line);
			if (conptr->oper == 0) (void)addstringtoinfstr(" or more");
			if (conptr->oper == 2) (void)addstringtoinfstr(" or less");
		}
	}

	/* decide whether to print the arc offset */
	doit = 0;
	if (xcons > 1 || ycons > 1) doit++;
	if (xcons == 0 && ycons == 0) doit++;
	if (conx != ai->end[1].xpos - ai->end[0].xpos || cony != ai->end[1].ypos - ai->end[0].ypos)
		doit++;
	if (doit != 0)
	{
		(void)sprintf(line, " [%s,%s]", latoa(ai->end[1].xpos - ai->end[0].xpos),
			latoa(ai->end[1].ypos - ai->end[0].ypos));
		(void)addstringtoinfstr(line);
	}
	(void)addstringtoinfstr(" to ");
	(void)addstringtoinfstr(cli_componentname(ai->end[1].nodeinst));
	if (cli_uniqueport(ai->end[1].nodeinst, ai->end[1].portarcinst->proto) == 0)
	{
		(void)addtoinfstr(':');
		(void)addstringtoinfstr(ai->end[1].portarcinst->proto->protoname);
	}
	(void)addstringtoinfstr(";");
	return(returninfstr());
}

/*
 * routine to handle the addition of the text associated with new port "pp"
 */
char *cli_makeportline(PORTPROTO *pp)
{
	REGISTER INTSML added, i;
	REGISTER VARIABLE *var;

	(void)initinfstr();
	(void)addstringtoinfstr("export ");
	if ((pp->userbits&STATEBITS) != 0)
	{
		(void)addstringtoinfstr("[type=");
		switch (pp->userbits&STATEBITS)
		{
			case INPORT:     (void)addstringtoinfstr("input");         break;
			case OUTPORT:    (void)addstringtoinfstr("output");        break;
			case BIDIRPORT:  (void)addstringtoinfstr("bidirectional"); break;
			case PWRPORT:    (void)addstringtoinfstr("power");         break;
			case GNDPORT:    (void)addstringtoinfstr("ground");        break;
			case CLKPORT:    (void)addstringtoinfstr("clock");         break;
			case C1PORT:     (void)addstringtoinfstr("clock1");        break;
			case C2PORT:     (void)addstringtoinfstr("clock2");        break;
			case C3PORT:     (void)addstringtoinfstr("clock3");        break;
			case C4PORT:     (void)addstringtoinfstr("clock4");        break;
			case C5PORT:     (void)addstringtoinfstr("clock5");        break;
			case C6PORT:     (void)addstringtoinfstr("clock6");        break;
			case REFOUTPORT: (void)addstringtoinfstr("refout");        break;
			case REFINPORT:  (void)addstringtoinfstr("refin");         break;
		}
		added = 1;
	} else added = 0;

	for(i=0; i<pp->numvar; i++)
	{
		var = &pp->firstvar[i];
		if ((var->type&VISARRAY) != 0) continue;
		if (added == 0) (void)addtoinfstr('['); else (void)addtoinfstr(',');
		(void)addstringtoinfstr((char *)var->key);
		(void)addtoinfstr('=');
		(void)addstringtoinfstr(describevariable(var, -1, 1));
		added++;
	}
	if (added != 0) (void)addstringtoinfstr("] ");

	(void)addstringtoinfstr(pp->protoname);
	(void)addstringtoinfstr(" is ");
	(void)addstringtoinfstr(cli_componentname(pp->subnodeinst));
	if (cli_uniqueport(pp->subnodeinst, pp->subportproto) == 0)
	{
		(void)addtoinfstr(':');
		(void)addstringtoinfstr(pp->subportproto->protoname);
	}
	(void)addtoinfstr(';');
	return(returninfstr());
}

/*
 * routine to tell whether port "pp" of node "ni" is unique on its node
 * returns nonzero if the port is unique
 */
INTSML cli_uniqueport(NODEINST *ni, PORTPROTO *pp)
{
	REGISTER PORTPROTO *opp;
	static POLYGON *poly1 = NOPOLYGON, *poly2 = NOPOLYGON;

	/* make sure polygons are allocated */
	if (poly1 == NOPOLYGON) poly1 = allocstaticpolygon(4, cli_constraint->cluster);
	if (poly2 == NOPOLYGON) poly2 = allocstaticpolygon(4, cli_constraint->cluster);

	/* get polygon for original port */
	shapeportpoly(ni, pp, poly1, 0);

	/* look for a different port */
	for(opp = ni->proto->firstportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
	{
		if (opp == pp) continue;
		shapeportpoly(ni, opp, poly2, 0);

		/* if the polygons differ, this port is not unique */
		if (polysame(poly1, poly2) == 0) return(0);
	}

	/* all polygons the same: port is unique */
	return(1);
}

/*
 * routine to tell whether arc "ai" could be in any other layer
 * Returns nonzero if the layer is unique
 */
INTSML cli_uniquelayer(ARCINST *ai)
{
	REGISTER TECHNOLOGY *tech;
	REGISTER ARCPROTO *ap;
	REGISTER PORTPROTO *pp;
	REGISTER INTSML i;

	/* clear all bits on arcs in this technology */
	tech = ai->proto->tech;
	for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
		ap->temp1 = 0;

	/* set bits for the allowable arcs at one end of the arc */
	pp = ai->end[0].portarcinst->proto;
	for(i=0; pp->connects[i] != NOARCPROTO; i++)
		pp->connects[i]->temp1++;

	/* set bits for the allowable arcs at the other end of the arc */
	pp = ai->end[1].portarcinst->proto;
	for(i=0; pp->connects[i] != NOARCPROTO; i++)
		pp->connects[i]->temp1++;

	/* count the number of arcs that can connect to both ends */
	i = 0;
	for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
		if (ap->temp1 == 2) i++;

	/* arc layer is not unique if more than one arc can make the connection */
	if (i > 1) return(0);
	return(1);
}

/*
 * routine to add to the infinite string the size/location/rotation factors for
 * the component "ni" if it is not the default.  Returns the number of characters
 * that were added to the string
 */
INTSML cli_addcomponentinfo(NODEINST *ni)
{
	char line[50];
	REGISTER char *st;
	REGISTER INTSML added, i;
	REGISTER VARIABLE *var;
	INTBIG plx, phx, ply, phy;

	/* add in component size */
	added = 0;
	if (ni->highx-ni->lowx != ni->proto->highx-ni->proto->lowx ||
		ni->highy-ni->lowy != ni->proto->highy-ni->proto->lowy)
	{
		nodesizeoffset(ni, &plx, &ply, &phx, &phy);
		(void)sprintf(line, "[size=(%s,%s)", latoa(ni->highx-ni->lowx-plx-phx),
			latoa(ni->highy-ni->lowy-ply-phy));
		(void)addstringtoinfstr(line);
		added += strlen(line);
	}

	/* add in component rotation */
	if (ni->rotation != 0 || ni->transpose != 0)
	{
		if (added != 0) (void)addtoinfstr(','); else (void)addtoinfstr('[');
		(void)sprintf(line, "rotation=%s", latoa(ni->rotation*el_curtech->deflambda/10));
		(void)addstringtoinfstr(line);
		added += strlen(line) + 1;
		if (ni->transpose != 0)
		{
			(void)addtoinfstr('t');
			added++;
		}
	}

	/* add in component location */
	if (ni->firstportarcinst == NOPORTARCINST)
	{
		if (added != 0) (void)addtoinfstr(','); else (void)addtoinfstr('[');
		(void)sprintf(line, "location=(%s,%s)", latoa((ni->highx+ni->lowx) / 2),
			latoa((ni->highy+ni->lowy) / 2));
		(void)addstringtoinfstr(line);
		added += strlen(line) + 1;
	}

	/* add additional variables */
	if (ni->numvar != 0) for(i=0; i<ni->numvar; i++)
	{
		var = &ni->firstvar[i];
		if (var->key == el_node_name) continue;
		if ((var->type&VISARRAY) != 0) continue;

		if (added != 0) (void)addtoinfstr(','); else (void)addtoinfstr('[');
		added++;
		(void)addstringtoinfstr((char *)var->key);
		(void)addtoinfstr('=');
		added += strlen((char *)var->key) + 1;
		st = describevariable(var, -1, 1);
		added += strlen(st);
		(void)addstringtoinfstr(st);
	}

	if (added == 0) return(0);
	(void)addtoinfstr(']');
	return(added+1);
}

/*
 * routine to add any options of component "compo" to the infinite string
 */
void cli_addcomponentdata(COMPONENT *compo)
{
	char line[50];
	REGISTER ATTR *a;

	if ((compo->flag&(COMPSIZE|COMPROT|COMPLOC|COMPATTR)) != 0)
	{
		(void)addtoinfstr('[');
		if ((compo->flag&COMPSIZE) != 0)
		{
			(void)sprintf(line, "size=(%s,%s)", latoa(compo->sizex), latoa(compo->sizey));
			(void)addstringtoinfstr(line);
		}
		if ((compo->flag&COMPROT) != 0)
		{
			if ((compo->flag&COMPSIZE) != 0) (void)addtoinfstr(',');
			(void)sprintf(line, "rotation=%s", latoa(compo->rot*el_curtech->deflambda/10));
			(void)addstringtoinfstr(line);
			if (compo->trans != 0) (void)addtoinfstr('t');
		}
		if ((compo->flag&COMPLOC) != 0)
		{
			if ((compo->flag&(COMPSIZE|COMPROT)) != 0) (void)addtoinfstr(',');
			(void)sprintf(line, "location=(%s,%s)", latoa(compo->locx), latoa(compo->locy));
			(void)addstringtoinfstr(line);
		}
		if ((compo->flag&COMPATTR) != 0)
		{
			if ((compo->flag&(COMPSIZE|COMPROT|COMPLOC)) != 0) (void)addtoinfstr(',');
			for(a = compo->firstattr; a != NOATTR; a = a->nextattr)
			{
				if (a != compo->firstattr) (void)addtoinfstr(',');
				cli_addattr(a);
			}
		}
		(void)addtoinfstr(']');
	}
}

/*
 * routine to return the component name for node "ni"
 */
char *cli_componentname(NODEINST *ni)
{
	static char line[50];
	REGISTER VARIABLE *var;

	var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
	if (var == NOVARIABLE)
	{
		(void)sprintf(line, "NODE%ld", (INTBIG)ni);
		return(line);
	}
	return((char *)var->addr);
}

void cli_addcomponentline(char *str, WINDOWPART *win)
{
	REGISTER INTBIG i, total;

	total = askaid(us_aid, "edit-totallines", (INTBIG)win);

	/* search for the last line with a valid component declaration */
	for(i = total-1; i >= 0; i--)
		if (cli_linetype((char *)askaid(us_aid, "edit-getline", (INTBIG)win, i)) == LINEDECL) break;
	if (i < 0) i = 0;
	(void)askaid(us_aid, "edit-addline", (INTBIG)win, i+1, (INTBIG)str);
	cli_textlines++;
}

void cli_addattr(ATTR *a)
{
	char line[50];

	(void)addstringtoinfstr(a->name);
	(void)addtoinfstr('=');
	if ((a->type&VTYPE) == VINTEGER)
	{
		(void)sprintf(line, "%ld", a->value);
		(void)addstringtoinfstr(line);
	} else
	{
		(void)addtoinfstr('"');
		(void)addstringtoinfstr((char *)a->value);
		(void)addtoinfstr('"');
	}
}

/*
 * routine to add the name of nodeproto "np" to the inifinte string
 */
void cli_addnodeprotoname(NODEPROTO *np)
{
	if (np->primindex != 0)
	{
		if (np->tech != el_curtech)
		{
			(void)addstringtoinfstr(np->tech->techname);
			(void)addtoinfstr(':');
		}
		(void)addstringtoinfstr(np->primname);
	} else (void)addstringtoinfstr(np->cell->cellname);
}
