/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AccessToken,
  Amounts,
  Duration,
  j2s,
  PreparePayResultType,
  succeedOrThrow,
  TalerCorebankApiClient,
  TalerMerchantApi,
  TalerMerchantInstanceHttpClient,
  TalerWireGatewayAuth,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import {
  createWalletDaemonWithClient,
  withdrawViaBankV3,
} from "../harness/environments.js";
import {
  BankService,
  DbInfo,
  ExchangeService,
  FakebankService,
  getTestHarnessPaytoForLabel,
  GlobalTestState,
  HarnessExchangeBankAccount,
  MerchantService,
  setupDb,
  WalletClient,
  WalletService,
} from "../harness/harness.js";

interface MyTestEnvironment {
  commonDb: DbInfo;
  bank: BankService;
  bankClient: TalerCorebankApiClient;
  exchange: ExchangeService;
  exchangeBankAccount: HarnessExchangeBankAccount;
  merchant: MerchantService;
  merchantAdminAccessToken: AccessToken;
  walletClient: WalletClient;
  walletService: WalletService;
}

async function createMyEnvironment(
  t: GlobalTestState,
): Promise<MyTestEnvironment> {
  const db = await setupDb(t);

  const db2 = await setupDb(t, {
    nameSuffix: "dbtwo",
  });

  const bc = {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  };

  const bank: BankService = await FakebankService.create(t, bc);

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    httpPort: 8083,
    database: db.connStr,
  });

  const receiverName = "Exchange";
  const exchangeBankUsername = "exchange";
  const exchangeBankPassword = "mypw-password";
  const exchangePaytoUri = getTestHarnessPaytoForLabel(exchangeBankUsername);
  const wireGatewayApiBaseUrl = new URL(
    `accounts/${exchangeBankUsername}/taler-wire-gateway/`,
    bank.corebankApiBaseUrl,
  ).href;
  const wireGatewayAuth: TalerWireGatewayAuth = {
    username: exchangeBankUsername,
    password: exchangeBankPassword,
  };

  const exchangeBankAccount: HarnessExchangeBankAccount = {
    wireGatewayApiBaseUrl,
    wireGatewayAuth,
    accountPaytoUri: exchangePaytoUri,
  };

  await exchange.addBankAccount("1", exchangeBankAccount);

  bank.setSuggestedExchange(exchange, exchangeBankAccount.accountPaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "admin-password",
    },
  });

  await bankClient.registerAccountExtended({
    name: receiverName,
    password: exchangeBankPassword,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePaytoUri,
  });

  exchange.addOfferedCoins(defaultCoinConfig);

  await exchange.start();

  const exchange2 = ExchangeService.create(t, {
    currency: "TESTKUDOS",
    database: db2.connStr,
    httpPort: 9080,
    name: "exchangetwo",
  });

  exchange2.addOfferedCoins(defaultCoinConfig);

  await exchange.enableAccount("payto://void/foo?receiver-name=Foo");

  await exchange2.start();

  merchant.addExchange(exchange);
  merchant.addExchange(exchange2);
  await merchant.start();
  await merchant.pingUntilAvailable();

  const {accessToken: merchantAdminAccessToken} = await merchant.addInstanceWithWireAccount({
    id: "admin",
    name: "Default Instance",
    paytoUris: [`payto://void/merchant-default?receiver-name=Merchant`],
    defaultWireTransferDelay: Duration.toTalerProtocolDuration(
      Duration.fromSpec({ minutes: 1 }),
    ),
  });

  const { walletClient, walletService } = await createWalletDaemonWithClient(
    t,
    {
      name: "wallet",
      persistent: true,
    },
  );

  console.log("setup done!");

  return {
    commonDb: db,
    exchange,
    merchant,
    walletClient,
    walletService,
    bank,
    bankClient,
    merchantAdminAccessToken,
    exchangeBankAccount,
  };
}

/**
 * Test wallet:
 * - balance error messages
 * - different types of insufficient balance.
 *
 * Related bugs:
 * https://bugs.taler.net/n/7299
 */
export async function runWalletBalanceTest(t: GlobalTestState) {
  // Set up test environment

  const { merchant, walletClient, exchange, bankClient, merchantAdminAccessToken } =
    await createMyEnvironment(t);

  const merchantClient = new TalerMerchantInstanceHttpClient(
    merchant.makeInstanceBaseUrl(),
  );

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  console.log("withdrawal finished");

  const order: TalerMerchantApi.Order = {
    summary: "Buy me!",
    amount: "TESTKUDOS:5",
    fulfillment_url: "taler://fulfillment-success/thx",
  };

  console.log("creating order");

  const orderResp = succeedOrThrow(
    await merchantClient.createOrder(merchantAdminAccessToken, {
      order,
      payment_target: "void",
    }),
  );

  console.log("created order with merchant");

  let orderStatus = succeedOrThrow(
    await merchantClient.getOrderDetails(merchantAdminAccessToken, orderResp.order_id),
  );

  t.assertTrue(orderStatus.order_status === "unpaid");

  console.log("queried order at merchant");

  // Make wallet pay for the order

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: orderStatus.taler_pay_uri,
    },
  );

  console.log(j2s(preparePayResult));

  t.assertTrue(
    preparePayResult.status === PreparePayResultType.InsufficientBalance,
  );

  t.assertDeepEqual(
    preparePayResult.status,
    PreparePayResultType.InsufficientBalance,
  );

  t.assertTrue(
    Amounts.isNonZero(
      preparePayResult.balanceDetails.balanceReceiverAcceptable,
    ),
  );

  t.assertTrue(
    Amounts.isZero(preparePayResult.balanceDetails.balanceReceiverDepositable),
  );

  console.log("waiting for transactions to finalize");

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
}

runWalletBalanceTest.suites = ["wallet"];
