/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU Affero General Public License as
  published by the Free Software Foundation; either version 3,
  or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not,
  see <http://www.gnu.org/licenses/>
*/

/**
 * @file taler-merchant-httpd_post-challenge-ID-confirm.c
 * @brief endpoint to solve MFA challenge
 * @author Christian Grothoff
 */
#include "platform.h"
#include "taler-merchant-httpd.h"
#include "taler-merchant-httpd_mfa.h"
#include "taler-merchant-httpd_post-challenge-ID-confirm.h"


MHD_RESULT
TMH_post_challenge_ID_confirm (const struct TMH_RequestHandler *rh,
                               struct MHD_Connection *connection,
                               struct TMH_HandlerContext *hc)
{
  uint64_t challenge_serial;
  struct TALER_MERCHANT_MFA_BodyHash h_body;
  const char *tan;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_string ("tan",
                             &tan),
    GNUNET_JSON_spec_end ()
  };
  enum GNUNET_DB_QueryStatus qs;
  bool solved;
  uint32_t retry_counter;
  enum GNUNET_GenericReturnValue ret;

  ret = TMH_mfa_parse_challenge_id (hc,
                                    hc->infix,
                                    &challenge_serial,
                                    &h_body);
  if (GNUNET_OK != ret)
    return (GNUNET_SYSERR == ret) ? MHD_NO : MHD_YES;
  {
    enum GNUNET_GenericReturnValue res;

    res = TALER_MHD_parse_json_data (hc->connection,
                                     hc->request_body,
                                     spec);
    if (GNUNET_OK != res)
    {
      GNUNET_break_op (0);
      return (GNUNET_NO == res)
             ? MHD_YES
             : MHD_NO;
    }
  }
  qs = TMH_db->solve_mfa_challenge (TMH_db->cls,
                                    challenge_serial,
                                    &h_body,
                                    tan,
                                    &solved,
                                    &retry_counter);
  switch (qs)
  {
  case GNUNET_DB_STATUS_HARD_ERROR:
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (
      hc->connection,

      MHD_HTTP_INTERNAL_SERVER_ERROR,
      TALER_EC_GENERIC_DB_COMMIT_FAILED,
      "solve_mfa_challenge");
  case GNUNET_DB_STATUS_SOFT_ERROR:
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (
      hc->connection,
      MHD_HTTP_INTERNAL_SERVER_ERROR,
      TALER_EC_GENERIC_DB_SOFT_FAILURE,
      "solve_mfa_challenge");
  case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
    GNUNET_break_op (0);
    return TALER_MHD_reply_with_error (
      hc->connection,
      MHD_HTTP_NOT_FOUND,
      TALER_EC_MERCHANT_TAN_CHALLENGE_UNKNOWN,
      hc->infix);
  case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
    break;
  }
  if (0 == retry_counter)
  {
    return TALER_MHD_reply_with_error (
      hc->connection,
      MHD_HTTP_TOO_MANY_REQUESTS,
      TALER_EC_MERCHANT_TAN_TOO_MANY_ATTEMPTS,
      NULL);
  }
  if (! solved)
  {
    return TALER_MHD_REPLY_JSON_PACK (
      hc->connection,
      MHD_HTTP_CONFLICT,
      TALER_MHD_PACK_EC (TALER_EC_MERCHANT_TAN_CHALLENGE_FAILED),
      GNUNET_JSON_pack_uint64 ("retry_counter",
                               retry_counter));
  }
  return TALER_MHD_reply_static (
    hc->connection,
    MHD_HTTP_NO_CONTENT,
    NULL,
    NULL,
    0);
}
