/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

/* This #if encloses the rest of the file. */
#if !NO_HTML

#include <config.h>
#include <assert.h>
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "common.h"
#include "error.h"		  
#include "output.h"
#include "som.h"
#include "filename.h"
#include "version.h"
#include "getline.h"
#include "tab.h"
#include "htmlP.h"

/* Prototypes. */
static int postopen (file_ext *);
static int preclose (file_ext *);

int
html_open_global (unused outp_class *this)
{
  return 1;
}

int
html_close_global (unused outp_class *this)
{
  return 1;
}

int
html_preopen_driver (outp_driver *this)
{
  html_driver_ext *x;

  assert (this->driver_open == 0);
  verbose_msg (1, _("HTML driver initializing as `%s'..."), this->name);

  this->ext = x = (html_driver_ext *) xmalloc (sizeof (html_driver_ext));
  this->res = 0;
  this->horiz = this->vert = 0;
  this->width = this->length = 0;

  this->cp_x = this->cp_y = 0;

  x->prologue_fn = NULL;

  x->file.filename = NULL;
  x->file.mode = "w";
  x->file.file = NULL;
  x->file.sequence_no = &x->sequence_no;
  x->file.param = this;
  x->file.postopen = postopen;
  x->file.preclose = preclose;

  x->sequence_no = 0;

  return 1;
}

int
html_postopen_driver (outp_driver *this)
{
  html_driver_ext *x = this->ext;

  assert (this->driver_open == 0);
  if (NULL == x->file.filename)
    x->file.filename = xstrdup ("pspp.html");
	
  if (x->prologue_fn == NULL)
    x->prologue_fn = xstrdup ("html-prologue");

  verbose_msg (2, _("%s: Initialization complete."), this->name);
  this->driver_open = 1;

  return 1;
}

int
html_close_driver (outp_driver *this)
{
  html_driver_ext *x = this->ext;

  assert (this->driver_open);
  verbose_msg (2, _("%s: Beginning closing..."), this->name);
  close_file_ext (&x->file);
  free (x->prologue_fn);
  free (x->file.filename);
  free (x);
  verbose_msg (3, _("%s: Finished closing."), this->name);
  this->driver_open = 0;
  
  return 1;
}

/* Generic option types. */
enum
{
  boolean_arg = -10,
  string_arg,
  nonneg_int_arg
};

/* All the options that the HTML driver supports. */
static outp_option option_tab[] =
{
  /* *INDENT-OFF* */
  {"output-file",		1,		0},
  {"prologue-file",		string_arg,	0},
  {"", 0, 0},
  /* *INDENT-ON* */
};
static outp_option_info option_info;

void
html_option (outp_driver *this, const char *key, const outp_value *val)
{
  html_driver_ext *x = this->ext;
  int cat, subcat;
  a_string value;

  value = val->string;

  cat = outp_match_keyword (key, option_tab, &option_info, &subcat);
  if (val->type != 'a')
    {
      msg (SE, _("String value expected for key `%s'."), key);
      return;
    }

  switch (cat)
    {
    case 0:
      msg (SE, _("Unknown configuration parameter `%s' for HTML device "
	   "driver."), key);
      break;
    case 1:
      free (x->file.filename);
      x->file.filename = sa_strdup (&value);
      break;
    case string_arg:
      {
	char **dest;
	switch (subcat)
	  {
	  case 0:
	    dest = &x->prologue_fn;
	    break;
	  default:
	    assert (0);
	  }
	if (*dest)
	  free (*dest);
	*dest = sa_strdup (&value);
      }
      break;
#if __CHECKER__
    case 42000:
      assert (0);
#endif
    default:
      assert (0);
    }
}

/* Variables for the prologue. */
typedef struct
  {
    const char *key;
    const char *value;
  }
html_variable;
  
static html_variable *html_var_tab;

/* Searches html_var_tab for a html_variable with key KEY, and returns
   the associated value. */
static const char *
html_get_var (const char *key)
{
  html_variable *v;

  for (v = html_var_tab; v->key; v++)
    if (streq (key, v->key))
      return v->value;
  return NULL;
}

/* Writes the HTML prologue to file F. */
static int
postopen (file_ext *f)
{
  static html_variable dict[] =
    {
      {"generator", 0},
      {"date", 0},
      {"user", 0},
      {"host", 0},
      {"title", 0},
      {"subtitle", 0},
      {"source-file", 0},
      {0, 0},
    };
#if HAVE_UNISTD_H
  char host[128];
#endif
  time_t curtime;
  struct tm *loctime;

  outp_driver *this = f->param;
  html_driver_ext *x = this->ext;

  char *prologue_fn = search_path (x->prologue_fn, config_path, NULL);
  FILE *prologue_file;

  char *buf = NULL;
  int buf_size = 0;

  if (prologue_fn == NULL)
    {
      msg (IE, _("Cannot find HTML prologue.  The use of `-vv' "
		 "on the command line is suggested as a debugging aid."));
      return 0;
    }

  verbose_msg (1, _("%s: %s: Opening HTML prologue..."),
	       this->name, prologue_fn);
  prologue_file = fopen (prologue_fn, "rb");
  if (prologue_file == NULL)
    {
      fclose (prologue_file);
      free (prologue_fn);
      msg (IE, "%s: %s", prologue_fn, strerror (errno));
      goto error;
    }

  dict[0].value = version;

  curtime = time (NULL);
  loctime = localtime (&curtime);
  dict[1].value = asctime (loctime);
  {
    char *cp = strchr (dict[1].value, '\n');
    if (cp)
      *cp = 0;
  }

  /* PORTME: Determine username, net address. */
#if HAVE_UNISTD_H
  dict[2].value = getenv ("LOGNAME");
  if (!dict[2].value)
    dict[2].value = getlogin ();
  if (!dict[2].value)
    dict[2].value = _("nobody");

  if (gethostname (host, 128) == -1)
    {
      if (errno == ENAMETOOLONG)
	host[127] = 0;
      else
	strcpy (host, _("nowhere"));
    }
  dict[3].value = host;
#else /* !HAVE_UNISTD_H */
  dict[2].value = _("nobody");
  dict[3].value = _("nowhere");
#endif /* !HAVE_UNISTD_H */

  dict[4].value = outp_title ? outp_title : "";
  dict[5].value = outp_subtitle ? outp_subtitle : "";
  
  dict[6].value = curfn ? curfn : "<stdin>";

  html_var_tab = dict;
  while (-1 != getline (&buf, &buf_size, prologue_file))
    {
      char *buf2;
      int len;

      if (strstr (buf, "!!!"))
	continue;
      
      {
	char *cp = strstr (buf, "!title");
	if (cp)
	  {
	    if (outp_title == NULL)
	      continue;
	    else
	      *cp = '\0';
	  }
      }
      
      {
	char *cp = strstr (buf, "!subtitle");
	if (cp)
	  {
	    if (outp_subtitle == NULL)
	      continue;
	    else
	      *cp = '\0';
	  }
      }
      
      /* PORTME: Line terminator. */
      buf2 = interp_vars (buf, html_get_var);
      len = strlen (buf2);
      fwrite (buf2, len, 1, f->file);
      if (buf2[len - 1] != '\n')
	putc ('\n', f->file);
      free (buf2);
    }
  if (ferror (f->file))
    msg (IE, _("Reading `%s': %s."), prologue_fn, strerror (errno));
  fclose (prologue_file);

  free (prologue_fn);
  free (buf);

  if (ferror (f->file))
    goto error;

  verbose_msg (2, _("%s: HTML prologue read successfully."), this->name);
  return 1;

error:
  verbose_msg (1, _("%s: Error reading HTML prologue."), this->name);
  return 0;
}

/* Writes the HTML epilogue to file F. */
static int
preclose (file_ext *f)
{
  fprintf (f->file,
	   "</BODY>\n"
	   "</HTML>\n"
	   "<!-- end of file -->\n");

  if (ferror (f->file))
    return 0;
  return 1;
}

int
html_open_page (outp_driver *this)
{
  html_driver_ext *x = this->ext;

  assert (this->driver_open && this->page_open == 0);
  x->sequence_no++;
  if (!open_file_ext (&x->file))
    {
      if (errno)
	msg (ME, _("HTML output driver: %s: %s"), x->file.filename,
	     strerror (errno));
      return 0;
    }

  if (!ferror (x->file.file))
    this->page_open = 1;
  return !ferror (x->file.file);
}

int
html_close_page (outp_driver *this)
{
  html_driver_ext *x = this->ext;

  assert (this->driver_open && this->page_open);
  this->page_open = 0;
  return !ferror (x->file.file);
}

static void output_tab_table (outp_driver *, tab_table *);

void
html_submit (outp_driver *this, som_table *s)
{
  extern som_table_class tab_table_class;
  html_driver_ext *x = this->ext;
  
  assert (this->driver_open && this->page_open);
  if (x->sequence_no == 0 && !html_open_page (this))
    {
      msg (ME, _("Cannot open first page on HTML device %s."), this->name);
      return;
    }

  if (s->class == &tab_table_class)
    output_tab_table (this, (tab_table *) s->ext);
  else
    assert (0);
}

/* Emit HTML to FILE to change from *OLD_ATTR attributes to NEW_ATTR.
   Sets *OLD_ATTR to NEW_ATTR when done. */
static void
change_attributes (FILE *f, int *old_attr, int new_attr)
{
  if (*old_attr == new_attr)
    return;

  if (*old_attr & OUTP_F_B)
    fputs ("</B>", f);
  if (*old_attr & OUTP_F_I)
    fputs ("</I>", f);
  if (new_attr & OUTP_F_I)
    fputs ("<I>", f);
  if (new_attr & OUTP_F_B)
    fputs ("<B>", f);

  *old_attr = new_attr;
}

/* Write string S of length LEN to file F, escaping characters as
   necessary for HTML.  If RICH is non-zero then interprets the string
   as rich text. */
static void
escape_string (FILE *f, char *s, int len, int rich)
{
  char *ep = &s[len];
  char *bp, *cp;
  int attr = 0;

  for (bp = cp = s; bp < ep; bp = cp)
    {
      while (cp < ep && *cp != '&' && *cp != '<' && *cp != '>' && *cp)
	cp++;
      if (cp > bp)
	fwrite (bp, 1, cp - bp, f);
      if (cp < ep)
	switch (*cp++)
	  {
	  case '&':
	    fputs ("&amp;", f);
	    break;
	  case '<':
	    fputs ("&lt;", f);
	    break;
	  case '>':
	    fputs ("&gt;", f);
	    break;
	  case 0:
	    if (!rich)
	      break;
	    if (cp[0] == 0 && cp[1] == 0)
	      {
		cp += 2;
		break;
	      }
	    
	    {
	      int escape_size = LOAD_2 (cp);

	      switch (cp[2])
		{
		case TAG_FONT_BY_NAME:
		  {
		    int name_len = escape_size - 4;
		    char *suffix = &cp[escape_size - 2];

		    if (name_len > 1)
		      {
			if (streq (&suffix[-2], "BI"))
			  change_attributes (f, &attr, OUTP_F_BI);
		      }
		    else if (name_len > 0)
		      {
			if (suffix[-1] == 'I')
			  change_attributes (f, &attr, OUTP_F_I);
			else if (suffix[-1] == 'B')
			  change_attributes (f, &attr, OUTP_F_B);
			else
			  change_attributes (f, &attr, OUTP_F_R);
		      }
		  }
		break;
		case TAG_FONT_BY_FAMILY:
		  /* no-op */
		  break;
		case TAG_FONT_BY_POSITION:
		  change_attributes (f, &attr, cp[3]);
		  break;
		case TAG_COLOR:
		  /* FIXME: add colors? */
		  break;
		case TAG_NO_OP:
		  /* no-op */
		  break;
		default:
		  assert (0);
		}
	      
	      cp += escape_size - 1;
	    }
	    break;
	      
	  default:
	    assert (0);
	  }
    }

  if (attr)
    change_attributes (f, &attr, 0);
}
  
/* Write table T to THIS output driver. */
static void
output_tab_table (outp_driver *this, tab_table *t)
{
  html_driver_ext *x = this->ext;
  
  tab_hit++;

  if (t->nr == 1 && t->nc == 1)
    {
      fputs ("<P>", x->file.file);
      if (t->cc->s != NULL)
	escape_string (x->file.file, t->cc->s, t->cc->l, 0);
      fputs ("</P>\n", x->file.file);
      
      return;
    }

  fputs ("<TABLE BORDER=1>\n", x->file.file);
  
  if (t->title.s)
    {
      fprintf (x->file.file, "  <TR>\n    <TH COLSPAN=%d>", t->nc);
      escape_string (x->file.file, t->title.s, t->title.l, 0);
      fputs ("</TH>\n  </TR>\n", x->file.file);
    }
  
  {
    int r;
    a_string *cc = t->cc;
    unsigned char *ct = t->ct;

    for (r = 0; r < t->nr; r++)
      {
	int c;
	
	fputs ("  <TR>\n", x->file.file);
	for (c = 0; c < t->nc; c++, cc++, ct++)
	  {
	    int tag;
	    char header[128];
	    char *cp;

	    if ((*ct & TAB_JOIN)
		&& ((tab_joined_cell *) cc->s)->hit == tab_hit)
	      continue;

	    if (r < t->t || r >= t->nr - t->b
		|| c < t->l || c >= t->nc - t->r)
	      tag = 'H';
	    else
	      tag = 'D';
	    cp = stpcpy (header, "    <T");
	    *cp++ = tag;
	    
	    switch (*ct & TAB_ALIGN_MASK)
	      {
	      case TAB_RIGHT:
		cp = stpcpy (cp, " ALIGN=RIGHT");
		break;
	      case TAB_LEFT:
		break;
	      case TAB_CENTER:
		cp = stpcpy (cp, " ALIGN=CENTER");
		break;
	      default:
		assert (0);
	      }

	    if (*ct & TAB_JOIN)
	      {
		tab_joined_cell *j = (tab_joined_cell *) cc->s;
		j->hit = tab_hit;
		
		if (j->x2 - j->x1 > 1)
		  cp = spprintf (cp, " COLSPAN=%d", j->x2 - j->x1);
		if (j->y2 - j->y1 > 1)
		  cp = spprintf (cp, " ROWSPAN=%d", j->y2 - j->y1);
	      }
	    
	    strcpy (cp, ">");
	    fputs (header, x->file.file);
	    
	    {
	      char *s = cc->s;
	      size_t l = cc->l;

	      while (l && isspace ((unsigned char) *s))
		{
		  l--;
		  s++;
		}
	      
	      escape_string (x->file.file, s, l, *ct & TAB_RICH);
	    }

	    fprintf (x->file.file, "</T%c>\n", tag);
	  }
	fputs ("  </TR>\n", x->file.file);
      }
  }
	      
  fputs ("</TABLE>\n\n", x->file.file);
}

/* HTML driver class. */
outp_class html_class =
{
  "html",
  0xfaeb,
  1,

  html_open_global,
  html_close_global,
  NULL,

  html_preopen_driver,
  html_option,
  html_postopen_driver,
  html_close_driver,

  html_open_page,
  html_close_page,

  html_submit,

  NULL,
  NULL,
  NULL,

  NULL,
  NULL,
  NULL,
  NULL,

  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
};

#endif /* !NO_HTML */

