/*
 *
 * $Source: /filesv/usr/local/proj/sphinx/spx2/src/lib/api/RCS/gssapi.c,v $
 *
 *
 *  MODULE NAME:    gssapi.c
 *
 *
 *  AUTHORS:
 *
 *	K. Alagappan
 *
 */


/*
 * COPYRIGHT (C) 1992 DIGITAL EQUIPMENT CORPORATION
 * ALL RIGHTS RESERVED
 *
 * "Digital Equipment Corporation authorizes the reproduction,
 * distribution and modification of this software subject to the following
 * restrictions:
 * 
 * 1.  Any partial or whole copy of this software, or any modification
 * thereof, must include this copyright notice in its entirety.
 *
 * 2.  This software is supplied "as is" with no warranty of any kind,
 * expressed or implied, for any purpose, including any warranty of fitness 
 * or merchantibility.  DIGITAL assumes no responsibility for the use or
 * reliability of this software, nor promises to provide any form of 
 * support for it on any basis.
 *
 * 3.  Distribution of this software is authorized only if no profit or
 * remuneration of any kind is received in exchange for such distribution. 
 * 
 * 4.  This software and all application programs are to be used only for
 * non-commercial purposes. However, media costs associated with the
 * distribution of the software or application programs may be recovered.
 *
 */


#include <stdio.h>
#include <syslog.h>
#include <pwd.h>
#include <sys/types.h>
#include "cdc.h"
#include "cdc_db.h"
#include "SPHINX-types.h"
#include "BigNum.h"
#include "BigRSA.h"
#include "random.h"
#include "spxapi_defs.h"
#include "objid.h"
#include "odd.h"
#include "gssapi_defs.h"

#ifdef sun
#define strncasecmp	strncmp_support
#define strcasecmp	strcmp_support
#endif

gss_cred_id_t  gss_default_credentials = NULL;
static char  expected_response[10];

int  encode_SPHINX_LongPosixTime();

/*
 * Offering "standard" GSS API for following mechanism(s) :  SPX
 *
 * Supported jacket routines :
 *
 *     gss_acquire_cred             Assume a global identity
 *
 *     gss_release_cred             Discard credentials
 *
 *     gss_init_sec_context         Initiate a security context with a
 *                                  peer application
 *
 *     gss_accept_sec_context       Accept a security context from a
 *                                  peer application
 *
 *     gss_display_status           Convert an API status code to text
 *
 *     gss_indicate_mechs           Determine underlying mechanism
 *
 *     gss_display_name             Convert opaque name to text
 *
 *     gss_import_name              Convert a textual name to API-format
 *
 *     gss_release_name             Deallocate API internal name
 *
 *     gss_release_buffer           Deallocate a buffer descriptor
 *
 *     gss_release_oid_set          Deallocate a set of object identifiers
 *
 *     gss_delete_sec_context	    Delete a security context.
 *
 * Unofficial jacket routines :
 *
 *     gss__stash_default_cred      Bind credential handle as default
 *
 *     gss__check_acl               Check if principal is listed in ACL
 *                                  file
 *
 *     gss__extract_key_from_context  Extract shared DES key from context
 *                                    handle
 *
 *     gss__globalname_to_local     Convert user's global identity to a local
 *                                  name based on a translation file
 *
 *     gss__release_context_handle  Deallocate context handle
 */

/*
 * gss_acquire_cred() - Allows an application to acquire a handle for
 *   a pre-existing credential by name.
 *
 *   OM_uint32   *minor_status  (output) - mechanism specific status code
 *   gss_name_t  desired_name   (input)  - name of principals whose
 *                                         credentials should be acquired
 *   OM_uint32   time_req       (input)  - number of seconds that credentials
 *                                         should remain valid
 *   gss_OID_set desired_mechs  (input)  - set of underlying security
 *                                         mechanisms that may be used
 *                                         (GSS_C_NULL_OID to obtain
 *                                         implementation specific default)
 *   int         cred_usage     (input)  - credentials usage (both, initiate,
 *                                         accept)
 *   gss_cred_id_t *output_cred_handle (output) - the returned credential
 *                                         handle
 *   gss_OID_set *actual_mechs  (output) - set of mechanisms for which the
 *                                         credential is valid
 *   OM_uint32   *time_rec      (output) - actual number of seconds for which
 *                                         the returned credentials will
 *                                         remain valid
 *
 */

OM_uint32 gss_acquire_cred(minor_status, desired_name, time_req,
			   desired_mechs, cred_usage, output_cred_handle,
			   actual_mechs, time_rec)
OM_uint32        *minor_status;
gss_name_t       desired_name;
OM_uint32        time_req;
gss_OID_set      desired_mechs;
int              cred_usage;
gss_cred_id_t    *output_cred_handle;
gss_OID_set      *actual_mechs;
OM_uint32        *time_rec;
{
  int  status, j, mechvalid=0;
  char *spx_cred_handl, *spx_ta_cred_handl;
  SPXCredentials *acquired_cred_handle;
  gss_OID_set    output_mechs;

  /*
   * Initial implementation only supports the default mech type
   */

  if (desired_mechs != GSS_C_NULL_OID_SET) {
    if (bcmp(desired_mechs->elements->elements, SPX_MECHTYPE_VALUE, SPX_MECHTYPE_SZ) != 0)
      return(GSS_S_BAD_MECH);
  }

  /*
   * Return actual mechanism used
   */

  output_mechs = (gss_OID_set) malloc(sizeof(gss_OID_set_d));
  *actual_mechs = output_mechs;
  output_mechs->count = 1;
  output_mechs->elements = &SPX_MECHTYPE_OID;

  acquired_cred_handle = (SPXCredentials *) malloc(sizeof(SPXCredentials));
  *output_cred_handle = (gss_cred_id_t) acquired_cred_handle;

  if (cred_usage == GSS_C_BOTH)  return(GSS_S_FAILURE);

  status = Sphinx_Acquire_cred((char *) desired_name,
			       NULL,
			       cred_usage,
			       &spx_cred_handl,
			       &spx_ta_cred_handl,
			       0);
  *minor_status = status;
  if (status < 0) {
    *output_cred_handle = GSS_C_NO_CREDENTIAL;
    return(GSS_S_FAILURE);
  }
  if (cred_usage == GSS_C_INITIATE) {
    acquired_cred_handle->clm_handle = (ClaimantCred *) spx_cred_handl;
    acquired_cred_handle->ver_handle = (VerifierCred *) NULL;
  } else {
    acquired_cred_handle->clm_handle = (ClaimantCred *) NULL;
    acquired_cred_handle->ver_handle = (VerifierCred *) spx_cred_handl;
  }
  acquired_cred_handle->ta_handle = (TrustedCred *) spx_ta_cred_handl;
  bzero(acquired_cred_handle->credential_file,sizeof(acquired_cred_handle->credential_file));
  if (time_rec != NULL) *time_rec = time_req;
  return(GSS_S_COMPLETE);
}

/*
 * gss_release_cred() - Informs GSS API that the specified credential handle
 * is no longer required by the process.
 *
 *   OM_uint32   *minor_status  (output) - mechanism specific status code
 *   gss_cred_id_t *cred_handle (input)  - handle for credentials
 *                                         claimed.  Supply GSS_C_NO_CREDENTIAL
 *                                         to use default credentials.
 *
 */

OM_uint32 gss_release_cred(minor_status, cred_handle)
OM_uint32        *minor_status;
gss_cred_id_t    *cred_handle;
{
  gss_cred_id_t  actual_cred_handle;
  SPXCredentials *gsscred;

  if (cred_handle == GSS_C_NO_CREDENTIAL) {
    if (!gss_default_credentials) return(GSS_S_COMPLETE);
    actual_cred_handle = gss_default_credentials;
  } else actual_cred_handle = *cred_handle;

  gsscred = (SPXCredentials *) actual_cred_handle;
  if (gsscred->clm_handle) free(gsscred->clm_handle);
  if (gsscred->ver_handle) free(gsscred->ver_handle);
  if (gsscred->ta_handle) free(gsscred->ta_handle);
  free(gsscred);

  if (GSS_C_NO_CREDENTIAL) gss_default_credentials = NULL;

  return(GSS_S_COMPLETE);
}

/*
 * gss_init_sec_context() - Initiates the establishment of a security context
 *   between the application and a remote peer.  Initially, the input_token
 *   parameter should be specified as GSS_C_NO_BUFFER.  The routine may return
 *   an output token which should be transferred to the peer application, where
 *   the peer application will present it to gss_accept_context..  If one or
 *   more reply tokens are required from the peer application, this routine
 *   will return a status value of GSS_S_CONTINUE_NEEDED in which case it
 *   should be called again when the reply token is received from the peer
 *   application, passing the token to gss_init_sec_context via the input
 *   token parameter.
 *
 *   OM_uint32   *minor_status  (output) - mechanism specific status code
 *   gss_cred_id_t claimant_cred_handle (input) - handle for credentials
 *                                         claimed.  Supply GSS_C_NO_CREDENTIAL
 *                                         to use default credentials.
 *   gss_ctx_id_t  *context_handle (output) - context handle for new context
 *   gss_name_t    target_name  (input)  - name of target
 *   gss_OID       mech_type    (input)  - mechanism type, GSS_C_NULL_OID
 *                                         specifies default
 *   int           req_flags    (input)  - request flags for context services
 *   int           time_req     (input)  - desired number of seconds for which
 *                                         context should remain valid,
 *                                         specify 0 for default
 *   gss_channel_bindings input_chan_bindings (input) - application specified
 *                                         bindings.  Allows application to
 *                                         securely bind channel information
 *                                         to security context
 *   gss_buffer_t input_token   (input)  - token received from peer, supply
 *                                         GSS_C_NO_BUFFER for initial call
 *   gss_OID      *actual_mech_type (output)  - actual mechanism used
 *   gss_buffer_t output_token  (output) - token to be sent to peer
 *   int          *ret_flags    (output) - returned flags which indicate that
 *                                         the context supports these services
 *   OM_uint32    *time_rec     (output) - number of seconds for which the
 *                                         context will remain valid
 *
 */

int gss_init_sec_context(minor_status, claimant_cred_handle, context_handle,
        target_name, mech_type, req_flags, time_req, input_chan_bindings,
	 input_token, actual_mech_type, output_token, ret_flags, time_rec)
OM_uint32        *minor_status;
gss_cred_id_t    claimant_cred_handle;
gss_ctx_id_t     *context_handle;
gss_name_t       target_name;
gss_OID          mech_type;
int              req_flags;
int              time_req;
gss_channel_bindings input_chan_bindings;
gss_buffer_t     input_token;
gss_OID          *actual_mech_type;
gss_buffer_t     output_token;
int              *ret_flags;
OM_uint32        *time_rec;
{
  int status, deleg_flag = 0, mutual_flag = 0;
  static char     output_token_buf[GSS_C_MAX_TOKEN];
  char            chanbinding[GSS_C_MAX_CHAN_BINDINGS];
  SPXCredentials  *gsscred;
  SPXContextHandle  *gssctxhandle;
  ContextHandle   *ctxhandle = NULL;
  gss_OID         output_mech;
  struct passwd   *pwd;
  int             wrapper_len, wrapper_adj, token_len, chanbinding_len;
  int             ret_flags_actual = 0, header_len = 0, i;
  char *spx_cred_handl, *spx_ta_cred_handl;
  DESblock   deskey;
  DESblock   mac;

  /*
   *  if we don't have claimant credentials, acquire default credentials
   *  for user
   */
  if (claimant_cred_handle == GSS_C_NO_CREDENTIAL) {
    if (!gss_default_credentials) {
      pwd = getpwuid(getuid());
      if (!pwd) return(GSS_S_NO_CRED);
      gsscred = (SPXCredentials *) malloc(sizeof(SPXCredentials));
      status = Sphinx_Acquire_cred(pwd->pw_name,
				   NULL,
				   GSS_C_INITIATE,
				   &spx_cred_handl,
				   &spx_ta_cred_handl,
				   0);
      *minor_status = status;
      if (status < 0)  return(GSS_S_NO_CRED);
      gsscred->clm_handle = (ClaimantCred *) spx_cred_handl;
      gsscred->ver_handle = (VerifierCred *) NULL;
      gsscred->ta_handle = (TrustedCred *) spx_ta_cred_handl;
      bzero(gsscred->credential_file, sizeof(gsscred->credential_file));
      gss_default_credentials = (void *) gsscred;
    }
  } else gsscred = (SPXCredentials *) claimant_cred_handle;

  if (input_token == GSS_C_NO_BUFFER) {
    /*
     *  Check mechanism type
     */
    if (mech_type != GSS_C_NULL_OID) {
      if (bcmp(mech_type->elements, SPX_MECHTYPE_VALUE, SPX_MECHTYPE_SZ) != 0)
	return(GSS_S_BAD_MECH);
    }
    output_mech = &SPX_MECHTYPE_OID;
    *actual_mech_type = output_mech;

    deleg_flag = (req_flags & GSS_C_DELEG_FLAG);
    mutual_flag = ((req_flags & GSS_C_MUTUAL_FLAG) >> 1);

    ret_flags_actual = ((deleg_flag) | (mutual_flag << 1));
    *ret_flags = ret_flags_actual;

/*
 *  Add GSSAPI wrapper  - support only SPX_MECHTYPE_OID for now
 */
    output_token_buf[0] = 0xa0;      /*  [APPLICATION 0] IMPLICIT SEQUENCE  */
    output_token_buf[1] = 0x82;      /*  definite length encoding           */
                                     /*    assuming that internal token     */
                                     /*    is between 256 and 65535 bytes   */
    output_token_buf[2] = 0x0;       /*  zero for now   */
    output_token_buf[3] = 0x0;       /*  zero for now   */
    bcopy(SPX_MECHTYPE_VALUE,&output_token_buf[4],SPX_MECHTYPE_SZ);
    wrapper_len = SPX_MECHTYPE_SZ+4;

    bcopy(input_chan_bindings->initiator_address.value, &chanbinding[0],
	  input_chan_bindings->initiator_address.length);
    chanbinding_len = input_chan_bindings->initiator_address.length;
    bcopy(input_chan_bindings->acceptor_address.value,
	  &chanbinding[chanbinding_len],
	  input_chan_bindings->acceptor_address.length);
    chanbinding_len = chanbinding_len +
      input_chan_bindings->acceptor_address.length;

    if (input_chan_bindings->application_data.length != 0) {
      bcopy(input_chan_bindings->application_data.value,
	    &chanbinding[chanbinding_len],
	    input_chan_bindings->application_data.length);
      chanbinding_len = chanbinding_len +
	input_chan_bindings->application_data.length;
    }

    status = Sphinx_Init_sec_context(gsscred->clm_handle,
				     gsscred->ta_handle,
				     &ctxhandle,
				     (char *) target_name,
				     deleg_flag,
				     mutual_flag,
				     chanbinding,
				     chanbinding_len,
				     NULL,   /*  input_token->value   */
				     0,      /*  input_token->length  */
				     &output_token_buf[wrapper_len],
				     &token_len,
				     0);
    *minor_status = status;

    if ((status != SPX_S_INIT_SUCCESS) && (status != SPX_S_INIT_CONTINUE)) {
      *context_handle = 0;
      return(GSS_S_FAILURE);
    }

    if (token_len+SPX_MECHTYPE_SZ > 256) {
      output_token_buf[2] = (token_len+SPX_MECHTYPE_SZ) / 256;
      output_token_buf[3] = (token_len+SPX_MECHTYPE_SZ) % 256;
      wrapper_adj = 0;
    } else {
      if (token_len+SPX_MECHTYPE_SZ > 128) {
	output_token_buf[1] = 0xa0;
	output_token_buf[2] = 0x81;
	wrapper_adj = 1;
      } else {
	output_token_buf[2] = 0xa0;
	wrapper_adj = 2;
      }
      output_token_buf[3] = (token_len+SPX_MECHTYPE_SZ);
    }
    output_token->length = token_len+wrapper_len - wrapper_adj;
    output_token->value = output_token_buf+wrapper_adj;

    gssctxhandle = (SPXContextHandle *) malloc(sizeof(SPXContextHandle));
    *context_handle = (gss_ctx_id_t) gssctxhandle;
    bcopy(SPX_MECHTYPE_VALUE, gssctxhandle->mechtype, SPX_MECHTYPE_SZ);
    gssctxhandle->mechlen = SPX_MECHTYPE_SZ;
    gssctxhandle->context_handle = ctxhandle;

    if (status == SPX_S_INIT_CONTINUE)  return(GSS_S_CONTINUE_NEEDED);
    return(GSS_S_COMPLETE);
  } else {
    header_len = 0;
    if (!context_handle)  return(GSS_S_FAILURE);
    gssctxhandle = (SPXContextHandle *) *context_handle;
    if (!gssctxhandle)  return(GSS_S_FAILURE);
    if (bcmp(gssctxhandle->mechtype, SPX_MECHTYPE_VALUE, gssctxhandle->mechlen)
	!= 0)
      return(GSS_S_BAD_MECH);
    ctxhandle = (ContextHandle *) gssctxhandle->context_handle;

    if (ctxhandle != NULL) {
      if (input_token->length >= ctxhandle->mutual_len) {
	output_token->length = 0;
	output_token->value = NULL;
	if ((unsigned char) input_token->value[0] == 0x81)
	  input_token->value[0] = 0x80;

	if (!bcmp(ctxhandle->mutual_resp, input_token->value,
		  ctxhandle->mutual_len)) {
	  if (input_token->length > ctxhandle->mutual_len) {
	    /*
	     *  mutual response matches, however we also have received
	     *  additional bytes.  This must be a challenge since our
	     *  clocks are out of sync.  Generate the response and return
	     *  as the output_token to be sent over.
	     */
	    bcopy((char *) ctxhandle->deskey.bytes, deskey.bytes, sizeof(deskey.bytes));
	    DES_X9_MAC(&deskey,
		       &input_token->value[ctxhandle->mutual_len],
		       (input_token->length - ctxhandle->mutual_len),
		       &mac);

	    output_token_buf[0] = 0xa7;
	    output_token_buf[1] = 8;
	    bcopy(&mac, &output_token_buf[2], 8);
	    output_token->length = 10;
	    output_token->value = (char *) output_token_buf;
	    printf("SPX-info : clocks out of sync, trying challenge/response\n");
	  }
	  return(GSS_S_COMPLETE);
	}
	*minor_status = SPX_S_MUTUAL_RESPONSE_BAD;
	return(GSS_S_FAILURE);
      }
    }

    if ((unsigned char) input_token->value[0] != 0xa6) {
      /*
       *  since input_token isn't valid, return bad MUTUAL_RESPONSE
       *  it is probably an error status message.
       */
      *minor_status = SPX_S_MUTUAL_RESPONSE_BAD;
      return(GSS_S_FAILURE);
    }

    header_len++;
    switch ((unsigned char) input_token->value[header_len]) {
      case 0x82 :
	header_len = header_len+3;
	break;
      case 0x81 :
	header_len = header_len + 2;
	break;
      default :  /* for indefinite length and token smaller than 128 bytes */
	header_len++;
	break;
    }

/*
 *  since input_token->length doesn't match expected mutual reponse length,
 *  process the input_token in Sphinx_Init_sec_context()
 */

    deleg_flag = (req_flags & GSS_C_DELEG_FLAG);
    mutual_flag = ((req_flags & GSS_C_MUTUAL_FLAG) >> 1);

    ret_flags_actual = ((deleg_flag) | (mutual_flag << 1));
    *ret_flags = ret_flags_actual;

    bcopy(input_chan_bindings->initiator_address.value, &chanbinding[0],
	  input_chan_bindings->initiator_address.length);
    chanbinding_len = input_chan_bindings->initiator_address.length;
    bcopy(input_chan_bindings->acceptor_address.value,
	  &chanbinding[chanbinding_len],
	  input_chan_bindings->acceptor_address.length);
    chanbinding_len = chanbinding_len +
      input_chan_bindings->acceptor_address.length;

    if (input_chan_bindings->application_data.length != 0) {
      bcopy(input_chan_bindings->application_data.value,
	    &chanbinding[chanbinding_len],
	    input_chan_bindings->application_data.length);
      chanbinding_len = chanbinding_len +
	input_chan_bindings->application_data.length;
    }

    status = Sphinx_Init_sec_context(gsscred->clm_handle,
				     gsscred->ta_handle,
				     &ctxhandle,
				     (char *) target_name,
				     deleg_flag,
				     mutual_flag,
				     chanbinding,
				     chanbinding_len,
				     &input_token->value[header_len],
				     input_token->length - header_len,
				     output_token_buf,
				     &token_len,
				     0);
    *minor_status = status;

    if ((status != SPX_S_INIT_SUCCESS) && (status != SPX_S_INIT_CONTINUE)) {
      *context_handle = 0;
      return(GSS_S_FAILURE);
    }

    output_token->length = token_len;
    output_token->value = output_token_buf;
    gssctxhandle->context_handle = ctxhandle;

    if (status == SPX_S_INIT_CONTINUE)  return(GSS_S_CONTINUE_NEEDED);
    return(GSS_S_COMPLETE);
  }
}

/*
 * gss_accept_sec_context() - Allows a remotely initiated security context
 *   between the application and a remote peer to be established.  The
 *   routine may return a output_token which should be transferred to the
 *   peer application, where the peer will present it to gss_init_sec_context.
 *   If one or more reply tokens are required from the peer application, this
 *   routine will return a status value of GSS_S_CONTINUE_NEEDED, in which case
 *   it should be called again when the reply token is received from the peer
 *   application, passing the token to gss_accept_sec_context via the input
 *   token parameter.
 *
 *   OM_uint32   *minor_status  (output) - mechanism specific status code
 *   gss_ctx_id_t *context_handle (output) - context handle for new context
 *   gss_cred_id_t verifier_cred_handle (input) - handle for credentials
 *                                         claimed by context acceptor.
 *                                         Supply GSS_C_NO_CREDENTIAL
 *                                         to use default credentials.
 *   gss_buffer_t input_token   (input)  - token received from peer, supply
 *                                         GSS_C_NO_BUFFER for initial call
 *   gss_channel_bindings input_chan_bindings (input) - application specified
 *                                         bindings.  Allows application to
 *                                         securely bind channel information
 *                                         to security context
 *   gss_name_t   *src_name     (output)  - authenticated name of context
 *                                         initiator.
 *   gss_OID      *mech_type    (output) - security mechanism used
 *   gss_buffer_t output_token  (output) - token to be sent to peer
 *   int          *ret_flags    (output) - returned flags which indicate that
 *                                         the context supports these services
 *   OM_uint32    *time_rec     (output) - number of seconds for which the
 *                                         context will remain valid
 *   gss_cred_id_t *delegated_cred_handle (output) - handle for credentials
 *                                         received from context initiator.
 *                                         Only valid if 'deleg_flag' is true.
 *
 */

int gss_accept_sec_context(minor_status, context_handle,
	 verifier_cred_handle, input_token, input_chan_bindings,
	 src_name, mech_type, output_token, ret_flags, time_rec,
	 delegated_cred_handle)
OM_uint32        *minor_status;
gss_ctx_id_t     *context_handle;
gss_cred_id_t    verifier_cred_handle;
gss_buffer_t     input_token;
gss_channel_bindings input_chan_bindings;
gss_name_t       *src_name;
gss_OID          *mech_type;
gss_buffer_t     output_token;
int              *ret_flags;
OM_uint32        *time_rec;
gss_cred_id_t    *delegated_cred_handle;
{
  int status, len, token_len, wrapper_len=0, header_len=0, ret_flags_actual=0;
  char  recv_wrapper[16], *internal_name;
  ClaimantCred   *spx_deleg_cred_handle;
  SPXCredentials *gsscred, *deleg_cred;
  SPXContextHandle *gssctxhandle;
  ContextHandle  *ctxhandle = NULL;
  gss_OID        output_mech;
  int            i, chanbinding_len, wrapper_adj, output_token_len, resp_len;
  time_t         time_now;
  char           chanbinding[GSS_C_MAX_CHAN_BINDINGS], output_token_buf[GSS_C_MAX_TOKEN];
  static KTEXT_ST pkt_st;
  KTEXT pkt = & pkt_st;       /* request packet  to CDC */
  static KTEXT_ST rpkt_st;
  KTEXT rpkt = &rpkt_st;      /* response packet from CDC */
  char    *cdc_domain_name;
  struct type_SPHINX_LongPosixTime         *challenge;
  unsigned char   *asn1_buf;
  int             asn1len;
  DESblock   deskey;
  DESblock   mac;

  if ((unsigned char) input_token->value[header_len] != 0xa0)
    if ((unsigned char) input_token->value[header_len] != 0x30)
      if ((unsigned char) input_token->value[header_len] != 0xa5)
	if ((unsigned char) input_token->value[header_len] != 0xa7) {
	  syslog(LOG_INFO, "gss_accept_context: illegal first char ... %2x", input_token->value[header_len]);
	  return(GSS_S_BAD_MECH);
	}

  if ((unsigned char) input_token->value[header_len] == 0xa0) {
    /*
     *  we need to unwrap the GSSAPI wrapper
     */
    header_len++;
    switch ((unsigned char) input_token->value[header_len]) {
      case 0x82 :
	header_len = header_len+3;
	if (input_token->value[header_len++] != 0x06)
	  return(GSS_S_BAD_MECH);
	recv_wrapper[wrapper_len++] = 0x06;
	len = recv_wrapper[wrapper_len++] = input_token->value[header_len++];
	bcopy(&input_token->value[header_len], &recv_wrapper[wrapper_len],len);
	wrapper_len = len + wrapper_len;
	header_len = len + header_len;
	token_len = input_token->length - header_len;
	break;
      case 0x81 :
	header_len = header_len + 2;
	if (input_token->value[header_len++] != 0x06)
	  return(GSS_S_BAD_MECH);
	recv_wrapper[wrapper_len++] = 0x06;
	len = recv_wrapper[wrapper_len++] = input_token->value[header_len++];
	bcopy(&input_token->value[header_len], &recv_wrapper[wrapper_len],len);
	wrapper_len = len + wrapper_len;
	header_len = len + header_len;
	token_len = input_token->length - header_len;
	break;
      default :  /* for indefinite length and token smaller than 128 bytes */
	header_len++;
	if (input_token->value[header_len++] != 0x06)
	  return(GSS_S_BAD_MECH);
	recv_wrapper[wrapper_len++] = 0x06;
	len = recv_wrapper[wrapper_len++] = input_token->value[header_len++];
	bcopy(&input_token->value[header_len], &recv_wrapper[wrapper_len],len);
	wrapper_len = len + wrapper_len;
	header_len = len + header_len;
	token_len = input_token->length - header_len - 2;
	break;
      }

    /*
     * check if GSS API wrapper is SPX_MECHTYPE_OID
     *      since this is the only supported type for now
     *
     */

    if (bcmp(recv_wrapper, SPX_MECHTYPE_VALUE, wrapper_len) != 0) {
      syslog(LOG_INFO, "gss_accept_context: recv_wrapper doesn't match");
      return(GSS_S_BAD_MECH);
    }
  } else {
    /*
     *  no GSSAPI wrapper to unwrap
     */
    header_len = 0;
    token_len = input_token->length;
  }

  /*
   *  if mech type is good, set actual mech type used
   */
  output_mech = &SPX_MECHTYPE_OID;
  *mech_type = output_mech;

  if ((unsigned char) input_token->value[header_len] == 0xa5) {
    /*
     *  This input token is actually a forwarded CDC request
     */
    header_len++;
    switch ((unsigned char) input_token->value[header_len]) {
      case 0x82 :
	header_len = header_len+3;
	break;
      case 0x81 :
	header_len = header_len + 2;
	break;
      default :  /* for indefinite length and token smaller than 128 bytes */
	header_len++;
	break;
      }

    pkt->length = input_token->length - header_len;
    bcopy(&input_token->value[header_len], pkt->dat, pkt->length);

    cdc_domain_name = (char *) get_domain_name(NULL);

    if ((status = send_to_cdc(pkt, rpkt, cdc_domain_name, NULL)) != ASUCCESS) {
      /*
       *  CDC is unavailable here also.
       */
      output_token_buf[0] = 0xa6;
      output_token_buf[1] = 0x1;
      output_token_buf[2] = 0x1;
      output_token->length = 3;
      output_token->value = output_token_buf;
      *delegated_cred_handle = 0;
      *minor_status = SPX_S_FWD_CLAIM_UNABLE_TO_ACCESS_CDC;
      return(GSS_S_FAILURE);
    }
    output_token_buf[0] = 0xa6;
    if (rpkt->length > 256) {
      output_token_buf[1] = 0x82;
      output_token_buf[2] = (rpkt->length) / 256;
      output_token_buf[3] = (rpkt->length) % 256;
      wrapper_adj = 3;
    } else {
      if (rpkt->length > 128) {
	output_token_buf[1] = 0x81;
	output_token_buf[2] = (rpkt->length);
	wrapper_adj = 2;
      } else {
	output_token_buf[1] = (rpkt->length);
	wrapper_adj = 1;
      }
    }

    bcopy(rpkt->dat, &output_token_buf[wrapper_adj+1], rpkt->length);
    output_token->length = rpkt->length + wrapper_adj + 1;
    output_token->value = output_token_buf;
    *delegated_cred_handle = 0;
    return(GSS_S_CONTINUE_NEEDED);
  }

  if ((unsigned char) input_token->value[header_len] == 0xa7) {
    /*
     *  This input token is response to previous challenge
     */
    header_len++;
    resp_len = input_token->value[header_len];
    header_len++;

    output_token->length = 0;
    output_token->value = NULL;

    if (!bcmp(&input_token->value[header_len], expected_response, resp_len))
      return(GSS_S_COMPLETE);
    else {
      *minor_status = SPX_S_FAILURE_WITH_CHALLENGE;
      return(GSS_S_FAILURE);
    }
  }

  if (verifier_cred_handle == GSS_C_NO_CREDENTIAL) return(GSS_S_FAILURE);
  gsscred = (SPXCredentials *) verifier_cred_handle;

  bcopy(input_chan_bindings->initiator_address.value, &chanbinding[0],
	input_chan_bindings->initiator_address.length);
  chanbinding_len = input_chan_bindings->initiator_address.length;
  bcopy(input_chan_bindings->acceptor_address.value,
	&chanbinding[chanbinding_len],
	input_chan_bindings->acceptor_address.length);
  chanbinding_len = chanbinding_len +
    input_chan_bindings->acceptor_address.length;

  if (input_chan_bindings->application_data.length != 0) {
    bcopy(input_chan_bindings->application_data.value,
	  &chanbinding[chanbinding_len],
	  input_chan_bindings->application_data.length);
    chanbinding_len = chanbinding_len +
      input_chan_bindings->application_data.length;
  }

  status = Sphinx_Accept_sec_context(&ctxhandle,
				     gsscred->ver_handle,
				     gsscred->ta_handle,
				     &input_token->value[header_len],
				     token_len,
				     chanbinding,
				     chanbinding_len,
				     &spx_deleg_cred_handle,
				     &ret_flags_actual,
				     0);

  *minor_status = status;
  *ret_flags = ret_flags_actual;
  gssctxhandle = (SPXContextHandle *) malloc(sizeof(SPXContextHandle));
  *context_handle = (gss_ctx_id_t) gssctxhandle;
  strcpy(gssctxhandle->mechtype, SPX_MECHTYPE_VALUE);

  if ((status != SPX_S_ACCEPT_SUCCESS) & (status != SPX_S_ACCEPT_WITH_CHALLENGE)) {
    bcopy(ctxhandle->mutual_resp, output_token_buf, ctxhandle->mutual_len);

    output_token->length = ctxhandle->mutual_len;
    output_token->value = output_token_buf;
    *delegated_cred_handle = 0;
    return(GSS_S_FAILURE);
  }

  gssctxhandle->context_handle = ctxhandle;
  bcopy(ctxhandle->mutual_resp, output_token_buf, ctxhandle->mutual_len);
  output_token_len = ctxhandle->mutual_len;

  if (status == SPX_S_ACCEPT_WITH_CHALLENGE) {
    output_token_buf[0] = 0x81;
    time_now = time(0);
    challenge = (struct type_SPHINX_LongPosixTime *) malloc(sizeof(struct type_SPHINX_LongPosixTime));
    challenge->seconds = time_now;
    challenge->nanoseconds = 0;
    ber_encode(challenge, encode_SPHINX_LongPosixTime,PS_LEN_LONG,&asn1_buf,
	       &asn1len);

    bcopy(asn1_buf, &output_token_buf[ctxhandle->mutual_len], asn1len);
    output_token_len += asn1len;

    {
      bcopy((char *) ctxhandle->deskey.bytes, deskey.bytes, sizeof(deskey.bytes));
      DES_X9_MAC(&deskey,
		 asn1_buf,
		 asn1len,
		 &mac);
      bcopy(&mac, expected_response, 8);
    }
  }

  output_token->length = output_token_len;
  output_token->value = output_token_buf;

  internal_name = (char *) malloc(GSS_C_MAX_PRINTABLE_NAME);
  *src_name = (gss_name_t) internal_name;
  strcpy(internal_name, ctxhandle->issuer_fn);
  deleg_cred = (SPXCredentials *) malloc(sizeof(SPXCredentials));
  *delegated_cred_handle = (gss_cred_id_t) deleg_cred;

  deleg_cred->clm_handle = spx_deleg_cred_handle;
  deleg_cred->ver_handle = NULL;
  deleg_cred->ta_handle = gsscred->ta_handle;
  strcpy(deleg_cred->credential_file, ctxhandle->claimant_file);

  if (status == SPX_S_ACCEPT_WITH_CHALLENGE)  return(GSS_S_CONTINUE_NEEDED);

  return(GSS_S_COMPLETE);
}

/*
 * gss_display_status() - Allows applications to obtain a textual
 *   representation of a GSSAPI status code, for display to the
 *   user or for logging purposes.
 *
 *   OM_uint32    *minor_status  (output) - mechanism specific status code
 *   int          status_value   (input)  - status value to be converted
 *   int          status_type    (input)  - specifies either GSS status code
 *                                          or mechanism status code
 *   gss_OID      mech_type      (input)  - underlying mech type used to
 *                                          interpret a minor status value
 *   int          *message_context (input/output) - indicates more messages
 *   gss_buffer_t status_string  (output) - textual interpretation of the
 *                                          status value
 *
 */

OM_uint32 gss_display_status(minor_status, status_value, status_type,
			     mech_type, message_context, status_string)
OM_uint32        *minor_status;
int              status_value;
int              status_type;
gss_OID          mech_type;
int              *message_context;
gss_buffer_t     status_string;
{
  int  msg_ctx;
  char error_buffer[256];

  msg_ctx = *message_context;
  if (msg_ctx > 0) {
    *message_context = 0;
    return(GSS_S_COMPLETE);
  }
  bzero(error_buffer, sizeof(error_buffer));
  if (status_type == GSS_C_GSS_CODE) {
    switch (status_value) {
      case GSS_S_CALL_INACCESSIBLE_READ :
	strcpy(error_buffer, "GSS : a required input parameter could not be read");
	break;
      case GSS_S_CALL_INACCESSIBLE_WRITE :
	strcpy(error_buffer, "GSS : a required output parameter could not be written");
	break;
      case GSS_S_CALL_BAD_STRUCTURE :
	strcpy(error_buffer, "GSS : a parameter was malformed");
	break;
      case GSS_S_BAD_MECH :
	strcpy(error_buffer, "GSS : an unsupported mechanism was requested");
	break;
      case GSS_S_BAD_NAME :
	strcpy(error_buffer, "GSS : an invalid name was supplied");
	break;
      case GSS_S_BAD_NAMETYPE :
	strcpy(error_buffer, "GSS : a supplied name was of an unsupported type");
	break;
      case GSS_S_BAD_BINDINGS :
	strcpy(error_buffer, "GSS : incorrect channel bindings were supplied");
	break;
      case GSS_S_BAD_STATUS :
	strcpy(error_buffer, "GSS : an invalid status code was supplied");
	break;
      case GSS_S_BAD_SIG :
	strcpy(error_buffer, "GSS : a token had an invalid signature");
	break;
      case GSS_S_NO_CRED :
	strcpy(error_buffer, "GSS : no credentials were supplied");
	break;
      case GSS_S_NO_CONTEXT :
	strcpy(error_buffer, "GSS : no context has been established");
	break;
      case GSS_S_DEFECTIVE_TOKEN :
	strcpy(error_buffer, "GSS : a token was invalid");
	break;
      case GSS_S_DEFECTIVE_CREDENTIAL :
	strcpy(error_buffer, "GSS : a credential was invalid");
	break;
      case GSS_S_CREDENTIALS_EXPIRED :
	strcpy(error_buffer, "GSS : supplied credentials have expired");
	break;
      case GSS_S_CONTEXT_EXPIRED :
	strcpy(error_buffer, "GSS : the context has expired");
	break;
      case GSS_S_FAILURE :
	strcpy(error_buffer, "GSS : mechanism-specific failure");
	break;
      case GSS_S_CONTINUE_NEEDED :
	strcpy(error_buffer, "GSS : the routine should be called again");
	break;
      case GSS_S_DUPLICATE_TOKEN :
	strcpy(error_buffer, "GSS : the token was a duplicate of an earlier token");
	break;
      case GSS_S_OLD_TOKEN :
	strcpy(error_buffer, "GSS : the token's validity period has expired");
	break;
      case GSS_S_UNSEQ_TOKEN :
	strcpy(error_buffer, "GSS : a later token has already been processed");
	break;
      default :
	return(GSS_S_BAD_STATUS);
	break;
      }
    status_string->length = strlen(error_buffer) + 1;
    status_string->value = error_buffer;
    return(GSS_S_COMPLETE);
  } else if (status_type == GSS_C_MECH_CODE) {
    if (mech_type != GSS_C_NULL_OID) {
      if (bcmp(mech_type->elements, SPX_MECHTYPE_VALUE, mech_type->length) != 0)
	return(GSS_S_BAD_MECH);
    }
    Sphinx_error_message(status_value, error_buffer);
    status_string->length = strlen(error_buffer) + 1;
    status_string->value = error_buffer;
    return(GSS_S_COMPLETE);
  }
  return(GSS_S_BAD_STATUS);
}

/*
 * gss_indicate_mechs() - Allow an application to determine which underlying
 *   security mechanisms are available.
 *
 *   OM_uint32    *minor_status  (output) - mechanism specific status code
 *   gss_OID_set  *mech_set      (output) - set of implementation-supported
 *                                          mechanisms.
 *
 */

OM_uint32 gss_indicate_mechs(minor_status, mech_set)
OM_uint32 *minor_status;
gss_OID_set *mech_set;
{
  gss_OID_set  actual_mech_set;

  actual_mech_set = (gss_OID_set) malloc(sizeof(gss_OID_set_d));
  *mech_set = actual_mech_set;
  actual_mech_set->count = 1;
  actual_mech_set->elements = &SPX_MECHTYPE_OID;
  return(GSS_S_COMPLETE);
}


/*
 * gss_display_name() - Convert an internal API name to a printable name.
 *
 *   OM_uint32   *minor_status  (output) - mechanism specific status code
 *   gss_name_t   input_name  (input)    - principal name in API
 *                                         internal format.
 *   gss_buffer_t output_name_buffer (output)  - printable name for principal
 *   gss_OID      *output_name_type   (output) - type of printable name.
 *                                         currently only GSS_C_NULL_OID is
 *                                         defined.
 *
 */

OM_uint32 gss_display_name(minor_status, input_name, output_name_buffer,
			   output_name_type)
OM_uint32        *minor_status;
gss_name_t       input_name;
gss_buffer_t     output_name_buffer;
gss_OID          *output_name_type;
{
  int  status;

  output_name_buffer->value = (char *) malloc(GSS_C_MAX_PRINTABLE_NAME);
  strcpy(output_name_buffer->value, "SPX:");
  strcat(output_name_buffer->value, (char *) input_name);
  output_name_buffer->length = strlen(output_name_buffer->value) + 1;
  *output_name_type = GSS_C_NULL_OID;
  return(GSS_S_COMPLETE);
}

/*
 * gss_import_name() - Convert a printable name to an internal API name.
 *
 *   OM_uint32   *minor_status  (output) - mechanism specific status code
 *   gss_buffer_t input_name_buffer (input)  - printable name for principal
 *   gss_OID      input_name_type   (input)  - type of printable name.
 *                                         currently only GSS_C_NULL_OID is
 *                                         defined.
 *   gss_name_t   *output_name  (output) - returned principal name in API
 *                                         internal format.
 *
 */

OM_uint32 gss_import_name(minor_status, input_name_buffer, input_name_type,
			  output_name)
OM_uint32        *minor_status;
gss_buffer_t     input_name_buffer;
gss_OID          input_name_type;
gss_name_t       *output_name;
{
  int  status;
  char *cp, *service, *host, *internal_name;

  if (input_name_type != GSS_C_NULL_OID)  return(GSS_S_BAD_NAMETYPE);

  if (!strncasecmp(input_name_buffer->value, "SERVICE:", 8)) {
    cp = index(input_name_buffer->value, ':');
    service = cp+1;
    if (!(cp = rindex(input_name_buffer->value, '@')))
      return(GSS_S_BAD_NAME);
    *cp = '\0';
    host = cp+1;
    internal_name = (char *) malloc(GSS_C_MAX_PRINTABLE_NAME);
    *output_name = (gss_name_t) internal_name;

    if (!Sphinx_build_name(1, internal_name, NULL, NULL, service, host))
      return(GSS_S_BAD_NAME);
    return(GSS_S_COMPLETE);
  }

  if (!strncasecmp(input_name_buffer->value, "SPX:", 4)) {
    internal_name = (char *) malloc(GSS_C_MAX_PRINTABLE_NAME);
    *output_name = (gss_name_t) internal_name;
    strcpy(internal_name, &input_name_buffer->value[4]);
    return(GSS_S_COMPLETE);
  }
  return(GSS_S_BAD_NAME);
}

/*
 * gss_release_name() - Deallocate an API internal name.
 *
 *   OM_uint32   *minor_status  (output) - mechanism specific status code
 *   gss_name_t           name  (input)  - name in API internal format
 *
 */

OM_uint32 gss_release_name(minor_status, name)
OM_uint32       *minor_status;
gss_name_t      name;
{
  free(name);
  return(GSS_S_COMPLETE);
}

/*
 * gss_release_buffer() - Deallocate a buffer descriptor
 *
 *   OM_uint32   *minor_status  (output) - mechanism specific status code
 *   gss_buffer_t       buffer  (input)  - buffer
 *
 */

OM_uint32 gss_release_buffer(minor_status, buffer)
OM_uint32         *minor_status;
gss_buffer_t      buffer;
{
  free(buffer->value);
  buffer->length = 0;
  return(GSS_S_COMPLETE);
}

/*
 * gss_release_oid_set() - Deallocate a set of object identifiers
 *
 *   OM_uint32   *minor_status  (output) - mechanism specific status code
 *   gss_OID_set          *set  (input)  - set of object identifiers
 *
 */

OM_uint32 gss_release_oid_set(minor_status, set)
OM_uint32        *minor_status;
gss_OID_set      *set;
{
  /*  fudge for now  */
  free(set);
  return(GSS_S_COMPLETE);
}

/*
 * gss_delete_sec_context() - Delete a security context
 *
 *   OM_uint32    *minor_status  (output) - mechanism specific status code
 *   gss_ctx_id_t *context_handle (input)  - handle for context_handle
 *   gss_buffer_t output_token   (output)
 *
 */

OM_uint32 gss_delete_sec_context(minor_status, context_handle, output_token)
OM_uint32        *minor_status;
gss_ctx_id_t     *context_handle;
gss_buffer_t     output_token;
{
  gss_ctx_id_t  actual_ctx_handle;
  ContextHandle *ctxhandle;
  SPXContextHandle *gssctxhandle;

  /*
   *  This mechanism doesn't return a token when contexts are deleted
   */
  output_token->length = 0;
  output_token->value = NULL;

  if (!context_handle) return(GSS_S_COMPLETE);
  actual_ctx_handle = *context_handle;
  if (!actual_ctx_handle) return(GSS_S_COMPLETE);

  gssctxhandle = (SPXContextHandle *) actual_ctx_handle;
  if (gssctxhandle->context_handle) free(gssctxhandle->context_handle);
  free(gssctxhandle);
  return(GSS_S_COMPLETE);
}


/*
 *
 * gss__stash_default_cred() - Allows remote peer to bind delegated credential
 *   handle with remote application.  Called by applications to set the
 *   delegated credentials as the default credentials for a process.
 *
 *   OM_uint32    *minor_status  (output) - mechanism specific status code
 *   gss_cred_id_t delegated_cred_handle (input) - handle for credentials
 *                                         received from context initiator.
 *
 */

OM_uint32 gss__stash_default_cred(minor_status, delegated_cred_handle)
OM_uint32     *minor_status;
gss_cred_id_t delegated_cred_handle;
{
  int             status;
  SPXCredentials  *gsscred;

  if (delegated_cred_handle == GSS_C_NO_CREDENTIAL)
    return(GSS_S_COMPLETE);
  gsscred = (SPXCredentials *) delegated_cred_handle;
  status = Sphinx_stash_cred(gsscred);
  if (status < 0) {
    *minor_status = status;
    return(GSS_S_FAILURE);
  }
  gss_default_credentials = (void *) gsscred;
  return(GSS_S_COMPLETE);
}

/*
 *
 * gss__check_acl() - Check if principal is listed in the specified ACL file.
 *
 *   OM_uint32    *minor_status  (output) - mechanism specific status code
 *   gss_buffer_t fullname_buffer (input) - principal's printable name
 *   gss_buffer_t acl_file_buffer (input) - acl file name
 *
 */

OM_uint32 gss__check_acl(minor_status,
			 fullname_buffer,
			 acl_file_buffer)
OM_uint32     *minor_status;
gss_buffer_t  fullname_buffer;
gss_buffer_t  acl_file_buffer;
{
  if (sphinxuserok(fullname_buffer->value, acl_file_buffer->value) != 1) {
    return(GSS_S_FAILURE);
  }
  return(GSS_S_COMPLETE);
}

int gss__extract_key_from_context(context_handle, key)
gss_ctx_id_t context_handle;
char         *key;
{
  ContextHandle *ctxhandle;
  SPXContextHandle *gssctxhandle;
  unsigned char deskey[8];
  int i;

  if (!context_handle)
    return(GSS_S_FAILURE);

  gssctxhandle = (SPXContextHandle *) context_handle;
  if (bcmp(gssctxhandle->mechtype, SPX_MECHTYPE_VALUE, SPX_MECHTYPE_SZ) != 0)
    return(GSS_S_BAD_MECH);
  ctxhandle = (ContextHandle *) gssctxhandle->context_handle;
  bcopy(ctxhandle->deskey.bytes, deskey, sizeof(DESblock));
  for (i=0;i<sizeof(DESblock);i++)
    deskey[i] = odd_parity[deskey[i]];
  bcopy(deskey, key, sizeof(DESblock));
  return(GSS_S_COMPLETE);
}

int gss__globalname_to_local(filename, globalname, localname)
char *filename;
gss_name_t globalname;
char *localname;
{
  FILE *hostf;
  char *princ;
  char fullname[256], *p, *fn_p, **cp;
  int  fn_len, fnmatch, quoted_name=0;

  if (!(hostf = fopen(filename, "r"))) {
      syslog(LOG_INFO, "gss__globalname_to_localname: unable to open '%s'", filename);
      return(GSS_S_FAILURE);
  }

  while (fgets(fullname, sizeof(fullname), hostf)) {
    /*
     * Skip beginning white space
     */

    p = fullname;
    while (*p == ' ' || *p == '\t')
      p++;
    fn_p = p;
    if (*p == '"') quoted_name = 1;

    if (!quoted_name) {
      while (*p != '\n' && *p != ' ' && *p != '\t' && *p != '\0')
        p++;
    } else {
      p++;
      fn_p = p;
      while (*p != '\n' && *p != '"' && *p != '\0')
        p++;
    }

    if (*p == ' ' || *p == '\t' || ((*p == '"') && (quoted_name))) {
      *p++ = '\0';
      /*
       * Skip middle white space
       */
      while (*p == ' ' || *p == '\t')
        p++;
      princ = p;

      while (*p != '\n' && *p != ' ' && *p != '\t' && *p != '\0')
        p++;
      /*
       * Stop before trailing white space
       */
    } else
      princ = p;
    *p = '\0';

    fnmatch = !strcasecmp(globalname, fn_p);
    if (fnmatch) {
      strcpy(localname, princ);
      return(GSS_S_COMPLETE);
    }
  }
  return(GSS_S_FAILURE);
}

/*
 * gss__release_context_handle() - Informs GSS API that the specified
 * context handle is no longer required by the process.
 *
 *   OM_uint32   *minor_status  (output) - mechanism specific status code
 *   gss_ctx_id_t *context_handle (input)  - handle for context_handle
 *
 */

OM_uint32 gss__release_context_handle(minor_status, context_handle)
OM_uint32        *minor_status;
gss_ctx_id_t    *context_handle;
{
  gss_ctx_id_t  actual_ctx_handle;
  ContextHandle *ctxhandle;
  SPXContextHandle *gssctxhandle;

  if (!context_handle) return(GSS_S_COMPLETE);
  actual_ctx_handle = *context_handle;
  if (!actual_ctx_handle) return(GSS_S_COMPLETE);

  gssctxhandle = (SPXContextHandle *) actual_ctx_handle;
  if (gssctxhandle->context_handle) free(gssctxhandle->context_handle);
  free(gssctxhandle);
  return(GSS_S_COMPLETE);
}

int gss__display_context_handle(context_handle)
gss_ctx_id_t context_handle;
{
  ContextHandle *ctxhandle;
  SPXContextHandle *gssctxhandle;

  if (!context_handle)
    return(GSS_S_FAILURE);

  gssctxhandle = (SPXContextHandle *) context_handle;
  if (bcmp(gssctxhandle->mechtype, SPX_MECHTYPE_VALUE, SPX_MECHTYPE_SZ) != 0)
    return(GSS_S_BAD_MECH);
  ctxhandle = (ContextHandle *) gssctxhandle->context_handle;

  printf("context handle is :\n");
  printf("  deskey is ");
  gss_hexdump(ctxhandle->deskey.bytes, sizeof(DESblock));
  printf("  issuer_ln  is  %s\n", ctxhandle->issuer_ln);
  printf("  issuer_fn  is  '%s'\n", ctxhandle->issuer_fn);
  printf("  claimant_file is '%s'\n", ctxhandle->claimant_file);
  printf("  deleg_flag is  %d\n", ctxhandle->deleg_flag);
  printf("  mutual len is  %d\n", ctxhandle->mutual_len);
  printf("  mutual response ");
  gss_hexdump(ctxhandle->mutual_resp, ctxhandle->mutual_len);
  return(GSS_S_COMPLETE);
}


gss_hexdump(p, l)
char *p;
int l;
{
  int i;

  for (i=0; i<l; i++)  {
    if ((i>0) && (i%16 == 0))  printf("\n");
    printf(" %02x", (unsigned char ) p[i]);
  }
  printf("\n");
}
