#!/usr/bin/perl -T

use File::Find qw(find);
use lib '.'; use lib 't';
use SATest; sa_t_init("bayessql");

use Test::More;
use Mail::SpamAssassin;

use constant HAS_DBI => eval { require DBI; }; # for our cleanup stuff
use constant SQLITE => eval { require DBD::SQLite; DBD::SQLite->VERSION(1.59_01); };
use constant SQL => conf_bool('run_bayes_sql_tests');

plan skip_all => "DBI is unavailable on this system" unless (HAS_DBI);
plan skip_all => "Bayes SQL tests are disabled or DBD::SQLite not found" unless (SQLITE || SQL);

my $tests = 0;
$tests += 59 if (SQLITE);
$tests += 59 if (SQL);
plan tests => $tests;

diag "Note: If there is a failure it may be due to an incorrect SQL configuration." if (SQL);

my ($dbconfig, $dbdsn, $dbusername, $dbpassword, $no_error_reported);

$no_error_reported = 1;

if (SQLITE) {
  # Try /dev/shm as it's likely memdisk, otherwise SQLite is sloow..
  my $dbdir = tempdir("bayessql.XXXXXX", DIR => -w "/dev/shm" ? "/dev/shm" : "log");
  die "FATAL: failed to create dbdir: $!" unless -d $dbdir;
  $dbdsn = "dbi:SQLite:dbname=$dbdir/bayes.db";
  $dbusername = "";
  $dbpassword = "";
  my $dbh = DBI->connect($dbdsn,$dbusername,$dbpassword);
  $dbh->do("PRAGMA synchronous = OFF");
  $dbh->do("PRAGMA cache_size = 10000");
  $dbh->do("
  CREATE TABLE bayes_expire (
    id int(11) NOT NULL default '0',
    runtime int(11) NOT NULL default '0',
    PRIMARY KEY (id)
  );
  ") or die "Failed to create $dbfile";
  $dbh->do("
  CREATE TABLE bayes_global_vars (
    variable varchar(30) NOT NULL default '',
    value varchar(200) NOT NULL default '',
    PRIMARY KEY (variable)
  );
  ") or die "Failed to create $dbfile";
  $dbh->do("
  INSERT INTO bayes_global_vars VALUES ('VERSION','3');
  ") or die "Failed to create $dbfile";
  $dbh->do("
  CREATE TABLE bayes_seen (
    id int(11) NOT NULL default '0',
    msgid varchar(200) NOT NULL default '' COLLATE binary,
    flag char(1) NOT NULL default '',
    PRIMARY KEY (id,msgid)
  );
  ") or die "Failed to create $dbfile";
  $dbh->do("
  CREATE TABLE bayes_token (
    id int(11) NOT NULL default '0',
    token char(5) NOT NULL default '' COLLATE binary,
    spam_count int(11) NOT NULL default '0',
    ham_count int(11) NOT NULL default '0',
    atime int(11) NOT NULL default '0',
    PRIMARY KEY (id, token)
  );
  ") or die "Failed to create $dbfile";
  $dbh->do("
  CREATE INDEX idx_id_atime ON bayes_token (id, atime);
  ") or die "Failed to create $dbfile";
  $dbh->do("
  CREATE TABLE bayes_vars (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    username varchar(200) NOT NULL default '',
    spam_count int(11) NOT NULL default '0',
    ham_count int(11) NOT NULL default '0',
    token_count int(11) NOT NULL default '0',
    last_expire int(11) NOT NULL default '0',
    last_atime_delta int(11) NOT NULL default '0',
    last_expire_reduce int(11) NOT NULL default '0',
    oldest_token_age int(11) NOT NULL default '2147483647',
    newest_token_age int(11) NOT NULL default '0'
  );
  ") or die "Failed to create $dbfile";
  $dbh->do("
  CREATE UNIQUE INDEX idx_username ON bayes_vars (username);
  ") or die "Failed to create $dbfile";

  $dbh->disconnect;
  undef $dbh;

  $dbconfig = "
    bayes_store_module Mail::SpamAssassin::BayesStore::SQL
    bayes_sql_dsn $dbdsn
  ";

  run_bayes();
  rmtree($dbdir);
}

if (SQL) {
  $dbdsn = conf('bayes_sql_dsn');
  $dbusername = conf('bayes_sql_username');
  $dbpassword = conf('bayes_sql_password');

  $dbconfig = '';
  foreach my $setting (qw(
    bayes_store_module
    bayes_sql_dsn
    bayes_sql_username
    bayes_sql_password
    ))
  {
    my $val = conf($setting);
    $dbconfig .= "$setting $val\n" if $val;
  }

  run_bayes();
}


#---------------------------------------------------------------------------
sub run_bayes {

my $testuser = 'tstusr.'.$$.'.'.time();

tstprefs ("
  $dbconfig
  bayes_sql_override_username $testuser
  loadplugin validuserplugin ../../../data/validuserplugin.pm
  bayes_sql_username_authorized 1
");

my $sa = create_saobj();

$sa->init();

_debug_dsn(1);
ok($sa);

my $learner = $sa->call_plugins("learner_get_implementation");

_debug_dsn(2);
ok($sa->{bayes_scanner} && $learner);

_debug_dsn(3);
ok($learner->{store}->tie_db_writable());

# This bit breaks abstraction a bit, the userid is an implementation detail,
# but is necessary to perform some of the tests.  Perhaps in the future we
# can add some sort of official API for this sort of thing.
my $testuserid = $learner->{store}->{_userid};
_debug_dsn(4);
ok(defined($testuserid));

_debug_dsn(5);
ok($learner->{store}->clear_database());

_debug_dsn(6);
ok(database_clear_p($testuser, $testuserid));

$sa->finish_learner();

undef $sa;

sa_t_init("bayessql");

tstprefs ("
  $dbconfig
  bayes_sql_override_username iwillfail
  loadplugin validuserplugin ../../../data/validuserplugin.pm
  bayes_sql_username_authorized 1
");

$sa = create_saobj();

$sa->init();

_debug_dsn(7);
ok($sa);

$learner = $sa->call_plugins("learner_get_implementation");

_debug_dsn(8);
ok($sa->{bayes_scanner});

_debug_dsn(9);
ok(!$learner->{store}->tie_db_writable());

$sa->finish_learner();

undef $sa;

sa_t_init("bayessql");

tstprefs ("
  $dbconfig
  bayes_sql_override_username $testuser
");

$sa = create_saobj();

$sa->init();

_debug_dsn(10);
ok($sa);

$learner = $sa->call_plugins("learner_get_implementation");

_debug_dsn(11);
ok($sa->{bayes_scanner});

_debug_dsn(12);
ok(!$sa->{bayes_scanner}->is_scan_available());

open(MAIL,"< data/spam/001");

my $raw_message = do {
  local $/;
  <MAIL>;
};

close(MAIL);
_debug_dsn(13);
ok($raw_message);

my @msg;
foreach my $line (split(/^/m,$raw_message)) {
  $line =~ s/\r$//;
  push(@msg, $line);
}

my $mail = $sa->parse( \@msg );

_debug_dsn(14);
ok($mail);

my $body = $learner->get_body_from_msg($mail);

_debug_dsn(15);
ok($body);

my $toks = $learner->tokenize($mail, $body);

_debug_dsn(16);
ok(scalar(keys %{$toks}) > 0);

my $msgid = $mail->generate_msgid();
my $msgid_hdr = $mail->get_msgid();

# $msgid is the generated hash messageid
# $msgid_hdr is the Message-Id header
_debug_dsn(17);
ok($msgid eq '71f849915d7e469ddc1890cd8175f6876843f99e@sa_generated');
_debug_dsn(18);
ok($msgid_hdr eq '9PS291LhupY');

_debug_dsn(19);
ok($learner->{store}->tie_db_writable());

_debug_dsn(20);
ok(!$learner->{store}->seen_get($msgid));

$learner->{store}->untie_db();

_debug_dsn(21);
ok($sa->{bayes_scanner}->learn(1, $mail));

_debug_dsn(22);
ok(!$sa->{bayes_scanner}->learn(1, $mail));

_debug_dsn(23);
ok($learner->{store}->tie_db_writable());

_debug_dsn(24);
ok($learner->{store}->seen_get($msgid) eq 's');

$learner->{store}->untie_db();

_debug_dsn(25);
ok($learner->{store}->tie_db_writable());

my $tokerror = 0;
foreach my $tok (keys %{$toks}) {
  _debug_dsn(26);
  my ($spam, $ham, $atime) = $learner->{store}->tok_get($tok);
  if ($spam == 0 || $ham > 0) {
    $tokerror = 1;
  }
}
_debug_dsn(27);
ok(!$tokerror);

_debug_dsn(28);
my $tokens = $learner->{store}->tok_get_all(keys %{$toks});

_debug_dsn(29);
ok($tokens);

$tokerror = 0;
foreach my $tok (@{$tokens}) {
  _debug_dsn(30);
  my ($token, $tok_spam, $tok_ham, $atime) = @{$tok};
  _debug_dsn(31);
  if ($tok_spam == 0 || $tok_ham > 0) {
    $tokerror = 1;
  }
}

_debug_dsn(32);
ok(!$tokerror);

$learner->{store}->untie_db();

_debug_dsn(33);
ok($sa->{bayes_scanner}->learn(0, $mail));

_debug_dsn(34);
ok($learner->{store}->tie_db_writable());

_debug_dsn(35);
ok($learner->{store}->seen_get($msgid) eq 'h');

$learner->{store}->untie_db();

_debug_dsn(36);
ok($learner->{store}->tie_db_writable());

$tokerror = 0;
foreach my $tok (keys %{$toks}) {
  _debug_dsn(37);
  my ($spam, $ham, $atime) = $learner->{store}->tok_get($tok);
  if ($spam  > 0 || $ham == 0) {
    $tokerror = 1;
  }
}
_debug_dsn(38);
ok(!$tokerror);

$learner->{store}->untie_db();

_debug_dsn(39);
ok($sa->{bayes_scanner}->forget($mail));

_debug_dsn(40);
ok($learner->{store}->tie_db_writable());

_debug_dsn(41);
ok(!$learner->{store}->seen_get($msgid));

$learner->{store}->untie_db();

# This bit breaks abstraction a bit, the userid is an implementation detail,
# but is necessary to perform some of the tests.  Perhaps in the future we
# can add some sort of official API for this sort of thing.
$testuserid = $learner->{store}->{_userid};
_debug_dsn(42);
ok(defined($testuserid));

_debug_dsn(43);
ok($learner->{store}->clear_database());

_debug_dsn(44);
ok(database_clear_p($testuser, $testuserid));

$sa->finish_learner();

undef $sa;

sa_t_init("bayessql"); # this wipes out what is there and begins anew

# make sure we learn to a journal
tstprefs ("
  $dbconfig
  bayes_min_spam_num 10
  bayes_min_ham_num 10
  bayes_sql_override_username $testuser
");

# we get to bastardize the existing pattern matching code here.  It lets us provide
_debug_dsn(45);
# our own checking callback and keep using the existing ok_all_patterns call
%patterns = ( 1 => 'Acted on message' );

$wanted_examined = count_files("data/spam");
_debug_dsn(46);
ok(salearnrun("--spam data/spam", \&check_examined));
_debug_dsn(47);
ok_all_patterns();

$wanted_examined = count_files("data/nice");
_debug_dsn(48);
ok(salearnrun("--ham data/nice", \&check_examined));
_debug_dsn(49);
ok_all_patterns();

$wanted_examined = count_files("data/welcomelists");
_debug_dsn(50);
ok(salearnrun("--ham data/welcomelists", \&check_examined));
_debug_dsn(51);
ok_all_patterns();

$wanted_examined = 3;
_debug_dsn(52);
ok(salearnrun("--ham --mbox data/nice.mbox", \&check_examined));
_debug_dsn(53);
ok_all_patterns();

$wanted_examined = 3;
_debug_dsn(54);
ok(salearnrun("--ham --mbox < data/nice.mbox", \&check_examined));
_debug_dsn(55);
ok_all_patterns();

$wanted_examined = 3;
_debug_dsn(56);
ok(salearnrun("--forget --mbox data/nice.mbox", \&check_examined));
_debug_dsn(57);
ok_all_patterns();

%patterns = ( 'non-token data: bayes db version' => 'db version' );
_debug_dsn(58);
ok(salearnrun("--dump magic", \&patterns_run_cb));
_debug_dsn(59);
ok_all_patterns();


use constant SCAN_USING_PERL_CODE_TEST => 1;
# jm: off! not working for some reason.   Mind you, this is
# not a supported way to call these APIs!  so no biggie

if (SCAN_USING_PERL_CODE_TEST) {
$sa = create_saobj();

$sa->init();

$learner = $sa->call_plugins("learner_get_implementation");

open(MAIL,"< ../sample-nonspam.txt");

$raw_message = do {
  local $/;
  <MAIL>;
};

close(MAIL);

@msg = ();
foreach my $line (split(/^/m,$raw_message)) {
  $line =~ s/\r$//;
  push(@msg, $line);
}

$mail = $sa->parse( \@msg );

$body = $learner->get_body_from_msg($mail);

my $msgstatus = Mail::SpamAssassin::PerMsgStatus->new($sa, $mail);

_debug_dsn(60);
ok($msgstatus);

my $score = $learner->scan($msgstatus, $mail, $body);

# Pretty much we can't count on the data returned with such little training
# so just make sure that the score wasn't equal to .5 which is the default
# return value.
print "\treturned score: $score\n";
_debug_dsn(61);
ok($score =~ /\d/ && $score <= 1.0 && $score != .5);

open(MAIL,"< ../sample-spam.txt");

$raw_message = do {
  local $/;
  <MAIL>;
};

close(MAIL);

@msg = ();
foreach my $line (split(/^/m,$raw_message)) {
  $line =~ s/\r$//;
  push(@msg, $line);
}

$mail = $sa->parse( \@msg );

$body = $learner->get_body_from_msg($mail);

$msgstatus = Mail::SpamAssassin::PerMsgStatus->new($sa, $mail);

$score = $learner->scan($msgstatus, $mail, $body);

# Pretty much we can't count on the data returned with such little training
# so just make sure that the score wasn't equal to .5 which is the default
# return value.
print "\treturned score: $score\n";
_debug_dsn(62);
ok($score =~ /\d/ && $score <= 1.0 && $score != .5);
}

# This bit breaks abstraction a bit, the userid is an implementation detail,
# but is necessary to perform some of the tests.  Perhaps in the future we
# can add some sort of official API for this sort of thing.
$testuserid = $learner->{store}->{_userid};
_debug_dsn(63);
ok(defined($testuserid));

_debug_dsn(64);
ok($learner->{store}->clear_database());

_debug_dsn(65);
ok(database_clear_p($testuser, $testuserid));

$sa->finish_learner();

}
#---------------------------------------------------------------------------

sub check_examined {
  local ($_);
  my $string = shift;

  if (defined $string) {
    $_ = $string;
  } else {
    $_ = join ('', <IN>);
  }

  if ($_ =~ /(?:Forgot|Learned) tokens from \d+ message\(s\) \((\d+) message\(s\) examined\)/) {
    #print STDERR "examined $1 messages\n";
    if (defined $wanted_examined && $wanted_examined == $1) {
      $found{'Acted on message'}++;
    }
  }
}

sub count_files {
  my $cnt = 0;
  find({wanted => sub { $cnt++ if -f $_; }, no_chdir => 1}, $_[0]);
  return $cnt;
}

# WARNING! Do not use this as an example, this breaks abstraction
# and is here strictly to help the regression tests.
sub database_clear_p {
  my ($username, $userid) = @_;

  my $dbh = DBI->connect($dbdsn,$dbusername,$dbpassword);

  if (!defined($dbh)) {
    return 0;
  }

  my @row_ary;

  my $sql = "SELECT count(*) from bayes_vars where username = ?";
  @row_ary = $dbh->selectrow_array($sql, undef, $username);
  return 0 if ($row_ary[0] != 0);

  $sql = "SELECT count(*) from bayes_token where id = ?";
  @row_ary = $dbh->selectrow_array($sql, undef, $userid);
  return 0 if ($row_ary[0] != 0);

  $sql = "SELECT count(*) from bayes_seen where id = ?";
  @row_ary = $dbh->selectrow_array($sql, undef, $userid);
  return 0 if ($row_ary[0] != 0);

  $sql = "SELECT count(*) from bayes_expire where id = ?";
  @row_ary = $dbh->selectrow_array($sql, undef, $userid);
  return 0 if ($row_ary[0] != 0);

  $dbh->disconnect();

  return 1;
}

sub _debug_dsn {
  my ($callid) = @_;
  if ($no_error_reported and ($dbconfig =~ /dbi:SQLite:dbname=(.*)\/bayes.db/)) {
    my $dbdir = $1;
    my $dbfile = "$dbdir/bayes.db";
    unless (-w $dbfile) {
      $no_error_reported = 0;
      my $direxists = (( -e $dbdir )? '' : 'not ').'exists';
      my $dirreadable = (( -r $dbdir )? '' : 'not ').'readable';
      my $dirwritable = (( -w $dbdir )? '' : 'not ').'writeable';
      my $fileexists = (( -e $dbfile )? '' : 'not ').'exists';
      my $filereadable = (( -r $dbfile )? '' : 'not ').'readable';
      my $filewritable = (( -w $dbfile )? '' : 'not ').'writeable';
      diag("DEBUG $callid $dbdir $direxists, $dirreadable, $dirwritable;  $dbfile $fileexists, $filereadable, $filewritable");
    }
  }
}

