use strict;
use warnings;

package Git::Helpers;
$Git::Helpers::VERSION = '0.000002';
use Carp qw( croak );
use File::pushd qw( pushd );
use Git::Sub;
use Sub::Exporter -setup => { exports => ['checkout_root'] };
use Try::Tiny;

sub checkout_root {
    my $dir = shift;

    my $new_dir;
    $new_dir = pushd($dir) if $dir;

    my $root;
    try {
        $root = scalar git::rev_parse qw(--show-toplevel);
    }
    catch {
        $dir ||= '.';
        croak "Error in $dir $_";
    };
    return $root;
}

1;

#ABSTRACT: Shortcuts for common Git commands

__END__

=pod

=encoding UTF-8

=head1 NAME

Git::Helpers - Shortcuts for common Git commands

=head1 VERSION

version 0.000002

=head1 SYNOPSIS

    use Git::Helpers qw( checkout_root );
    my $root = checkout_root();

=head2 checkout_root( $dir )

Gives you the root level of the git checkout which you are currently in.
Optionally accepts a directory parameter.  If you provide the directory
parameter, C<checkout_root> will temporarily C<chdir> to this directory and
find the top level of the repository.

This method will throw an exception if it cannot find a git repository at the
directory provided.

=head1 AUTHOR

Olaf Alders <olaf@wundercounter.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Olaf Alders.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
