package Getopt::Long::CGI;

use 5.006;
use strict;
use warnings;

use CGI;
use Getopt::Long();

require Exporter;

our @ISA = qw(Exporter);

our @EXPORT_OK = qw(GetOptions);

our @EXPORT = qw(GetOptions);

our $VERSION = '0.01';

sub GetOptions {
    my %options = @_;

    if (not $ENV{SERVER_SOFTWARE}) {
        # Not CGI - pass upstream for normal command line handling
        return Getopt::Long::GetOptions(%options);
    }

    my $cgi = CGI->new();

    if (not $cgi->param('__getopt__')) {
        # Not a form submission, so display form
        print $cgi->start_form(-method => "POST", -action => "");
        print $cgi->start_table();
        foreach my $key (sort keys %options) {
            my $val = $options{$key};
            if ($key !~ m/^([\w\d\|]+)([=:][isf])?([\+!\@\%])?$/) {
                next; # Not sure what this is
            }

            my $name = [split('|', $1, 1)]->[0];
            print $cgi->start_Tr();
            print $cgi->th($name . ($2 || ""));

            my $inner;
            if (($3 || "") eq '+') {
                die "Cannot mix + and params" if $2;
                $inner = $cgi->popup_menu(-name => $name, -values => [1..9]);
            } elsif (($3 || "") eq '!') {
                die "Cannot mix ! and params" if $2;
                $inner = $cgi->checkbox(-name => $name, -checked => 1);
            } elsif ($2) {
                $inner = $cgi->textfield(-name => $name);
                if (($3 || "") eq '%' or ($3 || "") eq '@' or ref($val) eq 'ARRAY' or ref($val) eq 'HASH') {
                    $inner = [$inner, $cgi->br(), $inner, $cgi->br(), $inner];
                }
            } else {
                $inner = $cgi->checkbox(-name => $name);
            }

            print $cgi->td($inner);
            print $cgi->end_Tr();
        }
        print $cgi->end_table();
        # This name is looked for on submit
        print $cgi->submit(-name => '__getopt__', -value => 'Go');
        print $cgi->end_form();
        return;
    }

    # CGI form submission, so grab responses
    foreach my $key (sort keys %options) {
        my $val = $options{$key};
        if ($key !~ m/^([\w\d\|]+)([=:][isf])?([\+!\@\%])?$/) {
            next; # Unknown item
        }

        my $name = [split('|', $1, 1)]->[0];

        if (($3 || "") eq '+') {
            $$val = $cgi->param($name); # "Incremental" integer
        } elsif ($2) {
            my @values = $cgi->param($name);
            my $type = $2;
            if (($3 || "") eq '%' or ref($val) eq 'HASH') {
                my %values = map { split /=/, $_, 1 } @values;
                if ($type =~ m/i$/) {
                    foreach my $k (keys %values) {
                        $values{$k} = int($values{$k})
                    }
                } elsif ($type =~ m/f$/) {
                    foreach my $k (keys %values) {
                        $values{$k} = 0 + $values{$k}
                    }
                }
                if (ref($val) eq 'CODE') {
                    while(my($k, $v) = each %values) {
                       $val->($name, $k, $v);
                    }
                } else {
                    %$val = %values;
                }
            } else {
                if ($type =~ m/i$/) {
                    @values = map { int($_) } @values;
                } elsif ($type =~ m/f$/) {
                    @values = map { 0 + $_ } @values;
                }
                if (($3 || "") eq '@' or ref($val) eq 'ARRAY') {
                    if (ref($val) eq 'CODE') {
                        $val->($name, \@values)
                    } else {
                        @$val = @values;
                    }
                } else {
                    if (ref($val) eq 'CODE') {
                        $val->($name, $values[0]);
                    } else {
                    	$$val = $values[0];
                    }
                }
            }
        } else {
            # Checkbox
            $$val = $cgi->param($name) ? 1 : 0;
        }
    }
}

1;
__END__

=head1 NAME

Getopt::Long::CGI - Perl extension for command-line and CGI argument parsing.

=head1 SYNOPSIS

  use Getopt::Long::CGI;

  my $data   = "file.dat";
  my $length = 24;
  my $verbose;
  GetOptions ("length=i" => \$length,    # numeric
              "file=s"   => \$data,      # string
              "verbose"  => \$verbose)   # flag
  or die("Error in command line arguments\n");

=head1 DESCRIPTION

Using the same syntax as L<Getopt::Long>, scripts work on the command-line
in the same way. If the same script is called in a CGI-like environment it'll
change mode and display an HTML form of the options accepted on the
command-line - from which you can run the script and see the output.

=head2 EXPORT

GetOptions

=head1 SEE ALSO

L<Getopt::Long>

=head1 AUTHOR

Luke Ross, E<lt>lr@lukeross.nameE<gt>

You can report bugs on the issue tracker at
L<https://github.com/lukeross/Getopt-Long-CGI>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2014-2015 by Luke Ross

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.6.0 or,
at your option, any later version of Perl 5 you may have available.

=cut
