#!env perl
use strict;
use diagnostics;
use Storable qw/nstore_fd/;
use POSIX qw/EXIT_SUCCESS/;

BEGIN {
    use File::Spec;
    unshift(@INC, 'lib');
}
use MarpaX::Database::Terminfo;
my $terminfo = MarpaX::Database::Terminfo->new();
my $buffer = do {local $/; <DATA>;};
print STDERR "Generating ncurses terminfo perl-ready database - be patient\n";
my $value = $terminfo->parse(\$buffer)->value();
my $outfile = File::Spec->catfile('share', 'ncurses-terminfo.storable');
open(OUTFILE, '>', $outfile) || die "Cannot open $outfile; $!";
print STDERR "Writing ncurses database with Storable into $outfile\n";
nstore_fd ${$value}, \*OUTFILE;
close(OUTFILE) || warn "Cannot close $outfile, $!\n";
exit(EXIT_SUCCESS);

__DATA__
######## TERMINAL TYPE DESCRIPTIONS SOURCE FILE
#
# This version of terminfo.src is distributed with ncurses and is maintained
# by Thomas E. Dickey (TD).
#
# Report bugs and new terminal descriptions to
#	bug-ncurses@gnu.org
#
#	$Revision: 1.478 $
#	$Date: 2013/06/07 23:27:32 $
#
# The original header is preserved below for reference.  It is noted that there
# is a "newer" version which differs in some cosmetic details (but actually
# stopped updates several years ago); we have decided to not change the header
# unless there is also a change in content.
#
# To further muddy the waters, it is noted that changes to this file as part of
# maintenance of ncurses (since 1996) are generally conceded to be copyright
# under the ncurses MIT-style license.  That was the effect of the agreement
# which the principal authors of ncurses made in 1998.  However, since much of
# the file itself is of unknown authorship (and the disclaimer below makes it
# obvious that Raymond cannot or will not convey rights over those parts),
# there is no explicit copyright notice on the file itself.
#
# It would also be a nuisance to split the file into unknown/known authorship
# and move pieces as they are maintained, since many of the maintenance changes
# have been small corrections to Raymond's translations to/from termcap format,
# correcting the data but not the accompanying annotations.
#
# In any case, note that almost half of this file is not data but annotations
# which reflect creative effort.  Furthermore, the structure of entries to
# reuse common chunks also is creative (and subject to copyright).  Finally,
# some portions of the data are derivative work under a compatible MIT-style
# license from xterm.
#
#------------------------------------------------------------------------------
#	Version 10.2.1
#	terminfo syntax
#
#	Eric S. Raymond		(current maintainer)
#	John Kunze, Berkeley
#	Craig Leres, Berkeley
#
# Please e-mail changes to terminfo@thyrsus.com; the old termcap@berkeley.edu
# address is no longer valid.  The latest version can always be found at
# <http://www.tuxedo.org/terminfo>.
#
# PURPOSE OF THIS FILE:
#
# This file describes the capabilities of various character-cell terminals,
# as needed by software such as screen-oriented editors.
#
# Other terminfo and termcap files exist, supported by various OS vendors
# or as relics of various older versions of UNIX.  This one is the longest
# and most comprehensive one in existence.  It subsumes not only the entirety
# of the historical 4.4BSD, GNU, System V and SCO termcap files and the BRL
# termcap file, but also large numbers of vendor-maintained termcap and
# terminfo entries more complete and carefully tested than those in historical
# termcap/terminfo versions.
#
# Pointers to related resources (including the ncurses distribution) may
# be found at <http://www.tuxedo.org/terminfo>.
#
# INTERNATIONALIZATION:
#
# This file uses only the US-ASCII character set (no ISO8859 characters).
#
# This file assumes a US-ASCII character set. If you need to fix this, start
# by global-replacing \E(B and \E)B with the appropriate ISO 6429 enablers
# for your character set.  \E(A and \E)A enables the British character set
# with the pound sign at position 2/3.
#
# In a Japanese-processing environment using EUC/Japanese or Shift-JIS,
# C1 characters are considered the first-byte set of the Japanese encodings,
# so \E)0 should be avoided in <enacs> and initialization strings.
#
# FILE FORMAT:
#
# The version you are looking at may be in any of three formats: master
# (terminfo with OT capabilities), stock terminfo, or termcap.  You can tell
# which by the format given in the header above.
#
# The master format is accepted and generated by the terminfo tools in the
# ncurses suite; it differs from stock (System V-compatible) terminfo only
# in that it admits a group of capabilities (prefixed `OT') equivalent to
# various obsolete termcap capabilities.  You can, thus, convert from master
# to stock terminfo simply by filtering with `sed "/OT[^,]*,/s///"'; but if
# you have ncurses `tic -I' is nicer (among other things, it automatically
# outputs entries in a canonical form).
#
# The termcap version is generated automatically from the master version
# using tic -C.  This filtering leaves in the OT capabilities under their
# original termcap names.  All translated entries fit within the 1023-byte
# string-table limit of archaic termcap libraries except where explicitly
# noted below.  Note that the termcap translation assumes that your termcap
# library can handle multiple tc capabilities in an entry. 4.4BSD has this
# capability.  Older versions of GNU termcap, through 1.3, do not.
#
# For details on these formats, see terminfo(5) in the ncurses distribution,
# and termcap(5) in the 4.4BSD Unix Programmer's Manual.  Be aware that 4.4BSD
# curses has been declared obsolete by the caretakers of the 4.4BSD sources
# as of June 1995; they are encouraging everyone to migrate to ncurses.
#
# Note: unlike some other distributed terminfo files (Novell Unix & SCO's),
# no entry in this file has embedded comments.  This is so source translation
# to termcap only has to carry over leading comments.  Also, no name field
# contains embedded whitespace (such whitespace confuses rdist).
#
# Further note: older versions of this file were often installed with an editor
# script (reorder) that moved the most common terminal types to the front of
# the file.  This should no longer be necessary, as the file is now ordered
# roughly by type frequency with ANSI/VT100 and other common types up front.
#
# Some information has been merged in from terminfo files distributed by
# USL and SCO (see COPYRIGHTS AND OTHER DELUSIONS below).  Much information
# comes from vendors who maintain official terminfos for their hardware
# (notably DEC and Wyse).
#
# A detailed change history is included at the end of this file.
#
# FILE ORGANIZATION:
#
# Comments in this file begin with # - they cannot appear in the middle
# of a terminfo/termcap entry (this feature had to be sacrificed in order
# to allow standard terminfo and termcap syntax to be generated cleanly from
# the master format).  Individual capabilities are commented out by
# placing a period between the colon and the capability name.
#
# The file is divided up into major sections (headed by lines beginning with
# the string "########") and minor sections (beginning with "####"); do
#
#	grep "^####" <file> | more
#
# to see a listing of section headings.  The intent of the divisions is
# (a) to make it easier to find things, and (b) to order the database so
# that important and frequently-encountered terminal types are near the
# front (so that you'll get reasonable search efficiency from a linear
# search of the termcap form even if you don't use reorder).  Minor sections
# usually correspond to manufacturers or standard terminal classes.
# Parenthesized words following manufacturer names are type prefixes or
# product line names used by that manufacturers.
#
# HOW TO READ THE ENTRIES:
#
# The first name in an entry is the canonical name for the model or
# type, last entry is a verbose description.  Others are mnemonic synonyms for
# the terminal.
#
# Terminal names look like <manufacturer> <model> - <modes/options>
# The part to the left of the dash, if a dash is present, describes the
# particular hardware of the terminal.  The part to the right may be used
# for flags indicating special ROMs, extra memory, particular terminal modes,
# or user preferences.
#
# All names should be in lower case, for consistency in typing.
#
# The following are conventionally used suffixes:
#	-2p	Has two pages of memory.  Likewise 4p, 8p, etc.
#	-am	Enable auto-margin.
#	-m	Monochrome.  Suppress color support
#	-mc	Magic-cookie.  Some terminals (notably older Wyses) can
#		only support one attribute without magic-cookie lossage.
#		Their base entry is usually paired with another that
#		uses magic cookies to support multiple attributes.
#	-nam	No auto-margin - suppress <am> capability
#	-nl	No labels - suppress soft labels
#	-ns	No status line - suppress status line
#	-rv	Terminal in reverse video mode (black on white)
#	-s	Enable status line.
#	-vb	Use visible bell (<flash>) rather than <bel>.
#	-w	Wide - in 132 column mode.
# If a name has multiple suffixes and one is a line height, that one should
# go first.  Thus `aaa-30-s-rv' is recommended over `aaa-s-rv-30'.
#
# Entries with embedded plus signs are designed to be included through use/tc
# capabilities, not used as standalone entries.
#
# To avoid search clashes, some older all-numeric names for terminals have
# been removed (i.e., "33" for the Model 33 Teletype, "2621" for the HP2621).
# All primary names of terminals now have alphanumeric prefixes.
#
# Comments marked "esr" are mostly results of applying the termcap-compiler
# code packaged with ncurses and contemplating the resulting error messages.
# In many cases, these indicated obvious fixes to syntax garbled by the
# composers.  In a few cases, I was able to deduce corrected forms for garbled
# capabilities by looking at context.  All the information in the original
# entries is preserved in the comments.
#
# In the comments, terminfo capability names are bracketed with <> (angle
# brackets).  Termcap capability names are bracketed with :: (colons).
#
# INTERPRETATION OF USER CAPABILITIES
#
# The System V Release 4 and XPG4 terminfo format defines ten string
# capabilities for use by applications, <u0>...<u9>.   In this file, we use
# certain of these capabilities to describe functions which are not covered
# by terminfo.  The mapping is as follows:
#
#	u9	terminal enquire string (equiv. to ANSI/ECMA-48 DA)
#	u8	terminal answerback description
#	u7	cursor position request (equiv. to VT100/ANSI/ECMA-48 DSR 6)
#	u6	cursor position report (equiv. to ANSI/ECMA-48 CPR)
#
# The terminal enquire string <u9> should elicit an answerback response
# from the terminal.  Common values for <u9> will be ^E (on older ASCII
# terminals) or \E[c (on newer VT100/ANSI/ECMA-48-compatible terminals).
#
# The cursor position request (<u7>) string should elicit a cursor position
# report.  A typical value (for VT100 terminals) is \E[6n.
#
# The terminal answerback description (u8) must consist of an expected
# answerback string.  The string may contain the following scanf(3)-like
# escapes:
#
#	%c	Accept any character
#	%[...]	Accept any number of characters in the given set
#
# The cursor position report (<u6>) string must contain two scanf(3)-style
# %d format elements.  The first of these must correspond to the Y coordinate
# and the second to the %d.  If the string contains the sequence %i, it is
# taken as an instruction to decrement each value after reading it (this is
# the inverse sense from the cup string).  The typical CPR value is
# \E[%i%d;%dR (on VT100/ANSI/ECMA-48-compatible terminals).
#
# These capabilities are used by tack(1m), the terminfo action checker
# (distributed with ncurses 5.0).
#
# TABSET FILES
#
# All the entries in this file have been edited to assume that the tabset
# files directory is /usr/share/tabset, in conformance with the File Hierarchy
# Standard for Linux and open-source BSD systems.  Some vendors (notably Sun)
# use /usr/lib/tabset or (more recently) /usr/share/lib/tabset.
#
# No curses package we know of actually uses these files.  If their location
# is an issue, you will have to hand-patch the file locations before compiling
# this file.
#
# REQUEST FOR CONTACT INFORMATION AND HISTORICAL MATERIAL
#
# As the ANSI/ECMA-48 standard and variants take firmer hold, and as
# character-cell terminals are increasingly replaced by X displays, much of
# this file is becoming a historical document (this is part of the reason for
# the new organization, which puts ANSI types, xterm, Unix consoles,
# and vt100 up front in confidence that this will catch 95% of new hardware).
#
# For the terminal types still alive, I'd like to have manufacturer's
# contact data (Internet address and/or snail-mail + phone).
#
# I'm also interested in enriching the comments so that the latter portions of
# the file do in fact become a potted history of VDT technology as seen by
# UNIX hackers.  Ideally, I'd like the headers for each manufacturer to
# include its live/dead/out-of-the-business status, and for as many
# terminal types as possible to be tagged with information like years
# of heaviest use, popularity, and interesting features.
#
# I'm especially interested in identifying the obscure entries listed under
# `Miscellaneous obsolete terminals, manufacturers unknown' before the tribal
# wisdom about them gets lost.  If you know a lot about obscure old terminals,
# please go to the terminfo resource page, grab the UFO file (ufo.ti), and
# eyeball it for things you can identify and describe.
#
# If you have been around long enough to contribute, please read the file
# with this in mind and send me your annotations.
#
# COPYRIGHTS AND OTHER DELUSIONS
#
# The BSD ancestor of this file had a standard Regents of the University of
# California copyright with dates from 1980 to 1993.
#
# Some information has been merged in from a terminfo file SCO distributes.
# It has an obnoxious boilerplate copyright which I'm ignoring because they
# took so much of the content from the ancestral BSD versions of this file
# and didn't attribute it, thereby violating the BSD Regents' copyright.
#
# Not that anyone should care.  However many valid functions copyrights may
# serve, putting one on a termcap/terminfo file with hundreds of anonymous
# contributors makes about as much sense as copyrighting a wall-full of
# graffiti -- it's legally dubious, ethically bogus, and patently ridiculous.
#
# This file deliberately has no copyright.  It belongs to no one and everyone.
# If you claim you own it, you will merely succeed in looking like a fool.
# Use it as you like.  Use it at your own risk.  Copy and redistribute freely.
# There are no guarantees anywhere.  Svaha!
#

######## ANSI, UNIX CONSOLE, AND SPECIAL TYPES
#
# This section describes terminal classes and brands that are still
# quite common.
#

#### Specials
#
# Special "terminals".  These are used to label tty lines when you don't
# know what kind of terminal is on it.  The characteristics of an unknown
# terminal are the lowest common denominator - they look about like a ti 700.
#

dumb|80-column dumb tty,
	am,
	cols#80,
	bel=^G, cr=^M, cud1=^J, ind=^J,
unknown|unknown terminal type,
	gn, use=dumb,
lpr|printer|line printer,
	OTbs, hc, os,
	cols#132, lines#66,
	bel=^G, cr=^M, cub1=^H, cud1=^J, ff=^L, ind=^J,
glasstty|classic glass tty interpreting ASCII control characters,
	OTbs, am,
	cols#80,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, ht=^I, kcub1=^H,
	kcud1=^J, nel=^M^J, .kbs=^H,

vanilla|dumb tty,
	OTbs,
	bel=^G, cr=^M, cud1=^J, ind=^J,

# This is almost the same as "dumb", but with no prespecified width.
# DEL and ^C are hardcoded to act as kill characters.
# ^D acts as a line break (just like newline).
# It also interprets
#      \033];xxx\007
# for compatibility with xterm -TD
9term|Plan9 terminal emulator for X,
	am,
	OTnl=^J, bel=^G, cud1=^J,

#### ANSI.SYS/ISO 6429/ECMA-48 Capabilities
#
# See the end-of-file comment for more on these.
#

# ANSI capabilities are broken up into pieces, so that a terminal
# implementing some ANSI subset can use many of them.
ansi+local1,
	cub1=\E[D, cud1=\E[B, cuf1=\E[C, cuu1=\E[A,
ansi+local,
	cub=\E[%p1%dD, cud=\E[%p1%dB, cuf=\E[%p1%dC,
	cuu=\E[%p1%dA, use=ansi+local1,
ansi+tabs,
	cbt=\E[Z, ht=^I, hts=\EH, tbc=\E[3g,
ansi+inittabs,
	it#8, use=ansi+tabs,
ansi+erase,
	clear=\E[H\E[J, ed=\E[J, el=\E[K,
ansi+rca,
	hpa=\E[%p1%{1}%+%dG, vpa=\E[%p1%{1}%+%dd,
ansi+cup,
	cup=\E[%i%p1%d;%p2%dH, home=\E[H,
ansi+rep,
	rep=%p1%c\E[%p2%{1}%-%db,
ansi+idl1,
	dl1=\E[M, il1=\E[L,
ansi+idl,
	dl=\E[%p1%dM, il=\E[%p1%dL, use=ansi+idl1,
ansi+idc,
	dch1=\E[P, ich=\E[%p1%d@, ich1=\E[@, rmir=\E6, smir=\E6,
ansi+arrows,
	kbs=^H, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	khome=\E[H,
ansi+sgr|ansi graphic renditions,
	blink=\E[5m, invis=\E[8m, rev=\E[7m,
	sgr=\E[0%?%p3%t;7%;%?%p4%t;5%;%?%p7%t;8%;m,
	sgr0=\E[0m,
ansi+sgrso|ansi standout only,
	rmso=\E[m, smso=\E[7m,
ansi+sgrul|ansi underline only,
	rmul=\E[m, smul=\E[4m,
ansi+sgrbold|ansi graphic renditions; assuming terminal has bold; not dim,
	bold=\E[1m,
	sgr=\E[%?%p1%t7;%;%?%p2%t4;%;%?%p3%t7;%;%?%p4%t5;%;%?%p6%t1;%;%?%p7%t8;%;m,
	use=ansi+sgr, use=ansi+sgrso, use=ansi+sgrul,
ansi+sgrdim|ansi graphic renditions; assuming terminal has dim; not bold,
	dim=\E[2m,
	sgr=\E[%?%p1%t7;%;%?%p2%t4;%;%?%p3%t7;%;%?%p4%t5;%;%?%p5%t2;%;%?%p7%t8;%;m,
	use=ansi+sgr, use=ansi+sgrso, use=ansi+sgrul,
ansi+csr|ansi scroll-region plus cursor save & restore,
	csr=\E[%i%p1%d;%p2%dr, rc=\E8, sc=\E7,

# The normal (ANSI) flavor of "media copy" building block asserts that
# characters sent to the printer do not echo on the screen. DEC terminals
# can also be put into autoprinter mode, where each line is sent to the
# printer as you move off that line, e.g., by a carriage return.
ansi+pp|ansi printer port,
	mc5i,
	mc0=\E[i, mc4=\E[4i, mc5=\E[5i,
dec+pp|DEC autoprinter mode,
	mc0=\E[i, mc4=\E[?4i, mc5=\E[?5i,

# The IBM PC alternate character set.  Plug this into any Intel console entry.
# We use \E[11m for rmacs rather than \E[12m so the <acsc> string can use the
# ROM graphics for control characters such as the diamond, up- and down-arrow.
# This works with the System V, Linux, and BSDI consoles.  It's a safe bet this
# will work with any Intel console, they all seem to have inherited \E[11m
# from the ANSI.SYS de-facto standard.
klone+acs|alternate character set for ansi.sys displays,
	acsc=+\020\,\021-\030.^Y0\333`\004a\261f\370g\361h\260j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	rmacs=\E[10m, smacs=\E[11m,

# Highlight controls corresponding to the ANSI.SYS standard.  Most
# console drivers for Intel boxes obey these.  Makes the same assumption
# about \E[11m as klone+acs.  True ANSI/ECMA-48 would have <rmso=\E[27m>,
# <rmul=\E[24m>, but this isn't a documented feature of ANSI.SYS.
klone+sgr|attribute control for ansi.sys displays,
	blink=\E[5m, bold=\E[1m, rev=\E[7m, rmpch=\E[10m,
	rmso=\E[m, rmul=\E[m,
	sgr=\E[0;10%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;%?%p9%t;11%;m,
	sgr0=\E[0;10m, smpch=\E[11m, smso=\E[7m, smul=\E[4m,
	use=klone+acs,

# Most Intel boxes do not treat "invis" (invisible) text.
klone+sgr8|attribute control for ansi.sys displays,
	invis=\E[8m,
	sgr=\E[0;10%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;%?%p7%t;8%;%?%p9%t;11%;m,
	use=klone+sgr,

# Highlight controls corresponding to the ANSI.SYS standard.  *All*
# console drivers for Intel boxes obey these.  Does not assume \E[11m will
# work; uses \E[12m instead, which is pretty bulletproof but loses you the ACS
# diamond and arrow characters under curses.
klone+sgr-dumb|attribute control for ansi.sys displays (no ESC [ 11 m),
	blink=\E[5m, bold=\E[1m, invis=\E[8m, rev=\E[7m, rmso=\E[m,
	rmul=\E[m,
	sgr=\E[0;10%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;%?%p7%t;8%;%?%p9%t;12%;m,
	sgr0=\E[0;10m, smacs=\E[12m, smso=\E[7m, smul=\E[4m,
	use=klone+acs,

# KOI8-R (RFC1489) acs (alternate character set)
# From: Qing Long <qinglong@Bolizm.ihep.su>, 24 Feb 1996.
klone+koi8acs|alternate character set for ansi.sys displays with KOI8 charset,
	acsc=+\020\,\021-\036.^_0\215`\004a\237f\234g\232h\222i\220j\205k\203l\202m\204n\212o\213p\216q\0r\217s\214t\206u\207v\210w\211x\201y\230z\231{\267|\274}L~\225,
	rmacs=\E[10m, smacs=\E[11m,

# ANSI.SYS color control.  The setab/setaf caps depend on the coincidence
# between SVr4/XPG4's color numbers and ANSI.SYS attributes.  Here are longer
# but equivalent strings that don't rely on that coincidence:
# setb=\E[4%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
# setf=\E[3%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
# The DOS 5 manual asserts that these sequences meet the ISO 6429 standard.
# They match a subset of ECMA-48.
klone+color|color control for ansi.sys and ISO6429-compatible displays,
	colors#8, ncv#3, pairs#64,
	op=\E[37;40m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,

# This is better than klone+color, it doesn't assume white-on-black as the
# default color pair,  but many `ANSI' terminals don't grok the <op> cap.
ecma+color|color control for ECMA-48-compatible terminals,
	AX,
	colors#8, ncv#3, pairs#64,
	op=\E[39;49m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,

# Attribute control for ECMA-48-compatible terminals
ecma+sgr|attribute capabilities for true ECMA-48 terminals,
	rmso=\E[27m, rmul=\E[24m, use=klone+sgr8,

# For comparison, here are all the capabilities implied by the Intel
# Binary Compatibility Standard (level 2) that fit within terminfo.
# For more detail on this rather pathetic standard, see the comments
# near the end of this file.
ibcs2|Intel Binary Compatibility Standard prescriptions,
	cbt=\E[Z, clear=\Ec, cub=\E[%p1%dD, cub1=\E[1D,
	cud=\E[%p1%dB, cud1=\E[1B, cuf=\E[%p1%dC, cuf1=\E[1C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[1A,
	dch=\E[%p1%dP, dispc=\E=%p1%dg, ech=\E[%p1%dX,
	hpa=\E[%i%p1%dG, hts=\EH, ich=\E[%p1%d@, ich1=\E[@,
	il=\E[%p1%dL, indn=\E[%p1%dS, rc=\E7, rin=\E[%p1%dT,
	rmam=\E[?7l, sc=\E7, smam=\E[?7h, tbc=\E[g,
	vpa=\E[%i%p1%dd,

#### ANSI/ECMA-48 terminals and terminal emulators
#
# See near the end of this file for details on ANSI conformance.
# Don't mess with these entries!  Lots of other entries depend on them!
#
# This section lists entries in a least-capable to most-capable order.
# if you're in doubt about what `ANSI' matches yours, try them in that
# order and back off from the first that breaks.

# ansi-mr is for ANSI terminals with ONLY relative cursor addressing
# and more than one page of memory.  It uses local motions instead of
# direct cursor addressing, and makes almost no assumptions. It does
# assume auto margins, no padding and/or xon/xoff, and a 24x80 screen.
ansi-mr|mem rel cup ansi,
	am, xon,
	cols#80, lines#24, use=vanilla, use=ansi+erase,
	use=ansi+local1,

# ansi-mini is a bare minimum ANSI terminal. This should work on anything, but
# beware of screen size problems and memory relative cursor addressing.
ansi-mini|any ansi terminal with pessimistic assumptions,
	am, xon,
	cols#80, lines#24, use=vanilla, use=ansi+cup,
	use=ansi+erase,

# ansi-mtabs adds relative addressing and minimal tab support
ansi-mtabs|any ansi terminal with pessimistic assumptions,
	it#8,
	ht=^I, use=ansi-mini, use=ansi+local1,

# ANSI X3.64 from emory!mlhhh (Hugh Hansard) via BRL
#
# The following is an entry for the full ANSI 3.64 (1977).  It lacks
# padding, but most terminals using the standard are "fast" enough
# not to require any -- even at 9600 bps.  If you encounter problems,
# try including the padding specifications.
#
# Note: the :as: and :ae: specifications are not implemented here, for
# the available termcap documentation does not make clear WHICH alternate
# character set to specify.  ANSI 3.64 seems to make allowances for several.
# Please make the appropriate adjustments to fit your needs -- that is
# if you will be using alternate character sets.
#
# There are very few terminals running the full ANSI 3.64 standard,
# so I could only test this entry on one verified terminal (Visual 102).
# I would appreciate the results on other terminals sent to me.
#
# Please report comments, changes, and problems to:
#
# U.S. MAIL:   Hugh Hansard
#              Box: 22830
#              Emory University
#              Atlanta, GA. 30322.
#
# USENET {akgua,msdc,sb1,sb6,gatech}!emory!mlhhh.
#
# (Added vt100 <rc>,<sc> to quiet a tic warning --esr)
ansi77|ansi 3.64 standard 1977 version,
	OTbs, am, mir,
	cols#80, it#8, lines#24,
	bel=^G, clear=\E[;H\E[2J, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub1=^H, cud1=\E[B, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu1=\E[A, dch1=\E[P, dl1=\E[M$<5*/>, ed=\E[J, el=\E[K,
	home=\E[H, ht=^I, il1=\E[L$<5*/>, ind=\ED, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf1=\EOP,
	kf2=\EOR, kf4=\EOS, khome=\E[H, nel=^M\ED, rc=\E8, ri=\EM,
	rmir=\E[4l, rmso=\E[m, rmul=\E[m, sc=\E7, smir=\E[4h,
	smso=\E[7m, smul=\E[4m,

# Procomm and some other ANSI emulations don't recognize all of the ANSI-
# standard capabilities.  This entry deletes <cuu>, <cuf>, <cud>, <cub>, and
# <vpa>/<hpa> capabilities, forcing curses to use repetitions of <cuu1>,
# <cuf1>, <cud1> and <cub1>.  Also deleted <ich> and <ich1>, as QModem up to
# 5.03 doesn't recognize these.  Finally, we delete <rep> and <ri>, which seem
# to confuse many emulators.  On the other hand, we can count on these programs
# doing <rmacs>/<smacs>/<sgr>. Older versions of this entry featured
# <invis=\E[9m>, but <invis=\E[8m> now seems to be more common under
# ANSI.SYS influence.
# From: Eric S. Raymond <esr@snark.thyrsus.com> Oct 30 1995
pcansi-m|pcansi-mono|ibm-pc terminal programs claiming to be ansi (mono mode),
	OTbs, am, mir, msgr,
	cols#80, it#8, lines#24,
	bel=^G, cbt=\E[Z, clear=\E[H\E[J, cr=^M, cub1=\E[D,
	cud1=\E[B, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A,
	dch1=\E[P, dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I,
	hts=\EH, il1=\E[L, ind=^J, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, khome=\E[H, tbc=\E[3g,
	use=klone+sgr-dumb,
pcansi-25-m|pcansi25m|ibm-pc terminal programs with 25 lines (mono mode),
	lines#25, use=pcansi-m,
pcansi-33-m|pcansi33m|ibm-pc terminal programs with 33 lines (mono mode),
	lines#33, use=pcansi-m,
pcansi-43-m|ansi43m|ibm-pc terminal programs with 43 lines (mono mode),
	lines#43, use=pcansi-m,
# The color versions.  All PC emulators do color...
pcansi|ibm-pc terminal programs claiming to be ansi,
	use=klone+color, use=pcansi-m,
pcansi-25|pcansi25|ibm-pc terminal programs with 25 lines,
	lines#25, use=pcansi,
pcansi-33|pcansi33|ibm-pc terminal programs with 33 lines,
	lines#33, use=pcansi,
pcansi-43|pcansi43|ibm-pc terminal programs with 43 lines,
	lines#43, use=pcansi,

# ansi-m -- full ANSI X3.64 with ANSI.SYS-compatible attributes, no color.
# If you want pound signs rather than dollars, replace `B' with `A'
# in the <s0ds>, <s1ds>, <s2ds>, and <s3ds> capabilities.
# From: Eric S. Raymond <esr@snark.thyrsus.com> Nov 6 1995
ansi-m|ansi-mono|ANSI X3.64-1979 terminal with ANSI.SYS compatible attributes,
	mc5i,
	cub=\E[%p1%dD, cud=\E[%p1%dB, cuf=\E[%p1%dC,
	cuu=\E[%p1%dA, dch=\E[%p1%dP, dl=\E[%p1%dM,
	ech=\E[%p1%dX, el1=\E[1K, hpa=\E[%i%p1%dG, ht=\E[I,
	ich=\E[%p1%d@, il=\E[%p1%dL, indn=\E[%p1%dS, kbs=^H,
	kcbt=\E[Z, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kich1=\E[L, mc4=\E[4i, mc5=\E[5i, nel=\r\E[S,
	rep=%p1%c\E[%p2%{1}%-%db, rin=\E[%p1%dT, s0ds=\E(B,
	s1ds=\E)B, s2ds=\E*B, s3ds=\E+B, tbc=\E[3g,
	vpa=\E[%i%p1%dd, use=pcansi-m,

ansi+enq|ncurses extension for ANSI ENQ,
	u6=\E[%i%d;%dR, u7=\E[6n, u8=\E[?%[;0123456789]c,
	u9=\E[c,

# ansi -- this terminfo expresses the largest subset of X3.64 that will fit in
# standard terminfo.  Assumes ANSI.SYS-compatible attributes and color.
# From: Eric S. Raymond <esr@snark.thyrsus.com> Nov 6 1995
ansi|ansi/pc-term compatible with color,
	use=ansi+enq, use=ecma+color, use=klone+sgr8, use=ansi-m,

# ansi-generic is a vanilla ANSI terminal. This is assumed to implement
# all the normal ANSI stuff with no extensions. It assumes
# insert/delete line/char is there, so it won't work with
# vt100 clones. It assumes video attributes for bold, blink,
# underline, and reverse, which won't matter much if the terminal
# can't do some of those. Padding is assumed to be zero, which
# shouldn't hurt since xon/xoff is assumed.
ansi-generic|generic ansi standard terminal,
	am, xon,
	cols#80, lines#24, use=vanilla, use=ansi+csr, use=ansi+cup,
	use=ansi+rca, use=ansi+erase, use=ansi+tabs,
	use=ansi+local, use=ansi+idc, use=ansi+idl, use=ansi+rep,
	use=ansi+sgrbold, use=ansi+arrows,

#### DOS ANSI.SYS variants
#
# This completely describes the sequences specified in the DOS 2.1 ANSI.SYS
# documentation (except for the keyboard key reassignment feature, which
# doesn't fit the <pfkey> model well).  The klone+acs sequences were valid
# though undocumented.  The <pfkey> capability is untested but should work for
# keys F1-F10 (%p1 values outside this range will yield unpredictable results).
# From: Eric S. Raymond <esr@snark.thyrsus.com> Nov 7 1995
ansi.sys-old|ANSI.SYS under PC-DOS 2.1,
	OTbs, am, mir, msgr, xon,
	cols#80, lines#25,
	clear=\E[2J, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, el=\E[k, home=\E[H,
	is2=\E[m\E[?7h, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K,
	khome=^^, pfkey=\E[0;%p1%{58}%+%d;%p2"%s"p, rc=\E[u,
	rmam=\E[?7l, sc=\E[s, smam=\E[?7h, u6=\E[%i%d;%dR,
	u7=\E[6n, use=klone+color, use=klone+sgr8,

# Keypad:	Home=\0G	Up=\0H	PrPag=\0I
#		ka1,kh		kcuu1		kpp,ka3
#
#		Left=\0K	5=\0L		Right=\0M
#		kcub1		kb2		kcuf1
#
#		End=\0O		Down=\0P	NxPag=\0Q
#		kc1,kend	kcud1		kc3,knp
#
#		Ins=\0R		Del=\0S
#		kich1		kdch1
#
# On keyboard with 12 function keys,
#	shifted f-keys: F13-F24
#	control f-keys: F25-F36
#	alt f-keys:     F37-F48
# The shift/control/alt keys do not modify each other, but alt overrides both,
# and control overrides shift.
#
# <pfkey> capability for F1-F48 -TD
ansi.sys|ANSI.SYS 3.1 and later versions,
	el=\E[K, ka1=\0G, ka3=\0I, kb2=\0L, kbs=^H, kc1=\0O, kc3=\0Q,
	kcbt=\0^O, kcub1=\0K, kcud1=\0P, kcuf1=\0M, kcuu1=\0H,
	kdch1=\0S, kend=\0O, kf1=\0;, kf10=\0D, kf11=\0\205,
	kf12=\0\206, kf13=\0T, kf14=\0U, kf15=\0V, kf16=\0W,
	kf17=\0X, kf18=\0Y, kf19=\0Z, kf2=\0<, kf20=\0[, kf21=\0\\,
	kf22=\0], kf23=\0\207, kf24=\0\210, kf25=\0\^, kf26=\0_,
	kf27=\0`, kf28=\0a, kf29=\0b, kf3=\0=, kf30=\0c, kf31=\0d,
	kf32=\0e, kf33=\0f, kf34=\0g, kf35=\0\211, kf36=\0\212,
	kf37=\0h, kf38=\0i, kf39=\0j, kf4=\0>, kf40=\0k, kf41=\0l,
	kf42=\0m, kf43=\0n, kf44=\0o, kf45=\0p, kf46=\0q,
	kf47=\0\213, kf48=\0\214, kf5=\0?, kf6=\0@, kf7=\0A, kf8=\0B,
	kf9=\0C, khome=\0G, kich1=\0R, knp=\0Q, kpp=\0I,
	pfkey=\E[0;%?%p1%{11}%<%t%'\:'%e%?%p1%{13}%<%t%'z'%e%?%p1%{23}%<%t%'G'%e%?%p1%{25}%<%t%'p'%e%?%p1%'#'%<%t%'E'%e%?%p1%'%'%<%t%'f'%e%?%p1%'/'%<%t%'C'%e%{92}%;%;%;%;%;%;%;%p1%+%d;%p2"%s"p,
	use=ansi.sys-old,

#
# Define IBM PC keypad keys for vi as per MS-Kermit while using ANSI.SYS.
# This should only be used when the terminal emulator cannot redefine the keys.
# Since redefining keys with ansi.sys also affects PC-DOS programs, the key
# definitions must be restored.  If the terminal emulator is quit while in vi
# or others using <smkx>/<rmkx>, the keypad will not be defined as per PC-DOS.
# The PgUp and PgDn are prefixed with ESC so that tn3270 can be used on Unix
# (^U and ^D are already defined for tn3270).  The ESC is safe for vi but it
# does "beep".  ESC ESC i is used for Ins to avoid tn3270 ESC i for coltab.
# Note that <kcub1> is always BS, because PC-dos can tolerate this change.
# Caution: vi is limited to 256 string bytes, longer crashes or weirds out vi.
# Consequently the End keypad key could not be set (it is relatively safe and
# actually useful because it sends ^@ O, which beeps and opens a line above).
ansi.sysk|ansisysk|PC-DOS 3.1 ANSI.SYS with keypad redefined for vi,
	is2=U2 PC-DOS 3.1 ANSI.SYS with keypad redefined for vi 9-29-86\n\E[;75;8p,
	rmkx=\E[;71;0;71p\E[;72;0;72p\E[;73;0;73p\E[;77;0;77p\E[;80;0;80p\E[;81;0;81p\E[;82;0;82p\E[;83;0;83p,
	smkx=\E[;71;30p\E[;72;11p\E[;73;27;21p\E[;77;12p\E[;80;10p\E[;81;27;4p\E[;82;27;27;105p\E[;83;127p,
	use=ansi.sys,
#
# Adds ins/del line/character, hence vi reverse scrolls/inserts/deletes nicer.
nansi.sys|nansisys|PC-DOS Public Domain NANSI.SYS,
	dch1=\E[1P, dl1=\E[1M, ich1=\E[1@, il1=\E[1L,
	is2=U3 PC-DOS Public Domain NANSI.SYS 9-23-86\n,
	use=ansi.sys,
#
# See ansi.sysk and nansi.sys above.
nansi.sysk|nansisysk|PC-DOS Public Domain NANSI.SYS with keypad redefined for vi,
	dch1=\E[1P, dl1=\E[1M, ich1=\E[1@, il1=\E[1L,
	is2=U4 PC-DOS Public Domain NANSI.SYS with keypad redefined for vi 9-29-86\n\E[;75;8p,
	use=ansi.sysk,

#### Atari ST terminals

# From Guido Flohr <gufl0000@stud.uni-sb.de>.
#
tw52|tw52-color|Toswin window manager with color,
	bce,
	colors#16, pairs#256,
	oc=\Eb?\Ec0, op=\Eb?\Ec0,
	setab=\Ec%?%p1%{0}%=%t?%e%p1%{7}%=%t0%e%p1%{15}%=%t7%e%p1%{48}%+%c,
	setaf=\Eb%?%p1%{0}%=%t?%e%p1%{7}%=%t0%e%p1%{15}%=%t7%e%p1%{48}%+%c,
	setb=\Ec%?%p1%{0}%=%t?%e%p1%{7}%=%t0%e%p1%{15}%=%t7%e%p1%{48}%+%c,
	setf=\Eb%?%p1%{0}%=%t?%e%p1%{7}%=%t0%e%p1%{15}%=%t7%e%p1%{48}%+%c,
	use=tw52-m,
tw52-m|Toswin window manager monochrome,
	ul,
	ma#999,
	bold=\Eya, dch1=\Ea, dim=\EyB,
	is2=\Ev\Eq\Ez_\Ee\Ei\Eb?\Ec0, rev=\EyP, rmso=\EzQ,
	rmul=\EzH, rs2=\Ev\Eq\Ez_\Ee\Ei\Eb?\Ec0, sgr0=\Ez_,
	smso=\EyQ, smul=\EyH, use=at-m,
tt52|Atari TT medium and high resolution,
	lines#30, use=at-color,
st52-color|at-color|atari-color|atari_st-color|Atari ST with color,
	bce,
	colors#16, pairs#256,
	is2=\Ev\Eq\Ee\Eb1\Ec0, rs2=\Ev\Eq\Ee\Eb1\Ec0,
	setab=\Ec%?%p1%{0}%=%t1%e%p1%{1}%=%t2%e%p1%{2}%=%t3%e%p1%{3}%=%t>%e%p1%{4}%=%t4%e%p1%{5}%=%t7%e%p1%{6}%=%t5%e%p1%{7}%=%t0%e%p1%{8}%=%t8%e%p1%{9}%=%t9%e%p1%{10}%=%t\:%e%p1%{11}%=%t;%e%p1%{12}%=%t<%e%p1%{13}%=%t=%e%p1%{14}%=%t6%e?,
	setaf=\Eb%?%p1%{0}%=%t1%e%p1%{1}%=%t2%e%p1%{2}%=%t3%e%p1%{3}%=%t>%e%p1%{4}%=%t4%e%p1%{5}%=%t7%e%p1%{6}%=%t5%e%p1%{7}%=%t0%e%p1%{8}%=%t8%e%p1%{9}%=%t9%e%p1%{10}%=%t\:%e%p1%{11}%=%t;%e%p1%{12}%=%t<%e%p1%{13}%=%t=%e%p1%{14}%=%t6%e?,
	setb=\Ec%?%p1%{0}%=%t1%e%p1%{1}%=%t2%e%p1%{2}%=%t3%e%p1%{3}%=%t>%e%p1%{4}%=%t4%e%p1%{5}%=%t7%e%p1%{6}%=%t5%e%p1%{7}%=%t0%e%p1%{8}%=%t8%e%p1%{9}%=%t9%e%p1%{10}%=%t\:%e%p1%{11}%=%t;%e%p1%{12}%=%t<%e%p1%{13}%=%t=%e%p1%{14}%=%t6%e?,
	setf=\Eb%?%p1%{0}%=%t1%e%p1%{1}%=%t2%e%p1%{2}%=%t3%e%p1%{3}%=%t>%e%p1%{4}%=%t4%e%p1%{5}%=%t7%e%p1%{6}%=%t5%e%p1%{7}%=%t0%e%p1%{8}%=%t8%e%p1%{9}%=%t9%e%p1%{10}%=%t\:%e%p1%{11}%=%t;%e%p1%{12}%=%t<%e%p1%{13}%=%t=%e%p1%{14}%=%t6%e?,
	use=st52,
st52|st52-m|at|at-m|atari|atari-m|atari_st|atarist-m|Atari ST,
	am, eo, mir, npc,
	cols#80, it#8, lines#24,
	bel=^G, civis=\Ef, clear=\EE, cnorm=\Ee, cr=^M, cub1=\ED,
	cud1=\EB, cuf1=\EC, cup=\EY%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=\EA, dl1=\EM, ed=\EJ, el=\EK, el1=\Eo, home=\EH, ht=^I,
	il1=\EL, ind=^J, is2=\Ev\Eq\Ee, kLFT=\Ed, kRIT=\Ec, kbs=^H,
	kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kdch1=\177,
	kf1=\EP, kf10=\EY, kf11=\Ep, kf12=\Eq, kf13=\Er, kf14=\Es,
	kf15=\Et, kf16=\Eu, kf17=\Ev, kf18=\Ew, kf19=\Ex, kf2=\EQ,
	kf20=\Ey, kf3=\ER, kf4=\ES, kf5=\ET, kf6=\EU, kf7=\EV, kf8=\EW,
	kf9=\EX, khlp=\EH, khome=\EE, kich1=\EI, knp=\Eb, kpp=\Ea,
	kund=\EK, nel=^M^J, rc=\Ek, rev=\Ep, ri=\EI, rmso=\Eq,
	rs2=\Ev\Eq\Ee, sc=\Ej, sgr0=\Eq, smso=\Ep,
tw100|toswin vt100 window mgr,
	eo, mir, msgr, xon,
	colors#8, cols#80, it#8, lines#24, pairs#64, vt#3,
	acsc=++\,\,--..00II``aaffgghhjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\Ef,
	clear=\E[2J\E[H, cnorm=\Ee, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=\EB,
	cuf=\E[%p1%dC, cuf1=\EC, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\EA, dch1=\Ea, dim=\E[2m, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K, home=\E[H, ht=^I,
	hts=\EH, ich=\E[%p1%d@, il1=\EL, ind=^J, is2=\E<\E)0, kbs=^H,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kdch1=\177,
	kf1=\EOP, kf10=\EOY, kf11=\Ep, kf12=\Eq, kf13=\Er, kf14=\Es,
	kf15=\Et, kf16=\Eu, kf17=\Ev, kf18=\Ew, kf19=\Ex, kf2=\EOQ,
	kf20=\Ey, kf3=\EOR, kf4=\EOS, kf5=\EOT, kf6=\EOU, kf7=\EOV,
	kf8=\EOW, kf9=\EOX, khlp=\EH, khome=\E\EE, kich1=\EI,
	knp=\Eb, kpp=\E\Ea, kund=\EK, ll=\E[24H, nel=\EE,
	oc=\E[30;47m, op=\E[30;47m, rc=\E8, rev=\E[7m, ri=\EM,
	rmacs=^O, rmcup=\E[?7h, rmir=\Ei, rmkx=\E[?1l\E>,
	rmso=\E[m, rmul=\E[m,
	rs1=\E<\E[20l\E[?3;6;9l\E[r\Eq\E(B\017\E)0\E>,
	sc=\E7,
	setb=\E[4%p1%'0'%+%Pa%?%ga%'0'%=%t0%e%ga%'1'%=%t4%e%ga%'2'%=%t2%e%ga%'3'%=%t6%e%ga%'4'%=%t1%e%ga%'5'%=%t5%e%ga%'6'%=%t3%e7%;m,
	setf=\E[3%p1%'0'%+%Pa%?%ga%'0'%=%t0%e%ga%'1'%=%t4%e%ga%'2'%=%t2%e%ga%'3'%=%t6%e%ga%'4'%=%t1%e%ga%'5'%=%t5%e%ga%'6'%=%t3%e7%;m,
	sgr0=\E[m, smacs=^N, smcup=\E[?7l, smir=\Eh,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
# The entries for stv52 and stv52pc probably need a revision.
stv52|MiNT virtual console,
	am, msgr,
	cols#80, it#8, lines#30,
	bel=^G, blink=\Er, bold=\EyA, civis=\Ef, clear=\EE,
	cnorm=\E. \Ee, cr=^M, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, cvvis=\E.",
	dim=\Em, dl1=\EM, ed=\EJ, el=\EK, home=\EH, ht=^I, il1=\EL,
	ind=\n$<2*/>, kbs=^H, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kdch1=\177, kf1=\EP, kf10=\EY, kf11=\Ep, kf12=\Eq,
	kf13=\Er, kf14=\Es, kf15=\Et, kf16=\Eu, kf17=\Ev, kf18=\Ew,
	kf19=\Ex, kf2=\EQ, kf20=\Ey, kf3=\ER, kf4=\ES, kf5=\ET,
	kf6=\EU, kf7=\EV, kf8=\EW, kf9=\EX, khlp=\EH, khome=\EE,
	kich1=\EI, knp=\Eb, kpp=\Ea, kund=\EK, nel=\r\n$<2*/>,
	op=\Eb@\EcO, rev=\Ep, ri=\EI$<2*/>, rmcup=\Ev\E. \Ee\Ez_,
	rmso=\Eq, rmul=\EzH, rs1=\Ez_\Eb@\EcA, sgr0=\Ez_,
	smcup=\Ev\Ee\Ez_, smso=\Ep, smul=\EyH,
stv52pc|MiNT virtual console with PC charset,
	am, msgr,
	cols#80, it#8, lines#30,
	acsc=+\257\,\256-\^.v0\333I\374`\177a\260f\370g\361h\261j\331k\277l\332m\300n\305o\377p-q\304r-s_t+u+v+w+x\263y\363z\362{\343|\366}\234~\371,
	bel=^G, blink=\Er, bold=\EyA, civis=\Ef, clear=\EE,
	cnorm=\E. \Ee, cr=^M, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, cvvis=\E.",
	dim=\Em, dl1=\EM, ed=\EJ, el=\EK, home=\EH, ht=^I, il1=\EL,
	ind=\n$<2*/>, kbs=^H, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kdch1=\177, kf1=\EP, kf10=\EY, kf11=\Ep, kf12=\Eq,
	kf13=\Er, kf14=\Es, kf15=\Et, kf16=\Eu, kf17=\Ev, kf18=\Ew,
	kf19=\Ex, kf2=\EQ, kf20=\Ey, kf3=\ER, kf4=\ES, kf5=\ET,
	kf6=\EU, kf7=\EV, kf8=\EW, kf9=\EX, khlp=\EH, khome=\EE,
	kich1=\EI, knp=\Eb, kpp=\Ea, kund=\EK, nel=\r\n$<2*/>,
	rev=\Ep, ri=\EI$<2*/>, rmcup=\Ev\E. \Ee\Ez_, rmso=\Eq,
	rmul=\EzH, rs1=\Ez_\Eb@\EcA, sgr0=\Ez_, smcup=\Ev\Ee\Ez_,
	smso=\Ep, smul=\EyH,

# From: Simson L. Garfinkel <simsong@media-lab.mit.edu>
atari-old|atari st,
	OTbs, am,
	cols#80, it#8, lines#25,
	clear=\EH\EJ, cub1=\ED, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dl1=\EM,
	ed=\EJ, el=\EK, ht=^I, il1=\EL, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, ri=\EI, rmso=\Eq, sgr0=\Eq, smso=\Ep,
# UniTerm terminal program for the Atari ST:  49-line VT220 emulation mode
# From: Paul M. Aoki <aoki@ucbvax.berkeley.edu>
uniterm|uniterm49|UniTerm VT220 emulator with 49 lines,
	lines#49,
	is2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h\E[1;49r\E[49;1H,
	use=vt220,
# MiNT VT52 emulation. 80 columns, 25 rows.
# MiNT is Now TOS, the operating system which comes with all Ataris now
# (mainly Atari Falcon). This termcap is for the VT52 emulation you get
# under tcsh/zsh/bash/sh/ksh/ash/csh when you run MiNT in `console' mode
# From: Per Persson <pp@gnu.ai.mit.edu>, 27 Feb 1996
st52-old|Atari ST with VT52 emulation,
	am, km,
	cols#80, lines#25,
	bel=^G, civis=\Ef, clear=\EH\EJ, cnorm=\Ee, cr=^M, cub1=\ED,
	cud1=\EB, cuf1=\EC, cup=\EY%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=\EA, dl1=\EM, ed=\EJ, el=\EK, home=\EH, ht=^I, il1=\EL,
	ind=^J, ka1=\E#7, ka3=\E#5, kb2=\E#9, kbs=^H, kc1=\E#1,
	kc3=\E#3, kclr=\E#7, kcub1=\E#K, kcud1=\E#P, kcuf1=\E#M,
	kcuu1=\E#H, kf0=\E#D, kf1=\E#;, kf2=\E#<, kf3=\E#=, kf4=\E#>,
	kf5=\E#?, kf6=\E#@, kf7=\E#A, kf8=\E#B, kf9=\E#C, khome=\E#G,
	kil1=\E#R, kind=\E#2, kri=\E#8, lf0=f10, nel=^M^J, rc=\Ek,
	ri=\EI, rmcup=, rmso=\Eq, rs1=\Ez_\Eb@\EcA, sc=\Ej, sgr0=\Eq,
	smcup=\Ee, smso=\Ep,

#### Apple Terminal.app

# nsterm*|Apple_Terminal - AppKit Terminal.app
#
# Terminal.app is a Terminal emulator bundled with NeXT's NeXTStep and
# OPENSTEP/Mach operating systems, and with Apple's Rhapsody, Mac OS X
# Server and Mac OS X operating systems. There is also a
# "terminal.app" in GNUStep, but I believe it to be an unrelated
# codebase and I have not attempted to describe it here.
#
# For NeXTStep, OPENSTEP/Mach, Rhapsody and Mac OS X Server 1.0, you
# are pretty much on your own. Use "nsterm-7-m" and hope for the best.
# You might also try "nsterm-7" and "nsterm-old" if you suspect your
# version supports color.
#
# To determine the version of Terminal.app you're using by running:
#
#     echo "$TERM_PROGRAM" "$TERM_PROGRAM_VERSION"
#
# For Apple_Terminal v309+, use "nsterm-256color" (or "nsterm-bce")
#
# For Apple_Terminal v200+, use "nsterm-16color" (a.k.a. "nsterm")
#
# For Apple_Terminal v71+/v100+, use "nsterm-bce".
#
# For Apple_Terminal v51+, use "nsterm-7-c" or "nsterm-7-c-s".
#
# For Apple_Terminal v41+, use "nsterm-old", or "nsterm-s".
#
# For all earlier versions (Apple_Terminal), try "nsterm-7-m"
# (monochrome) or "nsterm-7" (color); "nsterm-7-m-s" and "nsterm-7-s"
# might work too, but really you're on your own here since these
# systems are very obsolete and I can't test them. I do welcome
# patches, though :).

# Other Terminals:
#
# For GNUstep_Terminal, you're probably best off using "linux" or
# writing your own terminfo.

# For MacTelnet, you're on your own. It's a different codebase, and
# seems to be somewhere between "vt102", "ncsa" and "xterm-color".

# For iTerm.app, see "iterm".

#
# The AppKit Terminal.app descriptions all have names beginning with
# "nsterm". Note that the statusline (-s) versions use the window
# titlebar as a phony status line, and may produce warnings during
# compilation as a result ("tsl uses 0 parameters, expected 1".)
# Ignore these warnings, or even ignore these entries entirely. Apps
# which need to position the cursor or do other fancy stuff inside the
# status line won't work with these entries. They're primarily useful
# for programs like Pine which provide simple notifications in the
# status line. Please note that non-ASCII characters don't work right
# in the status line, since Terminal.app incorrectly interprets their
# Unicode codepoints as MacRoman codepoints (in earlier Mac OS X
# versions) or only accepts status lines consisting entirely of
# characters from the first 256 Unicode positions (including C1 but
# not C0 or DEL.)
#
# The Mythology* of AppKit Terminal.app:
#
# In the days of NeXTSTep 0.x and 1.x there were two incompatible
# bundled terminal emulators, Shell and Terminal. Scott Hess wrote a
# shareware replacement for Terminal called "Stuart" which NeXT bought
# and used as the basis for the Terminal.app in NeXTstep 2+,
# OPENSTEP/Mach, Apple Rhapsody, Mac OS X Server 1.0, and Mac OS X. I
# don't know the TERM_PROGRAM and TERM_PROGRAM_VERSION settings or
# capabilities for the early versions, but I believe that the
# TERM_PROGRAM_VERSION may have been reset at some point.
#
# The early versions were tailored to the NeXT character set. Sometime
# after the Apple aquisition the encoding was swiched to MacRoman
# (initally with serious altcharset bugs due to incomplete conversion
# of the old NeXT code,) and then later to UTF-8. Alos sometime during
# or just prior to the early days of Mac OS X, the Terminal grew ANSI
# 8-color support (initially buggy when combined with attributes, but
# that was later fixed.) More recently, around Mac OS X version 10.3
# or so (Terminal.app v100+) xterm-like 16-color support was added. In
# some versions (for instance 133-1 which shipped with Mac OS X
# version 10.4) this suffered from the <bce> bug, but that seems to
# have been fixed in Mac OS X version 10.5 (Terminal.app v240.2+).
#
# In the early days of Mac OS X the terminal was fairly buggy and
# would routinely crash under load. Many of these bugs seem to have
# been fixed around Mac OS X version 10.3 (Terminal.app v100+) but
# some may still remain. This change seems to correspond to
# Terminal.app reporting "xterm-color" as $TERM rather than "vt100" as
# it did previously.
#
# * This may correspond with what actually happened, but I don't
#   know. It is based on guesswork, hearsay, private correspondence,
#   my faulty memory, and the following online sources and references:
#
# [1] "Three Scotts and a Duane" by Simson L. Garfinkel
# http://www.nextcomputers.org/NeXTfiles/Articles/NeXTWORLD/93.8/93.8.Dec.Community1.html
#
# [2] NeXTSTEP entry from Wikipedia, the free encyclopedia
# https://secure.wikimedia.org/wikipedia/en/wiki/Nextstep
#
# * Renamed the AppKit Terminal.app entry from "Apple_Terminal" to
#   "nsterm" to comply with the name length and case conventions and
#   limitations of various software packages [notably Solaris terminfo
#   and UNIX.] A single Apple_Terminal alias is retained for
#   backwards-compatbility.
#
# * Added function key support (F1-F4). These only work in Terminal.app
#   version 51, hopefully the capabilities won't cause problems for people
#   using version 41.
#
# * Added "full color" (-c) entries which support the 16-color mode in
#   version 51.
#
# * By default, version 51 uses UTF-8 encoding with broken altcharset
#   support, so "ASCII" (-7) entries without altcharset support were
#   added.

# nsterm - AppKit Terminal.app
#
# Apple's Mac OS X includes a Terminal.app derived from the old NeXT
# Terminal.app. It is a partial VT100 emulation with some xterm-like
# extensions. This terminfo was written to describe versions 41
# (shipped with Mac OS X version 10.0) and 51 (shipped with Mac OS X
# version 10.1) of Terminal.app.
#
# Terminal.app runs under the Mac OS X Quartz windowing system (and
# other AppKit-supported windowing systems.)  On the Mac OS X machine I
# use, the executable for Terminal.app is:
# /Applications/Utilities/Terminal.app/Contents/MacOS/Terminal
#
# If you're looking for a description of the full-screen system
# console which runs under Apple's Darwin operating system on PowerPC
# platforms, see the "xnuppc" entry instead.
#
# There were no function keys in version 41. In version 51, there are
# four working function keys (F1, F2, F3 and F4.) The function keys
# are included in all of these entries.
#
# It does not support mouse pointer position reporting. Under some
# circumstances the cursor can be positioned using option-click; this
# works by comparing the cursor position and the selected position,
# and simulating enough cursor-key presses to move the cursor to the
# selected position. This technique fails in all but the simplest
# applications.
#
# It provides partial ANSI color support (background colors interacted
# badly with bold in version 41, though, as reflected in :ncv:.) The
# monochrome (-m) entries are useful if you've disabled color support
# or use a monochrome monitor. The full color (-c) entries are useful
# in version 51, which doesn't exhibit the background color bug. They
# also enable an xterm-compatible 16-color mode.
#
# The configurable titlebar is set using xterm-compatible sequences;
# it is used as a status bar in the statusline (-s) entries. Its width
# depends on font sizes and window sizes, but 50 characters seems to
# be the default for an 80x24 window.
#
# The MacRoman character encoding is used for some of the alternate
# characters in the "MacRoman" entries; the "ASCII" (-7) entries
# disable alternate character set support entirely, and the "VT100"
# (-acs) entries rely instead on Terminal.app's own buggy VT100
# graphics emulation, which seems to think the character encoding is
# the old NeXT charset instead of MacRoman. The "ASCII" (-7) entries
# are useful in Terminal.app version 51, which supports UTF-8 and
# other ASCII-compatible character encodings but does not correctly
# implement VT100 graphics; once VT100 graphics are correctly
# implemented in Terminal.app, the "VT100" (-acs) entries should be
# usable in any ASCII-compatible character encoding [except perhaps
# in UTF-8, where some experts argue for disallowing alternate
# characters entirely.]
#
# Terminal.app reports "vt100" as the terminal type, but exports
# several environment variables which may aid detection in a shell
# profile (i.e. .profile or .login):
#
# TERM=vt100
# TERM_PROGRAM=Apple_Terminal
# TERM_PROGRAM_VERSION=41      # in Terminal.app version 41
# TERM_PROGRAM_VERSION=51      # in Terminal.app version 51
#
# For example, the following Bourne shell script would detect the
# correct terminal type:
#
# if [ :"$TERM" = :"vt100" -a :"$TERM_PROGRAM" = :"Apple_Terminal" ]
# then
#     export TERM
#     if [ :"$TERM_PROGRAM_VERSION" = :41 ]
#     then
#         TERM="nsterm-old"
#     else
#         TERM="nsterm-c-7"
#     fi
# fi
#
# In a C shell derivative, this would be accomplished by:
#
# if ( $?TERM && $?TERM_PROGRAM && $?TERM_PROGRAM_VERSION) then
#     if ( :"$TERM" == :"vt100" && :"$TERM_PROGRAM" == :"Apple_Terminal" ) then
#          if ( :"$TERM_PROGRAM_VERSION" == :41 ) then
#              setenv TERM "nsterm-old"
#          else
#              setenv TERM "nsterm-c-7"
#          endif
#     endif
# endif

# The '+' entries are building blocks
nsterm+7|AppKit Terminal.app v41+ basic capabilities w/ASCII charset,
	am, bw, msgr, xenl, xon,
	cols#80, it#8, lines#24,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K,
	home=\E[H, ht=^I, hts=\EH, il=\E[%p1%dL, il1=\E[L, ind=^J,
	invis=\E[8m, kbs=\177, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC,
	kcuu1=\EOA, kent=\EOM, rc=\E8, rev=\E[7m, ri=\EM,
	rmam=\E[?7l, rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p7%t;8%;m,
	sgr0=\E[m, smam=\E[?7h, smkx=\E[?1h\E=, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g, use=vt100+enq, use=vt100+pfkeys,

nsterm+acs|AppKit Terminal.app v41+ basic capabilities w/VT100 alternate-charset,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	enacs=\E(B\E)0, rmacs=^O,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, use=nsterm+7,

nsterm+mac|AppKit Terminal.app v41+ basic capabilities w/MacRoman alternate-charset,
	acsc=+\335\,\334-\366.\3770#`\327a\:f\241g\261h#i\360jjkkllmmnno\370p\370q\321rrssttuuvvwwxxy\262z\263{\271|\255}\243~\245,
	enacs=\E(B\E)0, rmacs=^O,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, use=nsterm+7,

# compare with xterm+sl-twm
nsterm+s|AppKit Terminal.app v41+ status-line (window titlebar) support,
	hs,
	wsl#50,
	dsl=\E]2;\007, fsl=^G, tsl=\E]2;,

nsterm+c|AppKit Terminal.app v51+ full color support (including 16 colors),
	op=\E[0m, use=ibm+16color,

nsterm+c41|AppKit Terminal.app v41 color support,
	colors#8, ncv#37, pairs#64,
	op=\E[0m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,

# These are different combinations of the building blocks

# ASCII charset (-7)
nsterm-m-7|nsterm-7-m|AppKit Terminal.app v41+ w/ASCII charset (monochrome),
	use=nsterm+7,

nsterm-m-s-7|nsterm-7-m-s|AppKit Terminal.app v41+ w/ASCII charset (monochrome w/statusline),
	use=nsterm+s, use=nsterm+7,

nsterm-7|AppKit Terminal.app v41+ w/ASCII charset (color),
	use=nsterm+c41, use=nsterm+7,

nsterm-7-c|nsterm-c-7|AppKit Terminal.app v51+ w/ASCII charset (full color),
	use=nsterm+c, use=nsterm+7,

nsterm-s-7|nsterm-7-s|AppKit Terminal.app v41+ w/ASCII charset (color w/statusline),
	use=nsterm+s, use=nsterm+c41, use=nsterm+7,

nsterm-c-s-7|nsterm-7-c-s|AppKit Terminal.app v51+ w/ASCII charset (full color w/statusline),
	use=nsterm+s, use=nsterm+c, use=nsterm+7,

# VT100 alternate-charset (-acs)
nsterm-m-acs|nsterm-acs-m|AppKit Terminal.app v41+ w/VT100 alternate-charset (monochrome),
	use=nsterm+acs,

nsterm-m-s-acs|nsterm-acs-m-s|AppKit Terminal.app v41+ w/VT100 alternate-charset (monochrome w/statusline),
	use=nsterm+s, use=nsterm+acs,

nsterm-acs|AppKit Terminal.app v41+ w/VT100 alternate-charset (color),
	use=nsterm+c41, use=nsterm+acs,

nsterm-c-acs|nsterm-acs-c|AppKit Terminal.app v51+ w/VT100 alternate-charset (full color),
	use=nsterm+c, use=nsterm+acs,

nsterm-s-acs|nsterm-acs-s|AppKit Terminal.app v41+ w/VT100 alternate-charset (color w/statusline),
	use=nsterm+s, use=nsterm+c41, use=nsterm+acs,

nsterm-c-s-acs|nsterm-acs-c-s|AppKit Terminal.app v51+ w/VT100 alternate-charset (full color w/statusline),
	use=nsterm+s, use=nsterm+c, use=nsterm+acs,

# MacRoman charset
nsterm-m|AppKit Terminal.app v41+ w/MacRoman charset (monochrome),
	use=nsterm+mac,

nsterm-m-s|AppKit Terminal.app v41+ w/MacRoman charset (monochrome w/statusline),
	use=nsterm+s, use=nsterm+mac,

nsterm-old|AppKit Terminal.app v41+ w/MacRoman charset (color),
	use=nsterm+c41, use=nsterm+mac,

nsterm-c|AppKit Terminal.app v51+ w/MacRoman charset (full color),
	use=nsterm+c, use=nsterm+mac,

nsterm-s|AppKit Terminal.app v41+ w/MacRoman charset (color w/statusline),
	use=nsterm+s, use=nsterm+c41, use=nsterm+mac,

nsterm-c-s|AppKit Terminal.app v51+ w/MacRoman charset (full color w/statusline),
	use=nsterm+s, use=nsterm+c, use=nsterm+mac,

# In Mac OS X version 10.5 the format of the preferences has changed
# and a new, more complex technique is needed, e.g.,
#
#       python -c 'import sys,objc;NSUserDefaults=objc.lookUpClass(
#       "NSUserDefaults");ud=NSUserDefaults.alloc();
#       ud.init();prefs=ud.persistentDomainForName_(
#       "com.apple.Terminal");prefs["Window Settings"][
#       prefs["Default Window Settings"]]["TerminalType"
#       ]=sys.argv[1];ud.setPersistentDomain_forName_(prefs,
#       "com.apple.Terminal")' nsterm-16color
#
# and it is still not settable from the preferences dialog. This is
# tracked under rdar://problem/7365108 and rdar://problem/7365134
# in Apple's bug reporter.
#
# In OS X 10.7 (Leopard) the TERM which can be set in the preferences dialog
# defaults to xterm-color.  Alternative selections are ansi, dtterm, rxvt,
# vt52, vt100, vt102 and xterm.
nsterm-16color|AppKit Terminal.app v240.2+ with Mac OS X version 10.5,
	bw@, mir, npc,
	civis=\E[?25l, cnorm=\E[?25h, dch=\E[%p1%dP, dch1=\E[P,
	flash=\E[?5h$<200/>\E[?5l, hpa=\E[%i%p1%dG,
	ich=\E[%p1%d@, ich1=\E[@, kdch1=\E[3~, kend=\E[F,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf13=\E[25~,
	kf14=\E[26~, kf15=\E[28~, kf16=\E[29~, kf17=\E[31~,
	kf18=\E[22~, kf19=\E[33~, kf20=\E[34~, kf5=\E[15~,
	kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, khome=\E[H,
	knp=\E[6~, kpp=\E[5~, rmcup=\E[2J\E[?47l\E8, rmir=\E[4l,
	smcup=\E7\E[?47h, smir=\E[4h, vpa=\E[%i%p1%dd,
	kLFT5=\E[5D, kRIT5=\E[5C, use=nsterm-c-s-acs,

# The versions of Terminal.app in Mac OS X version 10.3.x seem to have
# the background color erase feature. The newer version 240.2 in Mac OS X
# version 10.5 does not.
#
# This entry is based on newsgroup comments by Alain Bench, Christian Ebert,
# and D P Schreber comparing to nsterm-c-s-acs.
#
# In Mac OS X version 10.4 and earlier, D P Schreber notes that $TERM
# can be set in Terminal.app, e.g.,
#
#	defaults write com.apple.Terminal TermCapString nsterm-bce
#
# and that it is not set in Terminal's preferences dialog.
#
# Modified for OS X 10.8, omitting bw based on testing with tack -TD
#
# Notes:
# * The terminal description matches the default settings.
# * The keyboard is configurable via a dialog.
# * By default khome, kend, knext and kprev are honored only with a
#   shift-modifier.
# * There are bindings for control left/right arrow (but not up/down).
#   Added those to nsterm-16color, which is the version used for OS X 10.6
# * "Allow VT100 application keypage mode" is by default disabled.
#   There is no way to press keypad-comma unless application mode is enabled
#   and used.
# * 132-column mode stopped working during vttest's tests.  Consider it broken.
# * CHT, REP, SU, SD are buggy.
# * ECH works (also in Leopard), but is not used here for compatibility.
# * The terminal preferences dialog replaces xterm-color by xterm-16color and
#   xterm-256color.  However, it adds "nsterm", so it is possible to use the
#   nsterm entry from this file to override the MacPorts (20110404) or
#   system (20081102) copy of this file.
# + In OS X 10.8 (Mountain Lion) the TERM which can be set in the preferences
#   dialog defaults to xterm-256color.  Alternative selections are ansi,
#   dtterm, rxvt, vt52, vt100, vt102, xterm and xterm-16color.  However,
#   the menu says "Declare terminal as" without promising to actually emulate
#   the corresponding terminals.  Indeed, changing TERM does not affect the
#   emulation itself.  This means that
#   + the function-keys do not match for dtterm for kf1-kf4 as well as
#     khome/kend
#   + the color model is the same for each setting of TERM (does not match
#     ansi or dtterm).
#   + the shift/control/meta key modifiers from rxvt and xterm variants are not
#     recognised except for a few special cases, i.e., kRIT5 and kLFT5.
#   + the vt52 emulation does not give a usable shell because screen-clearing
#     does not work as expected.
#   + selecting "xterm" or "xterm-16color" sets TERM to "xterm-256color".
nsterm-bce|AppKit Terminal.app v71+/v100.1.8+ with Mac OS X version 10.3/10.4 (bce),
	bce, use=nsterm-16color,

# This is tested with OS X 10.8 (Mountain Lion), 2012/08/11
#	TERM_PROGRAM_VERSION=309
# Earlier reports state that these differences also apply to OS X 10.7 (Lion),
#	TERM_PROGRAM_VERSION=303
nsterm-256color|Terminal.app in OS X 10.8,
	ccc@,
	initc@, use=xterm+256color, use=nsterm-bce,

# This is an alias which should always point to the "current" version
nsterm|Apple_Terminal|AppKit Terminal.app,
	use=nsterm-256color,

# iTerm.app from http://iterm.sourceforge.net/ is an alternative (and
# more featureful) terminal emulator for Mac OS X. It is similar
# enough in capabilities to nsterm-16color that I have derived this
# description from that one, but as far as I know they share no code.
# Many of the features are user-configurable, but I attempt only to
# describe the default configuration.
#
# NOTE: When tack tests (csr) + (nel) iTerm.app crashes, so (csr) is
# disabled.
iTerm.app|iTerm.app terminal emulator for Mac OS X,
	bce, bw@, ccc@,
	csr@, initc@, kend=\EOF, khome=\EOH, use=xterm+256color,
	use=nsterm-16color,

# xnuppc - Darwin PowerPC Console (a.k.a. "darwin")
#
# On PowerPC platforms, Apple's Darwin operating system uses a
# full-screen system console derived from a NetBSD framebuffer
# console. It is an ANSI-style terminal, and is not really VT-100
# compatible.
#
# Under Mac OS X, this is the system console driver used while in
# single-user mode [reachable by holding down Command-S during the
# boot process] and when logged in using console mode [reachable by
# typing ">console" at the graphical login prompt.]
#
# If you're looking for a description of the Terminal.app terminal
# emulator which runs under the Mac OS X Quartz windowing system (and
# other AppKit-supported windowing systems,) see the "nsterm"
# entry instead.
#
# NOTE: Under Mac OS X version 10.1, the default login window does not
# prompt for user name, instead requiring an icon to be selected from
# a list of known users. Since the special ">console" login is not in
# this list, you must make one of two changes in the Login Window
# panel of the Login section of System Prefs to make the special
# ">console" login accessible. The first option is to enable 'Show
# "Other User" in list for network users', which will add a special
# "Other..." icon to the graphical login panel. Selecting "Other..."
# will present the regular graphical login prompt. The second option
# is to change the 'Display Login Window as:' setting to 'Name and
# password entry fields', which replaces the login panel with a
# graphical login prompt.
#
# There are no function keys, at least not in Darwin 1.3.
#
# It has no mouse support.
#
# It has full ANSI color support, and color combines correctly with
# all three supported attributes: bold, inverse-video and underline.
# However, bold colored text is almost unreadable (bolding is
# accomplished using shifting and or-ing, and looks smeared) so bold
# has been excluded from the list of color-compatible attributes
# [using (ncv)]. The monochrome entry (-m) is useful if you use a
# monochrome monitor.
#
# There is one serious bug with this terminal emulation's color
# support: repositioning the cursor onto a cell with non-matching
# colors obliterates that cell's contents, replacing it with a blank
# and displaying a colored cursor in the "current" colors. There is
# no complete workaround at present [other than using the monochrome
# (-m) entries,] but removing the (msgr) capability seemed to help.
#
# The "standout" chosen was simple reverse-video, although a colorful
# standout might be more aesthetically pleasing. Similarly, the bold
# chosen is the terminal's own smeared bold, although a simple
# color-change might be more readable. The color-bold (-b) entries
# uses magenta colored text for bolding instead. The fancy color (-f
# and -f2) entries use color for bold, standout and underlined text
# (underlined text is still underlined, though.)
#
# Apparently the terminal emulator does support a VT-100-style
# alternate character set, but all the alternate character set
# positions have been left blank in the font. For this reason, no
# alternate character set capabilities have been included in this
# description. The console driver appears to be ASCII-only, so (enacs)
# has been excluded [although the VT-100 sequence does work.]
#
# The default Mac OS X and Darwin installation reports "vt100" as the
# terminal type, and exports no helpful environment variables. To fix
# this, change the "console" entry in /etc/ttys from "vt100" to
# "xnuppc-WxH", where W and H are the character dimensions of your
# console (see below.)
#
# The font used by the terminal emulator is apparently one originally
# drawn by Ka-Ping Yee, and uses 8x16-pixel characters. This
# file includes descriptions for the following geometries:
#
#     Pixels        Characters   Entry Name (append -m for monochrome)
#    -------------------------------------------------------------------
#     640x400       80x25        xnuppc-80x25
#     640x480       80x30        xnuppc-80x30
#     720x480       90x30        xnuppc-90x30
#     800x600       100x37       xnuppc-100x37
#     896x600       112x37       xnuppc-112x37
#     1024x640      128x40       xnuppc-128x40
#     1024x768      128x48       xnuppc-128x48
#     1152x768      144x48       xnuppc-144x48
#     1280x1024     160x64       xnuppc-160x64
#     1600x1024     200x64       xnuppc-200x64
#     1600x1200     200x75       xnuppc-200x75
#     2048x1536     256x96       xnuppc-256x96
#
# The basic "xnuppc" entry includes no size information, and the
# emulator includes no reporting capability, so you'll be at the mercy
# of the TTY device (which reports incorrectly on my hardware.) The
# color-bold entries do not include size information.

# The '+' entries are building blocks
xnuppc+basic|Darwin PowerPC Console basic capabilities,
	am, bce, mir, xenl,
	it#8,
	bold=\E[1m, clear=\E[H\E[J, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=\E[D, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dsl=\E]2;\007, ed=\E[J, el=\E[K,
	el1=\E[1K, home=\E[H, ht=^I, hts=\EH, ind=^J, kbs=\177,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, rc=\E8,
	rev=\E[7m, ri=\EM, rmam=\E[?7l, rmkx=\E[?1l\E>, rmso=\E[m,
	rmul=\E[m, rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h,
	sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;m,
	sgr0=\E[m, smam=\E[?7h, smkx=\E[?1h\E=, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g, use=vt100+keypad,

xnuppc+c|Darwin PowerPC Console ANSI color support,
	colors#8, ncv#32, pairs#64,
	op=\E[37;40m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,

xnuppc+b|Darwin PowerPC Console color-bold support,
	ncv#32,
	bold=\E[35m,
	sgr=\E[0%?%p6%t;35%;%?%p2%t;4%;%?%p1%p3%|%t;7%;m,
	use=xnuppc+basic,

xnuppc+f|Darwin PowerPC Console fancy color support,
	ncv#35,
	sgr=\E[0%?%p6%t;35%;%?%p2%t;36;4%;%?%p1%t;33;44%;%?%p3%t;7%;m,
	smso=\E[33;44m, smul=\E[36;4m, use=xnuppc+b,

xnuppc+f2|Darwin PowerPC Console alternate fancy color support,
	ncv#35,
	bold=\E[33m,
	sgr=\E[0%?%p6%t;33%;%?%p2%t;34%;%?%p1%t;31;47%;%?%p3%t;7%;m,
	smso=\E[31;47m, smul=\E[34m, use=xnuppc+basic,

# Building blocks for specific screen sizes
xnuppc+80x25|Darwin PowerPC Console 80x25 support (640x400 pixels),
	cols#80, lines#25,

xnuppc+80x30|Darwin PowerPC Console 80x30 support (640x480 pixels),
	cols#80, lines#30,

xnuppc+90x30|Darwin PowerPC Console 90x30 support (720x480 pixels),
	cols#90, lines#30,

xnuppc+100x37|Darwin PowerPC Console 100x37 support (800x600 pixels),
	cols#100, lines#37,

xnuppc+112x37|Darwin PowerPC Console 112x37 support (896x600 pixels),
	cols#112, lines#37,

xnuppc+128x40|Darwin PowerPC Console 128x40 support (1024x640 pixels),
	cols#128, lines#40,

xnuppc+128x48|Darwin PowerPC Console 128x48 support (1024x768 pixels),
	cols#128, lines#48,

xnuppc+144x48|Darwin PowerPC Console 144x48 support (1152x768 pixels),
	cols#144, lines#48,

xnuppc+160x64|Darwin PowerPC Console 160x64 support (1280x1024 pixels),
	cols#160, lines#64,

xnuppc+200x64|Darwin PowerPC Console 200x64 support (1600x1024 pixels),
	cols#200, lines#64,

xnuppc+200x75|Darwin PowerPC Console 200x75 support (1600x1200 pixels),
	cols#200, lines#75,

xnuppc+256x96|Darwin PowerPC Console 256x96 support (2048x1536 pixels),
	cols#256, lines#96,

# These are different combinations of the building blocks

xnuppc-m|darwin-m|Darwin PowerPC Console (monochrome),
	use=xnuppc+basic,

xnuppc|darwin|Darwin PowerPC Console (color),
	use=xnuppc+c, use=xnuppc+basic,

xnuppc-m-b|darwin-m-b|Darwin PowerPC Console (monochrome w/color-bold),
	use=xnuppc+b,

xnuppc-b|darwin-b|Darwin PowerPC Console (color w/color-bold),
	use=xnuppc+b, use=xnuppc+c,

xnuppc-m-f|darwin-m-f|Darwin PowerPC Console (fancy monochrome),
	use=xnuppc+f,

xnuppc-f|darwin-f|Darwin PowerPC Console (fancy color),
	use=xnuppc+f, use=xnuppc+c,

xnuppc-m-f2|darwin-m-f2|Darwin PowerPC Console (alternate fancy monochrome),
	use=xnuppc+f2,

xnuppc-f2|darwin-f2|Darwin PowerPC Console (alternate fancy color),
	use=xnuppc+f2, use=xnuppc+c,

# Combinations for specific screen sizes
xnuppc-80x25-m|darwin-80x25-m|Darwin PowerPC Console (monochrome) 80x25,
	use=xnuppc+80x25, use=xnuppc+basic,

xnuppc-80x25|darwin-80x25|Darwin PowerPC Console (color) 80x25,
	use=xnuppc+c, use=xnuppc+80x25, use=xnuppc+basic,

xnuppc-80x30-m|darwin-80x30-m|Darwin PowerPC Console (monochrome) 80x30,
	use=xnuppc+80x30, use=xnuppc+basic,

xnuppc-80x30|darwin-80x30|Darwin PowerPC Console (color) 80x30,
	use=xnuppc+c, use=xnuppc+80x30, use=xnuppc+basic,

xnuppc-90x30-m|darwin-90x30-m|Darwin PowerPC Console (monochrome) 90x30,
	use=xnuppc+90x30, use=xnuppc+basic,

xnuppc-90x30|darwin-90x30|Darwin PowerPC Console (color) 90x30,
	use=xnuppc+c, use=xnuppc+90x30, use=xnuppc+basic,

xnuppc-100x37-m|darwin-100x37-m|Darwin PowerPC Console (monochrome) 100x37,
	use=xnuppc+100x37, use=xnuppc+basic,

xnuppc-100x37|darwin-100x37|Darwin PowerPC Console (color) 100x37,
	use=xnuppc+c, use=xnuppc+100x37, use=xnuppc+basic,

xnuppc-112x37-m|darwin-112x37-m|Darwin PowerPC Console (monochrome) 112x37,
	use=xnuppc+112x37, use=xnuppc+basic,

xnuppc-112x37|darwin-112x37|Darwin PowerPC Console (color) 112x37,
	use=xnuppc+c, use=xnuppc+112x37, use=xnuppc+basic,

xnuppc-128x40-m|darwin-128x40-m|Darwin PowerPC Console (monochrome) 128x40,
	use=xnuppc+128x40, use=xnuppc+basic,

xnuppc-128x40|darwin-128x40|Darwin PowerPC Console (color) 128x40,
	use=xnuppc+c, use=xnuppc+128x40, use=xnuppc+basic,

xnuppc-128x48-m|darwin-128x48-m|Darwin PowerPC Console (monochrome) 128x48,
	use=xnuppc+128x48, use=xnuppc+basic,

xnuppc-128x48|darwin-128x48|Darwin PowerPC Console (color) 128x48,
	use=xnuppc+c, use=xnuppc+128x48, use=xnuppc+basic,

xnuppc-144x48-m|darwin-144x48-m|Darwin PowerPC Console (monochrome) 144x48,
	use=xnuppc+144x48, use=xnuppc+basic,

xnuppc-144x48|darwin-144x48|Darwin PowerPC Console (color) 144x48,
	use=xnuppc+c, use=xnuppc+144x48, use=xnuppc+basic,

xnuppc-160x64-m|darwin-160x64-m|Darwin PowerPC Console (monochrome) 160x64,
	use=xnuppc+160x64, use=xnuppc+basic,

xnuppc-160x64|darwin-160x64|Darwin PowerPC Console (color) 160x64,
	use=xnuppc+c, use=xnuppc+160x64, use=xnuppc+basic,

xnuppc-200x64-m|darwin-200x64-m|Darwin PowerPC Console (monochrome) 200x64,
	use=xnuppc+200x64, use=xnuppc+basic,

xnuppc-200x64|darwin-200x64|Darwin PowerPC Console (color) 200x64,
	use=xnuppc+c, use=xnuppc+200x64, use=xnuppc+basic,

xnuppc-200x75-m|darwin-200x75-m|Darwin PowerPC Console (monochrome) 200x75,
	use=xnuppc+200x75, use=xnuppc+basic,

xnuppc-200x75|darwin-200x75|Darwin PowerPC Console (color) 200x75,
	use=xnuppc+c, use=xnuppc+200x75, use=xnuppc+basic,

xnuppc-256x96-m|darwin-256x96-m|Darwin PowerPC Console (monochrome) 256x96,
	use=xnuppc+256x96, use=xnuppc+basic,

xnuppc-256x96|darwin-256x96|Darwin PowerPC Console (color) 256x96,
	use=xnuppc+c, use=xnuppc+256x96, use=xnuppc+basic,


#### BeOS
#
# BeOS entry for Terminal program Seems to be almost ANSI
beterm|BeOS Terminal,
	am, eo, mir, msgr, xenl, xon,
	colors#8, cols#80, it#8, lines#25, ncv#5, pairs#64,
	bel=^G, bold=\E[1m, clear=\E[H\E[J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K, home=\E[H,
	hpa=\E[%i%p1%dG, ht=^I, hts=\EH, ich=\E[%p1%d@, ich1=\E[@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, kbs=^H, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~,
	kend=\E[4~, kf1=\E[11~, kf10=\E[20~, kf11=\E[21~,
	kf12=\E[22~, kf2=\E[12~, kf3=\E[13~, kf4=\E[14~,
	kf5=\E[15~, kf6=\E[16~, kf7=\E[17~, kf8=\E[18~, kf9=\E[19~,
	khome=\E[1~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~, kspd=^Z,
	nel=^M^J, op=\E[m, rc=\E8, rev=\E[7m, ri=\EM, rmir=\E[4l,
	rmkx=\E[?4l, rmso=\E[m, rmul=\E[24m, rs1=\Ec, sc=\E7,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	setb=\E[%p1%{40}%+%cm, setf=\E[%p1%{30}%+%cm,
	sgr0=\E[0;10m, smir=\E[4h, smkx=\E[?4h, smso=\E[7m,
	smul=\E[4m, u6=\E[%i%p1%d;%p2%dR, u7=\E[6n,
	vpa=\E[%i%p1%dd,

#### Linux consoles
#

# This entry is good for the 1.2.13 or later version of the Linux console.
#
# ***************************************************************************
# *                                                                         *
# *                           WARNING:                                      *
# * Linuxes come with a default keyboard mapping kcbt=^I.  This entry, in   *
# * response to user requests, assumes kcbt=\E[Z, the ANSI/ECMA reverse-tab *
# * character. Here are the keymap replacement lines that will set this up: *
# *                                                                         *
#	keycode  15 = Tab             Tab
#		alt     keycode  15 = Meta_Tab
#		shift	keycode  15 = F26
#	string F26 ="\033[Z"
# *                                                                         *
# * This has to use a key slot which is unfortunate (any unused one will    *
# * do, F26 is the higher-numbered one).  The change ought to be built      *
# * into the kernel tables.                                                 *
# *                                                                         *
# ***************************************************************************
#
# All linux kernels since 1.2.13 (at least) set the screen size
# themselves; this entry assumes that capability.
#
linux-basic|linux console,
	am, bce, eo, mir, msgr, xenl, xon,
	it#8, ncv#18, U8#1,
	acsc=+\020\,\021-\030.^Y0\333`\004a\261f\370g\361h\260i\316j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	bel=^G, clear=\E[H\E[J, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dim=\E[2m, dl=\E[%p1%dM, dl1=\E[M, ech=\E[%p1%dX, ed=\E[J,
	el=\E[K, el1=\E[1K, flash=\E[?5h\E[?5l$<200/>, home=\E[H,
	hpa=\E[%i%p1%dG, ht=^I, hts=\EH, ich=\E[%p1%d@, ich1=\E[@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, kb2=\E[G, kbs=\177,
	kcbt=\E[Z, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[3~, kend=\E[4~, kf1=\E[[A, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf13=\E[25~, kf14=\E[26~,
	kf15=\E[28~, kf16=\E[29~, kf17=\E[31~, kf18=\E[32~,
	kf19=\E[33~, kf2=\E[[B, kf20=\E[34~, kf3=\E[[C, kf4=\E[[D,
	kf5=\E[[E, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, kich1=\E[2~, kmous=\E[M, knp=\E[6~, kpp=\E[5~,
	kspd=^Z, nel=^M^J, rc=\E8, rev=\E[7m, ri=\EM, rmam=\E[?7l,
	rmir=\E[4l, rmso=\E[27m, rmul=\E[24m, rs1=\Ec\E]R, sc=\E7,
	sgr=\E[0;10%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p9%t;11%;m,
	smam=\E[?7h, smir=\E[4h, smul=\E[4m, tbc=\E[3g,
	vpa=\E[%i%p1%dd, use=vt102+enq, use=klone+sgr,
	use=ecma+color,

linux-m|Linux console no color,
	colors@, pairs@,
	setab@, setaf@, setb@, setf@, use=linux,

# The 1.3.x kernels add color-change capabilities; if yours doesn't have this
# and it matters, turn off <ccc>.  The %02x escape used to implement this is
# not supposedly back-portable to older SV curses (although it has worked fine
# on Solaris for several years) and not supported in ncurses versions before
# 1.9.9.
linux-c-nc|linux console with color-change,
	ccc,
	initc=\E]P%p1%x%p2%{255}%*%{1000}%/%02x%p3%{255}%*%{1000}%/%02x%p4%{255}%*%{1000}%/%02x,
	oc=\E]R, use=linux-basic,
# From: Dennis Henriksen <opus@osrl.dk>, 9 July 1996
linux-c|linux console 1.3.6+ for older ncurses,
	ccc,
	initc=\E]P%?%p1%{9}%>%t%p1%{10}%-%'a'%+%c%e%p1%d%;%p2%{255}%*%{1000}%/%Pr%gr%{16}%/%Px%?%gx%{9}%>%t%gx%{10}%-%'a'%+%c%e%gx%d%;%gr%{15}%&%Px%?%gx%{9}%>%t%gx%{10}%-%'a'%+%c%e%gx%d%;%p3%{255}%*%{1000}%/%Pr%gr%{16}%/%Px%?%gx%{9}%>%t%gx%{10}%-%'a'%+%c%e%gx%d%;%gr%{15}%&%Px%?%gx%{9}%>%t%gx%{10}%-%'a'%+%c%e%gx%d%;%p4%{255}%*%{1000}%/%Pr%gr%{16}%/%Px%?%gx%{9}%>%t%gx%{10}%-%'a'%+%c%e%gx%d%;%gr%{15}%&%Px%?%gx%{9}%>%t%gx%{10}%-%'a'%+%c%e%gx%d%;,
	oc=\E]R, use=linux-basic,

# The 2.2.x kernels add a private mode that sets the cursor type; use that to
# get a block cursor for cvvis.
# reported by Frank Heckenbach <frank@g-n-u.de>.
linux2.2|linux 2.2.x console,
	civis=\E[?25l\E[?1c, cnorm=\E[?25h\E[?0c,
	cvvis=\E[?25h\E[?8c, use=linux-c-nc,

# Linux 2.6.x has a fix for SI/SO to work with UTF-8 encoding added here:
#	http://lkml.indiana.edu/hypermail/linux/kernel/0602.2/0868.html
# Using SI/SO has the drawback that it confuses screen.  SCS would work.
# However, SCS is buggy (see comment in Debian #515609) -TD
# Further, this breaks longstanding workarounds for Linux console's line
# drawing (see Debian 665959) -TD
linux2.6|linux 2.6.x console,
	rmacs=^O,
	sgr=\E[0;10%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, use=linux2.2,

# The 3.0 kernel adds support for clearing scrollback buffer (capability E3).
# It is the same as xterm's erase-saved-lines feature.
linux3.0|linux 3.0 kernels,
	E3=\E[3;J, use=linux2.6,

# This is Linux console for ncurses.
linux|linux console,
	use=linux2.2,

# Subject: linux 2.6.26 vt back_color_erase
# Changes to the Linux console driver broke bce model as reported in
#	https://bugzilla.novell.com/show_bug.cgi?id=418613
# apparently from
#	http://lkml.org/lkml/2008/4/26/305
#	http://groups.google.com/group/fa.linux.kernel/browse_thread/thread/87f98338f0d636bb/aa96e8b86cee0d1e?lnk=st&q=#aa96e8b86cee0d1e
linux2.6.26|linux console w/o bce,
	bce@, use=linux2.6,

# See the note on ICH/ICH1 VERSUS RMIR/SMIR near the end of file
linux-nic|linux with ich/ich1 suppressed for non-curses programs,
	ich@, ich1@, use=linux,

# This assumes you have used setfont(8) to load one of the Linux koi8-r fonts.
# acsc entry from Pavel Roskin" <pavel@absolute.spb.su>, 29 Sep 1997.
linux-koi8|linux with koi8 alternate character set,
	acsc=+\020\,\021-\030.^Y0\215`\004a\221f\234g\237h\220i\276j\205k\203l\202m\204n\212o~p\0q\0r\0s_t\206u\207v\211w\210x\201y\230z\231{\267|\274~\224,
	use=linux, use=klone+koi8acs,

# Another entry for KOI8-r with Qing Long's acsc.
# (which one better complies with the standard?)
linux-koi8r|linux with koi8-r alternate character set,
	use=linux, use=klone+koi8acs,

# Entry for the latin1 and latin2 fonts
linux-lat|linux with latin1 or latin2 alternate character set,
	acsc=+\020\,\021-\030.^Y0\333`\004a\013f\370g\361h\260i\316j\211k\214l\206m\203n\305o~p\304q\212r\304s_t\207u\215v\301w\302x\205y\363z\362{\343|\330}\234~\376,
	use=linux,

# This uses graphics from VT codeset instead of from cp437.
# reason: cp437 (aka "straight to font") is not functional under luit.
# from: Andrey V Lukyanov <land@long.yar.ru>.
linux-vt|linux console using VT codes for graphics,
	acsc=++\,\,--..00``aaffgghhiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz~~,
	rmacs=\E(K, rmpch@, sgr@, sgr0=\E[0m\E(K\017, smacs=\E(0,
	smpch@, use=linux,

# This is based on the Linux console (relies on the console to perform some
# of the functionality), but does not recognize as many control sequences.
# The program comes bundled with an old (circa 1998) copy of the Linux
# console terminfo.  It recognizes some non-ANSI/VT100 sequences such as
#	\E*	move cursor to home, as as \E[H
#	\E,X	same as \E(X
#	\EE	move cursor to beginning of row
#	\E[y,xf	same as \E[y,xH
#
# Note: The status-line support is buggy (dsl does not work).
kon|kon2|jfbterm|Kanji ON Linux console,
	ccc@, hs,
	civis@, cnorm@, cvvis@, dsl=\E[?H, flash@, fsl=\E[?F, initc@,
	initp@, kcbt@, oc@, op=\E[37;40m, rs1=\Ec, tsl=\E[?T,
	use=linux,

# 16-color linux console entry; this works with a 256-character
# console font but bright background colors turn into dim ones when
# you use a 512-character console font. This uses bold for bright
# foreground colors and blink for bright background colors.
linux-16color|linux console with 16 colors,
	colors#16, ncv#63, pairs#256,
	setab=\E[4%p1%{8}%m%d%?%p1%{7}%>%t;5%e;25%;m,
	setaf=\E[3%p1%{8}%m%d%?%p1%{7}%>%t;1%e;21%;m,
	use=linux,

# bterm (bogl 0.1.18)
# Implementation is in bogl-term.c
# Key capabilities from linux terminfo entry
#
# Notes:
# bterm only supports acs using wide-characters, has case for these: qjxamlkut
# bterm does not support sgr, since it only processes one parameter -TD
bterm|bogl virtual terminal,
	am, bce,
	colors#8, cols#80, lines#24, pairs#64,
	acsc=aajjkkllmmqqttuuxx, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[2J, cnorm=\E[?25h, cr=^M, cub1=^H, cud1=^J,
	cup=\E[%i%p1%d;%p2%dH, ed=\E[J, el=\E[K, home=\E[H, ind=^J,
	kb2=\E[G, kbs=\177, kcbt=\E[Z, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~, kend=\E[4~, kf1=\E[[A,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf13=\E[25~,
	kf14=\E[26~, kf15=\E[28~, kf16=\E[29~, kf17=\E[31~,
	kf18=\E[32~, kf19=\E[33~, kf2=\E[[B, kf20=\E[34~,
	kf3=\E[[C, kf4=\E[[D, kf5=\E[[E, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, khome=\E[1~, kich1=\E[2~,
	kmous=\E[M, knp=\E[6~, kpp=\E[5~, kspd=^Z, nel=^M^J,
	op=\E[49m\E[39m, rev=\E[7m, ri=\EM, rmacs=^O, rmso=\E[27m,
	rmul=\E[24m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr0=\E[0m, smacs=^N, smso=\E[7m, smul=\E[4m,

#### Mach
#

# From: Matthew Vernon <mcv21@pick.sel.cam.ac.uk>
mach|Mach Console,
	am, km,
	cols#80, it#8, lines#25,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\Ec, cr=^M,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, home=\E[H, ht=^I, il=\E[%p1%dL, il1=\E[L, ind=^J,
	kbs=\177, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[9, kend=\E[Y, kf1=\EOP, kf10=\EOY, kf2=\EOQ,
	kf3=\EOR, kf4=\EOS, kf5=\EOT, kf6=\EOU, kf7=\EOV, kf8=\EOW,
	kf9=\EOX, khome=\E[H, kich1=\E[@, kll=\E[F, knp=\E[U,
	kpp=\E[V, rev=\E[7m, rmso=\E[0m, rmul=\E[24m, sgr0=\E[0m,
	smso=\E[7m, smul=\E[4m,
mach-bold|Mach Console with bold instead of underline,
	rmul=\E[0m, smul=\E[1m, use=mach,
mach-color|Mach Console with ANSI color,
	colors#8, pairs#64,
	dim=\E[2m, invis=\E[8m, op=\E[37;40m, rmso=\E[27m,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm, use=mach,

# From: Samuel Thibault
# Source: git://git.sv.gnu.org/hurd/gnumach.git
# Files: i386/i386at/kd.c
#
# Added nel, hpa, sgr and removed rmacs, smacs based on source -TD
mach-gnu|GNU Mach,
	acsc=+>\,<-\^.v0\333`+a\261f\370g\361h\260i#j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m, ech=\E[%p1%dX,
	el1=\E[1K, hpa=\E[%i%p1%dG, ich=\E[%p1%d@, ich1=\E[@,
	indn=\E[%p1%dS, invis=\E[8m, nel=\EE, rin=\E[%p1%dT,
	sgr=\E[0%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p7%t;8%;m,
	use=mach,

mach-gnu-color|Mach Console with ANSI color,
	colors#8, pairs#64,
	op=\E[37;40m, rmso=\E[27m, setab=\E[4%p1%dm,
	setaf=\E[3%p1%dm, use=mach-gnu,

# From: Marcus Brinkmann
# http://savannah.gnu.org/cgi-bin/viewcvs/*checkout*/hurd/hurd/console/
#
# Comments in the original are summarized here:
#
# hurd uses 8-bit characters (km).
#
# Although it doesn't do XON/XOFF, we don't want padding characters (xon).
#
# Regarding compatibility to vt100:  hurd doesn't specify <xenl>, as we don't
# have the eat_newline_glitch.  It doesn't support setting or removing tab
# stops (hts/tbc).
#
# hurd uses ^H instead of \E[D for cub1, as only ^H implements <bw> and it is
# one byte instead three.
#
# <ich1> is not included because hurd has insert mode.
#
# hurd doesn't use ^J for scrolling, because this could put things into the
# scrollback buffer.
#
# gsbom/grbom are used to enable/disable real bold (not intensity bright) mode.
# This is a GNU extension.
#
# The original has commented-out ncv, but is restored here.
#
# Reading the source, RIS resets cnorm, but not xmous.
hurd|The GNU Hurd console server,
	am, bce, bw, eo, km, mir, msgr, xon,
	colors#8, it#8, ncv#18, pairs#64,
	acsc=++\,\,--..00``aaffgghhiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\Ec, cnorm=\E[?25h, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, cvvis=\E[34l, dch=\E[%p1%dP,
	dch1=\E[P, dim=\E[2m, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K, flash=\Eg,
	home=\E[H, hpa=\E[%i%p1%dG, ht=^I, ich=\E[%p1%d@,
	il=\E[%p1%dL, il1=\E[L, ind=\E[S, indn=\E[%p1%dS,
	invis=\E[8m, kb2=\E[G, kbs=\177, kcbt=\E[Z, kcub1=\EOD,
	kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kdch1=\E[3~,
	kend=\E[4~, kf1=\EOP, kf10=\E[21~, kf11=\E[23~,
	kf12=\E[24~, kf13=\E[25~, kf14=\E[26~, kf15=\E[28~,
	kf16=\E[29~, kf17=\E[31~, kf18=\E[32~, kf19=\E[33~,
	kf2=\EOQ, kf20=\E[34~, kf3=\EOR, kf4=\EOS, kf5=\E[15~,
	kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, kich1=\E[2~, kmous=\E[M, knp=\E[6~, kpp=\E[5~,
	kspd=^Z, nel=^M^J, op=\E[39;49m, rc=\E8, rev=\E[7m, ri=\E[T,
	rin=\E[%p1%dT, ritm=\E[23m, rmacs=\E[10m, rmir=\E[4l,
	rmso=\E[27m, rmul=\E[24m, rs1=\EM\E[?1000l, sc=\E7,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr=\E[0%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p7%t;8%;%?%p9%t;11%;m,
	sgr0=\E[0m, sitm=\E[3m, smacs=\E[11m, smir=\E[4h,
	smso=\E[7m, smul=\E[4m, vpa=\E[%i%p1%dd, grbom=\E[>1l,
	gsbom=\E[>1h,

#### QNX
#

# QNX 4.0 Console
# Michael's original version of this entry had <am@>, <smcup=\Ei>,
# <rmcup=\Eh\ER>; this was so terminfo applications could write the lower
# right corner without triggering a scroll.  The ncurses terminfo library can
# handle this case with the <ich1> capability, and prefers <am> for better
# optimization.  Bug: The <op> capability resets attributes.
# From: Michael Hunter <mphunter@qnx.com> 30 Jul 1996
# (removed: <sgr=%?%p1%t\E<%;%p2%t\E[%;%p3%t\E(%;%p4%t\E{%;%p6%t\E<%;,>)
qnx|qnx4|qnx console,
	daisy, km, mir, msgr, xhpa, xt,
	colors#8, cols#80, it#4, lines#25, ncv#3, pairs#8,
	acsc=O\333a\261j\331k\277l\332m\300n\305o\337q\304s\334t\303u\264v\301w\302x\263,
	bel=^G, blink=\E{, bold=\E<, civis=\Ey0, clear=\EH\EJ,
	cnorm=\Ey1, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, cvvis=\Ey2,
	dch1=\Ef, dl1=\EF, ed=\EJ, el=\EK, home=\EH, ht=^I, ich1=\Ee,
	il1=\EE, ind=^J, kBEG=\377\356, kCAN=\377\263,
	kCMD=\377\267, kCPY=\377\363, kCRT=\377\364,
	kDL=\377\366, kEND=\377\301, kEOL=\377\311,
	kEXT=\377\367, kFND=\377\370, kHLP=\377\371,
	kHOM=\377\260, kIC=\377\340, kLFT=\377\264,
	kMOV=\377\306, kMSG=\377\304, kNXT=\377\272,
	kOPT=\377\372, kPRT=\377\275, kPRV=\377\262,
	kRDO=\377\315, kRES=\377\374, kRIT=\377\266,
	kRPL=\377\373, kSAV=\377\307, kSPD=\377\303,
	kUND=\377\337, kbeg=\377\300, kcan=\377\243, kcbt=\377\0,
	kclo=\377\343, kclr=\377\341, kcmd=\377\245,
	kcpy=\377\265, kcrt=\377\305, kctab=\377\237,
	kcub1=\377\244, kcud1=\377\251, kcuf1=\377\246,
	kcuu1=\377\241, kdch1=\377\254, kdl1=\377\274,
	ked=\377\314, kel=\377\310, kend=\377\250, kent=\377\320,
	kext=\377\270, kf1=\377\201, kf10=\377\212,
	kf11=\377\256, kf12=\377\257, kf13=\377\213,
	kf14=\377\214, kf15=\377\215, kf16=\377\216,
	kf17=\377\217, kf18=\377\220, kf19=\377\221,
	kf2=\377\202, kf20=\377\222, kf21=\377\223,
	kf22=\377\224, kf23=\377\333, kf24=\377\334,
	kf25=\377\225, kf26=\377\226, kf27=\377\227,
	kf28=\377\230, kf29=\377\231, kf3=\377\203,
	kf30=\377\232, kf31=\377\233, kf32=\377\234,
	kf33=\377\235, kf34=\377\236, kf35=\377\276,
	kf36=\377\277, kf37=\377\321, kf38=\377\322,
	kf39=\377\323, kf4=\377\204, kf40=\377\324,
	kf41=\377\325, kf42=\377\326, kf43=\377\327,
	kf44=\377\330, kf45=\377\331, kf46=\377\332,
	kf47=\377\316, kf48=\377\317, kf5=\377\205, kf6=\377\206,
	kf7=\377\207, kf8=\377\210, kf9=\377\211, kfnd=\377\346,
	khlp=\377\350, khome=\377\240, khts=\377\342,
	kich1=\377\253, kil1=\377\273, kind=\377\261,
	kmov=\377\351, kmrk=\377\355, kmsg=\377\345,
	knp=\377\252, knxt=\377\312, kopn=\377\357,
	kopt=\377\353, kpp=\377\242, kprt=\377\255,
	kprv=\377\302, krdo=\377\336, kref=\377\354,
	kres=\377\360, krfr=\377\347, kri=\377\271,
	krmir=\377\313, krpl=\377\362, krst=\377\352,
	ksav=\377\361, kslt=\377\247, kspd=\377\335,
	ktbc=\377\344, kund=\377\365, mvpa=\E!%p1%02d, op=\ER,
	rep=\Eg%p2%{32}%+%c%p1%c, rev=\E(, ri=\EI, rmcup=\Eh\ER,
	rmso=\E), rmul=\E], rs1=\ER, setb=\E@%p1%Pb%gb%gf%d%d,
	setf=\E@%p1%Pf%gb%gf%d%d, sgr0=\E}\E]\E>\E), smcup=\Ei,
	smso=\E(, smul=\E[,
#
#
qnxt|qnxt4|QNX4 terminal,
	crxm, use=qnx4,
#
qnxm|QNX4 with mouse events,
	maddr#1,
	chr=\E/, cvr=\E", is1=\E/0t, mcub=\E/>1h, mcub1=\E/>7h,
	mcud=\E/>1h, mcud1=\E/>1l\E/>9h, mcuf=\E/>1h\E/>9l,
	mcuf1=\E/>7l, mcuu=\E/>6h, mcuu1=\E/>6l, rmicm=\E/>2l,
	smicm=\E/>2h, use=qnx4,
#
qnxw|QNX4 windows,
	xvpa, use=qnxm,
#
#	Monochrome QNX4 terminal or console. Setting this terminal type will
#	allow an application running on a color console to behave as if it
#	were a monochrome terminal. Output will be through stdout instead of
#	console writes because the term routines will recognize that the
#	terminal name starts with 'qnxt'.
#
qnxtmono|Monochrome QNX4 terminal or console,
	colors@, pairs@,
	scp@, use=qnx4,

# From: Federico Bianchi <bianchi@pc-arte2.arte.unipi.it>, 1 Jul 1998
# (esr: commented out <scp> and <rmcup> to avoid warnings.)
# (TD: derive from original qnx4 entry)
qnxt2|qnx 2.15 serial terminal,
	am,
	civis@, cnorm@, cvvis@, dch1@, ich1@, kRES@, kRPL@, kUND@, kspd@,
	rep@, rmcup@, rmso=\E>, setb@, setf@, smcup@, smso=\E<, use=qnx4,

# QNX ANSI terminal definition
qansi-g|QNX ANSI,
	am, eslok, hs, xon,
	colors#8, cols#80, it#8, lines#25, ncv#19, pairs#64, wsl#80,
	acsc=Oa``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[2J\E[H, cnorm=\E[?25h\E[?12l, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=\E[D,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH$<5>, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[?12;25h, dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[1M, dsl=\E[r, ech=\E[%p1%dX, ed=\E[J,
	el=\E[K, el1=\E[1K\E[X, flash=\E[?5h$<200>\E[?5l,
	fsl=\E[?6h\E8, home=\E[H, hpa=\E[%i%p1%dG, ht=^I, hts=\EH,
	ich=\E[%p1%d@, ich1=\E[1@, il=\E[%p1%dL, il1=\E[1L,
	ind=\E[S, indn=\E[%p1%dS, invis=\E[9m,
	is2=\E>\E[?1l\E[?7h\E[0;10;39;49m, is3=\E(B\E)0,
	kBEG=\ENn, kCAN=\E[s, kCMD=\E[t, kCPY=\ENs, kCRT=\ENt,
	kDL=\ENv, kEXT=\ENw, kFND=\ENx, kHLP=\ENy, kHOM=\E[h,
	kLFT=\E[d, kNXT=\E[u, kOPT=\ENz, kPRV=\E[v, kRIT=\E[c,
	kbs=^H, kcan=\E[S, kcbt=\E[Z, kclo=\ENc, kclr=\ENa,
	kcmd=\E[G, kcpy=\E[g, kctab=\E[z, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P, kdl1=\E[p, kend=\E[Y,
	kext=\E[y, kf1=\EOP, kf10=\EOY, kf11=\EOZ, kf12=\EOA,
	kf13=\EOp, kf14=\EOq, kf15=\EOr, kf16=\EOs, kf17=\EOt,
	kf18=\EOu, kf19=\EOv, kf2=\EOQ, kf20=\EOw, kf21=\EOx,
	kf22=\EOy, kf23=\EOz, kf24=\EOa, kf25=\E[1~, kf26=\E[2~,
	kf27=\E[3~, kf28=\E[4~, kf29=\E[5~, kf3=\EOR, kf30=\E[6~,
	kf31=\E[7~, kf32=\E[8~, kf33=\E[9~, kf34=\E[10~,
	kf35=\E[11~, kf36=\E[12~, kf37=\E[17~, kf38=\E[18~,
	kf39=\E[19~, kf4=\EOS, kf40=\E[20~, kf41=\E[21~,
	kf42=\E[22~, kf43=\E[23~, kf44=\E[24~, kf45=\E[25~,
	kf46=\E[26~, kf47=\E[27~, kf48=\E[28~, kf5=\EOT, kf6=\EOU,
	kf7=\EOV, kf8=\EOW, kf9=\EOX, kfnd=\ENf, khlp=\ENh,
	khome=\E[H, khts=\ENb, kich1=\E[@, kil1=\E[`, kind=\E[a,
	kmov=\ENi, kmrk=\ENm, kmsg=\ENe, knp=\E[U, kopn=\ENo,
	kopt=\ENk, kpp=\E[V, kref=\ENl, kres=\ENp, krfr=\ENg,
	kri=\E[b, krpl=\ENr, krst=\ENj, ksav=\ENq, kslt=\E[T,
	ktbc=\ENd, kund=\ENu, ll=\E[99H, nel=\EE, op=\E[39;49m,
	rep=%p1%c\E[%p2%{1}%-%db, rev=\E[7m, ri=\E[T,
	rin=\E[%p1%dT, rmacs=^O, rmam=\E[?7l, rmso=\E[27m,
	rmul=\E[24m, rs1=\017\E[?7h\E[0;39;49m$<2>\E>\E[?1l,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h,
	setb=\E[4%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	setf=\E[3%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	sgr=\E[0%?%p6%t;1%;%?%p5%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p3%p1%|%t;7%;%?%p7%t;9%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g,
	tsl=\E7\E1;24r\E[?6l\E[25;%i%p1%dH,
#
qansi|QNX ansi with console writes,
	daisy, xhpa, use=qansi-g,
#
qansi-t|QNX ansi without console writes,
	crxm, use=qansi,
#
qansi-m|QNX ansi with mouse,
	maddr#1,
	chr=\E[, cvr=\E], is1=\E[0t, mcub=\E[>1h, mcub1=\E[>7h,
	mcud=\E[>1h, mcud1=\E[>1l\E[>9h, mcuf=\E[>1h\E[>9l,
	mcuf1=\E[>7l, mcuu=\E[>6h, mcuu1=\E[>6l, rmicm=\E[>2l,
	smicm=\E[>2h, use=qansi,
#
qansi-w|QNX ansi for windows,
	xvpa, use=qansi-m,

#### SCO consoles

# SCO console and SOS-Syscons console for 386bsd
# (scoansi: had unknown capabilities
#	:Gc=N:Gd=K:Gh=M:Gl=L:Gu=J:Gv=\072:\
#	:GC=E:GD=B:GH=D:GL=\64:GU=A:GV=\63:GR=C:
#	:G1=?:G2=Z:G3=@:G4=Y:G5=;:G6=I:G7=H:G8=<:\
#	:CW=\E[M:NU=\E[N:RF=\E[O:RC=\E[P:\
#	:WL=\E[S:WR=\E[T:CL=\E[U:CR=\E[V:\
# I renamed GS/GE/HM/EN/PU/PD/RT and added klone+sgr-dumb, based
# on the <smacs>=\E[12m  -- esr)
#
# klone+sgr-dumb is an error since the acsc does not match -TD
#
# In this description based on SCO's keyboard(HW) manpage list of default
# function key values:
#	F13-F24 are shifted F1-F12
#	F25-F36 are control F1-F12
#	F37-F48 are shift+control F1-F12
#
# hpa/vpa work in the console, but not in scoterm:
#	hpa=\E[%p1%dG,
#	vpa=\E[%p1%dd,
#
# SCO's terminfo uses
#	kLFT=\E[d,
#	kRIT=\E[c,
# which do not work (console or scoterm).
#
# Console documents only 3 attributes can be set with SGR (so we don't use sgr).
scoansi-old|SCO Extended ANSI standard crt (5.0.5),
	OTbs, am, bce, eo, xon,
	colors#8, cols#80, it#8, lines#25, pairs#64,
	acsc=+/\,.-\230.\2310[5566778899\:\:;;<<==>>FFGGHHIIJJKKLLMMNNOOPPQQRRSSTTUUVVWWXX`\204a0fxgqh2jYk?lZm@nEqDtCu4vAwBx3yszr{c}\034~\207,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z,
	civis=\E[=14;12C, clear=\E[H\E[2J, cnorm=\E[=10;12C,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, cvvis=\E[=0;12C, dch=\E[%p1%dP,
	dch1=\E[P, dispc=\E[=%p1%dg, dl=\E[%p1%dM, dl1=\E[M,
	ed=\E[m\E[J, el=\E[m\E[K, el1=\E[1K, home=\E[H, ht=^I,
	hts=\EH, ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[L,
	ind=\E[S, indn=\E[%p1%dS, invis=\E[8m, kbeg=\E[E, kbs=^H,
	kcbt=\E[Z, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\177, kend=\E[F, kf1=\E[M, kf10=\E[V, kf11=\E[W,
	kf12=\E[X, kf13=\E[Y, kf15=\E[a, kf16=\E[b, kf17=\E[c,
	kf18=\E[d, kf19=\E[e, kf2=\E[N, kf20=\E[f, kf21=\E[g,
	kf22=\E[h, kf23=\E[i, kf24=\E[j, kf25=\E[k, kf26=\E[l,
	kf27=\E[m, kf28=\E[n, kf29=\E[o, kf3=\E[O, kf30=\E[p,
	kf31=\E[q, kf32=\E[r, kf33=\E[s, kf34=\E[t, kf35=\E[u,
	kf36=\E[v, kf37=\E[w, kf38=\E[x, kf39=\E[y, kf4=\E[P,
	kf40=\E[z, kf41=\E[@, kf42=\E[[, kf43=\E[\\, kf44=\E[],
	kf45=\E[\^, kf46=\E[_, kf47=\E[`, kf48=\E[{, kf5=\E[Q,
	kf6=\E[R, kf7=\E[S, kf8=\E[T, kf9=\E[U, khome=\E[H,
	kich1=\E[L, knp=\E[G, kpp=\E[I, op=\E[0;37;40m, rc=\E8,
	rev=\E[7m, ri=\E[T, rin=\E[%p1%dT, rmacs=\E[10m,
	rmam=\E[?7l, rmso=\E[m, rmul=\E[m, sc=\E7,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm, sgr0=\E[0;10m,
	smacs=\E[12m, smam=\E[?7h, smso=\E[7m, smul=\E[4m,
scoansi-new|SCO Extended ANSI standard crt (5.0.6),
	km,
	civis=\E[=0c, cnorm=\E[=1c, csr=\E[%i%p1%d;%p2%dr,
	cvvis=\E[=2c, mgc=\E[=r, oc=\E[51m, op=\E[50m,
	rep=\E[%p1%d;%p2%db, rmm=\E[=11L,
	sgr=\E[0%?%p1%p3%|%t;7%;%?%p2%t;4%;%?%p4%t;5%;%?%p6%t;1%;%?%p7%t;8%;%?%p9%t;12%e;10%;m,
	smgb=\E[=1;0m, smgbp=\E[=1;%i%p1%dm,
	smglp=\E[=2;%i%p1%dm, smgr=\E[=3;0m,
	smgrp=\E[=3;%i%p1%dm, smgt=\E[=0;0m,
	smgtp=\E[=0;%i%p1%dm, smm=\E[=10L,
	wind=\E[%i%p1%d;%p2%d;%i%p3%d;%p4%dr,
	use=scoansi-old,
# make this easy to change...
scoansi|SCO Extended ANSI standard crt,
	use=scoansi-old,

#### SGI consoles

# Sent by Stefan Stapelberg <stefan@rent-a-guru.de>, 24 Feb 1997, this is
# from SGI's terminfo database.  SGI's entry shows F9-F12 with the codes
# for the application keypad mode.  We have added iris-ansi-ap rather than
# change the original to keypad mode.
#
# (iris-ansi: added rmam/smam based on init string -- esr)
#
# This entry, and those derived from it, is used in xwsh (also known as
# winterm).  Some capabilities that do not fit into the terminfo model
# include the shift- and control-functionkeys:
#
# F1-F12 generate different codes when shift or control modifiers are used.
# For example:
#	F1      	\E[001q
#	shift F1	\E[013q
#	control-F1	\E[025q
#
# In application keypad mode, F9-F12 generate codes like vt100 PF1-PF4, i.e.,
# \EOP to \EOS.  The shifted and control modifiers still do the same thing.
#
# The cursor keys also have different codes:
#	control-up	\E[162q
#	control-down	\E[165q
#	control-left	\E[159q
#	control-right	\E[168q
#
#	shift-up	\E[161q
#	shift-down	\E[164q
#	shift-left	\E[158q
#	shift-right	\E[167q
#
#	control-tab	\[072q
#
iris-ansi|iris-ansi-net|IRIS emulating 40 line ANSI terminal (almost VT100),
	am,
	cols#80, it#8, lines#40,
	bel=^G, bold=\E[1m, clear=\E[H\E[2J,
	cnorm=\E[9/y\E[12/y\E[=6l, cr=^M, cub=\E[%p1%dD,
	cub1=\E[D, cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA,
	cuu1=\E[A, cvvis=\E[10/y\E[=1h\E[=2l\E[=6h,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K,
	home=\E[H, ht=^I, hts=\EH, il=\E[%p1%dL, il1=\E[L, ind=\ED,
	is2=\E[?1l\E>\E[?7h\E[100g\E[0m\E7\E[r\E8, kDC=\E[P,
	kEND=\E[147q, kHOM=\E[143q, kLFT=\E[158q, kPRT=\E[210q,
	kRIT=\E[167q, kSPD=\E[218q, kbs=^H, kcbt=\E[Z, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\177,
	kend=\E[146q, kent=^M, kf1=\E[001q, kf10=\E[010q,
	kf11=\E[011q, kf12=\E[012q, kf2=\E[002q, kf3=\E[003q,
	kf4=\E[004q, kf5=\E[005q, kf6=\E[006q, kf7=\E[007q,
	kf8=\E[008q, kf9=\E[009q, khome=\E[H, kich1=\E[139q,
	knp=\E[154q, kpp=\E[150q, kprt=\E[209q, krmir=\E[146q,
	kspd=\E[217q, nel=\EE, pfkey=\EP101;%p1%d.y%p2%s\E\\,
	rc=\E8, rev=\E[7m, ri=\EM, rmam=\E[?7l, rmso=\E[m, rmul=\E[m,
	sc=\E7, sgr0=\E[m, smam=\E[?7h, smso=\E[1;7m, smul=\E[4m,
	tbc=\E[3g,
iris-ansi-ap|IRIS ANSI in application-keypad mode,
	is2=\E[?1l\E=\E[?7h, kent=\EOM, kf10=\E[010q,
	kf11=\E[011q, kf12=\E[012q, kf9=\E[009q, use=iris-ansi,

# From the man-page, this is a quasi-vt100 emulator that runs on SGI's IRIX
# (T.Dickey 98/1/24)
iris-color|xwsh|IRIX ANSI with color,
	ncv#33,
	csr=\E[%i%p1%d;%p2%dr, dch=\E[%p1%dP, dim=\E[2m,
	ech=\E[%p1%dX, ich=\E[%p1%d@, rc=\E8, ritm=\E[23m,
	rmul=\E[24m, rs1=\Ec,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sitm=\E[3m, use=vt100+enq, use=klone+color,
	use=iris-ansi-ap,

#### OpenBSD consoles
#
# From: Alexei Malinin <Alexei.Malinin@mail.ru>; October, 2011.
#
# The following terminal descriptions for the  AMD/Intel PC console
# were prepared  based on information contained in  the OpenBSD-4.9
# termtypes.master and wscons(4) & vga(4) manuals (2010, November).
#
# Added bce based on testing with tack -TD
# Added several capabilities to pccon+base, reading wsemul_vt100_subr.c -TD
# Changed kbs to DEL and removed keys that duplicate stty settings -TD
#
pccon+keys|OpenBSD PC keyboard keys,
	kbs=\177, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[3~, kend=\E[8~, kent=^M, kf1=\E[11~, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf2=\E[12~, kf3=\E[13~,
	kf4=\E[14~, kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~,
	kf9=\E[20~, khome=\E[7~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~,
	krfr=^R,
pccon+sgr+acs0|sgr and simple ASCII pseudographics for OpenBSD PC console,
	acsc=+>\,<-\^.v0#`+a\:f\\h#i#j+k+l+m+n+o~p-q-r-s_t+u+v+w+x|y#z#{*|!}#~o,
	sgr=\E[0%?%p1%p3%|%t;7%;m, sgr0=\E[m,
pccon+sgr+acs|sgr and default ASCII pseudographics for OpenBSD PC console,
	acsc=++\,\,--..00``aaffgghhiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	enacs=\E)0$<5>, rmacs=\E(B$<5>,
	sgr=\E[0%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;$<5>,
	sgr0=\E[m\E(B$<5>, smacs=\E(0$<5>,
pccon+colors|ANSI colors for OpenBSD PC console,
	bce,
	colors#8, pairs#64,
	op=\E[m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
pccon+base|base capabilities for OpenBSD PC console,
	am, km, mc5i, msgr, npc, nxon, xenl, xon,
	cols#80, it#8, lines#24,
	bel=^G, clear=\E[H\E[J, cr=^M, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch=\E[%p1%dP,
	dch1=\E[P, dl1=\E[M, ech=\E[%p1%dX, ed=\E[J, el=\E[K,
	el1=\E[1K, home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@,
	il1=\E[L, ind=\ED, nel=\EE, rev=\E[7m, ri=\EM, rmam=\E[?7l,
	rmso=\E[m, rs2=\Ec$<50>, smam=\E[?7h, smso=\E[7m,
	tbc=\E[3g, u6=\E[%i%d;%dR, u7=\E[6n,
pccon0-m|OpenBSD PC console without colors & with simple ASCII pseudographics,
	use=pccon+base, use=pccon+sgr+acs0, use=pccon+keys,
pccon0|OpenBSD PC console with simple ASCII pseudographics,
	use=pccon0-m, use=pccon+colors,
pccon-m|OpenBSD PC console without colors,
	use=pccon+base, use=pccon+sgr+acs, use=pccon+keys,
pccon|OpenBSD PC console,
	use=pccon-m, use=pccon+colors,

#### NetBSD consoles
#
# pcvt termcap database entries (corresponding to release 3.31)
# Author's last edit-date: [Fri Sep 15 20:29:10 1995]
#
# (For the terminfo master file, I translated these into terminfo syntax.
# Then I dropped all the pseudo-HP entries. we don't want and can't use
# the :Xs: flag. Then I split :is: into a size-independent <is1> and a
# size-dependent <is2>.  Finally, I added <rmam>/<smam> -- esr)

# NOTE: <ich1> has been taken out of this entry. for reference, it should
# be <ich1=\E[@>.  For discussion, see ICH/ICH1 VERSUS RMIR/SMIR below.
# (esr: added <civis> and <cnorm> to resolve NetBSD Problem Report #4583)
pcvtXX|pcvt vt200 emulator (DEC VT220),
	am, km, mir, msgr, xenl,
	it#8, vt#3,
	acsc=++\,\,--..00``aaffgghhiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, el1=\E[1K, home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@,
	il=\E[%p1%dL, il1=\E[L, ind=\ED, indn=\E[%p1%dS,
	is1=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, kbs=\177,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kdch1=\E[3~, kf1=\E[17~, kf2=\E[18~, kf3=\E[19~,
	kf4=\E[20~, kf5=\E[21~, kf6=\E[23~, kf7=\E[24~, kf8=\E[25~,
	khome=\E[1~, kich1=\E[2~, kll=\E[4~, knp=\E[6~, kpp=\E[5~,
	nel=\EE, rc=\E8, rev=\E[7m, rf=/usr/share/tabset/vt100,
	ri=\EM, rin=\E[%p1%dT, rmacs=\E(B, rmam=\E[?7l, rmir=\E[4l,
	rmkx=\E[?1l\E>, rmso=\E[27m, rmul=\E[24m,
	rs1=\Ec\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr0=\E[m, smacs=\E(0, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,

#	NetBSD/FreeBSD vt220 terminal emulator console (pc keyboard & monitor)
#	termcap entries for pure VT220-Emulation and 25, 28, 35, 40, 43 and
#	50 lines entries; 80 columns
pcvt25|dec vt220 emulation with 25 lines,
	cols#80, lines#25,
	is2=\E[1;25r\E[25;1H, use=pcvtXX,
pcvt28|dec vt220 emulation with 28 lines,
	cols#80, lines#28,
	is2=\E[1;28r\E[28;1H, use=pcvtXX,
pcvt35|dec vt220 emulation with 35 lines,
	cols#80, lines#35,
	is2=\E[1;35r\E[35;1H, use=pcvtXX,
pcvt40|dec vt220 emulation with 40 lines,
	cols#80, lines#40,
	is2=\E[1;40r\E[40;1H, use=pcvtXX,
pcvt43|dec vt220 emulation with 43 lines,
	cols#80, lines#43,
	is2=\E[1;43r\E[43;1H, use=pcvtXX,
pcvt50|dec vt220 emulation with 50 lines,
	cols#80, lines#50,
	is2=\E[1;50r\E[50;1H, use=pcvtXX,

#	NetBSD/FreeBSD vt220 terminal emulator console (pc keyboard & monitor)
#	termcap entries for pure VT220-Emulation and 25, 28, 35, 40, 43 and
#	50 lines entries; 132 columns
pcvt25w|dec vt220 emulation with 25 lines and 132 cols,
	cols#132, lines#25,
	is2=\E[1;25r\E[25;1H, use=pcvtXX,
pcvt28w|dec vt220 emulation with 28 lines and 132 cols,
	cols#132, lines#28,
	is2=\E[1;28r\E[28;1H, use=pcvtXX,
pcvt35w|dec vt220 emulation with 35 lines and 132 cols,
	cols#132, lines#35,
	is2=\E[1;35r\E[35;1H, use=pcvtXX,
pcvt40w|dec vt220 emulation with 40 lines and 132 cols,
	cols#132, lines#40,
	is2=\E[1;40r\E[40;1H, use=pcvtXX,
pcvt43w|dec vt220 emulation with 43 lines and 132 cols,
	cols#132, lines#43,
	is2=\E[1;43r\E[43;1H, use=pcvtXX,
pcvt50w|dec vt220 emulation with 50 lines and 132 cols,
	cols#132, lines#50,
	is2=\E[1;50r\E[50;1H, use=pcvtXX,

#	OpenBSD implements a color variation
pcvt25-color|dec vt220 emulation with 25 lines and color,
	cols#80, lines#25,
	is2=\E[1;25r\E[25;1H, kf1=\EOP, kf10=\E[29~, kf11=\E[23~,
	kf12=\E[24~, kf13=\E[25~, kf14=\E[26~, kf15=\E[28~,
	kf16=\E[29~, kf17=\E[31~, kf18=\E[32~, kf19=\E[33~,
	kf2=\EOQ, kf20=\E[34~, kf3=\EOR, kf4=\EOS, kf5=\E[17~,
	kf6=\E[18~, kf7=\E[19~, kf8=\E[20~, kf9=\E[21~, use=pcvtXX,
	use=ecma+color,

# Terminfo entries to enable the use of the ncurses library in colour on a
# NetBSD-arm32 console (only tested on a RiscPC).
# Created by Dave Millen <dmill@globalnet.co.uk> 22.07.98
# modified codes for setf/setb to setaf/setab, then to klone+color, corrected
# typo in invis - TD
arm100|arm100-am|Arm(RiscPC) ncurses compatible (for 640x480),
	am, bce, msgr, xenl, xon,
	cols#80, it#8, lines#30,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m$<2>, bold=\E[1m$<2>,
	clear=\E[H\E[J$<50>, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C$<2>,
	cup=\E[%i%p1%d;%p2%dH$<5>, cuu=\E[%p1%dA,
	cuu1=\E[A$<2>, ed=\E[J$<50>, el=\E[K$<3>, el1=\E[1K$<3>,
	enacs=\E(B\E)0, home=\E[H, ht=^I, hts=\EH, ind=^J,
	invis=\E[8m$<2>, ka1=\E[q, ka3=\E[s, kb2=\E[r, kbs=^H,
	kc1=\E[p, kc3=\E[n, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kent=\E[M, kf0=\E[y, kf1=\E[P, kf10=\E[x,
	kf2=\E[Q, kf3=\E[R, kf4=\E[S, kf5=\E[t, kf6=\E[u, kf7=\E[v,
	kf8=\E[l, kf9=\E[w, rc=\E8, rev=\E[7m$<2>, ri=\EM$<5>,
	rmacs=^O, rmam=\E[?7l, rmkx=\E[?1l\E>, rmso=\E[m$<2>,
	rmul=\E[m$<2>, rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h,
	sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p7%t;8%;m%?%p9%t\016%e\017%;$<2>,
	sgr0=\E[m\017$<2>, smacs=^N, smam=\E[?7h, smkx=\E[?1h\E=,
	smso=\E[7m$<2>, smul=\E[4m$<2>, tbc=\E[3g, use=ecma+sgr,
	use=klone+color,

arm100-w|arm100-wam|Arm(RiscPC) ncurses compatible (for 1024x768),
	cols#132, lines#50, use=arm100,

# NetBSD/x68k console vt200 emulator. This port runs on a 68K machine
# manufactured by Sharp for the Japenese market.
# From Minoura Makoto <minoura@netlaputa.or.jp>, 12 May 1996
x68k|x68k-ite|NetBSD/x68k ITE,
	cols#96, lines#32,
	kclr=\E[9~, khlp=\E[28~, use=vt220,

# <tv@pobox.com>:
# Entry for the DNARD OpenFirmware console, close to ANSI but not quite.
#
# (still unfinished, but good enough so far.)
ofcons|DNARD OpenFirmware console,
	bw,
	cols#80, lines#30,
	bel=^G, blink=\2335m, bold=\2331m, clear=^L, cr=^M,
	cub=\233%p1%dD, cub1=\233D, cud=\233%p1%dB, cud1=\233B,
	cuf=\233%p1%dC, cuf1=\233C, cup=\233%i%p1%d;%p2%dH,
	cuu=\233%p1%dA, cuu1=\233A, dch=\233%p1%dP, dch1=\233P,
	dim=\2332m, dl=\233%p1%dM, dl1=\233M, ed=\233J, el=\233K,
	flash=^G, ht=^I, ich=\233%p1%d@, ich1=\233@, il=\233%p1%dL,
	il1=\233L, ind=^J, invis=\2338m, kbs=^H, kcub1=\233D,
	kcud1=\233B, kcuf1=\233C, kcuu1=\233A, kdch1=\233P,
	kf1=\2330P, kf10=\2330M, kf2=\2330Q, kf3=\2330W,
	kf4=\2330x, kf5=\2330t, kf6=\2330u, kf7=\2330q, kf8=\2330r,
	kf9=\2330p, knp=\233/, kpp=\233?, nel=^M^J, rev=\2337m,
	rmso=\2330m, rmul=\2330m,
	sgr=\2330%?%p6%t;1%;%?%p2%t;4%;%?%p4%t;5%;%?%p5%t2%;%?%p7%t8%;%?%p1%p3%|%t;7%;m,
	sgr0=\2330m, smso=\2337m, smul=\2334m,

# NetBSD "wscons" emulator in vt220 mode.
# This entry is based on the NetBSD termcap entry, correcting the ncv value.
# The emulator renders underlined text in red.  Colors are otherwise usable.
#
# Testing the emulator and reading the source code (NetBSD 2.0), it appears
# that "vt220" is inaccurate.  There are a few vt220-features, but most of the
# vt220 screens in vttest do not work with this emulator.  For instance, it
# identifies itself (primary DA response) as a vt220 with selective erase.  But
# the selective erase feature does not work.  The secondary response is copied
# from Kermit's emulation of vt220, does not correspond to actual vt220.  At
# the level of detail in a termcap, it is a passable emulator, since ECH does
# work.  Don't use it on a VMS system -TD
wsvt25|NetBSD wscons in 25 line DEC VT220 mode,
	bce, msgr,
	colors#8, cols#80, it#8, lines#25, ncv#2, pairs#64,
	civis=\E[?25l, cnorm=\E[?25h, is2=\E[r\E[25;1H,
	kdch1=\E[3~, kend=\E[8~, kf1=\E[11~, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf2=\E[12~, kf3=\E[13~,
	kf4=\E[14~, kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~,
	kf9=\E[20~, khome=\E[7~, op=\E[m, rs1=\Ec,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm, use=vt220,

wsvt25m|NetBSD wscons in 25 line DEC VT220 mode with Meta,
	km, use=wsvt25,

# `rasterconsole' provided by 4.4BSD, NetBSD and OpenBSD on SPARC, and
# DECstation/pmax.
rcons|BSD rasterconsole,
	use=sun-il,
# Color version of above. Color currently only provided by NetBSD.
rcons-color|BSD rasterconsole with ANSI color,
	bce,
	colors#8, pairs#64,
	op=\E[m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm, use=rcons,

# mgterm -- MGL/MGL2, MobileGear Graphic Library
# for PocketBSD,PocketLinux,NetBSD/{hpcmips,mac68k}
# -- the setf/setb are probably incorrect, more likely setaf/setab -TD
# -- compare with cons25w
mgterm,
	OTbs, OTpt, am, bce, bw, eo, km, msgr, npc,
	colors#8, cols#80, it#8, lines#18, pairs#64,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, clear=\E[H\E[J,
	cr=^M, csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dim=\E[30;1m, dl=\E[%p1%dM,
	dl1=\E[M, ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K,
	home=\E[H, hpa=\E[%i%p1%d`, ht=^I, ich=\E[%p1%d@,
	ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=\E[S,
	indn=\E[%p1%dS, kb2=\E[E, kbs=^H, kcbt=\E[Z, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\177, kend=\E[F,
	kf1=\E[M, kf10=\E[V, kf11=\E[W, kf12=\E[X, kf2=\E[N,
	kf3=\E[O, kf4=\E[P, kf5=\E[Q, kf6=\E[R, kf7=\E[S, kf8=\E[T,
	kf9=\E[U, khome=\E[H, kich1=\E[L, knp=\E[G, kpp=\E[I,
	nel=\E[E, op=\E[x, rc=\E8, rev=\E[7m, ri=\E[T, rin=\E[%p1%dT,
	rmso=\E[m, rs2=\E[x\E[m\Ec, sc=\E7, setb=\E[4%p1%dm,
	setf=\E[3%p1%dm, sgr0=\E[m, smso=\E[7m, vpa=\E[%i%p1%dd,

#### FreeBSD console entries
#
# From: Andrey Chernov <ache@astral.msk.su> 29 Mar 1996
# Andrey Chernov maintains the FreeBSD termcap distributions.
#
# Note: Users of FreeBSD 2.1.0 and older versions must either upgrade
# or comment out the :cb: capability in the console entry.
#
# Alexander Lukyanov reports:
# I have seen FreeBSD-2.1.5R... The old el1 bug changed, but it is still there.
# Now el1 clears not only to the line beginning, but also a large chunk
# of previous line. But there is another bug - ech does not work at all.
#

# for syscons
# common entry without semigraphics
# Bug: The <op> capability resets attributes.
# Bug? The ech and el1 attributes appear to move the cursor in some cases; for
# instance el1 does if the cursor is moved to the right margin first.  Removed
# by T.Dickey 97/5/3 (ech=\E[%p1%dX, el1=\E[1K)
#
# Setting colors turns off reverse; we cannot guarantee order, so use ncv.
# Note that this disables standout with color.
#
# The emulator sends difference strings based on shift- and control-keys,
# like scoansi:
#	F13-F24 are shifted F1-F12
#	F25-F36 are control F1-F12
#	F37-F48 are shift+control F1-F12
cons25w|ansiw|ansi80x25-raw|freebsd console (25-line raw mode),
	am, bce, bw, eo, msgr, npc,
	colors#8, cols#80, it#8, lines#25, ncv#21, pairs#64,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, clear=\E[H\E[J,
	cnorm=\E[=0C, cr=^M, cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB,
	cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[=1C, dch=\E[%p1%dP, dch1=\E[P, dim=\E[30;1m,
	dl=\E[%p1%dM, dl1=\E[M, ech=\E[%p1%dX, ed=\E[J, el=\E[K,
	home=\E[H, hpa=\E[%i%p1%d`, ht=^I, ich=\E[%p1%d@,
	ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=\E[S,
	indn=\E[%p1%dS, kb2=\E[E, kbs=^H, kcbt=\E[Z, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\177, kend=\E[F,
	kf1=\E[M, kf10=\E[V, kf11=\E[W, kf12=\E[X, kf13=\E[Y,
	kf14=\E[Z, kf15=\E[a, kf16=\E[b, kf17=\E[c, kf18=\E[d,
	kf19=\E[e, kf2=\E[N, kf20=\E[f, kf21=\E[g, kf22=\E[h,
	kf23=\E[i, kf24=\E[j, kf25=\E[k, kf26=\E[l, kf27=\E[m,
	kf28=\E[n, kf29=\E[o, kf3=\E[O, kf30=\E[p, kf31=\E[q,
	kf32=\E[r, kf33=\E[s, kf34=\E[t, kf35=\E[u, kf36=\E[v,
	kf37=\E[w, kf38=\E[x, kf39=\E[y, kf4=\E[P, kf40=\E[z,
	kf41=\E[@, kf42=\E[[, kf43=\E[\\, kf44=\E[], kf45=\E[\^,
	kf46=\E[_, kf47=\E[`, kf48=\E[{, kf5=\E[Q, kf6=\E[R,
	kf7=\E[S, kf8=\E[T, kf9=\E[U, khome=\E[H, kich1=\E[L,
	knp=\E[G, kpp=\E[I, nel=\E[E, op=\E[x, rc=\E8, rev=\E[7m,
	ri=\E[T, rin=\E[%p1%dT, rmso=\E[m, rs2=\E[x\E[m\Ec, sc=\E7,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr=\E[0%?%p1%t;2;7%;%?%p3%t;7%;%?%p4%t;5%;%?%p5%t;30;1%;%?%p6%t;1%;m,
	sgr0=\E[m, smso=\E[7m, vpa=\E[%i%p1%dd,
cons25|ansis|ansi80x25|freebsd console (25-line ansi mode),
	acsc=-\030.^Y0\333`\004a\260f\370g\361h\261i\025j\331k\277l\332m\300n\305q\304t\303u\264v\301w\302x\263y\363z\362~\371,
	use=cons25w,
cons25-debian|freebsd console with debian backspace (25-line ansi mode),
	kbs=\177, kdch1=\E[3~, use=cons25,
cons25-m|ansis-mono|ansi80x25-mono|freebsd console (25-line mono ansi mode),
	colors@, pairs@,
	bold@, dim@, op@, rmul=\E[m, setab@, setaf@,
	sgr=\E[0%?%p1%t;2;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;m,
	smul=\E[4m, use=cons25,
cons30|ansi80x30|freebsd console (30-line ansi mode),
	lines#30, use=cons25,
cons30-m|ansi80x30-mono|freebsd console (30-line mono ansi mode),
	lines#30, use=cons25-m,
cons43|ansi80x43|freebsd console (43-line ansi mode),
	lines#43, use=cons25,
cons43-m|ansi80x43-mono|freebsd console (43-line mono ansi mode),
	lines#43, use=cons25-m,
cons50|ansil|ansi80x50|freebsd console (50-line ansi mode),
	lines#50, use=cons25,
cons50-m|ansil-mono|ansi80x50-mono|freebsd console (50-line mono ansi mode),
	lines#50, use=cons25-m,
cons60|ansi80x60|freebsd console (60-line ansi mode),
	lines#60, use=cons25,
cons60-m|ansi80x60-mono|freebsd console (60-line mono ansi mode),
	lines#60, use=cons25-m,
cons25r|pc3r|ibmpc3r|cons25-koi8-r|freebsd console w/koi8-r cyrillic,
	acsc=-\030.^Y0\215`\004a\220f\234h\221i\025j\205k\203l\202m\204n\212q\0t\206u\207v\211w\210x\201y\230z\231~\225,
	use=cons25w,
cons25r-m|pc3r-m|ibmpc3r-mono|cons25-koi8r-m|freebsd console w/koi8-r cyrillic (mono),
	colors@, pairs@,
	op@, rmul=\E[m, setab@, setaf@,
	sgr=\E[0%?%p1%t;2;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p5%t;30;1%;%?%p6%t;1%;m,
	smul=\E[4m, use=cons25r,
cons50r|cons50-koi8r|freebsd console w/koi8-r cyrillic (50 lines),
	lines#50, use=cons25r,
cons50r-m|cons50-koi8r-m|freebsd console w/koi8-r cyrillic (50-line mono),
	lines#50, use=cons25r-m,
cons60r|cons60-koi8r|freebsd console w/koi8-r cyrillic (60 lines),
	lines#60, use=cons25r,
cons60r-m|cons60-koi8r-m|freebsd console w/koi8-r cyrillic (60-line mono),
	lines#60, use=cons25r-m,
# ISO 8859-1 FreeBSD console
cons25l1|cons25-iso8859|freebsd console w/iso 8859-1 chars,
	acsc=+\253\,\273-\030.\031`\201a\202f\207g\210i\247j\213k\214l\215m\216n\217o\220p\221q\222r\223s\224t\225u\226v\227w\230x\231y\232z\233~\237,
	use=cons25w,
cons25l1-m|cons25-iso-m|freebsd console w/iso 8859-1 chars (mono),
	colors@, pairs@,
	bold@, dim@, op@, rmul=\E[m, setab@, setaf@,
	sgr=\E[0%?%p1%t;2;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;m,
	smul=\E[4m, use=cons25l1,
cons50l1|cons50-iso8859|freebsd console w/iso 8859-1 chars (50 lines),
	lines#50, use=cons25l1,
cons50l1-m|cons50-iso-m|freebsd console w/iso 8859-1 chars (50-line mono),
	lines#50, use=cons25l1-m,
cons60l1|cons60-iso|freebsd console w/iso 8859-1 chars (60 lines),
	lines#60, use=cons25l1,
cons60l1-m|cons60-iso-m|freebsd console w/iso 8859-1 chars (60-line mono),
	lines#60, use=cons25l1-m,

#### 386BSD and BSD/OS Consoles
#

# This was the original 386BSD console entry (I think).
# Some places it's named oldpc3|oldibmpc3.
# From: Alex R.N. Wetmore <aw2t@andrew.cmu.edu>
origpc3|origibmpc3|IBM PC 386BSD Console,
	OTbs, am, bw, eo, xon,
	cols#80, lines#25,
	acsc=j\331k\277l\332m\300n\305q\304t\303u\264v\301w\302x\263,
	bold=\E[7m, clear=\Ec, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%2d;%p2%2dH, cuu1=\E[A, ed=\E[J, el=\E[K,
	home=\E[H, ind=\E[S, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, khome=\E[Y, ri=\E[T, rmso=\E[1;0x\E[2;7x,
	rmul=\E[1;0x\E[2;7x, sgr0=\E[m\E[1;0x\E[2;7x,
	smso=\E[1;7x\E[2;0x, smul=\E[1;7x\E[2;0x,

# description of BSD/386 console emulator in version 1.0 (supplied by BSDI)
oldpc3|oldibmpc3|old IBM PC BSD/386 Console,
	OTbs, km,
	lines#25,
	bel=^G, bold=\E[=15F, cr=^M, cud1=^J, dim=\E[=8F, dl1=\E[M,
	ht=^I, il1=\E[L, ind=^J, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, khome=\E[H, kich1=\E[L, kll=\E[F,
	knp=\E[G, kpp=\E[I, nel=^M^J, sgr0=\E[=R,

# Description of BSD/OS console emulator in version 1.1, 2.0, 2.1
# Note, the emulator supports many of the additional console features
# listed in the iBCS2 (e.g. character-set selection) though not all
# are described here.  This entry really ought to be upgraded.
# Also note, the console will also work with fewer lines after doing
# "stty rows NN", e.g. to use 24 lines.
# (Color support from Kevin Rosenberg <kevin@cyberport.com>, 2 May 1996)
# Bug: The <op> capability resets attributes.
bsdos-pc|IBM PC BSD/OS Console,
	sgr=\E[0;10%?%p1%t;7%;%?%p2%t;1%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;%?%p7%t;8%;%?%p9%t;11%;m,
	use=bsdos-pc-nobold,

bsdos-pc-nobold|BSD/OS PC console w/o bold,
	use=klone+color, use=bsdos-pc-m,

bsdos-pc-m|bsdos-pc-mono|BSD/OS PC console mono,
	OTbs, am, eo, km, xon,
	cols#80, it#8, lines#25,
	bel=^G, clear=\Ec, cr=^M, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I,
	il=\E[%p1%dL, il1=\E[L, ind=^J, kbs=^H, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, khome=\E[H, kich1=\E[L,
	kll=\E[F, knp=\E[G, kpp=\E[I, nel=^M^J, rc=\E8, sc=\E7,
	sgr=\E[0;10%?%p1%t;7%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;%?%p7%t;8%;%?%p9%t;11%;m%?%p5%t\E[=8F%;,
	use=klone+sgr8,

# Old names for BSD/OS PC console used in releases before 4.1.
pc3|BSD/OS on the PC Console,
	use=bsdos-pc-nobold,
ibmpc3|pc3-bold|BSD/OS on the PC Console with bold instead of underline,
	use=bsdos-pc,

# BSD/OS on the SPARC
bsdos-sparc|Sun SPARC BSD/OS Console,
	use=sun,

# BSD/OS on the PowerPC
bsdos-ppc|PowerPC BSD/OS Console,
	use=bsdos-pc,


#### DEC VT52
# (<acsc>/<rmacs>/<smacs> capabilities aren't in DEC's official entry -- esr)
#
# Actually (TD pointed this out at the time the acsc string was added):
# vt52 shouldn't define full acsc since most of the cells don't match.
# see vt100 manual page A-31.  This is the list that does match:
#	f degree
#	g plus/minus
#	h right-arrow
#	k down-arrow
#	m scan-1
#	o scan-3
#	q scan-5
#	s scan-7
# The line-drawing happens to work in several terminal emulators, but should
# not be used as a guide to the capabilities of the vt52.  Note in particular
# that vt52 does not support line-drawing characters (the scan-X values refer
# to a crude plotting feature) -TD
vt52|dec vt52,
	OTbs,
	cols#80, it#8, lines#24,
	acsc=+h.k0affggolpnqprrss, bel=^G, clear=\EH\EJ, cr=^M,
	cub1=\ED, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, ed=\EJ,
	el=\EK, home=\EH, ht=^I, ind=^J, kbs=^H, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, nel=^M^J, ri=\EI, rmacs=\EG, smacs=\EF,

#### DEC VT100 and compatibles
#
# DEC terminals from the vt100 forward are collected here. Older DEC terminals
# and micro consoles can be found in the `obsolete' section.  More details on
# the relationship between the VT100 and ANSI X3.64/ISO 6429/ECMA-48 may be
# found near the end of this file.
#
# Except where noted, these entries are DEC's official terminfos.
# Contact Bill Hedberg <hedberg@hannah.enet.dec.com> of Terminal Support
# Engineering for more information.  Updated terminfos and termcaps
# are kept available at ftp://gatekeeper.dec.com/pub/DEC/termcaps.
#
# In October 1995 DEC sold its terminals business, including the VT and Dorio
# line and trademark, to SunRiver Data Systems.  SunRiver has since changed
# its name to Boundless Technologies; see http://www.boundless.com.
#

# NOTE:  Any VT100 emulation, whether in hardware or software, almost
# certainly includes what DEC called the `Level 1 editing extension' codes;
# only the very oldest VT100s lacked these and there probably aren't any of
# those left alive.  To capture these, use one of the VT102 entries.
#
# Note that the <xenl> glitch in vt100 is not quite the same as on the Concept,
# since the cursor is left in a different position while in the
# weird state (concept at beginning of next line, vt100 at end
# of this line) so all versions of vi before 3.7 don't handle
# <xenl> right on vt100. The correct way to handle <xenl> is when
# you output the char in column 80, immediately output CR LF
# and then assume you are in column 1 of the next line. If <xenl>
# is on, am should be on too.
#
# I assume you have smooth scroll off or are at a slow enough baud
# rate that it doesn't matter (1200? or less). Also this assumes
# that you set auto-nl to "on", if you set it off use vt100-nam
# below.
#
# The padding requirements listed here are guesses. It is strongly
# recommended that xon/xoff be enabled, as this is assumed here.
#
# The vt100 uses <rs2> and <rf> rather than <is2>/<tbc>/<hts> because the
# tab settings are in non-volatile memory and don't need to be
# reset upon login. Also setting the number of columns glitches
# the screen annoyingly. You can type "reset" to get them set.
#
# The VT100 series terminals have cursor ("arrows") keys which can operate
# in two different modes: Cursor Mode and Application Mode.  Cursor Mode
# is the reset state, and is assumed to be the normal state.  Application
# Mode is the "set" state.  In Cursor Mode, the cursor keys transmit
# "Esc [ {code}" sequences, conforming to ANSI standards.  In Application
# Mode, the cursor keys transmit "Esc O <code>" sequences.  Application Mode
# was provided primarily as an aid to the porting of VT52 applications.  It is
# assumed that the cursor keys are normally in Cursor Mode, and expected that
# applications such as vi will always transmit the <smkx> string.  Therefore,
# the definitions for the cursor keys are made to match what the terminal
# transmits after the <smkx> string is transmitted.  If the <smkx> string
# is a null string or is not defined, then cursor keys are assumed to be in
# "Cursor Mode", and the cursor keys definitions should match that assumption,
# else the application may fail.  It is also expected that applications will
# always transmit the <rmkx> string to the terminal before they exit.
#
# The VT100 series terminals have an auxiliary keypad, commonly referred to as
# the "Numeric Keypad", because it is a cluster of numeric and function keys.
# The Numeric Keypad which can operate in two different modes: Numeric Mode and
# Application Mode.  Numeric Mode is the reset state, and is assumed to be
# the normal state.  Application Mode is the "set" state.  In Numeric Mode,
# the numeric and punctuation keys transmit ASCII 7-bit characters, and the
# Enter key transmits the same as the Return key (Note: the Return key
# can be configured to send either LF (\015) or CR LF).  In Application Mode,
# all the keypad keys transmit "Esc O {code}" sequences.  The PF1 - PF4 keys
# always send the same "Esc O {code}" sequences.  It is assumed that the keypad
# is normally in Numeric Mode.  If an application requires that the keypad be
# in Application Mode then it is expected that the user, or the application,
# will set the TERM environment variable to point to a terminfo entry which has
# defined the <smkx> string to include the codes that switch the keypad into
# Application Mode, and the terminfo entry will also define function key
# fields to match the Application Mode control codes.  If the <smkx> string
# is a null string or is not defined, then the keypad is assumed to be in
# Numeric Mode.  If the <smkx> string switches the keypad into Application
# Mode, it is expected that the <rmkx> string will contain the control codes
# necessary to reset the keypad to "Normal" mode, and it is also expected that
# applications which transmit the <smkx> string will also always transmit the
# <rmkx> string to the terminal before they exit.
#
# Here's a diagram of the VT100 keypad keys with their bindings.
# The top line is the name of the key (some DEC keyboards have the keys
# labelled somewhat differently, like GOLD instead of PF1, but this is
# the most "official" name).  The second line is the escape sequence it
# generates in Application Keypad mode (where "$" means the ESC
# character).  The third line contains two items, first the mapping of
# the key in terminfo, and then in termcap.
#   _______________________________________
#  |   PF1   |   PF2   |   PF3   |   PF4   |
#  |   $OP   |   $OQ   |   $OR   |   $OS   |
#  |_kf1__k1_|_kf2__k2_|_kf3__k3_|_kf4__k4_|
#  |    7         8         9         -    |
#  |   $Ow   |   $Ox   |   $Oy   |   $Om   |
#  |_kf9__k9_|_kf10_k;_|_kf0__k0_|_________|
#  |    4    |    5    |    6    |    ,    |
#  |   $Ot   |   $Ou   |   $Ov   |   $Ol   |
#  |_kf5__k5_|_kf6__k6_|_kf7__k7_|_kf8__k8_|
#  |    1    |    2    |    3    |         |
#  |   $Oq   |   $Or   |   $Os   |  enter  |
#  |_ka1__K1_|_kb2__K2_|_ka3__K3_|  $OM    |
#  |         0         |   .     |         |
#  |        $Op        |  $On    |         |
#  |___kc1_______K4____|_kc3__K5_|_kent_@8_|
#
# Note however, that the arrangement of the 5-key ka1-kc3 do not follow the
# terminfo guidelines.  That is a compromise used to assign the remaining
# keys on the keypad to kf5-kf0, used on older systems with legacy termcap
# support:
vt100+keypad|dec vt100 numeric keypad no fkeys,
	ka1=\EOq, ka3=\EOs, kb2=\EOr, kc1=\EOp, kc3=\EOn,
vt100+pfkeys|dec vt100 numeric keypad,
	kent=\EOM, kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS,
	use=vt100+keypad,
vt100+fnkeys|dec vt100 numeric keypad,
	kf0=\EOy, kf10=\EOx, kf5=\EOt, kf6=\EOu, kf7=\EOv, kf8=\EOl,
	kf9=\EOw, use=vt100+pfkeys,
#
# A better adaptation to modern keyboards such as the PC's, which have a dozen
# function keys and the keypad 2,4,6,8 keys are labeled with arrows keys, is to
# use the 5-key arrangement to model the arrow keys as suggested in the
# terminfo guidelines:
#   _______________________________________
#  |   PF1   |   PF2   |   PF3   |   PF4   |
#  |   $OP   |   $OQ   |   $OR   |   $OS   |
#  |_kf1__k1_|_kf2__k2_|_kf3__k3_|_kf4__k4_|
#  |    7         8         9         -    |
#  |   $Ow   |   $Ox   |   $Oy   |   $Om   |
#  |_ka1__K1_|_________|_ka3__K3_|_________|
#  |    4    |    5    |    6    |    ,    |
#  |   $Ot   |   $Ou   |   $Ov   |   $Ol   |
#  |_________|_kb2__K2_|_________|_________|
#  |    1    |    2    |    3    |         |
#  |   $Oq   |   $Or   |   $Os   |  enter  |
#  |_kc1__K4_|_________|_kc3__K5_|  $OM    |
#  |         0         |   .     |         |
#  |        $Op        |  $On    |         |
#  |___________________|_________|_kent_@8_|
#
vt220+keypad|dec vt220 numeric keypad,
	ka1=\EOw, ka3=\EOy, kb2=\EOu, kc1=\EOq, kc3=\EOs, kent=\EOM,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, ka2=\EOx, kb1=\EOt,
	kb3=\EOv, kc2=\EOr,
#
vt100+enq|ncurses extension for vt100-style ENQ,
	u8=\E[?1;2c, use=ansi+enq,
vt102+enq|ncurses extension for vt102-style ENQ,
	u8=\E[?6c, use=ansi+enq,
#
# And here, for those of you with orphaned VT100s lacking documentation, is
# a description of the soft switches invoked when you do `Set Up'.
#
#  Scroll 0-Jump               Shifted 3   0-#
#  |      1-Smooth             |           1-British pound sign
#  | Autorepeat 0-Off          | Wrap Around 0-Off
#  | |          1-On           | |           1-On
#  | | Screen 0-Dark Bkg       | | New Line 0-Off
#  | | |      1-Light Bkg      | | |        1-On
#  | | | Cursor 0-Underline    | | | Interlace 0-Off
#  | | | |      1-Block        | | | |         1-On
#  | | | |                     | | | |
#  1 1 0 1       1 1 1 1       0 1 0 0       0 0 1 0       <--Standard Settings
#                | | | |                     | | | |
#                | | | Auto XON/XOFF 0-Off   | | | Power 0-60 Hz
#                | | |               1-On    | | |       1-50 Hz
#                | | Ansi/VT52 0-VT52        | | Bits Per Char. 0-7 Bits
#                | |           1-ANSI        | |                1-8 Bits
#                | Keyclick 0-Off            | Parity 0-Off
#                |          1-On             |        1-On
#                Margin Bell 0-Off           Parity Sense 0-Odd
#                            1-On                         1-Even
#
# The following SET-UP modes are assumed for normal operation:
#	ANSI_MODE	AUTO_XON/XOFF_ON	NEWLINE_OFF	80_COLUMNS
#	WRAP_AROUND_ON  JUMP_SCROLL_OFF
# Other SET-UP modes may be set for operator convenience or communication
# requirements; I recommend
#	AUTOREPEAT_ON	BLOCK_CURSOR	MARGIN_BELL_OFF    SHIFTED_3_#
# Unless you have a graphics add-on such as Digital Engineering's VT640
# (and even then, whenever it can be arranged!) you should set
#	INTERLACE_OFF
#
# (vt100: I added <rmam>/<smam> based on the init string, also <OTbs>. -- esr)
vt100|vt100-am|dec vt100 (w/advanced video),
	OTbs, am, mc5i, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m$<2>, bold=\E[1m$<2>,
	clear=\E[H\E[J$<50>, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C$<2>,
	cup=\E[%i%p1%d;%p2%dH$<5>, cuu=\E[%p1%dA,
	cuu1=\E[A$<2>, ed=\E[J$<50>, el=\E[K$<3>, el1=\E[1K$<3>,
	enacs=\E(B\E)0, home=\E[H, ht=^I, hts=\EH, ind=^J, kbs=^H,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, lf1=pf1,
	lf2=pf2, lf3=pf3, lf4=pf4, mc0=\E[0i, mc4=\E[4i, mc5=\E[5i,
	rc=\E8, rev=\E[7m$<2>, ri=\EM$<5>, rmacs=^O, rmam=\E[?7l,
	rmkx=\E[?1l\E>, rmso=\E[m$<2>, rmul=\E[m$<2>,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;$<2>,
	sgr0=\E[m\017$<2>, smacs=^N, smam=\E[?7h, smkx=\E[?1h\E=,
	smso=\E[7m$<2>, smul=\E[4m$<2>, tbc=\E[3g,
	use=vt100+fnkeys,
vt100nam|vt100-nam|vt100 no automargins,
	am@, xenl@, use=vt100-am,
vt100-vb|dec vt100 (w/advanced video) & no beep,
	bel@, flash=\E[?5h\E[?5l, use=vt100,

# Ordinary vt100 in 132 column ("wide") mode.
vt100-w|vt100-w-am|dec vt100 132 cols (w/advanced video),
	cols#132, lines#24,
	rs2=\E>\E[?3h\E[?4l\E[?5l\E[?8h, use=vt100-am,
vt100-w-nam|vt100-nam-w|dec vt100 132 cols (w/advanced video no automargin),
	cols#132, lines#14, vt@,
	rs2=\E>\E[?3h\E[?4l\E[?5l\E[?8h, use=vt100-nam,

# vt100 with no advanced video.
vt100-nav|vt100 without advanced video option,
	xmc#1,
	blink@, bold@, rev@, rmso=\E[m, rmul@, sgr@, sgr0@, smso=\E[7m,
	smul@, use=vt100,
vt100-nav-w|vt100-w-nav|dec vt100 132 cols 14 lines (no advanced video option),
	cols#132, lines#14, use=vt100-nav,

# vt100 with one of the 24 lines used as a status line.
# We put the status line on the top.
vt100-s|vt100-s-top|vt100-top-s|vt100 for use with top sysline,
	eslok, hs,
	lines#23,
	clear=\E[2;1H\E[J$<50>, csr=\E[%i%i%p1%d;%p2%dr,
	cup=\E[%i%p1%{1}%+%d;%p2%dH$<5>, dsl=\E7\E[1;24r\E8,
	fsl=\E8, home=\E[2;1H, is2=\E7\E[2;24r\E8,
	tsl=\E7\E[1;%p1%dH\E[1K, use=vt100-am,

# Status line at bottom.
# Clearing the screen will clobber status line.
vt100-s-bot|vt100-bot-s|vt100 for use with bottom sysline,
	eslok, hs,
	lines#23,
	dsl=\E7\E[1;24r\E8, fsl=\E8, is2=\E[1;23r\E[23;1H,
	tsl=\E7\E[24;%p1%dH\E[1K, use=vt100-am,

# Most of the `vt100' emulators out there actually emulate a vt102
# This entry (or vt102-nsgr) is probably the right thing to use for
# these.
vt102|dec vt102,
	dch1=\E[P, dl1=\E[M, il1=\E[L, rmir=\E[4l, smir=\E[4h,
	use=vt100,
vt102-w|dec vt102 in wide mode,
	cols#132,
	rs3=\E[?3h, use=vt102,

# Many brain-dead PC comm programs that pretend to be `vt100-compatible'
# fail to interpret the ^O and ^N escapes properly.  Symptom: the <sgr0>
# string in the canonical vt100 entry above leaves the screen littered
# with little  snowflake or star characters (IBM PC ROM character \017 = ^O)
# after highlight turnoffs.  This entry should fix that, and even leave
# ACS support working, at the cost of making multiple-highlight changes
# slightly more expensive.
# From: Eric S. Raymond <esr@snark.thyrsus.com> July 22 1995
vt102-nsgr|vt102 no sgr (use if you see snowflakes after highlight changes),
	sgr@, sgr0=\E[m, use=vt102,

# VT125 Graphics CRT.  Clear screen also erases graphics
# Some vt125's came configured with vt102 support.
vt125|vt125 graphics terminal,
	mir,
	clear=\E[H\E[2J\EPpS(E)\E\\$<50>, use=vt100,

# This isn't a DEC entry, it came from University of Wisconsin.
# (vt131: I added <rmam>/<smam> based on the init string, also <OTbs> -- esr)
vt131|dec vt131,
	OTbs, am, xenl,
	cols#80, it#8, lines#24, vt#3,
	bel=^G, blink=\E[5m$<2/>, bold=\E[1m$<2/>,
	clear=\E[;H\E[2J$<50/>, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub1=^H, cud1=^J, cuf1=\E[C$<2/>,
	cup=\E[%i%p1%d;%p2%dH$<5/>, cuu1=\E[A$<2/>,
	ed=\E[J$<50/>, el=\E[K$<3/>, home=\E[H, ht=^I,
	is2=\E[1;24r\E[24;1H, kbs=^H, kcub1=\EOD, kcud1=\EOB,
	kcuf1=\EOC, kcuu1=\EOA, kf1=\EOP, kf2=\EOQ, kf3=\EOR,
	kf4=\EOS, nel=^M^J, rc=\E8, rev=\E[7m$<2/>, ri=\EM$<5/>,
	rmam=\E[?7h, rmkx=\E[?1l\E>, rmso=\E[m$<2/>,
	rmul=\E[m$<2/>,
	rs1=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr0=\E[m$<2/>, smam=\E[?7h, smkx=\E[?1h\E=,
	smso=\E[7m$<2/>, smul=\E[4m$<2/>,

# vt132 - like vt100 but slower and has ins/del line and such.
# I'm told that <smir>/<rmir> are backwards in the terminal from the
# manual and from the ANSI standard, this describes the actual
# terminal. I've never actually used a vt132 myself, so this
# is untested.
#
vt132|DEC vt132,
	xenl,
	dch1=\E[P$<7>, dl1=\E[M$<99>, il1=\E[L$<99>, ind=\n$<30>,
	ip=$<7>, rmir=\E[4h, smir=\E[4l, use=vt100,

# This vt220 description maps F5--F9 to the second block of function keys
# at the top of the keyboard.  The "DO" key is used as F10 to avoid conflict
# with the key marked (ESC) on the vt220.  See vt220d for an alternate mapping.
# PF1--PF4 are used as F1--F4.
#
# added msgr -TD
vt220-old|vt200-old|DEC VT220 in vt100 emulation mode,
	OTbs, OTpt, am, mir, msgr, xenl, xon,
	cols#80, lines#24, vt#3,
	OTnl=^J,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m$<2>, bold=\E[1m$<2>, civis=\E[?25l,
	clear=\E[H\E[2J$<50>, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH$<10>, cuu1=\E[A, dch1=\E[P,
	dl1=\E[M, ed=\E[J$<50>, el=\E[K$<3>, home=\E[H, ht=^I,
	if=/usr/share/tabset/vt100, il1=\E[L, ind=\ED$<20/>,
	is2=\E[1;24r\E[24;1H, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~, kend=\E[4~, kf1=\EOP,
	kf10=\E[29~, kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf5=\E[17~,
	kf6=\E[18~, kf7=\E[19~, kf8=\E[20~, kf9=\E[21~,
	khome=\E[1~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~, rc=\E8,
	rev=\E[7m$<2>, rf=/usr/share/tabset/vt100,
	ri=\EM$<14/>, rmacs=\E(B$<4>, rmam=\E[?7l, rmir=\E[4l,
	rmso=\E[27m, rmul=\E[24m,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p4%t;5%;%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;$<2>,
	sgr0=\E[m$<2>, smacs=\E(0$<2>, smam=\E[?7h, smir=\E[4h,
	smso=\E[7m, smul=\E[4m,

# A much better description of the VT200/220; used to be vt220-8
# changed rmacs/smacs from shift-in/shift-out to vt200-old's explicit G0/G1
# designation to accommodate bug in pcvt -TD
#
# Here's a picture of the VT220 editing keypad:
#	+--------+--------+--------+
#	| Find   | Insert | Remove |
#	+--------+--------+--------+
#	| Select | Prev   | Next   |
#	+--------+--------+--------+
vt220|vt200|dec vt220,
	OTbs, am, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K, enacs=\E)0,
	flash=\E[?5h$<200/>\E[?5l, home=\E[H, ht=^I, hts=\EH,
	ich=\E[%p1%d@, if=/usr/share/tabset/vt100,
	il=\E[%p1%dL, il1=\E[L, ind=\ED,
	is2=\E[?7h\E[>\E[?1l\E F\E[?4l, kbs=^H, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~, kf1=\EOP,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf13=\E[25~,
	kf14=\E[26~, kf17=\E[31~, kf18=\E[32~, kf19=\E[33~,
	kf2=\EOQ, kf20=\E[34~, kf3=\EOR, kf4=\EOS, kf6=\E[17~,
	kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, kfnd=\E[1~,
	khlp=\E[28~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~,
	krdo=\E[29~, kslt=\E[4~, lf1=pf1, lf2=pf2, lf3=pf3, lf4=pf4,
	nel=\EE, rc=\E8, rev=\E[7m, ri=\EM, rmacs=\E(B$<4>,
	rmam=\E[?7l, rmir=\E[4l, rmso=\E[27m, rmul=\E[24m,
	rs1=\E[?3l, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p4%t;5%;%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;$<2>,
	sgr0=\E[m\E(B, smacs=\E(0$<2>, smam=\E[?7h, smir=\E[4h,
	smso=\E[7m, smul=\E[4m, tbc=\E[3g, use=ansi+pp,
	use=ansi+enq,
vt220-w|vt200-w|DEC vt220 in wide mode,
	cols#132,
	rs3=\E[?3h, use=vt220,
vt220-8bit|vt220-8|vt200-8bit|vt200-8|dec vt220/200 in 8-bit mode,
	OTbs, am, mc5i, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\2335m, bold=\2331m, clear=\233H\233J, cr=^M,
	csr=\233%i%p1%d;%p2%dr, cub=\233%p1%dD, cub1=^H,
	cud=\233%p1%dB, cud1=^J, cuf=\233%p1%dC, cuf1=\233C,
	cup=\233%i%p1%d;%p2%dH, cuu=\233%p1%dA, cuu1=\233A,
	dch=\233%p1%dP, dch1=\233P, dl=\233%p1%dM, dl1=\233M,
	ech=\233%p1%dX, ed=\233J, el=\233K, el1=\2331K, enacs=\E)0,
	flash=\233?5h$<200/>\233?5l, home=\233H, ht=^I, hts=\EH,
	ich=\233%p1%d@, if=/usr/share/tabset/vt100,
	il=\233%p1%dL, il1=\233L, ind=\ED,
	is2=\233?7h\233>\233?1l\E F\233?4l, kbs=^H,
	kcub1=\233D, kcud1=\233B, kcuf1=\233C, kcuu1=\233A,
	kdch1=\2333~, kf1=\EOP, kf10=\23321~, kf11=\23323~,
	kf12=\23324~, kf13=\23325~, kf14=\23326~, kf17=\23331~,
	kf18=\23332~, kf19=\23333~, kf2=\EOQ, kf20=\23334~,
	kf3=\EOR, kf4=\EOS, kf6=\23317~, kf7=\23318~, kf8=\23319~,
	kf9=\23320~, kfnd=\2331~, khlp=\23328~, khome=\233H,
	kich1=\2332~, knp=\2336~, kpp=\2335~, krdo=\23329~,
	kslt=\2334~, lf1=pf1, lf2=pf2, lf3=pf3, lf4=pf4, mc0=\233i,
	mc4=\2334i, mc5=\2335i, nel=\EE, rc=\E8, rev=\2337m, ri=\EM,
	rmacs=\E(B, rmam=\233?7l, rmir=\2334l, rmso=\23327m,
	rmul=\23324m, rs1=\233?3l, sc=\E7,
	sgr=\2330%?%p6%t;1%;%?%p2%t;4%;%?%p4%t;5%;%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;$<2>,
	sgr0=\2330m\E(B, smacs=\E(0, smam=\233?7h, smir=\2334h,
	smso=\2337m, smul=\2334m, tbc=\2333g,

# vt220d:
# This vt220 description regards F6--F10 as the second block of function keys
# at the top of the keyboard.  This mapping follows the description given
# in the VT220 Programmer Reference Manual and agrees with the labeling
# on some terminals that emulate the vt220.  There is no support for an F5.
# See vt220 for an alternate mapping.
#
vt220d|DEC VT220 in vt100 mode with DEC function key labeling,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf13=\E[25~,
	kf14=\E[26~, kf15=\E[28~, kf16=\E[29~, kf17=\E[31~,
	kf18=\E[32~, kf19=\E[33~, kf20=\E[34~, kf5@, kf6=\E[17~,
	kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, use=vt220-old,

vt220-nam|v200-nam|VT220 in vt100 mode with no auto margins,
	am@,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7l\E[?8h, use=vt220,

# vt220 termcap written Tue Oct 25 20:41:10 1988 by Alex Latzko
# (not an official DEC entry!)
# The problem with real vt220 terminals is they don't send escapes when in
# in vt220 mode.  This can be gotten around two ways.  1> don't send
# escapes or 2> put the vt220 into vt100 mode and use all the nifty
# features of vt100 advanced video which it then has.
#
# This entry takes the view of putting a vt220 into vt100 mode so
# you can use the escape key in emacs and everything else which needs it.
#
# You probably don't want to use this on a VMS machine since VMS will think
# it has a vt220 and will get fouled up coming out of emacs
#
# From: Alexander Latzko <latzko@marsenius.rutgers.edu>, 30 Dec 1996
# (Added vt100 <rc>,<sc> to quiet a tic warning -- esr)
# added msgr -TD
vt200-js|vt220-js|dec vt200 series with jump scroll,
	am, msgr,
	cols#80,
	bel=^G, clear=\E[H\E[J, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub1=^H, cud1=^J, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu1=\E[A, dch1=\E[P, dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H,
	ht=^I, il1=\E[L, ind=\ED,
	is2=\E[61"p\E[H\E[?3l\E[?4l\E[?1h\E[?5l\E[?6l\E[?7h\E[?8h\E[?25h\E>\E[m,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, nel=^M\ED, rc=\E8,
	rf=/usr/share/tabset/vt100, ri=\EM, rmdc=, rmir=\E[4l,
	rmkx=\E[?1l\E>, rmso=\E[27m$<5/>, rmul=\E[24m,
	rs1=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7, smdc=,
	smir=\E[4h, smkx=\E[?1h\E=, smso=\E[7m$<5/>, smul=\E[4m,

# This was DEC's vt320.  Use the purpose-built one below instead
#vt320|DEC VT320 in vt100 emulation mode,
#	use=vt220,

# Use v320n for SCO's LYRIX.  Otherwise, use Adam Thompson's vt320-nam.
#
vt320nam|v320n|DEC VT320 in vt100 emul. mode with NO AUTO WRAP mode,
	am@,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7l\E[?8h, use=vt220,

# These entries are not DEC's official ones, they were purpose-built for the
# VT320.  Here are the designer's notes:
# <kel> is end on a PC kbd.  Actually 'select' on a VT.  Mapped to
# 'Erase to End of Field'... since nothing seems to use 'end' anyways...
# khome is Home on a PC kbd.  Actually 'FIND' on a VT.
# Things that use <knxt> usually use tab anyways... and things that don't use
# tab usually use <knxt> instead...
# kprv is same as tab - Backtab is useless...
# I left out <sgr> because of its RIDICULOUS complexity,
# and the resulting fact that it causes the termcap translation of the entry
# to SMASH the 1k-barrier...
# From: Adam Thompson <athompso@pangea.ca> Sept 10 1995
# (vt320: uncommented <fsl> --esr)
vt320|vt300|dec vt320 7 bit terminal,
	am, hs, mir, msgr, xenl,
	cols#80, lines#24, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[2J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K, home=\E[H, ht=^I,
	hts=\EH, ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L, ind=\ED,
	is2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h\E[1;24r\E[24;1H,
	kbs=\177, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kdch1=\E[3~, kel=\E[4~, kf10=\E[21~, kf11=\E[23~,
	kf12=\E[24~, kf13=\E[25~, kf14=\E[26~, kf15=\E[28~,
	kf16=\E[29~, kf17=\E[31~, kf18=\E[32~, kf19=\E[33~,
	kf20=\E[34~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~,
	kf9=\E[20~, khome=\E[1~, kich1=\E[2~, knp=\E[6~, knxt=^I,
	kpp=\E[5~, kprv=\E[Z, kslt=\E[4~, nel=\EE, rc=\E8, rev=\E[7m,
	rf=/usr/share/tabset/vt300, ri=\EM, rmacs=\E(B,
	rmam=\E[?7l, rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[m,
	rmul=\E[m,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h\E[1;24r\E[24;1H,
	sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p4%t;5%;%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;$<2>,
	sgr0=\E[m\E(B, smacs=\E(0, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	use=dec+pp, use=vt220+keypad, use=dec+sl, use=ansi+enq,
vt320-nam|vt300-nam|dec vt320 7 bit terminal with no am to make SAS happy,
	am@,
	is2=\E>\E[?3l\E[?4l\E[?5l\E[?7l\E[?8h\E[1;24r\E[24;1H,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7l\E[?8h\E[1;24r\E[24;1H,
	use=vt320,
# We have to init 132-col mode, not 80-col mode.
vt320-w|vt300-w|dec vt320 wide 7 bit terminal,
	cols#132, wsl#132,
	is2=\E>\E[?3h\E[?4l\E[?5l\E[?7h\E[?8h\E[1;24r\E[24;1H,
	rs2=\E>\E[?3h\E[?4l\E[?5l\E[?7h\E[?8h\E[1;24r\E[24;1H,
	use=vt320,
vt320-w-nam|vt300-w-nam|dec vt320 wide 7 bit terminal with no am,
	am@,
	is2=\E>\E[?3h\E[?4l\E[?5l\E[?7l\E[?8h\E[1;24r\E[24;1H,
	rs2=\E>\E[?3h\E[?4l\E[?5l\E[?7l\E[?8h\E[1;24r\E[24;1H,
	use=vt320-w,

# VT330 and VT340 -- These are ReGIS and SIXEL graphics terminals
#   which are pretty much a superset of the VT320.  They have the
#   host writable status line, yet another different DRCS matrix size,
#   and such, but they add the DEC Technical character set, Multiple text
#   pages, selectable length pages, and the like.  The difference between
#   the vt330 and vt340 is that the latter has only 2 planes and a monochrome
#   monitor, the former has 4 planes and a color monitor.  These terminals
#   support VT131 and ANSI block mode, but as with much of these things,
#   termcap/terminfo doesn't deal with these features.
#
# Note that this entry is are set up in what was the standard way for GNU
# Emacs v18 terminal modes to deal with the cursor keys in that the arrow
# keys were switched into application mode at the same time the numeric pad
# is switched into application mode.  This changes the definitions of the
# arrow keys.  Emacs v19 is smarter and mines its keys directly out of
# your termcap or terminfo entry,
#
# From: Daniel Glasser <dag@persoft.persoft.com>, 13 Oct 1993
# (vt340: string capability "sb=\E[M" corrected to "sr";
# also, added <rmam>/<smam> based on the init string -- esr)
vt340|dec-vt340|vt330|dec-vt330|dec vt340 graphics terminal with 24 line page,
	am, eslok, hs, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	blink=\E[5m, bold=\E[1m, civis=\E[?25l, clear=\E[H\E[J,
	cnorm=\E[?25h, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dl=\E[%p1%dM, dl1=\E[M, dsl=\E[2$~\r\E[1$}\E[K\E[$},
	ed=\E[J, el=\E[K, flash=\E[?5h\E[?5l$<200/>, fsl=\E[$},
	home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@, il=\E[%p1%dL,
	il1=\E[L, ind=\ED,
	is2=\E<\E F\E>\E[?1h\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h\E[1;24r\E[24;1H,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf6=\E[17~,
	kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, lf1=pf1, lf2=pf2,
	lf3=pf3, lf4=pf4, nel=^M\ED, rc=\E8, rev=\E[7m,
	rf=/usr/share/tabset/vt300, ri=\EM, rmacs=\E(B,
	rmam=\E[?7l, rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[27m,
	rmul=\E[24m, rs1=\E[?3l, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p4%t;5%;%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;$<2>,
	sgr0=\E[m\E(B, smacs=\E(0, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	tsl=\E[2$~\E[1$}\E[1;%dH,

# DEC doesn't supply a vt400 description, so we add Daniel Glasser's
# (originally written with vt420 as its primary name, and usable for it).
#
# VT400/420 -- This terminal is a superset of the vt320.  It adds the multiple
#    text pages and long text pages with selectable length of the vt340, along
#    with left and right margins, rectangular area text copy, fill, and erase
#    operations, selected region character attribute change operations,
#    page memory and rectangle checksums, insert/delete column, reception
#    macros, and other features too numerous to remember right now.  TERMCAP
#    can only take advantage of a few of these added features.
#
# Note that this entry is are set up in what was the standard way for GNU
# Emacs v18 terminal modes to deal with the cursor keys in that the arrow
# keys were switched into application mode at the same time the numeric pad
# is switched into application mode.  This changes the definitions of the
# arrow keys.  Emacs v19 is smarter and mines its keys directly out of
# your termcap entry,
#
# From: Daniel Glasser <dag@persoft.persoft.com>, 13 Oct 1993
# (vt400: string capability ":sb=\E[M:" corrected to ":sr=\E[M:";
# also, added <rmam>/<smam> based on the init string -- esr)
vt400|vt400-24|dec-vt400|dec vt400 24x80 column autowrap,
	am, eslok, hs, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[J$<10/>, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	dsl=\E[2$~\r\E[1$}\E[K\E[$}, ed=\E[J$<10/>,
	el=\E[K$<4/>, flash=\E[?5h\E[?5l$<200/>, fsl=\E[$},
	home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@, ich1=\E[@,
	il=\E[%p1%dL, il1=\E[L, ind=\ED,
	is2=\E<\E F\E>\E[?1h\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h\E[1;24r\E[24;1H,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf6=\E[17~,
	kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, lf1=pf1, lf2=pf2,
	lf3=pf3, lf4=pf4, nel=^M\ED, rc=\E8, rev=\E[7m,
	rf=/usr/share/tabset/vt300, ri=\EM, rmacs=\E(B,
	rmam=\E[?7l, rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[27m,
	rmul=\E[24m, rs1=\E<\E[?3l\E[!p\E[?7h, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p4%t;5%;%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;$<2>,
	sgr0=\E[m\E(B, smacs=\E(0, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	tsl=\E[2$~\E[1$}\E[1;%dH, use=dec+sl,

# (vt420: I removed <kf0>, it collided with <kf10>.  I also restored
# a missing <sc> -- esr)
# add msgr and other capabilities from vt220 -TD
vt420|DEC VT420,
	am, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m$<2>, bold=\E[1m$<2>, civis=\E[?25l,
	clear=\E[H\E[2J$<50>, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH$<10>, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J$<50>, el=\E[K$<3>, el1=\E[1K,
	enacs=\E)0, flash=\E[?5h$<200/>\E[?5l, home=\E[H, ht=^I,
	hts=\EH, ich=\E[%p1%d@, if=/usr/share/tabset/vt300,
	il=\E[%p1%dL, il1=\E[L, ind=\ED, is2=\E[1;24r\E[24;1H,
	is3=\E[?67h\E[64;1"p, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~, kf1=\EOP, kf10=\E[29~,
	kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf5=\E[17~, kf6=\E[18~,
	kf7=\E[19~, kf8=\E[20~, kf9=\E[21~, kfnd=\E[1~,
	kich1=\E[2~, knp=\E[6~, kpp=\E[5~, kslt=\E[4~, nel=\EE,
	rc=\E8, rev=\E[7m$<2>, rf=/usr/share/tabset/vt300,
	ri=\EM, rmacs=\E(B$<4>, rmam=\E[?7l, rmir=\E[4l, rmkx=\E>,
	rmsc=\E[?0;0r\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h,
	rmso=\E[27m, rmul=\E[24m, rs3=\E[?67h\E[64;1"p, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p4%t;5%;%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;$<2>,
	sgr0=\E[m\E(B$<2>, smacs=\E(0$<2>, smam=\E[?7h,
	smir=\E[4h, smkx=\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	use=ansi+pp, use=dec+sl, use=ansi+enq,

# DEC VT220 and up support DECUDK (user-defined keys).  DECUDK (i.e., pfx)
# takes two parameters, the key and the string.  Translating the key is
# straightforward (keys 1-5 are not defined on real terminals, though some
# emulators define these):
#
#               if (key < 16) then  value = key;
#               else if (key < 21) then value = key + 1;
#               else if (key < 25) then value = key + 2;
#               else if (key < 27) then value = key + 3;
#               else if (key < 30) then value = key + 4;
#               else value = key + 5;
#
# The string must be the hexadecimal equivalent, e.g., "5052494E" for "PRINT".
# There's no provision in terminfo for emitting a string in this format, so the
# application has to know it.
#
vt420pc|DEC VT420 w/PC keyboard,
	kdch1=\177, kend=\E[4~, kf1=\E[11~, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf13=\E[11;2~, kf14=\E[12;2~,
	kf15=\E[13;2~, kf16=\E[14;2~, kf17=\E[15;2~,
	kf18=\E[17;2~, kf19=\E[18;2~, kf2=\E[12~, kf20=\E[19;2~,
	kf21=\E[20;2~, kf22=\E[21;2~, kf23=\E[23;2~,
	kf24=\E[24;2~, kf25=\E[23~, kf26=\E[24~, kf27=\E[25~,
	kf28=\E[26~, kf29=\E[28~, kf3=\E[13~, kf30=\E[29~,
	kf31=\E[31~, kf32=\E[32~, kf33=\E[33~, kf34=\E[34~,
	kf35=\E[35~, kf36=\E[36~, kf37=\E[23;2~, kf38=\E[24;2~,
	kf39=\E[25;2~, kf4=\E[14~, kf40=\E[26;2~, kf41=\E[28;2~,
	kf42=\E[29;2~, kf43=\E[31;2~, kf44=\E[32;2~,
	kf45=\E[33;2~, kf46=\E[34;2~, kf47=\E[35;2~,
	kf48=\E[36;2~, kf5=\E[15~, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, khome=\E[H,
	pctrm=USR_TERM\:vt420pcdos\:,
	pfx=\EP1;1|%?%{16}%p1%>%t%{0}%e%{21}%p1%>%t%{1}%e%{25}%p1%>%t%{2}%e%{27}%p1%>%t%{3}%e%{30}%p1%>%t%{4}%e%{5}%;%p1%+%d/%p2%s\E\\,
	use=vt420,

vt420pcdos|DEC VT420 w/PC for DOS Merge,
	lines#25,
	dispc=%?%p1%{19}%=%t\E\023\021%e%p1%{32}%<%t\E%p1%c%e%p1%{127}%=%t\E\177%e%p1%c%;,
	pctrm@,
	rmsc=\E[?0;0r\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sgr@,
	sgr0=\E[m, smsc=\E[?1;2r\E[34h, use=vt420pc,

vt420f|DEC VT420 with VT kbd; VT400 mode; F1-F5 used as Fkeys,
	kdch1=\177, kf1=\E[11~, kf10=\E[21~, kf11=\E[23~,
	kf12=\E[24~, kf13=\E[25~, kf14=\E[26~, kf15=\E[28~,
	kf16=\E[29~, kf17=\E[31~, kf18=\E[32~, kf19=\E[33~,
	kf2=\E[12~, kf20=\E[34~, kf3=\E[13~, kf4=\E[14~,
	kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[H, lf1=\EOP, lf2=\EOQ, lf3=\EOR, lf4=\EOS,
	use=vt420,

vt510|DEC VT510,
	use=vt420,
vt510pc|DEC VT510 w/PC keyboard,
	use=vt420pc,
vt510pcdos|DEC VT510 w/PC for DOS Merge,
	use=vt420pcdos,

# VT520/VT525
#
# The VT520 is a monochrome text terminal capable of managing up to
# four independent sessions in the terminal.  It has multiple ANSI
# emulations (VT520, VT420, VT320, VT220, VT100, VT PCTerm, SCO Console)
# and ASCII emulations (WY160/60, PCTerm, 50/50+, 150/120, TVI 950,
# 925 910+, ADDS A2).  This terminfo data is for the ANSI emulations only.
#
# Terminal Set-Up is entered by pressing [F3], [Caps Lock]/[F3] or
# [Alt]/[Print Screen] depending upon which keyboard and which
# terminal mode is being used.  If Set-Up has been disabled or
# assigned to an unknown key, Set-Up may be entered by pressing
# [F3] as the first key after power up, regardless of keyboard type.
vt520|DEC VT520,
	use=ansi+rca, use=vt420, use=ansi+tabs,

vt525|DEC VT525,
	use=vt520,

#### VT100 emulations
#

# John Hawkinson <jhawk@MIT.EDU> tells us that the EWAN telnet for Windows
# (the best Windows telnet as of September 1995) presents the name `dec-vt100'
# to telnetd.   Michael Deutschmann <ldeutsch@mail.netshop.net> informs us
# that this works best with a stock vt100 entry.
dec-vt100|EWAN telnet's vt100 emulation,
	use=vt100,

# From: Adrian Garside <94ajg2@eng.cam.ac.uk>, 19 Nov 1996
dec-vt220|DOS tnvt200 terminal emulator,
	am@, use=vt220,

# Zstem340 is an (IMHO) excellent VT emulator for PC's.  I recommend it to
# anyone who needs PC VT340 emulation. (or anything below that level, for
# that matter -- DEC's ALL-in-1 seems happy with it, as does INFOPLUS's
# RDBM systems, it includes ReGIS and SiXel support!  I'm impressed...
# I can send the address if requested.
# (z340: changed garbled \E[5?l to \E[?5l, DEC smooth scroll off -- esr)
# From: Adam Thompson <athompso@pangea.ca> Sept 10 1995
z340|zstem vt340 terminal emulator 132col 42line,
	lines#42,
	is2=\E>\E[?3h\E[?4l\E[?5l\E[?7h\E[?8h\E[1;42r\E[42;1H,
	rs2=\E>\E[?3h\E[?4l\E[?5l\E[?7h\E[?8h\E[1;42r\E[42;1H,
	use=vt320-w,
z340-nam|zstem vt340 terminal emulator 132col 42line (no automatic margins),
	am@,
	is2=\E>\E[?3h\E[?4l\E[?5l\E[?7l\E[?8h\E[1;42r\E[42;1H,
	rs2=\E>\E[?3h\E[?4l\E[?5l\E[?7l\E[?8h\E[1;42r\E[42;1H,
	use=z340,

# CRT is shareware.  It implements some xterm features, including mouse.
crt|crt-vt220|CRT 2.3 emulating VT220,
	bce, msgr,
	ncv@,
	hts=\EH, use=vt100+enq, use=vt220, use=ecma+color,

# PuTTY 0.55 (released 3 August 2004)
# http://www.chiark.greenend.org.uk/~sgtatham/putty/
#
# Comparing with 0.51, vttest is much better (only a few problems with the
# cursor position reports and wrapping).
#
# PuTTY 0.51 (released 14 December 2000)
#
# This emulates vt100 + vt52 (plus a few vt220 features:  ech, SRM, DECTCEM, as
# well as SCO and Atari, color palettes from Linux console).  Reading the code,
# it is intended to be VT102 plus selected features.  By default, it sets $TERM
# to xterm, which is incorrect, since several features are misimplemented:
#
#	Alt+key always sends ESC+key, so 'km' capability is removed.
#
#	Control responses, wrapping and tabs are buggy, failing a couple of
#	screens in vttest.
#
#	xterm mouse support is not implemented (unrelease version may).
#
# Several features such as backspace/delete are optional; this entry documents
# the default behavior -TD

putty|PuTTY terminal emulator,
	am, bce, bw, ccc, mir, msgr, xenl, xon, XT,
	colors#8, it#8, ncv#22, pairs#64, U8#1,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\ED, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\EM,
	dch=\E[%p1%dP, dch1=\E[P,
	dispc=%?%p1%{8}%=%t\E%%G\342\227\230\E%%@%e%p1%{10}%=%t\E%%G\342\227\231\E%%@%e%p1%{12}%=%t\E%%G\342\231\0\E%%@%e%p1%{13}%=%t\E%%G\342\231\252\E%%@%e%p1%{14}%=%t\E%%G\342\231\253\E%%@%e%p1%{15}%=%t\E%%G\342\230\274\E%%@%e%p1%{27}%=%t\E%%G\342\206\220\E%%@%e%p1%{155}%=%t\E%%G\340\202\242\E%%@%e%p1%c%;,
	dl=\E[%p1%dM, dl1=\E[M, ech=\E[%p1%dX, ed=\E[J, el=\E[K,
	el1=\E[1K, enacs=\E(B\E)0, flash=\E[?5h\E[?5l, home=\E[H,
	hpa=\E[%i%p1%dG, ht=^I, hts=\EH, il=\E[%p1%dL, il1=\E[L,
	ind=^J, indn=\E[%p1%dS,
	initc=\E]P%p1%x%p2%{255}%*%{1000}%/%02x%p3%{255}%*%{1000}%/%02x%p4%{255}%*%{1000}%/%02x,
	is2=\E7\E[r\E[m\E[?7h\E[?1;4;6l\E[4l\E8\E>\E]R,
	kb2=\E[G, kbs=\177, kcbt=\E[Z, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~, kend=\E[4~,
	kf1=\E[11~, kf10=\E[21~, kf11=\E[23~, kf12=\E[24~,
	kf13=\E[25~, kf14=\E[26~, kf15=\E[28~, kf16=\E[29~,
	kf17=\E[31~, kf18=\E[32~, kf19=\E[33~, kf2=\E[12~,
	kf20=\E[34~, kf3=\E[13~, kf4=\E[14~, kf5=\E[15~,
	kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, kich1=\E[2~, kmous=\E[M, knp=\E[6~, kpp=\E[5~,
	kspd=^Z, nel=^M^J, oc=\E]R, op=\E[39;49m, rc=\E8, rev=\E[7m,
	ri=\EM, rin=\E[%p1%dT, rmacs=^O, rmam=\E[?7l,
	rmcup=\E[2J\E[?47l, rmir=\E[4l, rmpch=\E[10m,
	rmso=\E[27m, rmul=\E[24m,
	rs2=\E<\E["p\E[50;6"p\Ec\E[?3l\E]R\E[?1000l,
	s0ds=\E[10m, s1ds=\E[11m, s2ds=\E[12m, sc=\E7,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smcup=\E[?47h,
	smir=\E[4h, smpch=\E[11m, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g, vpa=\E[%i%p1%dd, E3=\E[3;J, use=vt102+enq,
	use=xterm+sl,
vt100-putty|Reset PuTTY to pure vt100,
	rs2=\E<\E["p\Ec\E[?3l\E]R\E[40"p\E[61"p\E[50;1;2"p,
	use=vt100,
# palette is hardcoded...
putty-256color|PuTTY 0.58 with xterm 256-colors,
	initc@, use=xterm+256color, use=putty,

# One of the keyboard selections is "VT100+".
# pterm (the X11 port) uses shifted F1-F10 as F11-F20
putty-vt100|VT100+ keyboard layout,
	kf1=\EOP, kf10=\EOY, kf11=\EOZ, kf12=\EO[, kf2=\EOQ,
	kf3=\EOR, kf4=\EOS, kf5=\EOT, kf6=\EOU, kf7=\EOV, kf8=\EOW,
	kf9=\EOX, use=putty,

# Unlike xterm-sco, this leaves kmous ambiguous with kf1.
#
# Use modifiers to obtain function keys past 12:
# F1-F12 - normal
# F13-F24 - shift
# F25-F36 - control/alt
# F37-F48 - control/shift
#
putty-sco|putty with SCO function keys,
	kbeg=\E[E, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\177, kend=\E[F, kf1=\E[M, kf10=\E[V, kf11=\E[W,
	kf12=\E[X, kf13=\E[Y, kf14=\E[Z, kf15=\E[a, kf16=\E[b,
	kf17=\E[c, kf18=\E[d, kf19=\E[e, kf2=\E[N, kf20=\E[f,
	kf21=\E[g, kf22=\E[h, kf23=\E[i, kf24=\E[j, kf25=\E[k,
	kf26=\E[l, kf27=\E[m, kf28=\E[n, kf29=\E[o, kf3=\E[O,
	kf30=\E[p, kf31=\E[q, kf32=\E[r, kf33=\E[s, kf34=\E[t,
	kf35=\E[u, kf36=\E[v, kf37=\E[w, kf38=\E[x, kf39=\E[y,
	kf4=\E[P, kf40=\E[z, kf41=\E[@, kf42=\E[[, kf43=\E[\\,
	kf44=\E[], kf45=\E[\^, kf46=\E[_, kf47=\E[`, kf48=\E[{,
	kf5=\E[Q, kf6=\E[R, kf7=\E[S, kf8=\E[T, kf9=\E[U, khome=\E[H,
	kich1=\E[L, knp=\E[G, kpp=\E[I, use=putty,

# This entry is for Tera Term Pro version 2.3, for MS-Windows 95/NT written by
# T. Teranishi dated Mar 10, 1998.  It is a free software terminal emulator
# (communication program) which supports:
#
#	- Serial port connections.
#	- TCP/IP (telnet) connections.
#	- VT100 emulation, and selected VT200/300 emulation.
#	- TEK4010 emulation.
#	- File transfer protocols (Kermit, XMODEM, ZMODEM, B-PLUS and
#	  Quick-VAN).
#	- Scripts using the "Tera Term Language".
#	- Japanese and Russian character sets.
#
# The program does not come with terminfo or termcap entries.  However, the
# emulation (testing with vttest and ncurses) is reasonably close to vt100 (no
# vt52 or doublesize character support; blinking is done with color).  Besides
# the HPA, VPA extensions it also implements CPL and CNL.
#
# All of the function keys can be remapped.  This description shows the default
# mapping, as installed.  Both vt100 PF1-PF4 keys and quasi-vt220 F1-F4 keys
# are supported.  F13-F20 are obtained by shifting F3-F10.  The editing keypad
# is laid out like vt220, rather than the face codes on the PC keyboard, i.e,
#	kfnd	Insert
#	kslt	Delete
#	kich1	Home
#	kdch1	PageUp
#	kpp	End
#	knp	PageDown
#
# ANSI colors are implemented, but cannot be combined with video attributes
# except for reverse.
#
# No fonts are supplied with the program, so the acsc string is chosen to
# correspond with the default Microsoft terminal font.
#
# Tera Term recognizes some xterm sequences, including those for setting and
# retrieving the window title, and for setting the window size (i.e., using
# "resize -s"), though it does not pass SIGWINCH to the application if the
# user resizes the window with the mouse.
teraterm2.3|Tera Term Pro,
	km, xon@,
	ncv#43, vt@,
	acsc=+\020\,\021-\030.^Y0\333`\004a\261f\370g\361h\260i\316j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	blink=\E[5m, bold=\E[1m, civis=\E[?25l, clear=\E[H\E[J,
	cnorm=\E[?25h, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K,
	flash=\E[?5h\E[?5l$<200/>, hpa=\E[%i%p1%dG,
	il=\E[%p1%dL, il1=\E[L, kdch1=\E[3~, kf1=\E[11~,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf13=\E[25~,
	kf14=\E[26~, kf15=\E[28~, kf16=\E[29~, kf17=\E[31~,
	kf18=\E[32~, kf19=\E[33~, kf2=\E[12~, kf20=\E[34~,
	kf3=\E[13~, kf4=\E[14~, kf5=\E[15~, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, kfnd=\E[1~, kich1=\E[2~, knp=\E[6~,
	kpp=\E[5~, kslt=\E[4~, op=\E[100m, rev=\E[7m, ri=\EM,
	rmso=\E[27m, rmul=\E[24m, sgr0=\E[0m\017, smso=\E[7m,
	smul=\E[4m, vpa=\E[%i%p1%dd, use=vt100+enq,
	use=klone+color, use=vt100,

# Version 4.59 has regular vt100 line-drawing (so it is no longer necessary
# to choose a Windows OEM font).
#
# Testing with tack:
#	- it does not have xenl (suppress that)
#	- underline seems to work with color (modify ncv).
# Testing with vttest:
#	- wrapping differs from vt100 (menu 1).
#	- it recognizes xterm's X10 and normal mouse tracking, but none of the
#	  other flavors.
#	- it recognizes the dtterm window controls for reporting size in
#	  characters and pixels.
#	- it passes SIGWINCH.
teraterm4.59|Tera Term Pro,
	bce, xenl@,
	ncv#41,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	kmous=\E[M, use=teraterm2.3,

teraterm|Tera Term,
	use=teraterm4.59,

# Tested with WinNT 4.0, the telnet application assumes the screensize is
# 25x80.  This entry uses the 'Terminal' font, to get line-drawing characters.
#
# Other notes:
# a) Fails tack's cup (cursor-addressing) test, though cup works well enough
#    for casual (occasional) use.  Also fails several of the vttest screens,
#    but that is not unusual for vt100 "emulators".
# b) Does not implement vt100 keypad
# c) Recognizes a subset of vt52 controls.
ms-vt100|MS telnet imitating dec vt100,
	lines#25,
	acsc=+\020\,\021-\030.^Y0\333`\004a\261f\370g\361h\260i\316j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	ka1@, ka3@, kb2@, kc1@, kc3@, kent@, kf0@, kf1@, kf10@, kf2@, kf3@, kf4@,
	kf5@, kf6@, kf7@, kf8@, kf9@, tbc@, use=vt102+enq, use=vt100,

# Tested with Windows 2000, the telnet application runs in a console window,
# also using 'Terminal' font.
#
# Other notes:
# a) This version has no function keys or numeric keypad.  Unlike the older
#    version, the numeric keypad is entirely ignored.
# b) The program sets $TERM to "ansi", which of course is inaccurate.
ms-vt100-color|vtnt|windows 2000 ansi (sic),
	bce,
	dch=\E[%p1%dP, ich=\E[%p1%d@, use=ecma+color,
	use=ms-vt100,

# Based on comments from Federico Bianchi:
#
#	vt100+ is basically a VT102-noSGR with ANSI.SYS colors and a different
#	scheme for PF keys.
#
# and PuTTY wishlist:
#
#	The modifiers are represented as the codes listed above, prefixed to
#	the normal sequences.  If the modifier is pressed alone, its sequence
#	is transmitted twice in succession.  If multiple modifiers apply,
#	they're transmitted in the order shift, control, alt.
#
#	Shift	\E^S
#	Alt	\E^A,
#	Ctrl	\E^C,
ms-vt100+|vt100+|windows XP vt100+ (sic),
	kdch1=\E-, kend=\Ek, kf1=\E1, kf10=\E0, kf11=\E!, kf12=\E@,
	kf13=\E\023\E1, kf14=\E\023\E2, kf15=\E\023\E3,
	kf16=\E\023\E4, kf17=\E\023\E5, kf18=\E\023\E6,
	kf19=\E\023\E7, kf2=\E2, kf20=\E\023\E8, kf21=\E\023\E9,
	kf22=\E\023\E0, kf23=\E\023\E!, kf24=\E\023\E@,
	kf25=\E\003\E1, kf26=\E\003\E2, kf27=\E\003\E3,
	kf28=\E\003\E4, kf29=\E\003\E5, kf3=\E3, kf30=\E\003\E6,
	kf31=\E\003\E7, kf32=\E\003\E8, kf33=\E\003\E9,
	kf34=\E\003\E0, kf35=\E\003\E!, kf36=\E\003\E@,
	kf37=\E\001\E1, kf38=\E\001\E2, kf39=\E\001\E3, kf4=\E4,
	kf40=\E\001\E4, kf41=\E\001\E5, kf42=\E\001\E6,
	kf43=\E\001\E7, kf44=\E\001\E8, kf45=\E\001\E9,
	kf46=\E\001\E0, kf47=\E\001\E!, kf48=\E\001\E@, kf5=\E5,
	kf6=\E6, kf7=\E7, kf8=\E8, kf9=\E9, khome=\Eh, kich1=\E+,
	knp=\E/, kpp=\E?, use=ms-vt100-color,

ms-vt-utf8|vt-utf8|UTF-8 flavor of vt100+,
	use=ms-vt100+,

# expect-5.44.1.15/example/tkterm
# a minimal subset of a vt100 (compare with "news-unk).
#
# The missing "=" in smkx is not a typo (here), but an error in tkterm.
tt|tkterm|Don Libes' tk text widget terminal emulator,
	clear=\E[H\E[J, cr=^M, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%p1%d;%p2%dH, cuu1=\E[A, ind=^J, kf1=\EOP, kf2=\EOQ,
	kf3=\EOR, kf4=\EOS, kf5=\EOT, kf6=\EOU, kf7=\EOV, kf8=\EOW,
	kf9=\EOX, rmkx=\E[?1l\E>, rmso=\E[m, smkx=\E[?1h\E,
	smso=\E[7m,

######## X TERMINAL EMULATORS
#### XTERM
#
# You can add the following line to your .Xdefaults to change the terminal type
# set by the xterms you start up to my-xterm:
#
# *termName:  my-xterm
#
# System administrators can change the default entry for xterm instances
# by adding a similar line to /usr/X11/lib/X11/app-defaults/XTerm.  In either
# case, xterm will detect and reject an invalid terminal type, falling back
# to the default of xterm.
#

# X10/6.6	11/7/86, minus alternate screen, plus (csr)
# (xterm: ":MT:" changed to ":km:"; added <smam>/<rmam> based on init string;
# removed (hs, eslok, tsl=\E[?E\E[?%i%dT, fsl=\E[?F, dsl=\E[?E)
# as these seem not to work -- esr)
x10term|vs100-x10|xterm terminal emulator (X10 window system),
	OTbs, am, km, mir, msgr, xenl, xon,
	cols#80, it#8, lines#65,
	bold=\E[1m, clear=\E[H\E[2J, csr=\E[%i%p1%d;%p2%dr,
	cub1=^H, cud1=^J, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I, il=\E[%p1%dL,
	il1=\E[L, ind=^J, is2=\E\E[m\E[?7h\E[?1;4l, kbs=^H,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kf1=\EOP,
	kf2=\EOQ, kf3=\EOR, kf4=\EOS, rev=\E[7m, ri=\EM, rmam=\E[?7l,
	rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m, smam=\E[?7h, smir=\E[4h, smkx=\E[?1h\E=,
	smso=\E[7m, smul=\E[4m,
# Compatible with the R5 xterm
# (from the XFree86 3.2 distribution, <blink=@> removed)
# added khome/kend, rmir/smir, rmul/smul, hts based on the R5 xterm code - TD
# corrected typos in rs2 string - TD
# added u6-u9 -TD
xterm-r5|xterm R5 version,
	OTbs, am, km, msgr, xenl,
	cols#80, it#8, lines#24,
	bel=^G, bold=\E[1m, clear=\E[H\E[2J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@, ich1=\E[@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, kbs=^H, kcub1=\EOD,
	kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kdch1=\E[3~,
	kdl1=\E[31~, kel=\E[8~, kend=\E[4~, kf0=\EOq, kf1=\E[11~,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf2=\E[12~,
	kf3=\E[13~, kf4=\E[14~, kf5=\E[15~, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, khome=\E[1~, kich1=\E[2~,
	kil1=\E[30~, kmous=\E[M, knp=\E[6~, kpp=\E[5~, rc=\E8,
	rev=\E[7m, ri=\EM, rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[m,
	rmul=\E[m,
	rs2=\E>\E[?1;3;4;5;6l\E[4l\E[?7h\E[m\E[r\E[2J\E[H,
	sc=\E7,
	sgr=\E[%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;m,
	sgr0=\E[m, smir=\E[4h, smkx=\E[?1h\E=, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g, use=vt100+enq,
# Compatible with the R6 xterm
# (from XFree86 3.2 distribution, <acsc> and <it> added, <blink@> removed)
# added khome/kend, hts based on the R6 xterm code - TD
# (khome/kend do not actually work in X11R5 or X11R6, but many people use this
# for compatibility with other emulators).
xterm-r6|xterm X11R6 version,
	OTbs, am, km, mir, msgr, xenl,
	cols#80, it#8, lines#24,
	acsc=``aaffggiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, bold=\E[1m, clear=\E[H\E[2J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, enacs=\E)0, home=\E[H, ht=^I, hts=\EH, il=\E[%p1%dL,
	il1=\E[L, ind=^J,
	is2=\E[m\E[?7h\E[4l\E>\E7\E[r\E[?1;3;4;6l\E8, kbs=^H,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kdch1=\E[3~, kf1=\E[11~, kf10=\E[21~, kf11=\E[23~,
	kf12=\E[24~, kf13=\E[25~, kf14=\E[26~, kf15=\E[28~,
	kf16=\E[29~, kf17=\E[31~, kf18=\E[32~, kf19=\E[33~,
	kf2=\E[12~, kf20=\E[34~, kf3=\E[13~, kf4=\E[14~,
	kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	kfnd=\E[1~, kich1=\E[2~, kmous=\E[M, knp=\E[6~, kpp=\E[5~,
	kslt=\E[4~, meml=\El, memu=\Em, rc=\E8, rev=\E[7m, ri=\EM,
	rmacs=^O, rmcup=\E[2J\E[?47l\E8, rmir=\E[4l,
	rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	rs2=\E[m\E[?7h\E[4l\E>\E7\E[r\E[?1;3;4;6l\E8, sc=\E7,
	sgr0=\E[m, smacs=^N, smcup=\E7\E[?47h, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	use=vt100+enq,
xterm-old|antique xterm version,
	use=xterm-r6,
# This is the base xterm entry for the xterm supplied with XFree86 3.2 & up.
# The name has been changed and some aliases have been removed.
xterm-xf86-v32|xterm terminal emulator (XFree86 3.2 Window System),
	OTbs, am, bce, km, mir, msgr, xenl, XT,
	cols#80, it#8, lines#24, ncv@,
	acsc=``aaffggiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[2J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K, enacs=\E(B\E)0,
	flash=\E[?5h$<100/>\E[?5l, home=\E[H, hpa=\E[%i%p1%dG,
	ht=^I, hts=\EH, ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL,
	il1=\E[L, ind=^J,
	is2=\E7\E[r\E[m\E[?7h\E[?1;3;4;6l\E[4l\E8\E>,
	kbeg=\EOE, kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC,
	kcuu1=\EOA, kdch1=\177, kend=\EOF, kf1=\E[11~, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf13=\E[25~, kf14=\E[26~,
	kf15=\E[28~, kf16=\E[29~, kf17=\E[31~, kf18=\E[32~,
	kf19=\E[33~, kf2=\E[12~, kf20=\E[34~, kf3=\E[13~,
	kf4=\E[14~, kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~,
	kf9=\E[20~, kfnd=\E[1~, khome=\EOH, kich1=\E[2~,
	kmous=\E[M, knp=\E[6~, kpp=\E[5~, kslt=\E[4~, meml=\El,
	memu=\Em, rc=\E8, rev=\E[7m, ri=\EM, rmacs=^O, rmam=\E[?7l,
	rmcup=\E[2J\E[?47l\E8, rmir=\E[4l, rmkx=\E[?1l\E>,
	rmso=\E[27m, rmul=\E[24m, rs1=^O,
	rs2=\E7\E[r\E8\E[m\E[?7h\E[?1;3;4;6l\E[4l\E>, sc=\E7,
	setb=\E[4%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	setf=\E[3%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smcup=\E7\E[?47h,
	smir=\E[4h, smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g, vpa=\E[%i%p1%dd, use=vt100+enq,
	use=ecma+color, use=vt220+keypad,

# This is the stock xterm entry supplied with XFree86 3.3, which uses VT100
# codes for F1-F4 except while in VT220 mode.
xterm-xf86-v33|xterm terminal emulator (XFree86 3.3 Window System),
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, use=xterm-xf86-v32,

# This version was released in XFree86 3.3.3 (November 1998).
# Besides providing printer support, it exploits a new feature that allows
# xterm to use terminfo-based descriptions with the titeInhibit resource.
# -- the distribution contained incorrect khome/kend values -TD
xterm-xf86-v333|xterm terminal emulator (XFree86 3.3.3 Window System),
	blink=\E[5m, ich1@, invis=\E[8m,
	is2=\E[!p\E[?3;4l\E[4l\E>, kdch1=\E[3~, kfnd@, kslt@,
	rmcup=\E[?1047l\E[?1048l, rs1=\Ec,
	rs2=\E[!p\E[?3;4l\E[4l\E>,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	smcup=\E[?1048h\E[?1047h, use=ansi+pp,
	use=xterm-xf86-v33,

# This version was released in XFree86 4.0.
xterm-xf86-v40|xterm terminal emulator (XFree86 4.0 Window System),
	npc,
	kDC=\E[3;5~, kEND=\EO5F, kHOM=\EO5H, kIC=\E[2;5~,
	kLFT=\EO5D, kNXT=\E[6;5~, kPRV=\E[5;5~, kRIT=\EO5C, ka1@,
	ka3@, kb2=\EOE, kc1@, kc3@, kcbt=\E[Z, kdch1=\E[3~, kend=\EOF,
	kf13=\EO2P, kf14=\EO2Q, kf15=\EO2R, kf16=\EO2S,
	kf17=\E[15;2~, kf18=\E[17;2~, kf19=\E[18;2~,
	kf20=\E[19;2~, kf21=\E[20;2~, kf22=\E[21;2~,
	kf23=\E[23;2~, kf24=\E[24;2~, kf25=\EO5P, kf26=\EO5Q,
	kf27=\EO5R, kf28=\EO5S, kf29=\E[15;5~, kf30=\E[17;5~,
	kf31=\E[18;5~, kf32=\E[19;5~, kf33=\E[20;5~,
	kf34=\E[21;5~, kf35=\E[23;5~, kf36=\E[24;5~, kf37=\EO6P,
	kf38=\EO6Q, kf39=\EO6R, kf40=\EO6S, kf41=\E[15;6~,
	kf42=\E[17;6~, kf43=\E[18;6~, kf44=\E[19;6~,
	kf45=\E[20;6~, kf46=\E[21;6~, kf47=\E[23;6~,
	kf48=\E[24;6~, khome=\EOH, rmcup=\E[?1049l,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;,
	smcup=\E[?1049h, use=xterm-xf86-v333,

# This version was released in XFree86 4.3.
xterm-xf86-v43|xterm terminal emulator (XFree86 4.3 Window System),
	kDC=\E[3;2~, kEND=\E[1;2F, kHOM=\E[1;2H, kIC=\E[2;2~,
	kLFT=\E[1;2D, kNXT=\E[6;2~, kPRV=\E[5;2~, kRIT=\E[1;2C,
	kbeg@,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	use=xterm-xf86-v40,

# This version was released in XFree86 4.4.
xterm-xf86-v44|xterm terminal emulator (XFree86 4.4 Window System),
	cnorm=\E[?12l\E[?25h, cvvis=\E[?12;25h, indn=\E[%p1%dS,
	rin=\E[%p1%dT, use=xterm-xf86-v43,

xterm-xfree86|xterm terminal emulator (XFree86),
	use=xterm-xf86-v44,

# This version reflects the current xterm features.
xterm-new|modern xterm terminal emulator,
	npc,
	indn=\E[%p1%dS, kb2=\EOE, kcbt=\E[Z, kent=\EOM,
	rin=\E[%p1%dT, use=xterm+pcfkeys, use=xterm+tmux,
	use=xterm-basic,

# This fragment is for people who cannot agree on what the backspace key
# should send.
xterm+kbs|fragment for backspace key,
	kbs=^H,
#
# This fragment describes as much of XFree86 xterm's "pc-style" function
# keys as will fit into terminfo's 60 function keys.
# From ctlseqs.ms:
#    Code     Modifiers
#  ---------------------------------
#     2       Shift
#     3       Alt
#     4       Shift + Alt
#     5       Control
#     6       Shift + Control
#     7       Alt + Control
#     8       Shift + Alt + Control
#  ---------------------------------
# The meta key may also be used as a modifier in this scheme, adding another
# bit to the parameter.
xterm+pcfkeys|fragment for PC-style fkeys,
	use=xterm+app, use=xterm+pcf2, use=xterm+pcc2,
	use=xterm+pce2,
#
xterm+noapp|fragment with cursor keys in normal mode,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kend=\E[F,
	khome=\E[H,

xterm+app|fragment with cursor keys in application mode,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kend=\EOF,
	khome=\EOH,
#
# The "PC-style" modifier scheme was introduced in xterm patch #94 (1999/3/27)
# and revised in patch #167 (2002/8/24).  Some other terminal emulators copied
# the earlier scheme, as noted in the "use=" clauses in this file.
#
# The original assignments from patch #94 for cursor-keys had some technical
# issues:
#
#	A parameter for a function-key to represent a modifier is just more
#	bits.  But for a cursor-key it may change the behavior of the
#	application.  For instance, emacs decodes the first parameter of a
#	cursor-key as a repeat count.
#
#	A parameterized string should (really) not begin with SS3 (\EO).
#	Rather, CSI (\E[) should be used.
#
# For these reasons, the original assignments were deprecated.  For
# compatibility reasons, they are still available as a setting of xterm's
# modifyCursorKeys resource.  These fragments list the modified cursor-keys
# that might apply to xterm+pcfkeys with different values of that resource.
xterm+pcc3|fragment with modifyCursorKeys:3,
	kLFT=\E[>1;2D, kRIT=\E[>1;2C, kind=\E[>1;2B,
	kri=\E[>1;2A, kDN=\E[>1;2B, kDN3=\E[>1;3B, kDN4=\E[>1;4B,
	kDN5=\E[>1;5B, kDN6=\E[>1;6B, kDN7=\E[>1;7B,
	kLFT3=\E[>1;3D, kLFT4=\E[>1;4D, kLFT5=\E[>1;5D,
	kLFT6=\E[>1;6D, kLFT7=\E[>1;7D, kRIT3=\E[>1;3C,
	kRIT4=\E[>1;4C, kRIT5=\E[>1;5C, kRIT6=\E[>1;6C,
	kRIT7=\E[>1;7C, kUP=\E[>1;2A, kUP3=\E[>1;3A,
	kUP4=\E[>1;4A, kUP5=\E[>1;5A, kUP6=\E[>1;6A,
	kUP7=\E[>1;7A,

xterm+pcc2|fragment with modifyCursorKeys:2,
	kLFT=\E[1;2D, kRIT=\E[1;2C, kind=\E[1;2B, kri=\E[1;2A,
	kDN=\E[1;2B, kDN3=\E[1;3B, kDN4=\E[1;4B, kDN5=\E[1;5B,
	kDN6=\E[1;6B, kDN7=\E[1;7B, kLFT3=\E[1;3D, kLFT4=\E[1;4D,
	kLFT5=\E[1;5D, kLFT6=\E[1;6D, kLFT7=\E[1;7D,
	kRIT3=\E[1;3C, kRIT4=\E[1;4C, kRIT5=\E[1;5C,
	kRIT6=\E[1;6C, kRIT7=\E[1;7C, kUP=\E[1;2A, kUP3=\E[1;3A,
	kUP4=\E[1;4A, kUP5=\E[1;5A, kUP6=\E[1;6A, kUP7=\E[1;7A,

xterm+pcc1|fragment with modifyCursorKeys:1,
	kLFT=\E[2D, kRIT=\E[2C, kind=\E[2B, kri=\E[2A, kDN=\E[2B,
	kDN3=\E[3B, kDN4=\E[4B, kDN5=\E[5B, kDN6=\E[6B, kDN7=\E[7B,
	kLFT3=\E[3D, kLFT4=\E[4D, kLFT5=\E[5D, kLFT6=\E[6D,
	kLFT7=\E[7D, kRIT3=\E[3C, kRIT4=\E[4C, kRIT5=\E[5C,
	kRIT6=\E[6C, kRIT7=\E[7C, kUP=\E[2A, kUP3=\E[3A,
	kUP4=\E[4A, kUP5=\E[5A, kUP6=\E[6A, kUP7=\E[7A,

xterm+pcc0|fragment with modifyCursorKeys:0,
	kLFT=\EO2D, kRIT=\EO2C, kind=\EO2B, kri=\EO2A, kDN=\EO2B,
	kDN3=\EO3B, kDN4=\EO4B, kDN5=\EO5B, kDN6=\EO6B, kDN7=\EO7B,
	kLFT3=\EO3D, kLFT4=\EO4D, kLFT5=\EO5D, kLFT6=\EO6D,
	kLFT7=\EO7D, kRIT3=\EO3C, kRIT4=\EO4C, kRIT5=\EO5C,
	kRIT6=\EO6C, kRIT7=\EO7C, kUP=\EO2A, kUP3=\EO3A,
	kUP4=\EO4A, kUP5=\EO5A, kUP6=\EO6A, kUP7=\EO7A,

#
# Here are corresponding fragments from xterm patch #216:
#
xterm+pcf0|fragment with modifyFunctionKeys:0,
	kf1=\EOP, kf10=\E[21~, kf11=\E[23~, kf12=\E[24~,
	kf13=\EO2P, kf14=\EO2Q, kf15=\EO2R, kf16=\EO2S,
	kf17=\E[15;2~, kf18=\E[17;2~, kf19=\E[18;2~, kf2=\EOQ,
	kf20=\E[19;2~, kf21=\E[20;2~, kf22=\E[21;2~,
	kf23=\E[23;2~, kf24=\E[24;2~, kf25=\EO5P, kf26=\EO5Q,
	kf27=\EO5R, kf28=\EO5S, kf29=\E[15;5~, kf3=\EOR,
	kf30=\E[17;5~, kf31=\E[18;5~, kf32=\E[19;5~,
	kf33=\E[20;5~, kf34=\E[21;5~, kf35=\E[23;5~,
	kf36=\E[24;5~, kf37=\EO6P, kf38=\EO6Q, kf39=\EO6R,
	kf4=\EOS, kf40=\EO6S, kf41=\E[15;6~, kf42=\E[17;6~,
	kf43=\E[18;6~, kf44=\E[19;6~, kf45=\E[20;6~,
	kf46=\E[21;6~, kf47=\E[23;6~, kf48=\E[24;6~, kf49=\EO3P,
	kf5=\E[15~, kf50=\EO3Q, kf51=\EO3R, kf52=\EO3S,
	kf53=\E[15;3~, kf54=\E[17;3~, kf55=\E[18;3~,
	kf56=\E[19;3~, kf57=\E[20;3~, kf58=\E[21;3~,
	kf59=\E[23;3~, kf6=\E[17~, kf60=\E[24;3~, kf61=\EO4P,
	kf62=\EO4Q, kf63=\EO4R, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
#
xterm+pcf2|fragment with modifyFunctionKeys:2,
	kf1=\EOP, kf10=\E[21~, kf11=\E[23~, kf12=\E[24~,
	kf13=\E[1;2P, kf14=\E[1;2Q, kf15=\E[1;2R, kf16=\E[1;2S,
	kf17=\E[15;2~, kf18=\E[17;2~, kf19=\E[18;2~, kf2=\EOQ,
	kf20=\E[19;2~, kf21=\E[20;2~, kf22=\E[21;2~,
	kf23=\E[23;2~, kf24=\E[24;2~, kf25=\E[1;5P, kf26=\E[1;5Q,
	kf27=\E[1;5R, kf28=\E[1;5S, kf29=\E[15;5~, kf3=\EOR,
	kf30=\E[17;5~, kf31=\E[18;5~, kf32=\E[19;5~,
	kf33=\E[20;5~, kf34=\E[21;5~, kf35=\E[23;5~,
	kf36=\E[24;5~, kf37=\E[1;6P, kf38=\E[1;6Q, kf39=\E[1;6R,
	kf4=\EOS, kf40=\E[1;6S, kf41=\E[15;6~, kf42=\E[17;6~,
	kf43=\E[18;6~, kf44=\E[19;6~, kf45=\E[20;6~,
	kf46=\E[21;6~, kf47=\E[23;6~, kf48=\E[24;6~,
	kf49=\E[1;3P, kf5=\E[15~, kf50=\E[1;3Q, kf51=\E[1;3R,
	kf52=\E[1;3S, kf53=\E[15;3~, kf54=\E[17;3~,
	kf55=\E[18;3~, kf56=\E[19;3~, kf57=\E[20;3~,
	kf58=\E[21;3~, kf59=\E[23;3~, kf6=\E[17~, kf60=\E[24;3~,
	kf61=\E[1;4P, kf62=\E[1;4Q, kf63=\E[1;4R, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~,
#
# Chunks from xterm #230:
xterm+pce2|fragment with modifyCursorKeys:2,
	kDC=\E[3;2~, kEND=\E[1;2F, kHOM=\E[1;2H, kIC=\E[2;2~,
	kNXT=\E[6;2~, kPRV=\E[5;2~, kich1=\E[2~, knp=\E[6~,
	kpp=\E[5~, kDC3=\E[3;3~, kDC4=\E[3;4~, kDC5=\E[3;5~,
	kDC6=\E[3;6~, kDC7=\E[3;7~, kEND3=\E[1;3F, kEND4=\E[1;4F,
	kEND5=\E[1;5F, kEND6=\E[1;6F, kEND7=\E[1;7F,
	kHOM3=\E[1;3H, kHOM4=\E[1;4H, kHOM5=\E[1;5H,
	kHOM6=\E[1;6H, kHOM7=\E[1;7H, kIC3=\E[2;3~, kIC4=\E[2;4~,
	kIC5=\E[2;5~, kIC6=\E[2;6~, kIC7=\E[2;7~, kNXT3=\E[6;3~,
	kNXT4=\E[6;4~, kNXT5=\E[6;5~, kNXT6=\E[6;6~,
	kNXT7=\E[6;7~, kPRV3=\E[5;3~, kPRV4=\E[5;4~,
	kPRV5=\E[5;5~, kPRV6=\E[5;6~, kPRV7=\E[5;7~,
	use=xterm+edit,

xterm+edit|fragment for 6-key editing-keypad,
	kdch1=\E[3~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~,
	use=xterm+pc+edit,

xterm+pc+edit|fragment for pc-style editing keypad,
	kend=\E[4~, khome=\E[1~,

xterm+vt+edit|fragment for vt220-style editing keypad,
	kfnd=\E[1~, kslt=\E[4~,

#
# Those chunks use the new-style (the xterm oldFunctionKeys resource is false).
# Alternatively, the same scheme with old-style function keys as in xterm-r6
# is shown here (because that is used in mrxvt and mlterm):
xterm+r6f2|xterm with oldFunctionKeys and modifyFunctionKeys:2,
	kf1=\E[11~, kf13=\E[11;2~, kf14=\E[12;2~, kf15=\E[13;2~,
	kf16=\E[14;2~, kf2=\E[12~, kf25=\E[11;5~, kf26=\E[12;5~,
	kf27=\E[13;5~, kf28=\E[14;5~, kf3=\E[13~, kf37=\E[11;6~,
	kf38=\E[12;6~, kf39=\E[13;6~, kf4=\E[14~, kf40=\E[14;6~,
	kf49=\E[11;3~, kf50=\E[12;3~, kf51=\E[13;3~,
	kf52=\E[14;3~, kf61=\E[11;4~, kf62=\E[12;4~,
	kf63=\E[13;4~, use=xterm+pcf2,
#
# This chunk is used for building the VT220/Sun/PC keyboard variants.
xterm-basic|modern xterm terminal emulator - common,
	OTbs, am, bce, km, mir, msgr, xenl, AX, XT,
	colors#8, cols#80, it#8, lines#24, pairs#64,
	acsc=``aaffggiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[2J, cnorm=\E[?12l\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[?12;25h, dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K,
	flash=\E[?5h$<100/>\E[?5l, home=\E[H, hpa=\E[%i%p1%dG,
	ht=^I, hts=\EH, ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L,
	ind=^J, invis=\E[8m, is2=\E[!p\E[?3;4l\E[4l\E>,
	kmous=\E[M, meml=\El, memu=\Em, op=\E[39;49m, rc=\E8,
	rev=\E[7m, ri=\EM, rmacs=\E(B, rmam=\E[?7l,
	rmcup=\E[?1049l, rmir=\E[4l, rmkx=\E[?1l\E>,
	rmm=\E[?1034l, rmso=\E[27m, rmul=\E[24m, rs1=\Ec,
	rs2=\E[!p\E[?3;4l\E[4l\E>, sc=\E7, setab=\E[4%p1%dm,
	setaf=\E[3%p1%dm,
	setb=\E[4%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	setf=\E[3%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	sgr=%?%p9%t\E(0%e\E(B%;\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p7%t;8%;m,
	sgr0=\E(B\E[m, smacs=\E(0, smam=\E[?7h, smcup=\E[?1049h,
	smir=\E[4h, smkx=\E[?1h\E=, smm=\E[?1034h, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g, vpa=\E[%i%p1%dd, E3=\E[3;J,
	use=ansi+pp, use=xterm+kbs, use=vt100+enq,

# From: David J. MacKenzie <djm@va.pubnix.com>, 14 Nov 1997
# In retrospect, something like xterm-r6 was intended here -TD
xterm-xi|xterm on XI Graphics Accelerated X under BSD/OS 3.1,
	rmso=\E[m, rmul=\E[m, use=xterm-xf86-v33,

# 16-colors is one of the variants of XFree86 3.3 xterm, updated for 4.0 (T.Dickey)
# If configured to support 88- or 256-colors (which is fairly common in 2009),
# xterm also recognizes the control sequences for initc -TD
xterm-16color|xterm with 16 colors like aixterm,
	ccc,
	initc=\E]4;%p1%d;rgb\:%p2%{255}%*%{1000}%/%2.2X/%p3%{255}%*%{1000}%/%2.2X/%p4%{255}%*%{1000}%/%2.2X\E\\,
	use=ibm+16color, use=xterm-new,

# 256-colors is a compile-time feature of XFree86 xterm beginning with
# patch #111 (1999/7/10) -TD
xterm+256color|xterm 256-color feature,
	ccc,
	colors#256, pairs#32767,
	initc=\E]4;%p1%d;rgb\:%p2%{255}%*%{1000}%/%2.2X/%p3%{255}%*%{1000}%/%2.2X/%p4%{255}%*%{1000}%/%2.2X\E\\,
	setab=\E[%?%p1%{8}%<%t4%p1%d%e%p1%{16}%<%t10%p1%{8}%-%d%e48;5;%p1%d%;m,
	setaf=\E[%?%p1%{8}%<%t3%p1%d%e%p1%{16}%<%t9%p1%{8}%-%d%e38;5;%p1%d%;m,
	setb@, setf@,

# 88-colors is a compile-time feature of XFree86 xterm beginning with
# patch #115 (1999/9/18) -TD
#
# Note that the escape sequences used are the same as for 256-colors - xterm
# has a different table of default color resource values.  If built for
# 256-colors, it can still handle an 88-color palette by using the initc
# capability.
#
# At this time (2007/7/14), except for rxvt 2.7.x, none of the other terminals
# which support the xterm+256color feature support the associated initc
# capability.  So it is cancelled in the entries which use this and/or the
# xterm+256color block.
#
# The default color palette for the 256- and 88-colors are different.  A
# given executable will have one palette (perhaps compiled-in).  If the program
# supports xterm's control sequence, it can be programmed using initc.
xterm+88color|xterm 88-color feature,
	colors#88, pairs#7744, use=xterm+256color,

# These variants of XFree86 3.9.16 xterm are built as a configure option.
xterm-256color|xterm with 256 colors,
	use=xterm+256color, use=xterm-new,
xterm-88color|xterm with 88 colors,
	use=xterm+88color, use=xterm-256color,

# These two are used to demonstrate the any-event mouse support, i.e., by
# using an extended name "XM" which tells ncurses to put the terminal into
# a special mode when initializing the xterm mouse.
xterm-1002|testing xterm-mouse,
	XM=\E[?1002%?%p1%{1}%=%th%el%;, use=xterm-new,
xterm-1003|testing xterm-mouse,
	XM=\E[?1003%?%p1%{1}%=%th%el%;, use=xterm-new,

# This chunk is based on suggestions by Ailin Nemui and Nicholas Marriott, who
# asked for some of xterm's advanced features to be added to its terminfo
# entry.  It defines extended capabilities not found in standard terminfo or
# termcap.  These are useful in tmux, for instance, hence the name.
#
# One caveat in adding extended capabilities in ncurses is that if the names
# are longer than two characters, then they will not be visible through the
# termcap interface.
#
# Ms modifies the selection/clipboard.  Its parameters are
#	p1 = the storage unit (clipboard, selection or cut buffer)
#	p2 = the base64-encoded clipboard content.
#
# Ss is used to set the cursor style as described by the DECSCUSR
#	function to a block or underline.
# Se resets the cursor style to the terminal power-on default.
#
# Cs and Cr set and reset the cursor colour.
xterm+tmux|advanced xterm features used in tmux,
	Cr=\E]112\007, Cs=\E]12;%p1%s\007,
	Ms=\E]52;%p1%s;%p2%s\007, Se=\E[2 q, Ss=\E[%p1%d q,

# This is another variant, for XFree86 4.0 xterm (T.Dickey)
# This is an 8-bit version of xterm, which emulates DEC vt220 with ANSI color.
# To use it, your decTerminalID resource must be set to 200 or above.
#
#	HTS	\E H	\210
#	RI	\E M	\215
#	SS3	\E O	\217
#	CSI	\E [	\233
#
xterm-8bit|xterm terminal emulator 8-bit controls (X Window System),
	OTbs, am, bce, km, mc5i, mir, msgr, npc, xenl, AX,
	colors#8, cols#80, it#8, lines#24, pairs#64,
	acsc=``aaffggiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\2335m, bold=\2331m, cbt=\233Z,
	civis=\233?25l, clear=\233H\2332J,
	cnorm=\233?25l\233?25h, cr=^M, csr=\233%i%p1%d;%p2%dr,
	cub=\233%p1%dD, cub1=^H, cud=\233%p1%dB, cud1=^J,
	cuf=\233%p1%dC, cuf1=\233C, cup=\233%i%p1%d;%p2%dH,
	cuu=\233%p1%dA, cuu1=\233A, cvvis=\233?12;25h,
	dch=\233%p1%dP, dch1=\233P, dl=\233%p1%dM, dl1=\233M,
	ech=\233%p1%dX, ed=\233J, el=\233K, el1=\2331K,
	flash=\233?5h$<100/>\233?5l, home=\233H,
	hpa=\233%i%p1%dG, ht=^I, hts=\210, ich=\233%p1%d@,
	il=\233%p1%dL, il1=\233L, ind=^J, invis=\2338m,
	is2=\E[62"p\E G\233m\233?7h\E>\E7\233?1;3;4;6l\2334l\233r\E8,
	ka1=\217w, ka3=\217u, kb2=\217y, kbeg=\217E, kc1=\217q,
	kc3=\217s, kcbt=\233Z, kcub1=\217D, kcud1=\217B,
	kcuf1=\217C, kcuu1=\217A, kdch1=\2333~, kend=\2334~,
	kent=\217M, kf1=\23311~, kf10=\23321~, kf11=\23323~,
	kf12=\23324~, kf13=\23325~, kf14=\23326~, kf15=\23328~,
	kf16=\23329~, kf17=\23331~, kf18=\23332~, kf19=\23333~,
	kf2=\23312~, kf20=\23334~, kf3=\23313~, kf4=\23314~,
	kf5=\23315~, kf6=\23317~, kf7=\23318~, kf8=\23319~,
	kf9=\23320~, khome=\2331~, kich1=\2332~, kmous=\233M,
	knp=\2336~, kpp=\2335~, mc0=\233i, mc4=\2334i, mc5=\2335i,
	meml=\El, memu=\Em, op=\23339;49m, rc=\E8, rev=\2337m,
	ri=\215, rmacs=\E(B, rmam=\233?7l, rmcup=\233?1049l,
	rmir=\2334l, rmkx=\233?1l\E>, rmso=\23327m, rmul=\23324m,
	rs1=\Ec,
	rs2=\E[62"p\E G\233m\233?7h\E>\E7\233?1;3;4;6l\2334l\233r\E8,
	sc=\E7, setab=\2334%p1%dm, setaf=\2333%p1%dm,
	setb=\2334%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	setf=\2333%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	sgr=\2330%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p7%t;8%;m%?%p9%t\E(0%e\E(B%;,
	sgr0=\2330m\E(B, smacs=\E(0, smam=\233?7h,
	smcup=\233?1049h, smir=\2334h, smkx=\233?1h\E=,
	smso=\2337m, smul=\2334m, tbc=\2333g, u6=\233[%i%d;%dR,
	u7=\E[6n, u8=\233[?1;2c, u9=\E[c, vpa=\233%i%p1%dd,
	use=xterm+kbs,

xterm-hp|xterm with hpterm function keys,
	kclr=\EJ, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA,
	kdch1=\EP, kend=\EF, kf1=\Ep, kf2=\Eq, kf3=\Er, kf4=\Es,
	kf5=\Et, kf6=\Eu, kf7=\Ev, kf8=\Ew, khome=\Eh, kich1=\EQ,
	knp=\ES, kpp=\ET, use=xterm-basic,

xterm-sco|xterm with SCO function keys,
	kbeg=\E[E, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\177, kend=\E[F, kf1=\E[M, kf10=\E[V, kf11=\E[W,
	kf12=\E[X, kf13=\E[Y, kf14=\E[Z, kf15=\E[a, kf16=\E[b,
	kf17=\E[c, kf18=\E[d, kf19=\E[e, kf2=\E[N, kf20=\E[f,
	kf21=\E[g, kf22=\E[h, kf23=\E[i, kf24=\E[j, kf25=\E[k,
	kf26=\E[l, kf27=\E[m, kf28=\E[n, kf29=\E[o, kf3=\E[O,
	kf30=\E[p, kf31=\E[q, kf32=\E[r, kf33=\E[s, kf34=\E[t,
	kf35=\E[u, kf36=\E[v, kf37=\E[w, kf38=\E[x, kf39=\E[y,
	kf4=\E[P, kf40=\E[z, kf41=\E[@, kf42=\E[[, kf43=\E[\\,
	kf44=\E[], kf45=\E[\^, kf46=\E[_, kf47=\E[`, kf48=\E[{,
	kf5=\E[Q, kf6=\E[R, kf7=\E[S, kf8=\E[T, kf9=\E[U, khome=\E[H,
	kich1=\E[L, kmous=\E[>M, knp=\E[G, kpp=\E[I,
	use=xterm-basic,

# The xterm-new description has all of the features, but is not completely
# compatible with vt220.  If you are using a Sun or PC keyboard, set the
# sunKeyboard resource to true:
#	+ maps the editing keypad
#	+ interprets control-function-key as a second array of keys, so a
#	  12-fkey keyboard can support vt220's 20-fkeys.
#	+ maps numeric keypad "+" to ",".
#	+ uses DEC-style control sequences for the application keypad.
#
xterm-vt220|xterm emulating vt220,
	kcbt=\E[Z, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kend=\E[4~, kf10=\E[21~, kf11=\E[23~, kf12=\E[24~,
	kf13=\E[25~, kf14=\E[26~, kf15=\E[28~, kf16=\E[29~,
	kf17=\E[31~, kf18=\E[32~, kf19=\E[33~, kf20=\E[34~,
	kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, kich1=\E[2~, kmous=\E[M, knp=\E[6~, kpp=\E[5~,
	use=xterm+app, use=xterm+edit, use=xterm-basic,
	use=vt220+keypad,

xterm-vt52|xterm emulating dec vt52,
	cols#80, it#8, lines#24,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, clear=\EH\EJ, cr=^M, cub1=\ED, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%' '%+%c%p2%' '%+%c, cuu1=\EA, ed=\EJ, el=\EK,
	home=\EH, ht=^I, ind=^J, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, nel=^M^J, ri=\EI, rmacs=\EG, smacs=\EF,
	use=xterm+kbs,

xterm-noapp|xterm with cursor keys in normal mode,
	rmcup@, rmkx=\E>, smcup@, smkx=\E=, use=xterm+noapp,
	use=xterm,

xterm-24|vs100|xterms|xterm terminal emulator (X Window System),
	lines#24, use=xterm-old,

# This is xterm for ncurses.
xterm|xterm terminal emulator (X Window System),
	use=xterm-new,

# This entry assumes that xterm's handling of VT100 SI/SO is disabled by
# setting the vt100Graphics resource to false.
xterm-utf8|xterm with no VT100 line-drawing in UTF-8 mode,
	U8#1, use=xterm,

# These building-blocks allow access to the X titlebar and icon name as a
# status line.  There are a few problems in using them in entries:
#
# a) tsl should have a parameter to denote the column on which to transfer to
#    the status line.
# b) the "0" code for xterm updates both icon-title and window title.  Some
#    window managers such as twm (and possibly window managers descended from
#    it such as tvtwm, ctwm, and vtwm) track windows by icon-name. Thus, you
#    don't want to mess with icon-name when using those window managers.
xterm+sl|access X title line and icon name,
	hs,
	dsl=\E]0;\007, fsl=^G, tsl=\E]0;,
xterm+sl-twm|access X title line (pacify twm-descended window managers),
	hs,
	dsl=\E]2;\007, fsl=^G, tsl=\E]2;,

# In contrast, this block can be used for a DEC vt320 and up.  There are two
# controls used.
#
# DECSASD (select active status display)
#	\E[0$}	Main display
#	\E[1$}	Status line
#
# DECSSDT (select status line type)
#	\E[0$~	No status line
#	\E[1$~	Indicator status line
#	\E[2$~	Host-writable status line
#
# The building block assumes that the terminal always shows something at the
# status line (either the indicator, or status line).  That is because if no
# status line is used, then the terminal makes that line part of the user
# window, changing its size without notice.
#
# Because there is no "esl" (enable status line) capability, the "tsl"
# capability ensures that the status line is host-writable.  A DEC terminal
# will clear the status line when changing from indicator to host-writable
# mode.
#
# Once on the status line, the row part of cursor addressing is ignored.  Since
# tsl expects a parameter (to specify the column), the shortest addressing that
# can be used for this purpose is HPA, e.g., \E[5d to go to column 5.
#
dec+sl|DEC VTxx status line,
	eslok, hs,
	dsl=\E[1$~, fsl=\E[0$}, tsl=\E[2$~\E[1$}\E[%i%p1%d`,

#
# The following xterm variants don't depend on your base version
#
# xterm with bold instead of underline
xterm-bold|xterm terminal emulator (X11R6 Window System) standout w/bold,
	sgr=%?%p9%t\016%e\017%;B\E[0%?%p6%t;1%;%?%p2%t;1%;%?%p1%p3%|%t;7%;m,
	smso=\E[7m, smul=\E[1m, use=xterm-old,

# See the note on ICH/ICH1 VERSUS RMIR/SMIR near the end of file
xterm-nic|xterm with ich/ich1 suppressed for non-curses programs,
	ich@, ich1@, use=xterm,
# From: Mark Sheppard <kimble@mistral.co.uk>, 4 May 1996
xterm1|xterm terminal emulator ignoring the alternate screen buffer,
	rmcup@, smcup@, use=xterm,

#### KTERM
# (kterm: this had extension capabilities ":KJ:TY=ascii:" -- esr)
# (kterm should not invoke DEC Graphics as the alternate character set
#  -- Kenji Rikitake)
# (proper setting of enacs, smacs, rmacs makes kterm to use DEC Graphics
#  -- MATSUMOTO Shoji)
# kterm implements acsc via built-in table of X Drawable's
kterm|kterm kanji terminal emulator (X window system),
	eslok, hs, XT,
	ncv@,
	acsc=``aajjkkllmmnnooppqqrrssttuuvvwwxx~~,
	csr=\E[%i%p1%d;%p2%dr, dsl=\E[?H, enacs=, fsl=\E[?F,
	kmous=\E[M, rc=\E8, rmacs=\E(B, rmam=\E[?7l, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;,
	sgr0=\E[m\E(B, smacs=\E(0, smam=\E[?7h,
	tsl=\E[?E\E[?%i%p1%dT, use=xterm-r6, use=ecma+color,
kterm-color|kterm-co|kterm with ANSI colors,
	ncv@, use=kterm, use=ecma+color,

#### Other XTERM
# These (xtermc and xtermm) are distributed with Solaris.  They refer to a
# variant of xterm which is apparently no longer supported, but are interesting
# because they illustrate SVr4 curses mouse controls - T.Dickey
xtermm|xterm terminal emulator (monocrome),
	OTbs, am, km, mir, msgr, xenl,
	btns#3, cols#80, it#8, lines#24,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink@, bold=\E[1m, clear=\E[H\E[2J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=\E[D,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, el1=\E[1K$<3>, enacs=\E(B\E)0, getm=\E[%p1%dY,
	home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@, ich1=\E[@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, kbs=^H, kcub1=\EOD,
	kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kend=\E[Y, kf0=\EOy,
	kf10=\EOY, kf11=\EOZ, kf12=\EOA, kf5=\EOT, kf6=\EOU,
	kf7=\EOV, kf8=\EOW, kf9=\EOX, khome=\E[H, kmous=\E[^_,
	knp=\E[U, kpp=\E[V, rc=\E8, reqmp=\E[492Z, rev=\E[7m, ri=\EM,
	rmacs=^O, rmcup=\E@0\E[?4r, rmso=\E[m,
	rs1=\E>\E[1;3;4;5;6l\E[?7h\E[m\E[r\E[2J\E[H,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smcup=\E@0\E[?4s\E[?4h\E@1,
	smso=\E[7m, tbc=\E[3g, use=vt100+fnkeys,

xtermc|xterm terminal emulator (color),
	colors#8, ncv#7, pairs#64,
	op=\E[100m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	setb=\E[4%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	setf=\E[3%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	use=xtermm,

# From: David J. MacKenzie <djm@va.pubnix.com> 20 Apr 1995
# Here's a termcap entry I've been using for xterm_color, which comes
# with BSD/OS 2.0, and the X11R6 contrib tape too I think.  Besides the
# color stuff, I also have a status line defined as the window manager
# title bar. [I have translated it to terminfo -- ESR]
xterm-pcolor|xterm with color used for highlights and status line,
	wsl#40,
	bold=\E[1;43m, rev=\E[7;34m,
	sgr=%?%p9%t\E(0%e\E(B%;\E[0%?%p6%t;1;43%;%?%p2%t;4;42%;%?%p1%t;7;31%;%?%p3%t;7;34%;%?%p4%t;5%;%?%p7%t;8%;m,
	smso=\E[7;31m, smul=\E[4;42m, use=xterm+sl, use=xterm-r6,

# This describes the capabilities of color_xterm, an xterm variant from
# before ECMA-64 color support was folded into the main-line xterm release.
# This entry is straight from color_xterm's maintainer.
# From: Jacob Mandelson <jlm@ugcs.caltech.edu>, 09 Nov 1996
# The README's with the distribution also say that it supports SGR 21, 24, 25
# and 27, but they are not present in the terminfo or termcap.
color_xterm|cx|cx100|color_xterm color terminal emulator for X,
	OTbs, am, km, mir, msgr, xenl, XT,
	cols#80, it#8, lines#65, ncv@,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, bold=\E[1m, clear=\E[H\E[2J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, el1=\E[1K, enacs=\E(B\E)0, home=\E[H, ht=^I,
	ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L, ind=^J,
	is1=\E[r\E[m\E[?7h\E[?4;6l\E[4l, kbs=^H, kcub1=\EOD,
	kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kend=\E[8~, kf1=\E[11~,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf2=\E[12~,
	kf3=\E[13~, kf4=\E[14~, kf5=\E[15~, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, khome=\E[7~, kich1=\E[2~,
	kmous=\E[M, knp=\E[6~, kpp=\E[5~, rc=\E8, rev=\E[7m, ri=\EM,
	rmacs=^O, rmam=\E[?7l, rmcup=\E>\E[?41;1r, rmir=\E[4l,
	rmso=\E[27m, rmul=\E[24m,
	rs1=\E(B\017\E[r\E[m\E[2J\E[H\E[?7h\E[?1;3;4;6l\E[4l\E<,
	sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;,
	sgr0=\E[0m\017, smacs=^N, smam=\E[?7h,
	smcup=\E[?1;41s\E[?1;41h\E=, smir=\E[4h, smso=\E[7m,
	smul=\E[4m, use=ecma+color, use=vt220+keypad,

# The 'nxterm' distributed with Redhat Linux 5.2 is a slight rehack of
# xterm-sb_right-ansi-3d, which implements ANSI colors, but does not support
# SGR 39 or 49.  SGR 0 does reset colors (along with everything else).  This
# description is "compatible" with color_xterm, rxvt and XFree86 xterm, except
# that each of those implements the home, end, delete keys differently.
#
# Redhat Linux 6.x distributes XFree86 xterm as "nxterm", which uses bce
# colors; note that this is not compatible with the 5.2 version.
# csw (2002-05-15): make xterm-color primary instead of nxterm, to
#   match XFree86's xterm.terminfo usage and prevent circular links
xterm-color|nxterm|generic color xterm,
	ncv@,
	op=\E[m, use=xterm-r6, use=klone+color,

# This entry describes an xterm with Sun-style function keys enabled
# via the X resource setting "xterm*sunFunctionKeys:true"
# To understand <kf11>/<kf12> note that L1,L2 and F11,F12 are the same.
# The <kf13>...<kf20> keys are L3-L10.  We don't set <kf16=\E[197z>
# because we want it to be seen as <kcpy>.
# The <kf31>...<kf45> keys are R1-R15.  We treat some of these in accordance
# with their Sun keyboard labels instead.
# From: Simon J. Gerraty <sjg@zen.void.oz.au> 10 Jan 1996
xterm-sun|xterm with sunFunctionKeys true,
	kb2=\E[218z, kcpy=\E[197z, kcub1=\EOD, kcud1=\EOB,
	kcuf1=\EOC, kcuu1=\EOA, kdch1=\E[3z, kend=\E[220z,
	kent=\EOM, kf1=\E[224z, kf10=\E[233z, kf11=\E[192z,
	kf12=\E[193z, kf13=\E[194z, kf14=\E[195z, kf15=\E[196z,
	kf17=\E[198z, kf18=\E[199z, kf19=\E[200z, kf2=\E[225z,
	kf20=\E[201z, kf3=\E[226z, kf31=\E[208z, kf32=\E[209z,
	kf33=\E[210z, kf34=\E[211z, kf35=\E[212z, kf36=\E[213z,
	kf38=\E[215z, kf4=\E[227z, kf40=\E[217z, kf42=\E[219z,
	kf44=\E[221z, kf45=\E[222z, kf46=\E[234z, kf47=\E[235z,
	kf5=\E[228z, kf6=\E[229z, kf7=\E[230z, kf8=\E[231z,
	kf9=\E[232z, kfnd=\E[200z, khlp=\E[196z, khome=\E[214z,
	kich1=\E[2z, knp=\E[222z, kpp=\E[216z, kund=\E[195z,
	use=xterm-basic,
xterms-sun|small (80x24) xterm with sunFunctionKeys true,
	cols#80, lines#24, use=xterm-sun,

#### GNOME (VTE)
# this describes the alpha-version of Gnome terminal shipped with Redhat 6.0
gnome-rh62|Gnome terminal,
	bce,
	kdch1=\177, kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS,
	use=xterm-color,

# GNOME Terminal 1.4.0.4 (Redhat 7.2)
#
# This implements a subset of vt102 with a random selection of features from
# other terminals such as color and function-keys.
#
# shift-f1 to shift-f10 are f11 to f20
#
# NumLock changes the application keypad to approximate vt100 keypad, except
# that there is no escape sequence matching comma (,).
#
# Other defects observed:
#	vt100 LNM mode is not implemented.
#	vt100 80/132 column mode is not implemented.
#	vt100 DECALN is not implemented.
#	vt100 DECSCNM mode is not implemented, so flash does not work.
#	vt100 TBC (tab reset) is not implemented.
#	xterm alternate screen controls do not restore cursor position properly
#	it hangs in tack after running function-keys test.
gnome-rh72|GNOME Terminal,
	bce, km@,
	civis=\E[?25l, cnorm=\E[?25h, kdch1=\E[3~, kf1=\EOP,
	kf2=\EOQ, kf3=\EOR, kf4=\EOS, rmam=\E[?7l,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;m%?%p9%t\016%e\017%;,
	sgr0=\E[0m\017, smam=\E[?7h, tbc@, use=xterm-color,

# GNOME Terminal 2.0.1 (Redhat 8.0)
#
# Documentation now claims it implements vt220 (which is demonstrably false).
# However, it does implement ECH, which is a vt220 feature.  And there are
# workable vt100 LNM, DECALN, DECSNM modes, making it possible to display
# more of its bugs using vttest.
#
# However, note that bce and msgr are broken in this release.  Tabs (tbc and
# hts) are broken as well.  Sometimes flash (as in xterm-new) works.
#
# kf1 and kf10 are not tested since they're assigned (hardcoded?) to menu
# operations.  Shift-tab generates a distinct sequence so it can be argued
# that it implements kcbt.
gnome-rh80|GNOME Terminal,
	bce@, msgr@,
	ech=\E[%p1%dX, flash=\E[?5h$<100/>\E[?5l, kbs=\177,
	kcbt=\E^I, op=\E[39;49m, use=gnome-rh72,

# GNOME Terminal 2.2.1 (Redhat 9.0)
#
# bce and msgr are repaired.
gnome-rh90|GNOME Terminal,
	bce, msgr, XT,
	hpa=\E[%i%p1%dG, kDC=\E[3;2~, kLFT=\EO2D, kRIT=\EO2C,
	kb2=\E[E, kcbt=\E[Z, kend=\EOF, khome=\EOH, tbc=\E[3g,
	vpa=\E[%i%p1%dd, use=xterm+pcf0, use=xterm+pcfkeys,
	use=gnome-rh80,

# GNOME Terminal 2.14.2 (Fedora Core 5)
# Ed Catmur notes that gnome-terminal has recognized soft-reset since May 2002.
gnome-fc5|GNOME Terminal,
	rs1=\Ec,
	rs2=\E7\E[r\E8\E[m\E[?7h\E[!p\E[?1;3;4;6l\E[4l\E>\E[?1000l\E[?25h,
	use=ansi+enq, use=xterm+pcc0, use=gnome-rh90,

# GNOME Terminal 2.18.1 (2007 snapshot)
#
# For any "recent" version of gnome-terminal, it is futile to attempt to
# support modifiers on cursor- and keypad keys because the program usually
# is hardcoded to set $TERM to "xterm", and on startup, it builds a subset
# of the keys (which more/less correspond to the termcap values), and will
# interpret those according to the $TERM value, but others not in the
# terminfo according to some constantly changing set of hacker guidelines -TD
vte-2007|VTE in GNOME Terminal snapshot 2.18.1,
	use=xterm+pcc2, use=gnome-fc5,
gnome-2007|GNOME Terminal snapshot 2.18.1,
	use=vte-2007,

# GNOME Terminal 2.22.3 (2008 snapshot)
#
# In vttest, it claims to be a vt220 with national replacement character-sets,
# but aside from the identifier string, implements only a small fraction of
# vt220's behavior, which will make it less usable on a VMS system (unclear
# what the intent of the developer is, since the NRC feature exposed in vttest
# by this change does not work).
vte-2008|VTE in GNOME Terminal snapshot 2.22.3,
	use=vte+pcfkeys, use=vte-2007,
gnome-2008|GNOME Terminal snapshot 2.22.3,
	use=vte-2008,

# GNOME Terminal 3.6.0 (2012)
# VTE 0.34.1 was marked in git 2012-10-15 (three days after patch was applied
# in ncurses).  It inherited from gnome-fc5, which broke the modified forms
# of f1-f4 -TD
#
# Testing with tack shows that flash does not/has not worked -TD
vte-2012|VTE 0.34.1,
	flash@, ritm=\E[23m, sitm=\E[3m, use=vte-2008,
# Version 3.6.1 sets TERM to xterm-256color (still hardcoded), which has
# 61 differences from a correct entry for gnome terminal.
gnome-2012|GNOME Terminal 3.6.0,
	use=vte-2012,

# GNOME terminal may automatically use the contents of the "xterm" terminfo to
# supply key information which is not built into the program.  With 2.22.3,
# this list is built into the program (which addresses the inadvertant use of
# random terminfo data, though using a set of values which does not correspond
# to any that xterm produces - still not solving the problem that GNOME
# terminal hardcodes the $TERM variable as "xterm").
#
#	terminfo	modifier	code	keys
#	kf13-kf24	shift		2	F1 to F12
#	kf25-kf36	control		5	F1 to F12
#	kf37-kf48	shift/control	6	F1 to F12
#	kf49-kf60	alt		3	F1 to F12
#	kf61-kf63	shift-alt	4	F1 to F3
#
# The parameters with \EO (SS3) are technically an error, since SS3 should have
# no parameters.  This appears to be rote copying based on xterm+pcc0.
vte+pcfkeys|VTE's variation on xterm+pcfkeys,
	kf1=\EOP, kf13=\EO1;2P, kf14=\EO1;2Q, kf15=\EO1;2R,
	kf16=\EO1;2S, kf2=\EOQ, kf25=\EO1;5P, kf26=\EO1;5Q,
	kf27=\EO1;5R, kf28=\EO1;5S, kf3=\EOR, kf37=\EO1;6P,
	kf38=\EO1;6Q, kf39=\EO1;6R, kf4=\EOS, kf40=\EO1;6S,
	kf49=\EO1;3P, kf50=\EO1;3Q, kf51=\EO1;3R, kf52=\EO1;3S,
	kf61=\EO1;4P, kf62=\EO1;4Q, kf63=\EO1;4R,
	use=xterm+pcfkeys,
gnome+pcfkeys|VTE's variation on xterm+pcfkeys,
	use=vte+pcfkeys,

vte|VTE aka GNOME Terminal,
	use=vte-2012,
gnome|GNOME Terminal,
	use=vte,

# palette is hardcoded...
vte-256color|VTE with xterm 256-colors,
	use=xterm+256color, use=vte,
gnome-256color|GNOME Terminal with xterm 256-colors,
	use=vte-256color,

# XFCE Terminal 0.2.5.4beta2
#
# This is based on some of the same source code, e.g., the VTE library, as
# gnome-terminal, but has fewer features, fails more screens in vttest.
# Since most of the terminfo-related behavior is due to the VTE library,
# the terminfo is the same as gnome-terminal.
xfce|Xfce Terminal,
	use=vte-2008,

#### Other GNOME
# Multi-Gnome-Terminal 1.6.2
#
# This does not use VTE, and does have different behavior (compare xfce and
# gnome).
mgt|Multi GNOME Terminal,
	indn=\E[%p1%dS, rin=\E[%p1%dT, use=xterm-xf86-v333,

#### KDE
# This is kvt 0-18.7, shipped with Redhat 6.0 (though whether it supports bce
# or not is debatable).
kvt|KDE terminal,
	bce, km@,
	kdch1=\177, kend=\E[F, khome=\E[H, use=xterm-color,

# Konsole 1.0.1
# (formerly known as kvt)
#
# This program hardcodes $TERM to 'xterm', which is not accurate.  However, to
# simplify this entry (and point out why konsole isn't xterm), we base this on
# xterm-r6.  The default keyboard appears to be 'linux'.
#
# Notes:
# a) konsole implements several features from XFree86 xterm, though none of
#    that is documented - except of course in its source code - apparently
#    because its implementors are unaccustomed to reading documentation - as
#    evidenced by the sparse and poorly edited documentation distributed with
#    konsole.  Some features such as the 1049 private mode are recognized but
#    incorrectly implemented as a duplicate of the 47 private mode.
# b) even with the "vt100 (historical)" keyboard setting, the numeric keypad
#    sends PC-style escapes rather than vt100.
# c) fails vttest menu 3 (Test of character sets) because it does not properly
#    parse some control sequences.  Also fails vttest Primary Device Attributes
#    by sending a bogus code (in the source it says it's supposed to be a
#    vt220, which is doubly incorrect because it does not implement vt220
#    control sequences except for a few special cases).  Treat it as a
#    mildly-broken vt102.
#
# Update for konsole 1.3.2:
#    The 1049 private mode works (but see the other xterm screens in vttest).
#    Primary Device Attributes now returns the code for a vt100 with advanced
#    video option.  Perhaps that's intended to be a "mildly-broken vt102".
#
# Updated for konsole 1.6.4:
#    add konsole-solaris
#
# Updated for konsole 1.6.6:
#    add control-key modifiers for function-keys, etc.
#
# vttest menu 1 shows that both konsole and gnome terminal do wrapping
# different from xterm (and vt100's).  They have the same behavior in this
# detail, but it is unclear which copies the other.
konsole-base|KDE console window,
	bce, km@, npc, XT,
	ncv@,
	bel@, blink=\E[5m, civis=\E[?25l, cnorm=\E[?25h,
	ech=\E[%p1%dX, flash=\E[?5h$<100/>\E[?5l,
	hpa=\E[%i%p1%dG, indn=\E[%p1%dS, kbs=\177, kdch1=\E[3~,
	kend=\E[4~, kf1@, kf10@, kf11@, kf12@, kf13@, kf14@, kf15@, kf16@,
	kf17@, kf18@, kf19@, kf2@, kf20@, kf3@, kf4@, kf5@, kf6@, kf7@, kf8@,
	kf9@, kfnd@, khome=\E[1~, kslt@, rin=\E[%p1%dT, rmam=\E[?7l,
	rmso=\E[27m, rmul=\E[24m,
	rs2=\E7\E[r\E8\E[m\E[?7h\E[?1;3;4;6l\E[4l\E>\E[?1000l\E[?25h,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;,
	sgr0=\E[0m\017, smam=\E[?7h, vpa=\E[%i%p1%dd,
	use=ecma+color, use=xterm-r6,
konsole-linux|KDE console window with linux keyboard,
	kf1=\E[[A, kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf13@,
	kf14@, kf15@, kf16@, kf17@, kf18@, kf19@, kf2=\E[[B, kf20@,
	kf3=\E[[C, kf4=\E[[D, kf5=\E[[E, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, use=konsole-base,
konsole-solaris|KDE console window with Solaris keyboard,
	kbs=^H, kend=\E[4~, khome=\E[1~, use=konsole-vt100,
# KDE's "XFree86 3.x.x" keyboard is based on reading the xterm terminfo rather
# than testing the code.
konsole-xf3x|KDE console window with keyboard for XFree86 3.x xterm,
	kend=\E[4~, khome=\E[1~, use=konsole-vt100,
# The value for kbs reflects local customization rather than the settings used
# for XFree86 xterm.
konsole-xf4x|KDE console window with keyboard for XFree86 4.x xterm,
	kend=\EOF, khome=\EOH, use=konsole+pcfkeys,
	use=konsole-vt100,
# Konsole does not implement shifted cursor-keys.
konsole+pcfkeys|konsole subset of xterm+pcfkeys,
	kLFT@, kRIT@, kcbt=\E[Z, kind@, kri@, kDN@, kUP@, use=xterm+pcc2,
	use=xterm+pcf0,
# KDE's "vt100" keyboard has no relationship to any terminal that DEC made, but
# it is still useful for deriving the other entries.
konsole-vt100|KDE console window with vt100 (sic) keyboard,
	kbs=\177, kdch1=\E[3~, kend=\E[F, kf1=\E[11~, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf13@, kf14@, kf15@, kf16@, kf17@,
	kf18@, kf19@, kf2=\E[12~, kf20@, kf3=\E[13~, kf4=\E[14~,
	kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[H, use=konsole-base,
konsole-vt420pc|KDE console window with vt420 pc keyboard,
	kbs=^H, kdch1=\177, use=konsole-vt100,
konsole-16color|klone of xterm-16color,
	ncv#32, use=ibm+16color, use=konsole,
# make a default entry for konsole
konsole|KDE console window,
	use=konsole-xf4x,

# palette is hardcoded...
konsole-256color|KDE console window with xterm 256-colors,
	initc@, use=xterm+256color, use=konsole,

#### MLTERM
# This is mlterm 2.9.3's mlterm.ti, with some additions/corrections -TD
#
# It is nominally a vt102 emulator, with features borrowed from rxvt and
# xterm.
#
# The function keys are numbered based on shift/control/alt modifiers, except
# that the control-modifier itself is used to spawn a new copy of mlterm (the
# "-P" option).  So control/F1 to control/F12 may not be usable, depending on
# how it is configured.
#
#				kf1 to kf12	\E[11~   to \E[24~
#	shift			kf1 to kf12	\E[11;2~ to \E[24;2~
#	alt			kf1 to kf12	\E[11;3~ to \E[24;3~
#	shift/alt		kf1 to kf12	\E[11;4~ to \E[24;4~
#	control			kf1 to kf12	\E[11;5~ to \E[24;5~ (maybe)
#	control/shift		kf1 to kf12	\E[11;6~ to \E[24;6~
#	control/alt		kf1 to kf12	\E[11;7~ to \E[24;7~
#	control/shift/alt	kf1 to kf12	\E[11;8~ to \E[24;8~
#
mlterm|multi lingual terminal emulator,
	am, eslok, km, mc5i, mir, msgr, npc, xenl, XT,
	colors#8, cols#80, it#8, lines#24, pairs#64,
	acsc=00``aaffgghhjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[2J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K, enacs=,
	home=\E[H, hpa=\E[%i%p1%dG, ht=^I, hts=\EH, ich=\E[%p1%d@,
	ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=^J, indn=\E[%p1%dS,
	is2=\E7\E[r\E[m\E[?7h\E[?1;3;4;6l\E[4l\E8\E>,
	kbs=\177, kcbt=\E[Z, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC,
	kcuu1=\EOA, kdch1=\E[3~, kend=\EOF, kent=\EOM, kfnd=\E[1~,
	khome=\EOH, kich1=\E[2~, kind=\EO1;2B, kmous=\E[M,
	knp=\E[6~, kpp=\E[5~, kri=\EO1;2A, kslt=\E[4~, mc0=\E[i,
	nel=\EE, op=\E[39;49m, rc=\E8, rev=\E[7m, ri=\EM,
	rin=\E[%p1%dT, rmacs=\E(B, rmam=\E[?7l, rmcup=\E[?1049l,
	rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[27m, rmul=\E[24m,
	rs2=\E7\E[r\E8\E[m\E[?7h\E[?1;3;4;6l\E[4l\E>\E[?1000l,
	sc=\E7, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;,
	sgr0=\E[m\E(B, smacs=\E(0, smam=\E[?7h, smcup=\E[?1049h,
	smir=\E[4h, smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g, u6=\E[%i%d;%dR, u7=\E[6n, u8=\E[?1;2c, u9=\E[c,
	vpa=\E[%i%p1%dd, use=mlterm+pcfkeys, use=xterm+r6f2,

# The insert/delete/home/end keys do not respond to modifiers because mlterm
# looks in its termcap to decide which string to send.  If it used terminfo
# (when available), it could use the extended names introduced for xterm.
mlterm+pcfkeys|fragment for PC-style fkeys,
	kLFT=\EO1;2D, kNXT=\E[6;2~, kPRV=\E[5;2~, kRIT=\EO1;2C,
	kDN=\EO1;2B, kDN3=\EO1;3B, kDN4=\EO1;4B, kDN5=\EO1;5B,
	kDN6=\EO1;6B, kDN7=\EO1;7B, kIC5=\E[2;5~, kIC6=\E[2;6~,
	kLFT3=\EO1;3D, kLFT4=\EO1;4D, kLFT5=\EO1;5D,
	kLFT6=\EO1;6D, kLFT7=\EO1;7D, kNXT5=\E[6;5~,
	kNXT6=\E[6;6~, kPRV5=\E[5;5~, kPRV6=\E[5;6~,
	kRIT3=\EO1;3C, kRIT4=\EO1;4C, kRIT5=\EO1;5C,
	kRIT6=\EO1;6C, kRIT7=\EO1;7C, kUP=\EO1;2A, kUP3=\EO1;3A,
	kUP4=\EO1;4A, kUP5=\EO1;5A, kUP6=\EO1;6A, kUP7=\EO1;7A,

mlterm-256color|mlterm 3.0 with xterm 256-colors,
	use=xterm+256color, use=rxvt,

#### RXVT
# From: Thomas Dickey <dickey@clark.net> 04 Oct 1997
# Updated: Oezguer Kesim <kesim@math.fu-berlin.de> 02 Nov 1997
# Notes:
# rxvt 2.21b uses
#	smacs=\E(B\E)U^N, rmacs=\E(B\E)0^O,
# but some applications don't work with that.
# It also has an AIX extension
#	box2=lqkxjmwuvtn,
# and
#	ech=\E[%p1%dX,
# but the latter does not work correctly.
#
# The distributed terminfo says it implements hpa and vpa, but they are not
# implemented correctly, using relative rather than absolute positioning.
#
# rxvt is normally configured to look for "xterm" or "xterm-color" as $TERM.
# Since rxvt is not really compatible with xterm, it should be configured as
# "rxvt" or "rxvt-color".
#
# removed dch/dch1 because they are inconsistent with bce/ech -TD
# remove km as per tack test -TD
rxvt-basic|rxvt terminal base (X Window System),
	OTbs, am, bce, eo, mir, msgr, xenl, xon, XT,
	cols#80, it#8, lines#24,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[2J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K,
	enacs=\E(B\E)0, flash=\E[?5h\E[?5l, home=\E[H, ht=^I,
	hts=\EH, ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[L,
	ind=^J, is1=\E[?47l\E=\E[?1l,
	is2=\E[r\E[m\E[2J\E[H\E[?7h\E[?1;3;4;6l\E[4l, kbs=^H,
	kcbt=\E[Z, kmous=\E[M, rc=\E8, rev=\E[7m, ri=\EM, rmacs=^O,
	rmcup=\E[2J\E[?47l\E8, rmir=\E[4l, rmkx=\E>, rmso=\E[27m,
	rmul=\E[24m,
	rs1=\E>\E[1;3;4;5;6l\E[?7h\E[m\E[r\E[2J\E[H,
	rs2=\E[r\E[m\E[2J\E[H\E[?7h\E[?1;3;4;6l\E[4l\E>\E[?1000l\E[?25h,
	s0ds=\E(B, s1ds=\E(0, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;,
	sgr0=\E[0m\017, smacs=^N, smcup=\E7\E[?47h, smir=\E[4h,
	smkx=\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g, use=vt100+enq,
	use=rxvt+pcfkeys, use=vt220+keypad,
# Key Codes from rxvt reference:
#
# Note: Shift + F1-F10 generates F11-F20
#
# For the keypad, use Shift to temporarily override Application-Keypad
# setting use Num_Lock to toggle Application-Keypad setting if Num_Lock
# is off, escape sequences toggle Application-Keypad setting.
# Also note that values of Home, End, Delete may have been compiled
# differently on your system.
#
#                   Normal       Shift        Control      Ctrl+Shift
#  Tab              ^I           ESC [ Z      ^I           ESC [ Z
#  BackSpace        ^H           ^?           ^?           ^?
#  Find             ESC [ 1 ~    ESC [ 1 $    ESC [ 1 ^    ESC [ 1 @
#  Insert           ESC [ 2 ~    paste        ESC [ 2 ^    ESC [ 2 @
#  Execute          ESC [ 3 ~    ESC [ 3 $    ESC [ 3 ^    ESC [ 3 @
#  Select           ESC [ 4 ~    ESC [ 4 $    ESC [ 4 ^    ESC [ 4 @
#  Prior            ESC [ 5 ~    scroll-up    ESC [ 5 ^    ESC [ 5 @
#  Next             ESC [ 6 ~    scroll-down  ESC [ 6 ^    ESC [ 6 @
#  Home             ESC [ 7 ~    ESC [ 7 $    ESC [ 7 ^    ESC [ 7 @
#  End              ESC [ 8 ~    ESC [ 8 $    ESC [ 8 ^    ESC [ 8 @
#  Delete           ESC [ 3 ~    ESC [ 3 $    ESC [ 3 ^    ESC [ 3 @
#  F1               ESC [ 11 ~   ESC [ 23 ~   ESC [ 11 ^   ESC [ 23 ^
#  F2               ESC [ 12 ~   ESC [ 24 ~   ESC [ 12 ^   ESC [ 24 ^
#  F3               ESC [ 13 ~   ESC [ 25 ~   ESC [ 13 ^   ESC [ 25 ^
#  F4               ESC [ 14 ~   ESC [ 26 ~   ESC [ 14 ^   ESC [ 26 ^
#  F5               ESC [ 15 ~   ESC [ 28 ~   ESC [ 15 ^   ESC [ 28 ^
#  F6               ESC [ 17 ~   ESC [ 29 ~   ESC [ 17 ^   ESC [ 29 ^
#  F7               ESC [ 18 ~   ESC [ 31 ~   ESC [ 18 ^   ESC [ 31 ^
#  F8               ESC [ 19 ~   ESC [ 32 ~   ESC [ 19 ^   ESC [ 32 ^
#  F9               ESC [ 20 ~   ESC [ 33 ~   ESC [ 20 ^   ESC [ 33 ^
#  F10              ESC [ 21 ~   ESC [ 34 ~   ESC [ 21 ^   ESC [ 34 ^
#  F11              ESC [ 23 ~   ESC [ 23 $   ESC [ 23 ^   ESC [ 23 @
#  F12              ESC [ 24 ~   ESC [ 24 $   ESC [ 24 ^   ESC [ 24 @
#  F13              ESC [ 25 ~   ESC [ 25 $   ESC [ 25 ^   ESC [ 25 @
#  F14              ESC [ 26 ~   ESC [ 26 $   ESC [ 26 ^   ESC [ 26 @
#  F15 (Help)       ESC [ 28 ~   ESC [ 28 $   ESC [ 28 ^   ESC [ 28 @
#  F16 (Menu)       ESC [ 29 ~   ESC [ 29 $   ESC [ 29 ^   ESC [ 29 @
#  F17              ESC [ 31 ~   ESC [ 31 $   ESC [ 31 ^   ESC [ 31 @
#  F18              ESC [ 32 ~   ESC [ 32 $   ESC [ 32 ^   ESC [ 32 @
#  F19              ESC [ 33 ~   ESC [ 33 $   ESC [ 33 ^   ESC [ 33 @
#  F20              ESC [ 34 ~   ESC [ 34 $   ESC [ 34 ^   ESC [ 34 @
#
#                                                          Application
#  Up               ESC [ A      ESC [ a      ESC O a      ESC O A
#  Down             ESC [ B      ESC [ b      ESC O b      ESC O B
#  Right            ESC [ C      ESC [ c      ESC O c      ESC O C
#  Left             ESC [ D      ESC [ d      ESC O d      ESC O D
#  KP_Enter         ^M                                     ESC O M
#  KP_F1            ESC O P                                ESC O P
#  KP_F2            ESC O Q                                ESC O Q
#  KP_F3            ESC O R                                ESC O R
#  KP_F4            ESC O S                                ESC O S
#  XK_KP_Multiply   *                                      ESC O j
#  XK_KP_Add        +                                      ESC O k
#  XK_KP_Separator  ,                                      ESC O l
#  XK_KP_Subtract   -                                      ESC O m
#  XK_KP_Decimal    .                                      ESC O n
#  XK_KP_Divide     /                                      ESC O o
#  XK_KP_0          0                                      ESC O p
#  XK_KP_1          1                                      ESC O q
#  XK_KP_2          2                                      ESC O r
#  XK_KP_3          3                                      ESC O s
#  XK_KP_4          4                                      ESC O t
#  XK_KP_5          5                                      ESC O u
#  XK_KP_6          6                                      ESC O v
#  XK_KP_7          7                                      ESC O w
#  XK_KP_8          8                                      ESC O x
#  XK_KP_9          9                                      ESC O y
#
# The source-code for rxvt actually defines mappings for F21-F35, using
# "ESC [ 35 ~" to "ESC [  49 ~".  Keyboards with more than 12 function keys
# are rare, so this entry uses the shift- and control-modifiers as in
# xterm+pcfkeys to define keys past F12.
#
# kIC is normally not used, since rxvt performs a paste for that (shifted
# insert), unless private mode 35 is set.
#
# kDN, kDN5, kDN6, etc are extensions based on the names from xterm+pcfkeys -TD
# Removed kDN6, etc (control+shift) since rxvt does not implement this -TD
rxvt+pcfkeys|fragment for PC-style fkeys,
	kDC=\E[3$, kEND=\E[8$, kHOM=\E[7$, kIC=\E[2$, kLFT=\E[d,
	kNXT=\E[6$, kPRV=\E[5$, kRIT=\E[c, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~, kel=\E[8\^,
	kend=\E[8~, kf1=\E[11~, kf10=\E[21~, kf11=\E[23~,
	kf12=\E[24~, kf13=\E[25~, kf14=\E[26~, kf15=\E[28~,
	kf16=\E[29~, kf17=\E[31~, kf18=\E[32~, kf19=\E[33~,
	kf2=\E[12~, kf20=\E[34~, kf21=\E[23$, kf22=\E[24$,
	kf23=\E[11\^, kf24=\E[12\^, kf25=\E[13\^, kf26=\E[14\^,
	kf27=\E[15\^, kf28=\E[17\^, kf29=\E[18\^, kf3=\E[13~,
	kf30=\E[19\^, kf31=\E[20\^, kf32=\E[21\^, kf33=\E[23\^,
	kf34=\E[24\^, kf35=\E[25\^, kf36=\E[26\^, kf37=\E[28\^,
	kf38=\E[29\^, kf39=\E[31\^, kf4=\E[14~, kf40=\E[32\^,
	kf41=\E[33\^, kf42=\E[34\^, kf43=\E[23@, kf44=\E[24@,
	kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	kfnd=\E[1~, khome=\E[7~, kich1=\E[2~, kind=\E[a, knp=\E[6~,
	kpp=\E[5~, kri=\E[b, kslt=\E[4~, kDC5=\E[3\^, kDC6=\E[3@,
	kDN=\E[b, kDN5=\EOb, kEND5=\E[8\^, kEND6=\E[8@,
	kHOM5=\E[7\^, kHOM6=\E[7@, kIC5=\E[2\^, kIC6=\E[2@,
	kLFT5=\EOd, kNXT5=\E[6\^, kNXT6=\E[6@, kPRV5=\E[5\^,
	kPRV6=\E[5@, kRIT5=\EOc, kUP=\E[a, kUP5=\EOa,

rxvt|rxvt terminal emulator (X Window System),
	ncv@,
	hpa=\E[%i%p1%dG, kf0=\E[21~, sgr0=\E[m\017,
	vpa=\E[%i%p1%dd, use=rxvt-basic, use=ecma+color,
rxvt-color|rxvt terminal emulator (X Window System),
	use=rxvt,
rxvt-256color|rxvt 2.7.9 with xterm 256-colors,
	use=xterm+256color, use=rxvt,
rxvt-88color|rxvt 2.7.9 with xterm 88-colors,
	use=xterm+88color, use=rxvt,
rxvt-xpm|rxvt terminal emulator (X Window System),
	use=rxvt,
rxvt-cygwin|rxvt terminal emulator (X Window System) on cygwin,
	acsc=+\257\,\256-\^0\333`\004a\261f\370g\361h\260j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	use=rxvt,
rxvt-cygwin-native|rxvt terminal emulator (native MS Window System port) on cygwin,
	acsc=+\257\,\256-\^0\333`\004a\261f\370g\361h\260j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330~\376,
	use=rxvt-cygwin,

# This variant is supposed to work with rxvt 2.7.7 when compiled with
# NO_BRIGHTCOLOR defined.  rxvt needs more work...
rxvt-16color|xterm with 16 colors like aixterm,
	ncv#32, use=ibm+16color, use=rxvt,

#### MRXVT
# mrxvt 0.5.4
#
# mrxvt is based on rxvt 2.7.11, but has by default XTERM_FKEYS defined, which
# makes its function-keys different from other flavors of rxvt -TD
mrxvt|multitabbed rxvt,
	XT,
	kEND=\E[8;2~, kHOM=\E[7;2~, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kend=\E[8~, khome=\E[7~,
	kEND3=\E[8;3~, kEND4=\E[8;4~, kEND5=\E[8;5~,
	kEND6=\E[8;6~, kEND7=\E[8;7~, kHOM3=\E[7;3~,
	kHOM4=\E[7;4~, kHOM5=\E[7;5~, kHOM6=\E[7;6~,
	kHOM7=\E[7;7~, use=xterm+r6f2, use=xterm+pcfkeys,
	use=rxvt,

mrxvt-256color|multitabbed rxvt with 256 colors,
	use=xterm+256color, use=mrxvt,

#### ETERM
# From: Michael Jennings <mej@valinux.com>
#
# Eterm 0.9.3
#
# removed kf0 which conflicts with kf10 -TD
# remove cvvis which conflicts with cnorm -TD
# Eterm does not implement control/shift cursor keys such as kDN6, or kPRV/kNXT
# but does otherwise follow the rxvt+pcfkeys model -TD
# remove nonworking flash -TD
# remove km as per tack test -TD
Eterm|Eterm-color|Eterm with xterm-style color support (X Window System),
	am, bce, bw, eo, mc5i, mir, msgr, xenl, xon, XT,
	btns#5, cols#80, it#8, lines#24, lm#0, ncv@,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[2J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K, enacs=\E)0,
	home=\E[H, hpa=\E[%i%p1%dG, ht=^I, hts=\EH, ich=\E[%p1%d@,
	ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=^J,
	is1=\E[?47l\E>\E[?1l,
	is2=\E[r\E[m\E[2J\E[H\E[?7h\E[?1;3;4;6l\E[4l, kNXT@,
	kPRV@, ka1=\E[7~, ka3=\E[5~, kb2=\EOu, kbeg=\EOu, kbs=\177,
	kc1=\E[8~, kc3=\E[6~, kent=\EOM, khlp=\E[28~, kmous=\E[M,
	mc4=\E[4i, mc5=\E[5i, rc=\E8, rev=\E[7m, ri=\EM, rmacs=^O,
	rmam=\E[?7l, rmcup=\E[2J\E[?47l\E8, rmir=\E[4l, rmkx=,
	rmso=\E[27m, rmul=\E[24m,
	rs1=\E>\E[1;3;4;5;6l\E[?7h\E[m\E[r\E[2J\E[H,
	rs2=\E[r\E[m\E[2J\E[H\E[?7h\E[?1;3;4;6l\E[4l\E>\E[?1000l\E[?25h,
	sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smcup=\E7\E[?47h,
	smir=\E[4h, smkx=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	vpa=\E[%i%p1%dd, use=vt100+enq, use=rxvt+pcfkeys,
	use=ecma+color,

Eterm-256color|Eterm with xterm 256-colors,
	use=xterm+256color, use=Eterm,

Eterm-88color|Eterm with 88 colors,
	use=xterm+88color, use=Eterm,

#### ATERM
# Based on rxvt 2.4.8, it has a few differences in key bindings
aterm|AfterStep terminal,
	XT,
	kbs=\177, kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, use=rxvt,

#### XITERM
# xiterm  0.5-5.2
# This is not based on xterm's source...
# vttest shows several problems with keyboard, cursor-movements.
# see also http://invisible-island.net/xterm/xterm.faq.html#bug_xiterm
xiterm|internationalized terminal emulator for X,
	km@,
	kbs=\177, kdch1=\E[3~, use=klone+color, use=xterm-r6,


#### HPTERM
# HP ships this (HPUX 9 and 10), except for the pb#9600 which was merged in
# from BSD termcap.  (hpterm:  added empty <acsc>, we have no idea what ACS
# chars look like --esr)
hpterm|X-hpterm|hp X11 terminal emulator,
	am, da, db, mir, xhp,
	cols#80, lh#2, lines#24, lm#0, lw#8, nlab#8, pb#9600, xmc#0,
	acsc=, bel=^G, bold=\E&dB, cbt=\Ei, clear=\E&a0y0C\EJ, cr=^M,
	cub1=^H, cud1=\EB, cuf1=\EC, cup=\E&a%p1%dy%p2%dC,
	cuu1=\EA, dch1=\EP, dim=\E&dH, dl1=\EM, ed=\EJ$<1>, el=\EK,
	hpa=\E&a%p1%dC, ht=^I, hts=\E1, il1=\EL, ind=^J, kbs=^H,
	kclr=\EJ, kctab=\E2, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kdch1=\EP, kdl1=\EM, ked=\EJ, kel=\EK, kf1=\Ep,
	kf2=\Eq, kf3=\Er, kf4=\Es, kf5=\Et, kf6=\Eu, kf7=\Ev, kf8=\Ew,
	khome=\Eh, khts=\E1, kich1=\EQ, kil1=\EL, kind=\ES, kll=\EF,
	knp=\EU, kpp=\EV, kri=\ET, krmir=\ER, ktbc=\E3, meml=\El,
	memu=\Em, pfkey=\E&f%p1%dk%p2%l%dL%p2%s,
	pfloc=\E&f1a%p1%dk%p2%l%dL%p2%s,
	pfx=\E&f2a%p1%dk%p2%l%dL%p2%s,
	pln=\E&f%p1%dk%p2%l%dd0L%p2%s, rev=\E&dB, ri=\ET,
	rmacs=^O, rmir=\ER, rmkx=\E&s0A, rmln=\E&j@, rmso=\E&d@,
	rmul=\E&d@,
	sgr=\E&d%?%p7%t%{115}%c%;%p1%p3%|%p6%|%{2}%*%p2%{4}%*%+%p4%+%p5%{8}%*%+%{64}%+%c%?%p9%t%'\016'%c%e%'\017'%c%;,
	sgr0=\E&d@, smacs=^N, smir=\EQ, smkx=\E&s1A, smln=\E&jB,
	smso=\E&dJ, smul=\E&dD, tbc=\E3, vpa=\E&a%p1%dY,
# HPUX 11 provides a color version.
hpterm-color|HP X11 terminal emulator with color,
	ccc,
	colors#64, pairs#8,
	home=\E&a0y0C,
	initp=\E&v%p2%da%p3%db%p4%dc%p5%dx%p6%dy%p7%dz%p1%dI,
	op=\E&v0S, scp=\E&v%p1%dS, use=hpterm,

#### EMU
# This is for the extensible terminal emulator on the X11R6 contrib tape.
# It corresponds to emu's internal emulation:
#	emu -term emu
# emu's default sets TERM to "xterm", but that doesn't work well -TD
# fixes: remove bogus rmacs/smacs, change oc to op, add bce, am -TD
# fixes: add civis, cnorm, sgr -TD
emu|emu native mode,
	am, bce, mir, msgr, xon,
	colors#15, cols#80, it#8, lines#24, pairs#64, vt#200,
	acsc=61a\202f\260g2j\213k\214l\215m\216n\217o\220q\222s\224t\225u\226v\227w\230x\231~\244,
	bel=^G, blink=\EW, bold=\EU, civis=\EZ, clear=\EP\EE0;0;,
	cnorm=\Ea, cr=^M, csr=\Ek%p1%d;%p2%d;, cub=\Eq-%p1%d;,
	cub1=^H, cud=\Ep%p1%d;, cud1=\EB, cuf=\Eq%p1%d;, cuf1=\EC,
	cup=\EE%p1%d;%p2%d;, cuu=\Ep-%p1%d;, cuu1=\EA,
	dch=\EI%p1%d;, dch1=\EI1;, dl=\ER%p1%d;, dl1=\ER1;,
	ech=\Ej%p1%d;, ed=\EN, el=\EK, el1=\EL, home=\EE0;0;, ht=^I,
	hts=\Eh, il=\EQ%p1%d;, il1=\EQ1;, ind=\EG,
	is2=\ES\Er0;\Es0;, kbs=^H, kcub1=\EC, kcud1=\EB, kcuf1=\ED,
	kcuu1=\EA, kdch1=\177, kent=^M, kf0=\EF00, kf1=\EF01,
	kf10=\EF10, kf11=\EF11, kf12=\EF12, kf13=\EF13, kf14=\EF14,
	kf15=\EF15, kf16=\EF16, kf17=\EF17, kf18=\EF18, kf19=\EF19,
	kf2=\EF02, kf20=\EF20, kf3=\EF03, kf4=\EF04, kf5=\EF05,
	kf6=\EF06, kf7=\EF07, kf8=\EF08, kf9=\EF09, kfnd=\Efind,
	kich1=\Eins, knp=\Enext, kpp=\Eprior, kslt=\Esel,
	op=\Es0;\Er0;, rev=\ET, ri=\EF, rmir=\EX, rmso=\ES, rmul=\ES,
	rs2=\ES\Es0;\Er0;, setab=\Es%i%p1%d;,
	setaf=\Er%i%p1%d;,
	sgr=\ES%?%p1%t\ET%;%?%p2%t\EV%;%?%p3%t\ET%;%?%p4%t\EW%;%?%p6%t\EU%;,
	sgr0=\ES, smir=\EY, smso=\ET, smul=\EV, tbc=\Ej,

# vt220 Terminfo entry for the Emu emulation, corresponds to
#	emu -term vt220
# with NumLock set (to make the keypad transmit kf0-kf9).
# fixes: add am, xenl, corrected sgr0 -TD
emu-220|Emu-220 (vt200-7bit mode),
	am, xenl, xon,
	cols#80, it#8, lines#24, vt#200,
	acsc=aaffggjjkkllmmnnooqqssttuuvvwwxx~~, bel=^G,
	blink=\E[0;5m, bold=\E[0;1m, civis=\E[?25l,
	clear=\E[2J\E[H, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=\E[1D,
	cud=\E[%p1%dB, cud1=\E[1B, cuf=\E[%p1%dC, cuf1=\E[1C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[1A,
	dch=\E[%p1%dP, dch1=\E[1P, dl=\E[%p1%dM, dl1=\E[1M,
	ed=\E[0J, el=\E[0K, el1=\E[1K, enacs=\E)0, home=\E[H, ht=^I,
	hts=\EH, if=/usr/share/tabset/vt300, il=\E[%p1%dL,
	il1=\E[1L, ind=\ED, is2=\E>\E[?1l\E[?3l\E[4l\E[?7h,
	kbs=^H, kcmd=\E[29~, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kdch1=\E[3~, kent=\EOM, kf0=\EOp, kf1=\EOq,
	kf10=\EOl, kf11=\EOm, kf12=\EOn, kf13=\EOP, kf14=\EOQ,
	kf15=\EOR, kf16=\EOS, kf2=\EOr, kf26=\E[17~, kf27=\E[18~,
	kf28=\E[19~, kf29=\E[20~, kf3=\EOs, kf30=\E[21~,
	kf34=\E[26~, kf37=\E[31~, kf38=\E[32~, kf39=\E[33~,
	kf4=\EOt, kf40=\E[34~, kf5=\EOu, kf6=\EOv, kf7=\EOw,
	kf8=\EOx, kf9=\EOy, kfnd=\E[1~, khlp=\E[28~, kich1=\E[2~,
	knp=\E[6~, kpp=\E[5~, kslt=\E[4~, rc=\E8, rev=\E[0;7m,
	ri=\EM, rmacs=^O, rmcup=\E>, rmkx=\E>, rmso=\E[m, rmul=\E[m,
	rs2=\E[4l\E[34l\E[?1l\E[?3l\E[?5l\E[?7h, sc=\E7,
	sgr=\E[0%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m, smacs=^N, smcup=\E[?1l\E=, smkx=\E=,
	smso=\E[0;7m, smul=\E[0;4m, tbc=\E[3g,

#### MVTERM
# A commercial product, Reportedly a version of Xterm with an OPEN LOOK UI,
# print interface, ANSI X3.64 colour escape sequences, etc.  Newsgroup postings
# indicate that it emulates more than one terminal, but incompletely.
#
# This is adapted from a FreeBSD bug-report by Daniel Rudy <dcrudy@pacbell.net>
# It is based on vt102's entry, with some subtle differences, but also
#	has status line
#	supports ANSI colors (except for 'op' string)
#	apparently implements alternate screen like xterm
#	does not use padding, of course.
mvterm|vv100|SwitchTerm aka mvTERM,
	am, eslok, hs, km, mir, msgr, xenl,
	colors#8, cols#80, it#8, lines#24, pairs#64,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[2J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	dsl=\E[?E, ed=\E[J, el=\E[K, el1=\E[1K$<3>, enacs=\E(B\E)0,
	fsl=\E[?F, home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@,
	ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=^J, kbs=^H,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kf0=\EOy,
	kf10=\EOx, kf5=\EOt, kf6=\EOu, kf7=\EOv, kf8=\EOl, kf9=\EOw,
	op=\E[100m, rc=\E8, rev=\E[7m, ri=\EM, rmacs=^O,
	rmcup=\E[2J\E[?47l\E8, rmkx=\E[?1l\E>, rmso=\E[m,
	rmul=\E[m,
	rs2=\E>\E[1;3;4;5;6l\E[?7h\E[100m\E[m\E[r\E[2J\E[H,
	sc=\E7, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smcup=\E7\E[?47h,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	tsl=\E[?E\E[?%i%p1%dT, use=vt100+fnkeys,

#### MTERM
#
# This application is available by email from <mouse@Rodents.Montreal.QC.CA>.
#
# "mterm -type ansi" sets $TERM to "ansi"
mterm-ansi|ANSI emulation,
	am, bw, mir, msgr,
	it#8,
	acsc=``aaffggiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[J, cr=^M,
	cub=\E[%p1%dD, cub1=\E[D, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dim=\E[2m, dl=\E[%p1%dM, dl1=\E[M, ech=\E[%p1%dX, ed=\E[J,
	el=\E[K, home=\E[H, hpa=\E[%i%p1%d`, ht=^I, ich1=,
	il=\E[%p1%dL, il1=\E[L, ind=\E[S, indn=\E[%p1%dS,
	invis=\E[8m, is2=\E)0\017, kbs=^H, nel=\EE, rev=\E[7m,
	ri=\E[T, rin=\E[%p1%dT, rmacs=^O, rmir=\E[4l, rmso=\E[27m,
	rmul=\E[24m,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m, smacs=^N, smir=\E[4h, smso=\E[7m, smul=\E[4m,
	vpa=\E[%i%p1%dd,
# mterm normally sets $TERM to "mterm"
mterm|mouse-sun|Der Mouse term,
	am, bw, mir,
	it#8,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^N, cuf1=^S,
	cup=\006%p1%d.%p2%d., cuu1=^X, dch1=^Y, dl1=^K, ed=^B, el=^C,
	home=^P, ht=^I, il1=^A, ind=^U, kbs=^H, ll=^R, nel=^M^U, ri=^W,
	rmir=^O, rmso=^T, smir=^Q, smso=^V,
# "mterm -type decansi" sets $TERM to "decansi"
#
# note: kdch1, kfnd, kslt are in the source code, but do not work -TD
decansi|ANSI emulation with DEC compatibility hacks,
	am, mir, msgr, xenl,
	colors#8, it#8, pairs#64,
	acsc=``aaffggiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=\E[D,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m, dl=\E[%p1%dM,
	dl1=\E[M, ech=\E[%p1%dX, ed=\E[J, el=\E[K, enacs=\E(B\E)0,
	home=\E[H, hpa=\E[%i%p1%d`, ht=^I, ich1=, il=\E[%p1%dL,
	il1=\E[L, ind=\E[S, indn=\E[%p1%dS, invis=\E[8m,
	is2=\E)0\E[r\017, kbs=^H, kcub1=\EOD, kcud1=\EOB,
	kcuf1=\EOC, kcuu1=\EOA, kdch1=\E[3~, kf1=\E[11~,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf13=\E[25~,
	kf14=\E[26~, kf15=\E[28~, kf16=\E[29~, kf17=\E[31~,
	kf18=\E[32~, kf19=\E[33~, kf2=\E[12~, kf20=\E[34~,
	kf3=\E[13~, kf4=\E[14~, kf5=\E[15~, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, kfnd=\E[1~, kich1=\E[2~, knp=\E[6~,
	kpp=\E[5~, kslt=\E[4~, nel=\EE, op=\E[0m, rc=\E8, rev=\E[7m,
	ri=\E[T, rin=\E[%p1%dT, rmacs=^O, rmam=\E[?7l, rmir=\E[4l,
	rmkx=\E[?1l\E>, rmso=\E[27m, rmul=\E[24m, sc=\E7,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m, smacs=^N, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m, u6=\E[%i%d;%dR,
	u7=\E[6n, vpa=\E[%i%p1%dd,

#### VWM
#
# vwmterm is a terminal emulator written for the VWM console window manager
#
vwmterm|(vwm term),
	am, bce, ccc, mir, msgr, npc, xenl, xon,
	colors#8, pairs#64,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[?25h, cr=^M, cub=\E[%p1%dD,
	cub1=^H, cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA,
	cuu1=\E[A, cvvis=\E[?25h, dim=\E[2m, ed=\E[J, el=\E[K,
	home=\E[H, il1=\E[L, ind=^J, invis=\E[8m, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~,
	kend=\E[4~, kf1=\E[[A, kf10=\E[21~, kf11=\E[22~,
	kf12=\E[23~, kf2=\E[[B, kf3=\E[[C, kf4=\E[[D, kf5=\E[[E,
	kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, knp=\E[6~, kpp=\E[5~, rev=\E[7m, rmacs=\E[10m,
	rmam=\E[?7l, rmcup=\E[?1049l, rs1=\E[H\E[J\E[m\Ec,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr=\E[0;10%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;%?%p7%t;8%;%?%p9%t;11%;m,
	sgr0=\E[0;10m, smacs=\E[11m, smam=\E[?7h,
	smcup=\E[?1049h, smso=\E[3m, smul=\E[4m,

#### MGR
#
# MGR is a Bell Labs window system lighter-weight than X.
# These entries describe MGR's xterm-equivalent.
# They are courtesy of Vincent Broman <broman@nosc.mil> 14 Jan 1997
#

mgr|Bellcore MGR (non X) window system terminal emulation,
	am, km,
	bel=^G, bold=\E2n, civis=\E9h, clear=^L, cnorm=\Eh, cr=^M,
	csr=\E%p1%d;%p2%dt, cub1=^H, cud1=\Ef, cuf1=\Er,
	cup=\E%p2%d;%p1%dM, cuu1=\Eu, cvvis=\E0h,
	dch=\E%p1%dE$<5>, dch1=\EE, dl=\E%p1%dd$<3*>,
	dl1=\Ed$<3>, ed=\EC, el=\Ec, hd=\E1;2f, ht=^I, hu=\E1;2u,
	ich=\E%p1%dA$<5>, ich1=\EA, il=\E%p1%da$<3*>,
	il1=\Ea$<3>, ind=^J, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, nel=^M^J, rev=\E1n, rmam=\E5S,
	rmso=\E0n, rmul=\E0n, sgr0=\E0n, smam=\E5s, smso=\E1n,
	smul=\E4n,
mgr-sun|Mgr window with Sun keyboard,
	ka1=\E[214z, ka3=\E[216z, kb2=\E[218z, kc1=\E[220z,
	kc3=\E[222z, kcpy=\E[197z, kend=\E[220z, kent=\E[250z,
	kf1=\E[224z, kf10=\E[233z, kf11=\E[234z, kf12=\E[235z,
	kf2=\E[225z, kf3=\E[226z, kf4=\E[227z, kf5=\E[228z,
	kf6=\E[229z, kf7=\E[230z, kf8=\E[231z, kf9=\E[232z,
	kfnd=\E[200z, khlp=\E[207z, khome=\E[214z, knp=\E[222z,
	kopn=\E[198z, kpp=\E[216z, kund=\E[195z, use=mgr,
mgr-linux|Mgr window with Linux keyboard,
	ka1=\E[H, ka3=\E[5~, kb2=\E[G, kc1=\E[Y, kc3=\E[6~,
	kdch1=\E[3~, kend=\E[4~, kf0=\E[[J, kf1=\E[[A, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf2=\E[[B, kf3=\E[[C, kf4=\E[[D,
	kf5=\E[[E, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, knp=\E[6~, kpp=\E[5~, use=mgr,

#### SIMPLETERM
# st.suckless.org
# st-0.1.1
#
# Note:  the original terminfo description uses leading blank to persuade
# ncurses to use "st" as its name.  Proper fix for that is to use "st" as an
# alias.
#
# Reading the code shows it should work for aixterm 16-colors
# - added st-16color
#
# Using tack:
# - set eo (erase-overstrike)
# - set xenl
# - tbc doesn't work
# - hts works
# - cbt doesn't work
# - shifted cursor-keys send sequences like rxvt
# - sgr referred to unimplemented "invis" mode.
# Fixes: add eo and xenl per tack, remove nonworking cbt, hts and tbc, invis
simpleterm|st| simpleterm,
	am, eo, mir, msgr, ul, xenl,
	colors#8, cols#80, it#8, lines#24, ncv#3, pairs#64,
	acsc=``aaffggiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, bold=\E[1m, civis=\E[?25l, clear=\E[H\E[2J,
	cnorm=\E[?12l\E[?25h, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub1=^H, cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K,
	home=\E[H, hpa=\E[%i%p1%dG, ht=^I, ich=\E[%p1%d@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, indn=\E[%p1%dS, kbs=\177,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[3~, kend=\E[4~, kf1=\EOP, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf2=\EOQ, kf3=\EOR, kf4=\EOS,
	kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, knp=\E[6~, kpp=\E[5~, op=\E[37;40m, rc=\E8,
	rev=\E[7m, rmacs=\E(B, rmso=\E[m, rmul=\E[m, sc=\E7,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr=%?%p9%t\E(0%e\E(B%;\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;m,
	sgr0=\E[0m, smacs=\E(0, smso=\E[7m, smul=\E[4m,
st-16color|simpleterm with 16-colors,
	use=ibm+16color, use=simpleterm,
# 256 colors "works", but when running xterm's test-scripts, some garbage is
# shown in the titlebar.
st-256color|simpleterm with 256 colors,
	ccc@,
	initc@, use=xterm+256color, use=simpleterm,

#### TERMINATOR
# http://software.jessies.org/terminator/
# Tested using their Debian package org.jessies.terminator 6.104.3256 on 64-bit
# Debian/current -TD (2011/8/20)
#
# There are some packaging problems:
# a) using Java, the program starts off using 50Mb, and climbs from there,
#    up to 114Mb after testing (no scrollback).
# b) it insists on reinstalling its terminal description in $HOME/.terminfo
#    (two copies, just in case the host happens to be Mac OS X).
#    I deleted this after testing with tack.
#
# Issues/features found with tack:
# a) tbc does not work (implying that hts also is broken).
#    Comparing with the tabs utility shows a problem with the last tabstop on
#    a line.
# b) has xterm-style shifted function-key strings
#    meta also is used, but control is ignored.
# c) has xterm-style modifiers for cursor keys (shift, control, shift+control, meta)
# d) some combinations of shift/control send xterm-style sequences for
#    insert/delete/home/end.
# e) numeric keypad sends only numbers (compare with vttest).
# f) meta mode (km) is not implemented.
#
# Issues found with ncurses test-program:
# a) bce is inconsistently implemented
# b) widths of Unicode values above 256 do not always agree with wcwidth.
#
# Checked with vttest, found low degree of compatibility there.
#
# Checked with xterm's scripts, found that the 256-color palette is fixed.
#
# Fixes:
# a) add sgr string
# b) corrected sgr0 to reset alternate character set
# c) modified smacs/rmacs to use SCS rather than SI/SO
# d) removed bce
# e) removed km
terminator|Terminator no line wrap,
	eo, mir, msgr, xenl, xon,
	colors#256, cols#80, it#8, lines#24, lm#0, pairs#32767,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bold=\E[1m, civis=\E[?25l, clear=\E[H\E[2J,
	cnorm=\E[?25h, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dl=\E[%p1%dM, dl1=\E[M, dsl=\E]2;\007, ed=\E[J, el=\E[K,
	el1=\E[1K, enacs=\E(B\E)0, flash=^G, fsl=^G, home=\E[H,
	hpa=\E[%i%p1%dG, ht=^I, hts=\EH, il=\E[%p1%dL, il1=\E[L,
	ind=^J, is1=\E[?47l\E=\E[?1l,
	is2=\E[r\E[m\E[2J\E[H\E[?7h\E[?1;3;4;6l\E[4l, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[3~, kend=\E[4~, kf1=\EOP, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf13=\E[25~, kf14=\E[26~,
	kf15=\E[28~, kf16=\E[29~, kf2=\EOQ, kf3=\EOR, kf4=\EOS,
	kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~,
	op=\E[39;49m, rc=\E8, rev=\E[7m, ri=\EM, rmacs=\E(B,
	rmcup=\E[?47l\E8, rmir=\E[4l, rmso=\E[27m, rmul=\E[24m,
	rs1=\E>\E[1;3;4;5;6l\E[?7h\E[m\E[r\E[2J\E[H,
	rs2=\E[r\E[m\E[2J\E[H\E[?7h\E[?1;3;4;6l\E[4l\E>,
	s0ds=\E(B, s1ds=\E(0, sc=\E7, setab=\E[48;5;%p1%dm,
	setaf=\E[38;5;%p1%dm,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;,
	sgr0=\E[m\E(B, smacs=\E(0, smcup=\E7\E[?47h, smir=\E[4h,
	smso=\E[7m, smul=\E[4m, tbc=\E[3g, tsl=\E]2;%p1,
	vpa=\E[%i%p1%dd,

######## UNIX VIRTUAL TERMINALS, VIRTUAL CONSOLES, AND TELNET CLIENTS
#

# Columbus UNIX virtual terminal. This terminal also appears in
# UNIX 4.0 and successors as line discipline 1 (?), but is
# undocumented and does not really work quite right.
cbunix|cb unix virtual terminal,
	OTbs, am, da, db,
	cols#80, lines#24, lm#0,
	bel=^G, clear=\EL, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EG%p2%c%p1%c, cuu1=\EA, dch1=\EM, dl1=\EN, ed=\EL,
	el=\EK, ich1=\EO, il1=\EP, ind=^J, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, khome=\EE, rmso=\Eb^D, rmul=\Eb^A,
	smso=\Ea^D, smul=\Ea^A,
# (vremote: removed obsolete ":nl@:" -- esr)
vremote|virtual remote terminal,
	am@,
	cols#79, use=cbunix,

pty|4bsd pseudo teletype,
	cup=\EG%p1%{32}%+%c%p2%{32}%+%c, rmso=\Eb$, rmul=\Eb!,
	smso=\Ea$, smul=\Ea!, use=cbunix,

#### Emacs

# The codes supported by the term.el terminal emulation in GNU Emacs 19.30
eterm|gnu emacs term.el terminal emulation,
	am, mir, xenl,
	cols#80, lines#24,
	bel=^G, bold=\E[1m, clear=\E[H\E[J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, el1=\E[1K, home=\E[H, ht=^I, ich=\E[%p1%d@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, rev=\E[7m,
	rmcup=\E[2J\E[?47l\E8, rmir=\E[4l, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m, smcup=\E7\E[?47h, smir=\E[4h, smso=\E[7m,
	smul=\E[4m,

# The codes supported by the term.el terminal emulation in GNU Emacs 22.2
eterm-color|Emacs term.el terminal emulator term-protocol-version 0.96,
	am, mir, msgr, xenl,
	colors#8, cols#80, lines#24, pairs#64,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, el1=\E[1K, home=\E[H, ht=^I, ich=\E[%p1%d@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, invis=\E[8m, kbs=\177,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kdch1=\E[3~, kend=\E[4~, khome=\E[1~, kich1=\E[2~,
	knp=\E[6~, kpp=\E[5~, op=\E[39;49m, rc=\E8, rev=\E[7m,
	ri=\EM, rmir=\E[4l, rmso=\E[27m, rmul=\E[24m, rs1=\Ec,
	sc=\E7, setab=\E[%p1%'('%+%dm, setaf=\E[%p1%{30}%+%dm,
	sgr=\E[0%?%p1%p3%|%t;7%;%?%p2%t;4%;%?%p4%t;5%;%?%p6%t;1%;%?%p7%t;8%;m,
	sgr0=\E[m, smir=\E[4h, smso=\E[7m, smul=\E[4m,
	u6=\E[%i%d;%dR, u7=\E[6n,

#### Screen

# Entries for use by the `screen' program by Juergen Weigert,
# Michael Schroeder, Oliver Laumann.  The screen and
# screen-w entries came with version 3.7.1.  The screen2 and screen3 entries
# come from University of Wisconsin and may be older.
# (screen: added <cnorm> on ANSI model -- esr)
#
# 'screen' defines extensions to termcap.  Some are used in its terminal
# description:
#      G0   (bool)  Terminal can deal with ISO 2022  font  selection sequences.
#      AX   (bool)  Does  understand ANSI set default fg/bg color
#                   (\E[39m / \E[49m).
#      S0   (str)   Switch charset 'G0' to the specified charset.
#      E0   (str)   Switch charset 'G0' back to standard charset.
#
# tested with screen 3.09.08
screen|VT 100/ANSI X3.64 virtual terminal,
	OTbs, OTpt, am, km, mir, msgr, xenl, G0,
	colors#8, cols#80, it#8, lines#24, ncv@, pairs#64, U8#1,
	acsc=++\,\,--..00``aaffgghhiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[34h\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\EM,
	cvvis=\E[34l, dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K, enacs=\E(B\E)0,
	flash=\Eg, home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, is2=\E)0, kbs=\177, kcbt=\E[Z,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kdch1=\E[3~, kend=\E[4~, kf1=\EOP, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf2=\EOQ, kf3=\EOR, kf4=\EOS,
	kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, kich1=\E[2~, kmous=\E[M, knp=\E[6~, kpp=\E[5~,
	nel=\EE, rc=\E8, rev=\E[7m, ri=\EM, rmacs=^O,
	rmcup=\E[?1049l, rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[23m,
	rmul=\E[24m, rs2=\Ec\E[?1000l\E[?25h, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p1%t;3%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smcup=\E[?1049h, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[3m, smul=\E[4m, tbc=\E[3g, E0=\E(B,
	S0=\E(%p1%c, use=ecma+color,
# The bce and status-line entries are from screen 3.9.13 (and require some
# changes to .screenrc).
screen-bce|VT 100/ANSI X3.64 virtual terminal with bce,
	bce,
	ech@, use=screen,
screen-s|VT 100/ANSI X3.64 virtual terminal with hardstatus line,
	dsl=\E_\E\\, fsl=\E\\, tsl=\E_, use=screen,

# ======================================================================
# Entries for GNU Screen with 16 colors.
# Those variations permit to benefit from 16 colors palette, and from
# bold font and blink attribute separated from bright colors. But they
# are less portable than the generic "screen" 8 color entries: Their
# usage makes real sense only if the terminals you attach and reattach
# do all support 16 color palette.

screen-16color|GNU Screen with 16 colors,
	use=ibm+16color, use=screen,

screen-16color-s|GNU Screen with 16 colors and status line,
	use=ibm+16color, use=screen-s,

screen-16color-bce|GNU Screen with 16 colors and BCE,
	use=ibm+16color, use=screen-bce,

screen-16color-bce-s|GNU Screen with 16 colors, BCE, and status line,
	bce, use=ibm+16color, use=screen-s,

# ======================================================================
# Entries for GNU Screen 4.02 with --enable-colors256.

screen-256color|GNU Screen with 256 colors,
	ccc@,
	initc@, use=xterm+256color, use=screen,

screen-256color-s|GNU Screen with 256 colors and status line,
	ccc@,
	initc@, use=xterm+256color, use=screen-s,

screen-256color-bce|GNU Screen with 256 colors and BCE,
	ccc@,
	initc@, use=xterm+256color, use=screen-bce,

screen-256color-bce-s|GNU Screen with 256 colors, BCE, and status line,
	bce, ccc@,
	initc@, use=xterm+256color, use=screen-s,

# ======================================================================

# Read the fine manpage:
#       When  screen  tries  to  figure  out  a  terminal name for
#       itself, it first looks for an entry named "screen.<term>",
#       where  <term>  is the contents of your $TERM variable.  If
#       no such entry exists, screen tries "screen" (or "screen-w"
#       if the terminal is wide (132 cols or more)).  If even this
#       entry cannot be found, "vt100" is used as a substitute.
#
# Notwithstanding the manpage, screen uses its own notion of the termcap
# and some keys from "screen.<term>" are ignored.  Here is an entry which
# covers those (tested with screen 4.00.02) -TD
screen+fkeys|function-keys according to screen,
	kend=\E[4~, kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, kfnd@,
	khome=\E[1~, kslt@,
#
# Here are a few customized entries which are useful -TD
#
# Notes:
# (a)	screen does not support invis.
# (b)	screen's implementation of bw is incorrect according to tack.
# (c)	screen appears to hardcode the strings for khome/kend, making it
#	necessary to override the "use=" clause's values (screen+fkeys).
# (d)	screen sets $TERMCAP to a termcap-formatted copy of the 'screen' entry,
#	which is NOT the same as the terminfo screen.<term>.
# (e)	when screen finds one of these customized entries, it sets $TERM to
#	match.  Hence, no "screen.xterm" entry is provided, since that would
#	create heartburn for people running remote xterm's.
#
#	xterm (-xfree86 or -r6) does not normally support kIC, kNXT and kPRV
#	since the default translations override the built-in keycode
#	translation.  They are suppressed here to show what is tested by tack.
screen.xterm-xfree86|screen.xterm-new|screen customized for modern xterm,
	bce@, bw,
	invis@, kIC@, kNXT@, kPRV@, meml@, memu@,
	sgr=%?%p9%t\E(0%e\E(B%;\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m,
	E3@, use=screen+fkeys, use=xterm-new,
# xterm-r6 does not really support khome/kend unless it is propped up by
# the translations resource.
screen.xterm-r6|screen customized for X11R6 xterm,
	bw, use=screen+fkeys, use=xterm-r6,
# Color applications running in screen and TeraTerm do not play well together
# on Solaris because Sun's curses implementation gets confused.
screen.teraterm|disable ncv in teraterm,
	ncv#127,
	acsc=+\020\,\021-\030.^Y0\333`\004a\261f\370g\361h\260i\316j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	use=screen+fkeys, use=screen,
# Other terminals
screen.rxvt|screen in rxvt,
	bw, XT,
	cvvis@, flash@, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC,
	kcuu1=\EOA, use=screen+fkeys, use=vt100+enq,
	use=rxvt+pcfkeys, use=vt220+keypad, use=screen,
screen.Eterm|screen in Eterm,
	use=screen+fkeys, use=Eterm,
screen.mrxvt|screen in mrxvt,
	use=screen+fkeys, use=mrxvt,
screen.vte|screen in any VTE-based terminal,
	use=screen+fkeys, use=vte,
screen.gnome|screen in GNOME Terminal,
	use=screen+fkeys, use=gnome,
screen.konsole|screen in KDE console window,
	use=screen+fkeys, use=konsole,
# fix the backspace key
screen.linux|screen in linux console,
	bw,
	kbs=\177, kcbt@, use=screen+fkeys, use=screen,
screen.mlterm|screen in mlterm,
	use=screen+fkeys, use=mlterm,

# The default "screen" entry is reasonably portable, but not optimal for the
# most widely-used terminal emulators.  The "bce" capability is supported in
# screen since 3.9.13, and when used, will require fewer characters to be sent
# to the terminal for updates.
#
# If you are using only terminals which support bce, then you can use this
# feature in your screen configuration.
#
# Adding these lines to your ".screenrc" file will allow using these customized
# entries:
#	term screen-bce
#	bce on
#	defbce on
screen-bce.xterm-new|screen optimized for modern xterm,
	bce,
	ech@, use=screen.xterm-new,
screen-bce.rxvt|screen optimized for rxvt,
	bce,
	ech@, use=screen.rxvt,
screen-bce.Eterm|screen optimized for Eterm,
	bce,
	ech@, use=screen.Eterm,
screen-bce.mrxvt|screen optimized for mrxvt,
	bce,
	ech@, use=screen.mrxvt,
screen-bce.gnome|screen optimized for GNOME-Terminal,
	bce,
	ech@, use=screen.gnome,
screen-bce.konsole|screen optimized for KDE console window,
	bce,
	ech@, use=screen.konsole,
screen-bce.linux|screen optimized for linux console,
	bce,
	ech@, use=screen.linux,
screen-bce.mlterm|screen optimized for mlterm,
	bce,
	ech@, use=screen.mlterm,

screen-w|VT 100/ANSI X3.64 virtual terminal with 132 cols,
	cols#132, use=screen,

screen2|old VT 100/ANSI X3.64 virtual terminal,
	cols#80, it#8, lines#24,
	cbt=\E[Z, clear=\E[2J\E[H, cr=^M, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, ht=^I, hts=\EH, ich=\E[%p1%d@, ich1=, il=\E[%p1%dL,
	il1=\E[L, ind=^J, kbs=^H, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kf0=\E~, kf1=\ES, kf2=\ET, kf3=\EU, kf4=\EV,
	kf5=\EW, kf6=\EP, kf7=\EQ, kf8=\ER, kf9=\E0I, khome=\EH,
	nel=^M^J, rc=\E8, ri=\EM, rmir=\E[4l, rmso=\E[23m,
	rmul=\E[24m, rs1=\Ec, sc=\E7, sgr0=\E[m, smir=\E[4h,
	smso=\E[3m, smul=\E[4m, tbc=\E[3g,
# (screen3: removed unknown ":xv:LP:G0:" -- esr)
screen3|older VT 100/ANSI X3.64 virtual terminal,
	km, mir, msgr,
	cols#80, it#8, lines#24,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, clear=\E[H\E[J,
	cr=^M, csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\EM,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, is2=\E)0, kbs=^H, kcub1=\EOD,
	kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kf1=\EOP, kf2=\EOQ,
	kf3=\EOR, kf4=\EOS, nel=\EE, rc=\E8, rev=\E[7m, ri=\EM,
	rmir=\E[4l, rmkx=\E>, rmso=\E[23m, rmul=\E[24m, rs1=\Ec,
	sc=\E7, sgr0=\E[m, smir=\E[4h, smkx=\E=, smso=\E[3m,
	smul=\E[4m, tbc=\E[3g,

#### NCSA Telnet

# Francesco Potorti <F.Potorti@cnuce.cnr.it>:
# NCSA telnet is one of the most used telnet clients for the Macintosh.  It has
# been maintained until recently by the National Center for Supercomputer
# Applications, and it is feature rich, stable and free.  It can be downloaded
# from www.ncsa.edu.  This terminfo description file is based on xterm-vt220,
# xterm+sl, and the docs at NCSA.  It works well.
#
# NCSA Telnet 2.6 for Macintosh in vt220 8-bit emulation mode
# The terminal options should be set as follows:
#         Xterm sequences ON
#         use VT wrap mode ON
#         use Emacs arrow keys OFF
#         CTRL-COMND is Emacs meta ON
#         8 bit mode ON
#         answerback string: "ncsa-vt220-8"
#         setup keys: all disabled
#
# Application mode is not used.
#
# Other special mappings:
#	Apple		VT220
#	HELP 		Find
#	HOME		Insert here
#	PAGEUP		Remove
#	DEL		Select
#	END		Prev Screen
#	PAGEDOWN	Next Screen
#
# Though it supports ANSI color, NCSA Telnet uses color to represent blinking
# text.
#
# The status-line manipulation is a mapping of the xterm-compatible control
# sequences for setting the window-title.  So you must use tsl and fsl in
# pairs, since the latter ends the string that is loaded to the window-title.
ncsa-m|ncsa-vt220-8|NCSA Telnet 2.6 for Macintosh in vt220-8 mode,
	am, km, mir, msgr, xenl,
	acsc=``aaffggiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[2J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, el1=\E[1K, enacs=\E)0, flash=\E[?5h\E[?5l,
	home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@,
	if=/usr/share/tabset/vt100, il=\E[%p1%dL, il1=\E[L,
	ind=\n$<150*>,
	is2=\E7\E[r\E[m\E[?7h\E[?1;4;6l\E[4l\E8\E>, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[4~, kend=\E[5~, kf1=\E[17~, kf10=\E[28~,
	kf11=\E[29~, kf12=\E[31~, kf13=\E[32~, kf14=\E[33~,
	kf15=\E[34~, kf2=\E[18, kf3=\E[19~, kf4=\E[20~, kf5=\E[21~,
	kf6=\E[23~, kf7=\E[24~, kf8=\E[25~, kf9=\E[26~, khlp=\E[1~,
	khome=\E[2~, knp=\E[6~, kpp=\E[3~, mc4=\E[4i, mc5=\E[5i,
	rc=\E8, rev=\E[7m, rf=/usr/share/tabset/vt100, ri=\EM,
	rmacs=\E(B, rmam=\E[?7l, rmcup=\E[2J\E8, rmir=\E[4l,
	rmso=\E[27m, rmul=\E[24m,
	rs2=\E7\E[r\E8\E[m\E[?7h\E[?1;4;6l\E[4l\E>, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p4%t;5%;%?%p1%p3%|%t;7%;m%?%p9%t\E(0%e\E(B%;,
	sgr0=\E[m\E(B, smacs=\E(0, smam=\E[?7h, smcup=\E7,
	smir=\E[4h, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	u8=\E[?62;1;6c, use=xterm+sl, use=ansi+enq,
ncsa|NCSA Telnet 2.7 for Macintosh in vt220-8 mode,
	use=ncsa-m, use=klone+color,
ncsa-ns|NCSA Telnet 2.7 for Macintosh in vt220-8 mode,
	hs@,
	dsl@, fsl@, tsl@, use=ncsa,
ncsa-m-ns|NCSA Telnet 2.6 for Macintosh in vt220-8 mode,
	hs@,
	dsl@, fsl@, tsl@, use=ncsa-m,
# alternate -TD:
# The documented function-key mapping refers to the Apple Extended Keyboard
# (e.g., NCSA Telnet's F1 corresponds to a VT220 F6).  We use the VT220-style
# codes, however, since the numeric keypad (VT100) PF1-PF4 are available on
# some keyboards and many applications require these as F1-F4.
#
ncsa-vt220|NCSA Telnet using vt220-compatible function keys,
	kf1=\EOP, kf10=\E[21~, kf11=\E[23~, kf12=\E[24~,
	kf13=\E[25~, kf14=\E[26~, kf15=\E[28~, kf16=\E[29~,
	kf17=\E[31~, kf18=\E[32~, kf19=\E[33~, kf2=\EOQ,
	kf20=\E[34~, kf3=\EOR, kf4=\EOS, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, use=ncsa,

#### Pilot Pro Palm-Top
#
# Termcap for Top Gun Telnet and SSH on the Palm Pilot.
# http://www.ai/~iang/TGssh/
pilot|tgtelnet|Top Gun Telnet on the Palm Pilot Professional,
	OTbs, am, xenl,
	cols#39, lines#16,
	bel=^G, clear=\Ec, cr=^M, cub1=^H, cud1=^J,
	cup=\Em%p1%{32}%+%c%p2%{32}%+%c, home=\Em\s\s, ht=^I,
	ind=^J, kbs=^H, kcub1=^H, kcud1=^J, knp=^L, kpp=^K, nel=\Em~\s,
	rmso=\EB, smso=\Eb,

# From: Federico Bianchi <bianchi@www.arte.unipi.it>
# These entries are for the Embeddable Linux Kernel System (ELKS)
# project - an heavily stripped down Linux to be run on 16 bit
# boxes or, eventually, to be used in embedded systems - and have been
# adapted from the stock ELKS termcap. The project itself looks stalled,
# and the latest improvements I know of date back to March 2000.
#
# To cope with the ELKS dumb console I added an "elks-glasstty" entry;
# as an added bonus, this deals with all the capabilities common to
# both VT52 and ANSI (or, eventually, "special") modes.

elks-glasstty|ELKS glass-TTY capabilities,
	OTbs, am,
	cols#80, it#8, lines#25,
	bel=^G, cr=^M, ht=^I, ind=^J, kbs=^H, kcub1=^H, kcud1=^J,
	nel=^M^J,

elks-vt52|ELKS vt52 console,
	clear=\EH\EJ, cub1=\ED, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, el=\EK,
	home=\EH, use=elks-glasstty,

elks-ansi|ELKS ANSI console,
	clear=\E[H\E[2J, cub1=\E[D, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, el=\E[K, home=\E[H,
	rmso=\E[m, smso=\E[7m, use=elks-glasstty,

# As a matter of fact, ELKS 0.0.83 on PCs defaults to ANSI emulation
# instead of VT52, but the "elks" entry still refers to the latter.

elks|default ELKS console,
	use=elks-vt52,

# Project SIBO (for Psion 3 palmtops) console is identical to the ELKS
# one but in screen size

sibo|ELKS SIBO console,
	cols#61, it#8, lines#20, use=elks-vt52,

######## COMMERCIAL WORKSTATION CONSOLES
#

#### Alpha consoles
#

# This is from the OSF/1 Release 1.0 termcap file
pccons|pcconsole|ANSI (mostly) Alpha PC console terminal emulation,
	am, xon,
	cols#80, lines#25,
	bel=^G, clear=\E[H\E[2J, cr=^M, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P, dl1=\E[M,
	el=\E[K, home=\E[H, ht=^I, ich1=\E[@, il1=\E[L, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, khome=\E[H,
	nel=^M^J, rev=\E[7m, rmso=\E[m, sgr0=\E[m, smso=\E[7m,

#### Sun consoles
#

# :is1: resets scrolling region in case a previous user had used "tset vt100"
oldsun|Sun Microsystems Workstation console,
	OTbs, am, km, mir, msgr,
	cols#80, it#8, lines#34,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch=\E[%p1%dP,
	dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, ht=^I,
	ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=^J,
	is1=\E[1r, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, khome=\E[H,
	rmso=\E[m, sgr0=\E[m, smso=\E[7m,
# From: Alexander Lukyanov <lav@video.yars.free.net>, 14 Nov 1995
# <lines> capability later corrected by J.T. Conklin <jtc@cygnus.com>
# SGR 1, 4 aren't supported - removed bold/underline (T.Dickey 17 Jan 1998)
sun-il|Sun Microsystems console with working insert-line,
	am, km, msgr,
	cols#80, lines#34,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch=\E[%p1%dP,
	dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, ht=^I,
	ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=^J,
	kb2=\E[218z, kbs=^H, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kdch1=\177, kend=\E[220z, kf1=\E[224z,
	kf10=\E[233z, kf11=\E[234z, kf12=\E[235z, kf2=\E[225z,
	kf3=\E[226z, kf4=\E[227z, kf5=\E[228z, kf6=\E[229z,
	kf7=\E[230z, kf8=\E[231z, kf9=\E[232z, khome=\E[214z,
	kich1=\E[247z, knp=\E[222z, kopt=\E[194z, kpp=\E[216z,
	kres=\E[193z, kund=\E[195z, rev=\E[7m, rmso=\E[m, rmul@,
	rs2=\E[s, sgr=\E[0%?%p1%p3%|%t;7%;m, sgr0=\E[m,
	smso=\E[7m, u8=\E[1t, u9=\E[11t,
# On some versions of CGSIX framebuffer firmware (SparcStation 5), <il1>/<il>
# flake out on the last line.  Unfortunately, without them the terminal has no
# way to scroll.
sun-cgsix|sun-ss5|Sun SparcStation 5 console,
	il@, il1@, use=sun-il,
# If you are using an SS5, change the sun definition to use sun-ss5.
sun|sun1|sun2|Sun Microsystems Inc. workstation console,
	use=sun-il,

sun+sl|Sun Workstation window status line,
	hs,
	dsl=\E]l\E\\, fsl=\E\\, tsl=\E]l,

# From: <john@ucbrenoir>  Tue Sep 24 13:14:44 1985
sun-s|Sun Microsystems Workstation window with status line,
	hs,
	dsl=\E]l\E\\, fsl=\E\\, tsl=\E]l, use=sun,
sun-e-s|sun-s-e|Sun Microsystems Workstation with status hacked for emacs,
	hs,
	dsl=\E]l\E\\, fsl=\E\\, tsl=\E]l, use=sun-e,
sun-48|Sun 48-line window,
	cols#80, lines#48, use=sun,
sun-34|Sun 34-line window,
	cols#80, lines#34, use=sun,
sun-24|Sun 24-line window,
	cols#80, lines#24, use=sun,
sun-17|Sun 17-line window,
	cols#80, lines#17, use=sun,
sun-12|Sun 12-line window,
	cols#80, lines#12, use=sun,
sun-1|Sun 1-line window for sysline,
	eslok, hs,
	cols#80, lines#1,
	dsl=^L, fsl=\E[K, tsl=^M, use=sun,
sun-e|sun-nic|sune|Sun Microsystems Workstation without insert character,
	ich1@, rmir@, smir@, use=sun,
sun-c|sun-cmd|Sun Microsystems Workstation console with scrollable history,
	lines#35,
	rmcup=\E[>4h, smcup=\E[>4l, use=sun,
sun-type4|Sun Workstation console with type 4 keyboard,
	kcub1=\E[217z, kcud1=\E[221z, kcuf1=\E[219z,
	kcuu1=\E[215z, use=sun-il,

# Most of the current references to sun-color are from users wondering why this
# is the default on install.  Details from reading the wscons manpage, adding
# cub, etc., here (rather than in the base sun-il entry) since it is not clear
# when those were added -TD (2005-05-28)
#
# According to wscons manpage, color is supported only on IA systems.
# Sun's terminfo entry documents bold and smul/rmul capabilities, but wscons
# does not list these.  It also sets ncv#3, however that corresponds to
# underline and standout.
#
# Since the documentation and terminfo do not agree, see also current code at
# http://src.opensolaris.org/source/xref/onnv/onnv-gate/usr/src/uts/common/io/tem_safe.c
#
# That (actually a different driver which "supports" sun-color) also supports
# these features:
#	vpa=\E[%i%p1%dd
#	hpa=\E[%i%p1%d`
#	cbt=\E[Z
#	dim=\E[2m
#	blink=\E[5m
# It supports bold, but not underline -TD (2009-09-19)
sun-color|Sun Microsystems Workstation console with color support (IA systems),
	colors#8, ncv#3, pairs#64,
	bold=\E[1m, cub=\E[%p1%dD, cud=\E[%p1%dB, cuf=\E[%p1%dC,
	cuu=\E[%p1%dA, home=\E[H, op=\E[0m, rs2=\E[s,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	setb=\E[4%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	setf=\E[3%?%p1%{1}%=%t4%e%p1%{3}%=%t6%e%p1%{4}%=%t1%e%p1%{6}%=%t3%e%p1%d%;m,
	sgr=\E[0%?%p6%t;1%;%?%p1%p3%|%t;7%;m, sgr0=\E[m,
	smso=\E[7m, use=sun,

#### Iris consoles
#

# (wsiris: this had extension capabilities
#	:HS=\E7F2:HE=\E7F7:\
#	:CT#2:CZ=*Bblack,red,green,yellow,blue,magenta,cyan,*Fwhite:
# See the note on Iris extensions near the end of this file.
# Finally, removed suboptimal <clear>=\EH\EJ and added <cud1> &
# <flash> from BRL -- esr)
wsiris|iris40|iris emulating a 40 line visual 50 (approximately),
	OTbs, OTnc, OTpt, am,
	OTkn#3, cols#80, it#8, lines#40,
	OTnl=\EB, bel=^G, clear=\Ev, cnorm=\E>, cub1=^H, cud1=\EB,
	cuf1=\EC, cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA,
	cvvis=\E;, dim=\E7F2, dl1=\EM, ed=\EJ, el=\EK,
	flash=\E7F4\E7B1\013\E7F7\E7B0, home=\EH, ht=^I, il1=\EL,
	ind=^J, is2=\E7B0\E7F7\E7C2\E7R3, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, kf0=\E0, kf1=\E1, kf2=\E2, kf3=\E3,
	kf4=\E4, kf5=\E5, kf6=\E6, kf7=\E7, kf8=\E8, kf9=\E9, ri=\EI,
	rmso=\E0@, rmul=\E7R3\E0@, sgr0=\E7F7, smso=\E9P,
	smul=\E7R2\E9P,

#### NeWS consoles
#
# Console terminal windows under the NeWS (Sun's Display Postscript windowing
# environment).   Note: these have nothing to do with Sony's News workstation
# line.
#

# Entry for NeWS's psterm from Eric Messick & Hugh Daniel
# (psterm: unknown ":sl=\EOl:el=\ENl:" removed -- esr)
psterm|psterm-basic|NeWS psterm-80x34,
	OTbs, am, hs, km, ul,
	cols#80, it#8, lines#34,
	blink=\EOb, bold=\EOd, clear=^L, csr=\EE%p1%d;%p2%d;,
	cub1=\ET, cud1=\EP, cuf1=\EV, cup=\E%p1%d;%p2%d;, cuu1=\EY,
	dch1=\EF, dl1=\EK, ed=\EB, el=\EC, flash=\EZ, fsl=\ENl,
	home=\ER, ht=^I, il1=\EA, ind=\EW, is1=\EN*, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, ll=\EU, rc=^\, rev=\EOr,
	ri=\EX, rmcup=\ENt, rmir=\ENi, rmso=\ENo, rmul=\ENu, sc=^],
	sgr0=\EN*, smcup=\EOt, smir=\EOi, smso=\EOo, smul=\EOu,
	tsl=\EOl,
psterm-96x48|NeWS psterm 96x48,
	cols#96, lines#48, use=psterm,
psterm-90x28|NeWS psterm 90x28,
	cols#90, lines#28, use=psterm,
psterm-80x24|NeWS psterm 80x24,
	cols#80, lines#24, use=psterm,
# This is a faster termcap for psterm.  Warning:  if you use this termcap,
# some control characters you type will do strange things to the screen.
# (psterm-fast: unknown ":sl=^Ol:el=^Nl:" -- esr)
psterm-fast|NeWS psterm fast version (flaky ctrl chars),
	OTbs, am, hs, km, ul,
	cols#80, it#8, lines#34,
	blink=^Ob, bold=^Od, clear=^L, csr=\005%p1%d;%p2%d;,
	cub1=^T, cud1=^P, cuf1=^V, cup=\004%p1%d;%p2%d;, cuu1=^Y,
	dch1=^F, dl1=^K, ed=^B, el=^C, flash=^Z, fsl=^Nl, home=^R, ht=^I,
	il1=^A, ind=^W, is1=^N*, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, ll=^U, rc=^\, rev=^Or, ri=^X, rmcup=^Nt, rmir=^Ni,
	rmso=^No, rmul=^Nu, sc=^], sgr0=^N*, smcup=^Ot, smir=^Oi,
	smso=^Oo, smul=^Ou, tsl=^Ol,

#### NeXT consoles
#
# Use `glasstty' for the Workspace application
#

# From: Dave Wetzel <dave@turbocat.snafu.de> 22 Dec 1995
next|NeXT console,
	am, xt,
	cols#80, it#8, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, el=\E[K, home=\E[H,
	ht=^I, ind=^J, kbs=^H, kcub1=^H, kcud1=^J, nel=^M^J,
	rmso=\E[4;1m, sgr0=\E[m, smso=\E[4;2m,
nextshell|NeXT Shell application,
	am,
	cols#80,
	bel=^G, cr=^M, cub1=^H, cud1=^J, ht=^I, kbs=^H, kcub1=^H,
	kcud1=^J, nel=^M^J,

#### Sony NEWS workstations
#

# (news-unk: this had :KB=news: -- esr)
news-unk|SONY NEWS vt100 emulator common entry,
	OTbs, OTpt, am, xenl,
	cols#80,
	OTnl=^J, bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[2J,
	cr=^M, csr=\E[%i%p1%d;%p2%dr, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dl=\E[%p1%dM, dl1=\E[M,
	ed=\E[J, el=\E[K, home=\E[H, ht=^I,
	if=/usr/share/tabset/vt100, il=\E[%p1%dL, il1=\E[L,
	is2=\E[?7h\E[?1h\E[?3l\E7\E8, kbs=^H, kcub1=\EOD,
	kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kf0=\EOY, kf1=\EOP,
	kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf5=\EOT, kf6=\EOU, kf7=\EOV,
	kf8=\EOW, kf9=\EOX, rc=\E8, rev=\E[7m, ri=\EM,
	rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h\E[r, sc=\E7,
	sgr0=\E[m, smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m,
#
# (news-29: this had :TY=ascii: --esr)
news-29,
	lines#29, use=news-unk,
# (news-29-euc: this had :TY=euc: --esr)
news-29-euc,
	use=news-29,
# (news-29-sjis: this had :TY=sjis: --esr)
news-29-sjis,
	use=news-29,
#
# (news-33: this had :TY=ascii: --esr)
news-33,
	lines#33, use=news-unk,
# (news-33-euc: this had :TY=euc: --esr)
news-33-euc,
	use=news-33,
# (news-33-sjis: this had :TY=sjis: --esr)
news-33-sjis,
	use=news-33,
#
# (news-42: this had :TY=ascii: --esr)
news-42,
	lines#42, use=news-unk,
# (news-42-euc: this had :TY=euc: --esr)
news-42-euc,
	use=news-42,
# (news-42-sjis: this had :TY=sjis: --esr)
news-42-sjis,
	use=news-42,
#
#	NEWS-OS old termcap entry
#
# (news-old-unk: this had :KB=news:TY=sjis: --esr)
news-old-unk|SONY NEWS vt100 emulator common entry,
	OTbs, OTpt, am, xenl,
	cols#80, vt#3,
	OTnl=^J, bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[;H\E[2J,
	cr=^M, csr=\E[%i%p1%d;%p2%dr, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, ed=\E[J, el=\E[K,
	home=\E[H, ht=^I, if=/usr/share/tabset/vt100, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf1=\EOP,
	kf2=\EOQ, kf3=\EOR, kf4=\EOS, rc=\E8, rev=\E[7m, ri=\EM,
	rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr0=\E[m, smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m,
#
# (nwp512: this had :DE=^H:, which I think means <OTbs> --esr)
nwp512|news|nwp514|news40|vt100-bm|old sony vt100 emulator 40 lines,
	OTbs,
	lines#40,
	is2=\E7\E[r\E8\EE\EE\EE\EM\EM\EM\E[?7h\E[?1l\E[?3l\E7\E[1;40r\E8,
	use=news-old-unk,
#
# (nwp512-a: this had :TY=ascii: and the alias vt100-bm --esr)
nwp512-a|nwp514-a|news-a|news42|news40-a|sony vt100 emulator 42 line,
	lines#42,
	is2=\E[?7h\E[?1l\E[?3l\E7\E[1;42r\E8,
	use=news-old-unk,
#
# (nwp-512-o: this had :KB=nwp410:DE=^H:  I interpret the latter as <OTbs>. --esr)
nwp512-o|nwp514-o|news-o|news40-o|vt100-bm-o|sony vt100 emulator 40 lines,
	OTbs,
	lines#40,
	is2=\E7\E[r\E8\EE\EE\EE\EM\EM\EM\E[?7h\E[?1l\E[?3l\E7\E[1;40r\E8,
	use=news-old-unk,
#
# (nwp513: this had :DE=^H: and the alias vt100-bm --esr)
nwp513|nwp518|nwe501|newscbm|news31|sony vt100 emulator 33 lines,
	OTbs,
	lines#31,
	is2=\E7\E[r\E8\EE\EE\EE\EM\EM\EM\E[?7h\E[?1l\E[?3l\E7\E[1;31r\E8,
	use=news-old-unk,
#
# (nwp513-a: this had :TY=ascii: and :DE=^H:, which I interpret as <OTbs>; --esr)
# also the alias vt100-bm.
nwp513-a|nwp518-a|nwe501-a|nwp251-a|newscbm-a|news31-a|newscbm33|news33|old sony vt100 emulator 33 lines,
	OTbs,
	lines#33,
	is2=\E7\E[r\E8\EE\EE\EE\EM\EM\EM\E[?7h\E[?1l\E[?3l\E7\E[1;33r\E8,
	use=news-old-unk,
#
# (nwp513-o: had :DE=^H:, I think that's <OTbs>; also the alias vt100-bm --esr)
nwp513-o|nwp518-o|nwe501-o|nwp251-o|newscbm-o|news31-o|old sony vt100 emulator 33 lines,
	OTbs,
	lines#31,
	is2=\E7\E[r\E8\EE\EE\EE\EM\EM\EM\E[?7h\E[?1l\E[?3l\E7\E[1;31r\E8,
	use=news-old-unk,
#
# (news28: this had :DE=^H:, I think that's <OTbs>, and :KB=nws1200: --esr)
news28|sony vt100 emulator 28 lines,
	OTbs,
	lines#28,
	is2=\E7\E[r\E8\EE\EE\EE\EM\EM\EM\E[?7h\E[?1l\E[?3l\E7\E[1;28r\E8,
	use=news-old-unk,
#
# (news29: this had :TY=ascii:KB=nws1200:\ --esr)
news29|news28-a|sony vt100 emulator 29 lines,
	lines#29,
	is2=\E7\E[r\E8\EE\EE\EE\EM\EM\EM\E[?7h\E[?1l\E[?3l\E7\E[1;29r\E8,
	use=news-old-unk,
#
# (news511: this had :TY=sjis: --esr)
nwp511|nwp-511|nwp-511 vt100,
	OTbs, OTpt, am, xenl,
	cols#80, lines#24,
	clear=\E[;H\E[2J$<20/>, cub1=\E[D, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A$<2/>, dl1=\E[M,
	ed=\E[J$<30/>, el=\E[K$<3/>,
	flash=\E[?5h\0\0\0\0\0\0\0\0\0\0\0\0\0\E[?5l,
	il1=\E[L, is2=\E[?5l\E[?1l\E>\E[?7h\E[?8h, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf1=\EOP, kf2=\EOQ,
	kf3=\EOR, kf4=\EOS, kf5=\EOT, kf6=\E#W, khome=\E[H,
	ri=\EM$<5/>, rmso=\E[m$<2/>, rmul=\E[m$<2/>,
	rs2=\E7\E[r\E8\E[?5l\E[?1l\E>\E[?7h\E[?8h,
	smso=\E[7m$<2/>, smul=\E[4m$<2/>,
# (news517: this had :TY=sjis:. --esr)
nwp517|nwp-517|nwp-517 vt200 80 cols 30 rows,
	eslok, hs,
	cols#80, lines#30,
	OTi2=\E[2$~\n, dsl=\E[1$~, fsl=\E[0$},
	is2=\E7\E[r\E8\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h,
	tsl=\E[1$}\E[;%df, use=vt200,
# (news517-w: this had :TY=sjis:. --esr)
nwp517-w|nwp-517-w|nwp-517 vt200 132 cols 50 rows,
	eslok, hs,
	cols#132, lines#50,
	OTi2=\E[2$~\n, dsl=\E[1$~, fsl=\E[0$},
	is2=\E7\E[r\E8\E>\E[?3h\E[?4l\E[?5l\E[?7h\E[?8h,
	rs2=\E>\E[?3h\E[?4l\E[?5l\E[?7h\E[?8h,
	tsl=\E[1$}\E[;%df, use=vt200,

#### Common Desktop Environment
#

# This ships with Sun's CDE in Solaris 2.5
# Corrected Sun Aug 9 1998 by Alexander V. Lukyanov <lav@video.yars.free.net>
dtterm|CDE desktop terminal,
	am, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, lm#0, ncv@,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m, dl=\E[%p1%dM,
	dl1=\E[M, ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K,
	enacs=\E(B\E)0, flash=\E[?5h$<200>\E[?5l, home=\E[H,
	ht=^I, hts=\EH, ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L,
	ind=\ED, invis=\E[8m, is2=\E F\E>\E[?1l\E[?7h\E[?45l,
	kbs=^H, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[3~, kf1=\E[11~, kf10=\E[21~, kf11=\E[23~,
	kf12=\E[24~, kf13=\E[25~, kf14=\E[26~, kf15=\E[28~,
	kf16=\E[29~, kf17=\E[31~, kf18=\E[32~, kf19=\E[33~,
	kf2=\E[12~, kf20=\E[34~, kf3=\E[13~, kf4=\E[14~,
	kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	kfnd=\E[1~, khlp=\E[28~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~,
	kslt=\E[4~, nel=\EE, rc=\E8, rev=\E[7m, ri=\EM, rmacs=^O,
	rmam=\E[?7l, rmir=\E[4l, rmso=\E[22;27m, rmul=\E[24m,
	sc=\E7,
	sgr=\E[0%?%p1%t;2;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smir=\E[4h,
	smso=\E[2;7m, smul=\E[4m, tbc=\E[3g, use=ecma+color,

######## Non-Unix Consoles
#

#### EMX termcap.dat compatibility modes
#
# Also (possibly only EMX, so we don't put it in ansi.sys, etc): set the
# no_color_video to inform the application that standout(1), underline(2)
# reverse(4) and invisible(64) don't work with color.
emx-base|DOS special keys,
	bce, bw,
	it#8, ncv#71,
	bel=^G, use=ansi.sys,

# Except for the "-emx" suffixes, these are as distributed with EMX 0.9b,
# a Unix-style environment used on OS/2.  (Note that the suffix makes some
# names longer than 14 characters, the nominal maximum).
#
# Removed: rmacs=\E[10m, smacs=\E[11m, because OS/2 does not implement acs.
ansi-emx|ANSI.SYS color,
	am, bce, eo, mir, msgr, xon,
	colors#8, cols#80, it#8, lines#25, pairs#64,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[1;33;44m\E[H\E[J, cnorm=\E[?25h, cr=^M, cub1=^H,
	cud1=^J, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A,
	dch=\E[%p1%dp, ed=\E[J, el=\E[K, flash=\E[?5h\E[?5l,
	home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@, ich1=\E[@, ind=^J,
	kb2=\E[G, kbs=^H, kf0=\0D, kll=\0O, kspd=^Z, nel=^M^J,
	rev=\E[5;37;41m, rmir=\E[4l, rmpch=\E[10m,
	rmso=\E[0;44m\E[1;33m, rmul=\E[0;44m\E[1;33m, rs1=\Ec,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr0=\E[0m\E[1;33;44m, smir=\E[4h, smpch=\E[11m,
	smso=\E[0;31;47m, smul=\E[1;31;44m, tbc=\E[3g, u8=\E[?6c,
	u9=\E[c, use=emx-base,
# nice colors for Emacs (white on blue, mode line white on cyan)
ansi-color-2-emx|ANSI.SYS color 2,
	clear=\E[0;37;44m\E[H\E[J, rev=\E[1;37;46m,
	rmso=\E[0;37;44m, rmul=\E[0;37;44m, rs1=\Ec,
	setaf=\E[3%p1%dm, sgr0=\E[0;37;44m, smso=\E[1;37;46m,
	smul=\E[1;36;44m, use=ansi-emx,
# nice colors for Emacs (white on black, mode line black on cyan)
ansi-color-3-emx|ANSI.SYS color 3,
	clear=\E[0;37;40m\E[H\E[J, rev=\E[1;37;46m,
	rmso=\E[0;37;40m, rmul=\E[0;37;40m, rs1=\Ec,
	setaf=\E[3%p1%dm, sgr0=\E[0;10m, smso=\E[1;37;46m,
	smul=\E[0;36;40m, use=ansi-emx,
mono-emx|stupid monochrome ansi terminal with only one kind of emphasis,
	am,
	cols#80, it#8, lines#24,
	clear=\E[H\E[2J$<50>, cub1=\E[D, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, el=\E[K, home=\E[H,
	ht=^I, kb2=\E[G, kbs=^H, kcub1=\0K, kcud1=\0P, kcuf1=\0M,
	kcuu1=\0H, kf0=\0D, kf1=\0;, kf2=\0<, kf3=\0=, kf4=\0>,
	kf5=\0?, kf6=\0@, kf7=\0A, kf8=\0B, kf9=\0C, khome=\0G,
	kich1=\0R, kll=\0O, knp=\0Q, kpp=\0I, nel=^M^J, rev=\E[7m,
	sgr0=\E[0m,

#### Cygwin

# Use this for cygwin32 (tested with beta 19.1)
# underline is colored bright magenta
# shifted kf1-kf12 are kf11-kf22
cygwinB19|ansi emulation for cygwin32,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[3~, kend=\E[4~, kf1=\E[[A, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf13=\E[25~, kf14=\E[26~,
	kf15=\E[28~, kf16=\E[29~, kf17=\E[31~, kf18=\E[32~,
	kf19=\E[33~, kf2=\E[[B, kf20=\E[34~, kf3=\E[[C, kf4=\E[[D,
	kf5=\E[[E, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~, rmam@, smam@,
	use=ansi.sys,

# Use this for cygwin (tested with version 1.1.0).
# I've combined pcansi and linux.  Some values of course were different and
# I've indicated which of these were and which I used.
# Cheers, earnie_boyd@yahoo.com
# several changes based on running with tack and comparing with older entry -TD
# more changes from csw:
#   add    cbt   [backtab]
#   remove eo    [erase overstrike with blank]
#   change clear was \E[H\E[J  now \E[2J  (faster?)
#   remove cols
#   remove lines
#   remove ncv#3 [colors collide with highlights, bitmask] not applicable
#                to MSDOS box?
#   add    cub   [cursor back param]
#   add    cuf   [cursor forward param]
#   add    cuu   [cursor up param]
#   add    cud   [cursor down param]
#   add    hs    [has status line]
#   add    fsl   [return from status line]
#   add    tsl   [go to status line]
#   add    smacs [Start alt charset] (not sure if this works)
#   add    rmacs [End alt charset]   (ditto)
#   add    smcup [enter_ca_mode] (save console; thanks Corinna)
#   add    rmcup [exit_ca_mode]  (restore console; thanks Corinna)
#   add    kb2   [center of keypad]
#   add    u8    [user string 8] \E[?6c
#   add    el    [clear to end of line] \E[K
# Notes:
#   cnorm [make cursor normal] not implemented
#   flash [flash] not implemented
#   blink [blink] not implemented very usefully in cygwin? \E[5m
#   dim   [dim] not implemented very usefully in cygwin? \E[2m
#   cub1  [cursor back 1] typically \E[D, but ^H is faster?
#   kNXT  [shifted next key] not implemented
#   kPRV  [shifted prev key] not implemented
#   khome [home key] really is \E[1~ NOT \E[H
#   tbc   [clear tab stops] not implemented
#   xenl  [newline ignnored after 80 cols] messes up last line? Ehud Karni
#   smpch [Start PC charset] is \E[11m, same as smacs
#   rmpch [End PC charset] is \E[10m, same as rmacs
#   mir   [move in insert mode] fails in tack?
#   bce   [back color erase] causes problems with change background color?
#   cvvis [make cursor very visible] causes a stackdump when testing with
#         testcurs using the output option? \E[?25h\E[?8c
#   civis [make cursor invisible] causes everything to stackdump? \E[?25l\E[?1c
#   ech   [erase characters param] broken \E[%p1%dX
#   kcbt  [back-tab key] not implemented in cygwin?  \E[Z
#
# 2005/11/12 -TD
#	Remove cbt since it does not work in current cygwin
#	Add 'mir' and 'in' flags based on tack
cygwin|ansi emulation for Cygwin,
	am, hs, mir, msgr, xon,
	colors#8, it#8, pairs#64,
	acsc=+\020\,\021-\030.^Y0\333`\004a\261f\370g\361h\260j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	bel=^G, bold=\E[1m, clear=\E[H\E[J, cr=^M, cub=\E[%p1%dD,
	cub1=^H, cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA,
	cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K, fsl=^G, home=\E[H,
	hpa=\E[%i%p1%dG, ht=^I, ich=\E[%p1%d@, ich1=\E[@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, invis=\E[8m, kb2=\E[G,
	kbs=^H, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[3~, kend=\E[4~, kf1=\E[[A, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf13=\E[25~, kf14=\E[26~,
	kf15=\E[28~, kf16=\E[29~, kf17=\E[31~, kf18=\E[32~,
	kf19=\E[33~, kf2=\E[[B, kf20=\E[34~, kf3=\E[[C, kf4=\E[[D,
	kf5=\E[[E, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~, kspd=^Z,
	nel=^M^J, op=\E[39;49m, rc=\E8, rev=\E[7m, ri=\EM,
	rmacs=\E[10m, rmcup=\E[2J\E[?47l\E8, rmir=\E[4l,
	rmpch=\E[10m, rmso=\E[27m, rmul=\E[24m, rs1=\Ec\E]R,
	sc=\E7, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr=\E[0;10%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p6%t;1%;%?%p7%t;8%;%?%p9%t;11%;m,
	sgr0=\E[0;10m, smacs=\E[11m, smcup=\E7\E[?47h,
	smir=\E[4h, smpch=\E[11m, smso=\E[7m, smul=\E[4m, tsl=\E];,
	vpa=\E[%i%p1%dd, use=vt102+enq,

# I've supplied this so that you can help test new values and add other
# features.  Cheers, earnie_boyd@yahoo.com.
#
# Some features are from pcansi.  The op value is from linux.  Function-keys
# are from linux.  These have been tested not to cause problems.  xenl was in
# this list, but DOES cause problems so it has been removed
cygwinDBG|Debug Version for Cygwin,
	am, eo, mir, msgr, xon,
	colors#8, cols#80, it#8, lines#24, ncv#3, pairs#64,
	acsc=+\020\,\021-\030.^Y0\333`\004a\261f\370g\361h\260j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, clear=\E[H\E[J,
	cnorm=\E[?25h, cr=^M, cub=\E[%p1%dD, cub1=\E[D,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K,
	flash=\E[?5h\E[?5l$<200/>, home=\E[H, hpa=\E[%i%p1%dG,
	ht=^I, hts=\EH, ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL,
	il1=\E[L, ind=^J, invis=\E[8m, kNXT=\E[6$, kPRV=\E[5$,
	kb2=\E[G, kbs=^H, kcbt=\E[Z, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~, kend=\E[4~, kf1=\E[[A,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf13=\E[25~,
	kf14=\E[26~, kf15=\E[28~, kf16=\E[29~, kf17=\E[31~,
	kf18=\E[32~, kf19=\E[33~, kf2=\E[[B, kf20=\E[34~,
	kf3=\E[[C, kf4=\E[[D, kf5=\E[[E, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, khome=\E[1~, kich1=\E[2~,
	knp=\E[6~, kpp=\E[5~, kspd=^Z, nel=^M^J, op=\E[39;49m,
	rc=\E8, rev=\E[7m, ri=\EM, rmacs=\E[10m, rmir=\E[4l,
	rmso=\E[m, rmul=\E[m, rs1=\Ec\E]R, sc=\E7,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr=\E[0;10%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p7%t;8%;%?%p9%t;12%;m,
	sgr0=\E[0;10m, smacs=\E[11m, smir=\E[4h, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g, vpa=\E[%i%p1%dd, use=vt102+enq,

#### DJGPP

# Key definitions:
# The encodings for unshifted arrow keys, F1-F12, Home, Insert, etc.  match the
# encodings used by other x86 environments.  All others are invented for DJGPP.
# Oddly enough, while several combinations of modifiers are tabulated, there is
# none for shifted cursor keys.
#
#	F1			\E[[A
#	F2			\E[[B
#	F3			\E[[C
#	F4			\E[[D
#	F5			\E[[E
#	F6			\E[17~
#	F7			\E[18~
#	F8			\E[19~
#	F9			\E[20~
#	F10			\E[21~
#	F11			\E[23~
#	F12			\E[24~
#
#	Delete			\E[3~
#	Down Arrow		\E[B
#	End			\E[4~
#	Home			\E[1~
#	Insert			\E[2~
#	Left Arrow		\E[D
#	Page Down		\E[6~
#	Page Up			\E[5~
#	Right Arrow		\E[C
#	Up Arrow		\E[A
#
#	Shift-F1		\E[25~
#	Shift-F2		\E[26~
#	Shift-F3		\E[27~
#	Shift-F4		\E[28~
#	Shift-F5		\E[29~
#	Shift-F6		\E[30~
#	Shift-F7		\E[31~
#	Shift-F8		\E[32~
#	Shift-F9		\E[33~
#	Shift-F10		\E[34~
#	Shift-F11		\E[35~
#	Shift-F12		\E[36~
#
#	Ctrl-F1			\E[47~
#	Ctrl-F2			\E[48~
#	Ctrl-F3			\E[49~
#	Ctrl-F4			\E[50~
#	Ctrl-F5			\E[51~
#	Ctrl-F6			\E[52~
#	Ctrl-F7			\E[53~
#	Ctrl-F8			\E[54~
#	Ctrl-F9			\E[55~
#	Ctrl-F10		\E[56~
#	Ctrl-F11		\E[57~
#	Ctrl-F12		\E[58~
#
#	Ctrl-Delete		\E[43~
#	Ctrl-Down Arrow		\E[38~
#	Ctrl-End		\E[44~
#	Ctrl-Home		\E[41~
#	Ctrl-Insert		\E[42~
#	Ctrl-Left Arrow		\E[39~
#	Ctrl-Page Down		\E[46~
#	Ctrl-Page Up		\E[45~
#	Ctrl-Right Arrow	\E[40~
#	Ctrl-Up Arrow		\E[37~
#
#	Alt-F1			\E[59~
#	Alt-F2			\E[60~
#	Alt-F3			\E[61~
#	Alt-F4			\E[62~
#	Alt-F5			\E[63~
#	Alt-F6			\E[64~
#	Alt-F7			\E[65~
#	Alt-F8			\E[66~
#	Alt-F9			\E[67~
#	Alt-F10			\E[68~
#	Alt-F11			\E[79~
#	Alt-F12			\E[80~
#
#	Alt-Delete		\E[65~
#	Alt-Down Arrow		\E[60~
#	Alt-End			\E[66~
#	Alt-Home		\E[41~
#	Alt-Insert		\E[64~
#	Alt-Left Arrow		\E[61~
#	Alt-Page Down		\E[68~
#	Alt-Page Up		\E[67~
#	Alt-Right Arrow		\E[62~
#	Alt-Up Arrow		\E[59~
#
# Also:
#	Alt-A			\E[82~
#	Alt-B			\E[82~
#	Alt-C			\E[83~
#	Alt-D			\E[84~
#	Alt-E			\E[85~
#	Alt-F			\E[86~
#	Alt-G			\E[87~
#	Alt-H			\E[88~
#	Alt-I			\E[89~
#	Alt-J			\E[90~
#	Alt-K			\E[91~
#	Alt-L			\E[92~
#	Alt-M			\E[93~
#	Alt-N			\E[94~
#	Alt-O			\E[95~
#	Alt-P			\E[96~
#	Alt-Q			\E[97~
#	Alt-R			\E[98~
#	Alt-S			\E[99~
#	Alt-T			\E[100~
#	Alt-U			\E[101~
#	Alt-V			\E[102~
#	Alt-W			\E[103~
#	Alt-X			\E[104~
#	Alt-Y			\E[105~
#	Alt-Z			\E[106~
djgpp|ansi emulation for DJGPP alpha,
	am, bce, msgr, xhp, xon, xt,
	colors#8, it#8, pairs#64,
	acsc=+\020\,\021-\030.^Y0\333`\004a\261f\370g\361h\260j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[1v,
	clear=\E[H\E[J, cnorm=\E[v, cr=^M, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[2v, dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ech=\E[%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K,
	home=\E[H, hpa=\E[%i%p1%dG, ht=^I, ich=\E[%p1%d@,
	ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=\E[S,
	indn=\E[%p1%dS, invis=\E[8m, kbs=^H, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~,
	kend=\E[4~, kf1=\E[[A, kf10=\E[21~, kf11=\E[23~,
	kf12=\E[24~, kf2=\E[[B, kf3=\E[[C, kf4=\E[[D, kf5=\E[[E,
	kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[1~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~, nel=^M^J,
	op=\E[37;40m, rev=\E[7m, ri=\E[T, rin=\E[%p1%dT, rmso=\E[m,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	sgr=\E[0%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%e;25%;%?%p6%t;1%;%?%p7%t;8%;m,
	sgr0=\E[m, smso=\E[7m, smul=\E[4m, vpa=\E[%i%p1%dd,

djgpp203|Entry for DJGPP 2.03,
	OTbs, am,
	cols#80, it#8, lines#25,
	bel=^G, cr=^M, cub1=^H, cud1=^J, ht=^I, ind=^J, kbs=^H, kcub1=^H,
	kcud1=^J, nel=^M^J,

djgpp204|Entry for DJGPP 2.04,
	OTbs, am, AX,
	colors#8, cols#80, it#8, lines#25, ncv#3, pairs#64,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[1v,
	clear=\E[H\E[2J, cnorm=\E[v, cr=^M, cub=\E[%p1%dD,
	cub1=\E[D, cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA,
	cuu1=\E[A, cvvis=\E[2v, dch=\E[%p1%dP, dch1=\E[P,
	dl=\E[%p1%dM, dl1=\E[M, ech=\E[%p1%dX, ed=\E[J, el=\E[K,
	home=\E[H, ht=^I, ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL,
	il1=\E[L, ind=\E[S, indn=\E[%p1%dS, invis=\E[8m, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[3~, kf0=\E[21~, kf1=\E[[A, kf10=\E[21~, kf2=\E[[B,
	kf3=\E[[C, kf4=\E[[D, kf5=\E[[E, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, khome=\E[1~, kich1=\E[2~,
	kll=\E[4~, knp=\E[6~, kpp=\E[5~, nel=^M^J, rev=\E[7m,
	ri=\E[T, rin=\E[%p1%dT, rmso=\E[m, setab=\E[4%p1%dm,
	setaf=\E[3%p1%dm, sgr0=\E[m, smso=\E[7m, smul=\E[4m,

#### U/Win

# This is tested using U/Win's telnet.  Scrolling is omitted because it is
# buggy.  Another odd bug appears when displaying "~" in alternate character
# set (the emulator spits out error messages).  Compare with att6386 -TD
uwin|U/Win 3.2 console,
	am, eo, in, msgr, xenl, xon,
	colors#8, it#8, ncv#58, pairs#64,
	acsc=+\020\,\021-\030.^Y0\333`\004a\261f\370g\361h\260i\316j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[?25h, cr=^M, cub1=^H, cud1=^J,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m, ech=\E[%p1%dX,
	ed=\E[J, el=\E[K, el1=\E[1K, home=\E[H, ht=^I, hts=\EH,
	ich=\E[%p1%d@, ich1=\E[@, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\177, kend=\E[Y, kf1=\EOP,
	kf10=\EOY, kf11=\EOZ, kf12=\EOA, kf2=\EOQ, kf3=\EOR,
	kf4=\EOS, kf5=\EOT, kf6=\EOU, kf7=\EOV, kf8=\EOW, kf9=\EOX,
	khome=\E[H, kich1=\E[@, nel=^M^J, op=\E[39;49m, rc=\E8,
	rev=\E[7m, rmacs=\E[10m, rmir=\E[4l, rmpch=\E[10m,
	rmso=\E[27m, rmul=\E[m, rs1=\Ec\E]R, sc=\E7,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm, sgr0=\E[0;10m,
	smacs=\E[11m, smir=\E[4h, smpch=\E[11m, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g, u6=\E[%i%d;%dR, u7=\E[6n,

#### Microsoft (miscellaneous)

# This entry fits the Windows NT console when the _POSIX_TERM environment
# variable is set to 'on'.  While the Windows NT POSIX console is seldom used,
# the Telnet client supplied with both the Windows for WorkGroup 3.11 TCP/IP
# stack and the Win32 (i.e., Windows 95 and Windows NT 3.1 or later) operating
# systems is not, and (surprise!) they match very well.
#
# See:  MS Knowledge Base item Q108581, dated 13-MAY-1997, titled "Setting Up
# VI POSIX Editor for Windows NT 3.1".  True to Microsoft form, not only
# are the installation instructions a pile of mind-numbing bureaucratese,
# but the termcap entry is actually broken and unusable as given; the :do:
# capability is misspelled "d".
#
# To use this, you need to a bunch of environment variables:
#
# SET _POSIX_TERM=on
# SET TERM=ansi
# SET TERMCAP=location of termcap file in POSIX file format
# which is case-sensitive.
# e.g. SET TERMCAP=//D/RESKIT35/posix/termcap
# SET TMP=//C/TEMP
#
# Important note: setting the TMP environment variable in POSIX style renders
# it incompatible with a lot of other applications, including Visual C++. So
# you should have a separate command window just for vi. All the other
# variables may be permanently set in the Control Panel\System applet.
#
# You can find out more about the restrictions of this facility at
# <http://www.nentug.org/unix-to-nt/ntposix.htm>.
#
# From: Federico Bianchi <bianchi@magna.cisid.unipi.it>, 15 Jan 1997
ansi-nt|psx_ansi|Microsoft Windows NT console POSIX ANSI mode,
	am, bw, msgr,
	cols#80, it#8, lines#25,
	bel=^G, clear=\E[2J, cr=^M, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, ed=\E[J, el=\E[K,
	home=\E[H, ht=^I, ind=\E[S, kbs=^H, kcub1=\E[D, kcud1=\E[V,
	kcuf1=\E[C, kcuu1=\E[A, nel=\r\E[S, rc=\E[u, rev=\E[7m,
	ri=\E[T, rmso=\E[m, sc=\E[s, sgr0=\E[0m, smso=\E[7m,
# From: jew@venus.sunquest.com
# Date: 19 Feb 93 23:41:07 GMT
# Here's  a  combination of  ansi and  vt100 termcap
# entries   that  works  nearly   perfectly  for  me
# (Gateway 2000 Handbook and Microsoft Works 3.0):
pcmw|PC running Microsoft Works,
	am, xenl,
	cols#80, it#8, lines#24, vt#3,
	bel=^G, blink=\E[5m$<2/>, bold=\E[1m$<2/>,
	clear=\E[;H\E[2J$<50/>, cr=^M, cub1=^H, cud1=^J,
	cuf1=\E[C$<2/>, cup=\E[%i%p1%d;%p2%dH$<5/>,
	cuu1=\E[A$<2/>, ed=\E[J$<50/>, el=\E[K$<3/>, home=\E[H,
	ht=^I, hts=\EH$<2/>, ind=\ED$<5/>, is2=\E[1;24r\E[24;1H,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, nel=\r\ED$<5/>,
	rc=\E8, rev=\E[7m$<2/>, rf=/usr/share/tabset/vt100,
	ri=\EM$<5/>, rmso=\E[m$<2/>, rmul=\E[m$<2/>,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr0=\E[m$<2/>, smso=\E[7m$<2/>, smul=\E[4m$<2/>,
	tbc=\E[3g$<2/>,

# From: Federico Bianchi
# This is the entry for the OpenNT terminal.
# The ntconsole name is for backward compatability.
# This is for OpenNT 2.0 and later.
# Later OpenNT was renamed to Interix.
#
# Presently it is distributed by Microsoft as Services For Unix (SFU).
# The 3.5 beta contains ncurses 4.2 (that is header files and executables,
# the documentation dates from 1.9.9e) -TD

interix|opennt|opennt-25|ntconsole|ntconsole-25|OpenNT-term compatible with color,
	am, bw, msgr,
	colors#8, cols#80, lines#25, ncv#3, pairs#64,
	acsc=+\020\,\021-\030.^Y0\333`\004a\261f\370g\361h\260j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	bel=^G, bold=\E[1m, cbt=\E[Z, clear=\E[2J, cub=\E[%p1%dD,
	cub1=\E[D, cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA,
	cuu1=\E[A, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K,
	home=\E[H, ht=^I, il=\E[%p1%dL, il1=\E[L, ind=\E[S,
	indn=\E[%p1%dS, kbs=^H, kcbt=\E[Z, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[M, kend=\E[U, kf0=\EFA,
	kf1=\EF1, kf10=\EFA, kf11=\EFB, kf12=\EFC, kf13=\EFD,
	kf14=\EFE, kf15=\EFF, kf16=\EFG, kf17=\EFH, kf18=\EFI,
	kf19=\EFJ, kf2=\EF2, kf20=\EFK, kf21=\EFL, kf22=\EFM,
	kf23=\EFN, kf24=\EFO, kf25=\EFP, kf26=\EFQ, kf27=\EFR,
	kf28=\EFS, kf29=\EFT, kf3=\EF3, kf30=\EFU, kf31=\EFV,
	kf32=\EFW, kf33=\EFX, kf34=\EFY, kf35=\EFZ, kf36=\EFa,
	kf37=\EFb, kf38=\EFc, kf39=\EFd, kf4=\EF4, kf40=\EFe,
	kf41=\EFf, kf42=\EFg, kf43=\EFh, kf44=\EFi, kf45=\EFj,
	kf46=\EFk, kf47=\EFm, kf48=\EFn, kf49=\EFo, kf5=\EF5,
	kf50=\EFp, kf51=\EFq, kf52=\EFr, kf53=\EFs, kf54=\EFt,
	kf55=\EFu, kf56=\EFv, kf57=\EFw, kf58=\EFx, kf59=\EFy,
	kf6=\EF6, kf60=\EFz, kf61=\EF+, kf62=\EF-,
	kf63=\EF\014 kf64=\EF$, kf7=\EF7, kf8=\EF8, kf9=\EF9,
	kich1=\E[L, kll=\E[U, knp=\E[T, kpp=\E[S, ll=\E[U, nel=^M^J,
	op=\E[m, rc=\E[u, rev=\E[7m, ri=\E[T, rin=\E[%p1%dT,
	rmcup=\E[2b\E[u\r\E[K, rmso=\E[m, rmul=\E[m, rs1=\Ec,
	sc=\E[s, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	setb=\E[%p1%{40}%+%dm, setf=\E[%p1%{30}%+%dm,
	sgr0=\E[0m, smcup=\E[s\E[1b, smso=\E[7m, smul=\E[4m,

opennt-35|ntconsole-35|OpenNT-term35 compatible with color,
	lines#35, use=opennt,

opennt-50|ntconsole-50|OpenNT-term50 compatible with color,
	lines#50, use=opennt,

opennt-60|ntconsole-60|OpenNT-term60 compatible with color,
	lines#60, use=opennt,

opennt-100|ntconsole-100|OpenNT-term100 compatible with color,
	lines#100, use=opennt,

# OpenNT wide terminals
opennt-w|opennt-25-w|ntconsole-w|ntconsole-25-w|OpenNT-term-w compat with color,
	cols#125, use=opennt,

opennt-35-w|ntconsole-35-w|OpenNT-term35-w compatible with color,
	lines#35, use=opennt-w,

opennt-50-w|ntconsole-50-w|OpenNT-term50-w compatible with color,
	lines#50, use=opennt-w,

opennt-60-w|ntconsole-60-w|OpenNT-term60-w compatible with color,
	lines#60, use=opennt-w,

opennt-w-vt|opennt-25-w-vt|ntconsole-w-vt|ntconsole-25-w-vt|OpenNT-term-w-vt compat with color,
	cols#132, use=opennt,

# OpenNT terminals with no smcup/rmcup (names match termcap entries)
interix-nti|opennt-nti|opennt-25-nti|ntconsole-25-nti|OpenNT-nti compatible with color,
	rmcup@, smcup@, use=opennt,

opennt-35-nti|ntconsole-35-nti|OpenNT-term35-nti compatible with color,
	lines#35, use=opennt-nti,

opennt-50-nti|ntconsole-50-nti|OpenNT-term50-nti compatible with color,
	lines#50, use=opennt-nti,

opennt-60-nti|ntconsole-60-nti|OpenNT-term60-nti compatible with color,
	lines#60, use=opennt-nti,

opennt-100-nti|ntconsole-100-nti|OpenNT-term100-nti compatible with color,
	lines#100, use=opennt-nti,

######## COMMON TERMINAL TYPES
#
# This section describes terminal classes and maker brands that are still
# quite common, but have proprietary command sets not blessed by ANSI.
#

#### Altos
#
# Altos made a moderately successful line of UNIX boxes.  In 1990 they were
# bought out by Acer, a major Taiwanese manufacturer of PC-clones.
# Acer has a web site at http://www.acer.com.
#
# Altos descriptions from Ted Mittelstaedt <tedm@agora.rain.com> 4 Sep 1993
# His comments suggest they were shipped with the system.
#

# (altos2: had extension capabilities
#	:c0=^A`\r:c1=^Aa\r:c2=^Ab\r:c3=^Ac\r:\
#	:c4=^Ad\r:c5=^Ae\r:c6=^Af\r:c7=^Ag\r:\
#	:c8=^Ah\r:c9=^Ai\r:cA=^Aj\r:cB=^Ak\r:\
#	:cC=^Al\r:cD=^Am\r:cE=^An\r:cF=^Ao\r:
#	:XU=^Aq\r:XD=^Ar\r:XR=^As\r:XL=^At\r:\
#	:YU=^AQ\r:YD=^AR\r:YR=^AS\r:YL=^AT\r:\
#	:HL=^AP\r:SP=\E[i:\
#	:IS=\E[@:DE=\E[P:IL=\E[L:NS=\E[S:PS=\E[T:\
#	:LO=\E[0q:LC=\E[5q:LL=\E[6q:\
# Comparison with the k* capabilities makes it obvious that the c* things are
# shift keys.  I have renamed them to keys 32 and up accordingly.  Also,
# :sr: was given as a boolean-- esr)
altos2|alt2|altos-2|altos II,
	cols#80, it#8, lines#24, xmc#0,
	clear=\E[H\E[2J, cr=^M, cub1=^H, cud1=\E[1B, cuf1=\E[1C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[1A, dch1=\E[P, dl1=\E[M,
	ed=\E[J, el=\E[K, home=\E[H, ht=^I, ich1=\E[@,
	if=/usr/share/tabset/vt100, il1=\E[L, ind=^J,
	is2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, kDL=^Am\r,
	kEOL=^An\r, kbs=^H, kcbt=^AK\r, kclr=^AL\r, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=^AM\r, kel=^AN\r,
	kf0=^AI\r, kf1=^A@\r, kf2=^AA\r, kf3=^AB\r, kf32=^A`\r,
	kf33=^Aa\r, kf34=^Ab\r, kf35=^Ac\r, kf36=^Ad\r, kf37=^Ae\r,
	kf38=^Af\r, kf39=^Ag\r, kf4=^AC\r, kf40=^Ah\r, kf41=^Ai\r,
	kf42=^Aj\r, kf43=^Ak\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r,
	kf8=^AG\r, kf9=^AH\r, khome=\E[f, kil1=^AJ\r, kind=^AO\r,
	nel=^M^J, rmam=\E[?7l, rmso=\E[m, rmul=\E[m, sgr0=\E[m,
	smam=\E[?7h, smso=\E[7m, smul=\E[4m,
# (altos3: had extension capabilities
#	:c0=^A`\r:c1=^Aa\r:c2=^Ab\r:c3=^Ac\r:\
#	:c4=^Ad\r:c5=^Ae\r:c6=^Af\r:c7=^Ag\r:\
#	:c8=^Ah\r:c9=^Ai\r:cA=^Aj\r:cB=^Ak\r:\
#	:cC=^Al\r:cD=^Am\r:cE=^An\r:cF=^Ao\r:
#	:XU=^Aq\r:XD=^Ar\r:XR=^As\r:XL=^At\r:\
#	:HL=^AP\r:SP=\E[i:\
#	:IS=\E[@:DE=\E[P:IL=\E[L:NS=\E[S:PS=\E[T:
altos3|altos5|alt3|alt5|altos-3|altos-5|altos III or V,
	blink=\E[5p, ri=\EM, sgr0=\E[p, use=altos2,
altos4|alt4|altos-4|altos IV,
	use=wy50,
# (altos7: had extension capabilities:
#	:GG#0:GI=\EH8:GF=\EH7:\
#	:c0=^A`\r:c1=^Aa\r:c2=^Ab\r:c3=^Ac\r:\
#	:c4=^Ad\r:c5=^Ae\r:c6=^Af\r:c7=^Ag\r:\
#	:c8=^Ah\r:c9=^Ai\r:cA=^Aj\r:cB=^Ak\r:\
#	:cC=^Al\r:cD=^Am\r:cE=^An\r:cF=^Ao\r:
# Comparison with the k* capabilities makes it obvious that the c* things are
# shift keys.  I have renamed them to keys 32 and up accordingly.  I have
# also made this entry relative to adm12 in order to give it an <sgr>. The
# <invis> imported by use=adm+sgr may work, let me know. -- esr)
altos7|alt7|altos VII,
	am, mir,
	cols#80, lines#24, xmc#0,
	acsc=j5k3l2m1n8q\:t4u9v=w0x6, blink=\EG2, bold=\EGt,
	clear=\E+^^, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dim=\EGp, dl1=\ER, ed=\EY, el=\ET, home=^^, ht=^I, il1=\EE,
	ind=^J, invis=\EG1,
	is2=\E`\:\Ee(\EO\Ee6\Ec41\E~4\Ec21\Eu\E~2, kDL=^Am\r,
	kEOL=^An\r, kbs=^H, kcbt=^AK\r, kclr=^AL\r, kcub1=^H,
	kcud1=^J, kcuf1=^L, kcuu1=^K, kdch1=^AM\r, kel=^AN\r,
	kf0=^AI\r, kf1=^A@\r, kf2=^AA\r, kf3=^AB\r, kf32=^A`\r,
	kf33=^Aa\r, kf34=^Ab\r, kf35=^Ac\r, kf36=^Ad\r, kf37=^Ae\r,
	kf38=^Af\r, kf39=^Ag\r, kf4=^AC\r, kf40=^Ah\r, kf41=^Ai\r,
	kf42=^Aj\r, kf43=^Ak\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r,
	kf8=^AG\r, kf9=^AH\r, khome=^^, kil1=^AJ\r, kind=^AO\r,
	knp=\EK, kpp=\EJ, mc4=\EJ, mc5=\Ed#, nel=^M^J, ri=\Ej,
	rmir=\Er, smir=\Eq, use=adm+sgr,
altos7pc|alt7pc|altos PC VII,
	kend=\ET, use=altos7,

#### Hewlett-Packard (hp)
#
#	Hewlett-Packard
#	8000 Foothills Blvd
#	Roseville, CA 95747
#	Vox: 1-(916)-785-4363	(Technical response line for VDTs)
#	     1-(800)-633-3600	(General customer support)
#
#
# As of March 1998, HP no longer has any terminals in production.
# The 700 series (22, 32, 41, 44, 92, 94, 96, 98) is still being
# supported (they still have parts). So are the 2392a and 2394a.
# See the WORKSTATION CONSOLES section for the 700s.
#

# Generic HP terminal - this should (hopefully) work on any HP terminal.
hpgeneric|hp|hewlett-packard generic terminal,
	OTbs, OTpt, am, da, db, mir, xhp,
	cols#80, lines#24, lm#0, vt#6,
	bel=^G, clear=\EH\EJ, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\E&a%p2%dc%p1%dY$<6>, cuu1=\EA, dch1=\EP, dl1=\EM,
	ed=\EJ, el=\EK, hpa=\E&a%p1%dC, ht=^I, hts=\E1, il1=\EL,
	ind=^J, kbs=^H, kcbt=\Ei, rmir=\ER, rmso=\E&d@, rmul=\E&d@,
	sgr0=\E&d@, smir=\EQ, smso=\E&dJ, smul=\E&dD, tbc=\E3,
	vpa=\E&a%p1%dY,

hp110|hewlett-packard model 110 portable,
	lines#16, use=hpgeneric,

hp+pfk+cr|hp function keys with CR,
	kf1=\Ep\r, kf2=\Eq\r, kf3=\Er\r, kf4=\Es\r, kf5=\Et\r,
	kf6=\Eu\r, kf7=\Ev\r, kf8=\Ew\r,

hp+pfk-cr|hp function keys w/o CR,
	kf1=\Ep, kf2=\Eq, kf3=\Er, kf4=\Es, kf5=\Et, kf6=\Eu, kf7=\Ev,
	kf8=\Ew,

# The hp2621s use the same keys for the arrows and function keys,
# but not separate escape sequences. These definitions allow the
# user to use those keys as arrow keys rather than as function
# keys.
hp+pfk+arrows|hp alternate arrow definitions,
	kcub1=\Eu\r, kcud1=\Ew\r, kcuf1=\Ev\r, kcuu1=\Et\r, kf1@,
	kf2@, kf3@, kf4@, kf5@, kf6@, kf7@, kf8@, khome=\Ep\r, kind=\Er\r,
	kll=\Eq\r, kri=\Es\r,

hp+arrows|hp arrow definitions,
	kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA, khome=\Eh,
	kind=\ES, kll=\EF, kri=\ET,

# Generic stuff from the HP 262x series
#
hp262x|HP 262x terminals,
	xhp,
	blink=\E&dA, dch1=\EP$<2>, ed=\EJ, ht=\011$<2>, ind=\ES,
	invis=\E&dS, ip=$<2>, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kdch1=\EP, kdl1=\EM, ked=\EJ, kel=\EK, khome=\Eh,
	kich1=\EQ, kil1=\EL, kind=\ES, knp=\EU, kpp=\EV, kri=\ET,
	krmir=\ER, rev=\E&dB, rmkx=\E&s0A, rmso=\E&d@, rmul=\E&d@,
	sgr=\E&d%{64}%?%p1%t%{66}%|%;%?%p2%t%{68}%|%;%?%p3%t%{66}%|%;%?%p4%t%{65}%|%;%c,
	sgr0=\E&d@, smkx=\E&s1A, smso=\E&dB, smul=\E&dD,

# Note: no <home> on HPs since that homes to top of memory, not screen.
# Due to severe 2621 braindamage, the only way to get the arrow keys to
# transmit anything at all is to turn on the function key labels
# with <smkx>, and even then the user has to hold down shift!
# The default 2621 turns off the labels except when it has to to
# enable the function keys. If your installation prefers labels
# on all the time, or off all the time (at the "expense" of the
# function keys), use 2621-nl or 2621-wl.
#
# Note: there are newer ROMs for 2621's that allow you to set
# strap A so the regular arrow keys xmit \EA, etc, as with the
# 2645. However, even with this strap set, the terminal stops
# xmitting if you reset it, until you unset and reset the strap!
# Since there is no way to set/unset the strap with an escape
# sequence, we don't use it in the default.
# If you like, you can use 2621-ba (brain-damaged arrow keys).
hp2621-ba|2621 w/new rom and strap A set,
	rmkx@, smkx@, use=hp+arrows, use=hp2621,

# hp2621 with function labels. Most of the time they are off,
# but inside vi, the function key labels appear. You have to
# hold down shift to get them to xmit.
hp2621|hp2621a|hp2621A|2621|2621a|2621A|hp2621-wl|2621-wl|hp 2621 w/labels,
	is2=\E&jA\r, rmkx=\E&jA, use=hp2621-fl,
hp2621-fl|hp 2621,
	xhp@, xon,
	pb#19200,
	cbt=\Ei, cup=\E&a%p2%dc%p1%dY, dch1=\EP$<2>, ht=\011$<2>,
	ip=$<2>, is2=\E&j@\r, rmkx=\E&j@, rmso=\E&d@, rmul=\E&d@,
	sgr0=\E&d@, smkx=\E&jB, smso=\E&dD, smul=\E&dD,
	use=hp+pfk+cr, use=hpgeneric,

# To use hp2621p printer, setenv TERM=2621p, PRINTER=2612p
hp2621p|hp 2621 with printer,
	mc4=\E&p13C, mc5=\E&p11C, use=hp2621,

hp2621p-a|hp2621p with fn as arrows,
	use=hp+pfk+arrows, use=hp2621p,

# hp2621 with k45 keyboard
hp2621-k45|hp2621k45|k45|hp 2621 with 45 keyboard,
	kbs=^H, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA,
	khome=\Eh, rmkx=\E&s0A, smkx=\E&s1A, use=hp2621,

# 2621 using all 48 lines of memory, only 24 visible at any time.
hp2621-48|48 line 2621,
	lines#48,
	cup=\E&a%p2%dc%p1%dR, home=\EH, vpa=\E&a%p1%dR,
	use=hp2621,

# 2621 with no labels ever. Also prevents vi delays on escape.
hp2621-nl|hp 2621 with no labels,
	kcub1@, kcud1@, kcuf1@, kcuu1@, khome@, rmkx@, smkx@,
	use=hp2621-fl,

# Needed for UCB ARPAVAX console, since lsi-11 expands tabs
# (wrong).
#
hp2621-nt|hp 2621 w/no tabs,
	ht@, use=hp2621,

# Hp 2624 B with 4 or 10 pages of memory.
#
# Some assumptions are made with this entry. These settings are
# NOT set up by the initialization strings.
#
# Port Configuration
# 	RecvPace=Xon/Xoff
# 	XmitPace=Xon/Xoff
# 	StripNulDel=Yes
#
# Terminal Configuration
# 	InhHndShk=Yes
# 	InhDC2=Yes
# 	XmitFnctn(A)=No
# 	InhEolWrp=No
#
# Note: the 2624 DOES have a true <home>, believe it or not!
#
# The 2624 has an "error line" to which messages can be sent.
# This is CLOSE to what is expected for a "status line". However,
# after a message is sent to the "error line", the next carriage
# return is EATEN and the "error line" is turned back off again!
# So I guess we can't define <hs>, <eslok>, <wsl>, <dsl>, <fsl>, <tsl>.
#
# This entry supports emacs (and any other program that uses raw
# mode) at 4800 baud and less. I couldn't get the padding right
# for 9600.
#
# (hp2624: replaced NUL sequences in flash with mandatory pauses -- esr)
hp2624|hp2624a|hp2624b|hp2624b-4p|Hewlett Packard 2624 B,
	da, db,
	lm#96,
	flash=\E&w13F$<66/>\E&w12F$<66/>\E&w13F$<66/>\E&w12F,
	use=hp+labels, use=scrhp,

# This hp2626 entry does not use any of the fancy windowing stuff
# of the 2626.
#
# Indeed, terminfo does not yet handle such stuff. Since changing
# any window clears memory, it is probably not possible to use
# this for screen opt.
#
# ed is incredibly slow most of the time - I am guessing at the
# exact padding. Since the terminal uses xoff/xon this is intended
# only for cost computation, so that the terminal will prefer el
# or even dl1 which is probably faster!
#
# \ED\EJ\EC hack for ed from Ed Bradford - apparently ed is only
# extra slow on the last line of the window.
#
# The padding probably should be changed.
#
hp2626|hp2626a|hp2626p|hp 2626,
	da, db,
	lm#0, pb#19200,
	ed=\ED\EJ$<500>\EC, indn=\E&r%p1%dD, ip=$<4>,
	is2=\E&j@\r, rin=\E&r%p1%dU, use=hp+pfk-cr,
	use=hp+labels, use=scrhp,

# This entry is for sysline. It allocates a 23 line window with
# a 115 line workspace for regular use, and a 1 line window for
# the status line.
#
# This assumes port 2 is being used.
# Turn off horizontal line, Create ws #1 with 115 lines,
# Create ws #2 with 1 line, Create window #1 lines 1-23,
# Create window #2 lines 24-24, Attach cursor to workspace #1.
# Note that this clears the tabs so it must be done by tset before
# it sets the tabs.
#
hp2626-s|hp 2626 using only 23 lines,
	eslok, hs,
	lines#23,
	fsl=\E&d@\E&w7f2p1I\E&w4f1I,
	is1=\E&q3t0{0H \E&w0f115n1I \E&w0f1n2I \E&w2f1i0d0u22l0S \E&w2f2i0d23u23l0S \E&w7f2p1I \r,
	tsl=\E&w7f2p2I\E&w4f2I\r\EK\E&a%p1%dC, use=hp2626,
# Force terminal back to 24 lines after being 23.
hp2626-ns|hp 2626 using all 24 lines,
	is1=\E&q3t0{0H \E&w0f118n1I \E&w0f1n2I \E&w2f1i0d0u23l0S \E&w3f2I \E&w7f2p1I \r,
	use=hp2626,
# Various entries useful for small windows on 2626.
hp2626-12|hewlett-packard 2626 12 lines,
	lines#12, use=hp2626,
hp2626-12x40|hewlett-packard 2626 12 lines 40 columns,
	cols#40, lines#12, use=hp2626,
hp2626-x40|hewlett-packard 2626 40 columns,
	cols#40, use=hp2626,
hp2626-12-s|hewlett-packard 2626 11 lines plus status,
	lines#11, use=hp2626-s,

#
# hp2627 color tubes from University of Wisconsin
#
hp2627a-rev|hp 2627 with reverse video colors,
	cr=^M, cud1=^J, ht=^I, ind=^J,
	is2=\E&v0m1a0b0c1x1y1z1i0a0b1c1x1y1z0i0S\E&j@\r\E3\r,
	kbs=^H, kcub1=^H, kcud1=^J, nel=^M^J, rmul=\E&v0S\E&d@,
	smul=\E&dD\E&v1S, use=hp2621-nl,
hp2627a|hp 2627 color terminal with no labels,
	cr=^M, cud1=^J, ht=^I, ind=^J,
	is2=\E&v0m1a1b0c1i0a1b1c2i1a0b0c0i0S\E&j@\r\E3\r,
	kbs=^H, kcub1=^H, kcud1=^J, nel=^M^J, rmso=\E&v0S,
	rmul=\E&v0S\E&d@, smso=\E&v2S, smul=\E&dD\E&v1S,
	use=hp2621-nl,
hp2627c|hp 2627 color (cyan) terminal with no labels,
	cr=^M, cud1=^J, ht=^I, ind=^J,
	is2=\E&v0m1a0b0c2i1a1b0c1i0a1b1c0i0S\E&j@\r\E3\r,
	kbs=^H, kcub1=^H, kcud1=^J, nel=^M^J, use=hp2627a,

# hp2640a doesn't have the Y cursor addressing feature, and C is
# memory relative instead of screen relative, as we need.
#
hp2640a|hp 2640a,
	cup@, rmkx@, smkx@, use=hp2645,

hp2640b|hp2644a|hp 264x series,
	rmkx@, smkx@, use=hp2645,

# (hp2641a: removed unknown :gu: -- esr)
hp2641a|hp2645a|hp2647a|HP 264?A series BRL entry,
	am, da, db, mir, xhp,
	cols#80, lines#24,
	bel=^G, clear=\EH\EJ, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\E&a%p2%2dc%p1%2dY, cuu1=\EA, dch1=\EP, dl1=\EM,
	ed=\EJ, el=\EK, hpa=\E&a%p1%2dC, ht=^I,
	if=/usr/share/tabset/std, il1=\EL, ind=^J,
	is2=\EE$<500/>, kbs=^H, kcub1=^H, kcud1=^J, nel=^M^J,
	rmir=\ER, rmso=\E&d@, smir=\EQ, smso=\E&dB,
	vpa=\E&a%p1%2dY,

# This terminal should be used at 4800 baud or less. It needs padding for
# plain characters at 9600, I guessed at an appropriate cr delay.  It really
# wants ^E/^F handshaking, but that doesn't work well even if you write
# software to support it.
hp2645|hp45|HP 2645 series,
	pb#9600,
	blink=\E&dA, cr=\r$<20>, dim=\E&dH, kctab=\E2, kcub1=\ED,
	kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kdch1=\EP, kdl1=\EM,
	ked=\EJ, kel=\EK, khome=\Eh, khts=\E1, kich1=\EQ, kil1=\EL,
	kind=\ES, knp=\EU, kpp=\EV, kri=\ET, krmir=\ER, rev=\E&dB,
	rmkx=\E&s0A,
	sgr=\E&d%{64}%?%p1%t%{66}%|%;%?%p2%t%{68}%|%;%?%p3%t%{66}%|%;%?%p4%t%{65}%|%;%?%p5%t%{72}%|%;%?%p6%t%{66}%|%;%c,
	sgr0=\E&d@, smkx=\E&s1A, smul=\E&dD, use=hpgeneric,
# You should use this terminal at 4800 baud or less.
hp2648|hp2648a|HP 2648a graphics terminal,
	clear=\EH\EJ$<50>, cup=\E&a%p2%dc%p1%dY$<20>,
	dch1=\EP$<7>, ip=$<5>, use=hp2645,

# The HP 150 terminal is a fairly vanilla HP terminal, with the
# clreol standout problem. It also has graphics capabilities and
# a touch screen, which we don't describe here.
hp150|hewlett packard Model 150,
	OTbs, use=hp2622,

# HP 2382a terminals, "the little ones." They don't have any
# alternate character set support and sending out ^N/^O will
# leave the screen blank.
hp2382a|hp2382|hewlett packard 2382a,
	da, db,
	lh#1, lm#48,
	acsc@,
	pln=\E&f0a%p1%dk%p2%l%Pa%?%ga%t%ga%d%e1%;d0L%?%ga%!%t %;%p2%s,
	rmacs@,
	sgr=\E&d%{0}%Pa%?%p4%t%{1}%ga%+%Pa%;%?%p1%p3%|%p6%|%t%{2}%ga%+%Pa%;%?%p2%p6%|%t%{4}%ga%+%Pa%;%?%p1%p5%|%t%{8}%ga%+%Pa%;%?%p7%t%?%ga%ts%ga%{64}%+%e%{83}%;%e%?%ga%t%ga%{64}%+%e%{64}%;%;%c,
	sgr0=\E&d@, smacs@, use=hp+labels, use=scrhp,

hp2621-a|hp2621a-a|hp2621 with fn as arrows,
	use=hp+pfk+arrows, use=hp2621-fl,

# newer hewlett packard terminals

newhpkeyboard|generic entry for HP extended keyboard,
	kbs=^H, kcbt=\Ei, kclr=\EJ, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kdch1=\EP, kdl1=\EM, ked=\EJ, kel=\EK, khome=\Eh,
	kich1=\EQ, kil1=\EL, kind=\ET, kll=\EF, knp=\EU, kpp=\EV,
	kri=\ES, krmir=\ER, rmkx=\E&s0A, smkx=\E&s1A,
	use=hp+pfk-cr,

newhp|generic entry for new hewlett packard terminals,
	am, bw, mir, xhp, xon,
	cols#80, lines#24, pb#4800,
	acsc=2[3@4>5I9(\:'JSKWLQMAO#P$Q;R!S"T1U2V4W3X\:Y+Z*dHjGkTlRmFn/q\,t5u6v8w7x.,
	bel=^G, blink=\E&dA, bold=\E&dF, cbt=\Ei, cr=^M, cub1=^H,
	cud1=^J, cuf1=\EC, cuu1=\EA, dch1=\EP$<2>, dim=\E&dH,
	dl1=\EM, ed=\EJ, el=\EK, ht=\011$<2>, hts=\E1, il1=\EL, ind=^J,
	invis=\E&dS, ip=$<2>, is1=\E&jB$<8>, nel=^M^J,
	pfkey=\E&f0a%p1%dk0d%p2%l%dL%p2%s,
	pfloc=\E&f1a%p1%dk0d%p2%l%dL%p2%s,
	pfx=\E&f2a%p1%dk0d%p2%l%dL%p2%s, rev=\E&dB, ri=\ET,
	rmacs=^O, rmir=\ER, rmso=\E&d@, rmul=\E&d@, rs1=\Eg,
	sgr=\E&d%{0}%Pa%?%p4%t%{1}%ga%+%Pa%;%?%p1%p3%|%p6%|%t%{2}%ga%+%Pa%;%?%p2%p6%|%t%{4}%ga%+%Pa%;%?%p1%p5%|%t%{8}%ga%+%Pa%;%?%p7%t%?%ga%ts%ga%{64}%+%e%{83}%;%e%?%ga%t%ga%{64}%+%e%{64}%;%;%c%?%p9%t\016%e\017%;,
	sgr0=\E&d@\017, smacs=^N, smir=\EQ, smso=\E&dJ, smul=\E&dD,
	tbc=\E3, use=newhpkeyboard,

memhp|memory relative addressing for new HP ttys,
	vt#6,
	clear=\EH\EJ$<40>, cub=\E&a-%p1%dC, cud=\E&a+%p1%dR,
	cuf=\E&a+%p1%dC, cup=\E&a%p1%dr%p2%dC, cuu=\E&a-%p1%dR,
	home=\EH, hpa=\E&a%p1%dC, ll=\E&a23R\r,
	mrcup=\E&a%p1%dr%p2%dC, vpa=\E&a%p1%dR, use=newhp,

scrhp|screen relative addressing for new HP ttys,
	clear=\E&a0c0Y\EJ$<40>, cub=\E&a-%p1%dC,
	cud=\E&a+%p1%dR, cuf=\E&a+%p1%dC,
	cup=\E&a%p1%dy%p2%dC$<10>, cuu=\E&a-%p1%dR,
	home=\E&a0y0C, hpa=\E&a%p1%dC, ll=\E&a0y0C\EA,
	mrcup=\E&a%p1%dr%p2%dC, vpa=\E&a%p1%dY, use=newhp,

# (hp+labels: added label values from a BRL termcap -- esr)
hp+labels|"standard" label info for new HP ttys,
	lh#2, lw#8, nlab#8,
	lf0=f1, lf1=f2, lf2=f3, lf3=f4, lf4=f5, lf5=f6, lf6=f7, lf7=f8,
	pln=\E&f2a%p1%dk%p2%l%Pa%?%ga%t%ga%d%e1%;d0L%?%ga%!%t %;%p2%s,
	rmln=\E&j@, smln=\E&jB,

hp+printer|"standard" printer info for HP ttys,
	ff=\E&p4u0C, mc0=\EH\E&p4dF, mc4=\E&p13C, mc5=\E&p11C,


# The new hp2621b is kind of a cross between the old 2621 and the
# new 262x series of machines. It has dip-switched options.
# The firmware has a bug in it such that if you give it a null
# length label, the following character is eaten!
hp2621b|hp 2621b with old style keyboard,
	lh#1, lm#48, lw#8, nlab#8,
	kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA, khome=\Eh,
	kind=\ET, kll=\EF, kri=\ES,
	pln=\E&f0a%p1%dk%p2%l%Pa%?%ga%t%ga%d%e1%;d3L%?%ga%!%t%{32}%c%;%p2%s\E%{111}%p1%+%c\r,
	smln=\E&jB, use=hp2621,

hp2621b-p|hp 2621b with printer,
	use=hp+printer, use=hp2621b,

# hp2621b - new 2621b with new extended keyboard
# these are closer to the new 26xx series than the other 2621b
hp2621b-kx|hp 2621b with extended keyboard,
	use=newhpkeyboard, use=hp2621b,

hp2621b-kx-p|hp 2621b with new keyboard & printer,
	use=hp+printer, use=hp2621b-kx,

# Some assumptions are made in the following entries.
# These settings are NOT set up by the initialization strings.
#
#    Port Configuration
# RecvPace=Xon/Xoff	XmitPace=Xon/Xoff	StripNulDel=Yes
#
#    Terminal Configuration
# InhHndShk(G)=Yes	InhDC2(H)=Yes
# XmitFnctn(A)=No		InhEolWrp=No
#
#
# Hp 2622a & hp2623a display and graphics terminals
#
hp2622|hp2622a|hp 2622,
	da, db,
	lm#0, pb#19200,
	is2=\E&dj@\r, use=hp+pfk-cr, use=hp+labels, use=scrhp,

# The 2623 is a 2622 with extra graphics hardware.
hp2623|hp2623a|hp 2623,
	use=hp2622,

hp2624b-p|hp2624b-4p-p|hewlett packard 2624 B with printer,
	use=hp+printer, use=hp2624,

# The hewlett packard B can have an optional extra 6 pages of memory.
hp2624-10p|hp2624a-10p|hp2624b-10p|hewlett packard 2624 B w/ 10 pages of memory,
	lm#240, use=hp2624,

hp2624b-10p-p|hewlett packard 2624 B w/ extra memory & printer,
	lm#240, use=hp2624b-p,

# Color manipulations for HP terminals
hp+color|hp with colors,
	ccc,
	colors#16, ncv#17, pairs#7,
	initp=\E&v%?%p2%{1000}%=%t1%e.%p2%d%;a%?%p3%{1000}%=%t1%e.%p3%d%;b%?%p4%{1000}%=%t1%e.%p4%d%;c%?%p5%{1000}%=%t1%e.%p5%d%;x%?%p6%{1000}%=%t1%e.%p6%d%;y%?%p7%{1000}%=%t1%e.%p7%d%;z%p1%dI,
	oc=\E&v0m1a1b1c0I\E&v1a1I\E&v1b2I\E&v1a1b3I\E&v1c4I\E&v1a1c5I\E&v1b1c6I\E&v1x1y7I,
	op=\E&v0S, scp=\E&v%p1%dS,

# <is2> sets the screen to be 80 columns wide
hp2397a|hp2397|hewlett packard 2397A color terminal,
	is2=\E&w6f80X, use=memhp, use=hp+labels, use=hp+color,

#  HP 700/44 Setup parameters:
# Terminal Mode		HP-PCterm
# Inhibit Auto Wrap	NO
# Status Line		Host Writable
# PC Character Set	YES
# Twenty-Five Line Mode	YES
# XON/XOFF		@128 or 64 (sc)
# Keycode Mode 		NO   or YES (sc)
# Backspace Key		BS or BS/DEL
#
# <is2> 	sets pcterm; autowrap; 25 lines; pc char set; prog DEL key;
# \E\\? does not turn off keycode mode
# <smsc>	sets alternate start/stop; keycode on
hpansi|hp700|hewlett packard 700/44 in HP-PCterm mode,
	am, eo, xenl, xon,
	cols#80, lines#25,
	acsc=j\331k\277l\332m\300n\305q\304t\303u\264v\301w\302x\263,
	bel=^G, cbt=\E[Z, civis=\E[?25l, clear=\E[2J\E[H,
	cnorm=\E[?25h, cr=^M, cub1=\E[D, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P, dl1=\E[M,
	ed=\E[J, el=\E[K, home=\E[H, ht=^I, ich1=\E[@, il1=\E[L,
	ind=^J,
	is2=\E[44"p\E[?7h\E[>10h\E[>12h\EP1;1|3/7F\E\\,
	kbs=^H, kcbt=\E[Z, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kend=\E[4~, kf1=\E[17~, kf10=\E[28~,
	kf2=\E[18~, kf3=\E[19~, kf4=\E[20~, kf5=\E[21~, kf6=\E[23~,
	kf7=\E[24~, kf8=\E[25~, kf9=\E[26~, khome=\E[1~, knp=\E[6~,
	kpp=\E[5~, rmam=\E[?7l,
	rmsc=\E[>11l\EP1**x0/11;1/13\E[m\E\\, rmso=\E[m,
	rmul=\E[m, sgr0=\E[m, smam=\E[?7h,
	smsc=\E[>11h\EPO**x0/65;1/67\E\\$<250>, smso=\E[7m,
	smul=\E[4m, xoffc=g, xonc=e,
#
# (hp2392: copied <rmir> here from hpex -- esr)
hp2392|239x series,
	cols#80,
	cbt=\Ei, cup=\E&a%p1%dy%p2%dC, kf1=\Ep\r, kf2=\Eq\r,
	kf3=\Er\r, kf4=\Es\r, kf5=\Et\r, kf6=\Eu\r, kf7=\Ev\r,
	kf8=\Ew\r, khome=\Eh, kind=\EU, knp=\Eu, kpp=\Ev, kri=\EV,
	rmir=\ER, rmul=\E&d@, smir=\EQ, smul=\E&dD, vpa=\E&a%p1%dY,
	use=hpsub,

hpsub|hp terminals -- capability subset,
	am, da, db, mir, xhp, xon,
	lines#24,
	bel=^G, clear=\EH\EJ, cr=^M, cub1=^H, cud1=\EB, cuf1=\EC,
	cuu1=\EA, dch1=\EP, dl1=\EM, ed=\EJ, el=\EK, hpa=\E&a%p1%dC,
	ht=^I, if=/usr/share/tabset/stdcrt, il1=\EL, ind=^J,
	is2=\E&s1A\E<\E&k0\\, kbs=^H, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, khome=\Eh, rmkx=\E&s0A, rmso=\E&d@,
	sgr0=\E&d@, smkx=\E&s1A, smso=\E&dB,

# hpex:
#	May be used for most 24 x 80 hp terminals,
# but has no padding added, so may allow runover in some terminals at high
# baud rates.  Will not work for hp2640a or hp2640b terminals, hp98x6 and
# hp98x5 terminal emulators or hp98x6 consoles.
# 	Adds xy-cursor addressing, vertical cursor addressing, home,
# last line, and underline capabilities.
#
# (hpex: removed memory-lock capabilities ":ml=\El:mu=\Em:",
# moved <rmir> here from hpsub -- esr)
hpex|hp extended capabilites,
	cr=^M, cud1=^J, cup=\E&a%p1%dy%p2%dC, ht=^I, ind=^J, kbs=^H,
	kcub1=^H, kcud1=^J, nel=^M^J, rmir=\ER, rmul=\E&d@, smir=\EQ,
	smul=\E&dD, vpa=\E&a%p1%dY, use=hpsub,

# From: Ville Sulko <Ville.Sulko@bip.atk.tpo.fi>, 05 Aug 1996
hp2|hpex2|hewlett-packard extended capabilities newer version,
	am, da, db, mir, xhp,
	cols#80, lh#2, lines#24, lm#0, lw#8, nlab#8, xmc#0,
	bel=^G, clear=\E&a0y0C\EJ, cr=^M, cub1=^H, cud1=\EB,
	cuf1=\EC, cup=\E&a%p1%dy%p2%dC, cuu1=\EA, dch1=\EP,
	dl1=\EM, ed=\EJ, el=\EK, hpa=\E&a%p1%dC, ht=^I, hts=\E1,
	il1=\EL, ind=^J, kbs=^H, kclr=\EJ, kctab=\E2, kcub1=\ED,
	kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kdch1=\EP, kdl1=\EM,
	ked=\EJ, kel=\EK, kf1=\Ep, kf2=\Eq, kf3=\Er, kf4=\Es, kf5=\Et,
	kf6=\Eu, kf7=\Ev, kf8=\Ew, khome=\Eh, khts=\E1, kich1=\EQ,
	kil1=\EL, kind=\ES, kll=\EF, knp=\EU, kpp=\EV, kri=\ET,
	krmir=\ER, ktbc=\E3, meml=\El, memu=\Em,
	pfkey=\E&f%p1%dk%p2%l%dL%p2%s,
	pfloc=\E&f1a%p1%dk%p2%l%dL%p2%s,
	pfx=\E&f2a%p1%dk%p2%l%dL%p2%s,
	pln=\E&f%p1%dk%p2%l%dd0L%p2%s, rmir=\ER, rmkx=\E&s0A,
	rmln=\E&j@, rmso=\E&d@, rmul=\E&d@,
	sgr=\E&d%?%p7%t%{115}%c%;%p1%p3%|%p6%|%{2}%*%p2%{4}%*%+%p4%+%p5%{8}%*%+%{64}%+%c%?%p9%t%'\016'%c%e%'\017'%c%;,
	sgr0=\E&d@, smir=\EQ, smkx=\E&s1A, smln=\E&jB, smso=\E&dB,
	smul=\E&dD, tbc=\E3, vpa=\E&a%p1%dY,

# HP 236 console
# From: <ddavis@ic.berkeley.edu>
hp236|hp236 internal terminal emulator,
	OTbs, am,
	cols#80, lines#24,
	clear=\EF, cnorm=\EDE, cub1=^H,
	cup=\EE%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, cvvis=\EDB,
	dch1=\EJ, dl1=\EH, el=\EK, ich1=\EI, il1=\EG, rmso=\ECI,
	sgr0=\ECI, smso=\EBI,

# This works on a hp300 console running Utah 4.3 BSD
# From: Craig Leres <leres@okeeffe.berkeley.edu>
hp300h|HP Catseye console,
	OTbs, am, da, db, mir, xhp,
	cols#128, lines#51, lm#0, xmc#0,
	bel=^G, cbt=\Ei, clear=\E&a0y0C\EJ, cr=^M, cub1=^H, cud1=\EB,
	cuf1=\EC, cup=\E&a%p1%dy%p2%dC, cuu1=\EA, dch1=\EP,
	dl1=\EM, ed=\EJ, el=\EK, hpa=\E&a%p1%dC, ht=^I,
	if=/usr/share/tabset/stdcrt, il1=\EL, ind=^J, kbs=^H,
	kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA, khome=\Eh,
	rmir=\ER, rmkx=\E&s0A, rmso=\E&d@, rmul=\E&d@, sgr0=\E&d@,
	smir=\EQ, smkx=\E&s1A, smso=\E&dB, smul=\E&dD, tbc=\E3,
	vpa=\E&a%p1%dY,
# From: Greg Couch <gregc@ernie.berkeley.edu>
hp9837|hp98720|hp98721|HP 9000/300 workstations,
	OTbs, am, da, db, mir, xhp,
	cols#128, it#8, lines#46, lm#0,
	bel=^G, cbt=\Ei, clear=\E&a0y0C\EJ, cub1=^H, cud1=\EB,
	cuf1=\EC, cup=\E&a%p1%dy%p2%dC, cuu1=\EA, dch1=\EP,
	dl1=\EM, ed=\EJ, el=\EK, hpa=\E&a%p1%dC, ht=^I, hts=\E1,
	il1=\EL, ind=^J, is2=\E&v0m1b0i&j@, kbs=^H, kcub1=\ED,
	kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kdch1=\EP, kdl1=\EM,
	ked=\EJ, kel=\EK, khome=\Eh, kich1=\EQ, kil1=\EL, knp=\EU,
	kpp=\EV, rmir=\ER, rmkx=\E&s0A, rmso=\E&v0S, rmul=\E&d@,
	sgr0=\E&d@, smir=\EQ, smkx=\E&s1A, smso=\E&v5S, smul=\E&dD,
	tbc=\E3, vpa=\E&a%p1%dY,
# HP 9845 desktop computer from BRL
# (hp9845: removed unknown capability :gu: -- esr)
hp9845|HP 9845,
	OTbs, am, da, db, eo, mir, xhp,
	cols#80, lines#21,
	OTbc=\ED, clear=\EH\EJ, cub1=\ED, cud1=\EB, cuf1=\EC,
	cup=\E&a%p2%2dc%p1%2dY, cuu1=\EA, dch1=\EP, dl1=\EM,
	ed=\EJ, el=\EK, if=/usr/share/tabset/std, il1=\EL,
	rmir=\ER, rmso=\E&d@, smir=\EQ, smso=\E&dB,
# From: Charles A. Finnell of MITRE <finnell@mitre.org>, developed 07SEP90
# (hp98550: replaced /usr/share/tabset/9837 with std because <it#8>,<hts=\E1>;
# added empty <acsc> to avoid warnings re <smacs>/<rmacs> --esr)
hp98550|hp98550a|HP 9000 Series 300 color console,
	OTbs, am, da, db, mir, xhp,
	cols#128, it#8, lines#49, lm#0,
	acsc=, bel=^G, blink=\E&dA, bold=\E&dJ, cbt=\Ei, civis=\E*dR,
	clear=\EH\EJ, cnorm=\E*dQ, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\E&a%p1%dy%p2%dC, cuu1=\EA, dch1=\EP, dim=\E&dH,
	dl1=\EM, ed=\EJ, el=\EK, hpa=\E&a%p1%dC, ht=^I, hts=\E1,
	if=/usr/share/tabset/std, il1=\EL, ind=^J, invis=\E&ds,
	kbs=^H, kclr=\EJ, kctab=\E2, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kdch1=\EP, kdl1=\EM, ked=\EJ, kel=\EK, kf1=\Ep,
	kf2=\Eq, kf3=\Er, kf4=\Es, kf5=\Et, kf6=\Eu, kf7=\Ev, kf8=\Ew,
	khome=\Eh, khts=\E1, kich1=\EQ, kil1=\EL, kind=\ES, kll=\EF,
	knp=\EU, kpp=\EV, kri=\ET, krmir=\ER, ktbc=\E3, rev=\E&dJ,
	rmacs=^O, rmir=\ER, rmkx=\E&s0A, rmso=\E&d@, rmul=\E&d@,
	sgr0=\E&d@, smacs=^N, smir=\EQ, smkx=\E&s1A, smso=\E&dJ,
	smul=\E&dD, tbc=\E3, vpa=\E&a%p1%dY,
# From: Victor Duchovni <vic@fine.princeton.edu>
# (hp700-wy: removed obsolete ":nl=^J:";
# replaced /usr/share/tabset/hp700-wy with std because <it#8>,<hts=\E1> -- esr)
hp700-wy|HP700/41 emulating wyse30,
	OTbs, am, bw, mir, msgr,
	cols#80, it#8, lines#24, xmc#1,
	cbt=\EI, clear=^Z, cr=^M, cub1=^H, cud1=^V, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\EY, el=\ET$<10/>, home=^^, ht=^I, hts=\E1,
	if=/usr/share/tabset/stdcrt, il1=\EE$<0.7*/>,
	is1=\E~"\EC\Er\E(\EG0\003\E`9\E`1, kbs=\177, kcbt=\EI,
	kclr=^Z, kcub1=^H, kcud1=^V, kcuf1=^L, kcuu1=^K, ked=\EY,
	kel=\ET, khome=^^, khts=\EI, kich1=\Eq, krmir=\Er, ll=^^^K,
	ri=\Ej, rmir=\Er, rmso=\EG0$<10/>, rmul=\EG0$<10/>,
	sgr0=\EG0$<10/>, smir=\Eq, smso=\EG4$<10/>,
	smul=\EG8$<10/>, tbc=\E0, vpa=\E[%p1%{32}%+%c,
hp70092|hp70092a|hp70092A|HP 700/92,
	am, da, db, xhp,
	cols#80, lh#2, lines#24, lm#0, lw#8, nlab#8,
	acsc=0cjgktlrmfn/q\,t5u6v8w7x., bel=^G, blink=\E&dA,
	bold=\E&dB, cbt=\Ei, clear=\E&a0y0C\EJ, cr=^M, cub1=^H,
	cud1=\EB, cuf1=\EC, cup=\E&a%p1%dy%p2%dC, cuu1=\EA,
	dch1=\EP, dim=\E&dH, dl1=\EM, el=\EK, hpa=\E&a%p1%dC, ht=^I,
	hts=\E1, il1=\EL, kbs=^H, kclr=\EJ, kctab=\E2, kcub1=\ED,
	kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kdch1=\EP, kdl1=\EM,
	ked=\EJ, kel=\EK, kf1=\Ep, kf2=\Eq, kf3=\Er, kf4=\Es, kf5=\Et,
	kf6=\Eu, kf7=\Ev, kf8=\Ew, khome=\Eh, khts=\E1, kich1=\EQ,
	kil1=\EL, kind=\ES, kll=\EF, knp=\EU, kpp=\EV, kri=\ET,
	krmir=\ER, ktbc=\E3, rev=\E&dB, ri=\ET, rmacs=^O, rmir=\ER,
	rmkx=\E&s0A, rmln=\E&j@, rmso=\E&d@, rmul=\E&d@,
	sgr0=\E&d@, smacs=^N, smir=\EQ, smkx=\E&s1A, smln=\E&jB,
	smso=\E&dJ, smul=\E&dD, tbc=\E3, vpa=\E&a%p1%dY,

bobcat|sbobcat|HP 9000 model 300 console,
	am, da, db, mir, xhp,
	cols#128, it#8, lines#47, xmc#0,
	cbt=\Ei, clear=\EH\EJ, cr=^M, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\E&a%p1%dy%p2%dC$<6/>, cuu1=\EA, dch1=\EP,
	dl1=\EM$<10*/>, ed=\EJ, el=\EK, hpa=\E&a%p1%dC$<6/>, ht=^I,
	il1=\EL$<10*/>, ind=^J, kbs=^H, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, khome=\Eh, nel=^M^J, rmir=\ER,
	rmkx=\E&s0A, rmso=\E&d@, rmul=\E&d@, sgr0=\E&d@, smir=\EQ,
	smkx=\E&s1A, smso=\E&dB, smul=\E&dD, vpa=\E&a%p1%dY$<6/>,
gator-t|HP 9000 model 237 emulating extra-tall AAA,
	lines#94, use=gator,
gator|HP 9000 model 237 emulating AAA,
	bw, km, mir, ul,
	cols#128, it#8, lines#47,
	bel=^G, cbt=\E[Z, clear=\E[H\E[J, cr=^M, cub1=^H, cud1=^J,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\EM,
	dch=\E[%p1%dP$<4/>, dch1=\E[P, dl=\E[%p1%dM$<1*/>,
	dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, hpa=\E[%i%p1%d`,
	ht=^I, ich=\E[%p1%d@$<4/>, ich1=\E[@, il=\E[%p1%dL$<1*/>,
	il1=\E[L, kbs=^H, kcub1=^H, kcud1=^J, nel=^M^J,
	rep=%p1%c\E[%p2%db$<1*/>, rev=\E[7m, rmso=\E[m,
	rmul=\E[m, sgr0=\E[m, smso=\E[7m, smul=\E[4m,
gator-52|HP 9000 model 237 emulating VT52,
	cols#128, lines#47, use=vt52,
gator-52t|HP 9000 model 237 emulating extra-tall VT52,
	lines#94, use=gator-52,

#### Honeywell-Bull
#
# From: Michael Haardt <michael@gandalf.moria> 11 Jan 93
#

# Honeywell Bull terminal.  Its cursor and function keys send single
# control characters and it has standout/underline glitch.  Most programs
# do not like these features/bugs.  Visual bell is realized by flashing the
# "keyboard locked" LED.
dku7003-dumb|Honeywell Bull DKU 7003 dumb mode,
	cols#80, lines#25,
	clear=^]^_, cr=^M, cub1=^Y, cud1=^K, cuf1=^X,
	cup=\E[%i%p1%d;%p2%dH, cuu1=^Z, ed=^_, el=\E[K,
	flash=\E[2h\E[2l, home=^], ht=^I, ind=^J, kbs=^H, kcub1=^Y,
	kcud1=^K, kcuf1=^X, kcuu1=^Z, khome=^], nel=^M^J,
dku7003|Honeywell Bull DKU 7003 all features described,
	msgr,
	xmc#1,
	blink=\E[5m, bold=\E[7m, dim=\E[2m, rev=\E[7m, rmso=\E[m,
	rmul=\E[m, sgr0=\E[m, smso=\E[7m, smul=\E[4m,
	use=dku7003-dumb,

#### Lear-Siegler (adm)
#
# These guys are long since out of the terminals business, but
# in 1995 many current terminals still have an adm type as one of their
# emulations (usually their stupidest, and usually labeled adm3, though
# these `adm3' emulations normally have adm3a+ capabilities).
#
# WARNING: Some early ADM terminals (including the ADM3 and ADM5) had a
# `diagnostic feature' that sending them a ^G while pin 22 (`Ring Indicator')
# was being held to ground would trigger a send of the top line on the screen.
# A quick fix might be to drop back to a cheesy 4-wire cable with pin 22
# hanging in the air. (Thanks to Eric Fischer, <eric@fudge.uchicago.edu>,
# for clearing up this point.)

adm1a|adm1|lsi adm1a,
	am,
	cols#80, lines#24,
	bel=^G, clear=\E;$<1>, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, home=^^,
	ind=^J,
adm2|lsi adm2,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=\E;, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\EY, el=\ET, home=^^, ich1=\EQ, il1=\EE, ind=^J,
	kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, khome=^^,
# (adm3: removed obsolete ":ma=^K^P:" -- esr)
adm3|lsi adm3,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=^Z, cr=^M, cub1=^H, cud1=^J, ind=^J,
# The following ADM-3A switch settings are assumed for normal operation:
#	SPACE		U/L_DISP	CLR_SCRN	24_LINE
#	CUR_CTL		LC_EN		AUTO_NL		FDX
# Other switches may be set for operator convenience or communication
# requirements.  I recommend
#	DISABLE_KB_LOCK	LOCAL_OFF	103		202_OFF
#	ETX_OFF		EOT_OFF
# Most of these terminals required an option ROM to support lower case display.
# Open the case and look at the motherboard; if you see an open 24-pin DIP
# socket, you may be out of luck.
#
# (adm3a: some capabilities merged in from BRl entry -- esr)
adm3a|lsi adm3a,
	OTbs, am,
	cols#80, lines#24,
	OTma=^K^P, OTnl=^J, bel=^G, clear=\032$<1/>, cr=^M, cub1=^H,
	cud1=^J, cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=^K, home=^^, ind=^J, kcub1=^H, kcud1=^J, kcuf1=^L,
	kcuu1=^K, rs2=^N,
adm3a+|adm3a plus,
	kbs=^H, use=adm3a,
# (adm5: removed obsolete ":ma=^Hh^Jj^Kk^Ll^^H:" & duplicate ":do=^J:" -- esr)
adm5|lsi adm5,
	xmc#1,
	bel=^G, cr=^M, cud1=^J, ed=\EY, el=\ET, kbs=^H, khome=^^,
	rmso=\EG, smso=\EG, use=adm3a+,
# A lot of terminals other than adm11s use these.  Wherever you see
# use=adm+sgr with some of its capabilities disabled, try the
# disabled ones.  They may well work but not have been documented or
# expressed in the using entry.  We'd like to cook up an <sgr> but the
# <rmacs>/<smacs> sequences of the using entries vary too much.
adm+sgr|adm style highlight capabilities,
	invis=\EG1, rev=\EG4, rmso=\EG0, rmul=\EG0, sgr0=\EG0,
	smso=\EG4, smul=\EG8,
# LSI ADM-11 from George William Hartwig, Jr. <geo@BRL-TGR.ARPA> via BRL
# Status line additions from Stephen J. Muir <stephen%comp.lancs.ac.uk@ucl-cs>
# <khome> from <stephen%comp.lancs.ac.uk@ucl-cs.arpa>.  <clear> could also
# be ^Z, according to his entry.
# (adm11: <smul>=\EG4 was obviously erroneous because it also said
# <rev>=\EG4.  Looking at other ADMs confirms this -- esr)
adm11|LSI ADM-11,
	OTbs, am, hs,
	OTkn#8, cols#80, lines#24,
	OTnl=^J, bel=^G, blink=\EG2, clear=\E*, cr=^M, cub1=^H,
	cud1=^J, cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=^K, dsl=\Eh, ed=\EY, el=\ET, fsl=\E(\r, home=^^, ht=^I,
	kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, kf1=^A@\r,
	kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r,
	kf7=^AF\r, kf8=^AG\r, khome=^^, nel=^M^J, tsl=\EF\E),
	use=adm+sgr,
# From: Andrew Scott Beals <bandy@lll-crg.ARPA>
# Corrected by Olaf Siebert <rhialto@polder.ubc.kun.nl>, 11 May 1995
# Supervisor mode info by Ari Wuolle, <awuolle@delta.hut.fi>, 27 Aug 1996
# (adm12: removed obsolete ":kn:ma=j^Jk^P^K^Pl ^R^L^L :".  This formerly had
# <is2>=\Eq but that looked wrong; this <is2> is from Dave Yost <esquire!yost>
# via BRL.  That entry asserted <xmc#1>, but I've left that out because
# neither earlier nor later ADMSs have it -- esr)
#
# You will need to get into the supervisor setup before you can set
# baudrate etc. for your ADM-12+. Press Shift-Ctrl-Setup and you should
# see a lot more setup options.
#
# While in supervisor setup you can also use following codes:
#
# Ctrl-P Personality character selections (configure for example what
#        arrow keys send, if I recall correctly)
# Ctrl-T tabs 1-80   use left&right to move and up to set and
# Ctrl-V tabs 81-158 down to clear tab. Shift-Ctrl-M sets right margin at cursor
# Ctrl-B Binary setup (probably not needed. I think that everything can
#        be set using normal setup)
# Ctrl-A Answerback mode (enter answerback message)
# Ctrl-U User friendly mode (normal setup)
# Ctrl-D Defaults entire setup and function keys from EPROM tables
# Ctrl-S Save both setup and functions keys. Takes from 6 to 10 seconds.
# Ctrl-R Reads both setup and functions keys from NVM.
# Shift-Ctrl-X Unlock keyboard and cancel received X-OFF status
#
# ADM-12+ supports hardware handshaking, but it is DTR/CTS as opposed to
# RTS/CTS used nowadays with virtually every modem and computer. 19200
# bps works fine with hardware flow control.
#
# The following null-modem cable should fix this and enable you to use
# RTS/CTS handshaking (which Linux supports, use CRTSCTS setting). Also
# set ADM-12+ for DTR handshaking from supervisor setup.
#
# PC Serial   ADM-12+
#  --------   -------
#         2 - 3
#         3 - 2
#         4 - 5
#         5 - 20
#       6,8 - 4
#         7 - 7
#        20 - 6,8
#
adm12|lsi adm12,
	OTbs, OTpt, am, mir,
	OTug#1, cols#80, it#8, lines#24,
	bel=^G, clear=^Z, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\EY, el=\ET, home=^^, hts=\E1, ich1=\EQ, il1=\EE,
	is2=\E0        \E1        \E1        \E1        \E1        \E1        \E1        \E1        \E1,
	kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, kf0=^A0\r, kf1=^A1\r,
	kf2=^A2\r, kf3=^A3\r, kf4=^A4\r, kf5=^A5\r, kf6=^A6\r,
	kf7=^A7\r, kf8=^A8\r, kf9=^A9\r, rmir=\Er, smir=\Eq, tbc=\E0,
	use=adm+sgr,
# (adm20: removed obsolete ":kn#7:" -- esr)
adm20|lear siegler adm20,
	OTbs, am,
	cols#80, it#8, lines#24,
	bel=^G, cbt=\EI, clear=^Z, cr=^M, cub1=^H, cuf1=^L,
	cup=\E=%i%p2%{31}%+%c%p1%{31}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\EY, el=\ET, home=^^, ht=^I, ich1=\EQ, il1=\EE,
	kf1=^A, kf2=^B, kf3=^W, kf4=^D, kf5=^E, kf6=^X, kf7=^Z, rmso=\E(,
	sgr0=\E(, smso=\E),
adm21|lear siegler adm21,
	xmc#1,
	bel=^G, cr=^M, cud1=^J, dch1=\EW, dl1=30*\ER, ed=\EY, el=\ET,
	ich1=\EQ, il1=30*\EE, ind=^J, invis@, kbs=^H, kcub1=^H,
	kcud1=^J, kcuf1=^L, kcuu1=^K, khome=^^, use=adm+sgr,
	use=adm3a,
# (adm22: ":em=:" was an obvious typo for ":ei=:"; also,
# removed obsolete ":kn#7:ma=j^Jk^P^K^Pl ^R^L^L :";
# removed bogus-looking \200 from before <cup>. -- esr)
adm22|lsi adm22,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, cbt=\EI, clear=\E+, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\Ey, el=\Et, home=^^, ht=\Ei, ich1=\EQ, il1=\EE,
	is2=\E%\014\014\014\016\003\0\003\002\003\002\0\0\0\0\0\0\0\0\0\0\0,
	kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, kf1=^A@\r,
	kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r,
	kf7=^AF\r, khome=^^, lf1=F1, lf2=F2, lf3=F3, lf4=F4, lf5=F5,
	lf6=F6, lf7=F7, rmso=\E(, sgr0=\E(, smso=\E),
# ADM 31 DIP Switches
#
# This information comes from two versions of the manual for the
# Lear-Siegler ADM 31.
#
# Main board:
#                  rear of case
#   +-||||-------------------------------------+
#   + S1S2                              ||S    +
#   +                                   ||3    +
#   +                                          +
#   +                                ||S       +
#   +                                ||4       +
#   +                                          +
#   +                                          +
#   +                                          +
#   +                                          +
#   +                                          +
# +-+                                          +-+
# +                                              +
# +                               S5 S6 S7       +
# +                               == == ==       +
# +----------------------------------------------+
#            front of case (keyboard)
#
#  S1 - Data Rate - Modem
#  S2 - Data Rate - Printer
# ------------------------
# Data Rate   Setting
# -------------------
# 50          0 0 0 0
# 75          1 0 0 0
# 110         0 1 0 0
# 134.5       1 1 0 0
# 150         0 0 1 0
# 300         1 0 1 0
# 600         0 1 1 0
# 1200        1 1 1 0
# 1800        0 0 0 1
# 2000        1 0 0 1
# 2400        0 1 0 1
# 3600        1 1 0 1
# 4800        0 0 1 1
# 7200        1 0 1 1
# 9600        0 1 1 1
# x           1 1 1 1
#
# S3 - Interface/Printer/Attributes
# ---------------------------------
# Printer Busy Control
# sw1   sw2   sw3
# ---------------
# off   off   off   Busy not active, CD disabled
# off   off   on    Busy not active, CD enabled
# off   on    off   Busy active on J5-20, CD disabled
# on    off   off   Busy active on J5-19, CD disabled - Factory Set.
# on    off   on    Busy active on J5-19, CD enabled
#
# sw4   Used in conjuction with S4 for comm interface control - Fact 0
#
# sw5   Secondary Channel Control (Hardware implementation only) - Fact 0
#
# sw6   ON enables printer BUSY active LOW - Factory Setting
#       OFF enables printer BUSY active HIGH - If set to this, ADM31 senses
#
# sw7   ON - steady cursor - Factory Setting
#       OFF - blinking cursor
#
# sw8   ON causes selected attribute character to be displayed
#       OFF causes SPACE to be displayed instead - Factory Setting
#
# S4 - Interface
# --------------
# Modem Interface
# S3    S4    S4    S4    S4
# sw4   sw1   sw2   sw3   sw4
# ---------------------------
# OFF   ON    OFF   ON    OFF   Enable RS-232C interface, Direct Connect and
#                               Current Loop disabled - Factory Setting
# ON    ON    OFF   ON    OFF   Enable Current Loop interface, Direct Connect
#                               disabled
# OFF   OFF   ON    OFF   ON    Enable Direct Connect interface, RS-232C and
#                               Current Loop Disabled
#
# sw5   ON disables dot stretching mode - Factory Setting
#       OFF enables dot stretching mode
# sw6   ON enables blanking function
#       OFF enables underline function - Factory Setting
# sw7   ON causes NULLS to be displayed as NULLS
#       OFF causes NULLS to be displayed as SPACES - Factory Setting
#
# S5 - Word Structure
# -------------------
# sw1   ON enables BREAK key - Factory Setting
#       OFF disables BREAK key
# sw2   ON selects 50Hz monitor refresh rate
#       OFF selects 60Hz monitor refresh rate - Factory Setting
#
# Modem Port Selection
# sw3   sw4   sw5
# ---------------
# ON    ON    ON    Selects 7 DATA bits, even parity, 2 STOP bits
# OFF   ON    ON    Selects 7 DATA bits, odd  parity, 2 STOP bits
# ON    OFF   ON    Selects 7 DATA bits, even parity, 1 STOP bit - Factory Set.
# OFF   OFF   ON    Selects 7 DATA bits, odd  parity, 1 STOP bit
# ON    ON    OFF   Selects 8 DATA bits, no   parity, 2 STOP bits
# OFF   ON    OFF   Selects 8 DATA bits, no   parity, 1 STOP bit
# ON    OFF   OFF   Selects 8 DATA bits, even parity, 1 STOP bit
# OFF   OFF   OFF   Selects 8 DATA bits, odd  parity, 1 STOP bit
#
# sw6   ON  sends bit 8 a 1 (mark)
#       OFF sends bit 8 as 0 (space) - Factory Setting
# sw7   ON  selects Block Mode
#       OFF selects Conversation Mode - Factory Setting
# sw8   ON  selects Full Duplex operation
#       OFF selects Half Duplex operation - Factory Setting
#
# S6 - Printer
# ------------
# sw1, sw2, sw6, sw7   Reserved - Factory 0
#
# Printer Port Selection
# same as Modem above, bit 8 (when 8 DATA bits) is always = 0
#
# sw8   ON   enables Printer Port
#       OFF disables Printer Port - Factory Setting
#
# S7 - Polling Address
# --------------------
# sw1-7 Establish ASCII character which designates terminal polling address
#       ON  = logic 0
#       OFF = logic 1 - Factory Setting
# sw8   ON   enables Polling Option
#       OFF disables Polling Option - Factory Setting
#
#
# On some older adm31s, S4 does not exist, and S5-sw6 is not defined.
#
# This adm31 entry uses underline as the standout mode.
# If the adm31 gives you trouble with standout mode, check the DIP switch in
# position 6, bank @c11, 25% from back end of the circuit board.  Should be
# OFF.  If there is no such switch, you have an old adm31 and must use oadm31.
# (adm31: removed obsolete ":ma=j^Jk^P^K^Pl ^R^L^L :" -- esr)
adm31|lsi adm31 with sw6 set for underline mode,
	OTbs, am, mir,
	cols#80, lines#24,
	bel=^G, clear=\E*, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\EY, el=\ET, home=^^, il1=\EE, ind=^J, is2=\Eu\E0,
	kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, kf0=^A0\r, kf1=^A1\r,
	kf2=^A2\r, kf3=^A3\r, kf4=^A4\r, kf5=^A5\r, kf6=^A6\r,
	kf7=^A7\r, kf8=^A8\r, kf9=^A9\r, rmir=\Er, rmso=\EG0,
	rmul=\EG0, sgr0=\EG0, smir=\Eq, smso=\EG1, smul=\EG1,
adm31-old|o31|old adm31,
	rmul@, smso=\EG4, smul@, use=adm31,
# LSI ADM-36 from Col. George L. Sicherman <gloria!colonel> via BRL
adm36|LSI ADM36,
	OTbs, OTpt,
	OTkn#4,
	if=/usr/share/tabset/vt100,
	is2=\E<\E>\E[6;?2;?7;?8h\E[4;20;?1;?3;?4;?5;?6;?18;?19l,
	use=vt100,
# (adm42: removed obsolete ":ma=^K^P:" -- esr)
adm42|lsi adm42,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, cbt=\EI, clear=\E;, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	cvvis=\EC\E3 \E3(, dch1=\EW, dl1=\ER, ed=\EY, el=\ET, ht=^I,
	il1=\EE$<270>, ind=^J, invis@, ip=$<6*>, kcub1=^H, kcud1=^J,
	kcuf1=^L, kcuu1=^K, khome=^^, pad=\177, rmir=\Er, rmul@,
	smir=\Eq, smul@, use=adm+sgr,
# The following termcap for the Lear Siegler ADM-42 leaves the
# "system line" at the bottom of the screen blank (for those who
# find it distracting otherwise)
adm42-ns|lsi adm-42 with no system line,
	cbt=\EI\EF \011, clear=\E;\EF \011,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c$<6>\EF \011,
	dch1=\EW\EF \011, dl1=\ER\EF \011, ed=\EY\EF \011,
	el=\ET\EF \011, il1=\EE\EF \011, rmir=\Er\EF \011,
	smir=\Eq\EF \011, use=adm42,
# ADM 1178 terminal -- rather like an ADM-42.  Manual is dated March 1 1985.
# The insert mode of this terminal is commented out because it's broken for our
# purposes in that it will shift the position of every character on the page,
# not just the cursor line!
# From: Michael Driscoll <fenris@lightspeed.net> 10 July 1996
adm1178|1178|lsi adm1178,
	am,
	cols#80, lines#24, xmc#1,
	bel=^G, bold=\E(, cbt=\EI, clear=\E+, cr=^M, cub1=^H, cud1=^J,
	cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	cvvis=\EC\E3 \E3(, dch1=\EW, dl1=\ER, ed=\EY, el=\ET,
	home=^^, ht=^I, il1=\EE, ind=^J, ip=$<6*/>, kbs=^H, kcub1=^H,
	kcud1=^J, nel=^M^J, pad=\177, rev=\EG4, rmso=\EG0, rmul=\EG0,
	sgr0=\E), smso=\EG4, smul=\EG1,

#### Prime
#
# Yes, Prime made terminals.  These entries were posted by Kevin J. Cummings
# <cummings@primerd.prime.com> on 14 Dec 1992 and lightly edited by esr.
# Prime merged with ComputerVision in the late 1980s; you can reach them at:
#
#	ComputerVision Services
#	500 Old Connecticut Path
#	Framingham, Mass.
#

# Standout mode is dim reverse-video.
pt100|pt200|wren|fenix|prime pt100/pt200,
	am, bw, mir, msgr,
	cols#80, it#8, lines#24,
	cbt=\E[Z, clear=\E?, cr=^M, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\ED, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E0%p1%{33}%+%c%p2%{33}%+%c, cuu=\E[%p1%dA,
	cuu1=\EM, dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m, dl1=\E[M,
	ed=\E[J\E[r, el=\E[K\E[t, flash=\E$$<200/>\E$P,
	home=\E$B, ht=^I, il1=\E[L\E[t, ind=^J, kbs=^H, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, khome=\E$A, nel=^M^J,
	rmcup=, rmir=\E[4l, rmkx=\E[>13l, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m,
	smcup=\E[>1l\E[>2l\E[>16l\E[4l\E[>9l\E[20l\E[>3l\E[>7h\E[>12l\E[1Q,
	smir=\E[4h, smkx=\E[>13h, smso=\E[2;7m, smul=\E[4m,
pt100w|pt200w|wrenw|fenixw|prime pt100/pt200 in 132-column mode,
	cols#132,
	cup=\E[%i%p1%d;%p2%dH, use=pt100,
pt250|Prime PT250,
	rmso@, smso@, use=pt100,
pt250w|Prime PT250 in 132-column mode,
	rmso@, smso@, use=pt100w,

#### Qume (qvt)
#
#	Qume, Inc.
#	3475-A North 1st Street
#	San Jose CA 95134
#	Vox: (800)-457-4447
#	Fax: (408)-473-1510
#	Net: josed@techsupp.wyse.com (Jose D'Oliveira)
#
# Qume was bought by Wyse, but still (as of early 1995) has its own support
# group and production division.
#
# Discontinued Qume models:
#
# The qvt101 and qvt102 listed here are long obsolete; so is the qvt101+
# built to replace them, and a qvt119+ which was a 101+ with available wide
# mode (132 columns).  There was a qvt103 which added vt100/vt131 emulations
# and an ANSI-compatible qvt203 that replaced it.  Qume started producing
# ANSI-compatible terminals with the qvt323 and qvt61.
#
# Current Qume models (as of February 1995):
#
# All current Qume terminals have ANSI-compatible operation modes.
# Qume is still producing the qvt62, which features emulations for other
# popular lines such as ADDS, and dual-host capabilities.  The qvt82 is
# designed for use as a SCO ANSI terminal.  The qvt70 is a color terminal
# with many emulations including Wyse370, Wyse 325, etc.  Their newest
# model is the qvt520, which is vt420-compatible.
#
# There are some ancient printing Qume terminals under `Daisy Wheel Printers'
#
# If you inherit a Qume without docs, try Ctrl-Shift-Setup to enter its
# setup mode.  Shift-s should be a configuration save to NVRAM.

qvt101|qvt108|qume qvt 101 and QVT 108,
	xmc#1, use=qvt101+,

# This used to have <cvvis=\E.2> but no <cnorm> or <civis>.  The BSD termcap
# file had <cvvis=\EM4 \200\200\200>.  I've done the safe thing and yanked
# both. The <rev> is from BSD, which also claimed bold=\E( and dim=\E).
# What seems to be going on here is that this entry was designed so that
# the normal highlight is bold and standout is dim plus something else
# (reverse-video maybe?  But then, are there two <rev> sequences?)
#
# Added kdch1, kil1, kdl1 based on screenshot -TD:
#	http://www.vintagecomputer.net/qume/qvt-108/qume_qvt-108_keyboard.jpg
qvt101+|qvt101p|qume qvt 101 PLUS product,
	am, bw, hs, ul,
	cols#80, lines#24, xmc#0,
	bel=^G, cbt=\EI, clear=^Z, cnorm=\E.4, cr=^M, cub1=^H, cud1=^J,
	cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	dch1=\EW, dl1=\ER, dsl=\Eg\Ef\r, ed=\EY, el=\ET,
	flash=\Eb$<200>\Ed, fsl=^M, home=^^, ht=^I, hts=\E1,
	ich1=\EQ, il1=\EE, ind=^J, invis@, kbs=^H, kcbt=\EI, kcub1=^H,
	kcud1=^J, kcuf1=^L, kcuu1=^K, kdch1=\EW, kdl1=\ER, ked=\EY,
	kel=\ET, kf1=^A@\r, kf10=^AI\r, kf2=^AA\r, kf3=^AB\r,
	kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r, kf8=^AG\r,
	kf9=^AH\r, khome=^^, kich1=\EQ, kil1=\EE, mc4=\EA, mc5=\E@,
	rmso=\E(, smso=\E0P\E), tbc=\E3, tsl=\Eg\Ef, use=adm+sgr,
qvt102|qume qvt 102,
	cnorm=\E., use=qvt101,
# (qvt103: added <rmam>/<smam> based on init string -- esr)
qvt103|qume qvt 103,
	am, xenl, xon,
	cols#80, it#8, lines#24, vt#3,
	bel=^G, blink=\E[5m$<2>, bold=\E[1m$<2>,
	clear=\E[H\E[2J$<50>, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C$<2>,
	cup=\E[%i%p1%d;%p2%dH$<5>, cuu=\E[%p1%dA,
	cuu1=\E[A$<2>, ed=\E[J$<50>, el=\E[K$<3>, home=\E[H, ht=^I,
	hts=\EH, ind=^J, kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC,
	kcuu1=\EOA, kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, rc=\E8,
	rev=\E[7m$<2>, ri=\EM$<5>, rmam=\E[?7l, rmkx=\E[?1l\E>,
	rmso=\E[m$<2>, rmul=\E[m$<2>,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr=\E[%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;m$<2>,
	sgr0=\E[m$<2>, smam=\E[?7h, smkx=\E[?1h\E=,
	smso=\E[7m$<2>, smul=\E[4m$<2>, tbc=\E[3g,
qvt103-w|qume qvt103 132 cols,
	cols#132, lines#24,
	rs2=\E>\E[?3h\E[?4l\E[?5l\E[?8h, use=qvt103,
qvt119+|qvt119p|qvt119|qume qvt 119 and 119PLUS terminals,
	am, hs, mir, msgr,
	cols#80, lines#24, xmc#0,
	bel=^G, cbt=\EI, clear=\E*1, cnorm=\E.4, cr=^M, cub1=^H,
	cud1=^J, cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=^K, cvvis=\E.2, dch1=\EW, dl1=\ER, dsl=\Eg\Ef\r, ed=\Ey,
	el=\Et, flash=\En0$<200>\En1, fsl=^M, home=^^, ht=^I,
	hts=\E1, il1=\EE, ind=^J, is2=\EDF\EC\EG0\Er\E(\E%EX,
	kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, kf0=^AI\r,
	kf1=^A@\r, kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r,
	kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^,
	mc4=\EA, mc5=\E@, ri=\EJ, rmir=\Er, smir=\Eq, smul=\EG8,
	tbc=\E3, tsl=\Eg\Ef, use=adm+sgr,
qvt119+-25|qvt119p-25|QVT 119 PLUS with 25 data lines,
	lines#25, use=qvt119+,
qvt119+-w|qvt119p-w|qvt119-w|QVT 119 and 119 PLUS in 132 column mode,
	cols#132,
	is2=\EDF\EC\EG0\Er\E(\E%\EX\En4, use=qvt119+,
qvt119+-25-w|qvt119p-25-w|qvt119-25-w|QVT 119 and 119 PLUS 132 by 25,
	lines#25, use=qvt119+,
qvt203|qvt203+|qume qvt 203 Plus,
	dch1=\E[P$<7>, dl1=\E[M$<99>, il1=\E[L$<99>, ind=\n$<30>,
	ip=$<7>, kf0=\E[29~, kf1=\E[17~, kf2=\E[18~, kf3=\E[19~,
	kf4=\E[20~, kf5=\E[21~, kf6=\E[23~, kf7=\E[24~, kf8=\E[25~,
	kf9=\E[28~, rmir=\E[4l, smir=\E[4h, use=qvt103,
qvt203-w|qvt203-w-am|qume qvt 203 PLUS in 132 cols (w/advanced video),
	cols#132, lines#24,
	rs2=\E>\E[?3h\E[?4l\E[?5l\E[?8h, use=qvt203,
#
#	Since a command is present for enabling 25 data lines,
#	a specific terminfo entry may be generated for the 203.
#	If one is desired for the QVT 119 PLUS then 25 lines must
#	be selected in the status line (setup line 9).
#
qvt203-25|QVT 203 PLUS with 25 by 80 column mode,
	cols#80, lines#25,
	is2=\E[=40h\E[?3l, use=qvt203,
qvt203-25-w|QVT 203 PLUS with 25 by 132 columns,
	cols#132, lines#25,
	rs2=\E[?3h\E[=40h, use=qvt203,

#### Televideo (tvi)
#
#	TeleVideo
#	550 East Brokaw Road
#	PO Box 49048    95161
#	San Jose CA 95112
#	Vox: (408)-954-8333
#	Fax: (408)-954-0623
#
#
# These require incredible amounts of padding.
#
# All of these terminals (912 to 970 and the tvipt) are discontinued.  Newer
# Televideo terminals are ANSI and PC-ANSI compatible.

tvi803|televideo 803,
	clear=\E*$<10>, use=tvi950,

# Vanilla tvi910 -- W. Gish <cswarren@violet> 10/29/86
# Switch settings are:
#
# S1  1 2 3 4
#     D D D D  9600
#     D D D U    50
#     D D U D    75
#     D D U U   110
#     D U D D   135
#     D U D U   150
#     D U U D   300
#     D U U U   600
#     U D D D  1200
#     U D D U  1800
#     U D U D  2400
#     U D U U  3600
#     U U D D  4800
#     U U D U  7200
#     U U U D  9600
#     U U U U 19200
#
# S1  5 6 7 8
#     U D X D  7N1 (data bits, parity, stop bits) (X means ignored)
#     U D X U  7N2
#     U U D D  7O1
#     U U D U  7O2
#     U U U D  7E1
#     U U U U  7E2
#     D D X D  8N1
#     D D X U  8N2
#     D U D D  8O1
#     D U U U  8E2
#
# S1  9  Autowrap
#     U  on
#     D  off
#
# S1 10  CR/LF
#     U  do CR/LF when CR received
#     D  do CR when CR received
#
# S2  1  Mode
#     U  block
#     D  conversational
#
# S2  2  Duplex
#     U  half
#     D  full
#
# S2  3  Hertz
#     U  50
#     D  60
#
# S2  4  Edit mode
#     U  local
#     D  duplex
#
# S2  5  Cursor type
#     U  underline
#     D  block
#
# S2  6  Cursor down key
#     U  send ^J
#     D  send ^V
#
# S2  7  Screen colour
#     U  green on black
#     D  black on green
#
# S2  8  DSR status (pin 6)
#     U  disconnected
#     D  connected
#
# S2  9  DCD status (pin 8)
#     U  disconnected
#     D  duplex
#
# S2 10  DTR status (pin 20)
#     U  disconnected
#     D  duplex
# (tvi910: removed obsolete ":ma=^Kk^Ll^R^L:"; added <khome>, <cub1>, <cud1>,
# <ind>, <hpa>, <vpa>, <am>, <msgr> from SCO entry -- esr)
tvi910|televideo model 910,
	OTbs, am, msgr,
	cols#80, it#8, lines#24, xmc#1,
	bel=^G, cbt=\EI, clear=^Z, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, ed=\EY, el=\ET,
	home=\E=\001\001, hpa=\E]%p1%{32}%+%c, ht=^I,
	if=/usr/share/tabset/stdcrt, ind=^J, invis@, kbs=^H,
	kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, kf0=^AI\r, kf1=^A@\r,
	kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r,
	kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^,
	vpa=\E[%p1%{32}%+%c, use=adm+sgr,
# From: Alan R. Rogers <rogers%albany@csnet-relay>
# as subsequently hacked over by someone at SCO
# (tvi910+: removed obsolete ":ma=^K^P^L :" -- esr)
#
# Here are the 910+'s DIP switches (U = up, D = down, X = don't care):
#
# S1  1 2 3 4:
#     D D D D  9600     D D D U    50     D D U D    75     D D U U   110
#     D U D D   135     D U D U   150     D U U D   300     D U U U   600
#     U D D D  1200     U D D U  1800     U D U D  2400     U D U U  3600
#     U U D D  4800     U U D U  7200     U U U D  9600     U U U U 19200
#
# S1  5 6 7 8:
#     U D X D  7N1     U D X U  7N2     U U D D  7O1     U U D U  7O2
#     U U U D  7E1     U U U U  7E2     D D X D  8N1     D D X U  8N2
#     D U D D  8O1     D U U U  8E2
#
# S1  9  Autowrap            (U = on, D = off)
# S1 10  CR/LF               (U = CR/LF on CR received, D = CR on CR received)
# S2  1  Mode                (U = block, D = conversational)
# S2  2  Duplex              (U =  half, D = full)
# S2  3  Hertz               (U = 50, D = 60)
# S2  4  Edit mode           (U = local, D = duplex)
# S2  5  Cursor type         (U = underline, D = block)
# S2  6  Cursor down key     (U = send ^J, D = send ^V)
# S2  7  Screen colour       (U = green on black, D = black on green)
# S2  8  DSR status (pin 6)  (U = disconnected, D = connected)
# S2  9  DCD status (pin 8)  (U = disconnected, D = connected)
# S2 10  DTR status (pin 20) (U = disconnected, D = connected)
#
tvi910+|televideo 910+,
	dch1=\EW, dl1=\ER$<33*>, home=^^, ich1=\EQ, il1=\EE$<33*>,
	kf0=^A@\r, kf1=^AA\r, kf2=^AB\r, kf3=^AC\r, kf4=^AD\r,
	kf5=^AE\r, kf6=^AF\r, kf7=^AG\r, kf8=^AH\r, kf9=^AI\r,
	ll=\E=7\s, use=tvi910,

# (tvi912: removed obsolete ":ma=^K^P^L :", added  <flash> and
# <khome> from BRL entry -- esr)
tvi912|tvi914|tvi920|old televideo 912/914/920,
	OTbs, OTpt, am, msgr,
	cols#80, it#8, lines#24, xmc#1,
	bel=^G, clear=^Z, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER$<33*>, ed=\Ey, el=\ET, flash=\Eb$<50/>\Ed, home=^^,
	ht=^I, hts=\E1, ich1=\EQ, if=/usr/share/tabset/stdcrt,
	il1=\EE$<33*>, ind=^J, kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^L,
	kcuu1=^K, kf0=^AI\r, kf1=^A@\r, kf2=^AA\r, kf3=^AB\r,
	kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r, kf8=^AG\r,
	kf9=^AH\r, khome=^^, rmso=\Ek, rmul=\Em, smso=\Ej, smul=\El,
	tbc=\E3,
# We got some new tvi912c terminals that act really weird on the regular
# termcap, so one of our gurus worked this up. Seems that cursor
# addressing is broken.
tvi912cc|tvi912 at cowell college,
	cup@, use=tvi912c,

# tvi{912,920}[bc] - TeleVideo TVI-912B/TVI-920B and TVI-912C/TVI-920C
# From: Benjamin C. W. Sittler
#
# Someone has put a scanned copy of the manual online at:
#     http://vt100.net/televideo/912b-om/
#
# These terminals were produced ca. 1979, and had a 12" monochrome
# screen, supported 75-9600 baud (no handshaking), monochrome, 7-bit
# ASCII, and were generally similar to adm3a but with attributes
# (including some with magic cookies), fancy half-duplex mode, and
# different bugs.
#
# Some operations reqire truly incredible amounts of padding. The
# insert_line (<il1>) and delete_line (<dl1>) operations in particular
# are so slow as to be nearly unusable.
#
# There may or may not have been a separate, earlier series of 912/920
# terminals (without the "B" and "C" suffix); I have never seen one,
# and the manual only describes the "B" and "C" series. The 912 and 920
# are quite distinct from the 914 and 924, which were much nicer non-
# magic-cookie terminals similar to the 950.
#
# This is a new description for the following TeleVideo terminals,
# distinguished chiefly by their keyboards:
#
#   TVI-912B - very odd layout, no function keys (84 keys)
#   TVI-920B - typewriter layout, no function keys (103 keys)
#   TVI-912C - very odd layout, function keys F1-F11 (82 keys)
#   TVI-920C - typewriter layout, function keys F1-F11 (101 keys)
#
# To choose a setting for the TERM variable, start with the model:
#
#    Model  || base name
# ----------||-----------
#  TVI-912B || tvi912b
#  TVI-912C || tvi912c
#  TVI-920B || tvi920b
#  TVI-920C || tvi920c
#
# Then add a suffix from the following table describing installed options
# and how you'd like to use the terminal:
#
#   Use Video | Second | Visual |  Magic  |  Page || feature
#  Attributes |  Page  |  Bell  | Cookies | Print || suffix
# ------------|--------|--------|---------|-------||---------
#      No     |   No   |  N/A   |   N/A   |   No  || -unk
#      No     |   No   |  N/A   |   N/A   |  Yes  || -p
#      No     |  Yes   |   No   |   N/A   |   No  || -2p-unk
#      No     |  Yes   |   No   |   N/A   |  Yes  || -2p-p
#      No     |  Yes   |  Yes   |   N/A   |   No  || -vb-unk
#      No     |  Yes   |  Yes   |   N/A   |  Yes  || -vb-p
#     Yes     |   No   |  N/A   |    No   |  N/A  ||
#     Yes     |   No   |  N/A   |   Yes   |  N/A  || -mc
#     Yes     |  Yes   |   No   |    No   |  N/A  || -2p
#     Yes     |  Yes   |   No   |   Yes   |  N/A  || -2p-mc
#     Yes     |  Yes   |  Yes   |    No   |  N/A  || -vb
#     Yes     |  Yes   |  Yes   |   Yes   |  N/A  || -vb-mc
#
# So e.g. a model 920 C with second page memory option, visual bell
# and no magic cookies would be tvi920c-vb; a model 912 B without the
# second page memory option and using magic cookies would be
# tvi912b-mc
#
# PADDING
#
# At 9600 baud, the terminal is prone to overflow its input buffer
# during complex operations (insert/delete
# character/line/screen/page), and it does not signal this over the
# RS232 cable. The typical symptom of an overrun is that the terminal
# starts beeping, and output becomes garbled.
#
# The padding delays in this terminfo were derived using tack(1)
# running on a Linux box connected to a TVI-920C with a later-model
# (A49C1-style) ROM running at 9600 baud, so your mileage may
# vary. The numbers below seem to give the terminal enough time so
# that it doesn't overflow its input buffer and start losing
# characters.
#
# KEYS
#
# If you want to use the FUNCT key on a tvi912[bc], use the
# corresponding tvi920[bc] terminfo with FUNCT + ... equivalents from
# the following table (these also work on the 920 series):
#
# Unshifted Function Keys:
#
#  Key | capname|| Equivalent
# -----|--------||------------
#   F1 |  <kf1> || FUNCT + @
#   F2 |  <kf2> || FUNCT + A
#   F3 |  <kf3> || FUNCT + B
#   F4 |  <kf4> || FUNCT + C
#   F5 |  <kf5> || FUNCT + D
#   F6 |  <kf6> || FUNCT + E
#   F7 |  <kf7> || FUNCT + F
#   F8 |  <kf8> || FUNCT + G
#   F9 |  <kf9> || FUNCT + H
#  F10 | <kf10> || FUNCT + I
#  F11 | <kf11> || FUNCT + J
#
# Shifted Function Keys:
#
#  SHIFT + Key | capname|| Equivalent
# -------------|--------||------------
#   SHIFT + F1 | <kf12> || FUNCT + `
#   SHIFT + F2 | <kf13> || FUNCT + a
#   SHIFT + F3 | <kf14> || FUNCT + b
#   SHIFT + F4 | <kf15> || FUNCT + c
#   SHIFT + F5 | <kf16> || FUNCT + d
#   SHIFT + F6 | <kf17> || FUNCT + e
#   SHIFT + F7 | <kf18> || FUNCT + f
#   SHIFT + F8 | <kf19> || FUNCT + g
#   SHIFT + F9 | <kf20> || FUNCT + h
#  SHIFT + F10 | <kf21> || FUNCT + i
#  SHIFT + F11 | <kf22> || FUNCT + j
#
# PORTS AND SWITCH SETTINGS
#
# Here are the switch settings for the TVI-912B/TVI-920B and
# TVI-912C/TVI-920C:
#
# S1 (Line), and S3 (Printer) baud rates -- put one, and only one, switch down:
# 2: 9600	3: 4800		4: 2400		5: 1200
# 6:  600	7:  300		8:  150		9:   75
# 10: 110
#
# S2 UART/Terminal options:
# 		Up			Down
# 1:		Not used		Not allowed
# 2:	Alternate character set	  Standard character set
# 3:	    Full duplex		    Half duplex
# 4:	    50 Hz refresh	    60 Hz refresh
# 5:	      No parity		     Send parity
# 6:	     2 stop bits	     1 stop bit
# 7:	     8 data bits	     7 data bits
# 8:		Not used		Not allowed on Rev E or lower
# 9:	     Even parity	     Odd parity
# 10:	    Steady cursor	    Blinking cursor
# 	(On Rev E or lower, use W25 instead of switch 10.)
#
# S5 UART/Terminal options:
# 		Open			Closed
# 1:	P3-6 Not connected	DSR received on P3-6
# 2:	P3-8 Not connected	DCD received on P3-8
#
# 3 Open, 4 Open:		P3-20 Not connected
# 3 Open, 4 Closed:	DTR on when terminal is on
# 3 Closed, 4 Open:	DTR is connected to RTS
# 3 Closed, 4 Closed:	Not allowed
#
# 5 Closed:	HDX printer (hardware control) Rev. K with extension port off,
# 		all data transmitted out of the modem port (P3) will also be
# 		transmitted out of the printer port (P4).
#
# 6 Open, 7 Open:		Not allowed
# 6 Open, 7 Closed:	20ma current loop input
# 6 Closed, 7 Open:	RS232 input
# 6 Closed, 7 Closed:	Not allowed
#
# Jumper options:
# If the jumper is installed, the effect will occur (the next time the terminal
# is switched on).
#
# S4/W31:	Enables automatic LF upon receipt of CR from
# 		remote or keyboard.
# S4/W32:	Enables transmission of EOT at the end of Send.  If not
# 		installed, a carriage return is sent.
# S4/W33:	Disables automatic carriage return in column 80.
# S4/W34:	Selects Page Print Mode as initial condition.  If not
# 		installed, Extension Mode is selected.
#
# NON-STANDARD CAPABILITIES
#
# Sending <u9> or <u7> returns a cursor position report in the format
# YX\r, where Y and X are as in <cup>. This format is described in
# <u8> and <u6>, but it's not clear how one should write an
# appropriate scanf string, since we need to subtract %' ' from the
# character after reading it. The <u9> capability is used by tack(1)
# to synchronize during padding tests, and seems to work for that
# purpose.
#
# This description also includes the obsolete termcap capabilities
# has_hardware_tabs (<OTpt>) and backspaces_with_bs (<OTbs>).
#
# FEATURES NOT YET DESCRIBED IN THIS TERMINFO
#
# The FUNCT modifier actually works with every normal key by sending
# ^AX\r, where X is the sequence normally sent by that key. This is a
# sort of meta key not currently describable in terminfo.
#
# There are quite a few other keys (especially on the 920 models,) but
# they are for the most part only useful in block mode.
#
# These terminals have lots of forms manipulation features, mainly
# useful in block mode, including "clear X to nulls" (vs. "clear X to
# spaces"; nulls are sentinels for "send X" operations); "send X"
# operations for uploading all or part of the screen; and block-mode
# editing keys (they don't send escape sequences, but manipulate video
# memory directly). Block mode is used for local editing, and protect
# mode (in conjunction with the "write protect" attribute,
# a.k.a. half-intensity outside of protect mode) is used to control
# which parts of the screen are edited/sent/printed (by <mc0>).
#
# There are at least two major families of ROM, "early" and
# A49B1/A49C1; the major difference seems to be that the latter ROMs
# support a few extra escape sequences for manipulating the off-screen
# memory page, and for sending whole pages back to the host (mainly
# useful in block mode.) The descriptions in this file don't use any
# of those sequences: set cursor position including page (\E-PYX,
# where P is \s for page 0 and ! for page 1 [actually only the LSB of
# P is taken into account, so e.g. 0 and 1 work too,] and Y and X are
# as in <cup>); read cursor position (\E/), which is analogous to <u9>
# and returns PYX\r, where P is \s for page 0 or ! for page 1, and YX
# are as in <cup>, and some "send page" features mainly useful for
# forms manipulation.
#
# The keyboard enable (\E") and disable (\E#) sequences are unused,
# except that a terminal reset (<is2>) enables the keyboard.
#
# Auto-flip mode (\Ev) is likely faster than the scrolling mode (\Ew)
# enabled in <is2>, but auto-flip is very jarring so we don't use it.
#
# BUGS
#
# At least up to the A49B1 and A49C1 ROMs, there are no \Eb and \Ed
# sequences (I infer that in some TeleVideo terminal they may invert
# and uninvert the display) so the <flash> sequence given here is a
# cheesy page-flip instead.
#
# The back_tab (<cbt>) sequence (\EI) doesn't work according to
# tack(1), so it is not included in the descriptions below.
#
# It's not clear whether auto_left_margin (<bw>) flag should be set
# for these terminals; tack says yes, so it is set here, but this
# differs from other descriptions I've seen.
#
# Extension print mode (<mc5>) echoes all characters to the printer
# port [in addition to displaying them] except for the page print mode
# sequence (<mc4>); this is a slight violation of the terminfo
# definition for <mc5> but I don't expect it to cause problems.  We
# reset to page print mode in <rs1> since it may have been enabled
# accidentally.
#
# The descriptions with plus signs (+) are building blocks.

tvi912b-unk|tvi912c-unk|TeleVideo TVI-912B or TVI-912C (no attributes),
	OTbs, OTpt, am, bw,
	cols#80, it#8, lines#24,
	bel=^G, clear=\032$<50>, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%' '%+%c%p2%' '%+%c, cuu1=^K, dch1=\EW$<30>,
	dl1=\ER$<1*>$<100>, ed=\Ey$<2*>$<10>, el=\ET$<15>,
	home=^^, ht=^I, hts=\E1, ich1=\EQ$<30>,
	if=/usr/share/tabset/stdcrt, il1=\EE$<1*>$<100>,
	ind=\n$<10>, is2=\Ew\EA\E'\E"\E(, kcub1=^H, kcud1=^J,
	kcuf1=^L, kcuu1=^K, kdch1=\177, kent=^M, khome=^^, mc4=\EA,
	mc5=\E@, rs1=\Ek\010\Em\010\Eq\032, tbc=\E3, u6=%c%c\r,
	u7=\E?, u8=%c%c\r, u9=\E?,

# This isn't included in the basic capabilities because it is
# typically unusable in combination with the full range of video
# attributes, since the magic cookie attributes turn into ASCII
# control characters, and the half-intensity ("protected") attribute
# converts all affected characters to spaces.

tvi912b+printer|TeleVideo TVI-912B/TVI-920B and TVI-912C/TVI-920C page print support,
	mc0=\EP,

# This uses half-intensity mode (<dim>) for standout (<smso>), and
# exposes no other attributes (half-intensity is the only attribute
# that does not generate a magic cookie.)

tvi912b+dim|TeleVideo TVI-912B/TVI-920B and TVI-912C/TVI-920C half-intensity attribute support,
	msgr,
	dim=\E), rmso=\E(, sgr=\E%?%p1%p5%|%t)%e(%;, sgr0=\E(,
	smso=\E),

# Full magic-cookie attribute support, with half-intensity reverse
# video for standout. Note that we add a space in the <dim> sequence
# to give a consistent magic-cookie count. Also note that <sgr> uses
# backspacing (in the TVI-supported order) to apply all requested
# attributes with only a single magic cookie.

tvi912b+mc|TeleVideo TVI-912B/TVI-920B and TVI-912C/TVI-920C full magic-cookie attribute support,
	xmc#1,
	blink=\E\^, dim=\E)\s, invis=\E_, rev=\Ej, rmso=\E(\Ek,
	rmul=\Em,
	sgr=\E%?%p1%p5%|%t)%e(%; \010\E%?%p1%p3%|%tj%ek%;\010\E%?%p2%tl%em%;\010\E%?%p7%t_%e%?%p4%t\^%eq%;%;,
	sgr0=\E(\Ek\010\Em\010\Eq, smso=\E)\Ej, smul=\El,

# This uses the second page memory option to save & restore screen
# contents. If your terminal is missing the option, this description
# should still work, but that has not been tested.

tvi912b+2p|TeleVideo TVI-912B/TVI-920B and TVI-912C/TVI-920C second page memory option support,
	flash=\EK$<100>\EK, rmcup=\032$<50>\EK\E=7\s,
	smcup=\EK\032$<50>\E(\Ek\010\Em\010\Eq\032$<50>,

# This simulates flashing by briefly toggling to the other page
# (kludge!)

tvi912b+vb|TeleVideo TVI-912B/TVI-920B and TVI-912C/TVI-920C second page memory option "visible bell" support,
	bel=\EK$<100>\EK, use=tvi912b+2p,

# Function keys (<kf12> .. <kf22> are shifted <kf1> .. <kf11>)

tvi920b+fn|TeleVideo TVI-920B and TVI-920C function key support,
	kf1=^A@\r, kf10=^AI\r, kf11=^AJ\r, kf12=^A`\r, kf13=^Aa\r,
	kf14=^Ab\r, kf15=^Ac\r, kf16=^Ad\r, kf17=^Ae\r, kf18=^Af\r,
	kf19=^Ag\r, kf2=^AA\r, kf20=^Ah\r, kf21=^Ai\r, kf22=^Aj\r,
	kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r,
	kf8=^AG\r, kf9=^AH\r,

# Combinations of the basic building blocks

tvi912b-2p-unk|tvi912c-2p-unk|tvi912b-unk-2p|tvi912c-unk-2p|TeleVideo TVI-912B or TVI-912C (second page memory option; no attributes),
	use=tvi912b+2p, use=tvi912b-unk,

tvi912b-vb-unk|tvi912c-vb-unk|tvi912b-unk-vb|tvi912c-unk-vb|TeleVideo TVI-912B or TVI-912C (second page memory option "visible bell"; no attributes),
	use=tvi912b+vb, use=tvi912b-unk,

tvi912b-p|tvi912c-p|TeleVideo TVI-912B or TVI-912C (no attributes; page print),
	use=tvi912b+printer, use=tvi912b-unk,

tvi912b-2p-p|tvi912c-2p-p|tvi912b-p-2p|tvi912c-p-2p|TeleVideo TVI-912B or TVI-912C (second page memory option; no attributes; page print),
	use=tvi912b+2p, use=tvi912b+printer, use=tvi912b-unk,

tvi912b-vb-p|tvi912c-vb-p|tvi912b-p-vb|tvi912c-p-vb|TeleVideo TVI-912B or TVI-912C (second page memory option "visible bell"; no attributes; page print),
	use=tvi912b+vb, use=tvi912b+printer, use=tvi912b-unk,

tvi912b-2p|tvi912c-2p|TeleVideo TVI-912B or TVI-912C (second page memory option; half-intensity attribute),
	use=tvi912b+2p, use=tvi912b+dim, use=tvi912b-unk,

tvi912b-2p-mc|tvi912c-2p-mc|tvi912b-mc-2p|tvi912c-mc-2p|TeleVideo TVI-912B or TVI-912C (second page memory option; magic cookies),
	use=tvi912b+2p, use=tvi912b+mc, use=tvi912b-unk,

tvi912b-vb|tvi912c-vb|TeleVideo TVI-912B or TVI-912C (second page memory option "visible bell"; half-intensity attribute),
	use=tvi912b+vb, use=tvi912b+dim, use=tvi912b-unk,

tvi912b-vb-mc|tvi912c-vb-mc|tvi912b-mc-vb|tvi912c-mc-vb|TeleVideo TVI-912B or TVI-912C (second page memory option "visible bell"; magic cookies),
	use=tvi912b+vb, use=tvi912b+mc, use=tvi912b-unk,

tvi912b|tvi912c|TeleVideo TVI-912B or TVI-912C (half-intensity attribute),
	use=tvi912b+dim, use=tvi912b-unk,

tvi912b-mc|tvi912c-mc|TeleVideo TVI-912B or TVI-912C (magic cookies),
	use=tvi912b+mc, use=tvi912b-unk,

tvi920b-unk|tvi920c-unk|TeleVideo TVI-920B or TVI-920C (no attributes),
	use=tvi920b+fn, use=tvi912b-unk,

tvi920b-2p-unk|tvi920c-2p-unk|tvi920b-unk-2p|tvi920c-unk-2p|TeleVideo TVI-920B or TVI-920C (second page memory option; no attributes),
	use=tvi920b+fn, use=tvi912b+2p, use=tvi912b-unk,

tvi920b-vb-unk|tvi920c-vb-unk|tvi920b-unk-vb|tvi920c-unk-vb|TeleVideo TVI-920B or TVI-920C (second page memory option "visible bell"; no attributes),
	use=tvi920b+fn, use=tvi912b+vb, use=tvi912b-unk,

tvi920b-p|tvi920c-p|TeleVideo TVI-920B or TVI-920C (no attributes; page print),
	use=tvi920b+fn, use=tvi912b+printer, use=tvi912b-unk,

tvi920b-2p-p|tvi920c-2p-p|tvi920b-p-2p|tvi920c-p-2p|TeleVideo TVI-920B or TVI-920C (second page memory option; no attributes; page print),
	use=tvi920b+fn, use=tvi912b+2p, use=tvi912b+printer,
	use=tvi912b-unk,

tvi920b-vb-p|tvi920c-vb-p|tvi920b-p-vb|tvi920c-p-vb|TeleVideo TVI-920B or TVI-920C (second page memory option "visible bell"; no attributes; page print),
	use=tvi920b+fn, use=tvi912b+vb, use=tvi912b+printer,
	use=tvi912b-unk,

tvi920b-2p|tvi920c-2p|TeleVideo TVI-920B or TVI-920C (second page memory option; half-intensity attribute),
	use=tvi920b+fn, use=tvi912b+2p, use=tvi912b+dim,
	use=tvi912b-unk,

tvi920b-2p-mc|tvi920c-2p-mc|tvi920b-mc-2p|tvi920c-mc-2p|TeleVideo TVI-920B or TVI-920C (second page memory option; magic cookies),
	use=tvi920b+fn, use=tvi912b+2p, use=tvi912b+mc,
	use=tvi912b-unk,

tvi920b-vb|tvi920c-vb|TeleVideo TVI-920B or TVI-920C (second page memory option "visible bell"; half-intensity attribute),
	use=tvi920b+fn, use=tvi912b+vb, use=tvi912b+dim,
	use=tvi912b-unk,

tvi920b-vb-mc|tvi920c-vb-mc|tvi920b-mc-vb|tvi920c-mc-vb|TeleVideo TVI-920B or TVI-920C (second page memory option "visible bell"; magic cookies),
	use=tvi920b+fn, use=tvi912b+vb, use=tvi912b+mc,
	use=tvi912b-unk,

tvi920b|tvi920c|TeleVideo TVI-920B or TVI-920C (half-intensity attribute),
	use=tvi920b+fn, use=tvi912b+dim, use=tvi912b-unk,

tvi920b-mc|tvi920c-mc|TeleVideo TVI-920B or TVI-920C (magic cookies),
	use=tvi920b+fn, use=tvi912b+mc, use=tvi912b-unk,

# Televideo 921 and variants
# From: Tim Theisen <tim@cs.wisc.edu> 22 Sept 1995
# (tvi921: removed :ko=bt: before translation, I see no backtab cap;
# also added empty <acsc> to suppress tic warning -- esr)
tvi921|televideo model 921 with sysline same as page & real vi function,
	OTbs, OTpt, am, hs, xenl, xhp,
	cols#80, lines#24, xmc#0,
	acsc=, clear=^Z, cnorm=\E.3, cr=^M, cub1=^H, cud1=^V, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c$<3/>, cuu1=^K,
	cvvis=\E.2, dch1=\EW, dl1=\ER$<1*/>, dsl=\Ef\r\Eg, ed=\EY,
	el=\ET, fsl=\Eg, home=^^, ht=^I, ich1=\EQ,
	if=/usr/share/tabset/stdcrt, il1=\EE, ind=^J, invis@,
	is2=\El\E"\EF1\E.3\017\EA\E<, kbs=^H, kclr=^Z, kcub1=^H,
	kcud1=^V, kcuf1=^L, kcuu1=^K, kdch1=\EW, kdl1=\ER$<1*/>,
	ked=\EY, kel=\ET, kich1=\EQ, kil1=\EE, nel=^M^J, rmacs=\E%%,
	rmir=, smacs=\E$, smir=, tsl=\Ef\EG0, use=adm+sgr,
# without the beeper
# (tvi92B: removed :ko=bt: before translation, I see no backtab cap;
# also added empty <acsc> to suppress tic warning -- esr)
tvi92B|televideo model 921 with sysline same as page & real vi function & no beeper,
	am, hs, xenl, xhp,
	cols#80, lines#24, xmc#0,
	acsc=, clear=^Z, cnorm=\E.3, cr=^M, cub1=^H, cud1=^V, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c$<3/>, cuu1=^K,
	cvvis=\E.2, dch1=\EW, dl1=\ER$<1*/>, dsl=\Ef\r\Eg, ed=\EY,
	el=\ET, flash=\Eb$<200/>\Ed, fsl=\Eg, home=^^, ht=^I,
	ich1=\EQ, if=/usr/share/tabset/stdcrt, il1=\EE, ind=^J,
	invis@, is2=\El\E"\EF1\E.3\017\EA\E<, kbs=^H, kclr=^Z,
	kcub1=^H, kcud1=^V, kcuf1=^L, kcuu1=^K, kdch1=\EW,
	kdl1=\ER$<1*/>, ked=\EY, kel=\ET, kich1=\EQ, kil1=\EE,
	nel=^M^J, rmacs=\E%%, smacs=\E$, tsl=\Ef\EG0, use=adm+sgr,
# (tvi92D: removed :ko=bt: before translation, I see no backtab cap -- esr)
tvi92D|tvi92B with DTR instead of XON/XOFF & better padding,
	dl1=\ER$<2*/>, il1=\EE$<2*/>,
	is2=\El\E"\EF1\E.3\016\EA\E<, kdl1=\ER$<2*/>,
	kil1=\EE$<2*/>, use=tvi92B,

# (tvi924: This used to have <dsl=\Es0>, <fsl=\031>.  I put the new strings
# in from a BSD termcap file because it looks like they do something the
# old ones skip -- esr)
tvi924|televideo tvi924,
	am, bw, hs, in, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, wsl#80, xmc#0,
	bel=^G, blink=\EG2, cbt=\EI, civis=\E.0, clear=\E*0,
	cnorm=\E.3, cr=^M, csr=\E_%p1%{32}%+%c%p2%{32}%+%c,
	cub1=^H, cud1=^V, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, cvvis=\E.1,
	dch1=\EW, dl1=\ER, dsl=\Es0\Ef\031, ed=\Ey, el=\Et,
	flash=\Eb$<200>\Ed, fsl=\031\Es1, home=^^, ht=^I, hts=\E1,
	ich1=\EQ, if=/usr/share/tabset/stdcrt, il1=\EE, ind=^J,
	invis@, is1=\017\E%\E'\E(\EDF\EC\EG0\EN0\Es0\Ev0,
	kbs=^H, kclr=\E*0, kcub1=^H, kcud1=^V, kcuf1=^L, kcuu1=^K,
	kdch1=\EW, kdl1=\ER, ked=\Ey, kel=\Et, kf0=^A@\r, kf1=^AA\r,
	kf10=^AJ\r, kf11=^AK\r, kf12=^AL\r, kf13=^AM\r, kf14=^AN\r,
	kf15=^AO\r, kf2=^AB\r, kf3=^AC\r, kf4=^AD\r, kf5=^AE\r,
	kf6=^AF\r, kf7=^AG\r, kf8=^AH\r, kf9=^AI\r, khome=^^,
	kich1=\EQ, kil1=\EE, lf0=F1, lf1=F2, lf10=F11, lf2=F3, lf3=F4,
	lf4=F5, lf5=F6, lf6=F7, lf7=F8, lf8=F9, lf9=F10,
	pfkey=\E|%p1%{49}%+%c%p2%s\031, ri=\Ej, tbc=\E3, tsl=\Ef,
	use=adm+sgr,

# TVI925 DIP switches.  In each of these, D = Down and U = Up,
#
# Here are the settings for the external (baud) switches (S1):
#
#             Position		Baud
# 7	8	9	10		[Printer]
# 1	2	3	4		[Main RS232]
# -----------------------------------------------------
# D	D	D	D	9600
# D	D	D	U	  50
# D	D	U	D	  75
# D	D	U	U	 110
# D	U	D	D	 135
# D	U	D	U	 150
# D	U	U	D	 300
# D	U	U	U	 600
# U	D	D	D	1200
# U	D	D	U	1800
# U	D	U	D	2400
# U	D	U	U	3600
# U	U	D	D	4800
# U	U	D	U	7200
# U	U	U	D	9600
# U	U	U	U	19200
#
#
# Settings for word length and stop-bits (S1)
#
#  Position	Description
# 5	6
# ---------------------------
# U	-	7-bit word
# D	-	8-bit word
# -	U	2 stop bits
# -	D	1 stop bit
#
#
# S2 (external) settings
#
# Position	Up	Dn	Description
# --------------------------------------------
# 1		X		Local edit
# 			X	Duplex edit (transmit editing keys)
# --------------------------------------------
# 2		X		912/920 emulation
# 			X	925
# --------------------------------------------
# 3			X
# 4			X	No parity
# 5			X
# --------------------------------------------
# 3			X
# 4			X	Odd parity
# 5		X
# --------------------------------------------
# 3			X
# 4		X		Even parity
# 5		X
# --------------------------------------------
# 3		X
# 4			X	Mark parity
# 5		X
# --------------------------------------------
# 3		X
# 4		X		Space parity
# 5		X
# --------------------------------------------
# 6		X		White on black display
# 			X	Black on white display
# --------------------------------------------
# 7			X	Half Duplex
# 8			X
# --------------------------------------------
# 7		X		Full Duplex
# 8			X
# --------------------------------------------
# 7			X	Block mode
# 8		X
# --------------------------------------------
# 9			X	50 Hz
# 		X		60 Hz
# --------------------------------------------
# 10		X		CR/LF (Auto LF)
# 			X	CR only
#
# S3 (internal switch) settings:
#
# Position	Up	Dn	Description
# --------------------------------------------
# 1		X		Keyclick off
# 			X	Keyclick on
# --------------------------------------------
# 2			X	English
# 3			X
# --------------------------------------------
# 2			X	German
# 3		X
# --------------------------------------------
# 2		X		French
# 3			X
# --------------------------------------------
# 2		X		Spanish
# 3		X
# --------------------------------------------
# 4			X	Blinking block cursor
# 5			X
# --------------------------------------------
# 4			X	Blinking underline cursor
# 5		X
# --------------------------------------------
# 4		X		Steady block cursor
# 5			X
# --------------------------------------------
# 4		X		Steady underline cursor
# 5		X
# --------------------------------------------
# 6		X		Screen blanking timer (ON)
# 			X	Screen blanking timer (OFF)
# --------------------------------------------
# 7		X		Page attributes
# 			X	Line attributes
# --------------------------------------------
# 8		X		DCD disconnected
# 			X	DCD connected
# --------------------------------------------
# 9		X		DSR disconnected
# 			X	DSR connected
# --------------------------------------------
# 10		X		DTR Disconnected
# 			X	DTR connected
# --------------------------------------------
#
# (tvi925: BSD has <clear=\E*>.  I got <is2> and <ri> from there -- esr)
tvi925|televideo 925,
	OTbs, am, bw, hs, ul,
	cols#80, lines#24, xmc#1,
	bel=^G, cbt=\EI, clear=^Z, cnorm=\E.4, cr=^M, cub1=^H, cud1=^V,
	cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	cvvis=\E.2, dch1=\EW, dl1=\ER, dsl=\Eh, ed=\EY, el=\ET,
	flash=\Eb$<200>\Ed, fsl=^M\Eg, home=^^, ht=^I, hts=\E1,
	ich1=\EQ, il1=\EE, ind=^J, invis@, is2=\El\E", kbs=^H, kclr=^Z,
	kcub1=^H, kcud1=^V, kcuf1=^L, kcuu1=^K, kdch1=\EW, kdl1=\ER,
	ked=\EY, kel=\ET, kf0=^AI\r, kf1=^A@\r, kf2=^AA\r, kf3=^AB\r,
	kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r, kf8=^AG\r,
	kf9=^AH\r, khome=^^, kich1=\EQ, kil1=\EE, ri=\Ej, tbc=\E3,
	tsl=\Eh\Ef, use=adm+sgr,
# TeleVideo 925 from Mitch Bradley <sun!wmb> via BRL
# to avoid "magic cookie" standout glitch:
tvi925-hi|TeleVideo Model 925 with half intensity standout mode,
	xmc@,
	kbs=^H, kcub1=^H, kcud1=^J, rmso=\E(, smso=\E), use=tvi925,

# From: Todd Litwin <litwin@litwin.jpl.nasa.gov> 28 May 1993
# Originally Tim Curry, Univ. of Central Fla., <duke!ucf-cs!tim> 5/21/82
# for additional capabilities,
# The following tvi descriptions from B:pjphar and virus!mike
# is for all 950s.  It sets the following attributes:
# full duplex (\EDF)		write protect off (\E()
# conversation mode (\EC)	graphics mode off (\E%)
# white on black (\Ed)		auto page flip off (\Ew)
# turn off status line (\Eg)	clear status line (\Ef\r)
# normal video (\E0)		monitor mode off (\EX or \Eu)
# edit mode (\Er)		load blank char to space (\Ee\040)
# line edit mode (\EO)		enable buffer control (^O)
# protect mode off (\E\047)	duplex edit keys (\El)
# program unshifted send key to send line all (\E016)
# program shifted send key to send line unprotected (\E004)
# set the following to nulls:
#	field delimiter (\Ex0\200\200)
#	line delimiter (\Ex1\200\200)
#	start-protected field delimiter (\Ex2\200\200)
#	end-protected field delimiter (\Ex3\200\200)
# set end of text delimiter to carriage return/null (\Ex4\r\200)
#
#                     TVI 950 Switch Setting Reference Charts
#
#                                     TABLE 1:
#
#      S1     1     2     3     4     5     6     7     8     9    10
#          +-----------------------+-----+-----+-----------------------+
#          | Computer Baud Rate    |Data |Stop | Printer Baud Rate     |
#          |                       |Bits |Bits |                       |
#   +------+-----------------------+-----+-----+-----------------------+
#   |  Up  |        See            |  7  |  2  |        See            |
#   +------+-----------------------+-----+-----+-----------------------+
#   | Down |      TABLE 2          |  8  |  1  |      TABLE 2          |
#   +------+-----------------------+-----+-----+-----------------------+
#
#
#      S2     1     2     3     4     5     6     7     8     9    10
#          +-----+-----+-----------------+-----+-----------+-----+-----+
#          |Edit |Cursr|    Parity       |Video|Transmiss'n| Hz  |Click|
#   +------+-----+-----+-----------------+-----+-----------+-----+-----+
#   |  Up  | Dplx|Blink|      See        |GonBk|   See     | 60  | Off |
#   +------+-----+-----+-----------------+-----+-----------+-----+-----+
#   | Down |Local|St'dy|    TABLE 3      |BkonG|  CHART    | 50  | On  |
#   +------+-----+-----+-----------------+-----+-----------+-----+-----+
#
#                                    TABLE 2:
#
#             +-----------+-----+-----+-----+-----+-----------+
#             | Display   |  1  |  2  |  3  |  4  |   Baud    |
#             +-----------+-----+-----+-----+-----+           |
#             | Printer   |  7  |  8  |  9  | 10  |   Rate    |
#             +-----------+-----+-----+-----+-----+-----------+
#                         |  D  |  D  |  D  |  D  |   9600    |
#                         |  U  |  D  |  D  |  D  |     50    |
#                         |  D  |  U  |  D  |  D  |     75    |
#                         |  U  |  U  |  D  |  D  |    110    |
#                         |  D  |  D  |  U  |  D  |    135    |
#                         |  U  |  D  |  U  |  D  |    150    |
#                         |  D  |  U  |  U  |  D  |    300    |
#                         |  U  |  U  |  U  |  D  |    600    |
#                         |  D  |  D  |  D  |  U  |   1200    |
#                         |  U  |  D  |  D  |  U  |   1800    |
#                         |  D  |  U  |  D  |  U  |   2400    |
#                         |  U  |  U  |  D  |  U  |   3600    |
#                         |  D  |  D  |  U  |  U  |   4800    |
#                         |  U  |  D  |  U  |  U  |   7200    |
#                         |  D  |  U  |  U  |  U  |   9600    |
#                         |  U  |  U  |  U  |  U  |  19200    |
#                         +-----+-----+-----+-----+-----------+
#
#                                    TABLE 3:
#                         +-----+-----+-----+-----------+
#                         |  3  |  4  |  5  |   Parity  |
#                         +-----+-----+-----+-----------+
#                         |  X  |  X  |  D  |    None   |
#                         |  D  |  D  |  U  |     Odd   |
#                         |  D  |  U  |  U  |    Even   |
#                         |  U  |  D  |  U  |    Mark   |
#                         |  U  |  U  |  U  |   Space   |
#                         +-----+-----+-----+-----------+
#                                 X = don't care
#
#                                     CHART:
#                         +-----+-----+-----------------+
#                         |  7  |  8  | Communication   |
#                         +-----+-----+-----------------+
#                         |  D  |  D  |  Half Duplex    |
#                         |  D  |  U  |  Full Duplex    |
#                         |  U  |  D  |     Block       |
#                         |  U  |  U  |     Local       |
#                         +-----+-----+-----------------+
#
# (tvi950: early versions had obsolete ":ma=^Vj^Kk^Hh^Ll^^H:".
# I also inserted <ich1> and <kich1>; the :ko: string indicated that <ich>
# should be present and all tvi native modes use the same string for this.
# Finally, note that BSD has cud1=^V. -- esr)
tvi950|televideo 950,
	OTbs, am, hs, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, xmc#1,
	acsc=b\011c\014d\re\ni\013, bel=^G, cbt=\EI, clear=\E*,
	cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, dsl=\Eg\Ef\r, ed=\Ey, el=\Et, flash=\Eb$<200/>\Ed,
	fsl=^M, home=^^, ht=^I, hts=\E1, ich1=\EQ, il1=\EE, ind=^J,
	invis@,
	is2=\EDF\EC\Ed\EG0\Eg\Er\EO\E'\E(\E%\Ew\EX\Ee \017\011\El\E016\E004\Ex0\0\0\Ex1\0\0\Ex2\0\0\011\Ex3\0\0\Ex4\r\0\Ef\r,
	kbs=^H, kcbt=\EI, kclr=\E*, kcub1=^H, kcud1=^V, kcuf1=^L,
	kcuu1=^K, kdch1=\EW, kdl1=\ER, ked=\Ey, kel=\Et, kf0=^A0\r,
	kf1=^A@\r, kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r,
	kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^,
	kich1=\EQ, kil1=\EE, mc4=\Ea, mc5=\E`, ri=\Ej, rmacs=^X,
	rmir=\Er, smacs=^U, smir=\Eq, tbc=\E3, tsl=\Eg\Ef,
	use=adm+sgr,
#
# is for 950 with two pages adds the following:
#	set 48 line page (\E\\2)
#	place cursor at page 0, line 24, column 1 (\E-07 )
#	set local (no send) edit keys (\Ek)
#
# two page 950 adds the following:
#	when entering ex, set 24 line page (\E\\1)
#	when exiting ex, reset 48 line page (\E\\2)
#			 place cursor at 0,24,1 (\E-07 )
#	set duplex (send) edit keys (\El) when entering vi
#	set local (no send) edit keys (\Ek) when exiting vi
#
tvi950-2p|televideo950 w/2 pages,
	is2=\EDF\EC\Ed\EG0\Eg\Er\EO\E'\E(\E%\Ew\EX\Ee \017\011\Ek\E016\E004\Ex0\0\0\Ex1\0\0\Ex2\0\0\011\Ex3\0\0\Ex4\r\0\E\\2\E-07 \011,
	rmcup=\E\\2\E-07\s, rmkx=\Ek, smcup=\E\\1\E-07\s,
	smkx=\El, use=tvi950,
#
# is for 950 with four pages adds the following:
#	set 96 line page (\E\\3)
#	place cursor at page 0, line 24, column 1 (\E-07 )
#
# four page 950 adds the following:
#	when entering ex, set 24 line page (\E\\1)
#	when exiting ex, reset 96 line page (\E\\3)
#			 place cursor at 0,24,1 (\E-07 )
#
tvi950-4p|televideo950 w/4 pages,
	is2=\EDF\EC\Ed\EG0\Eg\Er\EO\E'\E(\E%\Ew\EX\Ee \017\011\Ek\E016\E004\Ex0\0\0\Ex1\0\0\Ex2\0\0\011\Ex3\0\0\Ex4\r\0\E\\3\E-07 \011,
	rmcup=\E\\3\E-07\s, rmkx=\Ek, smcup=\E\\1\E-07\s,
	smkx=\El, use=tvi950,
#
# <is2> for reverse video 950 changes the following:
#	set reverse video (\Ed)
#
# set vb accordingly (\Ed ...delay... \Eb)
#
tvi950-rv|televideo950 rev video,
	flash=\Ed$<200/>\Eb,
	is2=\EDF\EC\Eb\EG0\Eg\Er\EO\E'\E(\E%\Ew\EX\Ee \017\011\El\E016\E004\Ex0\0\0\Ex1\0\0\Ex2\0\0\011\Ex3\0\0\Ex4\r\0,
	use=tvi950,

# tvi950-rv-2p uses the appropriate entries from 950-2p and 950-rv
tvi950-rv-2p|televideo950 rev video w/2 pages,
	flash=\Ed$<200/>\Eb,
	is2=\EDF\EC\Eb\EG0\Eg\Er\EO\E'\E(\E%\Ew\EX\Ee \017\011\Ek\E016\E004\Ex0\0\0\Ex1\0\0\Ex2\0\0\011\Ex3\0\0\Ex4\r\0\E\\2\E-07\s,
	rmcup=\E\\2\E-07\s, rmkx=\Ek, smcup=\E\\1\E-07\s,
	smkx=\El, use=tvi950,

# tvi950-rv uses the appropriate entries from 950-4p and 950-rv
tvi950-rv-4p|televideo950 rev video w/4 pages,
	flash=\Ed$<200/>\Eb,
	is2=\EDF\EC\Eb\EG0\Er\EO\E'\E(\E%\Ew\EX\Ee \017\011\Ek\E016\E004\Ex0\0\0\Ex1\0\0\Ex2\0\0\011\Ex3\0\0\Ex4\r\0\E\\3\E-07\s,
	rmcup=\E\\3\E-07\s, rmkx=\Ek, smcup=\E\\1\E-07\s,
	smkx=\El, use=tvi950,
# From: Andreas Stolcke <stolcke@icsi.berkeley.edu>
# (tvi955: removed obsolete ":ma:=^Vj^Kk^Hh^Ll^^H";
# removed incorrect (and overridden) ":do=^J:"; fixed broken continuations in
# the :rs: string, inserted the <ich> implied by the termcap :ko: string.  Note
# the :ko: string had :cl: in it, which means that one of the original
# <clear=\E*>, <kclr=\EY> had to be wrong; set <kclr=\E*> because that's what
# the 950 has.   Finally, corrected the <kel> string to match the 950 and what
# ko implies -- esr)
# If the BSD termcap file was right, <cup=\E=%p1%{32}%+%c%p2%{32}%+%c> would
# also work.
tvi955|televideo 955,
	OTbs, mc5i, msgr@,
	it#8, xmc@,
	acsc=0_`RjHkGlFmEnIoPqKsQtMuLvOwNxJ, blink=\EG2,
	civis=\E.0, cnorm=\E.2, cud1=^V, cup=\E[%i%p1%d;%p2%dH,
	cvvis=\E.1, dim=\E[=5h, ind@, invis=\EG1,
	is2=\E[=3l\EF1\Ed\EG0\E[=5l\E%\El, kctab=\E2, khts=\E1,
	knp=\EK, kpp=\EJ, krmir=\EQ, ktbc=\E3, mc0=\EP, rmacs=\E%%,
	rmam=\E[=7l, rmxon=^N,
	rs1=\EDF\EC\Eg\Er\EO\E'\E(\Ew\EX\Ee \017\E0P\E6\0\E0p\E4\0\Ef\r,
	sgr0=\EG0\E[=5l, smacs=\E$, smam=\E[=7h, smxon=^O,
	use=tvi950,
tvi955-w|955-w|televideo955 w/132 cols,
	cols#132,
	is2=\E[=3h\EF1\Ed\EG0\E[=5l\E%\El, use=tvi955,
# use half-intensity as normal mode, full intensity as <bold>
tvi955-hb|955-hb|televideo955 half-bright,
	bold=\E[=5l, dim@, is2=\E[=3l\EF1\Ed\EG0\E[=5h\E%\El,
	sgr0=\EG0\E[=5h, use=tvi955,
# From: Humberto Appleton <beto@cs.utexas.edu>, 880521 UT Austin
# (tvi970: removed ":sg#0:"; removed <rmso>=\E[m, <rmul>=\E[m;
# added <am>/<csr>/<home>/<hpa>/<vpa>/<smcup>/<rmcup> from BRL.
# According to BRL we could have <rmkx>=\E>, <smkx>=\E= but I'm not sure what
# it does to the function keys.  I deduced <rmam>/<smam>.
# also added empty <acsc> to suppress tic warning,  -- esr)
tvi970|televideo 970,
	OTbs, OTpt, am, da, db, mir, msgr,
	cols#80, it#8, lines#24,
	acsc=, cbt=\E[Z, clear=\E[H\E[2J, csr=\E[%i%p1%d;%p2%dr,
	cub1=^H, cud1=\ED, cuf1=\E[C, cup=\E[%i%p1%d;%p2%df,
	cuu1=\EM, cvvis=\E[1Q, dch1=\E[P, dl1=\E[M, dsl=\Eg\Ef\r,
	ed=\E[J, el=\E[K, flash=\E[5m$<200/>\E[m, home=\E[H,
	hpa=\E[%i%p1%dG, ht=^I, il1=\E[L,
	is2=\E<\E[?21l\E[19h\E[1Q\E[10l\E[7l\E[H\E[2J,
	kbs=^H, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kf1=\E?a, kf2=\E?b, kf3=\E?c, kf4=\E?d, kf5=\E?e, kf6=\E?f,
	kf7=\E?g, kf8=\E?h, kf9=\E?i, khome=\E[H, ri=\EM, rmacs=\E(B,
	rmam=\E[?7h, rmcup=, rmir=\E[4l, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m, smacs=\E(B, smam=\E[?7l,
	smcup=\E[?20l\E[?7h\E[1Q, smir=\E[4h, smso=\E[7m,
	smul=\E[4m, vpa=\E[%i%p1%dd,
tvi970-vb|televideo 970 with visual bell,
	flash=\E[?5h\0\0\0\0\0\0\0\0\0\0\0\0\0\E[?5l,
	use=tvi970,
tvi970-2p|televideo 970 with using 2 pages of memory,
	rmcup=\E[H\E[J\E[V, smcup=\E[U\E[?20l\E[?7h\E[1Q,
	use=tvi970,
# Works with vi and rogue.  NOTE: Esc v sets autowrap on, Esc u sets 80 chars
# per line (rather than 40), Esc K chooses the normal character set.  Not sure
# padding is needed, but adapted from the tvi920c termcap.  The <smso> and
# <smul> strings are klutzy, but at least use no screen space.
# (tvipt: removed obsolete ":ma=^Kk^Ll^R^L:".  I wish we knew <rmam>,
# its absence means <smam>=\Ev isn't safe to use. -- esr)
# From: Gene Rochlin <armsis@amber.berkeley.edu> 9/19/84.
# The <ed>/<kf0>/<kf1>/<khome>/<mc4>, and <mc5> caps are from BRL, which says:
# F1 and F2 should be programmed as ^A and ^B; required for UNIFY.
tvipt|televideo personal terminal,
	OTbs, am,
	cols#80, lines#24,
	cbt=\EI, clear=^Z, cub1=^H, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dl1=\ER$<5*>,
	ed=\EY, el=\ET, home=^^, if=/usr/share/tabset/stdcrt,
	il1=\EE$<5*>, is2=\Ev\Eu\EK, kbs=^H, kcub1=^H, kcud1=^J,
	kcuf1=^L, kcuu1=^K, kf0=^A, kf1=^B, khome=^^, mc4=^T, mc5=^R,
	rmso=\EF, rmul=\EF, smso=\EG1@A\EH, smul=\EG1B@\EH,
# From: Nathan Peterson <nathan@sco.com>, 03 Sep 1996
tvi9065|televideo 9065,
	am, bw, chts, hs, mc5i, mir, msgr, xenl, xon,
	cols#80, it#8, lh#1, lines#25, lm#0, lw#9, ma#4, nlab#8, vt#0,
	wnum#0, wsl#30,
	acsc='r0_jhkglfmeniopqksqtmulvownxj, bel=^G,
	blink=\EG2, bold=\EG\,, cbt=\EI, civis=\E.0, clear=^Z,
	cnorm=\E.3, cr=^M, csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD,
	cub1=^H, cud=\E[%p1%dB, cud1=^V, cuf=\E[%p1%dC, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu=\E[%p1%dA,
	cuu1=^K, cvvis=\E.2, dch=\E[%p1%dP, dch1=\EW, dim=\EGp,
	dl=\E[%p1%dM, dl1=\ER, dsl=\E_30\r, ech=\E[%p1%d@, ed=\EY,
	el=\ET, flash=\Eb$<15>\Ed, fsl=^M, home=^^, ht=^I, hts=\E1,
	ich=\E[%p1%d@, if=/usr/share/tabset/stdcrt,
	il=\E[%p1%dL, il1=\EE, ind=^J, indn=\E[%p1%dS, invis=\EG1,
	ip=$<3>,
	is1=\E"\E%\E'\E(\EG@\EO\EX\E[=5l\E[=6l\E[=7h\Ed\Er,
	is2=\EF2\EG0\E\\L, is3=\E<\E[=4l\E[=8h, kHOM=\E\s\s\s,
	kbs=^H, kcbt=\EI, kcub1=^H, kcud1=^V, kcuf1=^L, kcuu1=^K,
	kdch1=\EW, kf1=^A@\r, kf10=^AI\r, kf11=^AJ\r, kf12=^AK\r,
	kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r,
	kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^, ll=\E[25;1H,
	mc0=\E[0;0i, mc4=\Ea, mc5=\E`, nel=^M^J,
	pfkey=\E|%p1%{48}%+%c3%p2%s\031,
	pfloc=\E|%p1%{48}%+%c2%p2%s\031,
	pfx=\E|%p1%{48}%+%c1%p2%s\031,
	pln=\E_%p1%{63}%+%c%p2%s\r, prot=\E&,
	rep=\E[%p2%db%p1%c, rev=\EG4,
	rf=/usr/share/tabset/stdcrt, ri=\Ej, rin=\E[%p1%dT,
	rmacs=\E%%, rmam=\E[=7l, rmcup=\E.3\Er\E[1;25r\E[25;0H,
	rmdc=\0, rmir=\Er, rmln=\E[4;1v, rmso=\EG0, rmul=\EG0,
	rmxon=^N, rs1=\EC\EDF\E[0;0v\E[8;1v\E[=65l,
	rs2=\E.b\E[10;20v\E[14;1v\E[3;0v\E[7;0v\E[=11.h\E[=12.h\E[=13.h\E[=14.h\E[=15l\E[=20h\E[=60l\E[=61h\E[=9l\E[=10l\E[=21l\E[=23l\E[=3l\E_40\E_50\En\Ew\Ee \Ex0\0\0\Ex1\0\0\Ex2\0\0\Ex3\0\0\Ex4\0\0\E1,
	rs3=\E[=19h\E.3\E9\E0O\0\0\0\0\0\E0o\0\0\0\0\0\E0J\177\0\0\0\0,
	sgr=\EG0%?%p1%t\EGt%;%?%p2%t\EG8%;%?%p3%t\EG4%;%?%p4%t\EG2%;%?%p5%t\EGp%;%?%p6%t\EG\,%;%?%p7%t\EG1%;%?%p8%t\E&%;%?%p9%t\E$%e\E%%%;,
	sgr0=\EG0\E%, smacs=\E$, smam=\E=7h, smcup=\E.2, smdc=\Er,
	smir=\Eq, smln=\E[4;2v, smso=\EGt, smul=\EG8, smxon=^O,
	tbc=\E3, tsl=\E[4;1v\E_30, uc=\EG8\EG0,

#### Visual (vi)
#
# In September 1993, Visual Technology of Westboro, Massachusetts,
# merged with White Pine Software of Nashua, New Hampshire.
#
# White Pine Software may be contacted at +1 603/886-9050.
# Or visit White Pine on the World Wide Web at URL http://www.wpine.com.
#

# Visual 50 from Beau Shekita, BTL-Whippany <whuxlb!ejs>
# Recently I hacked together the following termcap for Visual
# Technology's Visual 50 terminal. It's a slight modification of
# the vt52 termcap.
# It's intended to run when the Visual 50 is in vt52 emulation mode
# (I know what you're thinking; if it's emulating a vt52, then why
# another termcap? Well, it turns out that the Visual 50 can handle
# <dl1> and db(?) among other things, which the vt52 can't)
# The termcap works OK for the most part. The only problem is on
# character inserts. The whole line gets painfully redrawn for each
# character typed. Any suggestions?
# Beau's entry is combined with the vi50 entry from University of Wisconsin.
# Note especially the <il1> function.  <kf4>-<kf6> are really l4-l6 in
# disguise; <kf7>-<kf9> are really l1-l3.
vi50|visual 50,
	OTbs, OTpt, am, da, db, msgr,
	cols#80, it#8, lines#24,
	OTnl=^J, bel=^G, cbt=\Ez$<4/>, clear=\EH\EJ, cr=^M, cub1=^H,
	cud1=\EB, cuf1=\EC, cup=\EY%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=\EA, dl1=\EM$<3*/>, ed=\EJ, el=\EK$<16/>, home=\EH,
	ht=^I, il1=\EL, ind=^J, kbs=^H, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, kf1=\EP, kf2=\EQ, kf3=\ER, kf4=\EV,
	kf5=\EE, kf6=\E], kf7=\EL, kf8=\Ev, kf9=\EM, khome=\EH,
	nel=^M^J, ri=\EI, rmso=\ET, rmul=\EW, smso=\EU, smul=\ES,
# this one was BSD & SCO's vi50
vi50adm|visual 50 in adm3a mode,
	am, msgr,
	cols#80, it#8, lines#24,
	bel=^G, clear=^Z, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dl1=\EM,
	ed=\Ek, el=\EK, home=\EH, ht=^I, il1=\EL, ind=^J, kbs=^H,
	kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA, khome=\EH,
	rmso=\ET, smso=\EU,
# From: Jeff Siegal <jbs@quiotix.com>
vi55|Visual 55,
	OTbs, am, mir, msgr,
	cols#80, it#8, lines#24,
	clear=\Ev, csr=\E_%p1%{65}%+%c%p2%{65}%+%c, cub1=^H,
	cud1=^J, cuf1=\EC, cup=\EY%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=\EA, dch1=\Ew, dl1=\EM, ed=\EJ, el=\EK, home=\EH, ht=^I,
	il1=\EL, is2=\Ev\E_AX\Eb\EW\E9P\ET, kbs=^H, kcub1=\ED,
	kcud1=\EB, kcuf1=\EC, kcuu1=\EA, ri=\EI, rmir=\Eb, rmso=\ET,
	smir=\Ea, smso=\EU,

# Visual 200 from BRL
# The following switch settings are assumed for normal operation:
#	FULL_DUPLEX		SCROLL			CR
#	AUTO_NEW_LINE_ON	VISUAL_200_EMULATION_MODE
# Other switches may be set for operator convenience or communication
# requirements.
# Character insertion is kludged in order to get around the "beep" misfeature.
# (This cap is commented out because <smir>/<rmir> is more efficient -- esr)
# Supposedly "4*" delays should be used for <il1>, <ed>, <clear>, <dch1>,
# and <dl1> strings, but we seem to get along fine without them.
vi200|visual 200,
	OTbs, OTpt, am, mir, msgr,
	OTkn#10, cols#80, it#8, lines#24,
	acsc=, bel=^G, cbt=\Ez, clear=\Ev, cnorm=\Ec, cr=^M, cub1=^H,
	cud1=^J, cuf1=\EC, cup=\EY%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=\EA, cvvis=\Ed, dch1=\EO, dim=\E4, dl1=\EM, ed=\Ey,
	el=\Ex, home=\EH, ht=^I, hts=\E1, il1=\EL, ind=^J, invis=\Ea,
	kbs=^H, kclr=\Ev, kctab=\E2, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kdch1=\EO, kdl1=\EM, ked=\EJ, kel=\Et, kf0=\E?p,
	kf1=\E?q, kf2=\E?r, kf3=\E?s, kf4=\E?t, kf5=\E?u, kf6=\E?v,
	kf7=\E?w, kf8=\E?x, kf9=\E?y, khome=\EH, khts=\E1, kich1=\Ei,
	kil1=\EL, krmir=\Ej, mc0=\EH\E], mc4=\EX, mc5=\EW, ri=\EI,
	rmacs=\EG, rmkx=\E>, rmso=\E3,
	rs1=\E3\Eb\Ej\E\El\EG\Ec\Ek\EX, sgr0=\E3\Eb, smacs=\EF,
	smkx=\E=, smso=\E4, tbc=\Eg,
# The older Visuals didn't come with function keys. This entry uses
# <smkx> and <rmkx> so that the keypad keys can be used as function keys.
# If your version of vi doesn't support function keys you may want
# to use vi200-f.
vi200-f|visual 200 no function keys,
	is2=\E3\Eb\Ej\E\\\El\EG\Ed\Ek, kf0=\E?p, kf1=\E?q,
	kf2=\E?r, kf3=\E?s, kf4=\E?t, kf5=\E?u, kf6=\E?v, kf7=\E?w,
	kf8=\E?x, kf9=\E?y, rmkx=\E>, rmso@, smkx=\E=, smso@,
	use=vi200,
vi200-rv|visual 200 reverse video,
	cnorm@, cvvis@, ri@, rmso=\E3, smso=\E4, use=vi200,

# the function keys are programmable but we don't reprogram them to their
# default values with <is2> because programming them is very verbose. maybe
# an initialization file should be made for the 300 and they could be stuck
# in it.
# (vi300: added <rmam>/<smam> based on init string -- esr)
vi300|visual 300 ansi x3.64,
	am, bw, mir, xenl,
	cols#80, lines#24,
	bel=^G, cbt=\E[Z, clear=\E[H\E[2J, cr=^M, cub1=^H, cud1=\E[B,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A,
	dch1=\E[P$<40>, dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I,
	il1=\E[L, ind=^J,
	is2=\E[7s\E[2;3;4;20;?5;?6l\E[12;?7h\E[1Q\E[0;1(D\E[8s,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kf1=\E_A\E\\, kf2=\E_B\E\\, kf3=\E_C\E\\, kf4=\E_D\E\\,
	kf5=\E_E\E\\, kf6=\E_F\E\\, kf7=\E_G\E\\, kf8=\E_H\E\\,
	kf9=\E_I\E\\, khome=\E[H, ri=\EM, rmam=\E[?7l, rmir=\E[4l,
	rmso=\E[m, rmul=\E[m, sgr0=\E[m, smam=\E[?7h, smir=\E[4h,
	smso=\E[1m, smul=\E[4m,
# some of the vi300s have older firmware that has the command
# sequence for setting editing extent reversed.
vi300-old|visual 300 with old firmware (set edit extent reversed),
	is2=\E[7s\E[2;3;4;20;?5;?6l\E[12;?7h\E[2Q\E[0;1(D\E[8s,
	use=vi300,

# Visual 500 prototype entry from University of Wisconsin.
# The best place to look for the escape sequences is page A1-1 of the
# Visual 500 manual.  The initialization sequence given here may be
# overkill, but it does leave out some of the initializations which can
# be done with the menus in set-up mode.
# The :xp: line below is so that emacs can understand the padding requirements
# of this slow terminal.  :xp: is 10 time the padding factor.
# (vi500: removed unknown :xp#4: termcap;
# also added empty <acsc> to suppress tic warning -- esr)
vi500|visual 500,
	am, mir, msgr,
	cols#80, it#8, lines#33,
	acsc=, cbt=\Ez$<4/>, clear=\Ev$<6*/>, cr=^M,
	csr=\E(%p1%{32}%+%c%p2%{32}%+%c, cub1=^H, cud1=\EB,
	cuf1=\EC, cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA,
	dch1=\EO$<3*/>, dl1=\EM$<3*/>, ed=\Ey$<3*/>,
	el=\Ex$<16/>, home=\EH, ht=\011$<8/>, il1=\EL\Ex$<3*/>,
	ind=^J,
	is2=\E3\E\001\E\007\E\003\Ek\EG\Ed\EX\El\E>\Eb\E\\,
	kbs=^H, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA,
	khome=\EH, nel=^M^J, rmacs=^O, rmir=\Ej, rmso=\E^G,
	rmul=\E^C, smacs=^N, smir=\Ei, smso=\E^H, smul=\E^D,

# The visual 550 is a visual 300 with tektronix graphics,
# and with 33 lines. clear screen is modified here to
# also clear the graphics.
vi550|visual 550 ansi x3.64,
	lines#33,
	clear=\030\E[H\E[2J, use=vi300,

vi603|visual603|visual 603,
	hs, mir,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[J,
	csr=\E[%i%p1%d;%p2%dr, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P, dl1=\E[M,
	dsl=\EP2;1~\E\\, ed=\E[J, el=\E[K, fsl=\E\\, il1=\E[L,
	ind=\ED, is1=\E>\E[?3l\E[?4l\E[?7h\E[?8h\E[1;24r,
	rev=\E[7m, ri=\EM, rmir=\E[4l, rmso=\E[27m, rmul=\E[24m,
	sgr0=\E[m, smir=\E[4h, smso=\E[7m, smul=\E[4m, tsl=\EP2~,
	use=vt100,

#### Wyse (wy)
#
#	Wyse Technology
#	3471 North First Street
#	San Jose, CA 95134
#	Vox: (408)-473-1200
#	Fax: (408) 473-1222
#	Web: http://www.wyse.com
#
# Wyse sales can be reached by phone at 1-800-GET-WYSE.  Tech support is at
# (800)-800-WYSE (option 5 gets you a human).  There's a Web page at the
# obvious address, <http://www.wyse.com>.  They keep terminfo entries at
# <http://www.wyse.co.uk/support/appnotes/idxappnt.htm>.
#
# Wyse bought out Link Technology, Inc. in 1990 and closed it down in 1995.
# They now own the Qume and Amdek brands, too.  So these are the people to
# talk with about all Link, Qume, and Amdek terminals.
#
# These entries include a few small fixes.
# I canceled the bel capacities in the vb entries.
# I made two trivial syntax fixes in the wyse30 entry.
# I made some entries relative to adm+sgr.
#
#
# Note: The wyse75, wyse85, and wyse99 have been discontinued.

#	   Although the Wyse 30 can support more than one attribute
#	it requires magic cookies to do so.  Many applications do not
#	function well with magic cookies.  The following terminfo uses
#	the protect mode to support one attribute (dim) without cookies.
#	If more than one attribute is needed then the wy30-mc terminfo
#	should be used.
#
wy30|wyse30|Wyse 30,
	am, bw, hs, mc5i, mir, msgr, xon,
	cols#80, lh#1, lines#24, lw#8, ma#1, nlab#8, wsl#45,
	acsc=0wa_h[jukslrmqnxqzttuyv]wpxv, bel=^G, cbt=\EI,
	civis=\E`0, clear=\E+$<80>, cnorm=\E`1, cr=^M, cub1=^H,
	cud1=^J, cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=^K, dch1=\EW$<10>, dim=\E`7\E), dl1=\ER$<1>,
	dsl=\EF\r, ed=\EY$<80>, el=\ET, flash=\E`8$<100/>\E`9,
	fsl=^M, home=^^, ht=\011$<1>, hts=\E1, il1=\EE$<2>,
	ind=\n$<2>, ip=$<2>, is2=\E'\E(\E\^3\E`9\016\024,
	kHOM=\E{, kbs=^H, kcbt=\EI, kcub1=^H, kcud1=^J, kcuf1=^L,
	kcuu1=^K, kdch1=\EW, kdl1=\ER, ked=\EY, kel=\ET, kent=\E7,
	kf1=^A@\r, kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r,
	kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, khome=^^, kich1=\EQ,
	kil1=\EE, knp=\EK, kpp=\EJ, krpl=\Er, ll=^^^K, mc0=\EP, mc4=^T,
	mc5=^X, nel=^M^J, pfx=\Ez%p1%{63}%+%c%p2%s\177,
	pln=\Ez%p1%{47}%+%c%p2%s\r, prot=\E`7\E), ri=\Ej$<3>,
	rmacs=\EH^C, rmir=\Er, rmln=\EA11, rmso=\E(,
	sgr=%?%p1%p5%p8%|%|%t\E`7\E)%e\E(%;%?%p9%t\EH\002%e\EH\003%;,
	sgr0=\E(\EH\003, smacs=\EH^B, smir=\Eq, smln=\EA10,
	smso=\E`7\E), tbc=\E0, tsl=\EF,
#
#	This terminal description uses the non-hidden attribute mode
#	(with magic cookie).
#
# (wy30-mc: added <smcup> to suppress tic warning --esr)
wy30-mc|wyse30-mc|wyse 30 with magic cookies,
	msgr@,
	ma@, xmc#1,
	blink=\EG2, dim=\EGp, prot=\EG0\E), rmacs=\EG0\EH\003,
	rmcup=\EG0, rmso=\EG0,
	sgr=\EG%{48}%?%p2%p6%|%t%{8}%|%;%?%p1%p3%|%p6%|%t%{4}%|%;%?%p4%t%{2}%|%;%?%p1%p5%|%t%{64}%|%;%?%p7%t%{1}%|%;%c%?%p8%t\E)%e\E(%;%?%p9%t\EH\002%e\EH\003%;,
	sgr0=\EG0\E(\EH\003, smacs=\EG0\EH\002, smcup=,
	smso=\EG4, use=wy30, use=adm+sgr,
#	The mandatory pause used by <flash> does not work with
#	older versions of terminfo.  If you see this effect then
#	unset xon and delete the / from the delay.
#	i.e. change $<100/> to $<100>
wy30-vb|wyse30-vb|wyse 30 visible bell,
	bel@, use=wy30,
#
#	   The Wyse 50 can support one attribute (e.g. Dim, Inverse,
#	Normal) without magic cookies by using the protect mode.
#	The following description uses this feature, but when more
#	than one attribute is put on the screen at once, all attributes
#	will be changed to be the same as the last attribute given.
#	   The Wyse 50 can support more attributes when used with magic
#	cookies.  The wy50-mc terminal description uses magic cookies
#	to correctly handle multiple attributes on a screen.
#
wy50|wyse50|Wyse 50,
	am, bw, hs, mc5i, mir, msgr, xon,
	cols#80, lh#1, lines#24, lw#8, ma#1, nlab#8, wsl#45,
	acsc=0wa_h[jukslrmqnxqzttuyv]wpxv, bel=^G, cbt=\EI,
	civis=\E`0, clear=\E+$<20>, cnorm=\E`1, cr=^M, cub1=^H,
	cud1=^J, cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=^K, dch1=\EW$<1>, dim=\E`7\E), dl1=\ER, dsl=\EF\r,
	ed=\EY$<20>, el=\ET, flash=\E`8$<100/>\E`9, fsl=^M,
	home=^^, ht=^I, hts=\E1, il1=\EE, ind=\n$<2>, ip=$<1>,
	is1=\E`\:\E`9$<30>, is2=\016\024\E'\E(, kHOM=\E{, kbs=^H,
	kcbt=\EI, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, kdch1=\EW,
	kdl1=\ER, ked=\EY, kel=\ET, kent=\E7, kf1=^A@\r, kf10=^AI\r,
	kf11=^AJ\r, kf12=^AK\r, kf13=^AL\r, kf14=^AM\r, kf15=^AN\r,
	kf16=^AO\r, kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r,
	kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^,
	kich1=\EQ, kil1=\EE, knp=\EK, kpp=\EJ, kprt=\EP, krpl=\Er,
	ll=^^^K, mc0=\EP, mc4=^T, mc5=^X, nel=^M^J,
	pfx=\Ez%p1%{63}%+%c%p2%s\177,
	pln=\Ez%p1%{47}%+%c%p2%s\r, prot=\E`7\E), rev=\E`6\E),
	ri=\Ej, rmacs=\EH^C, rmir=\Er, rmln=\EA11, rmso=\E(,
	sgr=%?%p1%p3%|%t\E`6\E)%e%p5%p8%|%t\E`7\E)%e\E(%;%?%p9%t\EH\002%e\EH\003%;,
	sgr0=\E(\EH\003, smacs=\EH^B, smir=\Eq, smln=\EA10,
	smso=\E`6\E), tbc=\E0, tsl=\EF,
#
#	This terminal description uses the non-hidden attribute mode
#	(with magic cookie).
#
#	The mandatory pause used by flash does not work with some
#	older versions of terminfo.  If you see this effect then
#	unset <xon> and delete the / from the delay.
#	i.e. change $<100/> to $<100>
# (wy50-mc: added <smcup> to suppress tic warning --esr)
wy50-mc|wyse50-mc|wyse 50 with magic cookies,
	msgr@,
	ma@, xmc#1,
	blink=\EG2, dim=\EGp, prot=\EG0\E), rev=\EG4,
	rmacs=\EG0\EH\003, rmcup=\EG0, rmso=\EG0,
	sgr=\EG%{48}%?%p2%p6%|%t%{8}%|%;%?%p1%p3%|%p6%|%t%{4}%|%;%?%p4%t%{2}%|%;%?%p1%p5%|%t%{64}%|%;%?%p7%t%{1}%|%;%c%?%p8%t\E)%e\E(%;%?%p9%t\EH\002%e\EH\003%;,
	sgr0=\EG0\E(\EH\003, smacs=\EG0\EH\002, smcup=,
	smso=\EGt, use=wy50, use=adm+sgr,
wy50-vb|wyse50-vb|wyse 50 visible bell,
	bel@, use=wy50,
wy50-w|wyse50-w|wyse 50 132-column,
	cols#132, lw#7, nlab#16, wsl#97,
	cup=\Ea%i%p1%dR%p2%dC, dch1=\EW$<2>, is1=\E`;\E`9$<30>,
	use=wy50,
wy50-wvb|wyse50-wvb|wyse 50 132-column visible bell,
	bel@, use=wy50-w,

#
#	The Wyse 350 is a Wyse 50 with color.
#	Unfortunately this means that it has magic cookies.
#	The color attributes are designed to overlap the reverse, dim and
#	underline attributes.  This is nice for monochrome applications
#	because you can make underline stuff green (or any other color)
#	but for true color applications it's not so hot because you cannot
#	mix color with reverse, dim or underline.
#	    To further complicate things one of the attributes must be
#	black (either the foreground or the background).  In reverse video
#	the background changes color with black letters.  In normal video
#	the foreground changes colors on a black background.
#	    This terminfo uses some of the more advanced features of curses
#	to display both color and blink.  In the final analysis I am not
#	sure that the wy350 runs better with this terminfo than it does
#	with the wy50 terminfo (with user adjusted colors).
#
#	The mandatory pause used by flash does not work with
#	older versions of terminfo.  If you see this effect then
#	unset xon and delete the / from the delay.
#	i.e. change $<100/> to $<100>
#
# Bug: The <op> capability resets attributes.
wy350|wyse350|Wyse 350,
	am, bw, hs, mc5i, mir, xon,
	colors#8, cols#80, lh#1, lines#24, lw#8, ncv#55, nlab#8, pairs#8,
	wsl#45, xmc#1,
	acsc=0wa_h[jukslrmqnxqzttuyv]wpxv, bel=^G, blink=\EG2,
	cbt=\EI, civis=\E`0, clear=\E+$<20>, cnorm=\E`1, cr=^M,
	cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW$<1>,
	dim=\EGp, dl1=\ER, dsl=\EF\r, ed=\EY$<20>, el=\ET,
	flash=\E`8$<100/>\E`9, fsl=^M, home=^^, ht=^I, hts=\E1,
	il1=\EE, ind=\n$<2>, ip=$<1>, is1=\E`\:\E`9$<30>,
	is2=\016\024\E'\E(, is3=\E%?, kHOM=\E{, kbs=^H, kcbt=\EI,
	kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, kdch1=\EW, kdl1=\ER,
	ked=\EY, kel=\ET, kent=\E7, kf1=^A@\r, kf10=^AI\r,
	kf11=^AJ\r, kf12=^AK\r, kf13=^AL\r, kf14=^AM\r, kf15=^AN\r,
	kf16=^AO\r, kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r,
	kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^,
	kich1=\EQ, kil1=\EE, knp=\EK, kpp=\EJ, kprt=\EP, krpl=\Er,
	ll=^^^K, mc0=\EP, mc4=^T, mc5=^X, nel=^M^J, oc=\E%?, op=\EG0,
	pfx=\Ez%p1%{63}%+%c%p2%s\177,
	pln=\Ez%p1%{47}%+%c%p2%s\r, prot=\EG0\E), ri=\Ej,
	rmacs=\EG0\EH\003, rmir=\Er, rmln=\EA11, setb=,
	setf=%?%p1%{0}%=%t%{76}%e%p1%{1}%=%t%{64}%e%p1%{2}%=%t%{8}%e%p1%{3}%=%t%{72}%e%p1%{4}%=%t%{4}%e%p1%{5}%=%t%{68}%e%p1%{6}%=%t%{12}%e%p1%{7}%=%t%{0}%;%PC\EG%gC%gA%+%{48}%+%c,
	sgr=%{0}%?%p4%t%{2}%|%;%?%p7%t%{1}%|%;%PA\EG%?%gC%t%gC%e%{0}%?%p1%t%{4}%|%;%?%p2%t%{8}%|%;%?%p3%t%{4}%|%;%?%p5%t%{64}%|%;%;%gA%+%{48}%+%c%?%p8%t\E)%e\E(%;%?%p9%t\EH\002%e\EH\003%;,
	sgr0=\EG0\E(\EH\003%{0}%PA%{0}%PC, smacs=\EG0\EH\002,
	smir=\Eq, smln=\EA10, tbc=\E0, tsl=\EF, use=adm+sgr,
wy350-vb|wyse350-vb|wyse 350 visible bell,
	bel@, use=wy350,
wy350-w|wyse350-w|wyse 350 132-column,
	cols#132, lw#7, nlab#16, wsl#97,
	cup=\Ea%i%p1%dR%p2%dC, dch1=\EW$<2>, is1=\E`;\E`9$<30>,
	use=wy350,
wy350-wvb|wyse350-wvb|wyse 350 132-column visible bell,
	bel@, use=wy350-w,
#
#	This terminfo description is untested.
#	The wyse100 emulates an adm31, so the adm31 entry should work.
#
wy100|wyse 100,
	hs, mir,
	cols#80, lines#24, xmc#1,
	bel=^G, clear=\E;, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, dsl=\EA31, ed=\EY, el=\ET, fsl=^M, il1=\EE, ind=^J,
	invis@, is2=\Eu\E0, kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^L,
	kcuu1=^K, kf1=^A@\r, kf2=^AA\r, kf3=^AB\r, kf4=^AC\r,
	kf5=^AD\r, kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, khome=\E{,
	rmir=\Er, smir=\Eq, tsl=\EF, use=adm+sgr,
#
#	The Wyse 120/150 has most of the features of the Wyse 60.
#	This terminal does not need padding up to 9600 baud!
#	<msgr> should be set but the clear screen fails when in
#	alt-charset mode.  Try \EcE\s\s\E+\s if the screen is really clear
#	then set <msgr>.
#
wy120|wyse120|wy150|wyse150|Wyse 120/150,
	am, bw, hs, km, mc5i, mir, msgr, xon,
	cols#80, it#8, lh#1, lines#24, lw#8, nlab#8, pb#9601, wsl#45,
	acsc=+/\,.0[a2fxgqh1ihjYk?lZm@nEqDtCu4vAwBx3yszr{c~~,
	bel=^G, blink=\EG2, cbt=\EI, civis=\E`0, clear=\E+$<50>,
	cnorm=\E`1, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW$<7>,
	dim=\EGp, dl1=\ER$<3>, dsl=\EF\r, ed=\EY$<50>, el=\ET$<4>,
	flash=\E`8$<100/>\E`9, fsl=^M, home=^^, ht=\011$<1>,
	hts=\E1, il1=\EE$<3>, ind=\n$<3>, ip=$<2>, is1=\EcB0\EcC1,
	is2=\Ed$\EcD\E'\Er\EH\003\Ed/\EO\Ee1\Ed*\E`@\E`9\E`1\016\024\El,
	is3=\EwJ\Ew1$<150>, kHOM=\E{, kbs=^H, kcbt=\EI, kcub1=^H,
	kcud1=^J, kcuf1=^L, kcuu1=^K, kdch1=\EW, kdl1=\ER, ked=\EY,
	kel=\ET, kent=\E7, kf1=^A@\r, kf10=^AI\r, kf11=^AJ\r,
	kf12=^AK\r, kf13=^AL\r, kf14=^AM\r, kf15=^AN\r, kf16=^AO\r,
	kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r,
	kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^, kich1=\EQ,
	kil1=\EE, knp=\EK, kpp=\EJ, kprt=\EP, krpl=\Er, ll=^^^K,
	mc0=\EP, mc4=^T, mc5=\Ed#, nel=\r\n$<3>,
	pfloc=\EZ2%p1%{63}%+%c%p2%s\177,
	pfx=\EZ1%p1%{63}%+%c%p2%s\177,
	pln=\Ez%p1%{47}%+%c%p2%s\r, prot=\E), ri=\Ej$<2>,
	rmacs=\EcD, rmam=\Ed., rmcup=\Ew1, rmir=\Er, rmln=\EA11,
	rmxon=\Ec20, rs1=\E~!\E~4$<30>, rs2=\EeF\E`\:$<70>,
	rs3=\EwG\Ee($<100>,
	sgr=%?%p8%t\E)%e\E(%;%?%p9%t\EcE%e\EcD%;\EG%{48}%?%p2%t%{8}%|%;%?%p1%p3%|%p6%|%t%{4}%|%;%?%p4%t%{2}%|%;%?%p1%p5%|%t%{64}%|%;%?%p7%t%{1}%|%;%c,
	sgr0=\E(\EH\003\EG0\EcD, smacs=\EcE, smam=\Ed/,
	smcup=\Ew0, smir=\Eq, smln=\EA10, smso=\EGt, smxon=\Ec21,
	tbc=\E0, tsl=\EF, use=adm+sgr,
#
wy120-w|wyse120-w|wy150-w|wyse150-w|wyse 120/150 132-column,
	cols#132, lw#7, nlab#16, wsl#97,
	cup=\Ea%i%p1%dR%p2%dC, dch1=\EW$<12>, ip=$<4>,
	rs2=\E`;$<70>, use=wy120,
#
wy120-25|wyse120-25|wy150-25|wyse150-25|wyse 120/150 80-column 25-lines,
	lh@, lines#25, lw@, nlab@,
	pln@, rs3=\EwG\Ee)$<100>, use=wy120,
#
wy120-25-w|wyse120-25-w|wy150-25-w|wyse150-25-w|wyse 120/150 132-column 25-lines,
	lh@, lines#25, lw@, nlab@,
	pln@, rs3=\EwG\Ee)$<100>, use=wy120-w,
#
wy120-vb|wyse120-vb|wy150-vb|wyse150-vb|Wyse 120/150 visible bell,
	bel@, use=wy120,
#
wy120-w-vb|wy120-wvb|wyse120-wvb|wy150-w-vb|wyse150-w-vb|Wyse 120/150 132-column visible bell,
	bel@, use=wy120-w,
#
#	The Wyse 60 is like the Wyse 50 but with more padding.
#	The reset strings are slow and the pad times very depending
#	on other parameters such as font loading.  I have tried
#	to follow the following outline:
#
#		<rs1> -> set personality
#		<rs2> -> set number of columns
#		<rs3> -> set number of lines
#		<is1> -> select the proper font
#		<is2> -> do the initialization
#		<is3> -> set up display memory (2 pages)
#
#	The Wyse 60's that have vt100 emulation are slower than the
#	older Wyse 60's.  This change happened mid-1987.
#	The capabilities effected are <dch1> <dl1> <il1> <ind> <ri>
#
#	The meta key is only half right.  This terminal will return the
#	high order bit set when you hit CTRL-function_key
#
#	It may be useful to assign two function keys with the
#	values  \E=(\s  look at old data in page 1
#	        \E=W,   look at bottom of page 1
#	where \s is a space ( ).
#
#	Note:
#	   The Wyse 60 runs faster when the XON/XOFF
#	   handshake is turned off.
#
# (wy60: we use \E{ rather than ^^ for home (both are documented) to avoid
# a bug reported by Robert Dunn, <rcdii@inlink.com> -- esr)
wy60|wyse60|Wyse 60,
	am, bw, hs, km, mc5i, mir, msgr,
	cols#80, lh#1, lines#24, lw#8, nlab#8, wsl#45,
	acsc=+/\,.0[a2fxgqh1ihjYk?lZm@nEqDtCu4vAwBx3yszr{c~~,
	bel=^G, blink=\EG2, cbt=\EI, civis=\E`0, clear=\E+$<100>,
	cnorm=\E`1, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	dch1=\EW$<11>, dclk=\E`b, dim=\EGp, dl1=\ER$<5>, dsl=\EF\r,
	ed=\EY$<100>, el=\ET, flash=\E`8$<100/>\E`9, fsl=^M,
	home=\E{, ht=\011$<1>, hts=\E1, il1=\EE$<4>, ind=\n$<5>,
	ip=$<3>, is1=\EcB0\EcC1,
	is2=\Ed$\EcD\E'\Er\EH\003\Ed/\EO\Ee1\Ed*\E`@\E`9\E`1\016\024\El,
	is3=\EwJ\Ew1$<150>, kHOM=\E{, kbs=^H, kcbt=\EI, kcub1=^H,
	kcud1=^J, kcuf1=^L, kcuu1=^K, kdch1=\EW, kdl1=\ER, ked=\EY,
	kel=\ET, kent=\E7, kf1=^A@\r, kf10=^AI\r, kf11=^AJ\r,
	kf12=^AK\r, kf13=^AL\r, kf14=^AM\r, kf15=^AN\r, kf16=^AO\r,
	kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r,
	kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^, kich1=\EQ,
	kil1=\EE, knp=\EK, kpp=\EJ, kprt=\EP, krpl=\Er, ll=\E{^K,
	mc0=\EP, mc4=^T, mc5=\Ed#, nel=\r\n$<3>,
	pfloc=\EZ2%p1%{63}%+%c%p2%s\177,
	pfx=\EZ1%p1%{63}%+%c%p2%s\177,
	pln=\Ez%p1%{47}%+%c%p2%s\r, prot=\E), ri=\Ej$<7>,
	rmacs=\EcD, rmam=\Ed., rmclk=\E`c, rmcup=\Ew1, rmir=\Er,
	rmln=\EA11, rmxon=\Ec20, rs1=\E~!\E~4$<150>,
	rs2=\EeG$<150>, rs3=\EwG\Ee($<200>,
	sgr=%?%p8%t\E)%e\E(%;%?%p9%t\EcE%e\EcD%;\EG%{48}%?%p2%t%{8}%|%;%?%p1%p3%|%p6%|%t%{4}%|%;%?%p4%t%{2}%|%;%?%p1%p5%|%t%{64}%|%;%?%p7%t%{1}%|%;%c,
	sgr0=\E(\EH\003\EG0\EcD, smacs=\EcE, smam=\Ed/,
	smcup=\Ew0, smir=\Eq, smln=\EA10, smso=\EGt, smxon=\Ec21,
	tbc=\E0, tsl=\EF, use=adm+sgr,
#
wy60-w|wyse60-w|wyse 60 132-column,
	cols#132, lw#7, nlab#16, wsl#97,
	cup=\Ea%i%p1%dR%p2%dC, dch1=\EW$<16>, ip=$<5>,
	rs2=\EeF$<150>\E`;$<150>, use=wy60,
#
wy60-25|wyse60-25|wyse 60 80-column 25-lines,
	lh@, lines#25, lw@, nlab@,
	pln@, rs3=\EwG\Ee)$<200>, use=wy60,
wy60-25-w|wyse60-25-w|wyse 60 132-column 25-lines,
	lh@, lines#25, lw@, nlab@,
	pln@, rs3=\EwG\Ee)$<200>, use=wy60-w,
#
wy60-42|wyse60-42|wyse 60 80-column 42-lines,
	lines#42,
	clear=\E+$<260>, cup=\E=%p1%{32}%+%c%p2%{32}%+%c$<2>,
	dch1=\EW$<16>, dl1=\ER$<11>, ed=\Ey$<260>, il1=\EE$<11>,
	ind=\n$<9>, ip=$<5>, is1=\EcB2\EcC3, nel=\r\n$<6>,
	ri=\Ej$<10>, rs3=\Ee*$<150>, use=wy60,
wy60-42-w|wyse60-42-w|wyse 60 132-column 42-lines,
	cols#132, lw#7, nlab#16, wsl#97,
	clear=\E+$<260>, cup=\Ea%i%p1%dR%p2%dC$<2>,
	dch1=\EW$<19>, ed=\Ey$<260>, home=\036$<2>, ip=$<6>,
	nel=\r\n$<11>, rs2=\EeF$<150>\E`;$<150>, use=wy60-42,
#
wy60-43|wyse60-43|wyse 60 80-column 43-lines,
	lh@, lines#43, lw@, nlab@,
	pln@, rs3=\Ee+$<150>, use=wy60-42,
wy60-43-w|wyse60-43-w|wyse 60 132-column 43-lines,
	lh@, lines#43, lw@, nlab@,
	pln@, rs3=\Ee+$<150>, use=wy60-42-w,
#
wy60-vb|wyse60-vb|Wyse 60 visible bell,
	bel@, use=wy60,
wy60-w-vb|wy60-wvb|wyse60-wvb|Wyse 60 132-column visible bell,
	bel@, use=wy60-w,

#	The Wyse-99GT looks at lot like the Wyse 60 except that it
#	does not have the 42/43 line mode.  In the Wyse-60 the "lines"
#	setup parameter controls the number of lines on the screen.
#	For the Wyse 99GT the "lines" setup parameter controls the
#	number of lines in a page.  The screen can display 25 lines max.
#	    The Wyse-99GT also has personalities for the VT220 and
#	Tektronix 4014.  But this has no bearing on the native mode.
#
#	(msgr) should be set but the clear screen fails when in
#	alt-charset mode.  Try \EcE\s\s\E+\s if the screen is really clear
#	then set msgr, else use msgr@.
#
#	u0 -> enter Tektronix mode
#	u1 -> exit Tektronix mode
#
wy99gt|wyse99gt|Wyse 99gt,
	msgr@,
	clear=\E+$<130>, dch1=\EW$<7>, dl1=\ER$<4>, ed=\Ey$<130>,
	el=\Et$<5>, flash=\E`8$<100/>\E`9, ht=\011$<1>,
	il1=\EE$<4>, ind=\n$<4>, ip=$<2>, is3=\Ew0$<20>, nel@,
	ri=\Ej$<3>, rmcup=\Ew0, rs2=\E`\:$<150>, smcup=\Ew1,
	u0=\E~>\E8, u1=\E[42h, use=wy60,
#
wy99gt-w|wyse99gt-w|wyse 99gt 132-column,
	cols#132, lw#7, nlab#16, wsl#97,
	clear=\E+$<160>, cup=\Ea%i%p1%dR%p2%dC$<2>,
	dch1=\EW$<9>, ed=\Ey$<160>, ip=$<4>, rs2=\E`;$<150>,
	use=wy99gt,
#
wy99gt-25|wyse99gt-25|wyse 99gt 80-column 25-lines,
	lh@, lines#25, lw@, nlab@,
	pln@, rs2=\E`\:$<150>, rs3=\EwG\Ee)$<200>, use=wy99gt,
#
wy99gt-25-w|wyse99gt-25-w|wyse 99gt 132-column 25-lines,
	lh@, lines#25, lw@, nlab@,
	pln@, rs2=\E`;$<150>, use=wy99gt-w,
#
wy99gt-vb|wyse99gt-vb|Wyse 99gt visible bell,
	bel@, use=wy99gt,
#
wy99gt-w-vb|wy99gt-wvb|wyse99gt-wvb|Wyse 99gt 132-column visible bell,
	bel@, use=wy99gt-w,

# Can't set tabs! Other bugs (ANSI mode only):
# - can't redefine function keys (anyway, key redefinition in ANSI mode
#   is too much complex to be described);
# - meta key can't be described (the terminal forgets it when reset);
# The xon-xoff handshaking can't be disabled while in ansi personality, so
# emacs can't work at speed greater than 9600 baud.  No padding is needed at
# this speed.
#   dch1 has been commented out because it causes annoying glittering when
# vi deletes one character at the beginning of a line with tabs in it.
#   dch makes sysgen(1M) have a horrible behaviour when deleting
# a screen and makes screen(1) behave badly, so it is disabled too. The nice
# thing is that vi goes crazy if smir-rmir are present and both dch-dch1 are
# not, so smir and rmir are commented out as well.
# From: Francesco Potorti` <F.Potorti@cnuce.cnr.it>, 24 Aug 1998
wy99-ansi|Wyse WY-99GT in ansi mode (int'l PC keyboard),
	am, km, mc5i, mir, msgr, xenl,
	cols#80, it#8, lines#25, vt#3,
	acsc=``aaffggjjkkllmmnnooqqssttuuvvwwxx{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J$<200>, cnorm=\E[34h\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD$<1>,
	cub1=\010$<1>, cud=\E[%p1%dB, cud1=\ED,
	cuf=\E[%p1%dC$<1>, cuf1=\E[C$<1>,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\EM,
	cvvis=\E[34l\E[?25h, dim=\E[2m, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J$<8*>, el=\E[K$<1>, el1=\E[1K$<1>,
	enacs=\E)0, flash=\E[?5h$<30/>\E[?5l, home=\E[H,
	hpa=\E[%i%p1%dG, ht=^I, ich=\E[%p1%d@, il=\E[%p1%dL,
	il1=\E[L, ind=\n$<1>, invis=\E[8m,
	is2=\E7\E[1r\E8\E[2;3;4;13;20;34;39;36l\E[12;16;34h\E[?1;3;4;5;10;18l\E[?7;8;25h\E>\E[?5W\E(B\017\E[4i,
	kbs=^H, kcbt=\E[z, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC,
	kcuu1=\EOA, kf1=\EOP, kf10=\E[21~, kf11=\E[23~,
	kf12=\E[24~, kf17=\E[K, kf18=\E[31~, kf19=\E[32~, kf2=\EOQ,
	kf20=\E[33~, kf21=\E[34~, kf22=\E[35~, kf23=\E[1~,
	kf24=\E[2~, kf3=\EOR, kf4=\EOS, kf5=\E[M, kf6=\E[17~,
	kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, ll=\E[24E, mc0=\E[?19h,
	mc4=\E[4i, mc5=\E[5i, nel=\EE, prot=\E[1"q, rc=\E8,
	rev=\E[7m, ri=\EM, rmacs=^O, rmam=\E[?7l, rmir=\E[4l,
	rmkx=\E[?1l, rmso=\E[27m, rmul=\E[24m,
	rs2=\E[61"p\E[40h\E[?6l\E[1r\E[2;3;4;13;20;34;39;36l\E[12;16;34h\E[?1;3;4;5;10;18l\E[?7;8;25h\E>\E[?5W\E(B\017\E[24E\E[4i,
	sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p2%t;4%;%?%p1%p3%O%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p7%t;8%;m\E[%?%p8%t1%;"q%?%p9%t\016%e\017%;,
	sgr0=\E[m\017\E["q, smacs=^N, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1h, smso=\E[7m, smul=\E[4m, vpa=\E[%i%p1%dd,

#   This is the american terminal. Here tabs work fine.
# From: Francesco Potorti` <F.Potorti@cnuce.cnr.it>, 24 Aug 1998
wy99a-ansi|Wyse WY-99GT in ansi mode (US PC keyboard),
	hts=\EH, is3=\E[?5l, rs3=\E[?5l, tbc=\E[3g, use=wy99-ansi,

# This terminal (firmware version 02) has a lot of bugs:
# - can't set tabs;
# - other bugs in ANSI modes (see above).
# This description disables handshaking when using cup. This is because
# GNU emacs doesn't like Xon-Xoff handshaking. This means the terminal
# cannot be used at speeds greater than 9600 baud, because at greater
# speeds handshaking is needed even for character sending. If you use
# DTR handshaking, you can use even greater speeds.
# From: Francesco Potorti` <F.Potorti@cnuce.cnr.it>, 24 Aug 1998
wy99f|wy99fgt|wy-99fgt|Wyse WY-99GT (int'l PC keyboard),
	am, bw, hs, km, mc5i, mir, msgr, xon,
	cols#80, it#8, lines#25, wsl#46,
	acsc='x+y.w_vi~j(k'l&m%n)o9q*s8t-u.v\,w+x=, bel=^G,
	blink=\EG2, cbt=\EI, civis=\E`0, clear=\E'\E(\032,
	cnorm=\E`4\E`1, cr=^M, cub1=^H, cud1=\Ej, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	cvvis=\E`2\E`1, dch1=\EW, dim=\EGp, dl1=\ER, dsl=\EF\r,
	ed=\EY$<8*>, el=\ET$<8>, enacs=\Ec@1J$<2000>,
	flash=\E\^1$<30/>\E\^0, fsl=^M, home=^^, ht=^I, il1=\EE,
	ind=^J, invis=\EG3,
	is2=\Eu\Ee6\EC\EDF\Ec21\Ec31\Ec62\Ec72\Ee;\016\E'\EeL\E`9\E\^0\E`1\E`4\Ee.\E`\:\Ee1\EG0\E(\Ed/\Ee4\Ed*\EO\E`I\Er\Ee"\EcD\024,
	ka1=^^, ka3=\EJ, kbs=^H, kc1=\ET, kc3=\EK, kcbt=\EI, kcub1=^H,
	kcud1=^J, kcuf1=^L, kcuu1=^K, kf1=^A@\r, kf10=^AI\r,
	kf11=^AJ\r, kf12=^AK\r, kf13=^A`\r, kf14=^Aa\r, kf15=^Ab\r,
	kf16=^Ac\r, kf17=^Ad\r, kf18=^Ae\r, kf19=^Af\r, kf2=^AA\r,
	kf20=^Ag\r, kf21=^Ah\r, kf22=^Ai\r, kf23=^Aj\r, kf24=^Ak\r,
	kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r,
	kf8=^AG\r, kf9=^AH\r, kprt=\EP, mc0=\EP, mc4=^T, mc5=\Ed#,
	nel=^_, prot=\E), rev=\EG4, ri=\Ej, rmacs=\EcD, rmam=\Ed.,
	rmcup=\Ec21\Ec31, rmir=\Er, rmso=\EG0, rmxon=\Ec20\Ec30,
	rs2=\Eu\E~4\Ee6\EC\EDF\Ec21\Ec31\Ec62\Ec72\Ee;\016\E'\EeL\E`9\E\^0\E`1\E`4\Ee.\E`\:\Ee)\Ew\EwG\Ew0\Ee1\EG0\E(\Ed/\Ee4\Ed*\EO\E`I\Er\Ee"\Ec@0B\EcD\024,
	sgr=\E(\EG%{48}%?%p1%p3%O%t%{4}%+%;%?%p2%t%{8}%+%;%?%p4%t%{2}%+%;%?%p5%t%{64}%+%;%?%p7%t%{1}%+%;%c%?%p8%t\E)%;%?%p9%t\EcE%e\EcD%;,
	sgr0=\E(\EG0, smacs=\EcE, smam=\Ed/, smcup=\Ec20\Ec30,
	smir=\Eq, smso=\EG4, smxon=\Ec21\Ec31, tsl=\EF,

# This is the american terminal. Here tabs work.
# From: Francesco Potorti` <F.Potorti@cnuce.cnr.it>, 24 Aug 1998
wy99fa|wy99fgta|wy-99fgta|Wyse WY-99GT (US PC keyboard),
	hts=\E1, tbc=\E0, use=wy99f,

#
#	The Wyse 160 is combination of the WY-60 and the WY-99gt.
#	The reset strings are slow and the pad times very depending
#	on other parameters such as font loading.  I have tried
#	to follow the following outline:
#
#		<rs1> -> set personality
#		<rs2> -> set number of columns
#		<rs3> -> set number of lines
#		<is1> -> select the proper font
#		<is2> -> do the initialization
#		<is3> -> set up display memory (2 pages)
#
#	The display memory may be used for either text or graphics.
#	When "Display Memory = Shared" the terminal will have more pages
#	but garbage may be left on the screen when you switch from
#	graphics to text.  If "Display Memory = Unshared" then the
#	text area will be only one page long.
#
# (wy160: we use \E{ rather than ^^ for home (both are documented) to avoid
# a bug reported by Robert Dunn, <rcdii@inlink.com> -- esr)
wy160|wyse160|Wyse 160,
	am, bw, hs, km, mc5i, mir, msgr,
	cols#80, lh#1, lines#24, lw#8, nlab#8, wsl#38,
	acsc=+/\,.0[a2fxgqh1ihjYk?lZm@nEqDtCu4vAwBx3yszr{c~~,
	bel=^G, blink=\EG2, cbt=\EI, civis=\E`0, clear=\E+$<30>,
	cnorm=\E`1, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW$<5>,
	dclk=\E`b, dim=\EGp, dl1=\ER$<1>, dsl=\EF\r, ed=\EY$<30>,
	el=\ET$<5>, flash=\E`8$<100/>\E`9, fsl=^M, home=\E{, ht=^I,
	hts=\E1, il1=\EE$<1>, ind=\n$<1>, ip=$<2>, is1=\EcB0\EcC1,
	is2=\Ed$\EcD\E'\Er\EH\003\Ed/\EO\Ee1\Ed*\E`@\E`9\E`1\016\024\El,
	is3=\Ew0$<100>, kHOM=\E{, kbs=^H, kcbt=\EI, kcub1=^H,
	kcud1=^J, kcuf1=^L, kcuu1=^K, kdch1=\EW, kdl1=\ER, ked=\EY,
	kel=\ET, kent=\E7, kf1=^A@\r, kf10=^AI\r, kf11=^AJ\r,
	kf12=^AK\r, kf13=^AL\r, kf14=^AM\r, kf15=^AN\r, kf16=^AO\r,
	kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r,
	kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^, kich1=\EQ,
	kil1=\EE, knp=\EK, kpp=\EJ, kprt=\EP, krpl=\Er, ll=\E{^K,
	mc0=\EP, mc4=^T, mc5=\Ed#, nel=\r\n$<1>,
	pfloc=\EZ2%p1%{63}%+%c%p2%s\177,
	pfx=\EZ1%p1%{63}%+%c%p2%s\177,
	pln=\Ez%p1%{47}%+%c%p2%s\r, prot=\E), ri=\Ej$<1>,
	rmacs=\EcD, rmam=\Ed., rmclk=\E`c, rmcup=\Ew0, rmir=\Er,
	rmln=\EA11, rmxon=\Ec20, rs1=\E~!\E~4$<70>,
	rs2=\E`\:$<100>, rs3=\EwG\Ee($<140>,
	sgr=%?%p8%t\E)%e\E(%;%?%p9%t\EcE%e\EcD%;\EG%{48}%?%p2%t%{8}%|%;%?%p1%p3%|%p6%|%t%{4}%|%;%?%p4%t%{2}%|%;%?%p1%p5%|%t%{64}%|%;%?%p7%t%{1}%|%;%c,
	sgr0=\E(\EH\003\EG0\EcD, smacs=\EcE, smam=\Ed/,
	smcup=\Ew1, smir=\Eq, smln=\EA10, smso=\EGt, smxon=\Ec21,
	tbc=\E0, tsl=\EF, use=adm+sgr,
#
wy160-w|wyse160-w|wyse 160 132-column,
	cols#132, lw#7, nlab#16, wsl#90,
	cup=\Ea%i%p1%dR%p2%dC, dch1=\EW$<9>,
	rs2=\EeF$<150>\E`;$<150>, use=wy160,
#
wy160-25|wyse160-25|wyse 160 80-column 25-lines,
	lh@, lines#25, lw@, nlab@,
	pln@, rs3=\EwG\Ee)$<200>, use=wy160,
wy160-25-w|wyse160-25-w|wyse 160 132-column 25-lines,
	lh@, lines#25, lw@, nlab@,
	pln@, rs3=\EwG\Ee)$<200>, use=wy160-w,
#
wy160-42|wyse160-42|wyse 160 80-column 42-lines,
	lines#42,
	clear=\E+$<50>, dl1=\ER$<2>, ed=\Ey$<50>, il1=\EE$<2>,
	ind=\n$<2>, is1=\EcB2\EcC3, nel=\r\n$<2>, ri=\Ej$<2>,
	rs3=\Ee*$<150>, use=wy160,
wy160-42-w|wyse160-42-w|wyse 160 132-column 42-lines,
	cols#132, lw#7, nlab#16, wsl#90,
	cup=\Ea%i%p1%dR%p2%dC, dch1=\EW$<8>, ip=$<3>,
	rs2=\EeF$<150>\E`;$<150>, use=wy160-42,
#
wy160-43|wyse160-43|wyse 160 80-column 43-lines,
	lh@, lines#43, lw@, nlab@,
	pln@, rs3=\Ee+$<150>, use=wy160-42,
wy160-43-w|wyse160-43-w|wyse 160 132-column 43-lines,
	lh@, lines#43, lw@, nlab@,
	pln@, rs3=\Ee+$<150>, use=wy160-42-w,
#
wy160-vb|wyse160-vb|Wyse 160 visible bell,
	bel@, use=wy160,
wy160-w-vb|wy160-wvb|wyse160-wvb|Wyse 160 132-column visible bell,
	bel@, use=wy160-w,
#
#	The Wyse 75 is a vt100 lookalike without advanced video.
#
#	   The Wyse 75 can support one attribute (e.g. Dim, Inverse,
#	Underline) without magic cookies.  The following description
#	uses this capability, but when more than one attribute is
#	put on the screen at once, all attributes will be changed
#	to be the same as the last attribute given.
#	   The Wyse 75 can support more attributes when used with magic
#	cookies.  The wy75-mc terminal description uses magic cookies
#	to correctly handle multiple attributes on a screen.
#
wy75|wyse75|wyse 75,
	am, hs, mc5i, mir, msgr, xenl, xon,
	cols#80, lines#24, ma#1, pb#1201, wsl#78,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, cbt=\E[Z, civis=\E[?25l, clear=\E[H\E[J$<30>,
	cnorm=\E[?25h, cr=^M, csr=\E[%i%p1%d;%p2%dr$<2>,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP$<3*>,
	dch1=\E[P$<3>, dim=\E[0t\E[2m, dl=\E[%p1%dM$<1*>,
	dl1=\E[M, dsl=\E[>\,\001\001\E[>-\001\001,
	ech=\E[%p1%dX, ed=\E[J$<30>, el=\E[K$<3>, el1=\E[1K$<3>,
	enacs=\E)0, flash=\E[30h\E\,\E[30l$<250>, fsl=^A,
	home=\E[H, hpa=\E[%i%p1%dG, ht=^I, hts=\EH,
	ich=\E[%p1%d@$<1*>, il=\E[%p1%dL$<2*>, il1=\E[L$<2>,
	ind=\n$<2>, ip=$<1>,
	is1=\E[2;4;20;30l\E[?1;10l\E[12h\E[?7;8;25h,
	is2=\E>\E(B\E)0\017, is3=\E[m, kbs=^H, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdl1=\E[M, kel=\E[K,
	kf1=\E[?5i, kf10=\E[21~, kf11=\E[23~, kf12=\E[24~,
	kf13=\E[25~, kf14=\E[26~, kf15=\E[28~, kf16=\E[29~,
	kf17=\E[31~, kf18=\E[32~, kf19=\E[33~, kf2=\E[?3i,
	kf20=\E[34~, kf21=\E[35~, kf3=\E[2i, kf4=\E[@, kf5=\E[M,
	kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, kfnd=\E[1~,
	khlp=\E[28~, khome=\E[H, kich1=\E[@, kil1=\E[L, knp=\E[6~,
	kpp=\E[5~, kprt=\E[?5i, kslt=\E[4~, mc0=\E[0i, mc4=\E[4i,
	mc5=\E[5i, rc=\E8, rev=\E[1t\E[7m, ri=\EM$<2>, rmacs=^O,
	rmam=\E[?7l, rmir=\E[4l, rmkx=\E>, rmso=\E[m, rmul=\E[m,
	rs1=\E[13l\E[3l\E!p, rs2=\E[35h\E[?3l$<80>, rs3=\E[?5l,
	sc=\E7,
	sgr=%?%p5%t\E[0t%;%?%p3%p1%|%t\E[1t%;%?%p2%t\E[2t%;%?%p4%t\E[3t%;%?%p1%p2%p3%p4%p5%|%|%|%|%t\E[7m%e\E[m%;%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1l\E[?7h\E=, smso=\E[1t\E[7m, smul=\E[2t\E[4m,
	tbc=\E[3g, tsl=\E[>\,\001, use=vt220+keypad,
#
#	This terminal description uses the non-hidden attribute mode
#	(with magic cookie).
#
wy75-mc|wyse75-mc|wyse 75 with magic cookies,
	msgr@,
	ma@, xmc#1,
	blink=\E[2p, dim=\E[1p, invis=\E[4p, is3=\E[m\E[p,
	rev=\E[16p, rmacs=\E[0p\017, rmso=\E[0p, rmul=\E[0p,
	sgr=\E[%{0}%?%p2%p6%|%t%{8}%|%;%?%p1%p3%|%p6%|%t%{16}%|%;%?%p4%t%{2}%|%;%?%p1%p5%|%t%{1}%|%;%?%p7%t%{4}%|%;%dp%?%p9%t\016%e\017%;,
	sgr0=\E[0p\017, smacs=\E[0p\016, smso=\E[17p, smul=\E[8p,
	use=wy75,
wy75-vb|wyse75-vb|wyse 75 with visible bell,
	pb@,
	bel@, use=wy75,
wy75-w|wyse75-w|wyse 75 in 132 column mode,
	cols#132, wsl#130,
	rs2=\E[35h\E[?3h$<80>, use=wy75,
wy75-wvb|wyse75-wvb|wyse 75 with visible bell 132 columns,
	pb@,
	bel@, use=wy75-w,
#
#	Wyse 85 emulating a vt220 7 bit mode.
#		24 line screen with status line.
#
#	The vt220 mode permits more function keys but it wipes out
#	the escape key.  I strongly recommend that <f11> be set to
#	escape (esc).
#	The terminal may have to be set for 8 data bits and 2 stop
#	bits for the arrow keys to work.
#	The Wyse 85 runs faster with XON/XOFF enabled.  Also the
#	<dch> and <ich> work best when XON/XOFF is set.  <ich> and
#	<dch> leave trash on the screen when used without XON/XOFF.
#
wy85|wyse85|wyse 85,
	am, hs, mc5i, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J$<110>, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH$<1>, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP$<3*>, dch1=\E[P$<3>, dim=\E[2m,
	dl=\E[%p1%dM$<3*>, dl1=\E[M$<3>, dsl=\E[40l,
	ech=\E[%p1%dX, ed=\E[J$<110>, el=\E[K$<1>, el1=\E[1K,
	enacs=\E)0, flash=\E[30h\E\,\E[30l$<300>,
	fsl=\E[1;24r\E8, home=\E[H, ht=\011$<1>, hts=\EH,
	ich=\E[%p1%d@$<4*>, il=\E[%p1%dL$<5*>, il1=\E[L$<5>,
	ind=\n$<3>, invis=\E[8m, ip=$<3>, is1=\E[62;1"p\E[?5W,
	is2=\E[2;4;20;30l\E[?1;4;10;16l\E[12h\E[?7;8;25h$<16>,
	is3=\E>\E(B\E)0\017\E[m, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf13=\E[25~, kf14=\E[26~,
	kf15=\E[28~, kf16=\E[29~, kf17=\E[31~, kf18=\E[32~,
	kf19=\E[33~, kf20=\E[34~, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, kfnd=\E[1~, khlp=\E[28~,
	khome=\E[26~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~,
	kslt=\E[4~, lf1=PF1, lf2=PF2, lf3=PF3, lf4=PF4, mc0=\E[0i,
	mc4=\E[4i, mc5=\E[5i, rc=\E8, rev=\E[7m, ri=\EM$<3>,
	rmacs=^O, rmam=\E[?7l, rmir=\E[4l, rmkx=\E>, rmso=\E[m,
	rmul=\E[m, rs1=\E[13l\E[3l\E!p, rs2=\E[35h\E[?3l$<70>,
	rs3=\E[?5l, sc=\E7,
	sgr=\E[0%?%p2%t;4%;%?%p3%p1%|%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1l\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	tsl=\E[40h\E7\E[25;%i%p1%dH, use=vt220+keypad,
#
#	Wyse 85 with visual bell.
wy85-vb|wyse85-vb|wyse 85 with visible bell,
	bel@, flash=\E[30h\E\,\E[30l$<300>, use=wy85,
#
#	Wyse 85 in 132-column mode.
wy85-w|wyse85-w|wyse 85 in 132-column mode,
	cols#132, wsl#132,
	rs2=\E[35h\E[?3h$<70>, use=wy85,
#
#	Wyse 85 in 132-column mode with visual bell.
wy85-wvb|wyse85-wvb|wyse 85 with visible bell 132-columns,
	bel@, use=wy85-w,

# From: Kevin Turner <kevint@aracnet.com>, 12 Jul 1998
# This copes with an apparent firmware bug in the wy85.  He writes:
# "What I did was change leave the terminal cursor keys set to Normal
# (instead of application), and change \E[ to \233 for all the keys in
# terminfo. At one point, I found some reference indicating that this
# terminal bug (not sending \E[) was acknowledged by Wyse (so it's not just
# me), but I can't find that and the server under my bookmark to "Wyse
# Technical" isn't responding.  So there's the question of wether the wy85
# terminfo should reflect the manufactuer's intended behaviour of the terminal
# or the actual."
wy85-8bit|wyse85-8bit|wyse 85 in 8-bit mode,
	am, hs, mc5i, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J$<110>, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH$<1>, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP$<3*>, dch1=\E[P$<3>, dim=\E[2m,
	dl=\E[%p1%dM$<3*>, dl1=\E[M$<3>, dsl=\E[40l,
	ech=\E[%p1%dX, ed=\E[J$<110>, el=\E[K$<1>, el1=\E[1K,
	enacs=\E)0, flash=\E[30h\E\,\E[30l$<300>,
	fsl=\E[1;24r\E8, home=\E[H, ht=\011$<1>, hts=\EH,
	ich=\E[%p1%d@$<4*>, il=\E[%p1%dL$<5*>, il1=\E[L$<5>,
	ind=\n$<3>, invis=\E[8m, ip=$<3>, is1=\E[62;1"p\E[?5W,
	is2=\E[2;4;20;30l\E[?1;4;10;16l\E[12h\E[?7;8;25h$<16>,
	is3=\E>\E(B\E)0\017\E[m, ka1=\EOw, ka3=\EOy, kb2=\EOu,
	kbs=^H, kc1=\EOq, kc3=\EOs, kcub1=\233D, kcud1=\233B,
	kcuf1=\233C, kcuu1=\233A, kdch1=\2333~, kent=\EOM,
	kf1=\EOP, kf10=\23321~, kf11=\23323~, kf12=\23324~,
	kf13=\23325~, kf14=\23326~, kf15=\23328~, kf16=\23329~,
	kf17=\23331~, kf18=\23332~, kf19=\23333~, kf2=\EOQ,
	kf20=\23334~, kf3=\EOR, kf4=\EOS, kf6=\23317~, kf7=\23318~,
	kf8=\23319~, kf9=\23320~, kfnd=\2331~, khlp=\23328~,
	khome=\23326~, kich1=\2332~, knp=\2336~, kpp=\2335~,
	kslt=\2334~, lf1=PF1, lf2=PF2, lf3=PF3, lf4=PF4, mc0=\E[0i,
	mc4=\E[4i, mc5=\E[5i, rc=\E8, rev=\E[7m, ri=\EM$<3>,
	rmacs=^O, rmam=\E[?7l, rmir=\E[4l, rmkx=\E>, rmso=\E[m,
	rmul=\E[m, rs1=\E[13l\E[3l\E!p, rs2=\E[35h\E[?3l$<70>,
	rs3=\E[?5l, sc=\E7,
	sgr=\E[0%?%p2%t;4%;%?%p3%p1%|%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p7%t;8%;+m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1l\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	tsl=\E[40h\E7\E[25;%i%p1%dH,
#
#	Wyse 185 emulating a vt320 7 bit mode.
#
#	This terminal always displays 25 lines.  These lines may be used
#	as 24 data lines and a terminal status line (top or bottom) or
#	25 data lines.  The 48 and 50 line modes change the page size
#	and not the number of lines on the screen.
#
#	The Compose Character key can be used as a meta key if changed
#	by set-up.
#
wy185|wyse185|wyse 185,
	am, hs, km, mc5i, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J$<40>, cnorm=\E[34h\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr$<20>, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[?25h\E[34l, dch=\E[%p1%dP$<3>, dch1=\E[P$<3>,
	dim=\E[2m, dl=\E[%p1%dM$<2*>, dl1=\E[M$<2>,
	dsl=\E7\E[99;0H\E[K\E8, ech=\E[%p1%dX, ed=\E[J$<40>,
	el=\E[K, el1=\E[1K, enacs=\E)0,
	flash=\E[30h\E\,\E[30l$<100>, fsl=\E[1;24r\E8,
	home=\E[H, hpa=\E[%i%p1%d`, ht=^I, hts=\EH,
	ich=\E[%p1%d@$<2>, il=\E[%p1%dL$<3*>, il1=\E[L$<3>,
	ind=\n$<2>, invis=\E[8m, ip=$<4>, is1=\E[?5W,
	is2=\E[2;4;20;30l\E[?1;4;10;16l\E[12h\E[?7;8;25h,
	is3=\E>\E(B\E)0\017\E[m, kbs=^H, kcbt=\E[Z, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~, kf1=\EOP,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf13=\E[25~,
	kf14=\E[26~, kf15=\E[28~, kf16=\E[29~, kf17=\E[31~,
	kf18=\E[32~, kf19=\E[33~, kf2=\EOQ, kf20=\E[34~, kf3=\EOR,
	kf4=\EOS, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	kfnd=\E[1~, khlp=\E[28~, khome=\E[26~, kich1=\E[2~,
	knp=\E[6~, kpp=\E[5~, kslt=\E[4~, lf1=PF1, lf2=PF2, lf3=PF3,
	lf4=PF4, mc0=\E[0i, mc4=\E[4i, mc5=\E[5i, rc=\E8, rev=\E[7m,
	ri=\EM$<2>, rmacs=^O, rmam=\E[?7l, rmcup=\E[ R, rmir=\E[4l,
	rmkx=\E>, rmso=\E[27m, rmul=\E[24m,
	rs1=\E[13l\E[3l\E\\\E[63;1"p\E[!p, rs2=\E[35h\E[?3l,
	rs3=\E[?5l\E[47h\E[40l\E[r, sc=\E7,
	sgr=\E[0%?%p2%t;4%;%?%p3%p1%|%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smcup=\E[ Q,
	smir=\E[4h, smkx=\E[?1l\E=, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g, tsl=\E7\E[99;%i%p1%dH, vpa=\E[%i%p1%dd,
	use=vt220+keypad,
#
#	Wyse 185 with 24 data lines and top status (terminal status)
wy185-24|wyse185-24|wyse 185 with 24 data lines,
	hs@,
	dsl@, fsl@, rs3=\E[?5l\E[47h\E[40l\E[1;24r, tsl@,
	use=wy185,
#
#	Wyse 185 with visual bell.
wy185-vb|wyse185-vb|wyse 185+flash,
	bel@, use=wy185,
#
#	Wyse 185 in 132-column mode.
wy185-w|wyse185-w|wyse 185 in 132-column mode,
	cols#132, wsl#132,
	dch=\E[%p1%dP$<7>, dch1=\E[P$<7>, ich=\E[%p1%d@$<7>,
	ip=$<7>, rs2=\E[35h\E[?3h, use=wy185,
#
#	Wyse 185 in 132-column mode with visual bell.
wy185-wvb|wyse185-wvb|wyse 185+flash+132 cols,
	bel@, use=wy185-w,

# wy325 terminfo entries
# Done by Joe H. Davis        3-9-92

# lines 25  columns 80
#
wy325|wyse325|Wyse epc,
	am, bw, hs, mc5i, mir,
	cols#80, lh#1, lines#24, lw#8, nlab#8, pb#9601, wsl#45,
	acsc=+/\,.0[a2fxgqh1ihjYk?lZm@nEqDtCu4vAwBx3yszr{c~~,
	bel=^G, blink=\EG2, cbt=\EI, civis=\E`0, clear=\E+$<50>,
	cnorm=\E`1, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW$<7>,
	dim=\EGp, dl1=\ER$<3>, dsl=\EF\r, ed=\EY$<50>, el=\ET$<4>,
	flash=\E`8$<100/>\E`9, fsl=^M, home=^^, ht=^I, hts=\E1,
	il1=\EE$<3>, ind=\n$<3>, ip=$<2>, is1=\EcB0\EcC1,
	is2=\EcD\E'\Er\EH\003\Ed/\EO\Ee1\Ed*\E`@\E`9\E`1\016\024\El,
	is3=\Ew0$<16>, kHOM=\E{, kbs=^H, kcbt=\EI, kcub1=^H,
	kcud1=^J, kcuf1=^L, kcuu1=^K, kdch1=\EW, kdl1=\ER, ked=\EY,
	kel=\ET, kent=\E7, kf1=^A@\r, kf10=^AI\r, kf11=^AJ\r,
	kf12=^AK\r, kf13=^AL\r, kf14=^AM\r, kf15=^AN\r, kf16=^AO\r,
	kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r,
	kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^, kich1=\Eq,
	kil1=\EE, knp=\EK, kpp=\EJ, kprt=\EP, krpl=\Er, ll=^^^K,
	mc0=\EP, mc4=^T, mc5=\Ed#,
	pfloc=\EZ2%p1%{63}%+%c%p2%s\177,
	pfx=\EZ1%p1%{63}%+%c%p2%s\177,
	pln=\Ez%p1%{47}%+%c%p2%s\r, prot=\E), ri=\Ej$<2>,
	rmacs=\EcD, rmam=\Ed., rmcup=\Ew0, rmir=\Er, rmln=\EA11,
	rs1=\E~!\E~4$<30>, rs2=\EeF\E`\:$<70>,
	rs3=\EwG\Ee($<100>,
	sgr=%?%p8%t\E)%e\E(%;%?%p9%t\EcE%e\EcD%;\EG%{48}%?%p2%t%{8}%|%;%?%p1%p3%|%p6%|%t%{4}%|%;%?%p4%t%{2}%|%;%?%p1%p5%|%t%{64}%|%;%?%p7%t%{1}%|%;%c,
	sgr0=\E(\EH\003\EG0\EcD, smacs=\EcE, smam=\Ed/,
	smcup=\Ew1, smir=\Eq, smln=\EA10, smso=\EGt, tbc=\E0,
	tsl=\EF, use=adm+sgr,

#
# lines 24  columns 80  vb
#
wy325-vb|wyse325-vb|wyse-325 with visual bell,
	bel@, use=wy325,

#
# lines 24  columns 132
#
wy325-w|wyse325-w|wy325w-24|wyse-325 in wide mode,
	cols#132, lw#7, nlab#16, wsl#97,
	cup=\Ea%i%p1%dR%p2%dC, dch1=\EW$<12>, ip=$<4>,
	rs2=\E`;$<70>, use=wy325,
#
# lines 25  columns 80
#
wy325-25|wyse325-25|wy325-80|wyse-325|wyse-325 25 lines,
	lh@, lines#25, lw@, nlab@,
	pln@, rs3=\EwG\Ee)$<100>, use=wy325,
#
# lines 25  columns 132
#
wy325-25w|wyse325-25w|wy325 132 columns,
	lh@, lines#25, lw@, nlab@,
	pln@, rs3=\EwG\Ee)$<100>, use=wy325-w,
#
# lines 25  columns 132  vb
#
wy325-w-vb|wy325-wvb|wyse325-wvb|wyse-325 wide mode reverse video,
	bel@, use=wy325-w,

#
# lines 42  columns 80
#
wy325-42|wyse325-42|wyse-325 42 lines,
	lh@, lines#42, lw@, nlab@,
	pln@, rs3=\EwG\Ee)$<100>, use=wy325,
#
# lines 42  columns 132
#
wy325-42w|wyse325-42w|wyse-325 42 lines wide mode,
	lh@, lines#42, lw@, nlab@,
	pln@, rs3=\EwG\Ee)$<100>, use=wy325-w,
#
# lines 42  columns 132  vb
#
wy325-42w-vb|wy325-42wvb|wyse-325 42 lines wide mode visual bell,
	bel@, use=wy325-w,
#
# lines 43  columns 80
#
wy325-43|wyse325-43|wyse-325 43 lines,
	lh@, lines#43, lw@, nlab@,
	pln@, use=wy325,
#
# lines 43  columns 132
#
wy325-43w|wyse325-43w|wyse-325 43 lines wide mode,
	lh@, lines#43, lw@, nlab@,
	pln@, rs3=\EwG\Ee)$<100>, use=wy325-w,
#
# lines 43  columns 132  vb
#
wy325-43w-vb|wy325-43wvb|wyse-325 43 lines wide mode visual bell,
	bel@, use=wy325-w,

#	Wyse 370 -- 24 line screen with status line.
#
#	The terminal may have to be set for 8 data bits and 2 stop
#	bits for the arrow keys to work.
#
#	If you change keyboards the terminal will send different
#	escape sequences.
#	The following definition is for the basic terminal without
#	function keys.
#
#	<u0> -> enter Tektronix 4010/4014 mode
#	<u1> -> exit  Tektronix 4010/4014 mode
#	<u2> -> enter ASCII mode (from any ANSI mode)
#	<u3> -> exit  ASCII mode (goto native ANSI mode)
#	<u4> -> enter Tek 4207 ANSI mode (from any ANSI mode)
#	<u5> -> exit  Tek 4207 mode (goto native ANSI mode)
#
# Bug: The <op> capability resets attributes.
wy370-nk|wyse 370 without function keys,
	am, ccc, hs, mc5i, mir, msgr, xenl, xon,
	colors#64, cols#80, it#8, lines#24, ncv#48, pairs#64, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J$<40>, cnorm=\E[34h\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH$<1>, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[?25h\E[34l, dch=\E[%p1%dP$<1*>, dch1=\E[P$<1>,
	dclk=\E[31h, dim=\E[2m, dl=\E[%p1%dM$<2*>, dl1=\E[M$<2>,
	dsl=\E[40l, ech=\E[%p1%dX$<.1*>, ed=\E[J$<40>,
	el=\E[K$<10>, el1=\E[1K$<12>, enacs=\E)0,
	flash=\E[30h\E\,\E[30l$<300>, fsl=\E[1;24r\E8,
	home=\E[H, hpa=\E[%i%p1%d`, ht=\011$<1>, hts=\EH,
	ich=\E[%p1%d@$<1*>, il=\E[%p1%dL$<2*>, il1=\E[L$<2>,
	ind=\n$<2>,
	initc=\E[66;%p1%d;%?%p2%{250}%<%t%{0}%e%p2%{500}%<%t%{16}%e%p2%{750}%<%t%{32}%e%{48}%;%?%p3%{250}%<%t%{0}%e%p3%{500}%<%t%{4}%e%p3%{750}%<%t%{8}%e%{12}%;%?%p4%{250}%<%t%{0}%e%p4%{500}%<%t%{1}%e%p4%{750}%<%t%{2}%e%{3}%;%{1}%+%+%+%dw,
	invis=\E[8m, ip=$<1>, is1=\E[90;1"p\E[?5W$<6>,
	is2=\E[2;4;20;30;40l\E[?1;10;16l\E[12h\E[?7;8;25h,
	is3=\E>\017\E)0\E(B\E[63;0w\E[m, mc0=\E[0i, mc4=\E[4i,
	mc5=\E[5i,
	oc=\E[60w\E[63;0w\E[66;1;4w\E[66;2;13w\E[66;3;16w\E[66;4;49w\E[66;5;51w\E[66;6;61w\E[66;7;64w,
	op=\E[m, rc=\E8, rev=\E[7m, ri=\EM$<2>, rmacs=^O,
	rmam=\E[?7l, rmclk=\E[31l, rmcup=\E[ R, rmir=\E[4l,
	rmkx=\E>, rmso=\E[27m, rmul=\E[24m,
	rs1=\E[13l\E[3l\E!p\E[?4i, rs2=\E[35h\E[?3l$<8>,
	rs3=\E[?5l, sc=\E7, setb=\E[62;%p1%dw, setf=\E[61;%p1%dw,
	sgr=\E[0%?%p2%t;4%;%?%p3%p1%|%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smcup=\E[ Q,
	smir=\E[4h, smkx=\E[?1l\E=, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g, tsl=\E[40l\E[40h\E7\E[99;%i%p1%dH,
	u0=\E[?38h\E8, u1=\E[?38l\E)0, u2=\E[92;52"p, u3=\E~B,
	u4=\E[92;76"p, u5=\E%!1\E[90;1"p, vpa=\E[%i%p1%dd,
#
#	Function key set for the ASCII (wy-50 compatible) keyboard
#	This is the default 370.
#
wy370|wyse370|wy370-101k|Wyse 370 with 101 key keyboard,
	kbs=^H, kcbt=\E[Z, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kdch1=\EOQ, kdl1=\EOQ, kent=\EOM, kf1=\E[?4i,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf13=\E[25~,
	kf14=\E[26~, kf15=\E[28~, kf16=\E[29~, kf2=\E[?3i,
	kf3=\E[2i, kf4=\E[@, kf5=\E[M, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, khome=\E[H, kich1=\EOP, kil1=\EOP,
	knp=\E[U, kpp=\E[V, use=wy370-nk,
#
#	Function key set for the VT-320 (and wy85) compatible keyboard
#
wy370-105k|Wyse 370 with 105 key keyboard,
	kbs=^H, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[3~, kf10=\E[21~, kf11=\E[23~, kf12=\E[24~,
	kf13=\E[25~, kf14=\E[26~, kf15=\E[28~, kf16=\E[29~,
	kf17=\E[31~, kf18=\E[32~, kf19=\E[33~, kf20=\E[34~,
	kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, kfnd=\E[1~,
	khlp=\E[28~, khome=\E[26~, kich1=\E[2~, knp=\E[6~,
	kpp=\E[5~, kslt=\E[4~, lf1=PF1, lf2=PF2, lf3=PF3, lf4=PF4,
	use=wy370-nk, use=vt220+keypad,
#
#	Function key set for the PC compatible keyboard
#
wy370-EPC|Wyse 370 with 102 key keyboard,
	kbs=^H, kcbt=\E[Z, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kend=\E[1~, kent=\EOM, kf1=\EOP, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf2=\EOQ, kf3=\EOR, kf4=\EOS,
	kf5=\E[M, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[H, kich1=\E[2~, knp=\E[U, kpp=\E[V, use=wy370-nk,
#
#	Wyse 370 with visual bell.
wy370-vb|Wyse 370 with visible bell,
	bel@, use=wy370,
#
#	Wyse 370 in 132-column mode.
wy370-w|Wyse 370 in 132-column mode,
	cols#132, wsl#132,
	rs2=\E[35h\E[?3h$<70>, use=wy370,
#
#	Wyse 370 in 132-column mode with visual bell.
wy370-wvb|Wyse 370 with visible bell 132-columns,
	flash=\E[30h\E\,\E[30l$<300>, use=wy370-w,
wy370-rv|Wyse 370 reverse video,
	rs3=\E[32h\E[?5h, use=wy370,
#
#	Wyse 99gt Tektronix 4010/4014 emulator,
#
wy99gt-tek|Wyse 99gt Tektronix 4010/4014 emulator,
	am, os,
	cols#74, lines#35,
	bel=^G, clear=\E^L, cr=^M, cub1=^H, cud1=^J, cuf1=\s,
	cup=\035%{3040}%{89}%p1%*%-%Py%p2%{55}%*%Px%gy%{128}%/%{31}%&%{32}%+%c%gy%{3}%&%{4}%*%gx%{3}%&%+%{96}%+%c%gy%{004}%/%{31}%&%{96}%+%c%gx%{128}%/%{31}%&%{32}%+%c%gx%{004}%/%{31}%&%{64}%+%c\037,
	cuu1=^K, ff=^L,
	hd=\036HHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHH\037,
	home=^]7`x @\037,
	hu=\036DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD\037,
	is2=\E8, nel=^M^J, u0=\E~>\E8, u1=\E[42h,
#
#	Wyse 160 Tektronix 4010/4014 emulator,
#
wy160-tek|Wyse 160 Tektronix 4010/4014 emulator,
	cup=\035%{3103}%{91}%p1%*%-%Py%p2%{55}%*%Px%gy%{128}%/%{31}%&%{32}%+%c%gy%{3}%&%{4}%*%gx%{3}%&%+%{96}%+%c%gy%{004}%/%{31}%&%{96}%+%c%gx%{128}%/%{31}%&%{32}%+%c%gx%{004}%/%{31}%&%{64}%+%c\037,
	home=^]8`g @\037, use=wy99gt-tek,
#
#	Wyse 370 Tektronix 4010/4014 emulator,
#
wy370-tek|Wyse 370 Tektronix 4010/4014 emulator,
	am, os,
	cols#80, lines#36,
	bel=^G, clear=\E^L, cr=^M, cub1=^H, cud1=^J, cuf1=\s,
	cup=\035%{775}%{108}%p1%*%{5}%/%-%Py%p2%{64}%*%{4}%+%{5}%/%Px%gy%{32}%/%{31}%&%{32}%+%c%gy%{31}%&%{96}%+%c%gx%{32}%/%{31}%&%{32}%+%c%gx%{31}%&%{64}%+%c\037,
	cuu1=^K, ff=^L,
	hd=\036HHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHH\037,
	home=^]8g @\037,
	hu=\036DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD\037,
	is2=\E8, kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^I, kcuu1=^K,
	nel=^M^J, u0=\E[?38h\E8, u1=\E[?38l\E)0,

# Vendor-supplied Wyse entries end here.

#
#TITLE:  TERMINFO ENTRY WY520
#DATE:   8/5/93
# The WY520 terminfo is based on the WY285 entry published on the WYSE
# BBS with the addition of more function keys and special keys.
#
#               rs1 -> set personality
#               rs2 -> set number of columns
#               rs3 -> set number of lines
#               is1 -> select the proper font
#               is2 -> do the initialization
#               is3 -> If this string is empty then rs3 gets sent.
#
#       Wyse 520 emulating a vt420 7 bit mode with default ANSI keyboard
#       - The BS key is programmed to generate BS in smcup since
#         is2 doesn't seem to work.
#       - Remove and shift/Remove: delete a character
#       - Insert : enter insert mode
#       - Find   : delete to end of file
#       - Select : clear a line
#       - F11, F12, F13: send default sequences (not ESC, BS, LF)
#       - F14 : Home key
#       - Bottom status line (host writable line) is used.
#       - smkx,rmkx are removed because this would put the numeric
#         keypad in Dec application mode which doesn't seem to work
#         with SCO applications.
#
wy520|wyse520|wyse 520,
	am, hs, km, mc5i, mir, xenl, xon,
	cols#80, it#8, lines#24, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J$<40>, cnorm=\E[34h\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr$<20>, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[?25h\E[34l, dch=\E[%p1%dP$<3>, dch1=\E[P$<30>,
	dim=\E[2m, dl=\E[%p1%dM$<2*>, dl1=\E[M$<2>, dsl=\E[0$~,
	ech=\E[%p1%dX, ed=\E[J$<40>, el=\E[K, el1=\E[1K,
	enacs=\E)0, fsl=\E[0$}, home=\E[H, hpa=\E[%i%p1%d`, ht=^I,
	hts=\EH, ich=\E[%p1%d@$<2>, il=\E[%p1%dL$<3*>,
	il1=\E[L$<3>, ind=\n$<2>, invis=\E[8m, ip=$<4>, is1=\E[?5W,
	is2=\E[2;4;20;30l\E[?1;4;10;16l\E[12h\E[?7;8;25;67h,
	is3=\E>\E(B\E)0\017\E[m, kbs=^H, kcbt=\E[Z, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~, ked=\E[1~,
	kel=\E[4~, kent=\EOM, kf10=\E[21~, kf11=\E[23~,
	kf12=\E[24~, kf13=\E[25~, kf14=\E[26~, kf15=\E[28~,
	kf16=\E[29~, kf17=\E[31~, kf18=\E[32~, kf19=\E[33~,
	kf20=\E[34~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~,
	kf9=\E[20~, kfnd=\E[1~, khlp=\E[28~, khome=\E[26~,
	kich1=\E[2~, knp=\E[6~, kpp=\E[5~, kslt=\E[4~, lf1=PF1,
	lf2=PF2, lf3=PF3, lf4=PF4, mc0=\E[0i, mc4=\E[4i, mc5=\E[5i,
	rc=\E8, rev=\E[7m, ri=\EM$<2>, rmacs=^O, rmam=\E[?7l,
	rmcup=\E[ R, rmir=\E[4l, rmso=\E[m, rmul=\E[24m,
	rs1=\E[13l\E[3l\E\\\E[63;1"p\E[!p, rs2=\E[35h\E[?3l,
	rs3=\E[?5l\E[47h\E[40l\E[r, sc=\E7,
	sgr=\E[0%?%p2%t;4%;%?%p3%p1%|%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h,
	smcup=\E[ Q\E[?67;8h, smir=\E[4h, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g, tsl=\E[2$~\E[1$}\E[%i%p1%d`,
	vpa=\E[%i%p1%dd, use=vt220+keypad,
#
#       Wyse 520 with 24 data lines and status (terminal status)
wy520-24|wyse520-24|wyse 520 with 24 data lines,
	hs@,
	dsl@, fsl@, rs3=\E[?5l\E[47h\E[40l\E[1;24r, tsl@,
	use=wy520,
#
#       Wyse 520 with visual bell.
wy520-vb|wyse520-vb|wyse 520 with visible bell,
	flash=\E[30h\E\,\E[30l$<100>, use=wy520,
#
#       Wyse 520 in 132-column mode.
wy520-w|wyse520-w|wyse 520 in 132-column mode,
	cols#132, wsl#132,
	dch=\E[%p1%dP$<7>, dch1=\E[P$<7>, ich=\E[%p1%d@$<7>,
	ip=$<7>, rs2=\E[35h\E[?3h, use=wy520,
#
#       Wyse 520 in 132-column mode with visual bell.
wy520-wvb|wyse520-wvb|wyse 520 with visible bell 132-columns,
	flash=\E[30h\E\,\E[30l$<100>, use=wy520-w,
#
#
#       Wyse 520 emulating a vt420 7 bit mode.
#       The DEL key is programmed to generate BS in is2.
#       With EPC keyboard.
#       - 'End' key will clear till end of line on EPC keyboard
#       - Shift/End : ignored.
#       - Insert : enter insert mode.
#       - Delete : delete a character (have to change interrupt character
#                  to CTRL-C: stty intr '^c') for it to work since the
#                  Delete key sends 7FH.
wy520-epc|wyse520-epc|wyse 520 with EPC keyboard,
	kdch1=\177, kel=\E[4~, kend=\E[4~, kf0=\E[21~, kf1=\E[11~,
	kf2=\E[12~, kf3=\E[13~, kf4=\E[14~, kf5=\E[15~, khome=\E[H,
	use=wy520,
#
#       Wyse 520 with 24 data lines and status (terminal status)
#       with EPC keyboard.
wy520-epc-24|wyse520-pc-24|wyse 520 with 24 data lines and EPC keyboard,
	hs@,
	dsl@, fsl@, rs3=\E[?5l\E[47h\E[40l\E[1;24r, tsl@,
	use=wy520-epc,
#
#       Wyse 520 with visual bell.
wy520-epc-vb|wyse520-pc-vb|wyse 520 with visible bell and EPC keyboard,
	flash=\E[30h\E\,\E[30l$<100>, use=wy520-epc,
#
#       Wyse 520 in 132-column mode.
wy520-epc-w|wyse520-epc-w|wyse 520 in 132-column mode with EPC keyboard,
	cols#132, wsl#132,
	dch=\E[%p1%dP$<7>, dch1=\E[P$<7>, ich=\E[%p1%d@$<7>,
	ip=$<7>, rs2=\E[35h\E[?3h, use=wy520-epc,
#
#       Wyse 520 in 132-column mode with visual bell.
wy520-epc-wvb|wyse520-p-wvb|wyse 520 with visible bell 132-columns and EPC keyboard,
	flash=\E[30h\E\,\E[30l$<100>, use=wy520-epc-w,
#
#       Wyse 520 in 80-column, 36 lines
wy520-36|wyse520-36|wyse 520 with 36 data lines,
	hs@,
	lines#36,
	dsl@, fsl@, rs3=\E[?5l\E[36*|\E[36t\E[40l\E[1;36r, tsl@,
	use=wy520,
#
#       Wyse 520 in 80-column, 48 lines
wy520-48|wyse520-48|wyse 520 with 48 data lines,
	hs@,
	lines#48,
	dsl@, fsl@, rs3=\E[?5l\E[48*|\E[48t\E[40l\E[1;48r, tsl@,
	use=wy520,
#
#       Wyse 520 in 132-column, 36 lines
wy520-36w|wyse520-36w|wyse 520 with 132 columns and 36 data lines,
	cols#132, wsl#132,
	rs2=\E[?3h,
	rs3=\E[?5l\E[36*|\E[36t\E[40l\E[1;36r\E[132$|,
	use=wy520-36,
#
#       Wyse 520 in 132-column, 48 lines
wy520-48w|wyse520-48w|wyse 520 with 48 data lines,
	cols#132, wsl#132,
	rs2=\E[?3h,
	rs3=\E[?5l\E[48*|\E[48t\E[40l\E[1;48r\E[132$|,
	use=wy520-48,
#
#
#       Wyse 520 in 80-column, 36 lines with EPC keyboard
wy520-36pc|wyse520-36pc|wyse 520 with 36 data lines and EPC keyboard,
	hs@,
	lines#36,
	dsl@, fsl@, rs3=\E[?5l\E[36*|\E[36t\E[40l\E[1;36r, tsl@,
	use=wy520-epc,
#
#       Wyse 520 in 80-column, 48 lines with EPC keyboard
wy520-48pc|wyse520-48pc|wyse 520 with 48 data lines and EPC keyboard,
	hs@,
	lines#48,
	dsl@, fsl@, rs3=\E[?5l\E[48*|\E[48t\E[40l\E[1;48r, tsl@,
	use=wy520-epc,
#
#       Wyse 520 in 132-column, 36 lines with EPC keyboard
wy520-36wpc|wyse520-36wpc|wyse 520 with 36 data lines and EPC keyboard,
	cols#132, wsl#132,
	rs2=\E[?3h,
	rs3=\E[?5l\E[36*|\E[36t\E[40l\E[1;36r\E[132$|,
	use=wy520-36pc,
#
#       Wyse 520 in 132-column, 48 lines with EPC keyboard
wy520-48wpc|wyse520-48wpc|wyse 520 with 48 data lines and EPC keyboard,
	cols#132, wsl#132,
	rs2=\E[?3h,
	rs3=\E[?5l\E[48*|\E[48t\E[40l\E[1;48r\E[132$|,
	use=wy520-48pc,

# From: John Gilmore <hoptoad!gnu@lll-crg.arpa>
# (wyse-vp: removed <if=/usr/share/tabset/wyse-adds>, there's no such
# file and we don't know what <hts> is -- esr)
wyse-vp|Wyse 50 in ADDS Viewpoint emulation mode with "enhance" on,
	OTbs, am,
	cols#80, it#8, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=^F,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=^Z, dch1=\EW,
	dl1=\El, ed=\Ek, el=\EK, home=^A, ht=^I, il1=\EM, ind=^J,
	is2=\E`\:\E`9\017\Er, kbs=^H, kcub1=^U, kcud1=^J, kcuf1=^F,
	kcuu1=^Z, khome=^A, ll=^A^Z, nel=^M^J, rmir=\Er, rmso=^O,
	rmul=^O, rs1=\E`\:\E`9\017\Er, sgr0=^O, smir=\Eq, smso=^N,
	smul=^N,

wy75ap|wyse75ap|wy-75ap|wyse-75ap|Wyse WY-75 Applications and Cursor keypad,
	is2=\E[1;24r\E[?10;3l\E[?1;25h\E[4l\E[m\E(B\E=,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	khome=\EOH, rmkx=\E[?1l\E>$<10/>, smkx=\E[?1h\E=$<10/>,
	use=wy75,

# From: Eric Freudenthal <freudent@eric.ultra.nyu.edu>
wy100q|Wyse 100 for Quotron,
	OTbs,
	cols#80, lines#24, xmc#1,
	cbt=\EI, clear=^Z, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\EY, el=\ET, home=^^, il1=\EE, invis@,
	is2=\E`\:\0\EC\EDF\E0\E'\E(\EA21, kcub1=^H, kcud1=^J,
	kcuf1=^L, kcuu1=^K, ri=\Ej, rmir=\Er, smir=\Eq, use=adm+sgr,

#### Kermit terminal emulations
#
# Obsolete Kermit versions may be listed in the section describing obsolete
# non-ANSI terminal emulators later in the file.
#

# KERMIT standard all versions.
# Straight ascii keyboard. :sr=\EI: not avail. many versions + bug prone in vi.
# (kermit: removed obsolete ":ma=^Hh^Jj^Kk^Ll^^H:" -- esr)
# From: greg small <gts@populi.berkeley.edu> 9-25-84
kermit|standard kermit,
	OTbs,
	cols#80, lines#24,
	clear=\EE, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, ed=\EJ,
	el=\EK, home=\EH, is2=K0 Standard Kermit  9-25-84\n,
	kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, khome=^^,
kermit-am|standard kermit plus auto-margin,
	am,
	is2=K1 Standard Kermit plus Automatic Margins\n,
	use=kermit,
# IBMPC Kermit 1.2.
# Bugs: <ed>, <el>: do not work except at beginning of line!  <clear> does
# not work, but fake with :cl=\EH\EJ (since :cd=\EJ: works at beginning of
# line).
# From: greg small <gts@populi.berkeley.edu> 8-30-84
pckermit|pckermit12|UCB IBMPC Kermit 1.2,
	am,
	lines#25,
	clear=\EH\EJ, ed@, el@,
	is2=K2 UCB IBMPC Kermit 1.2  8-30-84\n, use=kermit,
# IBMPC Kermit 1.20
# Cannot use line 25, now acts funny like ansi special scrolling region.
# Initialization must escape from that region by cursor position to line 24.
# Cannot use character insert because 1.20 goes crazy if insert at col 80.
# Does not use :am: because autowrap is lost when kermit dropped and restarted.
# From: greg small <gts@populi.berkeley.edu> 12-19-84
pckermit120|UCB IBMPC Kermit 1.20,
	it#8, lines#24,
	cud1=\EB, cvvis=\EO\Eq\EEK3, dch1=\EN, dl1=\EM, ht=^I,
	il1=\EL,
	is2=\EO\Eq\EJ\EY7 K3 UCB IBMPC Kermit 1.20  12-19-84\n,
	rmir@, rmso=\Eq, smir@, smso=\Ep, use=kermit,
# MS-DOS Kermit 2.27 for the IBMPC
# Straight ascii keyboard. :sr=\EI: not avail. many versions + bug prone in vi.
# Cannot use line 25, now acts funny like ansi special scrolling region.
# Initialization must escape from that region by cursor position to line 24.
# Does not use am: because autowrap is lost when kermit dropped and restarted.
# Reverse video for standout like H19.
# (msk227: removed obsolete ":ma=^Hh^Jj^Kk^Ll^^H:" -- esr)
# From: greg small <gts@populi.berkeley.edu> 3-17-85
msk227|mskermit227|MS-DOS Kermit 2.27 for the IBMPC,
	OTbs, am@,
	cols#80, it#8, lines#24,
	clear=\EE, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA,
	cvvis=\EO\Eq\EG\EwK4, dch1=\EN, dl1=\EM, ed=\EJ, el=\EK,
	home=\EH, ht=^I, il1=\EL,
	is2=\EO\Eq\EG\Ew\EJ\EY7 K4 MS Kermit 2.27 for the IBMPC 3-17-85\n,
	kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, khome=^^, rc=\Ek,
	rmir=\EO, rmso=\Eq, sc=\Ej, smir=\E@, smso=\Ep,
# MS-DOS Kermit 2.27 with automatic margins
# From:	greg small <gts@populi.berkeley.edu> 3-17-85
msk227am|mskermit227am|UCB MS-DOS Kermit 2.27 with automatic margins,
	am,
	cvvis=\EO\Eq\EG\EvK5,
	is2=\EO\Eq\EG\Ev\EJ\EY7 K5 MS Kermit 2.27 +automatic margins 3-17-85\n,
	use=msk227,
# MS-DOS Kermit 2.27 UCB 227.14 for the IBM PC
# Automatic margins now default.  Use ansi <sgr> for highlights.
# Define function keys.
# (msk22714: removed obsolete ":kn#10:" -- esr)
# From: greg small <gts@populi.berkeley.edu> 3-17-85
msk22714|mskermit22714|UCB MS-DOS Kermit 2.27 UCB 227.14 IBM PC,
	am,
	bold=\E[1m, cvvis=\EO\Eq\EG\EvK6,
	is2=\EO\Eq\EG\Ev\EJ\EY7 K6 MS Kermit 2.27 UCB 227.14 IBM PC 3-17-85\n,
	kf0=\E0, kf1=\E1, kf2=\E2, kf3=\E3, kf4=\E4, kf5=\E5, kf6=\E6,
	kf7=\E7, kf8=\E8, kf9=\E9, rev=\E[7m, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m, smso=\E[1m, smul=\E[4m, use=mskermit227,
# This was designed for a VT320 emulator, but it is probably a good start
# at support for the VT320 itself.
# Please send changes with explanations to bug-gnu-emacs@prep.ai.mit.edu.
# (vt320-k3: I added <rmam>/<smam> based on the init string -- esr)
vt320-k3|MS-Kermit 3.00's vt320 emulation,
	am, eslok, hs, km, mir, msgr, xenl,
	cols#80, it#8, lines#49, pb#9600, vt#3,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[J, cmdch=\E, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	dsl=\E[0$~, ech=\E[%p1%dX, ed=\E[J, el=\E[K,
	flash=\E[?5h\E[?5l\E[?5h\E[?5l\E[?5h\E[?5l,
	fsl=\E[0$}, home=\E[H, hpa=\E[%i%p1%dG, ht=^I, hts=\EH,
	ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L, ind=^J,
	is2=\E>\E F\E[?1h\E[?7h\E[r\E[2$~, kbs=^H, kcub1=\EOD,
	kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kdl1=\E[3~, kf0=\E[21~,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf6=\E[17~,
	kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, kich1=\E[2~, knp=\E[6~,
	kpp=\E[5~, mc0=\E[0i, mc4=\E[4i, mc5=\E[5i, nel=^M^J, rc=\E8,
	rev=\E[7m, ri=\EM, rin=\E[%p1%dL, rmacs=\E(B, rmam=\E[?7l,
	rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[27m, rmul=\E[24m,
	rs1=\E(B\E)B\E>\E F\E[4;20l\E[12h\E[?1;5;6;38;42l\E[?7;25h\E[4i\E[?4i\E[m\E[r\E[2$~,
	sc=\E7, sgr0=\E[m, smacs=\E(0, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	tsl=\E[1$}\r\E[K, vpa=\E[%i%p1%dd,
# From: Joseph Gil <yogi@cs.ubc.ca> 13 Dec 1991
# ACS capabilities from Philippe De Muyter  <phdm@info.ucl.ac.be> 30 May 1996
# (I removed a bogus boolean :mo: and added <msgr>, <smam>, <rmam> -- esr)
vt320-k311|dec vt320 series as defined by kermit 3.11,
	am, eslok, hs, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[;H\E[2J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	dsl=\E[2$~\r\E[1$}\E[K\E[$}, ed=\E[J, el=\E[K,
	flash=\E[?5h\E[?5l, fsl=\E[$}, home=\E[H, ht=^I, hts=\EH,
	ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L$<3/>, ind=\ED,
	is2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h\E[1;24r\E[24;1H,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf6=\E[17~,
	kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, lf1=pf1, lf2=pf2,
	lf3=pf3, lf4=pf4, nel=^M\ED, rc=\E8, rev=\E[7m,
	rf=/usr/share/tabset/vt100, ri=\EM, rmacs=^O,
	rmam=\E[?7l, rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[27m,
	rmul=\E[24m, rs1=\E[?3l, sc=\E7, sgr0=\E[m, smacs=^N,
	smam=\E[?7h, smir=\E[4h, smkx=\E[?1h\E=, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g, tsl=\E[2$~\E[1$}\E[1;%dH,

######## NON-ANSI TERMINAL EMULATIONS
#

#### Avatar
#
# These entries attempt to describe Avatar, a terminal emulation used with
# MS-DOS bulletin-board systems.  It was designed to give ANSI-like
# capabilities, but with cheaper (shorter) control sequences.  Messy design,
# excessively dependent on PC idiosyncracies, but apparently rather popular
# in the BBS world.
#
# No color support.  Avatar doesn't fit either of the Tektronix or HP color
# models that terminfo knows about.  An Avatar color attribute is the
# low 7 bits of the IBM-PC display-memory attribute.  Bletch.
#
# I wrote these entries while looking at the Avatar spec.  I don't have
# the facilities to test them.  Let me know if they work, or don't.
#
# Avatar escapes not used by these entries (because maybe you're smarter
# and more motivated than I am and can figure out how to wrap terminfo
# around some of them, and because they are weird enough to be funny):
#				level 0:
# ^L		-- clear window/reset current attribute to default
# ^V^A%p1%c	-- set current color attribute, parameter decodes as follows:
#
#      bit:         6   5   4   3   2   1   0
#                   |       |   |   |       |
#                   +---+---+   |   +---+---+
#                       |       |       |
#                       |       |  foreground color
#                       |  foreground intensity
#                  background color
#				level 0+:
# ^V^J%p1%c%p2%c%p3%c%p4%c%p5%c	-- scroll (p2,p3) to (p4,p5) up by p1 lines
# ^V^K%p1%c%p2%c%p3%c%p4%c%p5%c	-- scroll (p2,p3) to (p4,p5) down by p1 lines
# ^V^L%p1%c%p2%c%p3%c		-- clear p2 lines and p3 cols w/attr %p1
# ^V^M%p1%c%p2%c%p3%c%p4%c	-- fill p3 lines & p4 cols w/char p2+attr %p1
# (^V^L and ^V^M set the current attribute as a side-effect.)
# ^V ^Y <a> [...] <c>	-- repeat pattern. <a> specifies the number of bytes
#			   in the pattern, <c> the number of times the pattern
#		  	   should be repeated. If either value is 0, no-op.
#			   The pattern can contain Avatar console codes,
#			   including other ^V ^Y patterns.
#				level 1:
# ^V^O		-- clockwise mode on; turn print direction right each time you
#		   hit a window edge (yes, really).  Turned off by CR
# ^V^P		-- no-op
# ^V^Q%c	-- query the driver
# ^V^R		-- driver reset
# ^V^S		-- Sound tone (PC-specific)
# ^V^T			-- change highlight at current cursor poition to %c
# ^V^U%p1%c%p2%c	-- highlight window <a> with attribute <b>
# ^V^V%p1%c%p2%c%p3%c%p4%c%p5%c
#			-- define window
#
# From: Eric S. Raymond <esr@snark.thyrsus.com> 1 Nov 1995
# (The <blink>/<bold>/<rev>/<smacs>/<smul>/<smso> capabilities exist only to
# tell ncurses that the corresponding highlights exist; it should use <sgr>,
# which is the only method that will actually work for multiple highlights.)
#
# Update by TD - 2004: half of this was inconsistent.  Found documentation
# and repaired most of the damage.  sgr0 is probably incorrect, but the
# available documentation gives no clues for a workable string.
avatar0|avatar terminal emulator level 0,
	am, bce, msgr,
	cols#80, it#8, lines#25,
	blink=^V^B, bold=^V^A^P, cr=^M, cub1=^V^E, cud1=^V^D,
	cuf1=^V^F, cup=\026\010%p1%c%p2%c, cuu1=^V^C, el=^V^G,
	ind=^J, invis=^V^A\0, rep=\031%p1%c%p2%c, rev=^V^Ap,
	rmacs@, rs2=^L,
	sgr=%?%p1%p2%|%p3%|%p6%|%p7%|%t\026\001%?%p7%t%{128}%e%{0}%?%p1%t%{112}%|%;%?%p2%t%{1}%|%;%?%p3%t%{112}%|%;%?%p6%t%{16}%|%;%;%c%;%?%p4%t\026\002%;,
	sgr0=^V^A^G, smacs@, smso=^V^Ap, smul=^V^A^A,
	use=klone+acs,
# From: Eric S. Raymond <esr@snark.thyrsus.com> 1 Nov 1995
avatar0+|avatar terminal emulator level 0+,
	dch1=^V^N, rmir=\026\n\0\0\0\0, smir=^V^I, use=avatar0,
# From: Eric S. Raymond <esr@snark.thyrsus.com> 1 Nov 1995
avatar|avatar1|avatar terminal emulator level 1,
	civis=^V'^B, cnorm=^V'^A, cvvis=^V^C, dl1=^V-, il1=^V+,
	rmam=^V", rmir=^V^P, smam=^V$, use=avatar0+,

#### RBcomm
#
# RBComm is a lean and mean terminal emulator written by the Interrupt List
# maintainer, Ralf Brown. It was fairly popular in the late DOS years (early
# '90s), especially in the BBS world, and still has some loyal users due to
# its very small memory footprint and to a cute macro language.
rbcomm|IBM PC with RBcomm and EMACS keybindings,
	am, bw, mir, msgr, xenl,
	cols#80, it#8, lines#25,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=^L, cnorm=\E[?25h, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub1=^H, cud1=^C, cuf1=^B,
	cup=\037%p2%{32}%+%c%p1%{32}%+%c, cuu1=^^, dch1=^W,
	dl=\E[%p1%dM, dl1=^Z, ech=\E[%p1%dX, ed=^F5, el=^P^P, ht=^I,
	il=\E[%p1%dL, il1=^K, ind=\ED, invis=\E[8m,
	is2=\017\035\E(B\E)0\E[?7h\E[?3l\E[>8g, kbs=^H,
	kcub1=^B, kcud1=^N, kcuf1=^F, kcuu1=^P, khome=^A, nel=^M\ED,
	rc=\E8, rep=\030%p1%c%p2%c, rev=^R, ri=\EM, rmcup=, rmdc=,
	rmir=^], rmkx=\E>, rmso=^U, rmul=^U,
	rs1=\017\E(B\E)0\025\E[?3l\E[>8g, sc=\E7, sgr0=\E[m,
	smcup=, smdc=, smir=^\, smkx=\E=, smso=^R, smul=^T,
rbcomm-nam|IBM PC with RBcomm without autowrap,
	am@,
	bel=^G, cr=^M, cud1=^J, ht=^I, ind=^J,
	is2=\017\035\E(B\E)0\E[?7l\E[?3l\E[>8g, kbs=^H,
	kcub1=^H, kcud1=^J, nel=^M^J, use=rbcomm,
rbcomm-w|IBM PC with RBcomm in 132 column mode,
	cols#132,
	bel=^G, cr=^M, cud1=^J, ht=^I, ind=^J,
	is2=\017\035\E(B\E)0\E[?7h\E[?3h\E[>8g, kbs=^H,
	kcub1=^H, kcud1=^J, nel=^M^J, use=rbcomm,

######## LCD DISPLAYS
#

#### Matrix Orbital
# from: Eric Z. Ayers  (eric@ale.org)
#
# Matrix Orbital 20x4 LCD display
# Command Character is 0xFE (decimal 254, octal 376)
#
# On this device, cursor addressability isn't possible.  The LCD expects:
#      0xfe G <col> <row>
#      for cup: %p1 == row and %p2 is column
#
# This line:
#	cup=\376G%p2%c%p1%c
# LOOKS like it will work, but sometimes only one of the two numbers is sent.
# See the terminfo (5) manpage commented regarding 'Terminals which use "%c"'.
#
# Alas, there is no cursor upline capability on this display.
#
# These entries add some 'sanity stuff' to the clear function.  That is, it
# does a 'clear' and also turns OFF auto scroll, turns ON Auto Line Wrapping,
# and turns off the cursor blinking and stuff like that.
#
# NOTE: calling 'beep' turns on the backlight (bell)
# NOTE: calling 'flash' turns it on and back off (visual bell)
#
MtxOrb|Generic Matrix Orbital LCD display,
	bel=\376B^A, clear=\376X\376C\376R\376K\376T,
	cnorm=\376K\376T, cub1=\376L, cuf1=\376M,
	flash=\376B\001$<200>\376F, home=\376H,
MtxOrb204|20x4 Matrix Orbital LCD display,
	cols#20, lines#4, use=MtxOrb,
MtxOrb162|16x2 Matrix Orbital LCD display,
	cols#16, lines#2, use=MtxOrb,
# The end

######## OLDER TERMINAL TYPES
#
# This section is devoted to older commercial terminal brands that are now
# discontinued, but known to be still in use or represented by emulations.
#

#### AT&T (att, tty)
#
# This section also includes Teletype-branded VDTs.
#
# The AT&T/Teletype terminals group was sold to SunRiver Data Systems (now
# Boundless Technologies); for details, see the header comment on the ADDS
# section.
#
# These are AT&T's official terminfo entries.  All-caps aliases have been
# removed.
#
att2300|sv80|AT&T 2300 Video Information Terminal 80 column mode,
	am, eo, mir, msgr, xon,
	cols#80, it#8, lines#24,
	bel=^G, clear=\E[H\E[J, cr=^M, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, el1=\E[1K, home=\E[H, ht=^I, ich=\E[%p1%d@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, kbs=^H, kcbt=\E[Z, kclr=\E[J,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P,
	kdl1=\E[M, kf1=\E[1r, kf10=\E[10r, kf11=\E[11r,
	kf12=\E[12r, kf13=\E[13r, kf14=\E[14r, kf15=\E[15r,
	kf16=\E[16r, kf2=\E[2r, kf3=\E[3r, kf4=\E[4r, kf5=\E[5r,
	kf6=\E[6r, kf7=\E[7r, kf8=\E[8r, kf9=\E[9r, khome=\E[H,
	kich1=\E[@, kil1=\E[L, mc0=\E[0i, mc4=\E[4i, mc5=\E[5i,
	rev=\E[7m, rmir=\E[4l, rmso=\E[m, sgr0=\E[m, smir=\E[4h,
	smso=\E[7m,
att2350|AT&T 2350 Video Information Terminal 80 column mode,
	mc0@, mc4@, mc5@, use=att2300,

# Must setup RETURN KEY - CR, REC'VD LF - INDEX.
# Seems upward compatible with vt100, plus ins/del line/char.
# On sgr, the protection parameter is ignored.
# No check is made to make sure that only 3 parameters are output.
# 	standout= reverse + half-intensity = 3 | 5.
# 	bold= reverse + underline = 2 | 3.
# note that half-bright blinking doesn't look different from normal blinking.
# NOTE:you must program the function keys first, label second!
# (att4410: a BSD entry has been seen with the following capabilities:
# <is2=\E[?6l>, <kf1=\EOc>, <kf2=\EOd>, <kf3=\EOe>, <kf4=\EOg>,
# <kf6=\EOh>, <kf7=\EOi>, <kf8=\EOj>, -- esr)
att5410v1|att4410v1|tty5410v1|AT&T 4410/5410 80 columns - version 1,
	am, hs, mir, msgr, xon,
	cols#80, it#8, lh#2, lines#24, lw#8, nlab#8, wsl#80,
	acsc=++\,\,--..00``aaffgghhjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[2;7m, clear=\E[H\E[J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P, dim=\E[2m,
	dl1=\E[M, ed=\E[J, el=\E[K, fsl=\E8, home=\E[H, ht=^I,
	ich1=\E[@, il1=\E[L, ind=^J, invis=\E[8m, is1=\E[?3l\E)0,
	is3=\E[1;03q   f1           \EOP\E[2;03q   f2           \EOQ\E[3;03q   f3           \EOR\E[4;03q   f4           \EOS\E[5;03q   f5           \EOT\E[6;03q   f6           \EOU\E[7;03q   f7           \EOV\E[8;03q   f8           \EOW,
	kbs=^H, kclr=\E[2J, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf5=\EOT,
	kf6=\EOU, kf7=\EOV, kf8=\EOW, khome=\E[H, kll=\E[24;1H,
	ll=\E[24H, nel=^M^J,
	pfx=\E[%p1%1d;%p2%l%2.2dq   f%p1%1d           %p2%s,
	pln=\E[%p1%d;00q%p2%:-16s, rc=\E8, rev=\E[7m, ri=\EM,
	rmacs=^O, rmso=\E[m, rmul=\E[m, rs2=\Ec\E[?3l\E[2;0y,
	sc=\E7,
	sgr=\E[0%?%p1%p5%|%p6%|%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p3%p1%|%p6%|%t;7%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smso=\E[7m, smul=\E[4m,
	tsl=\E7\E[25;%p1%{1}%+%dH,

att4410v1-w|att5410v1-w|tty5410v1-w|AT&T 4410/5410 132 columns - version 1,
	cols#132, wsl#132,
	is1=\E[?3h\E)0, rs2=\Ec\E[?3h\E[2;0y, use=att5410v1,

att4410|att5410|tty5410|AT&T 4410/5410 80 columns - version 2,
	OTbs,
	pfx=\E[%p1%d;%p2%l%02dq   f%p1%d           %p2%s,
	use=att5410v1,

att5410-w|att4410-w|4410-w|tty5410-w|5410-w|AT&T 4410/5410 in 132 column mode,
	cols#132, wsl#132,
	is1=\E[?3h\E)0, rs2=\Ec\E[?3h\E[2;0y, use=att4410,

# 5410 in terms of a vt100
# (v5410: added <rmam>/<smam> based on init string -- esr)
v5410|att5410 in terms of a vt100,
	am, mir, msgr, xon,
	cols#80, it#8, lines#24, vt#3,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m$<2>, bold=\E[1m$<2>,
	clear=\E[H\E[J$<50>, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub1=^H, cud1=^J, cuf1=\E[C$<2>,
	cup=\E[%i%p1%d;%p2%dH$<5>, cuu1=\E[A$<2>, dch1=\E[P,
	dl1=\E[M, ed=\E[J$<50>, el=\E[K$<3>, el1=\E[1K$<3>,
	enacs=\E(B\E)0, home=\E[H, ht=^I, hts=\EH, ich1=\E[@,
	il1=\E[L, ind=^J, kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC,
	kcuu1=\EOA, rc=\E8, rev=\E[7m$<2>, ri=\EM$<5>, rmacs=^O,
	rmam=\E[?7l, rmkx=\E[?1l\E>, rmso=\E[m$<2>,
	rmul=\E[m$<2>, rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h,
	sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;$<2>,
	sgr0=\E[m\017$<2>, smacs=^N, smam=\E[?7h, smkx=\E[?1h\E=,
	smso=\E[1;7m$<2>, smul=\E[4m$<2>, tbc=\E[3g,
	use=vt100+fnkeys,

#
# Teletype Model 5420 -- A souped up 5410, with multiple windows,
# even! the 5420 has three modes: scroll, window or page mode
# this terminfo should work in scroll or window mode, but doesn't
# take advantage of any of the differences between them.
#
# Has memory below (2 lines!)
# 3 pages of memory (plus some spare)
# The 5410 sequences for <cup>, <cvvis>, <dch>, <dl>, <ech>, <flash>, <home>,
# <hpa>, <hts> would work for these, but these work in both scroll and window
# mode... Unset insert character so insert mode works
# <is1> sets 80 column mode,
# <is2> escape sequence:
# 1) turn off all fonts
# 2) function keys off, keyboard lock off, control display off,
#    insert mode off, erasure mode off,
# 3) full duplex, monitor mode off, send graphics off, nl on lf off
# 4) reset origin mode
# 5) set line wraparound
# 6) exit erasure mode, positional attribute mode, and erasure extent mode
# 7) clear margins
# 8) program ENTER to transmit ^J,
# We use \212 to program the ^J because a bare ^J will get translated by
# UNIX into a CR/LF. The enter key is needed for AT&T uOMS.
#     1      2            3              4     5     6    7  8
# <is3> set screen color to black,
# No representation in terminfo for the delete word key: kdw1=\Ed
# Key capabilities assume the power-up send sequence...
# This <rmcup> is not strictly necessary, but it helps maximize
# memory usefulness: <rmcup=\Ez>,
# Alternate sgr0:	<sgr0=\E[m\EW^O>,
# Alternate sgr:	<sgr=\E[%?%p1%t2;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p5%t;2%;%?%p7%t;8%;m%?%p8%t\EV%;%?%p9%t^N%e^O%;>,
# smkx programs the SYS PF keys to send a set sequence.
# It also sets up labels f1, f2, ..., f8, and sends edit keys.
# This string causes them to send the strings <kf1>-<kf8>
# when pressed in SYS PF mode.
# (att4415: I added <rmam>/<smam> based on the init string -- esr)
att4415|tty5420|att5420|AT&T 4415/5420 80 cols,
	OTbs, db, mir, xon,
	lh#2, lm#78, lw#8, nlab#8, wsl#55,
	cbt=\E[Z, clear=\E[x\E[J, cnorm=\E[11;0j, cub=\E[%p1%dD,
	cud=\E[%p1%dB, cuf=\E[%p1%dC, cup=\E[%i%p1%d;%p2%dx,
	cuu=\E[%p1%dA, cvvis=\E[11;1j, dch=\E[%p1%dP,
	dl=\E[%p1%dM, ech=\E[%p1%ds\E[%p1%dD,
	flash=\E[?5h$<200>\E[?5l, home=\E[x,
	hpa=\E[%p1%{1}%+%dG, hts=\EH, ich=\E[%p1%d@, ich1@,
	il=\E[%p1%dL, indn=\E[%p1%dE, is1=\E[?3l$<100>,
	is2=\E[m\017\E[1;2;3;4;6l\E[12;13;14;20l\E[?6;97;99l\E[?7h\E[4i\Ex\E[21;1j\212,
	is3=\E[?5l, kbeg=\Et, kcbt=\E[Z, kdch1=\E[P, kdl1=\E[M,
	kel=\E[2K, kend=\Ez, kent=\Eent, kf1=\EOc, kf2=\EOd,
	kf3=\EOe, kf4=\EOf, kf5=\EOg, kf6=\EOh, kf7=\EOi, kf8=\EOj,
	kich1=\E[4h, kil1=\E[L, kind=\E[T, kll=\Eu, knp=\E[U,
	kpp=\E[V, kri=\E[S, lf1=F1, lf2=F2, lf3=F3, lf4=F4, lf5=F5,
	lf6=F6, lf7=F7, lf8=F8, ll=\Ew, mc0=\E[?2i, mc4=\E[?9i,
	mc5=\E[?4i, mrcup=\E[%i%p1%d;%p2%dt,
	pfx=\E[%p1%d;%p2%l%02dq   F%p1%d           %p2%s,
	pln=\E[%p1%d;0;0;0q%p2%:-16.16s, prot=\EV,
	rin=\E[%p1%dF, rmam=\E[?7l, rmir=\E[4l,
	rmkx=\E[19;0j\E[21;1j\212, rmln=\E|,
	sgr=\E[0%?%p1%p5%|%p6%|%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p3%p1%|%p6%|%t;7%;%?%p7%t;8%;m%?%p8%t\EV%;%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smam=\E[?7h, smir=\E[4h,
	smkx=\E[19;1j\E[21;4j\Eent, smln=\E~, tbc=\E[3g,
	tsl=\E7\E[25;%p1%{8}%+%dH, vpa=\E[%p1%{1}%+%dd,
	use=att4410,

att4415-w|tty5420-w|att5420-w|AT&T 4415/5420 132 cols,
	cols#132, lm#54, wsl#97,
	is1=\E[?3h$<100>, use=att4415,

att4415-rv|tty5420-rv|att5420-rv|AT&T 4415/5420 80 cols/rv,
	flash=\E[?5l$<200>\E[?5h, is3=\E[?5h, use=att4415,

att4415-w-rv|tty5420-w-rv|att5420-w-rv|AT&T 4415/5420 132 cols/rv,
	cols#132, lm#54, wsl#97,
	flash=\E[?5l$<200>\E[?5h, is1=\E[?3h$<100>, is3=\E[?5h,
	use=att4415,

# Note that this mode permits programming USER PF KEYS and labels
# However, when you program user pf labels you have to reselect
# user pf keys to make them appear!
att4415+nl|tty5420+nl|att5420+nl|generic AT&T 4415/5420 changes for not changing labels,
	kf1@, kf2@, kf3@, kf4@, kf5@, kf6@, kf7@, kf8@,
	pfx=\E[%p1%d;%p2%l%02d;0;1q   F%p1%d           %p2%s,
	pln=\E[%p1%d;0;0;1q%p2%:-16.16s,

att4415-nl|tty5420-nl|att5420-nl|AT&T 4415/5420 without changing labels,
	kf1@, kf2@, kf3@, kf4@, kf5@, kf6@, kf7@, kf8@, use=att4415+nl,
	use=att4415,

att4415-rv-nl|tty5420-rv-nl|att5420-rv-nl|AT&T 4415/5420 reverse video without changing labels,
	kf1@, kf2@, kf3@, kf4@, kf5@, kf6@, kf7@, kf8@, use=att4415+nl,
	use=att4415-rv,

att4415-w-nl|tty5420-w-nl|att5420-w-nl|AT&T 4415/5420 132 cols without changing labels,
	kf1@, kf2@, kf3@, kf4@, kf5@, kf6@, kf7@, kf8@, use=att4415+nl,
	use=att4415-w,

att4415-w-rv-n|tty5420-w-rv-n|att5420-w-rv-n|AT&T 4415/5420 132 cols reverse without changing labels,
	kf1@, kf2@, kf3@, kf4@, kf5@, kf6@, kf7@, kf8@, use=att4415+nl,
	use=att4415-w-rv,

att5420_2|AT&T 5420 model 2 80 cols,
	am, db, hs, mir, msgr, xon,
	cols#80, it#8, lh#2, lines#24, lm#78, lw#8, nlab#8, wsl#55,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	blink=\E[5m, cbt=\E[1Z, clear=\EH\EJ, cnorm=\E[11;0j,
	cr=\EG, csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[1B, cuf=\E[%p1%dC, cuf1=\E[1C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[1A,
	cvvis=\E[11;1j, dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[M, ech=\E[%p1%ds\E[%p1%dD, ed=\E[0J,
	el=\E[0K, el1=\E[1K, flash=\E[?5h$<200>\E[?5l, fsl=\E8,
	home=\E[H, hpa=\E[%p1%{1}%+%dG, ht=^I, hts=\EH,
	ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=^J,
	indn=\E[%p1%dE, invis=\E[8m,
	is1=\E[0;23r\Ex\Ey\E[2;0j\E[3;3j\E[4;0j\E[5;0j\E[6;0j\E[7;0j\E[8;0j\E[9;1j\E[10;0j\E[15;0j\E[16;1j\E[19;0j\E[20;1j\E[29;0j\E[1;24r,
	kbeg=\Et, kbs=^H, kcbt=\E[Z, kclr=\E[2J, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P, kdl1=\E[M,
	kel=\E[2K, kend=\Ez, kent=^J, kf1=\EOc, kf2=\EOd, kf3=\EOe,
	kf4=\EOf, kf5=\EOg, kf6=\EOh, kf7=\EOi, kf8=\EOj, khome=\E[H,
	kich1=\E[4h, kil1=\E[L, kind=\E[T, kll=\Eu, knp=\E[U,
	kpp=\E[V, kri=\E[S, lf1=F1, lf2=F2, lf3=F3, lf4=F4, lf5=F5,
	lf6=F6, lf7=F7, lf8=F8, ll=\Ew, mc0=\E[?;2i, mc4=\E[4i,
	mc5=\E[5i, mrcup=\E[%i%p1%d;%p2%dt, nel=^M^J,
	pfx=\E[%p1%d;%p2%l%02dq   F%p1%d           %p2%s\E~,
	pln=\E[%p1%d;0;0;0q%p2%:-16.16s\E~, prot=\EV, rc=\E8,
	rev=\E[7m, ri=\EM, rin=\E[%p1%dF, rmacs=^O, rmkx=\E[19;0j,
	rmln=\E|, rmso=\E[m, rmul=\E[m, rs2=\Ec\E[?3l\E[2;0y,
	sc=\E7,
	sgr=\E[0%?%p1%p5%|%t;2%;%?%p2%p6%|%t;4%;%?%p4%t;5%;%?%p3%p1%|%p6%|%t;7%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smkx=\E[19;1j, smln=\E~,
	smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	tsl=\E7\E[25;%p1%{8}%+%dH, vpa=\E[%p1%{1}%+%dd,
att5420_2-w|AT&T 5420 model 2 in 132 column mode,
	cols#132,
	is1=\E[0;23r\Ex\Ey\E[2;0j\E[3;3j\E[4;0j\E[5;1j\E[6;0j\E[7;0j\E[8;0j\E[9;1j\E[10;0j\E[15;0j\E[16;1j\E[19;0j\E[20;1j\E[29;0j\E[1;24r,
	use=att5420_2,

att4418|att5418|AT&T 5418 80 cols,
	am, xon,
	cols#80, lines#24,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, clear=\E[H\E[2J, cr=^M, cub=\E[%p1%dD,
	cub1=\E[D, cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA,
	cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[1P, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[0J, el=\E[0K, home=\E[H,
	ich=\E[%p1%d@, ich1=\E[1@, il=\E[%p1%dL, il1=\E[1L, ind=^J,
	is1=\E[?3l, is2=\E)0\E?6l\E?5l, kclr=\E[%%, kcub1=\E@,
	kcud1=\EU, kcuf1=\EA, kcuu1=\ES, kent=\E[, kf1=\E[h,
	kf10=\E[m, kf11=\E[n, kf12=\E[o, kf13=\E[H, kf14=\E[I,
	kf15=\E[J, kf18=\E[K, kf19=\E[L, kf2=\E[i, kf20=\E[E,
	kf21=\E[_, kf22=\E[M, kf23=\E[N, kf24=\E[O, kf3=\E[j,
	kf6=\E[k, kf7=\E[l, kf8=\E[f, kf9=\E[w, khome=\Ec, rc=\E8,
	rev=\E[7m, rmacs=^O, rmso=\E[m, rmul=\E[m, sc=\E7,
	sgr0=\E[m\017, smacs=^N, smso=\E[7m, smul=\E[4m,
att4418-w|att5418-w|AT&T 5418 132 cols,
	cols#132,
	is1=\E[?3h, use=att5418,

att4420|tty4420|teletype 4420,
	OTbs, da, db, eo, msgr, ul, xon,
	cols#80, lines#24, lm#72,
	bel=^G, clear=\EH\EJ, cr=\EG, cub1=\ED, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dch1=\EP,
	dl1=\EM, ed=\EJ, el=\Ez, home=\EH, il1=\EL, ind=\EH\EM\EY7\s,
	kcbt=\EO, kclr=\EJ, kcub1=^H, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kdch1=\EP, kdl1=\EM, kf0=\EU, kf3=\E@, khome=\EH,
	kich1=\E\^, kil1=\EL, kind=\ES, kri=\ET,
	lf0=segment advance, lf3=cursor tab, rmdc@, rmso=\E~,
	rmul=\EZ, smdc@, smso=\E}, smul=\E\\,

#  The following is a terminfo entry for the Teletype 4424
#  asynchronous keyboard-display terminal.  It supports
#  the vi editor.  The terminal must be set up as follows,
#
# 	HIGHLIGHT DEFINITION	3-TONE
# 	DISPLAY FUNCTION	GROUP III
#
#  The second entry below provides limited (a la adm3a)
#  operation under GROUP II.
#
#  This must be used with DISPLAY FUNCTION GROUP I or III
# 	and HIGHLIGHT DEFINITION 3-TONE
# The terminal has either bold or blink, depending on options
#
# (att4424: commented out <smcup>=\E[1m, we don't need bright locked on -- esr)
att4424|tty4424|teletype 4424,
	OTbs, am, xon,
	cols#80, lines#24,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E3, bold=\E3, cbt=\EO, clear=\E[H\E[2J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\EB, cuf=\E[%p1%dC, cuf1=\EC,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\EA,
	dch=\E[%p1%dP, dch1=\EP, dim=\EW, dl=\E[%p1%dM, dl1=\EM,
	ed=\EJ, el=\Ez, home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@,
	ich1=\E\^, il=\E[%p1%dL, il1=\EL, ind=^J, is2=\E[20l\E[?7h,
	kbs=^H, kclr=\EJ, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS,
	khome=\E[H, nel=\EE, rev=\E}, ri=\ET, rmacs=\E(B, rmso=\E~,
	rmul=\EZ,
	sgr=\E[%?%p1%t7%;%?%p2%t;4%;%?%p3%t;7%;%?%p6%t;1%;%?%p6%p4%|%t;5%;%?%p5%t;0%;m,
	sgr0=\EX\E~\EZ\E4\E(B, smacs=\E(0, smso=\E}, smul=\E\\,
	tbc=\EF,

att4424-1|tty4424-1|teletype 4424 in display function group I,
	kclr@, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA, khome@,
	use=att4424,

# This entry is not one of AT&T's official ones, it was translated from the
# 4.4BSD termcap file.  The highlight strings are different from att4424.
# I have no idea why this is -- older firmware version, maybe?
# The following two lines are the comment originally attached to the entry:
# This entry appears to avoid the top line - I have no idea why.
# From: jwb Wed Mar 31 13:25:09 1982 remote from ihuxp
att4424m|tty4424m|teletype 4424M,
	am, da, db, mir,
	cols#80, it#8, lines#23,
	bel=^G, clear=\E[2;H\E[J, cr=^M, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%2d;%p2%2dH\E[B, cuu1=\E[A, dch1=\EP,
	dl1=\EM, el=\E[K, ht=^I, ich1=\E\^, il1=\EL, ind=^J, ip=$<2/>,
	is2=\E[m\E[2;24r, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kf1=\EOP, kf2=\EOQ, kf3=\EOR,
	kf4=\EOS, khome=\E[H, nel=^M^J, ri=\ET, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m, smso=\E[7m, smul=\E[4m,

# The Teletype 5425 is really version 2 of the Teletype 5420. It
# is quite similar, except for some minor differences. No page
# mode, for example, so all of the <cup> sequences used above have
# to change back to what's being used for the 5410. Many of the
# option settings have changed their numbering as well.
#
# This has been tested on a preliminary model.
#
# (att5425: added <rmam>/<smam> based on the init string -- esr)
att5425|tty5425|att4425|AT&T 4425/5425,
	am, da, db, hs, mir, msgr, xenl, xon,
	cols#80, it#8, lh#2, lines#24, lm#78, lw#8, nlab#8, wsl#55,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[2;7m, cbt=\E[Z,
	clear=\E[H\E[J, cnorm=\E[12;0j, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[12;1j, dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[M, ech=\E[%p1%ds\E[%p1%dD, ed=\E[J,
	el=\E[K, el1=\E[1K, enacs=\E(B\E)0,
	flash=\E[?5h$<200>\E[?5l, fsl=\E8, home=\E[H,
	hpa=\E[%p1%{1}%+%dG, ht=^I, hts=\EH, ich=\E[%p1%d@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, indn=\E[%p1%dE,
	invis=\E[8m, is1=\E<\E[?3l$<100>,
	is2=\E[m\017\E[1;2;3;4;6l\E[12;13;14;20l\E[?6;97;99l\E[?7h\E[4i\Ex\E[25;1j\212,
	is3=\E[?5l, kbeg=\Et, kbs=^H, kcbt=\E[Z, kclr=\E[J,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P,
	kdl1=\E[M, kel=\E[2K, kend=\Ez, kent=\Eent, kf1=\EOc,
	kf2=\EOd, kf3=\EOe, kf4=\EOf, kf5=\EOg, kf6=\EOh, kf7=\EOi,
	kf8=\EOj, khome=\E[H, kich1=\E[4h, kil1=\E[L, kind=\E[T,
	kri=\E[S, ll=\E[24H, mc0=\E[?2i, mc4=\E[?9i, mc5=\E[?4i,
	nel=^M^J,
	pfx=\E[%p1%d;%p2%l%02dq   F%p1%1d           %p2%s,
	pln=\E[%p1%d;0;0;0q%p2%:-16.16s, prot=\EV, rc=\E8,
	rev=\E[7m, ri=\EM, rin=\E[%p1%dF, rmacs=^O, rmam=\E[?7l,
	rmir=\E[4l, rmkx=\E[21;0j\E[25;1j\212, rmln=\E|,
	rmso=\E[m, rmul=\E[m, rs2=\Ec\E[?3l\E[2;0y, sc=\E7,
	sgr=\E[0%?%p5%p6%|%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p3%p1%|%p6%|%t;7%;%?%p7%t;8%;m%?%p8%t\EV%;%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smir=\E[4h,
	smkx=\E[21;1j\E[25;4j\Eent\E~, smln=\E~, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g, tsl=\E7\E[25;%p1%{8}%+%dH,
	vpa=\E[%p1%{1}%+%dd,

att5425-nl|tty5425-nl|att4425-nl|AT&T 4425/5425 80 columns no labels,
	smkx=\E[21;1j\E[25;4j\Eent, use=att4425,

att5425-w|att4425-w|tty5425-w|teletype 4425/5425 in 132 column mode,
	cols#132, lm#54, wsl#97,
	is1=\E[?3h$<100>, use=tty5425,

# (att4426: his had bogus capabilities: :ri=\EM:, :ri=\E[1U:.
# I also added <rmam>/<smam> -- esr)
att4426|tty4426|teletype 4426S,
	am, da, db, xon,
	cols#80, lines#24, lm#48,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, bold=\E[5m, clear=\E[H\E[2J\E[1U\E[H\E[2J\E[1V,
	cr=^M, cub=\E[%p1%dD, cub1=\E[D, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\EP,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[0K, home=\E[H,
	hpa=\E[%p1%dG, ht=^I, hts=\E1, ich=\E[%p1%d@, ich1=\E\^,
	il=\E[%p1%dL, il1=\EL, ind=^J, indn=\E[%p1%dS,
	is1=\Ec\E[?7h, is2=\E[m\E[1;24r, kbs=^H, kcbt=\EO,
	kclr=\E[2J, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf5=\EOT, kf6=\EOU,
	kf7=\EOV, kf8=\EOW, khome=\E[H, kll=\E[24;1H, ll=\E[24H,
	nel=^M^J, rc=\E8, rev=\E[7m, ri=\ET, rin=\E[%p1%dT,
	rmacs=\E(B, rmam=\E[?7l, rmso=\E[m, rmul=\E[m,
	rs2=\Ec\E[?3l\E[2;0y, sc=\E7, sgr0=\E[m\E(B, smacs=\E(0,
	smam=\E[?7h, smso=\E[5m, smul=\E[4m, tbc=\E[3g,
	vpa=\E[%p1%dd,

# Terminfo entry for the AT&T 510 A Personal Terminal
# Function keys 9 - 16 are available only after the
# screen labeled (soft keys/action blocks) are labeled.  Function key
# 9 corresponds to the leftmost touch target on the screen,
# function key 16 corresponds to the rightmost.
#
# This entry is based on one done by Ernie Rice at Summit, NJ and
# changed by Anne Gallup, Skokie, IL, ttrdc!anne
att510a|bct510a|AT&T 510A Personal Terminal,
	am, mir, msgr, xenl, xon,
	cols#80, lh#2, lines#24, lw#7, nlab#8,
	acsc=+g\,h-f.e`bhrisjjkkllmmnnqqttuuvvwwxx{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[2;7m, cbt=\E[Z,
	civis=\E[11;0|, clear=\E[H\E[J, cnorm=\E[11;3|, cr=^M,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, cvvis=\E[11;2|, dch=\E[%p1%dP,
	dch1=\E[P, dim=\E[2m, dl=\E[%p1%dM, dl1=\E[M, ed=\E[0J,
	el=\E[0K, el1=\E[1K, enacs=\E(B\E)1, ff=^L, home=\E[H, ht=^I,
	hts=\EH, il=\E[%p1%dL, il1=\E[L, ind=^J, is1=\E(B\E)1\E[2l,
	is3=\E[21;1|\212, kLFT=\E[u, kRIT=\E[v, kbs=^H, kcbt=\E[Z,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf1=\EOm,
	kf10=\EOd, kf11=\EOe, kf12=\EOf, kf13=\EOg, kf14=\EOh,
	kf15=\EOi, kf16=\EOj, kf2=\EOV, kf3=\EOu, kf4=\ENj, kf5=\ENe,
	kf6=\ENf, kf7=\ENh, kf8=\E[H, kf9=\EOc, kind=\E[S, kri=\E[T,
	mc0=\E[0i, mc4=\E[?8i, mc5=\E[?4i, nel=\EE,
	pln=\E[%p1%dp%p2%:-16s, rc=\E8, rev=\E[7m, ri=\EM,
	rmacs=^O, rmkx=\E[19;0|, rmso=\E[m, rmul=\E[m, sc=\E7,
	sgr=\E[0%?%p5%p6%|%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p3%p1%|%p6%|%t;7%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smkx=\E[19;1|, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g,

# Terminfo entry for the AT&T 510 D Personal Terminal
# Function keys 9 through 16 are accessed by bringing up the
# system blocks.
# Function key 9 corresponds to the leftmost touch target on the screen,
# function key 16 corresponds to the rightmost.
#
# There are problems with soft key labeling.  These are due to
# strangenesses in the native terminal that are impossible to
# describe in a terminfo.
att510d|bct510d|AT&T 510D Personal Terminal,
	am, da, db, mir, msgr, xenl, xon,
	cols#80, lh#2, lines#24, lm#48, lw#7, nlab#8,
	acsc=+g\,h-f.e`bhrisjjkkllmmnnqqttuuvvwwxx{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[2;7m, cbt=\E[Z,
	clear=\E[H\E[J, cnorm=\E[11;3|, cr=^M, cub=\E[%p1%dD,
	cub1=^H, cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA,
	cuu1=\E[A, cvvis=\E[11;2|, dch=\E[%p1%dP, dch1=\E[P,
	dim=\E[2m, dl=\E[%p1%dM, dl1=\E[M, ed=\E[0J, el=\E[0K,
	el1=\E[1K, enacs=\E(B\E)1, ff=^L, home=\E[H,
	hpa=\E[%p1%{1}%+%dG, ht=^I, hts=\EH, ich=\E[%p1%d@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, indn=\E[%p1%dS,
	invis=\E[8m, is1=\E(B\E)1\E[5;0|, is3=\E[21;1|\212,
	kLFT=\E[u, kRIT=\E[v, kbs=^H, kcbt=\E[Z, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf1=\EOm, kf10=\EOd,
	kf11=\EOe, kf12=\EOf, kf13=\EOg, kf14=\EOh, kf15=\EOi,
	kf16=\EOj, kf2=\EOV, kf3=\EOu, kf4=\ENj, kf5=\ENe, kf6=\ENf,
	kf7=\ENh, kf8=\E[H, kf9=\EOc, kind=\E[S, kri=\E[T, ll=\E#2,
	mc0=\E[0i, mc4=\E[?8i, mc5=\E[?4i, mgc=\E\:, nel=\EE,
	pln=\E[%p1%dp%p2%:-16s, rc=\E8,
	rep=%p1%c\E[%p2%{1}%-%db, rev=\E[7m, ri=\EM,
	rin=\E[%p1%dT, rmacs=^O, rmir=\E[4l, rmkx=\E[19;0|,
	rmln=\E<, rmso=\E[m, rmul=\E[m, rmxon=\E[29;1|,
	rs2=\E[5;0|, sc=\E7,
	sgr=\E[0%?%p5%p6%|%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p3%p1%|%p6%|%t;7%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smgl=\E4, smgr=\E5, smir=\E[4h,
	smkx=\E[19;1|, smln=\E?, smso=\E[7m, smul=\E[4m,
	smxon=\E[29;0|, tbc=\E[3g, vpa=\E[%p1%{1}%+%dd,

# (att500: I merged this with the att513 entry, att500 just used att513 -- esr)
att500|att513|AT&T 513 using page mode,
	am, chts, mir, msgr, xenl, xon,
	cols#80, lh#2, lines#24, lw#8, nlab#8,
	acsc=+g\,h-f.e`bhrisjjkkllmmnnqqttuuvvwwxx{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[2;7m, cbt=\E[Z,
	clear=\E[H\E[J, cnorm=\E[11;0|, cr=^M,
	csr=%i\E[%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[11;1|, dch=\E[%p1%dP, dch1=\E[P$<1>, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K,
	enacs=\E(B\E)1, home=\E[H, hpa=\E[%p1%{1}%+%dG, ht=^I,
	hts=\EH, ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L, ind=^J,
	indn=\E[%p1%dE, invis=\E[8m,
	is1=\E?\E[3;3|\E[10;0|\E[21;1|\212\E[6;1|\E[1{\E[?99l,
	kBEG=\ENB, kCAN=\EOW, kCMD=\EOU, kCPY=\END, kCRT=\EON,
	kDC=\ENF, kDL=\ENE, kEND=\ENN, kEOL=\EOA, kEXT=\EOK,
	kFND=\EOX, kHLP=\EOM, kHOM=\ENM, kIC=\ENJ, kLFT=\ENK,
	kMOV=\ENC, kMSG=\EOL, kNXT=\ENH, kOPT=\EOR, kPRT=\EOZ,
	kPRV=\ENG, kRDO=\EOT, kRES=\EOQ, kRIT=\ENL, kRPL=\EOY,
	kSAV=\EOO, kSPD=\EOP, kUND=\EOS, kbeg=\E9, kbs=^H, kcan=\EOw,
	kcbt=\E[Z, kclo=\EOV, kclr=\E[J, kcmd=\EOu, kcpy=\ENd,
	kcrt=\EOn, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\ENf, kdl1=\ENe, kel=\EOa, kend=\E0, kent=\Eent,
	kext=\EOk, kf1=\EOc, kf2=\EOd, kf3=\EOe, kf4=\EOf, kf5=\EOg,
	kf6=\EOh, kf7=\EOi, kf8=\EOj, kfnd=\EOx, khlp=\EOm,
	khome=\E[H, kich1=\ENj, kind=\E[S, kmov=\ENc, kmrk=\ENi,
	kmsg=\EOl, knp=\E[U, knxt=\ENh, kopn=\EOv, kopt=\EOr,
	kpp=\E[V, kprt=\EOz, kprv=\ENg, krdo=\EOt, kref=\EOb,
	kres=\EOq, krfr=\ENa, kri=\E[T, krpl=\EOy, krst=\EOB,
	ksav=\EOo, kslt=\ENI, kspd=\EOp, kund=\EOs, ll=\E#2,
	mc0=\E[?98l\E[0i, mc4=\E[?98l\E[?8i, mc5=\E[?98l\E[?4i,
	nel=\EE,
	pfkey=\E[%p1%d;%p2%l%d;3;0p   F%p1%d           %p2%s,
	pfloc=\E[%p1%d;%p2%l%d;2;0p   F%p1%d           %p2%s,
	pfx=\E[%p1%d;%p2%l%d;1;0p   F%p1%d           %p2%s,
	pln=\E[%p1%dp%p2%:-16s, rc=\E8,
	rep=%p1%c\E[%p2%{1}%-%db, rev=\E[7m, ri=\EM,
	rin=\E[%p1%dF, rmacs=^O, rmir=\E[4l,
	rmkx=\E[19;0|\E[21;1|\212, rmln=\E<, rmso=\E[m,
	rmul=\E[m,
	rs1=\E?\E[3;3|\E[10;0|\E[21;1|\212\E[6;1|\E[1{\E[?99l\E[2;0|\E[6;1|\E[8;0|\E[19;0|\E[1{\E[?99l,
	rs2=\E[5;0|, sc=\E7,
	sgr=\E[0%?%p1%p5%|%t;2%;%?%p2%p6%|%t;4%;%?%p4%t;5%;%?%p3%p1%|%p6%|%t;7%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smir=\E[4h,
	smkx=\E[19;1|\E[21;4|\Eent, smln=\E?, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g, vpa=\E[%p1%{1}%+%dd,

# 01-07-88
# printer must be set to EMUL ANSI to accept ESC codes
# <cuu1> stops at top margin
# <is1> sets cpi 10,lpi 6,form 66,left 1,right 132,top 1,bottom 66,font
#	and alt font ascii,wrap on,tabs cleared
# <is2> disables newline on LF,Emphasized off
# The <u0> capability sets form length
att5310|att5320|AT&T Model 53210 or 5320 matrix printer,
	xhpa, xvpa,
	bufsz#8192, cols#132, cps#120, it#8, lines#66, orc#10,
	orhi#100, orl#12, orvi#72,
	cpi=%?%p1%{10}%=%t\E[w%e%p1%{12}%=%t\E[2w%e%p1%{5}%=%t\E[5w%e%p1%{13}%=%p1%{14}%=%O%t\E[3w%e%p1%{16}%=%p1%{17}%=%O%t\E[4w%e%p1%{6}%=%t\E[6w%e%p1%{7}%=%t\E[7w%e%p1%{8}%=%t\E[8w%;,
	cr=^M,
	csnm=%?%p1%{0}%=%tusascii%e%p1%{1}%=%tenglish%e%p1%{2}%=%tfinnish%e%p1%{3}%=%tjapanese%e%p1%{4}%=%tnorwegian%e%p1%{5}%=%tswedish%e%p1%{6}%=%tgermanic%e%p1%{7}%=%tfrench%e%p1%{8}%=%tcanadian_french%e%p1%{9}%=%titalian%e%p1%{10}%=%tspanish%e%p1%{11}%=%tline%e%p1%{12}%=%tsecurity%e%p1%{13}%=%tebcdic%e%p1%{14}%=%tapl%e%p1%{15}%=%tmosaic%;,
	cud=\E[%p1%de, cud1=^J, cuf=\E[%p1%da, cuf1=\s, cuu1=\EM,
	ff=^L, hpa=\E[%p1%d`, ht=^I, is1=\Ec, is2=\E[20l\r,
	lpi=%?%p1%{2}%=%t\E[4z%e%p1%{3}%=%t\E[5z%e%p1%{4}%=%t\E[6z%e%p1%{6}%=%t\E[z%e%p1%{8}%=%t\E[2z%e%p1%{12}%=%t\E[3z%;,
	rshm=\E[m,
	scs=%?%p1%{0}%=%t\E(B%e%p1%{1}%=%t\E(A%e%p1%{2}%=%t\E(C%e%p1%{3}%=%t\E(D%e%p1%{4}%=%t\E(E%e%p1%{5}%=%t\E(H%e%p1%{6}%=%t\E(K%e%p1%{7}%=%t\E(R%e%p1%{8}%=%t\E(Q%e%p1%{9}%=%t\E(Y%e%p1%{10}%=%t\E(Z%e%p1%{11}%=%t\E(0%e%p1%{12}%=%t\E(1%e%p1%{13}%=%t\E(3%e%p1%{14}%=%t\E(8%e%p1%{15}%=%t\E(}%;,
	smgbp=\E[;%p1%dr, smglp=\E[%{1}%p1%+%ds,
	smgrp=\E[;%{1}%p1%+%ds, smgtp=\E[%p1%dr, sshm=\E[5m,
	u0=\E[%p1%dt, vpa=\E[%p1%dd,

# Teletype 5620, firmware version 1.1 (8;7;3) or earlier from BRL
# The following SET-UP modes are assumed for normal operation:
#	CR_DEF=CR	NL_DEF=INDEX	DUPLEX=FULL
# Other SET-UP modes may be set for operator convenience or communication
# requirements.  This termcap description is for the Resident Terminal Mode.
# No delays specified; use "stty ixon -ixany" to enable DC3/DC1 flow control!
# The BRL entry also said: UNSAFE :ll=\E[70H:
att5620-1|tty5620-1|dmd1|Teletype 5620 with old ROMs,
	am, xon,
	cols#88, it#8, lines#70, vt#3,
	bel=^G, clear=\E[H\E[J, cr=^M, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch=\E[%p1%dP,
	dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K,
	home=\E[H, ht=^I, ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL,
	il1=\E[L, ind=^J, indn=\E[%p1%dS, kbs=^H, kclr=\E[2J,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, khome=\E[H,
	kll=\E[70;1H, nel=^M^J, rc=\E8, ri=\E[T, rin=\E[%p1%dT,
	rs1=\Ec, sc=\E7,

# 5620 terminfo  (2.0 or later ROMS with char attributes)
# The following SET-UP modes are assumed for normal operation:
#	DUPLEX=FULL	GEN_FLOW=ON	NEWLINE=INDEX	RETURN=CR
# Other SET-UP modes may be set for operator convenience or communication
# requirements.  This termcap description is for Resident Terminal Mode.  No
# delays are specified; use "stty ixon -ixany" to enable DC3/DC1 flow control!
# assumptions: <ind> (scroll forward one line) is only done at screen bottom
# Be aware that older versions of the dmd have a firmware bug that affects
# parameter defaulting; for this terminal, the 0 in \E[0m is not optional.
# <msgr> is from an otherwise inferior BRL for this terminal.  That entry
# also has <ll>=\E[70H commented out and marked unsafe.
# For more, see the 5620 FAQ maintained by David Breneman <daveb@dgtl.com>.
att5620|dmd|tty5620|ttydmd|5620|5620 terminal 88 columns,
	OTbs, am, msgr, npc, xon,
	cols#88, it#8, lines#70,
	bel=^G, bold=\E[2m, clear=\E[H\E[J, cr=^M, cub1=^H,
	cud1=\E[B, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I, ich=\E[%p1%d@,
	ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=\E[S,
	indn=\E[%p1%dS, kbs=^H, kclr=\E[2J, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, khome=\E[H, kll=\E[70;1H, nel=^J,
	pfx=\E[%p1%d;%p2%l%dq%p2%s, rc=\E8, rev=\E[7m, ri=\E[T,
	rin=\E[%p1%dT, rmso=\E[0m, rmul=\E[0m, rs1=\Ec, sc=\E7,
	sgr0=\E[0m, smso=\E[7m, smul=\E[4m,
att5620-24|tty5620-24|dmd-24|teletype dmd 5620 in a 24x80 layer,
	lines#24, use=att5620,
att5620-34|tty5620-34|dmd-34|teletype dmd 5620 in a 34x80 layer,
	lines#34, use=att5620,
# 5620 layer running the "S" system's downloaded graphics handler:
att5620-s|tty5620-s|layer|vitty|5620 S layer,
	OTbs, OTpt, am,
	cols#80, it#8, lines#72,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J,
	cup=\EY%p2%{32}%+%c%p1%{32}%+%c, cuu1=^K, dl1=\ED,
	el=\EK, flash=\E^G, ht=^I, il1=\EI, ind=^J, kbs=^H, kclr=\E[2J,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, khome=\E[H,
	kll=\E[70;1H,

# Entries for <kf15> thru <kf28> refer to the shifted system pf keys.
#
# Entries for <kf29> thru <kf46> refer to the alternate keypad mode
# keys:  = * / + 7 8 9 - 4 5 6 , 1 2 3 0 . ENTER
att605|AT&T 605 80 column 102key keyboard,
	am, eo, xon,
	cols#80, lines#24, lw#8, nlab#8, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, clear=\E[H\E[J,
	cr=^M, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch=\E[%p1%dP,
	dch1=\E[P, dim=\E[2m, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K, el1=\E[1K, fsl=\E8, ht=^I, ich=\E[%p1%d@, ich1=\E[@,
	il1=\E[L, ind=^J, invis=\E[8m,
	is1=\E[8;0|\E[?\E[13;20l\E[?\E[12h, is2=\E[m\017,
	kLFT=\E[ A, kRIT=\E[ @, kbs=^H, kcbt=\E[Z, kclr=\E[2J,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P,
	kdl1=\E[M, kend=\E[24;1H, kf1=\EOc, kf10=\ENp, kf11=\ENq,
	kf12=\ENr, kf13=\ENs, kf14=\ENt, kf15=\EOC, kf16=\EOD,
	kf17=\EOE, kf18=\EOF, kf19=\EOG, kf2=\EOd, kf20=\EOH,
	kf21=\EOI, kf22=\EOJ, kf23=\ENO, kf24=\ENP, kf25=\ENQ,
	kf26=\ENR, kf27=\ENS, kf28=\ENT, kf29=\EOP, kf3=\EOe,
	kf30=\EOQ, kf31=\EOR, kf32=\EOS, kf33=\EOw, kf34=\EOx,
	kf35=\EOy, kf36=\EOm, kf37=\EOt, kf38=\EOu, kf39=\EOv,
	kf4=\EOf, kf40=\EOl, kf41=\EOq, kf42=\EOr, kf43=\EOs,
	kf44=\EOp, kf45=\EOn, kf46=\EOM, kf5=\EOg, kf6=\EOh,
	kf7=\EOi, kf8=\EOj, kf9=\ENo, khome=\E[H, kich1=\E[@,
	kil1=\E[L, kind=\E[S, knp=\E[U, kpp=\E[V, ll=\E[24H,
	mc4=\E[?4i, mc5=\E[?5i, nel=\EE,
	pfx=\E[%p1%d;%p2%l%02dq   F%p1%1d           %p2%s,
	pln=\E[%p1%d;0;0;0q%p2%:-16.16s, rc=\E8, rev=\E[7m,
	rmacs=^O, rmir=\E[4l, rmln=\E[2p, rmso=\E[m, rmul=\E[m,
	rs2=\Ec\E[?3l, sc=\E7, sgr0=\E[m\017, smacs=\E)0\016,
	smir=\E[4h, smln=\E[p, smso=\E[7m, smul=\E[4m,
	tsl=\E7\E[25;%i%p1%dx,
att605-pc|ATT 605 in pc term mode,
	acsc=j\331k\277l\332m\300n\305q\304t\303u\264v\301w\302x\263,
	cbt=\E[Z, cub1=\E[D, cud1=\E[B, cuf1=\E[C, cuu1=\E[A,
	dch1=\E[P, dl1=\E[M, ich1=\E[@, il1=\E[L, kcbt=\E[Z,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P,
	kdl1=\E[M, kend=\E[F, kf1=\E[M, kf10=\E[V, kf2=\E[N,
	kf3=\E[O, kf4=\E[P, kf5=\E[Q, kf6=\E[R, kf7=\E[S, kf8=\E[T,
	kf9=\E[U, khome=\E[H, kich1=\E[@, knp=\E[G, kpp=\E[I,
	rmsc=400\E[50;0|, smsc=250\E[?11l\E[50;1|, xoffc=g,
	xonc=e, use=att605,
att605-w|AT&T 605-w 132 column 102 key keyboard,
	cols#132, wsl#132,
	is1=\E[8;0|\E[?4;5;13;15l\E[13;20l\E[?3;7h\E[12h\E(B\E)0,
	use=att605,
# (att610: I added <rmam>/<smam> based on the init string.  I also
# added <indn> and <rin> because the BSD file says the att615s have them,
# and the 615 is like a 610 with a big keyboard, and most of their other
# smart terminals support the same sequence -- esr)
att610|AT&T 610; 80 column; 98key keyboard,
	am, eslok, hs, mir, msgr, xenl, xon,
	cols#80, it#8, lh#2, lines#24, lw#8, nlab#8, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[?25h\E[?12l, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[?12;25h, dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K,
	flash=\E[?5h$<200>\E[?5l, fsl=\E8, home=\E[H, ht=^I,
	ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L, ind=\ED,
	indn=\E[%p1%dS, invis=\E[8m,
	is1=\E[8;0|\E[?3;4;5;13;15l\E[13;20l\E[?7h\E[12h\E(B\E)0,
	is2=\E[m\017, is3=\E(B\E)0, kLFT=\E[ @, kRIT=\E[ A, kbs=^H,
	kcbt=\E[Z, kclr=\E[2J, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kf1=\EOc, kf10=\ENp, kf11=\ENq, kf12=\ENr,
	kf13=\ENs, kf14=\ENt, kf2=\EOd, kf3=\EOe, kf4=\EOf, kf5=\EOg,
	kf6=\EOh, kf7=\EOi, kf8=\EOj, kf9=\ENo, khome=\E[H,
	kind=\E[S, kri=\E[T, ll=\E[24H, mc4=\E[?4i, mc5=\E[?5i,
	nel=\EE,
	pfx=\E[%p1%d;%p2%l%02dq   F%p1%1d           %p2%s,
	pln=\E[%p1%d;0;0;0q%p2%:-16.16s, rc=\E8, rev=\E[7m,
	ri=\EM, rin=\E[%p1%dT, rmacs=^O, rmam=\E[?7l, rmir=\E[4l,
	rmln=\E[2p, rmso=\E[m, rmul=\E[m, rs2=\Ec\E[?3l, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p5%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p3%p1%|%t;7%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smir=\E[4h,
	smln=\E[p, smso=\E[7m, smul=\E[4m, tsl=\E7\E[25;%i%p1%dx,
att610-w|AT&T 610; 132 column; 98key keyboard,
	cols#132, wsl#132,
	is1=\E[8;0|\E[?4;5;13;15l\E[13;20l\E[?3;7h\E[12h,
	use=att610,

att610-103k|AT&T 610; 80 column; 103key keyboard,
	kBEG=\ENB, kCAN=\EOW, kCMD=\EOU, kCPY=\END, kCRT=\EON,
	kDC=\ENF, kDL=\ENE, kEND=\ENN, kEOL=\EOA, kEXT=\EOK,
	kFND=\EOX, kHLP=\EOM, kMOV=\ENC, kMSG=\EOL, kNXT=\ENH,
	kOPT=\EOR, kPRT=\EOZ, kPRV=\ENG, kRDO=\EOT, kRES=\EOQ,
	kRPL=\EOY, kSAV=\EOO, kSPD=\EOP, kUND=\EOS, kbeg=\E9,
	kcan=\EOw, kclo=\EOV, kcmd=\EOu, kcpy=\ENd, kcrt=\EOn,
	kdch1=\ENf, kdl1=\ENe, kel=\EOa, kend=\E0, kent=^M,
	kext=\EOk, kf10@, kf11@, kf12@, kf13@, kf14@, kf9@, kfnd=\EOx,
	khlp=\EOm, kich1=\ENj, kmov=\ENc, kmrk=\ENi, kmsg=\EOl,
	knp=\E[U, knxt=\ENh, kopn=\EOv, kopt=\EOr, kpp=\E[V,
	kprt=\EOz, kprv=\ENg, krdo=\EOt, kref=\EOb, kres=\EOq,
	krfr=\ENa, krmir=\ENj, krpl=\EOy, krst=\EOB, ksav=\EOo,
	kslt=\ENI, kspd=\EOp, kund=\EOs, use=att610,
att610-103k-w|AT&T 610; 132 column; 103key keyboard,
	cols#132, wsl#132,
	is1=\E[8;0|\E[?4;5;13;15l\E[13;20l\E[?3;7h\E[12h,
	use=att610-103k,
att615|AT&T 615; 80 column; 98key keyboard,
	kLFT=\E[ A, kRIT=\E[ @, kf15=\EOC, kf16=\EOD, kf17=\EOE,
	kf18=\EOF, kf19=\EOG, kf20=\EOH, kf21=\EOI, kf22=\EOJ,
	kf23=\ENO, kf24=\ENP, kf25=\ENQ, kf26=\ENR, kf27=\ENS,
	kf28=\ENT, kf29=\EOP, kf30=\EOQ, kf31=\EOR, kf32=\EOS,
	kf33=\EOw, kf34=\EOx, kf35=\EOy, kf36=\EOm, kf37=\EOt,
	kf38=\EOu, kf39=\EOv, kf40=\EOl, kf41=\EOq, kf42=\EOr,
	kf43=\EOs, kf44=\EOp, kf45=\EOn, kf46=\EOM, use=att610,
att615-w|AT&T 615; 132 column; 98key keyboard,
	kLFT=\E[ A, kRIT=\E[ @, kf15=\EOC, kf16=\EOD, kf17=\EOE,
	kf18=\EOF, kf19=\EOG, kf20=\EOH, kf21=\EOI, kf22=\EOJ,
	kf23=\ENO, kf24=\ENP, kf25=\ENQ, kf26=\ENR, kf27=\ENS,
	kf28=\ENT, kf29=\EOP, kf30=\EOQ, kf31=\EOR, kf32=\EOS,
	kf33=\EOw, kf34=\EOx, kf35=\EOy, kf36=\EOm, kf37=\EOt,
	kf38=\EOu, kf39=\EOv, kf40=\EOl, kf41=\EOq, kf42=\EOr,
	kf43=\EOs, kf44=\EOp, kf45=\EOn, kf46=\EOM, use=att610-w,
att615-103k|AT&T 615; 80 column; 103key keyboard,
	kLFT=\E[ A, kRIT=\E[ @, use=att610-103k,
att615-103k-w|AT&T 615; 132 column; 103key keyboard,
	kLFT=\E[ A, kRIT=\E[ @, use=att610-103k-w,
# (att620: I added <rmam>/<smam> based on the init string and
# <rin>/<indn> from a BSD termcap -- esr)
att620|AT&T 620; 80 column; 98key keyboard,
	am, eslok, hs, mir, msgr, xenl, xon,
	cols#80, it#8, lh#2, lines#24, lw#8, nlab#8, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[?25h\E[?12l, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[?12;25h, dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K,
	flash=\E[?5h$<200>\E[?5l, fsl=\E8, home=\E[H, ht=^I,
	ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L, ind=\ED,
	indn=\E[%p1%dS, invis=\E[8m,
	is1=\E[8;0|\E[?3;4;5;13;15l\E[13;20l\E[?7h\E[12h,
	is2=\E[m\017, is3=\E(B\E)0, kLFT=\E[ A, kRIT=\E[ @, kbs=^H,
	kcbt=\E[Z, kclr=\E[2J, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kf1=\EOc, kf10=\ENp, kf11=\ENq, kf12=\ENr,
	kf13=\ENs, kf14=\ENt, kf15=\EOC, kf16=\EOD, kf17=\EOE,
	kf18=\EOF, kf19=\EOG, kf2=\EOd, kf20=\EOH, kf21=\EOI,
	kf22=\EOJ, kf23=\ENO, kf24=\ENP, kf25=\ENQ, kf26=\ENR,
	kf27=\ENS, kf28=\ENT, kf29=\EOP, kf3=\EOe, kf30=\EOQ,
	kf31=\EOR, kf32=\EOS, kf33=\EOw, kf34=\EOx, kf35=\EOy,
	kf36=\EOm, kf37=\EOt, kf38=\EOu, kf39=\EOv, kf4=\EOf,
	kf40=\EOl, kf41=\EOq, kf42=\EOr, kf43=\EOs, kf44=\EOp,
	kf45=\EOn, kf46=\EOM, kf5=\EOg, kf6=\EOh, kf7=\EOi, kf8=\EOj,
	kf9=\ENo, khome=\E[H, kind=\E[S, kri=\E[T, ll=\E[24H,
	mc4=\E[?4i, mc5=\E[?5i, nel=\EE,
	pfx=\E[%p1%d;%p2%l%02dq   F%p1%1d           %p2%s,
	pln=\E[%p1%d;0;0;0q%p2%:-16.16s, rc=\E8, rev=\E[7m,
	ri=\EM, rin=\E[%p1%dT, rmacs=\E(B\017, rmam=\E[?7l,
	rmir=\E[4l, rmln=\E[2p, rmso=\E[m, rmul=\E[m,
	rs2=\Ec\E[?3l, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p5%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p3%p1%|%t;7%;%?%p7%t;8%;m%?%p9%t\E)0\016%e\E(B\017%;,
	sgr0=\E[m\E(B\017, smacs=\E)0\016, smam=\E[?7h,
	smir=\E[4h, smln=\E[p, smso=\E[7m, smul=\E[4m,
	tsl=\E7\E[25;%i%p1%dx,
att620-w|AT&T 620; 132 column; 98key keyboard,
	cols#132, wsl#132,
	is1=\E[8;0|\E[?4;5;13;15l\E[13;20l\E[?3;7h\E[12h,
	use=att620,
att620-103k|AT&T 620; 80 column; 103key keyboard,
	kBEG=\ENB, kCAN=\EOW, kCMD=\EOU, kCPY=\END, kCRT=\EON,
	kDC=\ENF, kDL=\ENE, kEND=\ENN, kEOL=\EOA, kEXT=\EOK,
	kFND=\EOX, kHLP=\EOM, kMOV=\ENC, kMSG=\EOL, kNXT=\ENH,
	kOPT=\EOR, kPRT=\EOZ, kPRV=\ENG, kRDO=\EOT, kRES=\EOQ,
	kRPL=\EOY, kSAV=\EOO, kSPD=\EOP, kUND=\EOS, kbeg=\E9,
	kcan=\EOw, kclo=\EOV, kcmd=\EOu, kcpy=\ENd, kcrt=\EOn,
	kdch1=\ENf, kdl1=\ENe, kel=\EOa, kend=\E0, kent=^M,
	kext=\EOk, kf10@, kf11@, kf12@, kf13@, kf14@, kf15@, kf16@, kf17@,
	kf18@, kf19@, kf20@, kf21@, kf22@, kf23@, kf24@, kf25@, kf26@, kf27@,
	kf28@, kf29@, kf30@, kf31@, kf32@, kf33@, kf34@, kf35@, kf36@, kf37@,
	kf38@, kf39@, kf40@, kf41@, kf42@, kf43@, kf44@, kf45@, kf46@, kf9@,
	kfnd=\EOx, khlp=\EOm, kich1=\ENj, kmov=\ENc, kmrk=\ENi,
	kmsg=\EOl, knp=\E[U, knxt=\ENh, kopn=\EOv, kopt=\EOr,
	kpp=\E[V, kprt=\EOz, kprv=\ENg, krdo=\EOt, kref=\EOb,
	kres=\EOq, krfr=\ENa, krmir=\ENj, krpl=\EOy, krst=\EOB,
	ksav=\EOo, kslt=\ENI, kspd=\EOp, kund=\EOs, use=att620,

att620-103k-w|AT&T 620; 132 column; 103key keyboard,
	cols#132, wsl#132,
	is1=\E[8;0|\E[?4;5;13;15l\E[13;20l\E[?3;7h\E[12h,
	use=att620-103k,

# AT&T (formerly Teletype) 630 Multi-Tasking Graphics terminal
# The following SETUP modes are assumed for normal operation:
#	Local_Echo=Off	Gen_Flow=On	Return=CR	Received_Newline=LF
#	Font_Size=Large		Non-Layers_Window_Cols=80
#				Non-Layers_Window_Rows=60
# Other SETUP modes may be set for operator convenience or communication
# requirements.  Some capabilities assume a printer attached to the Aux EIA
# port.  This termcap description is for the Fixed Non-Layers Window.  No
# delays are specified; use "stty ixon -ixany" to enable DC3/DC1 flow control!
# (att630: added <ich1>, <blink> and <dim> from a BSD termcap file -- esr)
att630|AT&T 630 windowing terminal,
	OTbs, am, da, db, mir, msgr, npc, xon,
	cols#80, it#8, lines#60, lm#0,
	bel=^G, blink=\E[5m, cbt=\E[Z, clear=\E[H\E[J, cr=^M,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dim=\E[2m, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K,
	el1=\E[1K, home=\E[H, ht=^I, ich=\E[%p1%d@, ich1=\E[@,
	il=\E[%p1%dL, il1=\E[L, ind=\ED, indn=\E[%p1%dS, is2=\E[m,
	kbs=^H, kcbt=\E[Z, kclr=\E[2J, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P, kdl1=\E[M, kent=^M,
	kf10=\ENp, kf11=\ENq, kf12=\ENr, kf13=\ENs, kf14=\ENt,
	kf15=\ENu, kf16=\ENv, kf17=\ENw, kf18=\ENx, kf19=\ENy,
	kf20=\ENz, kf21=\EN{, kf22=\EN|, kf23=\EN}, kf24=\EN~,
	kf9=\ENo, khome=\E[H, kich1=\E[@, kil1=\E[L, mc4=\E[?4i,
	mc5=\E[?5i, nel=^M^J, pfx=\E[%p1%d;%p2%l%dq%p2%s, rc=\E8,
	rev=\E[7m, ri=\EM, rin=\E[%p1%dT, rmir=\E[4l, rmso=\E[m,
	rmul=\E[m, rs2=\Ec, sc=\E7,
	sgr=\E[0%?%p5%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p1%p3%|%p4%|%t;7%;m,
	sgr0=\E[m, smir=\E[4h, smso=\E[7m, smul=\E[4m,
att630-24|5630-24|5630DMD-24|630MTG-24|AT&T 630 windowing terminal 24 lines,
	lines#24, use=att630,

# This is the att700 entry for 700 native emulation of the AT&T 700
# terminal.  Comments are relative to changes from the 605V2 entry and
# att730 on which the entry is based.  Comments show the terminfo
# capability name, termcap name, and description.
#
# Here is what's going onm in the init string:
#	ESC [ 50;4|	set 700 native mode (really is 605)
# x	ESC [ 56;ps| 	set lines to 24: ps=0; 40: ps=1 (plus status line)
#	ESC [ 53;0|	set GenFlow to Xon/Xoff
#	ESC [ 8 ;0|	set CR on NL
# x	ESC [ ? 3 l/h	set workspace: 80 col(l); 132 col(h)
#	ESC [ ? 4 l	jump scroll
#	ESC [ ? 5 l/h	video: normal (l); reverse (h)
#	ESC [ ?13 l	Labels on
#	ESC [ ?15 l	parity check = no
#	ESC [ 13 l	monitor mode off
#	ESC [ 20 l	LF on NL (not CRLF on NL)
#	ESC [ ? 7 h	autowrap on
#	ESC [ 12 h	local echo off
#	ESC ( B		GO = ASCII
#	ESC ) 0		G1 = Special Char & Line Drawing
#	ESC [ ? 31 l	Set 7 bit controls
#
# Note: Most terminals, especially the 600 family use Reverse Video for
# standout mode.  DEC also uses reverse video.  The VT100 uses bold in addition
# Assume we should stay with reverse video for 70..  However, the 605V2 exits
# standout mode with \E[m (all normal attributes).  The 730 entry simply
# exits reverse video which would leave other current attributes intact.  It
# was assumed the 730 entry to be more correct so rmso has changed.  The
# 605V2 has no sequences to turn individual attributes off, thus its setting
# and the rmso/smso settings from the 730.
#
# Note: For the same reason as above in rmso I changed exit under-score mode
# to specifically turn off underscore, rather than return to all normal
# attributes
#
# Note: The following pkey_xmit is taken from the 605V2 which contained the
# capability as pfxl.  It was changed here to pfx since pfxl
# will only compile successfully with Unix 4.0 tic.  Also note that pfx only
# allows strings to be parameters and label values must be programmed as
# constant strings.  Supposedly the pfxl of Version 4.0 allows both labels
# and strings to be parameters.  The 605V2 pfx entry should be examined later
# in this regard. For reference the 730 pfxl entry is shown here for comparison
# 730 pfx entry:
#     pfxl=\E[%?%p1%{25}%<%t%p1%e%p1%{24}%-%;%d;%p2%l%02d%?%p1%{25}%<%tq\s\s\s
# SYS\s\s\s\s\sF%p1%:-2d\s\s%e;0;3q%;%p2%s,
#
# (for 4.0 tic)
#     pfxl=\E[%p1%d;%p2%l%02dq%?%p1%{9}%<%t   F%p1%1d           %;%p2%s,
#
# (for <4.0 tic)
#     pfx=\E[%p1%d;%p2%l%02dq%?%p1%{9}%<%t   F%p1%1d           %;%p2%s,
#
# From the AT&T 705 Multi-tasking terminal user's guide Page 8-8,8-9
#
# Port1 Interface
#
# modular 10 pin Connector
# Left side       Right side
# Pin 1 2 3 4 5 6 7 8 9 10
#
#        Key (notch) at bottom
#
# Pin    1 DSR
#        3 DCD
#        4 DTR
#        5 Sig Ground
#        6 RD
#        7 SD
#        8 CTS
#        9 RTS
#        10 Frame Ground
#
# The manual is 189 pages and is loaded with details about the escape codes,
# etc..... Available from AT&T CIC 800-432-6600...
# ask for Document number 999-300-660..
#
att700|AT&T 700 24x80 column display w/102key keyboard,
	am, eslok, hs, mir, msgr, xenl, xon,
	cols#80, it#8, lh#2, lines#24, lw#8, nlab#8, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[?25h\E[?12l, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[?12;25h, dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K,
	enacs=\E(B\E)0, flash=\E[?5h$<200>\E[?5l, fln=4\,4,
	fsl=\E8, home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@,
	il=\E[%p1%dL, il1=\E[L, ind=\ED, invis=\E[8m,
	is2=\E[50;4|\E[53;0|\E[8;0|\E[?4;13;15l\E[13;20l\E[?7h\E[12h\E(B\E)0\E[?31l\E[0m\017,
	is3=\E(B\E)0, kLFT=\E[ A, kRIT=\E[ @, kbs=^H, kcbt=\E[Z,
	kclr=\E[2J, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[P, kdl1=\E[M, kend=\E[24;1H, kf1=\EOc, kf10=\ENp,
	kf11=\ENq, kf12=\ENr, kf13=\ENs, kf14=\ENt, kf15=\EOC,
	kf16=\EOD, kf17=\EOE, kf18=\EOF, kf19=\EOG, kf2=\EOd,
	kf20=\EOH, kf21=\EOI, kf22=\EOJ, kf23=\ENO, kf24=\ENP,
	kf25=\ENQ, kf26=\ENR, kf27=\ENS, kf28=\ENT, kf29=\EOq,
	kf3=\EOe, kf30=\EOr, kf31=\EOs, kf32=\EOt, kf33=\EOu,
	kf34=\EOv, kf35=\EOw, kf36=\EOx, kf37=\EOy, kf38=\EOu,
	kf39=\EOv, kf4=\EOf, kf40=\EOl, kf41=\EOq, kf42=\EOr,
	kf43=\EOs, kf44=\EOp, kf45=\EOn, kf46=\EOM, kf5=\EOg,
	kf6=\EOh, kf7=\EOi, kf8=\EOj, kf9=\ENo, khome=\E[H,
	kich1=\E[@, kil1=\E[L, knp=\E[U, kpp=\E[V, ll=\E[24H,
	mc0=\E[i, mc4=\E[?4i, mc5=\E[?5i, nel=\EE,
	pfx=\E[%p1%d;%p2%l%02dq%?%p1%{9}%<%t   F%p1%1d           %;%p2%s,
	pln=\E[%p1%d;0;0;0q%p2%:-16.16s, rc=\E8,
	rep=%p1%c\E[%p2%{1}%-%db, rev=\E[7m, ri=\EM, rmacs=^O,
	rmir=\E[4l, rmln=\E[2p, rmso=\E[27m, rmul=\E[24m,
	rmxon=\E[53;3|, rs1=\Ec\E[?3;5l\E[56;0|, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p5%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p3%p1%|%t;7%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smir=\E[4h, smln=\E[p, smso=\E[7m,
	smul=\E[4m, smxon=\E[53;0|, tbc=\E[3g,
	tsl=\E7\E[99;%i%p1%dx,

# This entry was modified 3/13/90 by JWE.
# fixes include additions of <enacs>, correcting <rep>, and modification
# of <kHOM>.  (See comments below)
# att730 has status line of 80 chars
# These were commented out: <indn=\E[%p1%dS>, <rin=\E[%p1%dT>,
# the <kf25> and up keys are used for shifted system Fkeys
# NOTE: JWE 3/13/90 The 98 key keyboard translation for shift/HOME is
# currently the same as <khome> (unshifted HOME or \E[H).  On the 102, 102+1
# and 122 key keyboards, the 730's translation is \E[2J.  For consistency
# <kHOM> has been commented out.  The user can uncomment <kHOM> if using the
# 102, 102+1, or 122 key keyboards
#       kHOM=\E[2J,
# (att730: I added <rmam>/<smam> based on the init string -- esr)
att730|AT&T 730 windowing terminal,
	am, da, db, eslok, hs, mir, msgr, npc, xenl, xon,
	cols#80, it#8, lh#2, lines#60, lm#0, lw#8, nlab#24, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[?25h\E[?12l, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[?12;25h, dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, el1=\E[1K,
	enacs=\E(B\E)0, flash=\E[?5h$<200>\E[?5l, fsl=\E8,
	home=\E[H, ht=^I, ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L,
	ind=\ED, invis=\E[8m,
	is1=\E[8;0|\E[?3;4;5;13;15l\E[13;20l\E[?7h\E[12h\E(B\E)B,
	is2=\E[m\017, is3=\E(B\E)0, kLFT=\E[ @, kRIT=\E[ A, kbs=^H,
	kcbt=\E[Z, kclr=\E[2J, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kf1=\EOc, kf10=\ENp, kf11=\ENq, kf12=\ENr,
	kf13=\ENs, kf14=\ENt, kf15=\ENu, kf16=\ENv, kf17=\ENw,
	kf18=\ENx, kf19=\ENy, kf2=\EOd, kf20=\ENz, kf21=\EN{,
	kf22=\EN|, kf23=\EN}, kf24=\EN~, kf25=\EOC, kf26=\EOD,
	kf27=\EOE, kf28=\EOF, kf29=\EOG, kf3=\EOe, kf30=\EOH,
	kf31=\EOI, kf32=\EOJ, kf33=\ENO, kf34=\ENP, kf35=\ENQ,
	kf36=\ENR, kf37=\ENS, kf38=\ENT, kf39=\EOU, kf4=\EOf,
	kf40=\EOV, kf41=\EOW, kf42=\EOX, kf43=\EOY, kf44=\EOZ,
	kf45=\EO[, kf46=\EO\s, kf47=\EO], kf48=\EO\^, kf5=\EOg,
	kf6=\EOh, kf7=\EOi, kf8=\EOj, kf9=\ENo, khome=\E[H,
	kich1=\E[@, kil1=\E[L, kind=\E[S, kri=\E[T,
	mc0=\E[?19h\E[0i, mc4=\E[?4i, mc5=\E[?5i, nel=\EE,
	pfx=\E[%?%p1%{25}%<%t%p1%e%p1%{24}%-%;%d;%p2%l%02d%?%p1%{25}%<%tq   SYS     F%p1%:-2d  %e;0;3q%;%p2%s,
	pfxl=\E[%p1%d;%p2%l%02d;0;0q%p3%:-16.16s%p2%s,
	pln=\E[%p1%d;0;0;0q%p2%:-16.16s, rc=\E8,
	rep=%p1%c\E[%p2%{1}%-%db, rev=\E[7m, ri=\EM, rmacs=^O,
	rmam=\E[?7l, rmir=\E[4l, rmln=\E[?13h, rmso=\E[27m,
	rmul=\E[24m, rmxon=\E[?21l, rs2=\Ec\E[?3l, sc=\E7,
	sgr=\E[0%?%p6%t;1%;%?%p5%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p3%p1%|%t;7%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smam=\E[?7h, smir=\E[4h,
	smln=\E[?13l, smso=\E[7m, smul=\E[4m, smxon=\E[?21h,
	swidm=\E#6, tsl=\E7\E[;%i%p1%dx,
att730-41|730MTG-41|AT&T 730-41 windowing terminal Version,
	lines#41, use=att730,
att730-24|730MTG-24|AT&T 730-24 windowing terminal Version,
	lines#24, use=att730,
att730r|730MTGr|AT&T 730 rev video windowing terminal Version,
	flash=\E[?5l$<200>\E[?5h,
	is1=\E[8;0|\E[?3;4;13;15l\E[?5h\E[13;20l\E[?7h\E[12h\E(B\E)B,
	use=att730,
att730r-41|730MTG-41r|AT&T 730r-41 rev video windowing terminal Version,
	lines#41, use=att730r,
att730r-24|730MTGr-24|AT&T 730r-24 rev video windowing terminal Version,
	lines#24, use=att730r,

# The following represents the screen layout along with the associated
# bezel buttons for the 5430/pt505 terminal. The "kf" designations do
# not appear on the screen but are shown to reference the bezel buttons.
# The "CMD", "MAIL", and "REDRAW" buttons are shown in their approximate
# position relative to the screen.
#
#
#
#      +----------------------------------------------------------------+
#      |                                                                |
# XXXX | kf0                                                       kf24 | XXXX
#      |                                                                |
#      |                                                                |
# XXXX | kf1                                                       kf23 | XXXX
#      |                                                                |
#      |                                                                |
# XXXX | kf2                                                       kf22 | XXXX
#      |                                                                |
#      |                                                                |
# XXXX | kf3                                                       kf21 | XXXX
#      |                                                                |
#      |                                                                |
# XXXX | kf4                                                       kf20 | XXXX
#      |                                                                |
#      |                                                                |
# XXXX | kf5                                                       kf19 | XXXX
#      |                                                                |
#      |                                                                |
# XXXX | kf6                                                       kf18 | XXXX
#      |                                                                |
#      |                                                                |
# XXXX |                                                                | XXXX
#      |                                                                |
#      |                                                                |
#      +----------------------------------------------------------------+
#
#          XXXX  XXXX  XXXX  XXXX  XXXX  XXXX  XXXX  XXXX  XXXX  XXXX
#
# Note: XXXX represents the screen buttons
#                                                          CMD   REDRAW
#
#                                                          MAIL
#
# version 1 note:
#	The character string sent by key 'kf26' may be user programmable
#       to send either \E[16s, or \E[26s.
#       The character string sent by key 'krfr' may be user programmable
#       to send either \E[17s, or \E[27s.
#
# Depression of the "CMD" key sends    \E!    (kcmd)
# Depression of the "MAIL" key sends   \E[26s (kf26)
# "REDRAW" same as "REFRESH" (krfr)
#
# "kf" functions adds carriage return to output string if terminal is in
# 'new line' mode.
#
# The following are functions not covered in the table above:
#
#       Set keyboard character (SKC): \EPn1;Pn2w
#                       Pn1= 0 Back Space key
#                       Pn1= 1 Break key
#                       Pn2=   Program char (hex)
#
#       Screen Definition (SDF): \E[Pn1;Pn2;Pn3;Pn4;Pn5t
#                       Pn1=     Window number (1-39)
#                       Pn2-Pn5= Y;X;Y;X coordinates
#
#       Screen Selection (SSL): \E[Pnu
#                       Pn= Window number
#
#       Set Terminal Modes (SM): \E[Pnh
#                       Pn= 3 Graphics mode
#                       Pn= > Cursor blink
#                       Pn= < Enter new line mode
#                       Pn= = Enter reverse insert/replace mode
#                       Pn= ? Enter no scroll mode
#
#       Reset Terminal Mode (RM): \E[Pnl
#                       Pn= 3 Exit graphics mode
#                       Pn= > Exit cursor blink
#                       Pn= < Exit new line mode
#                       Pn= = Exit reverse insert/replace mode
#                       Pn= ? Exit no scroll mode
#
#       Screen Status Report (SSR): \E[Pnp
#                       Pn= 0 Request current window number
#                       Pn= 1 Request current window dimensions
#
#       Device Status Report (DSR): \E[6n    Request cursor position
#
#       Call Status Report (CSR): \E[Pnv
#                       Pn= 0 Call failed
#                       Pn= 1 Call successful
#
#       Transparent Button String (TBS): \E[Pn1;Pn2;Pn3;{string
#                       Pn1= Button number to be loaded
#                       Pn2= Character count of "string"
#                       Pn3= Key mode being loaded:
#                               0= Unshifted
#                               1= Shifted
#                               2= Control
#                       String= Text string (15 chars max)
#
#       Screen Number Report (SNR): \E[Pnp
#                       Pn= Screen number
#
#       Screen Dimension Report (SDR): \E[Pn1;Pn2r
#                       Pn1= Number of rows available in window
#                       Pn2= Number of columns available in window
#
#       Cursor Position Report (CPR): \E[Pn1;Pn2R
#                       Pn1= "Y" Position of cursor
#                       Pn2= "X" Position of cursor
#
#       Request Answer Back (RAB): \E[c
#
#       Answer Back Response (ABR): \E[?;*;30;VSV
#                       *=  0 No printer available
#                       *=  2 Printer available
#                       V=  Software version number
#                       SV= Software sub version number
#	(printer-available field not documented in v1)
#
#       Screen Alignment Aid: \En
#
#       Bell (lower pitch): \E[x
#
#       Dial Phone Number: \EPdstring\
#                       string= Phone number to be dialed
#
#       Set Phone Labels: \EPpstring\
#                       string= Label for phone buttons
#
#       Set Clock: \EPchour;minute;second\
#
#       Position Clock: \EPsY;X\
#                       Y= "Y" coordinate
#                       X= "X" coordinate
#
#       Delete Clock: \Epr\
#
#       Programming The Function Buttons: \EPfPn;string\
#                       Pn= Button number (00-06, 18-24)
#                                         (kf00-kf06, kf18-kf24)
#                       string= Text to sent on button depression
#
# The following in version 2 only:
#
#       Request For Local Directory Data: \EPp12;\
#
#       Local Directory Data to host: \EPp11;LOCAL...DIRECTORY...DATA\
#
#	Request for Local Directory Data in print format: \EPp13;\
#
#	Enable 'Prt on Line' mode: \022 (DC2)
#
#	Disable 'Prt on Line' mode: \024 (DC4)
#

# 05-Aug-86:
# The following Terminfo entry describes functions which are supported by
# the AT&T 5430/pt505 terminal software version 2 and later.
att505|pt505|att5430|gs5430|AT&T Personal Terminal 505 or 5430 GETSET terminal,
	am, xon,
	cols#80, it#8, lines#24,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[2J\E[H,
	cnorm=\E[>l, cr=^M, cub=\E[%p1%dD, cub1=\E[D,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[>h, dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[0J, el=\E[0K, el1=\E[2K, home=\E[H, ht=^I,
	il=\E[%p1%dL, il1=\E[L, ind=^J,
	is1=\EPr\\E[0u\E[2J\E[0;0H\E[m\E[3l\E[<l\E[4l\E[>l\E[=l\E[?l,
	kbs=^H, kcmd=\E!, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kf0=\E[00s, kf1=\E[01s, kf18=\E[18s,
	kf19=\E[19s, kf2=\E[02s, kf20=\E[20s, kf21=\E[21s,
	kf22=\E[22s, kf23=\E[23s, kf24=\E[24s, kf26=\E[26s,
	kf3=\E[03s, kf4=\E[04s, kf5=\E[05s, kf6=\E[06s,
	krfr=\E[27s, mc4=\E[4i, mc5=\E[5i, rc=\E8, rev=\E[7m,
	rmacs=\E[10m, rmam=\E[11;1j, rmir=\E[4l, rmso=\E[m,
	rmul=\E[m, rs1=\Ec, sc=\E7, sgr0=\E[m, smacs=\E[11m,
	smam=\E[11;0j, smir=\E[4h, smso=\E[1m, smul=\E[4m,

# The following Terminfo entry describes functions which are supported by
# the AT&T 5430/pt505 terminal software version 1.
att505-24|pt505-24|gs5430-24|AT&T PT505 or 5430 GETSET version 1 24 lines,
	lines#24,
	mc4@, mc5@, rc@, rmam@, sc@, smam@, use=att505,
tt505-22|pt505-22|gs5430-22|AT&T PT505 or 5430 GETSET version 1 22 lines,
	lines#22, use=att505,
#
#### ------------------ TERMINFO FILE CAN BE SPLIT HERE ---------------------
# This cut mark helps make life less painful for people running ncurses tic
# on machines with relatively little RAM.  The file can be broken in half here
# cleanly and compiled in sections -- no `use' references cross this cut
# going forward.
#

#### Ampex (Dialogue)
#
# Yes, these are the same people who are better-known for making audio- and
# videotape.  I'm told they are located in Redwood City, CA.
#

# From: <cbosg!ucbvax!SRC:george> Fri Sep 11 22:38:32 1981
# (ampex80: some capabilities merged in from SCO's entry -- esr)
ampex80|a80|d80|dialogue|dialogue80|ampex dialogue 80,
	OTbs, am, bw, ul,
	cols#80, it#8, lines#24,
	bel=^G, cbt=\EI, clear=\E*$<75>, cr=^M, cub1=^H, cud1=^J,
	cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	dch1=\EW, dl1=\ER$<5*>, ed=\Ey, el=\Et, ht=^I, hts=\E1,
	ich1=\EQ, il1=\EE$<5*>, ind=^J, is2=\EA, rmso=\Ek, rmul=\Em,
	smso=\Ej, smul=\El, tbc=\E3,
# This entry was from somebody anonymous, Tue Aug  9 20:11:37 1983, who wrote:
ampex175|ampex d175,
	am,
	cols#80, lines#24,
	bel=^G, clear=\E+, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\Ey, el=\Et, home=^^, ich1=\EQ, il1=\EE, ind=^J,
	is2=\EX\EA\EF, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K,
	kdch1=\EW, kdl1=\ER, khome=^^, kich1=\EQ, kil1=\EE, ll=^^^K,
	rmcup=\EF, rmso=\Ek, rmul=\Em, smcup=\EN, smso=\Ej, smul=\El,
# No backspace key in the main QWERTY cluster. Fortunately, it has a
# NEWLINE/PAGE key just above RETURN that sends a strange single-character
# code.  Given a suitable Unix (one that lets you set an echo-erase-as-BS-SP-BS
# mode), this key can be used as the erase key; I find I like this. Because
# some people and some systems may not, there is another termcap ("ampex175")
# that suppresses this little eccentricity by omitting the relevant capability.
ampex175-b|ampex d175 using left arrow for erase,
	kbs=^_, use=ampex175,
# From: Richard Bascove <atd!dsd!rcb@ucbvax.berkeley.edu>
# (ampex210: removed obsolete ":kn#10:" -- esr)
ampex210|a210|ampex a210,
	OTbs, am, hs, xenl,
	cols#80, it#8, lines#24, xmc#1,
	cbt=\EI, clear=\E*, cub1=^H, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\Ey, el=\Et, flash=\EU\EX\EU\EX\EU\EX\EU\EX,
	fsl=\E.2, home=^^, ht=^I, ich1=\EQ,
	if=/usr/share/tabset/std, il1=\EE, invis@,
	is2=\EC\Eu\E'\E(\El\EA\E%\E{\E.2\EG0\Ed\En, kcub1=^H,
	kcud1=^V, kcuf1=^L, kcuu1=^K, kf0=^A0\r, kf1=^A1\r,
	kf2=^A2\r, kf3=^A3\r, kf4=^A4\r, kf5=^A5\r, kf6=^A6\r,
	kf7=^A7\r, kf8=^A8\r, kf9=^A9\r, khome=^^,
	tsl=\E.0\Eg\E}\Ef, use=adm+sgr,
# (ampex219: I added <rmam>/<smam> based on the init string, added <cvvis>
# from ampex219w, added <cnorm>=\E[?3l, irresistibly suggested by <cvvis>,
# and moved the padding to be *after* the caps -- esr)
ampex219|ampex-219|amp219|Ampex with Automargins,
	hs, xenl,
	cols#80, it#8, lines#24,
	bel=^G, blink=\E[5m$<2>, bold=\E[1m$<2>, cbt=\E[Z,
	clear=\E[H\E[2J$<50>, cnorm=\E[?3l, cr=^M,
	csr=%i\E[%p1%2d;%p2%2dr, cub1=^H, cud1=\E[B,
	cuf1=\E[C$<2>, cup=\E[%i%p1%d;%p2%dH$<5>,
	cuu1=\E[A$<2>, cvvis=\E[?3h, dim=\E[1m, ed=\E[J$<50>,
	el=\E[K$<3>, home=\E[H, ht=^I, ind=^J,
	is2=\E>\E[?1l\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf0=\E[21~,
	kf1=\E[7~, kf2=\E[8~, kf3=\E[9~, kf4=\E[10~, kf5=\E[11~,
	kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, khome=\E[H,
	rev=\E[7m, ri=\EM$<5>, rmam=\E[?7l, rmkx=\E>,
	rmso=\E[m$<2>, rmul=\E[m$<2>, sgr0=\E[m$<2>, smam=\E[?7h,
	smkx=\E=, smso=\E[7m$<2>, smul=\E[4m$<2>,
ampex219w|ampex-219w|amp219w|Ampex 132 cols,
	cols#132, lines#24,
	bel=^G, cr=^M, cud1=^J, ind=^J,
	is2=\E>\E[?3h\E[?4l\E[?5l\E[?7h\E[?8h, use=ampex219,
# (ampex232: removed <if=/usr/share/tabset/ampex>, no file and no <hts> --esr)
ampex232|ampex-232|Ampex Model 232,
	am,
	cols#80, lines#24, xmc#1,
	cbt=\EI, civis=\E.0, clear=\E+, cnorm=\E.4, cub1=^H, cud1=^V,
	cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	dch1=\EW, dl1=\ER$<5*/>, ed=\EY, el=\ET,
	flash=\Eb$<200/>\Ed, ht=^I, ich1=\EQ, il1=\EE$<5*/>,
	invis@, is2=\Eg\El, kbs=^H, kcub1=^H, kcud1=^V, kcuf1=^L,
	kcuu1=^K, kf0=^A@\r, kf1=^AA\r, kf2=^AB\r, kf3=^AC\r,
	kf4=^AD\r, kf5=^AE\r, kf6=^AF\r, kf7=^AG\r, kf8=^AH\r,
	kf9=^AI\r, khome=^^, use=adm+sgr,
# (ampex: removed <if=/usr/share/tabset/amp-132>, no file and no <hts> -- esr)
ampex232w|Ampex Model 232 / 132 columns,
	cols#132, lines#24,
	is2=\E\034Eg\El, use=ampex232,

#### Ann Arbor (aa)
#
# Ann Arbor made dream terminals for hackers -- large screen sizes and huge
# numbers of function keys.  At least some used monitors in portrait mode,
# allowing up to 76-character screen heights!  They were reachable at:
#
#	Ann Arbor Terminals
#	6175 Jackson Road
#	Ann Arbor, MI 48103
#	(313)-663-8000
#
# But in 1996 the phone number reaches some kitschy retail shop, and Ann Arbor
# can't be found on the Web; I fear they're long dead.  R.I.P.
#


# Originally from Mike O'Brien@Rand and Howard Katseff at Bell Labs.
# Highly modified 6/22 by Mike O'Brien.
# split out into several for the various screen sizes by dave-yost@rand
# Modifications made 3/82 by Mark Horton
# Modified by Tom Quarles at UCB for greater efficiency and more diversity
# status line moved to top of screen, <flash> removed 5/82
# Some unknown person at SCO then hacked the init strings to make them more
# efficient.
#
# assumes the following setup:
#   A menu: 0000 1010  0001 0000
#   B menu: 9600  0100 1000  0000 0000  1000 0000  17  19
#   C menu: 56   66   0    0    9600  0110 1100
#   D menu: 0110 1001   1   0
#
#	Briefly, the settings are for the following modes:
#	   (values are for bit set/clear with * indicating our preference
#	    and the value used to test these termcaps)
#	Note that many of these settings are irrelevent to the terminfo
#	and are just set to the default mode of the terminal as shipped
#	by the factory.
#
# A menu: 0000 1010  0001 0000
#	Block/underline cursor*
#	blinking/nonblinking cursor*
#	key click/no key click*
#	bell/no bell at column 72*
#
#	key pad is cursor control*/key pad is numeric
#	return and line feed/return for <cr> key *
#	repeat after .5 sec*/no repeat
#	repeat at 25/15 chars per sec. *
#
#	hold data until pause pressed/process data unless pause pressed*
#	slow scroll/no slow scroll*
#	Hold in area/don't hold in area*
#	functions keys have default*/function keys disabled on powerup
#
#	show/don't show position of cursor during page transmit*
#	unused
#	unused
#	unused
#
# B menu: 9600  0100 1000  0000 0000  1000 0000  17  19
#	Baud rate (9600*)
#
#	2 bits of parity - 00=odd,01=even*,10=space,11=mark
#	1 stop bit*/2 stop bits
#	parity error detection off*/on
#
#	keyboard local/on line*
#	half/full duplex*
#	disable/do not disable keyboard after data transmission*
#
#	transmit entire page/stop transmission at cursor*
#	transfer/do not transfer protected characters*
#	transmit all characters/transmit only selected characters*
#	transmit all selected areas/transmit only 1 selected area*
#
#	transmit/do not transmit line separators to host*
#	transmit/do not transmit page tab stops tabs to host*
#	transmit/do not transmit column tab stop tabs to host*
#	transmit/do not transmit graphics control (underline,inverse..)*
#
#	enable*/disable auto XON/XOFF control
#	require/do not require receipt of a DC1 from host after each LF*
#	pause key acts as a meta key/pause key is pause*
#	unused
#
#	unused
#	unused
#	unused
#	unused
#
#	XON character (17*)
#	XOFF character (19*)
#
# C menu: 56   66   0    0    9600  0110 1100
#	number of lines to print data on (printer) (56*)
#
#	number of lines on a sheet of paper (printer) (66*)
#
#	left margin (printer) (0*)
#
#	number of pad chars on new line to printer (0*)
#
#	printer baud rate (9600*)
#
#	printer parity: 00=odd,01=even*,10=space,11=mark
#	printer stop bits: 2*/1
#	print/do not print guarded areas*
#
#	new line is: 01=LF,10=CR,11=CRLF*
#	unused
#	unused
#
# D menu: 0110 1001   1   0
#	LF is newline/LF is down one line, same column*
#	wrap to preceding line if move left from col 1*/don't wrap
#	wrap to next line if move right from col 80*/don't wrap
#	backspace is/is not destructive*
#
#	display*/ignore DEL character
#	display will not/will scroll*
#	page/column tab stops*
#	erase everything*/erase unprotected only
#
#	editing extent: 0=display,1=line*,2=field,3=area
#
#	unused
#

annarbor4080|aa4080|ann arbor 4080,
	OTbs, am,
	cols#80, lines#40,
	bel=^G, clear=\014$<2>, cr=^M, cub1=^H, cud1=^J, cuf1=^_,
	cup=\017%p2%{10}%/%{16}%*%p2%{10}%m%+%c%p1%?%p1%{19}%>%t%{12}%+%;%{64}%+%c,
	cuu1=^N, home=^K, ht=^I, hts=^]^P1, ind=^J, kbs=^^, kcub1=^H,
	kcud1=^J, kcuf1=^_, kcuu1=^N, khome=^K, tbc=^\^P^P,

# Strange Ann Arbor terminal from BRL
aas1901|Ann Arbor K4080 w/S1901 mod,
	am,
	cols#80, lines#40,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=^_, cuu1=^N,
	home=^K, ht=^I, ind=^J, kbs=^H, kcub1=^H, kcud1=^J, ll=^O\0c,
	nel=^M^J,

# If you're using the GNU termcap library, add
#	:cS=\E[%p1%d;%p2%d;%p3%d;%p4%dp:
# to these capabilities.  This is the nonstandard GNU termcap scrolling
# capability, arguments are:
#   1. Total number of lines on the screen.
#   2. Number of lines above desired scroll region.
#   3. Number of lines below (outside of) desired scroll region.
#   4. Total number of lines on the screen, the same as the first parameter.
# The generic Ann Arbor entry is the only one that uses this.
aaa+unk|aaa-unk|ann arbor ambassador (internal - don't use this directly),
	OTbs, am, km, mc5i, mir, xon,
	cols#80, it#8,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z,
	clear=\E[H\E[J$<156>, cr=^M, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^K, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M, ed=\E[J,
	el=\E[K$<5>, home=\E[H, hpa=\E[%p1%{1}%+%dG, ht=^I,
	hts=\EH, ich=\E[%p1%d@$<4*>, ich1=\E[@$<4>, il=\E[%p1%dL,
	il1=\E[L$<3>, ind=^K, invis=\E[8m, is1=\E[m\E7\E[H\E9\E8,
	is3=\E[1Q\E[>20;30l\EP`+x~M\E\\, kbs=^H, kcbt=\E[Z,
	kclr=\E[J, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[P, kdl1=\E[M, kf1=\EOA, kf10=\EOJ, kf11=\EOK,
	kf12=\EOL, kf13=\EOM, kf14=\EON, kf15=\EOO, kf16=\EOP,
	kf17=\EOQ, kf18=\EOR, kf19=\EOS, kf2=\EOB, kf20=\EOT,
	kf21=\EOU, kf22=\EOV, kf23=\EOW, kf24=\EOX, kf3=\EOC,
	kf4=\EOD, kf5=\EOE, kf6=\EOF, kf7=\EOG, kf8=\EOH, kf9=\EOI,
	khome=\E[H, kich1=\E[@, kil1=\E[L, krmir=\E6, mc0=\E[0i,
	mc4=^C, mc5=\E[v, mc5p=\E[%p1%dv, rc=\E8,
	rep=%p1%c\E[%p2%{1}%-%db, rev=\E[7m,
	rmkx=\EP`>y~[[J`8xy~[[A`4xy~[[D`6xy~[[C`2xy~[[B\E\\,
	rmm=\E[>52l, rmso=\E[m, rmul=\E[m, sc=\E7,
	sgr=\E[%?%p1%t7;%;%?%p2%t4;%;%?%p3%t7;%;%?%p4%t5;%;%?%p6%t1;%;%?%p7%t8;%;m,
	sgr0=\E[m,
	smkx=\EP`>z~[[J`8xz~[[A`4xz~[[D`6xz~[[C`2xz~[[B\E\\,
	smm=\E[>52h, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	vpa=\E[%p1%{1}%+%dd,

aaa+rv|ann arbor ambassador in reverse video,
	blink=\E[5;7m, bold=\E[1;7m, invis=\E[7;8m,
	is1=\E[7m\E7\E[H\E9\E8, rev=\E[m, rmso=\E[7m, rmul=\E[7m,
	rs1=\E[H\E[7m\E[J$<156>,
	sgr=\E[%?%p2%t4;%;%?%p4%t5;%;%?%p6%t1;%;%?%p1%p2%|%p3%!%t7;%;%?%p7%t8;%;m\016,
	sgr0=\E[7m\016, smso=\E[m, smul=\E[4;7m,
# Ambassador with the DEC option, for partial vt100 compatibility.
aaa+dec|ann arbor ambassador in dec vt100 mode,
	acsc=aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}},
	csr=\E[%i%p1%d;%p2%dr, enacs=\E(0, rmacs=^N,
	sgr=\E[%?%p2%t4;%;%?%p4%t5;%;%?%p6%t1;%;%?%p1%p3%|%!%t7;%;%?%p7%t8;%;m%?%p9%t\017%e\016%;,
	smacs=^O,
aaa-18|ann arbor ambassador/18 lines,
	lines#18,
	is2=\E7\E[60;0;0;18p\E8,
	rmcup=\E[60;0;0;18p\E[60;1H\E[K, smcup=\E[18;0;0;18p,
	use=aaa+unk,
aaa-18-rv|ann arbor ambassador/18 lines+reverse video,
	use=aaa+rv, use=aaa-18,
aaa-20|ann arbor ambassador/20 lines,
	lines#20,
	is2=\E7\E[60;0;0;20p\E8,
	rmcup=\E[60;0;0;20p\E[60;1H\E[K, smcup=\E[20;0;0;20p,
	use=aaa+unk,
aaa-22|ann arbor ambassador/22 lines,
	lines#22,
	is2=\E7\E[60;0;0;22p\E8,
	rmcup=\E[60;0;0;22p\E[60;1H\E[K, smcup=\E[22;0;0;22p,
	use=aaa+unk,
aaa-24|ann arbor ambassador/24 lines,
	lines#24,
	is2=\E7\E[60;0;0;24p\E8,
	rmcup=\E[60;0;0;24p\E[60;1H\E[K, smcup=\E[24;0;0;24p,
	use=aaa+unk,
aaa-24-rv|ann arbor ambassador/24 lines+reverse video,
	use=aaa+rv, use=aaa-24,
aaa-26|ann arbor ambassador/26 lines,
	lines#26,
	is2=\E7\E[60;0;0;26p\E8,
	rmcup=\E[60;0;0;26p\E[26;1H\E[K,
	smcup=\E[H\E[J$<156>\E[26;0;0;26p, use=aaa+unk,
aaa-28|ann arbor ambassador/28 lines,
	lines#28,
	is2=\E7\E[60;0;0;28p\E8,
	rmcup=\E[60;0;0;28p\E[28;1H\E[K,
	smcup=\E[H\E[J$<156>\E[28;0;0;28p, use=aaa+unk,
aaa-30-s|aaa-s|ann arbor ambassador/30 lines w/status,
	eslok, hs,
	lines#29,
	dsl=\E7\E[60;0;0;30p\E[1;1H\E[K\E[H\E8\r\n\E[K,
	fsl=\E[>51l, is2=\r\n\E[A\E7\E[60;1;0;30p\E8,
	rmcup=\E[60;1;0;30p\E[29;1H\E[K,
	smcup=\E[H\E[J$<156>\E[30;1;0;30p\E[30;1H\E[K,
	tsl=\E[>51h\E[1;%p1%dH\E[2K, use=aaa+unk,
aaa-30-s-rv|aaa-s-rv|ann arbor ambassador/30 lines+status+reverse video,
	use=aaa+rv, use=aaa-30-s,
aaa-s-ctxt|aaa-30-s-ctxt|ann arbor ambassador/30 lines+status+save context,
	rmcup=\E[60;1;0;30p\E[59;1H\E[K,
	smcup=\E[30;1H\E[K\E[30;1;0;30p, use=aaa-30-s,
aaa-s-rv-ctxt|aaa-30-s-rv-ct|ann arbor ambassador/30 lines+status+save context+reverse video,
	rmcup=\E[60;1;0;30p\E[59;1H\E[K,
	smcup=\E[30;1H\E[K\E[30;1;0;30p, use=aaa-30-s-rv,
aaa|aaa-30|ambas|ambassador|ann arbor ambassador/30 lines,
	lines#30,
	is2=\E7\E[60;0;0;30p\E8,
	rmcup=\E[60;0;0;30p\E[30;1H\E[K,
	smcup=\E[H\E[J$<156>\E[30;0;0;30p, use=aaa+unk,
aaa-30-rv|aaa-rv|ann arbor ambassador/30 lines in reverse video,
	use=aaa+rv, use=aaa-30,
aaa-30-ctxt|aaa-ctxt|ann arbor ambassador/30 lines; saving context,
	rmcup=\E[60;0;0;30p\E[60;1H\E[K, smcup=\E[30;0;0;30p,
	use=aaa-30,
aaa-30-rv-ctxt|aaa-rv-ctxt|ann arbor ambassador/30 lines reverse video; saving context,
	rmcup=\E[60;0;0;30p\E[60;1H\E[K, smcup=\E[30;0;0;30p,
	use=aaa+rv, use=aaa-30,
aaa-36|ann arbor ambassador/36 lines,
	lines#36,
	is2=\E7\E[60;0;0;36p\E8,
	rmcup=\E[60;0;0;36p\E[36;1H\E[K,
	smcup=\E[H\E[J$<156>\E[36;0;0;36p, use=aaa+unk,
aaa-36-rv|ann arbor ambassador/36 lines+reverse video,
	use=aaa+rv, use=aaa-36,
aaa-40|ann arbor ambassador/40 lines,
	lines#40,
	is2=\E7\E[60;0;0;40p\E8,
	rmcup=\E[60;0;0;40p\E[40;1H\E[K,
	smcup=\E[H\E[J$<156>\E[40;0;0;40p, use=aaa+unk,
aaa-40-rv|ann arbor ambassador/40 lines+reverse video,
	use=aaa+rv, use=aaa-40,
aaa-48|ann arbor ambassador/48 lines,
	lines#48,
	is2=\E7\E[60;0;0;48p\E8,
	rmcup=\E[60;0;0;48p\E[48;1H\E[K,
	smcup=\E[H\E[J$<156>\E[48;0;0;48p, use=aaa+unk,
aaa-48-rv|ann arbor ambassador/48 lines+reverse video,
	use=aaa+rv, use=aaa-48,
aaa-60-s|ann arbor ambassador/59 lines+status,
	eslok, hs,
	lines#59,
	dsl=\E7\E[60;0;0;60p\E[1;1H\E[K\E[H\E8\r\n\E[K,
	fsl=\E[>51l, is2=\r\n\E[A\E7\E[60;1;0;60p\E8,
	tsl=\E[>51h\E[1;%p1%dH\E[2K, use=aaa+unk,
aaa-60-s-rv|ann arbor ambassador/59 lines+status+reverse video,
	use=aaa+rv, use=aaa-60-s,
aaa-60-dec-rv|ann arbor ambassador/dec mode+59 lines+status+rev video,
	use=aaa+dec, use=aaa+rv, use=aaa-60-s,
aaa-60|ann arbor ambassador/60 lines,
	lines#60,
	is2=\E7\E[60;0;0;60p\E[1Q\E[m\E[>20;30l\E8,
	use=aaa+unk,
aaa-60-rv|ann arbor ambassador/60 lines+reverse video,
	use=aaa+rv, use=aaa-60,
aaa-db|ann arbor ambassador 30/destructive backspace,
	OTbs@,
	cub1=\E[D, is3=\E[1Q\E[m\E[>20l\E[>30h, use=aaa-30,

guru|guru-33|guru+unk|ann arbor guru/33 lines 80 cols,
	lines#33,
	flash=\E[>59h$<100>\E[>59l,
	is2=\E7\E[255;0;0;33;80;80p\E8\E[J, is3=\E[>59l,
	rmcup=\E[255p\E[255;1H\E[K, smcup=\E[33p, use=aaa+unk,
guru+rv|guru changes for reverse video,
	flash=\E[>59l$<100>\E[>59h, is3=\E[>59h,
guru-rv|guru-33-rv|ann arbor guru/33 lines+reverse video,
	use=guru+rv, use=guru-33,
guru+s|guru status line,
	eslok, hs,
	dsl=\E7\E[;0p\E[1;1H\E[K\E[H\E8\r\n\E[K, fsl=\E[>51l,
	rmcup=\E[255;1p\E[255;1H\E[K, smcup=,
	tsl=\E[>51h\E[1;%p1%dH\E[2K,
guru-nctxt|guru with no saved context,
	smcup=\E[H\E[J$<156>\E[33p\E[255;1H\E[K, use=guru,
guru-s|guru-33-s|ann arbor guru/33 lines+status,
	lines#32,
	is2=\r\n\E[A\E7\E[255;1;0;33;80;80p\E8\E[J,
	smcup=\E[33;1p\E[255;1H\E[K, use=guru+s, use=guru+unk,
guru-24|ann arbor guru 24 lines,
	cols#80, lines#24,
	is2=\E7\E[255;0;0;24;80;80p\E8\E[J, smcup=\E[24p,
	use=guru+unk,
guru-44|ann arbor guru 44 lines,
	cols#97, lines#44,
	is2=\E7\E[255;0;0;44;97;100p\E8\E[J, smcup=\E[44p,
	use=guru+unk,
guru-44-s|ann arbor guru/44 lines+status,
	lines#43,
	is2=\r\n\E[A\E7\E[255;1;0;44;80;80p\E8\E[J,
	smcup=\E[44;1p\E[255;1H\E[K, use=guru+s, use=guru+unk,
guru-76|guru with 76 lines by 89 cols,
	cols#89, lines#76,
	is2=\E7\E[255;0;0;76;89;100p\E8\E[J, smcup=\E[76p,
	use=guru+unk,
guru-76-s|ann arbor guru/76 lines+status,
	cols#89, lines#75,
	is2=\r\n\E[A\E7\E[255;1;0;76;89;100p\E8\E[J,
	smcup=\E[76;1p\E[255;1H\E[K, use=guru+s, use=guru+unk,
guru-76-lp|guru-lp|guru with page bigger than line printer,
	cols#134, lines#76,
	is2=\E7\E[255;0;0;76;134;134p\E8\E[J, smcup=\E[76p,
	use=guru+unk,
guru-76-w|guru 76 lines by 178 cols,
	cols#178, lines#76,
	is2=\E7\E[255;0;0;76;178;178p\E8\E[J, smcup=\E[76p,
	use=guru+unk,
guru-76-w-s|ann arbor guru/76 lines+status+wide,
	cols#178, lines#75,
	is2=\r\n\E[A\E7\E[255;1;0;76;178;178p\E8\E[J,
	smcup=\E[76;1p\E[255;1H\E[K, use=guru+s, use=guru+unk,
guru-76-wm|guru 76 lines by 178 cols with 255 cols memory,
	cols#178, lines#76,
	is2=\E7\E[255;0;0;76;178;255p\E8\E[J, smcup=\E[76p,
	use=guru+unk,
aaa-rv-unk|ann arbor unknown type,
	lh#0, lw#0, nlab#0,
	blink=\E[5;7m, bold=\E[1;7m, home=\E[H, invis=\E[7;8m,
	is1=\E[7m\E7\E[H\E9\E8, rev=\E[m, rmso=\E[7m, rmul=\E[7m,
	rs1=\E[H\E[7m\E[J,
	sgr=\E[%?%p6%t1;%;%?%p2%t4;%;%?%p3%t7;%;%?%p4%t5;%;%?%p1%!%t7;%;%?%p7%t8;%;m,
	sgr0=\E[7m, smso=\E[m, smul=\E[4;7m,

#### Applied Digital Data Systems (adds)
#
# ADDS itself is long gone.  ADDS was bought by NCR, and the same group made
# ADDS and NCR terminals.  When AT&T and NCR merged, the engineering for
# terminals was merged again.  Then AT&T sold the terminal business to
# SunRiver, which later changed its  name to Boundless Technologies.  The
# engineers from Teletype, AT&T terminals, ADDS, and NCR (who are still there
# as of early 1995) are at:
#
#	Boundless Technologies
#	100 Marcus Boulevard
#	Hauppauge, NY 11788-3762
#	Vox: (800)-231-5445
#	Fax: (516)-342-7378
#	Web: http://boundless.com
#
# Their voice mail used to describe the place as "SunRiver (formerly ADDS)".
# In 1995 Boundless acquired DEC's terminals business.
#

# Regent: lowest common denominator, works on all regents.
# (regent: renamed ":bc:" to ":le:" -- esr)
regent|Adds Regent Series,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^U, cud1=^J, cuf1=^F, cuu1=^Z,
	home=\EY\s\s, ind=^J, ll=^A,
# Regent 100 has a bug where if computer sends escape when user is holding
# down shift key it gets confused, so we avoid escape.
regent100|Adds Regent 100,
	xmc#1,
	bel=^G,
	cup=\013%p1%' '%+%c\020%p2%{10}%/%{16}%*%p2%{10}%m%+%c,
	kf0=^B1\r, kf1=^B2\r, kf2=^B3\r, kf3=^B4\r, kf4=^B5\r,
	kf5=^B6\r, kf6=^B7\r, kf7=^B8\r, lf0=F1, lf1=F2, lf2=F3,
	lf3=F4, lf4=F5, lf5=F6, lf6=F7, lf7=F8, rmso=\E0@, rmul=\E0@,
	sgr0=\E0@, smso=\E0P, smul=\E0`, use=regent,
regent20|Adds Regent 20,
	bel=^G, cup=\EY%p1%{32}%+%c%p2%{32}%+%c, ed=\Ek, el=\EK,
	use=regent,
regent25|Adds Regent 25,
	bel=^G, kcub1=^U, kcud1=^J, kcuf1=^F, kcuu1=^Z, khome=^A,
	use=regent20,
regent40|Adds Regent 40,
	xmc#1,
	bel=^G, dl1=\El$<2*>, il1=\EM$<2*>, kf0=^B1\r, kf1=^B2\r,
	kf2=^B3\r, kf3=^B4\r, kf4=^B5\r, kf5=^B6\r, kf6=^B7\r,
	kf7=^B8\r, lf0=F1, lf1=F2, lf2=F3, lf3=F4, lf4=F5, lf5=F6,
	lf6=F7, lf7=F8, rmso=\E0@, rmul=\E0@, sgr0=\E0@, smso=\E0P,
	smul=\E0`, use=regent25,
regent40+|Adds Regent 40+,
	is2=\EB, use=regent40,
regent60|regent200|Adds Regent 60,
	dch1=\EE, is2=\EV\EB, kdch1=\EE, kich1=\EF, krmir=\EF,
	rmir=\EF, rmso=\ER\E0@\EV, smir=\EF, smso=\ER\E0P\EV,
	use=regent40+,
# From: <edward@onyx.berkeley.edu> Thu Jul  9 09:27:33 1981
# (viewpoint: added <kcuf1>, function key, and <dl1> capabilities -- esr)
viewpoint|addsviewpoint|adds viewpoint,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=^L, cnorm=\017\E0`, cr=^M, cub1=^H, cud1=^J,
	cuf1=^F, cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=^Z,
	cvvis=\017\E0P, dl1=\El, ed=\Ek$<16.1*>, el=\EK$<16>,
	ind=^J, is2=\017\E0`, kcub1=^U, kcud1=^J, kcuf1=^F, kcuu1=^Z,
	kf0=^B1, kf2=^B2, kf3=^B!, kf4=^B", kf5=^B#, khome=^A, ll=^A,
	rmso=^O, rmul=^O, sgr0=^O, smso=^N, smul=^N,
# Some viewpoints have bad ROMs that foo up on ^O
screwpoint|adds viewpoint with ^O bug,
	cvvis@, rmso@, rmul@, smso@, smul@, use=viewpoint,

# From: Jay S. Rouman <jsr@dexter.mi.org> 5 Jul 92
# The <civis>/<cnorm>/<sgr>/<sgr0> strings were added by ESR from specs.
# Theory; the vp3a+ wants \E0%c to set highlights, where normal=01000000,
# underline=01100000, rev=01010000, blink=01000010,dim=01000001,
# invis=01000100 and %c is the logical or of desired attributes.
# There is also a `tag bit' enabling attributes, set by \E) and unset by \E(.
#
# Update by TD - 2004:
# Adapted from
#	http://www.cs.utk.edu/~shuford/terminal/adds_viewpoint_news.txt
#
# COMMANDS                        ASCII CODE
#
# Address, Absolute               ESC,=,row,column
# Beep                            BEL
# Aux Port Enable                 ESC,@
# Aux Port Disable                ESC,A
# Backspace                       BS
# Cursor back                     BS
# Cursor down                     LF
# Cursor forward                  FF
# Cursor home                     RS
# Cursor up                       VT
# Cursor supress                  ETB
# Cursor enable                   CAN
# Erase to end of line            ESC,T
# Erase to end of page            ESC,Y
# Erase screen                    SUB
# Keyboard lock                   SI
# Keyboard unlock                 SO
# Read current cursor position    ESC,?
# Set Attribute                   ESC,0,x  (see below for values of x)
# Tag bit reset                   ESC,(
# Tag bit set                     ESC,)
# Transparent Print on            ESC,3
# Transparent Print off           ESC,4
#
#
# ATTRIBUTES
#
# Normal                          @	0100
# Half Intensity                  A	0101
# Blinking                        B	0102
# Half Intensity Blinking         C	0103
# Reverse Video                   P	0120
# Reverse Video Half Intensity    Q	0121
# Reverse Video Blinking          R	0122
# Reverse Video Half Intensity
#    Blinking                     S	0123
# Underlined                      `	0140
# Underlined Half Intensity       a	0141
# Underlined Blinking             b	0142
# Underlined Half Intensity
#    Blinking                     c	0143
# Video suppress                  D	0104
vp3a+|viewpoint3a+|adds viewpoint 3a+,
	am, bw,
	cols#80, it#8, lines#24,
	blink=\E0B\E), civis=^W, clear=\E*$<80>, cnorm=^X, cr=^M,
	cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dim=\E0A\E),
	ed=\EY$<80>, el=\ET, home=^^, ht=^I, ind=^J, invis=\E0D\E),
	kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, khome=^^,
	nel=^M^J, rev=\E0P\E), rmso=\E(,
	sgr=%?%p1%p2%|%p3%|%p4%|%p5%|%p7%|%t\E0%{64}%?%p1%t%{17}%|%;%?%p2%t%{32}%|%;%?%p3%t%{16}%|%;%?%p4%t%{2}%|%;%?%p5%t%{1}%|%;%c%?%p7%tD%;\E)%e\E(%;,
	sgr0=\E(, smso=\E0Q\E), smul=\E0`\E),
vp60|viewpoint60|addsvp60|adds viewpoint60,
	use=regent40,
#
# adds viewpoint 90 - from cornell
# Note:  emacs sends ei occasionally to insure the terminal is out of
#        insert mode. This unfortunately puts the viewpoint90 IN insert
#        mode.  A hack to get around this is <ich1=\EF\s\EF^U>.  (Also,
#   -    :ei=:im=: must be present in the termcap translation.)
#   -    <xhp> indicates glitch that attributes stick to location
#   -    <msgr> means it's safe to move in standout mode
#   -    <clear=\EG\Ek>: clears screen and visual attributes without affecting
#               the status line
# Function key and label capabilities merged in from SCO.
vp90|viewpoint90|adds viewpoint 90,
	OTbs, bw, msgr, xhp,
	cols#80, lines#24,
	clear=\EG\Ek, cub1=^H, cud1=^J, cuf1=^F,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=^Z, dch1=\EE,
	dl1=\El, ed=\Ek, el=\EK, home=\EY\s\s, ht=^I,
	ich1=\EF \EF\025, ind=^J, kbs=^H, kcub1=^U, kcud1=^J,
	kcuf1=^F, kcuu1=^Z, kf0=^B1\r, kf1=^B2\r, kf10=^B;\r,
	kf2=^B3\r, kf3=^B4\r, kf4=^B5\r, kf5=^B6\r, kf6=^B7\r,
	kf7=^B8\r, kf8=^B9\r, kf9=^B\:\r, khome=^A, lf0=F1, lf1=F2,
	lf10=F11, lf2=F3, lf3=F4, lf4=F5, lf5=F6, lf6=F7, lf7=F8, lf8=F9,
	lf9=F10, ll=^A, rmso=\ER\E0@\EV, rmul=\ER\E0@\EV,
	sgr0=\ER\E0@\EV, smso=\ER\E0Q\EV, smul=\ER\E0`\EV,
# Note: if return acts weird on a980, check internal switch #2
# on the top chip on the CONTROL pc board.
adds980|a980|adds consul 980,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=\014$<1>\013@, cr=^M, cub1=^H, cud1=^J,
	cuf1=\E^E01, cup=\013%p1%{64}%+%c\E\005%p2%2d,
	dl1=\E\017$<13>, il1=\E\016$<13>, ind=^J, kf0=\E0, kf1=\E1,
	kf2=\E2, kf3=\E3, kf4=\E4, kf5=\E5, kf6=\E6, kf7=\E7, kf8=\E8,
	kf9=\E9, rmso=^O, sgr0=^O, smso=^Y^^^N,

#### C. Itoh Electronics
#
# As of 1995 these people no longer make terminals (they're still in the
# printer business).  Their terminals were all clones of the DEC VT series.
# They're located in Orange County, CA.
#

# CIT 80  - vt-52 emulator, the termcap has been modified to remove
#           the delay times and do an auto tab set rather than the indirect
#           file used in vt100.
cit80|cit-80|citoh 80,
	OTbs, am,
	cols#80, lines#24,
	clear=\E[H\EJ, cr=^M, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%2d;%p2%2dH, cuu1=\E[A, ed=\EJ, el=\EK, ff=^L,
	ind=^J, is2=\E>, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC,
	kcuu1=\EOA, rmkx=\E[?1l\E>, smkx=\E[?1h\E=,
# From: Tim Wood <mtxinu!sybase!tim> Fri Sep 27 09:39:12 PDT 1985
# (cit101: added <rmam>/<smam> based on init string, merged this with c101 -- esr)
cit101|citc|C.itoh fast vt100,
	OTbs, am, xenl,
	cols#80, lines#24,
	bel=^G, clear=\E[H\E[2J, cnorm=\E[V\E8, cub1=^H, cud1=\E[B,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A,
	cvvis=\E7\E[U, dch1=\E[P, dl1=\E[M, ed=\E[J, el=\E[K,
	flash=\E[?5h$<200/>\E[?5l, ich1=\E[@, il1=\E[L,
	is2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h\E[3g\E[>5g,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	rmam=\E[?7l, rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m, smam=\E[?7h, smkx=\E[?1h\E=, smso=\E[7m,
	smul=\E[4m,
# CIE Terminals CIT-101e from Geoff Kuenning <callan!geoff> via BRL
# The following termcap entry was created from the Callan cd100 entry.  The
# last two lines (with the capabilities in caps) are used by RM-cobol to allow
# full selection of combinations of reverse video, underline, and blink.
# (cit101e: removed unknown :f0=\EOp:f1=\EOq:f2=\EOr:f3=\EOs:f4=\EOt:f5=\EOu:\
# f6=\EOv:f7=\EOw:f8=\EOx:f9=\EOy:AB=\E[0;5m:AL=\E[m:AR=\E[0;7m:AS=\E[0;5;7m:\
# :NB=\E[0;1;5m:NM=\E[0;1m:NR=\E[0;1;7m:NS=\E[0;1;5;7m: -- esr)
cit101e|C. Itoh CIT-101e,
	OTbs, OTpt, am, mir, msgr,
	cols#80, it#8, lines#24,
	acsc=, clear=\E[H\E[J, cnorm=, csr=\E[%i%p1%2d;%p2%2dr,
	cub1=\E[D, cud1=\E[B, cuf1=\E[C, cup=\E[%i%p1%2d;%p2%2dH,
	cuu1=\E[A, cvvis=\E[?1l\E[?4l\E[?7h, dch1=\E[P, dl1=\E[M,
	ed=\E[J, el=\E[K, if=/usr/share/tabset/vt100, il1=\E[L,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf0=\EOT,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf5=\EOm, kf6=\EOl,
	kf7=\EOM, kf8=\EOn, rc=\E8, ri=\EM, rmacs=^O, rmir=\E[4l,
	rmkx=\E>, rmso=\E[m, rmul=\E[m, sc=\E7, smacs=^N, smir=\E[4h,
	smkx=\E=, smso=\E[7m, smul=\E[4m,
# From: David S. Lawyer, June 1997:
# The CIT 101-e was made in Japan in 1983-4 and imported by CIE
# Terminals in Irvine, CA.  It was part of CITOH Electronics.  In the
# late 1980's CIT Terminals went out of business.
# There is no need to use the initialization string is=... (by invoking
# tset or setterm etc.) provided that the terminal has been manually set
# up (and the setup saved with ^S) to be compatible with this termcap.  To be
# compatible it should be in ANSI mode (not VT52).   A set-up that
# works is to set all the manually setable stuff to factory defaults
# by pressing ^D in set-up mode.  Then increse the brighness with the
# up-arrow key since the factory default will likely be dim on an old
# terminal.  Then change any options you want (provided that they are
# compatible with the termcap).  For my terminal I set: Screen
# Background: light; Keyclicks: silent; Auto wraparound: on; CRT saver:
# on.  I also set up mine for parity (but you may not need it).  Then
# save the setup with ^S.
# (cit101e-rv: added empty <rmcup> to suppress a tic warning. --esr)
cit101e-rv|Citoh CIT-101e (sets reverse video),
	am, eo, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24,
	OTnl=\EM, bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z,
	civis=\E[1v, clear=\E[H\E[J, cnorm=\E[0;3;4v, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[3;5v, dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el=\E[K, flash=\E[?5l\E[?5h$<200/>,
	home=\E[H, hpa=\E[%i%p1%dG, ht=^I, hts=\EH, ich=\E[%p1%d@,
	ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=^J, indn=\E[%p1%dS,
	is2=\E<\E>\E[?1l\E[?3l\E[?4l\E[?5h\E[?7h\E[?8h\E[3g\E[>5g\E(B\E[m\E[20l\E[1;24r\E[24;1H,
	kbs=\177, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, nel=\EE, rc=\E8,
	rev=\E[7m, ri=\EM, rin=\E[%p1%dT, rmcup=, rmir=\E[4l,
	rmso=\E[m, rmul=\E[m, rs1=\Ec\E[?7h\E[>5g, sc=\E7,
	sgr0=\E[m, smcup=\E[>5g\E[?7h\E[?5h, smir=\E[4h,
	smso=\E[7m, smul=\E[4m, tbc=\E[3g, u6=\E[%i%p1%d;%p2%dR,
	u7=\E[6n, u8=\E[?6c, u9=\E[c,
cit101e-n|CIT-101e w/o am,
	am@,
	cvvis=\E[?1l\E[?4l\E[?7l, kbs=^H, kcub1=^H, kcud1=^J,
	use=cit101e,
cit101e-132|CIT-101e with 132 cols,
	cols#132,
	kbs=^H, kcub1=^H, kcud1=^J, use=cit101e,
cit101e-n132|CIT-101e with 132 cols w/o am,
	am@,
	cols#132,
	cvvis=\E[?1l\E[?4l\E[?7l, kbs=^H, kcub1=^H, kcud1=^J,
	use=cit101e,
# CIE Terminals CIT-500 from BRL
# The following SET-UP modes are assumed for normal operation:
#	GENERATE_XON/XOFF:YES	DUPLEX:FULL		NEWLINE:OFF
#	AUTOWRAP:ON		MODE:ANSI		SCREEN_LENGTH:64_LINES
#	DSPLY_CNTRL_CODES?NO	PAGE_WIDTH:80		EDIT_MODE:OFF
# Other SET-UP modes may be set for operator convenience or communication
# requirements.
# Hardware tabs are assumed to be set every 8 columns; they can be set up
# by the "reset", "tset", or "tabs" utilities.  No delays are specified; use
# "stty ixon -ixany" to enable DC3/DC1 flow control!
# (cit500: I added <rmam>/<smam> based on the init string -- esr)
cit500|CIE Terminals CIT-500,
	OTbs, OTpt, mir, msgr, xon,
	OTkn#10, cols#80, it#8, lines#64, vt#3,
	acsc=, bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z,
	clear=\E[H\E[J, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\EM, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ed=\EJ, el=\EK, home=\E[H, ht=^I, hts=\EH, il=\E[%p1%dL,
	il1=\E[L, ind=^J, is2=\E<\E)0, kbs=^H, kcbt=\E[Z, kcub1=\EOD,
	kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kdch1=\E[P, kdl1=\E[M,
	ked=\EJ, kel=\EK, kf0=\EOP, kf1=\EOQ, kf2=\EOR, kf3=\EOS,
	kf4=\EOU, kf5=\EOV, kf6=\EOW, kf7=\EOX, kf8=\EOY, kf9=\EOZ,
	khome=\E[H, kich1=\E[4h, kil1=\E[L, krmir=\E[4l, lf0=PF1,
	lf1=PF2, lf2=PF3, lf3=PF4, lf4=F15, lf5=F16, lf6=F17, lf7=F18,
	lf8=F19, lf9=F20, ll=\E[64H, nel=\EE, rc=\E8, rev=\E[7m,
	ri=\EM, rmacs=^O, rmam=\E[?7l, rmir=\E[4l, rmkx=\E[?1l\E>,
	rmso=\E[m, rmul=\E[m,
	rs1=\E<\E2\E[20l\E[?6l\E[r\E[m\E[q\E(B\017\E)0\E>,
	sc=\E7, sgr0=\E[m, smacs=^N, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,

# C. Itoh printers begin here
citoh|ci8510|8510|c.itoh 8510a,
	cols#80, it#8,
	bold=\E!, cub1@,
	is2=\E(009\,017\,025\,033\,041\,049\,057\,065\,073.,
	rep=\ER%p2%03d%p1%c, ri=\Er, rmul=\EY, sgr0=\E"\EY,
	smul=\EX, use=lpr,
citoh-pica|citoh in pica,
	is1=\EN, use=citoh,
citoh-elite|citoh in elite,
	cols#96,
	is1=\EE,
	is2=\E(009\,017\,025\,033\,041\,049\,057\,065\,073\,081\,089.,
	use=citoh,
citoh-comp|citoh in compressed,
	cols#136,
	is1=\EQ,
	is2=\E(009\,017\,025\,033\,041\,049\,057\,065\,073\,081\,089\,097\,105\,113\,121\,129.,
	use=citoh,
# citoh has infinite cols because we don't want lp ever inserting \n\t**.
citoh-prop|citoh-ps|ips|citoh in proportional spacing mode,
	cols#32767,
	is1=\EP, use=citoh,
citoh-6lpi|citoh in 6 lines per inch mode,
	is3=\EA, use=citoh,
citoh-8lpi|citoh in 8 lines per inch mode,
	lines#88,
	is3=\EB, use=citoh,

#### Control Data (cdc)
#

cdc456|cdc 456 terminal,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=^Y^X, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E1%p1%{32}%+%c%p2%{32}%+%c, cuu1=^Z, dl1=\EJ, ed=^X,
	el=^V, home=^Y, il1=\EL, ind=^J,

# Assorted CDC terminals from BRL (improvements by DAG & Ferd Brundick)
cdc721|CDC Viking,
	OTbs, am,
	cols#80, lines#24,
	clear=^L, cuf1=^X, cup=\002%p2%{32}%+%c%p1%{32}%+%c,
	cuu1=^W, el=^K, home=^Y, kcub1=^H, kcud1=^J, kcuf1=^I,
	kcuu1=^W, khome=^Y,
cdc721ll|CDC Vikingll,
	OTbs, am,
	cols#132, lines#24,
	clear=^L, cuf1=^X, cup=\002%p2%{32}%+%c%p1%{32}%+%c,
	cuu1=^W, el=^K, home=^Y, kcub1=^H, kcud1=^J, kcuf1=^I,
	kcuu1=^W, khome=^Y,
# (cdc752: the BRL entry had :ll=\E1  ^Z: commented out
cdc752|CDC 752,
	OTbs, am, bw, xhp,
	cols#80, lines#24,
	bel=^G, clear=\030\E1\s\s, cr=^M, cub1=^H, cud1=^J, cuf1=^U,
	cup=\E1%p2%{32}%+%c%p1%{32}%+%c, cuu1=^Z, el=^V,
	home=\E1\s\s, ind=^J, ll=^Y, rs1=\E1  \030\002\003\017,
# CDC 756
# The following switch/key settings are assumed for normal operation:
#	96 chars	SCROLL		FULL duplex	not BLOCK
# Other switches may be set according to communication requirements.
# Insert/delete-character cannot be used, as the whole display is affected.
# "so" & "se" are commented out until jove handles "sg" correctly.
cdc756|CDC 756,
	OTbs, am, bw,
	OTkn#10, cols#80, lines#24,
	bel=^G, clear=^Y^X, cr=^M, cub1=^H, cud1=^J, cuf1=^U,
	cup=\E1%p2%{32}%+%c%p1%{32}%+%c, cuu1=^Z,
	dl1=\EJ$<6*/>, ed=^X, el=^V, home=^Y, il1=\EL$<6*/>, ind=^J,
	kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^U, kcuu1=^Z, kdch1=\EI,
	kdl1=\EL, ked=^X, kel=^V, kf0=\EA, kf1=\EB, kf2=\EC, kf3=\ED,
	kf4=\EE, kf5=\EF, kf6=\EG, kf7=\EH, kf8=\Ea, kf9=\Eb, khome=^Y,
	khts=^O, kich1=\EK, kil1=\EL, lf0=F1, lf1=F2, lf2=F3, lf3=F4,
	lf4=F5, lf5=F6, lf6=F7, lf7=F8, lf8=F9, lf9=F10, ll=^Y^Z,
	rs1=\031\030\002\003\017,
#
# CDC 721 from Robert Viduya, Ga. Tech. <ihnp4!gatech!gitpyr!robert> via BRL.
#
# Part of the long initialization string defines the "DOWN" key to the left
# of the tab key to send an ESC.  The real ESC key is positioned way out
# in right field.
#
# The termcap won't work in 132 column mode due to the way it it moves the
# cursor.  Termcap doesn't have the capability (as far as I could tell) to
# handle the 721 in 132 column mode.
#
# (cdc721: changed :ri: to :sr: -- esr)
cdc721-esc|Control Data 721,
	OTbs, OTpt, am, bw, msgr, xon,
	OTkn#10, cols#80, it#8, lines#30,
	bel=^G, blink=^N, cbt=^^^K, clear=^L, cub1=^H, cud1=^Z,
	cuf1=^X, cup=\002%p2%{32}%+%c%p1%{32}%+%c, cuu1=^W,
	dch1=^^N, dim=^\, dl1=^^Q, ed=^^P, el=^K, home=^Y, hts=^^^RW,
	ich1=^^O, il1=^^R, ind=\036W =\036U, invis=^^^R[,
	is2=\036\022B\003\036\035\017\022\025\035\036E\036\022H\036\022J\036\022L\036\022N\036\022P\036\022Q\036\022\036\022\^\036\022b\036\022i\036W =\036\022Z\036\011C1-` `!k/o,
	kbs=^H, kcub1=^H, kcud1=^Z, kcuf1=^X, kcuu1=^W, kf0=^^q,
	kf1=^^r, kf2=^^s, kf3=^^t, kf4=^^u, kf5=^^v, kf6=^^w, kf7=^^x,
	kf8=^^y, kf9=^^z, khome=^Y, ll=^B =, rev=^^D,
	ri=\036W =\036V, rmir=, rmkx=^^^Rl, rmso=^^E, rmul=^],
	sgr0=\017\025\035\036E\036\022\\, smir=, smkx=^^^Rk,
	smso=^^D, smul=^\, tbc=^^^RY,

#### Getronics
#
# Getronics is a Dutch electronics company that at one time was called
# `Geveke' and made async terminals; but (according to the company itself!)
# they've lost all their documentation on the command set.  The hardware
# documentation suggests the terminals were actually manufactured by a
# Taiwanese electronics company named Cal-Comp.  There are known
# to have been at least two models, the 33 and the 50.
#

# The 50 seems to be a top end vt220 clone, with the addition of a higher
# screen resolution, a larger screen, at least 1 page of memory above and
# below the screen, apparently pages of memory right and left of the screen
# which can be panned, and about 75 function keys (15 function keys x normal,
# shift, control, func A, func B). It also has more setup possibilities than
# the vt220. The monitor case is dated November 1978 and the keyboard case is
# May 1982.
#
# The vt100 emulation works as is.  The entry below describes the rather
# non-conformant (but more featureful) ANSI mode.
#
# From: Stephen Peterson <stv@utrecht.ow.nl>, 27 May 1995
visa50|geveke visa 50 terminal in ansi 80 character mode,
	bw, mir, msgr,
	cols#80, lines#25,
	acsc=0_aaffggh jjkkllmmnnooqqssttuuvvwwxx, bel=^G,
	blink=\E[5m, bold=\E[1m, cbt=\E[Z, clear=\E[H\E[2J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=\E[D,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dX, dch1=\E[X, dim=\E[2m, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el=\E[K, flash=\E[?5h\E[?5l, home=\E[H,
	hpa=\E[%i%p1%dG, ht=^I, ich=\E[%p1%d@, ich1=\E[@,
	il=\E[%p1%dL, il1=\E[L, ind=^J, invis=\E[8m,
	is2=\E0;2m\E[1;25r\E[25;1H\E[?3l\E[?7h\E[?8h,
	ka1=\E[f, ka3=\EOQ, kb2=\EOP, kbs=^H, kc1=\EOR, kc3=\EOS,
	kcub1=\E[D, kcud1=\E[A, kcuf1=\E[C, kcuu1=\E[A, kdch1=\177,
	kdl1=\EOS, kf0=\E010, kf1=\E001, kf10=\E011, kf2=\E002,
	kf3=\E003, kf4=\E004, kf5=\E005, kf6=\E006, kf7=\E007,
	kf8=\E008, kf9=\E009, khome=\E[f, lf2=A delete char,
	lf3=A insert line, lf4=A delete line, lf5=A clear,
	lf6=A ce of/cf gn, lf7=A print, lf8=A on-line,
	lf9=A funcl0=A send, nel=^M^J, rev=\E[7m, rmacs=\E[3l,
	rmam=\E[?7l, rmir=\E[4l, rmkx=\E>, rmso=\E[0;2m,
	rmul=\E[0m, sgr0=\E[0;2m, smacs=\E3h, smam=\E?7h,
	smir=\E[4h, smkx=\E=, smso=\E[2;7m, smul=\E[4m, tbc=\E[3g,
	vpa=\E[%i%p1%dd,

#### Human Designed Systems (Concept)
#
#	Human Designed Systems
#	400 Fehley Drive
#	King of Prussia, PA 19406
#	Vox: (610)-277-8300
#	Fax: (610)-275-5739
#	Net: support@hds.com
#
# John Martin <john@hds.com> is their termcap expert.  They're mostly out of
# the character-terminal business now (1995) and making X terminals.  In
# particular, the whole `Concept' line described here was discontinued long
# ago.
#

# From: <vax135!hpk>  Sat Jun 27 07:41:20 1981
# Extensive changes to c108 by arpavax:eric Feb 1982
# Some unknown person at SCO then translated it to terminfo.
#
# There seem to be a number of different versions of the C108 PROMS
# (with bug fixes in its Z-80 program).
#
# The first one that we had would lock out the keyboard of you
# sent lots of short lines (like /usr/dict/words) at 9600 baud.
# Try that on your C108 and see if it sends a ^S when you type it.
# If so, you have an old version of the PROMs.
#
# You should configure the C108 to send ^S/^Q before running this.
# It is much faster (at 9600 baud) than the c100 because the delays
# are not fixed.
# new status line display entries for c108-8p:
# <is3> - init str #3 - setup term for status display -
# set programmer mode, select window 2, define window at last
# line of memory, set bkgnd stat mesg there, select window 0.
#
# <tsl> - to status line - select window 2, home cursor, erase to
# end-of-window, 1/2 bright on, goto(line#0, col#?)
#
# <fsl> - from status line - 1/2 bright off, select window 0
#
# <dsl> - disable status display - set bkgnd status mesg with
# illegal window #
#
# There are probably more function keys that should be added but
# I don't know what they are.
#
# No delays needed on c108 because of ^S/^Q handshaking
#
c108|concept108|c108-8p|concept108-8p|concept 108 w/8 pages,
	is3=\EU\E z"\Ev\001\177 !p\E ;"\E z \Ev  \001\177p\Ep\n,
	rmcup=\Ev  \001\177p\Ep\r\n, use=c108-4p,
c108-4p|concept108-4p|concept 108 w/4 pages,
	OTbs, eslok, hs, xon,
	pb@,
	acsc=jEkTl\\mMqLxU, cnorm=\Ew, cr=^M,
	cup=\Ea%p1%?%p1%{95}%>%t\001%{96}%-%;%{32}%+%c%p2%?%p2%{95}%>%t\001%{96}%-%;%{32}%+%c,
	cvvis=\EW, dch1=\E 1$<16*>, dsl=\E ;\177, fsl=\Ee\E z\s,
	ind=^J, is1=\EK\E!\E F,
	is3=\EU\E z"\Ev\177 !p\E ;"\E z \Ev  \001 p\Ep\n,
	rmacs=\Ej\s, rmcup=\Ev  \001 p\Ep\r\n, smacs=\Ej!,
	smcup=\EU\Ev  8p\Ep\r\E\025,
	tsl=\E z"\E?\E\005\EE\Ea %+\s, use=c100,
c108-rv|c108-rv-8p|concept 108 w/8 pages in reverse video,
	rmcup=\Ev  \002 p\Ep\r\n, smcup=\EU\Ev  8p\Ep\r,
	use=c108-rv-4p,
c108-rv-4p|concept108rv4p|concept 108 w/4 pages in reverse video,
	flash=\EK$<200>\Ek, is1=\Ek, rmso=\Ee, smso=\EE,
	use=c108-4p,
c108-w|c108-w-8p|concept108-w-8|concept108-w8p|concept 108 w/8 pages in wide mode,
	cols#132,
	is1=\E F\E", rmcup=\Ev  ^A0\001D\Ep\r\n,
	smcup=\EU\Ev  8\001D\Ep\r, use=c108-8p,

# Concept 100:
# These have only window relative cursor addressing, not screen
# relative. To get it to work right here, smcup/rmcup (which
# were invented for the concept) lock you into a one page
# window for screen style programs.
#
# To get out of the one page window, we use a clever trick:
# we set the window size to zero ("\Ev    " in rmcup) which the
# terminal recognizes as an error and resets the window to all
# of memory.
#
# This trick works on c100 but does not on c108, sigh.
#
# Some tty drivers use cr3 for concept, others use nl3, hence
# the delays on cr and ind below. This padding is only needed at
# 9600 baud and up.  One or the other is commented out depending on
# local conventions.
#
# 2 ms padding on <rmcup> isn't always enough. 6 works fine. Maybe
# less than 6 but more than 2 will work.
#
# Note: can't use function keys f7-f10 because they are
# indistinguishable from arrow keys (!), also, del char and
# clear eol use xon/xoff so they probably won't work very well.
#
# Also note that we don't define insrt/del char/delline/eop/send
# because they don't transmit unless we reset them - I figured
# it was a bad idea to clobber their definitions.
#
# The <mc5> sequence changes the escape character to ^^ so that
# escapes will be passed through to the printer. Only trouble
# is that ^^ won't be - ^^ was chosen to be unlikely.
# Unfortunately, if you're sending raster bits through to be
# plotted, any character you choose will be likely, so we lose.
#
# \EQ"\EY(^W (send anything from printer to host, for xon/xoff)
# cannot be # in is2 because it will hang a c100 with no printer
# if sent twice.
c100|concept100|concept|c104|c100-4p|hds concept 100,
	OTbs, am, eo, mir, ul, xenl,
	cols#80, lines#24, pb#9600, vt#8,
	bel=^G, blink=\EC, clear=\E?\E\005$<2*>, cr=$<9>\r,
	cub1=^H, cud1=^J, cuf1=\E=,
	cup=\Ea%p1%{32}%+%c%p2%{32}%+%c, cuu1=\E;,
	dch1=\E\021$<16*>, dim=\EE, dl1=\E\002$<3*>,
	ed=\E\005$<16*>, el=\E\025$<16>, flash=\Ek$<200>\EK,
	ht=\011$<8>, il1=\E\022$<3*>, ind=^J, invis=\EH, ip=$<16*>,
	is1=\EK,
	is2=\EU\Ef\E7\E5\E8\El\ENH\E\0\Eo&\0\Eo'\E\Eo!\0\E\007!\E\010A@ \E4#\:"\E\:a\E4#;"\E\:b\E4#<"\E\:c,
	is3=\Ev    $<6>\Ep\n, kbs=^H, kcbt=\E', kctab=\E_,
	kcub1=\E>, kcud1=\E<, kcuf1=\E=, kcuu1=\E;, kdch1=\E^Q,
	kdl1=\E^B, ked=\E^C, kel=\E^S, kf1=\E5, kf2=\E6, kf3=\E7,
	kf4=\E8, kf5=\E9, kf6=\E\:a, kf7=\E\:b, kf8=\E\:c, khome=\E?,
	khts=\E], kich1=\E^P, kil1=\E^R, kind=\E[, knp=\E-, kpp=\E.,
	kri=\E\\, krmir=\E\0, mc4=\036o \E\EQ!\EYP\027,
	mc5=\EQ"\EY(\027\EYD\Eo \036, prot=\EI,
	rep=\Er%p1%c%p2%{32}%+%c$<.2*>, rev=\ED,
	rmcup=\Ev    $<6>\Ep\r\n, rmir=\E\s\s, rmkx=\Ex,
	rmso=\Ed, rmul=\Eg, sgr0=\EN@,
	smcup=\EU\Ev  8p\Ep\r\E\025$<16>, smir=\E^P, smkx=\EX,
	smso=\ED, smul=\EG,
c100-rv|c100-rv-4p|concept100-rv|c100 rev video,
	cnorm@, cvvis@, flash=\EK$<200>\Ek, is1=\Ek, rmso=\Ee,
	smso=\EE, use=c100,
oc100|oconcept|c100-1p|old 1 page concept 100,
	in,
	is3@, use=c100,

# From: Walter Skorski <walt@genetics1.JMP.TJU.EDU>, 16-oct-1996.
# Lots of notes, originally inline, but ncurses doesn't grok that.
#
# am: 	not available in power on mode, but turned on with \E[=107;207h in
#	is2=.  Also, \E=124l in is2= could have been used to prevent needing
#	to specify xenl:, but that would have rendered the last space on the
#	last line useless.
# bw:	Not available in power on mode, but turned on with \E[=107;207h in
#	is2=.
# clear: Could be done with \E[2J alone, except that vi (and probably most
#	other programs) assume that this also homes the cursor.
# dsl:	Go to window 2, go to the beginning of the line, use a line feed to
#	scroll the window, and go back to window 1.
# is2:	the string may cause a warning to be issued by tic that it
#	found a very long line and that it suspects that a comma is missing
#	somewhere.  This warning can be ignored (unless it comes up more than
#	once).  The initialization string contains the following commands:
#
#	 [Setup mode items changed from factory defaults:]
#		\E)0			set alternate character set to
#						graphics
#		^O			set character set to default
#	 [In case it wasn't]
#		\E[m			turn off all attributes
#	 [In case they weren't off]
#		\E[=107;		cursor wrap and
#			207h			character wrap on
#		\E[90;3u		set Fkey definitions to "transmit"
#						defaults
#		\E[92;3u		set cursor key definitions to
#						"transmit" defaults
#		\E[43;1u		set shift F13 to transmit...
#		\177\E$P\177
#		\E[44;1u		set shift F14 to transmit...
#			\177\E$Q\177
#		\E[45;1u		set shift F15 to transmit...
#			\177\E$R\177
#		\E[46;1u		set shift F16 to transmit...
#			\177\E$S\177
#		\E[200;1u		set shift up to transmit...
#			\177\E$A\177
#		\E[201;1u		set shift down to transmit...
#			\177\E$B\177
#		\E[202;1u		set shift right to transmit...
#			\177\E$C\177
#		\E[203;1u		set shift left to transmit...
#			\177\E$D\177
#		\E[204;1u		set shift home to transmit...
#			\177\E$H\177
#		\E[212;1u		set backtab to transmit...
#			\177\E$I\177
#		\E[213;1u		set shift backspace to transmit...
#			\177\E$^H\177
#		\E[214;1u		set shift del to transmit...
#			"\E$\177"
#	 [Necessary items not mentioned in setup mode:]
#		\E[2!w			move to window 2
#		\E[25;25w		define window as line 25 of memory
#		\E[!w			move to window 1
#		\E[2*w			show current line of window 2 as
#						status line
#		\E[2+x			set meta key to use high bit
#		\E[;3+}			move underline to bottom of character
#
#	All Fkeys are set to their default transmit definitions with \E[90;3u
#	in is2=.  IMPORTANT:  to use this terminal definition, the "quit" stty
#	setting MUST be redefined or deactivated, because the default is
#	contained in almost all of this terminal's Fkey strings!  If for some
#	reason "quit" cannot be altered, the Fkeys can, but it would be
#	necessary to change ^| to ^] in all of these definitions, and add
#	\E[2;029!t to is2.
# lines: is set to 24 because this terminal refuses to treat the 25th
#	line normally.
# ll:	Not available in power on mode, but turned on with \E[=107;207h in
#	is2=.
# lm:	Pointless, given that this definition locks a single screen of
#	memory into view, but what the hey...
# rmso: Could use \E[1;7!{ to turn off only bold and reverse (leaving any
#	other attributes alone), but some programs expect this to turn off
#	everything.
# rmul: Could use \E[4!{ to turn off only underline (leaving any other
#	attributes alone), but some programs expect this to turn off
#	everything.
# sgr:	Attributes are set on this terminal with the string \E[ followed by
#	a list of attribute code numbers (in decimal, separated by
#	semicolons), followed by the character m.  The attribute code
#	numbers are:
#		  1 for bold;
#		  2 for dim (which is ignored in power on mode);
#		  4 for underline;
#		  5 for blinking;
#		  7 for inverse;
#		  8 for not displayable; and
#		=99 for protected (except that there are strange side
#		effects to protected characters which make them inadvisable).
#	 The mapping of terminfo parameters to attributes is as follows:
#		%p1 (standout) = bold and inverse together;
#		%p2 (underline) = underline;
#		%p3 (reverse) = inverse;
#		%p4 (blink) = blinking;
#		%p5 (dim) is ignored;
#		%p6 (bold) = bold;
#		%p7 (invisible) = not displayable;
#		%p8 (protected) is ignored; and
#		%p9 (alt char set) = alt char set.
#	 The code to do this is:
#		\E[0		OUTPUT	\E[0
#		%?%p1%p6%O	IF	(standout; bold) OR
#		%t;1		THEN	OUTPUT	;1
#		%;		ENDIF
#		%?%p2		IF	underline
#		%t;4		THEN	OUTPUT	;4
#		%;		ENDIF
#		%?%p4		IF	blink
#		%t;5		THEN	OUTPUT	;5
#		%;		ENDIF
#		%?%p1%p3%O	IF	(standout; reverse) OR
#		%t;7		THEN	OUTPUT	;7
#		%;		ENDIF
#		%?%p7		IF	invisible
#		%t;8		THEN	OUTPUT	;8
#		%;		ENDIF
#		m		OUTPUT	m
#		%?%p9		IF	altcharset
#		%t^N		THEN	OUTPUT	^N
#		%e^O		ELSE	OUTPUT	^O
#		%;		ENDIF
# sgr0: Everything is turned off (including alternate character set), since
#	there is no way of knowing what it is that the program wants turned
#	off.
# smul: The "underline" attribute is reconfigurable to an overline or
#	strikethru, or (as done with \E[;3+} in is2=), to a line at the true
#	bottom of the character cell.  This was done to allow for more readable
#	underlined characters, and to be able to distinguish between an
#	underlined space, an underscore, and an underlined underscore.
# xenl: Terminal can be configured to not need this, but this "glitch"
#	behavior is actually preferable with autowrap terminals.
#
# Parameters kf31= thru kf53= actually contain the strings sent by the shifted
# Fkeys.  There are no parameters for shifted Fkeys in terminfo.  The is2
# string modifies the 'O' in kf43 to kf46 to a '$'.
#
# kcbt was originally ^I but redefined in is2=.
# kHOM was \E[H originally but redefined in is2=, as were a number of
# other keys.
# kDC was originally \177 but redefined in is2=.
#
# kbs:	Shift was also ^H originally but redefined as \E$^H in is2=.
# tsl:	Go to window 2, then do an hpa=.
#
#------- flash=\E[8;3!}^G\E[3;3!}
#------- flash=\E[?5h$<100>\E[?5l
# There are two ways to flash the screen, both of which have their drawbacks.
# The first is to set the bell mode to video, transmit a bell character, and
# set the bell mode back - but to what?  There is no way of knowing what the
# user's old bell setting was before we messed with it.  Worse, the command to
# set the bell mode also sets the key click volume, and there is no way to say
# "leave that alone", or to know what it's set to, either.
# The second way to do a flash is to set the screen to inverse video, pad for a
# tenth of a second, and set it back - but like before, there's no way to know
# that the screen wasn't ALREADY in inverse video, or that the user may prefer
# it that way.  The point is moot anyway, since vi (and probably other
# programs) assume that by defining flash=, you want the computer to use it
# INSTEAD of bel=, rather than as a secondary type of signal.
#
#------- cvvis=\E[+{
# The is the power on setting, which is also as visible as the cursor
# gets.
#-------  wind=\E[%i%p1%d;%p2%d;%p3%{1}%+%d;%p4%{1}%+%dw
# Windowing is possible, but not defined here because it is also used to
# emulate status line functions.  Allowing a program to set a window could
# clobber the status line or render it unusable.  There is additional memory,
# but screen scroll functions are destructive and do not make use of it.
#
#-------   dim=			Not available in power on mode.
# You have a choice of defining low intensity characters as "half bright" and
# high intensity as "normal", or defining low as "normal" and high as "bold".
# No matter which you choose, only one of either "half bright" or "bold" is
# available at any time, so taking the time to override the default is
# pointless.
#
#-------  prot=\E[=0;99m
# Not defined, because it appears to have some strange side effects.
#------- pfkey=%?%p1%{24}%<%p1%{30}%>%p1%{54}%<%A%O%t\E[%p1%du\177%p2%s\177%;
#------- pfloc=%?%p1%{24}%<%p1%{30}%>%p1%{54}%<%A%O%t\E[%p1%du\177%p2%s\177%;
#-------   pfx=%?%p1%{24}%<%p1%{30}%>%p1%{54}%<%A%O%t\E[%p1%d;1u\177%p2%s\177%;
#	 Available, but making them available to programs is inadvisable.
#	 The code to do this is:
#		%?%p1%{24}%<	IF	((key; 24) <;
#		%p1%{30}%>		 ((key; 30) >;
#		%p1%{54}%<		  (key; 54) <
#		%A			 ) AND
#		%O		  	) OR
#	 [that is, "IF key < 24 OR (key > 30 AND key < 54)",]
#		%t\E[		THEN	OUTPUT	\E[
#		%p1%d			OUTPUT	(key) as decimal
#	 [next line applies to pfx only]
#		;1			OUTPUT	;1
#		u			OUTPUT	u
#		\177			OUTPUT	\177
#		%p2%s			OUTPUT	(string) as string
#		\177			OUTPUT	\177
#	 [DEL chosen as delimiter, but could be any character]
#	 [implied:		ELSE	do nothing]
#		%;		ENDIF
#
#-------   rs2=
# Not defined since anything it might do could be done faster and easier with
# either Meta-Shift-Reset or the main power switch.
#
#-------  smkx=\E[1!z
#-------  rmkx=\E[!z
# These sequences apply to the cursor and setup keys only, not to the
# numeric keypad.  But it doesn't matter anyway, since making these
# available to programs is inadvisable.
# For the key definitions below, all sequences beginning with \E$ are
# custom and programmed into the terminal via is2.  \E$ also has no
# meaning to any other terminal.
#
#------- cmdch=\E[;%p1%d!t
# Available, but making it available to programs is inadvisable.
#------- smxon=\E[1*q
# Available, but making it available to programs is inadvisable.
# Terminal will send XON/XOFF on buffer overflow.
#------- rmxon=\E[*q
# Available, but making it available to programs is inadvisable.
# Terminal will not notify on buffer overflow.
#-------   smm=\E[2+x
#-------   rmm=\E[+x
# Available, but making them available to programs is inadvisable.
#
# Printing:
#	 It's not made clear in the manuals, but based on other ansi/vt type
#	 terminals, it's a good guess that this terminal is capable of both
#	 "transparent print" (which doesn't copy data to the screen, and
#	 therefore needs mc5i: specified to say so) and "auxilliary print"
#	 (which does duplicate printed data on the screen, in which case mc4=
#	 and mc5= should use the \E[?4i and \E[?5i strings instead).

hds200|Human Designed Systems HDS200,
	am, bw, eslok, hs, km, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, lm#0,
	acsc=``aaffggjjkkllmmnnooqqssttuuvvwwxx~~, bel=^G,
	blink=\E[0;5m, bold=\E[0;1m, cbt=\E[Z, civis=\E[6+{,
	clear=\E[H\E[J, cnorm=\E[+{, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=\E[D,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	dsl=\E[2!w\r\n\E[!w, ed=\E[J, el=\E[K, el1=\E[1K,
	fsl=\E[!w, home=\E[H, hpa=\E[%i%p1%dG, ht=^I, hts=\EH,
	ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L, ind=\ED,
	invis=\E[0;8m,
	is2=\E)0\017\E[m\E[=107;207h\E[90;3u\E[92;3u\E[43;1u\177\E$P\177\E[44;1u\177\E$Q\177\E[45;1u\177\E$R\177\E[46;1u\177\E$S\177\E[200;1u\177\E$A\177\E[201;1u\177\E$B\177\E[202;1u\177\E$C\177\E[203;1u\177\E$D\177\E[204;1u\177\E$H\177\E[212;1u\177\E$I\177\E[213;1u\177\E$\010\177\E[214;1u"\E$\177"\E[2!w\E[25;25w\E[!w\E[2*w\E[2+x\E[;3+},
	kDC=\E$\177, kHOM=\E$H, kLFT=\E$D, kRIT=\E$C, kbs=^H,
	kcbt=\E$I, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\177, kent=^M, kf1=^\001\r, kf10=^\010\r,
	kf11=^\011\r, kf12=^\012\r, kf13=\EOP, kf14=\EOQ,
	kf15=\EOR, kf16=\EOS, kf17=^\017\r, kf18=^\018\r,
	kf19=^\019\r, kf2=^\002\r, kf20=^\020\r, kf21=^\021\r,
	kf22=^\022\r, kf23=^\023\r, kf3=^\003\r, kf31=^\031\r,
	kf32=^\032\r, kf33=^\033\r, kf34=^\034\r, kf35=^\035\r,
	kf36=^\036\r, kf37=^\037\r, kf38=^\038\r, kf39=^\039\r,
	kf4=^\004\r, kf40=^\040\r, kf41=^\041\r, kf42=^\042\r,
	kf43=\E$P, kf44=\E$Q, kf45=\E$R, kf46=\E$S, kf47=^\047\r,
	kf48=^\048\r, kf49=^\049\r, kf5=^\005\r, kf50=^\050\r,
	kf51=^\051\r, kf52=^\052\r, kf53=^\053\r, kf6=^\006\r,
	kf7=^\007\r, kf8=^\008\r, kf9=^\009\r, khome=\E[H,
	kind=\E[T, knp=\E[U, kpp=\E[V, kri=\E[S, ll=\E[H\E[A,
	nel=\E[E, rc=\E8, rev=\E[0;7m, ri=\EM, rmacs=^O, rmir=\E[4l,
	rmso=\E[m\017, rmul=\E[m\017, sc=\E7,
	sgr=\E[0%?%p1%p6%O%t;1%;%?%p2%t;4%;%?%p4%t;5%;%?%p1%p3%O%t;7%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smir=\E[4h, smso=\E[0;1;7m,
	smul=\E[0;4m, tbc=\E[3g, tsl=\E[2!w\E[%i%p1%dG,
	vpa=\E[%i%p1%dd, use=ansi+pp,

# <ht> through <el> included to specify padding needed in raw mode.
# (avt-ns: added empty <acsc> to suppress a tic warning --esr)
avt-ns|concept avt no status line,
	OTbs, am, eo, mir, ul, xenl, xon,
	cols#80, it#8, lines#24, lm#192,
	acsc=, bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z,
	clear=\E[H\E[J$<38>, cnorm=\E[=119l, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[=119h, dch1=\E[P, dim=\E[1!{, dl=\E[%p1%dM$<4*>,
	dl1=\E[M$<4>, ed=\E[J$<96>, el=\E[K$<6>, home=\E[H,
	hpa=\E[%p1%{1}%+%dG, ht=\011$<4>, hts=\EH, ich=\E[%p1%d@,
	ich1=\E[@, il=\E[%p1%dL$<4*>, il1=\E[L$<4>, ind=\n$<8>,
	invis=\E[8m, ip=$<4>, is1=\E[=103l\E[=205l,
	is2=\E[1*q\E[2!t\E[7!t\E[=4;101;119;122l\E[=107;118;207h\E)1\E[1Q\EW\E[!y\E[!z\E>\E[0\:0\:32!r\E[0*w\E[w\E2\r\n\E[2;27!t,
	kbs=^H, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E^B\r, ked=\E^D\r, kf1=\EOP, kf2=\EOQ, kf3=\EOR,
	kf4=\EOS, khome=\E[H, kich1=\E^A\r, kil1=\E^C\r, ll=\E[24H,
	mc0=\E[0i, mc4=\E[4i, mc5=\E[5i,
	pfloc=\E[%p1%d;0u#%p2%s#, pfx=\E[%p1%d;1u#%p2%s#,
	prot=\E[99m, rc=\E8, rep=%p1%c\E[%p2%{1}%-%db, rev=\E[7m,
	ri=\EM$<4>, rmacs=\016$<1>, rmcup=\E[w\E2\r\n,
	rmir=\E[4l, rmkx=\E[!z\E[0;2u, rmso=\E[7!{, rmul=\E[4!{,
	sc=\E7,
	sgr=\E[%?%p1%t7;%;%?%p2%t4;%;%?%p3%t7;%;%?%p4%t5;%;%?%p6%t1;%;%?%p7%t8;%;%?%p8%t99;%;m%?%p5%t\E[1!{%;%?%p9%t\017%e\016%;$<1>,
	sgr0=\E[m\016$<1>, smacs=\017$<1>,
	smcup=\E[=4l\E[1;24w\E2\r, smir=\E[4h,
	smkx=\E[1!z\E[0;3u, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	vpa=\E[%p1%{1}%+%dd,
avt-rv-ns|concept avt in reverse video mode/no status line,
	flash=\E[=205l$<200>\E[=205h, is1=\E[=103l\E[=205h,
	use=avt-ns,
avt-w-ns|concept avt in 132 column mode/no status line,
	is1=\E[=103h\E[=205l, smcup=\E[H\E[1;24;1;132w,
	use=avt-ns,
avt-w-rv-ns|concept avt in 132 column mode/no status line/reverse video,
	flash=\E[=205l$<200>\E[=205h, is1=\E[=103h\E[=205h,
	smcup=\E[H\E[1;24;1;132w, use=avt-ns,

# Concept AVT with status line. We get the status line using the
# "Background status line" feature of the terminal. We swipe the
# first line of memory in window 2 for the status line, keeping
# 191 lines of memory and 24 screen lines for regular use.
# The first line is used instead of the last so that this works
# on both 4 and 8 page AVTs. (Note the lm#191 or 192 - this
# assumes an 8 page AVT but lm isn't currently used anywhere.)
#
avt+s|concept avt status line changes,
	eslok, hs,
	lm#191,
	dsl=\E[0*w, fsl=\E[1;1!w,
	is3=\E[2w\E[2!w\E[1;1;1;80w\E[H\E[2*w\E[1!w\E2\r\n,
	rmcup=\E[2w\E2\r\n, smcup=\E[2;25w\E2\r,
	tsl=\E[2;1!w\E[;%p1%dH\E[2K,
avt|avt-s|concept-avt|avt w/80 columns,
	use=avt+s, use=avt-ns,
avt-rv|avt-rv-s|avt reverse video w/sl,
	flash=\E[=205l$<200>\E[=205h, is1=\E[=103l\E[=205h,
	use=avt+s, use=avt-ns,
avt-w|avt-w-s|concept avt 132 cols+status,
	is1=\E[=103h\E[=205l, smcup=\E[H\E[1;24;1;132w,
	use=avt+s, use=avt-ns,
avt-w-rv|avt-w-rv-s|avt wide+status+rv,
	flash=\E[=205l$<200>\E[=205h, is1=\E[=103h\E[=205h,
	smcup=\E[H\E[1;24;1;132w, use=avt+s, use=avt-ns,

#### Contel Business Systems.
#

# Contel c300 and c320 terminals.
contel300|contel320|c300|Contel Business Systems C-300 or C-320,
	am, in, xon,
	cols#80, lines#24, xmc#1,
	bel=^G, clear=\EK, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EX%p1%{32}%+%c\EY%p2%{32}%+%c, cuu1=\EA,
	dch1=\EO$<5.5*>, dl1=\EM$<5.5*>, ed=\EJ$<5.5*>,
	el=\EI$<5.5>, flash=\020\002$<200/>\020\003, home=\EH,
	hts=\E1, ich1=\EN, il1=\EL$<5.5*>, ind=^J, ip=$<5.5*>,
	kbs=^H, kf0=\ERJ, kf1=\ERA, kf2=\ERB, kf3=\ERC, kf4=\ERD,
	kf5=\ERE, kf6=\ERF, kf7=\ERG, kf8=\ERH, kf9=\ERI, ll=\EH\EA,
	rmso=\E!\0, sgr0=\E!\0, smso=\E!\r, tbc=\E3,
# Contel c301 and c321 terminals.
contel301|contel321|c301|c321|Contel Business Systems C-301 or C-321,
	flash@, ich1@, ip@, rmso=\E!\0$<20>, smso=\E!\r$<20>,
	use=contel300,

#### Data General (dg)
#
# According to James Carlson <carlson@xylogics.com> writing in January 1995,
# the terminals group at Data General was shut down in 1991; all these
# terminals have thus been discontinued.
#
# DG terminals have function keys that respond to the SHIFT and CTRL keys,
# e.g., SHIFT-F1 generates a different code from F1.  To number the keys
# sequentially, first the unmodified key codes are listed as F1 through F15.
# Then their SHIFT versions are listed as F16 through F30, their CTRL versions
# are listed as F31 through F45, and their CTRL-SHIFT versions are listed as
# F46 through F60.  This is done in the private "includes" below whose names
# start with "dgkeys+".
#
# DG terminals generally support 8 bit characters.  For each of these terminals
# two descriptions are supplied:
#	1) A default description for 8 bits/character communications, which
#	   uses the default DG international character set and keyboard codes.
#	2) A description with suffix "-7b" for 7 bits/character communications.
#	   This description must use the NON-DEFAULT native keyboard language.

# Unmodified fkeys (kf1-kf11), Shift fkeys (kf12-kf22), Ctrl fkeys (kf23-kf33),
# Ctrl/Shift fdkeys (kf34-kf44).

dgkeys+8b|Private entry describing DG terminal 8-bit ANSI mode special keys,
	ka1=\233020z, ka3=\233021z, kc1=\233022z, kc3=\233023z,
	kclr=\2332J, kcub1=\233D, kcud1=\233B, kcuf1=\233C,
	kcuu1=\233A, kel=\233K, kf1=\233001z, kf10=\233010z,
	kf11=\233011z, kf12=\233012z, kf13=\233013z,
	kf14=\233014z, kf15=\233000z, kf16=\233101z,
	kf17=\233102z, kf18=\233103z, kf19=\233104z,
	kf2=\233002z, kf20=\233105z, kf21=\233106z,
	kf22=\233107z, kf23=\233108z, kf24=\233109z,
	kf25=\233110z, kf26=\233111z, kf27=\233112z,
	kf28=\233113z, kf29=\233114z, kf3=\233003z,
	kf30=\233100z, kf31=\233201z, kf32=\233202z,
	kf33=\233203z, kf34=\233204z, kf35=\233205z,
	kf36=\233206z, kf37=\233207z, kf38=\233208z,
	kf39=\233209z, kf4=\233004z, kf40=\233210z,
	kf41=\233211z, kf42=\233212z, kf43=\233213z,
	kf44=\233214z, kf45=\233200z, kf46=\233301z,
	kf47=\233302z, kf48=\233303z, kf49=\233304z,
	kf5=\233005z, kf50=\233305z, kf51=\233306z,
	kf52=\233307z, kf53=\233308z, kf54=\233309z,
	kf55=\233310z, kf56=\233311z, kf57=\233312z,
	kf58=\233313z, kf59=\233314z, kf6=\233006z,
	kf60=\233300z, kf7=\233007z, kf8=\233008z, kf9=\233009z,
	khome=\233H, kprt=\233i,

dgkeys+7b|Private entry describing DG terminal 7-bit ANSI mode special keys,
	ka1=\E[020z, ka3=\E[021z, kc1=\E[022z, kc3=\E[023z,
	kclr=\E[2J, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kel=\E[K, kf1=\E[001z, kf10=\E[010z, kf11=\E[011z,
	kf12=\E[012z, kf13=\E[013z, kf14=\E[014z, kf15=\E[000z,
	kf16=\E[101z, kf17=\E[102z, kf18=\E[103z, kf19=\E[104z,
	kf2=\E[002z, kf20=\E[105z, kf21=\E[106z, kf22=\E[107z,
	kf23=\E[108z, kf24=\E[109z, kf25=\E[110z, kf26=\E[111z,
	kf27=\E[112z, kf28=\E[113z, kf29=\E[114z, kf3=\E[003z,
	kf30=\E[100z, kf31=\E[201z, kf32=\E[202z, kf33=\E[203z,
	kf34=\E[204z, kf35=\E[205z, kf36=\E[206z, kf37=\E[207z,
	kf38=\E[208z, kf39=\E[209z, kf4=\E[004z, kf40=\E[210z,
	kf41=\E[211z, kf42=\E[212z, kf43=\E[213z, kf44=\E[214z,
	kf45=\E[200z, kf46=\E[301z, kf47=\E[302z, kf48=\E[303z,
	kf49=\E[304z, kf5=\E[005z, kf50=\E[305z, kf51=\E[306z,
	kf52=\E[307z, kf53=\E[308z, kf54=\E[309z, kf55=\E[310z,
	kf56=\E[311z, kf57=\E[312z, kf58=\E[313z, kf59=\E[314z,
	kf6=\E[006z, kf60=\E[300z, kf7=\E[007z, kf8=\E[008z,
	kf9=\E[009z, khome=\E[H, kprt=\E[i,

dgkeys+11|Private entry describing 11 minimal-subset DG mode special keys,
	kclr=^L, kcub1=^Y, kcud1=^Z, kcuf1=^X, kcuu1=^W, kel=^K,
	kf1=^^q, kf10=^^z, kf11=^^{, kf12=^^a, kf13=^^b, kf14=^^c,
	kf15=^^d, kf16=^^e, kf17=^^f, kf18=^^g, kf19=^^h, kf2=^^r,
	kf20=^^i, kf21=^^j, kf22=^^k, kf23=^^1, kf24=^^2, kf25=^^3,
	kf26=^^4, kf27=^^5, kf28=^^6, kf29=^^7, kf3=^^s, kf30=^^8,
	kf31=^^9, kf32=^^\:, kf33=^^;, kf34=^^!, kf35=^^", kf36=^^#,
	kf37=^^$, kf38=^^%%, kf39=^^&, kf4=^^t, kf40=^^', kf41=^^(,
	kf42=^^), kf43=^^*, kf44=^^+, kf5=^^u, kf6=^^v, kf7=^^w,
	kf8=^^x, kf9=^^y, khome=^H,

dgkeys+15|Private entry describing 15 DG mode special keys,
	kHOM=^^^H, kLFT=^^^Y, kRIT=^^^X, ka1=^^\\, ka3=^^], kc1=^^\^,
	kc3=^^_, kf1=^^q, kf10=^^z, kf11=^^{, kf12=^^|, kf13=^^},
	kf14=^^~, kf15=^^p, kf16=^^a, kf17=^^b, kf18=^^c, kf19=^^d,
	kf2=^^r, kf20=^^e, kf21=^^f, kf22=^^g, kf23=^^h, kf24=^^i,
	kf25=^^j, kf26=^^k, kf27=^^l, kf28=^^m, kf29=^^n, kf3=^^s,
	kf30=^^`, kf31=^^1, kf32=^^2, kf33=^^3, kf34=^^4, kf35=^^5,
	kf36=^^6, kf37=^^7, kf38=^^8, kf39=^^9, kf4=^^t, kf40=^^\:,
	kf41=^^;, kf42=^^<, kf43=^^=, kf44=^^>, kf45=^^0, kf46=^^!,
	kf47=^^", kf48=^^#, kf49=^^$, kf5=^^u, kf50=^^%%, kf51=^^&,
	kf52=^^', kf53=^^(, kf54=^^), kf55=^^*, kf56=^^+, kf57=^^\,,
	kf58=^^-, kf59=^^., kf6=^^v, kf60=^^\s, kf7=^^w, kf8=^^x,
	kf9=^^y,

# Data General color terminals use the "Tektronix" color model.  The total
# number of colors varies with the terminal model, as does support for
# attributes used in conjunction with color.

# Removed u7, u8 definitions since they conflict with tack:
#		Preserve user-defined colors in at least some cases.
#	u7=^^Fh,
#		Default is ACM mode.
#	u8=^^F}20^^Fi^^F}21,
#
dgunix+fixed|Fixed color info for DG D430C terminals in DG-UNIX mode,
	bce,
	colors#16, ncv#53, pairs#256,
	op=\036Ad\036Bd,
	setab=\036B%p1%?%p1%{8}%<%t%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%;%{48}%+%c,
	setaf=\036A%p1%?%p1%{8}%<%t%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%;%{48}%+%c,
	setb=\036B%p1%{48}%+%c, setf=\036A%p1%{48}%+%c,

dg+fixed|Fixed color info for DG D430C terminals in DG mode,
	use=dgunix+fixed,

# Video attributes are coordinated using static variables set by "sgr", then
# checked by "op", "seta[bf]", and "set[bf]" to refresh the attribute settings.
# (D=dim, U=underline, B=blink, R=reverse.)
dg+color8|Color info for Data General D220 and D230C terminals in ANSI mode,
	bce,
	colors#8, ncv#16, pairs#64,
	op=\E[%?%gD%t2;%;%?%gU%t4;%;%?%gB%t5;%;%?%gR%t7;%;m,
	setab=\E[4%p1%d%?%gD%t;2%;%?%gU%t;4%;%?%gB%t;5%;%?%gR%t;7%;m,
	setaf=\E[3%p1%d%?%gD%t;2%;%?%gU%t;4%;%?%gB%t;5%;%?%gR%t;7%;m,
	setb=\E[4%p1%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%d%?%gD%t;2%;%?%gU%t;4%;%?%gB%t;5%;%?%gR%t;7%;m,
	setf=\E[3%p1%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%d%?%gD%t;2%;%?%gU%t;4%;%?%gB%t;5%;%?%gR%t;7%;m,

dg+color|Color info for Data General D470C terminals in ANSI mode,
	colors#16, ncv#53, pairs#256,
	setab=\E[%?%p1%{8}%<%t4%p1%e=%p1%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%;%d%?%gD%t;2%;%?%gU%t;4%;%?%gB%t;5%;%?%gR%t;7%;m,
	setaf=\E[%?%p1%{8}%<%t3%p1%e<%p1%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%;%d%?%gD%t;2%;%?%gU%t;4%;%?%gB%t;5%;%?%gR%t;7%;m,
	setb=\E[%?%p1%{8}%<%t4%e=%;%p1%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%d%?%gD%t;2%;%?%gU%t;4%;%?%gB%t;5%;%?%gR%t;7%;m,
	setf=\E[%?%p1%{8}%<%t3%e<%;%p1%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%d%?%gD%t;2%;%?%gU%t;4%;%?%gB%t;5%;%?%gR%t;7%;m,
	use=dg+color8,

dgmode+color8|Color info for Data General D220/D230C terminals in DG mode,
	bce,
	colors#8, ncv#16, pairs#64,
	op=\036Ad\036Bd,
	setab=\036B%p1%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%{48}%+%c,
	setaf=\036A%p1%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%{48}%+%c,
	setb=\036B%p1%{48}%+%c, setf=\036A%p1%{48}%+%c,

dgmode+color|Color info for Data General D470C terminals in DG mode,
	colors#16, pairs#256,
	setab=\036B%p1%?%p1%{8}%<%t%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%;%{48}%+%c,
	setaf=\036A%p1%?%p1%{8}%<%t%{2}%&%?%p1%{1}%&%t%{4}%|%;%?%p1%{4}%&%t%{1}%|%;%;%{48}%+%c,
	use=dgmode+color8,

dgunix+ccc|Configurable color info for DG D430C terminals in DG-UNIX mode,
	bce, ccc,
	colors#52, ncv#53, pairs#26,
	initp=\036RG0%p1%02X%p2%{255}%*%{1000}%/%02X%p3%{255}%*%{1000}%/%02X%p4%{255}%*%{1000}%/%02X%p5%{255}%*%{1000}%/%02X%p6%{255}%*%{1000}%/%02X%p7%{255}%*%{1000}%/%02X,
	oc=\036RG01A00FF00000000\036RG01B00000000FF00\036RG01C007F00000000\036RG01D000000007F00,
	op=\036RF4831A\036RF2E31B\036RF1D31C\036RF3F31D,
	scp=\036RG2%p1%02X,

# Colors are in the order:  normal, reverse, dim, dim + reverse.
dg+ccc|Configurable color info for DG D430C terminals in DG mode,
	bce, ccc,
	colors#52, ncv#53, pairs#26,
	initp=\036RG0%p1%{16}%/%{48}%+%c%p1%{16}%m%{48}%+%c%p2%{255}%*%{1000}%/%Pa%ga%{16}%/%{48}%+%c%ga%{16}%m%{48}%+%c%p3%{255}%*%{1000}%/%Pa%ga%{16}%/%{48}%+%c%ga%{16}%m%{48}%+%c%p4%{255}%*%{1000}%/%Pa%ga%{16}%/%{48}%+%c%ga%{16}%m%{48}%+%c%p5%{255}%*%{1000}%/%Pa%ga%{16}%/%{48}%+%c%ga%{16}%m%{48}%+%c%p6%{255}%*%{1000}%/%Pa%ga%{16}%/%{48}%+%c%ga%{16}%m%{48}%+%c%p7%{255}%*%{1000}%/%Pa%ga%{16}%/%{48}%+%c%ga%{16}%m%{48}%+%c,
	oc=\036RG01\:00??00000000\036RG01;00000000??00\036RG01<007?00000000\036RG01=000000007?00,
	op=\036RF4831\:\036RF2>31;\036RF1=31<\036RF3?31=,
	scp=\036RG2%p1%{16}%/%{48}%+%c%p1%{16}%m%{48}%+%c,

# The generic DG terminal type (an 8-bit-clean subset of the 6053)
# Initialization string 1 sets:
#	^R		- vertical scrolling enabled
#	^C		- blinking enabled
dg-generic|Generic Data General terminal in DG mode,
	am, bw, msgr, xon,
	cols#80, lines#24,
	bel=^G, blink=^N, clear=^L, cr=^M, cub1=^Y, cud1=^Z, cuf1=^X,
	cup=\020%p2%c%p1%c, cuu1=^W, dim=^\, el=^K, ind=^J, is1=^R^C,
	mc0=^Q, nel=^J, rmso=^], rmul=^U, sgr0=^O^U^], smso=^\,
	smul=^T, use=dgkeys+11,

# According to the 4.4BSD termcap file, the dg200 <cup> should be the
# termcap equivalent of \020%p2%{128}%+%c%p1%{128}%+%c (in termcap
# notation that's "^P%r%+\200%+\200").  Those \200s are suspicious,
# maybe they were originally nuls (which would fit).

dg200|data general dasher 200,
	OTbs, am, bw,
	cols#80, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^Y, cud1=^Z, cuf1=^X,
	cup=\020%p2%c%p1%c, cuu1=^W, el=^K, home=^H, ind=^J,
	kcub1=^Y, kcud1=^Z, kcuf1=^X, kcuu1=^W, kf0=^^z, kf1=^^q,
	kf2=^^r, kf3=^^s, kf4=^^t, kf5=^^u, kf6=^^v, kf7=^^w, kf8=^^x,
	kf9=^^y, khome=^H, lf0=f10, nel=^J, rmso=^^E, rmul=^U,
	smso=^^D, smul=^T,

# Data General 210/211 (and 410?)	from Lee Pearson (umich!lp) via BRL
dg210|dg-ansi|Data General 210/211,
	am,
	cols#80, lines#24,
	OTnl=\E[B, clear=\E[2J, cub1=\E[D, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, ed=\E[J, el=\E[K,
	home=\E[H, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	khome=\E[H, nel=\r\E[H\E[A\n, rmso=\E[0;m, rmul=\E[0;m,
	smso=\E[7;m, smul=\E[4;m,
# From: Peter N. Wan <ihnp4!gatech!gacsr!wan>
# courtesy of Carlos Rucalde of Vantage Software, Inc.
# (dg211: this had <cup=\020%r%.%>., which was an ancient termcap hangover.
# I suspect the d200 function keys actually work on the dg211, check it out.)
dg211|Data General d211,
	cnorm=^L, cvvis=^L^R, ht=^I, ind@, kbs=^Y, kf0@, kf1@, kf2@, kf3@,
	kf4@, kf5@, kf6@, kf7@, kf8@, kf9@, lf0@, nel=^M^Z, rmcup=^L,
	rmso=\036E$<\0/>, smcup=^L^R, smso=\036D$<5/>, use=dg200,

# dg450 from Cornell (not official)
dg450|dg6134|data general 6134,
	cub1@, cuf1=^X, use=dg200,

# Not official...
# Note: lesser Dasher terminals will not work with vi because vi insists upon
# having a command to move straight down from any position on the bottom line
# and scroll the screen up, or a direct vertical scroll command.  The 460 and
# above have both, the D210/211, for instance, has neither.  We must use ANSI
# mode rather than DG mode because standard UNIX tty drivers assume that ^H is
# backspace on all terminals.  This is not so in DG mode.
# (dg460-ansi: removed obsolete ":kn#6:"; also removed ":mu=\EW:", on the
# grounds that there is no matching ":ml:"
dg460-ansi|Data General Dasher 460 in ANSI-mode,
	OTbs, am, msgr, ul,
	cols#80, it#8, lines#24,
	OTnl=\ED, blink=\E[5m, clear=\E[2J, cub1=^H, cud1=\E[B,
	cuf1=\E[C, cup=\E[%i%p1%2d;%p2%2dH, cuu1=\E[A, dch1=\E[P,
	dim=\E[2m, dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I,
	ich1=\E[@, il1=\E[L, ind=\E[S, is2=^^F@, kbs=\E[D,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kf0=\E[001z, kf1=\E[002z, kf2=\E[003z, kf3=\E[004z,
	kf4=\E[005z, kf5=\E[006z, kf6=\E[007z, kf7=\E[008z,
	kf8=\E[009z, kf9=\E[00\:z, khome=\E[H, lf0=f1, lf1=f2,
	lf2=f3, lf3=f4, lf4=f5, lf5=f6, lf6=f7, lf7=f8, lf9=f10,
	mc0=\E[i, rev=\E[7m, ri=\E[T, rmso=\E[m, rmul=\E[05,
	sgr=%?%p9%t\E(0%e\E(B%;\E[0%?%p5%t;2%;%?%p2%t;4%;%?%p4%t;5%;%?%p1%p3%|%t;7%;m,
	sgr0=\E(B\E[m, smso=\E[7m, smul=\E[4m, u6=\E[%d;%dR,
	u7=\E[6n, u8=\E[5n, u9=\E[0n,
# From: Wayne Throop <mcnc!rti-sel!rtp47!throopw> (not official)
# Data General 605x
# Ought to work for a Model 6242, Type D210 as well as a 605x.
# Note that the cursor-down key transmits ^Z.  Job control users, beware!
# This also matches a posted description of something called a `Dasher 100'
# so there's a dg100 alias here.
# (dg6053: the 4.4BSD file had <cub1=^H>, <cud1=^J>, <cuf1=^S>. -- esr)
dg6053-old|dg100|data general 6053,
	OTbs, am, bw, ul,
	cols#80, lines#24,
	OTbc=^Y, bel=^G, clear=^L, cnorm=^L, cr=^M, cub1=^Y, cud1=^Z,
	cuf1=^X, cup=\020%p2%c%p1%c, cuu1=^W, cvvis=^L^R, el=^K,
	home=^H, ht=^I, is2=^R, kbs=^Y, kcub1=^Y, kcud1=^Z, kcuf1=^X,
	kcuu1=^W, kf0=^^q, kf1=^^r, kf2=^^s, kf3=^^t, kf4=^^u, kf5=^^v,
	kf6=^^w, kf7=^^x, kf8=^^y, kf9=^^z, khome=^H, rmcup=^L,
	rmso=\0^^E, rmul=^U, smcup=^L^R, smso=\0\0\0\0\0\036D,
	smul=^T,

# (Some performance can be gained over the generic DG terminal type)
dg6053|6053|6053-dg|dg605x|605x|605x-dg|d2|d2-dg|Data General DASHER 6053,
	xon@,
	home=^P\0\0, ll=^P\0^W, use=dg-generic,

# Like 6053, but adds reverse video and more keypad and function keys.
d200|d200-dg|Data General DASHER D200,
	bold=^^D^T, home@, ll@, rev=^^D, rmso=^^E^],
	sgr=\036%?%p1%p3%|%p6%|%tD%eE%;%?%p2%p6%|%t\024%e\025%;%?%p4%t\016%e\017%;%?%p1%p5%|%t\034%e\035%;,
	sgr0=\017\025\035\036E, smso=^^D^\, use=dgkeys+15,
	use=dg6053,

# DASHER D210 series terminals in ANSI mode.
#	Reverse video, no insert/delete character/line, 7 bits/character only.
#
# Initialization string 1 sets:
#	<0		- scrolling enabled
#	<1		- blink enabled
#	<4		- print characters regardless of attributes
d210|d214|Data General DASHER D210 series,
	am, bw, msgr, xon,
	cols#80, lines#24,
	bel=^G, blink=\E[5m, bold=\E[4;7m, clear=\E[2J, cr=^M,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dim=\E[2m, ed=\E[J, el=\E[K,
	el1=\E[1K, home=\E[H, ind=^J, is1=\E[<0;<1;<4l,
	ll=\E[H\E[A, nel=^J, rev=\E[7m, rmso=\E[m, rmul=\E[m,
	sgr=\E[%?%p4%t5;%;%?%p2%p6%|%t4;%;%?%p1%p5%|%t2;%;%?%p1%p3%|%p6%|%t7;%;m,
	sgr0=\E[m, smso=\E[2;7m, smul=\E[4m, use=dgkeys+7b,

# DASHER D210 series terminals in DG mode.
# Like D200, but adds clear to end-of-screen and needs XON/XOFF.
d210-dg|d214-dg|Data General DASHER D210 series in DG mode,
	xon,
	ed=^^FF, use=d200-dg,

# DASHER D211 series terminals in ANSI mode.
# Like the D210, but with 8-bit characters and local printer support.
#
# Initialization string 2 sets:
#	\E[2;1;1;1v
#		2;1	- 8 bit operations
#		1;1	- 8 bit (international) keyboard language
#	\E(B		- default primary character set (U.S. ASCII)
#	\E)4		- default secondary character set (international)
#	^O		- primary character set
#
d211|d215|Data General DASHER D211 series,
	km,
	is2=\E[2;1;1;1v\E(B\E)4\017, mc0=\E[i, use=dgkeys+8b,
	use=d210,

# Initialization string 2 sets:
#	\E[2;0;1;0v
#		2;0	- 7 bit operations
#		1;0	- 7 bit (native) keyboard language
#	\E(0		- default character set (the keyboard native language)
#	^O		- primary character set
d211-7b|d215-7b|Data General DASHER D211 series in 7 bit mode,
	km@,
	is2=\E[2;0;1;0v\E(0\017, use=dgkeys+7b, use=d211,

# Like the D210 series, but adds support for 8-bit characters.
#
# Reset string 2 sets:
#	^^N	- secondary character set
#	^^FS0>	- 8 bit international character set
#	^^O	- primary character set
#	^^FS00	- default character set (matching the native keyboard language)
#
d211-dg|d215-dg|Data General DASHER D211 series in DG mode,
	km,
	rs2=\036N\036FS0>\036O\036FS00, use=d210-dg,

d216-dg|d216e-dg|d216+dg|d216e+dg|d217-dg|Data General DASHER D216 series in DG mode,
	use=d211-dg,

# Enhanced DG mode with changes to be more UNIX compatible.
d216-unix|d216e-unix|d216+|d216e+|Data General DASHER D216+ in DG-UNIX mode,
	mc5i,
	it#8,
	acsc=a\177j$k"l!m#n)q+t'u&v(w%x*, blink=^^PI,
	clear=^^PH, cub1=^^PD, cud1=^^PB, cuf1=^^PC, cuu1=^^PA,
	el=^^PE, home=^^PF, hpa=\020%p1%c\177, ht=^I, ind=^J,
	is1=\022\003\036P@1, is3=\036Fz0, kHOM=^^Pf, kLFT=^^Pd,
	kPRT=^^P1, kRIT=^^Pc, kclr=^^PH, kcub1=^^PD, kcud1=^^PB,
	kcuf1=^^PC, kcuu1=^^PA, kel=^^PE, khome=^^PF, kprt=^^P0,
	mc0=\036F?9, mc4=^^Fa, mc5=^^F`, rmacs=\036FS00,
	rs2=\036N\036FS0E\036O\036FS00,
	sgr=\036%?%p1%p3%|%p6%|%tD%eE%;%?%p2%p6%|%t\024%e\025%;\036P%?%p4%tI%eJ%;%?%p1%p5%|%t\034%e\035%;\036FS%?%p9%t11%e00%;,
	sgr0=\036PJ\025\035\036E\036FS00, smacs=\036FS11,
	vpa=\020\177%p1%c, use=dgkeys+15, use=d216-dg,
d216-unix-25|d216+25|Data General DASHER D216+ in DG-UNIX mode with 25 lines,
	lines#25,
	is3=\036Fz2, use=d216+,

d217-unix|Data General DASHER D217 in DG-UNIX mode,
	use=d216-unix,
d217-unix-25|Data General DASHER D217 in DG-UNIX mode with 25 lines,
	use=d216-unix-25,

# DASHER D220 color terminal in ANSI mode.
# Like the D470C but with fewer colors and screen editing features.
#
# Initialization string 1 sets:
#	\E[<0;<1;<4l
#		<0	- scrolling enabled
#		<1	- blink enabled
#		<4	- print characters regardless of attributes
#	\E[m		- all attributes off
# Reset string 1 sets:
#	\Ec		- initial mode defaults (RIS)
#
d220|Data General DASHER D220,
	mc5i@,
	dl@, dl1@, il@, il1@, is1=\E[<0;<1;<4l\E[m, mc4@, mc5@, rs1=\Ec,
	use=dg+color8, use=d470c,

d220-7b|Data General DASHER D220 in 7 bit mode,
	mc5i@,
	dl@, dl1@, il@, il1@, is1=\E[<0;<1;<4l\E[m, mc4@, mc5@, rs1=\Ec,
	use=dg+color8, use=d470c-7b,

# Initialization string 3 sets:
#	- default cursor (solid rectangle)
# Reset string 2 sets:
#	^^N     - secondary character set
#	^^FS0>  - 8 bit international character set
#	^^O     - primary character set
#       ^^FS00  - default character set (matching the native keyboard language)
#
d220-dg|Data General DASHER D220 color terminal in DG mode,
	mc5i@,
	dl1@, home@, il1@, is2@, is3=\036FQ2, ll@, mc4@, mc5@, rs1@,
	rs2=\036N\036FS0>\036O\036FS00, use=dgmode+color8,
	use=d470c-dg,

# DASHER D230C color terminal in ANSI mode.
# Like the D220 but with minor ANSI compatibility improvements.
#
d230c|d230|Data General DASHER D230C,
	blink=\E[5;50m, bold=\E[4;7;50m, dim=\E[2;50m, nel=^M^J,
	rev=\E[7;50m, rmkx=\E[2;1v, rmso=\E[50m, rmul=\E[50m,
	sgr=\E[50%?%p1%p3%|%p6%|%t;7%{1}%e%{0}%;%PR%?%p4%t;5%{1}%e%{0}%;%PB%?%p2%p6%|%t;4%{1}%e%{0}%;%PU%?%p1%p5%|%t;2%{1}%e%{0}%;%PDm\E)%?%p9%t6\016%e4\017%;,
	sgr0=\E[50m\E)4\017, smkx=\E[2;0v, smso=\E[2;7;50m,
	smul=\E[4;50m, use=dgkeys+7b, use=d220,

d230c-dg|d230-dg|Data General DASHER D230C in DG mode,
	use=d220-dg,

# DASHER D400/D450 series terminals.
# These add intelligent features like insert/delete to the D200 series.
#
# Initialization string 2 sets:
#	^^FQ2		- default cursor (solid rectangle)
#	^^FW		- character protection disabled
#	^^FJ		- normal (80 column) mode
#	^^F\^		- horizontal scrolling enabled (for alignment)
#	^^FX004?	- margins at columns 0 and 79
#	^^F]		- horizontal scrolling disabled
#	^^O		- primary character set
#	^^FS00		- default character set (the keyboard native language)
#	- (should reset scrolling regions, but that glitches the screen)
# Reset string 1 sets:
#	^^FA		- all terminal defaults except scroll rate
# Reset string 2 sets:
#	^^F]		- horizontal scrolling disabled
#	^^FT0		- jump scrolling
#
d400|d400-dg|d450|d450-dg|Data General DASHER D400/D450 series,
	mc5i,
	acsc=j$k"l!m#n)q+t'u&v(w%x*, civis=\036FQ0,
	cnorm=\036FQ2, dch1=^^K, dl1=^^FI,
	enacs=\036N\036FS11\036O, home=^^FG, hpa=\020%p1%c\177,
	ich1=^^J, il1=^^FH,
	is2=\036FQ2\036FW\036FJ\036F\^\036FX004?\036F]\036O\036FS00,
	ll=\036FG\027, mc4=^^Fa, mc5=^^F`, ri=^^I, rmacs=^^O,
	rs1=^^FA, rs2=\036F]\036FT0,
	sgr=\036%?%p1%p3%|%p6%|%tD%eE%;%?%p2%p6%|%t\024%e\025%;%?%p4%t\016%e\017%;%?%p1%p5%|%t\034%e\035%;\036%?%p9%tN%eO%;,
	sgr0=\017\025\035\036E\036O, smacs=^^N,
	vpa=\020\177%p1%c, use=d210-dg,

# DASHER D410/D460 series terminals in ANSI mode.
# These add a large number of intelligent terminal features.
#
# Initialization string 1 sets:
#	\E[<0;<1;<2;<4l
#		<0	- scrolling enabled
#		<1	- blink enabled
#		<2	- horizontal scrolling enabled (for alignment)
#		<4	- print characters regardless of attributes
#	\E[5;0v		- normal (80 column) mode
#	\E[1;1;80w	- margins at columns 1 and 80
#	\E[1;6;<2h
#		1	- print all characters even if protected
#		6	- character protection disabled
#		<2	- horizontal scrolling disabled
#	- (should reset scrolling regions, but that glitches the screen)
#
# Initialization string 2 sets:
#	\E[3;2;2;1;1;1v
#		3;2	- default cursor (solid rectangle)
#		2;1	- 8 bit operations
#		1;1	- international keyboard language
#	\E(B		- default primary character set (U.S. ASCII)
#	\E)4		- default secondary character set (international)
#	^O		- primary character set
#
#	Reset string 1 sets:
#	\Ec		- initial mode defaults (RIS)
#	\E[<2h		- horizontal scrolling disabled
#
# Reset string 2 sets:
#	\E[4;0;2;1;1;1v
#		4;0	- jump scrolling
#		2;1	- 8 bit operations
#		1;1	- 8 bit (international) keyboard language
#	\E(B		- default primary character set (U.S. ASCII)
#	\E)4		- default secondary character set (international)
#
d410|d411|d460|d461|Data General DASHER D410/D460 series,
	mc5i,
	acsc=j$k"l!m#n)q+t'u&v(w%x*, civis=\E[3;0v,
	cnorm=\E[3;2v, dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[L,
	is1=\E[<0;<1;<2;<4l\E[5;0v\E[1;1;80w\E[1;6;<2h,
	is2=\E[3;2;2;1;1;1v\E(B\E)4\017, mc4=\E[4i, mc5=\E[5i,
	ri=\EM, rmacs=\E)4\017, rs1=\Ec\E[<2h,
	rs2=\E[4;0;2;1;1;1v\E(B\E)4,
	sgr=\E[%?%p1%t2;7%;%?%p3%t7;%;%?%p4%t5;%;%?%p2%t4;%;%?%p1%p5%|%t2;%;%?%p6%t4;7;%;m\E)%?%p9%t6\016%e4\017%;,
	sgr0=\E[m\E)4\017, smacs=\E)6\016, use=d211,

# Initialization string 2 sets:
#	\E[3;2;2;0;1;0v
#		3;2	- default cursor (solid rectangle)
#		2;0	- 7 bit operations
#		1;0	- 7 bit (native) keyboard language
#	\E(0		- default character set (the keyboard native language)
#	^O		- primary character set
#
# Reset string 2 sets:
#	\E[4;0;2;0;1;0v
#		4;0	- jump scrolling
#		2;0	- 7 bit operations
#		1;0	- 7 bit (native) keyboard language
#	\E(0		- default character set (the keyboard native language)
#
d410-7b|d411-7b|d460-7b|d461-7b|Data General DASHER D410/D460 series in 7 bit mode,
	km@,
	enacs=\E)6, is2=\E[3;2;2;0;1;0v\E(0\017, rmacs=^O,
	rs2=\E[4;0;2;0;1;0v\E(0,
	sgr=\E[%?%p1%p3%|%p6%|%t7;%;%?%p4%t5;%;%?%p2%p6%|%t4;%;%?%p1%p5%|%t2;%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, use=dgkeys+7b, use=d410,

d410-dg|d460-dg|d411-dg|d461-dg|Data General DASHER D410/D460 series in DG mode,
	km,
	enacs@, rmacs=\036FS00,
	sgr=\036%?%p1%p3%|%p6%|%tD%eE%;%?%p2%p6%|%t\024%e\025%;%?%p4%t\016%e\017%;%?%p1%p5%|%t\034%e\035%;\036FS%?%p9%t11%e00%;,
	sgr0=\017\025\035\036E\036FS00, smacs=\036FS11,
	use=d400-dg,

# DASHER D410/D460 series terminals in wide (126 columns) ANSI mode.
#
# Initialization string 1 sets:
#	\E[<0;<1;<2;<4l
#		<0	- scrolling enabled
#		<1	- blink enabled
#		<2	- horizontal scrolling enabled (for alignment)
#		<4	- print characters regardless of attributes
#	\E[5;1v		- compressed (135 column) mode
#	\E[1;1;126	- margins at columns 1 and 126
#	\E[1;6;<2h
#		1	- print all characters even if protected
#		6	- character protection disabled
#		<2	- horizontal scrolling disabled
#	- (should reset scrolling regions, but that glitches the screen)
#
# Reset string 1 sets:
#	\Ec		- initial mode defaults (RIS)
#	\E[5;1v		- compressed (135 column) mode
#	\E[1;1;126w	- margins at columns 1 and 126
#	\E[<2h		- horizontal scrolling disabled
#
d410-w|d411-w|d460-w|d461-w|Data General DASHER D410/D460 series in wide mode,
	cols#126,
	is1=\E[<0;<1;<2;<4l\E[5;1v\E[1;1;126w\E[1;6;<2h,
	rs1=\Ec\E[5;1v\E[1;1;126w\E[<2h, use=d410,

d410-7b-w|d411-7b-w|d460-7b-w|d461-7b-w|Data General DASHER D410/D460 series in wide 7 bit mode,
	cols#126,
	is1=\E[<0;<1;<2;<4l\E[5;1v\E[1;1;126w\E[1;6;<2h,
	rs1=\Ec\E[5;1v\E[1;1;126w\E[<2h, use=d410-7b,

d412-dg|d462-dg|d462e-dg|d412+dg|d462+dg|d413-dg|d463-dg|Data General DASHER D412/D462 series in DG mode,
	use=d410-dg,

# These add intelligent features like scrolling regions.
d412-unix|d462-unix|d412+|d462+|Data General DASHER D412+/D462+ series in Unix mode,
	civis=\036FQ0, clear=^^FE, cnorm=\036FQ5,
	cup=\036FP%p2%2.2X%p1%2.2X, dch1=^^K, dl1=^^FI,
	home=^^FG, hpa=\036FP%p1%2.2XFF, ich1=^^J, il1=^^FH,
	is2=\036FQ5\036FW\036FJ\036F\^\036FX004F\036O\036FS00,
	ll=\036FG\036PA, mc0=^A, rc=\036F}11, ri=^^I,
	rs1=\036FA\036FT0, rs2=\036P@1, sc=\036F}10,
	vpa=\036FPFF%p1%2.2X,
	wind=\036FB%?%p1%t%p1%2.2X0%;%p2%p1%-%{1}%+%2.2X0%?%{23}%p2%>%t000%;\036FX%p3%2.2X%p4%2.2X,
	use=d216+,
d412-unix-w|d462-unix-w|d412+w|d462+w|Data General DASHER D412+/D462+ series in wide Unix mode,
	cols#132,
	is2=\036FQ5\036FW\036FK\036F\^\036FX0083\036O\036FS00,
	rs2=\036P@1\036FK\036FX0083,
	wind=\036FB%?%p1%t%p1%2.2X1%;%p2%p1%-%{1}%+%2.2X1%?%{23}%p2%>%t001%;\036FX%p3%2.2X%p4%2.2X,
	use=d412-unix,
d412-unix-25|d462-unix-25|d412+25|d462+25|Data General DASHER D412+/D462+ series in Unix mode with 25 lines,
	lines#25,
	is3=\036Fz2,
	wind=\036FB%?%p1%t%p1%2.2X0%;%p2%p1%-%{1}%+%2.2X0%?%{24}%p2%>%t000%;\036FX%p3%2.2X%p4%2.2X,
	use=d462+,
d412-unix-s|d462-unix-s|d412+s|d462+s|Data General DASHER D412+/D462+ in Unix mode with status line,
	eslok, hs,
	clear=\036FG\036PH, fsl=\036F}01\022,
	is3=\036Fz2\036F}00\036FB180000\036F}01, ll@,
	tsl=\036F}00\036FP%p1%2.2X18\036PG,
	wind=\036FB%?%p1%t%p1%2.2X0%;%p2%p1%-%{1}%+%2.2X0%?%{23}%p2%>%t%{23}%p2%-%2.2X0%;000\036FX%p3%2.2X%p4%2.2X,
	use=d462+,

#	Relative cursor motions are confined to the current window,
#	which is not what the scrolling region specification expects.
#	Thus, relative vertical cursor positioning must be deleted.
d412-unix-sr|d462-unix-sr|d412+sr|d462+sr|Data General DASHER D412+/D462+ in Unix mode with scrolling region,
	csr=\036FB%?%p1%t%p1%2.2X0%;%p2%p1%-%{1}%+%2.2X0%?%{23}%p2%>%t000%;,
	cud1@, cuu1@, ll@, use=d462+,

d413-unix|d463-unix|Data General DASHER D413/D463 series in DG-UNIX mode,
	use=d412-unix,
d413-unix-w|d463-unix-w|Data General DASHER D413/D463 series in wide DG-UNIX mode,
	use=d412-unix-w,
d413-unix-25|d463-unix-25|Data General DASHER D413/D463 series in DG-UNIX mode with 25 lines,
	use=d412-unix-25,
d413-unix-s|d463-unix-s|Data General DASHER D413/D463 in DG-UNIX mode with status line,
	use=d412-unix-s,
d413-unix-sr|d463-unix-sr|Data General DASHER D413/D463 in DG-UNIX mode with scrolling region,
	use=d412-unix-sr,

d414-unix|d464-unix|Data General D414/D464 in DG-UNIX mode,
	use=d413-unix,
d414-unix-w|d464-unix-w|Data General D414/D464 in wide DG-UNIX mode,
	use=d413-unix-w,
d414-unix-25|d464-unix-25|Data General D414/D464 in DG-UNIX mode with 25 lines,
	use=d413-unix-25,
d414-unix-s|d464-unix-s|Data General D414/D464 in DG-UNIX mode with status line,
	use=d413-unix-s,
d414-unix-sr|d464-unix-sr|Data General D414/D464 in DG-UNIX mode with scrolling region,
	use=d413-unix-sr,

d430c-dg|d430-dg|Data General D430C in DG mode,
	use=d413-dg, use=dg+fixed,
d430c-dg-ccc|d430-dg-ccc|Data General D430C in DG mode with configurable colors,
	use=d413-dg, use=dg+ccc,

d430c-unix|d430-unix|Data General D430C in DG-UNIX mode,
	use=d413-unix, use=dgunix+fixed,
d430c-unix-w|d430-unix-w|Data General D430C in wide DG-UNIX mode,
	use=d413-unix-w, use=dgunix+fixed,
d430c-unix-25|d430-unix-25|Data General D430C in DG-UNIX mode with 25 lines,
	use=d413-unix-25, use=dgunix+fixed,
d430c-unix-s|d430-unix-s|Data General D430C in DG-UNIX mode with status line,
	use=d413-unix-s, use=dgunix+fixed,
d430c-unix-sr|d430-unix-sr|Data General D430C in DG-UNIX mode with scrolling region,
	use=d413-unix-sr, use=dgunix+fixed,
d430c-unix-ccc|d430-unix-ccc|Data General D430C in DG-UNIX mode with configurable colors,
	use=d413-unix, use=dgunix+ccc,
d430c-unix-w-ccc|d430-unix-w-ccc|Data General D430C in wide DG-UNIX mode with configurable colors,
	use=d413-unix-w, use=dgunix+ccc,
d430c-unix-25-ccc|d430-unix-25-ccc|Data General D430C in DG-UNIX mode with 25 lines and configurable colors,
	use=d413-unix-25, use=dgunix+ccc,
d430c-unix-s-ccc|d430-unix-s-ccc|Data General D430C in DG-UNIX mode with status line and configurable colors,
	use=d413-unix-s, use=dgunix+ccc,
d430c-unix-sr-ccc|d430-unix-sr-ccc|Data General D430C in DG-UNIX mode with scrolling region and configurable colors,
	use=d413-unix-sr, use=dgunix+ccc,

# DASHER D470C color terminal in ANSI mode.
# Like the D460 but with 16 colors and without a compressed mode.
#
# Initialization string 1 sets:
#	\E[<0;<1;<2;<4l
#		<0	- scrolling enabled
#		<1	- blink enabled
#		<2	- horizontal scrolling enabled (for alignment)
#		<4	- print characters regardless of attributes
#	\E[1;1;80w	- margins at columns 1 and 80
#	\E[1;6;<2h
#		1	- print all characters even if protected
#		6	- character protection disabled
#		<2	- horizontal scrolling disabled
#	- (should reset scrolling regions, but that glitches the screen)
#
d470c|d470|Data General DASHER D470C,
	is1=\E[<0;<1;<2;<4l\E[1;1;80w\E[1;6;<2h,
	sgr=\E[%?%p3%t7;%;%?%p4%t5;%;%?%p2%t4;%;%?%p6%t4;7;%;%?%p1%t2;7;%;%?%p5%t2;%;m\E)%?%p9%t6\016%e4\017%;,
	use=dg+color, use=d460,

d470c-7b|d470-7b|Data General DASHER D470C in 7 bit mode,
	is1=\E[<0;<1;<2;<4l\E[1;1;80w\E[1;6;<2h,
	sgr=\E[%?%p3%t7;%;%?%p4%t5;%;%?%p2%t4;%;%?%p6%t4;7;%;%?%p1%t2;7;%;%?%p5%t2;%;m%?%p9%t\016%e\017%;,
	use=dg+color, use=d460-7b,

# Initialization string 2 sets:
#	^^FQ2		- default cursor (solid rectangle)
#	^^FW		- character protection disabled
#	^^F\^		- horizontal scrolling enabled (for alignment)
#	^^FX004?	- margins at columns 0 and 79
#	^^F]		- horizontal scrolling disabled
#	^^O		- primary character set
#	^^FS00		- default character set (the keyboard native language)
#	- (should reset scrolling regions, but that glitches the screen)
#
d470c-dg|d470-dg|Data General DASHER D470C in DG mode,
	is2=\036FQ2\036FW\036F\^\036FX004?\036F]\036O\036FS00,
	use=dgmode+color, use=d460-dg,

# DASHER D555 terminal in ANSI mode.
# Like a D411, but has an integrated phone.
d555|Data General DASHER D555,
	use=d411,
d555-7b|Data General DASHER D555 in 7-bit mode,
	use=d411-7b,
d555-w|Data General DASHER D555 in wide mode,
	use=d411-w,
d555-7b-w|Data General DASHER D555 in wide 7-bit mode,
	use=d411-7b-w,
d555-dg|Data General DASHER D555 series in DG mode,
	use=d411-dg,

# DASHER D577 terminal in ANSI mode.
# Like a D411, but acts as a keyboard for serial printers ("KSR" modes).
d577|Data General DASHER D577,
	use=d411,
d577-7b|Data General DASHER D577 in 7-bit mode,
	use=d411-7b,
d577-w|Data General DASHER D577 in wide mode,
	use=d411-w,
d577-7b-w|Data General DASHER D577 in wide 7-bit mode,
	use=d411-7b-w,

d577-dg|d578-dg|Data General DASHER D577/D578 series in DG mode,
	use=d411-dg,

# DASHER D578 terminal.
# Like a D577, but without compressed mode; like a D470C in this respect.
#
# Initialization string 1 sets:
#	\E[<0;<1;<2;<4l
#		<0	- scrolling enabled
#		<1	- blink enabled
#		<2	- horizontal scrolling enabled (for alignment)
#		<4	- print characters regardless of attributes
#	\E[1;1;80w	- margins at columns 1 and 80
#	\E[1;6;<2h
#		1	- print all characters even if protected
#		6	- character protection disabled
#		<2	- horizontal scrolling disabled
#	- (should reset scrolling regions, but that glitches the screen)
#
d578|Data General DASHER D578,
	is1=\E[<0;<1;<2;<4l\E[1;1;80w\E[1;6;<2h, use=d577,
d578-7b|Data General DASHER D578 in 7-bit mode,
	is1=\E[<0;<1;<2;<4l\E[1;1;80w\E[1;6;<2h, use=d577-7b,

#### Datamedia (dm)
#
# Datamedia was headquartered in Nashua, New Hampshire until it went
# out of business in 1993, but the ID plates on the terminals referred
# to the factory in Pennsauken, NJ.  The factory was sold to a PCB board
# manufacturer which threw out all information about the terminals.
#

cs10|colorscan|Datamedia Color Scan 10,
	msgr,
	cols#80, lines#24,
	bel=^G, clear=\E[H\E[J, cr=^M, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%02d;%p2%02dH, cuu1=\E[A, ed=\E[J, el=\E[K,
	ind=^J, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m, smso=\E[7m, smul=\E[4m,
cs10-w|Datamedia Color Scan 10 with 132 columns,
	cols#132,
	cup=\E[%i%p1%02d;%p2%03dH, use=cs10,

# (dm1520: removed obsolete ":ma=^\ ^_^P^YH:" -- esr)
dm1520|dm1521|datamedia 1520,
	OTbs, am, xenl,
	cols#80, it#8, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=^\,
	cup=\036%p2%{32}%+%c%p1%{32}%+%c, cuu1=^_, ed=^K, el=^],
	home=^Y, ht=^I, ind=^J, kcub1=^H, kcud1=^J, kcuf1=^\, kcuu1=^_,
	khome=^Y,
# dm2500: this terminal has both <ich> and <smir>. Applications using
# termcap/terminfo directly (rather than through ncurses) might be confused.
dm2500|datamedia2500|datamedia 2500,
	OTbs, OTnc,
	cols#80, lines#24,
	bel=^G, clear=^^^^\177, cub1=^H, cud1=^J, cuf1=^\,
	cup=\014%p2%{96}%^%c%p1%{96}%^%c, cuu1=^Z,
	dch1=\020\010\030\035$<10*>,
	dl1=\020\032\030\035$<10*>, el=^W, home=^B,
	ich1=\020\034\030\035$<10*>,
	il1=\020\n\030\035\030\035$<15>, ind=^J, pad=\377,
	rmdc=^X^], rmir=\377\377\030\035$<10>, rmso=^X^],
	smdc=^P, smir=^P, smso=^N,
# dmchat is like DM2500, but DOES need "all that padding" (jcm 1/31/82)
# also, has a meta-key.
# From: <goldberger@su-csli.arpa>
# (dmchat: ":MT:" changed to ":km:" -- esr)
dmchat|dmchat version of datamedia 2500,
	km,
	dl1=\020\032\030\035$<2/>,
	il1=\020\n\030\035\030\035$<1*/>, use=dm2500,
# (dm3025: ":MT:" changed to ":km:" -- esr)
dm3025|datamedia 3025a,
	OTbs, km,
	cols#80, it#8, lines#24,
	bel=^G, clear=\EM$<2>, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p2%{32}%+%c%p1%{32}%+%c, cuu1=\EA,
	dch1=\010$<6>, dl1=\EP\EA\EQ$<130>, ed=\EJ$<2>, el=\EK,
	home=\EH, ht=^I, il1=\EP\n\EQ$<130>, ind=^J, ip=$<6>,
	is2=\EQ\EU\EV, rmdc=\EQ, rmir=\EQ, rmso=\EO0, smdc=\EP,
	smir=\EP, smso=\EO1,
dm3045|datamedia 3045a,
	OTbs, am, eo, km@, ul, xenl,
	dch1=\EB$<6>, dl1@, il1@, is2=\EU\EV, kcuf1=\EC, kcuu1=\EA,
	kf0=\Ey\r, kf1=\Ep\r, kf2=\Eq\r, kf3=\Er\r, kf4=\Es\r,
	kf5=\Et\r, kf6=\Eu\r, kf7=\Ev\r, kf8=\Ew\r, kf9=\Ex\r,
	khome=\EH, pad=\177, rmdc@, rmir=\EP, rmso@, smdc@, smso@,
	use=dm3025,
# Datamedia DT80 soft switches:
# 1	0=Jump  1=Smooth
# 	Autorepeat 	0=off  1=on
# 	Screen		0=Dark 1=light
# 	Cursor		0=u/l  1=block
#
# 2	Margin Bell	0=off  1=on
# 	Keyclick	0=off  1=on
# 	Ansi/VT52	0=VT52 1=Ansi
# 	Xon/Xoff	0=Off  1=On
#
# 3	Shift3		0=Hash 1=UK Pound
# 	Wrap		0=Off  1=On
# 	Newline		0=Off  1=On
# 	Interlace	0=Off  1=On
#
# 4	Parity		0=Odd  1=Even
# 	Parity		0=Off  1=On
# 	Bits/Char	0=7    1=8
# 	Power		0=60Hz 1=50Hz
#
# 5	Line Interface  0=EIA  1=Loop
# 	Aux Interface   0=EIA  1=Loop
# 	Local Copy    	0=Off  1=On
# 	Spare
#
# 6	Aux Parity	0=Odd  1=Even
# 	Aux Parity	0=Off  1=On
# 	Aux Bits/Char   0=7    1=8
# 	CRT Saver	0=Off  1=On
# dm80/1 is a vt100 lookalike, but it doesn't seem to need any padding.
dm80|dmdt80|dt80|datamedia dt80/1,
	clear=\E[2J\E[H, cud1=^J, cuf1=\E[C,
	cup=%i\E[%p1%d;%p2%dH, cuu1=\E[A, ed=\E[J, el=\E[K,
	home=\E[H, mc0=\E[0i, mc4=\E[4i, mc5=\E[5i, ri=\EM,
	rmso=\E[m, rmul=\E[m, sgr0=\E[m, smso=\E[7m, smul=\E[4m,
	use=vt100,
# except in 132 column mode, where it needs a little padding.
# This is still less padding than the vt100, and you can always turn on
# the ^S/^Q handshaking, so you can use vt100 flavors for things like
# reverse video.
dm80w|dmdt80w|dt80w|datamedia dt80/1 in 132 char mode,
	cols#132,
	clear=\E[H\E[2J$<50/>, cud1=^J,
	cup=\E[%i%p1%d;%p2%dH$<5/>, cuu1=\E[A$<5/>,
	ed=\E[0J$<20/>, el=\E[0K$<20/>, use=dm80,
# From: Adam Thompson <athompso@pangea.ca> Sept 10 1995
dt80-sas|Datamedia DT803/DTX for SAS usage,
	am, bw,
	cols#80, lines#24,
	acsc=``a1fxgqh0jYk?lZm@nEooppqDrrsstCu4vAwBx3yyzz{{||}}~~,
	bel=^G, clear=^L, cr=^M,
	csr=\E=%p1%{32}%+%c%{32}%c\E#1\E=%p2%{32}%+%c%{32}%c\E#2,
	cub1=^H, cud1=\EB, cuf1=^\,
	cup=\E=%p2%{32}%+%c%p1%{32}%+%c, cuu1=^_, dl1=\EM, ed=^K,
	el=^], ff=^L, home=^Y, ht=^I, hts=\E'1, il1=\EL, ind=\EB,
	is2=\E)0\E<\EP\E'0\E$2, kclr=^L, kcub1=^H, kcud1=^J,
	kcuf1=^\, kcuu1=^_, ked=^K, kel=^], khome=^Y, mc4=^O, mc5=^N,
	rev=\E$2\004, ri=\EI, rmacs=\EG, rmso=^X, sgr0=^X, smacs=\EF,
	smso=\E$2\004, tbc=\E'0,

# Datamedia Excel 62, 64 from Gould/SEL UTX/32 via BRL
# These aren't end-all Excel termcaps; but do insert/delete char/line
# and name some of the extra function keys.  (Mike Feldman ccvaxa!feldman)
# The naming convention has been bent somewhat, with the use of E? (where
# E is for 'Excel') as # a name.  This was done to distinguish the entries
# from the other Datamedias in use here, and yet to associate a model of
# the Excel terminals with the regular datamedia terminals that share
# major characteristics.
excel62|excel64|datamedia Excel 62,
	dch1=\E[P, kbs=^H, kcub1=^H, kcud1=^J, kf5=\EOu, kf6=\EOv,
	kf7=\EOw, kf8=\EOx, kf9=\EOy, rmir=\E[4l, smir=\E[4h,
	use=dt80,
excel62-w|excel64-w|datamedia Excel 62 in 132 char mode,
	dch1=\E[P, kbs=^H, kcub1=^H, kcud1=^J, kf5=\EOu, kf6=\EOv,
	kf7=\EOw, kf8=\EOx, kf9=\EOy, rmir=\E[4l, smir=\E[4h,
	use=dt80w,
excel62-rv|excel64-rv|datamedia Excel 62 in reverse video mode,
	dch1=\E[P, flash=\E[?5l\E[?5h, kbs=^H, kcub1=^H, kcud1=^J,
	kf5=\EOu, kf6=\EOv, kf7=\EOw, kf8=\EOx, kf9=\EOy, rmir=\E[4l,
	smir=\E[4h, use=dt80,

#### Falco
#
#	Falco Data Products
#	440 Potrero Avenue
#	Sunnyvale, CA 940864-196
#	Vox: (800)-325-2648
#	Fax: (408)-745-7860
#	Net: techsup@charm.sys.falco.com
#
# Current Falco models as of 1995 are generally ANSI-compatible and support
# emulations of DEC VT-series, Wyse, and Televideo types.
#

# Test version for Falco ts-1. See <arpavax.hickman@ucb> for info
# This terminal was released around 1983 and was discontinued long ago.
# The standout and underline highlights are the same.
falco|ts1|ts-1|falco ts-1,
	OTbs, am,
	cols#80, it#8, lines#24,
	bel=^G, clear=\E*, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\EY, el=\ET\EG0\010, home=^^, ht=^I, il1=\EE,
	ind=^J, is2=\Eu\E3, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K,
	kf0=^A0\r, rmir=\Er, rmso=\Eg0, rmul=\Eg0, sgr0=\Eg0,
	smir=\Eq, smso=\Eg1, smul=\Eg1,
falco-p|ts1p|ts-1p|falco ts-1 with paging option,
	OTbs, am, da, db, mir, msgr, ul,
	cols#80, it#8, lines#24,
	bel=^G, cbt=\EI, clear=\E*, cr=^M, cub1=^H, cud1=\E[B,
	cuf1=\E[C, cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=\E[A,
	dch1=\EW, dl1=\ER, ed=\EY, el=\ET\EG0\010\Eg0, ht=^I,
	il1=\EE, ind=^J, is2=\EZ\E3\E_c, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, khome=\E[H, rmcup=\E_b, rmir=\Er,
	rmso=\Eg0, rmul=\Eg0, sgr0=\Eg0, smcup=\E_d, smir=\Eq,
	smso=\Eg4, smul=\Eg1,
# (ts100: I added <rmam>/<smam> based on the init string -- esr)
ts100|ts100-sp|falco ts100-sp,
	am, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m$<2>, bold=\E[1m$<2>,
	clear=\E[H\E[J$<50>, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C$<2>,
	cup=\E[%i%p1%d;%p2%dH$<5>, cuu=\E[%p1%dA,
	cuu1=\E[A$<2>, dch1=\E~W, dl1=\E~R, ed=\E[J$<50>,
	el=\E[K$<3>, el1=\E[1K$<3>, enacs=\E(B\E)0, home=\E[H,
	ht=^I, hts=\EH, ich1=\E~Q, il1=\E~E, ind=^J, is1=\E~)\E~ea,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	rc=\E8, rev=\E[7m$<2>, ri=\EM$<5>, rmacs=^O, rmam=\E[?7l,
	rmkx=\E[?1l\E>, rmso=\E[m$<2>, rmul=\E[m$<2>,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;$<2>,
	sgr0=\E[m\017$<2>, smacs=^N, smam=\E[?7h, smkx=\E[?1h\E=,
	smso=\E[1;7m$<2>, smul=\E[4m$<2>, tbc=\E[3g,
	use=vt100+fnkeys,
ts100-ctxt|falco ts-100 saving context,
	rmcup=\E~_b, smcup=\E~_d\E[2J, use=ts100,

#### Florida Computer Graphics
#

# Florida Computer Graphics Beacon System, using terminal emulator program
# "host.com", as provided by FCG.  This description is for an early release
# of the "host" program.  Known bug: <ed> clears the whole screen, so it's
# commented out.

# From: David Bryant <cbosg!djb> 1/7/83
beacon|FCG Beacon System,
	am, da, db,
	cols#80, lines#32,
	bel=\ESTART\r\E37\r\EEND\r$<1>,
	blink=\ESTART\r\E61\,1\r\EEND\r, clear=\EZ$<10>, cr=^M,
	cub1=^H, cud1=^J, cuf1=\EV,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c$<20>, cuu1=\EU,
	dch1=\EW, dl1=\ER, el=\ET, home=\EH$<10>, ich1=\EQ, il1=\EE,
	ind=^J, rev=\ESTART\r\E59\,1\r\EEND\r, rmcup=,
	rmso=\ESTART\r\E70\,0\r\EEND\r$<20>,
	rmul=\ESTART\r\E60\,0\r\EEND\r,
	sgr0=\ESTART\r\E78\r\E70\,0\r\EEND\r$<20>,
	smcup=\ESTART\r\E2\,0\r\E12\r\EEND\r$<10>,
	smso=\ESTART\r\E70\,6\r\EEND\r$<20>,
	smul=\ESTART\r\E60\,1\r\EEND\r,

#### Fluke
#

# The f1720a differences from ANSI: no auto margin, destructive
# tabs, # of lines, funny highlighting and underlining
f1720|f1720a|fluke 1720A,
	xt,
	cols#80, lines#16, xmc#1,
	bel=^G, clear=\E[H\E[2J, cr=^M, cub1=^H, cud1=\E[B,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, ed=\E[J,
	el=\E[K, ind=\ED, is2=\E[H\E[2J, kcub1=^_, kcud1=^],
	kcuf1=^^, kcuu1=^\, ri=\EM, rmso=\E[m, rmul=\E[m, sgr0=\E[m,
	smso=\E[7m, smul=\E[4m,

#### Liberty Electronics (Freedom)
#
#	Liberty Electronics
#	48089 Fremont Blvd
#	Fremont CA 94538
#	Vox: (510)-623-6000
#	Fax: (510)-623-7021

# From: <faletti@berkeley.edu>
# (f100: added empty <acsc> to suppress a tic warning;
# made this relative to adm+sgr -- note that <invis> isn't
# known to work for f100 but does on the f110. --esr)
f100|freedom|freedom100|freedom model 100,
	OTbs, am, bw, hs, mir, msgr, xon,
	cols#80, lines#24,
	acsc=, bel=^G, cbt=\EI, clear=^Z, cr=^M, cub1=^H, cud1=^J,
	cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	dch1=\EW, dl1=\ER$<11.5*>, dsl=\Eg\Ef\r, ed=\EY, el=\ET,
	flash=\Eb$<200>\Ed, fsl=^M, home=^^, hpa=\E]%p1%{32}%+%c,
	ht=^I, hts=\E1, il1=\EE$<8.5*>, ind=^J, ip=$<6>,
	is2=\Eg\Ef\r\Ed, kbs=^H, kcbt=\EI, kcub1=^H, kcud1=^V,
	kcuf1=^L, kcuu1=^K, kf1=^A@\r, kf10=^AI\r, kf2=^AA\r,
	kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r,
	kf8=^AG\r, kf9=^AH\r, khome=^^, ri=\Ej, rmacs=\E$, rmir=\Er,
	smacs=\E%%, smir=\Eq, tbc=\E3, tsl=\Eg\Ef,
	vpa=\E[%p1%{32}%+%c, use=adm+sgr,
f100-rv|freedom-rv|freedom 100 in reverse video,
	flash=\Ed$<200>\Eb, is2=\Eg\Ef\r\Eb, use=f100,
# The f110 and f200 have problems with vi(1).  They use the ^V
# code for the down cursor key. When kcud1 is defined in terminfo
# as ^V, the Control Character Quoting capability (^V in insert mode)
# is lost! It cannot be remapped in vi because it is necessary to enter
# a ^V to to quote the ^V that is being remapped!!!
#
# f110/f200 users will have to decide whether
# to lose the down cursor key or the quoting capability. We will opt
# initially for leaving the quoting capability out, since use of VI
# is not generally applicable to most interactive applications
# (f110: added <ht>, <khome> & <kcbt> from f100 -- esr)
f110|freedom110|Liberty Freedom 110,
	bw@, eslok,
	it#8, wsl#80,
	blink=\EG2, bold=\EG0, civis=\E.1, cnorm=\E.2, cud1=^V,
	dim=\EG@, dl1=\ER, dsl=\Ef\r, flash=\Eb$<200/>\Ed, il1=\EE,
	ip@, is2@, kclr=^^, kdch1=\EW, kdl1=\ER, ked=\EY, kel=\ET,
	kf0=^AI\r, kf10@, kich1=\EQ, kil1=\EE, mc4=\Ea, mc5=\E`,
	ri=\EJ, rmacs=\E%%, rmir=\Er\EO, smacs=\E$, smir=\EO\Eq,
	smso=\EG<, tsl=\Ef, use=f100,
f110-14|Liberty Freedom 110 14inch,
	dch1@, use=f110,
f110-w|Liberty Freedom 110 - 132 cols,
	cols#132, use=f110,
f110-14w|Liberty Freedom 110 14in/132 cols,
	cols#132,
	dch1@, use=f110,
# (f200: added <acsc> to suppress tic warnings re <smacs>/<rmacs> --esr)
f200|freedom200|Liberty Freedom 200,
	OTbs, am, eslok, hs, mir, msgr, xon,
	cols#80, it#8, lines#24, wsl#80,
	acsc=, bel=^G, blink=\EG2, bold=\EG0, cbt=\EI, civis=\E.0,
	clear=^Z, cnorm=\E.1, cr=^M,
	csr=\Em0%p1%{32}%+%c%p2%{32}%+%c, cub1=^H, cud1=^V,
	cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	dch1=\EW, dim=\EG@, dl1=\ER, dsl=\Ef\r, ed=\EY, el=\ET,
	flash=\Eo$<200/>\En, fsl=^M, home=^^,
	hpa=\E]%p1%{32}%+%c, hts=\E1, il1=\EE, ind=^J, kbs=^H,
	kclr=^^, kcub1=^H, kcud1=^V, kcuf1=^L, kcuu1=^K, kdch1=\EW,
	kdl1=\ER, ked=\EY, kel=\ET, kf0=^AI\r, kf1=^A@\r, kf2=^AA\r,
	kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r,
	kf8=^AG\r, kf9=^AH\r, kich1=\EQ, kil1=\EE, mc4=\Ea, mc5=\E`,
	ri=\EJ, rmacs=\E%%, rmir=\Er, smacs=\E$, smir=\Eq, smso=\EG<,
	tbc=\E3, tsl=\Ef, vpa=\E[%p1%{32}%+%c, use=adm+sgr,
f200-w|Liberty Freedom 200 - 132 cols,
	cols#132, use=f200,
# The f200 has the ability to reprogram the down cursor key. The key is
# reprogrammed to ^J (linefeed). This value is remembered in non-volatile RAM,
# so powering the terminal off and on will not cause the change to be lost.
f200vi|Liberty Freedom 200 for vi,
	flash=\Eb$<200/>\Ed, kcud1=^J, use=f200,
f200vi-w|Liberty Freedom 200 - 132 cols for vi,
	cols#132, use=f200vi,

#### GraphOn (go)
#
#	Graphon Corporation
#	544 Division Street
#	Campbell, CA 95008
#	Vox: (408)-370-4080
#	Fax: (408)-370-5047
#	Net: troy@graphon.com (Troy Morrison)
#
#
# The go140 and go225 have been discontinued.  GraphOn now makes X terminals,
# including one odd hybrid that starts out life on power-up as a character
# terminal, than can be switched to X graphics mode (driven over the serial
# line) by an escape sequence.  No info on this beast yet.
# (go140: I added <rmam>/<smam> based on the init string -- esr)
go140|graphon go-140,
	OTbs,
	cols#80, it#8, lines#24,
	clear=\E[H\E[2J$<10/>, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%2d;%p2%2dH, cuu1=\E[A, dch1=\E[P, dl1=\E[M,
	ed=\E[J$<10/>, el=\E[K, ht=^I,
	if=/usr/share/tabset/vt100, il1=\E[L,
	is2=\E<\E=\E[?3l\E[?7l\E(B\E[J\E7\E[;r\E8\E[m\E[q,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kf1=\EOP,
	kf2=\EOQ, kf3=\EOR, kf4=\EOS, khome=\E[H, ri=\EM,
	rmam=\E[?7l, rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[m,
	rmul=\E[m, sgr0=\E[m, smam=\E[?7h, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m,
go140w|graphon go-140 in 132 column mode,
	am,
	cols#132,
	is2=\E<\E=\E[?3h\E[?7h\E(B\E[J\E7\E[;r\E8\E[m\E[q,
	use=go140,
# Hacked up vt200 termcap to handle GO-225/VT220
# From: <edm@nwnexus.WA.COM>
# (go225: I added <rmam>/<smam> based on the init string -- esr)
go225|go-225|Graphon 225,
	OTbs, am, mir, xenl,
	cols#80, it#8, lines#25, vt#3,
	blink=\E[5m, bold=\E[1m, clear=\E[H\E[J,
	csr=\E[%i%p1%d;%p2%dr, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P, dl1=\E[M,
	ed=\E[J, el=\E[K, home=\E[H, ht=^I, il1=\E[L, ind=\ED,
	is2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf1=\EOP,
	kf2=\EOQ, kf3=\EOR, kf4=\EOS, khome=\E[H, rc=\E8, rev=\E[7m,
	rf=/usr/share/tabset/vt100, ri=\EM, rmam=\E[?7l,
	rmcup=\E[!p\E[?7h\E[2;1;1#w, rmir=\E[4l, rmkx=\E>,
	rmso=\E[27m, rmul=\E[24m, rs1=\E[!p\E[?7h\E[2;1;1#w,
	sc=\E7, sgr0=\E[m, smam=\E[?7h, smcup=\E[2;0#w\E[1;25r,
	smir=\E[4h, smkx=\E=, smso=\E[7m, smul=\E[4m,

#### Harris (Beehive)
#
# Bletch.  These guys shared the Terminal Brain Damage laurels with Hazeltine.
# Their terminal group is ancient history now (1995) though the parent
# company is still in business.
#

# Beehive documentation is undated and marked Preliminary and has no figures
# so we must have early Superbee2 (Model 600, according to phone conversation
# with mfr.). It has proved reliable except for some missing padding
# (notably after \EK and <nl> at bottom of screen).
#
# The key idea is that AEP mode is poison for <cup> & that US's in
# the local memory should be avoided like the plague. That means
# that the 2048 character local buffer is used as 25 lines of 80
# characters, period. No scrolling local memory, folks. It also
# appears that we cannot use naked INS LINE feature since it uses
# US. The sbi fakes <il1> with an 80-space insert that may be too
# slow at low speeds; also spaces get converted to \040 which is
# too long for some programs (not vi).  DEL LINE is ok but slow.
#
# The <nl> string is designed for last line of screen ONLY; cup to
# 25th line corrects the motion inherent in scrolling to Page 1.
#
# There is one understood bug. It is that the screen appears to
# pop to a new (blank) page after a <nel>, or leave a half-line
# ellipsis to a quad that is the extra 48 memory locations. The
# data received is dumped into memory but not displayed.  Not to
# worry if <cup> is being used; the lines not displayed will be,
# whenever the cursor is moved up there. Since <cup> is addressed
# relative to MEMORY of window, nothing is lost; but beware of
# relative cursor motion (<cuu1>,<cud1>,<cuf1>,<cub1>). Recommended,
# therefore, is setenv MORE -c .
#
# WARNING: Not all features tested.
#
# Timings are assembled from 3 sources. Some timings may reflect
# SB2/Model 300 that were used if more conservative.
# Tested on a Model 600 at 1200 and 9600 bd.
#
# The BACKSPACEkb option is cute. The NEWLINE key, so cleverly
# placed on the keyboard and useless because of AEP, is made
# into a backspace key. In use ESC must be pressed twice (to send)
# and sending ^C must be prefixed by ESC to avoid that weird
# transmit mode associated with ENTER key.
#
# IF TERMINAL EVER GOES CATATONIC with the cursor buzzing across
# the screen, then it has dropped into ENTER mode; hit
# RESET--ONLINE--!tset.
#
# As delivered this machine has a FATAL feature that will throw
# it into that strange transmit state (SPOW) if the space bar is
# hit after a CR is received, but before receiving a LF (or a
# few others).
#
# The circuits MUST be modified to eliminate the SPOW latch.
# This is done by strapping on chip A46 of the I/O board; cut
# the p.c. connection to Pin 5 and strap Pin 5 to Pin 8 of that
# chip. This mod has been checked out on a Mod 600 of Superbee II.
# With this modification absurdly high timings on cr are
# unnecessary.
#
# NOTE WELL that the rear panel switch should be set to CR/LF,
# not AEP!
#
sb1|beehive superbee,
	OTbs, am, bw, da, db, mir, ul, xsb,
	cols#80, lines#25, xmc#1,
	bel=^G, cbt=\E`$<650>, clear=\EH$<1>\EJ$<3>, cr=$<1>\r,
	cub1=^H, cud1=^J, cuf1=\EC$<3>, cup=\EF%p2%03d%p1%03d,
	cuu1=\EA$<3>, dch1=\EP$<3>, dl1=\EM$<100>, ed=\EJ$<3>,
	el=\EK$<3>, home=\EH$<1>, ht=^I, hts=\E1,
	il1=\EN\EL$<3>\EQ                                                                                \EP$<3> \EO\ER\EA$<3>,
	ind=^J, is2=\EE$<3>\EX\EZ\EO\Eb\Eg\ER, kbs=^_, kcub1=\ED,
	kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kdl1=\EM, ked=\EJ, kel=\EK,
	kf0=\E2, kf1=\Ep, kf2=\Eq, kf3=\Er, kf4=\Es, kf5=\Et, kf6=\Eu,
	kf7=\Ev, kf8=\Ew, kf9=\E1, khome=\EH, kich1=\EQ\EO,
	krmir=\ER, lf0=TAB CLEAR, lf9=TAB SET, rmcup=, rmir=\ER,
	rmso=\E_3, rmul=\E_3, sgr0=\E_3, smcup=\EO, smir=\EQ\EO,
	smso=\E_1, smul=\E_0, tbc=\E3,
sbi|superbee|beehive superbee at Indiana U.,
	xsb,
	cr=\r$<1>, il1=1\EN\EL$<9>\EQ \EP$<9> \EO\ER\EA,
	use=sb1,
# Alternate (older) description of Superbee - f1=escape, f2=^C.
# Note: there are at least 3 kinds of superbees in the world.  The sb1
# holds onto escapes and botches ^C's.  The sb2 is the best of the 3.
# The sb3 puts garbage on the bottom of the screen when you scroll with
# the switch in the back set to CRLF instead of AEP.  This description
# is tested on the sb2 but should work on all with either switch setting.
# The f1/f2 business is for the sb1 and the <xsb> can be taken out for
# the other two if you want to try to hit that tiny escape key.
# This description is tricky: being able to use cup depends on there being
# 2048 bytes of memory and the hairy <nl> string.
superbee-xsb|beehive super bee,
	am, da, db, xsb,
	cols#80, it#8, lines#25,
	clear=\EH\EJ$<3>, cnorm=^J, cr=\r$<1000>, cub1=^H, cud1=^J,
	cuf1=\EC, cup=\EF%p2%3d%p1%3d, cuu1=\EA$<3>,
	dch1=\EP$<3>, dl1=\EM$<100>, ed=\EJ$<3>, el=\EK$<3>,
	home=\EH, ht=^I, hts=\E1,
	ind=\n\0\0\0\n\0\0\0\EA\EK\0\0\0\ET\ET, is2=\EH\EJ,
	kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kf1=\Ep, kf2=\Eq,
	kf3=\Er, kf4=\Es, kf5=\Et, kf6=\Eu, kf7=\Ev, kf8=\Ew,
	khome=\EH, rmso=\E_3, sgr0=\E_3, smso=\E_1, tbc=\E3,
# This loses on lines > 80 chars long, use at your own risk
superbeeic|super bee with insert char,
	ich1=, rmir=\ER, smir=\EQ, use=superbee-xsb,
sb2|sb3|fixed superbee,
	xsb@, use=superbee,

#### Beehive Medical Electronics
#
# Steve Seymour <srseymour@mindspring.com> writes (Wed, 03 Feb 1999):
# Regarding your question though; Beehive terminals weren't made by Harris.
# They were made by Beehive Medical Electronics in Utah. They went out of
# business in the early '80s.
#
# (OK, then, I don't know why a couple of these say "harris beehive".)
#

# Reports are that most of these Beehive entries (except superbee) have not
# been tested and do not work right.  <rmso> is a trouble spot.  Be warned.

# (bee: <ich1> was empty, which is obviously bogus -- esr)
beehive|bee|harris beehive,
	OTbs, am, mir,
	cols#80, lines#24,
	cbt=\E>, clear=\EE, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EF%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dch1=\EP,
	dl1=\EM, ed=\EJ, el=\EK, home=\EH, il1=\EL, kbs=^H, kcbt=\E>,
	kclr=\EE, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA,
	kdch1=\EP, kdl1=\EM, kel=\EK, khome=\EH, kich1=\EQ, kil1=\EL,
	krmir=\E@, rmir=\E@, rmso=\Ed@, rmul=\Ed@, sgr0=\Ed@,
	smir=\EQ, smso=\EdP, smul=\Ed`,
# set tab is ^F, clear (one) tab is ^V, no way to clear all tabs.
# good grief - does this entry make :sg:/:ug: when it doesn't have to?
# look at those spaces in <rmso>/<smso>.  Seems strange to me...
# (beehive: <if=/usr/share/tabset/beehive> removed, no such file.  If you
# really care, cook up one using ^F -- esr)
beehive3|bh3m|beehiveIIIm|harris beehive 3m,
	OTbs, am,
	cols#80, it#8, lines#20,
	bel=^G, clear=^E^R, cr=^M, cub1=^H, cud1=^J, cuf1=^L, cuu1=^K,
	dl1=\021$<350>, ed=^R, el=^P, home=^E, ht=^I, hts=^F,
	il1=\023$<160>, ind=^J, ll=^E^K, rmso=\s^_, smso=^]\s,
beehive4|bh4|beehive 4,
	am,
	cols#80, lines#24,
	bel=^G, clear=\EE, cr=^M, cub1=\ED, cud1=^J, cuf1=\EC,
	cuu1=\EA, ed=\EJ, el=\EK, home=\EH, ind=^J,
# There was an early Australian kit-built computer called a "Microbee".
# It's not clear whether this is for one of those or for a relative
# of the Beehive.
microb|microbee|micro bee series,
	OTbs, am,
	cols#80, it#8, lines#24,
	bel=^G, clear=\EE, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EF%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, ed=\EJ,
	el=\EK, ht=^I, ind=^J, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kf1=\Ep, kf2=\Eq, kf3=\Er, kf4=\Es, kf5=\Et,
	kf6=\Eu, kf7=\Ev, kf8=\Ew, kf9=\Ex, khome=\EH, rmso=\Ed@,
	rmul=\Ed@, sgr0=\Ed@, smso=\s\EdP, smul=\Ed`,

# 8675, 8686, and bee from Cyrus Rahman
# (8675: changed k10, k11...k16 to k;, F1...F6 -- esr)
ha8675|harris 8675,
	is2=\ES\E#\E*\Eh\Em\E?\E1\E9\E@\EX\EU, kf1=^F,
	kf10=\Ed, kf11=^W, kf12=\ER, kf13=\EE, kf14=\EI, kf15=\Ei,
	kf16=\Eg, kf2=^P, kf3=^N, kf4=^V, kf5=^J, kf6=^T, kf7=^H,
	kf8=\177, kf9=\Ee, use=bee,
# (8686: changed k10, k11...k16 to k;, F1...F6; fixed broken continuation
# in :is: -- esr)
ha8686|harris 8686,
	is2=\ES\E#\E*\Eh\Em\E?\E1\E9\E@\EX\EU\E"*Z01\E"8F35021B7C83#\E"8F45021B7D83#\E"8F55021B7E83#\E"8F65021B7F83#\E"8F75021B7383#\E"8F851BD7#\E"8F95021B7083#\E"8FA5021B7183#\E"8FB5021B7283#,
	kf1=\002\Ep\003, kf10=\Ej, kf11=\EW, kf12=\002\E{\003,
	kf13=\002\E|\003, kf14=\002\E}\003, kf15=\002\E~\003,
	kf16=\002\E\177\003, kf2=\002\Eq\003, kf3=\002\Er\003,
	kf4=\002\Es\003, kf5=\E3, kf6=\EI, kf7=\ER, kf8=\EJ, kf9=\E(,
	use=bee,

#### Hazeltine
#
# Hazeltine appears to be out of the terminal business as of 1995.  These
# guys were co-owners of the Terminal Brain Damage Hall Of Fame along with
# Harris. They have a hazeltine.com domain (but no web page there ) and can
# be reached at:
#
#	Hazeltine
#	450 East Pulaski Road
#	Greenlawn, New York 11740
#
# As late as 1993, manuals for the terminal product line could still be
# purchased from:
#
#	TRW Customer Service Division
#	15 Law Drive
#	P.O. Box 2076
#	Fairfield, NJ 07007-2078
#
# They're now (1998) a subsidiary of General Electric, operating under the
# marque "GEC-Marconi Hazeltine" and doing military avionics.  Web page
# at <http://www.gec.com/cpd/1ncpd.htm#1.55>.
#

# Since <cuf1> is blank, when you want to erase something you
# are out of luck.  You will have to do ^L's a lot to
# redraw the screen.  h1000 is untested.  It doesn't work in
# vi - this terminal is too dumb for even vi.  (The code is
# there but it isn't debugged for this case.)
hz1000|hazeltine 1000,
	OTbs,
	cols#80, lines#12,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=\s, home=^K,
	ind=^J,
# From: <cbosg!ucbvax!pur-ee!cincy!chris> Thu Aug 20 09:09:18 1981
hz1420|hazeltine 1420,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=\E^\, cr=^M, cub1=^H, cud1=^J, cuf1=^P,
	cup=\E\021%p2%c%p1%{32}%+%c, cuu1=\E^L, dl1=\E^S,
	ed=\E^X, el=\E^O, ht=^N, il1=\E^Z, ind=^J, rmso=\E^Y,
	smso=\E^_,
# New "safe" cursor movement (11/87) from <cgs@umd5.umd.edu>.  Prevents
# freakout with out-of-range args and tn3270.  No hz since it needs to
# receive tildes.
hz1500|hazeltine 1500,
	OTbs, am, hz,
	cols#80, lines#24,
	bel=^G, clear=~^\, cr=^M, cub1=^H, cud1=~^K, cuf1=^P,
	cup=~\021%p2%p2%?%{30}%>%t%{32}%+%;%{96}%+%c%p1%{96}%+%c,
	cuu1=~^L, dl1=~\023$<40>, ed=~\030$<10>, el=~^O, home=~^R,
	il1=~\032$<40>, ind=^J, kcub1=^H, kcud1=^J, kcuf1=^P,
	kcuu1=~^L, khome=~^R, rmso=~^Y, smso=~^_,
# h1510 assumed to be in sane escape mode.  Else use h1500.
# (h1510: early versions of this entry apparently had "<rmso=\E^_>,
# <smso=\E^Y>, but these caps were commented out in 8.3; also,
# removed incorrect and overridden ":do=^J:" -- esr)
hz1510|hazeltine 1510,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=\E^\, cr=^M, cub1=^H, cud1=\E^K, cuf1=^P,
	cup=\E\021%p2%c%p1%c, cuu1=\E^L, dl1=\E^S, ed=\E^X,
	el=\E^O, il1=\E^Z, ind=^J,
# Hazeltine 1520
# The following switch settings are assumed for normal operation:
#	FULL		CR		U/L_CASE	ESCAPE
#	FORMAT_OFF	EOM_A_OFF	EOM_B_OFF	WRAPAROUND_ON
# Other switches may be set for operator convenience or communication
# requirements.
hz1520|Hazeltine 1520,
	OTbs, am, bw, msgr,
	cols#80, lines#24,
	bel=^G, bold=\E^_, clear=\E^\, cr=^M, cub1=^H, cud1=^J,
	cuf1=^P, cup=\E\021%p2%c%p1%c, cuu1=\E^L, dl1=\E^S,
	ed=\E^X, el=\E^O, home=\E^R, il1=\E^Z, ind=^J, kbs=^H,
	kclr=\E^\, kcub1=^H, kcud1=\E^K, kcuf1=^P, kcuu1=\E^L,
	kdl1=\E^S, ked=\E^X, kel=\E^O, khome=\E^R, kil1=\E^Z,
	rmso=\E^Y, rs1=\E$\E\005\E?\E\031, sgr0=\E^Y, smso=\E^_,
# This version works with the escape switch off
# (h1520: removed incorrect and overridden ":do=^J:" -- esr)
hz1520-noesc|hazeltine 1520,
	am, hz,
	cols#80, lines#24,
	bel=^G, clear=~^\, cr=^M, cub1=^H, cud1=~^K, cuf1=^P,
	cup=~\021%p2%c%p1%c$<1>, cuu1=~^L, dl1=~^S, ed=~^X, el=~^O,
	home=~^R, il1=~^Z, ind=^J, rmso=~^Y, smso=~^_,
# Note: the h1552 appears to be the first Hazeltine terminal which
# is not braindamaged.  It has tildes and backprimes and everything!
# Be sure the auto lf/cr switch is set to cr.
hz1552|hazeltine 1552,
	OTbs,
	cud1=^J, dl1=\EO, il1=\EE, kf1=\EP, kf2=\EQ, kf3=\ER, lf1=blue,
	lf2=red, lf3=green, use=vt52,
hz1552-rv|hazeltine 1552 reverse video,
	cud1=^J, rmso=\ET, smso=\ES, use=hz1552,
# Note: h2000 won't work well because of a clash between upper case and ~'s.
hz2000|hazeltine 2000,
	OTbs, OTnc, am,
	cols#74, lines#27,
	bel=^G, clear=~\034$<6>, cub1=^H, cud1=^J,
	cup=~\021%p2%c%p1%c, dl1=~\023$<6>, home=~^R,
	il1=~\032$<6>, ind=^J, pad=\177,
# Date: Fri Jul 23 10:27:53 1982.  Some unknown person wrote:
# I tested this termcap entry for the Hazeltine Esprit with vi. It seems
# to work ok. There is one problem though if one types a lot of garbage
# characters very fast vi seems not able to keep up and hangs while trying
# to insert. That's in insert mode while trying to insert in the middle of
# a line. It might be because the Esprit doesn't have insert char and delete
# char as a built in function. Vi has to delete to end of line and then
# redraw the rest of the line.
esprit|Hazeltine Esprit I,
	OTbs, am, bw,
	cols#80, lines#24,
	bel=^G, cbt=\E^T, clear=\E^\, cr=^M, cub1=^H, cud1=\E^K,
	cuf1=^P, cup=\E\021%p2%c%p1%c, cuu1=\E^L, dl1=\E^S,
	ed=\E^W, el=\E^O, home=\E^R, il1=\E^Z, ind=^J, is2=\E?, kbs=^H,
	kcub1=^H, kcud1=\E^K, kcuf1=^P, kcuu1=\E^L, kf0=^B0^J,
	kf1=^B1^J, kf2=^B2^J, kf3=^B3^J, kf4=^B4^J, kf5=^B5^J,
	kf6=^B6^J, kf7=^B7^J, kf8=^B8^J, kf9=^B9^J, khome=\E^R,
	lf0=0, lf1=1, lf2=2, lf3=3, lf4=4, lf5=5, lf6=6, lf7=7, lf8=8, lf9=9,
	rmkx=\E>, rmso=\E^Y, smkx=\E<, smso=\E^_,
esprit-am|hazeltine esprit auto-margin,
	am, use=esprit,
# Hazeltine Modular-1 from Cliff Shackelton <ittvax!ittral!shackelt> via BRL
# Vi it seems always wants to send a control J for "do" and it turned out
# that the terminal would work somewhat if the auto LF/CR was turned off.
# (hmod1: removed :dn=~^K: -- esr)
hmod1|Hazeltine Modular 1,
	OTbs, am, hz,
	cols#80, lines#24,
	bel=^G, cbt=~^T, clear=~^\, cr=^M, cub1=^H, cud1=~^K, cuf1=^P,
	cup=~\021%p2%c%p1%c, cuu1=~^L, dl1=~^S, home=~^R, il1=~^Z,
	ind=^J, kcub1=^H, kcud1=~^K, kcuf1=^P, kcuu1=~^L, khome=~^R,
	rc=~^Q, rmso=~^Y, sc=~^E, sgr0=~^Y, smso=~^_,
#
# Hazeltine Executive 80 Model 30 (1554?)
#	from  Will Martin <control@ALMSA-1.ARPA> via BRL
# Like VT100, except for different "am" behavior.
hazel|exec80|h80|he80|Hazeltine Executive 80,
	OTbs, OTpt, am,
	cols#80, it#8, lines#24, vt#3,
	OTnl=^J, bel=^G, blink=\E[5m$<2/>, bold=\E[1m$<2/>,
	clear=\E[;H\E[2J$<50/>, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub1=^H, cud1=^J, cuf1=\E[C$<2/>,
	cup=\E[%i%p1%d;%p2%dH$<5/>, cuu1=\E[A$<2/>,
	ed=\E[J$<50/>, el=\E[K$<3/>, home=\E[H, ht=^I,
	is2=\E[1;24r\E[24;1H, kbs=^H, kcub1=\EOD, kcud1=\EOB,
	kcuf1=\EOC, kcuu1=\EOA, kf1=\EOP, kf2=\EOQ, kf3=\EOR,
	kf4=\EOS, rc=\E8, rev=\E[7m$<2/>,
	rf=/usr/share/tabset/vt100, ri=\EM$<5/>,
	rmkx=\E[?1l\E>, rmso=\E[m$<2/>, rmul=\E[m$<2/>,
	rs1=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr0=\E[m$<2/>, smkx=\E[?1h\E=, smso=\E[7m$<2/>,
	smul=\E[4m$<2/>,

#### IBM
#

ibm327x|line mode IBM 3270 style,
	gn,
	clear=^M^J, el=^M, home=^M,

ibm3101|i3101|IBM 3101-10,
	OTbs, am, xon,
	cols#80, lines#24,
	bel=^G, clear=\EK, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, ed=\EJ,
	el=\EI, home=\EH, hts=\E0, ind=^J, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, nel=^M^J, tbc=\EH,
ibm3151|IBM 3151 display,
	is2=\E S, rmacs=\E>B, rmcup=\E>B, rs2=\E S, s0ds=\E>B,
	sgr=\E4%{64}%?%p1%t%{65}%|%;%?%p2%t%{66}%|%;%?%p3%t%{65}%|%;%?%p4%t%{68}%|%;%?%p5%t%{64}%|%;%?%p6%t%{72}%|%;%?%p7%t%{80}%|%;%c%?%p9%t\E>A%e\E>B%;,
	sgr0=\E4@\E>B, smacs=\E>A, smcup=\E>B, use=ibm3162,
# From: Mark Easter <marke@fsi-ssd.csg.ssd.fsi.com> 29 Oct 1992
# removed kend, knp, kpp -TD
ibm3161|ibm3163|wy60-316X|wyse60-316X|IBM 3161/3163 display,
	OTbs, am, mir, msgr,
	cols#80, it#8, lines#24,
	acsc=j\352k\353l\354m\355n\356q\361t\364u\365v\366w\367x\370,
	bel=^G, blink=\E4D, bold=\E4H, clear=\EH\EJ, cr=^M, cub1=\ED,
	cud1=\EB, cuf1=\EC, cup=\EY%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=\EA, dch1=\EQ, dl1=\EO, ed=\EJ, el=\EI, home=\EH, ind=^J,
	invis=\E4P, kbs=^H, kcbt=\E2, kclr=\EL\r, kctab=\E1,
	kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kdch1=\EQ,
	kdl1=\EO, ked=\EJ, kel=\EI, kf1=\Ea\r, kf10=\Ej\r,
	kf11=\Ek\r, kf12=\El\r, kf13=\E!a\r, kf14=\E!b\r,
	kf15=\E!c\r, kf16=\E!d\r, kf17=\E!e\r, kf18=\E!f\r,
	kf19=\E!g\r, kf2=\Eb\r, kf20=\E!h\r, kf21=\E!i\r,
	kf22=\E!j\r, kf23=\E!k\r, kf24=\E!l\r, kf3=\Ec\r,
	kf4=\Ed\r, kf5=\Ee\r, kf6=\Ef\r, kf7=\Eg\r, kf8=\Eh\r,
	kf9=\Ei\r, khome=\EH, khts=\E0, kich1=\EP \010, kil1=\EN,
	ktbc=\E 1, mc4=^P^T, mc5=^P^R, rev=\E4A, rmcup=\E>A,
	rmso=\E4@, rmul=\E4@,
	sgr=\E4%{64}%?%p1%t%{65}%|%;%?%p2%t%{66}%|%;%?%p3%t%{65}%|%;%?%p4%t%{68}%|%;%?%p5%t%{64}%|%;%?%p6%t%{72}%|%;%?%p7%t%{80}%|%;%c%?%p9%t\E>A%e\E<@%;,
	sgr0=\E4@\E<@, smcup=\E>A, smso=\E4A, smul=\E4B,

ibm3161-C|IBM 3161-C NLS terminal using cartridge,
	rmcup=\E>B, s0ds=\E>B, s1ds=\E>A, smcup=\E>B, use=ibm3161,
ibm3162|IBM 3162 display,
	blink=\E4$a, bold=\E4(a, il1=\EN, invis=\E40a, rev=\E4!a,
	rmso=\E4>b, rmul=\E4=b, sgr0=\E4@, smso=\E4!a, smul=\E4"a,
	use=ibm3161-C,

# This really should not use setab/setaf, but it is clear that the
# original terminfo does not toggle red/blue colors as in setb/setf.
ibm3164|i3164|IBM 3164,
	msgr,
	colors#8, pairs#64,
	op=\E4 "@, rmcup=\E!9(N\E>B, s0ds=\E>B, s1ds=\E>A,
	setab=\E4  %p1%{64}%+%c,
	setaf=\E4%?%p1%t %p1%{32}%+%c%e!'%;@,
	smcup=\E!9/N\E>B, use=ibm3161,

ibm5151|wy60-AT|wyse60-AT|IBM 5151 Monochrome display,
	am, bw, msgr, xon,
	cols#80, it#8, lines#25,
	acsc=j\331k\277l\332m\300n\305q\304t\303u\264v\301w\302x\263,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[J, cr=^M,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ech=\E[%p1%dX, ed=\E[J, el=\E[K, home=\E[H,
	hpa=\E[%i%p1%dG, il=\E[%p1%dL, il1=\E[L, ind=\E[S,
	indn=\E[%p1%dS, invis=\E[8m, is2=\Ec, kbs=^H, kcbt=\E[Z,
	kclr=\E[144q, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kdch1=\E[P, ked=\E[148q, kel=\E[142q,
	kend=\E[146q, kf1=\E[001q, kf10=\E[010q, kf11=\E[011q,
	kf12=\E[012q, kf13=\E[013q, kf14=\E[014q, kf15=\E[015q,
	kf16=\E[016q, kf17=\E[017q, kf18=\E[018q, kf19=\E[019q,
	kf2=\E[002q, kf20=\E[020q, kf21=\E[021q, kf22=\E[022q,
	kf23=\E[023q, kf24=\E[024q, kf25=\E[025q, kf26=\E[026q,
	kf27=\E[027q, kf28=\E[028q, kf29=\E[029q, kf3=\E[003q,
	kf30=\E[030q, kf31=\E[031q, kf32=\E[032q, kf33=\E[033q,
	kf34=\E[034q, kf35=\E[035q, kf36=\E[036q, kf4=\E[004q,
	kf5=\E[005q, kf6=\E[006q, kf7=\E[007q, kf8=\E[008q,
	kf9=\E[009q, khome=\E[H, kich1=\E[139q, kil1=\E[140q,
	kind=\E[151q, knp=\E[154q, kpp=\E[150q, kri=\E[155q,
	krmir=\E[4l, rev=\E[7m, ri=\E[T, rin=\E[%p1%dT, rmir=\E[4l,
	rmso=\E[m, rmul=\E[m, rs2=\Ec,
	sgr=\E[%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;%?%p7%t;8%;m,
	sgr0=\E[0m, smir=\E[4h, smso=\E[7m, smul=\E[4m,

ibmaed|IBM Experimental display,
	OTbs, am, eo, msgr,
	cols#80, it#8, lines#52,
	clear=\EH\EK, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dch1=\EQ,
	dl1=\EO, ed=\EJ, el=\EI, flash=\EG, home=\EH, ht=^I, ich1=\EP,
	il1=\EN, kbs=^H, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA,
	rmso=\E0, sgr0=\E0, smso=\E0,
ibm-apl|apl|IBM apl terminal simulator,
	lines#25, use=dm1520,
# (ibmmono: this had an unknown `sb' boolean, I changed it to `bs'.
# Also it had ":I0=f10:" which pretty obviously should be "l0=f10" -- esr)
ibmmono|IBM workstation monochrome,
	eslok, hs,
	bold=\EZ, dl1=\EM, dsl=\Ej\EY8 \EI\Ek, fsl=\Ek, il1=\EL,
	invis=\EF\Ef0;\Eb0;, kbs=^H, kf0=\E<, kf1=\ES, kf2=\ET,
	kf3=\EU, kf4=\EV, kf5=\EW, kf6=\EP, kf7=\EQ, kf8=\ER, kf9=\EY,
	khome=\EH, kich1=\0, kind=\EE, knp=\EE, kpp=\Eg, kri=\EG,
	lf0=f10, rev=\Ep, ri=\EA, rmso=\Ez, rmul=\Ew,
	sgr0=\Ew\Eq\Ez\EB, smso=\EZ, smul=\EW, tsl=\Ej\EY8%+ \Eo,
	use=ibm3101,
ibmega|IBM Enhanced Color Display,
	cr=^M, cud1=^J, ht=^I, ind=^J, kbs=^H, kcub1=^H, kcud1=^J,
	nel=^M^J, use=ibmmono,
# This color scheme is assumed in some recent IBM terminal descriptions
# (green on black, emulated on a 16-color terminal).
ibm+color|IBM color definitions,
	colors#8, ncv#3, pairs#64,
	op=\E[32m\E[40m,
	setb=\E[%?%p1%{0}%=%t40m%e%p1%{1}%=%t41m%e%p1%{2}%=%t42m%e%p1%{3}%=%t43m%e%p1%{4}%=%t44m%e%p1%{5}%=%t45m%e%p1%{6}%=%t46m%e%p1%{7}%=%t107m%;,
	setf=\E[%?%p1%{0}%=%t30m%e%p1%{1}%=%t31m%e%p1%{2}%=%t32m%e%p1%{3}%=%t33m%e%p1%{4}%=%t34m%e%p1%{5}%=%t35m%e%p1%{6}%=%t36m%e%p1%{7}%=%t97m%;,
ibm+16color|IBM aixterm color definitions,
	colors#16, pairs#256,
	setab=\E[%?%p1%{8}%<%t%p1%{40}%+%e%p1%{92}%+%;%dm,
	setaf=\E[%?%p1%{8}%<%t%p1%{30}%+%e%p1%{82}%+%;%dm,
	setb=%p1%{8}%/%{6}%*%{4}%+\E[%d%p1%{8}%m%Pa%?%ga%{1}%=%t4%e%ga%{3}%=%t6%e%ga%{4}%=%t1%e%ga%{6}%=%t3%e%ga%d%;m,
	setf=%p1%{8}%/%{6}%*%{3}%+\E[%d%p1%{8}%m%Pa%?%ga%{1}%=%t4%e%ga%{3}%=%t6%e%ga%{4}%=%t1%e%ga%{6}%=%t3%e%ga%d%;m,
ibm5154|IBM 5154 Color display,
	colors#8, ncv@, pairs#64,
	bold@, setab=\E[4%p1%dm, setaf=\E[3%p1%dm, use=ibm5151,
	use=ibm+color,
ibmega-c|ibm5154-c|IBM Enhanced Color Display with standout and underline,
	rmso=\EB, rmul=\EB, smso=\EF\Ef3;, smul=\EF\Ef2;,
	use=ibmmono,
ibmvga-c|IBM VGA display color termcap,
	cr=^M, cud1=^J, ht=^I, ind=^J, kbs=^H, kcub1=^H, kcud1=^J,
	nel=^M^J, use=ibmega-c,
ibmvga|IBM VGA display,
	cr=^M, cud1=^J, ht=^I, ind=^J, kbs=^H, kcub1=^H, kcud1=^J,
	nel=^M^J, use=ibmega,
# ibmapa* and ibmmono entries come from ACIS 4.3 distribution
rtpc|ibmapa16|IBM 6155 Extended Monochrome Graphics Display,
	lines#32,
	dsl=\Ej\EY@ \EI\Ek, tsl=\Ej\EY@%+ \Eo, use=ibmmono,
ibm6155|IBM 6155 Black & White display,
	blink@, bold@, use=ibm5151,
# Advanced Monochrome (6153) and Color (6154) Graphics Display:
ibmapa8c|ibmapa8|IBM 6154 Advanced Graphics Display,
	lines#31,
	dsl=\Ej\EY? \EI\Ek, tsl=\Ej\EY?%+ \Eo, use=ibmmono,
ibmapa8c-c|ibm6154-c|IBM 6154 Advanced Color Graphics Display,
	lines#31,
	dim=\EF\Ef7;, dsl=\Ej\EY? \EI\Ek, tsl=\Ej\EY?%+ \Eo,
	use=ibmega-c,
ibm6154|IBM 6154 Color displays,
	blink@, bold=\E[12m, s0ds=\E[10m, s1ds=\E[11m, s2ds=\E[12m,
	sgr=\E[%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;12%;m,
	sgr0=\E[0;10m, use=ibm5154,
ibm6153|IBM 6153 Black & White display,
	blink@, bold=\E[12m, s0ds=\E[10m, s1ds=\E[11m, s2ds=\E[12m,
	sgr=\E[%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;12%;m,
	sgr0=\E[0;10m, use=ibm5151,
ibm6153-90|IBM 6153 Black & White display,
	cols#90, lines#36,
	blink@, bold@, use=ibm5151,
ibm6153-40|IBM 6153 Black & White display,
	cols#40, lines#12, use=ibm6153-90,
ibm8512|ibm8513|IBM color VGA Terminal,
	am, mir, msgr,
	cols#80, it#8, lines#25,
	acsc=jjkkllmmnnqqttuuvvwwxx, blink=\E[5m, bold=\E[1m,
	clear=\E[H\E[J, cub1=\E[D, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J, el=\E[K, home=\E[H, il=\E[%p1%dL,
	il1=\E[L, is2=\Eb\E[m\017\E[?7h, kcud1=\E[B, kcuu1=\E[A,
	kf0=\E[010q, kf1=\E[001q, kf2=\E[002q, kf3=\E[003q,
	kf4=\E[004q, kf5=\E[005q, kf6=\E[006q, kf7=\E[007q,
	kf8=\E[008q, kf9=\E[009q, khome=\E[H, rc=\E[u, rev=\E[7m,
	rmacs=^O, rmam=\E[?7l, rmcup=\E[20h, rmdc=\E[4l,
	rmir=\E[4l, rmso=\E[m, rmul=\E[m,
	rs1=\Eb\E[m\017\E[?7h\E[H\E[J, sc=\E[s, sgr0=\E[m,
	smacs=^N, smam=\E[?7h, smcup=\E[20;4l\E[?7h\Eb,
	smdc=\E[4h, smir=\E[4h, smso=\E[7m, smul=\E[4m,
	use=ibm8503,
hft-c|HFT with Color,
	colors#8, pairs#64,
	acsc=jjkkllmmnnqqttuuvvwwxx, s0ds=\E(B, s1ds=\E(0,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm, sgr0=\E[0m\E(B,
	use=ibm5151, use=ibm+color,
hft-c-old|HFT with Color PC850,
	colors#8, pairs#64,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm, use=ibm5151,
	use=ibm+color,
hft-old|AIWS High Function Terminal,
	am, xon,
	cols#80, lines#25,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[J, cr=^M,
	cub1=^H, cud1=^J, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu1=\E[A, dch1=\E[P, dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H,
	ht=^I, ich1=\E[@, il1=\E[L, ind=^J, invis=\E[8m, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kf1=\E[001q, kf2=\E[002q, kf3=\E[003q, kf4=\E[004q,
	kf5=\E[005q, kf6=\E[006q, kf7=\E[007q, kf8=\E[008q,
	kf9=\E[009q, khome=\E[H, knp=\E[153q, kpp=\E[159q,
	ktbc=\E[010q, rev=\E[7m, rmir=\E6, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m, smir=\E6, smso=\E[7m, smul=\E[4m, use=ibm+color,
ibm-system1|system1|ibm system/1 computer,
	am, xt,
	cols#80, lines#24,
	bel=^G, clear=^Z, cub1=^H, cuf1=^\,
	cup=\005%p1%{32}%+%c%p2%{32}%+%c, cuu1=^^, home=^K,
	ind=^J,
#       lft-pc850 : IBM Low Function Terminal Device
#    lft "supports" underline, bold, and blink in the sense that the lft code
#    sets all the right bits.  HOWEVER, depending upon the adapter, these
#    attributes may or may not be supported by the device driver.
lft|lft-pc850|LFT-PC850|IBM LFT PC850 Device,
	am, bw, msgr, xon,
	cols#80, it#8, lines#25,
	acsc=j\331k\277l\332m\300n\305q\304t\303u\264v\301w\302x\263,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, clear=\E[H\E[J,
	cr=^M, cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dl=\E[%p1%dM, dl1=\E[M, ech=\E[%p1%dX, ed=\E[2J, el=\E[0K,
	home=\E[H, hpa=\E[%i%p1%dG, ich=\E[%p1%d@, il=\E[%p1%dL,
	il1=\E[L, ind=\ED, indn=\E[%p1%dS, invis=\E[8m, is2=\Ec,
	kbs=^H, kcbt=\E[Z, kclr=\E[144q, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P, ked=\E[148q,
	kel=\E[142q, kend=\E[146q, kf1=\E[001q, kf10=\E[010q,
	kf11=\E[011q, kf12=\E[012q, kf13=\E[013q, kf14=\E[014q,
	kf15=\E[015q, kf16=\E[016q, kf17=\E[017q, kf18=\E[018q,
	kf19=\E[019q, kf2=\E[002q, kf20=\E[020q, kf21=\E[021q,
	kf22=\E[022q, kf23=\E[023q, kf24=\E[024q, kf25=\E[025q,
	kf26=\E[026q, kf27=\E[027q, kf28=\E[028q, kf29=\E[029q,
	kf3=\E[003q, kf30=\E[030q, kf31=\E[031q, kf32=\E[032q,
	kf33=\E[033q, kf34=\E[034q, kf35=\E[035q, kf36=\E[036q,
	kf4=\E[004q, kf5=\E[005q, kf6=\E[006q, kf7=\E[007q,
	kf8=\E[008q, kf9=\E[009q, khome=\E[H, kich1=\E[139q,
	kil1=\E[140q, kind=\E[151q, knp=\E[154q, kpp=\E[150q,
	kri=\E[155q, krmir=\E[4l, rev=\E[7m, ri=\EL, rin=\E[%p1%dT,
	rmacs=\E(B, rmir=\E[4l, rmso=\E[0m, rmul=\E[0m, rs2=\Ec,
	sgr=\E[%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;%?%p7%t;8%;m%?%p9%t\E(0%e\E(B%;,
	sgr0=\E[0m, smacs=\E(0, smir=\E[4h, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g,
# "Megapel" refers to the display adapter, which was used with the IBM RT
# aka IBM 6150.
ibm5081|hft|IBM Megapel Color display,
	acsc=jjkkllmmnnqqttuuvvwwxx, blink@, bold@, s0ds=\E(B,
	s1ds=\E(0, sgr0=\E[0m\E(B, use=ibm5154,
ibm5081-c|ibmmpel-c|IBM 5081 1024x1024 256/4096 Megapel enhanced color display,
	eslok, hs,
	lines#33,
	dsl=\Ej\EYA \EI\Ek, fsl=\Ek, tsl=\Ej\EYA%+ \Eo,
	use=ibmega-c,
ibm8503|ibm8507|ibm8604|IBM 8503 B & W VGA display,
	use=hft-c,
ibm8514|IBM 8514/a color VGA display,
	eslok, hs,
	dsl=\Ej\EYI \EI\Ek, fsl=\Ek, tsl=\Ej\EYI%+ \Eo, use=hft,
ibm8514-c|IBM 8514 color display with standout and underline,
	eslok, hs,
	lines#41,
	cr=^M, cud1=^J, dsl=\Ej\EYI \EI\Ek, fsl=\Ek, ht=^I, ind=^J,
	kbs=^H, kcub1=^H, kcud1=^J, nel=^M^J, tsl=\Ej\EYI%+ \Eo,
	use=ibmega-c,

#
# AIX entries.  IBM ships these with AIX 3.2.5.
# -- added rc, sc based on manpage -TD
# Note that we could use ibm+16color, but that is not how IBM defines this one.
aixterm|IBM Aixterm Terminal Emulator,
	eslok, hs,
	acsc=jjkkllmmnnqqttuuvvwwxx, bold=\E[1m, dsl=\E[?E,
	fsl=\E[?F, rc=\E8, ri@, s0ds=\E(B, s1ds=\E(0, sc=\E7,
	sgr=\E[%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;%?%p7%t;8%;m%?%p9%t\E(0%e\E(B%;,
	sgr0=\E[0;10m\E(B, tsl=\E[?%p1%dT, use=ibm6154,
aixterm-m|IBM AIXterm Monochrome Terminal Emulator,
	eslok, hs,
	acsc=jjkkllmmnnqqttuuvvwwxx, bold=\E[1m, dsl=\E[?E,
	fsl=\E[?F, ri@, s0ds=\E(B, s1ds=\E(0,
	sgr=\E[%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;m,
	sgr0=\E[0;10m\E(B, tsl=\E[?%p1%dT, use=ibm6153,
aixterm-m-old|old IBM AIXterm Monochrome Terminal Emulator,
	eslok, hs,
	bold=\E[1m, dsl=\E[?E, fsl=\E[?F, ri@,
	sgr=\E[%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;m,
	tsl=\E[?%p1%dT, use=ibm6153,
jaixterm|IBM Kanji Aixterm Terminal Eemulator,
	acsc@, use=aixterm,
jaixterm-m|IBM Kanji AIXterm Monochrome Terminal Emulator,
	acsc@, use=aixterm-m,

# This flavor is adapted from xterm, in turn from aixterm documentation -TD
aixterm-16color|IBM Aixterm Terminal Emulator with 16 colors,
	use=ibm+16color, use=aixterm,

#### Infoton/General Terminal Corp.
#

# gt100 sounds like something DEC would come out with.  Let's hope they don't.
i100|gt100|gt100a|General Terminal 100A (formerly Infoton 100),
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\Ef%p2%{32}%+%c%p1%{32}%+%c, cuu1=\EA, dl1=\EM,
	ed=\EJ, el=\EK, flash=\Eb$<200/>\Ea, home=\EH, il1=\EL,
	ind=^J, rmso=\Ea, smso=\Eb,
i400|infoton 400,
	OTbs, am,
	cols#80, lines#25,
	bel=^G, clear=\E[2J, cr=^M, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%3d;%p2%3dH, cuu1=\E[A,
	dch1=\E[4h\E[2Q\E[P\E[4l\E[0Q, dl1=\E[M, el=\E[N,
	il1=\E[L, ind=^J, rmir=\E[4l\E[0Q, smir=\E[4h\E[2Q,
# (addrinfo: removed obsolete ":bc=^Z:" -- esr)
addrinfo,
	am,
	cols#80, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^Z, cud1=^J, cuf1=^Y,
	cup=\037%p1%c%p2%c, cuu1=^\, ed=^K, home=^H, ind=^J, ll=^H^\,
# (infoton: used to have the no-ops <lh#0>, <lw#0>, <nlab#0> -- esr)
infoton,
	am,
	cols#80, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^Z, cud1=^J, cuf1=^Y, cuu1=^\,
	ed=^K, ind=^J, ll=^H^\,

# The ICL6402 was actually the Kokusai Display System 6402.
# The 6404 was the KDS7372 (color version of the 6402).
#
# ICL6404 control codes follow:
#
#code            function
#~~~~~~~~~~~     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#ctrl-A          set SOM position at cursor position
#ctrl-G          Bell
#ctrl-H          Backspace
#ctrl-I          Horiz tab
#ctrl-J          Linefeed
#ctrl-K          Cursor up
#ctrl-L          Cursor right
#ctrl-M          Carriage return
#ctrl-N          Disable xon/xoff to host
#ctrl-O          Enable xon/xoff to host
#ctrl-R          Enable bidirectional mode
#ctrl-T          Disable bidirectional mode
#ctrl-V          Cursor down
#ctrl-Z          Clear unprotected data to insert char
#ctrl-^          Cursor home
#ctrl-_          Newline
#
#ESC             lead-in char for multiple character command
#
#ESC space R     execute power on sequence
#ESC ! p1 p2     define scroll region:
#                p1 = scroll top    line:  20h - 37h
#                p1 = scroll bottom line:  20h - 37h
#ESC "           unlock keyboard
#ESC #           lock keyboard
#ESC $           Semi-graphics mode on
#ESC %           Semi-graphics mode off
#ESC &           protect mode on
#ESC '           protect mode off
#ESC (           write protect mode off (full intensity)
#ESC )           write protect mode on (half intensity)
#
#ESC *           clear screen
#ESC +           clear unprotected data to insert char
#ESC ,           clear unprotected data to half intensity spaces
#ESC - p1 p2 p3 p4     address cursor to page, row, column:
#                      p1 = page number  0 - 3
#                      p2 = row          20h - 7fh
#                      p3 = column (lo)  20h - 7fh
#                      p4 = column (hi)  20h - 21h (only 132 col)
#ESC . p1        set cursor style:
#                p1 = 0  invisible cursor
#                p1 = 1  block blinking cursor
#                p1 = 2  block steady cursor
#                p1 = 3  underline blinking cursor
#                p1 = 4  underline steady cursor
#ESC /           transmit cursor location (page, row, column)
#ESC 0 p1 p2 p3 p4     program edit key:
#                      p1 = edit key code: '@'-'S', '`'-'s'
#                      p2 p3 p4 = program data (3 bytes)
#
#ESC 1           set tab
#ESC 2           clear tab at cursor
#ESC 3           clear all tabs
#ESC 4           send unprotect line to cursor
#ESC 5           send unprotect page to cursor
#ESC 6           send line to cursor
#ESC 7           send page to cursor
#ESC 8 n         set scroll mode:
#                n = 0   set jump scroll
#                n = 1   set smooth scroll
#ESC 9 n         control display:
#                n = 0   display off
#                n = 1   display on
#ESC :           clear unprotected data to null
#ESC ;           clear unprotected data to insert char
#
#ESC <           keyclick on
#ESC = p1 p2     address cursor to row, column
#                p1 = row          20h - 7fh
#                p2 = column (lo)  20h - 7fh
#                p3 = column (hi)  20h - 21h (only 132 col)
#ESC >           keyclick off
#ESC ?           transmit cursor location (row, column)
#
#ESC @           copy print mode on
#ESC A           copy print mode off
#ESC B           block mode on
#ESC C           block mode off (conversation mode)
#ESC D F         set full duplex
#ESC D H         set half duplex
#ESC E           line insert
#ESC F p1 p2     set page colour (p1 = f/grnd, p2 = b/grnd)
#                0 = black, 1 = red,     2 = green, 3 = yellow
#                4 = blue,  5 = magenta, 6 = cyan,  7 = white
#ESC G n         set serial field attribute (n = 30h - 3Fh)
#ESC H n         full graphics mode:
#                n = 0  exit full graphics mode
#                n = 1  enter full graphics mode
#ESC I           back tab
#ESC J           back page
#ESC K           forward page
#
#ESC L           unformatted page print
#ESC M L         move window left  (132 col mode only)
#ESC M R         move window right (132 col mode only)
#ESC N           set page edit (clear line edit)
#ESC O           set line edit (clear page edit)
#ESC P           formatted page print
#ESC Q           character insert
#ESC R           line delete
#ESC S           send message unprotected only
#ESC T           erase line to insert char
#ESC U           set monitor mode   (see ESC X, ESC u)
#
#ESC V n         select video attribute mode:
#                n = 0   serial field attribute mode
#                n = 1   parallel character attribute mode
#ESC V 2 n       define line attribute:
#                n = 0   single width single height
#                n = 1   single width double height
#                n = 2   double width single height
#                n = 3   double width double height
#ESC V 3 n       select character font:
#                n = 0   system font
#                n = 1   user defined font
#ESC V 4 n       select screen mode:
#                n = 0   page screen mode
#                n = 1   virtual screen mode
#ESC V 5 n       control mouse mode:
#                n = 0   disable mouse
#                n = 1   enable sample mode
#                n = 2   send mouse information
#                n = 3   enable request mode
#ESC W           character delete
#ESC X           clear monitor mode (see ESC U, ESC u)
#ESC Y           erase page to insert char
#
#ESC Z n         send user/status line:
#                n = 0   send user line
#                n = 1   send status line
#                n = 2   send terminal ID
#ESC [ p1 p2 p3  set character attribute (parallel char mode):
#                p1: 0 = normal
#                    1 = blank
#                    2 = blink
#                    3 = blink blank (= blank)
#                    4 = reverse
#                    5 = reverse blank
#                    6 = reverse blink
#                    7 = reverse blink blank (= reverse blank)
#                    8 = underline
#                    9 = underline blank
#                    : = underline blink
#                    ; = underline blink blank
#                    < = reverse underline
#                    = = reverse underline blank
#                    > = reverse underline blink
#                    ? = reverse underline blink blank
#                p2, p3: f/grnd, b/grnd colour
#                (see ESC F for colours)
#                use ZZ for mono, eg.
#                    ESC [ 0 Z Z for normal
#                    ESC [ 4 Z Z for inverse etc.
#
#ESC \ n         set page size:
#                n = 1   24 lines/page
#                n = 2   48 lines/page
#                n = 3   72 lines/page
#                n = 4   96 lines/page
#ESC ] n         set Wordstar mode:
#                n = 0   normal (KDS7372) mode
#                n = 1   Wordstar mode
#
#ESC b           set foreground colour screen
#
#ESC c n         enter self-test mode:
#                n = 0   exit self test mode
#                n = 1   ROM test
#                n = 2   RAM test
#                n = 3   NVRAM test
#                n = 4   screen display test
#                n = 5   main/printer port test
#                n = 6   mouse port test
#                n = 7   graphics board test
#                n = 8   graphics memory test
#                n = 9   display all 'E'
#                n = :   display all 'H'
#ESC d           set background colour screen
#
#ESC e n         program insert char (n = insert char)
#ESC f text CR   load user status line with 'text'
#
#ESC g           display user status line on 25th line
#ESC h           display system status line on 25th line
#ESC i           tab
#ESC j           reverse linefeed
#ESC k n         duplex/local edit mode:
#                n = 0   duplex edit mode
#                n = 1   local edit mode
#ESC l n         select virtual screen:
#                n = 0   screen 1
#                n = 1   screen 2
#ESC m           save current config to NVRAM
#ESC n p1        select display screen:
#                p1 = 0  screen 1
#                p1 = 1  screen 2
#                p1 = 2  screen 3
#                p1 = 3  screen 4
#ESC o p1 p2     set characters/line and attribute:
#                p1 = 0  80 chars/line
#
#ESC o p1 p2     set characters/line and attribute:
#                p1 = 0  80 chars/line
#                p1 = 1  132 chars/line
#                p2 = 0  single width single height
#                p2 = 1  single width double height
#                p2 = 2  double width single height
#                p2 = 3  double width double height
#
#ESC q           insert mode on
#ESC r           edit mode on
#ESC s           send message all
#ESC t           erase line to null
#ESC u           clear monitor mode (see ESC U, ESC X)
#ESC v           autopage mode on
#ESC w           autopage mode off
#ESC x p1 p2 p3  define delimiter code...
#ESC y           erase page to null
#
#ESC z 2 p1 p2 p3 p4   draw quadrangle:
#                      p1 = starting row
#                      p2 = starting column
#                      p3 = end row
#                      p4 = end column
#
#ESC { p1 p2 p3 p4     configure main port
#                      (baud, stop bits, parity, word length)
#
#ESC | p1 p2 text Ctrl-Y    program function key with 'text':
#                        p1 = function key code:
#                             '1' - ';'  normal f1- f11
#                             '<' - 'F'  shifted f1 - f11
#                        p2 = program mode:
#                             1 = FDX
#                             2 = LOC
#                             3 = HDX
#                        Ctrl-Y = terminator
#                        (use Ctrl-P to escape ^P, ^Y )
#
#ESC } p1 p2 p3 p4     configure printer port
#                      (baud, stop bits, parity, word length)
#ESC ~           send system status
#
# Codes and info from Peter Disdale <pete@pdlmail.demon.co.uk> 12 May 1997
#
# Entry is by esr going solely on above information and is UNTESTED.
# This actually looks a lot like a Televideo 9xx.
# This entry uses page 0 and is monochrome; I'm not brave enough to try
# to make color work without a test terminal.  The <am> capability is a guess.
# The initialization string sets conversation mode, blinking underline cursor,
# full duplex, parallel attribute mode, display user status line, white
# foreground, black background, normal highlight.
#
icl6404|kds7372|icl6402|kds6402|ICL 6404 aka Kokusai Display Systems 7372,
	OTbs, am, hs,
	cols#80, lines#24,
	bel=^G, blink=\E[2ZZ, cbt=\EI, civis=\E.0, clear=\E*,
	cnorm=\E.3, cr=^M,
	csr=\E!%+%p1%{32}%+%p2%{32} cud1=\026, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{80}%m%{32}%+%c%p2%{80}%>%{32}%+%c,
	cuu1=^K, cvvis=\E.1, dch1=\EW, dl1=\ER, home=^^, ht=^I,
	hts=\E1, il1=\EE, invis=\E[1ZZ,
	is1=\EC\E.3\EDF\EV1\Eg\E[0ZZ, nel=^_, rev=\E[4ZZ,
	rmir=\Er, rmso=\E[%gh%{4}%^%Ph%gh%dZZ,
	rmul=\E[%gh%{8}%^%Ph%gh%dZZ, rs2=\Eo1,
	sgr=\E[%'0'%?%p1%t%'8'%|%;%?%p2%t%'8'%|%;%?%p3%t%'4'%|%;%?%p4%t%'2'%|%;%?%p7%t%'1'%|%;%cZZ,
	sgr0=\E[0ZZ, smir=\Eq, smso=\E[8ZZ, smul=\E[8ZZ, tbc=\E3,
icl6404-w|kds7372-w|ICL 6404 aka Kokusai Display Systems 7372 132 cols,
	rs2=\Eo1, use=icl6404,

#### Interactive Systems Corp
#
# ISC used to sell OEMed and customized hardware to support ISC UNIX.
# ISC UNIX still exists in 1995, but ISC itself is no more; they got
# bought out by Sun.
#

# From: <cithep!eric>  Wed Sep 16 08:06:44 1981
# (intext: removed obsolete ":ma=^K^P^R^L^L ::bc=^_:", also the
# ":le=^_:" later overridden -- esr)
intext|Interactive Systems Corporation modified owl 1200,
	OTbs, am,
	cols#80, it#8, lines#24, xmc#1,
	bel=^G, cbt=^Y, clear=\014$<132>, cr=^M, cub1=^H, cud1=^J,
	cuf1=^^, cup=\017%p1%{32}%+%c%p2%{32}%+%c, cuu1=^\,
	dch1=\022$<5.5*>, dl1=\021$<5.5*>, ed=\026J$<5.5*>,
	el=^Kp^R, ht=^I, il1=\020$<5.5*>, ind=^J, ip=$<5.5*>, kbs=^H,
	kcub1=^_, kcud1=^J, kcuf1=^^, kcuu1=^\, kf0=^VJ\r, kf1=^VA\r,
	kf2=^VB\r, kf3=^VC\r, kf4=^VD\r, kf5=^VE\r, kf6=^VF\r,
	kf7=^VG\r, kf8=^VH\r, kf9=^VI\r, khome=^Z, rmir=^V<,
	rmkx=^V9, rmso=^V#\s, smir=^V;, smkx=\036\:\264\026%%,
	smso=^V$\,,
intext2|intextii|INTERACTIVE modified owl 1251,
	am, bw, ul,
	cols#80, lines#24, xmc#0,
	bel=^G, cbt=\E[Z, clear=\E[H\E[2J, cr=^M, cub1=\E[D,
	cud1=\E[B, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A,
	dch1=\E[P, dl1=\E[M, ed=\E[J, el=\E[K,
	flash=\E[;;;;;;;;;2;;u$<200/>\E[;;;;;;;;;1;;u,
	hpa=\E[%p1%{1}%+%dG, ht=^I, ich1=\E[@, il1=\E[L, ind=\E[S,
	kbs=^H, kcub1=\ED\r, kcud1=\EB\r, kcuf1=\EC\r, kcuu1=\EA\r,
	kf0=\E@\r, kf1=\EP\r, kf2=\EQ\r, kf3=\ES\r, kf4=\ET\r,
	kf5=\EU\r, kf6=\EV\r, kf7=\EW\r, kf8=\EX\r, kf9=\EY\r,
	khome=\ER\r, lf0=REFRSH, lf1=DEL CH, lf2=TABSET, lf3=GOTO,
	lf4=+PAGE, lf5=+SRCH, lf6=-PAGE, lf7=-SRCH, lf8=LEFT,
	lf9=RIGHT, ri=\E[T, rmso=\E[2 D, rmul=\E[2 D, smso=\E[6 D,
	smul=\E[18 D,

#### Kimtron (abm, kt)
#
# Kimtron seems to be history, but as March 1998 these people are still
# offering repair services for Kimtron equipment:
#
#    Com/Pair Monitor Service
#    1105 N. Cliff Ave.
#    Sioux Falls, South Dakota 57103
#
#    WATS voice:  1-800/398-4946
#    POTS   fax: +1 605/338-8709
#    POTS voice: +1 605/338-9650
#         Email: <compair@sd.cybernex.net>
#  Internet/Web: <http://www.com-pair.com>
#
# Kimtron entries include (undocumented) codes for: enter dim mode,
# enter bold mode, enter reverse mode, turn off all attributes.
#

# Kimtron ABM 85 added by Dual Systems
# (abm85: removed duplicated ":kd=^J:" -- esr)
abm85|Kimtron ABM 85,
	OTbs, am, bw, msgr,
	cols#80, it#8, lines#24, xmc#1,
	cbt=\EI, clear=\E*, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\Ey, el=\Et, ht=^I,
	if=/usr/share/tabset/stdcrt, il1=\EE,
	is2=\EC\EX\Eg\En\E%\Er\E(\Ek\Em\Eq, kbs=^H, kcub1=^H,
	kcud1=^J, kcuf1=^L, kcuu1=^K, khome=^^, rmir=\Er, rmso=\Ek,
	rmul=\Em, smir=\EQ, smso=\Ej, smul=\El,
# Kimtron ABM 85H added by Dual Systems.
# Some notes about the abm85h entries:
# 1) there are several firmware revs of 85H in the world. Use abm85h-old for
#    firmware revs prior to SP51
# 2) Make sure to use abm85h entry if the terminal is in 85h mode and the
#    abm85e entry if it is in tvi920 emulation mode. They are incompatible
#    in some places and NOT software settable i.e., <is2> can't fix it)
# 3) In 85h mode, the arrow keys and special functions transmit when
#    the terminal is in dup-edit, and work only locally in local-edit.
#    Vi won't swallow `del char' for instance, but <smcup> turns on
#    dup-edit anyway so that the arrow keys will work right. If the
#    arrow keys don't work the way you like, change <smcup>, <rmcup>, and
#    <is2>.  Note that 920E mode does not have software commands to toggle
#    between dup and local edit, so you get whatever was set last on the
#    terminal.
# 4) <flash> attribute is nice, but seems too slow to work correctly
#    (\Eb<pad>\Ed)
# 5) Make sure `hidden' attributes are selected. If `embedded' attributes
#    are selected, the <xmc@> entry should be removed.
# 6) auto new-line should be on (selectable from setup mode only)
#
# From: Erik Fair <fair@ucbarpa>  Sun Oct 27 07:21:05 1985
abm85h|Kimtron ABM 85H native mode,
	hs,
	xmc@,
	bel=^G, cnorm=\E.4, cvvis=\E.2, dim=\E), dsl=\Ee, flash@,
	fsl=^M, invis@,
	is2=\EC\EN\EX\024\016\EA\Ea\E%\E9\Ee\Er\En\E"\E}\E'\E(\Ef\r\EG0\Ed\E.4\El,
	kcud1=^V, sgr0=\E(\EG0, smir=\EZ, tsl=\Eg\Ef, use=adm+sgr,
	use=abm85,
abm85e|Kimtron ABM 85H in 920E mode,
	xmc@,
	bel=^G, dim=\E), flash@,
	is2=\EC\EX\EA\E%\E9\Ee\Er\En\E"\E}\E'\E(\Ef\r\Ek\Eq\Em,
	rev=\Ej, sgr0=\E(\Ek, smir=\EZ, use=abm85,
abm85h-old|oabm85h|o85h|Kimtron ABM 85H with old firmware rev.,
	xmc@,
	bel=^G, dim=\E),
	is2=\E}\EC\EX\Ee\En\E%\Er\E(\Ek\Em\Eq\Ed\ET\EC\E9\EF,
	rev=\Ej, sgr0=\E(\Ek, smir=\EZ, use=abm85,
# From: <malman@bbn-vax.arpa>
# (kt7: removed obsolete :ma=^V^J^L :" -- esr)
kt7|kimtron model kt-7,
	OTbs, am,
	cols#80, it#8, lines#24,
	cbt=\EI, clear=^Z, cub1=^H, cud1=^V, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\EY, el=\ET, fsl=\Eg, home=^^, ht=^I, ich1=\EQ,
	if=/usr/share/tabset/stdcrt, il1=\EE, invis@, is2=\El\E",
	kbs=^H, kcbt=\EI, kclr=^Z, kcub1=^H, kcud1=^V, kcuf1=^L,
	kcuu1=^K, kdch1=\EW, kdl1=\ER, ked=\EY, kel=\ET, kf0=^AI\r,
	kf1=^A@\r, kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r,
	kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^,
	kich1=\EQ, kil1=\EE, tsl=\Ef, use=adm+sgr,
# Renamed TB=^I to :ta:, BE=^G to :bl:, BS=^H to :kb:, N to :kS: (based on the
# other kt7 entry and the adjacent key capabilities).  Removed EE which is
# identical to :mh:.  Removed :ES=\EGD: which is some kind of highlight
# but we can't figure out what.
kt7ix|kimtron model kt-7 or 70 in IX mode,
	am, bw,
	cols#80, it#8, lines#25,
	acsc=jYk?lZm@nEqDt4uCvAwBx3, bel=^G, blink=\EG2, cbt=\EI,
	civis=\E.0, clear=\E*, cnorm=\E.3, cr=^M, cub1=^H, cud1=^V,
	cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	dch1=\EW, dim=\EG@, dl1=\ER, dsl=\Ef\r, ed=\EY, el=\ET, fsl=^M,
	home=^^, ht=^I, ich1=\EQ, il1=\EE, ind=^J,
	is2=\EG0\E s\017\E~, kbs=^H, kcbt=\EI, kclr=\E*,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdl1=\ER,
	ked=\EY, kel=\ET, kend=\EY, kf0=^AI\r, kf1=^A@\r, kf2=^AA\r,
	kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r,
	kf8=^AG\r, kf9=^AH\r, khome=^^, kich1=\EQ, kil1=\EE, knp=\EJ,
	nel=^M^J, pulse=\EK, rmacs=\E%%, rmir=, rmso=\EG0, rmul=\EG0,
	sgr0=\EG0, smacs=\E$, smir=, smso=\EG4, smul=\EG8, tsl=\Ef,

#### Microdata/MDIS
#
# This was a line of terminals made by McDonnell-Douglas Information Systems.
# These entries come direct from MDIS documentation.  I have edited them only
# to move primary names of the form p[0-9] * to aliases, and to comment out
# <rmacs>/<smacs> in a couple of entries without <acsc> strings.  I have
# also removed the change history; the last version indicates this is
# version 4.3 by A.Barkus, September 1990 (earliest entry is October 1989).
#

# McDonnell Information Systems Terminal Family History
# =========================================
#
# Prism-1, Prism-2 and P99:
#       Ancient Microdata and CMC terminals, vaguely like Adds Regent 25.
#
# Prism-4 and Prism-5:
#       Slightly less ancient range of Microdata terminals. Follow-on from
#       Prism-2, but with many enhancements. P5 has eight display pages.
#
# Prism-6:
#       A special terminal for use with library systems, primarily in Germany.
#       Limited numbers. Similar functionality to P5 (except attributes?).
#
# Prism-7, Prism-8 and Prism-9:
#       More recent range of MDIS terminals, in which P7 and P8
#       replace the P4 & P5, with added functionality, and P9 is the flagship.
#       The P9 has two emulation modes - P8 and ANSI - and includes a
#       large number of the DEC VT220 control sequences. Both
#       P8 and P9 support 80c/24ln/8pg and 132cl/24li/4pg formats.
#
# Prism-12 and Prism-14:
#       Latest range, functionally very similar to the P9.  The P14 has a
#       black-on-white overscanning screen.
#
# The terminfo definitions given here are:
#
# p2      - Prism-2 (or Prism-1 or P99).
#
# p4      - Prism-4 (and older P7s & P8s).
# p5      - Prism-5 (or Prism-6).
#
# p7      - Prism-7.
# p8      - Prism-8 (in national or multinational mode).
# p8-w    - 132 column version of p8.
# p9      - Prism-9 in ANSI mode.
# p9-w    - 132 column version of p9.
# p9-8    - Prism-9 in Prism-8 emulation mode.
# p9-8-w  - As p9-8, but with 132 columns.
#
# p12     - Prism-12 in ANSI mode.
# p12-w   - 132 column version of p12.
# p12-m   - Prism-12 in MDC emulation mode.
# p12-m-w - As p12-m, but with 132 columns.
# p14     - Prism-14 in ANSI mode.
# p14-w   - 132 column version of p14.
# p14-m   - Prism-14 in MDC emulation mode.
# p14-m-w - As p14-m, but with 132 columns.
#
# p2: Prism-2
# -----------
#
# Includes Prism-1 and basic P99 without SP or MP loaded.
# The simplest form of Prism-type terminal.
# Basic cursor movement and clearing operations only.
# No video attributes.
# Notes:
#  Horizontal cursor qualifiers of NUL, XON and XOFF are mapped to the next
#  value up, followed by backspace.
#
prism2|MDC Prism-2,
	am, bw, msgr,
	cols#80, lines#24,
	bel=^G, clear=\014$<20>, cr=^M, cub1=^H, cud1=^J, cuf1=^F,
	cup=\013%p1%{32}%+%c\020%p2%{10}%/%{16}%*%p2%{10}%m%+%Pc%?%{17}%gc%=%{19}%gc%=%|%gc%!%|%t%{1}%gc%+%c%{8}%e%gc%;%c,
	cuu1=^Z, ed=\EJ, el=\EK, home=^A,
	hpa=\020%p1%{10}%/%{16}%*%p1%{10}%m%+%Pc%?%{17}%gc%=%{19}%gc%=%|%gc%!%|%t%{1}%gc%+%c%{8}%e%gc%;%c,
	ind=^J, kbs=^H, khome=^A, vpa=\013%p1%{32}%+%c,

# p4: Prism-4
# -----------
#
# Includes early versions of P7 & P8.
# Basic family definition for most Prisms (except P2 and P9 ANSI).
# Notes:
#  Horizontal cursor qualifiers of NUL, XON and XOFF are mapped to the next
#  value up, followed by backspace.
#  Cursor key definitions removed because they interfere with vi and csh keys.
#
prism4|p4|P4|MDC Prism-4,
	am, bw, hs, mc5i, msgr,
	cols#80, lines#24, wsl#72, xmc#1,
	bel=^G, blink=^CB, civis=^]\344, clear=\014$<20>,
	cnorm=^]\342, cr=^M, cub1=^H, cud1=^J, cuf1=^F,
	cup=\013%p1%{32}%+%c\020%p2%{10}%/%{16}%*%p2%{10}%m%+%Pc%?%{17}%gc%=%{19}%gc%=%|%gc%!%|%t%{1}%gc%+%c%{8}%e%gc%;%c,
	cuu1=^Z, dim=^CA, dsl=\035\343\035\345, ed=\EJ, el=\EK,
	fsl=^]\345, home=^A,
	hpa=\020%p1%{10}%/%{16}%*%p1%{10}%m%+%Pc%?%{17}%gc%=%{19}%gc%=%|%gc%!%|%t%{1}%gc%+%c%{8}%e%gc%;%c,
	ind=^J, invis=^CH, kbs=^H, khome=^A, mc0=\EU, mc4=\ET, mc5=\ER,
	rev=^CD, rmso=^C\s, rmul=^C\s,
	sgr=\003%{64}%?%p1%p3%|%t%{4}%+%;%?%p2%t%{16}%+%;%?%p4%t%{2}%+%;%?%p5%t%{1}%+%;%?%p7%t%{8}%+%;%c%?%p9%t\016%e\017%;,
	sgr0=^C\s, smso=^CD, smul=^CP, tsl=^]\343,
	vpa=\013%p1%{32}%+%c,

# p5: Prism-5
# -----------
#
# Same definition as p4. Includes Prism-6 (not tested!).
# Does not use any multi-page features.
#
prism5|p5|P5|MDC Prism-5,
	use=p4,

# p7: Prism-7
# -----------
#
# Similar definition to p4. Uses ANSI cursor motion to avoid network problems.
# Notes:
#  Use p4 for very early models of P7.
#  Rev-index removed; can't send nulls to terminal in 8-bit modes.
#
prism7|p7|P7|MDC Prism-7,
	cup=\E[%i%p1%d;%p2%dH, hpa@, vpa@, use=p4,

# p8: Prism-8
# -----------
#
# Similar definition to p7. Uses ANSI cursor motion to avoid network problems.
# Supports national and multinational character sets.
# Notes:
#  Alternate char set operations only work in multinational mode.
#  Use p4 for very early models of P8.
#  Rev-index removed; can't send nulls to terminal in 8-bit modes.
# (esr: commented out <smacs>/<rmacs> because there's no <acsc>)
#
prism8|p8|P8|MDC Prism-8,
	cup=\E[%i%p1%d;%p2%dH, hpa=\E[%i%p1%d`, is2=\E[<12h,
	vpa=\E[%i%p1%dd, use=p4,

# p8-w: Prism-8 in 132 column mode
# --------------------------------
#
# 'Wide' version of p8.
# Notes:
#  Rev-index removed; can't send nulls to terminal in 8-bit modes.
#
prism8-w|p8-w|P8-W|MDC Prism-8 in 132 column mode,
	cols#132,
	is2=\E[<12h\E[<14h, use=p8,

# p9: Prism-9 in ANSI mode
# -------------------------
#
# The "flagship" model of this generation of terminals.
# ANSI X3.64 (ISO 6429) standard sequences, plus many DEC VT220 ones.
# Notes:
#  Tabs only reset by "reset". Otherwise assumes default (8 cols).
#  Fixes to deal with terminal firmware bugs:
#  . 'ri' uses insert-line since rev index doesn't always
#  . 'sgr0' has extra '0' since esc[m fails
#  . 'fsl' & 'dsl' use illegal char since cr is actioned wrong on line 25
#  Not covered in the current definition:
#  . Labels
#  . Programming Fn keys
#  . Graphic characters (defaults correctly to vt100)
#  . Padding values (sets xon)
# (esr: commented out <smacs>/<rmacs> because there's no <acsc>)
#
prism9|p9|P9|MDC Prism-9 in ANSII mode,
	am, bw, hs, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3, wsl#72,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[<4l,
	clear=^L, cnorm=\E[<4h, cr=^M, csr=\E[%i%p1%d;%p2%d%%v,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dl=\E[%p1%dM, dl1=\E[M, dsl=\E[%}\024, ech=\E[%p1%dX,
	ed=\E[J$<10>, el=\E[K, fsl=^T, home=\E[H, hpa=\E[%i%p1%d`,
	ht=^I, hts=\EH, il=\E[%p1%dL, il1=\E[L, ind=^J,
	is2=\E[&p\E[<12l\E F, kbs=^H, kclr=^L, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf1=\E[11~,
	kf10=\E[21~, kf11=\E[23~, kf12=\E[24~, kf13=\E[25~,
	kf14=\E[26~, kf15=\E[28~, kf16=\E[29~, kf17=\E[31~,
	kf18=\E[32~, kf2=\E[12~, kf3=\E[13~, kf4=\E[14~,
	kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	khome=\E[H, nel=^M^J, prot=\E[32%{, rc=\E[%z,
	rep=\E[%p2%db%p1%c, rev=\E[7m, ri=\E[L, rmir=\E[4l,
	rmso=\E[27m, rmul=\E[24m,
	rs2=\E[&p\E[<12l\E F\E[3g\E[9;17;25;33;41;49;57;65;73 N,
	sc=\E[%y,
	sgr=\E[0%?%p1%p3%|%t;7%;%?%p2%t;4%;%?%p4%t;5%;%?%p6%t;1%;m%?%p8%t\E[32%%{%;%?%p9%t\016%e\017%;,
	sgr0=\E[0m\017, smir=\E[4h, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g, tsl=\E[%i%p1%d%%}, vpa=\E[%i%p1%dd,
	use=ansi+pp,

# p9-w: Prism-9 in 132 column mode
# --------------------------------
#
# 'Wide' version of p9.
#
prism9-w|p9-w|P9-W|MDC Prism-9 in 132 column mode,
	cols#132,
	is2=\E[&p\E[<12l\E F\E[<14h,
	rs2=\E[&p\E[<12l\E F\E[<14h, use=p9,

# p9-8: Prism-9 in P8 mode
# ------------------------
#
# P9 terminal in P8 emulation mode.
# Similar to p8 definition.
# Insertion and deletion operations possible.
#
prism9-8|p9-8|P9-8|MDC Prism-9 in P8 mode,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[L, use=p8,

# p9-8-w: Prism-9 in P8 and 132 column modes
# ------------------------------------------
#
# P9 terminal in P8 emulation mode and 132 column mode.
#
prism9-8-w|p9-8-w|P9-8-W|MDC Prism-9 in Prism 8 emulation and 132 column mode,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[L, use=p8-w,

# p12: Prism-12 in ANSI mode
# ---------------------------
#
# See p9 definition.
#
prism12|p12|P12|MDC Prism-12 in ANSI mode,
	use=p9,

# p12-w: Prism-12 in 132 column mode
# ----------------------------------
#
# 'Wide' version of p12.
#
prism12-w|p12-w|P12-W|MDC Prism-12 in 132 column mode,
	use=p9-w,

# p12-m: Prism-12 in MDC emulation mode
# -------------------------------------
#
# P12 terminal in MDC emulation mode.
# Similar to p8 definition.
# Insertion and deletion operations possible.
#
prism12-m|p12-m|P12-M|MDC Prism-12 in MDC emulation mode,
	use=p9-8,

# p12-m-w: Prism-12 in MDC emulation and 132 column modes
# -------------------------------------------------------
#
# P12 terminal in MDC emulation mode and 132 column mode.
#
prism12-m-w|p12-m-w|P12-M-W|MDC Prism-12 in MDC emulation and 132 column mode,
	use=p9-8-w,

# p14: Prism-14 in ANSII mode
# ---------------------------
#
# See p9 definition.
#
prism14|p14|P14|MDC Prism-14 in ANSII mode,
	use=p9,

# p14-w: Prism-14 in 132 column mode
# ----------------------------------
#
# 'Wide' version of p14.
#
prism14-w|p14-w|P14-W|MDC Prism-14 in 132 column mode,
	use=p9-w,

# p14-m: Prism-14 in MDC emulation mode
# -------------------------------------
#
# P14 terminal in MDC emulation mode.
# Similar to p8 definition.
# Insertion and deletion operations possible.
#
prism14-m|p14-m|P14-M|MDC Prism-14 in MDC emulation mode,
	use=p9-8,

# p14-m-w: Prism-14 in MDC emulation and 132 column modes
# -------------------------------------------------------
#
# P14 terminal in MDC emulation mode and 132 column mode.
#
prism14-m-w|p14-m-w|P14-M-W|MDC Prism-14 in MDC emulation and 132 column mode,
	use=p9-8-w,

# End of McDonnell Information Systems Prism definitions

# These things were popular in the Pick database community at one time
# From: George Land <georgeland@aol.com> 24 Sep 1996
p8gl|prism8gl|McDonnell-Douglas Prism-8 alternate definition,
	am, bw, hs, mir,
	cols#80, lines#24, ma#1, wsl#78, xmc#1,
	bel=^G, blink=^CB, clear=^L, cr=^M, cub1=^U, cud1=^J, cuf1=^F,
	cup=\E[%i%p1%d;%p2%dH, cuu1=^Z, dch1=\s^H, dim=^CA, dl1=^P,
	ed=\EJ, el=\EK, home=^A, ind=^J, invis=^CH, kbs=^H, kcub1=^U,
	kcud1=^J, kcuf1=^F, kcuu1=^Z, kdch1=\s^H, kdl1=^P, ked=\EJ,
	kel=\EK, kf1=^A@\r, kf10=^AI\r, kf12=^AJ\r, kf13=^AK\r,
	kf14=^AL\r, kf15=^AM\r, kf16=^AN\r, kf17=^AO\r, kf2=^AA\r,
	kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r,
	kf8=^AG\r, kf9=^AH\r, khome=^A, lf1=F1, lf10=F10, lf2=F2,
	lf3=F3, lf4=F4, lf5=F5, lf6=F6, lf7=F7, lf8=F8, lf9=F9, nel=^J^M,
	pad=\0, rev=^CD, rmso=^C\s, rmul=^C\s, sgr0=^C\s, smso=^CE,
	smul=^C0,

#### Microterm (act, mime)
#
# The mime1 entries refer to the Microterm Mime I or Mime II.
# The default mime is assumed to be in enhanced act iv mode.
#

# New "safe" cursor movement (5/87) from <reuss@umd5.umd.edu>.  Prevents
# freakout with out-of-range args on Sytek multiplexors.  No <smso=^N> and
# <rmso=^N> since  it gets confused and it's too dim anyway.  No <ich1>
# since Sytek insists ^S means xoff.
# (act4: found ":ic=2^S:ei=:im=:ip=.1*^V:" commented out in 8.3 -- esr)
act4|microterm|microterm act iv,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=\014$<12/>, cr=^M, cub1=^H, cud1=^K, cuf1=^X,
	cup=\024%p1%{24}%+%c%p2%p2%?%{47}%>%t%{48}%+%;%{80}%+%c,
	cuu1=^Z, dch1=\004$<.1*/>, dl1=\027$<2.3*/>,
	ed=\037$<2.2*/>, el=\036$<.1*/>, home=^],
	il1=\001<2.3*/>, ind=^J, kcub1=^H, kcud1=^K, kcuf1=^X,
	kcuu1=^Z,
# The padding on :sr: and :ta: for act5 and mime is a guess and not final.
# The act 5 has hardware tabs, but they are in columns 8, 16, 24, 32, 41 (!)...
# (microterm5: removed obsolete ":ma==^Z^P^Xl^Kj:" -- esr)
act5|microterm5|microterm act v,
	kcub1=^H, kcud1=^K, kcuf1=^X, kcuu1=^Z, ri=\EH$<3>, uc=^H\EA,
	use=act4,
# Mimes using brightness for standout.  Half bright is really dim unless
# you turn up the brightness so far that lines show up on the screen.
mime-fb|full bright mime1,
	is2=^S\E, rmso=^S, smso=^Y, use=mime,
mime-hb|half bright mime1,
	is2=^Y\E, rmso=^Y, smso=^S, use=mime,
# (mime: removed obsolete ":ma=^X ^K^J^Z^P:"; removed ":do=^K:" that overrode
# the more plausible ":do=^J:" -- esr)
# uc was at one time disabled to get around a curses bug, be wary of it
mime|mime1|mime2|mimei|mimeii|microterm mime1,
	OTbs, am,
	cols#80, it#8, lines#24, vt#9,
	bel=^G, clear=^]^C, cr=^M, cub1=^H, cud1=^J, cuf1=^X,
	cup=\024%p1%{24}%+%c%p2%p2%?%{32}%>%t%{48}%+%;%{80}%+%c,
	cuu1=^Z, dl1=\027$<80>, ed=^_, el=^^, home=^], ht=\011$<2>,
	il1=\001$<80>, ind=^J, is2=^S\E^Q, kcub1=^H, kcud1=^K,
	kcuf1=^X, kcuu1=^Z, ri=\022$<3>, uc=^U,
# These termcaps (for mime2a) put the terminal in low intensity mode
# since high intensity mode is so obnoxious.
mime2a-s|microterm mime2a (emulating an enhanced soroc iq120),
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=\EL, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EI, dch1=\ED,
	dl1=\027$<20*>, ed=\EJ$<20*>, el=\EK, home=^^,
	il1=\001$<20*>, ind=^J, ip=$<2>, is2=\E), kcub1=^H, kcud1=^J,
	kcuf1=^L, kcuu1=^K, ri=\EI, rmir=^Z, rmso=\E;, rmul=\E7,
	smir=\EE, smso=\E\:, smul=\E6,
# This is the preferred mode (but ^X can't be used as a kill character)
mime2a|mime2a-v|microterm mime2a (emulating an enhanced vt52),
	OTbs,
	cols#80, it#8, lines#24,
	bel=^G, clear=\EL, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dch1=^N,
	dl1=\027$<20*>, ed=\EQ$<20*>, el=\EP, home=\EH, ht=^I,
	il1=\001$<20*>, ind=^J, ip=$<2>, is2=^Y, kcub1=\ED,
	kcud1=\EB, kcuf1=\EC, kcuu1=\EA, ri=\EA, rmir=^Z, rmso=\E9,
	rmul=\E5, smir=^O, smso=\E8, smul=\E4,
# (mime3a: removed obsolete ":ma=^X ^K^J^Z^P:" -- esr)
mime3a|mime1 emulating 3a,
	am@,
	kcub1=^H, kcud1=^K, kcuf1=^X, kcuu1=^Z, use=adm3a,
mime3ax|mime-3ax|mime1 emulating enhanced 3a,
	it#8,
	dl1=\027$<80>, ed=^_, el=^X, ht=\011$<3>, il1=\001$<80>,
	use=mime3a,
# Wed Mar  9 18:53:21 1983
# We run our terminals at 2400 baud, so there might be some timing problems at
# higher speeds. The major improvements in this model are the terminal now
# scrolls down and insert mode works without redrawing the rest of the line
# to the right of the cursor. This is done with a bit of a kludge using the
# exit graphics mode to get out of insert, but it does not appear to hurt
# anything when using vi at least. If you have some users using act4s with
# programs that use curses and graphics mode this could be a problem.
mime314|mm314|mime 314,
	am,
	cols#80, lines#24,
	clear=^L, cub1=^H, cuf1=^X, cup=\024%p1%c%p2%c, cuu1=^Z,
	dch1=^D, dl1=^W, ed=^_, el=^^, home=^], ht=^I, il1=^A, kcub1=^H,
	kcud1=^K, kcuf1=^X, kcuu1=^Z, rmir=^V, smir=^S,
# Microterm mime 340 from University of Wisconsin
mm340|mime340|mime 340,
	cols#80, lines#24,
	clear=\032$<12/>, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	dch1=\E#$<2.1*/>, dl1=\EV$<49.6/>, ed=\037$<2*/>,
	el=\EL$<2.1/>, ht=^I, il1=\EU$<46/>, ind=^J, is2=\E\,,
	kbs=^H, kcub1=^H, kcud1=^J, kcuu1=^K, nel=^M^J,
# This came from University of Wisconsin marked "astro termcap for jooss".
# (mt4520-rv: removed obsolete ":kn#4:" and incorrect ":ri=\E[C:";
# also added <rmam>/<smam> based  on the init string -- esr)
mt4520-rv|micro-term 4520 reverse video,
	am, hs, msgr, xenl, xon,
	cols#80, it#8, lines#24, wsl#80,
	bel=^G, clear=\E[H\E[J, cnorm=\E[0V\E8, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=\E[D,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E7\E[0U, dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el=\E[K, flash=\E[?5l$<200/>\E[?5h,
	fsl=\E[?5l\E[?5h, home=\E[H, ht=^I, hts=\EH, ich=\E[%p1%d@,
	ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=\ED,
	is2=\E(B\E[2l\E>\E[20l\E[?3l\E[?5h\E[?7h\E[1;24r\E[24;1H\E[H\E[J,
	kbs=^H, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, khome=\E[H,
	ll=\E[24;1H, nel=\EE, rc=\E8, rf=/usr/share/tabset/vt100,
	ri=\EM, rmam=\E[?7l, rmso=\E[0m, rmul=\E[24m,
	rs1=\E(B\E[2l\E>\E[20l\E[?3l\E[?5h\E[?7h\E[H\E[J,
	sc=\E7, sgr0=\E[m, smam=\E[?7h, smso=\E[7m, smul=\E[4m,
	tbc=\E[g, tsl=\E[25;1H,

# Fri Aug  5 08:11:57 1983
# This entry works for the ergo 4000 with the following setups:
# ansi,wraparound,newline disabled, xon/xoff disabled in both
# setup a & c.
#
# WARNING!!! There are multiple versions of ERGO 4000 microcode
# Be advised that very early versions DO NOT WORK RIGHT !!
# Microterm does have a ROM exchange program- use it or lose big
# (ergo400: added <rmam>/<smam> based on the init string -- esr)
ergo4000|microterm ergo 4000,
	da, db, msgr,
	cols#80, lines#66,
	bel=^G, clear=\E[H\E[2J$<80>, cr=^M, cub1=^H, cud1=\E[B,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A,
	dch1=\E[1P$<80>, dl1=\E[1M$<5*>, ed=\E[0J$<15>,
	el=\E[0K$<13>, ht=^I, il1=\E[1L$<5*>, ind=\ED$<20*>,
	is2=\E<\E=\E[?1l\E[?4l\E[?5l\E[?7h\E[?8h$<300>,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf1=\EOP,
	kf2=\EOQ, kf3=\EOR, kf4=\EOS, lf1=pf1, lf2=pf2, lf3=pf3,
	lf4=pf4, ri=\EM$<20*>, rmam=\E[?7l, rmir=\E[4l,
	rmkx=\E=$<4>, rmso=\E[m$<20>, sgr0=\E[m$<20>,
	smam=\E[?7m, smir=\E[4h$<6>, smkx=\E=$<4>,
	smso=\E[7m$<20>,

#### NCR
#
# NCR's terminal group was merged with AT&T's when AT&T bought the company.
# For what happened to that group, see the ADDS section.
#
# There is an NCR4103 terminal that's just a re-badged Wyse-50.
#

# The following vendor-supplied termcaps were captured from the Boundless
# Technologies site, 8 March 1998.  I removed all-upper-case names that were
# identical, except for case, to lower-case ones.  I also uncommented the acsc
# capabilities.X
#
# The Intecolor emulation of the NCR 2900/260C color terminal is basically a
# DEC vt200/300 with color capabilities added.
ncr260intan|NCR Intecolor emulation of the 2900_260C with an ANSI keyboard,
	colors#8, pairs#64,
	op=\E[0m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	use=ncr260vt300an,
# The Intecolor emulation of the NCR 2900/260C color terminal is basically a
# DEC vt200/300 with color capabilities added.
ncr260intwan|NCR Intecolor emulation of the 2900_260C with an ANSI keyboard,
	colors#8, pairs#64,
	op=\E[0m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	use=ncr260vt300wan,
# The Intecolor emulation of the NCR 2900/260C color terminal is basically a
# DEC vt200/300 with color capabilities added.
ncr260intpp|NCR Intecolor emulation of the 2900_260C with a PC+ keyboard,
	colors#8, pairs#64,
	op=\E[0m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	use=ncr260vt300pp,
# The Intecolor emulation of the NCR 2900/260C color terminal is basicly a
# DEC vt200/300 with color capabilities added.
ncr260intwpp|NCR Intecolor emulation of the 2900_260C with a PC+ keyboard in 132 column mode,
	colors#8, pairs#64,
	op=\E[0m, setab=\E[4%p1%dm, setaf=\E[3%p1%dm,
	use=ncr260vt300wpp,
# This definition for ViewPoint supports several attributes.  This means
# that it has magic cookies (extra spaces where the attributes begin).
# Some applications do not function well with magic cookies.  The System
# Administrator's Shell in NCR Unix SVR4 1.03 is one such application.
# If supporting various attributes is not vital, 'xmc#1' and the extra
# attributes can be removed.
# Mapping to ASCII character set ('acsc' capability) can also be
# restored if needed.
ncr260vppp|NCR 2900_260 viewpoint,
	am, bw, hs, km, mc5i, mir, msgr, xon,
	cols#80, lines#24, nlab#32, xmc#1,
	acsc=07a?h;j5k3l2m1n8q\:t4u9v=w0x6, bel=^G, blink=\EG2,
	cbt=\EI, civis=\E`0, clear=\014$<40>, cnorm=\E`5,
	cr=\r$<2>, cub1=\010$<2>, cud1=\n$<2>, cuf1=\006$<2>,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c$<5>, cuu1=\032$<2>,
	dch1=\EW$<2>, dim=\EGp, dl1=\El$<2>, dsl=\E`c, ed=\Ek$<2>,
	el=\EK$<2>, fsl=^M, home=\036$<2>, ht=^I, hts=\E1,
	il1=\EM$<2>, ind=\n$<2>, invis=\EG1,
	is2=\Ee6\E~%$<100>\E+\E`\:\Ed/\E`1\EO\Ee4\Ec@0@\Ec@1A\EcB0\EcC1\Ee7$<100>,
	kDC=\El, kEND=\Ek, kHOM=^A, kPRT=\E7, kRIT=^F, ka1=^A, ka3=\EJ,
	kbs=^H, kc1=\ET, kc3=\EJ, kcub1=^U, kcud1=^J, kcuf1=^F,
	kcuu1=^Z, kdch1=\EW, kend=\EK, kf1=^B1\r, kf10=^B\:\r,
	kf11=^B;\r, kf12=^B<\r, kf13=^B=\r, kf14=^B>\r, kf15=^B?\r,
	kf16=^B@\r, kf17=^B!\r, kf18=^B"\r, kf19=^B#\r, kf2=^B2\r,
	kf20=^B$\r, kf21=\002%^M, kf22=^B&\r, kf23=^B'\r,
	kf24=^B(\r, kf25=^B)\r, kf26=^B*\r, kf27=^B+\r,
	kf28=^B\,\r, kf29=^B-\r, kf3=^B3\r, kf30=^B.\r, kf31=^B/\r,
	kf32=^B0\r, kf4=^B4\r, kf5=^B5\r, kf6=^B6\r, kf7=^B7\r,
	kf8=^B8\r, kf9=^B9\r, khome=^A, kich1=\Eq, knp=\EJ, kpp=\EJ,
	kprt=\EP, ll=\001$<5>, mc0=\EP$<100>, mc4=^T, mc5=^R,
	mrcup=\Ew@%p1%{48}%+%c%p2%{32}%+%c%p3%{32}%+%c$<5>,
	nel=\037$<2>, rev=\EG4, ri=\Ej$<2>, rmacs=\EcB0\EH\003,
	rmir=\Er, rmso=\EG0, rmul=\EG0, rmxon=\Ec20,
	rs2=\Ee6\E~%$<100>\E+\E`\:\Ed/\E`1\EO\Ee4\Ec@0@\Ec@1A\EcB0\EcC1\Ee7$<100>,
	sgr0=\EG0\EH\003, smacs=\EcB1\EH\002, smir=\Eq,
	smso=\EG4, smul=\EG8, smxon=\Ec21, tsl=\EF,
ncr260vpwpp|NCR 2900_260 viewpoint wide mode,
	cols#132,
	cup=\Ea%i%p1%dR%p2%dC$<30>,
	is2=\Ee6\E~%$<100>\E+\E`;\Ed/\E`1\EO\Ee4\Ec@0@\Ec@1A\EcB0\EcC1\Ee7$<100>,
	rs2=\Ee6\E~%$<100>\E+\E`;\Ed/\E`1\EO\Ee4\Ec@0@\Ec@1A\EcB0\EcC1\Ee7$<100>,
	use=ncr260vppp,
ncr260vt100an|NCR 2900_260 vt100 with ansi kybd,
	am, hs, mir, msgr, xenl, xon,
	cols#80, lines#24, nlab#32,
	acsc=``aaffggjjkkllmmnnooqqssttuuvvwwxx~~, bel=^G,
	blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[2J\E[1;1H$<20>, cnorm=\E[?25h, cr=\r$<1>,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD$<5>,
	cub1=\E[D$<5>, cud=\E[%p1%dB$<5>, cud1=\E[B$<5>,
	cuf=\E[%p1%dC$<5>, cuf1=\E[C$<5>,
	cup=\E[%i%p1%d;%p2%dH$<10>, cuu=\E[%p1%dA$<5>,
	cuu1=\E[A$<5>, dch=\E[%p1%dP$<5>, dch1=\E[1P$<5>,
	dl=\E[%p1%dM$<5>, dl1=\E[M$<5>, dsl=\E[0$~\E[1$~,
	ech=\E[%p1%dX, ed=\E[0J$<5>, el=\E[0K$<3>, el1=\E[1K$<3>,
	fsl=\E[0$}, home=\E[H$<1>, hpa=\E[%p1%dG$<40>, ht=^I,
	hts=\EH, ich=\E[%p1%d@$<5>, il=\E[%p1%dL$<5>,
	il1=\E[L$<5>, ind=\ED$<5>, indn=\E[%p1%dE$<5>,
	invis=\E[8m,
	is2=\E[!p\E[?7;19;67h\E[?1;3;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kdch1=\E[3~, kfnd=\E[1~, khlp=\E[28~, kich1=\E[2~,
	knp=\E[6~, kpp=\E[5~, krdo=\E[29~, kslt=\E[4~, nel=\EE$<5>,
	rc=\E8, rev=\E[7m, ri=\EM$<5>, rmacs=^O, rmir=\E[4l,
	rmkx=\E[?1l\E>, rmso=\E[0m, rmul=\E[0m,
	rs2=\E[!p\E[?7;19;67h\E[?1;3;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p7%t;8%;m%?%p9%t\016%e\017%;$<20>,
	sgr0=\E[0m\017$<20>, smacs=^N, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[1;7m, smul=\E[4m, tbc=\E[3g,
	tsl=\E[2$~\E[1$}, vpa=\E[%p1%dd$<40>, use=vt220+keypad,
ncr260vt100wan|NCR 2900_260 vt100 wide mode ansi kybd,
	cols#132,
	cup=\E[%i%p1%d;%p2%dH$<30>,
	is2=\E[!p\E[?3;7;19;67h\E[?1;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	rs2=\E[!p\E[?3;7;19;67h\E[?1;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	use=ncr260vt100an,
ncr260vt100pp|NCR 2900_260 vt100 with PC+ kybd,
	is2=\E[!p\E[?7;19;67h\E[?1;3;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	ka1=\E[H, ka3=\EOu, kb2=\E[V, kc3=\E[U, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[4~,
	kend=\E[5~, khome=\E[2~, kich1=\E[1~, knp=\E[6~, kpp=\E[3~,
	lf1=pf1, lf2=pf2, lf3=pf3, lf4=pf4, rmkx=\E>,
	rs2=\E[!p\E[?7;19;67h\E[?1;3;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	smkx=\E=, use=ncr260vt100an,
ncr260vt100wpp|NCR 2900_260 vt100 wide mode pc+  kybd,
	cols#132,
	cup=\E[%i%p1%d;%p2%dH$<30>,
	is2=\E[!p\E[?3;7;19;67h\E[?1;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	rs2=\E[!p\E[?3;7;19;67h\E[?1;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	use=ncr260vt100pp,
ncr260vt200an|NCR 2900_260 vt200 with ansi kybd,
	am, hs, mir, msgr, xenl, xon,
	cols#80, lines#24, nlab#32,
	acsc=``aaffggjjkkllmmnnooqqssttuuvvwwxx~~, bel=^G,
	blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[2J\E[1;1H$<20>, cnorm=\E[?25h, cr=\r$<1>,
	csr=\E[%i%p1%d;%p2%dr$<5>, cub=\E[%p1%dD$<5>,
	cub1=\E[D$<5>, cud=\E[%p1%dB$<5>, cud1=\E[B$<5>,
	cuf=\E[%p1%dC$<5>, cuf1=\E[C$<5>,
	cup=\E[%i%p1%d;%p2%dH$<10>, cuu=\E[%p1%dA$<5>,
	cuu1=\E[A$<5>, dch=\E[%p1%dP$<5>, dch1=\E[1P$<5>,
	dl=\E[%p1%dM$<5>, dl1=\E[M$<5>, dsl=\E[0$~\E[1$~,
	ech=\E[%p1%dX$<5>, ed=\E[0J, el=\E[0K$<5>, el1=\E[1K$<5>,
	fsl=\E[0$}, home=\E[H, hpa=\E[%p1%dG$<40>, ht=^I, hts=\EH,
	ich=\E[%p1%d@$<5>, il=\E[%p1%dL$<5>, il1=\E[L$<5>,
	ind=\ED$<5>, indn=\E[%p1%dE$<5>, invis=\E[8m,
	is2=\E[!p\E[?7;19;67h\E[?1;3;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kdch1=\E[3~, kf0=\EOy, kf10=\E[21~, kf11=\E[23~,
	kf12=\E[24~, kf13=\E[25~, kf14=\E[26~, kf15=\E[28~,
	kf16=\E[29~, kf17=\E[31~, kf18=\E[32~, kf19=\E[33~,
	kf2=\EOQ, kf20=\E[34~, kf21=\E[31~, kf22=\E[32~,
	kf23=\E[33~, kf24=\E[34~, kf25=\E[35~, kf26=\E[1~,
	kf27=\E[2~, kf28=\E[3~, kf29=\E[4~, kf3=\EOR, kf30=\E[5~,
	kf31=\E[6~, kf32=\E[7~, kf33=\E[8~, kf34=\E[9~,
	kf35=\E[10~, kf4=\EOS, kf5=\E[M, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, kfnd=\E[1~, khlp=\E[28~,
	kich1=\E[2~, knp=\E[6~, kpp=\E[5~, krdo=\E[29~, kslt=\E[4~,
	mc0=\E[i, mc4=\E[4i, mc5=\E[5i, nel=\EE, rc=\E8, rev=\E[7m,
	ri=\EM$<5>, rmacs=\017$<20>, rmam=\E[?7l, rmir=\E[4l,
	rmkx=\E[?1l\E>, rmso=\E[27m, rmul=\E[24m,
	rs2=\E[!p\E[?7;19;67h\E[?1;3;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p7%t;8%;m%?%p9%t\016%e\017%;$<20>,
	sgr0=\E[0m\017$<20>, smacs=\016$<20>, smam=\E[?7h,
	smir=\E[4h, smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g, tsl=\E[2$~\E[1$}, vpa=\E[%p1%dd$<40>,
	use=vt220+keypad,
ncr260vt200wan|NCR 2900_260 vt200 wide mode ansi kybd,
	cols#132,
	cup=\E[%i%p1%d;%p2%dH$<30>,
	is2=\E[!p\E[?3;7;19;67h\E[?4l\E(B\E)0\017\E[2J\E[1;1H$<200>,
	rs2=\E[!p\E[?3;7;19;67h\E[?4l\E(B\E)0\017\E[2J\E[1;1H$<200>,
	use=ncr260vt200an,
ncr260vt200pp|NCR 2900_260 vt200 with pc+ kybd,
	ka1=\E[H, ka3=\EOu, kb2=\E[V, kc3=\E[U, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[4~,
	kend=\E[1~, khome=\E[H, kich1=\E[2~, knp=\E[6~, kpp=\E[5~,
	lf1=pf1, lf2=pf2, lf3=pf3, lf4=pf4, rmkx=\E>, smkx=\E=,
	use=ncr260vt200an,
ncr260vt200wpp|NCR 2900_260 vt200 wide mode pc+  kybd,
	cols#132,
	cup=\E[%i%p1%d;%p2%dH$<30>,
	is2=\E[!p\E[?3;7;19;67h\E[?1;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	rs2=\E[!p\E[?3;7;19;67h\E[?1;4l\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	use=ncr260vt200pp,
ncr260vt300an|NCR 2900_260 vt300 with ansi kybd,
	am, hs, mir, msgr, xenl, xon,
	cols#80, lines#24, nlab#32,
	acsc=``aaffggjjkkllmmnnooqqssttuuvvwwxx~~, bel=^G,
	blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[2J\E[1;1H$<20>, cnorm=\E[?25h, cr=\r$<1>,
	csr=\E[%i%p1%d;%p2%dr$<5>, cub=\E[%p1%dD$<5>,
	cub1=\E[D$<5>, cud=\E[%p1%dB$<5>, cud1=\E[B$<5>,
	cuf=\E[%p1%dC$<5>, cuf1=\E[C$<5>,
	cup=\E[%i%p1%d;%p2%dH$<10>, cuu=\E[%p1%dA$<5>,
	cuu1=\E[A$<5>, dch=\E[%p1%dP$<5>, dch1=\E[1P$<5>,
	dl=\E[%p1%dM$<5>, dl1=\E[M$<5>, dsl=\E[0$~\E[1$~,
	ech=\E[%p1%dX$<5>, ed=\E[0J, el=\E[0K$<5>, el1=\E[1K$<5>,
	fsl=\E[0$}, home=\E[H, hpa=\E[%p1%dG$<40>, ht=^I, hts=\EH,
	ich=\E[%p1%d@$<5>, il=\E[%p1%dL$<5>, il1=\E[L$<5>,
	ind=\ED$<5>, indn=\E[%p1%dE$<5>, invis=\E[8m,
	is2=\E[!p\E[?7;19;67h\E[?1;3;4l\E[1;0%w\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kdch1=\E[3~, kf0=\EOy, kf10=\E[21~, kf11=\E[23~,
	kf12=\E[24~, kf13=\E[25~, kf14=\E[26~, kf15=\E[28~,
	kf16=\E[29~, kf17=\E[31~, kf18=\E[32~, kf19=\E[33~,
	kf20=\E[34~, kf21=\E[31~, kf22=\E[32~, kf23=\E[33~,
	kf24=\E[34~, kf25=\E[35~, kf26=\E[1~, kf27=\E[2~,
	kf28=\E[3~, kf29=\E[4~, kf30=\E[5~, kf31=\E[6~, kf32=\E[7~,
	kf33=\E[8~, kf34=\E[9~, kf35=\E[10~, kf5=\E[M, kf6=\E[17~,
	kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, kfnd=\E[1~,
	khlp=\E[28~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~,
	krdo=\E[29~, kslt=\E[4~, mc0=\E[i, mc4=\E[4i, mc5=\E[5i,
	nel=\EE, rc=\E8, rev=\E[7m, ri=\EM$<5>, rmacs=\017$<20>,
	rmam=\E[?7l, rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[27m,
	rmul=\E[24m,
	rs2=\E[!p\E[?7;19;67h\E[?1;3;4l\E[1;0%w\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;%?%p7%t;8%;m%?%p9%t\016%e\017%;$<20>,
	sgr0=\E[0m\017$<20>, smacs=\016$<20>, smam=\E[?7h,
	smir=\E[4h, smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g, tsl=\E[2$~\E[1$}, vpa=\E[%p1%dd$<40>,
	use=vt220+keypad,
ncr260vt300wan|NCR 2900_260 vt300 wide mode ansi kybd,
	cols#132,
	cup=\E[%i%p1%d;%p2%dH$<30>,
	is2=\E[!p\E[?3;7;19;67h\E[?4l\E[1;0%w\E(B\E)0\017\E[2J\E[1;1H$<200>,
	rs2=\E[!p\E[?3;7;19;67h\E[?4l\E[1;0%w\E(B\E)0\017\E[2J\E[1;1H$<200>,
	use=ncr260vt300an,
ncr260vt300pp|NCR 2900_260 vt300 with pc+ kybd,
	ka1=\E[H, ka3=\EOu, kb2=\E[V, kc3=\E[U, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[4~,
	kend=\E[1~, khome=\E[H, kich1=\E[2~, knp=\E[6~, kpp=\E[5~,
	lf1=pf1, lf2=pf2, lf3=pf3, lf4=pf4, rmkx=\E>, smkx=\E=,
	use=ncr260vt300an,
NCR260VT300WPP|ncr260vt300wpp|NCR 2900_260 vt300 wide mode pc+  kybd,
	cols#132,
	cup=\E[%i%p1%d;%p2%dH$<30>,
	is2=\E[!p\E[?3;7;19;67h\E[?1;4l\E[1;0%w\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	rs2=\E[!p\E[?3;7;19;67h\E[?1;4l\E[1;0%w\E(B\E)0\017\E[2J\E[1;1H\E>$<200>,
	use=ncr260vt300pp,
# This terminfo file contains color capabilities for the Wyse325 emulation of
# the NCR 2900/260C color terminal.  Because of the structure of the command
# (escape sequence) used to set color attributes, one of the fore/background
# colors must be preset to a given value. I have set the background color to
# black.  The user can change this setup by altering the last section of the
# 'setf' definition.  The escape sequence to set color attributes is
#		ESC d y <foreground_color> <background_color> 1
# In addition, the background color can be changed through the desk accessories.
# The capablitiy 'op' sets colors to green on black (default combination).
#
# NOTE:  The NCR Unix System Administrator's Shell will not function properly
# 	    if the 'pairs' capability is defined. Un-Comment the 'pairs'
#	    capability and recompile if you wish to have it included.
#
ncr260wy325pp|NCR 2900_260 wyse 325,
	am, bw, hs, km, mc5i, mir, msgr, xon,
	colors#16, cols#80, lines#24, ncv#33, nlab#32,
	acsc=07a?h;j5k3l2m1n8q\:t4u9v=w0x6, bel=^G, blink=\EG2,
	cbt=\EI, civis=\E`0, clear=\E*$<10>, cnorm=\E`1, cr=^M,
	cub1=\010$<5>, cud1=\n$<5>, cuf1=\014$<5>,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c$<10>, cuu1=\013$<5>,
	cvvis=\E`5, dch1=\EW$<50>, dl1=\ER$<5>, dsl=\E`c,
	ed=\Ey$<5>, el=\Et$<5>, fsl=^M, home=\036$<5>, ht=^I,
	hts=\E1, il1=\EE$<5>, ind=\n$<5>, invis=\EG1,
	is2=\Ee6\Ed/\Ee1\Ed*\Er\EO\E`1\E`\:\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	kDC=\ER, kEND=\EY, kHOM=\E{, kNXT=\EK, kPRT=\E7, kPRV=\EJ,
	kRIT=^L, ka1=^^, kb2=\EJ, kbs=^H, kc1=\ET, kc3=\EK, kcbt=\EI,
	kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, kdch1=\EW, kend=\ET,
	kf1=^A@\r, kf10=^AI\r, kf11=^AJ\r, kf12=^AK\r, kf13=^AL\r,
	kf14=^AM\r, kf15=^AN\r, kf16=^AO\r, kf17=^A`\r, kf18=^Aa\r,
	kf19=^Ab\r, kf2=^AA\r, kf20=^Ac\r, kf21=^Ad\r, kf22=^Ae\r,
	kf23=^Af\r, kf24=^Ag\r, kf25=^Ah\r, kf26=^Ai\r, kf27=^Aj\r,
	kf28=^Ak\r, kf29=^Al\r, kf3=^AB\r, kf30=^Am\r, kf31=^An\r,
	kf32=^Ao\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r,
	kf8=^AG\r, kf9=^AH\r, khome=^^, kich1=\Eq, knp=\EK, kpp=\EJ,
	kprt=\EP, mc0=\EP, mc4=^T, mc5=^R,
	mrcup=\Ew@%p1%{48}%+%c%p2%{32}%+%c%p3%{32}%+%c$<10>,
	nel=\037$<5>, rev=\EG4, ri=\Ej$<5>, rmacs=\EH\003\EcB0,
	rmam=\Ed., rmir=\Er, rmso=\EG0, rmul=\EG0, rmxon=\Ec20,
	rs2=\Ee6\Ed/\Ee1\Ed*\Er\EO\E`1\E`\:\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	setb=\s,
	setf=%?%p1%{0}%=%t%{49}%e%p1%{1}%=%t%{50}%e%p1%{2}%=%t%{51}%e%p1%{3}%=%t%{52}%e%p1%{4}%=%t%{53}%e%p1%{5}%=%t%{54}%e%p1%{6}%=%t%{55}%e%p1%{7}%=%t%{64}%e%p1%{8}%=%t%{57}%e%p1%{9}%=%t%{58}%e%p1%{10}%=%t%{59}%e%p1%{11}%=%t%{60}%e%p1%{12}%=%t%{61}%e%p1%{13}%=%t%{62}%e%p1%{14}%=%t%{63}%e%p1%{15}%=%t%{56}%;\Edy%c11$<100>,
	sgr0=\EG0\EcB0\EcD$<15>, smacs=\EH\002\EcB1, smam=\Ed/,
	smir=\Eq, smso=\EGt, smul=\EG8, smxon=\Ec21, tbc=\E0,
	tsl=\EF,
ncr260wy325wpp|NCR 2900_260 wyse 325 wide mode,
	cols#132,
	cup=\Ea%i%p1%dR%p2%dC$<30>,
	is2=\Ee6\Ed/\Ee1\Ed*\Er\EO\E`1\E`;\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	rs2=\Ee6\Ed/\Ee1\Ed*\Er\EO\E`1\E`;\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	use=ncr260wy325pp,
# This definition for Wyse 350 supports several attributes.  This means
# that it has magic cookies (extra spaces where the attributes begin).
# Some applications do not function well with magic cookies.  The System
# Administrator's Shell in NCR Unix SVR4 1.03 is one such application.
# If supporting various attributes is not vital, 'xmc#1' and the extra
# attributes can be removed.
# Mapping to ASCII character set ('acsc' capability) can also be
# restored if needed.
# In addition, color capabilities have been added to this file.  The drawback,
# however, is that the background color has to be black.  The foreground colors
# are numbered 0 through 15.
#
# NOTE:  The NCR Unix System Administrator's Shell does not function properly
# 	    with the 'pairs' capability defined as below.  If you wish to
#	    have it included, Un-comment it and recompile (using 'tic').
#
ncr260wy350pp|NCR 2900_260 wyse 350,
	am, bw, hs, km, mc5i, mir, msgr, xon,
	colors#16, cols#80, lines#24, ncv#33, nlab#32, pairs#16, xmc#1,
	acsc=07a?h;j5k3l2m1n8q\:t4u9v=w0x6, bel=^G, blink=\EG2,
	cbt=\EI, civis=\E`0, clear=\E+$<20>, cnorm=\E`1, cr=^M,
	cub1=\010$<5>, cud1=\n$<5>, cuf1=\014$<5>,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c$<40>, cuu1=\013$<5>,
	cvvis=\E`5, dch1=\EW$<50>, dim=\EGp, dl1=\ER$<5>, dsl=\E`c,
	ed=\Ey$<5>, el=\Et$<5>, fsl=^M, home=\036$<10>, ht=^I,
	hts=\E1, il1=\EE$<5>, ind=\n$<5>, invis=\EG1,
	is2=\Ee6\Ed/\Ee1\Ed*\Er\EO\E`1\E`\:\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	kDC=\ER, kEND=\EY, kHOM=\E{, kPRT=\E7, kRIT=^L, ka1=^^, kbs=^H,
	kc1=\ET, kc3=\EK, kcbt=\EI, kcub1=^H, kcud1=^J, kcuf1=^L,
	kcuu1=^K, kdch1=\EW, kend=\ET, kf1=^A@\r, kf10=^AI\r,
	kf11=^AJ\r, kf12=^AK\r, kf13=^AL\r, kf14=^AM\r, kf15=^AN\r,
	kf16=^AO\r, kf17=^A`\r, kf18=^Aa\r, kf19=^Ab\r, kf2=^AA\r,
	kf20=^Ac\r, kf21=^Ad\r, kf22=^Ae\r, kf23=^Af\r, kf24=^Ag\r,
	kf25=^Ah\r, kf26=^Ai\r, kf27=^Aj\r, kf28=^Ak\r, kf29=^Al\r,
	kf3=^AB\r, kf30=^Am\r, kf31=^An\r, kf32=^Ao\r, kf4=^AC\r,
	kf5=^AD\r, kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, kf9=^AH\r,
	khome=^^, kich1=\Eq, knp=\EK, kpp=\EJ, kprt=\EP,
	mc0=\EP$<10>, mc4=^T, mc5=^R,
	mrcup=\Ew@%p1%{48}%+%c%p2%{32}%+%c%p3%{32}%+%c$<20>,
	nel=\037$<5>, rev=\EG4, ri=\Ej$<5>, rmacs=\EH\003\EcB0,
	rmam=\Ed., rmir=\Er, rmso=\EG0, rmul=\EG0, rmxon=\Ec20,
	rs2=\Ee6\Ed/\Ee1\Ed*\Er\EO\E`1\E`\:\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	setb=\s,
	setf=%?%p1%{0}%=%t%{49}%e%p1%{1}%=%t%{50}%e%p1%{2}%=%t%{51}%e%p1%{3}%=%t%{52}%e%p1%{4}%=%t%{53}%e%p1%{5}%=%t%{54}%e%p1%{6}%=%t%{55}%e%p1%{7}%=%t%{102}%e%p1%{8}%=%t%{97}%e%p1%{9}%=%t%{98}%e%p1%{10}%=%t%{99}%e%p1%{11}%=%t%{101}%e%p1%{12}%=%t%{106}%e%p1%{13}%=%t%{110}%e%p1%{14}%=%t%{111}%e%p1%{15}%=%t%{56}%;\Em0%c$<100>,
	sgr0=\EG0\EH\003\EcD, smacs=\EH\002\EcB1, smam=\Ed/,
	smir=\Eq, smso=\EGt, smul=\EG8, smxon=\Ec21, tbc=\E0,
	tsl=\EF,
ncr260wy350wpp|NCR 2900_260 wyse 350 wide mode,
	cols#132,
	cup=\Ea%i%p1%dR%p2%dC$<30>,
	is2=\Ee6\Ed/\Ee1\Ed*\Er\EO\E`1\E`;\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<200>,
	rs2=\Ee6\Ed/\Ee1\Ed*\Er\EO\E`1\E`;\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<200>,
	use=ncr260wy350pp,
# This definition for Wyse 50+ supports several attributes.  This means
# that it has magic cookies (extra spaces where the attributes begin).
# Some applications do not function well with magic cookies.  The System
# Administrator's Shell in NCR Unix SVR4 1.03 is one such application.
# If supporting various attributes is not vital, 'xmc#1' and the extra
# attributes can be removed.
# Mapping to ASCII character set ('acsc' capability) can also be
# restored if needed.
# (ncr260wy50+pp: originally contained commented-out
# <acsc=j5k3l2m1n8q:t4u9v=w0x6>, as well as the commented-out one there -- esr)
ncr260wy50+pp|NCR 2900_260 wyse 50+,
	am, bw, hs, km, mc5i, mir, msgr, xon,
	cols#80, lines#24, nlab#32, xmc#1,
	acsc=0wa_h[jukslrmqnxqzttuyv]wpxv, bel=^G, blink=\EG2,
	cbt=\EI$<5>, civis=\E`0, clear=\E+$<20>, cnorm=\E`1, cr=^M,
	cub1=\010$<5>, cud1=\n$<5>, cuf1=\014$<5>,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c$<30>, cuu1=\013$<5>,
	cvvis=\E`5, dch1=\EW$<50>, dim=\EGp, dl1=\ER$<5>, dsl=\E`c,
	ed=\EY$<5>, el=\ET$<5>, fsl=^M, home=\036$<10>,
	ht=\011$<5>, hts=\E1$<5>, il1=\EE$<5>, ind=\n$<5>,
	invis=\EG1,
	is2=\Ee6\E~"$<100>\E+\Ed/\Ee1\Ed*\Er\EO\E`1\E`\:\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	kDC=\ER, kEND=\EY, kHOM=\E{, kPRT=\E7, kRIT=^L, ka1=^^, kbs=^H,
	kc1=\ET, kc3=\EK, kcbt=\EI, kcub1=^H, kcud1=^J, kcuf1=^L,
	kcuu1=^K, kdch1=\EW, kend=\ET, kf1=^A@\r, kf10=^AI\r,
	kf11=^AJ\r, kf12=^AK\r, kf13=^AL\r, kf14=^AM\r, kf15=^AN\r,
	kf16=^AO\r, kf17=^A`\r, kf18=^Aa\r, kf19=^Ab\r, kf2=^AA\r,
	kf20=^Ac\r, kf21=^Ad\r, kf22=^Ae\r, kf23=^Af\r, kf24=^Ag\r,
	kf25=^Ah\r, kf26=^Ai\r, kf27=^Aj\r, kf28=^Ak\r, kf29=^Al\r,
	kf3=^AB\r, kf30=^Am\r, kf31=^An\r, kf32=^Ao\r, kf4=^AC\r,
	kf5=^AD\r, kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, kf9=^AH\r,
	khome=^^, kich1=\Eq, knp=\EK, kpp=\EJ, kprt=\EP,
	mc0=\EP$<10>, mc4=^T, mc5=^R,
	mrcup=\Ew@%p1%{48}%+%c%p2%{32}%+%c%p3%{32}%+%c$<10>,
	nel=\037$<5>, rev=\EG4, ri=\Ej$<5>, rmacs=\EH^C, rmam=\Ed.,
	rmir=\Er, rmso=\EG0, rmul=\EG0, rmxon=\Ec20,
	rs2=\Ee6\E~"$<100>\E+\Ed/\Ee1\Ed*\Er\EO\E`1\E`\:\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	sgr0=\EG0\EH\003$<15>, smacs=\EH^B, smam=\Ed/, smir=\Eq,
	smso=\EGt, smul=\EG8, smxon=\Ec21, tbc=\E0$<5>, tsl=\EF,
ncr260wy50+wpp|NCR 2900_260 wyse 50+ wide mode,
	cols#132,
	cup=\Ea%i%p1%dR%p2%dC$<30>,
	is2=\Ee6\E~"$<100>\E+\Ed/\Ee1\Ed*\Er\EO\E`1\E`;\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<200>,
	rs2=\Ee6\E~"$<100>\E+\Ed/\Ee1\Ed*\Er\EO\E`1\E`;\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<200>,
	use=ncr260wy50+pp,
ncr260wy60pp|NCR 2900_260 wyse 60,
	am, bw, hs, km, mc5i, mir, msgr, xon,
	cols#80, lines#24, nlab#32,
	acsc=07a?h;j5k3l2m1n8q\:t4u9v=w0x6, bel=^G, blink=\EG2,
	cbt=\EI$<15>, civis=\E`0, clear=\E*$<100>, cnorm=\E`1,
	cr=^M, cub1=\010$<5>, cud1=\n$<5>, cuf1=\014$<5>,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c$<10>, cuu1=\013$<5>,
	cvvis=\E`5, dch1=\EW$<50>, dl1=\ER$<5>, dsl=\E`c,
	ed=\Ey$<5>, el=\Et$<5>, fsl=^M, home=\036$<25>,
	ht=\011$<15>, hts=\E1$<15>, il1=\EE$<5>, ind=\n$<5>,
	invis=\EG1,
	is2=\Ee6\E~4$<100>\E+\Ed/\Ee1\Ed*\Er\EO\E`1\E`\:\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	kDC=\ER, kEND=\EY, kHOM=\E{, kNXT=\EK, kPRT=\E7, kPRV=\EJ,
	kRIT=^L, ka1=^^, kb2=\EJ, kbs=^H, kc1=\ET, kc3=\EK,
	kcbt=\EI$<15>, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K,
	kdch1=\EW, kend=\ET, kf1=^A@\r, kf10=^AI\r, kf11=^AJ\r,
	kf12=^AK\r, kf13=^AL\r, kf14=^AM\r, kf15=^AN\r, kf16=^AO\r,
	kf17=^A`\r, kf18=^Aa\r, kf19=^Ab\r, kf2=^AA\r, kf20=^Ac\r,
	kf21=^Ad\r, kf22=^Ae\r, kf23=^Af\r, kf24=^Ag\r, kf25=^Ah\r,
	kf26=^Ai\r, kf27=^Aj\r, kf28=^Ak\r, kf29=^Al\r, kf3=^AB\r,
	kf30=^Am\r, kf31=^An\r, kf32=^Ao\r, kf4=^AC\r, kf5=^AD\r,
	kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^,
	kich1=\Eq, knp=\EK, kpp=\EJ, kprt=\EP, mc0=\EP, mc4=^T, mc5=^R,
	mrcup=\Ew@%p1%{48}%+%c%p2%{32}%+%c%p3%{32}%+%c$<30>,
	nel=\037$<5>, rev=\EG4, ri=\Ej$<5>, rmacs=\EH^C, rmam=\Ed.,
	rmir=\Er, rmso=\EG0, rmul=\EG0, rmxon=\Ec20,
	rs2=\Ee6\E~4$<100>\E+\Ed/\Ee1\Ed*\Er\EO\E`1\E`\:\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	sgr0=\EG0\EcB0\EcD$<15>, smacs=\EH^B, smam=\Ed/,
	smir=\Eq, smso=\EGt, smul=\EG8, smxon=\Ec21, tbc=\E0$<15>,
	tsl=\EF,
ncr260wy60wpp|NCR 2900_260 wyse 60 wide mode,
	cols#132,
	cup=\Ea%i%p1%dR%p2%dC$<30>,
	is2=\Ee6\E~4$<100>\E+\Ed/\Ee1\Ed*\Er\EO\E`1\E`;\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	rs2=\Ee6\E~4$<100>\E+\Ed/\Ee1\Ed*\Er\EO\E`1\E`;\E`@\E~!\E"\Ee4\Ex@\E`9\Ee7$<100>,
	use=ncr260wy60pp,
ncr160vppp|NCR 2900_160 viewpoint,
	use=ncr260vppp,
ncr160vpwpp|NCR 2900_160 viewpoint wide mode,
	use=ncr260vpwpp,
ncr160vt100an|NCR 2900_160 vt100 with ansi kybd,
	use=ncr260vt100an,
ncr160vt100pp|NCR 2900_160 vt100 with PC+ kybd,
	use=ncr260vt100pp,
ncr160vt100wan|NCR 2900_160 vt100 wide mode ansi kybd,
	use=ncr260vt100wan,
ncr160vt100wpp|NCR 2900_160 vt100 wide mode pc+  kybd,
	use=ncr260vt100wpp,
ncr160vt200an|NCR 2900_160 vt200 with ansi kybd,
	use=ncr260vt200an,
ncr160vt200pp|NCR 2900_160 vt200 with pc+ kybd,
	use=ncr260vt200pp,
ncr160vt200wan|NCR 2900_160 vt200 wide mode ansi kybd,
	use=ncr260vt200wan,
ncr160vt200wpp|NCR 2900_160 vt200 wide mode pc+  kybd,
	use=ncr260vt200wpp,
ncr160vt300an|NCR 2900_160 vt300 with ansi kybd,
	use=ncr260vt300an,
ncr160vt300pp|NCR 2900_160 vt300 with pc+ kybd,
	use=ncr260vt300pp,
ncr160vt300wan|NCR 2900_160 vt300 wide mode ansi kybd,
	use=ncr260vt300wan,
ncr160vt300wpp|NCR 2900_160 vt300 wide mode pc+  kybd,
	use=ncr260vt300wpp,
ncr160wy50+pp|NCR 2900_160 wyse 50+,
	use=ncr260wy50+pp,
ncr160wy50+wpp|NCR 2900_160 wyse 50+ wide mode,
	use=ncr260wy50+wpp,
ncr160wy60pp|NCR 2900_160 wyse 60,
	use=ncr260wy60pp,
ncr160wy60wpp|NCR 2900_160 wyse 60 wide mode,
	use=ncr260wy60wpp,
ncrvt100an|ncrvt100pp|NCR vt100 for the 2900 terminal,
	am, hs, mc5i, mir, msgr, xon,
	cols#80, it#8, lines#24, nlab#32,
	acsc=``aaffgghhiijjkkllmmnnqqttuuvvwwxxyyzz~~,
	bel=^G, blink=\E[5m$<30>, bold=\E[1m$<30>,
	clear=\E[2J\E[1;1H$<300>, cr=^M,
	csr=\E[%i%p1%d;%p2%dr$<100>, cub=\E[%p1%dD$<30>,
	cub1=\E[D$<2>, cud=\E[%p1%dB$<30>, cud1=\E[B$<2>,
	cuf=\E[%p1%dC$<30>, cuf1=\E[C$<2>,
	cup=\E[%i%p1%d;%p2%dH$<100>, cuu=\E[%p1%dA$<30>,
	cuu1=\E[A$<2>, dch=\E[%p1%dP$<40>, dch1=\E[1P$<10>,
	dl=\E[%p1%dM$<70>, dl1=\E[M$<40>, dsl=\E[31l$<25>,
	ed=\E[0J$<300>, el=\E[0K$<30>, el1=\E[1K$<30>,
	enacs=\E(B\E)0$<40>, fsl=1$<10>, home=\E[H$<2>$<80>,
	ht=^I, hts=\EH, il=\E[%p1%dL$<80>, il1=\E[B\E[L$<80>,
	ind=\ED,
	is2=\E[12h\E[?10l\E%/0n\E[P\031\E[?3l\E(B\E)0$<200>,
	kLFT=\E[D, kRIT=\E[C, ka1=\E[H, kbs=^H, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kent=^M, kf1=\EOP,
	kf2=\EOQ, kf3=\EOR, kf4=\EOS, mc0=\E[i$<100>, nel=\EE,
	rc=\E8, rev=\E[7m$<30>, ri=\EM$<50>, rmacs=\017$<90>,
	rmir=\E[4l$<80>, rmso=\E[0m$<30>, rmul=\E[0m$<30>,
	rs2=\Ec\E[12;31h\E[?3;4;5;10l\E[?6;7;19;25h\E[33;34l\E[0m\E(B\E)0\E%/0n\E[P\031$<200>,
	sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;$<100>,
	sgr0=\017\E[0m$<120>, smacs=\016$<90>, smir=\E[4h$<80>,
	smso=\E[7m$<30>, smul=\E[4m$<30>, tbc=\E[3g$<40>,
	tsl=\E[>+1$<70>,
ncrvt100wan|NCRVT100WPP|ncrvt100wpp|NCR VT100 emulation of the 2900 terminal,
	cols#132,
	is2=\E[12h\E[?10l\E%/0n\E[P\031\E[?3h\E(B\E)0$<200>,
	rs2=\Ec\E[12;31h\E[?4;5;10l\E?3;6;7;19;25h\E[33;34l\E[0m\E(B\E)0\E%/0n\E[P\031$<200>,
	use=ncrvt100an,
#
# Vendor-supplied NCR termcaps end here

# NCR7900 DIP switches:
#
# Switch A:
# 1-4 - Baud Rate
# 5   - Parity (Odd/Even)
# 6   - Don't Send or Do Send Spaces
# 7   - Parity Enable
# 8   - Stop Bits (One/Two)
#
# Switch B:
# 1   - Upper/Lower Shift
# 2   - Typewriter Shift
# 3   - Half Duplex / Full Duplex
# 4   - Light/Dark Background
# 5-6 - Carriage Return Without / With Line Feed
# 7   - Extended Mode
# 8   - Suppress Keyboard Display
#
# Switch C:
# 1   - End of line entry disabled/enabled
# 2   - Conversational mode / (Local?) Mode
# 3   - Control characters displayed / not displayed
# 4   - (2-wire?) / 4-wire communications
# 5   - RTS on and off for each character
# 6   - (50Hz?) / 60 Hz
# 7   - Exit after level zero diagnostics
# 8   - RS-232 interface
#
# Switch D:
# 1   - Reverse Channel (yes / no)
# 2   - Manual answer (no / yes)
# 3-4 - Cursor appearance
# 5   - Communication Rate
# 6   - Enable / Disable EXT turnoff
# 7   - Enable / Disable CR turnoff
# 8   - Enable / Disable backspace
#
# Since each attribute parameter is 0 or 1, we shift each attribute (standout,
# reverse, blink, dim, and underline) the appropriate number of bits (by
# multiplying the 0 or 1 by a correct factor to shift) so the bias character,
# '@' is (effectively) "or"ed with each attribute to generate the proper third
# character in the <ESC>0 sequence.  The <sgr> string implements the following
# equation:
#
# ((((('@' + P5) | (P4 << 1)) | (P3 << 3)) | (P2 << 4)) | (p1 * 17))    =>
# ((((('@' + P5) + (P4 << 1)) + (P3 << 3)) + (P2 << 4)) + (p1 * 17))
#
#	Where:  P1 <==> Standout attribute parameter
#		P2 <==> Underline attribute parameter
#		P3 <==> Reverse attribute parameter
#		P4 <==> Blink attribute parameter
#		P5 <==> Dim attribute parameter
# From <root@goliath.un.atlantaga.NCR.COM>, init string hacked by SCO.
ncr7900i|ncr7900|ncr 7900 model 1,
	am, bw, ul,
	cols#80, lines#24, xmc#1,
	bel=^G, blink=\E0B, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=^F,
	cup=\E1%p2%c%p1%c, cuu1=^Z, dim=\E0A, ed=\Ek, el=\EK, ind=^J,
	is2=\E0@\010\E3\E4\E7, kcub1=^U, kcud1=^J, kcuf1=^F,
	kcuu1=^Z, khome=^A, ll=^A, mc4=^T, mc5=^R, rev=\E0P, rmso=\E0@,
	rmul=\E0@,
	sgr=\E0%p5%{64}%+%p4%{2}%*%+%p3%{16}%*%+%p2%{32}%*%+%p1%{17}%*%+%c,
	sgr0=\E0@, smso=\E0Q, smul=\E0`,
ncr7900iv|ncr 7900 model 4,
	am, bw, eslok, hs,
	cols#80, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J,
	cup=\013%p1%{64}%+%c\E\005%p2%02d, dl1=\E^O, dsl=\Ey1,
	fsl=\Ek\Ey5, home=\013@\E^E00, il1=\E^N, ind=^J, kbs=^H,
	kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kf1=\ES, kf2=\ET,
	kf3=\EU, kf4=\EV, kf5=\EW, kf6=\EP, kf7=\EQ, kf8=\ER,
	khome=\EH, lf6=blue, lf7=red, lf8=white, nel=^M^J,
	tsl=\Ej\Ex5\Ex1\EY8%p1%{32}%+%c\Eo,
# Warning: This terminal will lock out the keyboard when it receives a CTRL-D.
#	   The user can enter a CTRL-B to get out of this locked state.
# In <hpa>, we want to output the character given by the formula:
#		((col / 10) * 16) + (col % 10)		where "col" is "p1"
ncr7901|ncr 7901 model,
	am, bw, ul,
	cols#80, lines#24,
	bel=^G, blink=\E0B, civis=^W, clear=^L, cnorm=^X, cr=^M,
	cub1=^H, cud1=^J, cuf1=^F,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=^Z, dim=\E0A,
	ed=\Ek, el=\EK,
	hpa=\020%p1%{10}%/%{16}%*%p1%{10}%m%+%c, ind=^J,
	is2=\E4^O, kclr=^L, kcub1=^U, kcud1=^J, kcuf1=^F, kcuu1=^Z,
	khome=^H, ll=^A, mc4=^T, mc5=^R, rev=\E0P, rmso=^O, rmul=^O,
	sgr=\E0%p5%{64}%+%p4%{2}%*%+%p3%{16}%*%+%p2%{32}%*%+%p1%{17}%*%+%c\016,
	sgr0=^O, smso=\E0Q\016, smul=\E0`\016,
	vpa=\013%p1%{64}%+%c,

# Newbury Data Recording Limited (Newbury Data)
#
# Have been manufacturing and reselling  various peripherals for a long time
# They don't make terminals anymore, but are still in business (in 2007).
# Their e-mail address is at ndsales@newburydata.co.uk
# and their post address is:
#
# Newbury Data Recording Ltd,
# Premier Park, Road One,
# Winsford, Cheshire, CW7 3PT
#
# Their technical support is still good, they sent me for free a printed copy
# of the 9500 user manual and I got it just 1 week after I first contacted them
# (in 2005)!

# NDR 9500
# Manufactured in the early/mid eighties, behaves almost the same as a
# Televideo 950.  Take a 950, change its cabinet for a more 80s-ish one (but
# keep the same keyboard layout), add an optional 25-line mode, replace the DIP
# switches with a menu and remove the "lock line" feature (ESC !  1 and ESC !
# 2), here is the NDR 9500.  Even the line-lock, albeit disabled, is
# recognized:  if you type in "ESC !", the next (third) character is not
# echoed, showing that the terminal was actually waiting for a parameter!
ndr9500|nd9500|Newbury Data 9500,
	am, bw, hs, mc5i, mir, msgr, ul, xon,
	cols#80, lines#24, wsl#79,
	acsc=jDkClBmAnIqKtMuLvOwNxJ, bel=^G, cbt=\EI, civis=\E.0,
	clear=\E;, cnorm=\E.1, cr=^M, cub1=^H, cud1=^V, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dim=\E), dl1=\ER, dsl=\Eh, ed=\EY, el=\ET,
	flash=\Eb$<50/>\Ed, fsl=^M, home=^^, ht=^I, hts=\E1,
	ich1=\EQ, il1=\EE, ind=^J, is2=\Ew\E'\EDF\El\Er\EO,
	kDC=\Er, kDL=\EO, kEOL=\Et, kIC=\Eq, kcbt=\EI, kclr=^Z,
	kcub1=^H, kcud1=^V, kcuf1=^L, kcuu1=^K, kdch1=\EW, kdl1=\ER,
	ked=\EY, kel=\ET, kent=^M, kf1=^A@\r, kf10=^AI\r, kf11=^AJ\r,
	kf12=^A`\r, kf13=^Aa\r, kf14=^Ab\r, kf15=^Ac\r, kf16=^Ad\r,
	kf17=^Ae\r, kf18=^Af\r, kf19=^Ag\r, kf2=^AA\r, kf20=^Ah\r,
	kf21=^Ai\r, kf22=^Aj\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r,
	kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^,
	kich1=\EQ, kil1=\EE, kprt=\EP, mc4=\Ea, mc5=\E`, nel=^_,
	pfloc=\E|%{48}%p1%+%c2%p2\031,
	pfx=\E|%{48}%p1%+%c1%p2\031, prot=\E), ri=\Ej,
	rmacs=\E%%, rmir=\Er, rmso=\E(, rmxon=^N,
	sgr=\EG0\E%%%%\E(%?%p1%p5%p8%|%|%t\E)%;%?%p9%t\E$%;,
	sgr0=\EG0\E%%\E(, smacs=\E$, smir=\Eq, smso=\E), smxon=^O,
	tbc=\E3, tsl=\Eg\Ef\011%p1%{32}%+%c, .kbs=^H,

ndr9500-nl|NDR 9500 with no status line,
	hs@,
	wsl@,
	dsl@, fsl@, tsl@, use=ndr9500,

ndr9500-25|NDR 9500 with 25th line enabled,
	lines#25, use=ndr9500,

ndr9500-25-nl|NDR 9500 with 25 lines and no status line,
	lines#25, use=ndr9500-nl,

ndr9500-mc|NDR 9500 with magic cookies (enables underline inverse video invisible and blink),
	msgr@,
	xmc#1,
	blink=\EG2, invis=\EG1, rev=\EG4, rmso=\EG0, rmul=\EG0,
	sgr=\E%%\E(%?%p5%p8%|%t\E)%;%?%p9%t\E$%;\EG%{48}%?%p7%t%{1}%+%;%?%p4%t%{2}%+%;%?%p3%p1%|%t%{4}%+%;%?%p2%t%{8}%+%;%c,
	sgr0=\EG0\E%%\E(, smso=\EG4, smul=\EG8, use=ndr9500,

ndr9500-25-mc|NDR 500 with 25 lines and magic cookies,
	lines#25, use=ndr9500-mc,

ndr9500-mc-nl|NDR 9500 with magic cookies and no status line,
	hs@,
	wsl@,
	dsl@, fsl@, tsl@, use=ndr9500-mc,

ndr9500-25-mc-nl|NDR 9500 with 25 lines and magic cookies and no status line,
	lines#25, use=ndr9500-mc-nl,

#### Perkin-Elmer (Owl)
#
# These are official terminfo entries from within Perkin-Elmer.
#

bantam|pe550|pe6100|perkin elmer 550,
	OTbs,
	cols#80, lines#24,
	bel=^G, clear=\EK$<20>, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EX%p1%{32}%+%c\EY%p2%{32}%+%c, cuu1=\EA,
	el=\EI$<20>, home=\EH, ind=^J, ll=\EH\EA,
fox|pe1100|perkin elmer 1100,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=\EH\EJ$<132>, cr=^M, cub1=^H, cud1=^J,
	cuf1=\EC, cup=\EX%p1%{32}%+%c\EY%p2%{32}%+%c, cuu1=\EA,
	ed=\EJ$<5.5*>, el=\EI, flash=\020\002$<200/>\020\003,
	home=\EH, hts=\E1, ind=^J, ll=\EH\EA, tbc=\E3,
owl|pe1200|perkin elmer 1200,
	OTbs, am, in,
	cols#80, lines#24,
	bel=^G, clear=\EH\EJ$<132>, cr=^M, cub1=^H, cud1=^J,
	cuf1=\EC, cup=\EX%p1%{32}%+%c\EY%p2%{32}%+%c, cuu1=\EA,
	dch1=\EO$<5.5*>, dl1=\EM$<5.5*>, ed=\EJ$<5.5*>,
	el=\EI$<5.5>, flash=\020\002$<200/>\020\003, home=\EH,
	hts=\E1, ich1=\EN, il1=\EL$<5.5*>, ind=^J, ip=$<5.5*>,
	kbs=^H, kf0=\ERJ, kf1=\ERA, kf2=\ERB, kf3=\ERC, kf4=\ERD,
	kf5=\ERE, kf6=\ERF, kf7=\ERG, kf8=\ERH, kf9=\ERI, ll=\EH\EA,
	rmso=\E!\0, sgr0=\E!\0, smso=\E!^H, tbc=\E3,
pe1251|pe6300|pe6312|perkin elmer 1251,
	am,
	cols#80, it#8, lines#24, pb#300, vt#8, xmc#1,
	bel=^G, clear=\EK$<332>, cr=^M, cub1=\ED, cud1=\EB, cuf1=\EC,
	cup=\EX%p1%{32}%+%c\EY%p2%{32}%+%c, cuu1=\EA,
	ed=\EJ$<20*>, el=\EI$<10*>, home=\EH, hts=\E1, ind=^J,
	kf0=\ERA, kf1=\ERB, kf10=\ERK, kf2=\ERC, kf3=\ERD, kf4=\ERE,
	kf5=\ERF, kf6=\ERG, kf7=\ERH, kf8=\ERI, kf9=\ERJ, tbc=\E3,
# (pe7000m: this had
# 	rmul=\E!\0, smul=\E!\040,
# which is probably wrong, it collides with kf0
pe7000m|perkin elmer 7000 series monochrome monitor,
	am,
	cols#80, lines#24,
	bel=^G, cbt=\E!Y, clear=\EK, cr=^M, cub1=\ED, cud1=\EB,
	cuf1=\EC, cup=\ES%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA,
	ed=\EJ, el=\EI, home=\EH, ind=^J,
	is1=\E!\0\EW  7o\Egf\ES7\s, kbs=^H, kcub1=\E!V,
	kcud1=\E!U, kcuf1=\E!W, kcuu1=\E!T, kf0=\E!\0, kf1=\E!^A,
	kf10=\E!^J, kf2=\E!^B, kf3=\E!^C, kf4=\E!^D, kf5=\E!^E,
	kf6=\E!^F, kf7=\E!^G, kf8=\E!^H, kf9=\E!^I, khome=\E!S,
	ll=\ES7\s, ri=\ER,
pe7000c|perkin elmer 7000 series colour monitor,
	is1=\E!\0\EW  7o\Egf\Eb0\Ec7\ES7\s, rmso=\Eb0,
	rmul=\E!\0, smso=\Eb2, smul=\E!\s, use=pe7000m,

#### Sperry Univac
#
# Sperry Univac has merged with Burroughs to form Unisys.
#

# This entry is for the Sperry UTS30 terminal running the TTY
# utility under control of CP/M Plus 1R1. The functionality
# provided is comparable to the DEC vt100.
# (uts30: I added <rmam>/<smam> based on the init string -- esr)
uts30|sperry uts30 with cp/m@1R1,
	am, bw, hs,
	cols#80, lines#24, wsl#40,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\ER, clear=^L,
	cnorm=\ES, cr=^M, csr=\EU%p1%{32}%+%c%p2%{32}%+%c,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\EM,
	dl=\E[%p1%dM, dl1=\EL, ed=\E[J, el=\E[K, fsl=^M, home=\E[H,
	ht=^I, ich=\E[%p1%d@, ich1=\EO, il=\E[%p1%dL, il1=\EN,
	ind=^J, indn=\E[%p1%dB, is2=\E[U 7\E[24;1H, kbs=^H,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, khome=\E[H,
	rc=\EX, rev=\E[7m, rf=/usr/share/tabset/vt100, ri=\EI,
	rin=\E[%p1%dA, rmacs=\Ed, rmam=\E[?7l, rmso=\E[m,
	rmul=\E[m, rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h,
	sc=\EW, sgr0=\E[m, smacs=\EF, smam=\E[?7m, smso=\E[7m,
	smul=\E[4m, tsl=\E], uc=\EPB,

#### Tandem
#
# Tandem builds these things for use with its line of fault-tolerant
# transaction-processing computers.  They aren't generally available
# on the merchant market, and so are fairly uncommon.
#

tandem6510|adm3a repackaged by Tandem,
	use=adm3a,

# A funny series of terminal that TANDEM uses.  The actual model numbers
# have a fourth digit after 653 that designates minor variants.  These are
# natively block-mode and rather ugly, but they have a character mode which
# this doubtless(?) exploits.  There is a 6520 that is slightly dumber.
# (tandem653: had ":sb=\ES:", probably someone's mistake for sf; also,
# removed <if=/usr/share/tabset/tandem653>, no such file -- esr)
tandem653|t653x|Tandem 653x multipage terminal,
	OTbs, am, da, db, hs,
	cols#80, lines#24, wsl#64, xmc#1,
	clear=\EI, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\023%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dsl=\Eo\r,
	ed=\EJ, el=\EK, fsl=^M, home=\EH, ind=\ES, ri=\ET, rmso=\E6\s,
	rmul=\E6\s, sgr0=\E6\s, smso=\E6$, smul=\E60, tsl=\Eo,

#### Tandy/Radio Shack
#
# Tandy has a line of VDTs distinct from its microcomputers.
#

dmterm|deskmate terminal,
	am, bw,
	cols#80, lines#24,
	bel=^G, civis=\EG5, clear=\Ej, cnorm=\EG6, cr=^M, cub1=^H,
	cud1=\EB, cuf1=\EC, cup=\EY%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=\EA, dch1=\ES, dl1=\ER, ed=\EJ, el=\EK, home=\EH, ht=^I,
	ich1=\EQ, il1=\EP, ind=\EX, invis@, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, kf0=\E1, kf1=\E2, kf2=\E3, kf3=\E4,
	kf4=\E5, kf5=\E6, kf6=\E7, kf7=\E8, kf8=\E9, kf9=\E0,
	khome=\EH, lf0=f1, lf1=f2, lf2=f3, lf3=f4, lf4=f5, lf5=f6,
	lf6=f7, lf7=f8, lf8=f9, lf9=f10, ll=\EE, rmul@, smul@,
	use=adm+sgr,
dt100|dt-100|Tandy DT-100 terminal,
	xon,
	cols#80, lines#24, xmc#1,
	acsc=jjkkllmmnnqqttuuvvwwxx, bel=^G, civis=\E[?25l,
	clear=\E[H\E[2J, cnorm=\E[?25h, cr=^M,
	csr=\E[%p1%2d;%p2%2dr, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\010\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P,
	dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I, ich1=\E[@,
	il1=\E[L, ind=^J, is2=\E[?3l\E)0\E(B, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf1=\E[?3i,
	kf10=\E[?5i, kf2=\E[2i, kf3=\E[@, kf4=\E[M, kf5=\E[17~,
	kf6=\E[18~, kf7=\E[19~, kf8=\E[20~, kf9=\E[21~, khome=\E[H,
	knp=\E[29~, kpp=\E[28~, lf1=f1, lf2=f2, lf3=f3, lf4=f4, lf5=f5,
	lf6=f6, lf7=f7, lf8=f8, ri=\EM, rmacs=^O, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m, smacs=^N, smso=\E[7m, smul=\E[4m,
dt100w|dt-100w|Tandy DT-100 terminal (wide mode),
	cols#132, use=dt100,
dt110|Tandy DT-110 emulating ansi,
	xon,
	cols#80, lines#24,
	acsc=jjkkllmmnnqqttuuvvwwxx, bel=^G, civis=\E[?25l,
	clear=\E[H\E[2J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\010\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[0P,
	dl1=\E[0M, ed=\E[0J, el=\E[0K, enacs=\E(B\E)0, home=\E[H,
	ht=^I, ich1=\E[0@, il1=\E[0L, ind=^J, is2=\E[?3l\E)0\E(B,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kend=\E[K,
	kf1=\E[1~, kf10=\E[10~, kf2=\E[2~, kf3=\E[3~, kf4=\E[4~,
	kf5=\E[5~, kf6=\E[6~, kf7=\E[7~, kf8=\E[8~, kf9=\E[9~,
	khome=\E[G, kich1=\E[@, knp=\E[26~, kpp=\E[25~, lf0=f1,
	lf1=f2, lf2=f3, lf3=f4, lf4=f5, lf5=f6, lf6=f7, lf7=f8, lf8=f9,
	lf9=f10, ri=\EM, rmacs=^O, rmso=\E[m, rmul=\E[m, sgr0=\E[m,
	smacs=^N, smso=\E[7m, smul=\E[4m,
pt210|TRS-80 PT-210 printing terminal,
	hc, os,
	cols#80,
	bel=^G, cr=^M, cud1=^J, ind=^J,

#### Tektronix (tek)
#
# Tektronix tubes are graphics terminals.  Most of them use modified
# oscilloscope technology incorporating a long-persistence green phosphor,
# and support vector graphics on a main screen with an attached "dialogue
# area" for interactive text.
#

tek|tek4012|tektronix 4012,
	OTbs, os,
	cols#75, lines#35,
	bel=^G, clear=\E\014$<1000>, cr=^M, cub1=^H, cud1=^J,
	ff=\014$<1000>, is2=\E^O,
# (tek4013: added <acsc> to suppress tic warnings re <smacs>/<rmacs> --esr)
tek4013|tektronix 4013,
	acsc=, rmacs=\E^O, smacs=\E^N, use=tek4012,
tek4014|tektronix 4014,
	cols#81, lines#38,
	is2=\E\017\E9, use=tek4012,
# (tek4015: added <acsc> to suppress tic warnings re <smacs>/<rmacs> --esr)
tek4015|tektronix 4015,
	acsc=, rmacs=\E^O, smacs=\E^N, use=tek4014,
tek4014-sm|tektronix 4014 in small font,
	cols#121, lines#58,
	is2=\E\017\E\:, use=tek4014,
# (tek4015-sm: added <acsc> to suppress tic warnings re <smacs>/<rmacs> --esr)
tek4015-sm|tektronix 4015 in small font,
	acsc=, rmacs=\E^O, smacs=\E^N, use=tek4014-sm,
# Tektronix 4023 from Andrew Klossner <orca!andrew.tektronix@csnet-relay>
#
# You need to have "stty nl2" in effect.  Some versions of tset(1) know
# how to set it for you.
#
# It's got the Magic Cookie problem around stand-out mode.  If you can't
# live with Magic Cookie, remove the :so: and :se: fields and do without
# reverse video.  If you like reverse video stand-out mode but don't want
# it to flash, change the letter 'H' to 'P' in the :so: field.
tek4023|tektronix 4023,
	OTbs, am,
	OTdN#4, cols#80, lines#24, vt#4, xmc#1,
	OTnl=^J, bel=^G, clear=\E\014$<4/>, cr=^M, cub1=^H, cud1=^J,
	cuf1=^I, cup=\034%p2%{32}%+%c%p1%{32}%+%c, kbs=^H,
	rmso=^_@, smso=^_P,
# It is recommended that you run the 4025 at 4800 baud or less;
# various bugs in the terminal appear at 9600.  It wedges at the
# bottom of memory (try "cat /usr/dict/words"); ^S and ^Q typed
# on keyboard don't work.  You have to hit BREAK twice to get
# one break at any speed - this is a documented feature.
# Can't use cursor motion because it's memory relative, and
# because it only works in the workspace, not the monitor.
# Same for home. Likewise, standout only works in the workspace.
#
# <el> was commented out since vi and rogue seem to work better
# simulating it with lots of spaces!
#
# <il1> and <il> had 145ms of padding, but that slowed down vi's ^U
# and didn't seem necessary.
#
tek4024|tek4025|tek4027|tektronix 4024/4025/4027,
	OTbs, am, da, db,
	cols#80, it#8, lines#34, lm#0,
	bel=^G, clear=\037era\r\n\n, cmdch=^_, cr=^M,
	cub=\037lef %p1%d\r, cub1=^H, cud=\037dow %p1%d\r,
	cud1=^F^J, cuf=\037rig %p1%d\r, cuf1=\037rig\r,
	cuu=\037up %p1%d\r, cuu1=^K, dch1=\037dch\r,
	dl=\037dli %p1%d\r\006, dl1=\037dli\r\006,
	ed=\037dli 50\r, ht=^I, ich1=\037ich\r \010,
	il=\037up\r\037ili %p1%d\r, il1=\037up\r\037ili\r,
	ind=^F^J,
	is2=!com 31\r\n\037sto 9 17 25 33 41 49 57 65 73\r,
	rmkx=\037lea p2\r\037lea p4\r\037lea p6\r\037lea p8\r\037lea f5\r,
	smkx=\037lea p4 /h/\r\037lea p8 /k/\r\037lea p6 / /\r\037lea p2 /j/\r\037lea f5 /H/\r,
tek4025-17|tek 4025 17 line window,
	lines#17, use=tek4025,
tek4025-17-ws|tek 4025 17 line window in workspace,
	is2=!com 31\r\n\037sto 9 17 25 33 41 49 57 65 73\r\037wor 17\r\037mon 17\r,
	rmcup=\037mon h\r, rmso=\037att s\r, smcup=\037wor h\r,
	smso=\037att e\r, use=tek4025-17,
tek4025-ex|tek4027-ex|tek 4025/4027 w/!,
	is2=\037com 33\r\n!sto 9 17 25 33 41 49 57 65 73\r,
	rmcup=\037com 33\r, smcup=!com 31\r, use=tek4025,
# Tektronix 4025a
# From: Doug Gwyn <gwyn@brl-smoke.ARPA>
# The following status modes are assumed for normal operation (replace the
# initial "!" by whatever the current command character is):
#	!COM 29			# NOTE: changes command character to GS (^])
#	^]DUP
#	^]ECH R
#	^]EOL
#	^]RSS T
#	^]SNO N
#	^]STO 9 17 25 33 41 49 57 65 73
# Other modes may be set according to communication requirements.
# If the command character is inadvertently changed, termcap can't restore it.
# Insert-character cannot be made to work on both top and bottom rows.
# Clear-to-end-of-display emulation via !DLI 988 is too grotty to use, alas.
# There also seems to be a problem with vertical motion, perhaps involving
# delete/insert-line, following a typed carriage return.  This terminal sucks.
# Delays not specified; use "stty ixon -ixany" to enable DC3/DC1 flow control!
# (tek4025a: removed obsolete ":xx:". This may mean the tek4025a entry won't
# work any more. -- esr)
tek4025a|Tektronix 4025A,
	OTbs, OTpt, am, bw, da, db, xon,
	cols#80, it#8, lines#34,
	bel=^G, cbt=\035bac;, clear=\035era;\n\035rup;, cmdch=^],
	cr=^M, cub=\035lef %p1%d;, cub1=^H, cud=\035dow %p1%d;,
	cud1=^J, cuf=\035rig %p1%d;, cuf1=\035rig;,
	cuu=\035up %p1%d;, cuu1=^K, dch=\035dch %p1%d;,
	dch1=\035dch;, dl=\035dli %p1%d;, dl1=\035dli;,
	el=\035dch 80;, hpa=\r\035rig %p1%d;, ht=^I,
	il1=\013\035ili;, ind=^J, indn=\035dow %p1%d;,
	rs2=!com 29\035del 0\035rss t\035buf\035buf n\035cle\035dis\035dup\035ech r\035eol\035era g\035for n\035pad 203\035pad 209\035sno n\035sto 9 17 25 33 41 49 57 65 73\035wor 0;,
	tbc=\035sto;,
# From: cbosg!teklabs!davem Wed Sep 16 21:11:41 1981
# Here's the command file that I use to get rogue to work on the 4025.
# It should work with any program using the old curses (e.g. it better
# not try to scroll, or cursor addressing won't work.  Also, you can't
# see the cursor.)
# (This "learns" the arrow keys for rogue. I have adapted it for termcap - mrh)
tek4025-cr|tek 4025 for curses and rogue,
	OTbs, am,
	cols#80, it#8, lines#33,
	clear=\037era;, cub1=^H, cud1=^F^J, cuf1=\037rig;,
	cup=\037jum%i%p1%d\,%p2%d;, cuu1=^K, ht=^I, ind=^F^J,
	is2=!com 31\r\n\037sto 9 17 25 33 41 49 57 65 73\r,
	rmcup=\037wor 0, smcup=\037wor 33h,
# next two lines commented out since curses only allows 128 chars, sigh.
#	:ti=\037lea p1/b/\037lea p2/j/\037lea p3/n/\037lea p4/h/\037lea p5/ /\037lea p6/l/\037lea p7/y/\037lea p8/k/\037lea p9/u/\037lea p./f/\037lea pt/`era w/13\037lea p0/s/\037wor 33h:\
#	:te=\037lea p1\037lea p2\037lea p3\037lea p4\037lea pt\037lea p5\037lea p6\037lea p7\037lea p8\037lea p9/la/13\037lea p.\037lea p0\037wor 0:
tek4025ex|4025ex|4027ex|tek 4025 w/!,
	is2=\037com 33\r\n!sto 9\,17\,25\,33\,41\,49\,57\,65\,73\r,
	rmcup=\037com 33\r, smcup=!com 31\r, use=tek4025,
tek4105|tektronix 4105,
	OTbs, am, mir, msgr, ul, xenl, xt,
	cols#79, it#8, lines#29,
	acsc=, bel=^G, blink=\E[=3;<7m, bold=\E[=7;<4m, cbt=\E[Z,
	clear=\E[2J\E[H, cr=^M, cub1=\E[1D, cud1=\E[1B, cuf1=\E[1C,
	cup=\E[%i%p1%2d;%p2%2dH, cuu1=\E[1A, dch1=\E[1P,
	dim=\E[=1;<6m, dl1=\E[1M, ed=\E[J, el=\E[K, home=\E[H, ht=^I,
	il1=\E[1L, ind=\E[S, invis=\E[=6;<5, is1=\E%!1\E[m,
	is2=\E%!1\E[?6141\E[m, kbs=^H, kcub1=\E[1D, kcud1=\E[1B,
	kcuf1=\E[1C, kcuu1=\E[1A, rev=\E[=1;<3m, ri=\E[T,
	rmacs=\E[m, rmcup=, rmir=\E[4l, rmso=\E[=0;<1m,
	rmul=\E[=0;<1m, sgr0=\E[=0;<1m, smacs=\E[1m,
	smcup=\E%!1\E[?6l\E[2J, smir=\E[4h, smso=\E[=2;<3m,
	smul=\E[=5;<2m, tbc=\E[1g,

# (tek4105-30: I added <rmam>/<smam> based on the init string -- esr)
tek4105-30|4015 emulating 30 line vt100,
	am, mir, msgr, xenl, xon,
	cols#80, it#8, lines#30, vt#3,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m$<2>, bold=\E[1m$<2>,
	clear=\E[H\E[J$<50>, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C$<2>,
	cup=\E[%i%p1%d;%p2%dH$<5>, cuu=\E[%p1%dA,
	cuu1=\E[A$<2>, ed=\E[J$<50>, el=\E[K$<3>, el1=\E[1K$<3>,
	enacs=\E(B\E)0, home=\E[H, ht=^I, hts=\EH, ind=^J, kbs=^H,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, rc=\E8,
	rev=\E[7m$<2>, ri=\EM$<5>, rmacs=^O, rmam=\E[?7l,
	rmkx=\E[?1l\E>, rmso=\E[m$<2>, rmul=\E[m$<2>,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr=\E[0%?%p1%p6%|%t;1%;%?%p2%t;4%;%?%p1%p3%|%t;7%;%?%p4%t;5%;m%?%p9%t\016%e\017%;$<2>,
	sgr0=\E[m\017$<2>, smacs=^N, smam=\E[?7h, smkx=\E[?1h\E=,
	smso=\E[1;7m$<2>, smul=\E[4m$<2>, tbc=\E[3g,
	use=vt100+fnkeys,

# Tektronix 4105 from BRL
# The following setup modes are assumed for normal operation:
#	CODE ansi		CRLF no			DABUFFER 141
#	DAENABLE yes		DALINES 30		DAMODE replace
#	DAVISIBILITY yes	ECHO no			EDITMARGINS 1 30
#	FLAGGING input		INSERTREPLACE replace	LFCR no
#	ORIGINMODE relative	PROMPTMODE no		SELECTCHARSET G0 B
#	SELECTCHARSET G1 0	TABS -2
# Other setup modes may be set for operator convenience or communication
# requirements; I recommend
#	ACURSOR 1 0		AUTOREPEAT yes		AUTOWRAP yes
#	BYPASSCANCEL <LF>	CURSORKEYMODE no	DAINDEX 1 0 0
#	EOFSTRING ''		EOLSTRING <CR>		EOMCHARS <CR> <NU>
#	GAMODE overstrike	GCURSOR 0 100 0		GSPEED 10 1
#	IGNOREDEL no		KEYEXCHAR <DL>		NVDEFINE -53 "<NU>"
#	PROMPTSTRING ''		QUEUESIZE 2460		WINDOW 0 0 4095 3132
#	XMTDELAY 0
# and factory color maps.  After setting these modes, save them with NVSAVE. No
# delays are specified; use "stty ixon -ixany" to enable DC3/DC1 flow control!
# "IC" cannot be used in combination with "im" & "ei".
# "tek4105a" is just a guess:
tek4105a|Tektronix 4105,
	OTbs, OTpt, msgr, xon,
	OTkn#8, cols#80, it#8, lines#30, vt#3,
	acsc=, bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z,
	civis=\E%!0\ETD00\E%!1, clear=\E[H\E[J,
	cnorm=\E%!0\ETD10\E%!1, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\EM, cvvis=\E%!0\ETD70\E%!1,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J, el=\E[K, home=\E[H, ht=^I, hts=\EH,
	il=\E[%p1%dL, il1=\E[L, ind=^J, indn=\E[%p1%dS, is2=\E%!1,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kf0=\EOA, kf1=\EOB, kf2=\EOC, kf3=\EOD, kf4=\EOP, kf5=\EOQ,
	kf6=\EOR, kf7=\EOS, lf0=F1, lf1=F2, lf2=F3, lf3=F4, lf4=F5,
	lf5=F6, lf6=F8, ll=\E[30;H, nel=\EE, rc=\E8, rev=\E[7m, ri=\EM,
	rin=\E[%p1%dT, rmacs=^O, rmcup=\E%!0\ELBH=\E%!1,
	rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	rs2=\030\E%!0\EKC\E\014\EKR0\EKF0\ENM0\ELBH=\ETF8000010F40\ELI100\ELLA>\ELM0\EKE0\ENF1\EKS0\END0\E%!1\Ec\E[?3;5l\E[?7;8h\E[r\E[m\E>,
	sc=\E7, sgr0=\E[m, smacs=^N, smcup=\E[?6l, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m, tbc=\E[3g,

#
# Tektronix 4106/4107/4109 from BRL
# The following setup modes are assumed for normal operation:
#	CODE ansi		COLUMNMODE 80		CRLF no
#	DABUFFER 141		DAENABLE yes		DALINES 32
#	DAMODE replace		DAVISIBILITY yes	ECHO no
#	EDITMARGINS 1 32	FLAGGING input		INSERTREPLACE replace
#	LFCR no			LOCKKEYBOARD no		ORIGINMODE relative
#	PROMPTMODE no		SELECTCHARSET G0 B	SELECTCHARSET G1 0
#	TABS -2
# Other setup modes may be set for operator convenience or communication
# requirements; I recommend
#	ACURSOR 1 0		AUTOREPEAT yes		AUTOWRAP yes
#	BYPASSCANCEL <LF>	CURSORKEYMODE no	DAINDEX 1 0 0
#	EOFSTRING ''		EOLSTRING <CR>		EOMCHARS <CR> <NU>
#	GAMODE overstrike	GCURSOR 0 100 0		GSPEED 9 3
#	IGNOREDEL no		KEYEXCHAR <DL>		NVDEFINE -53 "<NU>"
#	PROMPTSTRING ''		QUEUESIZE 2620		WINDOW 0 0 4095 3132
#	XMTDELAY 0
# and factory color maps.  After setting these modes, save them with NVSAVE.  No
# delays are specified; use "stty ixon -ixany" to enable DC3/DC1 flow control!
# "IC" cannot be used in combination with "im" & "ei".
tek4106brl|tek4107brl|tek4109brl|Tektronix 4106 4107 or 4109,
	msgr, xon,
	cols#80, it#8, lines#32, vt#3,
	acsc=, bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z,
	civis=\E%!0\ETD00\E%!1, clear=\E[H\E[J,
	cnorm=\E%!0\ETD10\E%!1, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\EM, cvvis=\E%!0\ETD70\E%!1,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	ech=\E[%p1%dX, ed=\E[J, el=\E[K, home=\E[H, ht=^I, hts=\EH,
	il=\E[%p1%dL, il1=\E[L, ind=^J, indn=\E[%p1%dS, is2=\E%!1,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kf0=\EOA, kf1=\EOB, kf2=\EOC, kf3=\EOD, kf4=\EOP, kf5=\EOQ,
	kf6=\EOR, kf7=\EOS, lf0=F1, lf1=F2, lf2=F3, lf3=F4, lf4=F5,
	lf5=F6, lf6=F8, ll=\E[32;H, nel=\EE, rc=\E8, rev=\E[7m, ri=\EM,
	rin=\E[%p1%dT, rmacs=^O, rmcup=\E%!0\ELBH=\E%!1,
	rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	rs1=\030\E%!0\EKC\E\014\EKR0\EKF0\ENM0\ELBH=\ETF8000010F40\ELI100\ELLB0\ELM0\EKE0\ENF1\EKS0\END0\ERE0\E%!1\Ec\E[?3;5l\E[?7;8h\E[r\E[m\E>,
	sc=\E7, sgr0=\E[m, smacs=^N, smcup=\E[?6l, smir=\E[4h,
	smkx=\E[?1h\E=, smso=\E[7;42m, smul=\E[4m, tbc=\E[3g,

# Tektronix 4107/4109 interpret 4 modes using "\E%!" followed by a code:
# 0 selects Tek mode, i.e., \E%!0
# 1 selects ANSI mode
# 2 selects ANSI edit-mode
# 3 selects VT52 mode
#
# One odd thing about the description (which has been unchanged since the 90s)
# is that the cursor addressing is using VT52 mode, and a few others use the
# VT52's non-CSI versions of ANSI, e.g., \EJ.
tek4107|tek4109|tektronix terminals 4107 4109,
	OTbs, am, mir, msgr, ul, xenl, xt,
	cols#79, it#8, lines#29,
	bel=^G, blink=\E%!1\E[5m$<2>\E%!0,
	bold=\E%!1\E[1m$<2>\E%!0, clear=\ELZ, cnorm=\E%!0, cr=^M,
	cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, cvvis=\E%!3,
	dim=\E%!1\E[<0m$<2>\E%!0, ed=\EJ, el=\EK, ht=^I, ind=^J,
	kbs=^H, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA,
	rev=\E%!1\E[7m$<2>\E%!0, ri=\EI,
	rmso=\E%!1\E[m$<2>\E%!0, rmul=\E%!1\E[m$<2>\E%!0,
	sgr=\E%%!1\E[%?%p1%t;7;5%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p5%t<0%;%?%p6%t;1%;m$<2>\E%%!0,
	sgr0=\E%!1\E[m$<2>\E%!0, smso=\E%!1\E[7;5m$<2>\E%!0,
	smul=\E%!1\E[4m$<2>\E%!0,
# Tektronix 4207 with sysline.  In the ancestral termcap file this was 4107-s;
# see the note attached to tek4207.
tek4207-s|Tektronix 4207 with sysline but no memory,
	eslok, hs,
	dsl=\E7\E[?6l\E[2K\E[?6h\E8, fsl=\E[?6h\E8,
	is1=\E%!1\E[2;32r\E[132D\E[2g\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[J,
	is2=\E7\E[?6l\E[2K\E[?6h\E8,
	tsl=\E7\E[?6l\E[2K\E[;%i%df, use=tek4107,

# The 4110 series may be a wonderful graphics series, but they make the 4025
# look good for screen editing.  In the dialog area, you can't move the cursor
# off the bottom line.  Out of the dialog area, ^K moves it up, but there
# is no way to scroll.
#
# Note that there is a floppy for free from Tek that makes the
# 4112 emulate the vt52 (use the vt52 termcap). There is also
# an expected enhancement that will use ANSI standard sequences.
#
# 4112 in non-dialog area pretending to scroll. It really wraps
# but vi is said to work (more or less) in this mode.
#
# 'vi' works reasonably well with this entry.
#
otek4112|o4112-nd|otek4113|otek4114|old tektronix 4110 series,
	am,
	cols#80, lines#34,
	bel=^G, clear=\E^L, cr=^M, cub1=^H, cud1=^J, cuu1=^K, ind=^J,
	rmcup=\EKA1\ELV1, smcup=\EKA0\ELV0\EMG0,
# The 4112 with the ANSI compatibility enhancement
tek4112|tek4114|tektronix 4110 series,
	OTbs, am, db,
	cols#80, lines#34,
	cbt=\E[Z, clear=\E[2J\E[0;0H, cub1=^H, cud1=\E[B,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\EM, dch1=\E[P,
	dl1=\E[M, ed=\E[0J, el=\E[0K, ich1=\E[@, il1=\E[L,
	ind=\E7\E[0;0H\E[M\E8, is2=\E3!1, ri=\E7\E[0;0H\E[L\E8,
	rmso=\E[m, rmul=\E[m, sgr0=\E[m, smso=\E[7m, smul=\E[4m,
tek4112-nd|4112 not in dialog area,
	OTns,
	cuu1=^K, use=tek4112,
tek4112-5|4112 in 5 line dialog area,
	lines#5, use=tek4112,
# (tek4113: this used to have "<cuf1=\LM1\s\LM0>", someone's mistake;
# removed "<smacs=\E^N>, <rmacs=\E^O>", which had been commented out in 8.3.
# Note, the !0 and !1 sequences in <rmcup>/<smcup>/<cnorm>/<civis> were
# previously \0410 and \0411 sequences...I don't *think* they were supposed
# to be 4-digit octal -- esr)
tek4113|tektronix 4113 color graphics with 5 line dialog area,
	OTbs, am, da, eo,
	cols#80, lines#5,
	clear=\ELZ, cub1=^H, cud1=^J, cuf1=\ELM1 \ELM0,
	flash=\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERB0,
	is2=\EKA1\ELL5\ELV0\ELV1, uc=\010\ELM1_\ELM0,
tek4113-34|tektronix 4113 color graphics with 34 line dialog area,
	lines#34,
	is2=\EKA1\ELLB2\ELV0\ELV1, use=tek4113,
# :ns: left off to allow vi visual mode. APL font (:as=\E^N:/:ae=\E^O:) not
# supported here. :uc: is slow, but looks nice. Suggest setenv MORE -up .
# :vb: needs enough delay to let you see the background color being toggled.
tek4113-nd|tektronix 4113 color graphics with no dialog area,
	OTbs, am, eo,
	cols#80, it#8, lines#34,
	clear=\E^L, cub1=^H, cud1=^J, cuf1=^I, cuu1=^K,
	cvvis=\ELZ\EKA0,
	flash=\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERBA4\ERB0,
	home=\ELF7l\177 @, ht=^I, is2=\ELZ\EKA0\ELF7l\177 @,
	ll=\ELF hl @, rmso=\EMT1, smso=\EMT2, uc=\010\EMG1_\EMG0,
# This entry is from Tek. Inc.  (Brian Biehl)
# (tek4115: :bc: renamed to :le:, <rmam>/<smam> added based on init string -- esr)
otek4115|Tektronix 4115,
	OTbs, am, da, db, eo,
	cols#80, it#8, lines#34,
	cbt=\E[Z, clear=\E[H\E[2J,
	cnorm=\E%!0\ELBG8\E%!1\E[34;1H, cub1=\E[D, cud1=\E[B,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A,
	cvvis=\E%!0\ELBB2\E%!1, dch1=\E[P, dl1=\E[M, ed=\E[J,
	el=\E[K, home=\E[H, ht=^I, if=/usr/share/tabset/vt100,
	il1=\E[L,
	is2=\E%!0\E%\014\ELV0\EKA1\ELBB2\ENU@=\ELLB2\ELM0\ELV1\EKYA?\E%!1\E[<1l\E[?7h\E[?8h\E[34;1H\E[34B\E[m,
	kbs=^H, ri=\EM, rmam=\E[?7l,
	rmcup=\E%!0\ELBG8\E%!1\E[34;1H\E[J, rmir=\E[4l,
	rmkx=\E>, rmso=\E[m, rmul=\E[m, sgr0=\E[m, smam=\E[?7h,
	smcup=\E%!0\ELBB2\E%!1, smir=\E[4h, smkx=\E=, smso=\E[7m,
	smul=\E[4m,
tek4115|newer tektronix 4115 entry with more ANSI capabilities,
	am, xon,
	cols#80, lines#34,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, clear=\E[H\E[J,
	cr=^M, cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, hpa=\E[%p1%{1}%+%dG,
	ht=^I, hts=\EH, ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL,
	il1=\E[L, ind=^J, invis=\E[8m, kbs=^H, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, khome=\E[H,
	rep=%p1%c\E[%p2%{1}%-%db, rev=\E[7m, rmam=\E[?7l,
	rmso=\E[m, rmul=\E[m,
	sgr=\E[%?%p1%t7;%;%?%p2%t4;%;%?%p3%t7;%;%?%p4%t5;%;%?%p6%t1;%;%?%p7%t8;%;m,
	sgr0=\E[m, smam=\E[?7h, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	vpa=\E[%p1%{1}%+%dd,
# The tek4125 emulates a vt100 incorrectly - the scrolling region
# command is ignored.  The following entry replaces <csr> with the needed
# <il>, <il>, and <smir>; removes some cursor pad commands that the tek4125
# chokes on; and adds a lot of initialization for the tek dialog area.
# Note that this entry uses all 34 lines and sets the cursor color to green.
# Steve Jacobson 8/85
# (tek4125: there were two "\!"s in the is that I replaced with "\E!";
# commented out, <smir>=\E1 because there's no <rmir>  -- esr)
tek4125|tektronix 4125,
	lines#34,
	csr@, dl1=\E[1M, il1=\E[1L,
	is2=\E%\E!0\EQD1\EUX03\EKA\ELBB2\ELCE0\ELI100\ELJ2\ELLB2\ELM0\ELS1\ELX00\ELV1\E%\E!1\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h,
	rc@, sc@, smkx=\E=, use=vt100,

# From: <jcoker@ucbic>
# (tek4207: This was the termcap file's entry for the 4107/4207, but SCO
# supplied another, less capable 4107 entry.  So we'll use that for 4107 and
# note that if jcoker wasn't confused you may be able to use this one.
# I merged in <msgr>,<ind>,<ri>,<invis>,<tbc> from a BRL entry -- esr)
tek4207|Tektronix 4207 graphics terminal with memory,
	am, bw, mir, msgr, ul, xenl,
	cols#80, it#8, lines#32,
	blink=\E[5m, bold=\E[1m, cbt=\E[Z, clear=\E[H\E[J$<156/>,
	cub1=^H, cud1=^J, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu1=\EM, dch1=\E[P$<4/>, dl1=\E[M$<3/>, ed=\E[J,
	el=\E[K$<5/>, home=\E[H, ht=^I, ich1=\E[@$<4/>,
	il1=\E[L$<3/>, ind=\E[S, invis=\E[=6;<5,
	is2=\E%!0\ELBP0\E%!1\E[H\E[2g\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[8C\EH\E[J,
	kcub1=\E[D, kcud1=\ED, kcuf1=\E[C, kcuu1=\EM, khome=\E[H,
	rev=\E[7m, ri=\E[T,
	rmcup=\E[?6h\E%!0\ELBP0\E%!1\E[32;1f, rmso=\E[m,
	rmul=\E[m, sgr0=\E[m, smcup=\E[?6l\E[H\E[J, smso=\E[7m,
	smul=\E[4m, tbc=\E[1g,

# From: <carolyn@dali.berkeley.edu>  Thu Oct 31 12:54:27 1985
# (tek4404: There was a "\!" in <smcup> that I replaced with "\E!".
# Tab had been given as \E2I,that must be the tab-set capability -- esr)
tek4404|tektronix 4404,
	OTbs,
	cols#80, it#8, lines#32,
	blink=\E[5m, bold=\E[1m, clear=\E[H\E[2J,
	csr=\E[%i%p1%d;%p2%dr, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P, dl1=\E[1M,
	ed=\E[J, el=\E[K, home=\E[H, ht=^I, hts=\E[2I, il1=\E[1L,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, rc=\E8,
	rmcup=\E[1;1H\E[0J\E[?6h\E[?1l, rmir=\E[4l,
	rmkx=\E[?1h, rmso=\E[27m, rmul=\E[m, sc=\E7, sgr0=\E[m,
	smcup=\E%\E!1\E[1;32r\E[?6l\E>, smir=\E[4h,
	smkx=\E[?1l, smso=\E[7m, smul=\E[4m,
# Some unknown person wrote:
# I added the is string - straight Unix has ESC ; in the login
# string which sets a ct8500 into monitor mode (aka 4025 snoopy
# mode). The is string here cleans up a few things (but not
# everything).
ct8500|tektronix ct8500,
	am, bw, da, db,
	cols#80, lines#25,
	bel=^G, cbt=\E^I, clear=\E^E, cr=^M, cub1=^H, cud1=^J,
	cuf1=\ES, cup=\E|%p1%{32}%+%c%p2%{32}%+%c, cuu1=\ER,
	dch1=\E^], dl1=\E^M, ed=\E^U, el=\E^T, ht=^I, ich1=\E^\,
	il1=\E^L, ind=^J, is2=\037\EZ\Ek, ri=\E^A, rmso=\E\s,
	rmul=\E\s, sgr0=\E\s, smso=\E$, smul=\E!,

# Tektronix 4205 terminal.
#
# am is not defined because the wrap around occurs not when the char.
# is placed in the 80'th column, but when we are attempting to type
# the 81'st character on the line.  (esr: hmm, this is like the vt100
# version of xenl, perhaps am + xenl would work!)
#
# Bold, dim, and standout are simulated by colors and thus not allowed
# with colors.  The tektronix color table is mapped into the RGB color
# table by setf/setb. All colors are reset to factory specifications by oc.
# The <initc> cap uses RGB notation to define colors.  for arguments 1-3 the
# interval (0-1000) is broken into 8 smaller sub-intervals (125).  Each sub-
# interval then maps into pre-defined value.
tek4205|tektronix 4205,
	ccc, mir, msgr,
	colors#8, cols#80, it#8, lines#30, ncv#49, pairs#63,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[=7;<4m, cbt=\E[Z,
	clear=\E[2J\E[H, cr=^M, cub=\E[%p1%dD, cub1=\E[D,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch1=\E[1P, dim=\E[=1;<6m, dl=\E[%p1%dM, dl1=\E[1M,
	ech=\E%p1%dX, ed=\E[J, el=\E[K, el1=\E[1K, enacs=\E)0,
	home=\E[H, ht=^I, ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[1L,
	ind=\ED,
	initc=\E%%!0\ETF4%?%p1%{0}%=%t0%e%p1%{1}%=%t4%e%p1%{2}%=%t3%e%p1%{3}%=%t5%e%p1%{4}%=%t2%e%p1%{5}%=%t6%e%p1%{6}%=%t7%e1%;%?%p2%{125}%<%t0%e%p2%{250}%<%tA2%e%p2%{375}%<%tA?%e%p2%{500}%<%tC8%e%p2%{625}%<%tD4%e%p2%{750}%<%tE1%e%p2%{875}%<%tE\:%eF4%;%?%p3%{125}%<%t0%e%p3%{250}%<%tA2%e%p3%{375}%<%tA?%e%p3%{500}%<%tC8%e%p3%{625}%<%tD4%e%p3%{750}%<%tE1%e%p3%{875}%<%tE\:%eF4%;%?%p4%{125}%<%t0%e%p4%{250}%<%tA2%e%p4%{375}%<%tA?%e%p4%{500}%<%tC8%e%p4%{625}%<%tD4%e%p4%{750}%<%tE1%e%p4%{875}%<%tE\:%eF4%;\E%%!1,
	invis=\E[=6;<5, is1=\E%!0\ETM1\E%!1\E[m, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf0=\EOA,
	kf1=\EOB, kf2=\EOC, kf3=\EOD, kf4=\EP, kf5=\EQ, kf6=\ER,
	kf7=\ES,
	oc=\E%!0\ETFB000001F4F4F42F40030F404A4C<F450F4F46F40F47F4F40\E%!1,
	op=\E[39;40m, rev=\E[7m, ri=\EM, rmacs=^O, rmcup=,
	rmir=\E[4l, rmso=\E[=0;<1m, rmul=\E[24m,
	setb=\E[=%?%p1%{0}%=%t0m%e%p1%{1}%=%t4m%e%p1%{2}%=%t3m%e%p1%{3}%=%t5m%e%p1%{4}%=%t2m%e%p1%{5}%=%t6m%e%p1%{6}%=%t7m%e1m%;,
	setf=\E[<%?%p1%{0}%=%t0m%e%p1%{1}%=%t4m%e%p1%{2}%=%t3m%e%p1%{3}%=%t5m%e%p1%{4}%=%t2m%e%p1%{5}%=%t6m%e%p1%{6}%=%t7m%e1m%;,
	sgr0=\E[=0;<1m\E[24;25;27m\017, smacs=^N,
	smcup=\E%%!1\E[?6l\E[2J, smir=\E[4h, smso=\E[=2;<3m,
	smul=\E[4m, tbc=\E[1g,

#### Teletype (tty)
#
# These are the hardcopy Teletypes from before AT&T bought the company,
# clattering electromechanical dinosaurs in Bakelite cases that printed on
# pulpy yellow roll paper.  If you remember these you go back a ways.
# Teletype-branded VDTs are listed in the AT&T section.
#
# The earliest UNIXes were designed to use these clunkers; nroff and a few
# other programs still default to emitting codes for the Model 37.
#

tty33|tty35|model 33 or 35 teletype,
	hc, os, xon,
	cols#72,
	bel=^G, cr=^M, cud1=^J, ind=^J,
tty37|model 37 teletype,
	OTbs, hc, os, xon,
	bel=^G, cr=^M, cub1=^H, cud1=^J, cuu1=\E7, hd=\E9, hu=\E8,
	ind=^J,

# There are known to be at least three flavors of the tty40, all seem more
# like IBM half duplex forms fillers than ASCII terminals.  They have lots of
# awful braindamage, such as printing a visible newline indicator after each
# newline.  The 40-1 is a half duplex terminal and is hopeless.  The 40-2 is
# braindamaged but has hope and is described here.  The 40-4 is a 3270
# lookalike and beyond hope.  The terminal has visible bell but I don't know
# it - it's null here to prevent it from showing the BL character.
# There is an \EG in <nl> because of a bug in old vi (if stty says you have
# a "newline" style terminal (-crmode) vi figures all it needs is nl
# to get crlf, even if <cr> is not ^M.)
# (tty40: removed obsolete ":nl=\EG\EB:", it's just do+cr -- esr)
tty40|ds40|ds40-2|dataspeed40|teletype dataspeed 40/2,
	OTbs, xon,
	cols#80, lines#24,
	clear=\EH$<20>\EJ$<80>, cr=\EG, cub1=^H, cud1=\EB,
	cuf1=\EC, cuu1=\E7, dch1=\EP$<50>, dl1=\EM$<50>,
	ed=\EJ$<75>, home=\EH$<10>, ht=\E@$<10>, hts=\E1,
	ich1=\E\^$<50>, il1=\EL$<50>, ind=\ES$<20>, kbs=^],
	kcub1=^H, mc4=^T, mc5=\022$<2000>, ri=\ET$<10>, rmso=\E4,
	rs2=\023\ER$<60>, smso=\E3, tbc=\EH\E2$<80>,
tty43|model 43 teletype,
	OTbs, am, hc, os, xon,
	cols#132,
	bel=^G, cr=^M, cub1=^H, cud1=^J, ind=^J, kbs=^H,

#### Tymshare
#

# You can add <is2=\E<> to put this 40-column mode, though I can't
# for the life of me think why anyone would want to.
scanset|sc410|sc415|Tymshare Scan Set,
	am, bw, msgr,
	cols#80, lines#24,
	acsc=j%k4l<m-q\,x5, bel=^G, clear=\EH\EJ, cr=^M, cub1=^H,
	cud1=^J, cuf1=^I, cup=\EY%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=^K, ed=\EJ, el=\EK, home=\EH, ind=^J, kcub1=\ED,
	kcud1=\EB, kcuf1=\EC, kcuu1=\EA, mc0=\E;3, mc4=\E;0,
	mc5=\E;0, rc=^C, rmacs=^O, rs1=\E>, sc=^B, smacs=^N,

#### Volker-Craig (vc)
#
# If you saw a Byte Magazine cover with a terminal on it during the early
# 1980s, it was probably one of these.  Carl Helmers liked them because
# they could crank 19.2 and were cheap (that is, he liked them until he tried
# to program one...)
#

# Missing in vc303a and vc303 descriptions:  they scroll 2 lines at a time
# every other linefeed.
vc303|vc103|vc203|volker-craig 303,
	OTbs, OTns, am,
	cols#80, lines#24,
	bel=^G, clear=\014$<40>, cr=^M, cub1=^H, cud1=^J, cuf1=^I,
	cuu1=^N, home=\013$<40>, kcub1=^H, kcud1=^J, kcuf1=^I,
	kcuu1=^N, ll=\017$<1>W,
vc303a|vc403a|volker-craig 303a,
	clear=\030$<40>, cuf1=^U, cuu1=^Z, el=\026$<20>,
	home=\031$<40>, kcuf1=^U, kcuu1=^Z, ll=^P, use=vc303,
# (vc404: removed obsolete ":ma=^Z^P^U :" -- esr)
vc404|volker-craig 404,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=\030$<40>, cr=^M, cub1=^H, cud1=^J, cuf1=^U,
	cup=\020%p1%{32}%+%c%p2%{32}%+%c, cuu1=^Z,
	ed=\027$<40>, el=\026$<20>, home=\031$<40>, ind=^J,
	kcub1=^H, kcud1=^J, kcuf1=^U, kcuu1=^Z,
vc404-s|volker-craig 404 w/standout mode,
	cud1=^J, rmso=^O, smso=^N, use=vc404,
# From: <wolfgang@cs.sfu.ca>
# (vc414: merged in cup/dl1/home from an old vc414h-noxon)
vc414|vc414h|Volker-Craig 414H in sane escape mode.,
	OTbs, am,
	cols#80, lines#24,
	clear=\E\034$<40>, cud1=\E^K, cuf1=^P,
	cup=\E\021%p2%c%p1%c$<40>, cuu1=\E^L, dch1=\E3,
	dl1=\E\023$<40>, ed=\E^X, el=\E\017$<10/>, home=\E^R,
	ich1=\E\:, il1=\E\032$<40>, kcub1=^H, kcud1=\E^K, kcuf1=^P,
	kcuu1=\E^L, kf0=\EA, kf1=\EB, kf2=\EC, kf3=\ED, kf4=\EE,
	kf5=\EF, kf6=\EG, kf7=\EH, khome=\E^R, lf0=PF1, lf1=PF2,
	lf2=PF3, lf3=PF4, lf4=PF5, lf5=PF6, lf6=PF7, lf7=PF8,
	rmso=\E^_, smso=\E^Y,
vc415|volker-craig 415,
	clear=^L, use=vc404,

######## OBSOLETE PERSONAL-MICRO CONSOLES AND EMULATIONS
#

#### IBM PC and clones
#

# The pcplot IBM-PC terminal emulation program is really messed up. It is
# supposed to emulate a vt-100, but emulates the wraparound bug incorrectly,
# doesn't support scrolling regions, ignores add line commands, and ignores
# delete line commands. Consequently, the resulting behavior looks like a
# crude adm3a-type terminal.
# Steve Jacobson 8/85
pcplot|pc-plot terminal emulation program,
	xenl@,
	csr@, dl@, dl1@, il@, il1@, rc@, sc@, use=vt100,
# KayPro II from Richard G Turner <rturner at Darcom-Hq.ARPA>
# I've found that my KayPro II, running MDM730, continues to emulate an
# ADM-3A terminal, just like I was running TERM.COM. On our 4.2 UNIX
# system the following termcap entry works well:
# I have noticed a couple of minor glitches, but nothing I can't work
# around. (I added two capabilities from the BRL entry -- esr)
kaypro|kaypro2|kaypro II,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=\032$<1/>, cr=^M, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dl1=\ER, ed=^W,
	el=^X, home=^^, il1=\EE, ind=^J, kcud1=^J, kcuf1=^L, kcuu1=^K,

# From IBM, Thu May  5 19:35:27 1983
# (ibmpc: commented out <smir>=\200R because we don't know <rmir> -- esr)
ibm-pc|ibm5051|5051|IBM Personal Computer (no ANSI.SYS),
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=^L^K, cr=^M^^, cub1=^], cud1=^J, cuf1=^\,
	cuu1=^^, home=^K, ind=\n$<10>, kcud1=^_,

ibmpc|wy60-PC|wyse60-PC|IBM PC/XT running PC/IX,
	OTbs, am, bw, eo, hs, km, msgr, ul,
	cols#80, it#8, lines#24,
	acsc=j\331k\277l\332m\300n\305q\304t\303u\264v\301w\302x\263,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\Ec, cr=^M,
	cub=\E[%p1%dD, cub1=\E[D, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, ech=\E[%p1%dX, ed=\E[J, el=\E[K,
	home=\E[H, hpa=\E[%i%p1%dG, ind=\E[S\E[B,
	indn=\E[%p1%dS\E[%p1%dB, invis=\E[30;40m, kbs=^H,
	kcbt=^], kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\177, kend=\E[Y, kf1=\240, kf10=\251, kf2=\241,
	kf3=\242, kf4=\243, kf5=\244, kf6=\245, kf7=\246, kf8=\247,
	kf9=\250, khome=\E[H, kich1=\E[^H, knp=\E[U, kpp=\E[V,
	ll=\E[24;1H, nel=^M, rev=\E[7m, ri=\E[T\E[A,
	rin=\E[%p1%dT\E[%p1%dA, rmso=\E[m, rmul=\E[m,
	sgr=\E[%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;%?%p7%t30;40%;m,
	sgr0=\E[m, smso=\E[7m, smul=\E[4m,

#### Apple II
#
# Apple II firmware console first, then various 80-column cards and
# terminal emulators.  For two cents I'd toss all these in the UFO file
# along with the 40-column apple entries.
#

# From: brsmith@umn-cs.cs.umn.edu (Brian R. Smith) via BRL
#	'it#8' tells UNIX that you have tabs every 8 columns.  This is a
#		function of TIC, not the firmware.
#	The clear key on a IIgs will do something like clear-screen,
#		depending on what you're in.
appleIIgs|appleIIe|appleIIc|Apple 80 column firmware interface,
	OTbs, am, bw, eo, msgr,
	cols#80, it#8, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=^\,
	cup=\036%p2%{32}%+%c%p1%{32}%+%c, cuu1=^_, ed=^K, el=^],
	home=^Y, ht=^I, ind=^W, kbs=^H, kclr=^X, kcub1=^H, kcud1=^J,
	kcuf1=^U, kcuu1=^K, kdch1=\177, nel=^M^W, ri=^V, rmso=^N,
	smso=^O,
# Apple //e with 80-column card, entry from BRL
# The modem interface is permitted to discard LF (maybe DC1), otherwise
# passing characters to the 80-column firmware via COUT (PR#3 assumed).
# Auto-wrap does not work right due to newline scrolling delay, which also
# requires that you set "stty cr2".
# Note: Cursor addressing is only available via the Pascal V1.1 entry,
# not via the BASIC PR#3 hook.  All this nonsense can be avoided only by
# using a terminal emulation program instead of the built-in firmware.
apple2e|Apple //e,
	bw, msgr,
	cols#80, lines#24,
	bel=^G, clear=\014$<100/>, cub1=^H, cud1=^J, cuu1=^_,
	ed=\013$<4*/>, el=\035$<4/>, home=^Y, ht=^I, ind=^W,
	is2=^R^N, kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^U, kcuu1=^K,
	nel=\r$<100/>, rev=^O, ri=^V, rmso=^N, rs1=^R^N, sgr0=^N,
	smso=^O,
# mcvax!vu44!vu45!wilcke uses the "ap" entry together with Ascii Express Pro
# 4.20, with incoming and outgoing terminals both on 0, emulation On.
apple2e-p|Apple //e via Pascal,
	cup=\036%p2%{32}%+%c%p1%{32}%+%c, kbs=^H, kcub1=^H,
	kcud1=^J, use=apple2e,
# (ASCII Express) MouseTalk "Standard Apple //" emulation from BRL
# Enable DC3/DC1 flow control with "stty ixon -ixany".
apple-ae|ASCII Express,
	OTbs, am, bw, msgr, nxon, xon,
	cols#80, it#8, lines#24,
	bel=\007$<500/>, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=^U,
	cup=\036%p2%{32}%+%c%p1%{32}%+%c, cuu1=^_, ed=^K, el=^],
	home=^Y, ind=^W, is2=^R^N, kclr=^X, kcub1=^H, kcud1=^J,
	kcuf1=^U, kcuu1=^K, rev=^O, ri=^V, rmso=^N, rs1=^R^N, sgr0=^N,
	smso=^O,
appleII|apple ii plus,
	OTbs, am,
	cols#80, it#8, lines#24,
	clear=^L, cnorm=^TC2, cub1=^H, cud1=^J, cuf1=^\,
	cup=\036%p2%{32}%+%c%p1%{32}%+%c, cuu1=^_, cvvis=^TC6,
	ed=^K, el=^], flash=\024G1$<200/>\024T1, home=\E^Y, ht=^I,
	is2=\024T1\016, kcud1=^J, kcuf1=^U, rmso=^N, sgr0=^N,
	smso=^O,
# Originally by Gary Ford 21NOV83
# From: <ee178aci%sdcc7@SDCSVAX.ARPA>  Fri Oct 11 21:27:00 1985
apple-80|apple II with smarterm 80 col,
	OTbs, am, bw,
	cols#80, lines#24,
	cbt=^R, clear=\014$<10*/>, cr=\r$<10*/>, cub1=^H, cud1=^J,
	cuf1=^\, cup=\036%p2%{32}%+%c%p1%{32}%+%c, cuu1=^_,
	ed=\013$<10*/>, el=\035$<10/>, home=^Y,
apple-soroc|apple emulating soroc 120,
	am,
	cols#80, lines#24,
	bel=^G, clear=\E*$<300>, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, ed=\EY, el=\ET,
	home=^^, ind=^J, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K,
# From Peter Harrison, Computer Graphics Lab, San Francisco
#   ucbvax!ucsfmis!harrison  .....uucp
#   ucbvax!ucsfmis!harrison@BERKELEY   .......ARPA
# "These two work.  If you don't have the inverse video chip for the
# Apple with videx then remove the :so: and :se: fields."
# (apple-videx: this used to be called DaleApple -- esr)
apple-videx|Apple with videx videoterm 80 column board with inverse video,
	OTbs, am, xenl,
	cols#80, it#8, lines#24,
	clear=\014$<300/>, cub1=^H, cud1=^J, cuf1=^\,
	cup=\036%p2%{32}%+%c%p1%{32}%+%c, cuu1=^_, ed=^K, el=^],
	home=^Y, ht=^I, kcub1=^H, kcud1=^J, kcuf1=^U, khome=^Y,
	rmso=^Z2, sgr0=^Z2, smso=^Z3,
# My system [for reference] : Apple ][+, 64K, Ultraterm display card,
#			      Apple Cat ][ 212 modem, + more all
#			      controlled by ASCII Express: Pro.
# From Dave Shaver <isucs1!shaver>
apple-uterm-vb|Videx Ultraterm for Apple micros with Visible Bell,
	OTbs, am, eo, xt,
	cols#80, lines#24,
	acsc=, clear=^L, cuf1=^\,
	cup=\036%p2%{32}%+%c%p1%{32}%+%c, cuu1=^_, ed=^K, el=^],
	flash=^W35^W06, home=^Y,
	is2=^V4^W06\017\rVisible Bell Installed.\016\r\n,
	rmso=^N, smso=^O,
apple-uterm|Ultraterm for Apple micros,
	OTbs, am, eo, xt,
	cols#80, lines#24,
	acsc=, clear=^L, cuf1=^\,
	cup=\036%p2%{32}%+%c%p1%{32}%+%c, cuu1=^_, ed=^K, el=^],
	home=^Y, is2=^V4^W06\016, rmso=^N, smso=^O,
# from trwrba!bwong (Bradley W. Wong):
#
# This entry assumes that you are using an apple with the UCSD Pascal
# language card.  SYSTEM.MISCINFO is assumed to be the same as that
# supplied with the standard apple except that screenwidth should be set
# using SETUP to 80 columns.  Note that the right arrow is not mapped in
# this termcap entry.  This is because that key, on the Apple, transmits
# a ^U and would thus preempt the more useful "up" function of vi.
#
# HMH 2/23/81
apple80p|80-column apple with Pascal card,
	am, bw,
	cols#80, lines#24,
	clear=^Y^L, cuf1=^\\:,
	cup=\036%p2%{32}%+%c%p1%{32}%+%c, cuu1=^_, ed=^K, el=^],
	home=^Y, kcub1=^H,
#
# Apple II+ equipped with Videx 80 column card
#
# Terminfo from ihnp4!ihu1g!djc1 (Dave Christensen) via BRL;
# manually converted by D A Gwyn
#
# DO NOT use any terminal emulation with this data base, it works directly
# with the Videx card.  This has been tested with vi 1200 baud and works fine.
#
# This works great for vi, except I've noticed in pre-R2, ^U will scroll back
# 1 screen, while in R2 ^U doesn't.
# For inverse alternate character set add:
#	<smacs>=^O:<rmacs>=^N:
# (apple-v: added it#8 -- esr)
apple-videx2|Apple II+ w/ Videx card (similar to Datamedia h1520),
	am, xenl,
	cols#80, it#8, lines#24,
	bel=\007$<100/>, clear=\014$<16*/>, cr=^M, cub1=^H,
	cud1=^J, cuf1=^\, cup=\036%p2%{32}%+%c%p1%{32}%+%c,
	cuu1=^_, ed=\013$<16*/>, el=^], home=^Y, ht=\011$<8/>,
	ind=^J, kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^\, kcuu1=^_,
	khome=^Y, rmso=^Z2, smso=^Z3,
apple-videx3|vapple|Apple II with 80 col card,
	OTbs, am,
	cols#80, lines#24,
	clear=\Ev, cub1=\ED, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, el=\Ex,
	home=\EH, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA,
	kf0=\EP, kf1=\EQ, kf2=\ER, kf3=\E\s, kf4=\E!, kf5=\E", kf6=\E#,
	kf7=\E$, kf8=\E%%, kf9=\E&, khome=\EH,
#From: decvax!cbosgd!cbdkc1!mww Mike Warren via BRL
aepro|Apple II+ running ASCII Express Pro--vt52,
	OTbs,
	cols#80, lines#24,
	clear=\014$<300/>, cub1=\ED, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, ed=\EJ,
	el=\EK, home=\EH,
# UCSD addition: Yet another termcap from Brian Kantor's Micro Munger Factory
apple-vm80|ap-vm80|apple with viewmax-80,
	OTbs,
	cols#80, lines#24,
	clear=\014$<300/>, cuf1=^\\:,
	cup=\036%p1%{32}%+%c%p2%{32}%+%c$<100/>, cuu1=^_,
	ed=\013$<300/>, el=^], home=\031$<200/>,

#### Apple Lisa & Macintosh
#

# (lisa: changed <cvvis> to <cnorm> -- esr)
lisa|apple lisa console display (black on white),
	OTbs, am, eo, msgr,
	cols#88, it#8, lines#32,
	acsc=jdkclfmenbqattuvvuwsx`, civis=\E[5h, clear=^L,
	cnorm=\E[5l, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P, dl1=\E[M,
	ed=\E[J, el=\E[K, home=\E[H, ht=^I, ich1=\E[@, il1=\E[L,
	is2=\E>\E[m\014, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, rmacs=\E[10m, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m, smacs=\E[11m, smso=\E[7m, smul=\E[4m,
liswb|apple lisa console display (white on black),
	is2=\E>\E[0;7m\014, rmso=\E[0;7m, rmul=\E[0;7m,
	smso=\E[m, smul=\E[4m, use=lisa,

# lisaterm from ulysses!gamma!epsilon!mb2c!jed (John E. Duncan III) via BRL;
# <is2> revised by Ferd Brundick <fsbrn@BRL.ARPA>
#
# These entries assume that the 'Auto Wraparound' is enabled.
# Xon-Xoff flow control should also be enabled.
#
# The vt100 uses :rs2: and :rf: rather than :is2:/:tbc:/:hts: because the tab
# settings are in non-volatile memory and don't need to be reset upon login.
# Also setting the number of columns glitches the screen annoyingly.
# You can type "reset" to get them set.
#
lisaterm|Apple Lisa or Lisa/2 running LisaTerm vt100 emulation,
	OTbs, OTpt, am, xenl, xon,
	OTkn#4, cols#80, it#8, lines#24, vt#3,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[2J, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A, ed=\E[J,
	el=\E[K, home=\E[H, ht=^I, hts=\EH, ind=^J, kbs=^H, kcub1=\EOD,
	kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kf0=\EOP, kf1=\EOQ,
	kf2=\EOR, kf3=\EOS, lf0=F1, lf1=F2, lf2=F3, lf3=F4, rc=\E8,
	rev=\E[7m, ri=\EM, rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	rs1=\E>\E[?1l\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h\E[1;24r,
	sc=\E7, sgr0=\E[m, smkx=\E[?1h\E=, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g,
# Lisaterm in 132 column ("wide") mode.
lisaterm-w|Apple Lisa with Lisaterm in 132 column mode,
	cols#132,
	kbs=^H, kcub1=^H, kcud1=^J, use=lisaterm,
# Although MacTerminal has insert/delete line, it is commented out here
# since it is much faster and cleaner to use the "lock scrolling region"
# method of inserting and deleting lines due to the MacTerminal implementation.
# Also, the "Insert/delete ch" strings have an extra character appended to them
# due to a bug in MacTerminal V1.1.  Blink is disabled since it is not
# supported by MacTerminal.
mac|macintosh|Macintosh with MacTerminal,
	xenl,
	OTdN#30,
	blink@, dch1=\E[P$<7/>, ich1=\E[@$<9/>, ip=$<7/>, use=lisa,
# Lisaterm in 132 column ("wide") mode.
mac-w|macterminal-w|Apple Macintosh with Macterminal in 132 column mode,
	cols#132, use=mac,

#### Radio Shack/Tandy
#

# (coco3: This had "ta" used incorrectly as a boolean and bl given as "bl#7".
# I read these as mistakes for ":it#8:" and ":bl=\007:" respectively -- esr)
# From: <{pbrown,ctl}@ocf.berkeley.edu> 12 Mar 90
coco3|os9LII|Tandy CoCo3 24*80 OS9 Level II,
	OTbs, am,
	cols#80, it#8, lines#24,
	bel=^G, blink=^_", bold=\E\:^A, civis=^E\s,
	clear=\014$<5*/>, cnorm=^E!, cub1=^H, cud1=^J, cuf1=^F,
	cup=\002%p2%{32}%+%c%p1%{32}%+%c$<2/>, cuu1=^I,
	dl1=^_1, ed=^K, el=^D, home=^A, il1=^_0, kcub1=^H, kcud1=^J,
	kcuf1=^I, kcuu1=^L, rev=^_\s, rmso=^_!, rmul=^_#,
	sgr0=\037!\E\:\0, smso=^_\s, smul=^_",
# (trs2: removed obsolete ":nl=^_:" -- esr)
trs2|trsII|trs80II|Radio Shack Model II using P&T CP/M,
	OTbs, am, msgr,
	cols#80, it#8, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^_, cuf1=^],
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=^^, dl1=^K, ed=^B,
	el=^A, home=^F, ht=^I, il1=^D, ind=^J, kbs=^H, kcub1=^\,
	kcud1=^_, kcuf1=^], kcuu1=^^, rmso=^O, sgr0=^O, smso=^N,
# From: Kevin Braunsdorf <ksb@mentor.cc.purdue.edu>
# (This had extension capabilities
#	:BN=\E[?33h:BF=\E[?33l:UC=\E[_ q:BC=\E[\177 q:\
#	:CN=\ERC:CF=\ERc:NR=\ERD:NM=\ER@:
# I also deleted the unnecessary ":kn#2:", ":sg#0:" -- esr)
trs16|trs-80 model 16 console,
	OTbs, am,
	cols#80, it#8, lines#24,
	acsc=jak`l_mbquvewcxs, bel=^G, civis=\ERc, clear=^L,
	cnorm=\ERC, cr=^M, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dch1=\EQ,
	dl1=\EM, ed=\EJ, el=\EK, home=\EH, ht=^I, ich1=\EP, il1=\EL,
	ind=^J, kbs=^H, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA,
	kf0=^A, kf1=^B, kf2=^D, kf3=^L, kf4=^U, kf5=^P, kf6=^N, kf7=^S,
	khome=^W, lf0=f1, lf1=f2, lf2=f3, lf3=f4, lf4=f5, lf5=f6, lf6=f7,
	lf7=f8, mc4=\E]+, mc5=\E]=, rmacs=\ERg, rmso=\ER@, sgr0=\ER@,
	smacs=\ERG, smso=\ERD,

#### Commodore Business Machines
#
# Formerly located in West Chester, PA; went spectacularly bust in 1994
# after years of shaky engineering and egregious mismanagement.  Made one
# really nice machine (the Amiga) and boatloads of nasty ones (PET, C-64,
# C-128, VIC-20).  The C-64 is said to have been the most popular machine
# ever (most units sold); they can still be found gathering dust in closets
# everywhere.
#

# From: Kent Polk <kent@swrinde.nde.swri.edu>, 30 May 90
# Added a few more entries, converted caret-type control sequence (^x) entries
# to '\0xx' entries since a couple of people mentioned losing '^x' sequences.
# Corrections by Ty Sarna <tsarna@endicor.com>, Sat Feb 28 18:55:15 1998
#
# :as:, :ae:			Support for alternate character sets.
# :ve=\E[\040p:vi=\E[\060\040p:	cursor visible/invisible.
# :xn:  vt100 kludginess at column 80/NEWLINE ignore after 80 cols(Concept)
#     This one appears to fix a problem I always had with a line ending
#     at 'width+1' (I think) followed by a blank line in vi. The blank
#     line tended to disappear and reappear depending on how the screen
#     was refreshed. Note that this is probably needed only if you use
#     something like a Dnet Fterm with the window sized to some peculiar
#     dimension larger than 80 columns.
# :k0=\E9~:	map F10 to k0 - could have F0-9 -> k0-9, but ... F10 was 'k;'
# (amiga: removed obsolete :kn#10:,
# also added empty <acsc> to suppress a warning --esr)
amiga|Amiga ANSI,
	OTbs, am, bw, xenl,
	cols#80, lines#24,
	acsc=, bel=^G, blink=\E[7;2m, bold=\E[1m, cbt=\E[Z,
	civis=\E[0 p, clear=\E[H\E[J, cnorm=\E[ p, cub=\E[%p1%dD,
	cub1=\E[D, cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA,
	cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H,
	ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=\E[S,
	indn=\E[%p1%dS, invis=\E[8m, is2=\E[20l, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf0=\E[9~,
	kf1=\E[0~, kf2=\E[1~, kf3=\E[2~, kf4=\E[3~, kf5=\E[4~,
	kf6=\E[5~, kf7=\E[6~, kf8=\E[7~, kf9=\E[8~, rev=\E[7m,
	ri=\E[T, rin=\E[%p1%dT, rmacs=^O, rmso=\E[m, rmul=\E[m,
	rs1=\Ec, sgr0=\E[m, smacs=^N, smso=\E[7m, smul=\E[4m,

# From: Hans Verkuil <hans@wyst.hobby.nl>, 4 Dec 1995
# (amiga: added empty <acsc> to suppress a warning.
# I'm told this entry screws up badly with AS225, the Amiga
# TCP/IP package once from Commodore, and now sold by InterWorks.--esr)
amiga-h|Hans Verkuil's Amiga ANSI,
	OTbs, bw, msgr,
	cols#80, lines#24,
	acsc=, bel=^G, blink=\2337;2m, bold=\2331m, cbt=\233Z,
	civis=\2330 p, clear=\233H\233J, cnorm=\233 p, cr=^M,
	cub=\233%p1%dD, cub1=\233D, cud=\233%p1%dB, cud1=\233B,
	cuf=\233%p1%dC, cuf1=\233C, cup=\233%i%p1%d;%p2%dH,
	cuu=\233%p1%dA, cuu1=\233A, dch=\233%p1%dP, dch1=\233P,
	dim=\2332m, ech=\233%p1%dP, ed=\233J, el=\233K, flash=^G,
	home=\233H, ht=^I, ich=\233%p1%d@, ich1=\233@, ind=\233S,
	indn=\233%p1%dS, invis=\2338m, is2=\23320l, kbs=^H,
	kcub1=\233D, kcud1=\233B, kcuf1=\233C, kcuu1=\233A,
	kdch1=\177, kf0=\2339~, kf1=\2330~, kf2=\2331~, kf3=\2332~,
	kf4=\2333~, kf5=\2334~, kf6=\2335~, kf7=\2336~, kf8=\2337~,
	kf9=\2338~, nel=\233B\r, rev=\2337m, ri=\233T,
	rin=\233%p1%dT, rmacs=^O, rmcup=\233?7h, rmso=\2330m,
	rmul=\2330m, rs1=\Ec, sgr0=\2330m, smacs=^N, smcup=\233?7l,
	smso=\2337m, smul=\2334m,

# From: Henning 'Faroul' Peters <Faroul@beyond.kn-bremen.de>, 25 Sep 1999
#
# Pavel Fedin added
#	Home    Shift+Left
#	End     Shift+Right
#	PgUp    Shift+Up
#	PgDn    Shift+Down
amiga-8bit|Amiga ANSI using 8-bit controls,
	acsc=, dl=\233%p1%dM, dl1=\233M, il=\233%p1%dL, il1=\233L,
	ind=\204, indn@, kend=\233 @, khome=\233 A, knp=\233S,
	kpp=\233T, ri=\215, rin@, use=amiga-h,

# From: Ruediger Kuhlmann <terminfo@ruediger-kuhlmann.de>, 18 Jul 2000
# requires use of appropriate preferences settings.
amiga-vnc|Amiga using VNC console (black on light gray),
	am, da, db, msgr, ndscr,
	btns#1, colors#16, cols#80, lines#24, lm#0, ncv#0, pairs#256,
	bel=^G, blink=\E[7;2m, bold=\E[1m, civis=\E[0p,
	clear=\E[H\E[J, cnorm=\E[p\E[>?6l, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=\E[D,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\E[>?6h, dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[1M, ed=\E[J, el=\E[K, flash=^G,
	ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[1L, ind=\ED,
	indn=\E[%p1%dS, invis=\E8m,
	is2=\E[>?2;18l\E[>?26;?6;20;>?15;?7;>?22;>?8h,
	kbs=^H, kcbt=\233Z, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kdch1=\177, kf0=\E[9~, kf1=\E[0~, kf2=\E[1~,
	kf3=\E[2~, kf4=\E[3~, kf5=\E[4~, kf6=\E[5~, kf7=\E[6~,
	kf8=\E[7~, kf9=\E[8~, khlp=\E[?~, khome=\E[44~, kll=\E[45~,
	kmous=\E[M, knp=\E[42~, kpp=\E[41~, nel=\EE, oc=\E[0m,
	rev=\E[7m, ri=\EM, rin=\E[%p1%dT, rmcup=\E[?7h\E[r\E[J,
	rmkx=\E[?1l, rmso=\E[21m, rmul=\E[24m, rs1=\Ec,
	rs2=\E[>?2;18l\E[>?26;?6;20;>?15;?7;>?22;>?8h,
	setab=\E[%?%p1%{8}%>%t%'F'%p1%+%d%e4%p1%d%;m,
	setaf=\E[%?%p1%{8}%>%t%'2'%p1%+%d%e3%p1%d%;m,
	sgr0=\E[0m\017\E[30;85;>15m, smcup=\E[?7h, smkx=\E[?1h,
	smso=\E[1m, smul=\E[4m,

# MorphOS on Genesi Pegasos
# By Pavel Fedin <sonic_amiga@rambler.ru>
morphos,
	acsc=, dl=\233%p1%dM, dl1=\233M, il=\233%p1%dL, il1=\233L,
	ind=\204, indn@, kend=\23345~, kf11=\23320~, kf12=\23321~,
	khome=\23344~, kich1=\23340~, knp=\23342~, kpp=\23341~,
	ri=\215, rin@, use=amiga-h,

# Commodore B-128 microcomputer from Doug Tyrol <det@HEL-ACE.ARPA>
# 	I'm trying to write a termcap for a commodore b-128, and I'm
# having a little trouble. I've had to map most of my control characters
# to something that unix will accept (my delete-char is a ctrl-t, etc),
# and create some functions (like cm), but thats life.
# 	The problem is with the arrow keys - right, and up work fine, but
# left deletes the previous character and down I just can't figure out.
# Jove knows what I want, but I don't know what it's sending to me (it
# isn't thats bound to next-line in jove).
# 	Anybody got any ideas? Here's my termcap.
# DAG -- I changed his "^n" entries to "\n"; see if that works.
#
commodore|b-128|Commodore B-128 micro,
	am, bw,
	OTdN#20, cols#80, lines#24, pb#150,
	OTbc=^H, OTnl=^M, clear=\E\006$<10/>, cr=^M, cud1=^J,
	cuf1=^F, cup=\E\013%p1%2d\,%p2%2d\,$<20/>, cuu1=^P,
	dch1=\177$<10*/>, dl1=\Ed$<10*/>, el=\Eq$<10/>,
	home=\E^E, ht=\011$<5/>, ich1=\E\n$<5/>, il1=\Ei$<10/>,
	kcub1=^B, kcud1=^J, kcuf1=^F, kcuu1=^P, khome=\E^E, rmir=,
	smir=,

#### North Star
#
# North Star Advantage from Lt. Fickie <brl-ibd!fickie> via BRL
northstar|North Star Advantage,
	OTbs,
	cols#80, lines#24,
	clear=\004$<200/>,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c$<1/>, ed=\017$<200/>,
	el=\016$<200/>, home=\034\032$<200/>,

#### Osborne
#
# Thu Jul  7 03:55:16 1983
#
# As an aside, be careful; it may sound like an anomaly on the
# Osborne, but with the 80-column upgrade, it's too easy to
# enter lines >80 columns!
#
# I've already had several comments...
# The Osborne-1 with the 80-col option is capable of being
# 52, 80, or 104 characters wide; default to 80 for compatibility
# with most systems.
#
# The tab is destructive on the Ozzie; make sure to 'stty -tabs'.
osborne-w|osborne1-w|osborne I in 104-column mode,
	msgr, ul, xt,
	cols#104, lines#24,
	bel=^G, clear=^Z, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, el=\ET, ich1=\EQ, il1=\EE, ind=^J, kcub1=^H, kcud1=^J,
	kcuf1=^L, kcuu1=^K, rmso=\E(, rmul=\Em, smso=\E), smul=\El,
# Osborne I	from ptsfa!rhc (Robert Cohen) via BRL
osborne|osborne1|osborne I in 80-column mode,
	OTbs, am, mir, msgr, ul, xhp,
	OTdB#4, cols#80, lines#24,
	clear=^Z, cub1=\010$<4>, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K,
	dch1=\EW$<4/>, dl1=\ER, el=\ET, il1=\EE, is2=^Z, kbs=^H,
	kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, rmir=, rmso=\E),
	rmul=\Em, smir=\EQ, smso=\E(, smul=\El,
#
# Osborne Executive definition from BRL
# Similar to tvi920
# Added by David Milligan and Tom Smith (SMU)
osexec|Osborne executive,
	OTbs, am,
	OTug#1, cols#80, lines#24, xmc#1,
	OTnl=^J, bel=^G, clear=^Z, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dl1=\ER, ed=\EY, el=\ET, home=^^, hts=\E1, ich1=\EQ, il1=\EE,
	is2=\Eq\Ek\Em\EA\Ex0, kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^L,
	kcuu1=^K, kf0=^A@\r, kf1=^AA\r, kf2=^AB\r, kf3=^AC\r,
	kf4=^AD\r, kf5=^AE\r, kf6=^AF\r, kf7=^AG\r, kf8=^AH\r,
	kf9=^AI\r, rmir=, rmso=\Ek, rmul=\Em, smir=, smso=\Ej,
	smul=\El, tbc=\E3,

#### Console types for obsolete UNIX clones
#
# Coherent, Minix, Venix, and several lesser-known kin were OSs for 8088
# machines that tried to emulate the UNIX look'n'feel.  Coherent and Venix
# were commercial, Minix an educational tool sold in conjunction with a book.
# Memory-segmentation limits and a strong tendency to look like V7 long after
# it was obsolete made all three pretty lame.  Venix croaked early.  Coherent
# and Minix were ported to 32-bit Intel boxes, only to be run over by a
# steamroller named `Linux' (which, to be fair, traces some lineage to Minix).
# Coherent's vendor, the Mark Williams Company, went belly-up in 1994.  There
# are also, I'm told, Minix ports that ran on Amiga and Atari machines and
# even as single processes under SunOS and the Macintosh OS.
#

# See
#	http://www.minix3.org/manpages/man4/console.4.html
minix|minix console (v3),
	acsc=+\020\,\021-\030.^Y0\333`\004a\261f\370g\361h\260j\331k\277l\332m\300n\305o~p\304q\304r\304s_t\303u\264v\301w\302x\263y\363z\362{\343|\330}\234~\376,
	kdch1=\177, kend=\E[Y, kf0=\E[21~, kf1=\E[11~, kf10=\E[21~,
	kf11=\E[11;2~, kf12=\E[12;2~, kf13=\E[13;2~,
	kf14=\E[14;2~, kf15=\E[15;2~, kf16=\E[17;2~,
	kf17=\E[18;2~, kf18=\E[19;2~, kf19=\E[20;2~, kf2=\E[12~,
	kf20=\E[21;2~, kf21=\E[11;5~, kf22=\E[12;5~,
	kf23=\E[13;5~, kf24=\E[14;5~, kf25=\E[15;5~,
	kf26=\E[17;5~, kf27=\E[18;5~, kf28=\E[19;5~,
	kf29=\E[20;5~, kf3=\E[13~, kf30=\E[21;5~, kf31=\E[11;6~,
	kf32=\E[12;6~, kf33=\E[13;6~, kf34=\E[14;6~,
	kf35=\E[15;6~, kf36=\E[17;6~, kf37=\E[18;6~,
	kf38=\E[19;6~, kf39=\E[20;6~, kf4=\E[14~, kf40=\E[21;6~,
	kf5=\E[15~, kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~,
	kich1=\E[@, knp=\E[U, kpp=\E[V, lf0@, lf1@, lf2@, lf3@, lf4@, lf5@,
	use=minix-3.0,

minix-3.0|minix console (v3.0),
	use=ecma+color, use=minix-1.7,

# See
#	http://www.minix-vmd.org/pub/Minix-vmd/1.7.0/wwwman/man4/console.4.html
# This is the entry provided with minix 1.7.4, with bogus :ri: removed.
minix-1.7|minix console (v1.7),
	am, xenl,
	cols#80, it#8, lines#25,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[0J, cr=^M,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[0J, el=\E[K, el1=\E[2K,
	home=\E[H, ht=^I, ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL,
	il1=\E[L, ind=^J, is2=\E[0m, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kf0=\E[Y, kf1=\E[V, kf2=\E[U,
	kf3=\E[T, kf4=\E[S, kf5=\E[G, khome=\E[H, lf0=End, lf1=PgUp,
	lf2=PgDn, lf3=Num +, lf4=Num -, lf5=Num 5, nel=^M^J,
	rev=\E[7m, ri=\EM, rmso=\E[0m, rmul=\E[0m, sgr0=\E[0m,
	smso=\E[7m, smul=\E[4m,
# Corrected Jan 14, 1997 by Vincent Broman <broman@nosc.mil>
minix-old|minix-1.5|minix console (v1.5),
	xon,
	cols#80, it#8, lines#25,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[0J, cr=^M,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[0J, el=\E[K, home=\E[H, ht=^I,
	ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=^J,
	kbs=^H, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kf0=\E[Y, kf1=\E[V, kf2=\E[U, kf3=\E[T, kf4=\E[S, kf5=\E[G,
	khome=\E[H, nel=^M^J, rev=\E[7m, ri=\EM, rmso=\E[0m,
	rmul=\E[0m, sgr0=\E[0m, smso=\E[7m, smul=\E[4m,
# The linewrap option can be specified by editing /usr/include/minix/config.h
# before recompiling the minix 1.5 kernel.
minix-old-am|minix console with linewrap,
	am, use=minix-old,

pc-minix|minix console on an Intel box,
	use=klone+acs, use=minix-3.0,

# According to the Coherent 2.3 manual, the PC console is similar
# to a z19. The differences seem to be (1) 25 lines, (2) no status
# line, (3) standout is broken, (4) ins/del line is broken, (5)
# has blinking and bold.
pc-coherent|pcz19|coherent|IBM PC console running Coherent,
	am, mir,
	cols#80, it#8, lines#25,
	bel=^G, clear=\EE, cr=^M, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dch1=\EN,
	ed=\EJ, el=\EK, home=\EH, ht=^I, ind=^J, kbs=^H, kcub1=\ED,
	kcud1=\EB, kcuf1=\EC, kcuu1=\EA, khome=\EH, ri=\EI, rmir=\EO,
	rmso=\Eq, sgr0=\Eq, smir=\E@, smso=\Ep,

# According to the Venix 1.1 manual, the PC console is similar
# to a DEC vt52.  Differences seem to be (1) arrow keys send
# different strings, (2) enhanced standout, (3) added insert/delete line.
# Note in particular that it doesn't have automatic margins.
# There are other keys (f1-f10, kpp, knp, kcbt, kich1, kdch1) but they
# not described here because this derives from an old termcap entry.
pc-venix|venix|IBM PC console running Venix,
	cols#80, it#8, lines#25,
	bel=^G, clear=\EH\EJ, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dl1=\EM,
	ed=\EJ, el=\EK, ht=^I, il1=\EL, ind=^J, kbs=^H, kcub1=\EK,
	kcud1=\EP, kcuf1=\EM, kcuu1=\EH, khome=\EG, ri=\EI,

#### Miscellaneous microcomputer consoles
#
# If you know anything more about any of these, please tell me.
#

# The MAI Basic Four computer was obsolete at the end of the 1980s.
# It may be used as a terminal by putting it in "line" mode as seen on
# one of the status lines.
# Initialization is similar to CIT80. <is2> will set ANSI mode for you.
# Hardware tabs set by <if> at 8-spacing.  Auto line wrap causes glitches so
# wrap mode is reset by <cvvis>.  Using <ind>=\E[S caused errors so I
# used \ED instead.
# From: bf347@lafn.org (David Lawyer), 28 Jun 1997
mai|basic4|MAI Basic Four in ansi mode,
	am, da, db, mir, msgr,
	cols#82, it#8, lines#25,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=^]^_, cnorm=\E[?7h,
	cr=^M, csr=\E[%i%p1%d;%p2%dr, cub1=^H, cud1=^J, cuf1=^X,
	cup=\E[%i%p1%d;%p2%dH, cuu1=^Z, cvvis=\E[?7l, dch1=\E[1P,
	dl1=\E[M, ed=^_, el=^^, home=^], ht=^I,
	if=/usr/share/tabset/vt100, il1=\E[L, ind=\ED,
	is2=\E>\E[?1h\E[?7h\E[?5l\017\E(B\E[m\E[20l\E[1;24r\E[24;1H,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf5=\EOT, kf6=\EOU,
	kf7=\EOV, kf8=\EOW, nel=^M\ED, rc=\E8, rev=\E[7m, ri=\E[T,
	rmir=\E[4l, rmso=\E[m, rmul=\E[m, sc=\E7, sgr0=\E[m,
	smir=\E[4h, smso=\E[7m, smul=\E[4m,
# basis from Peter Harrison, Computer Graphics Lab, San Francisco
#   ucbvax!ucsfmis!harrison ...uucp / ucbvax!ucsfmis!harrison@BERKELEY ...ARPA
#
# On Sat, 7 Aug 1999, Torsten Jerzembeck <toje@nightingale.ms.sub.org> wrote:
# The Basis 108 was a Apple II clone, manufactured by the "Basis
# Mikrocomputer GmbH" in Munster, Germany (the company still exists today,
# about 1,5 km from where I live, but doesn't build own computers any
# more). A Basis 108 featured a really heavy (cast aluminium?) case, was
# equipped with one or two 5.25" disk drives, had a monochrome and colour
# video output for a TV set or a dedicated monitor and several slots for
# Apple II cards. Basis 108 were quite popular at german schools before
# the advent of the IBM PC. They run, for example, the UCSD Pascal
# development system (which I used even in 1993 to program the steering
# and data recording for our school's experimental solar panel :), Apple DOS
# or CP/M.
# (basis: removed obsolete ":ma=^K^P^R^L^L :nl=5000*^J:" -- esr)
basis|BASIS108 computer with terminal translation table active,
	clear=\E*$<300/>, cud1=\n$<5000/>, ed=\EY, el=\ET, kbs=^H,
	kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, rmso=\E), sgr0=\E),
	smso=\E(, use=adm3a,
# luna's BMC terminal emulator
luna|luna68k|LUNA68K Bitmap console,
	cols#88, lines#46, use=ansi-mini,
megatek|pegasus workstation terminal emulator,
	am, os,
	cols#83, lines#60,
# The Xerox 820 was a Z80 micro with a snazzy XEROX PARC-derived
# interface (pre-Macintosh by several years) that went nowhere.
xerox820|x820|Xerox 820,
	am,
	cols#80, lines#24,
	bel=^G, clear=1^Z, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, ed=^Q, el=^X,
	home=^^, ind=^J,

#### Videotex and teletext
#

# \E\:1}	switch to te'le'informatique mode (ascii terminal/ISO 6429)
# \E[?3l	80 columns
# \E[?4l	scrolling on
# \E[12h	local echo off
# \Ec		reset: G0 U.S. charset (to get #,@,{,},...), 80 cols, clear screen
# \E)0		G1 DEC set (line graphics)
#
# From: Igor Tamitegama <igor@ppp1493-ft.teaser.fr>, 18 Jan 1997
m2-nam|minitel|minitel-2|minitel-2-nam|France Telecom Minitel 2 mode te'le'informatique,
	OTbs, eslok, hs, xenl,
	cols#80, it#8, lines#24, wsl#72, xmc#0,
	acsc=aaffggjjkkllmmnnooqqssttuuvvwwxx, bel=^G,
	blink=\E[5m, bold=\E[1m, civis=\E[<1h, clear=\E[H\E[J,
	cnorm=\E[<1l, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub=\E[%p1%dD, cub1=\E[D, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, flash=^G, fsl=^J,
	home=\E[H, ht=^I, il=\E[%p1%dL, il1=\E[L, ind=^J, ip=$<7/>,
	is1=\E\:1}\Ec\E[?4l\E[12h, is2=\Ec\E[12h\E)0,
	is3=\E[?3l kbs=\010, kclr=\E[2J, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P, kdl1=\E[M, kf0=\EOp,
	kf1=\EOq, kf10=\EOp, kf2=\EOr, kf3=\EOs, kf4=\EOt, kf5=\EOu,
	kf6=\EOv, kf7=\EOw, kf8=\EOx, kf9=\EOy, khome=\E[H,
	kich1=\E[4h, kil1=\E[4l, knp=\EOn, kpp=\EOR, ll=\E[24;80H,
	mc0=\E[i, nel=^M^J, rc=\E8, rev=\E[7m, ri=\EM, rmacs=^O,
	rmir=\E[4l, rmso=\E[27m, rmul=\E[24m,
	rs1=\Ec\E[?4l\E[12h, rs2=\Ec\E)0, sc=\E7, sgr0=\E[m,
	smacs=^N, smir=\E[4h, smso=\E[7m, smul=\E[4m, tsl=^_@A,
	u6=\E[%i%d;%dR, u7=\E[6n,

# From: Alexandre Montaron <canal@mygale.org>, 18 Jun 1998
#
minitel1|minitel 1,
	am, bw, eslok, hs, hz, msgr,
	colors#8, cols#40, lines#24, pairs#8,
	acsc=+.\,\,./f0g1, bel=^G, blink=\EH, civis=^T, clear=^L,
	cnorm=^Q, cr=^M, cub1=^H, cud1=^J, cuf1=^I,
	cup=\037%p1%{65}%+%c%p2%{65}%+%c, cuu1=^K, el=^X,
	enacs=^Y, fsl=^J, home=^^, ind=^J,
	is2=\E;`ZQ\E\:iC\E\:iE\021, nel=^M^J, op=\EG,
	rep=%p1%c\022%p2%{63}%+%c, rev=\E], ri=^K, rmso=\E\\,
	setf=\E%?%p1%{1}%=%tD%e%p1%{3}%=%tF%e%p1%{4}%=%tA%e%p1%{6}%=%tC%e%p1%{64}%+%c%;,
	sgr=%?%p1%t\E]%;%?%p3%t\E]%;%?%p4%t\EH%;,
	sgr0=\EI\E\\, smso=\E], tsl=\037@%p1%{65}%+%c,
# is2=Fnct TE, Fnct MR, Fnct CM et pour finir: curseur ON.
minitel1b|minitel 1-bistandard (in 40cols mode),
	mir,
	cub=\E[%p1%dD, cud=\E[%p1%dB, cuf=\E[%p1%dC,
	cuu=\E[%p1%dA, dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el1=\E[1K, il=\E[%p1%dL, il1=\E[L,
	is1=\E;iYA\E;jYC, kclr=\E[2J, kctab=^I, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P, kdl1=\E[M,
	kel=^X, khome=\E[H, kich1=\E[4h, kil1=\E[L, rmir=\E[4l,
	smir=\E[4h, smkx=\E;iYA\E;jYC, use=minitel1,
# <rmkx> posait des problemes (logout en sortant de vi).
minitel1b-80|minitel 1-bistandard (standard teleinformatique),
	am@, bw@, hz@,
	colors@, cols#80, it#8, pairs@,
	blink=\E[5m, bold=\E[1m, civis=\037@A\024\n,
	clear=\E[H\E[2J, cnorm=\037@A\021\n, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, el=\E[K, home=\E[H,
	ht=^I, ind=\ED, is1@, is2@, kent=\EOM, kf0=\EOp, kf1=\EOq,
	kf2=\EOr, kf3=\EOs, kf4=\EOt, kf5=\EOu, kf6=\EOv, kf7=\EOw,
	kf8=\EOx, kf9=\EOy, nel=\EE, op@, rc=\E8, rep@, rev=\E[7m,
	ri=\EM, rmkx@, rmso=\E[27m, rmul=\E[24m, sc=\E7, setf@,
	sgr=\E[%?%p1%t7;%;%?%p2%t4;%;%?%p3%t7;%;%?%p4%t5;%;%?%p6%t1;%;m,
	sgr0=\E[m, smkx@, smso=\E[7m, smul=\E[4m, use=minitel1b,

######## OBSOLETE VDT TYPES
#
# These terminals are *long* dead -- these entries are retained for
# historical interest only.

#### Amtek Business Machines
#

# (abm80: early versions of this entry apparently had ":se=\E^_:so=\E^Y",
# but these caps were commented out in 8.3; also, removed overridden
# ":do=^J:" -- esr)
abm80|amtek business machines 80,
	OTbs, am, bw,
	cols#80, lines#24,
	cbt=^T, clear=\E^\, cub1=^H, cud1=\E^K, cuf1=^P,
	cup=\E\021%p2%{32}%+%c%p1%{32}%+%c, cuu1=\E^L,
	dl1=\E^S, ed=\E^X, el=\E^O, home=\E^R, il1=\E^Z,

#### Bell Labs blit terminals
#
# These were AT&T's official entries.  The 5620 FAQ maintained by
# David Breneman <daveb@dgtl.com> has this to say:
#
#  Actually, in the beginning was the Jerq, and the Jerq was white with a
#  green face, and Locanthi and Pike looked upon the Jerq and said the Jerq
#  was good.  But lo, upon the horizon loomed a mighty management-type person
#  (known now only by the initials VP) who said, the mighty Jerq must stay
#  alone, and could not go forth into the world. So Locanthi and Pike put the
#  Jerq to sleep, cloned its parts, and the Blit was brought forth unto the
#  world. And the Jerq lived the rest of its days in research, but never
#  strayed from those paths.
#
#  In all seriousness, the Blit was originally known as the Jerq, but when
#  it started to be shown outside of the halls of the Bell Labs Research
#  organization, the management powers that be decided that the name could
#  not remain. So it was renamed to be Blit. This was in late 1981.
#
# (The AT&T 5620 was the commercialized Blit.  Its successors were the 630,
# 730, and 730+.)
#

blit|jerq|blit running teletype rom,
	am, eo, ul, xon,
	cols#87, it#8, lines#72,
	bel=^G, clear=^L, cr=^M, cub1=\ED, cud1=^J, cuf1=\EC,
	cup=\EY%p2%{32}%+%c%p1%{32}%+%c, cuu1=\EA,
	dch=\Ee%p1%{32}%+%c, dch1=\Ee!, dl=\EE%p1%{32}%+%c,
	dl1=\EE!, el=\EK, ht=^I, ich=\Ef%p1%{32}%+%c, ich1=\Ef!,
	il=\EF%p1%{32}%+%c, il1=\EF!, ind=^J, kbs=^H, kcub1=\ED,
	kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kf1=\Ex, kf2=\Ey, kf3=\Ez,

# (cbblit: here's a BSD termcap that says <cud1=\EG> -- esr)
cbblit|fixterm|blit running columbus code,
	cols#88,
	ed=\EJ, flash=\E^G, ich1@, mc4=^T, mc5=^R, mc5p=\EP%p1%03d,
	rmir=\ER, rmso=\EV!, rmul=\EV", smir=\EQ, smso=\EU!,
	smul=\EU", use=blit,

oblit|ojerq|first version of blit rom,
	am, da, db, eo, mir, ul, xon,
	cols#88, it#8, lines#72,
	bel=^G, clear=^L, cr=^M, cub1=\ED, cud1=^J, cuf1=\EC,
	cup=\EY%p2%{32}%+%c%p1%{32}%+%c, cuu1=\EA, dch1=\EO,
	dl=\Ee%p1%{32}%+%c, dl1=\EE, ed=\EJ, el=\EK, flash=\E^G,
	ht=^I, il=\Ef%p1%{32}%+%c, il1=\EF, ind=^J, kbs=^H, rmir=\ER,
	smir=\EQ,

#### Bolt, Beranek & Newman (bbn)
#
# The BitGraph was a product of the now-defunct BBN Computer Corporation.
# The parent company, best known as the architects of the Internet, is
# still around.
#
# Jeff DelPapa <dp@world.std.com> writes:
# The bitgraph was a large white box that contained a monochrome bitmap
# display, and a 68000 to run it.  You could download code and run it on
# the cpu, it had 128kb (I think) of memory.  I used one in the late
# 70's, sure beat a vt100.  It had one strange feature tho -- it used
# the cpu to bitblt pixels to scroll, it took longer than the refresh
# rate, and looked like a rubber sheet stretching, then snapping
# upwards.  It had everything the early mac had, except a floppy drive a
# small screen (it had a 17" crisp beauty) and a real OS. They (Bolt
# Beranek and Neuman) sold at most a few hundred of them to the real
# world.  DOD may have bought more...
#

# Entries for the BitGraph terminals.  The problem
# with scrolling in vi can only be fixed by getting BBN to put
# smarter scroll logic in the terminal or changing vi or padding
# scrolls with about 500 ms delay.
#
# I always thought the problem was related to the terminal
# counting newlines in its input buffer before scrolling and
# then moving the screen that much. Then vi comes along and
# paints lines in on the bottom line of the screen, so you get
# this big white gap.

bitgraph|bg2.0nv|bg3.10nv|bbn bitgraph 2.0 or later (normal video),
	flash=\E[?5h$<200/>\E[?5l, is2=\E>\E[?5l\E[?7h,
	use=bg2.0,
bg2.0rv|bg3.10rv|bbn bitgraph 2.0 (reverse video),
	flash=\E[?5l$<200/>\E[?5h, is2=\E>\E[?5h\E[?7h,
	use=bg2.0,
bg2.0|bg3.10|bbn bitgraph 2.0 or later (no init),
	OTbs, xenl,
	cols#85, lines#64,
	bel=^G, clear=\E[H\E[J$<150>, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=%i\E[%p1%d;%p2%dH, cuu1=\E[A, dl1=\E[M$<2*>,
	ed=\E[J$<150>, el=\E[K$<2>, ht=^I, il1=\E[L$<2*>,
	ind=\n$<280>, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, lf1=PF1,
	lf2=PF2, lf3=PF3, lf4=PF4, rc=\E8, rmkx=\E>, rmso=\E[m, sc=\E7,
	sgr0=\E[m, smkx=\E=, smso=\E[7m,

bg1.25rv|bbn bitgraph 1.25 (reverse video),
	flash=\E[?5l$<200/>\E[?5h, is2=\E>\E[?5h\E[?7h,
	use=bg1.25,
bg1.25nv|bbn bitgraph 1.25 (normal video),
	flash=\E[?5h$<200/>\E[?5l, is2=\E>\E[?5l\E[?7h,
	use=bg1.25,
# (bg1.25: I added <rmam>/<smam> based on the init string -- esr)
bg1.25|bbn bitgraph 1.25,
	cols#85, lines#64,
	bel=^G, clear=\E[H\E[J$<150>, cr=^M, cub1=^H, cud1=\E[B,
	cuf1=\E[C, cup=%i\E[%p1%d;%p2%dH, cuu1=\E[A,
	dl1=\E[M$<2*>, ed=\E[J$<150>, el=\E[K$<2>, ht=^I,
	il1=\E[L$<2*>, ind=\n$<280>, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, kf1=\EP, kf2=\EQ, kf3=\ER, kf4=\ES,
	lf1=PF1, lf2=PF2, lf3=PF3, lf4=PF4, ll=\E[64;1H, rmam=\E[?7l,
	rmkx=\E>, rmso=\E[m, sgr0=\E[m, smam=\E[?7h, smkx=\E=,
	smso=\E[7m,

#### Bull (bq, dku, vip)
#
# (Adapted for terminfo; AIX extension capabilities translated -- esr)

#============================================#
# BULL QUESTAR 210 `SDP' terminals emulation #
#============================================#
#
# Description written by R.K.Saunders (Bull Transac)
#
# Modifications written by F. Girard (Bull MTS)
#		19-05-87 V02.00.01
#		17-12-87 V02.00.02
#		15-09-89 V02.00.05
#
#	Typical technical selections F1 (modes SDP/ROLL):
# -------------------------------------------------------
# |   01   02   03   04   05   06   07   08   09   10   |
# |  1010 0011 1010 0110 0110 0001 0100 0000 0000 0000  |
# |                                                     |
# |   11   12   13   14   15   16   17   18   19   20   |
# |  0000 0110 100? 0000 0000 0000 0001 0000 0000 0001  |
# |                                                     |
# |   21   22   23   24   25   26   27   28   29   30   |
# |  0011 0000 0001 1000 0000 0000 0000 0000 0000 0000  |
# |                                                     |
# |   31   32   33   34   35   36   37   38   39   40   |
# |  1010 0011 0000 0000 0000 0000 0000 0000 0000 0000  |
# -------------------------------------------------------
#	Typical firmware identification F5 "etat 6":
#  P287.02.04b	(AZERTY)
#  P297.11.04	(24-pin: 2732)	or P798.11.04	(28-pin: 2764)
#  P298.03.03	(monochrome)	or P374.03.02	(colour)
#
#	SM SDP mode (VIP command):	^[[?=h
#	RIS (erases screen):		^[c
#	DMI disable keyboard:		^[`
#	SM double rendition mode:	^[[?>h
#	RM solicited status mode:	^[[5l
#	RM character mode:		^[[>l
#	RM echoplex mode:		^[[12l
#	RM column tab mode:		^[[18l
#	RM forbid SS2 keyboard mode:	^[[?<l
#	SM scroll mode:			^[[=h
#	FCF enable XON/XOFF:		^[P1s^[\
#	MTL select end msg character:	^[[^Wp
#	EMI enable keyboard:		^[b
#	RIS retour etat initial:	^[c
#	enable FC keypad:		^[[?<h,
#	MPW map status line window:	^[PY99:98^[\
#	SCP select status line:		^[[0;98v
#	ED erase entire partition:	^[[2J
#	SCP select main partition:	^[[v
#	SM character insertion mode:	^[[4h
#	RM character replacement mode:	^[[4l
#	COO cursor on:			^[[r
#	COO cursor off:			^[[1r
#	SGR dim (turquoise) rev attr:	^[[2;7m
#	SGR Data normal attr:		^[[m
#	SO Line-graphic mode ON:	^N
#	SI Line-graphic mode OFF:	^O
#	MC start routing to printer:	^[[5i
#	MC stop routing to printer:	^M^[[4i
#

# This entry covers the following terminals:
# dku7102, tws2102, and tws models 2105 to 2112
tws-generic|dku7102|Bull Questar tws terminals,
	am, eslok, hs, mir, msgr, xenl, xhp@, xon,
	cols#80, it#8, lines#24, wsl#80,
	acsc=``aaffggj)k\,l&m#n/ooppq*rrsst'u-v+w.x%yyzz{{||}}~~,
	bel=^G, blink=\E[0;5m, cbt=\E[Z, civis=\E[1r, clear=\E[2J,
	cnorm=\E[r, cr=^M, cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB,
	cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%df,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dim=\E[0;2m, dl=\E[%p1%dM, dl1=\E[M,
	dsl=\EPY99\:98\E\\\E[0;98v\E[2J\E[v, ed=\E[J, el=\E[K,
	fsl=\E[v, home=\E[H, ht=\E[I, hts=\EH, il=\E[%p1%dL,
	il1=\E[L, ind=^J, invis=\E[0;8m,
	is1=\E[?=h\Ec\E`\E[?>h\EPY99\:98\E\\,
	is2=\E[5;>;12;18;?<l\E[=h\EP1s\E\\\E[\027p,
	is3=\Eb\E[?<h, kbs=^H, kcbt=\E[Z, kctab=\E[g, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P, kdl1=\E[M,
	ked=\E[J, kel=\E[K, kf1=\E[1u\027, kf2=\E[2u\027,
	kf3=\E[3u\027, kf4=\E[4u\027, kf5=\E[5u\027,
	kf6=\E[6u\027, kf7=\E[7u\027, kf8=\E[8u\027, khome=\E[H,
	khts=\EH, kil1=\E[L, krmir=\E[4l, ll=\E[H\E[A, mc0=\E[0i,
	mc4=\r\E[4i, mc5=\E[5i, rev=\E[0;7m, rmacs=^O,
	rmcup=\E[0;98v\E[2J\E[v, rmir=\E[4l, rmso=\E[m,
	rmul=\E[m, rs2=\E[?=h\Ec, s0ds=^O, s1ds=^N,
	sgr=\E[0%?%p1%p3%|%t;7%;%?%p2%t;4%;%?%p4%t;5%;%?%p5%t;2%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	sgr0=\E[m\017, smacs=^N, smcup=\E[?>h\EPY99\:98\E\\,
	smir=\E[4h, smso=\E[0;7m, smul=\E[0;4m, tbc=\E[3g,
	tsl=\EPY99\:98\E\\\E[0;98v\E[2;7m,
tws2102-sna|dku7102-sna|BULL Questar tws2102 for SNA,
	dsl=\E[0;98v\E[2J\E[v, fsl=\E[v, is3=\Eb, tsl=\E[0;98v,
	use=tws-generic,
tws2103|xdku|BULL Questar tws2103,
	ht=^I, use=tws-generic,
tws2103-sna|dku7103-sna|BULL Questar tws2103 for SNA,
	ht=^I, use=tws2102-sna,
dku7102-old|BULL Questar 200 DKU7102 (microcode version < 6),
	clear=\E[2J\E[H, cup@, dl@, dl1@,
	dsl=\EPY99\:98\E\\\E[0;98v\E[2J\E[H\E[v, el=\E[K\E[m,
	il@, il1@, tsl=\EPY99\:98\E\\\E[0;98v\E[H\E[2;7m,
	use=tws-generic,
dku7202|BULL Questar 200 DKU7202 (colour/character attributes),
	blink=\E[0;2;4m, dim=\E[0;5m, ht=^I, is3=\E[?3h\Eb,
	sgr=\E[0%?%p1%t;2;4;5;7%;%?%p3%t;7%;%?%p2%t;2%;%?%p4%t;2;4%;%?%p5%t;5%;%?%p7%t;8%;m%?%p9%t\016%e\017%;,
	smso=\E[0;4;5;7m, smul=\E[0;2m, use=tws-generic,

#=========================================================#
# BULL QUESTAR 303 & 310 `DEC VT 320' terminals emulation #
#=========================================================#
#
# Description written by J. Staerck (BULL SA)
#       Copyright (c) 1989 BULL SA
#---------------------------------------------------------------------------
#  This entry is used for terminals with vt320 emulation mode
#  and following set-up :
#    8 bit ISO Latin Character Set (ISO 8859-1),
#    7 bit Control Characters,
#    80 columns screen.
#  Hereafter are some DEC vt terminals' commands. (valid on vt200 and 300)
#  They are used in string capabilities with vt220-320 emulation mode.
#  In the following DEC definitions, two kinds of terminfo databases are
#    provided :
#    1. the first with Command Sequence Introducer starting with escape
#       sequence in 7 bits characters ex. ESC [ : 2 chars. in 7-bit mode.
#    2. the second with Command Sequence Introducer starting with escape
#       sequence in 8 bits characters ex. ESC [ : 1 char. 'CSI' =x9B.
#	Soft Terminal Reset		esc [ ! p
#	RIS (erases screen):		esc c
#	DECKPNM numeric keypad mode:	esc >
#	DECKPAM applic. keypad mode:	esc =
#	DECSTBM Scrolling region:	esc [ r
#	SCS select G0 = US:		esc ( B
#	SCS select G1 = line-graphic:	esc ) 0
#	Select 7-bit C1 controls:	esc sp F
#	Select 8-bit C1 controls:	esc sp G
#	Select cursor home:		esc [  H
#	Select erase screen:		esc [  J
#	SM KAM lock keyboard:		esc [ 2 h
#	RM KAM unlock keyboard:		esc [ 2 l
#	SM SRM local echo off:		esc [ 1 2 h
#	RM SRM local echo on:		esc [ 1 2 l
#	SM LNM New line :		esc [ 2 0 h
#	RM LNM return = CR only:	esc [ 2 0 l
#	SM DECCKM cursor keys mode:	esc [ ? 1 h
#	RM DECCKM appli. keys mode:	esc [ ? 1 l
#	SM DECANM ANSI mode on:		esc [ ? 2 h
#	RM DECANM ANSI mode off:	esc [ ? 2 l
#	SM DECCOLM 132-column screen:	esc [ ? 3 h
#	RM DECCOLM 80-column screen:	esc [ ? 3 l
#	SM DECSCLM Smooth scroll:	esc [ ? 4 h
#	RM DECSCLM Jump scroll:		esc [ ? 4 l
#	SM DECSCNM screen light backgr.	esc [ ? 5 h
#	RM DECSCNM screen dark backgr.	esc [ ? 5 l
#	SM DECOM move within margins:	esc [ ? 6 h
#	RM DECOM move outside margins:	esc [ ? 6 l
#	SM DECAWM auto right margin:	esc [ ? 7 h
#	RM DECAWM auto right margin:	esc [ ? 7 l
#	SM DECARM auto repeat:		esc [ ? 8 h
#	RM DECARM auto repeat:		esc [ ? 8 l
#	DECSASD Select active main:	esc [ 0 $ }
#	DECSASD Select active status:	esc [ 1 $ }
#	DECSSDT Select status none:	esc [ 0 $ ~
#	DECSSDT Select status indic.:	esc [ 1 $ ~
#	DECSSDT Select status host-wr:	esc [ 2 $ ~
#	SM DECTCEM Visible cursor:	esc [ ? 2 5 h
#	RM DECTCEM Invisible cursor:	esc [ ? 2 5 l
#	SM DECNCRM 7 bits NCR set:	esc [ ? 4 2 h
#	RM DECNCRM Multi or ISO latin:	esc [ ? 4 2 l
#	SM DECNKM numeric keypad mode:	esc [ ? 6 6 h
#	RM DECNKM numeric keypad appl.:	esc [ ? 6 6 l
#	SM DECKBUM clavier informatique	esc [ ? 6 8 h
#	RM DECKBUM clavier bureautique:	esc [ ? 6 8 l
#	DECSCL vt300 mode 8-bit ctrl:	esc [ 6 3 " p
# or	DECSCL vt300 mode 8-bit ctrl:	esc [ 6 3 ; 0 " p
# or	DECSCL vt300 mode 8-bit ctrl:	esc [ 6 3 ; 2 " p
#	DECSCL vt300 mode 7-bit ctrl:	esc [ 6 3 ; 1 " p
#	Char. and Line attributes:	esc [ Ps ... Ps m
# with:  0 All off, 1 Bold, 4 Underline, 5 Blinking, 7 Reverse
# and : 22 Bold off, 24 Underline off, 25 Blinking off, 27 Reverse off
#

# This entry covers BQ303, BQ306, BQ310, Q303, Q306, Q310
bq300|Bull vt320 ISO Latin 1 80 columns terminal,
	am, eo, eslok, hs, km, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[H\E[J, cnorm=\E[?25h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=\E[D,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[P, dl=\E[%p1%dM, dl1=\E[M,
	dsl=\E[1$}\E[2$~\n\E[0$}, ech=\E[%p1%dX, ed=\E[J,
	el=\E[K, el1=\E[1K, enacs=\E(B\E)0,
	flash=\E[?5h$<50>\E[?5l, fsl=\E[0$}, home=\E[H, ht=^I,
	hts=\EH, ich=\E[%p1%d@, il=\E[%p1%dL, il1=\E[L, ind=\ED,
	is1=\E[63;1"p\E[2h,
	is2=\E[?2h\E[?3l\E[?5l\E[?7h\E[?8h\E>\E[?1l\E F\E[?42l\E[?4l,
	is3=\E[0$}\E[?25h\E[2l\E[H\E[J, ka1=\EOw, ka3=\EOy,
	kb2=\EOu, kbs=^H, kc1=\EOq, kc3=\EOs, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[3~, kf1=\EOP, kf10=\E[21~,
	kf11=\E[23~, kf12=\E[24~, kf13=\E[25~, kf14=\E[26~,
	kf15=\E[28~, kf16=\E[29~, kf17=\E[31~, kf18=\E[32~,
	kf19=\E[33~, kf2=\EOQ, kf20=\E[34~, kf3=\EOR, kf4=\EOS,
	kf6=\E[17~, kf7=\E[18~, kf8=\E[19~, kf9=\E[20~, kfnd=\E[1~,
	khlp=\E[28~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~,
	krdo=\E[29~, kslt=\E[4~, lf1=pf1, lf2=pf2, lf3=pf3, lf4=pf4,
	nel=\EE, rc=\E8, rev=\E[7m, ri=\EM, rmacs=\E(B, rmam=\E[?7l,
	rmcup=\E[?7h, rmir=\E[4l, rmkx=\E[?1l\E>, rmso=\E[27m,
	rmul=\E[24m, rs1=\E[!p, rs2=\E[?3l, s0ds=\E(B, s1ds=\E(0,
	sc=\E7,
	sgr=\E[%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;m%?%p9%t\E(0%e\E(B%;,
	sgr0=\E[0m\E(B, smacs=\E(0, smam=\E[?7h,
	smcup=\E[?7l\E[?1l\E(B, smir=\E[4h, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g, tsl=\E[1$}\E[2$~, use=ansi+pp,
bq300-rv|Bull vt320 reverse 80 columns,
	flash=\E[?5l$<50>\E[?5h,
	is2=\E[?2h\E[?3l\E[?5h\E[?7h\E[?8h\E>\E[?1l\E F\E[?42l\E[?4l,
	use=bq300,
bq300-w|Bull vt320 132 columns,
	cols#132, wsl#132,
	is2=\E[?2h\E[?3h\E[?5l\E[?7h\E[?8h\E>\E[?1l\E F\E[?42l\E[?4l,
	rs2=\E[?3h, use=bq300,
bq300-w-rv|Bull vt320 reverse mode 132 columns,
	cols#132, wsl#132,
	flash=\E[?5l$<50>\E[?5h,
	is2=\E[?2h\E[?3h\E[?5h\E[?7h\E[?8h\E>\E[?1l\E F\E[?42l\E[?4l,
	rs2=\E[?3h, use=bq300,

#  This entry is used for terminals with vt320 emulation mode
#  and following set-up :
#    8 bit ISO Latin Character Set (ISO 8859-1),
#    8 bit Control Characters, (CSI coded as x9B for ESC [)
#    80 columns screen.
#	Soft Terminal Reset		csi ! p
#	RIS (erases screen):		esc c
#	DECKPNM numeric keypad mode:	esc >
#	DECKPAM applic. keypad mode:	esc =
#	DECSTBM Scrolling region:	esc [ r
#	SCS select G0 = US:		esc ( B
#	SCS select G1 = line-graphic:	esc ) 0
#	Select 7-bit C1 controls:	esc sp F
#	Select 8-bit C1 controls:	esc sp G
#	Select cursor home:		csi H
#	Select erase screen:		csi J
#	SM KAM lock keyboard:		csi 2 h
#	RM KAM unlock keyboard:		csi 2 l
#	SM SRM local echo off:		csi 1 2 h
#	RM SRM local echo on:		csi 1 2 l
#	SM LNM New line :		csi 2 0 h
#	RM LNM return = CR only:	csi 2 0 l
#	SM DECCKM cursor keys mode:	csi ? 1 h
#	RM DECCKM appli. keys mode:	csi ? 1 l
#	SM DECANM ANSI mode on:		csi ? 2 h
#	RM DECANM ANSI mode off:	csi ? 2 l
#	SM DECCOLM 132-column screen:	csi ? 3 h
#	RM DECCOLM 80-column screen:	csi ? 3 l
#	SM DECSCLM Smooth scroll:	csi ? 4 h
#	RM DECSCLM Jump scroll:		csi ? 4 l
#	SM DECSCNM screen light backgr.	csi ? 5 h
#	RM DECSCNM screen dark backgr.	csi ? 5 l
#	SM DECOM move within margins:	csi ? 6 h
#	RM DECOM move outside margins:	csi ? 6 l
#	SM DECAWM auto right margin:	csi ? 7 h
#	RM DECAWM auto right margin:	csi ? 7 l
#	SM DECARM auto repeat:		csi ? 8 h
#	RM DECARM auto repeat:		csi ? 8 l
#	DECSASD Select active main:	csi 0 $ }
#	DECSASD Select active status:	csi 1 $ }
#	DECSSDT Select status none:	csi 0 $ ~
#	DECSSDT Select status indic.:	csi 1 $ ~
#	DECSSDT Select status host-wr:	csi 2 $ ~
#	SM DECTCEM Visible cursor:	csi ? 2 5 h
#	RM DECTCEM Invisible cursor:	csi ? 2 5 l
#	SM DECNCRM 7 bits NCR set:	csi ? 4 2 h
#	RM DECNCRM Multi or ISO latin:	csi ? 4 2 l
#	DECSCL vt300 mode 8-bit ctrl:	csi 6 3 " p
# or	DECSCL vt300 mode 8-bit ctrl:	csi 6 3 ; 0 " p
#	DECSCL vt300 mode 7-bit ctrl:	csi 6 3 ; 1 " p
#	Char. and Line attributes:	csi Ps ... Ps m
# with:  0 All off, 1 Bold, 4 Underline, 5 Blinking, 7 Reverse
# and : 22 Bold off, 24 Underline off, 25 Blinking off, 27 Reverse off
# (bq300-8: <cub1>,<cuf1>,<cuu1>,<cud1>,<dl1>,<il1> to get under 1024 --esr)
bq300-8|Bull vt320 full 8 bits 80 columns,
	am, eo, eslok, hs, km, mc5i, mir, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3, wsl#80,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\2335m, bold=\2331m, civis=\233?25l,
	clear=\233H\233J, cnorm=\233?25h, cr=^M,
	csr=\233%i%p1%d;%p2%dr, cub=\233%p1%dD, cub1=\2331D,
	cud=\233%p1%dB, cud1=\2331B, cuf=\233%p1%dC, cuf1=\2331C,
	cup=\233%i%p1%d;%p2%dH, cuu=\233%p1%dA, cuu1=\2331A,
	dch=\233%p1%dP, dch1=\233P, dl=\233%p1%dM, dl1=\233M,
	dsl=\2331$}\2332$~\n\2330$}, ech=\233%p1%dX, ed=\233J,
	el=\233K, el1=\2331K, enacs=\E(B\E)0,
	flash=\233?5h$<50>\233?5l, fsl=\2330$}, home=\233H,
	ht=^I, hts=\EH, ich=\233%p1%d@, il=\233%p1%dL, il1=\233L,
	ind=\ED, is1=\E[63;2"p\E[2h,
	is2=\E[?2h\E[?3l\E[?5l\E[?7h\E[?8h\E>\E[?1l\E G\E[?42l\E[?4l,
	is3=\2330$}\233?25h\2332l\233H\233J, ka1=\217w,
	ka3=\217y, kb2=\217u, kbs=^H, kc1=\217q, kc3=\217s,
	kcub1=\233D, kcud1=\233B, kcuf1=\233C, kcuu1=\233A,
	kdch1=\2333~, kf1=\217P, kf10=\23321~, kf11=\23323~,
	kf12=\23324~, kf13=\23325~, kf14=\23326~, kf15=\23328~,
	kf16=\23329~, kf17=\23331~, kf18=\23332~, kf19=\23333~,
	kf2=\217Q, kf20=\23334~, kf3=\217R, kf4=\217S, kf6=\23317~,
	kf7=\23318~, kf8=\23319~, kf9=\23320~, kfnd=\2331~,
	khlp=\23328~, kich1=\2332~, knp=\2336~, kpp=\2335~,
	krdo=\23329~, kslt=\2334~, lf1=pf1, lf2=pf2, lf3=pf3,
	lf4=pf4, mc0=\233i, mc4=\2334i, mc5=\2335i, nel=\EE, rc=\E8,
	rev=\2337m, ri=\EM, rmacs=\E(B, rmam=\233?7l,
	rmcup=\233?7h, rmir=\2334l, rmkx=\233?1l\E>,
	rmso=\23327m, rmul=\23324m, rs1=\E[!p, rs2=\E[?3l,
	s0ds=\E(B, s1ds=\E(0, sc=\E7,
	sgr=\233%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;m%?%p9%t\E(0%e\E(B%;,
	sgr0=\2330m\E(B, smacs=\E(0, smam=\233?7h,
	smcup=\233?7l\233?1l\E(B, smir=\2334h, smso=\2337m,
	smul=\2334m, tbc=\2333g, tsl=\2331$}\2332$~,
bq300-8rv|Bull vt320 8-bit reverse mode 80 columns,
	flash=\233?5l$<50>\233?5h,
	is2=\E[?2h\E[?3l\E[?5h\E[?7h\E[?8h\E>\E[?1l\E G\E[?42l\E[?4l,
	use=bq300-8,
bq300-8w|Bull vt320 8-bit 132 columns,
	cols#132, wsl#132,
	is2=\E[?2h\E[?3h\E[?5l\E[?7h\E[?8h\E>\E[?1l\E G\E[?42l\E[?4l,
	rs2=\233?3h, use=bq300-8,
bq300-w-8rv|Bull vt320 8-bit reverse mode 132 columns,
	cols#132, wsl#132,
	flash=\233?5l$<50>\233?5h,
	is2=\E[?2h\E[?3h\E[?5h\E[?7h\E[?8h\E>\E[?1l\E G\E[?42l\E[?4l,
	rs2=\233?3h, use=bq300-8,

#  This entry is used for terminals with vt320 emulation mode
#  a 102 keys keyboard (PC scancode !) and following set-up :
#    8 bit ISO Latin Character Set (ISO 8859-1),
#    7 bit Control Characters,
#    80 columns screen.
bq300-pc|Questar 303 with PC keyboard ISO Latin 1 80 columns,
	kbs=^H, kdch1=\E[3~, kend=\E[4~, kf1=\E[17~, kf10=\E[28~,
	kf11=\E[29~, kf12=\E[31~, kf13@, kf14@, kf15@, kf16@, kf17@,
	kf18@, kf19@, kf2=\E[18~, kf20@, kf3=\E[19~, kf4=\E[20~,
	kf5=\E[21~, kf6=\E[23~, kf7=\E[24~, kf8=\E[25~, kf9=\E[26~,
	kfnd@, khlp@, khome=\E[1~, kich1=\E[2~, knp=\E[6~, kpp=\E[5~,
	krdo@, kslt@, lf1@, lf2@, lf3@, lf4@, use=bq300,
bq300-pc-rv|Questar 303 with PC keyboard reverse mode 80 columns,
	flash=\E[?5l$<50>\E[?5h,
	is2=\E[?2h\E[?3l\E[?5h\E[?7h\E[?8h\E>\E[?1l\E F\E[?42l\E[?4l,
	use=bq300-pc,
bq300-pc-w|Questar 303 with PC keyboard 132 columns terminal,
	cols#132, wsl#132,
	is2=\E[?2h\E[?3h\E[?5l\E[?7h\E[?8h\E>\E[?1l\E F\E[?42l\E[?4l,
	rs2=\E[?3h, use=bq300-pc,
bq300-pc-w-rv|Questar 303 with PC keyboard reverse mode 132 columns,
	cols#132, wsl#132,
	flash=\E[?5l$<50>\E[?5h,
	is2=\E[?2h\E[?3h\E[?5h\E[?7h\E[?8h\E>\E[?1l\E F\E[?42l\E[?4l,
	rs2=\E[?3h, use=bq300-pc,
#    8 bit ISO Latin Character Set (ISO 8859-1),
#    8 bit Control Characters,
#    80 columns screen.
bq300-8-pc|Q306-8-pc|Questar 303 with PC keyboard in full 8 bits 80 columns,
	kbs=^H, kdch1=\2333~, kend=\2334~, kf1=\23317~,
	kf10=\23328~, kf11=\23329~, kf12=\23331~, kf13@, kf14@,
	kf15@, kf16@, kf17@, kf18@, kf19@, kf2=\23318~, kf20@,
	kf3=\23319~, kf4=\23320~, kf5=\23321~, kf6=\23323~,
	kf7=\23324~, kf8=\23325~, kf9=\23326~, kfnd@, khlp@,
	khome=\2331~, kich1=\2332~, knp=\2336~, kpp=\2335~, krdo@,
	kslt@, lf1@, lf2@, lf3@, lf4@, use=bq300-8,
bq300-8-pc-rv|Questar 303 with PC keyboard full 8 bits reverse mode 80 columns,
	flash=\E[?5l$<50>\E[?5h,
	is2=\E[?2h\E[?3l\E[?5h\E[?7h\E[?8h\E>\E[?1l\E G\E[?42l\E[?4l,
	use=bq300-8-pc,
bq300-8-pc-w|Questar 303 with PC keyboard full 8 bits 132 columns,
	cols#132, wsl#132,
	is2=\E[?2h\E[?3h\E[?5l\E[?7h\E[?8h\E>\E[?1l\E G\E[?42l\E[?4l,
	rs2=\E[?3h, use=bq300-8-pc,
bq300-8-pc-w-rv|Questar 303 with PC keyboard full 8 bits reverse 132 columns,
	cols#132, wsl#132,
	flash=\E[?5l$<50>\E[?5h,
	is2=\E[?2h\E[?3h\E[?5h\E[?7h\E[?8h\E>\E[?1l\E G\E[?42l\E[?4l,
	rs2=\E[?3h, use=bq300-8-pc,

#======================================================#
# BULL QUESTAR 310 `VIP 7800/8800' terminals emulation #
#======================================================#

# normal mode, 8 bits, 80 columns terminal.
#	RES reset :			^[e
#	RIS reset initial state:	^[c
#	BLE bell enable			^[h
#	BLD bell disable		^[g
#	CAMS char. attr. mode set	^[[D
#	CAMR char. attr. mode reset	^[[G
#	CLR clear			^[`
#	KBU keyboard unlock (set)	^[[W
#	KBL keyboard lock (reset)	^[[X
#	CM  character mode (async.) 	^[k
#	NEP non echoplex mode (by host)	^[l
#	EP  echoplex mode (by host) 	^[m
#	IM  insert mode set		^[[I
#	IM  insert mode reset 		^[[J
#	RMS roll mode set 		^[r
#	RMR roll mode reset 		^[q
#	SM78 set mode vip7800	 	^[[1q
#	SD  scroll up  	(72 lines) 	^[[0s
#	SD  scroll down	(72 lines) 	^[[1s
#	RBM block mode reset		^[[E
#	SLS status line set 		^[w
#	SLR status line reset 		^[v
#	SLL status line lock 		^[O
#	LGS Line-graphic mode set 	^[G
#	LGR Line-graphic mode reset 	^[F
#	TBC tab clear (at cursor pos.)	^[[g
#	TBI tab initialize 		^[[N
#	TBS tab set (at cursor pos.)	^[p
#	PDS  print data space		^[[0p
#	PHD  print host data 		^[[3p
#	PDT  print data terminator	^[[<p
#	PRES print adapter reset	^[[2p
#	SSPR multi-part. reset		^[[<>u
#	SSP0 partition 0 set		^[[00u
#	SSP1 partition n format 1 	^[[PnPnSTRINGu
#	SSP2 partition n format 2 	^[[PnPnSTRINGu
#	SSP3 partition n format 3 	^[[PnPnu
#	ATR attribute (visual)
#	    blink :			^[sB
#	    dim :			^[sL
#	    hide (blank) :		^[sH
#	    restore :			^[sR
#	    inverse video :		^[sI
#	    prot. :			^[sP
#	    underline :			^[s_
#	    reset :			^{
#
# This covers the vip7800 and BQ3155-vip7800
vip|Bull Questar 3155-7800,
	am, eslok, hs, km, mc5i, msgr, xenl, xon,
	cols#80, it#8, lines#24, vt#3, wsl#80,
	acsc=0pjdkblamcnkqitgufvhwexj, bel=^G, blink=\EsB,
	cbt=\E[Z, clear=\E`, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\E[%i%p1%03d%p2%03df, cuu1=\EA, dch1=\E[P, dim=\EsL,
	dl1=\E[M, dsl=\Ev, ed=\EJ, el=\EK,
	flash=\007$<80>\007$<80>\007, fsl=\EO, home=\EH, ht=^I,
	hts=\Ep, ich1=\E[I, ind=^J, invis=\EsH,
	is2=\E[00u\E[<>001001024080024080u\E[01u,
	is3=\Er\E[W\E`, kHOM=\EH, kLFT=\Eo, kRIT=\Eu, kbs=^H,
	kcbt=\E[Z, kclr=\E`, kctab=\E[g, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, kdch1=\E[P, kdl1=\E[M, ked=\EJ,
	kel=\EK, kf1=\E0, kf10=\ET, kf11=\E\\, kf12=\E\^, kf13@, kf14@,
	kf15@, kf16@, kf17@, kf18@, kf19@, kf2=\E2, kf20@, kf21=\E1,
	kf22=\E5, kf23=\E7, kf24=\E9, kf25=\E;, kf26=\E=, kf27=\E?,
	kf28=\EQ, kf29=\ES, kf3=\E6, kf30=\EV, kf31=\E], kf32=\E_,
	kf4=\E8, kf5=\E\:, kf6=\E<, kf7=\E>, kf8=\EP, kf9=\ER,
	khome=\EH, khts=\Ep, kich1=\E[I, kil1=\E[L, kind=\E[0s,
	kll=\EH\EA, kri=\E[1s, krmir=\E[J, ktbc=\E[N, lf1=pf1,
	lf2=pf2, lf3=pf3, lf4=pf4, ll=\EH\EA, mc0=\E[0p, mc4=\E[<p,
	mc5=\E[3p, nel=^M, prot=\EsP, rev=\EsI,
	ri=\EA\EJ\EH\E[L$<10>, rmacs=\EF, rmir=\E[J, rmso=\EsR,
	rmul=\EsR, rs1=\Ec, rs2=\E[G, s0ds=\EF, s1ds=\EG,
	sgr0=\EsR\EsU\EF, smacs=\EG, smir=\E[I, smso=\EsI,
	smul=\Es_, tbc=\E[N, tsl=\Ew,
# normal screen, 8 bits, 132 columns terminal.
vip-w|vip7800-w|Q310-vip-w|Q310-vip-w-am|Questar 3155-vip7800 wide,
	cols#132, wsl#132,
	is2=\E[00u\E[<>001001024132024132u\E[01u, use=vip,
vip-H|vip7800-H|Q310-vip-H|Q310-vip-H-am|Questar 3155-vip7800 72 lines,
	lines#72,
	is2=\E[00u\E[<>001001024080072080u\E[01u, use=vip,
vip-Hw|vip7800-Hw|Q310-vip-Hw|Questar 3155-vip7800 wide 72 lines,
	cols#132, lines#72, wsl#132,
	is2=\E[00u\E[<>001001024132072132u\E[01u, use=vip,

#### Chromatics
#

# I have put the long strings in <smcup>/<rmcup>. Ti sets up a window
# that is smaller than the screen, and puts up a warning message
# outside the window. Te erases the warning message, puts the
# window back to be the whole screen, and puts the cursor at just
# below the small window. I defined <cnorm> and <civis> to really turn
# the cursor on and off, but I have taken this out since I don't
# like the cursor being turned off when vi exits.
cg7900|chromatics|chromatics 7900,
	am,
	cols#80, lines#40,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=^],
	cup=\001M%p2%d\,%p1%d\,, cuu1=^K, dch1=^A<1, dl1=^A<2,
	ed=^Al, el=^A`, home=^\, ich1=^A>1, il1=^A>2, ind=^J, ll=^A|,
	rmcup=\001W0\,40\,85\,48\,\014\001W0\,0\,85\,48\,\001M0\,40\,,
	rmso=\001C1\,\001c2\,,
	smcup=\001P0\001O1\001R1\001C4\,\001c0\,\014\001M0\,42\,WARNING DOUBLE ENTER ESCAPE and \025\001C1\,\001c2\,\001W0\,0\,79\,39\,,
	smso=\001C4\,\001c7\,, uc=\001\001_\001\0,

#### Computer Automation
#

ca22851|computer automation 22851,
	am,
	cols#80, lines#24,
	bel=^G, clear=\014$<8>, cr=^M, cub1=^U, cud1=^J, cuf1=^I,
	cup=\002%i%p1%c%p2%c, cuu1=^V, ed=^\, el=^], home=^^, ind=^J,
	kcub1=^U, kcud1=^W, kcuu1=^V, khome=^^,

#### Cybernex
#

# This entry has correct padding and the undocumented "ri" capability
cyb83|xl83|cybernex xl-83,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=\014$<62>, cr=^M, cub1=^H, cud1=^J, cuf1=^I,
	cup=\027%p1%{32}%+%c%p2%{32}%+%c, cuu1=^N,
	ed=\020$<62>, el=\017$<3>, home=^K, ind=^J, kcub1=^H,
	kcud1=^J, kcuf1=^I, kcuu1=^N, ri=^N,
# (mdl110: removed obsolete ":ma=^Z^P:" and overridden ":cd=145^NA^W:" -- esr)
cyb110|mdl110|cybernex mdl-110,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=\030$<70>, cr=^M, cub1=^H, cud1=^J, cuf1=^U,
	cup=\020%p1%{32}%+%c%p2%{32}%+%c, cuu1=^Z,
	dch1=\016A\036$<3.5>, dl1=\016A\016\036$<40>,
	ed=\016@\026$<6>, el=\016@\026$<145>, home=^Y,
	ht=\011$<43>, ich1=\016A\035$<3.5>,
	il1=\016A\016\035$<65>, ind=^J, rmso=^NG, smso=^NF,

#### Datapoint
#
# Datapoint is gone.  They used to be headquartered in Texas.
# They created ARCnet, an Ethernet competitor that flourished for a while
# in the early 1980s before 3COM got wise and cut its prices.  The service
# side of Datapoint still lives (1995) in the form of Intelogic Trace.
#

dp3360|datapoint|datapoint 3360,
	OTbs, am,
	cols#82, lines#25,
	bel=^G, clear=^]^_, cr=^M, cub1=^H, cud1=^J, cuf1=^X, cuu1=^Z,
	ed=^_, el=^^, home=^], ind=^J,

# From: Jan Willem Stumpel <jw.stumpel@inter.nl.net>, 11 May 1997
# The Datapoint 8242 Workstation was sold at least between 1985
# and 1989. To make the terminal work with this entry, press
# CONTROL-INT-INT to take the terminal off-line, and type (opt).
# Set the options AUTO ROLL, ROLL DN, and ESC KBD on, and AUTO
# CR/LF off. Use control-shift-[] as escape key, control-I as tab,
# shift-F1 to shift-F5 as F6 to F10 (unshifted F1 to F5 are in
# fact unusable because the strings sent by the terminal conflict
# with other keys).
# The terminal is capable of displaying "box draw" characters.
# For each graphic character you must send 2 ESC's (\E\E) followed
# by a control character as follows:
#         character        meaning
#         =========        =======
#         ctrl-E           top tee
#         ctrl-F           right tee
#         ctrl-G           bottom tee
#         ctrl-H           left tee
#         ctrl-I           cross
#         ctrl-J           top left corner
#         ctrl-K           top right corner
#         ctrl-L           bottom left corner
#         ctrl-M           bottom right corner
#         ctrl-N           horizontal line
#         ctrl-O           vertical line
# Unfortunately this cannot be fitted into the termcap/terminfo
# description scheme.
dp8242|datapoint 8242,
	msgr,
	cols#80, lines#25,
	bel=^G, civis=^Y, clear=\025\E\004\027\030, cnorm=^X,
	cr=^M, cub1=^H, cud1=^J,
	cup=\011%p2%'\0'%+%c%p1%'\0'%+%c, dl1=\E^Z, ed=^W, el=^V,
	home=^U, ht=^I, il1=\E^T, ind=^C,
	is1=\E\014\E\016\0\230\0\317\025\027\030\E\004,
	kbs=^H, kcub1=^D, kcud1=^B, kcuf1=^F, kcuu1=^E, kf1=^G\Ee,
	kf10=\EK\Ea, kf2=^I\Ed, kf3=^J\Ec, kf4=^J\Eb, kf5=^S\Ea,
	kf6=\EO\Ee, kf7=\EN\Ed, kf8=\EM\Ec, kf9=\EL\Eb, nel=^M^J,
	rep=\E\023%p1%c%p2%c, ri=^K, rmso=\E^D, rmul=\E^D,
	rs1=\E\014\E\016\0\230\0\317\025\027\030\E\004,
	smso=\E^E, smul=\E^F,
	wind=\E\014\E\016%p1%'\0'%+%c%p2%'\0'%+%c%p3%'\0'%+%c%p4%'\0'%+%c\025,

#### DEC terminals (Obsolete types: DECwriter and vt40/42/50)
#
# These entries are DEC's official terminfos for its older terminals.
# Contact Bill Hedberg <hedberg@hannah.enet.dec.com> of Terminal Support
# Engineering for more information.  Updated terminfos and termcaps
# are kept available at ftp://gatekeeper.dec.com/pub/DEC/termcaps.
#

gt40|dec gt40,
	OTbs, os,
	cols#72, lines#30,
	bel=^G, cr=^M, cub1=^H, cud1=^J,
gt42|dec gt42,
	OTbs, os,
	cols#72, lines#40,
	bel=^G, cr=^M, cub1=^H, cud1=^J,
vt50|dec vt50,
	OTbs,
	cols#80, lines#12,
	bel=^G, clear=\EH\EJ, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cuu1=\EA, ed=\EJ, el=\EK, ht=^I, ind=^J,
vt50h|dec vt50h,
	OTbs,
	cols#80, lines#12,
	bel=^G, clear=\EH\EJ, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, ed=\EJ,
	el=\EK, ht=^I, ind=^J, ri=\EI,
# (vt61: there's a BSD termcap that claims <dl1=\EPd>, <il1=\EPf.> <kbs=^H>)
vt61|vt-61|vt61.5|dec vt61,
	cols#80, lines#24,
	bel=^G, clear=\EH\EJ$<120>, cr=\r$<20>, cub1=^H, cud1=^J,
	cuf1=\EC$<20>, cup=\EY%p1%{32}%+%c%p2%{32}%+%c$<20>,
	cuu1=\EA$<20>, ed=\EJ$<120>, el=\EK$<70>, ht=^I,
	ind=\n$<20>, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA,
	ri=\E$<20>I,

# The gigi does standout with red!
# (gigi: I added <rmam>/<smam> based on the init string, corrected cub1 -- esr)
gigi|vk100|dec gigi graphics terminal,
	OTbs, am, xenl,
	cols#84, lines#24,
	bel=^G, clear=\E[H\E[2J, cr=^M, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A, ed=\E[J,
	el=\E[K, ht=^I, ind=^J,
	is2=\E>\E[?3l\E[?4l\E[?5l\E[?20l\E[?7h\E[?8h,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kf1=\EOP,
	kf2=\EOQ, kf3=\EOR, kf4=\EOS, khome=\E[H, ri=\EM,
	rmam=\E[?7l, rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	sgr0=\E[m, smam=\E[?7h, smkx=\E[?1h\E=, smso=\E[7;31m,
	smul=\E[4m,

# DEC PRO-350 console (VT220-style).  The 350 was DEC's attempt to produce
# a PC differentiated from the IBM clones.  It was a total, ludicrous,
# grossly-overpriced failure (among other things, DEC's OS didn't include
# a format program, so you had to buy pre-formatted floppies from DEC at
# a hefty premium!).
pro350|decpro|dec pro console,
	OTbs,
	cols#80, it#8, lines#24,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	clear=\EH\EJ, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, ed=\EJ,
	el=\EK, home=\EH, ht=^I, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kf0=\EE, kf1=\EF, kf2=\EG, kf3=\EH, kf4=\EI,
	kf5=\EJ, kf6=\Ei, kf7=\Ej, khome=\EH, ri=\EI, rmacs=\EG,
	rmso=\E^N, rmul=\E^C, smacs=\EF, smso=\E^H, smul=\E^D,

dw1|decwriter I,
	OTbs, hc, os,
	cols#72,
	bel=^G, cr=^M, cub1=^H, cud1=^J, ind=^J,
dw2|decwriter|dw|decwriter II,
	OTbs, hc, os,
	cols#132,
	bel=^G, cr=^M, cub1=^H, cud1=^J, ind=^J, kbs=^H,
# \E(B		Use U.S. character set (otherwise # => british pound !)
# \E[20l	Disable "linefeed newline" mode (else puts \r after \n,\f,\v)
# \E[w   	10 char/in pitch
# \E[1;132	full width horizontal margins
# \E[2g		clear all tab stops
# \E[z		6 lines/in
# \E[66t	66 lines/page (for \f)
# \E[1;66r	full vertical page can be printed
# \E[4g		clear vertical tab stops
# \E>		disable alternate keypad mode (so it transmits numbers!)
# \E[%i%p1%du	set tab stop at column %d (origin == 1)
#		(Full syntax is \E[n;n;n;n;n;...;nu where each 'n' is
#		a tab stop)
#
#       The dw3 does standout with wide characters.
#
dw3|la120|decwriter III,
	OTbs, hc, os,
	cols#132,
	bel=^G, cr=^M, cub1=^H, cud1=^J, ht=^I, ind=^J,
	is1=\E(B\E[20l\E[w\E[0;132s\E[2g\E[z\E[66t\E[1;66r\E[4g\E>,
	is2=\E[9;17;25;33;41;49;57;65;73;81;89;97;105;113;121;129u\r,
	kbs=^H, rmso=\E[w, sgr0=\E[w, smso=\E[6w,
dw4|decwriter IV,
	OTbs, am, hc, os,
	cols#132,
	bel=^G, cr=^M, cub1=^H, cud1=^J, ht=^I, ind=^J, is2=\Ec, kbs=^H,
	kf0=\EOP, kf1=\EOQ, kf2=\EOR, kf3=\EOS,

# These aren't official
ln03|dec ln03 laser printer,
	hc,
	cols#80, lines#66,
	bel=^G, cr=^M, cud1=^J, hd=\EK, ht=^I, hu=\EL, ind=^J, nel=^M^J,
	rmso=\E[22m, rmul=\E[24m, sgr0=\E[m, smso=\E[1m,
	smul=\E[4m,
ln03-w|dec ln03 laser printer 132 cols,
	cols#132,
	bel=^G, cr=^M, cud1=^J, ht=^I, ind=^J, kbs=^H, kcub1=^H,
	kcud1=^J, nel=^M^J, use=ln03,

#### Delta Data (dd)
#

# Untested. The cup sequence is hairy enough that it probably needs work.
# The idea is ctrl(O), dd(row), dd(col), where dd(x) is x - 2*(x%16) + '9'.
# There are BSD-derived termcap entries floating around for this puppy
# that are *certainly* wrong.
delta|dd5000|delta data 5000,
	OTbs, am,
	cols#80, lines#27,
	bel=^G, clear=^NR, cub1=^H, cud1=^J, cuf1=^Y,
	cup=\017%p1%p1%{16}%m%{2}%*%-%{57}%+%c%p2%p2%{16}%m%{2}%*%-%{57}%+%c,
	cuu1=^Z, dch1=^NV, el=^NU, home=^NQ, ind=^J,

#### Digital Data Research (ddr)
#

# (ddr: I added <rmam>/<smam> based on the init string -- esr)
ddr|rebus3180|ddr3180|Rebus/DDR 3180 vt100 emulator,
	OTbs, am, xenl,
	cols#80, it#8, lines#24, vt#3,
	blink=\E[5m$<2/>, bold=\E[1m$<2/>,
	clear=\E[H\E[2J$<50/>, csr=\E[%i%p1%d;%p2%dr, cub1=^H,
	cud1=^J, cuf1=\E[C$<2/>, cup=\E[%i%p1%d;%p2%dH$<5/>,
	cuu1=\E[A$<2/>, ed=\E[J$<50/>, el=\E[K$<3/>, home=\E[H,
	ht=^I, ind=\ED$<5/>, is2=\E[1;24r\E[24;1H, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf1=\EOP,
	kf2=\EOQ, kf3=\EOR, kf4=\EOS, rc=\E8, rev=\E[7m$<2/>,
	rf=/usr/share/tabset/vt100, ri=\EM$<5/>, rmam=\E[7l,
	rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m$<2/>,
	rs1=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sc=\E7,
	sgr0=\E[m$<2/>, smam=\E[7l, smkx=\E[?1h\E=, smso=\E[7m,
	smul=\E[4m$<2/>,

#### Evans & Sutherland
#

# Jon Leech <leech@cs.unc.edu> tells us:
# The ps300 was the Evans & Sutherland Picture System 300, a high
# performance 3D vector graphics system with a bunch of specialized hardware.
# Approximate date of release was 1982 (early 80s, anyway), and it had several
# evolutions including (limited) color versions such as the PS330C. PS300s
# were effectively obsolete by the late 80s, replaced by raster graphics
# systems, although specialized applications like molecular modelling
# hung onto them for a while longer.  AFAIK all E&S vector graphics systems
# are out of production, though of course E&S is very much alive (in 1996).
# (ps300: changed ":pt@:" to "it@" -- esr)
#
ps300|Picture System 300,
	xt,
	it@,
	rmso@, rmul@, smso@, smul@, use=vt100,

#### General Electric (ge)
#

terminet1200|terminet300|tn1200|tn300|terminet|GE terminet 1200,
	OTbs, hc, os,
	cols#120,
	bel=^G, cr=^M, cud1=^J, ind=^J,

#### Heathkit/Zenith
#

# Here is a description of the H19 DIP switches:
#
# S401
# 0-3 = baud rate as follows:
#
#         3       2       1       0
#	---	---	---	---
#         0       0       1       1       300 baud
#         0       1       0       1       1200 baud
#         1       0       0       0       2400 baud
#         1       0       1       0       4800 baud
#         1       1       0       0       9600 baud
#         1       1       0       1       19.2K baud
#
# 4 = parity (0 = no parity)
# 5 = even parity (0 = odd parity)
# 6 = stick parity (0 = normal parity)
# 7 = full duplex (0 = half duplex)
#
# S402
# 0 = block cursor (0 = underscore cursor)
# 1 = no key click (0 = keyclick)
# 2 = wrap at end of line (0 = no wrap)
# 3 = auto LF on CR (0 = no LF on CR)
# 4 = auto CR on LF (0 = no CR on LF)
# 5 = ANSI mode (0 = VT52 mode)
# 6 = keypad shifted (0 = keypad unshifted)
# 7 = 50Hz refresh (1 = 60Hz refresh)
#
# Factory Default settings are as follows:
#          7 6 5 4 3 2 1 0
# S401     1 0 0 0 1 1 0 0
# S402     0 0 0 0 0 0 0 0
# (h19: I added <rmam>/<smam> based on the init string;
# also added empty <acsc> to suppress a tic warning -- esr)
h19-a|h19a|heath-ansi|heathkit-a|heathkit h19 ansi mode,
	OTbs, am, mir, msgr,
	cols#80, it#8, lines#24,
	acsc=, bel=^G, clear=\E[2J, cnorm=\E[>4l, cr=^M, cub1=^H,
	cud1=\E[1B, cuf1=\E[1C, cup=\E[%i%p1%d;%p2%dH,
	cuu1=\E[1A, cvvis=\E[>4h, dch1=\E[1P, dl1=\E[1M$<1*>,
	ed=\E[J, el=\E[K, home=\E[H, ht=^I, il1=\E[1L$<1*>, ind=^J,
	is2=\E<\E[>1;2;3;4;5;6;7;8;9l\E[m\E[11m\E[?7h,
	kbs=^H, kcub1=\E[1D, kcud1=\E[1B, kcuf1=\E[1C, kcuu1=\E[1A,
	kf1=\EOS, kf2=\EOT, kf3=\EOU, kf4=\EOV, kf5=\EOW, kf6=\EOP,
	kf7=\EOQ, kf8=\EOR, khome=\E[H, lf6=blue, lf7=red, lf8=white,
	ri=\EM, rmacs=\E[11m, rmam=\E[?7l, rmir=\E[4l, rmso=\E[m,
	smacs=\E[10m, smam=\E[?7h, smir=\E[4h, smso=\E[7m,
h19-bs|heathkit w/keypad shifted,
	rmkx=\Eu, smkx=\Et, use=h19-b,
h19-us|h19us|h19-smul|heathkit w/keypad shifted/underscore cursor,
	rmkx=\Eu, smkx=\Et, use=h19-u,
# (h19: merged in <ip> from BSDI hp19-e entry>;
# also added empty <acsc> to suppress a tic warning --esr)
# From: Tim Pierce <twp@skepsis.com>, 23 Feb 1998
# Tim tells us that:
# I have an old Zenith-19 terminal at home that still gets a lot of use.
# This terminal suffers from the same famous insert-mode padding lossage
# that has been acknowledged for the Z29 terminal.  Emacs is nearly
# unusable on this box, since even a half-scroll up or down the window
# causes flaming terminal death.
#
# On the Z19, the only way I have found around this problem is to remove
# the :al: and :dl: entries entirely.  No amount of extra padding will
# help (I have tried up to 20000).  Removing <il1=\EL$> and <dl1=\EM$>
# makes Emacs a little slower, but it remains in the land of the living.
# Big win.
h19|heath|h19-b|heathkit|heath-19|z19|zenith|heathkit h19,
	OTbs, am, eslok, hs, mir, msgr,
	cols#80, it#8, lines#24,
	acsc=+h.kaiggjdkclfmenbozqas{tvutvuwsx`~\^, bel=^G,
	clear=\EE, cnorm=\Ey4, cr=^M, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, cvvis=\Ex4,
	dch1=\EN, ed=\EJ, el=\EK, fsl=\Ek\Ey5, home=\EH, ht=^I, ind=^J,
	ip=<1.5/>, kbs=^H, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kf1=\ES, kf2=\ET, kf3=\EU, kf4=\EV, kf5=\EW,
	kf6=\EP, kf7=\EQ, kf8=\ER, khome=\EH, lf6=blue, lf7=red,
	lf8=white, ri=\EI, rmacs=\EG, rmir=\EO, rmso=\Eq, smacs=\EF,
	smir=\E@, smso=\Ep, tsl=\Ej\Ex5\EY8%p1%{32}%+%c\Eo\Eo,
h19-u|heathkit with underscore cursor,
	cnorm@, cvvis@, use=h19-b,
h19-g|h19g|heathkit w/block cursor,
	cnorm=\Ex4, cvvis@, use=h19-b,
alto-h19|altoh19|altoheath|alto-heath|alto emulating heathkit h19,
	lines#60,
	dl1=\EM, il1=\EL, use=h19,

# The major problem with the Z29 is that it requires more padding than the Z19.
#
# The problem with declaring an H19 to be synonymous with a Z29 is that
# it needs more padding. It especially loses if a program attempts
# to put the Z29 into insert mode and insert text at 9600 baud. It
# even loses worse if the program attempts to insert tabs at 9600
# baud. Adding padding to text that is inserted loses because in
# order to make the Z29 not die, one must add so much padding that
# whenever the program tries to use insert mode, the effective
# rate is about 110 baud.
#
# What program would want to put the terminal into insert mode
# and shove stuff at it at 9600 baud you ask?
#
# Emacs. Emacs seems to want to do the mathematically optimal
# thing in doing a redisplay rather than the practical thing.
# When it is about to output a line on top of a line that is
# already on the screen, instead of just killing to the end of
# the line and outputting the new line, it compares the old line
# and the new line and if there are any similarities, it
# constructs the new line by deleting the text on the old line
# on the terminal that is already there and then inserting new
# text into the line to transform it into the new line that is
# to be displayed. The Z29 does not react kindly to this.
#
# But don't cry for too long.... There is a solution. You can make
# a termcap entry for the Z29 that says the Z29 has no insert mode.
# Then Emacs cannot use it. "Oh, no, but now inserting into a
# line will be really slow", you say. Well there is a sort of a
# solution to that too. There is an insert character option on
# the Z29 that will insert one character. Unfortunately, it
# involves putting the terminal into ansi mode, inserting the
# character, and changing it back to H19 mode. All this takes 12
# characters. Pretty expensive to insert one character, but it
# works. Either Emacs doesn't try to use its inserting hack when
# it's only given an insert character ability or the Z29 doesn't
# require padding with this (the former is probably more likely,
# but I haven't checked it out).
# (z29: added empty <acsc> to suppress a tic warning, merged in
# status line capabilities from BRL entry --esr)
z29|zenith29|z29b|zenith z29b,
	OTbs, OTpt, am, eslok, hs, mir, msgr,
	OTkn#10, cols#80, lines#24,
	OTbc=\ED, acsc=, bel=^G, cbt=\E-, clear=\EE$<14>, cnorm=\Ey4,
	cr=^M, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\E$<1>A,
	cvvis=\Ex4, dch1=\EN$<0.1*>, dl1=\EM$<1/>, dsl=\Ey1,
	ed=\EJ$<14>, el=\EK$<1>, fsl=\Ek\Ey5, home=\EH, ht=^I,
	ich1=\E<\E[1@\E[?2h$<1>, il1=\EL$<1/>, ind=\n$<2>,
	is2=\E<\E[?2h\Ev, kbs=^H, kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kf0=\E~, kf1=\ES, kf2=\ET, kf3=\EU, kf4=\EV,
	kf5=\EW, kf6=\EP, kf7=\EQ, kf8=\ER, kf9=\E0I, khome=\EH,
	lf0=home, ri=\EI$<2/>, rmacs=\EF, rmir=\EO, rmso=\Eq,
	rmul=\Es0, smacs=\EG, smir=\E@, smso=\Ep, smul=\Es8,
	tsl=\Ej\Ex5\Ex1\EY8%+ \Eo,
# z29 in ansi mode. Assumes that the cursor is in the correct state, and that
# the world is stable. <rs1> causes the terminal to be reset to the state
# indicated by the name. kc -> key click, nkc -> no key click, uc -> underscore
# cursor, bc -> block cursor.
# From: Mike Meyers
# (z29a: replaced nonexistent <if=/usr/share/tabset/zenith29> befause <hts>
# looks vt100-compatible -- esr)
z29a|z29a-kc-bc|h29a-kc-bc|heath/zenith 29 in ansi mode,
	OTbs, OTpt, am, eslok, hs, mir, msgr,
	OTkn#10, cols#80, it#8, lines#24,
	OTbc=\ED, bel=^G, blink=\E[5m, bold=\E[2m, clear=\E[2J,
	cr=^M, csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[1P, dim=\E[2m, dl=\E[%p1%dM,
	dl1=\E[M, dsl=\E[>1l, ed=\E[J, el=\E[K, fsl=\E[u\E[>5l,
	home=\E[H, ht=^I, hts=\EH, if=/usr/share/tabset/vt100,
	il=\E[%p1%dL, il1=\E[L, ind=\ED, kbs=^H, kclr=\E[J,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, ked=\E[J,
	kf0=\E[~, kf1=\EOS, kf2=\EOT, kf3=\EOU, kf4=\EOV, kf5=\EOW,
	kf6=\EOP, kf7=\EOQ, kf8=\EOR, kf9=\EOX, khome=\E[H, lf0=help,
	mc0=\E#7, nel=^M\ED, rc=\E[r, rev=\E[7m, ri=\EM,
	rmcup=\E[?7h, rmso=\E[m, rmul=\E[m,
	rs1=\E<\E[1;24r\E[24;1H\E[?7h\E[>4h\E[>1;2;3;5;6;7;8;9l\E[m\E[11m,
	sc=\E[s, sgr0=\E[m, smcup=\E[?7l, smso=\E[7;2m, smul=\E[4m,
	tbc=\E[3g, tsl=\E[s\E[>5;1h\E[25;%i%dH\E[1K,
z29a-kc-uc|h29a-kc-uc|z29 ansi mode with keyckick and underscore cursor,
	rs1=\E<\E[1;24r\E[24;1H\E[?7h\E[>1;2;3;4;5;6;7;8;9l\E[m\E[11m,
	use=z29a,
z29a-nkc-bc|h29a-nkc-bc|z29 ansi mode with block cursor and no keyclick,
	rs1=\E<\E[1;24r\E[24;1H\E[?7h\E[>2;4h\E[>1;3;5;6;7;8;9l\E[m\E[11m,
	use=z29a,
z29a-nkc-uc|h29a-nkc-uc|z29 ansi mode with underscore cursor and no keyclick,
	rs1=\E<\E[1;24r\E[24;1H\E[?7h\E[>2h\E[>1;3;4;5;6;7;8;9l\E[m\E[11m,
	use=z29a,
# From: Jeff Bartig <jeffb@dont.doit.wisc.edu> 31 Mar 1995
z39-a|z39a|zenith39-a|zenith39-ansi|Zenith 39 in ANSI mode,
	am, eslok, hs, mc5i, mir, msgr, xon,
	cols#80, lines#24,
	acsc=0a``aaffggjjkkllmmnnooqqssttuuvvwwxx~~, bel=^G,
	blink=\E[5m, bold=\E[1m, cbt=\E[1Z, civis=\E[>5h,
	clear=\E[2J\E[H, cnorm=\E[>5l, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\E[A,
	dch=\E[%p1%dP, dch1=\E[1P, dim=\E[2m, dl=\E[%p1%dM,
	dl1=\E[1M, dsl=\E[>1l, ed=\E[0J, el=\E[0K, el1=\E[1K,
	fsl=\E[u, home=\E[H, ht=^I, hts=\EH, il=\E[%p1%dL, il1=\E[1L,
	ind=^J, is2=\E<\E[>1;3;5;6;7l\E[0m\E[2J, ka1=\EOw,
	ka3=\EOu, kb2=\EOy, kbs=^H, kc1=\EOq, kc3=\EOs, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, ked=\E[J, kf1=\EOS,
	kf2=\EOT, kf3=\EOU, kf4=\EOV, kf5=\EOW, kf6=\EOP, kf7=\EOQ,
	kf8=\EOR, kf9=\EOX, khlp=\E[~, khome=\E[H, ll=\E[24;1H,
	mc0=\E[?19h\E[i, mc4=\E[4i, mc5=\E[5i, rc=\E[u, rev=\E[7m,
	rmacs=\E(B, rmir=\E[4l, rmkx=\E[>7l, rmso=\E[0m,
	rmul=\E[0m, rs2=\E<\Ec\0, sc=\E[s, sgr0=\E[0m, smacs=\E(0,
	smir=\E[4h, smkx=\E[>7h, smso=\E[7m, smul=\E[4m, tbc=\E[3g,
	tsl=\E[s\E[>1h\E[25;%i%p1%dH,

# From: Brad Brahms <Brahms@USC-ECLC>
z100|h100|z110|z-100|h-100|heath/zenith z-100 pc with color monitor,
	cnorm=\Ey4\Em70, cvvis=\Ex4\Em71, use=z100bw,
# (z100bw: removed obsolete ":kn#10:", added empty <acsc> -- esr)
z100bw|h100bw|z110bw|z-100bw|h-100bw|heath/zenith z-100 pc,
	OTbs, OTpt, mir, msgr,
	OTkn#10, cols#80, it#8, lines#24,
	acsc=+h.kaiggjdkclfmenbozqas{tvutvuwsx`~\^,
	clear=\EE$<5*/>, cnorm=\Ey4, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c$<1*/>, cuu1=\EA,
	cvvis=\Ex4, dch1=\EN$<1*/>, dl1=\EM$<5*/>, ed=\EJ, el=\EK,
	home=\EH, ht=^I, il1=\EL$<5*/>, kbs=^H, kcub1=\ED, kcud1=\EB,
	kcuf1=\EC, kcuu1=\EA, kf0=\EJ, kf1=\ES, kf2=\ET, kf3=\EU,
	kf4=\EV, kf5=\EW, kf6=\EP, kf7=\EQ, kf8=\ER, kf9=\EOI,
	khome=\EH, ri=\EI, rmacs=\EG, rmir=\EO, rmso=\Eq, smacs=\EF,
	smir=\E@, smso=\Ep,
p19|h19-b with il1/dl1,
	dl1=\EM$<2*/>, il1=\EL$<2*/>, use=h19-b,
# From: <ucscc!B.fiatlux@ucbvax.berkeley.edu>
# (ztx: removed duplicate :sr: -- esr)
ztx|ztx11|zt-1|htx11|ztx-1-a|ztx-10 or 11,
	OTbs, am, eslok, hs,
	cols#80, it#8, lines#24,
	clear=\EE, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dl1=\EM,
	dsl=\Ey1, ed=\EJ, el=\EK, fsl=\Ek\Ey5, home=\EH, ht=^I,
	il1=\EL, is2=\Ej\EH\Eq\Ek\Ev\Ey1\Ey5\EG\Ey8\Ey9\Ey>,
	kbs=^H, kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kf0=\ES,
	kf1=\EB, kf2=\EU, kf3=\EV, kf4=\EW, kf5=\EP, kf6=\EQ, kf7=\ER,
	ri=\EI, rmso=\Eq, rmul=\Eq, smso=\Es5, smul=\Es2,
	tsl=\Ej\Ex5\Ex1\EY8%+ \Eo,

#### IMS International (ims)
#
# There was a company called IMS International located in Carson City,
# Nevada, that flourished from the mid-70s to mid-80s.  They made S-100
# bus/Z80 hardware and a line of terminals called Ultimas.
#

# From: Erik Fair <fair@ucbarpa.berkeley.edu>  Sun Oct 27 07:21:05 1985
ims950-b|bare ims950 no init string,
	is2@, use=ims950,
# (ims950: removed obsolete ":ko@:" -- esr)
ims950|ims televideo 950 emulation,
	xenl@,
	flash@, kbs@, kcub1@, kcud1@, kcuf1@, kcuu1@, kf0@, kf1@, kf2@, kf3@,
	kf4@, kf5@, kf6@, kf7@, kf8@, kf9@, khome@, use=tvi950,
# (ims950-rv: removed obsolete ":ko@:" -- esr)
ims950-rv|ims tvi950 rev video,
	xenl@,
	flash@, kbs@, kcub1@, kcud1@, kcuf1@, kcuu1@, kf0@, kf1@, kf2@, kf3@,
	kf4@, kf5@, kf6@, kf7@, kf8@, kf9@, khome@, use=tvi950-rv,
ims-ansi|ultima2|ultimaII|IMS Ultima II,
	OTbs, am,
	cols#80, it#8, lines#24,
	clear=\E[H\E[2J, cub1=^H, cud1=\ED, cuf1=\EC,
	cup=\E[%i%p1%2d;%p2%2dH, cuu1=\EM, ed=\E[0J, el=\E[0K,
	ht=^I, if=/usr/share/tabset/vt100,
	is2=\E[m\E[>14l\E[?1;?5;20l\E>\E[1m\r, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, khome=\E[H, ri=\EM,
	rmso=\E[m\E[1m, rmul=\E[m\E[1m, sgr0=\E[m, smso=\E[7m,
	smul=\E[4m,

#### Intertec Data Systems
#
# I think this company is long dead as of 1995.  They made an early CP/M
# micro called the "Intertec Superbrain" that was moderately popular,
# then sank out of sight.
#

superbrain|intertec superbrain,
	OTbs, am, bw,
	cols#80, lines#24,
	OTbc=^U, bel=^G, clear=\014$<5*>, cr=^M, cub1=^H, cud1=^J,
	cuf1=^F, cup=\EY%p1%{32}%+%c%p2%{32}%+%c$<20>, cuu1=^K,
	ed=\E~k<10*>, el=\E~K$<15>, ht=^I, ind=^J, kcub1=^U,
	kcud1=^J, kcuf1=^F, kcuu1=^K, rmcup=^L, smcup=^L,
# (intertube: a Gould entry via BRL asserted smul=\E0@$<200/>,
# rmul=\E0A$<200/>; my guess is the highlight letter is bit-coded like an ADM,
# and the reverse is actually true.  Try it. -- esr)
intertube|intertec|Intertec InterTube,
	OTbs, am,
	cols#80, lines#25,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=^F,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c$<50>, cuu1=^Z, home=^A,
	ind=^J, rmso=\E0@, smso=\E0P,
# The intertube 2 has the "full duplex" problem like the tek 4025: if you
# are typing and a command comes in, the keystrokes you type get interspersed
# with the command and it messes up
intertube2|intertec data systems intertube 2,
	OTbs,
	cup=\016%p1%c\020%p2%{10}%/%{16}%*%p2%{10}%m%+%c,
	el=\EK, hpa=\020%p1%{10}%/%{16}%*%p1%{10}%m%+%c,
	ll=^K^X\r, vpa=\013%p1%c, use=intertube,

#### Ithaca Intersystems
#
# This company made S100-bus personal computers long ago in the pre-IBM-PC
# past.  They used to be reachable at:
#
#	Ithaca Intersystems
#	1650 Hanshaw Road
#	Ithaca, New York 14850
#
# However, the outfit went bankrupt years ago.
#

# The Graphos III was a color graphics terminal from Ithaca Intersystems.
# These entries were written (originally in termcap syntax) by Brian Yandell
# <yandell@stat.wisc.edu> and Mike Meyer <mikem@stat.wisc.edu> at the
# University of Wisconsin.

# (graphos: removed obsolete and syntactically incorrect :kn=4:,
# removed <if=/usr/share/tabset/init.graphos> and
# <rf=/usr/share/tabset/init.graphos> no such file & no <hts> -- esr)
graphos|graphos III,
	am, mir,
	cols#80, it#8, lines#24,
	clear=\E[H\E[2J, cnorm=\Ez56;2;0;0z\Ez73z\Ez4;1;1z,
	cr=^M, cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=\E[B,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A,
	cvvis=\Ez4;2;1z\Ez56;2;80;24z, dch1=\E[P, dl=\E[%p1%dM,
	dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I, il=\E[%p1%dL,
	il1=\E[L, ind=\ED, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kf1=\EOP, kf2=\EOQ, kf3=\EOR,
	kf4=\EOS, khome=\E[H, nel=^M\ED, rc=\E8, ri=\EM, rmdc=\E[4l,
	rmir=\E[4l, rmso=\E[m, sc=\E7, sgr0=\E[m, smdc=\E[4h,
	smir=\E[4h, smso=\E[7m,
graphos-30|graphos III with 30 lines,
	lines#30,
	cvvis=\Ez4;2;1z\Ez56;2;80;30z, use=graphos,

#### Modgraph
#
# These people used to be reachable at:
#
#	Modgraph, Inc
#	1393 Main Street,
#	Waltham, MA 02154
#	Vox: (617)-890-5796.
#
# However, if you call that number today you'll get an insurance company.
# I have mail from "Michael Berman, V.P. Sales, Modgraph" dated
# 26 Feb 1997 that says:
#
# Modgraph GX-1000, replaced by GX-2000.  Both are out of production, have been
# for ~7 years.  Modgraph still in business.  Products are rugged laptop and
# portable PC's and specialized CRT and LCD monitors (rugged, rack-mount
# panel-mount etc).  I can be emailed at sonfour@aol.com
#
# Peter D. Smith <pdsmith@nbbn.com> notes that his modgraph manual was
# dated 1984.  According to the manual, it featured Tek 4010/4014
# graphics and DEC VT100/VT52 + ADM-3A emulation with a VT220-style keyboard.
#

modgraph|mod24|modgraph terminal emulating vt100,
	xenl@,
	cols#80, lines#24,
	cvvis=\E\^9;0s\E\^7;1s,
	is2=\E\^9;0s\E\^7;1s\E[3g\E\^11;9s\E\^11;17s\E\^11;25s\E\^11;33s\E\^11;41s\E\^11;49s\E\^11;57s\E\^11;65s\E\^11;73s\E\^11;81s\E\^11;89s,
	rf@, ri=\EM\E[K$<5/>, use=vt100,
# The GX-1000 manual is dated 1984.  This looks rather like a VT-52.
modgraph2|modgraph gx-1000 80x24 with keypad not enabled,
	am, da, db,
	cols#80, it#8, lines#24,
	clear=\EH\EJ$<50/>, cub1=^H, cud1=\EB$<2/>,
	cuf1=\EC$<2/>, cup=\EY%p1%{32}%+%c%p2%{32}%+%c$<5/>,
	cuu1=\EA$<2/>, ed=\EJ$<50/>, el=\EK$<3/>, ht=^I,
	is2=\E<\E\^5;2s\E\^7;1s\E[3g\E\^11;9s\E\^11;17s\E\^11;25s\E\^11;33s\E\^11;41s\E\^11;49s\E\^11;57s\E\^11;65s\E\^11;73s\E\^11;81s\E\^11;89s\E\^12;0s\E\^14;2s\E\^15;9s\E\^25;1s\E\^9;1s\E\^27;1,
	ri=\EI$<5/>,
#
# Modgraph from Nancy L. Cider <nancyc@brl-tbd>
# BUG NOTE from Barbara E. Ringers <barb@brl-tbd>:
# If we set TERM=vt100, and set the Modgraph screen to 24 lines, setting a
# mark and using delete-to-killbuffer work correctly.  However, we would
# like normal mode of operation to be using a Modgraph with 48 line setting.
# If we set TERM=mod (which is a valid entry in termcap with 48 lines)
# the setting mark and delete-to-killbuffer results in the deletion of only
# the line the mark is set on.
# We've discovered that the delete-to-killbuffer works correctly
# with TERM=mod and screen set to 80x48 but it's not obvious.  Only
# the first line disappears but a ctrl-l shows that it did work
# correctly.
modgraph48|mod|Modgraph w/48 lines,
	OTbs, OTpt, am, xenl,
	cols#80, it#8, lines#48, vt#3,
	OTnl=^J, bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[;H\E[2J,
	cr=^M, csr=\E[%i%p1%d;%p2%dr, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, ed=\E[J, el=\E[K,
	flash=\E[?5h\E[0q\E[1;2q\E[?5l\E[0q\E[4;3q,
	home=\E[H, ht=^I, is2=\E<\E[1;48r\E[0q\E[3;4q\E=\E[?1h,
	kbs=^H, kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS, rc=\E8, rev=\E[7m,
	ri=\EM, rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	rs1=\E=\E[0q\E>, sc=\E7, sgr0=\E[m, smkx=\E[?1h\E=,
	smso=\E[7m, smul=\E[4m,

#### Morrow Designs
#
# This was George Morrow's company.  They started in the late 1970s making
# S100-bus machines.  They used to be reachable at:
#
#        Morrow
#        600 McCormick St.
#        San Leandro, CA 94577
#
# but they're long gone now (1995).
#

# The mt70 terminal was shipped with the Morrow MD-3 microcomputer.
# Jeff's specimen was dated June 1984.
# From: Jeff Wieland <wieland@acn.purdue.edu> 24 Feb 1995
mt70|mt-70|Morrow MD-70; native Morrow mode,
	am, mir, msgr, xon,
	cols#80, it#8, lines#24,
	acsc=+z\,{-x.yOi`|jGkFlEmDnHqJtLuKvNwMxI, bel=^G,
	cbt=\EI, civis=\E"0, clear=^Z, cnorm=\E"2, cr=^M, cub1=^H,
	cud1=^J, cuf1=^L, cup=\E=%p1%{32}%+%c%p2%{32}%+%c$<1>,
	cuu1=^K, dch1=\EW, dim=\EG2, dl1=\ER, ed=\EY, el=\ET$<10>,
	flash=\EK1$<200>\EK0, home=^^, ht=^I, ich1=\EQ, il1=\EE,
	ind=^J, invis@, is1=\E"2\EG0\E], kbs=^H, kcbt=^A^Z\r,
	kclr=^An\r, kcub1=^AL\r, kcud1=^AK\r, kcuf1=^AM\r,
	kcuu1=^AJ\r, kdch1=\177, kf1=^A@\r, kf10=^AI\r, kf11=^A`\r,
	kf12=^Aa\r, kf13=^Ab\r, kf14=^Ac\r, kf15=^Ad\r, kf16=^Ae\r,
	kf17=^Af\r, kf18=^Ag\r, kf19=^Ah\r, kf2=^AA\r, kf20=^Ai\r,
	kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r,
	kf8=^AG\r, kf9=^AH\r, khlp=^AO\r, khome=^AN\r, nel=^_,
	rmacs=\E%%, rmcup=, smacs=\E$, smcup=\E"2\EG0\E],
	smul=\EG1, tbc=\E0, use=adm+sgr,

#### Motorola
#

# Motorola EXORterm 155	from {decvax, ihnp4}!philabs!sbcs!megad!seth via BRL
# (Seth H Zirin)
ex155|Motorola Exorterm 155,
	OTbs, am, bw,
	OTkn#5, OTug#1, cols#80, lines#24,
	cbt=\E[, clear=\EX, cub1=\ED, cud1=\EB, cuf1=\EC,
	cup=\EE%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, ed=\ET,
	el=\EU, home=\E@, ht=\EZ, kbs=^H, kcbt=\E[, kclr=\EX, kcub1=^H,
	kcud1=^J, kcuf1=^L, kcuu1=^K, ked=\ET, kel=\EU, khome=\E@,
	rmso=\Ec\ED, rmul=\Eg\ED, smso=\Eb\ED, smul=\Ef\ED,

#### Omron
#
# This company is still around in 1995, manufacturing point-of-sale systems.

omron|Omron 8025AG,
	OTbs, am, da, db,
	cols#80, lines#24,
	bel=^G, clear=\EJ, cr=^M, cub1=^H, cud1=^J, cuf1=\EC, cuu1=\EA,
	cvvis=\EN, dch1=\EP, dl1=\EM, ed=\ER, el=\EK, home=\EH,
	il1=\EL, ind=\ES, ri=\ET, rmso=\E4, smso=\Ef,

#### Ramtek
#
# Ramtek was a vendor of high-end graphics terminals around 1979-1983; they
# were competition for things like the Tektronics 4025.
#

# Ramtek 6221 from BRL, probably by Doug Gwyn
# The following SET-UP modes are assumed for normal operation:
#	UNDERLINE_CURSOR	ANSI_MODE	AUTO_XON/XOFF_ON
#	NEWLINE_OFF		80_COLUMNS
# Other SET-UP modes may be set for operator convenience or communication
# requirements; I recommend
#	SMOOTH_SCROLL	AUTO_REPEAT_ON	3_#_SHIFTED	WRAP_AROUND_ON
# Hardware tabs are assumed to be every 8 columns; they can be set up by the
# "reset", "tset", or "tabs" utilities (use rt6221-w, 160 columns, for this).
# Note that the Control-E key is useless on this brain-damaged terminal.  No
# delays are specified; use "stty ixon -ixany" to enable DC3/DC1 flow control!
rt6221|Ramtek 6221 80x24,
	OTbs, OTpt, msgr, xon,
	OTkn#4, cols#80, it#8, lines#24, vt#3,
	acsc=, bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[>5l,
	clear=\E[1;1H\E[J, cnorm=\E[>5h\E[>9h, cr=^M,
	csr=\E[%i%p1%d;%p2%dr, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^K, cuf=\E[%p1%dC, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA, cuu1=\EM,
	cvvis=\E[>7h\E[>9l, ed=\E[J, el=\E[K, home=\E[1;1H, ht=^I,
	hts=\EH, ind=^J, is2=\E)0, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kf0=\EOP, kf1=\EOQ, kf2=\EOR,
	kf3=\EOS, lf0=PF1, lf1=PF2, lf2=PF3, lf3=PF4, ll=\E[24;1H,
	nel=\EE, rc=\E8, rev=\E[7m, ri=\EM, rmacs=^O, rmkx=\E>,
	rmso=\E[m, rmul=\E[m,
	rs1=\E[1w\E[>37m\E[>39m\E[1v\E[20l\E[?3l\E[?6l\E[>5h\E[>6h\E[>7h\E[>8l\E[>9h\E[>10l\E[1;24r\E[m\E[q\E(B\017\E)0\E#5\E>,
	sc=\E7, sgr0=\E[m, smacs=^N, smkx=\E=, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g,
# [TO DO: Check out: short forms of ho/cl and ll; reset (\Ec)].
rt6221-w|Ramtek 6221 160x48,
	cols#160, lines#48,
	ll=\E[48;1H, use=rt6221,

#### RCA
#

# RCA VP3301 or VP3501
rca|rca vp3301/vp3501,
	OTbs,
	cols#40, lines#24,
	clear=^L, cuf1=^U, cup=\EY%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=^K, home=^Z, rmso=\E\ES0, smso=\E\ES1,


#### Selanar
#

# Selanar HiREZ-100 from BRL, probably by Doug Gwyn
# The following SET-UP modes are assumed for normal operation:
#	SET_DEFAULT_TABS	48_LINES		80_COLUMNS
#	ONLINE			ANSI			CURSOR_VISIBLE
#	VT102_AUTO_WRAP_ON	VT102_NEWLINE_OFF	VT102_MONITOR_MODE_OFF
#	LOCAL_ECHO_OFF		US_CHAR_SET		WPS_TERMINAL_DISABLED
#	CPU_AUTO_XON/XOFF_ENABLED			PRINT_FULL_SCREEN
# For use with graphics software, all graphics modes should be set to factory
# default.  Other SET-UP modes may be set for operator convenience or
# communication requirements.  No delays are specified; use "stty ixon -ixany"
# to enable DC3/DC1 flow control!
# I commented out the scrolling capabilities since they are too slow.
hirez100|Selanar HiREZ-100,
	OTbs, OTpt, mir, msgr, xon,
	OTkn#4, cols#80, it#8, lines#48, vt#3,
	acsc=, bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[J,
	cr=^M, cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\EM, dch=\E[%p1%dP, dch1=\E[P,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I,
	hts=\EH, il=\E[%p1%dL, il1=\E[L, is2=\E<\E)0, kbs=^H,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kf0=\EOP,
	kf1=\EOQ, kf2=\EOR, kf3=\EOS, lf0=PF1, lf1=PF2, lf2=PF3,
	lf3=PF4, ll=\E[48H, mc0=\E[i, mc4=\E[4i\E[?4i,
	mc5=\E[?5i\E[5i, nel=\EE, rc=\E8, rev=\E[7m, rmacs=^O,
	rmkx=\E[?1l\E>, rmso=\E[m, rmul=\E[m,
	rs1=\030\E2\E<\E[4i\E[?4i\E[12h\E[2;4;20l\E[?0;7h\E[?1;3;6;19l\E[r\E[m\E(B\017\E)0\E>,
	sc=\E7, sgr0=\E[m, smacs=^N, smkx=\E[?1h\E=, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g,
hirez100-w|Selanar HiREZ-100 in 132-column mode,
	cols#132, use=hirez100,

#### Signetics
#

# From University of Wisconsin
vsc|Signetics Vsc Video driver by RMC,
	am, msgr,
	cols#80, it#8, lines#26,
	clear=\E[;H\E[2J$<50/>, cr=^M, cub1=^H, cud1=^J, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, el=\E[K, home=\E[H,
	ht=^I, ind=^J, kbs=^H, kcub1=^H, kcud1=^J, nel=^M^J, rev=^_\s,
	rmso=^_!, rmul=^_#, sgr0=^_!, smso=^_\s, smul=^_",

#### Soroc
#
# Alan Frisbie <frisbie@flying-disk.com> writes:
#
# As you may recall, the Soroc logo consisted of their name,
# with the letter "S" superimposed over an odd design.   This
# consisted of a circle with a slightly smaller 15 degree (approx.)
# wedge with rounded corners inside it.   The color was sort of
# a metallic gold/yellow.
#
# If I had been more of a beer drinker it might have been obvious
# to me, but it took a clue from their service department to make
# me exclaim, "Of course!"   The circular object was the top of
# a beer can (the old removable pop-top style) and "Soroc" was an
# anagram for "Coors".
#
# I can just imagine the founders of the company sitting around
# one evening, tossing back a few and trying to decide what to
# call their new company and what to use for a logo.
#

# (soroc120: removed obsolete ":ma=^K^P^R^L^L :" -- esr)
soroc120|iq120|soroc|soroc iq120,
	clear=\E*$<2>, cud1=^J, ed=\EY, el=\ET, kcub1=^H, kcud1=^J,
	kcuf1=^L, kcuu1=^K, use=adm3a,
soroc140|iq140|soroc iq140,
	OTbs, am, mir,
	cols#80, lines#24,
	bel=^G, cbt=\EI, clear=\E+, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\Ew,
	dl1=\Er$<.7*>, ed=\Ey, el=\Et, home=^^, il1=\Ee$<1*>, ind=^J,
	kbs=^H, kcuf1=^L, kcuu1=^K, kf0=^A0\r, kf1=^A@\r, kf2=^AA\r,
	kf3=^AB\r, kf4=^AC\r, kf5=^AD\r, kf6=^AE\r, kf7=^AF\r,
	kf8=^AG\r, kf9=^AH\r, khome=^^, ll=^^^K, rmir=\E8,
	rmso=\E\177, rmul=\E^A, smir=\E9, smso=\E\177, smul=\E^A,

#### Southwest Technical Products
#
# These guys made an early personal micro called the M6800.
# The ct82 was probably its console terminal.
#

# (swtp: removed obsolete ":bc=^D:" -- esr)
swtp|ct82|southwest technical products ct82,
	am,
	cols#82, lines#20,
	bel=^G, clear=^L, cr=^M, cub1=^D, cud1=^J, cuf1=^S,
	cup=\013%p2%c%p1%c, cuu1=^A, dch1=^\^H, dl1=^Z, ed=^V, el=^F,
	home=^P, ich1=^\^X, il1=^\^Y, ind=^N,
	is2=\034\022\036\023\036\004\035\027\011\023\036\035\036\017\035\027\022\011,
	ll=^C, ri=^O, rmso=^^^F, smso=^^^V,

#### Synertek
#
# Bob Manson <manson@pattyr.acs.ohio-state.edu> writes (28 Apr 1995):
#
# Synertek used to make ICs, various 6502-based single-board process
# control and hobbyist computers, and assorted peripherals including a
# series of small inexpensive terminals (I think they were one of the
# first to have a "terminal-on-a-keyboard", where the terminal itself
# was only slightly larger than the keyboard).
#
# They apparently had a KTM-1 model, which I've never seen. The KTM-2/40
# was a 40x24 terminal that could connect to a standard TV through a
# video modulator.  The KTM-2/80 was the 80-column version (the 2/40
# could be upgraded to the 2/80 by adding 2 2114 SRAMs and a new ROM).
# I have a KTM-2/80 still in working order.  The KTM-2s had fully
# socketed parts, used 2 6507s, a 6532 as keyboard scanner, a program
# ROM and 2 ROMs as character generators. They were incredibly simple,
# and I've never had any problems with mine (witness the fact that mine
# was made in 1981 and is still working great... I've blown the video
# output transistor a couple of times, but it's a 2N2222 :-)
#
# The KTM-3 (which is what is listed in the terminfo file) was their
# attempt at putting a KTM-2 in a box (and some models came with a
# CRT). It wasn't much different from the KTM-2 hardware-wise, but the
# control and escape sequences are very different. The KTM-3 was always
# real broken, at least according to the folks I've talked to about it.
#
# The padding in the entry is probably off--these terminals were very
# slow (it takes like 100ms for the KTM-2 to clear the screen...) And
# anyone with any sanity replaced the ROMs with something that provided
# a reasonable subset of VT100 functionality, since the usual ROMs were
# obviously very primitive... oh, you could get an upgraded ROM from
# Synertek for some incredible amount of money, but what hacker with an
# EPROM burner would do that? :)
#
# Sorry I don't have any contact info; I believe they were located in
# Sunnyvale, and I'm fairly sure they are still manufacturing ICs
# (they've gone to ASICs and FPGAs), but I doubt they're in the computer
# business these days.
#

# Tested, seems to work fine with vi.
synertek|ktm|synertek380|synertek ktm 3/80 tubeless terminal,
	am,
	cols#80, lines#24,
	clear=^Z, cub1=^H, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, ed=\EJ, el=\EK,

#### Tab Office Products
#
#	TAB Products Co. - Palo Alto, California
#	Electronic Office Products,
#	1451 California Avenue 94304
#
# I think they're out of business.
#

# The tab 132 uses xon/xoff, so no padding needed.
# <smkx>/<rmkx> have nothing to do with arrow keys.
# <is2> sets 80 col mode, normal video, autowrap on (for <am>).
# Seems to be no way to get rid of status line.
# The manual for this puppy was dated June 1981.  It claims to be VT52-
# compatible but looks more vt100-like.
tab132|tab|tab132-15|tab 132/15,
	da, db,
	OTdN@, cols#80, lines#24, lm#96,
	cud1=^J, cup=\E[%i%p1%d;%p2%dH, dch1=\E[P, dl1=\E[M,
	il1=\E[L, is2=\E[?7h\E[?3l\E[?5l, kcub1=\E[D, kcud1=\E[B,
	kcuu1=\E[A, rmir=\E[4l, rmkx@, smir=\E[4h, smkx@, use=vt100,
tab132-w|tab132 in wide mode,
	cols#132,
	is2=\E[?7h\E[?3h\E[?5l, use=tab132,
tab132-rv|tab132 in reverse-video mode,
	is2=\E[?7h\E[?3l\E[?5h, use=tab132,
tab132-w-rv|tab132 in reverse-video/wide mode,
	is2=\E[?7h\E[?3h\E[?5h, use=tab132-w,


#### Teleray
#
#	Research Incorporated
#	6425 Flying Cloud Drive
#	Eden Prairie, MN 55344
#	Vox: (612)-941-3300
#
# The Teleray terminals were all discontinued in 1992-93.  RI still services
# and repairs these beasts, but no longer manufactures them.  The Teleray
# people believe that all the types listed below are very rare now (1995).
# There was a newer line of Telerays (Model 7, Model 20, Model 30, and
# Model 100) that were ANSI-compatible.
#
# Note two things called "teleray".  Reorder should move the common one
# to the front if you have either.  A dumb teleray with the cursor stuck
# on the bottom and no obvious model number is probably a 3700.
#

t3700|dumb teleray 3700,
	OTbs,
	cols#80, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, ind=^J,
t3800|teleray 3800 series,
	OTbs,
	cols#80, it#8, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, ed=\EJ, el=\EK,
	home=\EH, ht=^I, ind=^J, ll=\EY7\s,
t1061|teleray|teleray 1061,
	OTbs, am, km, xhp, xt,
	cols#80, it#8, lines#24, xmc#1,
	bel=^G, clear=\014$<1>, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dch1=\EQ,
	dl1=\EM$<2*>, ed=\EJ$<1>, el=\EK, home=\EH, ht=^I, hts=\EF,
	ich1=\EP, il1=\EL$<2*>, ind=^J, ip=$<0.4*>,
	is2=\Ee\EU01^Z1\EV\EU02^Z2\EV\EU03^Z3\EV\EU04^Z4\EV\EU05^Z5\EV\EU06^Z6\EV\EU07^Z7\EV\EU08^Z8\EV\Ef,
	kf1=^Z1, kf2=^Z2, kf3=^Z3, kf4=^Z4, kf5=^Z5, kf6=^Z6, kf7=^Z7,
	kf8=^Z8, rmso=\ER@, rmul=\ER@, smso=\s\ERD, smul=\ERH,
	tbc=\EG,
t1061f|teleray 1061 with fast PROMs,
	dl1=\EM, il1=\EL, ip@, use=t1061,
# "Teleray Arpa Special", officially designated as
# "Teleray Arpa network model 10" with "Special feature 720".
# This is the new (1981) fast microcode updating the older "arpa" proms
# (which gave meta-key and programmable-fxn keys).  720 is much much faster,
# converts the keypad to programmable function keys, and has other goodies.
# Standout mode is still broken (magic cookie, etc) so is suppressed as no
# programs handle such lossage properly.
# Note: this is NOT the old termcap's "t1061f with fast proms."
# From: J. Lepreau <lepreau@utah-cs> Tue Feb  1 06:39:37 1983, Univ of Utah
# (t10: removed overridden ":so@:se@:us@:ue@:" -- esr)
t10|teleray 10 special,
	OTbs, km, xhp, xt,
	cols#80, it#8, lines#24, xmc#2,
	clear=\Ej$<30/>, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dch1=\EQ,
	dl1=\EM, ed=\EJ, el=\EK, home=\EH, ht=^I, ich1=\EP, il1=\EL,
	ind=\Eq, pad=\0, ri=\Ep, rmso=\ER@, rmul=\ER@, smso=\ERD,
	smul=\ERH,
# teleray 16 - map the arrow keys for vi/rogue, shifted to up/down page, and
# back/forth words. Put the function keys (f1-f10) where they can be
# found, and turn off the other magic keys along the top row, except
# for line/local. Do the magic appropriate to make the page shifts work.
# Also toggle ^S/^Q for those of us who use Emacs.
t16|teleray 16,
	am, da, db, mir, xhp, xt,
	cols#80, lines#24,
	bel=^G, clear=\E[H\E[2J, cr=^M, cub1=^H, cud1=\E[B,
	cuf1=\E[C, cup=%i\E[%p1%d;%p2%df, cuu1=\E[A, dch1=\E[P,
	dl1=\E[M, ed=\E[0J, el=\E[0K, home=\E[H, ht=^I, il1=\E[L,
	ind=^J, kf1=^Z1, kf10=^Z0, kf2=^Z2, kf3=^Z3, kf4=^Z4, kf5=^Z5,
	kf6=^Z6, kf7=^Z7, kf8=^Z8, kf9=^Z9, ri=\E[T,
	rmcup=\E[V\E[24;1f\E[?38h, rmir=\E[4l, rmso=\E[m,
	rmul=\E[m, sgr0=\E[m, smcup=\E[U\E[?38l, smir=\E[4h,
	smso=\E[7m, smul=\E[4m,

#### Texas Instruments (ti)
#

# The Silent 700 was so called because it was built around a quiet thermal
# printer.  It was portable, equipped with an acoustic coupler, and pretty
# neat for its day.
ti700|ti733|ti735|ti745|ti800|ti silent 700/733/735/745 or omni 800,
	OTbs, hc, os,
	cols#80,
	bel=^G, cr=\r$<162>, cub1=^H, cud1=^J, ind=^J,

#
# Texas Instruments 916 VDT 7 bit control mode
#
ti916|ti916-220-7|Texas Instruments 916 VDT 8859/1 vt220 mode 7 bit CTRL,
	da, db, in, msgr,
	cbt=\E[Z, civis=\E[?25l, clear=\E[H\E[2J$<6>,
	cnorm=\E[?25h, cub=\E[%p1%dD, cud=\E[%p1%dB,
	cuf=\E[%p1%dC, cup=\E[%p1%i%p1%d;%p2%dH, cuu=\E[%p1%dA,
	dch=\E[%p1%dP$<250>, dch1=\E[P, dl=\E[%p1%dM,
	ech=\E[%p1%dX$<20>, ed=\E[J$<6>, el=\E[0K, el1=\E[1K,
	enacs=\E(B\E)0, ff=^L, flash=\E[?5h\E[?5l$<6>,
	hpa=\E[%p1%{1}%+%dG, hts=\E[0W, ich=\E[%p1%d@$<250>,
	il=\E[%p1%dL$<36>, ip=$<10>, is2=\E[1;24r\E[24;1H,
	kcmd=\E[29~, kdch1=\E[P, kent=^J, kf1=\E[17~, kf10=\E[28~,
	kf11=\E[29~, kf12=\E[31~, kf2=\E[18~, kf3=\E[19~,
	kf4=\E[20~, kf5=\E[21~, kf6=\E[23~, kf7=\E[24~, kf8=\E[25~,
	kf9=\E[26~, khome=\E[H, kich1=\E[@, knp=\E[S, kpp=\E[T,
	kprt=^X, prot=\E&, rmacs=\017$<2>, rs2=\E[!p, sgr@,
	smacs=\016$<2>, tbc=\E[3g, vpa=\E[%p1%{1}%+%dd,
	use=vt220,
#
# Texas Instruments 916 VDT 8 bit control mode
#
ti916-8|ti916-220-8|Texas Instruments 916 VDT 8859/1 8 vt220 mode bit CTRL,
	kcmd=\23329~, kcub1=\233D, kcud1=\233B, kcuf1=\233C,
	kcuu1=\233A, kdch1=\233P, kent=^J, kf1=\23317~,
	kf10=\23328~, kf11=\23329~, kf12=\23331~, kf2=\23318~,
	kf3=\23319~, kf4=\23320~, kf5=\23321~, kf6=\23323~,
	kf7=\23324~, kf8=\23325~, kf9=\23326~, khome=\233H,
	kich1=\233@, knp=\233S, kpp=\233T, kprt=^X, use=ti916,
#
# Texas Instruments 916 VDT 8859/1 7 bit control 132 column mode
#
ti916-132|Texas Instruments 916 VDT vt220 132 column,
	cols#132, use=ti916,
#
# Texas Instruments 916 VDT 8859/1 8 bit control 132 column mode
#
ti916-8-132|Texas Instruments 916 VDT 8-bit vt220 132 column,
	cols#132, use=ti916-8,
ti924|Texas Instruments 924 VDT 8859/1 7 bit CTRL,
	OTbs, am, xon,
	cols#80, it#8, lines#24,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[2J\E[H, cnorm=\E[?25h, cr=^M,
	csr=%i\E[%p1%d;%p2%dr, cub1=\E[D, cud1=\E[B, cuf1=\E[C,
	cup=%i\E[%p1%d;%p2%dH, cuu1=\E[A, cvvis=\E[?31h,
	dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I, hts=\EH,
	il1=\E[L, ind=\ED, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\E[P, kf1=\EOP, kf2=\EOQ,
	kf3=\EOR, kf4=\EOS, kf5=\E[16~, kf6=\E[17~, kf7=\E[18~,
	kf8=\E[19~, kf9=\E[20~, kich1=\E[@, rc=\E8, rev=\E[7m,
	ri=\EM, rmso=\E[m, rmul=\E[m, sc=\E7, sgr0=\E[m, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g,
ti924-8|Texas Instruments 924 VDT 8859/1 8 bit CTRL,
	am, xon,
	cols#80, it#8, lines#24,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[?25l,
	clear=\E[2J\E[H, cnorm=\E[?25h, cr=^M,
	csr=%i\E[%p1%d;%p2%dr, cub1=\E[D, cud1=\E[B, cuf1=\E[C,
	cup=%i\E[%p1%d;%p2%dH, cuu1=\E[A, cvvis=\E[?31h,
	dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I, hts=\EH,
	il1=\E[L, ind=\ED, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=P$<\233>, kf1=P$<\217>,
	kf2=Q$<\217>, kf3=R$<\217>, kf4=S$<\217>, kf5=~$<\23316>,
	kf6=~$<\23317>, kf7=~$<\23318>, kf8=~$<\23319>,
	kf9=~$<\23320>, kich1=@$<\233>, rc=\E8, rev=\E[7m, ri=\EM,
	rmso=\E[m, rmul=\E[m, sc=\E7, sgr0=\E[m, smso=\E[7m,
	smul=\E[4m, tbc=\E[3g,
ti924w|Texas Instruments 924 VDT 7 bit - 132 column mode,
	cols#132, use=ti924,
ti924-8w|Texas Instruments 924 VDT 8 bit - 132 column mode,
	cols#132, use=ti924-8,
ti931|Texas Instruments 931 VDT,
	OTbs, am, xon,
	cols#80, lines#24,
	bel=^G, blink=\E4P, clear=\EL, cnorm=\E4@, cr=^M, cub1=\ED,
	cud1=\EB, cuf1=\EC, cup=\EY%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=\EA, dch1=\EQ, dl1=\EO, ed=\EJ, el=\EI, home=\EH,
	ich1=\ER\EP\EM, il1=\EN, ind=\Ea, invis=\E4H,
	is2=\EGB\E(@B@@\E), kcub1=\ED, kcud1=\EB, kcuf1=\EC,
	kcuu1=\EA, kdch1=\EQ, kdl1=\EO, kf1=\Ei1, kf2=\Ei2, kf3=\Ei3,
	kf4=\Ei4, kf5=\Ei5, kf6=\Ei6, kf7=\Ei7, kf8=\Ei8, kf9=\Ei9,
	kich1=\EP, kil1=\EN, rev=\E4B, ri=\Eb, rmso=\E4@, rmul=\E4@,
	sgr0=\E4@, smso=\E4A, smul=\E4D,
ti926|Texas Instruments 926 VDT 8859/1 7 bit CTRL,
	csr@, ind=\E[1S, ri=\E[1T, use=ti924,
# (ti926-8: I corrected this from the broken SCO entry -- esr)
ti926-8|Texas Instruments 926 VDT 8859/1 8 bit CTRL,
	csr@, ind=\2331S, ri=\2331T, use=ti924-8,
ti_ansi|basic entry for ti928,
	am, bce, eo, xenl, xon,
	colors#8, cols#80, it#8, lines#25, pairs#64,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, clear=\E[2J\E[H,
	cr=^M, csr=\E[%i%p1%d;%p2%dr, cub1=^H, cud1=\E[B,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P,
	dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I, ich1=\E[@,
	il1=\E[L, ind=\E[S, kbs=^H, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kend=\E[F, kf0=\E[V, kf1=\E[M,
	kf2=\E[N, kf3=\E[O, kf4=\E[P, kf5=\E[Q, kf6=\E[R, kf7=\E[S,
	kf8=\E[T, kf9=\E[U, khome=\E[H, knp=\E[G, kpp=\E[I,
	op=\E[37;40m, ri=\E[T, rmso=\E[m, rmul=\E[m,
	setab=\E[4%p1%dm, setaf=\E[3%p1%dm, sgr0=\E[m,
	smso=\E[7m, smul=\E[4m,
#
#       928 VDT 7 bit control mode
#
ti928|Texas Instruments 928 VDT 8859/1 7 bit CTRL,
	kdch1=\E[P, kend=\E_1\E\\, kent=\E[8~, kf1=\E[17~,
	kf10=\E[28~, kf11=\E[29~, kf12=\E[31~, kf13=\E[32~,
	kf15=\E[34~, kf2=\E[18~, kf3=\E[19~, kf4=\E[20~,
	kf5=\E[21~, kf6=\E[23~, kf7=\E[24~, kf8=\E[25~, kf9=\E[26~,
	kich1=\E[@, knp=\E[S, kpp=\E[T, kprt=\E[35~, use=ti_ansi,
#
#       928 VDT 8 bit control mode
#
ti928-8|Texas Instruments 928 VDT 8859/1 8 bit CTRL,
	kdch1=\233P, kend=\2371\234, kent=\2338~, kf1=\23317~,
	kf10=\23328~, kf11=\23329~, kf12=\23331~, kf13=\23332~,
	kf15=\23334~, kf2=\23318~, kf3=\23319~, kf4=\23320~,
	kf5=\23321~, kf6=\23323~, kf7=\23324~, kf8=\23325~,
	kf9=\23326~, khome=\233H, kich1=\233@, knp=\233S,
	kpp=\233T, kprt=\23335~, use=ti_ansi,

#### Zentec (zen)
#

# (zen30: removed obsolete :ma=^L ^R^L^K^P:.  This entry originally
# had just <smso>=\EG6 which I think means standout was supposed to be
# dim-reverse using ADM12-style attributes. ADM12 <smul>/<rmul> and
# <invis> might work-- esr)
zen30|z30|zentec 30,
	OTbs, am, mir, ul,
	cols#80, lines#24,
	bel=^G, clear=\E*, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, dch1=\EW,
	dim=\EG2, dl1=\ER$<1.5*>, ed=\EY, el=\ET$<1.0*>, home=^^,
	il1=\EE$<1.5*>, ind=^J, rmir=\Er, rmul@, smir=\Eq, smso=\EG6,
	smul@, use=adm+sgr,
# (zen50: this had extension capabilities
#	:BS=^U:CL=^V:CR=^B:
# UK/DK/RK/LK/HM were someone's aliases for ku/kd/kl/kr/kh,
# which were also in the original entry -- esr)
# (zen50: removed obsolete ":ma=^Hh^Ll^Jj^Kk:" -- esr)
zen50|z50|zentec zephyr,
	OTbs, am,
	cols#80, lines#24, xmc#1,
	clear=\E+, cub1=^H, cup=\E=%p1%{32}%+%c%p2%{32}%+%c,
	cuu1=^K, dch1=\EW, dl1=\ER, ed=\EY, el=\ET, ich1=\EQ, il1=\EE,
	invis@, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K, khome=^^,
	rmul@, smul@, use=adm+sgr,

# CCI 4574 (Office Power) from Will Martin <wmartin@BRL.ARPA> via BRL
cci|cci1|z8001|zen8001|CCI Custom Zentec 8001,
	OTbs, am, bw,
	cols#80, lines#24,
	blink=\EM", clear=\EH\EJ, cnorm=\EP,
	csr=\ER%p1%{32}%+%c%p2%{32}%+%c, cub1=^H, cud1=^J,
	cuf1=\EC, cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA,
	cvvis=\EF\EQ\EM \ER 7, dim=\EM!, ed=\EJ, el=\EK, home=\EH,
	invis=\EM(, is2=\EM \EF\ET\EP\ER 7, kbs=^H, kcub1=\ED,
	kcud1=\EB, kcuf1=\EC, kcuu1=\EA, khome=\EH, mc4=^T, mc5=^R,
	rev=\EM$, ri=\EI, rmso=\EM\s, rmul=\EM\s, sgr0=\EM\s,
	smso=\EM$, smul=\EM0,

######## OBSOLETE UNIX CONSOLES
#

#### Apollo consoles
#
# Apollo got bought by Hewlett-Packard.  The Apollo workstations are
# labeled HP700s now.
#

# From: Gary Darland <goodmanc@garnet.berkeley.edu>
apollo|apollo console,
	OTbs, am, mir,
	cols#88, lines#53,
	clear=^L, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\EM%p1%{32}%+%c%p2%d), cuu1=\EA, dch1=\EP, dl1=\EL,
	ed=\EJ, el=\EK, hpa=\EN%p1%d, il1=\EI, ind=\EE, ri=\ED,
	rmcup=\EX, rmir=\ER, rmso=\ET, rmul=\EV, smcup=\EW, smir=\EQ,
	smso=\ES, smul=\EU, vpa=\EO+\s,

# We don't know whether or not the apollo guys replicated DEC's firmware bug
# in the VT132 that reversed <rmir>/<smir>.  To be on the safe side, disable
# both these capabilities.
apollo_15P|apollo 15 inch display,
	rmir@, smir@, use=vt132,
apollo_19L|apollo 19 inch display,
	rmir@, smir@, use=vt132,
apollo_color|apollo color display,
	rmir@, smir@, use=vt132,

#### AT&T consoles

# This actually describes the generic SVr4 display driver for Intel boxes.
# The <dim=\E[2m> isn't documented and therefore may not be reliable.
# From: Eric Raymond <esr@snark.thyrsus.com> Mon Nov 27 19:00:53 EST 1995
att6386|at386|386at|AT&T WGS 6386 console,
	am, bw, eo, xon,
	cols#80, it#8, lines#25,
	acsc=``a1fxgqh0jYk?lZm@nEooppqDrrsstCu4vAwBx3yyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, cbt=\E[Z, civis=\E[=C,
	clear=\E[2J\E[H, cnorm=\E[=1C, cr=^M, cub=\E[%p1%dD,
	cub1=\E[D, cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA,
	cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P, dim=\E[2m,
	dl=\E[%p1%dM, dl1=\E[1M, ech=\E[%p1%dX, ed=\E[J, el=\E[K,
	home=\E[H, hpa=\E[%i%p1%dG, ht=^I, hts=\EH, ich=\E[%p1%d@,
	ich1=\E[1@, il=\E[%p1%dL, il1=\E[1L, ind=\E[S,
	indn=\E[%p1%dS, invis=\E[9m, is2=\E[0;10;39m, kbs=^H,
	kcbt=^], kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kdch1=\E[P, kend=\E[Y, kf1=\EOP, kf10=\EOY, kf11=\EOZ,
	kf12=\EOA, kf2=\EOQ, kf3=\EOR, kf4=\EOS, kf5=\EOT, kf6=\EOU,
	kf7=\EOV, kf8=\EOW, kf9=\EOX, khome=\E[H, kich1=\E[@,
	knp=\E[U, kpp=\E[V, krmir=\E0, nel=\r\E[S, rc=\E8, rev=\E[7m,
	ri=\E[T, rin=\E[%p1%dT, rmacs=\E[10m, rmso=\E[m, rmul=\E[m,
	sc=\E7,
	sgr=\E[10m\E[0%?%p1%p3%|%t;7%;%?%p2%t;4%;%?%p4%t;5%;%?%p5%t;2%;%?%p6%t;1%;%?%p9%t;12%e;10%;%?%p7%t;9%;m,
	sgr0=\E[0;10m, smacs=\E[12m, smso=\E[7m, smul=\E[4m,
	tbc=\E[3g, vpa=\E[%i%p1%dd, use=klone+color,
# (pc6300plus: removed ":KM=/usr/lib/ua/kmap.s5:"; renamed BO/EE/CI/CV -- esr)
pc6300plus|AT&T 6300 plus,
	OTbs, am, xon,
	cols#80, lines#24,
	bel=^G, blink=\E[5m, bold=\E[1m, civis=\E[=C,
	clear=\E[2J\E[H, cnorm=\E[=1C, cr=^M, cub1=^H, cud1=\E[B,
	cuf1=\E[C, cup=\E[%i%p1%2d;%p2%2dH, cuu1=\E[A,
	dch1=\E[1P, dim=\E[2m, dl1=\E[1M, ed=\E[0J, el=\E[0K,
	home=\E[H, hts=\EH, ich1=\E[1@, il1=\E[1L, ind=^J,
	invis=\E[9m, kbs=^H, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C,
	kcuu1=\E[A, kf1=\EOc, kf10=\EOu, kf2=\EOd, kf3=\EOe,
	kf4=\EOf, kf5=\EOg, kf6=\EOh, kf7=\EOi, kf8=\EOj, kf9=\EOk,
	nel=^M^J, rev=\E[7m, rmso=\E[m, rmul=\E[m, sgr0=\E[m,
	smso=\E[7m, smul=\E[4m, tbc=\E[3g,

# From: Benjamin C. W. Sittler <bsittler@nmt.edu>
#
# I have a UNIX PC which I use as a terminal attached to my Linux PC.
# Unfortunately, the UNIX PC terminfo entry that comes with ncurses
# is broken. All the special key sequences are broken, making it unusable
# with Emacs. The problem stems from the following:
#
# The UNIX PC has a plethora of keys (103 of them, and there's no numeric
# keypad!), loadable fonts, and strange highlighting modes ("dithered"
# half-intensity, "smeared" bold, and real strike-out, for example.) It also
# uses resizable terminal windows, but the bundled terminal program always
# uses an 80x24 window (and doesn't support seem to support a 132-column
# mode.)
#
# HISTORY: The UNIX PC was one of the first machines with a GUI, and used a
# library which was a superset of SVr3.5 curses (called tam, for "terminal
# access method".) tam includes support for real, overlapping windows,
# onscreen function key labels, and bitmap graphics. But since the primary
# user interface on the UNIX PC was a GUI program (ua, for "user
# assistant",) and remote administration was considered important for the
# machine, tam also supported VT100-compatible terminals attached to the
# serial port or used across the StarLan network. To simulate the extra keys
# not present on a VT100, users could press ESC and a two-letter sequence,
# such as u d (Undo) or U D (Shift-Undo.) These two-letter sequences,
# however, were not the same as those sent by the actual Undo key. The
# actual Undo key sends ESC 0 s unshifted, and ESC 0 S shifted, for example.
# (If you're interested in adding some of the tam calls to ncurses, btw, I
# have the full documentation and several programs which use tam. It also
# used an extended terminfo format to describe key sequences, special
# highlighting modes, etc.)
#
# KEYS: This means that ncurses would quite painful on the UNIX PC, since
# there are two sequences for every key-modifier combination (local keyboard
# sequence and remote "VT100" sequence.) But I doubt many people are trying
# to use ncurses on the UNIX PC, since ncurses doesn't properly handle the
# GUI. Unfortunately, the terminfo entry (and the termcap, too, I presume)
# seem to have been built from the manual describing the VT100 sequences.
# This means it doesn't work for a real live UNIX PC.
#
# FONTS: The UNIX PC also has a strange interpretation of "alternate
# character set". Rather than the VT100 graphics you might expect, it allows
# up to 8 custom fonts to be loaded at any given time. This means that
# programs expecting VT100 graphics will usually be disappointed. For this
# reason I have disabled the smacs/rmacs sequences, but they could easily be
# re-enabled. Here are the relevant control sequences (from the ESCAPE(7)
# manpage), should you wish to do so:
#
# SGR10 - Select font 0 - ESC [ 10 m or SO
# SGR11 - Select font 1 - ESC [ 11 m or SI
# SGR12 - Select font 2 - ESC [ 12 m
# ... (etc.)
# SGR17 - Select font 7 - ESC [ 17 m
#
# Graphics for line drawing are not reliably found at *any* character
# location because the UNIX PC has dynamically reloadable fonts. I use font
# 0 for regular text and font 1 for italics, but this is by no means
# universal. So ASCII line drawing is in order if smacs/rmacs are enabled.
#
# MISC: The cursor visible/cursor invisible sequences were swapped in the
# distributed terminfo.
#
# To ameliorate these problems (and fix a few highlighting bugs) I rewrote
# the UNIX PC terminfo entry. The modified version works great with Lynx,
# Emacs, and XEmacs running on my Linux PC and displaying on the UNIX PC
# attached by serial cable. In Emacs, even the Undo key works, and many
# applications can now use the F1-F8 keys.
#
# esr's notes:
#	Terminfo entry for the AT&T Unix PC 7300
#	from escape(7) in Unix PC 7300 Manual.
#	Somewhat similar to a vt100-am (but different enough
#	to redo this from scratch.)
#
#	/***************************************************************
#	*
#	*           FONT LOADING PROGRAM FOR THE UNIX PC
#	*
#	*     This routine loads a font defined in the file ALTFONT
#	*     into font memory slot #1.  Once the font has been loaded,
#	*     it can be used as an alternative character set.
#	*
#	*     The call to ioctl with the argument WIOCLFONT is the key
#	*     to this routine.  For more information, see window(7) in
#	*     the PC 7300 documentation.
#	***************************************************************/
#	#include <string.h>		/* needed for strcpy call */
#	#include <sys/window.h>         /* needed for ioctl call */
#	#define FNSIZE	60		/* font name size */
#	#define ALTFONT  "/usr/lib/wfont/special.8.ft"  /* font file */
#	/*
#	*     The file /usr/lib/wfont/special.8.ft comes with the
#	*     standard PC software.  It defines a graphics character set
#	*     similar to that of the Teletype 5425 terminal.  To view
#	*     this or other fonts in /usr/lib/wfont, use the command
#	*     cfont <filename>.  For further information on fonts see
#	*     cfont(1) in the PC 7300 documentation.
#	*/
#
#	struct altfdata 	/* structure for alt font data */
#	{
#	short	altf_slot;		/* memory slot number */
#	char	altf_name[FNSIZE];	/* font name (file name) */
#	};
#	ldfont()
#	{
#		int wd;		/* window in which altfont will be */
#		struct altfdata altf;
#		altf.altf_slot=1;
#		strcpy(altf.altf_name,ALTFONT);
#		for (wd =1; wd < 12; wd++) {
#		     ioctl(wd, WIOCLFONT,&altf);
#	        }
#	}
#
# (att7300: added <civis>/<cnorm>/<ich1>/<invis> from the BSDI entry,
# they're confirmed by the man page for the System V display---esr)
#
att7300|unixpc|pc7300|3b1|s4|AT&T UNIX PC Model 7300,
	am, xon,
	cols#80, it#8, lines#24,
	bel=^G, blink=\E[9m, bold=\E[1m, cbt=\E^I, civis=\E[=1C,
	clear=\E[2J\E[H, cnorm=\E[=0C, cr=^M, cub=\E[%p1%dD,
	cub1=^H, cud=\E[%p1%dB, cud1=\E[B, cuf=\E[%p1%dC,
	cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu=\E[%p1%dA,
	cuu1=\E[A, dch1=\E[P, dim=\E[2m, dl=\E[%p1%dM, dl1=\E[M,
	ed=\E[0J, el=\E[0K, home=\E[H, ich1=\E[@, il=\E[%p1%dL,
	il1=\E[L, ind=^J, invis=\E[9m, is1=\017\E[=1w, kBEG=\ENB,
	kCAN=\EOW, kCPY=\END, kCRT=\EON, kDC=\ENF, kDL=\ENE,
	kEND=\ENN, kEOL=\EOA, kFND=\EOX, kHLP=\EOM, kHOM=\ENM,
	kIC=\ENJ, kLFT=\ENK, kMOV=\ENC, kNXT=\ENH, kOPT=\EOR,
	kPRV=\ENG, kRDO=\EOT, kRIT=\ENL, kRPL=\EOY, kSAV=\EOO,
	kUND=\EOS, kbeg=\ENb, kbs=^H, kcan=\EOw, kcbt=\E[Z,
	kclo=\EOV, kclr=\E[J, kcmd=\EOu, kcpy=\ENd, kcrt=\EOn,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kdch1=\ENf,
	ked=\E[J, kel=\EOa, kend=\E0, kext=\EOk, kf1=\EOc, kf2=\EOd,
	kf3=\EOe, kf4=\EOf, kf5=\EOg, kf6=\EOh, kf7=\EOi, kf8=\EOj,
	kfnd=\EOx, khlp=\EOm, khome=\E[H, kich1=\ENj, kind=\E[B,
	kmov=\ENc, kmrk=\ENi, knp=\E[U, knxt=\ENh, kopn=\EOv,
	kopt=\EOr, kpp=\E[V, kprt=\EOz, kprv=\ENg, krdo=\EOt,
	kref=\EOb, krfr=\ENa, kri=\E[A, krpl=\EOy, krst=\EOB,
	ksav=\EOo, kslt=\ENI, kund=\EOs, nel=\EE, rev=\E[7m, ri=\EM,
	rmso=\E[m, rmul=\E[m, sgr0=\E[0;10m, smso=\E[7m,
	smul=\E[4m,

#### Convergent Technology
#
# Burroughs bought Convergent shortly before it merged with Univac.
# CTOS is (I believe) dead.  Probably the aws is too (this entry dates
# from 1991 or earlier).
#

# Convergent AWS workstation from Gould/SEL UTX/32 via BRL
# (aws: removed unknown :dn=^K: -- esr)
aws|Convergent Technologies AWS workstation under UTX and Xenix,
	am,
	OTug#0, cols#80, lines#28, xmc#0,
	OTbc=^H, OTma=\016h\013j\001k\022l\002m, OTnl=^J, acsc=,
	clear=^L, cud1=^K, cuf1=^R, cup=\EC%p2%c%p1%c, cuu1=^A,
	dch1=\EDC, dl1=\EDL, ed=\EEF, el=\EEL, hpa=\EH%p1%c,
	ich1=\EIC, il1=\EIL, ind=\ESU, kbs=^H, kcub1=^N, kcud1=^K,
	kcuf1=^R, kcuu1=^A, ri=\ESD, rmacs=\EAAF, rmso=\EARF,
	rmul=\EAUF, smacs=\EAAN, smso=\EARN, smul=\EAUN,
	vpa=\EV%p1%c,
awsc|Convergent Technologies AWS workstation under CTOS,
	am,
	OTug#0, cols#80, lines#24, xmc#0,
	OTbc=^N, OTma=\016h\013j\001k\022l\002m, acsc=, clear=^L,
	cud1=^K, cuf1=^R, cup=\EC%p2%c%p1%c, cuu1=^A, ed=\EEF,
	el=\EEL, kbs=^H, kcub1=^N, kcud1=^K, kcuf1=^R, kcuu1=^A,
	rmacs=\EAAF, rmso=\EAA, rmul=\EAA, smacs=\EAAN, smso=\EAE,
	smul=\EAC,

#### DEC consoles
#

# The MicroVax console.  Tim Theisen <tim@cs.wisc.edu> writes:
# The digital uVax II's had a graphic display called a qdss.  It was
# supposed to be a high performance graphic accelerator, but it was
# late to market and barely appeared before faster dumb frame buffers
# appeared.  I have only used this display while running X11.  However,
# during bootup, it was in text mode, and probably had a terminal emulator
# within it.  And that is what your termcap entry is for.  In graphics
# mode the screen size is 1024x864 pixels.
qdss|qdcons|qdss glass tty,
	OTbs, am,
	cols#128, lines#57,
	clear=\032$<1/>, cub1=^H, cud1=^J, cuf1=^L,
	cup=\E=%p1%c%p2%c, cuu1=^K,

#### Fortune Systems consoles
#
# Fortune made a line of 68K-based UNIX boxes that were pretty nifty
# in their day; I (esr) used one myself for a year or so around 1984.
# They had no graphics, though, and couldn't compete against Suns and
# the like.  R.I.P.
#

# From: Robert Nathanson <c160-3bp@Coral> via tut   Wed Oct 5, 1983
# (This had extension capabilities
#	:rv=\EH:re=\EI:rg=0:GG=0:\
#	:CO=\E\\:WL=^Aa\r:WR=^Ab\r:CL=^Ac\r:CR=^Ad\r:DL=^Ae\r:RF=^Af\r:\
#	:RC=^Ag\r:CW=^Ah\r:NU=^Aj\r:EN=^Ak\r:HM=^Al:PL=^Am\r:\
#	:PU=^An\r:PD=^Ao\r:PR=^Ap\r:HP=^A@\r:RT=^Aq\r:TB=\r:CN=\177:MP=\E+F:
# It had both ":bs:" and ":bs=^H:"; I removed the latter.  Also, it had
# ":sg=0:" and ":ug=0:"; evidently the composer was trying (unnecessarily)
# to force both magic cookie glitches off.  Once upon a time, I
# used a Fortune myself, so I know the capabilities of the form ^A[a-z]\r are
# function keys; thus the "Al" value for HM was certainly an error.  I renamed
# EN/PD/PU/CO/CF/RT according to the XENIX/TC mappings, but not HM/DL/RF/RC.
# I think :rv: and :re: are start/end reverse video and :rg: is a nonexistent
# "reverse-video-glitch" capability; I have put :rv: and :re: in with standard
# names below.  I've removed obsolete ":nl=5^J:" as there is a :do: -- esr)
fos|fortune|Fortune system,
	OTbs, am, bw,
	cols#80, lines#25,
	acsc=j*k(l m"q&v%w#x-, bel=^G, blink=\EN, civis=\E],
	clear=\014$<20>, cnorm=\E\\, cr=^M, cub1=^H, cud1=\n$<3>,
	cup=\034C%p1%{32}%+%c%p2%{32}%+%c, cuu1=\013$<3>,
	cvvis=\E\:, dch1=\034W$<5>, dl1=\034R$<15>,
	ed=\034Y$<3*>, el=^\Z, home=\036$<10>, ht=^Z,
	ich1=\034Q$<5>, il1=\034E$<15>, ind=^J, is2=^_.., kbs=^H,
	kcub1=^Aw\r, kcud1=^Ay\r, kcuf1=^Az\r, kcuu1=^Ax\r,
	kend=^Ak\r, kent=^Aq, kf1=^Aa\r, kf2=^Ab\r, kf3=^Ac\r,
	kf4=^Ad\r, kf5=^Ae\r, kf6=^Af\r, kf7=^Ag\r, kf8=^Ah\r,
	khome=^A?\r, knp=^Ao\r, kpp=^An\r, nel=^M^J, rev=\EH,
	rmacs=^O, rmso=^\I`, rmul=^\IP, sgr0=\EI, smacs=\Eo,
	smso=^\H`, smul=^\HP,

#### Masscomp consoles
#
# Masscomp has gone out of business.  Their product line was purchased by
# comany in Georgia (US) called "XS International", parts and service may
# still be available through them.
#

# (masscomp: ":MT:" changed to ":km:";  -- esr)
masscomp|masscomp workstation console,
	OTbs, km, mir,
	cols#80, it#8, lines#24,
	clear=\E[2J, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P, dl1=\E[M,
	ed=\E[J, el=\E[K, ht=^I, il1=\E[L, is2=\EGc\EGb\EGw, kbs=^H,
	kcub1=\EOD, kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, rmir=\E[4l,
	rmso=\E[m, rmul=\EGau, smir=\E[4h, smso=\E[7m, smul=\EGu,
masscomp1|masscomp large screen version 1,
	cols#104, lines#36, use=masscomp,
masscomp2|masscomp large screen version 2,
	cols#64, lines#21, use=masscomp,

#### OSF Unix
#

# OSF/1 1.1 Snapshot 2
pmcons|pmconsole|PMAX console,
	am,
	cols#128, lines#57,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuu1=^K, ht=^I,
	ind=^J, kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A,
	kf1=\EOP, kf2=\EOQ, kf3=\EOR, kf4=\EOS,

#### Other consoles
# The following is a version of the ibm-pc entry distributed with PC/IX,
# (Interactive Systems' System 3 for the Big Blue), modified by Richard
# McIntosh at UCB/CSM.  The :pt: and :uc: have been removed from the original,
# (the former is untrue, and the latter failed under UCB/man); standout and
# underline modes have been added.  Note: this entry describes the "native"
# capabilities of the PC monochrome display, without ANY emulation; most
# communications packages (but NOT PC/IX connect) do some kind of emulation.
pcix|PC/IX console,
	am, bw, eo,
	cols#80, lines#24,
	clear=\Ec, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%2d;%p2%2dH, cuu1=\E[A, ed=\E[J, el=\E[K,
	home=\E[H, rmso=\E[m, rmul=\E[m, sgr0=\E[m, smso=\E[7m,
	smul=\E[4m,

# (ibmpcx: this entry used to be known as ibmx.
# It formerly included the following extension capabilities:
#	:GC=b:GL=v:GR=t:RT=^J:\
#	:GH=\E[196g:GV=\E[179g:\
#	:GU=\E[193g:GD=\E[194g:\
#	:G1=\E[191g:G2=\E[218g:G3=\E[192g:G4=\E[217g:\
#	:CW=\E[E:NU=\E[F:RF=\E[G:RC=\E[H:\
#	:WL=\E[K:WR=\E[L:CL=\E[M:CR=\E[N:\
# I renamed GS/GE/WL/WR/CL/CR/PU/PD/HM/EN; also, removed a duplicate
# ":kh=\E[Y:".  Added IBM-PC forms characters and highlights, they match
# what was there before. -- esr)
ibmpcx|xenix|ibmx|IBM PC xenix console display,
	OTbs, am, msgr,
	cols#80, lines#25,
	clear=^L, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P, dl1=\E[M,
	ed=\E[J, el=\E[K, home=\E[H, ich1=\E[@, il1=\E[L, kbs=^H,
	kcub1=\E[D, kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kend=\E[d,
	kf1=\E[K, kf2=\E[L, kf3=\E[M, kf4=\E[N, khome=\E[Y, knp=\E[e,
	kpp=\E[Z, use=klone+acs, use=klone+sgr8,

######## OTHER OBSOLETE TYPES
#
# These terminals are *long* dead -- these entries are retained for
# historical interest only.
#

#### Obsolete non-ANSI software emulations
#

# CTRM terminal emulator
# 1. underlining is not allowed with colors: first, is is simulated by
# black on white, second, it disables background color manipulations.
# 2. BLINKING, REVERSE and BOLD are allowed with colors,
# so we have to save their status in the static registers A, B and H
# respectively, to be able to restore them when color changes
# (because any color change turns off ALL attributes)
# 3. <bold> and <rev> sequences alternate modes,
# rather than simply  entering them.  Thus we have to check the
# static register B and H to determine the status, before sending the
# escape sequence.
# 4. <sgr0> now must set the status of all 3 register (A,B,H) to zero
# and then reset colors
# 5. implementation of the protect mode would badly penalize the performance.
# we would have to use \E&bn sequence to turn off colors (as well as all
# other attributes), and keep the status of protect mode in yet another
# static variable.  If someone really needs this mode, they would have to
# create another terminfo entry.
# 6. original color-pair is white on black.
# store the information about colors into static registers
# 7. set foreground color.  it performs the following steps.
#   1) turn off all attributes
#   2) turn on the background and video attributes that have been turned
#      on before (this information is stored in static registers X,Y,Z,A,B,H,D).
#   3) turn on foreground attributes
#   4) store information about foreground into U,V,W static registers
# 8. turn on background: similar to turn on foreground above
ctrm|C terminal emulator,
	am, bce, xon,
	colors#8, cols#80, lh#0, lines#24, lm#0, lw#0, ncv#2, nlab#0,
	pairs#63, pb#19200, vt#6,
	bel=^G, blink=\E&dA%{1}%PA,
	bold=%?%gH%{0}%=%t\E&dH%{1}%PH%;, cbt=\Ei,
	clear=\EH\EJ, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\E&a%p2%dc%p1%dY, cuu1=\EA, dch1=\EP$<2>, dl1=\EM,
	ed=\EJ, el=\EK, hpa=\E&a%p1%dC, ht=\011$<2>, hts=\E1,
	il1=\EL, ind=^J, ip=$<2>, is2=\E&jA\r, kbs=^H, kcub1=\Eu\r,
	kcud1=\Ew\r, kcuf1=\Ev\r, kcuu1=\Et\r, kf1=\Ep\r,
	kf2=\Eq\r, kf3=\Er\r, kf4=\Es\r, kf5=\Et\r, kf6=\Eu\r,
	kf7=\Ev\r, kf8=\Ew\r, khome=\Ep\r,
	op=\E&bn\E&bB\E&bG\E&bR%{0}%PX%{0}%PY%{0}%PZ%{1}%PW%{1}%PV%{1}%PU,
	rev=%?%gB%{0}%=%t\E&dB%{1}%PB%;, rmir=\ER, rmkx=\E&jA,
	setb=\E&bn%?%gA%t\E&dA%;%?%gB%t\E&dB%;%?%gH%t\E&dH%;%?%gU%t\E&bR%;%?%gV%t\E&bG%;%?%gW%t\E&bB%;%?%p1%{1}%&%t\E&bb%{1}%e%{0}%;%PZ%?%p1%{2}%&%t\E&bg%{1}%e%{0}%;%PY%?%p1%{4}%&%t\E&br%{1}%e%{0}%;%PX,
	setf=\E&bn%?%gA%t\E&dA%;%?%gB%t\E&dB%;%?%gH%t\E&dH%;%?%gX%t\E&br%;%?%gY%t\E&bg%;%?%gZ%t\E&bb%;%?%p1%{1}%&%t\E&bB%{1}%e%{0}%;%PW%?%p1%{2}%&%t\E&bG%{1}%e%{0}%;%PV%?%p1%{4}%&%t\E&bR%{1}%e%{0}%;%PU,
	sgr=\E&d@%{0}%PA%{0}%PB%{0}%PD%{0}%PH%?%p1%p3%p5%|%|%t\E&dB%{1}%PB%;%?%p4%t\E&dA%{1}%PA%;%?%p6%t\E&dH%{1}%PH%;%?%p2%t\E&dD%;,
	sgr0=\E&d@%{0}%PA%{0}%PB%{0}%PH, smir=\EQ, smkx=\E&jB,
	smso=\E&dD, smul=\E&dD, tbc=\E3, vpa=\E&a%p1%dY,

# gs6300 - can't use blue foreground, it clashes with underline;
# it's simulated with cyan
# Bug: The <op> capability probably resets attributes.
# (gs6300: commented out <rmln> (no <smln>) --esr)
gs6300|emots|AT&T PC6300 with EMOTS terminal emulator,
	am, bce, msgr, xon,
	colors#8, cols#80, it#8, lines#24, pairs#63,
	acsc=++\,\,--..``aaffgghhiijjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, blink=\E[5m, bold=\E[1m, clear=\E[H\E[J, cr=^M,
	cub=\E[%p1%dD, cub1=^H, cud=\E[%p1%dB, cud1=^J,
	cuf=\E[%p1%dC, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH,
	cuu=\E[%p1%dA, cuu1=\E[A, dch=\E[%p1%dP, dch1=\E[P,
	dl=\E[%p1%dM, dl1=\E[M, ed=\E[J, el=\E[K, home=\E[H, ht=^I,
	ich=\E[%p1%d@, ich1=\E[@, il=\E[%p1%dL, il1=\E[L, ind=^J,
	is2=\E[m, kbs=^H, kcbt=^R^I, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kf1=\E[0s, kf2=\E[24s, kf3=\E[1s,
	kf4=\E[23s, kf5=\E[2s, kf6=\E[22s, kf7=\E[3s, kf8=\E[21s,
	khome=\E[H, mc4=\E[4i, mc5=\E[5i, op=\E[?;m, rev=\E[7m,
	ri=\E[L, rmacs=\E[10m, rs1=\Ec, setb=\E[?;%p1%dm,
	setf=\E[?%?%p1%{0}%=%t0%e%p1%{1}%=%t2%e%p1%{1}%-%d%;m,
	sgr0=\E[m\E[10m, smacs=\E[11m, smso=\E[1m, smul=\E[4m,

# From: <earle@smeagol.UUCP> 29 Oct 85 05:40:18 GMT
# MS-Kermit with Heath-19 emulation mode enabled
# (h19k: changed ":pt@:" to ":it@"
h19k|h19kermit|heathkit emulation provided by Kermit (no auto margin),
	am@, da, db, xt,
	it@,
	ht@, use=h19-u,

# Apple Macintosh with Versaterm, a terminal emulator distributed by Synergy
# Software (formerly Peripherals Computers & Supplies, Inc) of
# 2457 Perkiomen Ave., Reading, PA 19606, 1-800-876-8376.  They can
# also be reached at support@synergy.com.
versaterm|versaterm vt100 emulator for the macintosh,
	am, xenl,
	cols#80, it#8, lines#24,
	bel=^G, blink=\E[5m$<2/>, bold=\E[1m$<2/>,
	clear=\E[;H\E[2J$<50/>, cr=^M, csr=\E[%i%p1%d;%p2%dr,
	cub1=^H, cud1=^J, cuf1=\E[C$<2/>,
	cup=\E[%i%p1%d;%p2%dH$<5/>, cuu1=\E[A$<2/>,
	dch1=\E[1P$<7/>, dl1=\E[1M$<9/>, ed=\E[J$<50/>,
	el=\E[K$<3/>, home=\E[H, ht=^I, ich1=\E[1@$<7/>,
	il1=\E[1L$<9/>, is2=\E[1;24r\E[24;1H, kbs=^H, kcub1=\EOD,
	kcud1=\EOB, kcuf1=\EOC, kcuu1=\EOA, kf1=\EOP, kf2=\EOQ,
	kf3=\EOR, kf4=\EOS, nel=^M^J, rc=\E8, rev=\E[7m$<2/>,
	rf=/usr/share/tabset/vt100, ri=\EM$<5/>,
	rmkx=\E>\E[?1l, rmso=\E[m$<2/>, rmul=\E[m$<2/>, rs1=\E>,
	sc=\E7, sgr0=\E[m$<2/>, smkx=\E=\E[?1h, smso=\E[7m$<2/>,
	smul=\E[4m$<2/>,

# From: Rick Thomas <ihnp4!btlunix!rbt>
# (xtalk: I added <rmam>/<smam> based on the init string.
xtalk|IBM PC with xtalk communication program (versions up to 3.4),
	am, mir, msgr, xon,
	cols#80, it#8, lines#24, vt#3, xmc#1,
	acsc=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~,
	bel=^G, clear=\E[H\E[J$<50>, cr=^M, cub=\E[%p1%dD, cub1=^H,
	cud=\E[%p1%dB, cud1=^J, cuf=\E[%p1%dC, cuf1=\E[C$<2>,
	cup=\E[%i%p1%d;%p2%dH$<5>, cuu=\E[%p1%dA,
	cuu1=\E[A$<2>, dl1=\E[M$<99>, ed=\E[J$<50>, el=\E[K$<3>,
	el1=\E[1K$<3>, enacs=\E(B\E)0, home=\E[H, ht=^I, hts=\EH,
	il1=\E[L$<99>, ind=^J, kbs=^H, kcub1=\EOD, kcud1=\EOB,
	kcuf1=\EOC, kcuu1=\EOA, ri=\EM$<5>, rmacs=^O, rmam=\E[?7l,
	rmkx=\E[?1l\E>, rmso=\E[m\s,
	rs2=\E>\E[?3l\E[?4l\E[?5l\E[?7h\E[?8h, sgr0=\E[m,
	smacs=^N, smam=\E[?7h, smkx=\E[?1h\E=, smso=\E[7m\s,
	tbc=\E[3g, use=vt100+fnkeys,

# The official PC terminal emulator program of the AT&T Product Centers.
# Note - insert mode commented out - doesn't seem to work on AT&T PC.
simterm|attpc running simterm,
	am,
	cols#80, lines#24,
	bel=^G, clear=\EH\EJ, cr=^M, cub1=^H, cud1=\EB, cuf1=\EC,
	cup=\E=%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dch1=\ER,
	dl1=\EM, ed=\EJ, el=\EK, home=\EH, il1=\EL, ind=^J, rmcup=\EVE,
	rmso=\E&d@, sgr0=\E&d@, smcup=\EVS, smso=\E&dB,

#### Daisy wheel printers
#
# This section collects Diablo, DTC, Xerox, Qume, and other daisy
# wheel terminals.  These are now largely obsolete.
#

# (diablo1620: removed <if=/usr/share/tabset/xerox1720>, no such file -- esr)
diablo1620|diablo1720|diablo450|ipsi|diablo 1620,
	hc, os,
	cols#132, it#8,
	cub1=^H, cud1=^J, cuu1=\E^J, hd=\ED, hpa=\E\011%i%p1%c,
	ht=^I, hts=\E1, hu=\EU, kbs=^H, tbc=\E2,
diablo1620-m8|diablo1640-m8|diablo 1620 w/8 column left margin,
	cols#124,
	is2=\r        \E9, use=diablo1620,
# (diablo1640: removed <if=/usr/share/tabset/xerox1730>, no such file -- esr)
diablo1640|diablo1730|diablo1740|diablo630|x1700|diablo|xerox|diablo 1640,
	bel=^G, rmso=\E&, rmul=\ER, smso=\EW, smul=\EE,
	use=diablo1620,
# (diablo1640-lm: removed <if=/usr/share/tabset/xerox1730-lm>, no such
# file -- esr)
diablo1640-lm|diablo-lm|xerox-lm|diablo 1640 with indented left margin,
	cols#124,
	rmso=\E&, rmul=\ER, smso=\EW, smul=\EE, use=diablo1620,
diablo1740-lm|630-lm|1730-lm|x1700-lm|diablo 1740 printer,
	use=diablo1640-lm,
# DTC 382 with VDU.  Has no <ed> so we fake it with <el>.  Standout
# <smso=^P\s\002^PF> works but won't go away without dynamite <rmso=^P\s\0>.
# The terminal has tabs, but I'm getting tired of fighting the braindamage.
# If no tab is set or the terminal's in a bad mood, it glitches the screen
# around all of memory.  Note that return puts a blank ("a return character")
# in the space the cursor was at, so we use ^P return (and thus ^P newline for
# newline).  Note also that if you turn off :pt: and let Unix expand tabs,
# curses won't work (some old BSD versions) because it doesn't clear this bit,
# and cursor addressing sends a tab for row/column 9.  What a losing terminal!
# I have been unable to get tabs set in all 96 lines - it always leaves at
# least one line with no tabs in it, and once you tab through that line,
# it completely weirds out.
# (dtc382: change <rmcup> to <smcup> -- it  just does a clear --esr)
dtc382|DTC 382,
	am, da, db, xhp,
	cols#80, lines#24, lm#96,
	bel=^G, clear=\020\035$<20>, cnorm=^Pb, cr=^P^M, cub1=^H,
	cuf1=^PR, cup=\020\021%p2%c%p1%c, cuu1=^P^L, cvvis=^PB,
	dch1=^X, dl1=^P^S, ed=\020\025\020\023\020\023, el=^P^U,
	home=^P^R, il1=^P^Z, ind=^J, pad=\177, rmcup=, rmir=^Pi,
	rmul=^P \0, smcup=\020\035$<20>, smir=^PI, smul=^P ^P,
dtc300s|DTC 300s,
	hc, os,
	cols#132,
	bel=^G, cr=^M, cub1=^H, cud1=^J, cuu1=^Z, ff=^L, hd=\Eh, ht=^I,
	hts=\E1, hu=\EH, ind=^J, kbs=^H, tbc=\E3,
gsi|mystery gsi terminal,
	hc, os,
	cols#132,
	bel=^G, cr=^M, cub1=^H, cud1=^J, cuu1=^Z, hd=\Eh, ht=^I, hu=\EH,
	ind=^J,
aj830|aj832|aj|anderson jacobson,
	hc, os,
	bel=^G, cr=^M, cub1=^H, cud1=^J, cuu1=\E7, hd=\E9, hu=\E8,
	ind=^J,
# From: Chris Torek <chris@gyre.umd.edu> Thu, 7 Nov 85 18:21:58 EST
aj510|Anderson-Jacobson model 510,
	am, mir,
	cols#80, lines#24,
	clear=^L, cub1=^H, cuf1=\EX,
	cup=\E#%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EY,
	dch1=.1*\E'D, dl1=\E&D$<2*/>, ed=\E'P, el=\E'L, ich1=,
	il1=\E&I$<2*/>, ip=$<.1*/>, kcub1=\EW, kcud1=\EZ,
	kcuf1=\EX, kcuu1=\EY, pad=\177, rmcup=\E"N, rmir=\E'J,
	rmso=\E"I, rmul=\E"U, smcup=\E"N, smir=\E'I, smso=\E"I,
	smul=\E"U,
# From: <cbosg!ucbvax!pur-ee!cincy!chris> Thu Aug 20 09:09:18 1981
# This is incomplete, but it's a start.
nec5520|nec|spinwriter|nec 5520,
	hc, os,
	cols#132, it#8,
	bel=^G, cr=^M, cub1=^H, cud1=^J, cuu1=\E9, ff=^L,
	hd=\E]s\n\E]W, ht=^I, hts=\E1, hu=\E]s\E9\E]W, ind=^J,
	kbs=^H, tbc=\E3,
qume5|qume|Qume Sprint 5,
	hc, os,
	cols#80, it#8,
	bel=^G, cr=^M, cub1=^H, cud1=^J, cuu1=^Z, ff=^L, hd=\Eh, ht=^I,
	hts=\E1, hu=\EH, ind=^J, kbs=^H, tbc=\E3,
# I suspect the xerox 1720 is the same as the diablo 1620.
xerox1720|x1720|x1750|xerox 1720,
	hc, os,
	cols#132, it#8,
	bel=^G, cr=^M, cub1=^H, cud1=^J, ff=^L, ht=^I, hts=\E1, ind=^J,
	tbc=\E2,

#### Miscellaneous obsolete terminals, manufacturers unknown
#
# If you have any information about these (like, a manufacturer's name,
# and a date on the serial-number plate) please send it!

cad68-3|cgc3|cad68 basic monitor transparent mode size 3 chars,
	OTbs, am,
	cols#73, lines#36,
	clear=^Z, cub1=^H, cuf1=^L, cuu1=^K, home=^^,
cad68-2|cgc2|cad68 basic monitor transparent mode size 2 chars,
	OTbs, am,
	cols#85, lines#39,
	clear=^Z, cub1=^H, cuf1=^L, cuu1=^K, home=^^, kcub1=\E3,
	kcud1=\E2, kcuf1=\E4, kcuu1=\E1, kf1=\E5, kf2=\E6, kf3=\E7,
	kf4=\E8, rmso=\Em^C, smso=\Em^L,
cops10|cops|cops-10|cops 10,
	am, bw,
	cols#80, lines#24,
	bel=^G, clear=\030$<30/>, cr=^M, cub1=^H, cud1=^J, cuf1=^L,
	cup=\020%p1%{32}%+%c%p2%{32}%+%c, cuu1=^K, ed=^W, el=^V,
	ind=^J, kbs=^H, kcub1=^H, kcud1=^J, kcuf1=^L, kcuu1=^K,
	khome=^Y,
# (d132: removed duplicate :ic=\E5:,
# merged in capabilities from a BRL entry -- esr)
d132|datagraphix|datagraphix 132a,
	da, db, in,
	cols#80, lines#30,
	bel=^G, clear=^L, cnorm=\Em\En, cr=^M, cub1=^H, cud1=^J,
	cuf1=\EL, cup=\E8%i%p1%3d%p2%3d, cuu1=\EK, cvvis=\Ex,
	dch1=\E6, home=\ET, ht=^I, ich1=\E5, il1=\E3, ind=^J, kbs=^H,
	kcub1=^H, kcud1=^J, nel=^M^J, ri=\Ew,
# The d800 was an early portable terminal from c.1984-85 that looked a lot
# like the original Compaq `lunchbox' portable (but no handle).  It had a vt220
# mode (which is what this entry looks like) and several other lesser-known
# emulations.
d800|Direct 800/A,
	OTbs, am, da, db, msgr, xhp,
	cols#80, it#8, lines#24,
	acsc=``a1fxgqh0jYk?lZm@nEooppqDrrsstCu4vAwBx3yyzz{{||}}~~,
	bel=^G, clear=\E[1;1H\E[2J, cnorm=\E[>12h, cr=^M, cub1=^H,
	cud1=^J, cuf1=\E[C, cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A,
	cvvis=\E[>12l, ed=\E[J, el=\E[K, ht=^I, ind=\ED, kcub1=\E[D,
	kcud1=\E[B, kcuf1=\E[C, kcuu1=\E[A, kf1=\EOP, kf2=\EOQ,
	kf3=\EOR, kf4=\EOS, kf5=\EOT, kf6=\EOU, kf7=\EOV, kf8=\EOW,
	ri=\EM, rmacs=\E[m, rmso=\E[m, rmul=\E[m, sgr0=\E[m,
	smacs=\E[1m, smso=\E[7m, smul=\E[4m,
digilog|digilog 333,
	OTbs,
	cols#80, lines#16,
	bel=^G, cr=^M, cub1=^H, cud1=^J, cuf1=^I, cuu1=^O, el=^X,
	home=^N, ind=^J,
# The DWK was a terminal manufactured in the Soviet Union c.1986
dwk|dwk-vt|dwk terminal,
	am,
	cols#80, it#8, lines#24,
	acsc=+\^\,Q-S.M0\177`+a\:f'g#h#i#jXkClJmFnNo~qUs_tEuPv\\wKxW~_,
	bel=^G, clear=\EH\EJ, cr=^M, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p1%{32}%+%c%p2%{32}%+%c, cuu1=\EA, dch1=\EP,
	ed=\EJ, el=\EK, home=\EH, ht=^I, ich1=\EQ, ind=^J, kbs=\177,
	kcub1=\ED, kcud1=\EB, kcuf1=\EC, kcuu1=\EA, kdch1=\Ee,
	kf1=\Ef1, kf10=\Ef0, kf2=\Ef2, kf3=\Ef3, kf4=\Ef4, kf5=\Ef5,
	kf6=\Ef6, kf7=\Ef7, kf8=\Ef8, kf9=\Ef9, kich1=\Ed, knp=\Eh,
	kpp=\Eg, nel=^M^J, rev=\ET, ri=\ES, rmacs=\EG, rmso=\EX,
	sgr0=\EX, smacs=\EF, smso=\ET,
env230|envision230|envision 230 graphics terminal,
	xenl@,
	enacs@, mc0=\E[0i, mc4=\E[4i, mc5=\E[5i, rmacs@,
	sgr=\E[0%?%p1%t;7%;%?%p2%t;4%;%?%p3%t;7%;%?%p4%t;5%;%?%p6%t;1%;m$<2>,
	sgr0=\E[0m$<2>, smacs@, use=vt100,
# These execuports were impact-printer ttys with a 30- or maybe 15-cps acoustic
# coupler attached, the whole rig fitting in a suitcase and more or less
# portable.  Hot stuff for c.1977 :-) -- esr
ep48|ep4080|execuport 4080,
	OTbs, am, os,
	cols#80,
	bel=^G, cr=^M, cub1=^H, cud1=^J, hd=^\, hu=^^, ind=^J,
ep40|ep4000|execuport 4000,
	cols#136, use=ep4080,
# Adam Thompson <athompso@pangea.ca> tells us:
# Informer series - these are all portable units, resembling older
# automatic bread-baking machines.  The terminal looks like a `clamshell'
# design, but isn't.  The structure is similar to the Direct terminals,
# but only half the width.  The entire unit is only about 10" wide.
# It features an 8" screen (6" or 7" if you have color!), and an 9"x6"
# keyboard.  All the keys are crammed together, much like some laptop
# PCs today, but perhaps less well organized...all these units have a
# bewildering array of plugs on the back, including a built-in modem.
# The 305 was a color version of the 304; the 306 and 307 were mono and
# color terminals built for IBM bisync protocols.
# From: Paul Leondis <unllab@amber.berkeley.edu>
ifmr|Informer D304,
	OTbs, am,
	cols#80, lines#24,
	clear=\EZ, cub1=^H, cud1=^J, cuf1=\EC,
	cup=\EY%p2%{32}%+%c%p1%{32}%+%c, cuu1=\EA, dch1=\E\\,
	ed=\E/, el=\EQ, home=\EH, ich1=\E[, ri=\En, rmso=\EK, sgr0=\EK,
	smso=\EJ,
# Entry largely based on wy60 and has the features of wy60ak.
opus3n1+|Esprit Opus3n1+ in wy60 mode with ANSI arrow keys,
	am, bw, hs, km, mir, msgr, ul, xon,
	cols#80, lh#1, lines#24, lw#8, nlab#8, wsl#80,
	acsc=0wa_h[jukslrmqnxqzttuyv]wpxv, bel=^G, blink=\EG2,
	cbt=\EI, civis=\E`0, clear=\E*$<100>, cnorm=\E`1, cr=^M,
	cub1=^H, cud1=^J, cuf1=^L, cup=\Ea%i%p1%dR%p2%dC, cuu1=^K,
	dch1=\EW$<11>, dim=\EGp, dl1=\ER$<5>, dsl=\Ez(\r,
	ed=\EY$<100>, el=\ET, fsl=^M, home=\036$<2>, ht=\011$<5>,
	hts=\E1, if=/usr/share/tabset/std, il1=\EE$<4>, ind=^J,
	ip=$<3>,
	is2=\E`\:\Ee(\EO\Ee6\Ec41\E~4\Ec21\Ed/\Ezz&\E[A\177\Ezz'\E[B\177\Ezz(\E[D\177\Ezz)\E[C\177\Ezz<\E[Q\177\Ezz`\E[F\177\EA1*\EZH12,
	kHOM=\E{, kbs=^H, kcbt=\EI, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kdch1=\EW, kdl1=\ER, ked=\EY,
	kel=\ET, kend=\E[F, kent=\E7, kf1=^A@\r, kf10=^AI\r,
	kf11=^AJ\r, kf12=^AK\r, kf13=^AL\r, kf14=^AM\r, kf15=^AN\r,
	kf16=^AO\r, kf2=^AA\r, kf3=^AB\r, kf4=^AC\r, kf5=^AD\r,
	kf6=^AE\r, kf7=^AF\r, kf8=^AG\r, kf9=^AH\r, khome=^^,
	kich1=\EQ, kil1=\EE, knp=\EK, kpp=\EJ, kprt=\EP, krpl=\Er,
	mc0=\EP, mc4=^T, mc5=^R, nel=\r\n$<3>,
	pfloc=\EZ2%p1%{63}%+%c%p2%s\177,
	pfx=\EZ1%p1%{63}%+%c%p2%s\177,
	pln=\Ez%p1%{47}%+%c%p2%s\r, prot=\E), ri=\Ej$<7>,
	rmacs=\EH^C, rmam=\Ed., rmcup=, rmir=\Er, rmln=\EA11,
	rmxon=\Ec20, rs1=\E~!\E~4$<150>, rs2=\EeF$<150>,
	rs3=\EwG\Ee($<150>,
	sgr=%?%p8%t\E)%e\E(%;%?%p9%t\EH\002%e\EH\003%;\EG%{48}%?%p2%t%{8}%|%;%?%p1%p3%|%t%{4}%|%;%?%p4%t%{2}%|%;%?%p1%p5%|%t%{64}%|%;%?%p7%t%{1}%|%;%c,
	sgr0=\E(\EH\003\EG0\EcD, smacs=\EH^B, smam=\Ed/,
	smcup=\Ezz&\E[A\177\Ezz'\E[B\177\Ezz(\E[D\177\Ezz)\E[C\177\Ezz<\E[Q\177,
	smir=\Eq, smln=\EA10, smxon=\Ec21, tbc=\E0, tsl=\Ez(,
	uc=\EG8\EG0, use=adm+sgr,
teletec|Teletec Datascreen,
	OTbs, am,
	cols#80, lines#24,
	bel=^G, clear=^L, cr=^M, cub1=^H, cud1=^J, cuf1=^_, cuu1=^K,
	home=^^, ind=^J,
# From: Mark Dornfeld <romwa@ucbvax.berkeley.edu>
# This description is for the LANPAR Technologies VISION 3220
# terminal from 1984/85.  The function key definitions k0-k5 represent the
# edit keypad: FIND, INSERT HERE, REMOVE, SELECT, PREV SCREEN,
# NEXT SCREEN. The key definitions k6-k9 represent the PF1 to PF4 keys.
#
# Kenneth Randell <kenr@datametrics.com> writes on 31 Dec 1998:
# I had a couple of scopes (3221) like this once where I used to work, around
# the 1987 time frame if memory serves me correctly.  These scopes were made
# by an outfit called LANPAR Technologies, and were meant to me DEC VT 220
# compatible.  The 3220 was a plain text terminal like the VT-220, the 3221
# was a like the VT-240 (monochrome with Regis + Sixel graphics), and the 3222
# was like the VT-241 (color with Regis + Sixel Graphics).  These terminals
# (3221) cost about $1500 each, and one was always broken -- had to be sent
# back to the shop for repairs.
# The only real advantage these scopes had over the VT-240's were:
# 1) They were faster in the Regis display, or at least the ones I did
# 2) They had a handy debugging feature where you could split-screen the
# scope, the graphics would appear on the top, and the REGIS commands would
# appear on the bottom.  I don't remember the VT-240s being able to do that.
# I would swear that LANPAR Technologies was in MA someplace, but since I
# don't work at the same place anymore, and those terminals and manuals were
# long since junked, I cannot be any more sure than that.
#
# (v3220: removed obsolete ":kn#10:",
# I added <rmam>/<smam> based on the init string -- esr)
v3220|LANPAR Vision II model 3220/3221/3222,
	OTbs, am, mir, xenl,
	cols#80, it#8, lines#24,
	clear=\E[H\E[J, cub1=^H, cud1=\E[B, cuf1=\E[C,
	cup=\E[%i%p1%d;%p2%dH, cuu1=\E[A, dch1=\E[P, dl1=\E[M,
	ed=\E[J, el=\E[K, ht=^I, il1=\E[L,
	is2=\E>\E[?3l\E[?7h\E[?8h\E[p, kcub1=\E[D, kcud1=\E[B,
	kcuf1=\E[C, kcuu1=\E[A, kf0=\E[1~, kf1=\E[2~, kf2=\E[3~,
	kf3=\E[4~, kf4=\E[5~, kf5=\E[6~, kf6=\E[OP, kf7=\E[OQ,
	kf8=\E[OR, kf9=\E[OS, khome=\E[H, ri=\EM, rmam=\E[?7l,
	rmir=\E[4l, rmkx=\E>, rmso=\E[m, rmul=\E[m, sgr0=\E[m,
	smam=\E[?7h, smir=\E[4h, smkx=\E=, smso=\E[7m, smul=\E[4m,
######## ICH/ICH1 VERSUS RMIR/SMIR
#
# Some non-curses applications get confused if both ich/ich1 and rmir/smir
# are present; the symptom is doubled characters in an update using insert.
# These applications are technically correct; in both 4.3BSD termcap and
# terminfo, you're not actually supposed to specify both ich/ich1 and rmir/smir
# unless the terminal needs both.  To my knowledge, no terminal still in this
# file requires both other than the very obsolete dm2500.
#
# For ncurses-based applications this is not a problem, as ncurses uses
# one or the other as appropriate but never mixes the two.  Therefore we
# have not corrected entries like `linux' and `xterm' that specify both.
# If you see doubled characters from these, use the linux-nic and xterm-nic
# entries that suppress ich/ich1.  And upgrade to ncurses!
#

######## VT100/ANSI/ISO 6429/ECMA-48/PC-TERM TERMINAL STANDARDS
#
# ANSI X3.64 has been withdrawn and replaced by ECMA-48.  The ISO 6429 and
# ECMA-48 standards are said to be almost identical, but are not the same
# as X3.64 (though for practical purposes they are close supersets of it).
#
# You can obtain ECMA-48 for free by sending email to helpdesk@ecma.ch
# requesting the standard(s) you want (i.e. ECMA-48, "Control Functions for
# Coded Character Sets"), include your snail-mail address, and you should
# receive the document in due course.  Don't expect an email acknowledgement.
#
# Related standards include "X3.4-1977: American National Standard Code for
# Information Interchange" (the ASCII standard) and "X3.41.1974:
# Code-Extension Techniques for Use with the 7-Bit Coded Character Set of
# American National Standard for Information Interchange."  I believe (but
# am not certain) that these are effectively identical to ECMA-6 and ECMA-35
# respectively.
#

#### VT100/ANSI/ECMA-48
#
# ANSI Standard (X3.64) Control Sequences for Video Terminals and Peripherals
# and ECMA-48 Control Functions for Coded Character Sets.
#
# Much of the content of this comment is adapted from a table prepared by
# Richard Shuford, based on a 1984 Byte article.  Terminfo correspondences,
# discussion of some terminfo-related issues, and updates to capture ECMA-48
# have been added.  Control functions described in ECMA-48 only are tagged
# with * after their names.
#
# The table is a complete list of the defined ANSI X3.64/ECMA-48 control
# sequences.  In the main table, \E stands for an escape (\033) character,
# SPC for space.  Pn stands for a single numeric parameter to be inserted
# in decimal ASCII.  Ps stands for a list of such parameters separated by
# semicolons.  Parameter meanings for most parametrized sequences are
# decribed in the notes.
#
# Sequence     Sequence                             Parameter   or
# Mnemonic     Name              Sequence           Value      Mode   terminfo
# -----------------------------------------------------------------------------
# APC  Applicatn Program Command \E _                -         Delim  -
# BEL  Bell *                    ^G                  -         -      bel
# BPH  Break Permitted Here *    \E B                -         *      -
# BS   Backpace *                ^H                  -         EF     -
# CAN  Cancel *                  ^X                  -         -      -   (A)
# CBT  Cursor Backward Tab       \E [ Pn Z           1         eF     cbt
# CCH  Cancel Previous Character \E T                -         -      -
# CHA  Cursor Horizntal Absolute \E [ Pn G           1         eF     hpa (B)
# CHT  Cursor Horizontal Tab     \E [ Pn I           1         eF     tab (C)
# CMD  Coding Method Delimiter * \E
# CNL  Cursor Next Line          \E [ Pn E           1         eF     nel (D)
# CPL  Cursor Preceding Line     \E [ Pn F           1         eF     -
# CPR  Cursor Position Report    \E [ Pn ; Pn R      1, 1      -      -   (E)
# CSI  Control Sequence Intro    \E [                -         Intro  -
# CTC  Cursor Tabulation Control \E [ Ps W           0         eF     -   (F)
# CUB  Cursor Backward           \E [ Pn D           1         eF     cub
# CUD  Cursor Down               \E [ Pn B           1         eF     cud
# CUF  Cursor Forward            \E [ Pn C           1         eF     cuf
# CUP  Cursor Position           \E [ Pn ; Pn H      1, 1      eF     cup (G)
# CUU  Cursor Up                 \E [ Pn A           1         eF     cuu
# CVT  Cursor Vertical Tab       \E [ Pn Y           -         eF     -   (H)
# DA   Device Attributes         \E [ Pn c           0         -      -
# DAQ  Define Area Qualification \E [ Ps o           0         -      -
# DCH  Delete Character          \E [ Pn P           1         eF     dch
# DCS  Device Control String     \E P                -         Delim  -
# DL   Delete Line               \E [ Pn M           1         eF     dl
# DLE  Data Link Escape *        ^P                  -         -      -
# DMI  Disable Manual Input      \E \                -         Fs     -
# DSR  Device Status Report      \E [ Ps n           0         -      -   (I)
# DTA  Dimension Text Area *     \E [ Pn ; Pn SPC T  -         PC     -
# EA   Erase in Area             \E [ Ps O           0         eF     -   (J)
# ECH  Erase Character           \E [ Pn X           1         eF     ech
# ED   Erase in Display          \E [ Ps J           0         eF     ed  (J)
# EF   Erase in Field            \E [ Ps N           0         eF     -
# EL   Erase in Line             \E [ Ps K           0         eF     el  (J)
# EM   End of Medium *           ^Y                  -         -      -
# EMI  Enable Manual Input       \E b                          Fs     -
# ENQ  Enquire                   ^E                  -         -      -
# EOT  End Of Transmission       ^D                  -         *      -
# EPA  End of Protected Area     \E W                -         -      -   (K)
# ESA  End of Selected Area      \E G                -         -      -
# ESC  Escape                    ^[                  -         -      -
# ETB  End Transmission Block    ^W                  -         -      -
# ETX  End of Text               ^C                  -         -      -
# FF   Form Feed                 ^L                  -         -      -
# FNK  Function Key *            \E [ Pn SPC W       -         -      -
# GCC  Graphic Char Combination* \E [ Pn ; Pn SPC B  -         -      -
# FNT  Font Selection            \E [ Pn ; Pn SPC D  0, 0      FE     -
# GSM  Graphic Size Modify       \E [ Pn ; Pn SPC B  100, 100  FE     -   (L)
# GSS  Graphic Size Selection    \E [ Pn SPC C       none      FE     -
# HPA  Horz Position Absolute    \E [ Pn `           1         FE     -   (B)
# HPB  Char Position Backward    \E [ j              1         FE     -
# HPR  Horz Position Relative    \E [ Pn a           1         FE     -   (M)
# HT   Horizontal Tab *          ^I                  -         FE     -   (N)
# HTJ  Horz Tab w/Justification  \E I                -         FE     -
# HTS  Horizontal Tab Set        \E H                -         FE     hts
# HVP  Horz & Vertical Position  \E [ Pn ; Pn f      1, 1      FE     -   (G)
# ICH  Insert Character          \E [ Pn @           1         eF     ich
# IDCS ID Device Control String  \E [ SPC O          -         *      -
# IGS  ID Graphic Subrepertoire  \E [ SPC M          -         *      -
# IL   Insert Line               \E [ Pn L           1         eF     il
# IND  Index                     \E D                -         FE     -
# INT  Interrupt                 \E a                -         Fs     -
# JFY  Justify                   \E [ Ps SPC F       0         FE     -
# IS1  Info Separator #1 *       ^_                  -         *      -
# IS2  Info Separator #1 *       ^^                  -         *      -
# IS3  Info Separator #1 *       ^]                  -         *      -
# IS4  Info Separator #1 *       ^\                  -         *      -
# LF   Line Feed                 ^J                  -         -      -
# LS1R Locking Shift Right 1 *   \E ~                -         -      -
# LS2  Locking Shift 2 *         \E n                -         -      -
# LS2R Locking Shift Right 2 *   \E }                -         -      -
# LS3  Locking Shift 3 *         \E o                -         -      -
# LS3R Locking Shift Right 3 *   \E |                -         -      -
# MC   Media Copy                \E [ Ps i           0         -      -   (S)
# MW   Message Waiting           \E U                -         -      -
# NAK  Negative Acknowledge *    ^U                  -         *      -
# NBH  No Break Here *           \E C                -         -      -
# NEL  Next Line                 \E E                -         FE     nel (D)
# NP   Next Page                 \E [ Pn U           1         eF     -
# NUL  Null *                    ^@                  -         -      -
# OSC  Operating System Command  \E ]                -         Delim  -
# PEC  Pres. Expand/Contract *   \E Pn SPC Z         0         -      -
# PFS  Page Format Selection *   \E Pn SPC J         0         -      -
# PLD  Partial Line Down         \E K                -         FE     -   (T)
# PLU  Partial Line Up           \E L                -         FE     -   (U)
# PM   Privacy Message           \E ^                -         Delim  -
# PP   Preceding Page            \E [ Pn V           1         eF     -
# PPA  Page Position Absolute *  \E [ Pn SPC P       1         FE     -
# PPB  Page Position Backward *  \E [ Pn SPC R       1         FE     -
# PPR  Page Position Forward *   \E [ Pn SPC Q       1         FE     -
# PTX  Parallel Texts *          \E [ \              -         -      -
# PU1  Private Use 1             \E Q                -         -      -
# PU2  Private Use 2             \E R                -         -      -
# QUAD Typographic Quadding      \E [ Ps SPC H       0         FE     -
# REP  Repeat Char or Control    \E [ Pn b           1         -      rep
# RI   Reverse Index             \E M                -         FE     -   (V)
# RIS  Reset to Initial State    \E c                -         Fs     -
# RM   Reset Mode *              \E [ Ps l           -         -      -   (W)
# SACS Set Add. Char. Sep. *     \E [ Pn SPC /       0         -      -
# SAPV Sel. Alt. Present. Var. * \E [ Ps SPC ]       0         -      -   (X)
# SCI  Single-Char Introducer    \E Z                -         -      -
# SCO  Sel. Char. Orientation *  \E [ Pn ; Pn SPC k  -         -      -
# SCS  Set Char. Spacing *       \E [ Pn SPC g       -         -      -
# SD   Scroll Down               \E [ Pn T           1         eF     rin
# SDS  Start Directed String *   \E [ Pn ]           1         -      -
# SEE  Select Editing Extent     \E [ Ps Q           0         -      -   (Y)
# SEF  Sheet Eject & Feed *      \E [ Ps ; Ps SPC Y  0,0       -      -
# SGR  Select Graphic Rendition  \E [ Ps m           0         FE     sgr (O)
# SHS  Select Char. Spacing *    \E [ Ps SPC K       0         -      -
# SI   Shift In                  ^O                  -         -      -   (P)
# SIMD Sel. Imp. Move Direct. *  \E [ Ps ^           -         -      -
# SL   Scroll Left               \E [ Pn SPC @       1         eF     -
# SLH  Set Line Home *           \E [ Pn SPC U       -         -      -
# SLL  Set Line Limit *          \E [ Pn SPC V       -         -      -
# SLS  Set Line Spacing *        \E [ Pn SPC h       -         -      -
# SM   Select Mode               \E [ Ps h           none      -      -   (W)
# SO   Shift Out                 ^N                  -         -      -   (Q)
# SOH  Start Of Heading *        ^A                  -         -      -
# SOS  Start of String *         \E X                -         -      -
# SPA  Start of Protected Area   \E V                -         -      -   (Z)
# SPD  Select Pres. Direction *  \E [ Ps ; Ps SPC S  0,0       -      -
# SPH  Set Page Home *           \E [ Ps SPC G       -         -      -
# SPI  Spacing Increment         \E [ Pn ; Pn SPC G  none      FE     -
# SPL  Set Page Limit *          \E [ Ps SPC j       -         -      -
# SPQR Set Pr. Qual. & Rapid. *  \E [ Ps SPC X       0         -      -
# SR   Scroll Right              \E [ Pn SPC A       1         eF     -
# SRCS Set Reduced Char. Sep. *  \E [ Pn SPC f       0         -      -
# SRS  Start Reversed String *   \E [ Ps [           0         -      -
# SSA  Start of Selected Area    \E F                -         -      -
# SSU  Select Size Unit *        \E [ Pn SPC I       0         -      -
# SSW  Set Space Width *         \E [ Pn SPC [       none      -      -
# SS2  Single Shift 2 (G2 set)   \E N                -         Intro  -
# SS3  Single Shift 3 (G3 set)   \E O                -         Intro  -
# ST   String Terminator         \E \                -         Delim  -
# STAB Selective Tabulation *    \E [ Pn SPC ^       -         -      -
# STS  Set Transmit State        \E S                -         -      -
# STX  Start pf Text *           ^B                  -         -      -
# SU   Scroll Up                 \E [ Pn S           1         eF     indn
# SUB  Substitute *              ^Z                  -         -      -
# SVS  Select Line Spacing *     \E [ Pn SPC \       1         -      -
# SYN  Synchronous Idle *        ^F                  -         -      -
# TAC  Tabul. Aligned Centered * \E [ Pn SPC b       -         -      -
# TALE Tabul. Al. Leading Edge * \E [ Pn SPC a       -         -      -
# TATE Tabul. Al. Trailing Edge* \E [ Pn SPC `       -         -      -
# TBC  Tab Clear                 \E [ Ps g           0         FE     tbc
# TCC  Tabul. Centered on Char * \E [ Pn SPC c       -         -      -
# TSR  Tabulation Stop Remove  * \E [ Pn SPC d       -         FE     -
# TSS  Thin Space Specification  \E [ Pn SC E        none      FE     -
# VPA  Vert. Position Absolute   \E [ Pn d           1         FE     vpa
# VPB  Line Position Backward *  \E [ Pn k           1         FE     -
# VPR  Vert. Position Relative   \E [ Pn e           1         FE     -   (R)
# VT   Vertical Tabulation *     ^K                  -         FE     -
# VTS  Vertical Tabulation Set   \E J                -         FE     -
#
# ---------------------------------------------------------------------------
#
# Notes:
#
# Some control characters are listed in the ECMA-48 standard without
# being assigned functions relevant to terminal control there (they
# referred to other standards such as ISO 1745 or ECMA-35).  They are listed
# here anyway for completeness.
#
# (A) ECMA-48 calls this "CancelCharacter" but retains the CCH abbreviation.
#
# (B) There seems to be some confusion abroad between CHA and HPA.  Most
# `ANSI' terminals accept the CHA sequence, not the HPA. but terminfo calls
# the capability (hpa).  ECMA-48 calls this "Cursor Character Absolute" but
# preserved the CHA abbreviation.
#
# (C) CHT corresponds to terminfo (tab).  Usually it has the value ^I.
# Occasionally (as on, for example, certain HP terminals) this has the HTJ
# value.  ECMA-48 calls this "Cursor Forward Tabulation" but preserved the
# CHT abbreviation.
#
# (D) terminfo (nel) is usually \r\n rather than ANSI \EE.
#
# (E) ECMA-48 calls this "Active Position Report" but preserves the CPR
# abbreviation.
#
# (F) CTC parameter values: 0 = set char tab, 1 = set line tab, 2 = clear
# char tab, 3 = clear line tab, 4 = clear all char tabs on current line,
# 5 = clear all char tabs, 6 = clear all line tabs.
#
# (G) CUP and HVP are identical in effect.  Some ANSI.SYS versions accept
# HVP, but always allow CUP as an alternate.  ECMA-48 calls HVP "Character
# Position Absolute" but retains the HVP abbreviation.
#
# (H) ECMA calls this "Cursor Line Tabulation" but preserves the CVT
# abbreviation.
#
# (I) DSR parameter values: 0 = ready, 1 = busy, 2 = busy, will send DSR
# later, 3 = malfunction, 4 = malfunction, will send DSR later, 5 = request
# DSR, 6 = request CPR response.
#
# (J) ECMA calls ED "Erase In Page". EA/ED/EL parameters: 0 = clear to end,
# 1 = clear from beginning, 2 = clear.
#
# (K) ECMA calls this "End of Guarded Area" but preserves the EPA abbreviation.
#
# (L) The GSM parameters are vertical and horizontal parameters to scale by.
#
# (M) Some ANSI.SYS versions accept HPR, but more commonly `ANSI' terminals
# use CUF for this function and ignore HPR.  ECMA-48 calls this "Character
# Position Relative" but retains the HPR abbreviation.
#
# (N) ECMA-48 calls this "Character Tabulation" but retains the HT
# abbreviation.
#
# (O) SGR parameter values: 0 = default mode (attributes off), 1 = bold,
# 2 = dim, 3 = italicized, 4 = underlined, 5 = slow blink, 6 = fast blink,
# 7 = reverse video, 8 = invisible, 9 = crossed-out (marked for deletion),
# 10 = primary font, 10 + n (n in 1..9) = nth alternative font, 20 = Fraktur,
# 21 = double underline, 22 = turn off 2, 23 = turn off 3, 24 = turn off 4,
# 25 = turn off 5, 26 = proportional spacing, 27 = turn off 7, 28 = turn off
# 8, 29 = turn off 9, 30 = black fg, 31 = red fg, 32 = green fg, 33 = yellow
# fg, 34 = blue fg, 35 = magenta fg, 36 = cyan fg, 37 = white fg, 38 = set
# fg color as in CCIT T.416, 39 = set default fg color, 40 = black bg
# 41 = red bg, 42 = green bg, 43 = yellow bg, 44 = blue bg, 45 = magenta bg,
# 46 = cyan bg, 47 = white bg, 48 = set bg color as in CCIT T.416, 39 = set
# default bg color, 50 = turn off 26, 51 = framed, 52 = encircled, 53 =
# overlined, 54 = turn off 51 & 52, 55 = not overlined, 56-59 = reserved,
# 61-65 = variable highlights for ideograms.
#
# (P) SI is also called LSO, Locking Shift Zero.
#
# (Q) SI is also called LS1, Locking Shift One.
#
# (R) Some ANSI.SYS versions accept VPR, but more commonly `ANSI' terminals
# use CUD for this function and ignore VPR.  ECMA calls it `Line Position
# Absolute' but retains the VPA abbreviation.
#
# (S) MC parameters: 0 = start xfer to primary aux device, 1 = start xfer from
# primary aux device, 2 = start xfer to secondary aux device, 3 = start xfer
# from secondary aux device, 4 = stop relay to primary aux device, 5 =
# start relay to primary aux device, 6 = stop relay to secondary aux device,
# 7 = start relay to secondary aux device.
#
# (T) ECMA-48 calls this "Partial Line Forward" but retains the PLD
# abbreviation.
#
# (U) ECMA-48 calls this "Partial Line Backward" but retains the PLU
# abbreviation.
#
# (V) ECMA-48 calls this "Reverse Line Feed" but retains the RI abbreviation.
#
# (W) RM/SM modes are as follows: 1 = Guarded Area Transfer Mode (GATM),
# 2 = Keyboard Action Mode (KAM), 3 = Control Representation Mode (CRM),
# 4 = Insertion Replacement Mode, 5 = Status Report Transfer Mode (SRTM),
# 6 = Erasure Mode (ERM), 7 = Line Editing Mode (LEM), 8 = Bi-Directional
# Support Mode (BDSM), 9 = Device Component Select Mode (DCSM),
# 10 = Character Editing Mode (HEM), 11 = Positioning Unit Mode (PUM),
# 12 = Send/Receive Mode, 13 = Format Effector Action Mode (FEAM),
# 14 = Format Effector Transfer Mode (FETM), 15 = Multiple Area Transfer
# Mode (MATM), 16 = Transfer Termination Mode, 17 = Selected Area Transfer
# Mode, 18 = Tabulation Stop Mode, 19 = Editing Boundary Mode, 20 = Line Feed
# New Line Mode (LF/NL), Graphic Rendition Combination Mode (GRCM), 22 =
# Zero Default Mode (ZDM).  The EBM and LF/NL modes have actually been removed
# from ECMA-48's 5th edition but are listed here for reference.
#
# (X) Select Alternate Presentation Variants is used only for non-Latin
# alphabets.
#
# (Y) "Select Editing Extent" (SEE) was ANSI "Select Edit Extent Mode" (SEM).
#
# (Z) ECMA-48 calls this "Start of Guarded Area" but retains the SPA
# abbreviation.
#
# ---------------------------------------------------------------------------
#
# Abbreviations:
#
# Intro  an Introducer of some kind of defined sequence; the normal 7-bit
#        X3.64 Control Sequence Introducer is the two characters "Escape ["
#
# Delim  a Delimiter
#
# x/y    identifies a character by position in the ASCII table (column/row)
#
# eF     editor function (see explanation)
#
# FE     format effector (see explanation)
#
# F      is a Final character in
#             an Escape sequence (F from 3/0 to 7/14 in the ASCII table)
#             a control sequence (F from 4/0 to 7/14)
#
# Gs     is a graphic character appearing in strings (Gs ranges from
#        2/0 to 7/14) in the ASCII table
#
# Ce     is a control represented as a single bit combination in the C1 set
#        of controls in an 8-bit character set
#
# C0     the familiar set of 7-bit ASCII control characters
#
# C1     roughly, the set of control chars available only in 8-bit systems.
#        This is too complicated to explain fully here, so read Jim Fleming's
#        article in the February 1983 BYTE, especially pages 214 through 224.
#
# Fe     is a Final character of a 2-character Escape sequence that has an
#        equivalent representation in an 8-bit environment as a Ce-type
#        (Fe ranges from 4/0 to 5/15)
#
# Fs     is a Final character of a 2-character Escape sequence that is
#        standardized internationally with identical representation in 7-bit
#        and 8-bit environments and is independent of the currently
#        designated C0 and C1 control sets (Fs ranges from 6/0 to 7/14)
#
# I      is an Intermediate character from 2/0 to 2/15 (inclusive) in the
#        ASCII table
#
# P      is a parameter character from 3/0 to 3/15 (inclusive) in the ASCII
#        table
#
# Pn     is a numeric parameter in a control sequence, a string of zero or
#        more characters ranging from 3/0 to 3/9 in the ASCII table
#
# Ps     is a variable number of selective parameters in a control sequence
#        with each selective parameter separated from the other by the code
#        3/11 (which usually represents a semicolon); Ps ranges from
#        3/0 to 3/9 and includes 3/11
#
# *      Not relevant to terminal control, listed for completeness only.
#
# Format Effectors versus Editor Functions
#
# A format effector specifies how following output is to be displayed.
# An editor function allows you to modify the display.  Informally
# format effectors may be destructive; format effectors should not be.
#
# For instance, a format effector that moves the "active position" (the
# cursor or equivalent) one space to the left would be useful when you want to
# create an overstrike, a compound character made of two standard characters
# overlaid. Control-H, the Backspace character, is actually supposed to be a
# format effector, so you can do this. But many systems use it in a
# nonstandard fashion, as an editor function, deleting the character to the
# left of the cursor and moving the cursor left. When Control-H is assumed to
# be an editor function, you cannot predict whether its use will create an
# overstrike unless you also know whether the output device is in an "insert
# mode" or an "overwrite mode". When Control-H is used as a format effector,
# its effect can always be predicted. The familiar characters carriage
# return, linefeed, formfeed, etc., are defined as format effectors.
#
# NOTES ON THE DEC VT100 IMPLEMENTATION
#
# Control sequences implemented in the VT100 are as follows:
#
#      CPR, CUB, CUD, CUF, CUP, CUU, DA, DSR, ED, EL, HTS, HVP, IND,
#      LNM, NEL, RI, RIS, RM, SGR, SM, TBC
#
# plus several private DEC commands.
#
# Erasing parts of the display (EL and ED) in the VT100 is performed thus:
#
#      Erase from cursor to end of line           Esc [ 0 K    or Esc [ K
#      Erase from beginning of line to cursor     Esc [ 1 K
#      Erase line containing cursor               Esc [ 2 K
#      Erase from cursor to end of screen         Esc [ 0 J    or Esc [ J
#      Erase from beginning of screen to cursor   Esc [ 1 J
#      Erase entire screen                        Esc [ 2 J
#
# Some brain-damaged terminal/emulators respond to Esc [ J as if it were
# Esc [ 2 J, but this is wrong; the default is 0.
#
# The VT100 responds to receiving the DA (Device Attributes) control
#
#      Esc [ c    (or Esc [ 0 c)
#
# by transmitting the sequence
#
#      Esc [ ? l ; Ps c
#
# where Ps is a character that describes installed options.
#
# The VT100's cursor location can be read with the DSR (Device Status
# Report) control
#
#      Esc [ 6 n
#
# The VT100 reports by transmitting the CPR sequence
#
#      Esc [ Pl ; Pc R
#
# where Pl is the line number and Pc is the column number (in decimal).
#
# The specification for the DEC VT100 is document EK-VT100-UG-003.

#### ANSI.SYS
#
# Here is a description of the color and attribute controls supported in the
# the ANSI.SYS driver under MS-DOS.  Most console drivers and ANSI
# terminal emulators for Intel boxes obey these.  They are a proper subset
# of the ECMA-48 escapes.
#
# 0	all attributes off
# 1	foreground bright
# 4	underscore on
# 5	blink on/background bright (not reliable with brown)
# 7	reverse-video
# 8	set blank (non-display)
# 10	set primary font
# 11	set first alternate font (on PCs, display ROM characters 1-31)
# 12	set second alternate font (on PCs, display IBM high-half chars)
#
#			Color attribute sets
# 3n	set foreground color       / 0=black, 1=red,     2=green, 3=brown,
# 4n	set background color       \ 4=blue,  5=magenta, 6=cyan,  7=white
# Bright black becomes gray.  Bright brown becomes yellow,
# These coincide with the prescriptions of the ISO 6429/ECMA-48 standard.
#
# * If the 5 attribute is on and you set a background color (40-47) it is
#   supposed to enable bright background.
#
# * Many VGA cards (such as the Paradise and compatibles) do the wrong thing
#   when you try to set a "bright brown" (yellow) background with attribute
#   5 (you get a blinking yellow foreground instead).  A few displays
#   (including the System V console) support an attribute 6 that undoes this
#   braindamage (this is required by iBCS2).
#
# * Some older versions of ANSI.SYS have a bug that causes thems to require
#   ESC [ Pn k as EL rather than the ANSI ESC [ Pn K.  (This is not ECMA-48
#   compatible.)

#### Intel Binary Compatibility Standard
#
# For comparison, here are the capabilities implied by the Intel Binary
# Compatibility Standard for UNIX systems (Intel order number 468366-001).
# These recommendations are optional.  IBCS2 allows the leading escape to
# be either the 7-bit \E[ or 8-bit \0233 introducer, in accordance with
# the ANSI X.364/ISO 6429/ECMA-48 standard.  Here are the iBCS2 capabilities
# (as described in figure 9-3 of the standard).  Those expressed in the ibcs2
# terminfo entry are followed with the corresponding capability in parens:
#
#	CSI <n>k		disable (n=0) or enable (n=1) keyclick
#	CSI 2h   		lock keyboard
#	CSI 2i  		send screen as input
#	CSI 2l  		unlock keyboard
#	CSI 6m  		enable background color intensity
#	CSI <0-2>c		reserved
#	CSI <0-59>m		select graphic rendition
#	CSI <n>;<m>H	(cup)	cursor to line n and column m
#	CSI <n>;<m>f		cursor to line n and column m
#	CSI <n>@	(ich)	insert characters
#	CSI <n>A	(cuu)	cursor up n lines
#	CSI <n>B	(cud)	cursor down n lines
#	CSI <n>C	(cuu)	cursor right n characters
#	CSI <n>D	(cud)	cursor left n characters
#	CSI <n>E		cursor down n lines and in first column
#	CSI <n>F		cursor up n lines and in first column
#	CSI <n>G	(hpa)	position cursor at column n-1
#	CSI <n>J	(ed)	erase in display
#	CSI <n>K	(el)	erase in line
#	CSI <n>L	(il)	insert line(s)
#	CSI <n>P	(dch)	delete characters
#	CSI <n>S	(indn)	scroll up n lines
#	CSI <n>T	(rin)	scroll down n lines
#	CSI <n>X	(ech)	erase characters
#	CSI <n>Z	(cbt)	back up n tab stops
#	CSI <n>`		cursor to column n on line
#	CSI <n>a	(cuu)	cursor right n characters
#	CSI <n>d	(vpa)	cursor to line n
#	CSI <n>e		cursor down n lines and in first column
#	CSI <n>g	(cbt)	clear all tabs
#	CSI <n>z		make virtual terminal n active
#	CSI ?7h 	(smam)	turn automargin on
#	CSI ?7l 	(rmam)	turn automargin off
#	CSI s     		save cursor position
#	CSI u   		restore cursor position to saved value
#	CSI =<c>A		set overscan color
#	CSI =<c>F		set normal foreground color
#	CSI =<c>G		set normal background color
#	CSI =<c>H		set reverse foreground color
#	CSI =<c>I		set reverse foreground color
#	CSI =<c>J		set graphic foreground color
#	CSI =<c>K		set graphic foreground color
#	CSI =<n>g	(dispc) display n from alternate graphics character set
#	CSI =<p>;<d>B		set bell parameters
#	CSI =<s>;<e>C		set cursor parameters
#	CSI =<x>D		enable/disable intensity of background color
#	CSI =<x>E		set/clear blink vs. bold background
#	CSI 7     	(sc)	(sc) save cursor position
#	CSI 8   	(rc)	(rc) restore cursor position to saved value
#	CSI H		(hts)	(hts) set tab stop
#	CSI Q<n><string>	define function key string
#				(string must begin and end with delimiter char)
#	CSI c   	(clear) clear screen
#
# The lack of any specification for attributes in SGR (among other things)
# makes this a wretchedly weak standard. The table above is literally
# everything iBSC2 has to say about terminal escape sequences; there is
# no further discussion of their meaning or how to set the parameters
# in these sequences at all.
#

######## NONSTANDARD CAPABILITY TRANSLATIONS USED IN THIS FILE
#
# The historical termcap file entries were written primarily in 4.4BSD termcap.
# The 4.4BSD termcap set was substantially larger than the original 4.1BSD set,
# with the extension names chosen for compatibility with the termcap names
# assigned in System V terminfo.  There are some variant extension sets out
# there.  We try to describe them here.
#
#### XENIX extensions:
#
# The XENIX extensions include a set of function-key capabilities as follows:
#
#       code	XENIX variable name	terminfo name	name clashes?
#	----	-------------------	-------------	-----------------------
#	CL	key_char_left
#	CR	key_char_right
#	CW	key_change_window			create_window
#	EN	key_end          	kend
#	HM	key_home		khome
#	HP	??
#	LD	key_delete_line  	kdl1
#	LF	key_linefeed     			label_off
#	NU	key_next_unlocked_cell
#	PD	key_page_down   	knp
#	PL	??
#	PN	start_print		mc5
#	PR	??
#	PS	stop_print		mc4
#	PU	key_page_up     	kpp		pulse
#	RC	key_recalc				remove_clock
#	RF	key_toggle_ref				req_for_input
#	RT	key_return      	kent
#	UP	key_up_arrow           	kcuu1   	parm_up_cursor
#	WL	key_word_left
#	WR	key_word_right
#
# The XENIX extensions also include the following character-set and highlight
# capabilities:
#
#	XENIX	terminfo	function
#	-----	--------	------------------------------
#	GS	smacs		start alternate character set
#	GE	rmacs		end alternate character set
#	GG			:as:/:ae: glitch (analogous to :sg:/:ug:)
#	bo	blink		begin blink (not used in /etc/termcap)
#	be			end blink (not used in /etc/termcap)
#	bb			blink glitch  (not used in /etc/termcap)
#	it	dim		begin dim (not used in /etc/termcap)
#	ie			end dim (not used in /etc/termcap)
#	ig			dim glitch  (not used in /etc/termcap)
#
# Finally, XENIX also used the following forms-drawing capabilities:
#
#	single	double  type             ASCII approximation
#	------	------	-------------    -------------------
#	GV	Gv	vertical line             |
#	GH	Gv	horizontal line       -   _
#	G1	G5	top right corner       _   |
#	G2	G6	top left corner       |
#	G3	G7	bottom left corner         |_
#	G4	G8	bottom right corner   _|
#	GD	Gd	down-tick character        T
#	GL	Gl	left-tick character   -|
#	GR	Gr	right-tick character       |-
#	GC	Gc	middle intersection   -|-
#	GU	Gu	up-tick character          _|_
#
# These were invented to take advantage of the IBM PC ROM character set.  One
# can compose an acsc string from the single-width characters as follows
#	"j{G4}k{G1}l{G2}m{G3}q{GH}x{GV}t{GR}u{GL}v{GU}w{GD}n{GC}"
# When translating a termcap file, ncurses tic will do this automatically.
# The double forms characters don't fit the SVr4 terminfo model.
#
#### AT&T Extensions:
#
# The old AT&T 5410, 5420, 5425, pc6300plus, 610, and s4 entries used a set of
# nonstandard capabilities.  Its signature is the KM capability, used to name
# some sort of keymap file.  EE, BO, CI, CV, XS, DS, FL and FE are in this
# set.  Comments in the original, and a little cross-checking with other AT&T
# documentation, seem to establish that BO=:mr: (start reverse video), DS=:mh:
# (start dim), XS=:mk: (secure/invisible mode), EE=:me: (end highlights),
# FL=:LO: (enable soft labels), FE=:LF: (disable soft labels), CI=:vi: (make
# cursor invisible), and CV=:ve: (make cursor normal).
#
#### HP Extensions
#
# The HP library (as of mid-1995, their term.h file version 70.1) appears to
# have the System V capabilities up to SVr1 level.  After that, it supports
# two nonstandard caps meml and memu corresponding to the old termcap :ml:,
# :mu: capabilities.  After that, it supports caps plab_norm, label_on,
# label_off, and key_f11..key_f63 capabilities like SVr4's.  This makes the
# HP binary format incompatible with SVr4's.
#
#### IBM Extensions
#
# There is a set of nonstandard terminfos used by IBM's AIX operating system.
# The AIX terminfo library diverged from SVr1 terminfo, and replaces all
# capabilities following prtr_non with the following special capabilties:
# box[12], batt[12], colb[0123456789], colf[0123456789], f[01234567], kbtab,
# kdo, kcmd, kcpn, kend, khlp, knl, knpn, kppn, kppn, kquit, ksel, kscl, kscr,
# ktab, kmpf[123456789], apstr, ksf1..ksf10, kf11...kf63, kact, topl, btml,
# rvert, lvert.   Some of these are identical to XPG4/SVr4 equivalents:
# kcmd, kend, khlp, and kf11...kf63.  Two others (kbtab and ksel) can be
# renamed (to kcbt and kslt).  The places in the box[12] capabilities
# correspond to acsc chars, here is the mapping:
#
#	box1[0]  = ACS_ULCORNER
#	box1[1]  = ACS_HLINE
#	box1[2]  = ACS_URCORNER
#	box1[3]  = ACS_VLINE
#	box1[4]  = ACS_LRCORNER
#	box1[5]  = ACS_LLCORNER
#	box1[6]  = ACS_TTEE
#	box1[7]  = ACS_RTEE
#	box1[8]  = ACS_BTEE
#	box1[9]  = ACS_LTEE
#	box1[10] = ACS_PLUS
#
# The box2 characters are the double-line versions of these forms graphics.
# The AIX binary terminfo format is incompatible with SVr4's.
#
#### Iris console extensions:
#
# HS is half-intensity start; HE is half-intensity end
# CT is color terminal type (for Curses & rogue)
# CP is color change escape sequence
# CZ are color names (for Curses & rogue)
#
# The ncurses tic utility recognizes HS as an alias for mh <dim>.
#
#### TC Extensions:
#
# There is a set of extended termcaps associated with something
# called the "Terminal Control" or TC package created by MainStream Systems,
# Winfield Kansas.  This one also uses GS/GE for as/ae, and also uses
# CF for civis and CO for cvvis.  Finally, they define a boolean :ct:
# that flags color terminals.
#
######## NCURSES USER-DEFINABLE CAPABILITIES
#
# Extensions added after ncurses 5.0 generally use the "-x" option of tic and
# infocmp to manipulate user-definable capabilities.  Those that are intended
# for use in either terminfo or termcap use 2-character names.  Extended
# function keys do not use 2-character names, and are available only with
# terminfo.
#
# As of mid-2012, no other terminfo/termcap implementation than ncurses
# supports this extension; termcap libraries can as noted above make limited
# use of the feature.
#
# ncurses makes explicit checks for a few user-definable capabilities:  AX, U8,
# XM.
#
#### SCREEN Extensions:
#
# The screen program uses the termcap interface.  It recognizes a few useful
# nonstandard capabilities.  Those are used in this file.
#
#       AX   (bool)  Does  understand  ANSI  set  default fg/bg color (\E[39m /
#                    \E[49m).
#       G0   (bool)  Terminal can deal with ISO 2022  font  selection sequences.
#       E0   (str)   Switch charset 'G0' back to standard charset.
#       S0   (str)   Switch charset 'G0' to the specified charset.
#       XT   (bool)  Terminal understands special xterm sequences  (OSC,  mouse
#                    tracking).
#
# AX is relatively straightforward; it is interpreted by ncurses to say that
# SGR 39/49 reset the terminal's foreground and background colors to their
# "default".
#
# XT is harder, since screen's manpage does not give more details.  For that,
# we must read screen's source-code.  When XT is set, screen assumes
#
# a) OSC 1 sets the title string, e.g., for the icon.  Recent versions of
#    screen may also set the terminal's name, which is (for xterm) distinct
#    from the icon name.
# b) OSC 20 sets the background pixmap.  This is an rxvt feature.
# c) OSC 39 and OSC 49 set the default foreground/background colors.  Again
#    this is an rxvt feature.
# d) certain mode settings enable the mouse: 9, 1000, 1001, 1002, 1003.
#    These are from xterm, although xterm accepts mouse codes that may not be
#    recognized by screen, e.g., 1005, 1006.
# e) colors beyond 0..7 are implemented by xterm's aixterm-like 16-color
#    sequence.  However, because screen uses only termcap, the values returned
#    by Af/Ab are not usable because they rely on expressions that termcap
#    does not support.  Therefore, screen uses a hardcoded string to work
#    around the limitation.
# f) all entries named "*xterm*" or "*rxvt*" have the bce flag set.
#
# The other ISO-2022 features are rarely used, but provided here to make
# screen's termcap features available.
#
#### XTERM Extensions:
#
# Most of the xterm extensions are for function-keys.  Since patch #94 (in
# 1999), xterm has supported shift/control/alt/meta modifiers which produce
# additional function-key strings.  Some other developers copied the feature,
# though they did not follow xterm's lead in patch #167 (in 2002), to make
# these key definitions less ambiguous.
#
# A few terminals provide similar functionality (sending distinct keys when
# a modifier is used), including rxvt.
#
# These are the extended keys defined in this file:
#
# kDC3 kDC4 kDC5 kDC6 kDC7 kDN kDN3 kDN4 kDN5 kDN6 kDN7 kEND3 kEND4 kEND5 kEND6
# kEND7 kHOM3 kHOM4 kHOM5 kHOM6 kHOM7 kIC3 kIC4 kIC5 kIC6 kIC7 kLFT3 kLFT4
# kLFT5 kLFT6 kLFT7 kNXT3 kNXT4 kNXT5 kNXT6 kNXT7 kPRV3 kPRV4 kPRV5 kPRV6 kPRV7
# kRIT3 kRIT4 kRIT5 kRIT6 kRIT7 kUP kUP3 kUP4 kUP5 kUP6 kUP7 ka2 kb1 kb3 kc2
#
# Here are the other xterm-related extensions which are used in this file:
#
# Cr is a string capability which resets the cursor color
# Cs is a string capability which sets the cursor color to a given value.
#    The single string parameter is the color name/number, according to the
#    implementation.
# Ms modifies the selection/clipboard.  Its parameters are
#	p1 = the storage unit (clipboard, selection or cut buffer)
#	p2 = the base64-encoded clipboard content.
# Se resets the cursor style to the terminal power-on default.
# Ss is a string capability with one numeric parameter.  It is used to set the
#    cursor style as described by the DECSCUSR function to a block or
#    underline.
# XM is a string capability which overrides ncurses's built-in string which
#    enables xterm mouse mode.
#
#### Miscellaneous extensions:
#
# gsbom/grbom are used to enable/disable real bold (not intensity bright) mode.
#    This was implemented for the Hurd.
# E3 clears the terminal's scrollback buffer.  This was implemented in the
#    Linux 3.0 kernel as a security feature.  It matches a feature which was
#    added in xterm patch #107.
# U8 is a numeric capability which denotes a terminal emulator which does not
#    support VT100 SI/SO when processing UTF-8 encoding.  Set this to a nonzero
#    value to enable it.
#
######## CHANGE HISTORY
#
# The last /etc/termcap version maintained by John Kunze was 8.3, dated 8/5/94.
# Releases 9 and 10 (up until the release of ncurses 4.2 in 1998) were
# maintained by Eric S. Raymond as part of the ncurses project.
#
# This file contains all the capability information present in John Kunze's
# last version of the termcap master file, except as noted in the change
# comments at end of file.  Some information about very ancient obsolete
# capabilities has been moved to comments.  Some all-numeric names of older
# terminals have been retired.
#
# I changed :MT: to :km: (the 4.4BSD name) everywhere.  I commented out some
# capabilities (EP, dF, dT, dV, kn, ma, ml, mu, xr, xx) that are no longer
# used by BSD curses.
#
# The 9.1.0 version of this file was translated from my lightly-edited copy of
# 8.3, then mechanically checked against 8.3 using Emacs Lisp code written for
# the purpose.  Unless the ncurses tic implementation and the Lisp code were
# making perfectly synchronized mistakes which I then failed to catch by
# eyeball, the translation was correct and perfectly information-preserving.
#
# Major version number bumps correspond to major version changes in ncurses.
#
# Here is a log of the changes since then:
#
# 9.1.0 (Wed Feb  1 04:50:32 EST 1995):
#	* First terminfo master translated from 8.3.
# 9.2.0 (Wed Feb  1 12:21:45 EST 1995):
#	* Replaced Wyse entries with updated entries supplied by vendor.
#
# 9.3.0 (Mon Feb  6 19:14:40 EST 1995):
#	* Added contact & status info from G. Clark Brown <clark@sssi.com>.
# 9.3.1 (Tue Feb  7 12:00:24 EST 1995):
#	* Better XENIX keycap translation.  Describe TC termcaps.
#	* Contact and history info supplied by Qume.
# 9.3.2 (Sat Feb 11 23:40:02 EST 1995):
#	* Raided the Shuford FTP site for recent termcaps/terminfos.
#	* Added information on X3.64 and VT100 standard escape sequences.
# 9.3.3 (Mon Feb 13 12:26:15 EST 1995):
#	* Added a correct X11R6 xterm entry.
#	* Fixed terminfo translations of padding.
# 9.3.4 (Wed Feb 22 19:27:34 EST 1995):
#	* Added correct acsc/smacs/rmacs strings for vt100 and xterm.
#	* Added u6/u7/u8/u9 capabilities.
#	* Added PCVT entry.
# 9.3.5 (Thu Feb 23 09:37:12 EST 1995):
#	* Emacs uses :so:, not :mr:, for its mode line.  Fix linux entry
#	  to use reverse-video standout so Emacs will look right.
#	* Added el1 capability to ansi.
#	* Added smacs/rmacs to ansi.sys.
#
# 9.4.0 (Sat Feb 25 16:43:25 EST 1995):
#	* New mt70 entry.
#	* Added COPYRIGHTS AND OTHER DELUSIONS.
#	* Added AT&T 23xx & 500/513, vt220 and vt420, opus3n1+, netronics
#	  smartvid & smarterm, ampex 175 & 219 & 232,
#	  env230, falco ts100, fluke, intertube, superbrain, ncr7901, vic20,
#	  ozzie, trs200, tr600, Tandy & Texas Instruments VDTs, intext2,
#	  screwpoint, fviewpoint, Contel Business Systems, Datamedia Colorscan,
#	  adm36, mime314, ergo4000, ca22851.  Replaced att7300, esprit, dd5500.
#	* Replaced the Perkin-Elmer entries with vendor's official ones.
#	* Restored the old minimal-ansi entry, luna needs it.
#	* Fixed some incorrect ip and proportional-padding translations.
# 9.4.1 (Mon Feb 27 14:18:33 EST 1995):
#	* Fix linux & AT386 sgr strings to do A_ALTCHARSET turnoff correctly.
#	* Make the xterm entry 65 lines again; create xterm25 and xterm24
#	  to force a particular height.
#	* Added beehive4 and reorganized other Harris entries.
# 9.4.2 (Thu Mar  9 01:45:44 EST 1995):
#	* Merged in DEC's official entries for its terminals.  The only old
#	  entry I kept was Doug Gwyn's alternate vt100 (as vt100-avo).
#	* Replaced the translated BBN Bitgraph entries with purpose-built
#	  ones from AT&T's SVr3.
#	* Replaced the AT&T entries with AT&T's official terminfos.
#	* Added teleray 16, vc415, cops10.
#	* Merged in many individual capabilities from SCO terminfo files.
# 9.4.3 (Mon Mar 13 02:37:53 EST 1995):
#	* Typo fixes.
#	* Change linux entry so A_PROTECT enables IBM-PC ROM characters.
# 9.4.4 (Mon Mar 27 12:32:35 EST 1995):
#	* Added tty35, Ann Arbor Guru series. vi300 and 550, cg7900, tvi803,
#	  pt210, ibm3164, IBM System 1, ctrm, Tymshare scanset, dt200, adm21,
#	  simterm, citoh and variants.
#	* Replaced sol entry with sol1 and sol2.
#	* Replaced Qume QVT and Freedom-series entries with purpose-built
#	  terminfo entries.
#	* Enhanced vt220, tvi910, tvi924, hpterm, hp2645, adm42, tek
#	  and dg200 entries using caps from from SCO.
#	* Added the usual set of function-key mappings to ANSI entry.
#	* Corrected xterm's function-key capabilities.
# 9.4.5 (Tue Mar 28 14:27:49 EST 1995):
#	* Fix in xterm entry, cub and cud are not reliable under X11R6.
# 9.4.6 (Thu Mar 30 14:52:15 EST 1995):
#	* Fix in xterm entry, get the arrow keys right.
#	* Change some \0 escapes to \200.
# 9.4.7 (Tue Apr  4 11:27:11 EDT 1995)
#	* Added apple (Videx card), adm1a, oadm31.
#	* Fixed malformed ampex csr.
#	* Fixed act4, cyb110; they had old-style prefix padding left in.
#	* Changed mandatory to advisory padding in many entries.
#	* Replaced HP entries up to hpsub with purpose-built ones.
#	* Blank rmir/smir/rmdc/smdc capabilities removed.
#	* Small fixes merged in from SCO entries for lpr, fos, tvi910+, tvi924.
# 9.4.8 (Fri Apr  7 09:36:34 EDT 1995):
#	* Replaced the Ann Arbor entries with SCO's, the init strings are
#	  more efficient (but the entries otherwise identical).
#	* Added dg211 from Shuford archive.
#	* Added synertek, apple-soroc, ibmpc, pc-venix, pc-coherent, xtalk,
#	  adm42-nl, pc52, gs6300, xerox820, uts30.
#	* Pull SCO's padding into vi200 entry.
#	* Improved capabilities for tvi4107 and other Televideo and Viewpoint
#	  entries merged in from SCO's descriptions.
#	* Fixed old-style prefix padding on zen50, h1500.
#	* Moved old superbee entry to superbee-xsb, pulled in new superbee
#	  entry from SCO's description.
#	* Reorganized the special entries.
#	* Added lm#0 to cbunix and virtual entries.
#
# 9.5.0 (Mon Apr 10 11:30:00 EDT 1995):
#	* Restored cdc456tst.
#	* Fixed sb1 entry, SCO erroneously left out the xsb glitch.
#	* Added megatek, beacon, microkit.
#	* Freeze for ncurses-1.9 release.
# 9.5.1 (Fri Apr 21 12:46:42 EDT 1995):
#	* Added historical data for TAB.
#	* Comment fixes from David MacKenzie.
#	* Added the new BSDI pc3 entry.
# 9.5.2 (Tue Apr 25 17:27:52 EDT 1995)
#	* A change in the tic -C logic now ensures that all entries in
#	  the termcap translation will fit in < 1024 bytes.
#	* Added `bobcat' and `gator' HP consoles and the Nu machine entries
#	  from GNU termcap file.  This merges in all their local information.
# 9.5.3 (Tue Apr 25 22:28:13 EDT 1995)
#	* Changed tic -C logic to dump all capabilities used by GNU termcap.
#	* Added warnings about entries with long translations (restoring
#	  all the GNU termcaps pushes a few over the edge).
# 9.5.4 (Wed Apr 26 15:35:09 EDT 1995)
#	* Yet another tic change, and a couple of entry tweaks, to reduce the
#	  number of long (> 1024) termcap translations back to 0.
#
# 9.6.0 (Mon May  1 10:35:54 EDT 1995)
#	* Added kf13-kf20 to Linux entry.
#	* Regularize Prime terminal names.
#	* Historical data on Synertek.
#	* Freeze for ncurses-1.9.1.
# 9.6.1 (Sat May  6 02:00:52 EDT 1995):
#	* Added true xterm-color entry, renamed djm's pseudo-color entry.
#	* Eliminate whitespace in short name fields, this tanks some scripts.
#	* Name field changes to shorten some long entries.
#	* Termcap translation now automatically generates empty rmir/smir
#	  when ich1/ich is present (copes with an ancient vi bug).
#	* Added `screen' entries from FSF's screen-3.6.2.
#	* Added linux-nic and xterm-nic entries.
# 9.6.2 (Sat May  6 17:00:55 EDT 1995):
#	* Change linux entry to use smacs=\E[11m and have an explicit acsc,
#	  eliminating some special-case code in ncurses.
#
# 9.7.0 (Tue May  9 18:03:12 EDT 1995):
#	* Added vt320-k3, rsvidtx from the Emacs termcap.dat file.  I think
#	  that captures everything unique from it.
#	* Added reorder script generator.
#	* Freeze for ncurses 1.9.2 release.
# 9.7.1 (Thu Jun 29 09:35:22 EDT 1995):
#	* Added Sean Farley's kspd, flash, rs1 capabilities for linux.
#	* Added Olaf Siebert's corrections for adm12.
#	* ansi-pc-color now includes the colors and pairs caps, so that
#	  entries which use it will inherit them automatically.
#	* The linux entry can now recognize the center (keypad 5) key.
#	* Removed some junk that found its way into Linux acsc.
#
# 9.8.0 (Fri Jul  7 04:46:57 EDT 1995):
#	* Add 50% cut mark as a desperate hack to reduce tic's core usage.
#	* xterm doesn't try to use application keypad mode any more.
#	* Freeze for ncurses-1.9.3 release.
# 9.8.1 (Thu Jul 19 17:02:12 EDT 1995):
#	* Added corrected sun entry from vendor.
#	* Added csr capability to linux entry.
#	* Peter Wemm says the at386 hpa should be \E[%i%p1%dG, not \E[%p1%dG.
#	* Added vt102-nsgr to cope with stupid IBM PC `VT100' emulators.
#	* Some commented-out caps in long entries come back in, my code
#	  for computing string-table lengths had a bug in it.
#	* pcansi series modified to fit comm-program reality better.
# 9.8.2 (Sat Sep  9 23:35:00 EDT 1995):
#	* BSD/OS actually ships the ibmpc3 bold entry as its console.
#	* Correct some bad aliases in the pcansi series
#	* Added entry for QNX console.
#	* Clean up duplicate long names for use with 4.4 library.
#	* Change vt100 standout to be normal reverse vide, not bright reverse;
#	  this makes the Emacs status line look better.
# 9.8.3 (Sun Sep 10 13:07:34 EDT 1995):
#	* Added Adam Thompson's VT320 entries, also his dtx-sas and z340.
#	* Minor surgery, mostly on name strings, to shorten termcap version.
#
# 9.9.0 (Sat Sep 16 23:03:48 EDT 1995):
#	* Added dec-vt100 for use with the EWAN emulator.
#	* Added kmous to xterm for use with xterm's mouse-tracking facility.
#	* Freeze for 1.9.5 alpha release.
# 9.9.1 (Wed Sep 20 13:46:09 EDT 1995):
#	* Changed xterm lines to 24, the X11R6 default.
# 9.9.2 (Sat Sep 23 21:29:21 EDT 1995):
#	* Added 7 newly discovered, undocumented acsc characters to linux
#	  entry (the pryz{|} characters).
#	* ncurses no longer steals A_PROTECT.  Simplify linux sgr accordingly.
#	* Correct two typos in the xterm entries introduced in 9.9.1.
#	* I finally figured out how to translate ko capabilities.  Done.
#	* Added tvi921 entries from Tim Theisen.
#	* Cleanup: dgd211 -> dg211, adm42-nl -> adm42-nsl.
#	* Removed mystery tec entry, it was neither interesting nor useful.
#	* shortened altos3, qvt203, tvi910+, tvi92D, tvi921-g, tvi955, vi200-f,
#	  vi300-ss, att505-24, contel301, dm3045, f200vi, pe7000c, vc303a,
#	  trs200, wind26, wind40, wind50, cdc456tst, dku7003, f110, dg211,
#	  by making them relative to use capabilities
#	* Added cuf1=^L to tvi925 from deleted variant tvi925a.
#	* fixed cup in adm22 entry and parametrized strings in vt320-k3.
#	* added it#8 to entries that used to have :pt: -- tvi912, vi200,
#	  ampex80,
#	* Translate all home=\E[;H capabilities to home=\E[H, they're
#	  equivalent.
#	* Translate \E[0m -> \E[m in [rs]mso, [rs]mul, and init strings of
#	  vt100 and ANSI-like terminals.
# 9.9.3 (Tue Sep 26 20:11:15 EDT 1995):
#	* Added it#8 and ht=\t to *all* entries with :pt:; the ncurses tic
#	  does this now, too.
#	* fviewpoint is gone, it duplicated screwpoint.
#	* Added hp2627, graphos, graphos-30, hpex, ibmega, ibm8514, ibm8514-c,
#	  ibmvga, ibmvga-c, minix, mm340, mt4520-rv, screen2, screen3,
#	  versaterm, vi500, vsc, vt131, vt340, vt400 entries from UW.
#	  The UW vi50 replaces the old one, which becomes vi50adm,
#	* No more embedded commas in name fields.
#
# 9.10.0 (Wed Oct  4 15:39:37 EDT 1995):
#	* XENIX forms characters in fos, trs16, scoansi become acsc strings,
#	* Introduced klone+* entries for describing Intel-console behavior.
#	* Linux kbs is default-mapped to delete for some brain-dead reason.
#	* -nsl -> -ns.  The -pp syntax is obsolete.
#	* Eliminate [A-Z]* primaries in accordance with SVr4 terminfo docs.
#	* Make xterm entry do application-keypad mode again.  I got complaints
#	  that it was messing up someone's 3270 emulator.
#	* Added some longname fields in order to avoid warning messages from
#	  older tic implementations.
#	* According to ctlseqs.ms, xterm has a full vt100 graphics set.  Use
#	  it! (This gives us pi, greater than, less than, and a few more.)
#	* Freeze for ncurses-1.9.6 release.
# 9.10.1 (Sat Oct 21 22:18:09 EDT 1995):
#	* Add xon to a number of console entries, they're memory-mapped and
#	  don't need padding.
#	* Correct the use dependencies in the ansi series.
#	* Hand-translate more XENIX capabilities.
#	* Added hpterm entry for HP's X terminal emulator.
#	* Added aixterm entries.
#	* Shortened four names so everything fits in 14 chars.
#
# 9.11.0 (Thu Nov  2 17:29:35 EST 1995):
#	* Added ibcs2 entry and info on iBCS2 standard.
#	* Corrected hpa/vpa in linux entry.  They still fail the worm test.
#	* We can handle the HP meml/memu capability now.
#	* Added smacs to klone entries, just as documentation.
#	* Carrected ansi.sys and cit-500 entries.
#	* Added z39, vt320-k311, v220c, and avatar entries.
#	* Make pcansi use the ansi.sys invis capability.
#	* Added DIP switch descriptions for vt100, adm31, tvi910, tvi920c,
#	  tvi925, tvi950, dt80, ncr7900i, h19.
#	* X3.64 has been withdrawn, change some references.
#	* Removed function keys from ansi-m entry.
#	* Corrected ansi.sys entry.
#	* Freeze for ncurses-1.9.7 release.
# 9.11.1 (Tue Nov  6 18:18:38 EST 1995):
#	* Added rmam/smam capabilities to many entries based on init strings.
#	* Added correct hpa/vpa to linux.
#	* Reduced several entries relative to vt52.
# 9.11.2 (Tue Nov  7 00:21:06 EST 1995):
#	* Exiled some utterly unidentifiable custom and homebrew types to the
#	  UFO file; also, obsolete small-screen hardware; also, entries which
#	  look flat-out incorrect, garbled, or redundant.  These include the
#	  following entries: carlock, cdc456tst, microkit, qdss, ramtek, tec,
#	  tec400, tec500, ubell, wind, wind16, wind40, wind50, plasma, agile,
#	  apple, bch, daleblit, nucterm, ttywilliams, nuterminal, nu24, bnu,
#	  fnu, nunix-30, nunix-61, exidy, ex3000, sexidy, pc52, sanyo55,
#	  yterm10, yterm11, yterm10nat, aed, aed-ucb, compucolor, compucolor2,
#	  vic20, dg1, act5s, netx, smartvid, smarterm, sol, sol2, dt200,
#	  trs80, trs100, trs200, trs600, xitex, rsvidtx, vid, att2300-x40,
#	  att2350-x40, att4410-nfk, att5410-ns, otty5410, att5425-nl-w,
#	  tty5425-fk, tty5425-w-fk, cita, c108-na, c108-rv-na, c100-rv-na,
#	  c108-na-acs, c108-rv-na-acs, ims950-ns, infotonKAS, ncr7900i-na,
#	  regent60na, scanset-n, tvi921-g, tvi925n, tvi925vbn, tvi925vb,
#	  vc404-na, vc404-s-na, vt420nam, vt420f-nam, vt420pc-nam, vt510nam,
#	  vt510pc-nam, vt520nam, vt525nam, xterm25, xterm50, xterm65, xterms.
#	* Corrected pcvt25h as suggested by Brian C. Grayson
#	  <bgrayson@pine.ece.utexas.edu>.
# 9.11.3 (Thu Nov  9 12:14:40 EST 1995):
#	* Added kspd=\E[P, kcbt=\E[Z, to linux entry, changed kbs back to ^H.
#	* Added kent=\EOM to xterm entry.
#
# 9.11.4 (Fri Nov 10 08:31:35 EST 1995):
#	* Corrected gigi entry.
#	* Restored cuf/cud1 to xterm, their apparent bugginess was due to
#	  bad hpa/vpa capabilities.
#	* Corrected flash strings to have a uniform delay of .2 sec.  No
#	  more speed-dependent NUL-padding!
#	* terminfo capabilities in comments bracketed with <>.
# 9.11.5 (Fri Nov 10 15:35:02 EST 1995):
#	* Replaced pcvt with the 3.31 pcvt entries.
#	* Freeze for 1.9.7a.
# 9.11.6 (Mon Nov 13 10:20:24 EST 1995):
#	* Added emu entry from the X11R6 contrib tape sources.
#
# 9.12.0 (Wed Nov 29 04:22:25 EST 1995):
#	* Improved iris-ansi and sun entries.
#	* More flash string improvements.
#	* Corrected wy160 & wy160 as suggested by Robert Dunn
#	* Added dim to at386.
#	* Reconciled pc3 and ibmpc3 with the BSDI termcap file.  Keith says
#	  he's ready to start using the termcap generated from this one.
#	* Added vt102-w, vt220-w, xterm-bold, wyse-vp, wy75ap, att4424m,
#	  ln03, lno3-w, h19-g, z29a*, qdss.  Made vt200 an alias of vt220.
#	* Improved hpterm, apollo consoles, fos, qvt101, tvi924. tvi925,
#	  att610, att620, att630,
#	* Changed hazeltine name prefix from h to hz.
#	* Sent t500 to the UFI file.
#	* I think we've sucked all the juice out of BSDI's termcap file now.
#	* Freeze for ncurses 1.9.8 release
# 9.12.1 (Thu Nov 30 03:14:06 EST 1995)
#	* Unfreeze, linux kbs needed to be fixed.
#	* Tim Theisen pinned down a bug in the DMD firmware.
# 9.12.2 (Thu Nov 30 19:08:55 EST 1995):
#	* Fixes to ansi and klone capabilities (thank you, Aaron Ucko).
#	  (The broken ones had been shadowed by sgr.)
# 9.12.3 (Thu Dec  7 17:47:22 EST 1995):
#	* Added documentation on ECMA-48 standard.
#	* New Amiga entry.
# 9.12.4 (Thu Dec 14 04:16:39 EST 1995):
#	* More ECMA-48 stuff
#	* Corrected typo in minix entry, added pc-minix.
#	* Corrected khome/kend in xterm (thank you again, Aaron Ucko).
#	* Added rxvt entry.
#	* Added 1.3.x color-change capabilities to linux entry.
# 9.12.5 (Tue Dec 19 00:22:10 EST 1995):
#	* Corrected rxvt entry khome/kend.
#	* Corrected linux color change capabilities.
#	* NeXT entries from Dave Wetzel.
#	* Cleaned up if and rf file names (all in /usr/share now).
#	* Changed linux op capability to avoid screwing up a background color
#	  pair set by setterm.
# 9.12.6 (Wed Feb  7 16:14:35 EST 1996):
#	* Added xterm-sun.
# 9.12.7 (Fri Feb  9 13:27:35 EST 1996):
#	* Added visa50.
#
# 9.13.0 (Sun Mar 10 00:13:08 EST 1996):
#	* Another sweep through the Shuford archive looking for new info.
#	* Added dg100 alias to dg6053 based on a comp.terminals posting.
# 	* Added st52 from Per Persson.
#	* Added eterm from the GNU Emacs 19.30 distribution.
#	* Freeze for 1.9.9.
# 9.13.1 (Fri Mar 29 14:06:46 EST 1996):
#	* FreeBSD console entries from Andrew Chernov.
#	* Removed duplicate Atari st52 name.
# 9.13.2 (Tue May  7 16:10:06 EDT 1996)
#	* xterm doesn't actually have ACS_BLOCK.
#	* Change klone+color setf/setb to simpler forms that can be
#	  translated into termcap.
#	* Added xterm1.
#	* Removed mechanically-generated junk capabilities from cons* entries.
#	* Added color support to bsdos.
# 9.13.3 (Thu May  9 10:35:51 EDT 1996):
#	* Added Wyse 520 entries from Wm. Randolph Franklin <wrf@ecse.rpi.edu>.
#	* Created ecma+color, linux can use it.  Also added ech to linux.
#	* Teach xterm about more keys. Add Thomas Dickey's 3.1.2E updates.
#	* Add descriptions to FreeBSD console entries.  Also shorten
#	  some aliases to <= 14 chars for portability.
#	* Added x68k console
#	* Added OTbs to several VT-series entries.
# 9.13.4 (Wed May 22 10:54:09 EDT 1996):
#	* screen entry update for 3.7.1 from Michael Alan Dorman.
# 9.13.5 (Wed Jun  5 11:22:41 EDT 1996):
#	* kterm correction due to Kenji Rikitake.
#	* ACS correction in vt320-kll due to Phillippe De Muyter.
# 9.13.6 (Sun Jun 16 15:01:07 EDT 1996):
#	* Sun console entry correction from J.T. Conklin.
#	* Changed all DEC VT300 and up terminals to use VT300 tab set
# 9.13.7 (Mon Jul  8 20:14:32 EDT 1996):
#	* Added smul to linux entry (we never noticed it was missing
#	  because of sgr!).
#	* Added rmln to hp+labels (deduced from other HP entries).
#	* Added vt100 acsc capability to vt220, vt340, vt400, d800, dt80-sas,
#	  pro350, att7300, 5420_2, att4418, att4424, att4426, att505, vt320-k3.
#	* Corrected vt220 acsc.
#	* The klone+sgr and klone+sgr-dumb entries now use klone+acs;
#	  this corresponds to reality and helps prevent some tic warnings.
#	* Added sgr0 to c101, pcix, vt100-nav, screen2, oldsun, next, altos2,
#	  hpgeneric, hpansi, hpsub, hp236, hp700-wy, bobcat, dku7003, adm11,
#	  adm12, adm20, adm21, adm22, adm31, adm36, adm42, pt100, pt200,
#	  qvt101, tvi910, tvi921, tvi92B, tvi925, tvi950, tvi970, wy30-mc,
#	  wy50-mc, wy100, wyse-vp, ampex232, regent100, viewpoint, vp90,
#	  adds980, cit101, cit500, contel300, cs10, dm80, falco, falco-p,
#	  f1720a, go140, sb1, superbeeic, microb, ibm8512, kt7, ergo4000,
#	  owl, uts30, dmterm, dt100, dt100, dt110, appleII, apple-videx,
#	  lisa, trsII, atari, st52, pc-coherent, basis, m2-man, bg2.0, bg1.25,
#	  dw3, ln03, ims-ansi, graphos, t16, zen30, xtalk, simterm, d800,
#	  ifmr, v3220, wy100q, tandem653, ibmaed.
#	* Added DWK terminal description.
# 9.13.8 (Wed Jul 10 11:45:21 EDT 1996):
#	* Many entries now have highlights inherited from adm+sgr.
#	* xterm entry now corresponds to XFree86 3.1.2E, with color.
#	* xtitle and xtitle-twm enable access to the X status line.
#	* Added linux-1.3.6 color palette caps in conventional format.
#	* Added adm1178 terminal.
#	* Move fos and apollo terminals to obsolete category.
#	* Aha! The BRL terminals file told us what the Iris extensions mean.
#	* Added, from the BRL termcap file: rt6221, rt6221-w, northstar,
#	  commodore, cdc721-esc, excel62, osexec.  Replaced from the BRL file:
#	  cit500, adm11.
# 9.13.9 (Mon Jul 15 00:32:51 EDT 1996):
#	* Added, from the BRL termcap file: cdc721, cdc721l, cdc752, cdc756,
#	  aws, awsc, zentec8001, modgraph48, rca vp3301/vp3501, ex155.
#	* Corrected, from BRL termcap file: vi50.
#	* Better rxvt entry & corrected xterm entries from Thomas Dickey.
# 9.13.10 (Mon Jul 15 12:20:13 EDT 1996):
#	* Added from BRL: cit101e & variants, hmod1, vi200, ansi77, att5620-1,
#	  att5620-s, att5620-s, dg210, aas1901, hz1520, hp9845, osborne
#	  (old osborne moved to osborne-w), tvi970-vb, tvi970-2p, tvi925-hi,
#	  tek4105brl, tek4106brl, tek4107brl,tek4109brl, hazel, aepro,
#	  apple40p, apple80p, appleIIgs, apple2e, apple2e-p, apple-ae.
#	* Paired-attribute fixes to various terminals.
#	* Sun entry corrections from A. Lukyanov & Gert-Jan Vons.
#	* xterm entry corrections from Thomas Dickey.
# 9.13.11 (Tue Jul 30 16:42:58 EDT 1996):
#	* Added t916 entry, translated from a termcap in SCO's support area.
#	* New qnx entry from Michael Hunter.
# 9.13.12 (Mon Aug  5 14:31:11 EDT 1996):
#	* Added hpex2 from Ville Sulko.
#	* Fixed a bug that ran the qnx and pcvtXX together.
# 9.13.13 (Fri Aug  9 01:16:17 EDT 1996):
#	* Added dtterm entry from Solaris CDE.
# 9.13.14 (Tue Sep 10 15:31:56 EDT 1996):
#	* corrected pairs#8 typo in dtterm entry.
#	* added tvi9065.
# 9.13.15 (Sun Sep 15 02:47:05 EDT 1996):
#	* updated xterm entry to cover 3.1.2E's new features.
# 9.13.16 (Tue Sep 24 12:47:43 EDT 1996):
#	* Added new minix entry
#	* Removed aliases of the form ^[0-9]* for obsolete terminals.
#	* Commented out linux-old, nobody's using pre-1.2 kernels now.
# 9.13.17 (Fri Sep 27 13:25:38 EDT 1996):
#	* Added Prism entries and kt7ix.
#	* Caution notes about EWAN and tabset files.
#	* Changed /usr/lib/tabset -> /usr/share/tabset.
#	* Added acsc/rmacs/smacs to vt52.
# 9.13.18 (Mon Oct 28 13:24:59 EST 1996):
#	* Merged in Thomas Dickey's reorganization of the xterm entries;
#	  added technical corrections to avoid warning messages.
# 9.13.19 (Sat Nov 16 16:05:49 EST 1996):
#	* Added rmso=\E[27m in Linux entry.
#	* Added koi8-r support for Linux console.
#	* Replace xterm entries with canonical ones from XFree86 3.2.
# 9.13.20 (Sun Nov 17 23:02:51 EST 1996):
#	* Added color_xterm from Jacob Mandelson
# 9.13.21 (Mon Nov 18 12:43:42 EST 1996):
#	* Back off the xterm entry to use r6 as a base.
# 9.13.22 (Sat Nov 30 11:51:31 EST 1996):
#	* Added dec-vt220 at Adrian Garside's request.
#
#-(original-changelog-1996/12/29-to-1998/02/28-by-TD)---------------------------
#
# 10.1.0 (Sun Dec 29 02:36:31 EST 1996): withdrawn
#	* Minor corrections to xterm entries.
#	* Replaced EWAN telnet entry.
#	* Dropped the reorder script generator.  It was a fossil.
# 9.13.23 (Fri Feb 21 16:36:06 EST 1997):
#	* Replaced minitel-2 entry.
#	* Added MGR, ansi-nt.
# 9.13.24 (Sun Feb 23 20:55:23 EST 1997):
#	* Thorsten Lockert added termcap `bs' to a lot of types, working from
#	  the 4.4BSD Lite2 file.
#
# 10.1.1 (Sat May  3 21:41:27 EDT 1997):
#	* Use setaf/setab consistently with SVr4.
#	* Remove ech, el1 from cons25w, they do not work in FreeBSD 2.1.5
# 10.1.2 (Sat May 24 21:10:57 EDT 1997)
#	* update xterm-xf86-v32 to match XFree86 3.2A (changes F1-F4)
#	* add xterm-16color, for XFree86 3.3
# 10.1.3 (Sat May 31 12:21:05 EDT 1997)
#	* correct typo in emu
#	* correct typo in vt102-w (Robert Wuest)
#	* make new entry xterm-xf86-v33, restored xterm-xf86-v32.
# 10.1.4 (Sun Jun 15 08:29:05 EDT 1997)
#	* remove ech capability from rxvt (it does the wrong thing)
# 10.1.5 (Sat Jun 28 21:34:36 EDT 1997)
#	* remove spurious newlines from several entries (hp+color, wy50,
#	  wy350, wy370-nk, wy99gt-tek, wy370-tek, ibm3161, tek4205, ctrm,
#	  gs6300)
# 10.1.6 (Sat Jul  5 15:08:16 EDT 1997)
#	* correct rmso capability of wy50-mc
# 10.1.7 (Sat Jul 12 20:05:55 EDT 1997)
#	* add cbt to xterm-xf86-v32
#	* disentangle some entries from 'xterm', preferring xterm-r6 in case
#	  'xterm' is derived from xterm-xf86-v32, which implements ech and
#	  other capabilities not in xterm-r6.
#	* remove alternate character set from kterm entry.
# 10.1.8 (Sat Aug  2 18:43:18 EDT 1997)
#	* correct acsc entries for ACS_LANTERN, which is 'i', not 'I'.
# 10.1.9 (Sat Aug 23 17:54:38 EDT 1997)
#	* add xterm-8bit entry.
# 10.1.10 (Sat Oct  4 18:17:13 EDT 1997)
#	* repair several places where early version of tic replaced \, with \\\,
#	* make acsc entries canonical form (sorted, uniq).
#	* modify acsc entries for linux, linux-koi8
#	* new rxvt entry, from corrected copy of distribution in rxvt 2.21b
#	* add color, mouse support to kterm.
# 10.1.11 (Sat Oct 11 14:57:10 EDT 1997)
#	* correct wy120 smxon/tbc capabilities which were stuck together.
# 10.1.12 (Sat Oct 18 17:38:41 EDT 1997)
#	* add entry for xterm-xf86-v39t
# 10.1.13 (Sat Nov  8 13:43:33 EST 1997)
#	* add u8,u9 to sun-il description
# 10.1.14 (Sat Nov 22 19:59:03 EST 1997)
#	* add vt220-js, pilot, rbcomm, datapoint entries from esr's 27-jun-97
#	  version.
#	* add hds200 description (Walter Skorski)
#	* add EMX 0.9b descriptions
#	* correct rmso/smso capabilities in wy30-mc and wy50-mc (Daniel Weaver)
#	* rename xhpterm back to hpterm.
# 10.1.15 (Sat Nov 29 19:21:59 EST 1997)
#	* change initc in linux-c-nc to use 0..1000 range.
# 10.1.16 (Sat Dec 13 19:41:59 EST 1997)
#	* remove hpa/vpa from rxvt, which implements them incorrectly.
#	* add sgr0 for rxvt.
#	* remove bogus smacs/rmacs from EMX descriptions.
# 10.1.17 (Sat Dec 20 17:54:10 EST 1997)
#	* revised entry for att7300
# 10.1.18 (Sat Jan  3 17:58:49 EST 1998)
#	* use \0 rather than \200.
#	* rename rxvt-color to rxvt to match rxvt 2.4.5 distribution.
# 10.1.19 (Sat Jan 17 14:24:57 EST 1998)
#	* change xterm (xterm-xf86-v40), xterm-8bit rs1 to use hard reset.
#	* rename xterm-xf86-v39t to xterm-xf86-v40
#	* remove bold/underline from sun console entries since they're not
#	  implemented.
# 10.1.20 (Sat Jan 24 11:02:51 EST 1998)
#	* add beterm entry (Fred Fish)
#	* add irix-color/xwsh entry.
#	* turn ncv off for linux.
# 10.1.21 (Sat Jan 31 17:39:16 EST 1998)
#	* set ncv for FreeBSD console (treat colors with reverse specially).
#	* remove sgr string from qnx based on report by Xiaodan Tang
# 10.1.22 (Wed Feb 11 18:40:12 EST 1998)
#	* remove spurious commas from descriptions
#	* correct xterm-8bit to match XFree86 3.9Ad F1-F4.
# 10.1.23 (Sat Feb 28 17:48:38 EST 1998)
#	* add linux-koi8r to replace linux-koi8 (which uses a corrupt acsc,
#	  apparently based on cp-866).
#
#-(replaced-changelog-1998/02/28-by-ESR)----------------------------------------
#
# 9.13.23 (Fri Feb 21 16:36:06 EST 1997):
#	* Replaced minitel-2 entry.
#	* Added MGR, ansi-nt.
#	* Minor corrections to xterm entries.
#	* Replaced EWAN telnet entry.
#	* Dropped the reorder script generator.  It was a fossil.
# 9.13.24 (Sun Feb 23 20:55:23 EST 1997):
#	* Thorsten Lockert added termcap `bs' to a lot of types, working from
#	  the 4.4BSD Lite2 file.
# 9.13.25 (Fri Jun 20 12:33:36 EDT 1997):
#	* Added Datapoint 8242, pilot, ansi_psx, rbcomm, vt220js.
#	* Updated iris-ansi; corrected vt102-w.
#	* Switch base xterm entry to 3.3 level.
# 9.13.26 (Mon Jun 30 22:45:45 EDT 1997)
#	* Added basic4.
#	* Removed rmir/smir from tv92B.
#
# 10.2.0 (Sat Feb 28 12:47:36 EST 1998):
#	* add hds200 description (Walter Skorski)
#	* add beterm entry (Fred Fish)
#	* add Thomas Dickey's xterm-xf86-v40, xterm-8bit, xterm-16color,
#	  iris-color entries.
#	* add emx entries.
#	* Replaced unixpc entry with Benjamin Sittler's corrected version.
#	* Replaced xterm/rxvt/emu/syscons entries with Thomas Dickey's
#	  versions.
#	* remove sgr string from qnx based on report by Xiaodan Tang
#	* Added u8/u9, removed rmul/smul from sun-il.
#	* 4.2 tic displays \0 rather than \200.
#	* add linux-koi8r to replace linux-koi8 (which uses a corrupt acsc,
#	  apparently based on cp-866).
#	* Merged in Pavel Roskin's acsc for linux-koi8
#	* Corrected some erroneous \\'s to \.
#	* 4.2 ncurses has been changed to use setaf/setab, consistent w/SysV.
#	* II -> ii in pcvtXX, screen, xterm.
#	* Removed \n chars following ANSI escapes in sgr & friends.
#	* Updated Wyse entries.
#	* h19 corrections from Tim Pierce.
#	* Noted that the dm2500 has both ich and smir.
#	* added pccons for the Alpha under OSF/1.
#	* Added Sony NEWS workstation entries and cit101e-rv.
#	* Reverted `amiga'; to Kent Polk's version, as I'm told
#	  the Verkuil entry messes up with Amiga Telnet.
# 10.2.1 (Sun Mar  8 18:32:04 EST 1998):
#	* Corrected attributions in 10.2.0 release notes.
#	* Scanned the Shuford archive for new terminfos and information.
#	* Removed sgr from qnx entry (Thomas Dickey).
#	* Added entries for ICL and Kokusai Data Systems terminals.
#	* Incorporated NCR terminfos from the Boundless Technology FTP site.
#	* Incorporated att700 from the Boundless Technology FTP site.
#	* Miscellaneous contact-address and Web-page updates.
#
#-(changelog-beginning-ncurses-4.2)---------------------------------------------
#
# 1998/5/9
#	* add nxterm and xterm-color terminfo description (request by Cristian
#	  Gafton <gafton@redhat.com>).
#	* modify rxvt terminfo description to clear alternate screen before
#	  switching back to normal screen, for compatibility with applications
#	  which use xterm (reported by Manoj Kasichainula <manojk@io.com>).
#	* modify linux terminfo description to reset color palette (reported
#	  by Telford Tendys <telford@eng.uts.edu.au>).
#
# 1998/7/4
#	* merge changes from current XFree86 xterm terminfo descriptions.
#
# 1998/7/25
#	* Added minitel1 entries from Alexander Montaron.
#	* Added qnxt2 from Federico Bianchi.
#	* Added arm100 terminfo entries from Dave Millen.
#
# 1998/8/6
#	* Added ncsa telnet entries from Francesco Potorti
#
# 1998/8/15
#	* modify ncsa telnet entry to reflect color, other capabilities based on
#	  examination of the source code - T.Dickey.
#
# 1998/8/22
#	* Corrected some erroneous \\'s to \ (eterm, osborne) - TD.
#
# 1998/8/29
#	* Added Francesco Potorti's tuned Wyse 99 entries.
#	* dtterm enacs correction from Alexander V. Lukyanov.
#	* Add ncsa-ns, ncsa-m-ns and ncsa-m entries from esr version.
#	* correct a typo in icl6404 entry.
#	* add xtermm and xtermc
#
# 1998/9/26
#	* format most %'char' sequences to %{number}
#	* adapt IBM AIX 3.2.5 terminfo - T.Dickey
#	* merge Data General terminfo from Hasufin <hasufin@vidnet.net> - TD
#
# 1998/10/10
#	* update xterm-xfree86 to current (patch 84), for is2/rs2 changes - TD
#	* correct initialization string in xterm-r5, add misc other features
#	  to correspond with xterm patch 84 - TD
#
# 1998/12/19
#	* update xterm-xfree86 to current (patch 90), smcur/rmcur changes - TD
#	* add Mathew Vernon's mach console entries
#	* corrections for ncsa function-keys (report by Larry Virden)
#
# 1998/12/19
#	* change linux to use ncv#2, since underline does not work with color - TD
#
# 1999/1/9
#	* add kbt to iris-ansi, document other shift/control functionkeys - TD
#	* correct iris-ansi and iris-ansi-ap with respect to normal vs keypad
#	  application modes, change kent to use the correct keypad code - TD
#
# 1999/1/10
#	* add entry for Tera Term - TD
#
# 1999/1/23
#	* minor improvements for teraterm entry - TD
#	* rename several entries used by BSDI: bsdos to bsdos-pc-nobold,
#	  and bsdos-bold to bsdos-pc (Jeffrey C Honig)
#
# 1999/2/20
#	* resolve ambiguity of kend/kll/kslt and khome/kfnd/kich1 strings in
#	  xterm and ncsa entries by removing the unneeded ones.  Note that
#	  some entries will return kend & khome versus kslt and kfnd, for
#	  PC-style keyboards versus strict vt220 compatiblity - TD
#
# 1999/3/13
#	* adjust xterm-xfree86 khome/kend to match default PC-style keyboard
#	  tables - TD
#	* add 'crt' entry - TD
#	* correct typos in 'linux-c' entry - TD
#
# 1999/3/14
#	* update entries for BSD/OS console to use klone+sgr and klone+color
#	  (Jeffrey C Honig)
#
# 1999/3/27
#	* adjust xterm-xfree86 miscellaneous keypad keys, as per patch #94 - TD.
#
# 1999/4/10
#	* add linux-lat, from RedHat patches to ncurses 4.2
#
# 1999/4/17
#	* add complete set of default function-key definitions for scoansi - TD.
#
# 1999/7/3
#	* add cnorm, cvvis for Linux 2.2 kernels
#
# 1999/7/24
#	* add kmous to xterm-r5 -TD
#	* correct entries xterm+sl and xterm+sl-twm, which were missing the
#	  parent "use" clause -TD
#
# 1999/7/31
#	* corrected cnorm, added el1 in 'screen' description -TD
#
# 1999/8/14
#	* add ms-vt100 -TD
#
# 1999/8/21
#	* corrections to beterm entry -TD
#
# 1999/8/28
#	* add cygwin entry -TD
#
# 1999/9/4
#	* minor corrections for beterm entry -TD
#
# 1999/9/18
#	* add acsc string to HP 70092 terminfo entry -Joerg Wunsch
#
# 1999/9/25
#	* add amiga-8bit entry
#	* add console entries from NetBSD: ofcons, wsvt25, wsvt25m, rcons,
#	  rcons-color, based on
#	  ftp://ftp.netbsd.org/pub/NetBSD/NetBSD-current/src/share/termcap/termcap.src
#	* add alias for iris-ansi-net
#
# 1999/10/2
#	* corrected scoansi entry's acsc, some function keys, add color -TD
#
# 1999/10/23
#	* add cnorm, cvvis to cons25w, and modify ncv to add 'dim' -TD
#	* reorder ncsa entries to make ncsa-vt220 use the alternate function
#	  key mapping, leaving Potorti's entries more like he named them -TD
#	* remove enter/exit am-mode from cygwin -TD
#
# 1999/10/30
#	* correct typos in several entries (missing '[' from CSI):
#	  mgr-sun, ncsa-m, vt320-k3, att505, avt-ns, as well as smir/rmir
#	  strings for avt-ns -TD
#	* add 'dim' to ncv mask for linux (report by Klaus Weide).
#
# 1999/11/27
#	* correct kf1-kf4 in xterm-r6 which were vt100-style PF1-PF4 -TD
#	* add hts to xterm-r6, and u6-u9 to xterm-r5 -TD
#	* add xterm-88color and xterm-256color -TD
#
# 1999/12/4
#	* add "obsolete" termcap strings -TD
#	* add kvt and gnome entries -TD
#
# 1999/12/11
#	* correct cup string for regent100 -TD
#
# 2000/1/1
#	* update mach, add mach-color based on Debian diffs for ncurses 5.0 -TD
#	* add entries for xterm-hp, xterm-vt220, xterm-vt52 and xterm-noapp -TD
#	* change OTrs capabilities to rs2 -TD
#	* add obsolete and extended capabilities to 'screen' -TD
#
# 2000/1/5
#	* remove kf0 from rxvt, vt520, vt525 and ibm5151 since it conflicts
#	  with kf10 -TD
#	* updated xterm-xf86-v40, making kdch1 correspond to vt220 'Remove',
#	  and adding kcbt -TD
#
# 2000/1/12
#	* remove incorrect khome/kend from xterm-xf86-v333, which was based on
#	  nonstandard resource settings -TD
#
# 2000/2/26
#	* minor fixes for xterm-*, based on Debian #58530 -TD
#
# 2000/3/4
#	* add several terminal types from esr's "11.0", as well as comments.
#	  bq300*, dku7102-old, dku7202, hft, lft, pcmw, pmcons, tws*, vip*,
#	  vt220-8bit, vt220-old, wy85-8bit
#
# 2000/3/18
#	* add several terminal types from esr's "11.0.1" (ansi-*).
#	* update OTxx capabilities for changes on 2000/3/4.
#	* revert part of vt220 change (request by Todd C Miller for OpenBSD)
#
# 2000/3/26
#	* move screen's AX extension to ecma+color, modify several entries to
#	  use that, adjusting ncv as needed -TD
#
# 2000/4/8
#	* add bsdos-pc-m, bsdos-pc-mono (Jeffrey C Honig)
#	* correct spelling error in entry name: bq300-rv was given as bg300-rv
#	  in esr's version.
#
# 2000/4/15
#	* add cud, ech, etc., to beterm based on feedback from Rico Tudor -TD
#	* correct color definition for ibm3164, make minor changes to other
#	  IBM terminal definitions based on recent terminfo descriptions -TD
#
# 2000/4/22
#	* add mgterm, from NetBSD -TD
#	* add alias sun-cgsix for sun-ss5 as per NetBSD
#	* change cons25w to use rs2 for reset rather than rs1 -TD
#	* add rc/sc to aixterm based on manpage -TD
#
# 2000/5/13
#	* remove ncv from xterm-16color, xterm-256color
#
# 2000/6/10
#	* add kmous capability to linux to use Joerg Schoen's gpm patch.
#
# 2000/7/1
#	* add Eterm (Michael Jennings)
#
# 2000-07-18
#       * add amiga-vnc entry.
#
# 2000-08-12
#	* correct description of Top Gun Telnet.
#	* add kterm-color
#
# 2000-08-26
#	* add qansi* entries from QNX ftp site.
#
# 2000-09-16
#	* add Matrix Orbital entries by Eric Z. Ayers).
#	* add xterm-basic, xterm-sco entries, update related entries to XFree86
#	  4.0.1c -TD
#
# 2000-09-17
#	* add S0, E0 extensions to screen's entry -TD
#
# 2000-09-23
#	* several corrections based on tic's new parameter-checking code -TD
#	* modify xterm-r6 and similar rs2 sequences which had \E7...\E8
#	  bracketing sequences that reset video attributes (\E8 would restore
#	  them) -TD
#
# 2000-11-11
#	* rename cygwin to cygwinB19, adapt newer entry from Earnie Boyd -TD
#
# 2000-12-16
#	* improved scoansi, based on SCO man-page, and testing console,
#	  scoterm with tack -TD
#
# 2001-01-27
#	* modify kterm to use acsc via SCS controls.
#
# 2001-02-10
#	* screen 3.9.8 allows xterm mouse controls to pass-through
#
# 2001-03-11
#	* remove spurious "%|" from some xterm entries.
#
# 2001-03-31
#	* modify 'screen' khome/kend to match screen 3.09.08
#	* add examples of 'screen' customization (screen.xterm-xfree86,
#	  screen.xterm-r6, screen.teraterm) -TD
#
# 2001-04-14
#	* correct definitions of shifted editing keys for xterm-xfree86 -TD
#	* add "Apple_Terminal" entries -Benjamin Sittler
#	* remove time-delays from "Apple_Terminal" entries -TD
#	* make sgr entries time-delays consistent with individual caps -TD
#
# 2001-05-05
#	* corrected/updated screen.xterm-xfree86
#
# 2001-05-19
#	* ELKS descriptions, from Federico Bianchi
#	* add u6 (CSR) to Eterm (Michael Jennings).
#
# 2001-07-21
#	* renamed "Apple_Terminal" entries to "nsterm" to work with Solaris's
#	  tic which handles names no longer than 14 characters.  Add
#	  corresponding descriptions for the Darwin PowerPC console named
#	  "xnuppc" -Benjamin Sittler
#
# 2001-09-01
#	* change kbs in mach entries to ^? (Marcus Brinkmann).
#
# 2001-11-17
#	* add "putty" entry -TD
#	* updated "Apple_Terminal" entries -Benjamin Sittler
#
# 2001-11-24
#	* add ms-vt100-color entry -TD
#	* add "konsole" entries -TD
#
# 2001-12-08
#	* update gnome entry to Redhat 7.2 -TD
#
# 2002-05-25
#	* add kf13-kf48 strings to cons25w -TD
#	* add pcvt25-color entry -TD
#	* changed a few /usr/lib/tabset -> /usr/share/tabset.
#	* improve some features of scoansi entry based on SCO's version -TD
#	* add scoansi-new entry corresponding to OpenServer 5.0.6
#
# 2002-06-15
#	* add kcbt to screen entry -TD
#
# 2002-06-22
#	* add rxvt-16color, ibm+16color, mvterm entries -TD
#
# 2002-09-28
#	* split out linux-basic entry, making linux-c inherit from that, and
#	  in turn linux (with cnorm, etc) inherit from linux-c-nc to reflect
#	  the history of this console type -TD
#	* scaled the linux-c terminfo entry to match linux-c-nc, i.e., the
#	  r/g/b parameters of initc are in the range 0 to 1000 -TD
#
# 2002-10-05
#	* minor fix for scale-factor of linux-c and linux-c-nc -TD
#
# 2002-11-09
#	* split-out vt100+keypad and vt220+keypad, fix interchanged ka3/kb2
#	  in the latter -TD
#
# 2002-11-16
#	* add entries for mterm (mterm, mterm-ansi, decansi) -TD
#	* ncr260wy350pp has only 16 color pairs -TD
#	* add sun-type4 from NetBSD -TD
#	* update xterm-xfree86 to current (patch 170) -TD
#	* add screen-bce, screen-s entries -TD
#	* add xterm-1002, xterm-1003 entries -TD
#
# 2003-01-11
#	* update homepage for Top Gun Telnet/SSH
#
# 2003-01-25
#	* reduce duplication in emx entries, added emx-base -TD
#
# 2003-05-24
#	* corrected acs for screen.teraterm -TD
#	* add tkterm entry -TD
#
# 2003-07-15
#	* cygwin changes from Charles Wilson:
#	  misc/terminfo.src (nxterm|xterm-color): make xterm-color
#	  primary instead of nxterm, to match XFree86's xterm.terminfo
#	  usage and to prevent circular links.
#	  (rxvt): add additional codes from rxvt.org.
#	  (rxvt-color): new alias
#	  (rxvt-xpm): new alias
#	  (rxvt-cygwin): like rxvt, but with special acsc codes.
#	  (rxvt-cygwin-native): ditto.  rxvt may be run under XWindows, or
#	  with a "native" MSWin GUI.  Each takes different acsc codes,
#	  which are both different from the "normal" rxvt's acsc.
#	  (cygwin): cygwin-in-cmd.exe window.  Lots of fixes.
#	  (cygwinDBG): ditto.
#
# 2003-09-27
#	* update gnome terminal entries -TD
#
# 2003-10-04
#	* add entries for djgpp 2.03 and 2.04 -TD
#
# 2003-10-25
#	* add alias for vtnt -TD
#	* update xterm-xfree86 for XFree86 4.4 -TD
#
# 2003-11-22
#	* add linux-vt (Andrey V Lukyanov)
#
# 2003-12-20
#	* add screen.linux -TD
#
# 2004-01-10
#	* revised/improved entries for tvi912b, tvi920b (Benjamin Sittler)
#
# 2004-01-17
#	* add OpenNT/Interix/SFU entries (Federico Bianchi)
#	* add vt100+ and vt-utf8 entries -TD
#	* add uwin entry -TD
#
# 2004-03-27
#	* add sgr strings to several common entries lacking them, e.g.,
#	  screen, to make the entries more portable -TD
#	* remove cvvis from rxvt entry, since it is the same as cnorm -TD
#	* similar fixups for cvvis/cnorm various entries -TD
#
# 2004-05-22
#	* remove 'ncv' from xterm-256color (patch 188) -TD
#
# 2004-06-26
#	* add mlterm -TD
#	* add xterm-xf86-v44 -TD
#	* modify xterm-new aka xterm-xfree86 to accommodate luit, which relies
#	  on G1 being used via an ISO-2022 escape sequence (report by
#	  Juliusz Chroboczek) -TD
#	* add 'hurd' entry -TD
#
# 2004-07-03
#	* make xterm-xf86-v43 derived from xterm-xf86-v40 rather than
#	  xterm-basic -TD
#	* align with xterm #192's use of xterm-new -TD
#	* update xterm-new and xterm-8bit for cvvis/cnorm strings -TD
#	* make xterm-new the default "xterm" -TD
#
# 2004-07-10
#	* minor fixes for emu -TD
#	* add emu-220
#	* add rmam/smam to linux (Trevor Van Bremen)
#	* change wyse acsc strings to use 'i' map rather than 'I' -TD
#	* fixes for avatar0 -TD
#	* fixes for vp3a+ -TD
#
# 2004-07-17
#	* add xterm-pc-fkeys -TD
#	* review/update gnome and gnome-rh90 entries (prompted by
#	  Redhat Bugzilla #122815) -TD
#	* review/update konsole entries -TD
#	* add sgr, correct sgr0 for kterm and mlterm -TD
#	* correct tsl string in kterm -TD
#
# 2004-07-24
#	* make ncsa-m rmacs/smacs consistent with sgr -TD
#	* add sgr, rc/sc and ech to syscons entries -TD
#	* add function-keys to decansi -TD
#	* add sgr to mterm-ansi -TD
#	* add sgr, civis, cnorm to emu -TD
#	* correct/simplify cup in addrinfo -TD
#	* corrections for gnome and konsole entries
#	  (Redhat Bugzilla #122815) -Hans de Goede
#	* modify DEC entries (vt220, etc), to add sgr string, and to use
#	  ISO-2022 strings for rmacs/smacs -TD
#
# 2004-07-31
#	* rename xterm-pc-fkeys to xterm+pcfkeys -TD
#
# 2004-08-07
#	* improved putty entry -Robert de Bath
#
# 2004-08-14
#	* remove dch/dch1 from rxvt because they are implemented inconsistently
#	  with the common usage of bce/ech -TD
#	* remove khome from vt220 (vt220's have no home key) -TD
#	* add rxvt+pcfkeys -TD
#
# 2004-08-21
#	* modify several entries to ensure xterm mouse and cursor visibility
#	  are reset in rs2 string:  hurd, putty, gnome, konsole-base, mlterm,
#	  Eterm, screen.  (The xterm entries are left alone - old ones for
#	  compatibility, and the new ones do not require this change) -TD
#
# 2004-08-28
#	* add morphos entry -Pavel Fedin
#	* modify amiga-8bit to add khome/kend/knp/kpp -Pavel Fedin
#	* corrected \E[5?l to \E[?5l in vt320 entries -TD
#
# 2004-11-20
#	* update wsvt25 entry -TD
#
# 2005-01-29
#	* update pairs for xterm-88color and xterm-256color to reflect the
#	  ncurses extended-color support -TD
#
# 2005-02-26
#	* modify sgr/sgr0 in xterm-new to improve tgetent's derived "me" -TD
#	* add aixterm-16color to demonstrate 16-color capability -TD
#
# 2005-04-23
#	* add media-copy to vt100 -TD
#	* corrected acsc string for vt52 -TD
#
# 2005-04-30
#	* add kUP, kDN (user-defined shifted up/down arrow) definitions for
#	  xterm-new -TD
#	* add kUP5, kUP6, etc., for xterm-new and rxvt -TD
#
# 2005-05-07
#	* re-corrected acsc string for vt52 -TD
#
# 2005-05-28
#	* corrected sun-il sgr string which referred to bold and underline -TD
#	* add sun-color entry -TD
#
# 2005-07-23
#	* modify sgr0 in several entries to reset alternate-charset as in the
#	  sgr string -TD
#	* modify sgr string of prism9 to better match the individual
#	  attributes -TD
#
# 2005-10-15
#	* correct order of use= in rxvt-basic -TD
#
# 2005-10-26
#	* use kind/kri as shifted up/down cursor keys for xterm-new -TD
#
# 2005-11-12
#	* other minor fixes to cygwin based on tack -TD
#	* correct smacs in cygwin (report by Baurzhan Ismagulov).
#
# 2006-02-18
#	* add nsterm-16color entry -TD
#	* remove ncv flag from xterm-16color -TD
#	* remove setf/setb from xterm-256color to match xterm #209 -TD
#	* update mlterm entry to 2.9.2 -TD
#
# 2006-02-25
#	* fixes to make nsterm-16color match report
#	  by Christian Ebert -Alain Bench
#
# 2006-04-22
#	* add xterm+256color building block -TD
#	* add gnome-256color, putty-256color, rxvt-256color -TD
#
# 2006-05-06
#	* add hpterm-color -TD
#
# 2006-06-24
#	* add xterm+pcc0, xterm+pcc1, xterm+pcc2, xterm+pcc3 -TD
#	* add gnome-fc5 (prompted by GenToo #122566) -TD
#	* remove obsolete/misleading comments about kcbt on Linux -Alain Bench
#	* improve xterm-256color by combining the ibm+16color setaf/setab
#	  strings with SGR 48.  The setf/setb strings also are cancelled here
#	  rather than omitted so derived entries will cancel those also -Alain
#	  Bench
#
# 2006-07-01
#	* add some notes regarding copyright to terminfo.src -TD
#	* use rxvt+pcfkeys in Eterm -TD
#	* remove km and flash from gnome, Eterm and rxvt since they do not work
#	  as one would expect (km sends ESC rather than setting the 8th bit
#	  of the key) -TD
#	* add/use ansi+enq, vt100+enq and vt102+enq -TD
#	* add konsole-solaris -TD
#
# 2006-07-22
#	* update xterm-sun and xterm-sco entries to match xterm #216 -TD
#	* modify is2/rs2 strings for xterm-r6 as per fix in xterm #148 -TD
#	* modify xterm-24 to inherit from "xterm" -TD
#	* add xiterm entry -TD
#	* add putty-vt100 entry -TD
#	* corrected spelling of Michael A Dorman's name, prompted by
#	  http://www.advogato.org/person/mdorman/diary.html -TD
#
# 2006-08-05
#	* add xterm+pcf0, xterm+pcf2 from xterm #216 -TD
#	* update xterm+pcfkeys to match xterm #216 -TD
#
# 2006-08-17
#	* make descriptions of xterm entries consistent with its terminfo -TD
#
# 2006-08-26
#	* add xfce, mgt -TD
#
# 2006-09-02
#	* correct acsc string in kterm -TD
#
# 2006-09-09
#	* add kon entry -TD
#	* remove invis from linux and related entries, add klone+sgr8 for those
#	  that implement the feature (or have not been shown to lack it) -TD
#
# 2006-09-23
#	* add ka2, kb1, kb3, kc2 to vt220-keypad as an extension -TD
#	* minor improvements to rxvt+pcfkeys -TD
#
# 2006-09-30
#	* fix a few typos in if/then/else expressions -TD
#
# 2006-10-07
#	* add several GNU Screen variations with 16- and 256-colors, and
#	  status line (Alain Bench).
#
# 2007-03-03
#	* add Newbury Data entries (Jean-Charles Billaud).
#
# 2007-06-10
#	* corrected xterm+pcf2 modifiers for F1-F4, match xterm #226 -TD
#
# 2007-07-14
#	* restore section of pre-ncurses-4.2 changelog to fix attribution -TD
#	* add konsole-256color entry -TD
#
# 2007-08-18
#	* add 9term entry (request by Juhapekka Tolvanen) -TD
#
# 2007-10-13
#	* correct kIC in rxvt+pcfkeys (prompted by Debian #446444) -TD
#	* add shift-control- and control-modified keys for rxvt editing
#	  keypad -TD
#	* update mlterm entry to 2.9.3 -TD
#	* add mlterm+pcfkeys -TD
#
# 2007-10-20
#	* move kLFT, kRIT, kind and kri capabilities from xterm-new to
#	  xterm+pcc0, etc., to make the corresponding building blocks reflect
#	  xterm's capabilities -TD
#	* add mrxvt entry -TD
#	* add xterm+r6f2, use in mlterm and mrxvt entries -TD
#
# 2007-11-03
#	* correct acsc strings for h19 and z100 (Benjamin Sittler)
#
# 2007-11-11
#	* use xterm-xf86-v44 for "xterm-xfree86", reflecting changes to
#	  xterm starting with patch #216 -TD
#	* make legacy xterm entries such as xterm-24 inherit from xterm-old,
#	  to match xterm #230 -TD
#	* extend xterm+pccX entries to match xterm #230 -TD
#	* add xterm+app, xterm+noapp, from xterm #230 -TD
#	* add/use xterm+pce2 from xterm #230, in xterm+pcfkeys -TD
#
# 2008-04-19
#	* add screen.rxvt -TD
#
# 2008-04-28
#	* add screen+fkeys (prompted by Debian #478094) -TD
#
# 2008-06-28
#	* add screen.mlterm -TD
#	* improve mlterm and mlterm+pcfkeys -TD
#
# 2008-08-23
#	* add Eterm-256color, Eterm-88color -TD
#	* add rxvt-88color -TD
#
# 2008-10-12
#	* add teraterm4.59 entry, use that as primary teraterm entry, rename
#	  original to teraterm2.3 -TD
#	* update "gnome" to 2.22.3 -TD
#	* update "konsole" to 1.6.6 -TD
#	* add "aterm" -TD
#	* add "linux2.6.26" -TD
#
# 2008-11-15
#	* change several \E[2g (clear tab at current column) to \E[3g
#	  (clear all tabs) to match definition for tbc capability -TD
#
# 2008-11-29
#	* add eterm-color -TD
#
# 2009-01-10
#	* add screen.Eterm -TD
#
# 2009-03-28
#	* correct typo in pfkey of ansi.sys-old
#	  (report by Kalle Olavi Niemitalo)
#	* move function- and cursor-keys from emx-base to ansi.sys, and create
#	  a pfkey capability which handles F1-F48 -TD
#
# 2009-05-02
#	* add vwmterm entry (Bryan Christ)
#
# 2009-09-19
#	* change ncv and op capabilities in sun-color to match Sun's entry for
#	  this (report by Laszlo Peter)
#	* improve interix smso by using reverse rather than bold (report by
#	  Kristof Zelechovski).
#
# 2009-10-03
#	* remove unnecessary kcan assignment to ^C from putty (Sven Joachim)
#	* add linux-16color (Benjamin Sittler)
#	* correct initc capability of linux-c-nc end-of-range (Benjamin Sittler)
#	* similar change for dg+ccc and dgunix+ccc (Benjamin Sittler)
#	* add ccc and initc capabilities to xterm-16color -TD
#
# 2009-10-31
#	* updated nsterm* entries (Benjamin Sittler, prompted by GenToo #206201)
#
# 2009-12-12
#	* updated nsterm* entries (Benjamin Sittler, Emanuele Giaquinta)
#
# 2009-12-19
#	* add bw (auto-left-margin) to nsterm* entries (Benjamin Sittler)
#	* rename minix to minix-1.7, add minix entry for Minux3 -TD
#
# 2009-12-26
#	* add bterm (bogl 0.1.18) -TD
#	* minor fix to rxvt+pcfkeys -TD
#
# 2010-02-06
#	* update mrxvt to 0.5.4, add mrxvt-256color -TD
#
# 2010-02-13
#	* add several screen-bce.XXX entries -TD
#
# 2010-02-23
#	* modify screen-bce.XXX entries to exclude ech, since screen's color
#	  model does not clear with color for that feature -TD
#
# 2010-03-20
#	* rename atari and st52 to atari-old, st52-old, use newer entries from
#	  FreeMiNT by Guido Flohr (from patch/report by Alan Hourihane).
#
# 2010-06-12
#	* add mlterm+256color entry -TD
#
# 2010-07-17
#	* add hard-reset for rs2 to wsvt25 to help ensure that reset ends
#	  the alternate character set (patch by Nicholas Marriott)
#
# 2010-08-28
#	* improve acsc for vt52 (Benjamin Sittler)
#	* modify nsterm entries for consistent sgr/sgr0 -TD
#	* modify xnuppc entries for consistent sgr/sgr0 -TD
#	* add invis to tek4115 sgr -TD
#
# 2010-09-11
#	* reformat acsc strings to canonical format -TD
#
# 2010-09-25
#	* add "XT" capability to entries for terminals that support both
#	  xterm-style mouse- and title-controls, for "screen" which
#	  special-cases TERM beginning with "xterm" or "rxvt" -TD
#
# 2010-10-02
#	* fill in no-parameter forms of cursor-movement where a parameterized
#	  form is available -TD
#	* fill in missing cursor controls where the form of the controls is
#	  ANSI -TD
#	* add parameterized cursor-controls to linux-basic (report by Dae) -TD
#
# 2010-10-09
#	* correct comparison used for setting 16-colors in linux-16color
#	  entry (Novell #644831) -TD
#	* improve linux-16color entry, using "dim" for color-8 which makes it
#	  gray rather than black like color-0 -TD
#
# 2010-11-20
#	* make "vte" the principal entry defining "gnome", since GNOME terminal
#	  is merely one of several terminals whose behavior is provided by this
#	  library -TD
#
# 2010-11-27
#	* fix typo in rmso for tek4106 -Goran Weinholt
#
# 2010-12-11
#	* suppress ncv in screen entry, allowing underline -Alejandro R. Sedeno
#	* also suppress ncv in konsole-base -TD
#
# 2011-02-05
#	* add U8 feature to denote entries for terminal emulators which do not
#	  support VT100 SI/SO when processing UTF-8 encoding -TD
#	* add xterm-utf8 as a demo of the U8 feature -TD
#
# 2011-02-20
#	* add cons25-debian entry (Brian M Carlson, Debian #607662).
#
# 2011-06-11
#	* update minix entry to minix 3.2 (Thomas Cort).
#
# 2011-07-09
#	* fix inconsistent tabset path in pcmw (Todd C. Miller).
#	* remove a backslash which continued comment, obscuring altos3
#	  definition with OpenBSD toolset (Nicholas Marriott).
#
# 2011-07-16
#	* add/use xterm+tmux chunk from xterm #271 -TD
#	* resync xterm-new entry from xterm #271 -TD
#	* add E3 extended capability to linux-basic (Miroslav Lichvar)
#	* add linux2.2, linux2.6, linux3.0 entries to give context for E3 -TD
#	* add SI/SO change to linux2.6 entry (Debian #515609) -TD
#
# 2011-07-21
#	* add kich1 to sun (Yuri Pankov)
#	* use bold rather than reverse for smso in sun-color (Yuri Pankov).
#
# 2011-08-06
#	* corrected k9 in dg460-ansi, add other features based on manuals -TD
#
# 2011-08-20
#	* minor cleanup of X-terminal emulator section -TD
#	* add terminator entry -TD
#	* add simpleterm entry -TD
#
# 2011-09-10
#	* add xterm+kbs fragment from xterm #272 -TD
#
# 2011-11-12
#	* add pccon entries for OpenBSD console (Alexei Malinin)
#
# 2011-12-17
#	* corrected old changelog comments -TD
#
# 2011-11-24
#	* add putty-sco -TD
#
# 2012-01-28
#	* add mach-gnu (Samuel Thibault)
#	* add mach-gnu-color, tweaks to mach-gnu -TD
#	* make sgr for sun-color agree with smso -TD
#	* make sgr for prism9 agree with other caps -TD
#	* make sgr for icl6404 agree with other caps -TD
#	* make sgr for ofcons agree with other caps -TD
#	* make sgr for att5410v1, att4415, att620 agree with other caps -TD
#	* make sgr for aaa-unk, aaa-rv agree with other caps -TD
#	* make sgr for avt-ns agree with other caps -TD
#
# 2012-02-11
#	* make sgr for xterm-pcolor agree with other caps -TD
#	* make sgr for att5425 agree with other caps -TD
#	* make sgr for att630 agree with other caps -TD
#	* make sgr for linux entries agree with other caps -TD
#	* make sgr for tvi9065 agree with other caps -TD
#	* make sgr for ncr260vt200an agree with other caps -TD
#	* make sgr for ncr160vt100pp agree with other caps -TD
#	* make sgr for ncr260vt300an agree with other caps -TD
#	* make sgr for aaa-60-dec-rv, aaa+dec agree with other caps -TD
#	* make sgr for cygwin, cygwinDBG agree with other caps -TD
#
# 2012-03-31
#	* correct order of use-clauses in st-256color -TD
#
# 2012-04-01
#	* revert 2011-07-16 change to "linux" alias, return to "linux2.2" -TD
#
# 2012-04-14
#	* document all of the user-defined capabilities in one place -TD
#	* add XT to some places to improve usefulness for other applications
#	  than screen, which would like to pretend that xterm's title is
#	  a status-line. -TD
#	* change use-clauses in ansi-mtabs, hp2626, and hp2622 based on review
#	  of ordering and overrides -TD
#
# 2012-04-21
#	* add msgr to vt420, similar DEC vtXXX entries -TD
#	* add several missing vt420 capabilities from vt220 -TD
#	* factor out ansi+pp from several entries -TD
#	* change xterm+sl and xterm+sl-twm to include only the status-line
#	  capabilities and not "use=xterm", making them more generally useful
#	  as building-blocks -TD
#	* add dec+sl building block, as example -TD
#
# 2012-04-28
#	* fix some inconsistencies between vt320/vt420, e.g., cnorm/civis -TD
#	* add eslok flag to dec+sl -TD
#	* dec+sl applies to vt320 and up -TD
#	* drop wsl width from xterm+sl -TD
#	* reuse xterm+sl in putty and nsca-m -TD
#	* add ansi+tabs to vt520 -TD
#	* add ansi+enq to vt220-vt520 -TD
#
# 2012-05-05
#	* remove p6 (bold) from opus3n1+ for consistency -TD
#	* remove acs stuff from env230 per clues in Ingres termcap -TD
#	* modify env230 sgr/sgr0 to match other capabilities -TD
#	* modify smacs/rmacs in bq300-8 to match sgr/sgr0 -TD
#	* make sgr for dku7202 agree with other caps -TD
#	* make sgr for ibmpc agree with other caps -TD
#	* make sgr for tek4107 agree with other caps -TD
#	* make sgr for ndr9500 agree with other caps -TD
#	* make sgr for sco-ansi agree with other caps -TD
#	* make sgr for d410 agree with other caps -TD
#	* make sgr for d210 agree with other caps -TD
#	* make sgr for d470c, d470c-7b agree with other caps -TD
#
# 2012-05-12
#	* rewrite vt520 entry based on vt420 -TD
#	* corrected 'op' for bterm (report by Samuel Thibault) -TD
#
# 2012-06-02
#	* add kdch1 to wsvt25 entry from NetBSD CVS (reported by David Lord,
#	  analysis by Martin Husemann).
#	* add cnorm/civis to wsvt25 entry from NetBSD CVS (report/analysis by
#	  Onno van der Linden).
#	* add kdch1 aka "Remove" to vt220 and vt220-8 entries -TD
#	* add kdch1, etc., to qvt108 -TD
#	* add dl1/il1 to some entries based on dl/il values -TD
#	* add dl to simpleterm -TD
#
# 2012-06-10
#	* modify some older xterm entries to align with xterm source -TD
#	* separate "xterm-old" alias from "xterm-r6" -TD
#
# 2012-07-28
#	* add E3 to xterm-basic and putty -TD
#
# 2012-08-11
#	* add nsterm-256color, make this the default nsterm -TD
#	* remove bw from nsterm-bce, per testing with tack -TD
#
# 2012-10-12
#       * add vte-2012, gnome-2012, making these the defaults for vte/gnome
#	  (patch by Christian Persch).
#
# 2012-11-02
#	* reviewed vte-2012, reverted most of the change since it was incorrect
#	  based on testing with tack -TD
#	* un-cancel the initc