=head1 NAME

Log::Handler::Output::Forward - Forward messages.

=head1 SYNOPSIS

    use Log::Handler::Output::Forward;

    my $forwarder = Log::Handler::Output::Forward->new(
        forward_to => sub { },
        arguments  => [ 'foo' ],
    );

    $forwarder->log($message);

=head1 DESCRIPTION

This output module makes it possible to forward messages to sub routines.

=head1 METHODS

=head2 new()

Call C<new()> to create a new Log::Handler::Output::Forward object.

The C<new()> method expected the options for the forwarder.

=head2 log()

Call C<log()> if you want to forward messages to the subroutines.

Example:

    $forwarder->log('this message will be forwarded to all sub routines');

=head2 errstr()

This function returns the last error message.

=head1 OPTIONS

=head2 forward_to

This option excepts a array reference with code references.

=head2 arguments

With this option you can define arguments that will be passed to the sub routine.

In the following example the arguments would be passed as a array to C<my_func()>.

    my $forwarder = Log::Handler::Output::Forward->new(
        forward_to => \&Class::method,
        arguments  => [ $self, 'foo' ],
    );

This would call intern:

    Class::method(@arguments, $message);

If this option is not set then the message will be passed as first argument.

=head1 FORWARDED MESSAGE

Note that the message will be forwarded as a hash reference.

If you make changes to the reference it affects all other outputs.

The hash key C<message> contains the message.

=head1 PREREQUISITES

    Carp
    Params::Validate

=head1 EXPORTS

No exports.

=head1 REPORT BUGS

Please report all bugs to <jschulz.cpan(at)bloonix.de>.

=head1 AUTHOR

Jonny Schulz <jschulz.cpan(at)bloonix.de>.

=head1 QUESTIONS

Do you have any questions or ideas?

MAIL: <jschulz.cpan(at)bloonix.de>

If you send me a mail then add Log::Handler into the subject.

=head1 COPYRIGHT

Copyright (C) 2007 by Jonny Schulz. All rights reserved.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE IS WITH
YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL
NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE SOFTWARE AS PERMITTED BY THE ABOVE LICENCE, BE
LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE
THE SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.

=cut

package Log::Handler::Output::Forward;

use strict;
use warnings;
our $VERSION = '0.00_04';
our $ERRSTR  = '';

use Carp;
use Params::Validate;

sub new {
    my $class   = shift;
    my $options = $class->_validate(@_);
    return bless $options, $class;
}

sub log {
    my ($self, $message) = @_;
    my $coderef = $self->{forward_to};

    if ($self->{arguments}) {
        eval { &$coderef(@{$self->{arguments}}, $message) };
    } else {
        eval { &$coderef($message) };
    }

    if ($@) {
        return $self->_raise_error($@);
    }

    return 1;
}

sub errstr { $ERRSTR }

#
# private stuff
#

sub _validate {
    my $class   = shift;

    my %options = Params::Validate::validate(@_, {
        forward_to => {
            type => Params::Validate::CODEREF,
        },
        arguments => {
            type => Params::Validate::ARRAYREF,
            default => undef,
        },
    });

    return \%options;
}

sub _raise_error {
    my $self = shift;
    $ERRSTR = shift;
    return undef;
}

1;
