#!/usr/bin/env raku

use lib <./lib>;
use Gen-Test :ALL;

# data from Meeus
my $ifilM = './meeus/test-data.txt';
# data generated by file: gen-jpl-test-data.raku:
my $ifilJ = './jpl/jpl-test-data.dat';

my $ofil2  = '02-APC-jpl-time-tests.t';
my $ofil2n = '02-APC-jpl-time-tests-BC.t';

# archived after being added to roast:
# tests for Rakudo PR #
my $ofilr  = 'juliandate.t';
my $ofilrd = 'juliandate.t.draft';

# tests for Rakudo PR #
my $ofiln  = 'negative-dates.t';
my $ofilnd = 'negative-dates.t.draft';

if not @*ARGS {
    say qq:to/HERE/;
    Usage: {$*PROGRAM.basename} go | raku [debug]

    Extracts test data from files:

      Meeus: $ifilM
      JPL:   $ifilJ

    and creates a test file at:

        $ofilnd

    The 'raku' option writes the final test file:

        $ofiln
    HERE
    exit;
}

my $raku  = 0;
my $debug = 0;
for @*ARGS {
    when /:i ^d/ { $debug = 1 }
    when /:i ^r/ { $raku  = 1 }
}

# collect the test data
my @M = get-meeus-test-data($ifilM.IO, :$debug);
my @J = get-jpl-test-data($ifilJ.IO, :$debug);

my @ofils;
my $fh;
if $raku {
    $fh = open $ofiln, :w;
    @ofils.push: $ofiln;
}
else {
    $fh = open $ofilnd, :w;
    @ofils.push: $ofilnd;
}

gen-raku-test-neg $fh, :@M, :@J, :$debug;
$fh.close if $fh;

say "Normal end.";
my $s = @ofils.elems > 1 ?? 's' !! '';
say "See output file$s:";
say "  $_" for @ofils;

#### subs ####
sub gen-raku-test-neg($fh, :@M, :@J, :$debug) {
    # test JPL dates from 4000 B.C. to 4000 A.D.
    constant MJD-offset = 2_400_000.5;

    # this will have to be changed to the final value as tests are stabilized:
    my $ntests = 126;

    $fh.say: qq:to/HERE/;
    use Test;

    plan $ntests;

    my \%jpl =
        # using test data from the JPL website:
        #     https://ssd.jpl.nasa.gov/tc.cgi
    HERE
    for @J -> $t {
        # still probs with negative dates
        #next if $t.year < 1970;
        next if $t.year > 100;
        $fh.say: "    '{$t.dts}' => [{$t.jd}, '{$t.dow}', {$t.day-of-week}, {$t.day-frac}, {$t.doy} ],";
    }
    $fh.say: ";\n";

    $fh.say: q:to/HERE/;

    # the difference between MJD and JD
    constant MJD-offset = 2_400_000.5;

    for %jpl.kv -> $JPL-utc, $JPL-out {
        # get all the data of interest in desired formats
        my $JPL-jd      = $JPL-out[0];
        my $JPL-jdday   = $JPL-jd.truncate;
        my $JPL-mjd     = $JPL-jd - MJD-offset;
        my $JPL-mjdday  = $JPL-mjd.truncate;
        my $JPL-dow     = $JPL-out[2];
        my $JPL-mjdfrac = frac $JPL-mjd;
        my $JPL-dayfrac = $JPL-out[3];
        my $JPL-doy     = $JPL-out[4];

        # get our data from the input utc as a DateTime object
        my $dt       = DateTime.new: $JPL-utc;
        my $mjd      = $dt.modified-julian-date;
        my $jd       = $dt.julian-date;
        my $day-frac = $dt.day-fraction;
        my $mjdday   = $dt.daycount;
        my $dow      = $dt.day-of-week;
        my $doy      = $dt.day-of-year;

        # check integral values
        is $mjdday, $JPL-mjdday, "cmp MJD.Int: got ($mjdday) vs exp ($JPL-mjdday)";
        is $dow, $JPL-dow, "cmp day-of-week: got ($dow) vs exp ($JPL-dow)";
        is $doy, $JPL-doy, "cmp day-of-year: got ($doy) vs exp ($JPL-doy)";

        # check decimal values for MJD and JD
        my $np-JPL-mjd     = ndp $JPL-mjd;
        my $np-JPL-jd      = ndp $JPL-jd;
        my $np-JPL-mjdfrac = ndp $JPL-mjdfrac;
        my $np-JPL-dayfrac = ndp $JPL-dayfrac;

        $mjd      = sprintf '%.*f', $np-JPL-mjd, $mjd;
        $jd       = sprintf '%.*f', $np-JPL-jd, $jd;
        $day-frac = sprintf '%.*f', $np-JPL-mjdfrac, $day-frac;

        is $mjd, $JPL-mjd, "cmp MJD: got ($mjd) vs exp ($JPL-mjd)";
        is $jd, $JPL-jd, "cmp JD: got ($jd) vs exp ($JPL-jd)";
        is $day-frac, $JPL-mjdfrac, "cmp day-fraction: got ($day-frac) vs exp ($JPL-mjdfrac)";
        is $day-frac, $JPL-dayfrac, "cmp day-fraction: got ($day-frac) vs exp2 ($JPL-dayfrac)";
    }

    # Subs needed for stand-alone testing with no external dependencies
    # From Math::FractionalPart):
    sub frac($x) {
        $x - floor($x)
    }
    # From Math::FractionalPart):
    sub ndp($x) {
        my $f = frac $x;
        $f == 0 ?? 0
                !! ($f.chars-2)
    }
    # From Astro::Montenbruck (from Meeus' "Astronomical Algorithms"):
    # From Astro::Montenbruck (from Meeus' "Astronomical Algorithms"):
    # From Astro::Montenbruck (from Meeus' "Astronomical Algorithms"):

    HERE

    # mandatory close, caller expects it
    $fh.close;
} # end sub gen-raku-test-neg


=finish
sub gen-raku-test($fh, @T, :$debug) {
    constant MJD-offset = 2_400_000.5;

    # this will have to be changed to the final value as tests are stabilized:
    my $ntests = 126;

    $fh.say: qq:to/HERE/;
    use Test;

    plan $ntests;

    my \%jpl =
        # using test data from the JPL website:
        #     https://ssd.jpl.nasa.gov/tc.cgi
    HERE
    for @T -> $t {
        # the current Raku doesn't handle negative MJDs
        next if $t.jd < MJD-offset;
        $fh.say: "    '{$t.dts}' => [{$t.jd}, '{$t.dow}', {$t.day-of-week}, {$t.day-frac}, {$t.doy} ],";
    }
    $fh.say: ";\n";

    $fh.say: q:to/HERE/;

    # the difference between MJD and JD
    constant MJD-offset = 2_400_000.5;

    for %jpl.kv -> $JPL-utc, $JPL-out {
        # get all the data of interest in desired formats
        my $JPL-jd      = $JPL-out[0];
        my $JPL-jdday   = $JPL-jd.truncate;
        my $JPL-mjd     = $JPL-jd - MJD-offset;
        my $JPL-mjdday  = $JPL-mjd.truncate;
        my $JPL-dow     = $JPL-out[2];
        my $JPL-mjdfrac = frac $JPL-mjd;
        my $JPL-dayfrac = $JPL-out[3];
        my $JPL-doy     = $JPL-out[4];

        # get our data from the input utc as a DateTime object
        my $dt       = DateTime.new: $JPL-utc;
        my $mjd      = $dt.modified-julian-date;
        my $jd       = $dt.julian-date;
        my $day-frac = $dt.day-fraction;
        my $mjdday   = $dt.daycount;
        my $dow      = $dt.day-of-week;
        my $doy      = $dt.day-of-year;

        # check integral values
        is $mjdday, $JPL-mjdday, "cmp MJD.Int: got ($mjdday) vs exp ($JPL-mjdday)";
        is $dow, $JPL-dow, "cmp day-of-week: got ($dow) vs exp ($JPL-dow)";
        is $doy, $JPL-doy, "cmp day-of-year: got ($doy) vs exp ($JPL-doy)";

        # check decimal values for MJD and JD
        my $np-JPL-mjd     = ndp $JPL-mjd;
        my $np-JPL-jd      = ndp $JPL-jd;
        my $np-JPL-mjdfrac = ndp $JPL-mjdfrac;
        my $np-JPL-dayfrac = ndp $JPL-dayfrac;

        $mjd      = sprintf '%.*f', $np-JPL-mjd, $mjd;
        $jd       = sprintf '%.*f', $np-JPL-jd, $jd;
        $day-frac = sprintf '%.*f', $np-JPL-mjdfrac, $day-frac;

        is $mjd, $JPL-mjd, "cmp MJD: got ($mjd) vs exp ($JPL-mjd)";
        is $jd, $JPL-jd, "cmp JD: got ($jd) vs exp ($JPL-jd)";
        is $day-frac, $JPL-mjdfrac, "cmp day-fraction: got ($day-frac) vs exp ($JPL-mjdfrac)";
        is $day-frac, $JPL-dayfrac, "cmp day-fraction: got ($day-frac) vs exp2 ($JPL-dayfrac)";
    }

    # Two subs needed for stand-alone testing with no external dependencies
    # (both are from Math::FractionalPart):
    sub frac($x) {
        $x - floor($x)
    }
    sub ndp($x) {
        my $f = frac $x;
        $f == 0 ?? 0
                !! ($f.chars-2)
    }
    HERE

    # mandatory close, caller expects it
    $fh.close;
} # sub gen-raku-test


sub gen-test3($fh, @T, :$debug) {
    # this will have to be increased as more tests are added:
    my $ntests = @T.elems;
    $fh.say: qq:to/HERE/;
    use Test;
    use DateTime::Julian :ALL;

    plan $ntests;

    # test data from JPL
    #   see:  https://ssd.jpl.nasa.gov/tc.cgi
    my \%jpl =
        # using test data from the JPL website:
        #     https://ssd.jpl.nasa.gov/tc.cgi
    HERE
    for @T -> $t {
        $fh.say: "    '{$t.dts}' => [{$t.jd}, '{$t.dow}'],";
    }
    $fh.say: ";\n";

    $fh.say: q:to/HERE/;
    for %jpl.keys.sort -> $ut {
        # with key and value JPL test data:
        my $jd  = %jpl{$ut}[0];
        my $dow = %jpl{$ut}[1];

        my $ut-dt = DateTime::Julian.new: $ut;
        my $jd-dt = DateTime::Julian.new: :juliandate($jd);
        is $ut-dt, $jd-ut;
    }
    HERE

    =begin comment
    my ($year, $month, $day, $hour, $minute, $second) =
        $utin.year, $utin.month, $utin.day, $utin.hour, $utin.minute, $utin.second;
    # convert day/hour/minute/second to day.decimalhms
    my $decimalday = hms2days $hour, $minute, $second;
    $decimalday += $day;
    my $jd = utc2jd $year, $month, $decimalday;

    # how many decimal places?
    my $nplaces = nplaces $jdexp;
    # round output to same number decimal places
    my $jdround = $jd;
    if $nplaces {
        $jdround = sprintf '%0.*f', $nplaces, $jd;
    }
    is $jdround, $jdexp, 'convert UT to JD';
    =end comment
}

# reverse to test the key/values to ensure they round trip okay
my %jd = %utc.invert;

for %jd.keys.sort -> $jd {
    my $jdin  = $jd;
    my $utexp = %jd{$jd};
    my ($year, $month, $day, $hr) = jd2utc $jd;

    # convert decimal hours to hms format
    my ($hour, $minute, $second) = hours2hms $hr;

    # get a DateTime object
    my $ut = DateTime.new: :$year, :$month, :$day, :$hour, :$minute, :$second;
    is $ut.Str, $utexp.Str, 'convert JD to UT';
}
