#-------------------------------------------------------------------------------
# Lint, run, install an Android App
# Philip R Brenan at gmail dot com, Appa Apps Ltd, 2017
#-------------------------------------------------------------------------------

package Android::Build;
require v5.16.0;
use warnings FATAL => qw(all);
use strict;
use Carp;
use Data::Dump qw(dump);
use Data::Table::Text qw(:all);
use File::Copy;
use POSIX qw(strftime);                                                         # http://www.cplusplus.com/reference/ctime/strftime/

our $VERSION = '2017.403';

#-------------------------------------------------------------------------------
# Constants
#-------------------------------------------------------------------------------

sub new()
 {package Android::Build;
  my $home        = qx(pwd) =~ s/\w+\Z//r; chomp($home);                        # Home directory
  my $name        = [split /\//, $home]->[-1];                                  # The name of the app is the directory above perl
  my $libs        = $home."libs";                                               # Libraries directory, place any additional jars the app needs here
  my $version     = strftime('%Y%m%d', localtime);                              # Version number without dots
  my $src         = $home.'src/';                                               # Edit source code here
  my $perl        = $home.'perl/';                                              # Perl folder
  my $tmp         = $home.'tmp/';                                               # App layout directories
  my $app         = $tmp.'app/';
  my $ant         = $app."ant.properties";                                      # Ant properties file
  my $lib         = $app.'libs/';                                               # App library folder
  my $res         = $app.'res/';
  my $man         = $app.'AndroidManifest.xml';                                 # Manifest file
  my $permissions =                                                             # Default permissions
   [qw(INTERNET ACCESS_WIFI_STATE ACCESS_NETWORK_STATE WRITE_EXTERNAL_STORAGE),
       qw(READ_EXTERNAL_STORAGE RECEIVE_BOOT_COMPLETED)];

  bless{home=>$home, name=>$name, libs=>$libs, version=>$version, src=>$src,
        perl=>$perl, tmp=>$tmp, app=>$app, ant=>$ant, lib=>$lib, res=>$res,
        man=>$man,   permissions=>$permissions, debug=>0, s3SaveOnRun=>0,
        target=>1,   device=>qq(emulator-5554)};
 }

if (1)                                                                          # Scalar Methods
 {Data::Table::Text::genLValueScalarMethods(qw(
    home name libs version src perl tmp app ant lib res man title
    domain activity usefulFolder icon debug sdk sdkLevels androidJar target
    device permissions s3SaveOnRun s3Save keyStoreFile keyStorePwd action
 ))}

sub package                                                     # Package for app
 {my ($a) = @_;                                                                 # Android build
  $a->domain.".".lc($a->name);
 }

sub apk                                                         # Apk name
 {my ($a) = @_;                                                                 # Android build
  $a->app.'bin/'.$a->name.($a->debug ? '-debug.apk' : '-release.apk');
 }
#-------------------------------------------------------------------------------
# Create icons for app
#-------------------------------------------------------------------------------

sub pushIcon                                                    # Create and transfer each icon  using Imagemagick
 {my ($android, $size, $dir) = @_;
  my $tmp         = $android->tmp;
  my $icon        = $android->icon;
  my $res         = $android->res;
  my $man         = $android->man;
  for my $i(qw(ic_launcher))
   {for my $d(qw(drawable))
     {my $t = $tmp.'icon.png';
      makePath($t);
      my $s = $size;
      my $I = $icon;
      my $c = "convert -strip $I -resize $s\\\!x$s}\\\! $t";
      qx($c);

      my $res = $android->res;
      my $T = $res.$d.'-'.$dir.'dpi/'.$i.'.png';
      makePath($T);
      print STDERR qx(rsync $t $T);
     }
   }
 }

sub pushIcons                                                   # Create icons
 {my ($android) = @_;
  $android->pushIcon(@$_)
    for ([48, "m"], [72, "h"], [96, "xh"], [144, "xxh"]);
 }

#-------------------------------------------------------------------------------
# Create manifest for app
#-------------------------------------------------------------------------------

sub addPermissions                                              # Create permissions
 {my ($android) = @_;
  my $P = "android.permission";
  my %p = (map {$_=>1} @{$android->permissions});
  my $p = "\n";

  for(sort keys %p)
   {$p .= "  <uses-permission android:name=\"$P.$_\"/>\n";
   }

  $p
 }

sub manifest
 {my ($android) = @_;
  my $permissions = $android->addPermissions;

  my ($minSdk, $targetSdk) = @{$android->sdkLevels};
  my $package     = $android->package;
  my $version     = $android->version;
  my $debug       = $android->debug;
  my $man         = $android->man;

  my $manifest = << "END";
<?xml version="1.0" encoding="utf-8"?>
  <manifest xmlns:android="http://schemas.android.com/apk/res/android"
    package="$package"
    android:installLocation="auto"
    android:versionCode="$version"
    android:versionName="\@string/versionName">

  <uses-sdk
    android:minSdkVersion="$minSdk"
    android:targetSdkVersion="$targetSdk"/>
  <application
    android:allowBackup="true"
    android:icon="\@drawable/ic_launcher"
    android:largeHeap="true"
    android:debuggable="true"
    android:hardwareAccelerated="true"
    android:label="\@string/app_name">
    <activity
      android:name=".Activity"
      android:configChanges="keyboard|keyboardHidden|orientation|screenSize"
      android:screenOrientation="sensor"
      android:theme="\@android:style/Theme.NoTitleBar"
      android:label="\@string/app_name">
      <intent-filter>
        <action android:name="android.intent.action.MAIN"/>
        <category android:name="android.intent.category.LAUNCHER"/>
      </intent-filter>
    </activity>
  </application>
  $permissions
</manifest>
END
  $manifest =~ s/android:debuggable="true"//gs unless $debug;
  writeFile($man, $manifest);
 }

#-------------------------------------------------------------------------------
# Create resources for app
#-------------------------------------------------------------------------------

sub resources()
 {my ($android) = @_;
  my $title     = $android->title;
  my $version   = $android->version;
  my $res       = $android->res;
  my $t = << "END";
<?xml version="1.0" encoding="utf-8"?>
<resources>
    <string name="app_name">$title</string>
    <string name="versionName">$version</string>
</resources>
END
  writeFile($res."values/strings.xml", $t);
 }

#-------------------------------------------------------------------------------
# Copy source - could be improved by adding package name automatically
#-------------------------------------------------------------------------------

sub copySource
 {my ($android) = @_;
  my $s = $android->src;
  my $p = $android->package  =~ s/\./\//gr;
  my $t = $android->tmp."app/src/$p/";
  makePath($t);
  print STDERR qx(rsync -r $s $t);
 }

#-------------------------------------------------------------------------------
# Save source code in S3
#-------------------------------------------------------------------------------

sub save
 {my ($android) = @_;
  my $s3Save      = $android->s3Save;
  my $s3SaveOnRun = $android->s3SaveOnRun;
  my $tmp         = $android->tmp;
  my $name        = $android->name;
  my $home        = $android->home;
  return unless $s3SaveOnRun;
   unless (fork())
   {my $z = $tmp."$name.zip";
    makePath($z);
    for("cd $home && zip -r $z perl src",
        "aws s3 cp $z s3://$s3Save/")
     {say STDERR;
      say STDERR for qx($_);
     }
    exit;
   }
 }

#-------------------------------------------------------------------------------
# Copy libraries
#-------------------------------------------------------------------------------

sub copyLibs
 {my ($android) = @_;
  my $lib       = $android->lib;
  my $libs      = $android->libs;
  print STDERR qx(rsync -r $_ $lib) for glob("$libs/*")
 }

#-------------------------------------------------------------------------------
# Ant properties plus signing key for Appa Apps
#-------------------------------------------------------------------------------

sub antProperties
 {my ($android) = @_;
  my $keyStoreFile = $android->keyStoreFile;
  my $keyStorePwd  = $android->keyStorePwd;
  my $ant          = $android->ant;
  my $s = <<END;
key.store=$keyStoreFile
key.alias=vocabulary
key.store.password=$keyStorePwd
key.alias.password=$keyStorePwd
dex.force.jumbo=true
END
  writeFile($ant, $s);
 }

#-------------------------------------------------------------------------------
# Create app
#-------------------------------------------------------------------------------

sub create
 {my ($android) = @_;
  my $app       = $android->app;
  my $sdk       = $android->sdk;
  my $ant       = $android->ant;
  my $target    = $android->target;
  my $name      = $android->name;
  my $activity  = $android->activity;
  my $package   = $android->package;
  qx(rm -r $app);
  my $cmd = "$sdk./tools/android create project --target $target ".
            " --name $name --path $app --activity $activity --package $package";
  print STDERR qx($cmd);                                                        # Need check of result
  $android->pushIcons;                                                          # Create icons
  $android->copySource;                                                         # Copy source
  $android->copyLibs;                                                           # Copy libraries
  $android->antProperties;                                                      # Ant properties
  $android->manifest;                                                           # Create manifest
  $android->resources;                                                          # Create resources
 }

#-------------------------------------------------------------------------------
# Make app
#-------------------------------------------------------------------------------

sub make
 {my ($android) = @_;
  my $app   = $android->app;
  my $debug = $android->debug;
  my $cmd = "ant -f $app./build.xml ".($debug ? 'debug' : 'release');           # Command
  say STDERR $cmd;
  my $r = qx($cmd);                                                             # Perform compile
  confess $r if $r !~ m/BUILD SUCCESSFUL/;
 }

#-------------------------------------------------------------------------------
# Lint app
#-------------------------------------------------------------------------------

sub lint
 {my ($android)  = @_;
  my $src        = $android->src;
  my $androidJar = $android->androidJar;
  my $cmd = qq(cd $src && javac *.java -cp  $androidJar);
  say STDERR $cmd;
  if (my $r = qx($cmd))                                                         # Perform compile
   {confess $r;
   }
 }

#-------------------------------------------------------------------------------
# Install app
#-------------------------------------------------------------------------------

sub install
 {my ($android)  = @_;
  my $sdk        = $android->sdk;
  my $apk        = $android->apk;
  my $device     = $android->device;
  my $package    = $android->package;
  my $adb        = $sdk."platform-tools/adb";
  for("$adb -s $device install -r $apk",
      "$adb -s $device shell am start $package/.Activity")
   {say STDERR qx($_)                                                           # Perform compile
   }
 }

#-------------------------------------------------------------------------------
# Actions
#-------------------------------------------------------------------------------

sub cInstall                                                                    # Install on emulator
 {my ($android)  = @_;
  say STDERR "Install";
  $android->install;
 }

sub cLint                                                                       # Lint the source code
 {my ($android)  = @_;
  say STDERR "Lint";
  $android->lint;
 }

sub cRun                                                                        # Create, make, install
 {my ($android)  = @_;
  say STDERR "Run";
  $android->save;
  $android->create;
  $android->make;                                                               # Command
  $android->install;                                                            # Perform compile
 }

#-------------------------------------------------------------------------------
# Perform actions
# lint    - lint the source code
# run     - compile and install
# install - install last compiled version
#-------------------------------------------------------------------------------

sub build
 {my ($android, @actions) = @_;
  @actions = ($android->action) unless @actions;                                # Default action if no action supplied

  while(@actions)
   {local $_ = shift @actions;

    if    (/\A-*run\z/i)     {$android->cRun}                                   # Run app
    elsif (/\A-*lint\z/i)    {$android->cLint}                                  # Lint source
    elsif (/\A-*install\z/i) {$android->cInstall}                               # Install on emulator
    else
     {confess"Ignored unknown command: $_";
     }
   }
  say STDERR "Normal finish";
 }

#-------------------------------------------------------------------------------
# Test
#-------------------------------------------------------------------------------

sub test
 {eval join('', <Android::Build::DATA>) || die $@
 }

test unless caller();

# Documentation
#extractDocumentation unless caller;

#-------------------------------------------------------------------------------
# Export
#-------------------------------------------------------------------------------

require Exporter;

use vars qw(@ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

@ISA          = qw(Exporter);
@EXPORT       = qw();
@EXPORT_OK    = qw();
%EXPORT_TAGS  = (all=>[@EXPORT, @EXPORT_OK]);

1;

=encoding utf-8

=head1 Name

Android::Build - Lint, run, install an Android App

=head1 Prerequisites

 sudo apt-get install imagemagick zip
 sudo cpan install Data::Table::Text Data::Dump Carp POSIX File::Copy;

And a version of the Android Software Development Kit.

=head1 Synopsis

This file which can be found in the tar.gz file containing this module:

  SampleApp/perl/makeWithperl.pl

contains:

 use Android::Build;

 my $a = &Android::Build::new();

 $a->title        = qq(Generic App);                                            # Title of the app as seen under the icon
 $a->domain       = qq(com.appaapps);                                           # Domain name in reverse order
 $a->activity     = qq(Activity);                                               # Name of Activity = $activity.java file containing onCreate() for this app
 $a->icon         = qq(~/images/Jets/EEL.jpg);                                  # Image that will be scaled to make an icon using Imagemagick
 $a->sdk          = qq(~/Android/sdk/);                                         # Android SDK on the local machine
 $a->sdkLevels    = [15,25];                                                    # Min sdk, target sdk for manifest
 $a->androidJar   = $a->sdk."platforms/android-25/android.jar";                 # Android sdk jar
 $a->keyStoreFile = qq(~/keystore/release-key.keystore);                        # Keystore file
 $a->keyStorePwd  = qq(xxx);                                                    # Password for keystore

 $a->build(qw(run));                                                            # Compile and run the app

Modify the values above to reflect your local environment, then start an
emulator and run:

 perl SampleApp/perl/makeWithPerl.pl

to compile the sample app and load it into the emulator.

=head1 File layout

If your Android build description is in file:

 /somewhere/$folder/perl/makeWithPerl.pl

then the Java source and libs for your app should be in:

 /somewhere/$folder/src/*.java
 /somewhere/$folder/libs/*.jar

and the java package name for your app should be:

 package $domain.$folder

where:

 $domain

is your reversed domain name written in lowercase.

 use Android::Build;

 my $a = &Android::Build::new();
 ...
 $a->build(qw(run));

will copy the files in the b<src> and b<lib> folders into an Android project
created in the b<tmp> folder which will then be compiled and launched on the
specified device or emulator

=head1 Installation

Standard Module::Build process for building and installing modules:

  perl Build.PL
  ./Build
  ./Build test
  ./Build install

=head1 Author

philiprbrenan@gmail.com

http://www.appaapps.com

=head1 Copyright

Copyright (c) 2016 Philip R Brenan.

This module is free software. It may be used, redistributed and/or
modified under the same terms as Perl itself.

=cut

__DATA__
use Test::More tests => 1;

ok 1;
