# -*- mode: perl; coding: iso-8859-1-unix; -*-
#
# Author:      Peter J. Acklam
# Time-stamp:  2008-04-20 11:48:55 +02:00
# E-mail:      pjacklam@cpan.org
# URL:         http://home.online.no/~pjacklam

=pod

=head1 NAME

Text::Unaccent::PurePerl - remove accents from a string

=head1 SYNOPSIS

  use Text::Unaccent::PurePerl qw(unac_string);

  $unaccented = unac_string($string);

  # For compatibility with Text::Unaccent, and
  # for dealing with strings of raw octets:

  $unaccented = unac_string($charset, $octets);
  $unaccented = unac_string_utf16($octets);

  # For compatibility with Text::Unaccent, but
  # have no useful purpose in this module.
  $version = unac_version();
  unac_debug($level);

=head1 DESCRIPTION

Text::Unaccent::PurePerl is a module for removing accents from a string.  It is
essentially a pure Perl equivalent to the Text::Unaccent module, but this one
also properly handles character strings, whereas Text::Unaccent only deals with
raw octet strings with an associated character coding. In addition, this
module, as the name suggests, does not require a C compiler to build. The
disadvantage is that this module is much slower than the compiled
Text::Unaccent.

=head1 EXPORT

Functions exported by default: C<unac_string>, C<unac_string_utf16>,
C<unac_version>, and C<unac_debug>.

=cut

package Text::Unaccent::PurePerl;

use 5.006;              # for the 'utf8' and 'warnings' pragmas
use strict;             # restrict unsafe constructs
use warnings;           # control optional warnings

#use utf8;               # enable/disable UTF-8 (or UTF-EBCDIC) in source code

use Carp;
use Exporter;

our @ISA = qw(Exporter);
our @EXPORT = qw(
                    unac_string
                    unac_string_utf16
                    unac_version
                    unac_debug
               );
our @EXPORT_OK = qw();

our $VERSION = '0.01';

# These three are only included for compatibility with Text::Unaccent. They
# have no effect on the behaviour of this module.

our $DEBUG_NONE = undef;
our $DEBUG_LOW  = undef;
our $DEBUG_HIGH = undef;

my $map =
{

 # 00A0 NO-BREAK SPACE
 # ->   0020 SPACE
 "\xA0" => " ",

 # 00A8 DIAERESIS
 # ->   0020 SPACE
 "\xA8" => " ",

 # 00AA FEMININE ORDINAL INDICATOR
 # ->   0061 LATIN SMALL LETTER A
 "\xAA" => "a",

 # 00AF MACRON
 # ->   0020 SPACE
 "\xAF" => " ",

 # 00B2 SUPERSCRIPT TWO
 # ->   0032 DIGIT TWO
 "\xB2" => "2",

 # 00B3 SUPERSCRIPT THREE
 # ->   0033 DIGIT THREE
 "\xB3" => "3",

 # 00B4 ACUTE ACCENT
 # ->   0020 SPACE
 "\xB4" => " ",

 # 00B5 MICRO SIGN
 # ->   03BC GREEK SMALL LETTER MU
 "\xB5" => "\x{03BC}",

 # 00B8 CEDILLA
 # ->   0020 SPACE
 "\xB8" => " ",

 # 00B9 SUPERSCRIPT ONE
 # ->   0031 DIGIT ONE
 "\xB9" => "1",

 # 00BA MASCULINE ORDINAL INDICATOR
 # ->   006F LATIN SMALL LETTER O
 "\xBA" => "o",

 # 00BC VULGAR FRACTION ONE QUARTER
 # ->   0031 DIGIT ONE
 #    + 2044 FRACTION SLASH
 #    + 0034 DIGIT FOUR
 "\xBC" => "1\x{2044}4",

 # 00BD VULGAR FRACTION ONE HALF
 # ->   0031 DIGIT ONE
 #    + 2044 FRACTION SLASH
 #    + 0032 DIGIT TWO
 "\xBD" => "1\x{2044}2",

 # 00BE VULGAR FRACTION THREE QUARTERS
 # ->   0033 DIGIT THREE
 #    + 2044 FRACTION SLASH
 #    + 0034 DIGIT FOUR
 "\xBE" => "3\x{2044}4",

 # 00C0 LATIN CAPITAL LETTER A WITH GRAVE
 # ->   0041 LATIN CAPITAL LETTER A
 "\xC0" => "A",

 # 00C1 LATIN CAPITAL LETTER A WITH ACUTE
 # ->   0041 LATIN CAPITAL LETTER A
 "\xC1" => "A",

 # 00C2 LATIN CAPITAL LETTER A WITH CIRCUMFLEX
 # ->   0041 LATIN CAPITAL LETTER A
 "\xC2" => "A",

 # 00C3 LATIN CAPITAL LETTER A WITH TILDE
 # ->   0041 LATIN CAPITAL LETTER A
 "\xC3" => "A",

 # 00C4 LATIN CAPITAL LETTER A WITH DIAERESIS
 # ->   0041 LATIN CAPITAL LETTER A
 "\xC4" => "A",

 # 00C5 LATIN CAPITAL LETTER A WITH RING ABOVE
 # ->   0041 LATIN CAPITAL LETTER A
 "\xC5" => "A",

 # 00C6 LATIN CAPITAL LETTER AE
 # ->   0041 LATIN CAPITAL LETTER A
 #    + 0045 LATIN CAPITAL LETTER E
 "\xC6" => "AE",

 # 00C7 LATIN CAPITAL LETTER C WITH CEDILLA
 # ->   0043 LATIN CAPITAL LETTER C
 "\xC7" => "C",

 # 00C8 LATIN CAPITAL LETTER E WITH GRAVE
 # ->   0045 LATIN CAPITAL LETTER E
 "\xC8" => "E",

 # 00C9 LATIN CAPITAL LETTER E WITH ACUTE
 # ->   0045 LATIN CAPITAL LETTER E
 "\xC9" => "E",

 # 00CA LATIN CAPITAL LETTER E WITH CIRCUMFLEX
 # ->   0045 LATIN CAPITAL LETTER E
 "\xCA" => "E",

 # 00CB LATIN CAPITAL LETTER E WITH DIAERESIS
 # ->   0045 LATIN CAPITAL LETTER E
 "\xCB" => "E",

 # 00CC LATIN CAPITAL LETTER I WITH GRAVE
 # ->   0049 LATIN CAPITAL LETTER I
 "\xCC" => "I",

 # 00CD LATIN CAPITAL LETTER I WITH ACUTE
 # ->   0049 LATIN CAPITAL LETTER I
 "\xCD" => "I",

 # 00CE LATIN CAPITAL LETTER I WITH CIRCUMFLEX
 # ->   0049 LATIN CAPITAL LETTER I
 "\xCE" => "I",

 # 00CF LATIN CAPITAL LETTER I WITH DIAERESIS
 # ->   0049 LATIN CAPITAL LETTER I
 "\xCF" => "I",

 # 00D1 LATIN CAPITAL LETTER N WITH TILDE
 # ->   004E LATIN CAPITAL LETTER N
 "\xD1" => "N",

 # 00D2 LATIN CAPITAL LETTER O WITH GRAVE
 # ->   004F LATIN CAPITAL LETTER O
 "\xD2" => "O",

 # 00D3 LATIN CAPITAL LETTER O WITH ACUTE
 # ->   004F LATIN CAPITAL LETTER O
 "\xD3" => "O",

 # 00D4 LATIN CAPITAL LETTER O WITH CIRCUMFLEX
 # ->   004F LATIN CAPITAL LETTER O
 "\xD4" => "O",

 # 00D5 LATIN CAPITAL LETTER O WITH TILDE
 # ->   004F LATIN CAPITAL LETTER O
 "\xD5" => "O",

 # 00D6 LATIN CAPITAL LETTER O WITH DIAERESIS
 # ->   004F LATIN CAPITAL LETTER O
 "\xD6" => "O",

 # 00D8 LATIN CAPITAL LETTER O WITH STROKE
 # ->   004F LATIN CAPITAL LETTER O
 "\xD8" => "O",

 # 00D9 LATIN CAPITAL LETTER U WITH GRAVE
 # ->   0055 LATIN CAPITAL LETTER U
 "\xD9" => "U",

 # 00DA LATIN CAPITAL LETTER U WITH ACUTE
 # ->   0055 LATIN CAPITAL LETTER U
 "\xDA" => "U",

 # 00DB LATIN CAPITAL LETTER U WITH CIRCUMFLEX
 # ->   0055 LATIN CAPITAL LETTER U
 "\xDB" => "U",

 # 00DC LATIN CAPITAL LETTER U WITH DIAERESIS
 # ->   0055 LATIN CAPITAL LETTER U
 "\xDC" => "U",

 # 00DD LATIN CAPITAL LETTER Y WITH ACUTE
 # ->   0059 LATIN CAPITAL LETTER Y
 "\xDD" => "Y",

 # 00E0 LATIN SMALL LETTER A WITH GRAVE
 # ->   0061 LATIN SMALL LETTER A
 "\xE0" => "a",

 # 00E1 LATIN SMALL LETTER A WITH ACUTE
 # ->   0061 LATIN SMALL LETTER A
 "\xE1" => "a",

 # 00E2 LATIN SMALL LETTER A WITH CIRCUMFLEX
 # ->   0061 LATIN SMALL LETTER A
 "\xE2" => "a",

 # 00E3 LATIN SMALL LETTER A WITH TILDE
 # ->   0061 LATIN SMALL LETTER A
 "\xE3" => "a",

 # 00E4 LATIN SMALL LETTER A WITH DIAERESIS
 # ->   0061 LATIN SMALL LETTER A
 "\xE4" => "a",

 # 00E5 LATIN SMALL LETTER A WITH RING ABOVE
 # ->   0061 LATIN SMALL LETTER A
 "\xE5" => "a",

 # 00E6 LATIN SMALL LETTER AE
 # ->   0061 LATIN SMALL LETTER A
 #    + 0065 LATIN SMALL LETTER E
 "\xE6" => "ae",

 # 00E7 LATIN SMALL LETTER C WITH CEDILLA
 # ->   0063 LATIN SMALL LETTER C
 "\xE7" => "c",

 # 00E8 LATIN SMALL LETTER E WITH GRAVE
 # ->   0065 LATIN SMALL LETTER E
 "\xE8" => "e",

 # 00E9 LATIN SMALL LETTER E WITH ACUTE
 # ->   0065 LATIN SMALL LETTER E
 "\xE9" => "e",

 # 00EA LATIN SMALL LETTER E WITH CIRCUMFLEX
 # ->   0065 LATIN SMALL LETTER E
 "\xEA" => "e",

 # 00EB LATIN SMALL LETTER E WITH DIAERESIS
 # ->   0065 LATIN SMALL LETTER E
 "\xEB" => "e",

 # 00EC LATIN SMALL LETTER I WITH GRAVE
 # ->   0069 LATIN SMALL LETTER I
 "\xEC" => "i",

 # 00ED LATIN SMALL LETTER I WITH ACUTE
 # ->   0069 LATIN SMALL LETTER I
 "\xED" => "i",

 # 00EE LATIN SMALL LETTER I WITH CIRCUMFLEX
 # ->   0069 LATIN SMALL LETTER I
 "\xEE" => "i",

 # 00EF LATIN SMALL LETTER I WITH DIAERESIS
 # ->   0069 LATIN SMALL LETTER I
 "\xEF" => "i",

 # 00F1 LATIN SMALL LETTER N WITH TILDE
 # ->   006E LATIN SMALL LETTER N
 "\xF1" => "n",

 # 00F2 LATIN SMALL LETTER O WITH GRAVE
 # ->   006F LATIN SMALL LETTER O
 "\xF2" => "o",

 # 00F3 LATIN SMALL LETTER O WITH ACUTE
 # ->   006F LATIN SMALL LETTER O
 "\xF3" => "o",

 # 00F4 LATIN SMALL LETTER O WITH CIRCUMFLEX
 # ->   006F LATIN SMALL LETTER O
 "\xF4" => "o",

 # 00F5 LATIN SMALL LETTER O WITH TILDE
 # ->   006F LATIN SMALL LETTER O
 "\xF5" => "o",

 # 00F6 LATIN SMALL LETTER O WITH DIAERESIS
 # ->   006F LATIN SMALL LETTER O
 "\xF6" => "o",

 # 00F8 LATIN SMALL LETTER O WITH STROKE
 # ->   006F LATIN SMALL LETTER O
 "\xF8" => "o",

 # 00F9 LATIN SMALL LETTER U WITH GRAVE
 # ->   0075 LATIN SMALL LETTER U
 "\xF9" => "u",

 # 00FA LATIN SMALL LETTER U WITH ACUTE
 # ->   0075 LATIN SMALL LETTER U
 "\xFA" => "u",

 # 00FB LATIN SMALL LETTER U WITH CIRCUMFLEX
 # ->   0075 LATIN SMALL LETTER U
 "\xFB" => "u",

 # 00FC LATIN SMALL LETTER U WITH DIAERESIS
 # ->   0075 LATIN SMALL LETTER U
 "\xFC" => "u",

 # 00FD LATIN SMALL LETTER Y WITH ACUTE
 # ->   0079 LATIN SMALL LETTER Y
 "\xFD" => "y",

 # 00FF LATIN SMALL LETTER Y WITH DIAERESIS
 # ->   0079 LATIN SMALL LETTER Y
 "\xFF" => "y",

 # 0100 LATIN CAPITAL LETTER A WITH MACRON
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{0100}" => "A",

 # 0101 LATIN SMALL LETTER A WITH MACRON
 # ->   0061 LATIN SMALL LETTER A
 "\x{0101}" => "a",

 # 0102 LATIN CAPITAL LETTER A WITH BREVE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{0102}" => "A",

 # 0103 LATIN SMALL LETTER A WITH BREVE
 # ->   0061 LATIN SMALL LETTER A
 "\x{0103}" => "a",

 # 0104 LATIN CAPITAL LETTER A WITH OGONEK
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{0104}" => "A",

 # 0105 LATIN SMALL LETTER A WITH OGONEK
 # ->   0061 LATIN SMALL LETTER A
 "\x{0105}" => "a",

 # 0106 LATIN CAPITAL LETTER C WITH ACUTE
 # ->   0043 LATIN CAPITAL LETTER C
 "\x{0106}" => "C",

 # 0107 LATIN SMALL LETTER C WITH ACUTE
 # ->   0063 LATIN SMALL LETTER C
 "\x{0107}" => "c",

 # 0108 LATIN CAPITAL LETTER C WITH CIRCUMFLEX
 # ->   0043 LATIN CAPITAL LETTER C
 "\x{0108}" => "C",

 # 0109 LATIN SMALL LETTER C WITH CIRCUMFLEX
 # ->   0063 LATIN SMALL LETTER C
 "\x{0109}" => "c",

 # 010A LATIN CAPITAL LETTER C WITH DOT ABOVE
 # ->   0043 LATIN CAPITAL LETTER C
 "\x{010A}" => "C",

 # 010B LATIN SMALL LETTER C WITH DOT ABOVE
 # ->   0063 LATIN SMALL LETTER C
 "\x{010B}" => "c",

 # 010C LATIN CAPITAL LETTER C WITH CARON
 # ->   0043 LATIN CAPITAL LETTER C
 "\x{010C}" => "C",

 # 010D LATIN SMALL LETTER C WITH CARON
 # ->   0063 LATIN SMALL LETTER C
 "\x{010D}" => "c",

 # 010E LATIN CAPITAL LETTER D WITH CARON
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{010E}" => "D",

 # 010F LATIN SMALL LETTER D WITH CARON
 # ->   0064 LATIN SMALL LETTER D
 "\x{010F}" => "d",

 # 0110 LATIN CAPITAL LETTER D WITH STROKE
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{0110}" => "D",

 # 0111 LATIN SMALL LETTER D WITH STROKE
 # ->   0064 LATIN SMALL LETTER D
 "\x{0111}" => "d",

 # 0112 LATIN CAPITAL LETTER E WITH MACRON
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{0112}" => "E",

 # 0113 LATIN SMALL LETTER E WITH MACRON
 # ->   0065 LATIN SMALL LETTER E
 "\x{0113}" => "e",

 # 0114 LATIN CAPITAL LETTER E WITH BREVE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{0114}" => "E",

 # 0115 LATIN SMALL LETTER E WITH BREVE
 # ->   0065 LATIN SMALL LETTER E
 "\x{0115}" => "e",

 # 0116 LATIN CAPITAL LETTER E WITH DOT ABOVE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{0116}" => "E",

 # 0117 LATIN SMALL LETTER E WITH DOT ABOVE
 # ->   0065 LATIN SMALL LETTER E
 "\x{0117}" => "e",

 # 0118 LATIN CAPITAL LETTER E WITH OGONEK
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{0118}" => "E",

 # 0119 LATIN SMALL LETTER E WITH OGONEK
 # ->   0065 LATIN SMALL LETTER E
 "\x{0119}" => "e",

 # 011A LATIN CAPITAL LETTER E WITH CARON
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{011A}" => "E",

 # 011B LATIN SMALL LETTER E WITH CARON
 # ->   0065 LATIN SMALL LETTER E
 "\x{011B}" => "e",

 # 011C LATIN CAPITAL LETTER G WITH CIRCUMFLEX
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{011C}" => "G",

 # 011D LATIN SMALL LETTER G WITH CIRCUMFLEX
 # ->   0067 LATIN SMALL LETTER G
 "\x{011D}" => "g",

 # 011E LATIN CAPITAL LETTER G WITH BREVE
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{011E}" => "G",

 # 011F LATIN SMALL LETTER G WITH BREVE
 # ->   0067 LATIN SMALL LETTER G
 "\x{011F}" => "g",

 # 0120 LATIN CAPITAL LETTER G WITH DOT ABOVE
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{0120}" => "G",

 # 0121 LATIN SMALL LETTER G WITH DOT ABOVE
 # ->   0067 LATIN SMALL LETTER G
 "\x{0121}" => "g",

 # 0122 LATIN CAPITAL LETTER G WITH CEDILLA
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{0122}" => "G",

 # 0123 LATIN SMALL LETTER G WITH CEDILLA
 # ->   0067 LATIN SMALL LETTER G
 "\x{0123}" => "g",

 # 0124 LATIN CAPITAL LETTER H WITH CIRCUMFLEX
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{0124}" => "H",

 # 0125 LATIN SMALL LETTER H WITH CIRCUMFLEX
 # ->   0068 LATIN SMALL LETTER H
 "\x{0125}" => "h",

 # 0126 LATIN CAPITAL LETTER H WITH STROKE
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{0126}" => "H",

 # 0127 LATIN SMALL LETTER H WITH STROKE
 # ->   0068 LATIN SMALL LETTER H
 "\x{0127}" => "h",

 # 0128 LATIN CAPITAL LETTER I WITH TILDE
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{0128}" => "I",

 # 0129 LATIN SMALL LETTER I WITH TILDE
 # ->   0069 LATIN SMALL LETTER I
 "\x{0129}" => "i",

 # 012A LATIN CAPITAL LETTER I WITH MACRON
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{012A}" => "I",

 # 012B LATIN SMALL LETTER I WITH MACRON
 # ->   0069 LATIN SMALL LETTER I
 "\x{012B}" => "i",

 # 012C LATIN CAPITAL LETTER I WITH BREVE
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{012C}" => "I",

 # 012D LATIN SMALL LETTER I WITH BREVE
 # ->   0069 LATIN SMALL LETTER I
 "\x{012D}" => "i",

 # 012E LATIN CAPITAL LETTER I WITH OGONEK
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{012E}" => "I",

 # 012F LATIN SMALL LETTER I WITH OGONEK
 # ->   0069 LATIN SMALL LETTER I
 "\x{012F}" => "i",

 # 0130 LATIN CAPITAL LETTER I WITH DOT ABOVE
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{0130}" => "I",

 # 0132 LATIN CAPITAL LIGATURE IJ
 # ->   0049 LATIN CAPITAL LETTER I
 #    + 004A LATIN CAPITAL LETTER J
 "\x{0132}" => "IJ",

 # 0133 LATIN SMALL LIGATURE IJ
 # ->   0069 LATIN SMALL LETTER I
 #    + 006A LATIN SMALL LETTER J
 "\x{0133}" => "ij",

 # 0134 LATIN CAPITAL LETTER J WITH CIRCUMFLEX
 # ->   004A LATIN CAPITAL LETTER J
 "\x{0134}" => "J",

 # 0135 LATIN SMALL LETTER J WITH CIRCUMFLEX
 # ->   006A LATIN SMALL LETTER J
 "\x{0135}" => "j",

 # 0136 LATIN CAPITAL LETTER K WITH CEDILLA
 # ->   004B LATIN CAPITAL LETTER K
 "\x{0136}" => "K",

 # 0137 LATIN SMALL LETTER K WITH CEDILLA
 # ->   006B LATIN SMALL LETTER K
 "\x{0137}" => "k",

 # 0139 LATIN CAPITAL LETTER L WITH ACUTE
 # ->   004C LATIN CAPITAL LETTER L
 "\x{0139}" => "L",

 # 013A LATIN SMALL LETTER L WITH ACUTE
 # ->   006C LATIN SMALL LETTER L
 "\x{013A}" => "l",

 # 013B LATIN CAPITAL LETTER L WITH CEDILLA
 # ->   004C LATIN CAPITAL LETTER L
 "\x{013B}" => "L",

 # 013C LATIN SMALL LETTER L WITH CEDILLA
 # ->   006C LATIN SMALL LETTER L
 "\x{013C}" => "l",

 # 013D LATIN CAPITAL LETTER L WITH CARON
 # ->   004C LATIN CAPITAL LETTER L
 "\x{013D}" => "L",

 # 013E LATIN SMALL LETTER L WITH CARON
 # ->   006C LATIN SMALL LETTER L
 "\x{013E}" => "l",

 # 013F LATIN CAPITAL LETTER L WITH MIDDLE DOT
 # ->   004C LATIN CAPITAL LETTER L
 #    + 00B7 MIDDLE DOT
 "\x{013F}" => "L\xB7",

 # 0140 LATIN SMALL LETTER L WITH MIDDLE DOT
 # ->   006C LATIN SMALL LETTER L
 #    + 00B7 MIDDLE DOT
 "\x{0140}" => "l\xB7",

 # 0141 LATIN CAPITAL LETTER L WITH STROKE
 # ->   004C LATIN CAPITAL LETTER L
 "\x{0141}" => "L",

 # 0142 LATIN SMALL LETTER L WITH STROKE
 # ->   006C LATIN SMALL LETTER L
 "\x{0142}" => "l",

 # 0143 LATIN CAPITAL LETTER N WITH ACUTE
 # ->   004E LATIN CAPITAL LETTER N
 "\x{0143}" => "N",

 # 0144 LATIN SMALL LETTER N WITH ACUTE
 # ->   006E LATIN SMALL LETTER N
 "\x{0144}" => "n",

 # 0145 LATIN CAPITAL LETTER N WITH CEDILLA
 # ->   004E LATIN CAPITAL LETTER N
 "\x{0145}" => "N",

 # 0146 LATIN SMALL LETTER N WITH CEDILLA
 # ->   006E LATIN SMALL LETTER N
 "\x{0146}" => "n",

 # 0147 LATIN CAPITAL LETTER N WITH CARON
 # ->   004E LATIN CAPITAL LETTER N
 "\x{0147}" => "N",

 # 0148 LATIN SMALL LETTER N WITH CARON
 # ->   006E LATIN SMALL LETTER N
 "\x{0148}" => "n",

 # 0149 LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
 # ->   02BC MODIFIER LETTER APOSTROPHE
 #    + 006E LATIN SMALL LETTER N
 "\x{0149}" => "\x{02BC}n",

 # 014C LATIN CAPITAL LETTER O WITH MACRON
 # ->   004F LATIN CAPITAL LETTER O
 "\x{014C}" => "O",

 # 014D LATIN SMALL LETTER O WITH MACRON
 # ->   006F LATIN SMALL LETTER O
 "\x{014D}" => "o",

 # 014E LATIN CAPITAL LETTER O WITH BREVE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{014E}" => "O",

 # 014F LATIN SMALL LETTER O WITH BREVE
 # ->   006F LATIN SMALL LETTER O
 "\x{014F}" => "o",

 # 0150 LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{0150}" => "O",

 # 0151 LATIN SMALL LETTER O WITH DOUBLE ACUTE
 # ->   006F LATIN SMALL LETTER O
 "\x{0151}" => "o",

 # 0154 LATIN CAPITAL LETTER R WITH ACUTE
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{0154}" => "R",

 # 0155 LATIN SMALL LETTER R WITH ACUTE
 # ->   0072 LATIN SMALL LETTER R
 "\x{0155}" => "r",

 # 0156 LATIN CAPITAL LETTER R WITH CEDILLA
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{0156}" => "R",

 # 0157 LATIN SMALL LETTER R WITH CEDILLA
 # ->   0072 LATIN SMALL LETTER R
 "\x{0157}" => "r",

 # 0158 LATIN CAPITAL LETTER R WITH CARON
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{0158}" => "R",

 # 0159 LATIN SMALL LETTER R WITH CARON
 # ->   0072 LATIN SMALL LETTER R
 "\x{0159}" => "r",

 # 015A LATIN CAPITAL LETTER S WITH ACUTE
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{015A}" => "S",

 # 015B LATIN SMALL LETTER S WITH ACUTE
 # ->   0073 LATIN SMALL LETTER S
 "\x{015B}" => "s",

 # 015C LATIN CAPITAL LETTER S WITH CIRCUMFLEX
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{015C}" => "S",

 # 015D LATIN SMALL LETTER S WITH CIRCUMFLEX
 # ->   0073 LATIN SMALL LETTER S
 "\x{015D}" => "s",

 # 015E LATIN CAPITAL LETTER S WITH CEDILLA
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{015E}" => "S",

 # 015F LATIN SMALL LETTER S WITH CEDILLA
 # ->   0073 LATIN SMALL LETTER S
 "\x{015F}" => "s",

 # 0160 LATIN CAPITAL LETTER S WITH CARON
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{0160}" => "S",

 # 0161 LATIN SMALL LETTER S WITH CARON
 # ->   0073 LATIN SMALL LETTER S
 "\x{0161}" => "s",

 # 0162 LATIN CAPITAL LETTER T WITH CEDILLA
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{0162}" => "T",

 # 0163 LATIN SMALL LETTER T WITH CEDILLA
 # ->   0074 LATIN SMALL LETTER T
 "\x{0163}" => "t",

 # 0164 LATIN CAPITAL LETTER T WITH CARON
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{0164}" => "T",

 # 0165 LATIN SMALL LETTER T WITH CARON
 # ->   0074 LATIN SMALL LETTER T
 "\x{0165}" => "t",

 # 0166 LATIN CAPITAL LETTER T WITH STROKE
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{0166}" => "T",

 # 0167 LATIN SMALL LETTER T WITH STROKE
 # ->   0074 LATIN SMALL LETTER T
 "\x{0167}" => "t",

 # 0168 LATIN CAPITAL LETTER U WITH TILDE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{0168}" => "U",

 # 0169 LATIN SMALL LETTER U WITH TILDE
 # ->   0075 LATIN SMALL LETTER U
 "\x{0169}" => "u",

 # 016A LATIN CAPITAL LETTER U WITH MACRON
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{016A}" => "U",

 # 016B LATIN SMALL LETTER U WITH MACRON
 # ->   0075 LATIN SMALL LETTER U
 "\x{016B}" => "u",

 # 016C LATIN CAPITAL LETTER U WITH BREVE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{016C}" => "U",

 # 016D LATIN SMALL LETTER U WITH BREVE
 # ->   0075 LATIN SMALL LETTER U
 "\x{016D}" => "u",

 # 016E LATIN CAPITAL LETTER U WITH RING ABOVE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{016E}" => "U",

 # 016F LATIN SMALL LETTER U WITH RING ABOVE
 # ->   0075 LATIN SMALL LETTER U
 "\x{016F}" => "u",

 # 0170 LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{0170}" => "U",

 # 0171 LATIN SMALL LETTER U WITH DOUBLE ACUTE
 # ->   0075 LATIN SMALL LETTER U
 "\x{0171}" => "u",

 # 0172 LATIN CAPITAL LETTER U WITH OGONEK
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{0172}" => "U",

 # 0173 LATIN SMALL LETTER U WITH OGONEK
 # ->   0075 LATIN SMALL LETTER U
 "\x{0173}" => "u",

 # 0174 LATIN CAPITAL LETTER W WITH CIRCUMFLEX
 # ->   0057 LATIN CAPITAL LETTER W
 "\x{0174}" => "W",

 # 0175 LATIN SMALL LETTER W WITH CIRCUMFLEX
 # ->   0077 LATIN SMALL LETTER W
 "\x{0175}" => "w",

 # 0176 LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
 # ->   0059 LATIN CAPITAL LETTER Y
 "\x{0176}" => "Y",

 # 0177 LATIN SMALL LETTER Y WITH CIRCUMFLEX
 # ->   0079 LATIN SMALL LETTER Y
 "\x{0177}" => "y",

 # 0178 LATIN CAPITAL LETTER Y WITH DIAERESIS
 # ->   0059 LATIN CAPITAL LETTER Y
 "\x{0178}" => "Y",

 # 0179 LATIN CAPITAL LETTER Z WITH ACUTE
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{0179}" => "Z",

 # 017A LATIN SMALL LETTER Z WITH ACUTE
 # ->   007A LATIN SMALL LETTER Z
 "\x{017A}" => "z",

 # 017B LATIN CAPITAL LETTER Z WITH DOT ABOVE
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{017B}" => "Z",

 # 017C LATIN SMALL LETTER Z WITH DOT ABOVE
 # ->   007A LATIN SMALL LETTER Z
 "\x{017C}" => "z",

 # 017D LATIN CAPITAL LETTER Z WITH CARON
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{017D}" => "Z",

 # 017E LATIN SMALL LETTER Z WITH CARON
 # ->   007A LATIN SMALL LETTER Z
 "\x{017E}" => "z",

 # 017F LATIN SMALL LETTER LONG S
 # ->   0073 LATIN SMALL LETTER S
 "\x{017F}" => "s",

 # 0180 LATIN SMALL LETTER B WITH STROKE
 # ->   0062 LATIN SMALL LETTER B
 "\x{0180}" => "b",

 # 0181 LATIN CAPITAL LETTER B WITH HOOK
 # ->   0042 LATIN CAPITAL LETTER B
 "\x{0181}" => "B",

 # 0182 LATIN CAPITAL LETTER B WITH TOPBAR
 # ->   0042 LATIN CAPITAL LETTER B
 "\x{0182}" => "B",

 # 0183 LATIN SMALL LETTER B WITH TOPBAR
 # ->   0062 LATIN SMALL LETTER B
 "\x{0183}" => "b",

 # 0187 LATIN CAPITAL LETTER C WITH HOOK
 # ->   0043 LATIN CAPITAL LETTER C
 "\x{0187}" => "C",

 # 0188 LATIN SMALL LETTER C WITH HOOK
 # ->   0063 LATIN SMALL LETTER C
 "\x{0188}" => "c",

 # 018A LATIN CAPITAL LETTER D WITH HOOK
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{018A}" => "D",

 # 018B LATIN CAPITAL LETTER D WITH TOPBAR
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{018B}" => "D",

 # 018C LATIN SMALL LETTER D WITH TOPBAR
 # ->   0064 LATIN SMALL LETTER D
 "\x{018C}" => "d",

 # 0191 LATIN CAPITAL LETTER F WITH HOOK
 # ->   0046 LATIN CAPITAL LETTER F
 "\x{0191}" => "F",

 # 0192 LATIN SMALL LETTER F WITH HOOK
 # ->   0066 LATIN SMALL LETTER F
 "\x{0192}" => "f",

 # 0193 LATIN CAPITAL LETTER G WITH HOOK
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{0193}" => "G",

 # 0197 LATIN CAPITAL LETTER I WITH STROKE
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{0197}" => "I",

 # 0198 LATIN CAPITAL LETTER K WITH HOOK
 # ->   004B LATIN CAPITAL LETTER K
 "\x{0198}" => "K",

 # 0199 LATIN SMALL LETTER K WITH HOOK
 # ->   006B LATIN SMALL LETTER K
 "\x{0199}" => "k",

 # 019A LATIN SMALL LETTER L WITH BAR
 # ->   006C LATIN SMALL LETTER L
 "\x{019A}" => "l",

 # 019D LATIN CAPITAL LETTER N WITH LEFT HOOK
 # ->   004E LATIN CAPITAL LETTER N
 "\x{019D}" => "N",

 # 019E LATIN SMALL LETTER N WITH LONG RIGHT LEG
 # ->   006E LATIN SMALL LETTER N
 "\x{019E}" => "n",

 # 019F LATIN CAPITAL LETTER O WITH MIDDLE TILDE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{019F}" => "O",

 # 01A0 LATIN CAPITAL LETTER O WITH HORN
 # ->   004F LATIN CAPITAL LETTER O
 "\x{01A0}" => "O",

 # 01A1 LATIN SMALL LETTER O WITH HORN
 # ->   006F LATIN SMALL LETTER O
 "\x{01A1}" => "o",

 # 01A4 LATIN CAPITAL LETTER P WITH HOOK
 # ->   0050 LATIN CAPITAL LETTER P
 "\x{01A4}" => "P",

 # 01A5 LATIN SMALL LETTER P WITH HOOK
 # ->   0070 LATIN SMALL LETTER P
 "\x{01A5}" => "p",

 # 01AB LATIN SMALL LETTER T WITH PALATAL HOOK
 # ->   0074 LATIN SMALL LETTER T
 "\x{01AB}" => "t",

 # 01AC LATIN CAPITAL LETTER T WITH HOOK
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{01AC}" => "T",

 # 01AD LATIN SMALL LETTER T WITH HOOK
 # ->   0074 LATIN SMALL LETTER T
 "\x{01AD}" => "t",

 # 01AE LATIN CAPITAL LETTER T WITH RETROFLEX HOOK
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{01AE}" => "T",

 # 01AF LATIN CAPITAL LETTER U WITH HORN
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{01AF}" => "U",

 # 01B0 LATIN SMALL LETTER U WITH HORN
 # ->   0075 LATIN SMALL LETTER U
 "\x{01B0}" => "u",

 # 01B2 LATIN CAPITAL LETTER V WITH HOOK
 # ->   0056 LATIN CAPITAL LETTER V
 "\x{01B2}" => "V",

 # 01B3 LATIN CAPITAL LETTER Y WITH HOOK
 # ->   0059 LATIN CAPITAL LETTER Y
 "\x{01B3}" => "Y",

 # 01B4 LATIN SMALL LETTER Y WITH HOOK
 # ->   0079 LATIN SMALL LETTER Y
 "\x{01B4}" => "y",

 # 01B5 LATIN CAPITAL LETTER Z WITH STROKE
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{01B5}" => "Z",

 # 01B6 LATIN SMALL LETTER Z WITH STROKE
 # ->   007A LATIN SMALL LETTER Z
 "\x{01B6}" => "z",

 # 01BA LATIN SMALL LETTER EZH WITH TAIL
 # ->   0292 LATIN SMALL LETTER EZH
 "\x{01BA}" => "\x{0292}",

 # 01BE LATIN LETTER INVERTED GLOTTAL STOP WITH STROKE
 # ->   0296 LATIN LETTER INVERTED GLOTTAL STOP
 "\x{01BE}" => "\x{0296}",

 # 01C4 LATIN CAPITAL LETTER DZ WITH CARON
 # ->   0044 LATIN CAPITAL LETTER D
 #    + 005A LATIN CAPITAL LETTER Z
 "\x{01C4}" => "DZ",

 # 01C5 LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON
 # ->   0044 LATIN CAPITAL LETTER D
 #    + 007A LATIN SMALL LETTER Z
 "\x{01C5}" => "Dz",

 # 01C6 LATIN SMALL LETTER DZ WITH CARON
 # ->   0064 LATIN SMALL LETTER D
 #    + 007A LATIN SMALL LETTER Z
 "\x{01C6}" => "dz",

 # 01C7 LATIN CAPITAL LETTER LJ
 # ->   004C LATIN CAPITAL LETTER L
 #    + 004A LATIN CAPITAL LETTER J
 "\x{01C7}" => "LJ",

 # 01C8 LATIN CAPITAL LETTER L WITH SMALL LETTER J
 # ->   004C LATIN CAPITAL LETTER L
 #    + 006A LATIN SMALL LETTER J
 "\x{01C8}" => "Lj",

 # 01C9 LATIN SMALL LETTER LJ
 # ->   006C LATIN SMALL LETTER L
 #    + 006A LATIN SMALL LETTER J
 "\x{01C9}" => "lj",

 # 01CA LATIN CAPITAL LETTER NJ
 # ->   004E LATIN CAPITAL LETTER N
 #    + 004A LATIN CAPITAL LETTER J
 "\x{01CA}" => "NJ",

 # 01CB LATIN CAPITAL LETTER N WITH SMALL LETTER J
 # ->   004E LATIN CAPITAL LETTER N
 #    + 006A LATIN SMALL LETTER J
 "\x{01CB}" => "Nj",

 # 01CC LATIN SMALL LETTER NJ
 # ->   006E LATIN SMALL LETTER N
 #    + 006A LATIN SMALL LETTER J
 "\x{01CC}" => "nj",

 # 01CD LATIN CAPITAL LETTER A WITH CARON
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{01CD}" => "A",

 # 01CE LATIN SMALL LETTER A WITH CARON
 # ->   0061 LATIN SMALL LETTER A
 "\x{01CE}" => "a",

 # 01CF LATIN CAPITAL LETTER I WITH CARON
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{01CF}" => "I",

 # 01D0 LATIN SMALL LETTER I WITH CARON
 # ->   0069 LATIN SMALL LETTER I
 "\x{01D0}" => "i",

 # 01D1 LATIN CAPITAL LETTER O WITH CARON
 # ->   004F LATIN CAPITAL LETTER O
 "\x{01D1}" => "O",

 # 01D2 LATIN SMALL LETTER O WITH CARON
 # ->   006F LATIN SMALL LETTER O
 "\x{01D2}" => "o",

 # 01D3 LATIN CAPITAL LETTER U WITH CARON
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{01D3}" => "U",

 # 01D4 LATIN SMALL LETTER U WITH CARON
 # ->   0075 LATIN SMALL LETTER U
 "\x{01D4}" => "u",

 # 01D5 LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{01D5}" => "U",

 # 01D6 LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
 # ->   0075 LATIN SMALL LETTER U
 "\x{01D6}" => "u",

 # 01D7 LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{01D7}" => "U",

 # 01D8 LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
 # ->   0075 LATIN SMALL LETTER U
 "\x{01D8}" => "u",

 # 01D9 LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{01D9}" => "U",

 # 01DA LATIN SMALL LETTER U WITH DIAERESIS AND CARON
 # ->   0075 LATIN SMALL LETTER U
 "\x{01DA}" => "u",

 # 01DB LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{01DB}" => "U",

 # 01DC LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE
 # ->   0075 LATIN SMALL LETTER U
 "\x{01DC}" => "u",

 # 01DE LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{01DE}" => "A",

 # 01DF LATIN SMALL LETTER A WITH DIAERESIS AND MACRON
 # ->   0061 LATIN SMALL LETTER A
 "\x{01DF}" => "a",

 # 01E0 LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{01E0}" => "A",

 # 01E1 LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON
 # ->   0061 LATIN SMALL LETTER A
 "\x{01E1}" => "a",

 # 01E2 LATIN CAPITAL LETTER AE WITH MACRON
 # ->   0041 LATIN CAPITAL LETTER A
 #    + 0045 LATIN CAPITAL LETTER E
 "\x{01E2}" => "AE",

 # 01E3 LATIN SMALL LETTER AE WITH MACRON
 # ->   0061 LATIN SMALL LETTER A
 #    + 0065 LATIN SMALL LETTER E
 "\x{01E3}" => "ae",

 # 01E4 LATIN CAPITAL LETTER G WITH STROKE
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{01E4}" => "G",

 # 01E5 LATIN SMALL LETTER G WITH STROKE
 # ->   0067 LATIN SMALL LETTER G
 "\x{01E5}" => "g",

 # 01E6 LATIN CAPITAL LETTER G WITH CARON
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{01E6}" => "G",

 # 01E7 LATIN SMALL LETTER G WITH CARON
 # ->   0067 LATIN SMALL LETTER G
 "\x{01E7}" => "g",

 # 01E8 LATIN CAPITAL LETTER K WITH CARON
 # ->   004B LATIN CAPITAL LETTER K
 "\x{01E8}" => "K",

 # 01E9 LATIN SMALL LETTER K WITH CARON
 # ->   006B LATIN SMALL LETTER K
 "\x{01E9}" => "k",

 # 01EA LATIN CAPITAL LETTER O WITH OGONEK
 # ->   004F LATIN CAPITAL LETTER O
 "\x{01EA}" => "O",

 # 01EB LATIN SMALL LETTER O WITH OGONEK
 # ->   006F LATIN SMALL LETTER O
 "\x{01EB}" => "o",

 # 01EC LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
 # ->   004F LATIN CAPITAL LETTER O
 "\x{01EC}" => "O",

 # 01ED LATIN SMALL LETTER O WITH OGONEK AND MACRON
 # ->   006F LATIN SMALL LETTER O
 "\x{01ED}" => "o",

 # 01EE LATIN CAPITAL LETTER EZH WITH CARON
 # ->   01B7 LATIN CAPITAL LETTER EZH
 "\x{01EE}" => "\x{01B7}",

 # 01EF LATIN SMALL LETTER EZH WITH CARON
 # ->   0292 LATIN SMALL LETTER EZH
 "\x{01EF}" => "\x{0292}",

 # 01F0 LATIN SMALL LETTER J WITH CARON
 # ->   006A LATIN SMALL LETTER J
 "\x{01F0}" => "j",

 # 01F1 LATIN CAPITAL LETTER DZ
 # ->   0044 LATIN CAPITAL LETTER D
 #    + 005A LATIN CAPITAL LETTER Z
 "\x{01F1}" => "DZ",

 # 01F2 LATIN CAPITAL LETTER D WITH SMALL LETTER Z
 # ->   0044 LATIN CAPITAL LETTER D
 #    + 007A LATIN SMALL LETTER Z
 "\x{01F2}" => "Dz",

 # 01F3 LATIN SMALL LETTER DZ
 # ->   0064 LATIN SMALL LETTER D
 #    + 007A LATIN SMALL LETTER Z
 "\x{01F3}" => "dz",

 # 01F4 LATIN CAPITAL LETTER G WITH ACUTE
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{01F4}" => "G",

 # 01F5 LATIN SMALL LETTER G WITH ACUTE
 # ->   0067 LATIN SMALL LETTER G
 "\x{01F5}" => "g",

 # 01F8 LATIN CAPITAL LETTER N WITH GRAVE
 # ->   004E LATIN CAPITAL LETTER N
 "\x{01F8}" => "N",

 # 01F9 LATIN SMALL LETTER N WITH GRAVE
 # ->   006E LATIN SMALL LETTER N
 "\x{01F9}" => "n",

 # 01FA LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{01FA}" => "A",

 # 01FB LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE
 # ->   0061 LATIN SMALL LETTER A
 "\x{01FB}" => "a",

 # 01FC LATIN CAPITAL LETTER AE WITH ACUTE
 # ->   0041 LATIN CAPITAL LETTER A
 #    + 0045 LATIN CAPITAL LETTER E
 "\x{01FC}" => "AE",

 # 01FD LATIN SMALL LETTER AE WITH ACUTE
 # ->   0061 LATIN SMALL LETTER A
 #    + 0065 LATIN SMALL LETTER E
 "\x{01FD}" => "ae",

 # 01FE LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{01FE}" => "O",

 # 01FF LATIN SMALL LETTER O WITH STROKE AND ACUTE
 # ->   006F LATIN SMALL LETTER O
 "\x{01FF}" => "o",

 # 0200 LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{0200}" => "A",

 # 0201 LATIN SMALL LETTER A WITH DOUBLE GRAVE
 # ->   0061 LATIN SMALL LETTER A
 "\x{0201}" => "a",

 # 0202 LATIN CAPITAL LETTER A WITH INVERTED BREVE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{0202}" => "A",

 # 0203 LATIN SMALL LETTER A WITH INVERTED BREVE
 # ->   0061 LATIN SMALL LETTER A
 "\x{0203}" => "a",

 # 0204 LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{0204}" => "E",

 # 0205 LATIN SMALL LETTER E WITH DOUBLE GRAVE
 # ->   0065 LATIN SMALL LETTER E
 "\x{0205}" => "e",

 # 0206 LATIN CAPITAL LETTER E WITH INVERTED BREVE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{0206}" => "E",

 # 0207 LATIN SMALL LETTER E WITH INVERTED BREVE
 # ->   0065 LATIN SMALL LETTER E
 "\x{0207}" => "e",

 # 0208 LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{0208}" => "I",

 # 0209 LATIN SMALL LETTER I WITH DOUBLE GRAVE
 # ->   0069 LATIN SMALL LETTER I
 "\x{0209}" => "i",

 # 020A LATIN CAPITAL LETTER I WITH INVERTED BREVE
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{020A}" => "I",

 # 020B LATIN SMALL LETTER I WITH INVERTED BREVE
 # ->   0069 LATIN SMALL LETTER I
 "\x{020B}" => "i",

 # 020C LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{020C}" => "O",

 # 020D LATIN SMALL LETTER O WITH DOUBLE GRAVE
 # ->   006F LATIN SMALL LETTER O
 "\x{020D}" => "o",

 # 020E LATIN CAPITAL LETTER O WITH INVERTED BREVE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{020E}" => "O",

 # 020F LATIN SMALL LETTER O WITH INVERTED BREVE
 # ->   006F LATIN SMALL LETTER O
 "\x{020F}" => "o",

 # 0210 LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{0210}" => "R",

 # 0211 LATIN SMALL LETTER R WITH DOUBLE GRAVE
 # ->   0072 LATIN SMALL LETTER R
 "\x{0211}" => "r",

 # 0212 LATIN CAPITAL LETTER R WITH INVERTED BREVE
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{0212}" => "R",

 # 0213 LATIN SMALL LETTER R WITH INVERTED BREVE
 # ->   0072 LATIN SMALL LETTER R
 "\x{0213}" => "r",

 # 0214 LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{0214}" => "U",

 # 0215 LATIN SMALL LETTER U WITH DOUBLE GRAVE
 # ->   0075 LATIN SMALL LETTER U
 "\x{0215}" => "u",

 # 0216 LATIN CAPITAL LETTER U WITH INVERTED BREVE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{0216}" => "U",

 # 0217 LATIN SMALL LETTER U WITH INVERTED BREVE
 # ->   0075 LATIN SMALL LETTER U
 "\x{0217}" => "u",

 # 0218 LATIN CAPITAL LETTER S WITH COMMA BELOW
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{0218}" => "S",

 # 0219 LATIN SMALL LETTER S WITH COMMA BELOW
 # ->   0073 LATIN SMALL LETTER S
 "\x{0219}" => "s",

 # 021A LATIN CAPITAL LETTER T WITH COMMA BELOW
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{021A}" => "T",

 # 021B LATIN SMALL LETTER T WITH COMMA BELOW
 # ->   0074 LATIN SMALL LETTER T
 "\x{021B}" => "t",

 # 021E LATIN CAPITAL LETTER H WITH CARON
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{021E}" => "H",

 # 021F LATIN SMALL LETTER H WITH CARON
 # ->   0068 LATIN SMALL LETTER H
 "\x{021F}" => "h",

 # 0220 LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
 # ->   004E LATIN CAPITAL LETTER N
 "\x{0220}" => "N",

 # 0221 LATIN SMALL LETTER D WITH CURL
 # ->   0064 LATIN SMALL LETTER D
 "\x{0221}" => "d",

 # 0224 LATIN CAPITAL LETTER Z WITH HOOK
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{0224}" => "Z",

 # 0225 LATIN SMALL LETTER Z WITH HOOK
 # ->   007A LATIN SMALL LETTER Z
 "\x{0225}" => "z",

 # 0226 LATIN CAPITAL LETTER A WITH DOT ABOVE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{0226}" => "A",

 # 0227 LATIN SMALL LETTER A WITH DOT ABOVE
 # ->   0061 LATIN SMALL LETTER A
 "\x{0227}" => "a",

 # 0228 LATIN CAPITAL LETTER E WITH CEDILLA
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{0228}" => "E",

 # 0229 LATIN SMALL LETTER E WITH CEDILLA
 # ->   0065 LATIN SMALL LETTER E
 "\x{0229}" => "e",

 # 022A LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON
 # ->   004F LATIN CAPITAL LETTER O
 "\x{022A}" => "O",

 # 022B LATIN SMALL LETTER O WITH DIAERESIS AND MACRON
 # ->   006F LATIN SMALL LETTER O
 "\x{022B}" => "o",

 # 022C LATIN CAPITAL LETTER O WITH TILDE AND MACRON
 # ->   004F LATIN CAPITAL LETTER O
 "\x{022C}" => "O",

 # 022D LATIN SMALL LETTER O WITH TILDE AND MACRON
 # ->   006F LATIN SMALL LETTER O
 "\x{022D}" => "o",

 # 022E LATIN CAPITAL LETTER O WITH DOT ABOVE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{022E}" => "O",

 # 022F LATIN SMALL LETTER O WITH DOT ABOVE
 # ->   006F LATIN SMALL LETTER O
 "\x{022F}" => "o",

 # 0230 LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON
 # ->   004F LATIN CAPITAL LETTER O
 "\x{0230}" => "O",

 # 0231 LATIN SMALL LETTER O WITH DOT ABOVE AND MACRON
 # ->   006F LATIN SMALL LETTER O
 "\x{0231}" => "o",

 # 0232 LATIN CAPITAL LETTER Y WITH MACRON
 # ->   0059 LATIN CAPITAL LETTER Y
 "\x{0232}" => "Y",

 # 0233 LATIN SMALL LETTER Y WITH MACRON
 # ->   0079 LATIN SMALL LETTER Y
 "\x{0233}" => "y",

 # 0234 LATIN SMALL LETTER L WITH CURL
 # ->   006C LATIN SMALL LETTER L
 "\x{0234}" => "l",

 # 0235 LATIN SMALL LETTER N WITH CURL
 # ->   006E LATIN SMALL LETTER N
 "\x{0235}" => "n",

 # 0236 LATIN SMALL LETTER T WITH CURL
 # ->   0074 LATIN SMALL LETTER T
 "\x{0236}" => "t",

 # 0253 LATIN SMALL LETTER B WITH HOOK
 # ->   0062 LATIN SMALL LETTER B
 "\x{0253}" => "b",

 # 0255 LATIN SMALL LETTER C WITH CURL
 # ->   0063 LATIN SMALL LETTER C
 "\x{0255}" => "c",

 # 0256 LATIN SMALL LETTER D WITH TAIL
 # ->   0064 LATIN SMALL LETTER D
 "\x{0256}" => "d",

 # 0257 LATIN SMALL LETTER D WITH HOOK
 # ->   0064 LATIN SMALL LETTER D
 "\x{0257}" => "d",

 # 025A LATIN SMALL LETTER SCHWA WITH HOOK
 # ->   0259 LATIN SMALL LETTER SCHWA
 "\x{025A}" => "\x{0259}",

 # 025D LATIN SMALL LETTER REVERSED OPEN E WITH HOOK
 # ->   025C LATIN SMALL LETTER REVERSED OPEN E
 "\x{025D}" => "\x{025C}",

 # 0260 LATIN SMALL LETTER G WITH HOOK
 # ->   0067 LATIN SMALL LETTER G
 "\x{0260}" => "g",

 # 0266 LATIN SMALL LETTER H WITH HOOK
 # ->   0068 LATIN SMALL LETTER H
 "\x{0266}" => "h",

 # 0268 LATIN SMALL LETTER I WITH STROKE
 # ->   0069 LATIN SMALL LETTER I
 "\x{0268}" => "i",

 # 026B LATIN SMALL LETTER L WITH MIDDLE TILDE
 # ->   006C LATIN SMALL LETTER L
 "\x{026B}" => "l",

 # 026C LATIN SMALL LETTER L WITH BELT
 # ->   006C LATIN SMALL LETTER L
 "\x{026C}" => "l",

 # 026D LATIN SMALL LETTER L WITH RETROFLEX HOOK
 # ->   006C LATIN SMALL LETTER L
 "\x{026D}" => "l",

 # 0270 LATIN SMALL LETTER TURNED M WITH LONG LEG
 # ->   026F LATIN SMALL LETTER TURNED M
 "\x{0270}" => "\x{026F}",

 # 0271 LATIN SMALL LETTER M WITH HOOK
 # ->   006D LATIN SMALL LETTER M
 "\x{0271}" => "m",

 # 0272 LATIN SMALL LETTER N WITH LEFT HOOK
 # ->   006E LATIN SMALL LETTER N
 "\x{0272}" => "n",

 # 0273 LATIN SMALL LETTER N WITH RETROFLEX HOOK
 # ->   006E LATIN SMALL LETTER N
 "\x{0273}" => "n",

 # 027A LATIN SMALL LETTER TURNED R WITH LONG LEG
 # ->   0279 LATIN SMALL LETTER TURNED R
 "\x{027A}" => "\x{0279}",

 # 027B LATIN SMALL LETTER TURNED R WITH HOOK
 # ->   0279 LATIN SMALL LETTER TURNED R
 "\x{027B}" => "\x{0279}",

 # 027C LATIN SMALL LETTER R WITH LONG LEG
 # ->   0072 LATIN SMALL LETTER R
 "\x{027C}" => "r",

 # 027D LATIN SMALL LETTER R WITH TAIL
 # ->   0072 LATIN SMALL LETTER R
 "\x{027D}" => "r",

 # 027E LATIN SMALL LETTER R WITH FISHHOOK
 # ->   0072 LATIN SMALL LETTER R
 "\x{027E}" => "r",

 # 0282 LATIN SMALL LETTER S WITH HOOK
 # ->   0073 LATIN SMALL LETTER S
 "\x{0282}" => "s",

 # 0286 LATIN SMALL LETTER ESH WITH CURL
 # ->   0283 LATIN SMALL LETTER ESH
 "\x{0286}" => "\x{0283}",

 # 0288 LATIN SMALL LETTER T WITH RETROFLEX HOOK
 # ->   0074 LATIN SMALL LETTER T
 "\x{0288}" => "t",

 # 028B LATIN SMALL LETTER V WITH HOOK
 # ->   0076 LATIN SMALL LETTER V
 "\x{028B}" => "v",

 # 0290 LATIN SMALL LETTER Z WITH RETROFLEX HOOK
 # ->   007A LATIN SMALL LETTER Z
 "\x{0290}" => "z",

 # 0291 LATIN SMALL LETTER Z WITH CURL
 # ->   007A LATIN SMALL LETTER Z
 "\x{0291}" => "z",

 # 0293 LATIN SMALL LETTER EZH WITH CURL
 # ->   0292 LATIN SMALL LETTER EZH
 "\x{0293}" => "\x{0292}",

 # 029B LATIN LETTER SMALL CAPITAL G WITH HOOK
 # ->   0262 LATIN LETTER SMALL CAPITAL G
 "\x{029B}" => "\x{0262}",

 # 029D LATIN SMALL LETTER J WITH CROSSED-TAIL
 # ->   006A LATIN SMALL LETTER J
 "\x{029D}" => "j",

 # 02A0 LATIN SMALL LETTER Q WITH HOOK
 # ->   0071 LATIN SMALL LETTER Q
 "\x{02A0}" => "q",

 # 02A1 LATIN LETTER GLOTTAL STOP WITH STROKE
 # ->   0294 LATIN LETTER GLOTTAL STOP
 "\x{02A1}" => "\x{0294}",

 # 02A5 LATIN SMALL LETTER DZ DIGRAPH WITH CURL
 # ->   02A3 LATIN SMALL LETTER DZ DIGRAPH
 "\x{02A5}" => "\x{02A3}",

 # 02AE LATIN SMALL LETTER TURNED H WITH FISHHOOK
 # ->   0265 LATIN SMALL LETTER TURNED H
 "\x{02AE}" => "\x{0265}",

 # 02AF LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
 # ->   0265 LATIN SMALL LETTER TURNED H
 "\x{02AF}" => "\x{0265}",

 # 02B0 MODIFIER LETTER SMALL H
 # ->   0068 LATIN SMALL LETTER H
 "\x{02B0}" => "h",

 # 02B1 MODIFIER LETTER SMALL H WITH HOOK
 # ->   0068 LATIN SMALL LETTER H
 "\x{02B1}" => "h",

 # 02B2 MODIFIER LETTER SMALL J
 # ->   006A LATIN SMALL LETTER J
 "\x{02B2}" => "j",

 # 02B3 MODIFIER LETTER SMALL R
 # ->   0072 LATIN SMALL LETTER R
 "\x{02B3}" => "r",

 # 02B4 MODIFIER LETTER SMALL TURNED R
 # ->   0279 LATIN SMALL LETTER TURNED R
 "\x{02B4}" => "\x{0279}",

 # 02B5 MODIFIER LETTER SMALL TURNED R WITH HOOK
 # ->   0279 LATIN SMALL LETTER TURNED R
 "\x{02B5}" => "\x{0279}",

 # 02B6 MODIFIER LETTER SMALL CAPITAL INVERTED R
 # ->   0281 LATIN LETTER SMALL CAPITAL INVERTED R
 "\x{02B6}" => "\x{0281}",

 # 02B7 MODIFIER LETTER SMALL W
 # ->   0077 LATIN SMALL LETTER W
 "\x{02B7}" => "w",

 # 02B8 MODIFIER LETTER SMALL Y
 # ->   0079 LATIN SMALL LETTER Y
 "\x{02B8}" => "y",

 # 02D8 BREVE
 # ->   0020 SPACE
 "\x{02D8}" => " ",

 # 02D9 DOT ABOVE
 # ->   0020 SPACE
 "\x{02D9}" => " ",

 # 02DA RING ABOVE
 # ->   0020 SPACE
 "\x{02DA}" => " ",

 # 02DB OGONEK
 # ->   0020 SPACE
 "\x{02DB}" => " ",

 # 02DC SMALL TILDE
 # ->   0020 SPACE
 "\x{02DC}" => " ",

 # 02DD DOUBLE ACUTE ACCENT
 # ->   0020 SPACE
 "\x{02DD}" => " ",

 # 02E0 MODIFIER LETTER SMALL GAMMA
 # ->   0263 LATIN SMALL LETTER GAMMA
 "\x{02E0}" => "\x{0263}",

 # 02E1 MODIFIER LETTER SMALL L
 # ->   006C LATIN SMALL LETTER L
 "\x{02E1}" => "l",

 # 02E2 MODIFIER LETTER SMALL S
 # ->   0073 LATIN SMALL LETTER S
 "\x{02E2}" => "s",

 # 02E3 MODIFIER LETTER SMALL X
 # ->   0078 LATIN SMALL LETTER X
 "\x{02E3}" => "x",

 # 02E4 MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
 # ->   0295 LATIN LETTER PHARYNGEAL VOICED FRICATIVE
 "\x{02E4}" => "\x{0295}",

 # 0374 GREEK NUMERAL SIGN
 # ->   02B9 MODIFIER LETTER PRIME
 "\x{0374}" => "\x{02B9}",

 # 037A GREEK YPOGEGRAMMENI
 # ->   0020 SPACE
 "\x{037A}" => " ",

 # 037E GREEK QUESTION MARK
 # ->   003B SEMICOLON
 "\x{037E}" => ";",

 # 0384 GREEK TONOS
 # ->   0020 SPACE
 "\x{0384}" => " ",

 # 0385 GREEK DIALYTIKA TONOS
 # ->   0020 SPACE
 "\x{0385}" => " ",

 # 0386 GREEK CAPITAL LETTER ALPHA WITH TONOS
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{0386}" => "\x{0391}",

 # 0387 GREEK ANO TELEIA
 # ->   00B7 MIDDLE DOT
 "\x{0387}" => "\xB7",

 # 0388 GREEK CAPITAL LETTER EPSILON WITH TONOS
 # ->   0395 GREEK CAPITAL LETTER EPSILON
 "\x{0388}" => "\x{0395}",

 # 0389 GREEK CAPITAL LETTER ETA WITH TONOS
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{0389}" => "\x{0397}",

 # 038A GREEK CAPITAL LETTER IOTA WITH TONOS
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{038A}" => "\x{0399}",

 # 038C GREEK CAPITAL LETTER OMICRON WITH TONOS
 # ->   039F GREEK CAPITAL LETTER OMICRON
 "\x{038C}" => "\x{039F}",

 # 038E GREEK CAPITAL LETTER UPSILON WITH TONOS
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{038E}" => "\x{03A5}",

 # 038F GREEK CAPITAL LETTER OMEGA WITH TONOS
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{038F}" => "\x{03A9}",

 # 0390 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{0390}" => "\x{03B9}",

 # 03AA GREEK CAPITAL LETTER IOTA WITH DIALYTIKA
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{03AA}" => "\x{0399}",

 # 03AB GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{03AB}" => "\x{03A5}",

 # 03AC GREEK SMALL LETTER ALPHA WITH TONOS
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{03AC}" => "\x{03B1}",

 # 03AD GREEK SMALL LETTER EPSILON WITH TONOS
 # ->   03B5 GREEK SMALL LETTER EPSILON
 "\x{03AD}" => "\x{03B5}",

 # 03AE GREEK SMALL LETTER ETA WITH TONOS
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{03AE}" => "\x{03B7}",

 # 03AF GREEK SMALL LETTER IOTA WITH TONOS
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{03AF}" => "\x{03B9}",

 # 03B0 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{03B0}" => "\x{03C5}",

 # 03CA GREEK SMALL LETTER IOTA WITH DIALYTIKA
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{03CA}" => "\x{03B9}",

 # 03CB GREEK SMALL LETTER UPSILON WITH DIALYTIKA
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{03CB}" => "\x{03C5}",

 # 03CC GREEK SMALL LETTER OMICRON WITH TONOS
 # ->   03BF GREEK SMALL LETTER OMICRON
 "\x{03CC}" => "\x{03BF}",

 # 03CD GREEK SMALL LETTER UPSILON WITH TONOS
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{03CD}" => "\x{03C5}",

 # 03CE GREEK SMALL LETTER OMEGA WITH TONOS
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{03CE}" => "\x{03C9}",

 # 03D0 GREEK BETA SYMBOL
 # ->   03B2 GREEK SMALL LETTER BETA
 "\x{03D0}" => "\x{03B2}",

 # 03D1 GREEK THETA SYMBOL
 # ->   03B8 GREEK SMALL LETTER THETA
 "\x{03D1}" => "\x{03B8}",

 # 03D2 GREEK UPSILON WITH HOOK SYMBOL
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{03D2}" => "\x{03A5}",

 # 03D3 GREEK UPSILON WITH ACUTE AND HOOK SYMBOL
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{03D3}" => "\x{03A5}",

 # 03D4 GREEK UPSILON WITH DIAERESIS AND HOOK SYMBOL
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{03D4}" => "\x{03A5}",

 # 03D5 GREEK PHI SYMBOL
 # ->   03C6 GREEK SMALL LETTER PHI
 "\x{03D5}" => "\x{03C6}",

 # 03D6 GREEK PI SYMBOL
 # ->   03C0 GREEK SMALL LETTER PI
 "\x{03D6}" => "\x{03C0}",

 # 03F0 GREEK KAPPA SYMBOL
 # ->   03BA GREEK SMALL LETTER KAPPA
 "\x{03F0}" => "\x{03BA}",

 # 03F1 GREEK RHO SYMBOL
 # ->   03C1 GREEK SMALL LETTER RHO
 "\x{03F1}" => "\x{03C1}",

 # 03F2 GREEK LUNATE SIGMA SYMBOL
 # ->   03C2 GREEK SMALL LETTER FINAL SIGMA
 "\x{03F2}" => "\x{03C2}",

 # 03F4 GREEK CAPITAL THETA SYMBOL
 # ->   0398 GREEK CAPITAL LETTER THETA
 "\x{03F4}" => "\x{0398}",

 # 03F5 GREEK LUNATE EPSILON SYMBOL
 # ->   03B5 GREEK SMALL LETTER EPSILON
 "\x{03F5}" => "\x{03B5}",

 # 03F9 GREEK CAPITAL LUNATE SIGMA SYMBOL
 # ->   03A3 GREEK CAPITAL LETTER SIGMA
 "\x{03F9}" => "\x{03A3}",

 # 0400 CYRILLIC CAPITAL LETTER IE WITH GRAVE
 # ->   0415 CYRILLIC CAPITAL LETTER IE
 "\x{0400}" => "\x{0415}",

 # 0401 CYRILLIC CAPITAL LETTER IO
 # ->   0415 CYRILLIC CAPITAL LETTER IE
 "\x{0401}" => "\x{0415}",

 # 0403 CYRILLIC CAPITAL LETTER GJE
 # ->   0413 CYRILLIC CAPITAL LETTER GHE
 "\x{0403}" => "\x{0413}",

 # 0407 CYRILLIC CAPITAL LETTER YI
 # ->   0406 CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I
 "\x{0407}" => "\x{0406}",

 # 040C CYRILLIC CAPITAL LETTER KJE
 # ->   041A CYRILLIC CAPITAL LETTER KA
 "\x{040C}" => "\x{041A}",

 # 040D CYRILLIC CAPITAL LETTER I WITH GRAVE
 # ->   0418 CYRILLIC CAPITAL LETTER I
 "\x{040D}" => "\x{0418}",

 # 040E CYRILLIC CAPITAL LETTER SHORT U
 # ->   0423 CYRILLIC CAPITAL LETTER U
 "\x{040E}" => "\x{0423}",

 # 0419 CYRILLIC CAPITAL LETTER SHORT I
 # ->   0418 CYRILLIC CAPITAL LETTER I
 "\x{0419}" => "\x{0418}",

 # 0439 CYRILLIC SMALL LETTER SHORT I
 # ->   0438 CYRILLIC SMALL LETTER I
 "\x{0439}" => "\x{0438}",

 # 0450 CYRILLIC SMALL LETTER IE WITH GRAVE
 # ->   0435 CYRILLIC SMALL LETTER IE
 "\x{0450}" => "\x{0435}",

 # 0451 CYRILLIC SMALL LETTER IO
 # ->   0435 CYRILLIC SMALL LETTER IE
 "\x{0451}" => "\x{0435}",

 # 0453 CYRILLIC SMALL LETTER GJE
 # ->   0433 CYRILLIC SMALL LETTER GHE
 "\x{0453}" => "\x{0433}",

 # 0457 CYRILLIC SMALL LETTER YI
 # ->   0456 CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
 "\x{0457}" => "\x{0456}",

 # 045C CYRILLIC SMALL LETTER KJE
 # ->   043A CYRILLIC SMALL LETTER KA
 "\x{045C}" => "\x{043A}",

 # 045D CYRILLIC SMALL LETTER I WITH GRAVE
 # ->   0438 CYRILLIC SMALL LETTER I
 "\x{045D}" => "\x{0438}",

 # 045E CYRILLIC SMALL LETTER SHORT U
 # ->   0443 CYRILLIC SMALL LETTER U
 "\x{045E}" => "\x{0443}",

 # 0476 CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
 # ->   0474 CYRILLIC CAPITAL LETTER IZHITSA
 "\x{0476}" => "\x{0474}",

 # 0477 CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
 # ->   0475 CYRILLIC SMALL LETTER IZHITSA
 "\x{0477}" => "\x{0475}",

 # 047C CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
 # ->   0460 CYRILLIC CAPITAL LETTER OMEGA
 "\x{047C}" => "\x{0460}",

 # 047D CYRILLIC SMALL LETTER OMEGA WITH TITLO
 # ->   0461 CYRILLIC SMALL LETTER OMEGA
 "\x{047D}" => "\x{0461}",

 # 048A CYRILLIC CAPITAL LETTER SHORT I WITH TAIL
 # ->   0418 CYRILLIC CAPITAL LETTER I
 "\x{048A}" => "\x{0418}",

 # 048B CYRILLIC SMALL LETTER SHORT I WITH TAIL
 # ->   0438 CYRILLIC SMALL LETTER I
 "\x{048B}" => "\x{0438}",

 # 048E CYRILLIC CAPITAL LETTER ER WITH TICK
 # ->   0420 CYRILLIC CAPITAL LETTER ER
 "\x{048E}" => "\x{0420}",

 # 048F CYRILLIC SMALL LETTER ER WITH TICK
 # ->   0440 CYRILLIC SMALL LETTER ER
 "\x{048F}" => "\x{0440}",

 # 0490 CYRILLIC CAPITAL LETTER GHE WITH UPTURN
 # ->   0413 CYRILLIC CAPITAL LETTER GHE
 "\x{0490}" => "\x{0413}",

 # 0491 CYRILLIC SMALL LETTER GHE WITH UPTURN
 # ->   0433 CYRILLIC SMALL LETTER GHE
 "\x{0491}" => "\x{0433}",

 # 0492 CYRILLIC CAPITAL LETTER GHE WITH STROKE
 # ->   0413 CYRILLIC CAPITAL LETTER GHE
 "\x{0492}" => "\x{0413}",

 # 0493 CYRILLIC SMALL LETTER GHE WITH STROKE
 # ->   0433 CYRILLIC SMALL LETTER GHE
 "\x{0493}" => "\x{0433}",

 # 0494 CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
 # ->   0413 CYRILLIC CAPITAL LETTER GHE
 "\x{0494}" => "\x{0413}",

 # 0495 CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK
 # ->   0433 CYRILLIC SMALL LETTER GHE
 "\x{0495}" => "\x{0433}",

 # 0496 CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
 # ->   0416 CYRILLIC CAPITAL LETTER ZHE
 "\x{0496}" => "\x{0416}",

 # 0497 CYRILLIC SMALL LETTER ZHE WITH DESCENDER
 # ->   0436 CYRILLIC SMALL LETTER ZHE
 "\x{0497}" => "\x{0436}",

 # 0498 CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
 # ->   0417 CYRILLIC CAPITAL LETTER ZE
 "\x{0498}" => "\x{0417}",

 # 0499 CYRILLIC SMALL LETTER ZE WITH DESCENDER
 # ->   0437 CYRILLIC SMALL LETTER ZE
 "\x{0499}" => "\x{0437}",

 # 049A CYRILLIC CAPITAL LETTER KA WITH DESCENDER
 # ->   041A CYRILLIC CAPITAL LETTER KA
 "\x{049A}" => "\x{041A}",

 # 049B CYRILLIC SMALL LETTER KA WITH DESCENDER
 # ->   043A CYRILLIC SMALL LETTER KA
 "\x{049B}" => "\x{043A}",

 # 049C CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
 # ->   041A CYRILLIC CAPITAL LETTER KA
 "\x{049C}" => "\x{041A}",

 # 049D CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE
 # ->   043A CYRILLIC SMALL LETTER KA
 "\x{049D}" => "\x{043A}",

 # 049E CYRILLIC CAPITAL LETTER KA WITH STROKE
 # ->   041A CYRILLIC CAPITAL LETTER KA
 "\x{049E}" => "\x{041A}",

 # 049F CYRILLIC SMALL LETTER KA WITH STROKE
 # ->   043A CYRILLIC SMALL LETTER KA
 "\x{049F}" => "\x{043A}",

 # 04A2 CYRILLIC CAPITAL LETTER EN WITH DESCENDER
 # ->   041D CYRILLIC CAPITAL LETTER EN
 "\x{04A2}" => "\x{041D}",

 # 04A3 CYRILLIC SMALL LETTER EN WITH DESCENDER
 # ->   043D CYRILLIC SMALL LETTER EN
 "\x{04A3}" => "\x{043D}",

 # 04A6 CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
 # ->   041F CYRILLIC CAPITAL LETTER PE
 "\x{04A6}" => "\x{041F}",

 # 04A7 CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK
 # ->   043F CYRILLIC SMALL LETTER PE
 "\x{04A7}" => "\x{043F}",

 # 04AA CYRILLIC CAPITAL LETTER ES WITH DESCENDER
 # ->   0421 CYRILLIC CAPITAL LETTER ES
 "\x{04AA}" => "\x{0421}",

 # 04AB CYRILLIC SMALL LETTER ES WITH DESCENDER
 # ->   0441 CYRILLIC SMALL LETTER ES
 "\x{04AB}" => "\x{0441}",

 # 04AC CYRILLIC CAPITAL LETTER TE WITH DESCENDER
 # ->   0422 CYRILLIC CAPITAL LETTER TE
 "\x{04AC}" => "\x{0422}",

 # 04AD CYRILLIC SMALL LETTER TE WITH DESCENDER
 # ->   0442 CYRILLIC SMALL LETTER TE
 "\x{04AD}" => "\x{0442}",

 # 04B0 CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
 # ->   04AE CYRILLIC CAPITAL LETTER STRAIGHT U
 "\x{04B0}" => "\x{04AE}",

 # 04B1 CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
 # ->   04AF CYRILLIC SMALL LETTER STRAIGHT U
 "\x{04B1}" => "\x{04AF}",

 # 04B2 CYRILLIC CAPITAL LETTER HA WITH DESCENDER
 # ->   0425 CYRILLIC CAPITAL LETTER HA
 "\x{04B2}" => "\x{0425}",

 # 04B3 CYRILLIC SMALL LETTER HA WITH DESCENDER
 # ->   0445 CYRILLIC SMALL LETTER HA
 "\x{04B3}" => "\x{0445}",

 # 04B6 CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
 # ->   0427 CYRILLIC CAPITAL LETTER CHE
 "\x{04B6}" => "\x{0427}",

 # 04B7 CYRILLIC SMALL LETTER CHE WITH DESCENDER
 # ->   0447 CYRILLIC SMALL LETTER CHE
 "\x{04B7}" => "\x{0447}",

 # 04B8 CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
 # ->   0427 CYRILLIC CAPITAL LETTER CHE
 "\x{04B8}" => "\x{0427}",

 # 04B9 CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE
 # ->   0447 CYRILLIC SMALL LETTER CHE
 "\x{04B9}" => "\x{0447}",

 # 04BE CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
 # ->   04BC CYRILLIC CAPITAL LETTER ABKHASIAN CHE
 "\x{04BE}" => "\x{04BC}",

 # 04BF CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER
 # ->   04BD CYRILLIC SMALL LETTER ABKHASIAN CHE
 "\x{04BF}" => "\x{04BD}",

 # 04C1 CYRILLIC CAPITAL LETTER ZHE WITH BREVE
 # ->   0416 CYRILLIC CAPITAL LETTER ZHE
 "\x{04C1}" => "\x{0416}",

 # 04C2 CYRILLIC SMALL LETTER ZHE WITH BREVE
 # ->   0436 CYRILLIC SMALL LETTER ZHE
 "\x{04C2}" => "\x{0436}",

 # 04C3 CYRILLIC CAPITAL LETTER KA WITH HOOK
 # ->   041A CYRILLIC CAPITAL LETTER KA
 "\x{04C3}" => "\x{041A}",

 # 04C4 CYRILLIC SMALL LETTER KA WITH HOOK
 # ->   043A CYRILLIC SMALL LETTER KA
 "\x{04C4}" => "\x{043A}",

 # 04C5 CYRILLIC CAPITAL LETTER EL WITH TAIL
 # ->   041B CYRILLIC CAPITAL LETTER EL
 "\x{04C5}" => "\x{041B}",

 # 04C6 CYRILLIC SMALL LETTER EL WITH TAIL
 # ->   043B CYRILLIC SMALL LETTER EL
 "\x{04C6}" => "\x{043B}",

 # 04C7 CYRILLIC CAPITAL LETTER EN WITH HOOK
 # ->   041D CYRILLIC CAPITAL LETTER EN
 "\x{04C7}" => "\x{041D}",

 # 04C8 CYRILLIC SMALL LETTER EN WITH HOOK
 # ->   043D CYRILLIC SMALL LETTER EN
 "\x{04C8}" => "\x{043D}",

 # 04C9 CYRILLIC CAPITAL LETTER EN WITH TAIL
 # ->   041D CYRILLIC CAPITAL LETTER EN
 "\x{04C9}" => "\x{041D}",

 # 04CA CYRILLIC SMALL LETTER EN WITH TAIL
 # ->   043D CYRILLIC SMALL LETTER EN
 "\x{04CA}" => "\x{043D}",

 # 04CD CYRILLIC CAPITAL LETTER EM WITH TAIL
 # ->   041C CYRILLIC CAPITAL LETTER EM
 "\x{04CD}" => "\x{041C}",

 # 04CE CYRILLIC SMALL LETTER EM WITH TAIL
 # ->   043C CYRILLIC SMALL LETTER EM
 "\x{04CE}" => "\x{043C}",

 # 04D0 CYRILLIC CAPITAL LETTER A WITH BREVE
 # ->   0410 CYRILLIC CAPITAL LETTER A
 "\x{04D0}" => "\x{0410}",

 # 04D1 CYRILLIC SMALL LETTER A WITH BREVE
 # ->   0430 CYRILLIC SMALL LETTER A
 "\x{04D1}" => "\x{0430}",

 # 04D2 CYRILLIC CAPITAL LETTER A WITH DIAERESIS
 # ->   0410 CYRILLIC CAPITAL LETTER A
 "\x{04D2}" => "\x{0410}",

 # 04D3 CYRILLIC SMALL LETTER A WITH DIAERESIS
 # ->   0430 CYRILLIC SMALL LETTER A
 "\x{04D3}" => "\x{0430}",

 # 04D6 CYRILLIC CAPITAL LETTER IE WITH BREVE
 # ->   0415 CYRILLIC CAPITAL LETTER IE
 "\x{04D6}" => "\x{0415}",

 # 04D7 CYRILLIC SMALL LETTER IE WITH BREVE
 # ->   0435 CYRILLIC SMALL LETTER IE
 "\x{04D7}" => "\x{0435}",

 # 04DA CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
 # ->   04D8 CYRILLIC CAPITAL LETTER SCHWA
 "\x{04DA}" => "\x{04D8}",

 # 04DB CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS
 # ->   04D9 CYRILLIC SMALL LETTER SCHWA
 "\x{04DB}" => "\x{04D9}",

 # 04DC CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
 # ->   0416 CYRILLIC CAPITAL LETTER ZHE
 "\x{04DC}" => "\x{0416}",

 # 04DD CYRILLIC SMALL LETTER ZHE WITH DIAERESIS
 # ->   0436 CYRILLIC SMALL LETTER ZHE
 "\x{04DD}" => "\x{0436}",

 # 04DE CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
 # ->   0417 CYRILLIC CAPITAL LETTER ZE
 "\x{04DE}" => "\x{0417}",

 # 04DF CYRILLIC SMALL LETTER ZE WITH DIAERESIS
 # ->   0437 CYRILLIC SMALL LETTER ZE
 "\x{04DF}" => "\x{0437}",

 # 04E2 CYRILLIC CAPITAL LETTER I WITH MACRON
 # ->   0418 CYRILLIC CAPITAL LETTER I
 "\x{04E2}" => "\x{0418}",

 # 04E3 CYRILLIC SMALL LETTER I WITH MACRON
 # ->   0438 CYRILLIC SMALL LETTER I
 "\x{04E3}" => "\x{0438}",

 # 04E4 CYRILLIC CAPITAL LETTER I WITH DIAERESIS
 # ->   0418 CYRILLIC CAPITAL LETTER I
 "\x{04E4}" => "\x{0418}",

 # 04E5 CYRILLIC SMALL LETTER I WITH DIAERESIS
 # ->   0438 CYRILLIC SMALL LETTER I
 "\x{04E5}" => "\x{0438}",

 # 04E6 CYRILLIC CAPITAL LETTER O WITH DIAERESIS
 # ->   041E CYRILLIC CAPITAL LETTER O
 "\x{04E6}" => "\x{041E}",

 # 04E7 CYRILLIC SMALL LETTER O WITH DIAERESIS
 # ->   043E CYRILLIC SMALL LETTER O
 "\x{04E7}" => "\x{043E}",

 # 04EA CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
 # ->   04E8 CYRILLIC CAPITAL LETTER BARRED O
 "\x{04EA}" => "\x{04E8}",

 # 04EB CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS
 # ->   04E9 CYRILLIC SMALL LETTER BARRED O
 "\x{04EB}" => "\x{04E9}",

 # 04EC CYRILLIC CAPITAL LETTER E WITH DIAERESIS
 # ->   042D CYRILLIC CAPITAL LETTER E
 "\x{04EC}" => "\x{042D}",

 # 04ED CYRILLIC SMALL LETTER E WITH DIAERESIS
 # ->   044D CYRILLIC SMALL LETTER E
 "\x{04ED}" => "\x{044D}",

 # 04EE CYRILLIC CAPITAL LETTER U WITH MACRON
 # ->   0423 CYRILLIC CAPITAL LETTER U
 "\x{04EE}" => "\x{0423}",

 # 04EF CYRILLIC SMALL LETTER U WITH MACRON
 # ->   0443 CYRILLIC SMALL LETTER U
 "\x{04EF}" => "\x{0443}",

 # 04F0 CYRILLIC CAPITAL LETTER U WITH DIAERESIS
 # ->   0423 CYRILLIC CAPITAL LETTER U
 "\x{04F0}" => "\x{0423}",

 # 04F1 CYRILLIC SMALL LETTER U WITH DIAERESIS
 # ->   0443 CYRILLIC SMALL LETTER U
 "\x{04F1}" => "\x{0443}",

 # 04F2 CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
 # ->   0423 CYRILLIC CAPITAL LETTER U
 "\x{04F2}" => "\x{0423}",

 # 04F3 CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE
 # ->   0443 CYRILLIC SMALL LETTER U
 "\x{04F3}" => "\x{0443}",

 # 04F4 CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
 # ->   0427 CYRILLIC CAPITAL LETTER CHE
 "\x{04F4}" => "\x{0427}",

 # 04F5 CYRILLIC SMALL LETTER CHE WITH DIAERESIS
 # ->   0447 CYRILLIC SMALL LETTER CHE
 "\x{04F5}" => "\x{0447}",

 # 04F8 CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
 # ->   042B CYRILLIC CAPITAL LETTER YERU
 "\x{04F8}" => "\x{042B}",

 # 04F9 CYRILLIC SMALL LETTER YERU WITH DIAERESIS
 # ->   044B CYRILLIC SMALL LETTER YERU
 "\x{04F9}" => "\x{044B}",

 # 0587 ARMENIAN SMALL LIGATURE ECH YIWN
 # ->   0565 ARMENIAN SMALL LETTER ECH
 #    + 0582 ARMENIAN SMALL LETTER YIWN
 "\x{0587}" => "\x{0565}\x{0582}",

 # 0622 ARABIC LETTER ALEF WITH MADDA ABOVE
 # ->   0627 ARABIC LETTER ALEF
 "\x{0622}" => "\x{0627}",

 # 0623 ARABIC LETTER ALEF WITH HAMZA ABOVE
 # ->   0627 ARABIC LETTER ALEF
 "\x{0623}" => "\x{0627}",

 # 0624 ARABIC LETTER WAW WITH HAMZA ABOVE
 # ->   0648 ARABIC LETTER WAW
 "\x{0624}" => "\x{0648}",

 # 0625 ARABIC LETTER ALEF WITH HAMZA BELOW
 # ->   0627 ARABIC LETTER ALEF
 "\x{0625}" => "\x{0627}",

 # 0626 ARABIC LETTER YEH WITH HAMZA ABOVE
 # ->   064A ARABIC LETTER YEH
 "\x{0626}" => "\x{064A}",

 # 0672 ARABIC LETTER ALEF WITH WAVY HAMZA ABOVE
 # ->   0627 ARABIC LETTER ALEF
 "\x{0672}" => "\x{0627}",

 # 0673 ARABIC LETTER ALEF WITH WAVY HAMZA BELOW
 # ->   0627 ARABIC LETTER ALEF
 "\x{0673}" => "\x{0627}",

 # 0675 ARABIC LETTER HIGH HAMZA ALEF
 # ->   0627 ARABIC LETTER ALEF
 #    + 0674 ARABIC LETTER HIGH HAMZA
 "\x{0675}" => "\x{0627}\x{0674}",

 # 0676 ARABIC LETTER HIGH HAMZA WAW
 # ->   0648 ARABIC LETTER WAW
 #    + 0674 ARABIC LETTER HIGH HAMZA
 "\x{0676}" => "\x{0648}\x{0674}",

 # 0677 ARABIC LETTER U WITH HAMZA ABOVE
 # ->   06C7 ARABIC LETTER U
 #    + 0674 ARABIC LETTER HIGH HAMZA
 "\x{0677}" => "\x{06C7}\x{0674}",

 # 0678 ARABIC LETTER HIGH HAMZA YEH
 # ->   064A ARABIC LETTER YEH
 #    + 0674 ARABIC LETTER HIGH HAMZA
 "\x{0678}" => "\x{064A}\x{0674}",

 # 067C ARABIC LETTER TEH WITH RING
 # ->   062A ARABIC LETTER TEH
 "\x{067C}" => "\x{062A}",

 # 067D ARABIC LETTER TEH WITH THREE DOTS ABOVE DOWNWARDS
 # ->   062A ARABIC LETTER TEH
 "\x{067D}" => "\x{062A}",

 # 0681 ARABIC LETTER HAH WITH HAMZA ABOVE
 # ->   062D ARABIC LETTER HAH
 "\x{0681}" => "\x{062D}",

 # 0682 ARABIC LETTER HAH WITH TWO DOTS VERTICAL ABOVE
 # ->   062D ARABIC LETTER HAH
 "\x{0682}" => "\x{062D}",

 # 0685 ARABIC LETTER HAH WITH THREE DOTS ABOVE
 # ->   062D ARABIC LETTER HAH
 "\x{0685}" => "\x{062D}",

 # 0689 ARABIC LETTER DAL WITH RING
 # ->   062F ARABIC LETTER DAL
 "\x{0689}" => "\x{062F}",

 # 068A ARABIC LETTER DAL WITH DOT BELOW
 # ->   062F ARABIC LETTER DAL
 "\x{068A}" => "\x{062F}",

 # 068B ARABIC LETTER DAL WITH DOT BELOW AND SMALL TAH
 # ->   062F ARABIC LETTER DAL
 "\x{068B}" => "\x{062F}",

 # 068F ARABIC LETTER DAL WITH THREE DOTS ABOVE DOWNWARDS
 # ->   062F ARABIC LETTER DAL
 "\x{068F}" => "\x{062F}",

 # 0690 ARABIC LETTER DAL WITH FOUR DOTS ABOVE
 # ->   062F ARABIC LETTER DAL
 "\x{0690}" => "\x{062F}",

 # 0692 ARABIC LETTER REH WITH SMALL V
 # ->   0631 ARABIC LETTER REH
 "\x{0692}" => "\x{0631}",

 # 0693 ARABIC LETTER REH WITH RING
 # ->   0631 ARABIC LETTER REH
 "\x{0693}" => "\x{0631}",

 # 0694 ARABIC LETTER REH WITH DOT BELOW
 # ->   0631 ARABIC LETTER REH
 "\x{0694}" => "\x{0631}",

 # 0695 ARABIC LETTER REH WITH SMALL V BELOW
 # ->   0631 ARABIC LETTER REH
 "\x{0695}" => "\x{0631}",

 # 0696 ARABIC LETTER REH WITH DOT BELOW AND DOT ABOVE
 # ->   0631 ARABIC LETTER REH
 "\x{0696}" => "\x{0631}",

 # 0697 ARABIC LETTER REH WITH TWO DOTS ABOVE
 # ->   0631 ARABIC LETTER REH
 "\x{0697}" => "\x{0631}",

 # 0699 ARABIC LETTER REH WITH FOUR DOTS ABOVE
 # ->   0631 ARABIC LETTER REH
 "\x{0699}" => "\x{0631}",

 # 069A ARABIC LETTER SEEN WITH DOT BELOW AND DOT ABOVE
 # ->   0633 ARABIC LETTER SEEN
 "\x{069A}" => "\x{0633}",

 # 069B ARABIC LETTER SEEN WITH THREE DOTS BELOW
 # ->   0633 ARABIC LETTER SEEN
 "\x{069B}" => "\x{0633}",

 # 069C ARABIC LETTER SEEN WITH THREE DOTS BELOW AND THREE DOTS ABOVE
 # ->   0633 ARABIC LETTER SEEN
 "\x{069C}" => "\x{0633}",

 # 069D ARABIC LETTER SAD WITH TWO DOTS BELOW
 # ->   0635 ARABIC LETTER SAD
 "\x{069D}" => "\x{0635}",

 # 069E ARABIC LETTER SAD WITH THREE DOTS ABOVE
 # ->   0635 ARABIC LETTER SAD
 "\x{069E}" => "\x{0635}",

 # 069F ARABIC LETTER TAH WITH THREE DOTS ABOVE
 # ->   0637 ARABIC LETTER TAH
 "\x{069F}" => "\x{0637}",

 # 06A0 ARABIC LETTER AIN WITH THREE DOTS ABOVE
 # ->   0639 ARABIC LETTER AIN
 "\x{06A0}" => "\x{0639}",

 # 06A2 ARABIC LETTER FEH WITH DOT MOVED BELOW
 # ->   0641 ARABIC LETTER FEH
 "\x{06A2}" => "\x{0641}",

 # 06A3 ARABIC LETTER FEH WITH DOT BELOW
 # ->   0641 ARABIC LETTER FEH
 "\x{06A3}" => "\x{0641}",

 # 06A5 ARABIC LETTER FEH WITH THREE DOTS BELOW
 # ->   0641 ARABIC LETTER FEH
 "\x{06A5}" => "\x{0641}",

 # 06A7 ARABIC LETTER QAF WITH DOT ABOVE
 # ->   0642 ARABIC LETTER QAF
 "\x{06A7}" => "\x{0642}",

 # 06A8 ARABIC LETTER QAF WITH THREE DOTS ABOVE
 # ->   0642 ARABIC LETTER QAF
 "\x{06A8}" => "\x{0642}",

 # 06AB ARABIC LETTER KAF WITH RING
 # ->   0643 ARABIC LETTER KAF
 "\x{06AB}" => "\x{0643}",

 # 06AC ARABIC LETTER KAF WITH DOT ABOVE
 # ->   0643 ARABIC LETTER KAF
 "\x{06AC}" => "\x{0643}",

 # 06AE ARABIC LETTER KAF WITH THREE DOTS BELOW
 # ->   0643 ARABIC LETTER KAF
 "\x{06AE}" => "\x{0643}",

 # 06B0 ARABIC LETTER GAF WITH RING
 # ->   06AF ARABIC LETTER GAF
 "\x{06B0}" => "\x{06AF}",

 # 06B2 ARABIC LETTER GAF WITH TWO DOTS BELOW
 # ->   06AF ARABIC LETTER GAF
 "\x{06B2}" => "\x{06AF}",

 # 06B4 ARABIC LETTER GAF WITH THREE DOTS ABOVE
 # ->   06AF ARABIC LETTER GAF
 "\x{06B4}" => "\x{06AF}",

 # 06B5 ARABIC LETTER LAM WITH SMALL V
 # ->   0644 ARABIC LETTER LAM
 "\x{06B5}" => "\x{0644}",

 # 06B6 ARABIC LETTER LAM WITH DOT ABOVE
 # ->   0644 ARABIC LETTER LAM
 "\x{06B6}" => "\x{0644}",

 # 06B7 ARABIC LETTER LAM WITH THREE DOTS ABOVE
 # ->   0644 ARABIC LETTER LAM
 "\x{06B7}" => "\x{0644}",

 # 06B8 ARABIC LETTER LAM WITH THREE DOTS BELOW
 # ->   0644 ARABIC LETTER LAM
 "\x{06B8}" => "\x{0644}",

 # 06B9 ARABIC LETTER NOON WITH DOT BELOW
 # ->   0646 ARABIC LETTER NOON
 "\x{06B9}" => "\x{0646}",

 # 06BC ARABIC LETTER NOON WITH RING
 # ->   0646 ARABIC LETTER NOON
 "\x{06BC}" => "\x{0646}",

 # 06BD ARABIC LETTER NOON WITH THREE DOTS ABOVE
 # ->   0646 ARABIC LETTER NOON
 "\x{06BD}" => "\x{0646}",

 # 06BF ARABIC LETTER TCHEH WITH DOT ABOVE
 # ->   0686 ARABIC LETTER TCHEH
 "\x{06BF}" => "\x{0686}",

 # 06C0 ARABIC LETTER HEH WITH YEH ABOVE
 # ->   06D5 ARABIC LETTER AE
 "\x{06C0}" => "\x{06D5}",

 # 06C2 ARABIC LETTER HEH GOAL WITH HAMZA ABOVE
 # ->   06C1 ARABIC LETTER HEH GOAL
 "\x{06C2}" => "\x{06C1}",

 # 06C4 ARABIC LETTER WAW WITH RING
 # ->   0648 ARABIC LETTER WAW
 "\x{06C4}" => "\x{0648}",

 # 06CA ARABIC LETTER WAW WITH TWO DOTS ABOVE
 # ->   0648 ARABIC LETTER WAW
 "\x{06CA}" => "\x{0648}",

 # 06CD ARABIC LETTER YEH WITH TAIL
 # ->   064A ARABIC LETTER YEH
 "\x{06CD}" => "\x{064A}",

 # 06CE ARABIC LETTER YEH WITH SMALL V
 # ->   064A ARABIC LETTER YEH
 "\x{06CE}" => "\x{064A}",

 # 06CF ARABIC LETTER WAW WITH DOT ABOVE
 # ->   0648 ARABIC LETTER WAW
 "\x{06CF}" => "\x{0648}",

 # 06D1 ARABIC LETTER YEH WITH THREE DOTS BELOW
 # ->   064A ARABIC LETTER YEH
 "\x{06D1}" => "\x{064A}",

 # 06D3 ARABIC LETTER YEH BARREE WITH HAMZA ABOVE
 # ->   06D2 ARABIC LETTER YEH BARREE
 "\x{06D3}" => "\x{06D2}",

 # 06EE ARABIC LETTER DAL WITH INVERTED V
 # ->   062F ARABIC LETTER DAL
 "\x{06EE}" => "\x{062F}",

 # 06EF ARABIC LETTER REH WITH INVERTED V
 # ->   0631 ARABIC LETTER REH
 "\x{06EF}" => "\x{0631}",

 # 06FA ARABIC LETTER SHEEN WITH DOT BELOW
 # ->   0634 ARABIC LETTER SHEEN
 "\x{06FA}" => "\x{0634}",

 # 06FB ARABIC LETTER DAD WITH DOT BELOW
 # ->   0636 ARABIC LETTER DAD
 "\x{06FB}" => "\x{0636}",

 # 06FC ARABIC LETTER GHAIN WITH DOT BELOW
 # ->   063A ARABIC LETTER GHAIN
 "\x{06FC}" => "\x{063A}",

 # 06FF ARABIC LETTER HEH WITH INVERTED V
 # ->   0647 ARABIC LETTER HEH
 "\x{06FF}" => "\x{0647}",

 # 0929 DEVANAGARI LETTER NNNA
 # ->   0928 DEVANAGARI LETTER NA
 "\x{0929}" => "\x{0928}",

 # 0931 DEVANAGARI LETTER RRA
 # ->   0930 DEVANAGARI LETTER RA
 "\x{0931}" => "\x{0930}",

 # 0934 DEVANAGARI LETTER LLLA
 # ->   0933 DEVANAGARI LETTER LLA
 "\x{0934}" => "\x{0933}",

 # 0958 DEVANAGARI LETTER QA
 # ->   0915 DEVANAGARI LETTER KA
 "\x{0958}" => "\x{0915}",

 # 0959 DEVANAGARI LETTER KHHA
 # ->   0916 DEVANAGARI LETTER KHA
 "\x{0959}" => "\x{0916}",

 # 095A DEVANAGARI LETTER GHHA
 # ->   0917 DEVANAGARI LETTER GA
 "\x{095A}" => "\x{0917}",

 # 095B DEVANAGARI LETTER ZA
 # ->   091C DEVANAGARI LETTER JA
 "\x{095B}" => "\x{091C}",

 # 095C DEVANAGARI LETTER DDDHA
 # ->   0921 DEVANAGARI LETTER DDA
 "\x{095C}" => "\x{0921}",

 # 095D DEVANAGARI LETTER RHA
 # ->   0922 DEVANAGARI LETTER DDHA
 "\x{095D}" => "\x{0922}",

 # 095E DEVANAGARI LETTER FA
 # ->   092B DEVANAGARI LETTER PHA
 "\x{095E}" => "\x{092B}",

 # 095F DEVANAGARI LETTER YYA
 # ->   092F DEVANAGARI LETTER YA
 "\x{095F}" => "\x{092F}",

 # 09DC BENGALI LETTER RRA
 # ->   09A1 BENGALI LETTER DDA
 "\x{09DC}" => "\x{09A1}",

 # 09DD BENGALI LETTER RHA
 # ->   09A2 BENGALI LETTER DDHA
 "\x{09DD}" => "\x{09A2}",

 # 09DF BENGALI LETTER YYA
 # ->   09AF BENGALI LETTER YA
 "\x{09DF}" => "\x{09AF}",

 # 09F0 BENGALI LETTER RA WITH MIDDLE DIAGONAL
 # ->   09B0 BENGALI LETTER RA
 "\x{09F0}" => "\x{09B0}",

 # 09F1 BENGALI LETTER RA WITH LOWER DIAGONAL
 # ->   09B0 BENGALI LETTER RA
 "\x{09F1}" => "\x{09B0}",

 # 0A33 GURMUKHI LETTER LLA
 # ->   0A32 GURMUKHI LETTER LA
 "\x{0A33}" => "\x{0A32}",

 # 0A36 GURMUKHI LETTER SHA
 # ->   0A38 GURMUKHI LETTER SA
 "\x{0A36}" => "\x{0A38}",

 # 0A59 GURMUKHI LETTER KHHA
 # ->   0A16 GURMUKHI LETTER KHA
 "\x{0A59}" => "\x{0A16}",

 # 0A5A GURMUKHI LETTER GHHA
 # ->   0A17 GURMUKHI LETTER GA
 "\x{0A5A}" => "\x{0A17}",

 # 0A5B GURMUKHI LETTER ZA
 # ->   0A1C GURMUKHI LETTER JA
 "\x{0A5B}" => "\x{0A1C}",

 # 0A5E GURMUKHI LETTER FA
 # ->   0A2B GURMUKHI LETTER PHA
 "\x{0A5E}" => "\x{0A2B}",

 # 0B5C ORIYA LETTER RRA
 # ->   0B21 ORIYA LETTER DDA
 "\x{0B5C}" => "\x{0B21}",

 # 0B5D ORIYA LETTER RHA
 # ->   0B22 ORIYA LETTER DDHA
 "\x{0B5D}" => "\x{0B22}",

 # 0B94 TAMIL LETTER AU
 # ->   0B92 TAMIL LETTER O
 "\x{0B94}" => "\x{0B92}",

 # 0E33 THAI CHARACTER SARA AM
 # ->   0E32 THAI CHARACTER SARA AA
 "\x{0E33}" => "\x{0E32}",

 # 0EB3 LAO VOWEL SIGN AM
 # ->   0EB2 LAO VOWEL SIGN AA
 "\x{0EB3}" => "\x{0EB2}",

 # 0EDC LAO HO NO
 # ->   0EAB LAO LETTER HO SUNG
 #    + 0E99 LAO LETTER NO
 "\x{0EDC}" => "\x{0EAB}\x{0E99}",

 # 0EDD LAO HO MO
 # ->   0EAB LAO LETTER HO SUNG
 #    + 0EA1 LAO LETTER MO
 "\x{0EDD}" => "\x{0EAB}\x{0EA1}",

 # 0F0C TIBETAN MARK DELIMITER TSHEG BSTAR
 # ->   0F0B TIBETAN MARK INTERSYLLABIC TSHEG
 "\x{0F0C}" => "\x{0F0B}",

 # 0F43 TIBETAN LETTER GHA
 # ->   0F42 TIBETAN LETTER GA
 "\x{0F43}" => "\x{0F42}",

 # 0F4D TIBETAN LETTER DDHA
 # ->   0F4C TIBETAN LETTER DDA
 "\x{0F4D}" => "\x{0F4C}",

 # 0F52 TIBETAN LETTER DHA
 # ->   0F51 TIBETAN LETTER DA
 "\x{0F52}" => "\x{0F51}",

 # 0F57 TIBETAN LETTER BHA
 # ->   0F56 TIBETAN LETTER BA
 "\x{0F57}" => "\x{0F56}",

 # 0F5C TIBETAN LETTER DZHA
 # ->   0F5B TIBETAN LETTER DZA
 "\x{0F5C}" => "\x{0F5B}",

 # 0F69 TIBETAN LETTER KSSA
 # ->   0F40 TIBETAN LETTER KA
 "\x{0F69}" => "\x{0F40}",

 # 1026 MYANMAR LETTER UU
 # ->   1025 MYANMAR LETTER U
 "\x{1026}" => "\x{1025}",

 # 1D0C LATIN LETTER SMALL CAPITAL L WITH STROKE
 # ->   029F LATIN LETTER SMALL CAPITAL L
 "\x{1D0C}" => "\x{029F}",

 # 1D13 LATIN SMALL LETTER SIDEWAYS O WITH STROKE
 # ->   1D11 LATIN SMALL LETTER SIDEWAYS O
 "\x{1D13}" => "\x{1D11}",

 # 1D2C MODIFIER LETTER CAPITAL A
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1D2C}" => "A",

 # 1D2D MODIFIER LETTER CAPITAL AE
 # ->   0041 LATIN CAPITAL LETTER A
 #    + 0045 LATIN CAPITAL LETTER E
 "\x{1D2D}" => "AE",

 # 1D2E MODIFIER LETTER CAPITAL B
 # ->   0042 LATIN CAPITAL LETTER B
 "\x{1D2E}" => "B",

 # 1D30 MODIFIER LETTER CAPITAL D
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{1D30}" => "D",

 # 1D31 MODIFIER LETTER CAPITAL E
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1D31}" => "E",

 # 1D32 MODIFIER LETTER CAPITAL REVERSED E
 # ->   018E LATIN CAPITAL LETTER REVERSED E
 "\x{1D32}" => "\x{018E}",

 # 1D33 MODIFIER LETTER CAPITAL G
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{1D33}" => "G",

 # 1D34 MODIFIER LETTER CAPITAL H
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{1D34}" => "H",

 # 1D35 MODIFIER LETTER CAPITAL I
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{1D35}" => "I",

 # 1D36 MODIFIER LETTER CAPITAL J
 # ->   004A LATIN CAPITAL LETTER J
 "\x{1D36}" => "J",

 # 1D37 MODIFIER LETTER CAPITAL K
 # ->   004B LATIN CAPITAL LETTER K
 "\x{1D37}" => "K",

 # 1D38 MODIFIER LETTER CAPITAL L
 # ->   004C LATIN CAPITAL LETTER L
 "\x{1D38}" => "L",

 # 1D39 MODIFIER LETTER CAPITAL M
 # ->   004D LATIN CAPITAL LETTER M
 "\x{1D39}" => "M",

 # 1D3A MODIFIER LETTER CAPITAL N
 # ->   004E LATIN CAPITAL LETTER N
 "\x{1D3A}" => "N",

 # 1D3C MODIFIER LETTER CAPITAL O
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1D3C}" => "O",

 # 1D3D MODIFIER LETTER CAPITAL OU
 # ->   0222 LATIN CAPITAL LETTER OU
 "\x{1D3D}" => "\x{0222}",

 # 1D3E MODIFIER LETTER CAPITAL P
 # ->   0050 LATIN CAPITAL LETTER P
 "\x{1D3E}" => "P",

 # 1D3F MODIFIER LETTER CAPITAL R
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{1D3F}" => "R",

 # 1D40 MODIFIER LETTER CAPITAL T
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{1D40}" => "T",

 # 1D41 MODIFIER LETTER CAPITAL U
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1D41}" => "U",

 # 1D42 MODIFIER LETTER CAPITAL W
 # ->   0057 LATIN CAPITAL LETTER W
 "\x{1D42}" => "W",

 # 1D43 MODIFIER LETTER SMALL A
 # ->   0061 LATIN SMALL LETTER A
 "\x{1D43}" => "a",

 # 1D44 MODIFIER LETTER SMALL TURNED A
 # ->   0250 LATIN SMALL LETTER TURNED A
 "\x{1D44}" => "\x{0250}",

 # 1D45 MODIFIER LETTER SMALL ALPHA
 # ->   0251 LATIN SMALL LETTER ALPHA
 "\x{1D45}" => "\x{0251}",

 # 1D46 MODIFIER LETTER SMALL TURNED AE
 # ->   1D02 LATIN SMALL LETTER TURNED AE
 "\x{1D46}" => "\x{1D02}",

 # 1D47 MODIFIER LETTER SMALL B
 # ->   0062 LATIN SMALL LETTER B
 "\x{1D47}" => "b",

 # 1D48 MODIFIER LETTER SMALL D
 # ->   0064 LATIN SMALL LETTER D
 "\x{1D48}" => "d",

 # 1D49 MODIFIER LETTER SMALL E
 # ->   0065 LATIN SMALL LETTER E
 "\x{1D49}" => "e",

 # 1D4A MODIFIER LETTER SMALL SCHWA
 # ->   0259 LATIN SMALL LETTER SCHWA
 "\x{1D4A}" => "\x{0259}",

 # 1D4B MODIFIER LETTER SMALL OPEN E
 # ->   025B LATIN SMALL LETTER OPEN E
 "\x{1D4B}" => "\x{025B}",

 # 1D4C MODIFIER LETTER SMALL TURNED OPEN E
 # ->   025C LATIN SMALL LETTER REVERSED OPEN E
 "\x{1D4C}" => "\x{025C}",

 # 1D4D MODIFIER LETTER SMALL G
 # ->   0067 LATIN SMALL LETTER G
 "\x{1D4D}" => "g",

 # 1D4F MODIFIER LETTER SMALL K
 # ->   006B LATIN SMALL LETTER K
 "\x{1D4F}" => "k",

 # 1D50 MODIFIER LETTER SMALL M
 # ->   006D LATIN SMALL LETTER M
 "\x{1D50}" => "m",

 # 1D51 MODIFIER LETTER SMALL ENG
 # ->   014B LATIN SMALL LETTER ENG
 "\x{1D51}" => "\x{014B}",

 # 1D52 MODIFIER LETTER SMALL O
 # ->   006F LATIN SMALL LETTER O
 "\x{1D52}" => "o",

 # 1D53 MODIFIER LETTER SMALL OPEN O
 # ->   0254 LATIN SMALL LETTER OPEN O
 "\x{1D53}" => "\x{0254}",

 # 1D54 MODIFIER LETTER SMALL TOP HALF O
 # ->   1D16 LATIN SMALL LETTER TOP HALF O
 "\x{1D54}" => "\x{1D16}",

 # 1D55 MODIFIER LETTER SMALL BOTTOM HALF O
 # ->   1D17 LATIN SMALL LETTER BOTTOM HALF O
 "\x{1D55}" => "\x{1D17}",

 # 1D56 MODIFIER LETTER SMALL P
 # ->   0070 LATIN SMALL LETTER P
 "\x{1D56}" => "p",

 # 1D57 MODIFIER LETTER SMALL T
 # ->   0074 LATIN SMALL LETTER T
 "\x{1D57}" => "t",

 # 1D58 MODIFIER LETTER SMALL U
 # ->   0075 LATIN SMALL LETTER U
 "\x{1D58}" => "u",

 # 1D59 MODIFIER LETTER SMALL SIDEWAYS U
 # ->   1D1D LATIN SMALL LETTER SIDEWAYS U
 "\x{1D59}" => "\x{1D1D}",

 # 1D5A MODIFIER LETTER SMALL TURNED M
 # ->   026F LATIN SMALL LETTER TURNED M
 "\x{1D5A}" => "\x{026F}",

 # 1D5B MODIFIER LETTER SMALL V
 # ->   0076 LATIN SMALL LETTER V
 "\x{1D5B}" => "v",

 # 1D5C MODIFIER LETTER SMALL AIN
 # ->   1D25 LATIN LETTER AIN
 "\x{1D5C}" => "\x{1D25}",

 # 1D5D MODIFIER LETTER SMALL BETA
 # ->   03B2 GREEK SMALL LETTER BETA
 "\x{1D5D}" => "\x{03B2}",

 # 1D5E MODIFIER LETTER SMALL GREEK GAMMA
 # ->   03B3 GREEK SMALL LETTER GAMMA
 "\x{1D5E}" => "\x{03B3}",

 # 1D5F MODIFIER LETTER SMALL DELTA
 # ->   03B4 GREEK SMALL LETTER DELTA
 "\x{1D5F}" => "\x{03B4}",

 # 1D60 MODIFIER LETTER SMALL GREEK PHI
 # ->   03C6 GREEK SMALL LETTER PHI
 "\x{1D60}" => "\x{03C6}",

 # 1D61 MODIFIER LETTER SMALL CHI
 # ->   03C7 GREEK SMALL LETTER CHI
 "\x{1D61}" => "\x{03C7}",

 # 1D62 LATIN SUBSCRIPT SMALL LETTER I
 # ->   0069 LATIN SMALL LETTER I
 "\x{1D62}" => "i",

 # 1D63 LATIN SUBSCRIPT SMALL LETTER R
 # ->   0072 LATIN SMALL LETTER R
 "\x{1D63}" => "r",

 # 1D64 LATIN SUBSCRIPT SMALL LETTER U
 # ->   0075 LATIN SMALL LETTER U
 "\x{1D64}" => "u",

 # 1D65 LATIN SUBSCRIPT SMALL LETTER V
 # ->   0076 LATIN SMALL LETTER V
 "\x{1D65}" => "v",

 # 1D66 GREEK SUBSCRIPT SMALL LETTER BETA
 # ->   03B2 GREEK SMALL LETTER BETA
 "\x{1D66}" => "\x{03B2}",

 # 1D67 GREEK SUBSCRIPT SMALL LETTER GAMMA
 # ->   03B3 GREEK SMALL LETTER GAMMA
 "\x{1D67}" => "\x{03B3}",

 # 1D68 GREEK SUBSCRIPT SMALL LETTER RHO
 # ->   03C1 GREEK SMALL LETTER RHO
 "\x{1D68}" => "\x{03C1}",

 # 1D69 GREEK SUBSCRIPT SMALL LETTER PHI
 # ->   03C6 GREEK SMALL LETTER PHI
 "\x{1D69}" => "\x{03C6}",

 # 1D6A GREEK SUBSCRIPT SMALL LETTER CHI
 # ->   03C7 GREEK SMALL LETTER CHI
 "\x{1D6A}" => "\x{03C7}",

 # 1E00 LATIN CAPITAL LETTER A WITH RING BELOW
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1E00}" => "A",

 # 1E01 LATIN SMALL LETTER A WITH RING BELOW
 # ->   0061 LATIN SMALL LETTER A
 "\x{1E01}" => "a",

 # 1E02 LATIN CAPITAL LETTER B WITH DOT ABOVE
 # ->   0042 LATIN CAPITAL LETTER B
 "\x{1E02}" => "B",

 # 1E03 LATIN SMALL LETTER B WITH DOT ABOVE
 # ->   0062 LATIN SMALL LETTER B
 "\x{1E03}" => "b",

 # 1E04 LATIN CAPITAL LETTER B WITH DOT BELOW
 # ->   0042 LATIN CAPITAL LETTER B
 "\x{1E04}" => "B",

 # 1E05 LATIN SMALL LETTER B WITH DOT BELOW
 # ->   0062 LATIN SMALL LETTER B
 "\x{1E05}" => "b",

 # 1E06 LATIN CAPITAL LETTER B WITH LINE BELOW
 # ->   0042 LATIN CAPITAL LETTER B
 "\x{1E06}" => "B",

 # 1E07 LATIN SMALL LETTER B WITH LINE BELOW
 # ->   0062 LATIN SMALL LETTER B
 "\x{1E07}" => "b",

 # 1E08 LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
 # ->   0043 LATIN CAPITAL LETTER C
 "\x{1E08}" => "C",

 # 1E09 LATIN SMALL LETTER C WITH CEDILLA AND ACUTE
 # ->   0063 LATIN SMALL LETTER C
 "\x{1E09}" => "c",

 # 1E0A LATIN CAPITAL LETTER D WITH DOT ABOVE
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{1E0A}" => "D",

 # 1E0B LATIN SMALL LETTER D WITH DOT ABOVE
 # ->   0064 LATIN SMALL LETTER D
 "\x{1E0B}" => "d",

 # 1E0C LATIN CAPITAL LETTER D WITH DOT BELOW
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{1E0C}" => "D",

 # 1E0D LATIN SMALL LETTER D WITH DOT BELOW
 # ->   0064 LATIN SMALL LETTER D
 "\x{1E0D}" => "d",

 # 1E0E LATIN CAPITAL LETTER D WITH LINE BELOW
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{1E0E}" => "D",

 # 1E0F LATIN SMALL LETTER D WITH LINE BELOW
 # ->   0064 LATIN SMALL LETTER D
 "\x{1E0F}" => "d",

 # 1E10 LATIN CAPITAL LETTER D WITH CEDILLA
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{1E10}" => "D",

 # 1E11 LATIN SMALL LETTER D WITH CEDILLA
 # ->   0064 LATIN SMALL LETTER D
 "\x{1E11}" => "d",

 # 1E12 LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{1E12}" => "D",

 # 1E13 LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW
 # ->   0064 LATIN SMALL LETTER D
 "\x{1E13}" => "d",

 # 1E14 LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1E14}" => "E",

 # 1E15 LATIN SMALL LETTER E WITH MACRON AND GRAVE
 # ->   0065 LATIN SMALL LETTER E
 "\x{1E15}" => "e",

 # 1E16 LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1E16}" => "E",

 # 1E17 LATIN SMALL LETTER E WITH MACRON AND ACUTE
 # ->   0065 LATIN SMALL LETTER E
 "\x{1E17}" => "e",

 # 1E18 LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1E18}" => "E",

 # 1E19 LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW
 # ->   0065 LATIN SMALL LETTER E
 "\x{1E19}" => "e",

 # 1E1A LATIN CAPITAL LETTER E WITH TILDE BELOW
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1E1A}" => "E",

 # 1E1B LATIN SMALL LETTER E WITH TILDE BELOW
 # ->   0065 LATIN SMALL LETTER E
 "\x{1E1B}" => "e",

 # 1E1C LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1E1C}" => "E",

 # 1E1D LATIN SMALL LETTER E WITH CEDILLA AND BREVE
 # ->   0065 LATIN SMALL LETTER E
 "\x{1E1D}" => "e",

 # 1E1E LATIN CAPITAL LETTER F WITH DOT ABOVE
 # ->   0046 LATIN CAPITAL LETTER F
 "\x{1E1E}" => "F",

 # 1E1F LATIN SMALL LETTER F WITH DOT ABOVE
 # ->   0066 LATIN SMALL LETTER F
 "\x{1E1F}" => "f",

 # 1E20 LATIN CAPITAL LETTER G WITH MACRON
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{1E20}" => "G",

 # 1E21 LATIN SMALL LETTER G WITH MACRON
 # ->   0067 LATIN SMALL LETTER G
 "\x{1E21}" => "g",

 # 1E22 LATIN CAPITAL LETTER H WITH DOT ABOVE
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{1E22}" => "H",

 # 1E23 LATIN SMALL LETTER H WITH DOT ABOVE
 # ->   0068 LATIN SMALL LETTER H
 "\x{1E23}" => "h",

 # 1E24 LATIN CAPITAL LETTER H WITH DOT BELOW
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{1E24}" => "H",

 # 1E25 LATIN SMALL LETTER H WITH DOT BELOW
 # ->   0068 LATIN SMALL LETTER H
 "\x{1E25}" => "h",

 # 1E26 LATIN CAPITAL LETTER H WITH DIAERESIS
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{1E26}" => "H",

 # 1E27 LATIN SMALL LETTER H WITH DIAERESIS
 # ->   0068 LATIN SMALL LETTER H
 "\x{1E27}" => "h",

 # 1E28 LATIN CAPITAL LETTER H WITH CEDILLA
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{1E28}" => "H",

 # 1E29 LATIN SMALL LETTER H WITH CEDILLA
 # ->   0068 LATIN SMALL LETTER H
 "\x{1E29}" => "h",

 # 1E2A LATIN CAPITAL LETTER H WITH BREVE BELOW
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{1E2A}" => "H",

 # 1E2B LATIN SMALL LETTER H WITH BREVE BELOW
 # ->   0068 LATIN SMALL LETTER H
 "\x{1E2B}" => "h",

 # 1E2C LATIN CAPITAL LETTER I WITH TILDE BELOW
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{1E2C}" => "I",

 # 1E2D LATIN SMALL LETTER I WITH TILDE BELOW
 # ->   0069 LATIN SMALL LETTER I
 "\x{1E2D}" => "i",

 # 1E2E LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{1E2E}" => "I",

 # 1E2F LATIN SMALL LETTER I WITH DIAERESIS AND ACUTE
 # ->   0069 LATIN SMALL LETTER I
 "\x{1E2F}" => "i",

 # 1E30 LATIN CAPITAL LETTER K WITH ACUTE
 # ->   004B LATIN CAPITAL LETTER K
 "\x{1E30}" => "K",

 # 1E31 LATIN SMALL LETTER K WITH ACUTE
 # ->   006B LATIN SMALL LETTER K
 "\x{1E31}" => "k",

 # 1E32 LATIN CAPITAL LETTER K WITH DOT BELOW
 # ->   004B LATIN CAPITAL LETTER K
 "\x{1E32}" => "K",

 # 1E33 LATIN SMALL LETTER K WITH DOT BELOW
 # ->   006B LATIN SMALL LETTER K
 "\x{1E33}" => "k",

 # 1E34 LATIN CAPITAL LETTER K WITH LINE BELOW
 # ->   004B LATIN CAPITAL LETTER K
 "\x{1E34}" => "K",

 # 1E35 LATIN SMALL LETTER K WITH LINE BELOW
 # ->   006B LATIN SMALL LETTER K
 "\x{1E35}" => "k",

 # 1E36 LATIN CAPITAL LETTER L WITH DOT BELOW
 # ->   004C LATIN CAPITAL LETTER L
 "\x{1E36}" => "L",

 # 1E37 LATIN SMALL LETTER L WITH DOT BELOW
 # ->   006C LATIN SMALL LETTER L
 "\x{1E37}" => "l",

 # 1E38 LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
 # ->   004C LATIN CAPITAL LETTER L
 "\x{1E38}" => "L",

 # 1E39 LATIN SMALL LETTER L WITH DOT BELOW AND MACRON
 # ->   006C LATIN SMALL LETTER L
 "\x{1E39}" => "l",

 # 1E3A LATIN CAPITAL LETTER L WITH LINE BELOW
 # ->   004C LATIN CAPITAL LETTER L
 "\x{1E3A}" => "L",

 # 1E3B LATIN SMALL LETTER L WITH LINE BELOW
 # ->   006C LATIN SMALL LETTER L
 "\x{1E3B}" => "l",

 # 1E3C LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
 # ->   004C LATIN CAPITAL LETTER L
 "\x{1E3C}" => "L",

 # 1E3D LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW
 # ->   006C LATIN SMALL LETTER L
 "\x{1E3D}" => "l",

 # 1E3E LATIN CAPITAL LETTER M WITH ACUTE
 # ->   004D LATIN CAPITAL LETTER M
 "\x{1E3E}" => "M",

 # 1E3F LATIN SMALL LETTER M WITH ACUTE
 # ->   006D LATIN SMALL LETTER M
 "\x{1E3F}" => "m",

 # 1E40 LATIN CAPITAL LETTER M WITH DOT ABOVE
 # ->   004D LATIN CAPITAL LETTER M
 "\x{1E40}" => "M",

 # 1E41 LATIN SMALL LETTER M WITH DOT ABOVE
 # ->   006D LATIN SMALL LETTER M
 "\x{1E41}" => "m",

 # 1E42 LATIN CAPITAL LETTER M WITH DOT BELOW
 # ->   004D LATIN CAPITAL LETTER M
 "\x{1E42}" => "M",

 # 1E43 LATIN SMALL LETTER M WITH DOT BELOW
 # ->   006D LATIN SMALL LETTER M
 "\x{1E43}" => "m",

 # 1E44 LATIN CAPITAL LETTER N WITH DOT ABOVE
 # ->   004E LATIN CAPITAL LETTER N
 "\x{1E44}" => "N",

 # 1E45 LATIN SMALL LETTER N WITH DOT ABOVE
 # ->   006E LATIN SMALL LETTER N
 "\x{1E45}" => "n",

 # 1E46 LATIN CAPITAL LETTER N WITH DOT BELOW
 # ->   004E LATIN CAPITAL LETTER N
 "\x{1E46}" => "N",

 # 1E47 LATIN SMALL LETTER N WITH DOT BELOW
 # ->   006E LATIN SMALL LETTER N
 "\x{1E47}" => "n",

 # 1E48 LATIN CAPITAL LETTER N WITH LINE BELOW
 # ->   004E LATIN CAPITAL LETTER N
 "\x{1E48}" => "N",

 # 1E49 LATIN SMALL LETTER N WITH LINE BELOW
 # ->   006E LATIN SMALL LETTER N
 "\x{1E49}" => "n",

 # 1E4A LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
 # ->   004E LATIN CAPITAL LETTER N
 "\x{1E4A}" => "N",

 # 1E4B LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW
 # ->   006E LATIN SMALL LETTER N
 "\x{1E4B}" => "n",

 # 1E4C LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1E4C}" => "O",

 # 1E4D LATIN SMALL LETTER O WITH TILDE AND ACUTE
 # ->   006F LATIN SMALL LETTER O
 "\x{1E4D}" => "o",

 # 1E4E LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1E4E}" => "O",

 # 1E4F LATIN SMALL LETTER O WITH TILDE AND DIAERESIS
 # ->   006F LATIN SMALL LETTER O
 "\x{1E4F}" => "o",

 # 1E50 LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1E50}" => "O",

 # 1E51 LATIN SMALL LETTER O WITH MACRON AND GRAVE
 # ->   006F LATIN SMALL LETTER O
 "\x{1E51}" => "o",

 # 1E52 LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1E52}" => "O",

 # 1E53 LATIN SMALL LETTER O WITH MACRON AND ACUTE
 # ->   006F LATIN SMALL LETTER O
 "\x{1E53}" => "o",

 # 1E54 LATIN CAPITAL LETTER P WITH ACUTE
 # ->   0050 LATIN CAPITAL LETTER P
 "\x{1E54}" => "P",

 # 1E55 LATIN SMALL LETTER P WITH ACUTE
 # ->   0070 LATIN SMALL LETTER P
 "\x{1E55}" => "p",

 # 1E56 LATIN CAPITAL LETTER P WITH DOT ABOVE
 # ->   0050 LATIN CAPITAL LETTER P
 "\x{1E56}" => "P",

 # 1E57 LATIN SMALL LETTER P WITH DOT ABOVE
 # ->   0070 LATIN SMALL LETTER P
 "\x{1E57}" => "p",

 # 1E58 LATIN CAPITAL LETTER R WITH DOT ABOVE
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{1E58}" => "R",

 # 1E59 LATIN SMALL LETTER R WITH DOT ABOVE
 # ->   0072 LATIN SMALL LETTER R
 "\x{1E59}" => "r",

 # 1E5A LATIN CAPITAL LETTER R WITH DOT BELOW
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{1E5A}" => "R",

 # 1E5B LATIN SMALL LETTER R WITH DOT BELOW
 # ->   0072 LATIN SMALL LETTER R
 "\x{1E5B}" => "r",

 # 1E5C LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{1E5C}" => "R",

 # 1E5D LATIN SMALL LETTER R WITH DOT BELOW AND MACRON
 # ->   0072 LATIN SMALL LETTER R
 "\x{1E5D}" => "r",

 # 1E5E LATIN CAPITAL LETTER R WITH LINE BELOW
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{1E5E}" => "R",

 # 1E5F LATIN SMALL LETTER R WITH LINE BELOW
 # ->   0072 LATIN SMALL LETTER R
 "\x{1E5F}" => "r",

 # 1E60 LATIN CAPITAL LETTER S WITH DOT ABOVE
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{1E60}" => "S",

 # 1E61 LATIN SMALL LETTER S WITH DOT ABOVE
 # ->   0073 LATIN SMALL LETTER S
 "\x{1E61}" => "s",

 # 1E62 LATIN CAPITAL LETTER S WITH DOT BELOW
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{1E62}" => "S",

 # 1E63 LATIN SMALL LETTER S WITH DOT BELOW
 # ->   0073 LATIN SMALL LETTER S
 "\x{1E63}" => "s",

 # 1E64 LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{1E64}" => "S",

 # 1E65 LATIN SMALL LETTER S WITH ACUTE AND DOT ABOVE
 # ->   0073 LATIN SMALL LETTER S
 "\x{1E65}" => "s",

 # 1E66 LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{1E66}" => "S",

 # 1E67 LATIN SMALL LETTER S WITH CARON AND DOT ABOVE
 # ->   0073 LATIN SMALL LETTER S
 "\x{1E67}" => "s",

 # 1E68 LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{1E68}" => "S",

 # 1E69 LATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVE
 # ->   0073 LATIN SMALL LETTER S
 "\x{1E69}" => "s",

 # 1E6A LATIN CAPITAL LETTER T WITH DOT ABOVE
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{1E6A}" => "T",

 # 1E6B LATIN SMALL LETTER T WITH DOT ABOVE
 # ->   0074 LATIN SMALL LETTER T
 "\x{1E6B}" => "t",

 # 1E6C LATIN CAPITAL LETTER T WITH DOT BELOW
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{1E6C}" => "T",

 # 1E6D LATIN SMALL LETTER T WITH DOT BELOW
 # ->   0074 LATIN SMALL LETTER T
 "\x{1E6D}" => "t",

 # 1E6E LATIN CAPITAL LETTER T WITH LINE BELOW
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{1E6E}" => "T",

 # 1E6F LATIN SMALL LETTER T WITH LINE BELOW
 # ->   0074 LATIN SMALL LETTER T
 "\x{1E6F}" => "t",

 # 1E70 LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{1E70}" => "T",

 # 1E71 LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW
 # ->   0074 LATIN SMALL LETTER T
 "\x{1E71}" => "t",

 # 1E72 LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1E72}" => "U",

 # 1E73 LATIN SMALL LETTER U WITH DIAERESIS BELOW
 # ->   0075 LATIN SMALL LETTER U
 "\x{1E73}" => "u",

 # 1E74 LATIN CAPITAL LETTER U WITH TILDE BELOW
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1E74}" => "U",

 # 1E75 LATIN SMALL LETTER U WITH TILDE BELOW
 # ->   0075 LATIN SMALL LETTER U
 "\x{1E75}" => "u",

 # 1E76 LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1E76}" => "U",

 # 1E77 LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW
 # ->   0075 LATIN SMALL LETTER U
 "\x{1E77}" => "u",

 # 1E78 LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1E78}" => "U",

 # 1E79 LATIN SMALL LETTER U WITH TILDE AND ACUTE
 # ->   0075 LATIN SMALL LETTER U
 "\x{1E79}" => "u",

 # 1E7A LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1E7A}" => "U",

 # 1E7B LATIN SMALL LETTER U WITH MACRON AND DIAERESIS
 # ->   0075 LATIN SMALL LETTER U
 "\x{1E7B}" => "u",

 # 1E7C LATIN CAPITAL LETTER V WITH TILDE
 # ->   0056 LATIN CAPITAL LETTER V
 "\x{1E7C}" => "V",

 # 1E7D LATIN SMALL LETTER V WITH TILDE
 # ->   0076 LATIN SMALL LETTER V
 "\x{1E7D}" => "v",

 # 1E7E LATIN CAPITAL LETTER V WITH DOT BELOW
 # ->   0056 LATIN CAPITAL LETTER V
 "\x{1E7E}" => "V",

 # 1E7F LATIN SMALL LETTER V WITH DOT BELOW
 # ->   0076 LATIN SMALL LETTER V
 "\x{1E7F}" => "v",

 # 1E80 LATIN CAPITAL LETTER W WITH GRAVE
 # ->   0057 LATIN CAPITAL LETTER W
 "\x{1E80}" => "W",

 # 1E81 LATIN SMALL LETTER W WITH GRAVE
 # ->   0077 LATIN SMALL LETTER W
 "\x{1E81}" => "w",

 # 1E82 LATIN CAPITAL LETTER W WITH ACUTE
 # ->   0057 LATIN CAPITAL LETTER W
 "\x{1E82}" => "W",

 # 1E83 LATIN SMALL LETTER W WITH ACUTE
 # ->   0077 LATIN SMALL LETTER W
 "\x{1E83}" => "w",

 # 1E84 LATIN CAPITAL LETTER W WITH DIAERESIS
 # ->   0057 LATIN CAPITAL LETTER W
 "\x{1E84}" => "W",

 # 1E85 LATIN SMALL LETTER W WITH DIAERESIS
 # ->   0077 LATIN SMALL LETTER W
 "\x{1E85}" => "w",

 # 1E86 LATIN CAPITAL LETTER W WITH DOT ABOVE
 # ->   0057 LATIN CAPITAL LETTER W
 "\x{1E86}" => "W",

 # 1E87 LATIN SMALL LETTER W WITH DOT ABOVE
 # ->   0077 LATIN SMALL LETTER W
 "\x{1E87}" => "w",

 # 1E88 LATIN CAPITAL LETTER W WITH DOT BELOW
 # ->   0057 LATIN CAPITAL LETTER W
 "\x{1E88}" => "W",

 # 1E89 LATIN SMALL LETTER W WITH DOT BELOW
 # ->   0077 LATIN SMALL LETTER W
 "\x{1E89}" => "w",

 # 1E8A LATIN CAPITAL LETTER X WITH DOT ABOVE
 # ->   0058 LATIN CAPITAL LETTER X
 "\x{1E8A}" => "X",

 # 1E8B LATIN SMALL LETTER X WITH DOT ABOVE
 # ->   0078 LATIN SMALL LETTER X
 "\x{1E8B}" => "x",

 # 1E8C LATIN CAPITAL LETTER X WITH DIAERESIS
 # ->   0058 LATIN CAPITAL LETTER X
 "\x{1E8C}" => "X",

 # 1E8D LATIN SMALL LETTER X WITH DIAERESIS
 # ->   0078 LATIN SMALL LETTER X
 "\x{1E8D}" => "x",

 # 1E8E LATIN CAPITAL LETTER Y WITH DOT ABOVE
 # ->   0059 LATIN CAPITAL LETTER Y
 "\x{1E8E}" => "Y",

 # 1E8F LATIN SMALL LETTER Y WITH DOT ABOVE
 # ->   0079 LATIN SMALL LETTER Y
 "\x{1E8F}" => "y",

 # 1E90 LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{1E90}" => "Z",

 # 1E91 LATIN SMALL LETTER Z WITH CIRCUMFLEX
 # ->   007A LATIN SMALL LETTER Z
 "\x{1E91}" => "z",

 # 1E92 LATIN CAPITAL LETTER Z WITH DOT BELOW
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{1E92}" => "Z",

 # 1E93 LATIN SMALL LETTER Z WITH DOT BELOW
 # ->   007A LATIN SMALL LETTER Z
 "\x{1E93}" => "z",

 # 1E94 LATIN CAPITAL LETTER Z WITH LINE BELOW
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{1E94}" => "Z",

 # 1E95 LATIN SMALL LETTER Z WITH LINE BELOW
 # ->   007A LATIN SMALL LETTER Z
 "\x{1E95}" => "z",

 # 1E96 LATIN SMALL LETTER H WITH LINE BELOW
 # ->   0068 LATIN SMALL LETTER H
 "\x{1E96}" => "h",

 # 1E97 LATIN SMALL LETTER T WITH DIAERESIS
 # ->   0074 LATIN SMALL LETTER T
 "\x{1E97}" => "t",

 # 1E98 LATIN SMALL LETTER W WITH RING ABOVE
 # ->   0077 LATIN SMALL LETTER W
 "\x{1E98}" => "w",

 # 1E99 LATIN SMALL LETTER Y WITH RING ABOVE
 # ->   0079 LATIN SMALL LETTER Y
 "\x{1E99}" => "y",

 # 1E9A LATIN SMALL LETTER A WITH RIGHT HALF RING
 # ->   0061 LATIN SMALL LETTER A
 #    + 02BE MODIFIER LETTER RIGHT HALF RING
 "\x{1E9A}" => "a\x{02BE}",

 # 1E9B LATIN SMALL LETTER LONG S WITH DOT ABOVE
 # ->   0073 LATIN SMALL LETTER S
 "\x{1E9B}" => "s",

 # 1EA0 LATIN CAPITAL LETTER A WITH DOT BELOW
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EA0}" => "A",

 # 1EA1 LATIN SMALL LETTER A WITH DOT BELOW
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EA1}" => "a",

 # 1EA2 LATIN CAPITAL LETTER A WITH HOOK ABOVE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EA2}" => "A",

 # 1EA3 LATIN SMALL LETTER A WITH HOOK ABOVE
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EA3}" => "a",

 # 1EA4 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EA4}" => "A",

 # 1EA5 LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EA5}" => "a",

 # 1EA6 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EA6}" => "A",

 # 1EA7 LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EA7}" => "a",

 # 1EA8 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EA8}" => "A",

 # 1EA9 LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EA9}" => "a",

 # 1EAA LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EAA}" => "A",

 # 1EAB LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EAB}" => "a",

 # 1EAC LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EAC}" => "A",

 # 1EAD LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EAD}" => "a",

 # 1EAE LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EAE}" => "A",

 # 1EAF LATIN SMALL LETTER A WITH BREVE AND ACUTE
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EAF}" => "a",

 # 1EB0 LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EB0}" => "A",

 # 1EB1 LATIN SMALL LETTER A WITH BREVE AND GRAVE
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EB1}" => "a",

 # 1EB2 LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EB2}" => "A",

 # 1EB3 LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EB3}" => "a",

 # 1EB4 LATIN CAPITAL LETTER A WITH BREVE AND TILDE
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EB4}" => "A",

 # 1EB5 LATIN SMALL LETTER A WITH BREVE AND TILDE
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EB5}" => "a",

 # 1EB6 LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{1EB6}" => "A",

 # 1EB7 LATIN SMALL LETTER A WITH BREVE AND DOT BELOW
 # ->   0061 LATIN SMALL LETTER A
 "\x{1EB7}" => "a",

 # 1EB8 LATIN CAPITAL LETTER E WITH DOT BELOW
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1EB8}" => "E",

 # 1EB9 LATIN SMALL LETTER E WITH DOT BELOW
 # ->   0065 LATIN SMALL LETTER E
 "\x{1EB9}" => "e",

 # 1EBA LATIN CAPITAL LETTER E WITH HOOK ABOVE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1EBA}" => "E",

 # 1EBB LATIN SMALL LETTER E WITH HOOK ABOVE
 # ->   0065 LATIN SMALL LETTER E
 "\x{1EBB}" => "e",

 # 1EBC LATIN CAPITAL LETTER E WITH TILDE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1EBC}" => "E",

 # 1EBD LATIN SMALL LETTER E WITH TILDE
 # ->   0065 LATIN SMALL LETTER E
 "\x{1EBD}" => "e",

 # 1EBE LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1EBE}" => "E",

 # 1EBF LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE
 # ->   0065 LATIN SMALL LETTER E
 "\x{1EBF}" => "e",

 # 1EC0 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1EC0}" => "E",

 # 1EC1 LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE
 # ->   0065 LATIN SMALL LETTER E
 "\x{1EC1}" => "e",

 # 1EC2 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1EC2}" => "E",

 # 1EC3 LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
 # ->   0065 LATIN SMALL LETTER E
 "\x{1EC3}" => "e",

 # 1EC4 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1EC4}" => "E",

 # 1EC5 LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE
 # ->   0065 LATIN SMALL LETTER E
 "\x{1EC5}" => "e",

 # 1EC6 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{1EC6}" => "E",

 # 1EC7 LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW
 # ->   0065 LATIN SMALL LETTER E
 "\x{1EC7}" => "e",

 # 1EC8 LATIN CAPITAL LETTER I WITH HOOK ABOVE
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{1EC8}" => "I",

 # 1EC9 LATIN SMALL LETTER I WITH HOOK ABOVE
 # ->   0069 LATIN SMALL LETTER I
 "\x{1EC9}" => "i",

 # 1ECA LATIN CAPITAL LETTER I WITH DOT BELOW
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{1ECA}" => "I",

 # 1ECB LATIN SMALL LETTER I WITH DOT BELOW
 # ->   0069 LATIN SMALL LETTER I
 "\x{1ECB}" => "i",

 # 1ECC LATIN CAPITAL LETTER O WITH DOT BELOW
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1ECC}" => "O",

 # 1ECD LATIN SMALL LETTER O WITH DOT BELOW
 # ->   006F LATIN SMALL LETTER O
 "\x{1ECD}" => "o",

 # 1ECE LATIN CAPITAL LETTER O WITH HOOK ABOVE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1ECE}" => "O",

 # 1ECF LATIN SMALL LETTER O WITH HOOK ABOVE
 # ->   006F LATIN SMALL LETTER O
 "\x{1ECF}" => "o",

 # 1ED0 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1ED0}" => "O",

 # 1ED1 LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE
 # ->   006F LATIN SMALL LETTER O
 "\x{1ED1}" => "o",

 # 1ED2 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1ED2}" => "O",

 # 1ED3 LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE
 # ->   006F LATIN SMALL LETTER O
 "\x{1ED3}" => "o",

 # 1ED4 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1ED4}" => "O",

 # 1ED5 LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
 # ->   006F LATIN SMALL LETTER O
 "\x{1ED5}" => "o",

 # 1ED6 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1ED6}" => "O",

 # 1ED7 LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE
 # ->   006F LATIN SMALL LETTER O
 "\x{1ED7}" => "o",

 # 1ED8 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1ED8}" => "O",

 # 1ED9 LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW
 # ->   006F LATIN SMALL LETTER O
 "\x{1ED9}" => "o",

 # 1EDA LATIN CAPITAL LETTER O WITH HORN AND ACUTE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1EDA}" => "O",

 # 1EDB LATIN SMALL LETTER O WITH HORN AND ACUTE
 # ->   006F LATIN SMALL LETTER O
 "\x{1EDB}" => "o",

 # 1EDC LATIN CAPITAL LETTER O WITH HORN AND GRAVE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1EDC}" => "O",

 # 1EDD LATIN SMALL LETTER O WITH HORN AND GRAVE
 # ->   006F LATIN SMALL LETTER O
 "\x{1EDD}" => "o",

 # 1EDE LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1EDE}" => "O",

 # 1EDF LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE
 # ->   006F LATIN SMALL LETTER O
 "\x{1EDF}" => "o",

 # 1EE0 LATIN CAPITAL LETTER O WITH HORN AND TILDE
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1EE0}" => "O",

 # 1EE1 LATIN SMALL LETTER O WITH HORN AND TILDE
 # ->   006F LATIN SMALL LETTER O
 "\x{1EE1}" => "o",

 # 1EE2 LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
 # ->   004F LATIN CAPITAL LETTER O
 "\x{1EE2}" => "O",

 # 1EE3 LATIN SMALL LETTER O WITH HORN AND DOT BELOW
 # ->   006F LATIN SMALL LETTER O
 "\x{1EE3}" => "o",

 # 1EE4 LATIN CAPITAL LETTER U WITH DOT BELOW
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1EE4}" => "U",

 # 1EE5 LATIN SMALL LETTER U WITH DOT BELOW
 # ->   0075 LATIN SMALL LETTER U
 "\x{1EE5}" => "u",

 # 1EE6 LATIN CAPITAL LETTER U WITH HOOK ABOVE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1EE6}" => "U",

 # 1EE7 LATIN SMALL LETTER U WITH HOOK ABOVE
 # ->   0075 LATIN SMALL LETTER U
 "\x{1EE7}" => "u",

 # 1EE8 LATIN CAPITAL LETTER U WITH HORN AND ACUTE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1EE8}" => "U",

 # 1EE9 LATIN SMALL LETTER U WITH HORN AND ACUTE
 # ->   0075 LATIN SMALL LETTER U
 "\x{1EE9}" => "u",

 # 1EEA LATIN CAPITAL LETTER U WITH HORN AND GRAVE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1EEA}" => "U",

 # 1EEB LATIN SMALL LETTER U WITH HORN AND GRAVE
 # ->   0075 LATIN SMALL LETTER U
 "\x{1EEB}" => "u",

 # 1EEC LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1EEC}" => "U",

 # 1EED LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE
 # ->   0075 LATIN SMALL LETTER U
 "\x{1EED}" => "u",

 # 1EEE LATIN CAPITAL LETTER U WITH HORN AND TILDE
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1EEE}" => "U",

 # 1EEF LATIN SMALL LETTER U WITH HORN AND TILDE
 # ->   0075 LATIN SMALL LETTER U
 "\x{1EEF}" => "u",

 # 1EF0 LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{1EF0}" => "U",

 # 1EF1 LATIN SMALL LETTER U WITH HORN AND DOT BELOW
 # ->   0075 LATIN SMALL LETTER U
 "\x{1EF1}" => "u",

 # 1EF2 LATIN CAPITAL LETTER Y WITH GRAVE
 # ->   0059 LATIN CAPITAL LETTER Y
 "\x{1EF2}" => "Y",

 # 1EF3 LATIN SMALL LETTER Y WITH GRAVE
 # ->   0079 LATIN SMALL LETTER Y
 "\x{1EF3}" => "y",

 # 1EF4 LATIN CAPITAL LETTER Y WITH DOT BELOW
 # ->   0059 LATIN CAPITAL LETTER Y
 "\x{1EF4}" => "Y",

 # 1EF5 LATIN SMALL LETTER Y WITH DOT BELOW
 # ->   0079 LATIN SMALL LETTER Y
 "\x{1EF5}" => "y",

 # 1EF6 LATIN CAPITAL LETTER Y WITH HOOK ABOVE
 # ->   0059 LATIN CAPITAL LETTER Y
 "\x{1EF6}" => "Y",

 # 1EF7 LATIN SMALL LETTER Y WITH HOOK ABOVE
 # ->   0079 LATIN SMALL LETTER Y
 "\x{1EF7}" => "y",

 # 1EF8 LATIN CAPITAL LETTER Y WITH TILDE
 # ->   0059 LATIN CAPITAL LETTER Y
 "\x{1EF8}" => "Y",

 # 1EF9 LATIN SMALL LETTER Y WITH TILDE
 # ->   0079 LATIN SMALL LETTER Y
 "\x{1EF9}" => "y",

 # 1F00 GREEK SMALL LETTER ALPHA WITH PSILI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F00}" => "\x{03B1}",

 # 1F01 GREEK SMALL LETTER ALPHA WITH DASIA
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F01}" => "\x{03B1}",

 # 1F02 GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F02}" => "\x{03B1}",

 # 1F03 GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F03}" => "\x{03B1}",

 # 1F04 GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F04}" => "\x{03B1}",

 # 1F05 GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F05}" => "\x{03B1}",

 # 1F06 GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F06}" => "\x{03B1}",

 # 1F07 GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F07}" => "\x{03B1}",

 # 1F08 GREEK CAPITAL LETTER ALPHA WITH PSILI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F08}" => "\x{0391}",

 # 1F09 GREEK CAPITAL LETTER ALPHA WITH DASIA
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F09}" => "\x{0391}",

 # 1F0A GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F0A}" => "\x{0391}",

 # 1F0B GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F0B}" => "\x{0391}",

 # 1F0C GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F0C}" => "\x{0391}",

 # 1F0D GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F0D}" => "\x{0391}",

 # 1F0E GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F0E}" => "\x{0391}",

 # 1F0F GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F0F}" => "\x{0391}",

 # 1F10 GREEK SMALL LETTER EPSILON WITH PSILI
 # ->   03B5 GREEK SMALL LETTER EPSILON
 "\x{1F10}" => "\x{03B5}",

 # 1F11 GREEK SMALL LETTER EPSILON WITH DASIA
 # ->   03B5 GREEK SMALL LETTER EPSILON
 "\x{1F11}" => "\x{03B5}",

 # 1F12 GREEK SMALL LETTER EPSILON WITH PSILI AND VARIA
 # ->   03B5 GREEK SMALL LETTER EPSILON
 "\x{1F12}" => "\x{03B5}",

 # 1F13 GREEK SMALL LETTER EPSILON WITH DASIA AND VARIA
 # ->   03B5 GREEK SMALL LETTER EPSILON
 "\x{1F13}" => "\x{03B5}",

 # 1F14 GREEK SMALL LETTER EPSILON WITH PSILI AND OXIA
 # ->   03B5 GREEK SMALL LETTER EPSILON
 "\x{1F14}" => "\x{03B5}",

 # 1F15 GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
 # ->   03B5 GREEK SMALL LETTER EPSILON
 "\x{1F15}" => "\x{03B5}",

 # 1F18 GREEK CAPITAL LETTER EPSILON WITH PSILI
 # ->   0395 GREEK CAPITAL LETTER EPSILON
 "\x{1F18}" => "\x{0395}",

 # 1F19 GREEK CAPITAL LETTER EPSILON WITH DASIA
 # ->   0395 GREEK CAPITAL LETTER EPSILON
 "\x{1F19}" => "\x{0395}",

 # 1F1A GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA
 # ->   0395 GREEK CAPITAL LETTER EPSILON
 "\x{1F1A}" => "\x{0395}",

 # 1F1B GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA
 # ->   0395 GREEK CAPITAL LETTER EPSILON
 "\x{1F1B}" => "\x{0395}",

 # 1F1C GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA
 # ->   0395 GREEK CAPITAL LETTER EPSILON
 "\x{1F1C}" => "\x{0395}",

 # 1F1D GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
 # ->   0395 GREEK CAPITAL LETTER EPSILON
 "\x{1F1D}" => "\x{0395}",

 # 1F20 GREEK SMALL LETTER ETA WITH PSILI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F20}" => "\x{03B7}",

 # 1F21 GREEK SMALL LETTER ETA WITH DASIA
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F21}" => "\x{03B7}",

 # 1F22 GREEK SMALL LETTER ETA WITH PSILI AND VARIA
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F22}" => "\x{03B7}",

 # 1F23 GREEK SMALL LETTER ETA WITH DASIA AND VARIA
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F23}" => "\x{03B7}",

 # 1F24 GREEK SMALL LETTER ETA WITH PSILI AND OXIA
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F24}" => "\x{03B7}",

 # 1F25 GREEK SMALL LETTER ETA WITH DASIA AND OXIA
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F25}" => "\x{03B7}",

 # 1F26 GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F26}" => "\x{03B7}",

 # 1F27 GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F27}" => "\x{03B7}",

 # 1F28 GREEK CAPITAL LETTER ETA WITH PSILI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F28}" => "\x{0397}",

 # 1F29 GREEK CAPITAL LETTER ETA WITH DASIA
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F29}" => "\x{0397}",

 # 1F2A GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F2A}" => "\x{0397}",

 # 1F2B GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F2B}" => "\x{0397}",

 # 1F2C GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F2C}" => "\x{0397}",

 # 1F2D GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F2D}" => "\x{0397}",

 # 1F2E GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F2E}" => "\x{0397}",

 # 1F2F GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F2F}" => "\x{0397}",

 # 1F30 GREEK SMALL LETTER IOTA WITH PSILI
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1F30}" => "\x{03B9}",

 # 1F31 GREEK SMALL LETTER IOTA WITH DASIA
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1F31}" => "\x{03B9}",

 # 1F32 GREEK SMALL LETTER IOTA WITH PSILI AND VARIA
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1F32}" => "\x{03B9}",

 # 1F33 GREEK SMALL LETTER IOTA WITH DASIA AND VARIA
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1F33}" => "\x{03B9}",

 # 1F34 GREEK SMALL LETTER IOTA WITH PSILI AND OXIA
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1F34}" => "\x{03B9}",

 # 1F35 GREEK SMALL LETTER IOTA WITH DASIA AND OXIA
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1F35}" => "\x{03B9}",

 # 1F36 GREEK SMALL LETTER IOTA WITH PSILI AND PERISPOMENI
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1F36}" => "\x{03B9}",

 # 1F37 GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1F37}" => "\x{03B9}",

 # 1F38 GREEK CAPITAL LETTER IOTA WITH PSILI
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1F38}" => "\x{0399}",

 # 1F39 GREEK CAPITAL LETTER IOTA WITH DASIA
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1F39}" => "\x{0399}",

 # 1F3A GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1F3A}" => "\x{0399}",

 # 1F3B GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1F3B}" => "\x{0399}",

 # 1F3C GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1F3C}" => "\x{0399}",

 # 1F3D GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1F3D}" => "\x{0399}",

 # 1F3E GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1F3E}" => "\x{0399}",

 # 1F3F GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1F3F}" => "\x{0399}",

 # 1F40 GREEK SMALL LETTER OMICRON WITH PSILI
 # ->   03BF GREEK SMALL LETTER OMICRON
 "\x{1F40}" => "\x{03BF}",

 # 1F41 GREEK SMALL LETTER OMICRON WITH DASIA
 # ->   03BF GREEK SMALL LETTER OMICRON
 "\x{1F41}" => "\x{03BF}",

 # 1F42 GREEK SMALL LETTER OMICRON WITH PSILI AND VARIA
 # ->   03BF GREEK SMALL LETTER OMICRON
 "\x{1F42}" => "\x{03BF}",

 # 1F43 GREEK SMALL LETTER OMICRON WITH DASIA AND VARIA
 # ->   03BF GREEK SMALL LETTER OMICRON
 "\x{1F43}" => "\x{03BF}",

 # 1F44 GREEK SMALL LETTER OMICRON WITH PSILI AND OXIA
 # ->   03BF GREEK SMALL LETTER OMICRON
 "\x{1F44}" => "\x{03BF}",

 # 1F45 GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
 # ->   03BF GREEK SMALL LETTER OMICRON
 "\x{1F45}" => "\x{03BF}",

 # 1F48 GREEK CAPITAL LETTER OMICRON WITH PSILI
 # ->   039F GREEK CAPITAL LETTER OMICRON
 "\x{1F48}" => "\x{039F}",

 # 1F49 GREEK CAPITAL LETTER OMICRON WITH DASIA
 # ->   039F GREEK CAPITAL LETTER OMICRON
 "\x{1F49}" => "\x{039F}",

 # 1F4A GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA
 # ->   039F GREEK CAPITAL LETTER OMICRON
 "\x{1F4A}" => "\x{039F}",

 # 1F4B GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA
 # ->   039F GREEK CAPITAL LETTER OMICRON
 "\x{1F4B}" => "\x{039F}",

 # 1F4C GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA
 # ->   039F GREEK CAPITAL LETTER OMICRON
 "\x{1F4C}" => "\x{039F}",

 # 1F4D GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
 # ->   039F GREEK CAPITAL LETTER OMICRON
 "\x{1F4D}" => "\x{039F}",

 # 1F50 GREEK SMALL LETTER UPSILON WITH PSILI
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1F50}" => "\x{03C5}",

 # 1F51 GREEK SMALL LETTER UPSILON WITH DASIA
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1F51}" => "\x{03C5}",

 # 1F52 GREEK SMALL LETTER UPSILON WITH PSILI AND VARIA
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1F52}" => "\x{03C5}",

 # 1F53 GREEK SMALL LETTER UPSILON WITH DASIA AND VARIA
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1F53}" => "\x{03C5}",

 # 1F54 GREEK SMALL LETTER UPSILON WITH PSILI AND OXIA
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1F54}" => "\x{03C5}",

 # 1F55 GREEK SMALL LETTER UPSILON WITH DASIA AND OXIA
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1F55}" => "\x{03C5}",

 # 1F56 GREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENI
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1F56}" => "\x{03C5}",

 # 1F57 GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1F57}" => "\x{03C5}",

 # 1F59 GREEK CAPITAL LETTER UPSILON WITH DASIA
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{1F59}" => "\x{03A5}",

 # 1F5B GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{1F5B}" => "\x{03A5}",

 # 1F5D GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{1F5D}" => "\x{03A5}",

 # 1F5F GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{1F5F}" => "\x{03A5}",

 # 1F60 GREEK SMALL LETTER OMEGA WITH PSILI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1F60}" => "\x{03C9}",

 # 1F61 GREEK SMALL LETTER OMEGA WITH DASIA
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1F61}" => "\x{03C9}",

 # 1F62 GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1F62}" => "\x{03C9}",

 # 1F63 GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1F63}" => "\x{03C9}",

 # 1F64 GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1F64}" => "\x{03C9}",

 # 1F65 GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1F65}" => "\x{03C9}",

 # 1F66 GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1F66}" => "\x{03C9}",

 # 1F67 GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1F67}" => "\x{03C9}",

 # 1F68 GREEK CAPITAL LETTER OMEGA WITH PSILI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1F68}" => "\x{03A9}",

 # 1F69 GREEK CAPITAL LETTER OMEGA WITH DASIA
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1F69}" => "\x{03A9}",

 # 1F6A GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1F6A}" => "\x{03A9}",

 # 1F6B GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1F6B}" => "\x{03A9}",

 # 1F6C GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1F6C}" => "\x{03A9}",

 # 1F6D GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1F6D}" => "\x{03A9}",

 # 1F6E GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1F6E}" => "\x{03A9}",

 # 1F6F GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1F6F}" => "\x{03A9}",

 # 1F70 GREEK SMALL LETTER ALPHA WITH VARIA
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F70}" => "\x{03B1}",

 # 1F71 GREEK SMALL LETTER ALPHA WITH OXIA
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F71}" => "\x{03B1}",

 # 1F72 GREEK SMALL LETTER EPSILON WITH VARIA
 # ->   03B5 GREEK SMALL LETTER EPSILON
 "\x{1F72}" => "\x{03B5}",

 # 1F73 GREEK SMALL LETTER EPSILON WITH OXIA
 # ->   03B5 GREEK SMALL LETTER EPSILON
 "\x{1F73}" => "\x{03B5}",

 # 1F74 GREEK SMALL LETTER ETA WITH VARIA
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F74}" => "\x{03B7}",

 # 1F75 GREEK SMALL LETTER ETA WITH OXIA
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F75}" => "\x{03B7}",

 # 1F76 GREEK SMALL LETTER IOTA WITH VARIA
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1F76}" => "\x{03B9}",

 # 1F77 GREEK SMALL LETTER IOTA WITH OXIA
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1F77}" => "\x{03B9}",

 # 1F78 GREEK SMALL LETTER OMICRON WITH VARIA
 # ->   03BF GREEK SMALL LETTER OMICRON
 "\x{1F78}" => "\x{03BF}",

 # 1F79 GREEK SMALL LETTER OMICRON WITH OXIA
 # ->   03BF GREEK SMALL LETTER OMICRON
 "\x{1F79}" => "\x{03BF}",

 # 1F7A GREEK SMALL LETTER UPSILON WITH VARIA
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1F7A}" => "\x{03C5}",

 # 1F7B GREEK SMALL LETTER UPSILON WITH OXIA
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1F7B}" => "\x{03C5}",

 # 1F7C GREEK SMALL LETTER OMEGA WITH VARIA
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1F7C}" => "\x{03C9}",

 # 1F7D GREEK SMALL LETTER OMEGA WITH OXIA
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1F7D}" => "\x{03C9}",

 # 1F80 GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F80}" => "\x{03B1}",

 # 1F81 GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F81}" => "\x{03B1}",

 # 1F82 GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F82}" => "\x{03B1}",

 # 1F83 GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F83}" => "\x{03B1}",

 # 1F84 GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F84}" => "\x{03B1}",

 # 1F85 GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F85}" => "\x{03B1}",

 # 1F86 GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F86}" => "\x{03B1}",

 # 1F87 GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1F87}" => "\x{03B1}",

 # 1F88 GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F88}" => "\x{0391}",

 # 1F89 GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F89}" => "\x{0391}",

 # 1F8A GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F8A}" => "\x{0391}",

 # 1F8B GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F8B}" => "\x{0391}",

 # 1F8C GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F8C}" => "\x{0391}",

 # 1F8D GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F8D}" => "\x{0391}",

 # 1F8E GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F8E}" => "\x{0391}",

 # 1F8F GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1F8F}" => "\x{0391}",

 # 1F90 GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F90}" => "\x{03B7}",

 # 1F91 GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F91}" => "\x{03B7}",

 # 1F92 GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F92}" => "\x{03B7}",

 # 1F93 GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F93}" => "\x{03B7}",

 # 1F94 GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F94}" => "\x{03B7}",

 # 1F95 GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F95}" => "\x{03B7}",

 # 1F96 GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F96}" => "\x{03B7}",

 # 1F97 GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1F97}" => "\x{03B7}",

 # 1F98 GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F98}" => "\x{0397}",

 # 1F99 GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F99}" => "\x{0397}",

 # 1F9A GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F9A}" => "\x{0397}",

 # 1F9B GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F9B}" => "\x{0397}",

 # 1F9C GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F9C}" => "\x{0397}",

 # 1F9D GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F9D}" => "\x{0397}",

 # 1F9E GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F9E}" => "\x{0397}",

 # 1F9F GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1F9F}" => "\x{0397}",

 # 1FA0 GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FA0}" => "\x{03C9}",

 # 1FA1 GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FA1}" => "\x{03C9}",

 # 1FA2 GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FA2}" => "\x{03C9}",

 # 1FA3 GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FA3}" => "\x{03C9}",

 # 1FA4 GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FA4}" => "\x{03C9}",

 # 1FA5 GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FA5}" => "\x{03C9}",

 # 1FA6 GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FA6}" => "\x{03C9}",

 # 1FA7 GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FA7}" => "\x{03C9}",

 # 1FA8 GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1FA8}" => "\x{03A9}",

 # 1FA9 GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1FA9}" => "\x{03A9}",

 # 1FAA GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1FAA}" => "\x{03A9}",

 # 1FAB GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1FAB}" => "\x{03A9}",

 # 1FAC GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1FAC}" => "\x{03A9}",

 # 1FAD GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1FAD}" => "\x{03A9}",

 # 1FAE GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1FAE}" => "\x{03A9}",

 # 1FAF GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1FAF}" => "\x{03A9}",

 # 1FB0 GREEK SMALL LETTER ALPHA WITH VRACHY
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1FB0}" => "\x{03B1}",

 # 1FB1 GREEK SMALL LETTER ALPHA WITH MACRON
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1FB1}" => "\x{03B1}",

 # 1FB2 GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1FB2}" => "\x{03B1}",

 # 1FB3 GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1FB3}" => "\x{03B1}",

 # 1FB4 GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1FB4}" => "\x{03B1}",

 # 1FB6 GREEK SMALL LETTER ALPHA WITH PERISPOMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1FB6}" => "\x{03B1}",

 # 1FB7 GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI
 # ->   03B1 GREEK SMALL LETTER ALPHA
 "\x{1FB7}" => "\x{03B1}",

 # 1FB8 GREEK CAPITAL LETTER ALPHA WITH VRACHY
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1FB8}" => "\x{0391}",

 # 1FB9 GREEK CAPITAL LETTER ALPHA WITH MACRON
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1FB9}" => "\x{0391}",

 # 1FBA GREEK CAPITAL LETTER ALPHA WITH VARIA
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1FBA}" => "\x{0391}",

 # 1FBB GREEK CAPITAL LETTER ALPHA WITH OXIA
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1FBB}" => "\x{0391}",

 # 1FBC GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
 # ->   0391 GREEK CAPITAL LETTER ALPHA
 "\x{1FBC}" => "\x{0391}",

 # 1FBD GREEK KORONIS
 # ->   0020 SPACE
 "\x{1FBD}" => " ",

 # 1FBE GREEK PROSGEGRAMMENI
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1FBE}" => "\x{03B9}",

 # 1FBF GREEK PSILI
 # ->   0020 SPACE
 "\x{1FBF}" => " ",

 # 1FC0 GREEK PERISPOMENI
 # ->   0020 SPACE
 "\x{1FC0}" => " ",

 # 1FC1 GREEK DIALYTIKA AND PERISPOMENI
 # ->   0020 SPACE
 "\x{1FC1}" => " ",

 # 1FC2 GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1FC2}" => "\x{03B7}",

 # 1FC3 GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1FC3}" => "\x{03B7}",

 # 1FC4 GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1FC4}" => "\x{03B7}",

 # 1FC6 GREEK SMALL LETTER ETA WITH PERISPOMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1FC6}" => "\x{03B7}",

 # 1FC7 GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI
 # ->   03B7 GREEK SMALL LETTER ETA
 "\x{1FC7}" => "\x{03B7}",

 # 1FC8 GREEK CAPITAL LETTER EPSILON WITH VARIA
 # ->   0395 GREEK CAPITAL LETTER EPSILON
 "\x{1FC8}" => "\x{0395}",

 # 1FC9 GREEK CAPITAL LETTER EPSILON WITH OXIA
 # ->   0395 GREEK CAPITAL LETTER EPSILON
 "\x{1FC9}" => "\x{0395}",

 # 1FCA GREEK CAPITAL LETTER ETA WITH VARIA
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1FCA}" => "\x{0397}",

 # 1FCB GREEK CAPITAL LETTER ETA WITH OXIA
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1FCB}" => "\x{0397}",

 # 1FCC GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
 # ->   0397 GREEK CAPITAL LETTER ETA
 "\x{1FCC}" => "\x{0397}",

 # 1FCD GREEK PSILI AND VARIA
 # ->   0020 SPACE
 "\x{1FCD}" => " ",

 # 1FCE GREEK PSILI AND OXIA
 # ->   0020 SPACE
 "\x{1FCE}" => " ",

 # 1FCF GREEK PSILI AND PERISPOMENI
 # ->   0020 SPACE
 "\x{1FCF}" => " ",

 # 1FD0 GREEK SMALL LETTER IOTA WITH VRACHY
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1FD0}" => "\x{03B9}",

 # 1FD1 GREEK SMALL LETTER IOTA WITH MACRON
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1FD1}" => "\x{03B9}",

 # 1FD2 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1FD2}" => "\x{03B9}",

 # 1FD3 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1FD3}" => "\x{03B9}",

 # 1FD6 GREEK SMALL LETTER IOTA WITH PERISPOMENI
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1FD6}" => "\x{03B9}",

 # 1FD7 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI
 # ->   03B9 GREEK SMALL LETTER IOTA
 "\x{1FD7}" => "\x{03B9}",

 # 1FD8 GREEK CAPITAL LETTER IOTA WITH VRACHY
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1FD8}" => "\x{0399}",

 # 1FD9 GREEK CAPITAL LETTER IOTA WITH MACRON
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1FD9}" => "\x{0399}",

 # 1FDA GREEK CAPITAL LETTER IOTA WITH VARIA
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1FDA}" => "\x{0399}",

 # 1FDB GREEK CAPITAL LETTER IOTA WITH OXIA
 # ->   0399 GREEK CAPITAL LETTER IOTA
 "\x{1FDB}" => "\x{0399}",

 # 1FDD GREEK DASIA AND VARIA
 # ->   0020 SPACE
 "\x{1FDD}" => " ",

 # 1FDE GREEK DASIA AND OXIA
 # ->   0020 SPACE
 "\x{1FDE}" => " ",

 # 1FDF GREEK DASIA AND PERISPOMENI
 # ->   0020 SPACE
 "\x{1FDF}" => " ",

 # 1FE0 GREEK SMALL LETTER UPSILON WITH VRACHY
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1FE0}" => "\x{03C5}",

 # 1FE1 GREEK SMALL LETTER UPSILON WITH MACRON
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1FE1}" => "\x{03C5}",

 # 1FE2 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1FE2}" => "\x{03C5}",

 # 1FE3 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIA
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1FE3}" => "\x{03C5}",

 # 1FE4 GREEK SMALL LETTER RHO WITH PSILI
 # ->   03C1 GREEK SMALL LETTER RHO
 "\x{1FE4}" => "\x{03C1}",

 # 1FE5 GREEK SMALL LETTER RHO WITH DASIA
 # ->   03C1 GREEK SMALL LETTER RHO
 "\x{1FE5}" => "\x{03C1}",

 # 1FE6 GREEK SMALL LETTER UPSILON WITH PERISPOMENI
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1FE6}" => "\x{03C5}",

 # 1FE7 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI
 # ->   03C5 GREEK SMALL LETTER UPSILON
 "\x{1FE7}" => "\x{03C5}",

 # 1FE8 GREEK CAPITAL LETTER UPSILON WITH VRACHY
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{1FE8}" => "\x{03A5}",

 # 1FE9 GREEK CAPITAL LETTER UPSILON WITH MACRON
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{1FE9}" => "\x{03A5}",

 # 1FEA GREEK CAPITAL LETTER UPSILON WITH VARIA
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{1FEA}" => "\x{03A5}",

 # 1FEB GREEK CAPITAL LETTER UPSILON WITH OXIA
 # ->   03A5 GREEK CAPITAL LETTER UPSILON
 "\x{1FEB}" => "\x{03A5}",

 # 1FEC GREEK CAPITAL LETTER RHO WITH DASIA
 # ->   03A1 GREEK CAPITAL LETTER RHO
 "\x{1FEC}" => "\x{03A1}",

 # 1FED GREEK DIALYTIKA AND VARIA
 # ->   0020 SPACE
 "\x{1FED}" => " ",

 # 1FEE GREEK DIALYTIKA AND OXIA
 # ->   0020 SPACE
 "\x{1FEE}" => " ",

 # 1FEF GREEK VARIA
 # ->   0060 GRAVE ACCENT
 "\x{1FEF}" => "`",

 # 1FF2 GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FF2}" => "\x{03C9}",

 # 1FF3 GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FF3}" => "\x{03C9}",

 # 1FF4 GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FF4}" => "\x{03C9}",

 # 1FF6 GREEK SMALL LETTER OMEGA WITH PERISPOMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FF6}" => "\x{03C9}",

 # 1FF7 GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI
 # ->   03C9 GREEK SMALL LETTER OMEGA
 "\x{1FF7}" => "\x{03C9}",

 # 1FF8 GREEK CAPITAL LETTER OMICRON WITH VARIA
 # ->   039F GREEK CAPITAL LETTER OMICRON
 "\x{1FF8}" => "\x{039F}",

 # 1FF9 GREEK CAPITAL LETTER OMICRON WITH OXIA
 # ->   039F GREEK CAPITAL LETTER OMICRON
 "\x{1FF9}" => "\x{039F}",

 # 1FFA GREEK CAPITAL LETTER OMEGA WITH VARIA
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1FFA}" => "\x{03A9}",

 # 1FFB GREEK CAPITAL LETTER OMEGA WITH OXIA
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1FFB}" => "\x{03A9}",

 # 1FFC GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{1FFC}" => "\x{03A9}",

 # 1FFD GREEK OXIA
 # ->   0020 SPACE
 "\x{1FFD}" => " ",

 # 1FFE GREEK DASIA
 # ->   0020 SPACE
 "\x{1FFE}" => " ",

 # 2000 EN QUAD
 # ->   0020 SPACE
 "\x{2000}" => " ",

 # 2001 EM QUAD
 # ->   0020 SPACE
 "\x{2001}" => " ",

 # 2002 EN SPACE
 # ->   0020 SPACE
 "\x{2002}" => " ",

 # 2003 EM SPACE
 # ->   0020 SPACE
 "\x{2003}" => " ",

 # 2004 THREE-PER-EM SPACE
 # ->   0020 SPACE
 "\x{2004}" => " ",

 # 2005 FOUR-PER-EM SPACE
 # ->   0020 SPACE
 "\x{2005}" => " ",

 # 2006 SIX-PER-EM SPACE
 # ->   0020 SPACE
 "\x{2006}" => " ",

 # 2007 FIGURE SPACE
 # ->   0020 SPACE
 "\x{2007}" => " ",

 # 2008 PUNCTUATION SPACE
 # ->   0020 SPACE
 "\x{2008}" => " ",

 # 2009 THIN SPACE
 # ->   0020 SPACE
 "\x{2009}" => " ",

 # 200A HAIR SPACE
 # ->   0020 SPACE
 "\x{200A}" => " ",

 # 2011 NON-BREAKING HYPHEN
 # ->   2010 HYPHEN
 "\x{2011}" => "\x{2010}",

 # 2017 DOUBLE LOW LINE
 # ->   0020 SPACE
 "\x{2017}" => " ",

 # 2024 ONE DOT LEADER
 # ->   002E FULL STOP
 "\x{2024}" => ".",

 # 2025 TWO DOT LEADER
 # ->   002E FULL STOP
 #    + 002E FULL STOP
 "\x{2025}" => "..",

 # 2026 HORIZONTAL ELLIPSIS
 # ->   002E FULL STOP
 #    + 002E FULL STOP
 #    + 002E FULL STOP
 "\x{2026}" => "...",

 # 202F NARROW NO-BREAK SPACE
 # ->   0020 SPACE
 "\x{202F}" => " ",

 # 2033 DOUBLE PRIME
 # ->   2032 PRIME
 #    + 2032 PRIME
 "\x{2033}" => "\x{2032}\x{2032}",

 # 2034 TRIPLE PRIME
 # ->   2032 PRIME
 #    + 2032 PRIME
 #    + 2032 PRIME
 "\x{2034}" => "\x{2032}\x{2032}\x{2032}",

 # 2036 REVERSED DOUBLE PRIME
 # ->   2035 REVERSED PRIME
 #    + 2035 REVERSED PRIME
 "\x{2036}" => "\x{2035}\x{2035}",

 # 2037 REVERSED TRIPLE PRIME
 # ->   2035 REVERSED PRIME
 #    + 2035 REVERSED PRIME
 #    + 2035 REVERSED PRIME
 "\x{2037}" => "\x{2035}\x{2035}\x{2035}",

 # 203C DOUBLE EXCLAMATION MARK
 # ->   0021 EXCLAMATION MARK
 #    + 0021 EXCLAMATION MARK
 "\x{203C}" => "!!",

 # 203E OVERLINE
 # ->   0020 SPACE
 "\x{203E}" => " ",

 # 2045 LEFT SQUARE BRACKET WITH QUILL
 # ->   005B LEFT SQUARE BRACKET
 "\x{2045}" => "[",

 # 2046 RIGHT SQUARE BRACKET WITH QUILL
 # ->   005D RIGHT SQUARE BRACKET
 "\x{2046}" => "]",

 # 2047 DOUBLE QUESTION MARK
 # ->   003F QUESTION MARK
 #    + 003F QUESTION MARK
 "\x{2047}" => "??",

 # 2048 QUESTION EXCLAMATION MARK
 # ->   003F QUESTION MARK
 #    + 0021 EXCLAMATION MARK
 "\x{2048}" => "?!",

 # 2049 EXCLAMATION QUESTION MARK
 # ->   0021 EXCLAMATION MARK
 #    + 003F QUESTION MARK
 "\x{2049}" => "!?",

 # 2057 QUADRUPLE PRIME
 # ->   2032 PRIME
 #    + 2032 PRIME
 #    + 2032 PRIME
 #    + 2032 PRIME
 "\x{2057}" => "\x{2032}\x{2032}\x{2032}\x{2032}",

 # 205F MEDIUM MATHEMATICAL SPACE
 # ->   0020 SPACE
 "\x{205F}" => " ",

 # 2070 SUPERSCRIPT ZERO
 # ->   0030 DIGIT ZERO
 "\x{2070}" => "0",

 # 2071 SUPERSCRIPT LATIN SMALL LETTER I
 # ->   0069 LATIN SMALL LETTER I
 "\x{2071}" => "i",

 # 2074 SUPERSCRIPT FOUR
 # ->   0034 DIGIT FOUR
 "\x{2074}" => "4",

 # 2075 SUPERSCRIPT FIVE
 # ->   0035 DIGIT FIVE
 "\x{2075}" => "5",

 # 2076 SUPERSCRIPT SIX
 # ->   0036 DIGIT SIX
 "\x{2076}" => "6",

 # 2077 SUPERSCRIPT SEVEN
 # ->   0037 DIGIT SEVEN
 "\x{2077}" => "7",

 # 2078 SUPERSCRIPT EIGHT
 # ->   0038 DIGIT EIGHT
 "\x{2078}" => "8",

 # 2079 SUPERSCRIPT NINE
 # ->   0039 DIGIT NINE
 "\x{2079}" => "9",

 # 207A SUPERSCRIPT PLUS SIGN
 # ->   002B PLUS SIGN
 "\x{207A}" => "+",

 # 207B SUPERSCRIPT MINUS
 # ->   2212 MINUS SIGN
 "\x{207B}" => "\x{2212}",

 # 207C SUPERSCRIPT EQUALS SIGN
 # ->   003D EQUALS SIGN
 "\x{207C}" => "=",

 # 207D SUPERSCRIPT LEFT PARENTHESIS
 # ->   0028 LEFT PARENTHESIS
 "\x{207D}" => "(",

 # 207E SUPERSCRIPT RIGHT PARENTHESIS
 # ->   0029 RIGHT PARENTHESIS
 "\x{207E}" => ")",

 # 207F SUPERSCRIPT LATIN SMALL LETTER N
 # ->   006E LATIN SMALL LETTER N
 "\x{207F}" => "n",

 # 2080 SUBSCRIPT ZERO
 # ->   0030 DIGIT ZERO
 "\x{2080}" => "0",

 # 2081 SUBSCRIPT ONE
 # ->   0031 DIGIT ONE
 "\x{2081}" => "1",

 # 2082 SUBSCRIPT TWO
 # ->   0032 DIGIT TWO
 "\x{2082}" => "2",

 # 2083 SUBSCRIPT THREE
 # ->   0033 DIGIT THREE
 "\x{2083}" => "3",

 # 2084 SUBSCRIPT FOUR
 # ->   0034 DIGIT FOUR
 "\x{2084}" => "4",

 # 2085 SUBSCRIPT FIVE
 # ->   0035 DIGIT FIVE
 "\x{2085}" => "5",

 # 2086 SUBSCRIPT SIX
 # ->   0036 DIGIT SIX
 "\x{2086}" => "6",

 # 2087 SUBSCRIPT SEVEN
 # ->   0037 DIGIT SEVEN
 "\x{2087}" => "7",

 # 2088 SUBSCRIPT EIGHT
 # ->   0038 DIGIT EIGHT
 "\x{2088}" => "8",

 # 2089 SUBSCRIPT NINE
 # ->   0039 DIGIT NINE
 "\x{2089}" => "9",

 # 208A SUBSCRIPT PLUS SIGN
 # ->   002B PLUS SIGN
 "\x{208A}" => "+",

 # 208B SUBSCRIPT MINUS
 # ->   2212 MINUS SIGN
 "\x{208B}" => "\x{2212}",

 # 208C SUBSCRIPT EQUALS SIGN
 # ->   003D EQUALS SIGN
 "\x{208C}" => "=",

 # 208D SUBSCRIPT LEFT PARENTHESIS
 # ->   0028 LEFT PARENTHESIS
 "\x{208D}" => "(",

 # 208E SUBSCRIPT RIGHT PARENTHESIS
 # ->   0029 RIGHT PARENTHESIS
 "\x{208E}" => ")",

 # 20A8 RUPEE SIGN
 # ->   0052 LATIN CAPITAL LETTER R
 #    + 0073 LATIN SMALL LETTER S
 "\x{20A8}" => "Rs",

 # 2100 ACCOUNT OF
 # ->   0061 LATIN SMALL LETTER A
 #    + 002F SOLIDUS
 #    + 0063 LATIN SMALL LETTER C
 "\x{2100}" => "a/c",

 # 2101 ADDRESSED TO THE SUBJECT
 # ->   0061 LATIN SMALL LETTER A
 #    + 002F SOLIDUS
 #    + 0073 LATIN SMALL LETTER S
 "\x{2101}" => "a/s",

 # 2102 DOUBLE-STRUCK CAPITAL C
 # ->   0043 LATIN CAPITAL LETTER C
 "\x{2102}" => "C",

 # 2103 DEGREE CELSIUS
 # ->   00B0 DEGREE SIGN
 #    + 0043 LATIN CAPITAL LETTER C
 "\x{2103}" => "\xB0C",

 # 2105 CARE OF
 # ->   0063 LATIN SMALL LETTER C
 #    + 002F SOLIDUS
 #    + 006F LATIN SMALL LETTER O
 "\x{2105}" => "c/o",

 # 2106 CADA UNA
 # ->   0063 LATIN SMALL LETTER C
 #    + 002F SOLIDUS
 #    + 0075 LATIN SMALL LETTER U
 "\x{2106}" => "c/u",

 # 2107 EULER CONSTANT
 # ->   0190 LATIN CAPITAL LETTER OPEN E
 "\x{2107}" => "\x{0190}",

 # 2109 DEGREE FAHRENHEIT
 # ->   00B0 DEGREE SIGN
 #    + 0046 LATIN CAPITAL LETTER F
 "\x{2109}" => "\xB0F",

 # 210A SCRIPT SMALL G
 # ->   0067 LATIN SMALL LETTER G
 "\x{210A}" => "g",

 # 210B SCRIPT CAPITAL H
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{210B}" => "H",

 # 210C BLACK-LETTER CAPITAL H
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{210C}" => "H",

 # 210D DOUBLE-STRUCK CAPITAL H
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{210D}" => "H",

 # 210E PLANCK CONSTANT
 # ->   0068 LATIN SMALL LETTER H
 "\x{210E}" => "h",

 # 210F PLANCK CONSTANT OVER TWO PI
 # ->   0068 LATIN SMALL LETTER H
 "\x{210F}" => "h",

 # 2110 SCRIPT CAPITAL I
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{2110}" => "I",

 # 2111 BLACK-LETTER CAPITAL I
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{2111}" => "I",

 # 2112 SCRIPT CAPITAL L
 # ->   004C LATIN CAPITAL LETTER L
 "\x{2112}" => "L",

 # 2113 SCRIPT SMALL L
 # ->   006C LATIN SMALL LETTER L
 "\x{2113}" => "l",

 # 2115 DOUBLE-STRUCK CAPITAL N
 # ->   004E LATIN CAPITAL LETTER N
 "\x{2115}" => "N",

 # 2116 NUMERO SIGN
 # ->   004E LATIN CAPITAL LETTER N
 #    + 006F LATIN SMALL LETTER O
 "\x{2116}" => "No",

 # 2119 DOUBLE-STRUCK CAPITAL P
 # ->   0050 LATIN CAPITAL LETTER P
 "\x{2119}" => "P",

 # 211A DOUBLE-STRUCK CAPITAL Q
 # ->   0051 LATIN CAPITAL LETTER Q
 "\x{211A}" => "Q",

 # 211B SCRIPT CAPITAL R
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{211B}" => "R",

 # 211C BLACK-LETTER CAPITAL R
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{211C}" => "R",

 # 211D DOUBLE-STRUCK CAPITAL R
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{211D}" => "R",

 # 2120 SERVICE MARK
 # ->   0053 LATIN CAPITAL LETTER S
 #    + 004D LATIN CAPITAL LETTER M
 "\x{2120}" => "SM",

 # 2121 TELEPHONE SIGN
 # ->   0054 LATIN CAPITAL LETTER T
 #    + 0045 LATIN CAPITAL LETTER E
 #    + 004C LATIN CAPITAL LETTER L
 "\x{2121}" => "TEL",

 # 2122 TRADE MARK SIGN
 # ->   0054 LATIN CAPITAL LETTER T
 #    + 004D LATIN CAPITAL LETTER M
 "\x{2122}" => "TM",

 # 2124 DOUBLE-STRUCK CAPITAL Z
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{2124}" => "Z",

 # 2126 OHM SIGN
 # ->   03A9 GREEK CAPITAL LETTER OMEGA
 "\x{2126}" => "\x{03A9}",

 # 2128 BLACK-LETTER CAPITAL Z
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{2128}" => "Z",

 # 212A KELVIN SIGN
 # ->   004B LATIN CAPITAL LETTER K
 "\x{212A}" => "K",

 # 212B ANGSTROM SIGN
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{212B}" => "A",

 # 212C SCRIPT CAPITAL B
 # ->   0042 LATIN CAPITAL LETTER B
 "\x{212C}" => "B",

 # 212D BLACK-LETTER CAPITAL C
 # ->   0043 LATIN CAPITAL LETTER C
 "\x{212D}" => "C",

 # 212F SCRIPT SMALL E
 # ->   0065 LATIN SMALL LETTER E
 "\x{212F}" => "e",

 # 2130 SCRIPT CAPITAL E
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{2130}" => "E",

 # 2131 SCRIPT CAPITAL F
 # ->   0046 LATIN CAPITAL LETTER F
 "\x{2131}" => "F",

 # 2133 SCRIPT CAPITAL M
 # ->   004D LATIN CAPITAL LETTER M
 "\x{2133}" => "M",

 # 2134 SCRIPT SMALL O
 # ->   006F LATIN SMALL LETTER O
 "\x{2134}" => "o",

 # 2135 ALEF SYMBOL
 # ->   05D0 HEBREW LETTER ALEF
 "\x{2135}" => "\x{05D0}",

 # 2136 BET SYMBOL
 # ->   05D1 HEBREW LETTER BET
 "\x{2136}" => "\x{05D1}",

 # 2137 GIMEL SYMBOL
 # ->   05D2 HEBREW LETTER GIMEL
 "\x{2137}" => "\x{05D2}",

 # 2138 DALET SYMBOL
 # ->   05D3 HEBREW LETTER DALET
 "\x{2138}" => "\x{05D3}",

 # 2139 INFORMATION SOURCE
 # ->   0069 LATIN SMALL LETTER I
 "\x{2139}" => "i",

 # 213B FACSIMILE SIGN
 # ->   0046 LATIN CAPITAL LETTER F
 #    + 0041 LATIN CAPITAL LETTER A
 #    + 0058 LATIN CAPITAL LETTER X
 "\x{213B}" => "FAX",

 # 213D DOUBLE-STRUCK SMALL GAMMA
 # ->   03B3 GREEK SMALL LETTER GAMMA
 "\x{213D}" => "\x{03B3}",

 # 213E DOUBLE-STRUCK CAPITAL GAMMA
 # ->   0393 GREEK CAPITAL LETTER GAMMA
 "\x{213E}" => "\x{0393}",

 # 213F DOUBLE-STRUCK CAPITAL PI
 # ->   03A0 GREEK CAPITAL LETTER PI
 "\x{213F}" => "\x{03A0}",

 # 2140 DOUBLE-STRUCK N-ARY SUMMATION
 # ->   2211 N-ARY SUMMATION
 "\x{2140}" => "\x{2211}",

 # 2145 DOUBLE-STRUCK ITALIC CAPITAL D
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{2145}" => "D",

 # 2146 DOUBLE-STRUCK ITALIC SMALL D
 # ->   0064 LATIN SMALL LETTER D
 "\x{2146}" => "d",

 # 2147 DOUBLE-STRUCK ITALIC SMALL E
 # ->   0065 LATIN SMALL LETTER E
 "\x{2147}" => "e",

 # 2148 DOUBLE-STRUCK ITALIC SMALL I
 # ->   0069 LATIN SMALL LETTER I
 "\x{2148}" => "i",

 # 2149 DOUBLE-STRUCK ITALIC SMALL J
 # ->   006A LATIN SMALL LETTER J
 "\x{2149}" => "j",

 # 2153 VULGAR FRACTION ONE THIRD
 # ->   0031 DIGIT ONE
 #    + 2044 FRACTION SLASH
 #    + 0033 DIGIT THREE
 "\x{2153}" => "1\x{2044}3",

 # 2154 VULGAR FRACTION TWO THIRDS
 # ->   0032 DIGIT TWO
 #    + 2044 FRACTION SLASH
 #    + 0033 DIGIT THREE
 "\x{2154}" => "2\x{2044}3",

 # 2155 VULGAR FRACTION ONE FIFTH
 # ->   0031 DIGIT ONE
 #    + 2044 FRACTION SLASH
 #    + 0035 DIGIT FIVE
 "\x{2155}" => "1\x{2044}5",

 # 2156 VULGAR FRACTION TWO FIFTHS
 # ->   0032 DIGIT TWO
 #    + 2044 FRACTION SLASH
 #    + 0035 DIGIT FIVE
 "\x{2156}" => "2\x{2044}5",

 # 2157 VULGAR FRACTION THREE FIFTHS
 # ->   0033 DIGIT THREE
 #    + 2044 FRACTION SLASH
 #    + 0035 DIGIT FIVE
 "\x{2157}" => "3\x{2044}5",

 # 2158 VULGAR FRACTION FOUR FIFTHS
 # ->   0034 DIGIT FOUR
 #    + 2044 FRACTION SLASH
 #    + 0035 DIGIT FIVE
 "\x{2158}" => "4\x{2044}5",

 # 2159 VULGAR FRACTION ONE SIXTH
 # ->   0031 DIGIT ONE
 #    + 2044 FRACTION SLASH
 #    + 0036 DIGIT SIX
 "\x{2159}" => "1\x{2044}6",

 # 215A VULGAR FRACTION FIVE SIXTHS
 # ->   0035 DIGIT FIVE
 #    + 2044 FRACTION SLASH
 #    + 0036 DIGIT SIX
 "\x{215A}" => "5\x{2044}6",

 # 215B VULGAR FRACTION ONE EIGHTH
 # ->   0031 DIGIT ONE
 #    + 2044 FRACTION SLASH
 #    + 0038 DIGIT EIGHT
 "\x{215B}" => "1\x{2044}8",

 # 215C VULGAR FRACTION THREE EIGHTHS
 # ->   0033 DIGIT THREE
 #    + 2044 FRACTION SLASH
 #    + 0038 DIGIT EIGHT
 "\x{215C}" => "3\x{2044}8",

 # 215D VULGAR FRACTION FIVE EIGHTHS
 # ->   0035 DIGIT FIVE
 #    + 2044 FRACTION SLASH
 #    + 0038 DIGIT EIGHT
 "\x{215D}" => "5\x{2044}8",

 # 215E VULGAR FRACTION SEVEN EIGHTHS
 # ->   0037 DIGIT SEVEN
 #    + 2044 FRACTION SLASH
 #    + 0038 DIGIT EIGHT
 "\x{215E}" => "7\x{2044}8",

 # 215F FRACTION NUMERATOR ONE
 # ->   0031 DIGIT ONE
 #    + 2044 FRACTION SLASH
 "\x{215F}" => "1\x{2044}",

 # 2160 ROMAN NUMERAL ONE
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{2160}" => "I",

 # 2161 ROMAN NUMERAL TWO
 # ->   0049 LATIN CAPITAL LETTER I
 #    + 0049 LATIN CAPITAL LETTER I
 "\x{2161}" => "II",

 # 2162 ROMAN NUMERAL THREE
 # ->   0049 LATIN CAPITAL LETTER I
 #    + 0049 LATIN CAPITAL LETTER I
 #    + 0049 LATIN CAPITAL LETTER I
 "\x{2162}" => "III",

 # 2163 ROMAN NUMERAL FOUR
 # ->   0049 LATIN CAPITAL LETTER I
 #    + 0056 LATIN CAPITAL LETTER V
 "\x{2163}" => "IV",

 # 2164 ROMAN NUMERAL FIVE
 # ->   0056 LATIN CAPITAL LETTER V
 "\x{2164}" => "V",

 # 2165 ROMAN NUMERAL SIX
 # ->   0056 LATIN CAPITAL LETTER V
 #    + 0049 LATIN CAPITAL LETTER I
 "\x{2165}" => "VI",

 # 2166 ROMAN NUMERAL SEVEN
 # ->   0056 LATIN CAPITAL LETTER V
 #    + 0049 LATIN CAPITAL LETTER I
 #    + 0049 LATIN CAPITAL LETTER I
 "\x{2166}" => "VII",

 # 2167 ROMAN NUMERAL EIGHT
 # ->   0056 LATIN CAPITAL LETTER V
 #    + 0049 LATIN CAPITAL LETTER I
 #    + 0049 LATIN CAPITAL LETTER I
 #    + 0049 LATIN CAPITAL LETTER I
 "\x{2167}" => "VIII",

 # 2168 ROMAN NUMERAL NINE
 # ->   0049 LATIN CAPITAL LETTER I
 #    + 0058 LATIN CAPITAL LETTER X
 "\x{2168}" => "IX",

 # 2169 ROMAN NUMERAL TEN
 # ->   0058 LATIN CAPITAL LETTER X
 "\x{2169}" => "X",

 # 216A ROMAN NUMERAL ELEVEN
 # ->   0058 LATIN CAPITAL LETTER X
 #    + 0049 LATIN CAPITAL LETTER I
 "\x{216A}" => "XI",

 # 216B ROMAN NUMERAL TWELVE
 # ->   0058 LATIN CAPITAL LETTER X
 #    + 0049 LATIN CAPITAL LETTER I
 #    + 0049 LATIN CAPITAL LETTER I
 "\x{216B}" => "XII",

 # 216C ROMAN NUMERAL FIFTY
 # ->   004C LATIN CAPITAL LETTER L
 "\x{216C}" => "L",

 # 216D ROMAN NUMERAL ONE HUNDRED
 # ->   0043 LATIN CAPITAL LETTER C
 "\x{216D}" => "C",

 # 216E ROMAN NUMERAL FIVE HUNDRED
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{216E}" => "D",

 # 216F ROMAN NUMERAL ONE THOUSAND
 # ->   004D LATIN CAPITAL LETTER M
 "\x{216F}" => "M",

 # 2170 SMALL ROMAN NUMERAL ONE
 # ->   0069 LATIN SMALL LETTER I
 "\x{2170}" => "i",

 # 2171 SMALL ROMAN NUMERAL TWO
 # ->   0069 LATIN SMALL LETTER I
 #    + 0069 LATIN SMALL LETTER I
 "\x{2171}" => "ii",

 # 2172 SMALL ROMAN NUMERAL THREE
 # ->   0069 LATIN SMALL LETTER I
 #    + 0069 LATIN SMALL LETTER I
 #    + 0069 LATIN SMALL LETTER I
 "\x{2172}" => "iii",

 # 2173 SMALL ROMAN NUMERAL FOUR
 # ->   0069 LATIN SMALL LETTER I
 #    + 0076 LATIN SMALL LETTER V
 "\x{2173}" => "iv",

 # 2174 SMALL ROMAN NUMERAL FIVE
 # ->   0076 LATIN SMALL LETTER V
 "\x{2174}" => "v",

 # 2175 SMALL ROMAN NUMERAL SIX
 # ->   0076 LATIN SMALL LETTER V
 #    + 0069 LATIN SMALL LETTER I
 "\x{2175}" => "vi",

 # 2176 SMALL ROMAN NUMERAL SEVEN
 # ->   0076 LATIN SMALL LETTER V
 #    + 0069 LATIN SMALL LETTER I
 #    + 0069 LATIN SMALL LETTER I
 "\x{2176}" => "vii",

 # 2177 SMALL ROMAN NUMERAL EIGHT
 # ->   0076 LATIN SMALL LETTER V
 #    + 0069 LATIN SMALL LETTER I
 #    + 0069 LATIN SMALL LETTER I
 #    + 0069 LATIN SMALL LETTER I
 "\x{2177}" => "viii",

 # 2178 SMALL ROMAN NUMERAL NINE
 # ->   0069 LATIN SMALL LETTER I
 #    + 0078 LATIN SMALL LETTER X
 "\x{2178}" => "ix",

 # 2179 SMALL ROMAN NUMERAL TEN
 # ->   0078 LATIN SMALL LETTER X
 "\x{2179}" => "x",

 # 217A SMALL ROMAN NUMERAL ELEVEN
 # ->   0078 LATIN SMALL LETTER X
 #    + 0069 LATIN SMALL LETTER I
 "\x{217A}" => "xi",

 # 217B SMALL ROMAN NUMERAL TWELVE
 # ->   0078 LATIN SMALL LETTER X
 #    + 0069 LATIN SMALL LETTER I
 #    + 0069 LATIN SMALL LETTER I
 "\x{217B}" => "xii",

 # 217C SMALL ROMAN NUMERAL FIFTY
 # ->   006C LATIN SMALL LETTER L
 "\x{217C}" => "l",

 # 217D SMALL ROMAN NUMERAL ONE HUNDRED
 # ->   0063 LATIN SMALL LETTER C
 "\x{217D}" => "c",

 # 217E SMALL ROMAN NUMERAL FIVE HUNDRED
 # ->   0064 LATIN SMALL LETTER D
 "\x{217E}" => "d",

 # 217F SMALL ROMAN NUMERAL ONE THOUSAND
 # ->   006D LATIN SMALL LETTER M
 "\x{217F}" => "m",

 # 219A LEFTWARDS ARROW WITH STROKE
 # ->   2190 LEFTWARDS ARROW
 "\x{219A}" => "\x{2190}",

 # 219B RIGHTWARDS ARROW WITH STROKE
 # ->   2192 RIGHTWARDS ARROW
 "\x{219B}" => "\x{2192}",

 # 21A2 LEFTWARDS ARROW WITH TAIL
 # ->   2190 LEFTWARDS ARROW
 "\x{21A2}" => "\x{2190}",

 # 21A3 RIGHTWARDS ARROW WITH TAIL
 # ->   2192 RIGHTWARDS ARROW
 "\x{21A3}" => "\x{2192}",

 # 21A8 UP DOWN ARROW WITH BASE
 # ->   2195 UP DOWN ARROW
 "\x{21A8}" => "\x{2195}",

 # 21A9 LEFTWARDS ARROW WITH HOOK
 # ->   2190 LEFTWARDS ARROW
 "\x{21A9}" => "\x{2190}",

 # 21AA RIGHTWARDS ARROW WITH HOOK
 # ->   2192 RIGHTWARDS ARROW
 "\x{21AA}" => "\x{2192}",

 # 21AB LEFTWARDS ARROW WITH LOOP
 # ->   2190 LEFTWARDS ARROW
 "\x{21AB}" => "\x{2190}",

 # 21AC RIGHTWARDS ARROW WITH LOOP
 # ->   2192 RIGHTWARDS ARROW
 "\x{21AC}" => "\x{2192}",

 # 21AE LEFT RIGHT ARROW WITH STROKE
 # ->   2194 LEFT RIGHT ARROW
 "\x{21AE}" => "\x{2194}",

 # 21B0 UPWARDS ARROW WITH TIP LEFTWARDS
 # ->   2191 UPWARDS ARROW
 "\x{21B0}" => "\x{2191}",

 # 21B1 UPWARDS ARROW WITH TIP RIGHTWARDS
 # ->   2191 UPWARDS ARROW
 "\x{21B1}" => "\x{2191}",

 # 21B2 DOWNWARDS ARROW WITH TIP LEFTWARDS
 # ->   2193 DOWNWARDS ARROW
 "\x{21B2}" => "\x{2193}",

 # 21B3 DOWNWARDS ARROW WITH TIP RIGHTWARDS
 # ->   2193 DOWNWARDS ARROW
 "\x{21B3}" => "\x{2193}",

 # 21B4 RIGHTWARDS ARROW WITH CORNER DOWNWARDS
 # ->   2192 RIGHTWARDS ARROW
 "\x{21B4}" => "\x{2192}",

 # 21B5 DOWNWARDS ARROW WITH CORNER LEFTWARDS
 # ->   2193 DOWNWARDS ARROW
 "\x{21B5}" => "\x{2193}",

 # 21CD LEFTWARDS DOUBLE ARROW WITH STROKE
 # ->   21D0 LEFTWARDS DOUBLE ARROW
 "\x{21CD}" => "\x{21D0}",

 # 21CE LEFT RIGHT DOUBLE ARROW WITH STROKE
 # ->   21D4 LEFT RIGHT DOUBLE ARROW
 "\x{21CE}" => "\x{21D4}",

 # 21CF RIGHTWARDS DOUBLE ARROW WITH STROKE
 # ->   21D2 RIGHTWARDS DOUBLE ARROW
 "\x{21CF}" => "\x{21D2}",

 # 21DE UPWARDS ARROW WITH DOUBLE STROKE
 # ->   2191 UPWARDS ARROW
 "\x{21DE}" => "\x{2191}",

 # 21DF DOWNWARDS ARROW WITH DOUBLE STROKE
 # ->   2193 DOWNWARDS ARROW
 "\x{21DF}" => "\x{2193}",

 # 21EC UPWARDS WHITE ARROW ON PEDESTAL WITH HORIZONTAL BAR
 # ->   21EB UPWARDS WHITE ARROW ON PEDESTAL
 "\x{21EC}" => "\x{21EB}",

 # 21ED UPWARDS WHITE ARROW ON PEDESTAL WITH VERTICAL BAR
 # ->   21EB UPWARDS WHITE ARROW ON PEDESTAL
 "\x{21ED}" => "\x{21EB}",

 # 21F7 LEFTWARDS ARROW WITH VERTICAL STROKE
 # ->   2190 LEFTWARDS ARROW
 "\x{21F7}" => "\x{2190}",

 # 21F8 RIGHTWARDS ARROW WITH VERTICAL STROKE
 # ->   2192 RIGHTWARDS ARROW
 "\x{21F8}" => "\x{2192}",

 # 21F9 LEFT RIGHT ARROW WITH VERTICAL STROKE
 # ->   2194 LEFT RIGHT ARROW
 "\x{21F9}" => "\x{2194}",

 # 21FA LEFTWARDS ARROW WITH DOUBLE VERTICAL STROKE
 # ->   2190 LEFTWARDS ARROW
 "\x{21FA}" => "\x{2190}",

 # 21FB RIGHTWARDS ARROW WITH DOUBLE VERTICAL STROKE
 # ->   2192 RIGHTWARDS ARROW
 "\x{21FB}" => "\x{2192}",

 # 21FC LEFT RIGHT ARROW WITH DOUBLE VERTICAL STROKE
 # ->   2194 LEFT RIGHT ARROW
 "\x{21FC}" => "\x{2194}",

 # 2204 THERE DOES NOT EXIST
 # ->   2203 THERE EXISTS
 "\x{2204}" => "\x{2203}",

 # 2209 NOT AN ELEMENT OF
 # ->   2208 ELEMENT OF
 "\x{2209}" => "\x{2208}",

 # 220C DOES NOT CONTAIN AS MEMBER
 # ->   220B CONTAINS AS MEMBER
 "\x{220C}" => "\x{220B}",

 # 2224 DOES NOT DIVIDE
 # ->   2223 DIVIDES
 "\x{2224}" => "\x{2223}",

 # 2226 NOT PARALLEL TO
 # ->   2225 PARALLEL TO
 "\x{2226}" => "\x{2225}",

 # 222C DOUBLE INTEGRAL
 # ->   222B INTEGRAL
 #    + 222B INTEGRAL
 "\x{222C}" => "\x{222B}\x{222B}",

 # 222D TRIPLE INTEGRAL
 # ->   222B INTEGRAL
 #    + 222B INTEGRAL
 #    + 222B INTEGRAL
 "\x{222D}" => "\x{222B}\x{222B}\x{222B}",

 # 222F SURFACE INTEGRAL
 # ->   222E CONTOUR INTEGRAL
 #    + 222E CONTOUR INTEGRAL
 "\x{222F}" => "\x{222E}\x{222E}",

 # 2230 VOLUME INTEGRAL
 # ->   222E CONTOUR INTEGRAL
 #    + 222E CONTOUR INTEGRAL
 #    + 222E CONTOUR INTEGRAL
 "\x{2230}" => "\x{222E}\x{222E}\x{222E}",

 # 2241 NOT TILDE
 # ->   223C TILDE OPERATOR
 "\x{2241}" => "\x{223C}",

 # 2244 NOT ASYMPTOTICALLY EQUAL TO
 # ->   2243 ASYMPTOTICALLY EQUAL TO
 "\x{2244}" => "\x{2243}",

 # 2247 NEITHER APPROXIMATELY NOR ACTUALLY EQUAL TO
 # ->   2245 APPROXIMATELY EQUAL TO
 "\x{2247}" => "\x{2245}",

 # 2249 NOT ALMOST EQUAL TO
 # ->   2248 ALMOST EQUAL TO
 "\x{2249}" => "\x{2248}",

 # 2260 NOT EQUAL TO
 # ->   003D EQUALS SIGN
 "\x{2260}" => "=",

 # 2262 NOT IDENTICAL TO
 # ->   2261 IDENTICAL TO
 "\x{2262}" => "\x{2261}",

 # 226D NOT EQUIVALENT TO
 # ->   224D EQUIVALENT TO
 "\x{226D}" => "\x{224D}",

 # 226E NOT LESS-THAN
 # ->   003C LESS-THAN SIGN
 "\x{226E}" => "<",

 # 226F NOT GREATER-THAN
 # ->   003E GREATER-THAN SIGN
 "\x{226F}" => ">",

 # 2270 NEITHER LESS-THAN NOR EQUAL TO
 # ->   2264 LESS-THAN OR EQUAL TO
 "\x{2270}" => "\x{2264}",

 # 2271 NEITHER GREATER-THAN NOR EQUAL TO
 # ->   2265 GREATER-THAN OR EQUAL TO
 "\x{2271}" => "\x{2265}",

 # 2274 NEITHER LESS-THAN NOR EQUIVALENT TO
 # ->   2272 LESS-THAN OR EQUIVALENT TO
 "\x{2274}" => "\x{2272}",

 # 2275 NEITHER GREATER-THAN NOR EQUIVALENT TO
 # ->   2273 GREATER-THAN OR EQUIVALENT TO
 "\x{2275}" => "\x{2273}",

 # 2278 NEITHER LESS-THAN NOR GREATER-THAN
 # ->   2276 LESS-THAN OR GREATER-THAN
 "\x{2278}" => "\x{2276}",

 # 2279 NEITHER GREATER-THAN NOR LESS-THAN
 # ->   2277 GREATER-THAN OR LESS-THAN
 "\x{2279}" => "\x{2277}",

 # 2280 DOES NOT PRECEDE
 # ->   227A PRECEDES
 "\x{2280}" => "\x{227A}",

 # 2281 DOES NOT SUCCEED
 # ->   227B SUCCEEDS
 "\x{2281}" => "\x{227B}",

 # 2284 NOT A SUBSET OF
 # ->   2282 SUBSET OF
 "\x{2284}" => "\x{2282}",

 # 2285 NOT A SUPERSET OF
 # ->   2283 SUPERSET OF
 "\x{2285}" => "\x{2283}",

 # 2288 NEITHER A SUBSET OF NOR EQUAL TO
 # ->   2286 SUBSET OF OR EQUAL TO
 "\x{2288}" => "\x{2286}",

 # 2289 NEITHER A SUPERSET OF NOR EQUAL TO
 # ->   2287 SUPERSET OF OR EQUAL TO
 "\x{2289}" => "\x{2287}",

 # 228A SUBSET OF WITH NOT EQUAL TO
 # ->   2282 SUBSET OF
 "\x{228A}" => "\x{2282}",

 # 228B SUPERSET OF WITH NOT EQUAL TO
 # ->   2283 SUPERSET OF
 "\x{228B}" => "\x{2283}",

 # 22AC DOES NOT PROVE
 # ->   22A2 RIGHT TACK
 "\x{22AC}" => "\x{22A2}",

 # 22AD NOT TRUE
 # ->   22A8 TRUE
 "\x{22AD}" => "\x{22A8}",

 # 22AE DOES NOT FORCE
 # ->   22A9 FORCES
 "\x{22AE}" => "\x{22A9}",

 # 22AF NEGATED DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE
 # ->   22AB DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE
 "\x{22AF}" => "\x{22AB}",

 # 22BE RIGHT ANGLE WITH ARC
 # ->   221F RIGHT ANGLE
 "\x{22BE}" => "\x{221F}",

 # 22E0 DOES NOT PRECEDE OR EQUAL
 # ->   227C PRECEDES OR EQUAL TO
 "\x{22E0}" => "\x{227C}",

 # 22E1 DOES NOT SUCCEED OR EQUAL
 # ->   227D SUCCEEDS OR EQUAL TO
 "\x{22E1}" => "\x{227D}",

 # 22E2 NOT SQUARE IMAGE OF OR EQUAL TO
 # ->   2291 SQUARE IMAGE OF OR EQUAL TO
 "\x{22E2}" => "\x{2291}",

 # 22E3 NOT SQUARE ORIGINAL OF OR EQUAL TO
 # ->   2292 SQUARE ORIGINAL OF OR EQUAL TO
 "\x{22E3}" => "\x{2292}",

 # 22EA NOT NORMAL SUBGROUP OF
 # ->   22B2 NORMAL SUBGROUP OF
 "\x{22EA}" => "\x{22B2}",

 # 22EB DOES NOT CONTAIN AS NORMAL SUBGROUP
 # ->   22B3 CONTAINS AS NORMAL SUBGROUP
 "\x{22EB}" => "\x{22B3}",

 # 22EC NOT NORMAL SUBGROUP OF OR EQUAL TO
 # ->   22B4 NORMAL SUBGROUP OF OR EQUAL TO
 "\x{22EC}" => "\x{22B4}",

 # 22ED DOES NOT CONTAIN AS NORMAL SUBGROUP OR EQUAL
 # ->   22B5 CONTAINS AS NORMAL SUBGROUP OR EQUAL TO
 "\x{22ED}" => "\x{22B5}",

 # 22F2 ELEMENT OF WITH LONG HORIZONTAL STROKE
 # ->   2208 ELEMENT OF
 "\x{22F2}" => "\x{2208}",

 # 22F3 ELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKE
 # ->   2208 ELEMENT OF
 "\x{22F3}" => "\x{2208}",

 # 22F4 SMALL ELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKE
 # ->   220A SMALL ELEMENT OF
 "\x{22F4}" => "\x{220A}",

 # 22F5 ELEMENT OF WITH DOT ABOVE
 # ->   2208 ELEMENT OF
 "\x{22F5}" => "\x{2208}",

 # 22F6 ELEMENT OF WITH OVERBAR
 # ->   2208 ELEMENT OF
 "\x{22F6}" => "\x{2208}",

 # 22F7 SMALL ELEMENT OF WITH OVERBAR
 # ->   220A SMALL ELEMENT OF
 "\x{22F7}" => "\x{220A}",

 # 22F8 ELEMENT OF WITH UNDERBAR
 # ->   2208 ELEMENT OF
 "\x{22F8}" => "\x{2208}",

 # 22F9 ELEMENT OF WITH TWO HORIZONTAL STROKES
 # ->   2208 ELEMENT OF
 "\x{22F9}" => "\x{2208}",

 # 2329 LEFT-POINTING ANGLE BRACKET
 # ->   3008 LEFT ANGLE BRACKET
 "\x{2329}" => "\x{3008}",

 # 232A RIGHT-POINTING ANGLE BRACKET
 # ->   3009 RIGHT ANGLE BRACKET
 "\x{232A}" => "\x{3009}",

 # 237C RIGHT ANGLE WITH DOWNWARDS ZIGZAG ARROW
 # ->   221F RIGHT ANGLE
 "\x{237C}" => "\x{221F}",

 # 237F VERTICAL LINE WITH MIDDLE DOT
 # ->   007C VERTICAL LINE
 "\x{237F}" => "|",

 # 2385 WHITE SQUARE WITH CENTRE VERTICAL LINE
 # ->   25A1 WHITE SQUARE
 "\x{2385}" => "\x{25A1}",

 # 23C1 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH CIRCLE
 # ->   23C9 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL
 "\x{23C1}" => "\x{23C9}",

 # 23C2 DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH CIRCLE
 # ->   23CA DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL
 "\x{23C2}" => "\x{23CA}",

 # 23C4 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH TRIANGLE
 # ->   23C9 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL
 "\x{23C4}" => "\x{23C9}",

 # 23C5 DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH TRIANGLE
 # ->   23CA DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL
 "\x{23C5}" => "\x{23CA}",

 # 23C7 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH WAVE
 # ->   23C9 DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL
 "\x{23C7}" => "\x{23C9}",

 # 23C8 DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH WAVE
 # ->   23CA DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL
 "\x{23C8}" => "\x{23CA}",

 # 2460 CIRCLED DIGIT ONE
 # ->   0031 DIGIT ONE
 "\x{2460}" => "1",

 # 2461 CIRCLED DIGIT TWO
 # ->   0032 DIGIT TWO
 "\x{2461}" => "2",

 # 2462 CIRCLED DIGIT THREE
 # ->   0033 DIGIT THREE
 "\x{2462}" => "3",

 # 2463 CIRCLED DIGIT FOUR
 # ->   0034 DIGIT FOUR
 "\x{2463}" => "4",

 # 2464 CIRCLED DIGIT FIVE
 # ->   0035 DIGIT FIVE
 "\x{2464}" => "5",

 # 2465 CIRCLED DIGIT SIX
 # ->   0036 DIGIT SIX
 "\x{2465}" => "6",

 # 2466 CIRCLED DIGIT SEVEN
 # ->   0037 DIGIT SEVEN
 "\x{2466}" => "7",

 # 2467 CIRCLED DIGIT EIGHT
 # ->   0038 DIGIT EIGHT
 "\x{2467}" => "8",

 # 2468 CIRCLED DIGIT NINE
 # ->   0039 DIGIT NINE
 "\x{2468}" => "9",

 # 2469 CIRCLED NUMBER TEN
 # ->   0031 DIGIT ONE
 #    + 0030 DIGIT ZERO
 "\x{2469}" => "10",

 # 246A CIRCLED NUMBER ELEVEN
 # ->   0031 DIGIT ONE
 #    + 0031 DIGIT ONE
 "\x{246A}" => "11",

 # 246B CIRCLED NUMBER TWELVE
 # ->   0031 DIGIT ONE
 #    + 0032 DIGIT TWO
 "\x{246B}" => "12",

 # 246C CIRCLED NUMBER THIRTEEN
 # ->   0031 DIGIT ONE
 #    + 0033 DIGIT THREE
 "\x{246C}" => "13",

 # 246D CIRCLED NUMBER FOURTEEN
 # ->   0031 DIGIT ONE
 #    + 0034 DIGIT FOUR
 "\x{246D}" => "14",

 # 246E CIRCLED NUMBER FIFTEEN
 # ->   0031 DIGIT ONE
 #    + 0035 DIGIT FIVE
 "\x{246E}" => "15",

 # 246F CIRCLED NUMBER SIXTEEN
 # ->   0031 DIGIT ONE
 #    + 0036 DIGIT SIX
 "\x{246F}" => "16",

 # 2470 CIRCLED NUMBER SEVENTEEN
 # ->   0031 DIGIT ONE
 #    + 0037 DIGIT SEVEN
 "\x{2470}" => "17",

 # 2471 CIRCLED NUMBER EIGHTEEN
 # ->   0031 DIGIT ONE
 #    + 0038 DIGIT EIGHT
 "\x{2471}" => "18",

 # 2472 CIRCLED NUMBER NINETEEN
 # ->   0031 DIGIT ONE
 #    + 0039 DIGIT NINE
 "\x{2472}" => "19",

 # 2473 CIRCLED NUMBER TWENTY
 # ->   0032 DIGIT TWO
 #    + 0030 DIGIT ZERO
 "\x{2473}" => "20",

 # 2474 PARENTHESIZED DIGIT ONE
 # ->   0028 LEFT PARENTHESIS
 #    + 0031 DIGIT ONE
 #    + 0029 RIGHT PARENTHESIS
 "\x{2474}" => "(1)",

 # 2475 PARENTHESIZED DIGIT TWO
 # ->   0028 LEFT PARENTHESIS
 #    + 0032 DIGIT TWO
 #    + 0029 RIGHT PARENTHESIS
 "\x{2475}" => "(2)",

 # 2476 PARENTHESIZED DIGIT THREE
 # ->   0028 LEFT PARENTHESIS
 #    + 0033 DIGIT THREE
 #    + 0029 RIGHT PARENTHESIS
 "\x{2476}" => "(3)",

 # 2477 PARENTHESIZED DIGIT FOUR
 # ->   0028 LEFT PARENTHESIS
 #    + 0034 DIGIT FOUR
 #    + 0029 RIGHT PARENTHESIS
 "\x{2477}" => "(4)",

 # 2478 PARENTHESIZED DIGIT FIVE
 # ->   0028 LEFT PARENTHESIS
 #    + 0035 DIGIT FIVE
 #    + 0029 RIGHT PARENTHESIS
 "\x{2478}" => "(5)",

 # 2479 PARENTHESIZED DIGIT SIX
 # ->   0028 LEFT PARENTHESIS
 #    + 0036 DIGIT SIX
 #    + 0029 RIGHT PARENTHESIS
 "\x{2479}" => "(6)",

 # 247A PARENTHESIZED DIGIT SEVEN
 # ->   0028 LEFT PARENTHESIS
 #    + 0037 DIGIT SEVEN
 #    + 0029 RIGHT PARENTHESIS
 "\x{247A}" => "(7)",

 # 247B PARENTHESIZED DIGIT EIGHT
 # ->   0028 LEFT PARENTHESIS
 #    + 0038 DIGIT EIGHT
 #    + 0029 RIGHT PARENTHESIS
 "\x{247B}" => "(8)",

 # 247C PARENTHESIZED DIGIT NINE
 # ->   0028 LEFT PARENTHESIS
 #    + 0039 DIGIT NINE
 #    + 0029 RIGHT PARENTHESIS
 "\x{247C}" => "(9)",

 # 247D PARENTHESIZED NUMBER TEN
 # ->   0028 LEFT PARENTHESIS
 #    + 0031 DIGIT ONE
 #    + 0030 DIGIT ZERO
 #    + 0029 RIGHT PARENTHESIS
 "\x{247D}" => "(10)",

 # 247E PARENTHESIZED NUMBER ELEVEN
 # ->   0028 LEFT PARENTHESIS
 #    + 0031 DIGIT ONE
 #    + 0031 DIGIT ONE
 #    + 0029 RIGHT PARENTHESIS
 "\x{247E}" => "(11)",

 # 247F PARENTHESIZED NUMBER TWELVE
 # ->   0028 LEFT PARENTHESIS
 #    + 0031 DIGIT ONE
 #    + 0032 DIGIT TWO
 #    + 0029 RIGHT PARENTHESIS
 "\x{247F}" => "(12)",

 # 2480 PARENTHESIZED NUMBER THIRTEEN
 # ->   0028 LEFT PARENTHESIS
 #    + 0031 DIGIT ONE
 #    + 0033 DIGIT THREE
 #    + 0029 RIGHT PARENTHESIS
 "\x{2480}" => "(13)",

 # 2481 PARENTHESIZED NUMBER FOURTEEN
 # ->   0028 LEFT PARENTHESIS
 #    + 0031 DIGIT ONE
 #    + 0034 DIGIT FOUR
 #    + 0029 RIGHT PARENTHESIS
 "\x{2481}" => "(14)",

 # 2482 PARENTHESIZED NUMBER FIFTEEN
 # ->   0028 LEFT PARENTHESIS
 #    + 0031 DIGIT ONE
 #    + 0035 DIGIT FIVE
 #    + 0029 RIGHT PARENTHESIS
 "\x{2482}" => "(15)",

 # 2483 PARENTHESIZED NUMBER SIXTEEN
 # ->   0028 LEFT PARENTHESIS
 #    + 0031 DIGIT ONE
 #    + 0036 DIGIT SIX
 #    + 0029 RIGHT PARENTHESIS
 "\x{2483}" => "(16)",

 # 2484 PARENTHESIZED NUMBER SEVENTEEN
 # ->   0028 LEFT PARENTHESIS
 #    + 0031 DIGIT ONE
 #    + 0037 DIGIT SEVEN
 #    + 0029 RIGHT PARENTHESIS
 "\x{2484}" => "(17)",

 # 2485 PARENTHESIZED NUMBER EIGHTEEN
 # ->   0028 LEFT PARENTHESIS
 #    + 0031 DIGIT ONE
 #    + 0038 DIGIT EIGHT
 #    + 0029 RIGHT PARENTHESIS
 "\x{2485}" => "(18)",

 # 2486 PARENTHESIZED NUMBER NINETEEN
 # ->   0028 LEFT PARENTHESIS
 #    + 0031 DIGIT ONE
 #    + 0039 DIGIT NINE
 #    + 0029 RIGHT PARENTHESIS
 "\x{2486}" => "(19)",

 # 2487 PARENTHESIZED NUMBER TWENTY
 # ->   0028 LEFT PARENTHESIS
 #    + 0032 DIGIT TWO
 #    + 0030 DIGIT ZERO
 #    + 0029 RIGHT PARENTHESIS
 "\x{2487}" => "(20)",

 # 2488 DIGIT ONE FULL STOP
 # ->   0031 DIGIT ONE
 #    + 002E FULL STOP
 "\x{2488}" => "1.",

 # 2489 DIGIT TWO FULL STOP
 # ->   0032 DIGIT TWO
 #    + 002E FULL STOP
 "\x{2489}" => "2.",

 # 248A DIGIT THREE FULL STOP
 # ->   0033 DIGIT THREE
 #    + 002E FULL STOP
 "\x{248A}" => "3.",

 # 248B DIGIT FOUR FULL STOP
 # ->   0034 DIGIT FOUR
 #    + 002E FULL STOP
 "\x{248B}" => "4.",

 # 248C DIGIT FIVE FULL STOP
 # ->   0035 DIGIT FIVE
 #    + 002E FULL STOP
 "\x{248C}" => "5.",

 # 248D DIGIT SIX FULL STOP
 # ->   0036 DIGIT SIX
 #    + 002E FULL STOP
 "\x{248D}" => "6.",

 # 248E DIGIT SEVEN FULL STOP
 # ->   0037 DIGIT SEVEN
 #    + 002E FULL STOP
 "\x{248E}" => "7.",

 # 248F DIGIT EIGHT FULL STOP
 # ->   0038 DIGIT EIGHT
 #    + 002E FULL STOP
 "\x{248F}" => "8.",

 # 2490 DIGIT NINE FULL STOP
 # ->   0039 DIGIT NINE
 #    + 002E FULL STOP
 "\x{2490}" => "9.",

 # 2491 NUMBER TEN FULL STOP
 # ->   0031 DIGIT ONE
 #    + 0030 DIGIT ZERO
 #    + 002E FULL STOP
 "\x{2491}" => "10.",

 # 2492 NUMBER ELEVEN FULL STOP
 # ->   0031 DIGIT ONE
 #    + 0031 DIGIT ONE
 #    + 002E FULL STOP
 "\x{2492}" => "11.",

 # 2493 NUMBER TWELVE FULL STOP
 # ->   0031 DIGIT ONE
 #    + 0032 DIGIT TWO
 #    + 002E FULL STOP
 "\x{2493}" => "12.",

 # 2494 NUMBER THIRTEEN FULL STOP
 # ->   0031 DIGIT ONE
 #    + 0033 DIGIT THREE
 #    + 002E FULL STOP
 "\x{2494}" => "13.",

 # 2495 NUMBER FOURTEEN FULL STOP
 # ->   0031 DIGIT ONE
 #    + 0034 DIGIT FOUR
 #    + 002E FULL STOP
 "\x{2495}" => "14.",

 # 2496 NUMBER FIFTEEN FULL STOP
 # ->   0031 DIGIT ONE
 #    + 0035 DIGIT FIVE
 #    + 002E FULL STOP
 "\x{2496}" => "15.",

 # 2497 NUMBER SIXTEEN FULL STOP
 # ->   0031 DIGIT ONE
 #    + 0036 DIGIT SIX
 #    + 002E FULL STOP
 "\x{2497}" => "16.",

 # 2498 NUMBER SEVENTEEN FULL STOP
 # ->   0031 DIGIT ONE
 #    + 0037 DIGIT SEVEN
 #    + 002E FULL STOP
 "\x{2498}" => "17.",

 # 2499 NUMBER EIGHTEEN FULL STOP
 # ->   0031 DIGIT ONE
 #    + 0038 DIGIT EIGHT
 #    + 002E FULL STOP
 "\x{2499}" => "18.",

 # 249A NUMBER NINETEEN FULL STOP
 # ->   0031 DIGIT ONE
 #    + 0039 DIGIT NINE
 #    + 002E FULL STOP
 "\x{249A}" => "19.",

 # 249B NUMBER TWENTY FULL STOP
 # ->   0032 DIGIT TWO
 #    + 0030 DIGIT ZERO
 #    + 002E FULL STOP
 "\x{249B}" => "20.",

 # 249C PARENTHESIZED LATIN SMALL LETTER A
 # ->   0028 LEFT PARENTHESIS
 #    + 0061 LATIN SMALL LETTER A
 #    + 0029 RIGHT PARENTHESIS
 "\x{249C}" => "(a)",

 # 249D PARENTHESIZED LATIN SMALL LETTER B
 # ->   0028 LEFT PARENTHESIS
 #    + 0062 LATIN SMALL LETTER B
 #    + 0029 RIGHT PARENTHESIS
 "\x{249D}" => "(b)",

 # 249E PARENTHESIZED LATIN SMALL LETTER C
 # ->   0028 LEFT PARENTHESIS
 #    + 0063 LATIN SMALL LETTER C
 #    + 0029 RIGHT PARENTHESIS
 "\x{249E}" => "(c)",

 # 249F PARENTHESIZED LATIN SMALL LETTER D
 # ->   0028 LEFT PARENTHESIS
 #    + 0064 LATIN SMALL LETTER D
 #    + 0029 RIGHT PARENTHESIS
 "\x{249F}" => "(d)",

 # 24A0 PARENTHESIZED LATIN SMALL LETTER E
 # ->   0028 LEFT PARENTHESIS
 #    + 0065 LATIN SMALL LETTER E
 #    + 0029 RIGHT PARENTHESIS
 "\x{24A0}" => "(e)",

 # 24A1 PARENTHESIZED LATIN SMALL LETTER F
 # ->   0028 LEFT PARENTHESIS
 #    + 0066 LATIN SMALL LETTER F
 #    + 0029 RIGHT PARENTHESIS
 "\x{24A1}" => "(f)",

 # 24A2 PARENTHESIZED LATIN SMALL LETTER G
 # ->   0028 LEFT PARENTHESIS
 #    + 0067 LATIN SMALL LETTER G
 #    + 0029 RIGHT PARENTHESIS
 "\x{24A2}" => "(g)",

 # 24A3 PARENTHESIZED LATIN SMALL LETTER H
 # ->   0028 LEFT PARENTHESIS
 #    + 0068 LATIN SMALL LETTER H
 #    + 0029 RIGHT PARENTHESIS
 "\x{24A3}" => "(h)",

 # 24A4 PARENTHESIZED LATIN SMALL LETTER I
 # ->   0028 LEFT PARENTHESIS
 #    + 0069 LATIN SMALL LETTER I
 #    + 0029 RIGHT PARENTHESIS
 "\x{24A4}" => "(i)",

 # 24A5 PARENTHESIZED LATIN SMALL LETTER J
 # ->   0028 LEFT PARENTHESIS
 #    + 006A LATIN SMALL LETTER J
 #    + 0029 RIGHT PARENTHESIS
 "\x{24A5}" => "(j)",

 # 24A6 PARENTHESIZED LATIN SMALL LETTER K
 # ->   0028 LEFT PARENTHESIS
 #    + 006B LATIN SMALL LETTER K
 #    + 0029 RIGHT PARENTHESIS
 "\x{24A6}" => "(k)",

 # 24A7 PARENTHESIZED LATIN SMALL LETTER L
 # ->   0028 LEFT PARENTHESIS
 #    + 006C LATIN SMALL LETTER L
 #    + 0029 RIGHT PARENTHESIS
 "\x{24A7}" => "(l)",

 # 24A8 PARENTHESIZED LATIN SMALL LETTER M
 # ->   0028 LEFT PARENTHESIS
 #    + 006D LATIN SMALL LETTER M
 #    + 0029 RIGHT PARENTHESIS
 "\x{24A8}" => "(m)",

 # 24A9 PARENTHESIZED LATIN SMALL LETTER N
 # ->   0028 LEFT PARENTHESIS
 #    + 006E LATIN SMALL LETTER N
 #    + 0029 RIGHT PARENTHESIS
 "\x{24A9}" => "(n)",

 # 24AA PARENTHESIZED LATIN SMALL LETTER O
 # ->   0028 LEFT PARENTHESIS
 #    + 006F LATIN SMALL LETTER O
 #    + 0029 RIGHT PARENTHESIS
 "\x{24AA}" => "(o)",

 # 24AB PARENTHESIZED LATIN SMALL LETTER P
 # ->   0028 LEFT PARENTHESIS
 #    + 0070 LATIN SMALL LETTER P
 #    + 0029 RIGHT PARENTHESIS
 "\x{24AB}" => "(p)",

 # 24AC PARENTHESIZED LATIN SMALL LETTER Q
 # ->   0028 LEFT PARENTHESIS
 #    + 0071 LATIN SMALL LETTER Q
 #    + 0029 RIGHT PARENTHESIS
 "\x{24AC}" => "(q)",

 # 24AD PARENTHESIZED LATIN SMALL LETTER R
 # ->   0028 LEFT PARENTHESIS
 #    + 0072 LATIN SMALL LETTER R
 #    + 0029 RIGHT PARENTHESIS
 "\x{24AD}" => "(r)",

 # 24AE PARENTHESIZED LATIN SMALL LETTER S
 # ->   0028 LEFT PARENTHESIS
 #    + 0073 LATIN SMALL LETTER S
 #    + 0029 RIGHT PARENTHESIS
 "\x{24AE}" => "(s)",

 # 24AF PARENTHESIZED LATIN SMALL LETTER T
 # ->   0028 LEFT PARENTHESIS
 #    + 0074 LATIN SMALL LETTER T
 #    + 0029 RIGHT PARENTHESIS
 "\x{24AF}" => "(t)",

 # 24B0 PARENTHESIZED LATIN SMALL LETTER U
 # ->   0028 LEFT PARENTHESIS
 #    + 0075 LATIN SMALL LETTER U
 #    + 0029 RIGHT PARENTHESIS
 "\x{24B0}" => "(u)",

 # 24B1 PARENTHESIZED LATIN SMALL LETTER V
 # ->   0028 LEFT PARENTHESIS
 #    + 0076 LATIN SMALL LETTER V
 #    + 0029 RIGHT PARENTHESIS
 "\x{24B1}" => "(v)",

 # 24B2 PARENTHESIZED LATIN SMALL LETTER W
 # ->   0028 LEFT PARENTHESIS
 #    + 0077 LATIN SMALL LETTER W
 #    + 0029 RIGHT PARENTHESIS
 "\x{24B2}" => "(w)",

 # 24B3 PARENTHESIZED LATIN SMALL LETTER X
 # ->   0028 LEFT PARENTHESIS
 #    + 0078 LATIN SMALL LETTER X
 #    + 0029 RIGHT PARENTHESIS
 "\x{24B3}" => "(x)",

 # 24B4 PARENTHESIZED LATIN SMALL LETTER Y
 # ->   0028 LEFT PARENTHESIS
 #    + 0079 LATIN SMALL LETTER Y
 #    + 0029 RIGHT PARENTHESIS
 "\x{24B4}" => "(y)",

 # 24B5 PARENTHESIZED LATIN SMALL LETTER Z
 # ->   0028 LEFT PARENTHESIS
 #    + 007A LATIN SMALL LETTER Z
 #    + 0029 RIGHT PARENTHESIS
 "\x{24B5}" => "(z)",

 # 24B6 CIRCLED LATIN CAPITAL LETTER A
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{24B6}" => "A",

 # 24B7 CIRCLED LATIN CAPITAL LETTER B
 # ->   0042 LATIN CAPITAL LETTER B
 "\x{24B7}" => "B",

 # 24B8 CIRCLED LATIN CAPITAL LETTER C
 # ->   0043 LATIN CAPITAL LETTER C
 "\x{24B8}" => "C",

 # 24B9 CIRCLED LATIN CAPITAL LETTER D
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{24B9}" => "D",

 # 24BA CIRCLED LATIN CAPITAL LETTER E
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{24BA}" => "E",

 # 24BB CIRCLED LATIN CAPITAL LETTER F
 # ->   0046 LATIN CAPITAL LETTER F
 "\x{24BB}" => "F",

 # 24BC CIRCLED LATIN CAPITAL LETTER G
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{24BC}" => "G",

 # 24BD CIRCLED LATIN CAPITAL LETTER H
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{24BD}" => "H",

 # 24BE CIRCLED LATIN CAPITAL LETTER I
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{24BE}" => "I",

 # 24BF CIRCLED LATIN CAPITAL LETTER J
 # ->   004A LATIN CAPITAL LETTER J
 "\x{24BF}" => "J",

 # 24C0 CIRCLED LATIN CAPITAL LETTER K
 # ->   004B LATIN CAPITAL LETTER K
 "\x{24C0}" => "K",

 # 24C1 CIRCLED LATIN CAPITAL LETTER L
 # ->   004C LATIN CAPITAL LETTER L
 "\x{24C1}" => "L",

 # 24C2 CIRCLED LATIN CAPITAL LETTER M
 # ->   004D LATIN CAPITAL LETTER M
 "\x{24C2}" => "M",

 # 24C3 CIRCLED LATIN CAPITAL LETTER N
 # ->   004E LATIN CAPITAL LETTER N
 "\x{24C3}" => "N",

 # 24C4 CIRCLED LATIN CAPITAL LETTER O
 # ->   004F LATIN CAPITAL LETTER O
 "\x{24C4}" => "O",

 # 24C5 CIRCLED LATIN CAPITAL LETTER P
 # ->   0050 LATIN CAPITAL LETTER P
 "\x{24C5}" => "P",

 # 24C6 CIRCLED LATIN CAPITAL LETTER Q
 # ->   0051 LATIN CAPITAL LETTER Q
 "\x{24C6}" => "Q",

 # 24C7 CIRCLED LATIN CAPITAL LETTER R
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{24C7}" => "R",

 # 24C8 CIRCLED LATIN CAPITAL LETTER S
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{24C8}" => "S",

 # 24C9 CIRCLED LATIN CAPITAL LETTER T
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{24C9}" => "T",

 # 24CA CIRCLED LATIN CAPITAL LETTER U
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{24CA}" => "U",

 # 24CB CIRCLED LATIN CAPITAL LETTER V
 # ->   0056 LATIN CAPITAL LETTER V
 "\x{24CB}" => "V",

 # 24CC CIRCLED LATIN CAPITAL LETTER W
 # ->   0057 LATIN CAPITAL LETTER W
 "\x{24CC}" => "W",

 # 24CD CIRCLED LATIN CAPITAL LETTER X
 # ->   0058 LATIN CAPITAL LETTER X
 "\x{24CD}" => "X",

 # 24CE CIRCLED LATIN CAPITAL LETTER Y
 # ->   0059 LATIN CAPITAL LETTER Y
 "\x{24CE}" => "Y",

 # 24CF CIRCLED LATIN CAPITAL LETTER Z
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{24CF}" => "Z",

 # 24D0 CIRCLED LATIN SMALL LETTER A
 # ->   0061 LATIN SMALL LETTER A
 "\x{24D0}" => "a",

 # 24D1 CIRCLED LATIN SMALL LETTER B
 # ->   0062 LATIN SMALL LETTER B
 "\x{24D1}" => "b",

 # 24D2 CIRCLED LATIN SMALL LETTER C
 # ->   0063 LATIN SMALL LETTER C
 "\x{24D2}" => "c",

 # 24D3 CIRCLED LATIN SMALL LETTER D
 # ->   0064 LATIN SMALL LETTER D
 "\x{24D3}" => "d",

 # 24D4 CIRCLED LATIN SMALL LETTER E
 # ->   0065 LATIN SMALL LETTER E
 "\x{24D4}" => "e",

 # 24D5 CIRCLED LATIN SMALL LETTER F
 # ->   0066 LATIN SMALL LETTER F
 "\x{24D5}" => "f",

 # 24D6 CIRCLED LATIN SMALL LETTER G
 # ->   0067 LATIN SMALL LETTER G
 "\x{24D6}" => "g",

 # 24D7 CIRCLED LATIN SMALL LETTER H
 # ->   0068 LATIN SMALL LETTER H
 "\x{24D7}" => "h",

 # 24D8 CIRCLED LATIN SMALL LETTER I
 # ->   0069 LATIN SMALL LETTER I
 "\x{24D8}" => "i",

 # 24D9 CIRCLED LATIN SMALL LETTER J
 # ->   006A LATIN SMALL LETTER J
 "\x{24D9}" => "j",

 # 24DA CIRCLED LATIN SMALL LETTER K
 # ->   006B LATIN SMALL LETTER K
 "\x{24DA}" => "k",

 # 24DB CIRCLED LATIN SMALL LETTER L
 # ->   006C LATIN SMALL LETTER L
 "\x{24DB}" => "l",

 # 24DC CIRCLED LATIN SMALL LETTER M
 # ->   006D LATIN SMALL LETTER M
 "\x{24DC}" => "m",

 # 24DD CIRCLED LATIN SMALL LETTER N
 # ->   006E LATIN SMALL LETTER N
 "\x{24DD}" => "n",

 # 24DE CIRCLED LATIN SMALL LETTER O
 # ->   006F LATIN SMALL LETTER O
 "\x{24DE}" => "o",

 # 24DF CIRCLED LATIN SMALL LETTER P
 # ->   0070 LATIN SMALL LETTER P
 "\x{24DF}" => "p",

 # 24E0 CIRCLED LATIN SMALL LETTER Q
 # ->   0071 LATIN SMALL LETTER Q
 "\x{24E0}" => "q",

 # 24E1 CIRCLED LATIN SMALL LETTER R
 # ->   0072 LATIN SMALL LETTER R
 "\x{24E1}" => "r",

 # 24E2 CIRCLED LATIN SMALL LETTER S
 # ->   0073 LATIN SMALL LETTER S
 "\x{24E2}" => "s",

 # 24E3 CIRCLED LATIN SMALL LETTER T
 # ->   0074 LATIN SMALL LETTER T
 "\x{24E3}" => "t",

 # 24E4 CIRCLED LATIN SMALL LETTER U
 # ->   0075 LATIN SMALL LETTER U
 "\x{24E4}" => "u",

 # 24E5 CIRCLED LATIN SMALL LETTER V
 # ->   0076 LATIN SMALL LETTER V
 "\x{24E5}" => "v",

 # 24E6 CIRCLED LATIN SMALL LETTER W
 # ->   0077 LATIN SMALL LETTER W
 "\x{24E6}" => "w",

 # 24E7 CIRCLED LATIN SMALL LETTER X
 # ->   0078 LATIN SMALL LETTER X
 "\x{24E7}" => "x",

 # 24E8 CIRCLED LATIN SMALL LETTER Y
 # ->   0079 LATIN SMALL LETTER Y
 "\x{24E8}" => "y",

 # 24E9 CIRCLED LATIN SMALL LETTER Z
 # ->   007A LATIN SMALL LETTER Z
 "\x{24E9}" => "z",

 # 24EA CIRCLED DIGIT ZERO
 # ->   0030 DIGIT ZERO
 "\x{24EA}" => "0",

 # 25A2 WHITE SQUARE WITH ROUNDED CORNERS
 # ->   25A1 WHITE SQUARE
 "\x{25A2}" => "\x{25A1}",

 # 25EB WHITE SQUARE WITH VERTICAL BISECTING LINE
 # ->   25A1 WHITE SQUARE
 "\x{25EB}" => "\x{25A1}",

 # 25EC WHITE UP-POINTING TRIANGLE WITH DOT
 # ->   25B3 WHITE UP-POINTING TRIANGLE
 "\x{25EC}" => "\x{25B3}",

 # 25F0 WHITE SQUARE WITH UPPER LEFT QUADRANT
 # ->   25A1 WHITE SQUARE
 "\x{25F0}" => "\x{25A1}",

 # 25F1 WHITE SQUARE WITH LOWER LEFT QUADRANT
 # ->   25A1 WHITE SQUARE
 "\x{25F1}" => "\x{25A1}",

 # 25F2 WHITE SQUARE WITH LOWER RIGHT QUADRANT
 # ->   25A1 WHITE SQUARE
 "\x{25F2}" => "\x{25A1}",

 # 25F3 WHITE SQUARE WITH UPPER RIGHT QUADRANT
 # ->   25A1 WHITE SQUARE
 "\x{25F3}" => "\x{25A1}",

 # 25F4 WHITE CIRCLE WITH UPPER LEFT QUADRANT
 # ->   25CB WHITE CIRCLE
 "\x{25F4}" => "\x{25CB}",

 # 25F5 WHITE CIRCLE WITH LOWER LEFT QUADRANT
 # ->   25CB WHITE CIRCLE
 "\x{25F5}" => "\x{25CB}",

 # 25F6 WHITE CIRCLE WITH LOWER RIGHT QUADRANT
 # ->   25CB WHITE CIRCLE
 "\x{25F6}" => "\x{25CB}",

 # 25F7 WHITE CIRCLE WITH UPPER RIGHT QUADRANT
 # ->   25CB WHITE CIRCLE
 "\x{25F7}" => "\x{25CB}",

 # 2611 BALLOT BOX WITH CHECK
 # ->   2610 BALLOT BOX
 "\x{2611}" => "\x{2610}",

 # 2612 BALLOT BOX WITH X
 # ->   2610 BALLOT BOX
 "\x{2612}" => "\x{2610}",

 # 2614 UMBRELLA WITH RAIN DROPS
 # ->   2602 UMBRELLA
 "\x{2614}" => "\x{2602}",

 # 2686 WHITE CIRCLE WITH DOT RIGHT
 # ->   25CB WHITE CIRCLE
 "\x{2686}" => "\x{25CB}",

 # 2687 WHITE CIRCLE WITH TWO DOTS
 # ->   25CB WHITE CIRCLE
 "\x{2687}" => "\x{25CB}",

 # 2688 BLACK CIRCLE WITH WHITE DOT RIGHT
 # ->   25CF BLACK CIRCLE
 "\x{2688}" => "\x{25CF}",

 # 2689 BLACK CIRCLE WITH TWO WHITE DOTS
 # ->   25CF BLACK CIRCLE
 "\x{2689}" => "\x{25CF}",

 # 27D0 WHITE DIAMOND WITH CENTRED DOT
 # ->   25C7 WHITE DIAMOND
 "\x{27D0}" => "\x{25C7}",

 # 27DF UP TACK WITH CIRCLE ABOVE
 # ->   22A5 UP TACK
 "\x{27DF}" => "\x{22A5}",

 # 27E2 WHITE CONCAVE-SIDED DIAMOND WITH LEFTWARDS TICK
 # ->   27E1 WHITE CONCAVE-SIDED DIAMOND
 "\x{27E2}" => "\x{27E1}",

 # 27E3 WHITE CONCAVE-SIDED DIAMOND WITH RIGHTWARDS TICK
 # ->   27E1 WHITE CONCAVE-SIDED DIAMOND
 "\x{27E3}" => "\x{27E1}",

 # 27E4 WHITE SQUARE WITH LEFTWARDS TICK
 # ->   25A1 WHITE SQUARE
 "\x{27E4}" => "\x{25A1}",

 # 27E5 WHITE SQUARE WITH RIGHTWARDS TICK
 # ->   25A1 WHITE SQUARE
 "\x{27E5}" => "\x{25A1}",

 # 2902 LEFTWARDS DOUBLE ARROW WITH VERTICAL STROKE
 # ->   21D0 LEFTWARDS DOUBLE ARROW
 "\x{2902}" => "\x{21D0}",

 # 2903 RIGHTWARDS DOUBLE ARROW WITH VERTICAL STROKE
 # ->   21D2 RIGHTWARDS DOUBLE ARROW
 "\x{2903}" => "\x{21D2}",

 # 2904 LEFT RIGHT DOUBLE ARROW WITH VERTICAL STROKE
 # ->   21D4 LEFT RIGHT DOUBLE ARROW
 "\x{2904}" => "\x{21D4}",

 # 2908 DOWNWARDS ARROW WITH HORIZONTAL STROKE
 # ->   2193 DOWNWARDS ARROW
 "\x{2908}" => "\x{2193}",

 # 2909 UPWARDS ARROW WITH HORIZONTAL STROKE
 # ->   2191 UPWARDS ARROW
 "\x{2909}" => "\x{2191}",

 # 2911 RIGHTWARDS ARROW WITH DOTTED STEM
 # ->   2192 RIGHTWARDS ARROW
 "\x{2911}" => "\x{2192}",

 # 2914 RIGHTWARDS ARROW WITH TAIL WITH VERTICAL STROKE
 # ->   2192 RIGHTWARDS ARROW
 "\x{2914}" => "\x{2192}",

 # 2915 RIGHTWARDS ARROW WITH TAIL WITH DOUBLE VERTICAL STROKE
 # ->   2192 RIGHTWARDS ARROW
 "\x{2915}" => "\x{2192}",

 # 2923 NORTH WEST ARROW WITH HOOK
 # ->   2196 NORTH WEST ARROW
 "\x{2923}" => "\x{2196}",

 # 2924 NORTH EAST ARROW WITH HOOK
 # ->   2197 NORTH EAST ARROW
 "\x{2924}" => "\x{2197}",

 # 2925 SOUTH EAST ARROW WITH HOOK
 # ->   2198 SOUTH EAST ARROW
 "\x{2925}" => "\x{2198}",

 # 2926 SOUTH WEST ARROW WITH HOOK
 # ->   2199 SOUTH WEST ARROW
 "\x{2926}" => "\x{2199}",

 # 293D TOP ARC ANTICLOCKWISE ARROW WITH PLUS
 # ->   293A TOP ARC ANTICLOCKWISE ARROW
 "\x{293D}" => "\x{293A}",

 # 2945 RIGHTWARDS ARROW WITH PLUS BELOW
 # ->   2192 RIGHTWARDS ARROW
 "\x{2945}" => "\x{2192}",

 # 2946 LEFTWARDS ARROW WITH PLUS BELOW
 # ->   2190 LEFTWARDS ARROW
 "\x{2946}" => "\x{2190}",

 # 298B LEFT SQUARE BRACKET WITH UNDERBAR
 # ->   005B LEFT SQUARE BRACKET
 "\x{298B}" => "[",

 # 298C RIGHT SQUARE BRACKET WITH UNDERBAR
 # ->   005D RIGHT SQUARE BRACKET
 "\x{298C}" => "]",

 # 298D LEFT SQUARE BRACKET WITH TICK IN TOP CORNER
 # ->   005B LEFT SQUARE BRACKET
 "\x{298D}" => "[",

 # 298E RIGHT SQUARE BRACKET WITH TICK IN BOTTOM CORNER
 # ->   005D RIGHT SQUARE BRACKET
 "\x{298E}" => "]",

 # 298F LEFT SQUARE BRACKET WITH TICK IN BOTTOM CORNER
 # ->   005B LEFT SQUARE BRACKET
 "\x{298F}" => "[",

 # 2990 RIGHT SQUARE BRACKET WITH TICK IN TOP CORNER
 # ->   005D RIGHT SQUARE BRACKET
 "\x{2990}" => "]",

 # 2991 LEFT ANGLE BRACKET WITH DOT
 # ->   3008 LEFT ANGLE BRACKET
 "\x{2991}" => "\x{3008}",

 # 2992 RIGHT ANGLE BRACKET WITH DOT
 # ->   3009 RIGHT ANGLE BRACKET
 "\x{2992}" => "\x{3009}",

 # 299E ANGLE WITH S INSIDE
 # ->   2220 ANGLE
 "\x{299E}" => "\x{2220}",

 # 29A4 ANGLE WITH UNDERBAR
 # ->   2220 ANGLE
 "\x{29A4}" => "\x{2220}",

 # 29A5 REVERSED ANGLE WITH UNDERBAR
 # ->   29A3 REVERSED ANGLE
 "\x{29A5}" => "\x{29A3}",

 # 29A8 MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND RIGHT
 # ->   2221 MEASURED ANGLE
 "\x{29A8}" => "\x{2221}",

 # 29A9 MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND LEFT
 # ->   2221 MEASURED ANGLE
 "\x{29A9}" => "\x{2221}",

 # 29AA MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING DOWN AND RIGHT
 # ->   2221 MEASURED ANGLE
 "\x{29AA}" => "\x{2221}",

 # 29AB MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING DOWN AND LEFT
 # ->   2221 MEASURED ANGLE
 "\x{29AB}" => "\x{2221}",

 # 29AC MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND UP
 # ->   2221 MEASURED ANGLE
 "\x{29AC}" => "\x{2221}",

 # 29AD MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING LEFT AND UP
 # ->   2221 MEASURED ANGLE
 "\x{29AD}" => "\x{2221}",

 # 29AE MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND DOWN
 # ->   2221 MEASURED ANGLE
 "\x{29AE}" => "\x{2221}",

 # 29AF MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING LEFT AND DOWN
 # ->   2221 MEASURED ANGLE
 "\x{29AF}" => "\x{2221}",

 # 29B1 EMPTY SET WITH OVERBAR
 # ->   2205 EMPTY SET
 "\x{29B1}" => "\x{2205}",

 # 29B2 EMPTY SET WITH SMALL CIRCLE ABOVE
 # ->   2205 EMPTY SET
 "\x{29B2}" => "\x{2205}",

 # 29B3 EMPTY SET WITH RIGHT ARROW ABOVE
 # ->   2205 EMPTY SET
 "\x{29B3}" => "\x{2205}",

 # 29B4 EMPTY SET WITH LEFT ARROW ABOVE
 # ->   2205 EMPTY SET
 "\x{29B4}" => "\x{2205}",

 # 29D1 BOWTIE WITH LEFT HALF BLACK
 # ->   22C8 BOWTIE
 "\x{29D1}" => "\x{22C8}",

 # 29D2 BOWTIE WITH RIGHT HALF BLACK
 # ->   22C8 BOWTIE
 "\x{29D2}" => "\x{22C8}",

 # 29E4 EQUALS SIGN AND SLANTED PARALLEL WITH TILDE ABOVE
 # ->   29E3 EQUALS SIGN AND SLANTED PARALLEL
 "\x{29E4}" => "\x{29E3}",

 # 29EA BLACK DIAMOND WITH DOWN ARROW
 # ->   25C6 BLACK DIAMOND
 "\x{29EA}" => "\x{25C6}",

 # 29EC WHITE CIRCLE WITH DOWN ARROW
 # ->   25CB WHITE CIRCLE
 "\x{29EC}" => "\x{25CB}",

 # 29ED BLACK CIRCLE WITH DOWN ARROW
 # ->   25CF BLACK CIRCLE
 "\x{29ED}" => "\x{25CF}",

 # 29F6 SOLIDUS WITH OVERBAR
 # ->   002F SOLIDUS
 "\x{29F6}" => "/",

 # 29F7 REVERSE SOLIDUS WITH HORIZONTAL STROKE
 # ->   005C REVERSE SOLIDUS
 "\x{29F7}" => "\\",

 # 2A0C QUADRUPLE INTEGRAL OPERATOR
 # ->   222B INTEGRAL
 #    + 222B INTEGRAL
 #    + 222B INTEGRAL
 #    + 222B INTEGRAL
 "\x{2A0C}" => "\x{222B}\x{222B}\x{222B}\x{222B}",

 # 2A0E INTEGRAL WITH DOUBLE STROKE
 # ->   222B INTEGRAL
 "\x{2A0E}" => "\x{222B}",

 # 2A17 INTEGRAL WITH LEFTWARDS ARROW WITH HOOK
 # ->   222B INTEGRAL
 "\x{2A17}" => "\x{222B}",

 # 2A18 INTEGRAL WITH TIMES SIGN
 # ->   222B INTEGRAL
 "\x{2A18}" => "\x{222B}",

 # 2A19 INTEGRAL WITH INTERSECTION
 # ->   222B INTEGRAL
 "\x{2A19}" => "\x{222B}",

 # 2A1A INTEGRAL WITH UNION
 # ->   222B INTEGRAL
 "\x{2A1A}" => "\x{222B}",

 # 2A1B INTEGRAL WITH OVERBAR
 # ->   222B INTEGRAL
 "\x{2A1B}" => "\x{222B}",

 # 2A1C INTEGRAL WITH UNDERBAR
 # ->   222B INTEGRAL
 "\x{2A1C}" => "\x{222B}",

 # 2A22 PLUS SIGN WITH SMALL CIRCLE ABOVE
 # ->   002B PLUS SIGN
 "\x{2A22}" => "+",

 # 2A23 PLUS SIGN WITH CIRCUMFLEX ACCENT ABOVE
 # ->   002B PLUS SIGN
 "\x{2A23}" => "+",

 # 2A24 PLUS SIGN WITH TILDE ABOVE
 # ->   002B PLUS SIGN
 "\x{2A24}" => "+",

 # 2A25 PLUS SIGN WITH DOT BELOW
 # ->   002B PLUS SIGN
 "\x{2A25}" => "+",

 # 2A26 PLUS SIGN WITH TILDE BELOW
 # ->   002B PLUS SIGN
 "\x{2A26}" => "+",

 # 2A27 PLUS SIGN WITH SUBSCRIPT TWO
 # ->   002B PLUS SIGN
 "\x{2A27}" => "+",

 # 2A28 PLUS SIGN WITH BLACK TRIANGLE
 # ->   002B PLUS SIGN
 "\x{2A28}" => "+",

 # 2A29 MINUS SIGN WITH COMMA ABOVE
 # ->   2212 MINUS SIGN
 "\x{2A29}" => "\x{2212}",

 # 2A2A MINUS SIGN WITH DOT BELOW
 # ->   2212 MINUS SIGN
 "\x{2A2A}" => "\x{2212}",

 # 2A2B MINUS SIGN WITH FALLING DOTS
 # ->   2212 MINUS SIGN
 "\x{2A2B}" => "\x{2212}",

 # 2A2C MINUS SIGN WITH RISING DOTS
 # ->   2212 MINUS SIGN
 "\x{2A2C}" => "\x{2212}",

 # 2A30 MULTIPLICATION SIGN WITH DOT ABOVE
 # ->   00D7 MULTIPLICATION SIGN
 "\x{2A30}" => "\xD7",

 # 2A31 MULTIPLICATION SIGN WITH UNDERBAR
 # ->   00D7 MULTIPLICATION SIGN
 "\x{2A31}" => "\xD7",

 # 2A40 INTERSECTION WITH DOT
 # ->   2229 INTERSECTION
 "\x{2A40}" => "\x{2229}",

 # 2A41 UNION WITH MINUS SIGN
 # ->   222A UNION
 "\x{2A41}" => "\x{222A}",

 # 2A42 UNION WITH OVERBAR
 # ->   222A UNION
 "\x{2A42}" => "\x{222A}",

 # 2A43 INTERSECTION WITH OVERBAR
 # ->   2229 INTERSECTION
 "\x{2A43}" => "\x{2229}",

 # 2A44 INTERSECTION WITH LOGICAL AND
 # ->   2229 INTERSECTION
 "\x{2A44}" => "\x{2229}",

 # 2A45 UNION WITH LOGICAL OR
 # ->   222A UNION
 "\x{2A45}" => "\x{222A}",

 # 2A51 LOGICAL AND WITH DOT ABOVE
 # ->   2227 LOGICAL AND
 "\x{2A51}" => "\x{2227}",

 # 2A52 LOGICAL OR WITH DOT ABOVE
 # ->   2228 LOGICAL OR
 "\x{2A52}" => "\x{2228}",

 # 2A5A LOGICAL AND WITH MIDDLE STEM
 # ->   2227 LOGICAL AND
 "\x{2A5A}" => "\x{2227}",

 # 2A5B LOGICAL OR WITH MIDDLE STEM
 # ->   2228 LOGICAL OR
 "\x{2A5B}" => "\x{2228}",

 # 2A5C LOGICAL AND WITH HORIZONTAL DASH
 # ->   2227 LOGICAL AND
 "\x{2A5C}" => "\x{2227}",

 # 2A5D LOGICAL OR WITH HORIZONTAL DASH
 # ->   2228 LOGICAL OR
 "\x{2A5D}" => "\x{2228}",

 # 2A5E LOGICAL AND WITH DOUBLE OVERBAR
 # ->   2227 LOGICAL AND
 "\x{2A5E}" => "\x{2227}",

 # 2A5F LOGICAL AND WITH UNDERBAR
 # ->   2227 LOGICAL AND
 "\x{2A5F}" => "\x{2227}",

 # 2A60 LOGICAL AND WITH DOUBLE UNDERBAR
 # ->   2227 LOGICAL AND
 "\x{2A60}" => "\x{2227}",

 # 2A62 LOGICAL OR WITH DOUBLE OVERBAR
 # ->   2228 LOGICAL OR
 "\x{2A62}" => "\x{2228}",

 # 2A63 LOGICAL OR WITH DOUBLE UNDERBAR
 # ->   2228 LOGICAL OR
 "\x{2A63}" => "\x{2228}",

 # 2A66 EQUALS SIGN WITH DOT BELOW
 # ->   003D EQUALS SIGN
 "\x{2A66}" => "=",

 # 2A6A TILDE OPERATOR WITH DOT ABOVE
 # ->   223C TILDE OPERATOR
 "\x{2A6A}" => "\x{223C}",

 # 2A6B TILDE OPERATOR WITH RISING DOTS
 # ->   223C TILDE OPERATOR
 "\x{2A6B}" => "\x{223C}",

 # 2A6F ALMOST EQUAL TO WITH CIRCUMFLEX ACCENT
 # ->   2248 ALMOST EQUAL TO
 "\x{2A6F}" => "\x{2248}",

 # 2A74 DOUBLE COLON EQUAL
 # ->   003A COLON
 #    + 003A COLON
 #    + 003D EQUALS SIGN
 "\x{2A74}" => "::=",

 # 2A75 TWO CONSECUTIVE EQUALS SIGNS
 # ->   003D EQUALS SIGN
 #    + 003D EQUALS SIGN
 "\x{2A75}" => "==",

 # 2A76 THREE CONSECUTIVE EQUALS SIGNS
 # ->   003D EQUALS SIGN
 #    + 003D EQUALS SIGN
 #    + 003D EQUALS SIGN
 "\x{2A76}" => "===",

 # 2A77 EQUALS SIGN WITH TWO DOTS ABOVE AND TWO DOTS BELOW
 # ->   003D EQUALS SIGN
 "\x{2A77}" => "=",

 # 2A7F LESS-THAN OR SLANTED EQUAL TO WITH DOT INSIDE
 # ->   2A7D LESS-THAN OR SLANTED EQUAL TO
 "\x{2A7F}" => "\x{2A7D}",

 # 2A80 GREATER-THAN OR SLANTED EQUAL TO WITH DOT INSIDE
 # ->   2A7E GREATER-THAN OR SLANTED EQUAL TO
 "\x{2A80}" => "\x{2A7E}",

 # 2A81 LESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVE
 # ->   2A7D LESS-THAN OR SLANTED EQUAL TO
 "\x{2A81}" => "\x{2A7D}",

 # 2A82 GREATER-THAN OR SLANTED EQUAL TO WITH DOT ABOVE
 # ->   2A7E GREATER-THAN OR SLANTED EQUAL TO
 "\x{2A82}" => "\x{2A7E}",

 # 2A83 LESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVE RIGHT
 # ->   2A7D LESS-THAN OR SLANTED EQUAL TO
 "\x{2A83}" => "\x{2A7D}",

 # 2A84 GREATER-THAN OR SLANTED EQUAL TO WITH DOT ABOVE LEFT
 # ->   2A7E GREATER-THAN OR SLANTED EQUAL TO
 "\x{2A84}" => "\x{2A7E}",

 # 2A97 SLANTED EQUAL TO OR LESS-THAN WITH DOT INSIDE
 # ->   2A95 SLANTED EQUAL TO OR LESS-THAN
 "\x{2A97}" => "\x{2A95}",

 # 2A98 SLANTED EQUAL TO OR GREATER-THAN WITH DOT INSIDE
 # ->   2A96 SLANTED EQUAL TO OR GREATER-THAN
 "\x{2A98}" => "\x{2A96}",

 # 2AA3 DOUBLE NESTED LESS-THAN WITH UNDERBAR
 # ->   2AA1 DOUBLE NESTED LESS-THAN
 "\x{2AA3}" => "\x{2AA1}",

 # 2AAE EQUALS SIGN WITH BUMPY ABOVE
 # ->   003D EQUALS SIGN
 "\x{2AAE}" => "=",

 # 2AC3 SUBSET OF OR EQUAL TO WITH DOT ABOVE
 # ->   2286 SUBSET OF OR EQUAL TO
 "\x{2AC3}" => "\x{2286}",

 # 2AC4 SUPERSET OF OR EQUAL TO WITH DOT ABOVE
 # ->   2287 SUPERSET OF OR EQUAL TO
 "\x{2AC4}" => "\x{2287}",

 # 2ADA PITCHFORK WITH TEE TOP
 # ->   22D4 PITCHFORK
 "\x{2ADA}" => "\x{22D4}",

 # 2ADC FORKING
 # ->   2ADD NONFORKING
 "\x{2ADC}" => "\x{2ADD}",

 # 2AE7 SHORT DOWN TACK WITH OVERBAR
 # ->   2ADF SHORT DOWN TACK
 "\x{2AE7}" => "\x{2ADF}",

 # 2AE8 SHORT UP TACK WITH UNDERBAR
 # ->   2AE0 SHORT UP TACK
 "\x{2AE8}" => "\x{2AE0}",

 # 2AEE DOES NOT DIVIDE WITH REVERSED NEGATION SLASH
 # ->   2223 DIVIDES
 "\x{2AEE}" => "\x{2223}",

 # 2AEF VERTICAL LINE WITH CIRCLE ABOVE
 # ->   007C VERTICAL LINE
 "\x{2AEF}" => "|",

 # 2AF0 VERTICAL LINE WITH CIRCLE BELOW
 # ->   007C VERTICAL LINE
 "\x{2AF0}" => "|",

 # 2AF1 DOWN TACK WITH CIRCLE BELOW
 # ->   22A4 DOWN TACK
 "\x{2AF1}" => "\x{22A4}",

 # 2E9F CJK RADICAL MOTHER
 # ->   6BCD CJK UNIFIED IDEOGRAPH 6BCD
 "\x{2E9F}" => "\x{6BCD}",

 # 2EF3 CJK RADICAL C-SIMPLIFIED TURTLE
 # ->   9F9F CJK UNIFIED IDEOGRAPH 9F9F
 "\x{2EF3}" => "\x{9F9F}",

 # 2F00 KANGXI RADICAL ONE
 # ->   4E00 CJK UNIFIED IDEOGRAPH 4E00
 "\x{2F00}" => "\x{4E00}",

 # 2F01 KANGXI RADICAL LINE
 # ->   4E28 CJK UNIFIED IDEOGRAPH 4E28
 "\x{2F01}" => "\x{4E28}",

 # 2F02 KANGXI RADICAL DOT
 # ->   4E36 CJK UNIFIED IDEOGRAPH 4E36
 "\x{2F02}" => "\x{4E36}",

 # 2F03 KANGXI RADICAL SLASH
 # ->   4E3F CJK UNIFIED IDEOGRAPH 4E3F
 "\x{2F03}" => "\x{4E3F}",

 # 2F04 KANGXI RADICAL SECOND
 # ->   4E59 CJK UNIFIED IDEOGRAPH 4E59
 "\x{2F04}" => "\x{4E59}",

 # 2F05 KANGXI RADICAL HOOK
 # ->   4E85 CJK UNIFIED IDEOGRAPH 4E85
 "\x{2F05}" => "\x{4E85}",

 # 2F06 KANGXI RADICAL TWO
 # ->   4E8C CJK UNIFIED IDEOGRAPH 4E8C
 "\x{2F06}" => "\x{4E8C}",

 # 2F07 KANGXI RADICAL LID
 # ->   4EA0 CJK UNIFIED IDEOGRAPH 4EA0
 "\x{2F07}" => "\x{4EA0}",

 # 2F08 KANGXI RADICAL MAN
 # ->   4EBA CJK UNIFIED IDEOGRAPH 4EBA
 "\x{2F08}" => "\x{4EBA}",

 # 2F09 KANGXI RADICAL LEGS
 # ->   513F CJK UNIFIED IDEOGRAPH 513F
 "\x{2F09}" => "\x{513F}",

 # 2F0A KANGXI RADICAL ENTER
 # ->   5165 CJK UNIFIED IDEOGRAPH 5165
 "\x{2F0A}" => "\x{5165}",

 # 2F0B KANGXI RADICAL EIGHT
 # ->   516B CJK UNIFIED IDEOGRAPH 516B
 "\x{2F0B}" => "\x{516B}",

 # 2F0C KANGXI RADICAL DOWN BOX
 # ->   5182 CJK UNIFIED IDEOGRAPH 5182
 "\x{2F0C}" => "\x{5182}",

 # 2F0D KANGXI RADICAL COVER
 # ->   5196 CJK UNIFIED IDEOGRAPH 5196
 "\x{2F0D}" => "\x{5196}",

 # 2F0E KANGXI RADICAL ICE
 # ->   51AB CJK UNIFIED IDEOGRAPH 51AB
 "\x{2F0E}" => "\x{51AB}",

 # 2F0F KANGXI RADICAL TABLE
 # ->   51E0 CJK UNIFIED IDEOGRAPH 51E0
 "\x{2F0F}" => "\x{51E0}",

 # 2F10 KANGXI RADICAL OPEN BOX
 # ->   51F5 CJK UNIFIED IDEOGRAPH 51F5
 "\x{2F10}" => "\x{51F5}",

 # 2F11 KANGXI RADICAL KNIFE
 # ->   5200 CJK UNIFIED IDEOGRAPH 5200
 "\x{2F11}" => "\x{5200}",

 # 2F12 KANGXI RADICAL POWER
 # ->   529B CJK UNIFIED IDEOGRAPH 529B
 "\x{2F12}" => "\x{529B}",

 # 2F13 KANGXI RADICAL WRAP
 # ->   52F9 CJK UNIFIED IDEOGRAPH 52F9
 "\x{2F13}" => "\x{52F9}",

 # 2F14 KANGXI RADICAL SPOON
 # ->   5315 CJK UNIFIED IDEOGRAPH 5315
 "\x{2F14}" => "\x{5315}",

 # 2F15 KANGXI RADICAL RIGHT OPEN BOX
 # ->   531A CJK UNIFIED IDEOGRAPH 531A
 "\x{2F15}" => "\x{531A}",

 # 2F16 KANGXI RADICAL HIDING ENCLOSURE
 # ->   5338 CJK UNIFIED IDEOGRAPH 5338
 "\x{2F16}" => "\x{5338}",

 # 2F17 KANGXI RADICAL TEN
 # ->   5341 CJK UNIFIED IDEOGRAPH 5341
 "\x{2F17}" => "\x{5341}",

 # 2F18 KANGXI RADICAL DIVINATION
 # ->   535C CJK UNIFIED IDEOGRAPH 535C
 "\x{2F18}" => "\x{535C}",

 # 2F19 KANGXI RADICAL SEAL
 # ->   5369 CJK UNIFIED IDEOGRAPH 5369
 "\x{2F19}" => "\x{5369}",

 # 2F1A KANGXI RADICAL CLIFF
 # ->   5382 CJK UNIFIED IDEOGRAPH 5382
 "\x{2F1A}" => "\x{5382}",

 # 2F1B KANGXI RADICAL PRIVATE
 # ->   53B6 CJK UNIFIED IDEOGRAPH 53B6
 "\x{2F1B}" => "\x{53B6}",

 # 2F1C KANGXI RADICAL AGAIN
 # ->   53C8 CJK UNIFIED IDEOGRAPH 53C8
 "\x{2F1C}" => "\x{53C8}",

 # 2F1D KANGXI RADICAL MOUTH
 # ->   53E3 CJK UNIFIED IDEOGRAPH 53E3
 "\x{2F1D}" => "\x{53E3}",

 # 2F1E KANGXI RADICAL ENCLOSURE
 # ->   56D7 CJK UNIFIED IDEOGRAPH 56D7
 "\x{2F1E}" => "\x{56D7}",

 # 2F1F KANGXI RADICAL EARTH
 # ->   571F CJK UNIFIED IDEOGRAPH 571F
 "\x{2F1F}" => "\x{571F}",

 # 2F20 KANGXI RADICAL SCHOLAR
 # ->   58EB CJK UNIFIED IDEOGRAPH 58EB
 "\x{2F20}" => "\x{58EB}",

 # 2F21 KANGXI RADICAL GO
 # ->   5902 CJK UNIFIED IDEOGRAPH 5902
 "\x{2F21}" => "\x{5902}",

 # 2F22 KANGXI RADICAL GO SLOWLY
 # ->   590A CJK UNIFIED IDEOGRAPH 590A
 "\x{2F22}" => "\x{590A}",

 # 2F23 KANGXI RADICAL EVENING
 # ->   5915 CJK UNIFIED IDEOGRAPH 5915
 "\x{2F23}" => "\x{5915}",

 # 2F24 KANGXI RADICAL BIG
 # ->   5927 CJK UNIFIED IDEOGRAPH 5927
 "\x{2F24}" => "\x{5927}",

 # 2F25 KANGXI RADICAL WOMAN
 # ->   5973 CJK UNIFIED IDEOGRAPH 5973
 "\x{2F25}" => "\x{5973}",

 # 2F26 KANGXI RADICAL CHILD
 # ->   5B50 CJK UNIFIED IDEOGRAPH 5B50
 "\x{2F26}" => "\x{5B50}",

 # 2F27 KANGXI RADICAL ROOF
 # ->   5B80 CJK UNIFIED IDEOGRAPH 5B80
 "\x{2F27}" => "\x{5B80}",

 # 2F28 KANGXI RADICAL INCH
 # ->   5BF8 CJK UNIFIED IDEOGRAPH 5BF8
 "\x{2F28}" => "\x{5BF8}",

 # 2F29 KANGXI RADICAL SMALL
 # ->   5C0F CJK UNIFIED IDEOGRAPH 5C0F
 "\x{2F29}" => "\x{5C0F}",

 # 2F2A KANGXI RADICAL LAME
 # ->   5C22 CJK UNIFIED IDEOGRAPH 5C22
 "\x{2F2A}" => "\x{5C22}",

 # 2F2B KANGXI RADICAL CORPSE
 # ->   5C38 CJK UNIFIED IDEOGRAPH 5C38
 "\x{2F2B}" => "\x{5C38}",

 # 2F2C KANGXI RADICAL SPROUT
 # ->   5C6E CJK UNIFIED IDEOGRAPH 5C6E
 "\x{2F2C}" => "\x{5C6E}",

 # 2F2D KANGXI RADICAL MOUNTAIN
 # ->   5C71 CJK UNIFIED IDEOGRAPH 5C71
 "\x{2F2D}" => "\x{5C71}",

 # 2F2E KANGXI RADICAL RIVER
 # ->   5DDB CJK UNIFIED IDEOGRAPH 5DDB
 "\x{2F2E}" => "\x{5DDB}",

 # 2F2F KANGXI RADICAL WORK
 # ->   5DE5 CJK UNIFIED IDEOGRAPH 5DE5
 "\x{2F2F}" => "\x{5DE5}",

 # 2F30 KANGXI RADICAL ONESELF
 # ->   5DF1 CJK UNIFIED IDEOGRAPH 5DF1
 "\x{2F30}" => "\x{5DF1}",

 # 2F31 KANGXI RADICAL TURBAN
 # ->   5DFE CJK UNIFIED IDEOGRAPH 5DFE
 "\x{2F31}" => "\x{5DFE}",

 # 2F32 KANGXI RADICAL DRY
 # ->   5E72 CJK UNIFIED IDEOGRAPH 5E72
 "\x{2F32}" => "\x{5E72}",

 # 2F33 KANGXI RADICAL SHORT THREAD
 # ->   5E7A CJK UNIFIED IDEOGRAPH 5E7A
 "\x{2F33}" => "\x{5E7A}",

 # 2F34 KANGXI RADICAL DOTTED CLIFF
 # ->   5E7F CJK UNIFIED IDEOGRAPH 5E7F
 "\x{2F34}" => "\x{5E7F}",

 # 2F35 KANGXI RADICAL LONG STRIDE
 # ->   5EF4 CJK UNIFIED IDEOGRAPH 5EF4
 "\x{2F35}" => "\x{5EF4}",

 # 2F36 KANGXI RADICAL TWO HANDS
 # ->   5EFE CJK UNIFIED IDEOGRAPH 5EFE
 "\x{2F36}" => "\x{5EFE}",

 # 2F37 KANGXI RADICAL SHOOT
 # ->   5F0B CJK UNIFIED IDEOGRAPH 5F0B
 "\x{2F37}" => "\x{5F0B}",

 # 2F38 KANGXI RADICAL BOW
 # ->   5F13 CJK UNIFIED IDEOGRAPH 5F13
 "\x{2F38}" => "\x{5F13}",

 # 2F39 KANGXI RADICAL SNOUT
 # ->   5F50 CJK UNIFIED IDEOGRAPH 5F50
 "\x{2F39}" => "\x{5F50}",

 # 2F3A KANGXI RADICAL BRISTLE
 # ->   5F61 CJK UNIFIED IDEOGRAPH 5F61
 "\x{2F3A}" => "\x{5F61}",

 # 2F3B KANGXI RADICAL STEP
 # ->   5F73 CJK UNIFIED IDEOGRAPH 5F73
 "\x{2F3B}" => "\x{5F73}",

 # 2F3C KANGXI RADICAL HEART
 # ->   5FC3 CJK UNIFIED IDEOGRAPH 5FC3
 "\x{2F3C}" => "\x{5FC3}",

 # 2F3D KANGXI RADICAL HALBERD
 # ->   6208 CJK UNIFIED IDEOGRAPH 6208
 "\x{2F3D}" => "\x{6208}",

 # 2F3E KANGXI RADICAL DOOR
 # ->   6236 CJK UNIFIED IDEOGRAPH 6236
 "\x{2F3E}" => "\x{6236}",

 # 2F3F KANGXI RADICAL HAND
 # ->   624B CJK UNIFIED IDEOGRAPH 624B
 "\x{2F3F}" => "\x{624B}",

 # 2F40 KANGXI RADICAL BRANCH
 # ->   652F CJK UNIFIED IDEOGRAPH 652F
 "\x{2F40}" => "\x{652F}",

 # 2F41 KANGXI RADICAL RAP
 # ->   6534 CJK UNIFIED IDEOGRAPH 6534
 "\x{2F41}" => "\x{6534}",

 # 2F42 KANGXI RADICAL SCRIPT
 # ->   6587 CJK UNIFIED IDEOGRAPH 6587
 "\x{2F42}" => "\x{6587}",

 # 2F43 KANGXI RADICAL DIPPER
 # ->   6597 CJK UNIFIED IDEOGRAPH 6597
 "\x{2F43}" => "\x{6597}",

 # 2F44 KANGXI RADICAL AXE
 # ->   65A4 CJK UNIFIED IDEOGRAPH 65A4
 "\x{2F44}" => "\x{65A4}",

 # 2F45 KANGXI RADICAL SQUARE
 # ->   65B9 CJK UNIFIED IDEOGRAPH 65B9
 "\x{2F45}" => "\x{65B9}",

 # 2F46 KANGXI RADICAL NOT
 # ->   65E0 CJK UNIFIED IDEOGRAPH 65E0
 "\x{2F46}" => "\x{65E0}",

 # 2F47 KANGXI RADICAL SUN
 # ->   65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{2F47}" => "\x{65E5}",

 # 2F48 KANGXI RADICAL SAY
 # ->   66F0 CJK UNIFIED IDEOGRAPH 66F0
 "\x{2F48}" => "\x{66F0}",

 # 2F49 KANGXI RADICAL MOON
 # ->   6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{2F49}" => "\x{6708}",

 # 2F4A KANGXI RADICAL TREE
 # ->   6728 CJK UNIFIED IDEOGRAPH 6728
 "\x{2F4A}" => "\x{6728}",

 # 2F4B KANGXI RADICAL LACK
 # ->   6B20 CJK UNIFIED IDEOGRAPH 6B20
 "\x{2F4B}" => "\x{6B20}",

 # 2F4C KANGXI RADICAL STOP
 # ->   6B62 CJK UNIFIED IDEOGRAPH 6B62
 "\x{2F4C}" => "\x{6B62}",

 # 2F4D KANGXI RADICAL DEATH
 # ->   6B79 CJK UNIFIED IDEOGRAPH 6B79
 "\x{2F4D}" => "\x{6B79}",

 # 2F4E KANGXI RADICAL WEAPON
 # ->   6BB3 CJK UNIFIED IDEOGRAPH 6BB3
 "\x{2F4E}" => "\x{6BB3}",

 # 2F4F KANGXI RADICAL DO NOT
 # ->   6BCB CJK UNIFIED IDEOGRAPH 6BCB
 "\x{2F4F}" => "\x{6BCB}",

 # 2F50 KANGXI RADICAL COMPARE
 # ->   6BD4 CJK UNIFIED IDEOGRAPH 6BD4
 "\x{2F50}" => "\x{6BD4}",

 # 2F51 KANGXI RADICAL FUR
 # ->   6BDB CJK UNIFIED IDEOGRAPH 6BDB
 "\x{2F51}" => "\x{6BDB}",

 # 2F52 KANGXI RADICAL CLAN
 # ->   6C0F CJK UNIFIED IDEOGRAPH 6C0F
 "\x{2F52}" => "\x{6C0F}",

 # 2F53 KANGXI RADICAL STEAM
 # ->   6C14 CJK UNIFIED IDEOGRAPH 6C14
 "\x{2F53}" => "\x{6C14}",

 # 2F54 KANGXI RADICAL WATER
 # ->   6C34 CJK UNIFIED IDEOGRAPH 6C34
 "\x{2F54}" => "\x{6C34}",

 # 2F55 KANGXI RADICAL FIRE
 # ->   706B CJK UNIFIED IDEOGRAPH 706B
 "\x{2F55}" => "\x{706B}",

 # 2F56 KANGXI RADICAL CLAW
 # ->   722A CJK UNIFIED IDEOGRAPH 722A
 "\x{2F56}" => "\x{722A}",

 # 2F57 KANGXI RADICAL FATHER
 # ->   7236 CJK UNIFIED IDEOGRAPH 7236
 "\x{2F57}" => "\x{7236}",

 # 2F58 KANGXI RADICAL DOUBLE X
 # ->   723B CJK UNIFIED IDEOGRAPH 723B
 "\x{2F58}" => "\x{723B}",

 # 2F59 KANGXI RADICAL HALF TREE TRUNK
 # ->   723F CJK UNIFIED IDEOGRAPH 723F
 "\x{2F59}" => "\x{723F}",

 # 2F5A KANGXI RADICAL SLICE
 # ->   7247 CJK UNIFIED IDEOGRAPH 7247
 "\x{2F5A}" => "\x{7247}",

 # 2F5B KANGXI RADICAL FANG
 # ->   7259 CJK UNIFIED IDEOGRAPH 7259
 "\x{2F5B}" => "\x{7259}",

 # 2F5C KANGXI RADICAL COW
 # ->   725B CJK UNIFIED IDEOGRAPH 725B
 "\x{2F5C}" => "\x{725B}",

 # 2F5D KANGXI RADICAL DOG
 # ->   72AC CJK UNIFIED IDEOGRAPH 72AC
 "\x{2F5D}" => "\x{72AC}",

 # 2F5E KANGXI RADICAL PROFOUND
 # ->   7384 CJK UNIFIED IDEOGRAPH 7384
 "\x{2F5E}" => "\x{7384}",

 # 2F5F KANGXI RADICAL JADE
 # ->   7389 CJK UNIFIED IDEOGRAPH 7389
 "\x{2F5F}" => "\x{7389}",

 # 2F60 KANGXI RADICAL MELON
 # ->   74DC CJK UNIFIED IDEOGRAPH 74DC
 "\x{2F60}" => "\x{74DC}",

 # 2F61 KANGXI RADICAL TILE
 # ->   74E6 CJK UNIFIED IDEOGRAPH 74E6
 "\x{2F61}" => "\x{74E6}",

 # 2F62 KANGXI RADICAL SWEET
 # ->   7518 CJK UNIFIED IDEOGRAPH 7518
 "\x{2F62}" => "\x{7518}",

 # 2F63 KANGXI RADICAL LIFE
 # ->   751F CJK UNIFIED IDEOGRAPH 751F
 "\x{2F63}" => "\x{751F}",

 # 2F64 KANGXI RADICAL USE
 # ->   7528 CJK UNIFIED IDEOGRAPH 7528
 "\x{2F64}" => "\x{7528}",

 # 2F65 KANGXI RADICAL FIELD
 # ->   7530 CJK UNIFIED IDEOGRAPH 7530
 "\x{2F65}" => "\x{7530}",

 # 2F66 KANGXI RADICAL BOLT OF CLOTH
 # ->   758B CJK UNIFIED IDEOGRAPH 758B
 "\x{2F66}" => "\x{758B}",

 # 2F67 KANGXI RADICAL SICKNESS
 # ->   7592 CJK UNIFIED IDEOGRAPH 7592
 "\x{2F67}" => "\x{7592}",

 # 2F68 KANGXI RADICAL DOTTED TENT
 # ->   7676 CJK UNIFIED IDEOGRAPH 7676
 "\x{2F68}" => "\x{7676}",

 # 2F69 KANGXI RADICAL WHITE
 # ->   767D CJK UNIFIED IDEOGRAPH 767D
 "\x{2F69}" => "\x{767D}",

 # 2F6A KANGXI RADICAL SKIN
 # ->   76AE CJK UNIFIED IDEOGRAPH 76AE
 "\x{2F6A}" => "\x{76AE}",

 # 2F6B KANGXI RADICAL DISH
 # ->   76BF CJK UNIFIED IDEOGRAPH 76BF
 "\x{2F6B}" => "\x{76BF}",

 # 2F6C KANGXI RADICAL EYE
 # ->   76EE CJK UNIFIED IDEOGRAPH 76EE
 "\x{2F6C}" => "\x{76EE}",

 # 2F6D KANGXI RADICAL SPEAR
 # ->   77DB CJK UNIFIED IDEOGRAPH 77DB
 "\x{2F6D}" => "\x{77DB}",

 # 2F6E KANGXI RADICAL ARROW
 # ->   77E2 CJK UNIFIED IDEOGRAPH 77E2
 "\x{2F6E}" => "\x{77E2}",

 # 2F6F KANGXI RADICAL STONE
 # ->   77F3 CJK UNIFIED IDEOGRAPH 77F3
 "\x{2F6F}" => "\x{77F3}",

 # 2F70 KANGXI RADICAL SPIRIT
 # ->   793A CJK UNIFIED IDEOGRAPH 793A
 "\x{2F70}" => "\x{793A}",

 # 2F71 KANGXI RADICAL TRACK
 # ->   79B8 CJK UNIFIED IDEOGRAPH 79B8
 "\x{2F71}" => "\x{79B8}",

 # 2F72 KANGXI RADICAL GRAIN
 # ->   79BE CJK UNIFIED IDEOGRAPH 79BE
 "\x{2F72}" => "\x{79BE}",

 # 2F73 KANGXI RADICAL CAVE
 # ->   7A74 CJK UNIFIED IDEOGRAPH 7A74
 "\x{2F73}" => "\x{7A74}",

 # 2F74 KANGXI RADICAL STAND
 # ->   7ACB CJK UNIFIED IDEOGRAPH 7ACB
 "\x{2F74}" => "\x{7ACB}",

 # 2F75 KANGXI RADICAL BAMBOO
 # ->   7AF9 CJK UNIFIED IDEOGRAPH 7AF9
 "\x{2F75}" => "\x{7AF9}",

 # 2F76 KANGXI RADICAL RICE
 # ->   7C73 CJK UNIFIED IDEOGRAPH 7C73
 "\x{2F76}" => "\x{7C73}",

 # 2F77 KANGXI RADICAL SILK
 # ->   7CF8 CJK UNIFIED IDEOGRAPH 7CF8
 "\x{2F77}" => "\x{7CF8}",

 # 2F78 KANGXI RADICAL JAR
 # ->   7F36 CJK UNIFIED IDEOGRAPH 7F36
 "\x{2F78}" => "\x{7F36}",

 # 2F79 KANGXI RADICAL NET
 # ->   7F51 CJK UNIFIED IDEOGRAPH 7F51
 "\x{2F79}" => "\x{7F51}",

 # 2F7A KANGXI RADICAL SHEEP
 # ->   7F8A CJK UNIFIED IDEOGRAPH 7F8A
 "\x{2F7A}" => "\x{7F8A}",

 # 2F7B KANGXI RADICAL FEATHER
 # ->   7FBD CJK UNIFIED IDEOGRAPH 7FBD
 "\x{2F7B}" => "\x{7FBD}",

 # 2F7C KANGXI RADICAL OLD
 # ->   8001 CJK UNIFIED IDEOGRAPH 8001
 "\x{2F7C}" => "\x{8001}",

 # 2F7D KANGXI RADICAL AND
 # ->   800C CJK UNIFIED IDEOGRAPH 800C
 "\x{2F7D}" => "\x{800C}",

 # 2F7E KANGXI RADICAL PLOW
 # ->   8012 CJK UNIFIED IDEOGRAPH 8012
 "\x{2F7E}" => "\x{8012}",

 # 2F7F KANGXI RADICAL EAR
 # ->   8033 CJK UNIFIED IDEOGRAPH 8033
 "\x{2F7F}" => "\x{8033}",

 # 2F80 KANGXI RADICAL BRUSH
 # ->   807F CJK UNIFIED IDEOGRAPH 807F
 "\x{2F80}" => "\x{807F}",

 # 2F81 KANGXI RADICAL MEAT
 # ->   8089 CJK UNIFIED IDEOGRAPH 8089
 "\x{2F81}" => "\x{8089}",

 # 2F82 KANGXI RADICAL MINISTER
 # ->   81E3 CJK UNIFIED IDEOGRAPH 81E3
 "\x{2F82}" => "\x{81E3}",

 # 2F83 KANGXI RADICAL SELF
 # ->   81EA CJK UNIFIED IDEOGRAPH 81EA
 "\x{2F83}" => "\x{81EA}",

 # 2F84 KANGXI RADICAL ARRIVE
 # ->   81F3 CJK UNIFIED IDEOGRAPH 81F3
 "\x{2F84}" => "\x{81F3}",

 # 2F85 KANGXI RADICAL MORTAR
 # ->   81FC CJK UNIFIED IDEOGRAPH 81FC
 "\x{2F85}" => "\x{81FC}",

 # 2F86 KANGXI RADICAL TONGUE
 # ->   820C CJK UNIFIED IDEOGRAPH 820C
 "\x{2F86}" => "\x{820C}",

 # 2F87 KANGXI RADICAL OPPOSE
 # ->   821B CJK UNIFIED IDEOGRAPH 821B
 "\x{2F87}" => "\x{821B}",

 # 2F88 KANGXI RADICAL BOAT
 # ->   821F CJK UNIFIED IDEOGRAPH 821F
 "\x{2F88}" => "\x{821F}",

 # 2F89 KANGXI RADICAL STOPPING
 # ->   826E CJK UNIFIED IDEOGRAPH 826E
 "\x{2F89}" => "\x{826E}",

 # 2F8A KANGXI RADICAL COLOR
 # ->   8272 CJK UNIFIED IDEOGRAPH 8272
 "\x{2F8A}" => "\x{8272}",

 # 2F8B KANGXI RADICAL GRASS
 # ->   8278 CJK UNIFIED IDEOGRAPH 8278
 "\x{2F8B}" => "\x{8278}",

 # 2F8C KANGXI RADICAL TIGER
 # ->   864D CJK UNIFIED IDEOGRAPH 864D
 "\x{2F8C}" => "\x{864D}",

 # 2F8D KANGXI RADICAL INSECT
 # ->   866B CJK UNIFIED IDEOGRAPH 866B
 "\x{2F8D}" => "\x{866B}",

 # 2F8E KANGXI RADICAL BLOOD
 # ->   8840 CJK UNIFIED IDEOGRAPH 8840
 "\x{2F8E}" => "\x{8840}",

 # 2F8F KANGXI RADICAL WALK ENCLOSURE
 # ->   884C CJK UNIFIED IDEOGRAPH 884C
 "\x{2F8F}" => "\x{884C}",

 # 2F90 KANGXI RADICAL CLOTHES
 # ->   8863 CJK UNIFIED IDEOGRAPH 8863
 "\x{2F90}" => "\x{8863}",

 # 2F91 KANGXI RADICAL WEST
 # ->   897E CJK UNIFIED IDEOGRAPH 897E
 "\x{2F91}" => "\x{897E}",

 # 2F92 KANGXI RADICAL SEE
 # ->   898B CJK UNIFIED IDEOGRAPH 898B
 "\x{2F92}" => "\x{898B}",

 # 2F93 KANGXI RADICAL HORN
 # ->   89D2 CJK UNIFIED IDEOGRAPH 89D2
 "\x{2F93}" => "\x{89D2}",

 # 2F94 KANGXI RADICAL SPEECH
 # ->   8A00 CJK UNIFIED IDEOGRAPH 8A00
 "\x{2F94}" => "\x{8A00}",

 # 2F95 KANGXI RADICAL VALLEY
 # ->   8C37 CJK UNIFIED IDEOGRAPH 8C37
 "\x{2F95}" => "\x{8C37}",

 # 2F96 KANGXI RADICAL BEAN
 # ->   8C46 CJK UNIFIED IDEOGRAPH 8C46
 "\x{2F96}" => "\x{8C46}",

 # 2F97 KANGXI RADICAL PIG
 # ->   8C55 CJK UNIFIED IDEOGRAPH 8C55
 "\x{2F97}" => "\x{8C55}",

 # 2F98 KANGXI RADICAL BADGER
 # ->   8C78 CJK UNIFIED IDEOGRAPH 8C78
 "\x{2F98}" => "\x{8C78}",

 # 2F99 KANGXI RADICAL SHELL
 # ->   8C9D CJK UNIFIED IDEOGRAPH 8C9D
 "\x{2F99}" => "\x{8C9D}",

 # 2F9A KANGXI RADICAL RED
 # ->   8D64 CJK UNIFIED IDEOGRAPH 8D64
 "\x{2F9A}" => "\x{8D64}",

 # 2F9B KANGXI RADICAL RUN
 # ->   8D70 CJK UNIFIED IDEOGRAPH 8D70
 "\x{2F9B}" => "\x{8D70}",

 # 2F9C KANGXI RADICAL FOOT
 # ->   8DB3 CJK UNIFIED IDEOGRAPH 8DB3
 "\x{2F9C}" => "\x{8DB3}",

 # 2F9D KANGXI RADICAL BODY
 # ->   8EAB CJK UNIFIED IDEOGRAPH 8EAB
 "\x{2F9D}" => "\x{8EAB}",

 # 2F9E KANGXI RADICAL CART
 # ->   8ECA CJK UNIFIED IDEOGRAPH 8ECA
 "\x{2F9E}" => "\x{8ECA}",

 # 2F9F KANGXI RADICAL BITTER
 # ->   8F9B CJK UNIFIED IDEOGRAPH 8F9B
 "\x{2F9F}" => "\x{8F9B}",

 # 2FA0 KANGXI RADICAL MORNING
 # ->   8FB0 CJK UNIFIED IDEOGRAPH 8FB0
 "\x{2FA0}" => "\x{8FB0}",

 # 2FA1 KANGXI RADICAL WALK
 # ->   8FB5 CJK UNIFIED IDEOGRAPH 8FB5
 "\x{2FA1}" => "\x{8FB5}",

 # 2FA2 KANGXI RADICAL CITY
 # ->   9091 CJK UNIFIED IDEOGRAPH 9091
 "\x{2FA2}" => "\x{9091}",

 # 2FA3 KANGXI RADICAL WINE
 # ->   9149 CJK UNIFIED IDEOGRAPH 9149
 "\x{2FA3}" => "\x{9149}",

 # 2FA4 KANGXI RADICAL DISTINGUISH
 # ->   91C6 CJK UNIFIED IDEOGRAPH 91C6
 "\x{2FA4}" => "\x{91C6}",

 # 2FA5 KANGXI RADICAL VILLAGE
 # ->   91CC CJK UNIFIED IDEOGRAPH 91CC
 "\x{2FA5}" => "\x{91CC}",

 # 2FA6 KANGXI RADICAL GOLD
 # ->   91D1 CJK UNIFIED IDEOGRAPH 91D1
 "\x{2FA6}" => "\x{91D1}",

 # 2FA7 KANGXI RADICAL LONG
 # ->   9577 CJK UNIFIED IDEOGRAPH 9577
 "\x{2FA7}" => "\x{9577}",

 # 2FA8 KANGXI RADICAL GATE
 # ->   9580 CJK UNIFIED IDEOGRAPH 9580
 "\x{2FA8}" => "\x{9580}",

 # 2FA9 KANGXI RADICAL MOUND
 # ->   961C CJK UNIFIED IDEOGRAPH 961C
 "\x{2FA9}" => "\x{961C}",

 # 2FAA KANGXI RADICAL SLAVE
 # ->   96B6 CJK UNIFIED IDEOGRAPH 96B6
 "\x{2FAA}" => "\x{96B6}",

 # 2FAB KANGXI RADICAL SHORT TAILED BIRD
 # ->   96B9 CJK UNIFIED IDEOGRAPH 96B9
 "\x{2FAB}" => "\x{96B9}",

 # 2FAC KANGXI RADICAL RAIN
 # ->   96E8 CJK UNIFIED IDEOGRAPH 96E8
 "\x{2FAC}" => "\x{96E8}",

 # 2FAD KANGXI RADICAL BLUE
 # ->   9751 CJK UNIFIED IDEOGRAPH 9751
 "\x{2FAD}" => "\x{9751}",

 # 2FAE KANGXI RADICAL WRONG
 # ->   975E CJK UNIFIED IDEOGRAPH 975E
 "\x{2FAE}" => "\x{975E}",

 # 2FAF KANGXI RADICAL FACE
 # ->   9762 CJK UNIFIED IDEOGRAPH 9762
 "\x{2FAF}" => "\x{9762}",

 # 2FB0 KANGXI RADICAL LEATHER
 # ->   9769 CJK UNIFIED IDEOGRAPH 9769
 "\x{2FB0}" => "\x{9769}",

 # 2FB1 KANGXI RADICAL TANNED LEATHER
 # ->   97CB CJK UNIFIED IDEOGRAPH 97CB
 "\x{2FB1}" => "\x{97CB}",

 # 2FB2 KANGXI RADICAL LEEK
 # ->   97ED CJK UNIFIED IDEOGRAPH 97ED
 "\x{2FB2}" => "\x{97ED}",

 # 2FB3 KANGXI RADICAL SOUND
 # ->   97F3 CJK UNIFIED IDEOGRAPH 97F3
 "\x{2FB3}" => "\x{97F3}",

 # 2FB4 KANGXI RADICAL LEAF
 # ->   9801 CJK UNIFIED IDEOGRAPH 9801
 "\x{2FB4}" => "\x{9801}",

 # 2FB5 KANGXI RADICAL WIND
 # ->   98A8 CJK UNIFIED IDEOGRAPH 98A8
 "\x{2FB5}" => "\x{98A8}",

 # 2FB6 KANGXI RADICAL FLY
 # ->   98DB CJK UNIFIED IDEOGRAPH 98DB
 "\x{2FB6}" => "\x{98DB}",

 # 2FB7 KANGXI RADICAL EAT
 # ->   98DF CJK UNIFIED IDEOGRAPH 98DF
 "\x{2FB7}" => "\x{98DF}",

 # 2FB8 KANGXI RADICAL HEAD
 # ->   9996 CJK UNIFIED IDEOGRAPH 9996
 "\x{2FB8}" => "\x{9996}",

 # 2FB9 KANGXI RADICAL FRAGRANT
 # ->   9999 CJK UNIFIED IDEOGRAPH 9999
 "\x{2FB9}" => "\x{9999}",

 # 2FBA KANGXI RADICAL HORSE
 # ->   99AC CJK UNIFIED IDEOGRAPH 99AC
 "\x{2FBA}" => "\x{99AC}",

 # 2FBB KANGXI RADICAL BONE
 # ->   9AA8 CJK UNIFIED IDEOGRAPH 9AA8
 "\x{2FBB}" => "\x{9AA8}",

 # 2FBC KANGXI RADICAL TALL
 # ->   9AD8 CJK UNIFIED IDEOGRAPH 9AD8
 "\x{2FBC}" => "\x{9AD8}",

 # 2FBD KANGXI RADICAL HAIR
 # ->   9ADF CJK UNIFIED IDEOGRAPH 9ADF
 "\x{2FBD}" => "\x{9ADF}",

 # 2FBE KANGXI RADICAL FIGHT
 # ->   9B25 CJK UNIFIED IDEOGRAPH 9B25
 "\x{2FBE}" => "\x{9B25}",

 # 2FBF KANGXI RADICAL SACRIFICIAL WINE
 # ->   9B2F CJK UNIFIED IDEOGRAPH 9B2F
 "\x{2FBF}" => "\x{9B2F}",

 # 2FC0 KANGXI RADICAL CAULDRON
 # ->   9B32 CJK UNIFIED IDEOGRAPH 9B32
 "\x{2FC0}" => "\x{9B32}",

 # 2FC1 KANGXI RADICAL GHOST
 # ->   9B3C CJK UNIFIED IDEOGRAPH 9B3C
 "\x{2FC1}" => "\x{9B3C}",

 # 2FC2 KANGXI RADICAL FISH
 # ->   9B5A CJK UNIFIED IDEOGRAPH 9B5A
 "\x{2FC2}" => "\x{9B5A}",

 # 2FC3 KANGXI RADICAL BIRD
 # ->   9CE5 CJK UNIFIED IDEOGRAPH 9CE5
 "\x{2FC3}" => "\x{9CE5}",

 # 2FC4 KANGXI RADICAL SALT
 # ->   9E75 CJK UNIFIED IDEOGRAPH 9E75
 "\x{2FC4}" => "\x{9E75}",

 # 2FC5 KANGXI RADICAL DEER
 # ->   9E7F CJK UNIFIED IDEOGRAPH 9E7F
 "\x{2FC5}" => "\x{9E7F}",

 # 2FC6 KANGXI RADICAL WHEAT
 # ->   9EA5 CJK UNIFIED IDEOGRAPH 9EA5
 "\x{2FC6}" => "\x{9EA5}",

 # 2FC7 KANGXI RADICAL HEMP
 # ->   9EBB CJK UNIFIED IDEOGRAPH 9EBB
 "\x{2FC7}" => "\x{9EBB}",

 # 2FC8 KANGXI RADICAL YELLOW
 # ->   9EC3 CJK UNIFIED IDEOGRAPH 9EC3
 "\x{2FC8}" => "\x{9EC3}",

 # 2FC9 KANGXI RADICAL MILLET
 # ->   9ECD CJK UNIFIED IDEOGRAPH 9ECD
 "\x{2FC9}" => "\x{9ECD}",

 # 2FCA KANGXI RADICAL BLACK
 # ->   9ED1 CJK UNIFIED IDEOGRAPH 9ED1
 "\x{2FCA}" => "\x{9ED1}",

 # 2FCB KANGXI RADICAL EMBROIDERY
 # ->   9EF9 CJK UNIFIED IDEOGRAPH 9EF9
 "\x{2FCB}" => "\x{9EF9}",

 # 2FCC KANGXI RADICAL FROG
 # ->   9EFD CJK UNIFIED IDEOGRAPH 9EFD
 "\x{2FCC}" => "\x{9EFD}",

 # 2FCD KANGXI RADICAL TRIPOD
 # ->   9F0E CJK UNIFIED IDEOGRAPH 9F0E
 "\x{2FCD}" => "\x{9F0E}",

 # 2FCE KANGXI RADICAL DRUM
 # ->   9F13 CJK UNIFIED IDEOGRAPH 9F13
 "\x{2FCE}" => "\x{9F13}",

 # 2FCF KANGXI RADICAL RAT
 # ->   9F20 CJK UNIFIED IDEOGRAPH 9F20
 "\x{2FCF}" => "\x{9F20}",

 # 2FD0 KANGXI RADICAL NOSE
 # ->   9F3B CJK UNIFIED IDEOGRAPH 9F3B
 "\x{2FD0}" => "\x{9F3B}",

 # 2FD1 KANGXI RADICAL EVEN
 # ->   9F4A CJK UNIFIED IDEOGRAPH 9F4A
 "\x{2FD1}" => "\x{9F4A}",

 # 2FD2 KANGXI RADICAL TOOTH
 # ->   9F52 CJK UNIFIED IDEOGRAPH 9F52
 "\x{2FD2}" => "\x{9F52}",

 # 2FD3 KANGXI RADICAL DRAGON
 # ->   9F8D CJK UNIFIED IDEOGRAPH 9F8D
 "\x{2FD3}" => "\x{9F8D}",

 # 2FD4 KANGXI RADICAL TURTLE
 # ->   9F9C CJK UNIFIED IDEOGRAPH 9F9C
 "\x{2FD4}" => "\x{9F9C}",

 # 2FD5 KANGXI RADICAL FLUTE
 # ->   9FA0 CJK UNIFIED IDEOGRAPH 9FA0
 "\x{2FD5}" => "\x{9FA0}",

 # 3000 IDEOGRAPHIC SPACE
 # ->   0020 SPACE
 "\x{3000}" => " ",

 # 3036 CIRCLED POSTAL MARK
 # ->   3012 POSTAL MARK
 "\x{3036}" => "\x{3012}",

 # 3038 HANGZHOU NUMERAL TEN
 # ->   5341 CJK UNIFIED IDEOGRAPH 5341
 "\x{3038}" => "\x{5341}",

 # 3039 HANGZHOU NUMERAL TWENTY
 # ->   5344 CJK UNIFIED IDEOGRAPH 5344
 "\x{3039}" => "\x{5344}",

 # 303A HANGZHOU NUMERAL THIRTY
 # ->   5345 CJK UNIFIED IDEOGRAPH 5345
 "\x{303A}" => "\x{5345}",

 # 304C HIRAGANA LETTER GA
 # ->   304B HIRAGANA LETTER KA
 "\x{304C}" => "\x{304B}",

 # 304E HIRAGANA LETTER GI
 # ->   304D HIRAGANA LETTER KI
 "\x{304E}" => "\x{304D}",

 # 3050 HIRAGANA LETTER GU
 # ->   304F HIRAGANA LETTER KU
 "\x{3050}" => "\x{304F}",

 # 3052 HIRAGANA LETTER GE
 # ->   3051 HIRAGANA LETTER KE
 "\x{3052}" => "\x{3051}",

 # 3054 HIRAGANA LETTER GO
 # ->   3053 HIRAGANA LETTER KO
 "\x{3054}" => "\x{3053}",

 # 3056 HIRAGANA LETTER ZA
 # ->   3055 HIRAGANA LETTER SA
 "\x{3056}" => "\x{3055}",

 # 3058 HIRAGANA LETTER ZI
 # ->   3057 HIRAGANA LETTER SI
 "\x{3058}" => "\x{3057}",

 # 305A HIRAGANA LETTER ZU
 # ->   3059 HIRAGANA LETTER SU
 "\x{305A}" => "\x{3059}",

 # 305C HIRAGANA LETTER ZE
 # ->   305B HIRAGANA LETTER SE
 "\x{305C}" => "\x{305B}",

 # 305E HIRAGANA LETTER ZO
 # ->   305D HIRAGANA LETTER SO
 "\x{305E}" => "\x{305D}",

 # 3060 HIRAGANA LETTER DA
 # ->   305F HIRAGANA LETTER TA
 "\x{3060}" => "\x{305F}",

 # 3062 HIRAGANA LETTER DI
 # ->   3061 HIRAGANA LETTER TI
 "\x{3062}" => "\x{3061}",

 # 3065 HIRAGANA LETTER DU
 # ->   3064 HIRAGANA LETTER TU
 "\x{3065}" => "\x{3064}",

 # 3067 HIRAGANA LETTER DE
 # ->   3066 HIRAGANA LETTER TE
 "\x{3067}" => "\x{3066}",

 # 3069 HIRAGANA LETTER DO
 # ->   3068 HIRAGANA LETTER TO
 "\x{3069}" => "\x{3068}",

 # 3070 HIRAGANA LETTER BA
 # ->   306F HIRAGANA LETTER HA
 "\x{3070}" => "\x{306F}",

 # 3071 HIRAGANA LETTER PA
 # ->   306F HIRAGANA LETTER HA
 "\x{3071}" => "\x{306F}",

 # 3073 HIRAGANA LETTER BI
 # ->   3072 HIRAGANA LETTER HI
 "\x{3073}" => "\x{3072}",

 # 3074 HIRAGANA LETTER PI
 # ->   3072 HIRAGANA LETTER HI
 "\x{3074}" => "\x{3072}",

 # 3076 HIRAGANA LETTER BU
 # ->   3075 HIRAGANA LETTER HU
 "\x{3076}" => "\x{3075}",

 # 3077 HIRAGANA LETTER PU
 # ->   3075 HIRAGANA LETTER HU
 "\x{3077}" => "\x{3075}",

 # 3079 HIRAGANA LETTER BE
 # ->   3078 HIRAGANA LETTER HE
 "\x{3079}" => "\x{3078}",

 # 307A HIRAGANA LETTER PE
 # ->   3078 HIRAGANA LETTER HE
 "\x{307A}" => "\x{3078}",

 # 307C HIRAGANA LETTER BO
 # ->   307B HIRAGANA LETTER HO
 "\x{307C}" => "\x{307B}",

 # 307D HIRAGANA LETTER PO
 # ->   307B HIRAGANA LETTER HO
 "\x{307D}" => "\x{307B}",

 # 3094 HIRAGANA LETTER VU
 # ->   3046 HIRAGANA LETTER U
 "\x{3094}" => "\x{3046}",

 # 309B KATAKANA-HIRAGANA VOICED SOUND MARK
 # ->   0020 SPACE
 "\x{309B}" => " ",

 # 309C KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
 # ->   0020 SPACE
 "\x{309C}" => " ",

 # 309E HIRAGANA VOICED ITERATION MARK
 # ->   309D HIRAGANA ITERATION MARK
 "\x{309E}" => "\x{309D}",

 # 309F HIRAGANA DIGRAPH YORI
 # ->   3088 HIRAGANA LETTER YO
 #    + 308A HIRAGANA LETTER RI
 "\x{309F}" => "\x{3088}\x{308A}",

 # 30AC KATAKANA LETTER GA
 # ->   30AB KATAKANA LETTER KA
 "\x{30AC}" => "\x{30AB}",

 # 30AE KATAKANA LETTER GI
 # ->   30AD KATAKANA LETTER KI
 "\x{30AE}" => "\x{30AD}",

 # 30B0 KATAKANA LETTER GU
 # ->   30AF KATAKANA LETTER KU
 "\x{30B0}" => "\x{30AF}",

 # 30B2 KATAKANA LETTER GE
 # ->   30B1 KATAKANA LETTER KE
 "\x{30B2}" => "\x{30B1}",

 # 30B4 KATAKANA LETTER GO
 # ->   30B3 KATAKANA LETTER KO
 "\x{30B4}" => "\x{30B3}",

 # 30B6 KATAKANA LETTER ZA
 # ->   30B5 KATAKANA LETTER SA
 "\x{30B6}" => "\x{30B5}",

 # 30B8 KATAKANA LETTER ZI
 # ->   30B7 KATAKANA LETTER SI
 "\x{30B8}" => "\x{30B7}",

 # 30BA KATAKANA LETTER ZU
 # ->   30B9 KATAKANA LETTER SU
 "\x{30BA}" => "\x{30B9}",

 # 30BC KATAKANA LETTER ZE
 # ->   30BB KATAKANA LETTER SE
 "\x{30BC}" => "\x{30BB}",

 # 30BE KATAKANA LETTER ZO
 # ->   30BD KATAKANA LETTER SO
 "\x{30BE}" => "\x{30BD}",

 # 30C0 KATAKANA LETTER DA
 # ->   30BF KATAKANA LETTER TA
 "\x{30C0}" => "\x{30BF}",

 # 30C2 KATAKANA LETTER DI
 # ->   30C1 KATAKANA LETTER TI
 "\x{30C2}" => "\x{30C1}",

 # 30C5 KATAKANA LETTER DU
 # ->   30C4 KATAKANA LETTER TU
 "\x{30C5}" => "\x{30C4}",

 # 30C7 KATAKANA LETTER DE
 # ->   30C6 KATAKANA LETTER TE
 "\x{30C7}" => "\x{30C6}",

 # 30C9 KATAKANA LETTER DO
 # ->   30C8 KATAKANA LETTER TO
 "\x{30C9}" => "\x{30C8}",

 # 30D0 KATAKANA LETTER BA
 # ->   30CF KATAKANA LETTER HA
 "\x{30D0}" => "\x{30CF}",

 # 30D1 KATAKANA LETTER PA
 # ->   30CF KATAKANA LETTER HA
 "\x{30D1}" => "\x{30CF}",

 # 30D3 KATAKANA LETTER BI
 # ->   30D2 KATAKANA LETTER HI
 "\x{30D3}" => "\x{30D2}",

 # 30D4 KATAKANA LETTER PI
 # ->   30D2 KATAKANA LETTER HI
 "\x{30D4}" => "\x{30D2}",

 # 30D6 KATAKANA LETTER BU
 # ->   30D5 KATAKANA LETTER HU
 "\x{30D6}" => "\x{30D5}",

 # 30D7 KATAKANA LETTER PU
 # ->   30D5 KATAKANA LETTER HU
 "\x{30D7}" => "\x{30D5}",

 # 30D9 KATAKANA LETTER BE
 # ->   30D8 KATAKANA LETTER HE
 "\x{30D9}" => "\x{30D8}",

 # 30DA KATAKANA LETTER PE
 # ->   30D8 KATAKANA LETTER HE
 "\x{30DA}" => "\x{30D8}",

 # 30DC KATAKANA LETTER BO
 # ->   30DB KATAKANA LETTER HO
 "\x{30DC}" => "\x{30DB}",

 # 30DD KATAKANA LETTER PO
 # ->   30DB KATAKANA LETTER HO
 "\x{30DD}" => "\x{30DB}",

 # 30F4 KATAKANA LETTER VU
 # ->   30A6 KATAKANA LETTER U
 "\x{30F4}" => "\x{30A6}",

 # 30F7 KATAKANA LETTER VA
 # ->   30EF KATAKANA LETTER WA
 "\x{30F7}" => "\x{30EF}",

 # 30F8 KATAKANA LETTER VI
 # ->   30F0 KATAKANA LETTER WI
 "\x{30F8}" => "\x{30F0}",

 # 30F9 KATAKANA LETTER VE
 # ->   30F1 KATAKANA LETTER WE
 "\x{30F9}" => "\x{30F1}",

 # 30FA KATAKANA LETTER VO
 # ->   30F2 KATAKANA LETTER WO
 "\x{30FA}" => "\x{30F2}",

 # 30FE KATAKANA VOICED ITERATION MARK
 # ->   30FD KATAKANA ITERATION MARK
 "\x{30FE}" => "\x{30FD}",

 # 30FF KATAKANA DIGRAPH KOTO
 # ->   30B3 KATAKANA LETTER KO
 #    + 30C8 KATAKANA LETTER TO
 "\x{30FF}" => "\x{30B3}\x{30C8}",

 # 3131 HANGUL LETTER KIYEOK
 # ->   1100 HANGUL CHOSEONG KIYEOK
 "\x{3131}" => "\x{1100}",

 # 3132 HANGUL LETTER SSANGKIYEOK
 # ->   1101 HANGUL CHOSEONG SSANGKIYEOK
 "\x{3132}" => "\x{1101}",

 # 3133 HANGUL LETTER KIYEOK-SIOS
 # ->   11AA HANGUL JONGSEONG KIYEOK-SIOS
 "\x{3133}" => "\x{11AA}",

 # 3134 HANGUL LETTER NIEUN
 # ->   1102 HANGUL CHOSEONG NIEUN
 "\x{3134}" => "\x{1102}",

 # 3135 HANGUL LETTER NIEUN-CIEUC
 # ->   11AC HANGUL JONGSEONG NIEUN-CIEUC
 "\x{3135}" => "\x{11AC}",

 # 3136 HANGUL LETTER NIEUN-HIEUH
 # ->   11AD HANGUL JONGSEONG NIEUN-HIEUH
 "\x{3136}" => "\x{11AD}",

 # 3137 HANGUL LETTER TIKEUT
 # ->   1103 HANGUL CHOSEONG TIKEUT
 "\x{3137}" => "\x{1103}",

 # 3138 HANGUL LETTER SSANGTIKEUT
 # ->   1104 HANGUL CHOSEONG SSANGTIKEUT
 "\x{3138}" => "\x{1104}",

 # 3139 HANGUL LETTER RIEUL
 # ->   1105 HANGUL CHOSEONG RIEUL
 "\x{3139}" => "\x{1105}",

 # 313A HANGUL LETTER RIEUL-KIYEOK
 # ->   11B0 HANGUL JONGSEONG RIEUL-KIYEOK
 "\x{313A}" => "\x{11B0}",

 # 313B HANGUL LETTER RIEUL-MIEUM
 # ->   11B1 HANGUL JONGSEONG RIEUL-MIEUM
 "\x{313B}" => "\x{11B1}",

 # 313C HANGUL LETTER RIEUL-PIEUP
 # ->   11B2 HANGUL JONGSEONG RIEUL-PIEUP
 "\x{313C}" => "\x{11B2}",

 # 313D HANGUL LETTER RIEUL-SIOS
 # ->   11B3 HANGUL JONGSEONG RIEUL-SIOS
 "\x{313D}" => "\x{11B3}",

 # 313E HANGUL LETTER RIEUL-THIEUTH
 # ->   11B4 HANGUL JONGSEONG RIEUL-THIEUTH
 "\x{313E}" => "\x{11B4}",

 # 313F HANGUL LETTER RIEUL-PHIEUPH
 # ->   11B5 HANGUL JONGSEONG RIEUL-PHIEUPH
 "\x{313F}" => "\x{11B5}",

 # 3140 HANGUL LETTER RIEUL-HIEUH
 # ->   111A HANGUL CHOSEONG RIEUL-HIEUH
 "\x{3140}" => "\x{111A}",

 # 3141 HANGUL LETTER MIEUM
 # ->   1106 HANGUL CHOSEONG MIEUM
 "\x{3141}" => "\x{1106}",

 # 3142 HANGUL LETTER PIEUP
 # ->   1107 HANGUL CHOSEONG PIEUP
 "\x{3142}" => "\x{1107}",

 # 3143 HANGUL LETTER SSANGPIEUP
 # ->   1108 HANGUL CHOSEONG SSANGPIEUP
 "\x{3143}" => "\x{1108}",

 # 3144 HANGUL LETTER PIEUP-SIOS
 # ->   1121 HANGUL CHOSEONG PIEUP-SIOS
 "\x{3144}" => "\x{1121}",

 # 3145 HANGUL LETTER SIOS
 # ->   1109 HANGUL CHOSEONG SIOS
 "\x{3145}" => "\x{1109}",

 # 3146 HANGUL LETTER SSANGSIOS
 # ->   110A HANGUL CHOSEONG SSANGSIOS
 "\x{3146}" => "\x{110A}",

 # 3147 HANGUL LETTER IEUNG
 # ->   110B HANGUL CHOSEONG IEUNG
 "\x{3147}" => "\x{110B}",

 # 3148 HANGUL LETTER CIEUC
 # ->   110C HANGUL CHOSEONG CIEUC
 "\x{3148}" => "\x{110C}",

 # 3149 HANGUL LETTER SSANGCIEUC
 # ->   110D HANGUL CHOSEONG SSANGCIEUC
 "\x{3149}" => "\x{110D}",

 # 314A HANGUL LETTER CHIEUCH
 # ->   110E HANGUL CHOSEONG CHIEUCH
 "\x{314A}" => "\x{110E}",

 # 314B HANGUL LETTER KHIEUKH
 # ->   110F HANGUL CHOSEONG KHIEUKH
 "\x{314B}" => "\x{110F}",

 # 314C HANGUL LETTER THIEUTH
 # ->   1110 HANGUL CHOSEONG THIEUTH
 "\x{314C}" => "\x{1110}",

 # 314D HANGUL LETTER PHIEUPH
 # ->   1111 HANGUL CHOSEONG PHIEUPH
 "\x{314D}" => "\x{1111}",

 # 314E HANGUL LETTER HIEUH
 # ->   1112 HANGUL CHOSEONG HIEUH
 "\x{314E}" => "\x{1112}",

 # 314F HANGUL LETTER A
 # ->   1161 HANGUL JUNGSEONG A
 "\x{314F}" => "\x{1161}",

 # 3150 HANGUL LETTER AE
 # ->   1162 HANGUL JUNGSEONG AE
 "\x{3150}" => "\x{1162}",

 # 3151 HANGUL LETTER YA
 # ->   1163 HANGUL JUNGSEONG YA
 "\x{3151}" => "\x{1163}",

 # 3152 HANGUL LETTER YAE
 # ->   1164 HANGUL JUNGSEONG YAE
 "\x{3152}" => "\x{1164}",

 # 3153 HANGUL LETTER EO
 # ->   1165 HANGUL JUNGSEONG EO
 "\x{3153}" => "\x{1165}",

 # 3154 HANGUL LETTER E
 # ->   1166 HANGUL JUNGSEONG E
 "\x{3154}" => "\x{1166}",

 # 3155 HANGUL LETTER YEO
 # ->   1167 HANGUL JUNGSEONG YEO
 "\x{3155}" => "\x{1167}",

 # 3156 HANGUL LETTER YE
 # ->   1168 HANGUL JUNGSEONG YE
 "\x{3156}" => "\x{1168}",

 # 3157 HANGUL LETTER O
 # ->   1169 HANGUL JUNGSEONG O
 "\x{3157}" => "\x{1169}",

 # 3158 HANGUL LETTER WA
 # ->   116A HANGUL JUNGSEONG WA
 "\x{3158}" => "\x{116A}",

 # 3159 HANGUL LETTER WAE
 # ->   116B HANGUL JUNGSEONG WAE
 "\x{3159}" => "\x{116B}",

 # 315A HANGUL LETTER OE
 # ->   116C HANGUL JUNGSEONG OE
 "\x{315A}" => "\x{116C}",

 # 315B HANGUL LETTER YO
 # ->   116D HANGUL JUNGSEONG YO
 "\x{315B}" => "\x{116D}",

 # 315C HANGUL LETTER U
 # ->   116E HANGUL JUNGSEONG U
 "\x{315C}" => "\x{116E}",

 # 315D HANGUL LETTER WEO
 # ->   116F HANGUL JUNGSEONG WEO
 "\x{315D}" => "\x{116F}",

 # 315E HANGUL LETTER WE
 # ->   1170 HANGUL JUNGSEONG WE
 "\x{315E}" => "\x{1170}",

 # 315F HANGUL LETTER WI
 # ->   1171 HANGUL JUNGSEONG WI
 "\x{315F}" => "\x{1171}",

 # 3160 HANGUL LETTER YU
 # ->   1172 HANGUL JUNGSEONG YU
 "\x{3160}" => "\x{1172}",

 # 3161 HANGUL LETTER EU
 # ->   1173 HANGUL JUNGSEONG EU
 "\x{3161}" => "\x{1173}",

 # 3162 HANGUL LETTER YI
 # ->   1174 HANGUL JUNGSEONG YI
 "\x{3162}" => "\x{1174}",

 # 3163 HANGUL LETTER I
 # ->   1175 HANGUL JUNGSEONG I
 "\x{3163}" => "\x{1175}",

 # 3164 HANGUL FILLER
 # ->   1160 HANGUL JUNGSEONG FILLER
 "\x{3164}" => "\x{1160}",

 # 3165 HANGUL LETTER SSANGNIEUN
 # ->   1114 HANGUL CHOSEONG SSANGNIEUN
 "\x{3165}" => "\x{1114}",

 # 3166 HANGUL LETTER NIEUN-TIKEUT
 # ->   1115 HANGUL CHOSEONG NIEUN-TIKEUT
 "\x{3166}" => "\x{1115}",

 # 3167 HANGUL LETTER NIEUN-SIOS
 # ->   11C7 HANGUL JONGSEONG NIEUN-SIOS
 "\x{3167}" => "\x{11C7}",

 # 3168 HANGUL LETTER NIEUN-PANSIOS
 # ->   11C8 HANGUL JONGSEONG NIEUN-PANSIOS
 "\x{3168}" => "\x{11C8}",

 # 3169 HANGUL LETTER RIEUL-KIYEOK-SIOS
 # ->   11CC HANGUL JONGSEONG RIEUL-KIYEOK-SIOS
 "\x{3169}" => "\x{11CC}",

 # 316A HANGUL LETTER RIEUL-TIKEUT
 # ->   11CE HANGUL JONGSEONG RIEUL-TIKEUT
 "\x{316A}" => "\x{11CE}",

 # 316B HANGUL LETTER RIEUL-PIEUP-SIOS
 # ->   11D3 HANGUL JONGSEONG RIEUL-PIEUP-SIOS
 "\x{316B}" => "\x{11D3}",

 # 316C HANGUL LETTER RIEUL-PANSIOS
 # ->   11D7 HANGUL JONGSEONG RIEUL-PANSIOS
 "\x{316C}" => "\x{11D7}",

 # 316D HANGUL LETTER RIEUL-YEORINHIEUH
 # ->   11D9 HANGUL JONGSEONG RIEUL-YEORINHIEUH
 "\x{316D}" => "\x{11D9}",

 # 316E HANGUL LETTER MIEUM-PIEUP
 # ->   111C HANGUL CHOSEONG MIEUM-PIEUP
 "\x{316E}" => "\x{111C}",

 # 316F HANGUL LETTER MIEUM-SIOS
 # ->   11DD HANGUL JONGSEONG MIEUM-SIOS
 "\x{316F}" => "\x{11DD}",

 # 3170 HANGUL LETTER MIEUM-PANSIOS
 # ->   11DF HANGUL JONGSEONG MIEUM-PANSIOS
 "\x{3170}" => "\x{11DF}",

 # 3171 HANGUL LETTER KAPYEOUNMIEUM
 # ->   111D HANGUL CHOSEONG KAPYEOUNMIEUM
 "\x{3171}" => "\x{111D}",

 # 3172 HANGUL LETTER PIEUP-KIYEOK
 # ->   111E HANGUL CHOSEONG PIEUP-KIYEOK
 "\x{3172}" => "\x{111E}",

 # 3173 HANGUL LETTER PIEUP-TIKEUT
 # ->   1120 HANGUL CHOSEONG PIEUP-TIKEUT
 "\x{3173}" => "\x{1120}",

 # 3174 HANGUL LETTER PIEUP-SIOS-KIYEOK
 # ->   1122 HANGUL CHOSEONG PIEUP-SIOS-KIYEOK
 "\x{3174}" => "\x{1122}",

 # 3175 HANGUL LETTER PIEUP-SIOS-TIKEUT
 # ->   1123 HANGUL CHOSEONG PIEUP-SIOS-TIKEUT
 "\x{3175}" => "\x{1123}",

 # 3176 HANGUL LETTER PIEUP-CIEUC
 # ->   1127 HANGUL CHOSEONG PIEUP-CIEUC
 "\x{3176}" => "\x{1127}",

 # 3177 HANGUL LETTER PIEUP-THIEUTH
 # ->   1129 HANGUL CHOSEONG PIEUP-THIEUTH
 "\x{3177}" => "\x{1129}",

 # 3178 HANGUL LETTER KAPYEOUNPIEUP
 # ->   112B HANGUL CHOSEONG KAPYEOUNPIEUP
 "\x{3178}" => "\x{112B}",

 # 3179 HANGUL LETTER KAPYEOUNSSANGPIEUP
 # ->   112C HANGUL CHOSEONG KAPYEOUNSSANGPIEUP
 "\x{3179}" => "\x{112C}",

 # 317A HANGUL LETTER SIOS-KIYEOK
 # ->   112D HANGUL CHOSEONG SIOS-KIYEOK
 "\x{317A}" => "\x{112D}",

 # 317B HANGUL LETTER SIOS-NIEUN
 # ->   112E HANGUL CHOSEONG SIOS-NIEUN
 "\x{317B}" => "\x{112E}",

 # 317C HANGUL LETTER SIOS-TIKEUT
 # ->   112F HANGUL CHOSEONG SIOS-TIKEUT
 "\x{317C}" => "\x{112F}",

 # 317D HANGUL LETTER SIOS-PIEUP
 # ->   1132 HANGUL CHOSEONG SIOS-PIEUP
 "\x{317D}" => "\x{1132}",

 # 317E HANGUL LETTER SIOS-CIEUC
 # ->   1136 HANGUL CHOSEONG SIOS-CIEUC
 "\x{317E}" => "\x{1136}",

 # 317F HANGUL LETTER PANSIOS
 # ->   1140 HANGUL CHOSEONG PANSIOS
 "\x{317F}" => "\x{1140}",

 # 3180 HANGUL LETTER SSANGIEUNG
 # ->   1147 HANGUL CHOSEONG SSANGIEUNG
 "\x{3180}" => "\x{1147}",

 # 3181 HANGUL LETTER YESIEUNG
 # ->   114C HANGUL CHOSEONG YESIEUNG
 "\x{3181}" => "\x{114C}",

 # 3182 HANGUL LETTER YESIEUNG-SIOS
 # ->   11F1 HANGUL JONGSEONG YESIEUNG-SIOS
 "\x{3182}" => "\x{11F1}",

 # 3183 HANGUL LETTER YESIEUNG-PANSIOS
 # ->   11F2 HANGUL JONGSEONG YESIEUNG-PANSIOS
 "\x{3183}" => "\x{11F2}",

 # 3184 HANGUL LETTER KAPYEOUNPHIEUPH
 # ->   1157 HANGUL CHOSEONG KAPYEOUNPHIEUPH
 "\x{3184}" => "\x{1157}",

 # 3185 HANGUL LETTER SSANGHIEUH
 # ->   1158 HANGUL CHOSEONG SSANGHIEUH
 "\x{3185}" => "\x{1158}",

 # 3186 HANGUL LETTER YEORINHIEUH
 # ->   1159 HANGUL CHOSEONG YEORINHIEUH
 "\x{3186}" => "\x{1159}",

 # 3187 HANGUL LETTER YO-YA
 # ->   1184 HANGUL JUNGSEONG YO-YA
 "\x{3187}" => "\x{1184}",

 # 3188 HANGUL LETTER YO-YAE
 # ->   1185 HANGUL JUNGSEONG YO-YAE
 "\x{3188}" => "\x{1185}",

 # 3189 HANGUL LETTER YO-I
 # ->   1188 HANGUL JUNGSEONG YO-I
 "\x{3189}" => "\x{1188}",

 # 318A HANGUL LETTER YU-YEO
 # ->   1191 HANGUL JUNGSEONG YU-YEO
 "\x{318A}" => "\x{1191}",

 # 318B HANGUL LETTER YU-YE
 # ->   1192 HANGUL JUNGSEONG YU-YE
 "\x{318B}" => "\x{1192}",

 # 318C HANGUL LETTER YU-I
 # ->   1194 HANGUL JUNGSEONG YU-I
 "\x{318C}" => "\x{1194}",

 # 318D HANGUL LETTER ARAEA
 # ->   119E HANGUL JUNGSEONG ARAEA
 "\x{318D}" => "\x{119E}",

 # 318E HANGUL LETTER ARAEAE
 # ->   11A1 HANGUL JUNGSEONG ARAEA-I
 "\x{318E}" => "\x{11A1}",

 # 3192 IDEOGRAPHIC ANNOTATION ONE MARK
 # ->   4E00 CJK UNIFIED IDEOGRAPH 4E00
 "\x{3192}" => "\x{4E00}",

 # 3193 IDEOGRAPHIC ANNOTATION TWO MARK
 # ->   4E8C CJK UNIFIED IDEOGRAPH 4E8C
 "\x{3193}" => "\x{4E8C}",

 # 3194 IDEOGRAPHIC ANNOTATION THREE MARK
 # ->   4E09 CJK UNIFIED IDEOGRAPH 4E09
 "\x{3194}" => "\x{4E09}",

 # 3195 IDEOGRAPHIC ANNOTATION FOUR MARK
 # ->   56DB CJK UNIFIED IDEOGRAPH 56DB
 "\x{3195}" => "\x{56DB}",

 # 3196 IDEOGRAPHIC ANNOTATION TOP MARK
 # ->   4E0A CJK UNIFIED IDEOGRAPH 4E0A
 "\x{3196}" => "\x{4E0A}",

 # 3197 IDEOGRAPHIC ANNOTATION MIDDLE MARK
 # ->   4E2D CJK UNIFIED IDEOGRAPH 4E2D
 "\x{3197}" => "\x{4E2D}",

 # 3198 IDEOGRAPHIC ANNOTATION BOTTOM MARK
 # ->   4E0B CJK UNIFIED IDEOGRAPH 4E0B
 "\x{3198}" => "\x{4E0B}",

 # 3199 IDEOGRAPHIC ANNOTATION FIRST MARK
 # ->   7532 CJK UNIFIED IDEOGRAPH 7532
 "\x{3199}" => "\x{7532}",

 # 319A IDEOGRAPHIC ANNOTATION SECOND MARK
 # ->   4E59 CJK UNIFIED IDEOGRAPH 4E59
 "\x{319A}" => "\x{4E59}",

 # 319B IDEOGRAPHIC ANNOTATION THIRD MARK
 # ->   4E19 CJK UNIFIED IDEOGRAPH 4E19
 "\x{319B}" => "\x{4E19}",

 # 319C IDEOGRAPHIC ANNOTATION FOURTH MARK
 # ->   4E01 CJK UNIFIED IDEOGRAPH 4E01
 "\x{319C}" => "\x{4E01}",

 # 319D IDEOGRAPHIC ANNOTATION HEAVEN MARK
 # ->   5929 CJK UNIFIED IDEOGRAPH 5929
 "\x{319D}" => "\x{5929}",

 # 319E IDEOGRAPHIC ANNOTATION EARTH MARK
 # ->   5730 CJK UNIFIED IDEOGRAPH 5730
 "\x{319E}" => "\x{5730}",

 # 319F IDEOGRAPHIC ANNOTATION MAN MARK
 # ->   4EBA CJK UNIFIED IDEOGRAPH 4EBA
 "\x{319F}" => "\x{4EBA}",

 # 3200 PARENTHESIZED HANGUL KIYEOK
 # ->   0028 LEFT PARENTHESIS
 #    + 1100 HANGUL CHOSEONG KIYEOK
 #    + 0029 RIGHT PARENTHESIS
 "\x{3200}" => "(\x{1100})",

 # 3201 PARENTHESIZED HANGUL NIEUN
 # ->   0028 LEFT PARENTHESIS
 #    + 1102 HANGUL CHOSEONG NIEUN
 #    + 0029 RIGHT PARENTHESIS
 "\x{3201}" => "(\x{1102})",

 # 3202 PARENTHESIZED HANGUL TIKEUT
 # ->   0028 LEFT PARENTHESIS
 #    + 1103 HANGUL CHOSEONG TIKEUT
 #    + 0029 RIGHT PARENTHESIS
 "\x{3202}" => "(\x{1103})",

 # 3203 PARENTHESIZED HANGUL RIEUL
 # ->   0028 LEFT PARENTHESIS
 #    + 1105 HANGUL CHOSEONG RIEUL
 #    + 0029 RIGHT PARENTHESIS
 "\x{3203}" => "(\x{1105})",

 # 3204 PARENTHESIZED HANGUL MIEUM
 # ->   0028 LEFT PARENTHESIS
 #    + 1106 HANGUL CHOSEONG MIEUM
 #    + 0029 RIGHT PARENTHESIS
 "\x{3204}" => "(\x{1106})",

 # 3205 PARENTHESIZED HANGUL PIEUP
 # ->   0028 LEFT PARENTHESIS
 #    + 1107 HANGUL CHOSEONG PIEUP
 #    + 0029 RIGHT PARENTHESIS
 "\x{3205}" => "(\x{1107})",

 # 3206 PARENTHESIZED HANGUL SIOS
 # ->   0028 LEFT PARENTHESIS
 #    + 1109 HANGUL CHOSEONG SIOS
 #    + 0029 RIGHT PARENTHESIS
 "\x{3206}" => "(\x{1109})",

 # 3207 PARENTHESIZED HANGUL IEUNG
 # ->   0028 LEFT PARENTHESIS
 #    + 110B HANGUL CHOSEONG IEUNG
 #    + 0029 RIGHT PARENTHESIS
 "\x{3207}" => "(\x{110B})",

 # 3208 PARENTHESIZED HANGUL CIEUC
 # ->   0028 LEFT PARENTHESIS
 #    + 110C HANGUL CHOSEONG CIEUC
 #    + 0029 RIGHT PARENTHESIS
 "\x{3208}" => "(\x{110C})",

 # 3209 PARENTHESIZED HANGUL CHIEUCH
 # ->   0028 LEFT PARENTHESIS
 #    + 110E HANGUL CHOSEONG CHIEUCH
 #    + 0029 RIGHT PARENTHESIS
 "\x{3209}" => "(\x{110E})",

 # 320A PARENTHESIZED HANGUL KHIEUKH
 # ->   0028 LEFT PARENTHESIS
 #    + 110F HANGUL CHOSEONG KHIEUKH
 #    + 0029 RIGHT PARENTHESIS
 "\x{320A}" => "(\x{110F})",

 # 320B PARENTHESIZED HANGUL THIEUTH
 # ->   0028 LEFT PARENTHESIS
 #    + 1110 HANGUL CHOSEONG THIEUTH
 #    + 0029 RIGHT PARENTHESIS
 "\x{320B}" => "(\x{1110})",

 # 320C PARENTHESIZED HANGUL PHIEUPH
 # ->   0028 LEFT PARENTHESIS
 #    + 1111 HANGUL CHOSEONG PHIEUPH
 #    + 0029 RIGHT PARENTHESIS
 "\x{320C}" => "(\x{1111})",

 # 320D PARENTHESIZED HANGUL HIEUH
 # ->   0028 LEFT PARENTHESIS
 #    + 1112 HANGUL CHOSEONG HIEUH
 #    + 0029 RIGHT PARENTHESIS
 "\x{320D}" => "(\x{1112})",

 # 320E PARENTHESIZED HANGUL KIYEOK A
 # ->   0028 LEFT PARENTHESIS
 #    + 1100 HANGUL CHOSEONG KIYEOK
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{320E}" => "(\x{1100}\x{1161})",

 # 320F PARENTHESIZED HANGUL NIEUN A
 # ->   0028 LEFT PARENTHESIS
 #    + 1102 HANGUL CHOSEONG NIEUN
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{320F}" => "(\x{1102}\x{1161})",

 # 3210 PARENTHESIZED HANGUL TIKEUT A
 # ->   0028 LEFT PARENTHESIS
 #    + 1103 HANGUL CHOSEONG TIKEUT
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{3210}" => "(\x{1103}\x{1161})",

 # 3211 PARENTHESIZED HANGUL RIEUL A
 # ->   0028 LEFT PARENTHESIS
 #    + 1105 HANGUL CHOSEONG RIEUL
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{3211}" => "(\x{1105}\x{1161})",

 # 3212 PARENTHESIZED HANGUL MIEUM A
 # ->   0028 LEFT PARENTHESIS
 #    + 1106 HANGUL CHOSEONG MIEUM
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{3212}" => "(\x{1106}\x{1161})",

 # 3213 PARENTHESIZED HANGUL PIEUP A
 # ->   0028 LEFT PARENTHESIS
 #    + 1107 HANGUL CHOSEONG PIEUP
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{3213}" => "(\x{1107}\x{1161})",

 # 3214 PARENTHESIZED HANGUL SIOS A
 # ->   0028 LEFT PARENTHESIS
 #    + 1109 HANGUL CHOSEONG SIOS
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{3214}" => "(\x{1109}\x{1161})",

 # 3215 PARENTHESIZED HANGUL IEUNG A
 # ->   0028 LEFT PARENTHESIS
 #    + 110B HANGUL CHOSEONG IEUNG
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{3215}" => "(\x{110B}\x{1161})",

 # 3216 PARENTHESIZED HANGUL CIEUC A
 # ->   0028 LEFT PARENTHESIS
 #    + 110C HANGUL CHOSEONG CIEUC
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{3216}" => "(\x{110C}\x{1161})",

 # 3217 PARENTHESIZED HANGUL CHIEUCH A
 # ->   0028 LEFT PARENTHESIS
 #    + 110E HANGUL CHOSEONG CHIEUCH
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{3217}" => "(\x{110E}\x{1161})",

 # 3218 PARENTHESIZED HANGUL KHIEUKH A
 # ->   0028 LEFT PARENTHESIS
 #    + 110F HANGUL CHOSEONG KHIEUKH
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{3218}" => "(\x{110F}\x{1161})",

 # 3219 PARENTHESIZED HANGUL THIEUTH A
 # ->   0028 LEFT PARENTHESIS
 #    + 1110 HANGUL CHOSEONG THIEUTH
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{3219}" => "(\x{1110}\x{1161})",

 # 321A PARENTHESIZED HANGUL PHIEUPH A
 # ->   0028 LEFT PARENTHESIS
 #    + 1111 HANGUL CHOSEONG PHIEUPH
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{321A}" => "(\x{1111}\x{1161})",

 # 321B PARENTHESIZED HANGUL HIEUH A
 # ->   0028 LEFT PARENTHESIS
 #    + 1112 HANGUL CHOSEONG HIEUH
 #    + 1161 HANGUL JUNGSEONG A
 #    + 0029 RIGHT PARENTHESIS
 "\x{321B}" => "(\x{1112}\x{1161})",

 # 321C PARENTHESIZED HANGUL CIEUC U
 # ->   0028 LEFT PARENTHESIS
 #    + 110C HANGUL CHOSEONG CIEUC
 #    + 116E HANGUL JUNGSEONG U
 #    + 0029 RIGHT PARENTHESIS
 "\x{321C}" => "(\x{110C}\x{116E})",

 # 321D PARENTHESIZED KOREAN CHARACTER OJEON
 # ->   0028 LEFT PARENTHESIS
 #    + 110B HANGUL CHOSEONG IEUNG
 #    + 1169 HANGUL JUNGSEONG O
 #    + 110C HANGUL CHOSEONG CIEUC
 #    + 1165 HANGUL JUNGSEONG EO
 #    + 11AB HANGUL JONGSEONG NIEUN
 #    + 0029 RIGHT PARENTHESIS
 "\x{321D}" => "(\x{110B}\x{1169}\x{110C}\x{1165}\x{11AB})",

 # 321E PARENTHESIZED KOREAN CHARACTER O HU
 # ->   0028 LEFT PARENTHESIS
 #    + 110B HANGUL CHOSEONG IEUNG
 #    + 1169 HANGUL JUNGSEONG O
 #    + 1112 HANGUL CHOSEONG HIEUH
 #    + 116E HANGUL JUNGSEONG U
 #    + 0029 RIGHT PARENTHESIS
 "\x{321E}" => "(\x{110B}\x{1169}\x{1112}\x{116E})",

 # 3220 PARENTHESIZED IDEOGRAPH ONE
 # ->   0028 LEFT PARENTHESIS
 #    + 4E00 CJK UNIFIED IDEOGRAPH 4E00
 #    + 0029 RIGHT PARENTHESIS
 "\x{3220}" => "(\x{4E00})",

 # 3221 PARENTHESIZED IDEOGRAPH TWO
 # ->   0028 LEFT PARENTHESIS
 #    + 4E8C CJK UNIFIED IDEOGRAPH 4E8C
 #    + 0029 RIGHT PARENTHESIS
 "\x{3221}" => "(\x{4E8C})",

 # 3222 PARENTHESIZED IDEOGRAPH THREE
 # ->   0028 LEFT PARENTHESIS
 #    + 4E09 CJK UNIFIED IDEOGRAPH 4E09
 #    + 0029 RIGHT PARENTHESIS
 "\x{3222}" => "(\x{4E09})",

 # 3223 PARENTHESIZED IDEOGRAPH FOUR
 # ->   0028 LEFT PARENTHESIS
 #    + 56DB CJK UNIFIED IDEOGRAPH 56DB
 #    + 0029 RIGHT PARENTHESIS
 "\x{3223}" => "(\x{56DB})",

 # 3224 PARENTHESIZED IDEOGRAPH FIVE
 # ->   0028 LEFT PARENTHESIS
 #    + 4E94 CJK UNIFIED IDEOGRAPH 4E94
 #    + 0029 RIGHT PARENTHESIS
 "\x{3224}" => "(\x{4E94})",

 # 3225 PARENTHESIZED IDEOGRAPH SIX
 # ->   0028 LEFT PARENTHESIS
 #    + 516D CJK UNIFIED IDEOGRAPH 516D
 #    + 0029 RIGHT PARENTHESIS
 "\x{3225}" => "(\x{516D})",

 # 3226 PARENTHESIZED IDEOGRAPH SEVEN
 # ->   0028 LEFT PARENTHESIS
 #    + 4E03 CJK UNIFIED IDEOGRAPH 4E03
 #    + 0029 RIGHT PARENTHESIS
 "\x{3226}" => "(\x{4E03})",

 # 3227 PARENTHESIZED IDEOGRAPH EIGHT
 # ->   0028 LEFT PARENTHESIS
 #    + 516B CJK UNIFIED IDEOGRAPH 516B
 #    + 0029 RIGHT PARENTHESIS
 "\x{3227}" => "(\x{516B})",

 # 3228 PARENTHESIZED IDEOGRAPH NINE
 # ->   0028 LEFT PARENTHESIS
 #    + 4E5D CJK UNIFIED IDEOGRAPH 4E5D
 #    + 0029 RIGHT PARENTHESIS
 "\x{3228}" => "(\x{4E5D})",

 # 3229 PARENTHESIZED IDEOGRAPH TEN
 # ->   0028 LEFT PARENTHESIS
 #    + 5341 CJK UNIFIED IDEOGRAPH 5341
 #    + 0029 RIGHT PARENTHESIS
 "\x{3229}" => "(\x{5341})",

 # 322A PARENTHESIZED IDEOGRAPH MOON
 # ->   0028 LEFT PARENTHESIS
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 #    + 0029 RIGHT PARENTHESIS
 "\x{322A}" => "(\x{6708})",

 # 322B PARENTHESIZED IDEOGRAPH FIRE
 # ->   0028 LEFT PARENTHESIS
 #    + 706B CJK UNIFIED IDEOGRAPH 706B
 #    + 0029 RIGHT PARENTHESIS
 "\x{322B}" => "(\x{706B})",

 # 322C PARENTHESIZED IDEOGRAPH WATER
 # ->   0028 LEFT PARENTHESIS
 #    + 6C34 CJK UNIFIED IDEOGRAPH 6C34
 #    + 0029 RIGHT PARENTHESIS
 "\x{322C}" => "(\x{6C34})",

 # 322D PARENTHESIZED IDEOGRAPH WOOD
 # ->   0028 LEFT PARENTHESIS
 #    + 6728 CJK UNIFIED IDEOGRAPH 6728
 #    + 0029 RIGHT PARENTHESIS
 "\x{322D}" => "(\x{6728})",

 # 322E PARENTHESIZED IDEOGRAPH METAL
 # ->   0028 LEFT PARENTHESIS
 #    + 91D1 CJK UNIFIED IDEOGRAPH 91D1
 #    + 0029 RIGHT PARENTHESIS
 "\x{322E}" => "(\x{91D1})",

 # 322F PARENTHESIZED IDEOGRAPH EARTH
 # ->   0028 LEFT PARENTHESIS
 #    + 571F CJK UNIFIED IDEOGRAPH 571F
 #    + 0029 RIGHT PARENTHESIS
 "\x{322F}" => "(\x{571F})",

 # 3230 PARENTHESIZED IDEOGRAPH SUN
 # ->   0028 LEFT PARENTHESIS
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 #    + 0029 RIGHT PARENTHESIS
 "\x{3230}" => "(\x{65E5})",

 # 3231 PARENTHESIZED IDEOGRAPH STOCK
 # ->   0028 LEFT PARENTHESIS
 #    + 682A CJK UNIFIED IDEOGRAPH 682A
 #    + 0029 RIGHT PARENTHESIS
 "\x{3231}" => "(\x{682A})",

 # 3232 PARENTHESIZED IDEOGRAPH HAVE
 # ->   0028 LEFT PARENTHESIS
 #    + 6709 CJK UNIFIED IDEOGRAPH 6709
 #    + 0029 RIGHT PARENTHESIS
 "\x{3232}" => "(\x{6709})",

 # 3233 PARENTHESIZED IDEOGRAPH SOCIETY
 # ->   0028 LEFT PARENTHESIS
 #    + 793E CJK UNIFIED IDEOGRAPH 793E
 #    + 0029 RIGHT PARENTHESIS
 "\x{3233}" => "(\x{793E})",

 # 3234 PARENTHESIZED IDEOGRAPH NAME
 # ->   0028 LEFT PARENTHESIS
 #    + 540D CJK UNIFIED IDEOGRAPH 540D
 #    + 0029 RIGHT PARENTHESIS
 "\x{3234}" => "(\x{540D})",

 # 3235 PARENTHESIZED IDEOGRAPH SPECIAL
 # ->   0028 LEFT PARENTHESIS
 #    + 7279 CJK UNIFIED IDEOGRAPH 7279
 #    + 0029 RIGHT PARENTHESIS
 "\x{3235}" => "(\x{7279})",

 # 3236 PARENTHESIZED IDEOGRAPH FINANCIAL
 # ->   0028 LEFT PARENTHESIS
 #    + 8CA1 CJK UNIFIED IDEOGRAPH 8CA1
 #    + 0029 RIGHT PARENTHESIS
 "\x{3236}" => "(\x{8CA1})",

 # 3237 PARENTHESIZED IDEOGRAPH CONGRATULATION
 # ->   0028 LEFT PARENTHESIS
 #    + 795D CJK UNIFIED IDEOGRAPH 795D
 #    + 0029 RIGHT PARENTHESIS
 "\x{3237}" => "(\x{795D})",

 # 3238 PARENTHESIZED IDEOGRAPH LABOR
 # ->   0028 LEFT PARENTHESIS
 #    + 52B4 CJK UNIFIED IDEOGRAPH 52B4
 #    + 0029 RIGHT PARENTHESIS
 "\x{3238}" => "(\x{52B4})",

 # 3239 PARENTHESIZED IDEOGRAPH REPRESENT
 # ->   0028 LEFT PARENTHESIS
 #    + 4EE3 CJK UNIFIED IDEOGRAPH 4EE3
 #    + 0029 RIGHT PARENTHESIS
 "\x{3239}" => "(\x{4EE3})",

 # 323A PARENTHESIZED IDEOGRAPH CALL
 # ->   0028 LEFT PARENTHESIS
 #    + 547C CJK UNIFIED IDEOGRAPH 547C
 #    + 0029 RIGHT PARENTHESIS
 "\x{323A}" => "(\x{547C})",

 # 323B PARENTHESIZED IDEOGRAPH STUDY
 # ->   0028 LEFT PARENTHESIS
 #    + 5B66 CJK UNIFIED IDEOGRAPH 5B66
 #    + 0029 RIGHT PARENTHESIS
 "\x{323B}" => "(\x{5B66})",

 # 323C PARENTHESIZED IDEOGRAPH SUPERVISE
 # ->   0028 LEFT PARENTHESIS
 #    + 76E3 CJK UNIFIED IDEOGRAPH 76E3
 #    + 0029 RIGHT PARENTHESIS
 "\x{323C}" => "(\x{76E3})",

 # 323D PARENTHESIZED IDEOGRAPH ENTERPRISE
 # ->   0028 LEFT PARENTHESIS
 #    + 4F01 CJK UNIFIED IDEOGRAPH 4F01
 #    + 0029 RIGHT PARENTHESIS
 "\x{323D}" => "(\x{4F01})",

 # 323E PARENTHESIZED IDEOGRAPH RESOURCE
 # ->   0028 LEFT PARENTHESIS
 #    + 8CC7 CJK UNIFIED IDEOGRAPH 8CC7
 #    + 0029 RIGHT PARENTHESIS
 "\x{323E}" => "(\x{8CC7})",

 # 323F PARENTHESIZED IDEOGRAPH ALLIANCE
 # ->   0028 LEFT PARENTHESIS
 #    + 5354 CJK UNIFIED IDEOGRAPH 5354
 #    + 0029 RIGHT PARENTHESIS
 "\x{323F}" => "(\x{5354})",

 # 3240 PARENTHESIZED IDEOGRAPH FESTIVAL
 # ->   0028 LEFT PARENTHESIS
 #    + 796D CJK UNIFIED IDEOGRAPH 796D
 #    + 0029 RIGHT PARENTHESIS
 "\x{3240}" => "(\x{796D})",

 # 3241 PARENTHESIZED IDEOGRAPH REST
 # ->   0028 LEFT PARENTHESIS
 #    + 4F11 CJK UNIFIED IDEOGRAPH 4F11
 #    + 0029 RIGHT PARENTHESIS
 "\x{3241}" => "(\x{4F11})",

 # 3242 PARENTHESIZED IDEOGRAPH SELF
 # ->   0028 LEFT PARENTHESIS
 #    + 81EA CJK UNIFIED IDEOGRAPH 81EA
 #    + 0029 RIGHT PARENTHESIS
 "\x{3242}" => "(\x{81EA})",

 # 3243 PARENTHESIZED IDEOGRAPH REACH
 # ->   0028 LEFT PARENTHESIS
 #    + 81F3 CJK UNIFIED IDEOGRAPH 81F3
 #    + 0029 RIGHT PARENTHESIS
 "\x{3243}" => "(\x{81F3})",

 # 3250 PARTNERSHIP SIGN
 # ->   0050 LATIN CAPITAL LETTER P
 #    + 0054 LATIN CAPITAL LETTER T
 #    + 0045 LATIN CAPITAL LETTER E
 "\x{3250}" => "PTE",

 # 3251 CIRCLED NUMBER TWENTY ONE
 # ->   0032 DIGIT TWO
 #    + 0031 DIGIT ONE
 "\x{3251}" => "21",

 # 3252 CIRCLED NUMBER TWENTY TWO
 # ->   0032 DIGIT TWO
 #    + 0032 DIGIT TWO
 "\x{3252}" => "22",

 # 3253 CIRCLED NUMBER TWENTY THREE
 # ->   0032 DIGIT TWO
 #    + 0033 DIGIT THREE
 "\x{3253}" => "23",

 # 3254 CIRCLED NUMBER TWENTY FOUR
 # ->   0032 DIGIT TWO
 #    + 0034 DIGIT FOUR
 "\x{3254}" => "24",

 # 3255 CIRCLED NUMBER TWENTY FIVE
 # ->   0032 DIGIT TWO
 #    + 0035 DIGIT FIVE
 "\x{3255}" => "25",

 # 3256 CIRCLED NUMBER TWENTY SIX
 # ->   0032 DIGIT TWO
 #    + 0036 DIGIT SIX
 "\x{3256}" => "26",

 # 3257 CIRCLED NUMBER TWENTY SEVEN
 # ->   0032 DIGIT TWO
 #    + 0037 DIGIT SEVEN
 "\x{3257}" => "27",

 # 3258 CIRCLED NUMBER TWENTY EIGHT
 # ->   0032 DIGIT TWO
 #    + 0038 DIGIT EIGHT
 "\x{3258}" => "28",

 # 3259 CIRCLED NUMBER TWENTY NINE
 # ->   0032 DIGIT TWO
 #    + 0039 DIGIT NINE
 "\x{3259}" => "29",

 # 325A CIRCLED NUMBER THIRTY
 # ->   0033 DIGIT THREE
 #    + 0030 DIGIT ZERO
 "\x{325A}" => "30",

 # 325B CIRCLED NUMBER THIRTY ONE
 # ->   0033 DIGIT THREE
 #    + 0031 DIGIT ONE
 "\x{325B}" => "31",

 # 325C CIRCLED NUMBER THIRTY TWO
 # ->   0033 DIGIT THREE
 #    + 0032 DIGIT TWO
 "\x{325C}" => "32",

 # 325D CIRCLED NUMBER THIRTY THREE
 # ->   0033 DIGIT THREE
 #    + 0033 DIGIT THREE
 "\x{325D}" => "33",

 # 325E CIRCLED NUMBER THIRTY FOUR
 # ->   0033 DIGIT THREE
 #    + 0034 DIGIT FOUR
 "\x{325E}" => "34",

 # 325F CIRCLED NUMBER THIRTY FIVE
 # ->   0033 DIGIT THREE
 #    + 0035 DIGIT FIVE
 "\x{325F}" => "35",

 # 3260 CIRCLED HANGUL KIYEOK
 # ->   1100 HANGUL CHOSEONG KIYEOK
 "\x{3260}" => "\x{1100}",

 # 3261 CIRCLED HANGUL NIEUN
 # ->   1102 HANGUL CHOSEONG NIEUN
 "\x{3261}" => "\x{1102}",

 # 3262 CIRCLED HANGUL TIKEUT
 # ->   1103 HANGUL CHOSEONG TIKEUT
 "\x{3262}" => "\x{1103}",

 # 3263 CIRCLED HANGUL RIEUL
 # ->   1105 HANGUL CHOSEONG RIEUL
 "\x{3263}" => "\x{1105}",

 # 3264 CIRCLED HANGUL MIEUM
 # ->   1106 HANGUL CHOSEONG MIEUM
 "\x{3264}" => "\x{1106}",

 # 3265 CIRCLED HANGUL PIEUP
 # ->   1107 HANGUL CHOSEONG PIEUP
 "\x{3265}" => "\x{1107}",

 # 3266 CIRCLED HANGUL SIOS
 # ->   1109 HANGUL CHOSEONG SIOS
 "\x{3266}" => "\x{1109}",

 # 3267 CIRCLED HANGUL IEUNG
 # ->   110B HANGUL CHOSEONG IEUNG
 "\x{3267}" => "\x{110B}",

 # 3268 CIRCLED HANGUL CIEUC
 # ->   110C HANGUL CHOSEONG CIEUC
 "\x{3268}" => "\x{110C}",

 # 3269 CIRCLED HANGUL CHIEUCH
 # ->   110E HANGUL CHOSEONG CHIEUCH
 "\x{3269}" => "\x{110E}",

 # 326A CIRCLED HANGUL KHIEUKH
 # ->   110F HANGUL CHOSEONG KHIEUKH
 "\x{326A}" => "\x{110F}",

 # 326B CIRCLED HANGUL THIEUTH
 # ->   1110 HANGUL CHOSEONG THIEUTH
 "\x{326B}" => "\x{1110}",

 # 326C CIRCLED HANGUL PHIEUPH
 # ->   1111 HANGUL CHOSEONG PHIEUPH
 "\x{326C}" => "\x{1111}",

 # 326D CIRCLED HANGUL HIEUH
 # ->   1112 HANGUL CHOSEONG HIEUH
 "\x{326D}" => "\x{1112}",

 # 326E CIRCLED HANGUL KIYEOK A
 # ->   1100 HANGUL CHOSEONG KIYEOK
 #    + 1161 HANGUL JUNGSEONG A
 "\x{326E}" => "\x{1100}\x{1161}",

 # 326F CIRCLED HANGUL NIEUN A
 # ->   1102 HANGUL CHOSEONG NIEUN
 #    + 1161 HANGUL JUNGSEONG A
 "\x{326F}" => "\x{1102}\x{1161}",

 # 3270 CIRCLED HANGUL TIKEUT A
 # ->   1103 HANGUL CHOSEONG TIKEUT
 #    + 1161 HANGUL JUNGSEONG A
 "\x{3270}" => "\x{1103}\x{1161}",

 # 3271 CIRCLED HANGUL RIEUL A
 # ->   1105 HANGUL CHOSEONG RIEUL
 #    + 1161 HANGUL JUNGSEONG A
 "\x{3271}" => "\x{1105}\x{1161}",

 # 3272 CIRCLED HANGUL MIEUM A
 # ->   1106 HANGUL CHOSEONG MIEUM
 #    + 1161 HANGUL JUNGSEONG A
 "\x{3272}" => "\x{1106}\x{1161}",

 # 3273 CIRCLED HANGUL PIEUP A
 # ->   1107 HANGUL CHOSEONG PIEUP
 #    + 1161 HANGUL JUNGSEONG A
 "\x{3273}" => "\x{1107}\x{1161}",

 # 3274 CIRCLED HANGUL SIOS A
 # ->   1109 HANGUL CHOSEONG SIOS
 #    + 1161 HANGUL JUNGSEONG A
 "\x{3274}" => "\x{1109}\x{1161}",

 # 3275 CIRCLED HANGUL IEUNG A
 # ->   110B HANGUL CHOSEONG IEUNG
 #    + 1161 HANGUL JUNGSEONG A
 "\x{3275}" => "\x{110B}\x{1161}",

 # 3276 CIRCLED HANGUL CIEUC A
 # ->   110C HANGUL CHOSEONG CIEUC
 #    + 1161 HANGUL JUNGSEONG A
 "\x{3276}" => "\x{110C}\x{1161}",

 # 3277 CIRCLED HANGUL CHIEUCH A
 # ->   110E HANGUL CHOSEONG CHIEUCH
 #    + 1161 HANGUL JUNGSEONG A
 "\x{3277}" => "\x{110E}\x{1161}",

 # 3278 CIRCLED HANGUL KHIEUKH A
 # ->   110F HANGUL CHOSEONG KHIEUKH
 #    + 1161 HANGUL JUNGSEONG A
 "\x{3278}" => "\x{110F}\x{1161}",

 # 3279 CIRCLED HANGUL THIEUTH A
 # ->   1110 HANGUL CHOSEONG THIEUTH
 #    + 1161 HANGUL JUNGSEONG A
 "\x{3279}" => "\x{1110}\x{1161}",

 # 327A CIRCLED HANGUL PHIEUPH A
 # ->   1111 HANGUL CHOSEONG PHIEUPH
 #    + 1161 HANGUL JUNGSEONG A
 "\x{327A}" => "\x{1111}\x{1161}",

 # 327B CIRCLED HANGUL HIEUH A
 # ->   1112 HANGUL CHOSEONG HIEUH
 #    + 1161 HANGUL JUNGSEONG A
 "\x{327B}" => "\x{1112}\x{1161}",

 # 327C CIRCLED KOREAN CHARACTER CHAMKO
 # ->   110E HANGUL CHOSEONG CHIEUCH
 #    + 1161 HANGUL JUNGSEONG A
 #    + 11B7 HANGUL JONGSEONG MIEUM
 #    + 1100 HANGUL CHOSEONG KIYEOK
 #    + 1169 HANGUL JUNGSEONG O
 "\x{327C}" => "\x{110E}\x{1161}\x{11B7}\x{1100}\x{1169}",

 # 327D CIRCLED KOREAN CHARACTER JUEUI
 # ->   110C HANGUL CHOSEONG CIEUC
 #    + 116E HANGUL JUNGSEONG U
 #    + 110B HANGUL CHOSEONG IEUNG
 #    + 1174 HANGUL JUNGSEONG YI
 "\x{327D}" => "\x{110C}\x{116E}\x{110B}\x{1174}",

 # 3280 CIRCLED IDEOGRAPH ONE
 # ->   4E00 CJK UNIFIED IDEOGRAPH 4E00
 "\x{3280}" => "\x{4E00}",

 # 3281 CIRCLED IDEOGRAPH TWO
 # ->   4E8C CJK UNIFIED IDEOGRAPH 4E8C
 "\x{3281}" => "\x{4E8C}",

 # 3282 CIRCLED IDEOGRAPH THREE
 # ->   4E09 CJK UNIFIED IDEOGRAPH 4E09
 "\x{3282}" => "\x{4E09}",

 # 3283 CIRCLED IDEOGRAPH FOUR
 # ->   56DB CJK UNIFIED IDEOGRAPH 56DB
 "\x{3283}" => "\x{56DB}",

 # 3284 CIRCLED IDEOGRAPH FIVE
 # ->   4E94 CJK UNIFIED IDEOGRAPH 4E94
 "\x{3284}" => "\x{4E94}",

 # 3285 CIRCLED IDEOGRAPH SIX
 # ->   516D CJK UNIFIED IDEOGRAPH 516D
 "\x{3285}" => "\x{516D}",

 # 3286 CIRCLED IDEOGRAPH SEVEN
 # ->   4E03 CJK UNIFIED IDEOGRAPH 4E03
 "\x{3286}" => "\x{4E03}",

 # 3287 CIRCLED IDEOGRAPH EIGHT
 # ->   516B CJK UNIFIED IDEOGRAPH 516B
 "\x{3287}" => "\x{516B}",

 # 3288 CIRCLED IDEOGRAPH NINE
 # ->   4E5D CJK UNIFIED IDEOGRAPH 4E5D
 "\x{3288}" => "\x{4E5D}",

 # 3289 CIRCLED IDEOGRAPH TEN
 # ->   5341 CJK UNIFIED IDEOGRAPH 5341
 "\x{3289}" => "\x{5341}",

 # 328A CIRCLED IDEOGRAPH MOON
 # ->   6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{328A}" => "\x{6708}",

 # 328B CIRCLED IDEOGRAPH FIRE
 # ->   706B CJK UNIFIED IDEOGRAPH 706B
 "\x{328B}" => "\x{706B}",

 # 328C CIRCLED IDEOGRAPH WATER
 # ->   6C34 CJK UNIFIED IDEOGRAPH 6C34
 "\x{328C}" => "\x{6C34}",

 # 328D CIRCLED IDEOGRAPH WOOD
 # ->   6728 CJK UNIFIED IDEOGRAPH 6728
 "\x{328D}" => "\x{6728}",

 # 328E CIRCLED IDEOGRAPH METAL
 # ->   91D1 CJK UNIFIED IDEOGRAPH 91D1
 "\x{328E}" => "\x{91D1}",

 # 328F CIRCLED IDEOGRAPH EARTH
 # ->   571F CJK UNIFIED IDEOGRAPH 571F
 "\x{328F}" => "\x{571F}",

 # 3290 CIRCLED IDEOGRAPH SUN
 # ->   65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{3290}" => "\x{65E5}",

 # 3291 CIRCLED IDEOGRAPH STOCK
 # ->   682A CJK UNIFIED IDEOGRAPH 682A
 "\x{3291}" => "\x{682A}",

 # 3292 CIRCLED IDEOGRAPH HAVE
 # ->   6709 CJK UNIFIED IDEOGRAPH 6709
 "\x{3292}" => "\x{6709}",

 # 3293 CIRCLED IDEOGRAPH SOCIETY
 # ->   793E CJK UNIFIED IDEOGRAPH 793E
 "\x{3293}" => "\x{793E}",

 # 3294 CIRCLED IDEOGRAPH NAME
 # ->   540D CJK UNIFIED IDEOGRAPH 540D
 "\x{3294}" => "\x{540D}",

 # 3295 CIRCLED IDEOGRAPH SPECIAL
 # ->   7279 CJK UNIFIED IDEOGRAPH 7279
 "\x{3295}" => "\x{7279}",

 # 3296 CIRCLED IDEOGRAPH FINANCIAL
 # ->   8CA1 CJK UNIFIED IDEOGRAPH 8CA1
 "\x{3296}" => "\x{8CA1}",

 # 3297 CIRCLED IDEOGRAPH CONGRATULATION
 # ->   795D CJK UNIFIED IDEOGRAPH 795D
 "\x{3297}" => "\x{795D}",

 # 3298 CIRCLED IDEOGRAPH LABOR
 # ->   52B4 CJK UNIFIED IDEOGRAPH 52B4
 "\x{3298}" => "\x{52B4}",

 # 3299 CIRCLED IDEOGRAPH SECRET
 # ->   79D8 CJK UNIFIED IDEOGRAPH 79D8
 "\x{3299}" => "\x{79D8}",

 # 329A CIRCLED IDEOGRAPH MALE
 # ->   7537 CJK UNIFIED IDEOGRAPH 7537
 "\x{329A}" => "\x{7537}",

 # 329B CIRCLED IDEOGRAPH FEMALE
 # ->   5973 CJK UNIFIED IDEOGRAPH 5973
 "\x{329B}" => "\x{5973}",

 # 329C CIRCLED IDEOGRAPH SUITABLE
 # ->   9069 CJK UNIFIED IDEOGRAPH 9069
 "\x{329C}" => "\x{9069}",

 # 329D CIRCLED IDEOGRAPH EXCELLENT
 # ->   512A CJK UNIFIED IDEOGRAPH 512A
 "\x{329D}" => "\x{512A}",

 # 329E CIRCLED IDEOGRAPH PRINT
 # ->   5370 CJK UNIFIED IDEOGRAPH 5370
 "\x{329E}" => "\x{5370}",

 # 329F CIRCLED IDEOGRAPH ATTENTION
 # ->   6CE8 CJK UNIFIED IDEOGRAPH 6CE8
 "\x{329F}" => "\x{6CE8}",

 # 32A0 CIRCLED IDEOGRAPH ITEM
 # ->   9805 CJK UNIFIED IDEOGRAPH 9805
 "\x{32A0}" => "\x{9805}",

 # 32A1 CIRCLED IDEOGRAPH REST
 # ->   4F11 CJK UNIFIED IDEOGRAPH 4F11
 "\x{32A1}" => "\x{4F11}",

 # 32A2 CIRCLED IDEOGRAPH COPY
 # ->   5199 CJK UNIFIED IDEOGRAPH 5199
 "\x{32A2}" => "\x{5199}",

 # 32A3 CIRCLED IDEOGRAPH CORRECT
 # ->   6B63 CJK UNIFIED IDEOGRAPH 6B63
 "\x{32A3}" => "\x{6B63}",

 # 32A4 CIRCLED IDEOGRAPH HIGH
 # ->   4E0A CJK UNIFIED IDEOGRAPH 4E0A
 "\x{32A4}" => "\x{4E0A}",

 # 32A5 CIRCLED IDEOGRAPH CENTRE
 # ->   4E2D CJK UNIFIED IDEOGRAPH 4E2D
 "\x{32A5}" => "\x{4E2D}",

 # 32A6 CIRCLED IDEOGRAPH LOW
 # ->   4E0B CJK UNIFIED IDEOGRAPH 4E0B
 "\x{32A6}" => "\x{4E0B}",

 # 32A7 CIRCLED IDEOGRAPH LEFT
 # ->   5DE6 CJK UNIFIED IDEOGRAPH 5DE6
 "\x{32A7}" => "\x{5DE6}",

 # 32A8 CIRCLED IDEOGRAPH RIGHT
 # ->   53F3 CJK UNIFIED IDEOGRAPH 53F3
 "\x{32A8}" => "\x{53F3}",

 # 32A9 CIRCLED IDEOGRAPH MEDICINE
 # ->   533B CJK UNIFIED IDEOGRAPH 533B
 "\x{32A9}" => "\x{533B}",

 # 32AA CIRCLED IDEOGRAPH RELIGION
 # ->   5B97 CJK UNIFIED IDEOGRAPH 5B97
 "\x{32AA}" => "\x{5B97}",

 # 32AB CIRCLED IDEOGRAPH STUDY
 # ->   5B66 CJK UNIFIED IDEOGRAPH 5B66
 "\x{32AB}" => "\x{5B66}",

 # 32AC CIRCLED IDEOGRAPH SUPERVISE
 # ->   76E3 CJK UNIFIED IDEOGRAPH 76E3
 "\x{32AC}" => "\x{76E3}",

 # 32AD CIRCLED IDEOGRAPH ENTERPRISE
 # ->   4F01 CJK UNIFIED IDEOGRAPH 4F01
 "\x{32AD}" => "\x{4F01}",

 # 32AE CIRCLED IDEOGRAPH RESOURCE
 # ->   8CC7 CJK UNIFIED IDEOGRAPH 8CC7
 "\x{32AE}" => "\x{8CC7}",

 # 32AF CIRCLED IDEOGRAPH ALLIANCE
 # ->   5354 CJK UNIFIED IDEOGRAPH 5354
 "\x{32AF}" => "\x{5354}",

 # 32B0 CIRCLED IDEOGRAPH NIGHT
 # ->   591C CJK UNIFIED IDEOGRAPH 591C
 "\x{32B0}" => "\x{591C}",

 # 32B1 CIRCLED NUMBER THIRTY SIX
 # ->   0033 DIGIT THREE
 #    + 0036 DIGIT SIX
 "\x{32B1}" => "36",

 # 32B2 CIRCLED NUMBER THIRTY SEVEN
 # ->   0033 DIGIT THREE
 #    + 0037 DIGIT SEVEN
 "\x{32B2}" => "37",

 # 32B3 CIRCLED NUMBER THIRTY EIGHT
 # ->   0033 DIGIT THREE
 #    + 0038 DIGIT EIGHT
 "\x{32B3}" => "38",

 # 32B4 CIRCLED NUMBER THIRTY NINE
 # ->   0033 DIGIT THREE
 #    + 0039 DIGIT NINE
 "\x{32B4}" => "39",

 # 32B5 CIRCLED NUMBER FORTY
 # ->   0034 DIGIT FOUR
 #    + 0030 DIGIT ZERO
 "\x{32B5}" => "40",

 # 32B6 CIRCLED NUMBER FORTY ONE
 # ->   0034 DIGIT FOUR
 #    + 0031 DIGIT ONE
 "\x{32B6}" => "41",

 # 32B7 CIRCLED NUMBER FORTY TWO
 # ->   0034 DIGIT FOUR
 #    + 0032 DIGIT TWO
 "\x{32B7}" => "42",

 # 32B8 CIRCLED NUMBER FORTY THREE
 # ->   0034 DIGIT FOUR
 #    + 0033 DIGIT THREE
 "\x{32B8}" => "43",

 # 32B9 CIRCLED NUMBER FORTY FOUR
 # ->   0034 DIGIT FOUR
 #    + 0034 DIGIT FOUR
 "\x{32B9}" => "44",

 # 32BA CIRCLED NUMBER FORTY FIVE
 # ->   0034 DIGIT FOUR
 #    + 0035 DIGIT FIVE
 "\x{32BA}" => "45",

 # 32BB CIRCLED NUMBER FORTY SIX
 # ->   0034 DIGIT FOUR
 #    + 0036 DIGIT SIX
 "\x{32BB}" => "46",

 # 32BC CIRCLED NUMBER FORTY SEVEN
 # ->   0034 DIGIT FOUR
 #    + 0037 DIGIT SEVEN
 "\x{32BC}" => "47",

 # 32BD CIRCLED NUMBER FORTY EIGHT
 # ->   0034 DIGIT FOUR
 #    + 0038 DIGIT EIGHT
 "\x{32BD}" => "48",

 # 32BE CIRCLED NUMBER FORTY NINE
 # ->   0034 DIGIT FOUR
 #    + 0039 DIGIT NINE
 "\x{32BE}" => "49",

 # 32BF CIRCLED NUMBER FIFTY
 # ->   0035 DIGIT FIVE
 #    + 0030 DIGIT ZERO
 "\x{32BF}" => "50",

 # 32C0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JANUARY
 # ->   0031 DIGIT ONE
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32C0}" => "1\x{6708}",

 # 32C1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR FEBRUARY
 # ->   0032 DIGIT TWO
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32C1}" => "2\x{6708}",

 # 32C2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR MARCH
 # ->   0033 DIGIT THREE
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32C2}" => "3\x{6708}",

 # 32C3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR APRIL
 # ->   0034 DIGIT FOUR
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32C3}" => "4\x{6708}",

 # 32C4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR MAY
 # ->   0035 DIGIT FIVE
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32C4}" => "5\x{6708}",

 # 32C5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JUNE
 # ->   0036 DIGIT SIX
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32C5}" => "6\x{6708}",

 # 32C6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR JULY
 # ->   0037 DIGIT SEVEN
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32C6}" => "7\x{6708}",

 # 32C7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR AUGUST
 # ->   0038 DIGIT EIGHT
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32C7}" => "8\x{6708}",

 # 32C8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR SEPTEMBER
 # ->   0039 DIGIT NINE
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32C8}" => "9\x{6708}",

 # 32C9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR OCTOBER
 # ->   0031 DIGIT ONE
 #    + 0030 DIGIT ZERO
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32C9}" => "10\x{6708}",

 # 32CA IDEOGRAPHIC TELEGRAPH SYMBOL FOR NOVEMBER
 # ->   0031 DIGIT ONE
 #    + 0031 DIGIT ONE
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32CA}" => "11\x{6708}",

 # 32CB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DECEMBER
 # ->   0031 DIGIT ONE
 #    + 0032 DIGIT TWO
 #    + 6708 CJK UNIFIED IDEOGRAPH 6708
 "\x{32CB}" => "12\x{6708}",

 # 32CC SQUARE HG
 # ->   0048 LATIN CAPITAL LETTER H
 #    + 0067 LATIN SMALL LETTER G
 "\x{32CC}" => "Hg",

 # 32CD SQUARE ERG
 # ->   0065 LATIN SMALL LETTER E
 #    + 0072 LATIN SMALL LETTER R
 #    + 0067 LATIN SMALL LETTER G
 "\x{32CD}" => "erg",

 # 32CE SQUARE EV
 # ->   0065 LATIN SMALL LETTER E
 #    + 0056 LATIN CAPITAL LETTER V
 "\x{32CE}" => "eV",

 # 32CF LIMITED LIABILITY SIGN
 # ->   004C LATIN CAPITAL LETTER L
 #    + 0054 LATIN CAPITAL LETTER T
 #    + 0044 LATIN CAPITAL LETTER D
 "\x{32CF}" => "LTD",

 # 32D0 CIRCLED KATAKANA A
 # ->   30A2 KATAKANA LETTER A
 "\x{32D0}" => "\x{30A2}",

 # 32D1 CIRCLED KATAKANA I
 # ->   30A4 KATAKANA LETTER I
 "\x{32D1}" => "\x{30A4}",

 # 32D2 CIRCLED KATAKANA U
 # ->   30A6 KATAKANA LETTER U
 "\x{32D2}" => "\x{30A6}",

 # 32D3 CIRCLED KATAKANA E
 # ->   30A8 KATAKANA LETTER E
 "\x{32D3}" => "\x{30A8}",

 # 32D4 CIRCLED KATAKANA O
 # ->   30AA KATAKANA LETTER O
 "\x{32D4}" => "\x{30AA}",

 # 32D5 CIRCLED KATAKANA KA
 # ->   30AB KATAKANA LETTER KA
 "\x{32D5}" => "\x{30AB}",

 # 32D6 CIRCLED KATAKANA KI
 # ->   30AD KATAKANA LETTER KI
 "\x{32D6}" => "\x{30AD}",

 # 32D7 CIRCLED KATAKANA KU
 # ->   30AF KATAKANA LETTER KU
 "\x{32D7}" => "\x{30AF}",

 # 32D8 CIRCLED KATAKANA KE
 # ->   30B1 KATAKANA LETTER KE
 "\x{32D8}" => "\x{30B1}",

 # 32D9 CIRCLED KATAKANA KO
 # ->   30B3 KATAKANA LETTER KO
 "\x{32D9}" => "\x{30B3}",

 # 32DA CIRCLED KATAKANA SA
 # ->   30B5 KATAKANA LETTER SA
 "\x{32DA}" => "\x{30B5}",

 # 32DB CIRCLED KATAKANA SI
 # ->   30B7 KATAKANA LETTER SI
 "\x{32DB}" => "\x{30B7}",

 # 32DC CIRCLED KATAKANA SU
 # ->   30B9 KATAKANA LETTER SU
 "\x{32DC}" => "\x{30B9}",

 # 32DD CIRCLED KATAKANA SE
 # ->   30BB KATAKANA LETTER SE
 "\x{32DD}" => "\x{30BB}",

 # 32DE CIRCLED KATAKANA SO
 # ->   30BD KATAKANA LETTER SO
 "\x{32DE}" => "\x{30BD}",

 # 32DF CIRCLED KATAKANA TA
 # ->   30BF KATAKANA LETTER TA
 "\x{32DF}" => "\x{30BF}",

 # 32E0 CIRCLED KATAKANA TI
 # ->   30C1 KATAKANA LETTER TI
 "\x{32E0}" => "\x{30C1}",

 # 32E1 CIRCLED KATAKANA TU
 # ->   30C4 KATAKANA LETTER TU
 "\x{32E1}" => "\x{30C4}",

 # 32E2 CIRCLED KATAKANA TE
 # ->   30C6 KATAKANA LETTER TE
 "\x{32E2}" => "\x{30C6}",

 # 32E3 CIRCLED KATAKANA TO
 # ->   30C8 KATAKANA LETTER TO
 "\x{32E3}" => "\x{30C8}",

 # 32E4 CIRCLED KATAKANA NA
 # ->   30CA KATAKANA LETTER NA
 "\x{32E4}" => "\x{30CA}",

 # 32E5 CIRCLED KATAKANA NI
 # ->   30CB KATAKANA LETTER NI
 "\x{32E5}" => "\x{30CB}",

 # 32E6 CIRCLED KATAKANA NU
 # ->   30CC KATAKANA LETTER NU
 "\x{32E6}" => "\x{30CC}",

 # 32E7 CIRCLED KATAKANA NE
 # ->   30CD KATAKANA LETTER NE
 "\x{32E7}" => "\x{30CD}",

 # 32E8 CIRCLED KATAKANA NO
 # ->   30CE KATAKANA LETTER NO
 "\x{32E8}" => "\x{30CE}",

 # 32E9 CIRCLED KATAKANA HA
 # ->   30CF KATAKANA LETTER HA
 "\x{32E9}" => "\x{30CF}",

 # 32EA CIRCLED KATAKANA HI
 # ->   30D2 KATAKANA LETTER HI
 "\x{32EA}" => "\x{30D2}",

 # 32EB CIRCLED KATAKANA HU
 # ->   30D5 KATAKANA LETTER HU
 "\x{32EB}" => "\x{30D5}",

 # 32EC CIRCLED KATAKANA HE
 # ->   30D8 KATAKANA LETTER HE
 "\x{32EC}" => "\x{30D8}",

 # 32ED CIRCLED KATAKANA HO
 # ->   30DB KATAKANA LETTER HO
 "\x{32ED}" => "\x{30DB}",

 # 32EE CIRCLED KATAKANA MA
 # ->   30DE KATAKANA LETTER MA
 "\x{32EE}" => "\x{30DE}",

 # 32EF CIRCLED KATAKANA MI
 # ->   30DF KATAKANA LETTER MI
 "\x{32EF}" => "\x{30DF}",

 # 32F0 CIRCLED KATAKANA MU
 # ->   30E0 KATAKANA LETTER MU
 "\x{32F0}" => "\x{30E0}",

 # 32F1 CIRCLED KATAKANA ME
 # ->   30E1 KATAKANA LETTER ME
 "\x{32F1}" => "\x{30E1}",

 # 32F2 CIRCLED KATAKANA MO
 # ->   30E2 KATAKANA LETTER MO
 "\x{32F2}" => "\x{30E2}",

 # 32F3 CIRCLED KATAKANA YA
 # ->   30E4 KATAKANA LETTER YA
 "\x{32F3}" => "\x{30E4}",

 # 32F4 CIRCLED KATAKANA YU
 # ->   30E6 KATAKANA LETTER YU
 "\x{32F4}" => "\x{30E6}",

 # 32F5 CIRCLED KATAKANA YO
 # ->   30E8 KATAKANA LETTER YO
 "\x{32F5}" => "\x{30E8}",

 # 32F6 CIRCLED KATAKANA RA
 # ->   30E9 KATAKANA LETTER RA
 "\x{32F6}" => "\x{30E9}",

 # 32F7 CIRCLED KATAKANA RI
 # ->   30EA KATAKANA LETTER RI
 "\x{32F7}" => "\x{30EA}",

 # 32F8 CIRCLED KATAKANA RU
 # ->   30EB KATAKANA LETTER RU
 "\x{32F8}" => "\x{30EB}",

 # 32F9 CIRCLED KATAKANA RE
 # ->   30EC KATAKANA LETTER RE
 "\x{32F9}" => "\x{30EC}",

 # 32FA CIRCLED KATAKANA RO
 # ->   30ED KATAKANA LETTER RO
 "\x{32FA}" => "\x{30ED}",

 # 32FB CIRCLED KATAKANA WA
 # ->   30EF KATAKANA LETTER WA
 "\x{32FB}" => "\x{30EF}",

 # 32FC CIRCLED KATAKANA WI
 # ->   30F0 KATAKANA LETTER WI
 "\x{32FC}" => "\x{30F0}",

 # 32FD CIRCLED KATAKANA WE
 # ->   30F1 KATAKANA LETTER WE
 "\x{32FD}" => "\x{30F1}",

 # 32FE CIRCLED KATAKANA WO
 # ->   30F2 KATAKANA LETTER WO
 "\x{32FE}" => "\x{30F2}",

 # 3300 SQUARE APAATO
 # ->   30A2 KATAKANA LETTER A
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30C8 KATAKANA LETTER TO
 #    + 30CF KATAKANA LETTER HA
 "\x{3300}" => "\x{30A2}\x{30FC}\x{30C8}\x{30CF}",

 # 3301 SQUARE ARUHUA
 # ->   30A2 KATAKANA LETTER A
 #    + 30EB KATAKANA LETTER RU
 #    + 30D5 KATAKANA LETTER HU
 #    + 30A1 KATAKANA LETTER SMALL A
 "\x{3301}" => "\x{30A2}\x{30EB}\x{30D5}\x{30A1}",

 # 3302 SQUARE ANPEA
 # ->   30A2 KATAKANA LETTER A
 #    + 30F3 KATAKANA LETTER N
 #    + 30A2 KATAKANA LETTER A
 #    + 30D8 KATAKANA LETTER HE
 "\x{3302}" => "\x{30A2}\x{30F3}\x{30A2}\x{30D8}",

 # 3303 SQUARE AARU
 # ->   30A2 KATAKANA LETTER A
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30EB KATAKANA LETTER RU
 "\x{3303}" => "\x{30A2}\x{30FC}\x{30EB}",

 # 3304 SQUARE ININGU
 # ->   30A4 KATAKANA LETTER I
 #    + 30CB KATAKANA LETTER NI
 #    + 30F3 KATAKANA LETTER N
 #    + 30AF KATAKANA LETTER KU
 "\x{3304}" => "\x{30A4}\x{30CB}\x{30F3}\x{30AF}",

 # 3305 SQUARE INTI
 # ->   30A4 KATAKANA LETTER I
 #    + 30F3 KATAKANA LETTER N
 #    + 30C1 KATAKANA LETTER TI
 "\x{3305}" => "\x{30A4}\x{30F3}\x{30C1}",

 # 3306 SQUARE UON
 # ->   30A6 KATAKANA LETTER U
 #    + 30A9 KATAKANA LETTER SMALL O
 #    + 30F3 KATAKANA LETTER N
 "\x{3306}" => "\x{30A6}\x{30A9}\x{30F3}",

 # 3307 SQUARE ESUKUUDO
 # ->   30A8 KATAKANA LETTER E
 #    + 30B9 KATAKANA LETTER SU
 #    + 30AF KATAKANA LETTER KU
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30C8 KATAKANA LETTER TO
 "\x{3307}" => "\x{30A8}\x{30B9}\x{30AF}\x{30FC}\x{30C8}",

 # 3308 SQUARE EEKAA
 # ->   30A8 KATAKANA LETTER E
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30AB KATAKANA LETTER KA
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 "\x{3308}" => "\x{30A8}\x{30FC}\x{30AB}\x{30FC}",

 # 3309 SQUARE ONSU
 # ->   30AA KATAKANA LETTER O
 #    + 30F3 KATAKANA LETTER N
 #    + 30B9 KATAKANA LETTER SU
 "\x{3309}" => "\x{30AA}\x{30F3}\x{30B9}",

 # 330A SQUARE OOMU
 # ->   30AA KATAKANA LETTER O
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30E0 KATAKANA LETTER MU
 "\x{330A}" => "\x{30AA}\x{30FC}\x{30E0}",

 # 330B SQUARE KAIRI
 # ->   30AB KATAKANA LETTER KA
 #    + 30A4 KATAKANA LETTER I
 #    + 30EA KATAKANA LETTER RI
 "\x{330B}" => "\x{30AB}\x{30A4}\x{30EA}",

 # 330C SQUARE KARATTO
 # ->   30AB KATAKANA LETTER KA
 #    + 30E9 KATAKANA LETTER RA
 #    + 30C3 KATAKANA LETTER SMALL TU
 #    + 30C8 KATAKANA LETTER TO
 "\x{330C}" => "\x{30AB}\x{30E9}\x{30C3}\x{30C8}",

 # 330D SQUARE KARORII
 # ->   30AB KATAKANA LETTER KA
 #    + 30ED KATAKANA LETTER RO
 #    + 30EA KATAKANA LETTER RI
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 "\x{330D}" => "\x{30AB}\x{30ED}\x{30EA}\x{30FC}",

 # 330E SQUARE GARON
 # ->   30ED KATAKANA LETTER RO
 #    + 30F3 KATAKANA LETTER N
 #    + 30AB KATAKANA LETTER KA
 "\x{330E}" => "\x{30ED}\x{30F3}\x{30AB}",

 # 330F SQUARE GANMA
 # ->   30F3 KATAKANA LETTER N
 #    + 30DE KATAKANA LETTER MA
 #    + 30AB KATAKANA LETTER KA
 "\x{330F}" => "\x{30F3}\x{30DE}\x{30AB}",

 # 3310 SQUARE GIGA
 # ->   30AD KATAKANA LETTER KI
 #    + 30AB KATAKANA LETTER KA
 "\x{3310}" => "\x{30AD}\x{30AB}",

 # 3311 SQUARE GINII
 # ->   30CB KATAKANA LETTER NI
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30AD KATAKANA LETTER KI
 "\x{3311}" => "\x{30CB}\x{30FC}\x{30AD}",

 # 3312 SQUARE KYURII
 # ->   30AD KATAKANA LETTER KI
 #    + 30E5 KATAKANA LETTER SMALL YU
 #    + 30EA KATAKANA LETTER RI
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 "\x{3312}" => "\x{30AD}\x{30E5}\x{30EA}\x{30FC}",

 # 3313 SQUARE GIRUDAA
 # ->   30EB KATAKANA LETTER RU
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30AD KATAKANA LETTER KI
 #    + 30BF KATAKANA LETTER TA
 "\x{3313}" => "\x{30EB}\x{30FC}\x{30AD}\x{30BF}",

 # 3314 SQUARE KIRO
 # ->   30AD KATAKANA LETTER KI
 #    + 30ED KATAKANA LETTER RO
 "\x{3314}" => "\x{30AD}\x{30ED}",

 # 3315 SQUARE KIROGURAMU
 # ->   30AD KATAKANA LETTER KI
 #    + 30ED KATAKANA LETTER RO
 #    + 30E9 KATAKANA LETTER RA
 #    + 30E0 KATAKANA LETTER MU
 #    + 30AF KATAKANA LETTER KU
 "\x{3315}" => "\x{30AD}\x{30ED}\x{30E9}\x{30E0}\x{30AF}",

 # 3316 SQUARE KIROMEETORU
 # ->   30AD KATAKANA LETTER KI
 #    + 30ED KATAKANA LETTER RO
 #    + 30E1 KATAKANA LETTER ME
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30C8 KATAKANA LETTER TO
 #    + 30EB KATAKANA LETTER RU
 "\x{3316}" => "\x{30AD}\x{30ED}\x{30E1}\x{30FC}\x{30C8}\x{30EB}",

 # 3317 SQUARE KIROWATTO
 # ->   30AD KATAKANA LETTER KI
 #    + 30ED KATAKANA LETTER RO
 #    + 30EF KATAKANA LETTER WA
 #    + 30C3 KATAKANA LETTER SMALL TU
 #    + 30C8 KATAKANA LETTER TO
 "\x{3317}" => "\x{30AD}\x{30ED}\x{30EF}\x{30C3}\x{30C8}",

 # 3318 SQUARE GURAMU
 # ->   30E9 KATAKANA LETTER RA
 #    + 30E0 KATAKANA LETTER MU
 #    + 30AF KATAKANA LETTER KU
 "\x{3318}" => "\x{30E9}\x{30E0}\x{30AF}",

 # 3319 SQUARE GURAMUTON
 # ->   30E9 KATAKANA LETTER RA
 #    + 30E0 KATAKANA LETTER MU
 #    + 30C8 KATAKANA LETTER TO
 #    + 30F3 KATAKANA LETTER N
 #    + 30AF KATAKANA LETTER KU
 "\x{3319}" => "\x{30E9}\x{30E0}\x{30C8}\x{30F3}\x{30AF}",

 # 331A SQUARE KURUZEIRO
 # ->   30AF KATAKANA LETTER KU
 #    + 30EB KATAKANA LETTER RU
 #    + 30A4 KATAKANA LETTER I
 #    + 30ED KATAKANA LETTER RO
 #    + 30BB KATAKANA LETTER SE
 "\x{331A}" => "\x{30AF}\x{30EB}\x{30A4}\x{30ED}\x{30BB}",

 # 331B SQUARE KUROONE
 # ->   30AF KATAKANA LETTER KU
 #    + 30ED KATAKANA LETTER RO
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30CD KATAKANA LETTER NE
 "\x{331B}" => "\x{30AF}\x{30ED}\x{30FC}\x{30CD}",

 # 331C SQUARE KEESU
 # ->   30B1 KATAKANA LETTER KE
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30B9 KATAKANA LETTER SU
 "\x{331C}" => "\x{30B1}\x{30FC}\x{30B9}",

 # 331D SQUARE KORUNA
 # ->   30B3 KATAKANA LETTER KO
 #    + 30EB KATAKANA LETTER RU
 #    + 30CA KATAKANA LETTER NA
 "\x{331D}" => "\x{30B3}\x{30EB}\x{30CA}",

 # 331E SQUARE KOOPO
 # ->   30B3 KATAKANA LETTER KO
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30DB KATAKANA LETTER HO
 "\x{331E}" => "\x{30B3}\x{30FC}\x{30DB}",

 # 331F SQUARE SAIKURU
 # ->   30B5 KATAKANA LETTER SA
 #    + 30A4 KATAKANA LETTER I
 #    + 30AF KATAKANA LETTER KU
 #    + 30EB KATAKANA LETTER RU
 "\x{331F}" => "\x{30B5}\x{30A4}\x{30AF}\x{30EB}",

 # 3320 SQUARE SANTIIMU
 # ->   30B5 KATAKANA LETTER SA
 #    + 30F3 KATAKANA LETTER N
 #    + 30C1 KATAKANA LETTER TI
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30E0 KATAKANA LETTER MU
 "\x{3320}" => "\x{30B5}\x{30F3}\x{30C1}\x{30FC}\x{30E0}",

 # 3321 SQUARE SIRINGU
 # ->   30B7 KATAKANA LETTER SI
 #    + 30EA KATAKANA LETTER RI
 #    + 30F3 KATAKANA LETTER N
 #    + 30AF KATAKANA LETTER KU
 "\x{3321}" => "\x{30B7}\x{30EA}\x{30F3}\x{30AF}",

 # 3322 SQUARE SENTI
 # ->   30BB KATAKANA LETTER SE
 #    + 30F3 KATAKANA LETTER N
 #    + 30C1 KATAKANA LETTER TI
 "\x{3322}" => "\x{30BB}\x{30F3}\x{30C1}",

 # 3323 SQUARE SENTO
 # ->   30BB KATAKANA LETTER SE
 #    + 30F3 KATAKANA LETTER N
 #    + 30C8 KATAKANA LETTER TO
 "\x{3323}" => "\x{30BB}\x{30F3}\x{30C8}",

 # 3324 SQUARE DAASU
 # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30B9 KATAKANA LETTER SU
 #    + 30BF KATAKANA LETTER TA
 "\x{3324}" => "\x{30FC}\x{30B9}\x{30BF}",

 # 3325 SQUARE DESI
 # ->   30B7 KATAKANA LETTER SI
 #    + 30C6 KATAKANA LETTER TE
 "\x{3325}" => "\x{30B7}\x{30C6}",

 # 3326 SQUARE DORU
 # ->   30EB KATAKANA LETTER RU
 #    + 30C8 KATAKANA LETTER TO
 "\x{3326}" => "\x{30EB}\x{30C8}",

 # 3327 SQUARE TON
 # ->   30C8 KATAKANA LETTER TO
 #    + 30F3 KATAKANA LETTER N
 "\x{3327}" => "\x{30C8}\x{30F3}",

 # 3328 SQUARE NANO
 # ->   30CA KATAKANA LETTER NA
 #    + 30CE KATAKANA LETTER NO
 "\x{3328}" => "\x{30CA}\x{30CE}",

 # 3329 SQUARE NOTTO
 # ->   30CE KATAKANA LETTER NO
 #    + 30C3 KATAKANA LETTER SMALL TU
 #    + 30C8 KATAKANA LETTER TO
 "\x{3329}" => "\x{30CE}\x{30C3}\x{30C8}",

 # 332A SQUARE HAITU
 # ->   30CF KATAKANA LETTER HA
 #    + 30A4 KATAKANA LETTER I
 #    + 30C4 KATAKANA LETTER TU
 "\x{332A}" => "\x{30CF}\x{30A4}\x{30C4}",

 # 332B SQUARE PAASENTO
 # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30BB KATAKANA LETTER SE
 #    + 30F3 KATAKANA LETTER N
 #    + 30C8 KATAKANA LETTER TO
 #    + 30CF KATAKANA LETTER HA
 "\x{332B}" => "\x{30FC}\x{30BB}\x{30F3}\x{30C8}\x{30CF}",

 # 332C SQUARE PAATU
 # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30C4 KATAKANA LETTER TU
 #    + 30CF KATAKANA LETTER HA
 "\x{332C}" => "\x{30FC}\x{30C4}\x{30CF}",

 # 332D SQUARE BAARERU
 # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30EC KATAKANA LETTER RE
 #    + 30EB KATAKANA LETTER RU
 #    + 30CF KATAKANA LETTER HA
 "\x{332D}" => "\x{30FC}\x{30EC}\x{30EB}\x{30CF}",

 # 332E SQUARE PIASUTORU
 # ->   30A2 KATAKANA LETTER A
 #    + 30B9 KATAKANA LETTER SU
 #    + 30C8 KATAKANA LETTER TO
 #    + 30EB KATAKANA LETTER RU
 #    + 30D2 KATAKANA LETTER HI
 "\x{332E}" => "\x{30A2}\x{30B9}\x{30C8}\x{30EB}\x{30D2}",

 # 332F SQUARE PIKURU
 # ->   30AF KATAKANA LETTER KU
 #    + 30EB KATAKANA LETTER RU
 #    + 30D2 KATAKANA LETTER HI
 "\x{332F}" => "\x{30AF}\x{30EB}\x{30D2}",

 # 3330 SQUARE PIKO
 # ->   30B3 KATAKANA LETTER KO
 #    + 30D2 KATAKANA LETTER HI
 "\x{3330}" => "\x{30B3}\x{30D2}",

 # 3331 SQUARE BIRU
 # ->   30EB KATAKANA LETTER RU
 #    + 30D2 KATAKANA LETTER HI
 "\x{3331}" => "\x{30EB}\x{30D2}",

 # 3332 SQUARE HUARADDO
 # ->   30D5 KATAKANA LETTER HU
 #    + 30A1 KATAKANA LETTER SMALL A
 #    + 30E9 KATAKANA LETTER RA
 #    + 30C3 KATAKANA LETTER SMALL TU
 #    + 30C8 KATAKANA LETTER TO
 "\x{3332}" => "\x{30D5}\x{30A1}\x{30E9}\x{30C3}\x{30C8}",

 # 3333 SQUARE HUIITO
 # ->   30D5 KATAKANA LETTER HU
 #    + 30A3 KATAKANA LETTER SMALL I
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30C8 KATAKANA LETTER TO
 "\x{3333}" => "\x{30D5}\x{30A3}\x{30FC}\x{30C8}",

 # 3334 SQUARE BUSSYERU
 # ->   30C3 KATAKANA LETTER SMALL TU
 #    + 30B7 KATAKANA LETTER SI
 #    + 30A7 KATAKANA LETTER SMALL E
 #    + 30EB KATAKANA LETTER RU
 #    + 30D5 KATAKANA LETTER HU
 "\x{3334}" => "\x{30C3}\x{30B7}\x{30A7}\x{30EB}\x{30D5}",

 # 3335 SQUARE HURAN
 # ->   30D5 KATAKANA LETTER HU
 #    + 30E9 KATAKANA LETTER RA
 #    + 30F3 KATAKANA LETTER N
 "\x{3335}" => "\x{30D5}\x{30E9}\x{30F3}",

 # 3336 SQUARE HEKUTAARU
 # ->   30D8 KATAKANA LETTER HE
 #    + 30AF KATAKANA LETTER KU
 #    + 30BF KATAKANA LETTER TA
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30EB KATAKANA LETTER RU
 "\x{3336}" => "\x{30D8}\x{30AF}\x{30BF}\x{30FC}\x{30EB}",

 # 3337 SQUARE PESO
 # ->   30BD KATAKANA LETTER SO
 #    + 30D8 KATAKANA LETTER HE
 "\x{3337}" => "\x{30BD}\x{30D8}",

 # 3338 SQUARE PENIHI
 # ->   30CB KATAKANA LETTER NI
 #    + 30D2 KATAKANA LETTER HI
 #    + 30D8 KATAKANA LETTER HE
 "\x{3338}" => "\x{30CB}\x{30D2}\x{30D8}",

 # 3339 SQUARE HERUTU
 # ->   30D8 KATAKANA LETTER HE
 #    + 30EB KATAKANA LETTER RU
 #    + 30C4 KATAKANA LETTER TU
 "\x{3339}" => "\x{30D8}\x{30EB}\x{30C4}",

 # 333A SQUARE PENSU
 # ->   30F3 KATAKANA LETTER N
 #    + 30B9 KATAKANA LETTER SU
 #    + 30D8 KATAKANA LETTER HE
 "\x{333A}" => "\x{30F3}\x{30B9}\x{30D8}",

 # 333B SQUARE PEEZI
 # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30D8 KATAKANA LETTER HE
 #    + 30B7 KATAKANA LETTER SI
 "\x{333B}" => "\x{30FC}\x{30D8}\x{30B7}",

 # 333C SQUARE BEETA
 # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30BF KATAKANA LETTER TA
 #    + 30D8 KATAKANA LETTER HE
 "\x{333C}" => "\x{30FC}\x{30BF}\x{30D8}",

 # 333D SQUARE POINTO
 # ->   30A4 KATAKANA LETTER I
 #    + 30F3 KATAKANA LETTER N
 #    + 30C8 KATAKANA LETTER TO
 #    + 30DB KATAKANA LETTER HO
 "\x{333D}" => "\x{30A4}\x{30F3}\x{30C8}\x{30DB}",

 # 333E SQUARE BORUTO
 # ->   30EB KATAKANA LETTER RU
 #    + 30C8 KATAKANA LETTER TO
 #    + 30DB KATAKANA LETTER HO
 "\x{333E}" => "\x{30EB}\x{30C8}\x{30DB}",

 # 333F SQUARE HON
 # ->   30DB KATAKANA LETTER HO
 #    + 30F3 KATAKANA LETTER N
 "\x{333F}" => "\x{30DB}\x{30F3}",

 # 3340 SQUARE PONDO
 # ->   30F3 KATAKANA LETTER N
 #    + 30DB KATAKANA LETTER HO
 #    + 30C8 KATAKANA LETTER TO
 "\x{3340}" => "\x{30F3}\x{30DB}\x{30C8}",

 # 3341 SQUARE HOORU
 # ->   30DB KATAKANA LETTER HO
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30EB KATAKANA LETTER RU
 "\x{3341}" => "\x{30DB}\x{30FC}\x{30EB}",

 # 3342 SQUARE HOON
 # ->   30DB KATAKANA LETTER HO
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30F3 KATAKANA LETTER N
 "\x{3342}" => "\x{30DB}\x{30FC}\x{30F3}",

 # 3343 SQUARE MAIKURO
 # ->   30DE KATAKANA LETTER MA
 #    + 30A4 KATAKANA LETTER I
 #    + 30AF KATAKANA LETTER KU
 #    + 30ED KATAKANA LETTER RO
 "\x{3343}" => "\x{30DE}\x{30A4}\x{30AF}\x{30ED}",

 # 3344 SQUARE MAIRU
 # ->   30DE KATAKANA LETTER MA
 #    + 30A4 KATAKANA LETTER I
 #    + 30EB KATAKANA LETTER RU
 "\x{3344}" => "\x{30DE}\x{30A4}\x{30EB}",

 # 3345 SQUARE MAHHA
 # ->   30DE KATAKANA LETTER MA
 #    + 30C3 KATAKANA LETTER SMALL TU
 #    + 30CF KATAKANA LETTER HA
 "\x{3345}" => "\x{30DE}\x{30C3}\x{30CF}",

 # 3346 SQUARE MARUKU
 # ->   30DE KATAKANA LETTER MA
 #    + 30EB KATAKANA LETTER RU
 #    + 30AF KATAKANA LETTER KU
 "\x{3346}" => "\x{30DE}\x{30EB}\x{30AF}",

 # 3347 SQUARE MANSYON
 # ->   30DE KATAKANA LETTER MA
 #    + 30F3 KATAKANA LETTER N
 #    + 30B7 KATAKANA LETTER SI
 #    + 30E7 KATAKANA LETTER SMALL YO
 #    + 30F3 KATAKANA LETTER N
 "\x{3347}" => "\x{30DE}\x{30F3}\x{30B7}\x{30E7}\x{30F3}",

 # 3348 SQUARE MIKURON
 # ->   30DF KATAKANA LETTER MI
 #    + 30AF KATAKANA LETTER KU
 #    + 30ED KATAKANA LETTER RO
 #    + 30F3 KATAKANA LETTER N
 "\x{3348}" => "\x{30DF}\x{30AF}\x{30ED}\x{30F3}",

 # 3349 SQUARE MIRI
 # ->   30DF KATAKANA LETTER MI
 #    + 30EA KATAKANA LETTER RI
 "\x{3349}" => "\x{30DF}\x{30EA}",

 # 334A SQUARE MIRIBAARU
 # ->   30DF KATAKANA LETTER MI
 #    + 30EA KATAKANA LETTER RI
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30EB KATAKANA LETTER RU
 #    + 30CF KATAKANA LETTER HA
 "\x{334A}" => "\x{30DF}\x{30EA}\x{30FC}\x{30EB}\x{30CF}",

 # 334B SQUARE MEGA
 # ->   30E1 KATAKANA LETTER ME
 #    + 30AB KATAKANA LETTER KA
 "\x{334B}" => "\x{30E1}\x{30AB}",

 # 334C SQUARE MEGATON
 # ->   30E1 KATAKANA LETTER ME
 #    + 30C8 KATAKANA LETTER TO
 #    + 30F3 KATAKANA LETTER N
 #    + 30AB KATAKANA LETTER KA
 "\x{334C}" => "\x{30E1}\x{30C8}\x{30F3}\x{30AB}",

 # 334D SQUARE MEETORU
 # ->   30E1 KATAKANA LETTER ME
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30C8 KATAKANA LETTER TO
 #    + 30EB KATAKANA LETTER RU
 "\x{334D}" => "\x{30E1}\x{30FC}\x{30C8}\x{30EB}",

 # 334E SQUARE YAADO
 # ->   30E4 KATAKANA LETTER YA
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30C8 KATAKANA LETTER TO
 "\x{334E}" => "\x{30E4}\x{30FC}\x{30C8}",

 # 334F SQUARE YAARU
 # ->   30E4 KATAKANA LETTER YA
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30EB KATAKANA LETTER RU
 "\x{334F}" => "\x{30E4}\x{30FC}\x{30EB}",

 # 3350 SQUARE YUAN
 # ->   30E6 KATAKANA LETTER YU
 #    + 30A2 KATAKANA LETTER A
 #    + 30F3 KATAKANA LETTER N
 "\x{3350}" => "\x{30E6}\x{30A2}\x{30F3}",

 # 3351 SQUARE RITTORU
 # ->   30EA KATAKANA LETTER RI
 #    + 30C3 KATAKANA LETTER SMALL TU
 #    + 30C8 KATAKANA LETTER TO
 #    + 30EB KATAKANA LETTER RU
 "\x{3351}" => "\x{30EA}\x{30C3}\x{30C8}\x{30EB}",

 # 3352 SQUARE RIRA
 # ->   30EA KATAKANA LETTER RI
 #    + 30E9 KATAKANA LETTER RA
 "\x{3352}" => "\x{30EA}\x{30E9}",

 # 3353 SQUARE RUPII
 # ->   30EB KATAKANA LETTER RU
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30D2 KATAKANA LETTER HI
 "\x{3353}" => "\x{30EB}\x{30FC}\x{30D2}",

 # 3354 SQUARE RUUBURU
 # ->   30EB KATAKANA LETTER RU
 #    + 30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 #    + 30EB KATAKANA LETTER RU
 #    + 30D5 KATAKANA LETTER HU
 "\x{3354}" => "\x{30EB}\x{30FC}\x{30EB}\x{30D5}",

 # 3355 SQUARE REMU
 # ->   30EC KATAKANA LETTER RE
 #    + 30E0 KATAKANA LETTER MU
 "\x{3355}" => "\x{30EC}\x{30E0}",

 # 3356 SQUARE RENTOGEN
 # ->   30EC KATAKANA LETTER RE
 #    + 30F3 KATAKANA LETTER N
 #    + 30C8 KATAKANA LETTER TO
 #    + 30F3 KATAKANA LETTER N
 #    + 30B1 KATAKANA LETTER KE
 "\x{3356}" => "\x{30EC}\x{30F3}\x{30C8}\x{30F3}\x{30B1}",

 # 3357 SQUARE WATTO
 # ->   30EF KATAKANA LETTER WA
 #    + 30C3 KATAKANA LETTER SMALL TU
 #    + 30C8 KATAKANA LETTER TO
 "\x{3357}" => "\x{30EF}\x{30C3}\x{30C8}",

 # 3358 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ZERO
 # ->   0030 DIGIT ZERO
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3358}" => "0\x{70B9}",

 # 3359 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ONE
 # ->   0031 DIGIT ONE
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3359}" => "1\x{70B9}",

 # 335A IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWO
 # ->   0032 DIGIT TWO
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{335A}" => "2\x{70B9}",

 # 335B IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THREE
 # ->   0033 DIGIT THREE
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{335B}" => "3\x{70B9}",

 # 335C IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOUR
 # ->   0034 DIGIT FOUR
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{335C}" => "4\x{70B9}",

 # 335D IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIVE
 # ->   0035 DIGIT FIVE
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{335D}" => "5\x{70B9}",

 # 335E IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIX
 # ->   0036 DIGIT SIX
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{335E}" => "6\x{70B9}",

 # 335F IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVEN
 # ->   0037 DIGIT SEVEN
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{335F}" => "7\x{70B9}",

 # 3360 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHT
 # ->   0038 DIGIT EIGHT
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3360}" => "8\x{70B9}",

 # 3361 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINE
 # ->   0039 DIGIT NINE
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3361}" => "9\x{70B9}",

 # 3362 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TEN
 # ->   0031 DIGIT ONE
 #    + 0030 DIGIT ZERO
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3362}" => "10\x{70B9}",

 # 3363 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ELEVEN
 # ->   0031 DIGIT ONE
 #    + 0031 DIGIT ONE
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3363}" => "11\x{70B9}",

 # 3364 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWELVE
 # ->   0031 DIGIT ONE
 #    + 0032 DIGIT TWO
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3364}" => "12\x{70B9}",

 # 3365 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THIRTEEN
 # ->   0031 DIGIT ONE
 #    + 0033 DIGIT THREE
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3365}" => "13\x{70B9}",

 # 3366 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOURTEEN
 # ->   0031 DIGIT ONE
 #    + 0034 DIGIT FOUR
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3366}" => "14\x{70B9}",

 # 3367 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIFTEEN
 # ->   0031 DIGIT ONE
 #    + 0035 DIGIT FIVE
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3367}" => "15\x{70B9}",

 # 3368 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIXTEEN
 # ->   0031 DIGIT ONE
 #    + 0036 DIGIT SIX
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3368}" => "16\x{70B9}",

 # 3369 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVENTEEN
 # ->   0031 DIGIT ONE
 #    + 0037 DIGIT SEVEN
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3369}" => "17\x{70B9}",

 # 336A IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHTEEN
 # ->   0031 DIGIT ONE
 #    + 0038 DIGIT EIGHT
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{336A}" => "18\x{70B9}",

 # 336B IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINETEEN
 # ->   0031 DIGIT ONE
 #    + 0039 DIGIT NINE
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{336B}" => "19\x{70B9}",

 # 336C IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY
 # ->   0032 DIGIT TWO
 #    + 0030 DIGIT ZERO
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{336C}" => "20\x{70B9}",

 # 336D IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-ONE
 # ->   0032 DIGIT TWO
 #    + 0031 DIGIT ONE
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{336D}" => "21\x{70B9}",

 # 336E IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-TWO
 # ->   0032 DIGIT TWO
 #    + 0032 DIGIT TWO
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{336E}" => "22\x{70B9}",

 # 336F IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-THREE
 # ->   0032 DIGIT TWO
 #    + 0033 DIGIT THREE
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{336F}" => "23\x{70B9}",

 # 3370 IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-FOUR
 # ->   0032 DIGIT TWO
 #    + 0034 DIGIT FOUR
 #    + 70B9 CJK UNIFIED IDEOGRAPH 70B9
 "\x{3370}" => "24\x{70B9}",

 # 3371 SQUARE HPA
 # ->   0068 LATIN SMALL LETTER H
 #    + 0050 LATIN CAPITAL LETTER P
 #    + 0061 LATIN SMALL LETTER A
 "\x{3371}" => "hPa",

 # 3372 SQUARE DA
 # ->   0064 LATIN SMALL LETTER D
 #    + 0061 LATIN SMALL LETTER A
 "\x{3372}" => "da",

 # 3373 SQUARE AU
 # ->   0041 LATIN CAPITAL LETTER A
 #    + 0055 LATIN CAPITAL LETTER U
 "\x{3373}" => "AU",

 # 3374 SQUARE BAR
 # ->   0062 LATIN SMALL LETTER B
 #    + 0061 LATIN SMALL LETTER A
 #    + 0072 LATIN SMALL LETTER R
 "\x{3374}" => "bar",

 # 3375 SQUARE OV
 # ->   006F LATIN SMALL LETTER O
 #    + 0056 LATIN CAPITAL LETTER V
 "\x{3375}" => "oV",

 # 3376 SQUARE PC
 # ->   0070 LATIN SMALL LETTER P
 #    + 0063 LATIN SMALL LETTER C
 "\x{3376}" => "pc",

 # 3377 SQUARE DM
 # ->   0064 LATIN SMALL LETTER D
 #    + 006D LATIN SMALL LETTER M
 "\x{3377}" => "dm",

 # 3378 SQUARE DM SQUARED
 # ->   0064 LATIN SMALL LETTER D
 #    + 006D LATIN SMALL LETTER M
 #    + 0032 DIGIT TWO
 "\x{3378}" => "dm2",

 # 3379 SQUARE DM CUBED
 # ->   0064 LATIN SMALL LETTER D
 #    + 006D LATIN SMALL LETTER M
 #    + 0033 DIGIT THREE
 "\x{3379}" => "dm3",

 # 337A SQUARE IU
 # ->   0049 LATIN CAPITAL LETTER I
 #    + 0055 LATIN CAPITAL LETTER U
 "\x{337A}" => "IU",

 # 337B SQUARE ERA NAME HEISEI
 # ->   5E73 CJK UNIFIED IDEOGRAPH 5E73
 #    + 6210 CJK UNIFIED IDEOGRAPH 6210
 "\x{337B}" => "\x{5E73}\x{6210}",

 # 337C SQUARE ERA NAME SYOUWA
 # ->   662D CJK UNIFIED IDEOGRAPH 662D
 #    + 548C CJK UNIFIED IDEOGRAPH 548C
 "\x{337C}" => "\x{662D}\x{548C}",

 # 337D SQUARE ERA NAME TAISYOU
 # ->   5927 CJK UNIFIED IDEOGRAPH 5927
 #    + 6B63 CJK UNIFIED IDEOGRAPH 6B63
 "\x{337D}" => "\x{5927}\x{6B63}",

 # 337E SQUARE ERA NAME MEIZI
 # ->   660E CJK UNIFIED IDEOGRAPH 660E
 #    + 6CBB CJK UNIFIED IDEOGRAPH 6CBB
 "\x{337E}" => "\x{660E}\x{6CBB}",

 # 337F SQUARE CORPORATION
 # ->   682A CJK UNIFIED IDEOGRAPH 682A
 #    + 5F0F CJK UNIFIED IDEOGRAPH 5F0F
 #    + 4F1A CJK UNIFIED IDEOGRAPH 4F1A
 #    + 793E CJK UNIFIED IDEOGRAPH 793E
 "\x{337F}" => "\x{682A}\x{5F0F}\x{4F1A}\x{793E}",

 # 3380 SQUARE PA AMPS
 # ->   0070 LATIN SMALL LETTER P
 #    + 0041 LATIN CAPITAL LETTER A
 "\x{3380}" => "pA",

 # 3381 SQUARE NA
 # ->   006E LATIN SMALL LETTER N
 #    + 0041 LATIN CAPITAL LETTER A
 "\x{3381}" => "nA",

 # 3382 SQUARE MU A
 # ->   03BC GREEK SMALL LETTER MU
 #    + 0041 LATIN CAPITAL LETTER A
 "\x{3382}" => "\x{03BC}A",

 # 3383 SQUARE MA
 # ->   006D LATIN SMALL LETTER M
 #    + 0041 LATIN CAPITAL LETTER A
 "\x{3383}" => "mA",

 # 3384 SQUARE KA
 # ->   006B LATIN SMALL LETTER K
 #    + 0041 LATIN CAPITAL LETTER A
 "\x{3384}" => "kA",

 # 3385 SQUARE KB
 # ->   004B LATIN CAPITAL LETTER K
 #    + 0042 LATIN CAPITAL LETTER B
 "\x{3385}" => "KB",

 # 3386 SQUARE MB
 # ->   004D LATIN CAPITAL LETTER M
 #    + 0042 LATIN CAPITAL LETTER B
 "\x{3386}" => "MB",

 # 3387 SQUARE GB
 # ->   0047 LATIN CAPITAL LETTER G
 #    + 0042 LATIN CAPITAL LETTER B
 "\x{3387}" => "GB",

 # 3388 SQUARE CAL
 # ->   0063 LATIN SMALL LETTER C
 #    + 0061 LATIN SMALL LETTER A
 #    + 006C LATIN SMALL LETTER L
 "\x{3388}" => "cal",

 # 3389 SQUARE KCAL
 # ->   006B LATIN SMALL LETTER K
 #    + 0063 LATIN SMALL LETTER C
 #    + 0061 LATIN SMALL LETTER A
 #    + 006C LATIN SMALL LETTER L
 "\x{3389}" => "kcal",

 # 338A SQUARE PF
 # ->   0070 LATIN SMALL LETTER P
 #    + 0046 LATIN CAPITAL LETTER F
 "\x{338A}" => "pF",

 # 338B SQUARE NF
 # ->   006E LATIN SMALL LETTER N
 #    + 0046 LATIN CAPITAL LETTER F
 "\x{338B}" => "nF",

 # 338C SQUARE MU F
 # ->   03BC GREEK SMALL LETTER MU
 #    + 0046 LATIN CAPITAL LETTER F
 "\x{338C}" => "\x{03BC}F",

 # 338D SQUARE MU G
 # ->   03BC GREEK SMALL LETTER MU
 #    + 0067 LATIN SMALL LETTER G
 "\x{338D}" => "\x{03BC}g",

 # 338E SQUARE MG
 # ->   006D LATIN SMALL LETTER M
 #    + 0067 LATIN SMALL LETTER G
 "\x{338E}" => "mg",

 # 338F SQUARE KG
 # ->   006B LATIN SMALL LETTER K
 #    + 0067 LATIN SMALL LETTER G
 "\x{338F}" => "kg",

 # 3390 SQUARE HZ
 # ->   0048 LATIN CAPITAL LETTER H
 #    + 007A LATIN SMALL LETTER Z
 "\x{3390}" => "Hz",

 # 3391 SQUARE KHZ
 # ->   006B LATIN SMALL LETTER K
 #    + 0048 LATIN CAPITAL LETTER H
 #    + 007A LATIN SMALL LETTER Z
 "\x{3391}" => "kHz",

 # 3392 SQUARE MHZ
 # ->   004D LATIN CAPITAL LETTER M
 #    + 0048 LATIN CAPITAL LETTER H
 #    + 007A LATIN SMALL LETTER Z
 "\x{3392}" => "MHz",

 # 3393 SQUARE GHZ
 # ->   0047 LATIN CAPITAL LETTER G
 #    + 0048 LATIN CAPITAL LETTER H
 #    + 007A LATIN SMALL LETTER Z
 "\x{3393}" => "GHz",

 # 3394 SQUARE THZ
 # ->   0054 LATIN CAPITAL LETTER T
 #    + 0048 LATIN CAPITAL LETTER H
 #    + 007A LATIN SMALL LETTER Z
 "\x{3394}" => "THz",

 # 3395 SQUARE MU L
 # ->   03BC GREEK SMALL LETTER MU
 #    + 006C LATIN SMALL LETTER L
 "\x{3395}" => "\x{03BC}l",

 # 3396 SQUARE ML
 # ->   006D LATIN SMALL LETTER M
 #    + 006C LATIN SMALL LETTER L
 "\x{3396}" => "ml",

 # 3397 SQUARE DL
 # ->   0064 LATIN SMALL LETTER D
 #    + 006C LATIN SMALL LETTER L
 "\x{3397}" => "dl",

 # 3398 SQUARE KL
 # ->   006B LATIN SMALL LETTER K
 #    + 006C LATIN SMALL LETTER L
 "\x{3398}" => "kl",

 # 3399 SQUARE FM
 # ->   0066 LATIN SMALL LETTER F
 #    + 006D LATIN SMALL LETTER M
 "\x{3399}" => "fm",

 # 339A SQUARE NM
 # ->   006E LATIN SMALL LETTER N
 #    + 006D LATIN SMALL LETTER M
 "\x{339A}" => "nm",

 # 339B SQUARE MU M
 # ->   03BC GREEK SMALL LETTER MU
 #    + 006D LATIN SMALL LETTER M
 "\x{339B}" => "\x{03BC}m",

 # 339C SQUARE MM
 # ->   006D LATIN SMALL LETTER M
 #    + 006D LATIN SMALL LETTER M
 "\x{339C}" => "mm",

 # 339D SQUARE CM
 # ->   0063 LATIN SMALL LETTER C
 #    + 006D LATIN SMALL LETTER M
 "\x{339D}" => "cm",

 # 339E SQUARE KM
 # ->   006B LATIN SMALL LETTER K
 #    + 006D LATIN SMALL LETTER M
 "\x{339E}" => "km",

 # 339F SQUARE MM SQUARED
 # ->   006D LATIN SMALL LETTER M
 #    + 006D LATIN SMALL LETTER M
 #    + 0032 DIGIT TWO
 "\x{339F}" => "mm2",

 # 33A0 SQUARE CM SQUARED
 # ->   0063 LATIN SMALL LETTER C
 #    + 006D LATIN SMALL LETTER M
 #    + 0032 DIGIT TWO
 "\x{33A0}" => "cm2",

 # 33A1 SQUARE M SQUARED
 # ->   006D LATIN SMALL LETTER M
 #    + 0032 DIGIT TWO
 "\x{33A1}" => "m2",

 # 33A2 SQUARE KM SQUARED
 # ->   006B LATIN SMALL LETTER K
 #    + 006D LATIN SMALL LETTER M
 #    + 0032 DIGIT TWO
 "\x{33A2}" => "km2",

 # 33A3 SQUARE MM CUBED
 # ->   006D LATIN SMALL LETTER M
 #    + 006D LATIN SMALL LETTER M
 #    + 0033 DIGIT THREE
 "\x{33A3}" => "mm3",

 # 33A4 SQUARE CM CUBED
 # ->   0063 LATIN SMALL LETTER C
 #    + 006D LATIN SMALL LETTER M
 #    + 0033 DIGIT THREE
 "\x{33A4}" => "cm3",

 # 33A5 SQUARE M CUBED
 # ->   006D LATIN SMALL LETTER M
 #    + 0033 DIGIT THREE
 "\x{33A5}" => "m3",

 # 33A6 SQUARE KM CUBED
 # ->   006B LATIN SMALL LETTER K
 #    + 006D LATIN SMALL LETTER M
 #    + 0033 DIGIT THREE
 "\x{33A6}" => "km3",

 # 33A7 SQUARE M OVER S
 # ->   006D LATIN SMALL LETTER M
 #    + 2215 DIVISION SLASH
 #    + 0073 LATIN SMALL LETTER S
 "\x{33A7}" => "m\x{2215}s",

 # 33A8 SQUARE M OVER S SQUARED
 # ->   006D LATIN SMALL LETTER M
 #    + 2215 DIVISION SLASH
 #    + 0073 LATIN SMALL LETTER S
 #    + 0032 DIGIT TWO
 "\x{33A8}" => "m\x{2215}s2",

 # 33A9 SQUARE PA
 # ->   0050 LATIN CAPITAL LETTER P
 #    + 0061 LATIN SMALL LETTER A
 "\x{33A9}" => "Pa",

 # 33AA SQUARE KPA
 # ->   006B LATIN SMALL LETTER K
 #    + 0050 LATIN CAPITAL LETTER P
 #    + 0061 LATIN SMALL LETTER A
 "\x{33AA}" => "kPa",

 # 33AB SQUARE MPA
 # ->   004D LATIN CAPITAL LETTER M
 #    + 0050 LATIN CAPITAL LETTER P
 #    + 0061 LATIN SMALL LETTER A
 "\x{33AB}" => "MPa",

 # 33AC SQUARE GPA
 # ->   0047 LATIN CAPITAL LETTER G
 #    + 0050 LATIN CAPITAL LETTER P
 #    + 0061 LATIN SMALL LETTER A
 "\x{33AC}" => "GPa",

 # 33AD SQUARE RAD
 # ->   0072 LATIN SMALL LETTER R
 #    + 0061 LATIN SMALL LETTER A
 #    + 0064 LATIN SMALL LETTER D
 "\x{33AD}" => "rad",

 # 33AE SQUARE RAD OVER S
 # ->   0072 LATIN SMALL LETTER R
 #    + 0061 LATIN SMALL LETTER A
 #    + 0064 LATIN SMALL LETTER D
 #    + 2215 DIVISION SLASH
 #    + 0073 LATIN SMALL LETTER S
 "\x{33AE}" => "rad\x{2215}s",

 # 33AF SQUARE RAD OVER S SQUARED
 # ->   0072 LATIN SMALL LETTER R
 #    + 0061 LATIN SMALL LETTER A
 #    + 0064 LATIN SMALL LETTER D
 #    + 2215 DIVISION SLASH
 #    + 0073 LATIN SMALL LETTER S
 #    + 0032 DIGIT TWO
 "\x{33AF}" => "rad\x{2215}s2",

 # 33B0 SQUARE PS
 # ->   0070 LATIN SMALL LETTER P
 #    + 0073 LATIN SMALL LETTER S
 "\x{33B0}" => "ps",

 # 33B1 SQUARE NS
 # ->   006E LATIN SMALL LETTER N
 #    + 0073 LATIN SMALL LETTER S
 "\x{33B1}" => "ns",

 # 33B2 SQUARE MU S
 # ->   03BC GREEK SMALL LETTER MU
 #    + 0073 LATIN SMALL LETTER S
 "\x{33B2}" => "\x{03BC}s",

 # 33B3 SQUARE MS
 # ->   006D LATIN SMALL LETTER M
 #    + 0073 LATIN SMALL LETTER S
 "\x{33B3}" => "ms",

 # 33B4 SQUARE PV
 # ->   0070 LATIN SMALL LETTER P
 #    + 0056 LATIN CAPITAL LETTER V
 "\x{33B4}" => "pV",

 # 33B5 SQUARE NV
 # ->   006E LATIN SMALL LETTER N
 #    + 0056 LATIN CAPITAL LETTER V
 "\x{33B5}" => "nV",

 # 33B6 SQUARE MU V
 # ->   03BC GREEK SMALL LETTER MU
 #    + 0056 LATIN CAPITAL LETTER V
 "\x{33B6}" => "\x{03BC}V",

 # 33B7 SQUARE MV
 # ->   006D LATIN SMALL LETTER M
 #    + 0056 LATIN CAPITAL LETTER V
 "\x{33B7}" => "mV",

 # 33B8 SQUARE KV
 # ->   006B LATIN SMALL LETTER K
 #    + 0056 LATIN CAPITAL LETTER V
 "\x{33B8}" => "kV",

 # 33B9 SQUARE MV MEGA
 # ->   004D LATIN CAPITAL LETTER M
 #    + 0056 LATIN CAPITAL LETTER V
 "\x{33B9}" => "MV",

 # 33BA SQUARE PW
 # ->   0070 LATIN SMALL LETTER P
 #    + 0057 LATIN CAPITAL LETTER W
 "\x{33BA}" => "pW",

 # 33BB SQUARE NW
 # ->   006E LATIN SMALL LETTER N
 #    + 0057 LATIN CAPITAL LETTER W
 "\x{33BB}" => "nW",

 # 33BC SQUARE MU W
 # ->   03BC GREEK SMALL LETTER MU
 #    + 0057 LATIN CAPITAL LETTER W
 "\x{33BC}" => "\x{03BC}W",

 # 33BD SQUARE MW
 # ->   006D LATIN SMALL LETTER M
 #    + 0057 LATIN CAPITAL LETTER W
 "\x{33BD}" => "mW",

 # 33BE SQUARE KW
 # ->   006B LATIN SMALL LETTER K
 #    + 0057 LATIN CAPITAL LETTER W
 "\x{33BE}" => "kW",

 # 33BF SQUARE MW MEGA
 # ->   004D LATIN CAPITAL LETTER M
 #    + 0057 LATIN CAPITAL LETTER W
 "\x{33BF}" => "MW",

 # 33C0 SQUARE K OHM
 # ->   006B LATIN SMALL LETTER K
 #    + 03A9 GREEK CAPITAL LETTER OMEGA
 "\x{33C0}" => "k\x{03A9}",

 # 33C1 SQUARE M OHM
 # ->   004D LATIN CAPITAL LETTER M
 #    + 03A9 GREEK CAPITAL LETTER OMEGA
 "\x{33C1}" => "M\x{03A9}",

 # 33C2 SQUARE AM
 # ->   0061 LATIN SMALL LETTER A
 #    + 002E FULL STOP
 #    + 006D LATIN SMALL LETTER M
 #    + 002E FULL STOP
 "\x{33C2}" => "a.m.",

 # 33C3 SQUARE BQ
 # ->   0042 LATIN CAPITAL LETTER B
 #    + 0071 LATIN SMALL LETTER Q
 "\x{33C3}" => "Bq",

 # 33C4 SQUARE CC
 # ->   0063 LATIN SMALL LETTER C
 #    + 0063 LATIN SMALL LETTER C
 "\x{33C4}" => "cc",

 # 33C5 SQUARE CD
 # ->   0063 LATIN SMALL LETTER C
 #    + 0064 LATIN SMALL LETTER D
 "\x{33C5}" => "cd",

 # 33C6 SQUARE C OVER KG
 # ->   0043 LATIN CAPITAL LETTER C
 #    + 2215 DIVISION SLASH
 #    + 006B LATIN SMALL LETTER K
 #    + 0067 LATIN SMALL LETTER G
 "\x{33C6}" => "C\x{2215}kg",

 # 33C7 SQUARE CO
 # ->   0043 LATIN CAPITAL LETTER C
 #    + 006F LATIN SMALL LETTER O
 #    + 002E FULL STOP
 "\x{33C7}" => "Co.",

 # 33C8 SQUARE DB
 # ->   0064 LATIN SMALL LETTER D
 #    + 0042 LATIN CAPITAL LETTER B
 "\x{33C8}" => "dB",

 # 33C9 SQUARE GY
 # ->   0047 LATIN CAPITAL LETTER G
 #    + 0079 LATIN SMALL LETTER Y
 "\x{33C9}" => "Gy",

 # 33CA SQUARE HA
 # ->   0068 LATIN SMALL LETTER H
 #    + 0061 LATIN SMALL LETTER A
 "\x{33CA}" => "ha",

 # 33CB SQUARE HP
 # ->   0048 LATIN CAPITAL LETTER H
 #    + 0050 LATIN CAPITAL LETTER P
 "\x{33CB}" => "HP",

 # 33CC SQUARE IN
 # ->   0069 LATIN SMALL LETTER I
 #    + 006E LATIN SMALL LETTER N
 "\x{33CC}" => "in",

 # 33CD SQUARE KK
 # ->   004B LATIN CAPITAL LETTER K
 #    + 004B LATIN CAPITAL LETTER K
 "\x{33CD}" => "KK",

 # 33CE SQUARE KM CAPITAL
 # ->   004B LATIN CAPITAL LETTER K
 #    + 004D LATIN CAPITAL LETTER M
 "\x{33CE}" => "KM",

 # 33CF SQUARE KT
 # ->   006B LATIN SMALL LETTER K
 #    + 0074 LATIN SMALL LETTER T
 "\x{33CF}" => "kt",

 # 33D0 SQUARE LM
 # ->   006C LATIN SMALL LETTER L
 #    + 006D LATIN SMALL LETTER M
 "\x{33D0}" => "lm",

 # 33D1 SQUARE LN
 # ->   006C LATIN SMALL LETTER L
 #    + 006E LATIN SMALL LETTER N
 "\x{33D1}" => "ln",

 # 33D2 SQUARE LOG
 # ->   006C LATIN SMALL LETTER L
 #    + 006F LATIN SMALL LETTER O
 #    + 0067 LATIN SMALL LETTER G
 "\x{33D2}" => "log",

 # 33D3 SQUARE LX
 # ->   006C LATIN SMALL LETTER L
 #    + 0078 LATIN SMALL LETTER X
 "\x{33D3}" => "lx",

 # 33D4 SQUARE MB SMALL
 # ->   006D LATIN SMALL LETTER M
 #    + 0062 LATIN SMALL LETTER B
 "\x{33D4}" => "mb",

 # 33D5 SQUARE MIL
 # ->   006D LATIN SMALL LETTER M
 #    + 0069 LATIN SMALL LETTER I
 #    + 006C LATIN SMALL LETTER L
 "\x{33D5}" => "mil",

 # 33D6 SQUARE MOL
 # ->   006D LATIN SMALL LETTER M
 #    + 006F LATIN SMALL LETTER O
 #    + 006C LATIN SMALL LETTER L
 "\x{33D6}" => "mol",

 # 33D7 SQUARE PH
 # ->   0050 LATIN CAPITAL LETTER P
 #    + 0048 LATIN CAPITAL LETTER H
 "\x{33D7}" => "PH",

 # 33D8 SQUARE PM
 # ->   0070 LATIN SMALL LETTER P
 #    + 002E FULL STOP
 #    + 006D LATIN SMALL LETTER M
 #    + 002E FULL STOP
 "\x{33D8}" => "p.m.",

 # 33D9 SQUARE PPM
 # ->   0050 LATIN CAPITAL LETTER P
 #    + 0050 LATIN CAPITAL LETTER P
 #    + 004D LATIN CAPITAL LETTER M
 "\x{33D9}" => "PPM",

 # 33DA SQUARE PR
 # ->   0050 LATIN CAPITAL LETTER P
 #    + 0052 LATIN CAPITAL LETTER R
 "\x{33DA}" => "PR",

 # 33DB SQUARE SR
 # ->   0073 LATIN SMALL LETTER S
 #    + 0072 LATIN SMALL LETTER R
 "\x{33DB}" => "sr",

 # 33DC SQUARE SV
 # ->   0053 LATIN CAPITAL LETTER S
 #    + 0076 LATIN SMALL LETTER V
 "\x{33DC}" => "Sv",

 # 33DD SQUARE WB
 # ->   0057 LATIN CAPITAL LETTER W
 #    + 0062 LATIN SMALL LETTER B
 "\x{33DD}" => "Wb",

 # 33DE SQUARE V OVER M
 # ->   0056 LATIN CAPITAL LETTER V
 #    + 2215 DIVISION SLASH
 #    + 006D LATIN SMALL LETTER M
 "\x{33DE}" => "V\x{2215}m",

 # 33DF SQUARE A OVER M
 # ->   0041 LATIN CAPITAL LETTER A
 #    + 2215 DIVISION SLASH
 #    + 006D LATIN SMALL LETTER M
 "\x{33DF}" => "A\x{2215}m",

 # 33E0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ONE
 # ->   0031 DIGIT ONE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33E0}" => "1\x{65E5}",

 # 33E1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWO
 # ->   0032 DIGIT TWO
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33E1}" => "2\x{65E5}",

 # 33E2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THREE
 # ->   0033 DIGIT THREE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33E2}" => "3\x{65E5}",

 # 33E3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOUR
 # ->   0034 DIGIT FOUR
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33E3}" => "4\x{65E5}",

 # 33E4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIVE
 # ->   0035 DIGIT FIVE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33E4}" => "5\x{65E5}",

 # 33E5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIX
 # ->   0036 DIGIT SIX
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33E5}" => "6\x{65E5}",

 # 33E6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVEN
 # ->   0037 DIGIT SEVEN
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33E6}" => "7\x{65E5}",

 # 33E7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHT
 # ->   0038 DIGIT EIGHT
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33E7}" => "8\x{65E5}",

 # 33E8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINE
 # ->   0039 DIGIT NINE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33E8}" => "9\x{65E5}",

 # 33E9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TEN
 # ->   0031 DIGIT ONE
 #    + 0030 DIGIT ZERO
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33E9}" => "10\x{65E5}",

 # 33EA IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ELEVEN
 # ->   0031 DIGIT ONE
 #    + 0031 DIGIT ONE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33EA}" => "11\x{65E5}",

 # 33EB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWELVE
 # ->   0031 DIGIT ONE
 #    + 0032 DIGIT TWO
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33EB}" => "12\x{65E5}",

 # 33EC IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTEEN
 # ->   0031 DIGIT ONE
 #    + 0033 DIGIT THREE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33EC}" => "13\x{65E5}",

 # 33ED IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOURTEEN
 # ->   0031 DIGIT ONE
 #    + 0034 DIGIT FOUR
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33ED}" => "14\x{65E5}",

 # 33EE IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIFTEEN
 # ->   0031 DIGIT ONE
 #    + 0035 DIGIT FIVE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33EE}" => "15\x{65E5}",

 # 33EF IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIXTEEN
 # ->   0031 DIGIT ONE
 #    + 0036 DIGIT SIX
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33EF}" => "16\x{65E5}",

 # 33F0 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVENTEEN
 # ->   0031 DIGIT ONE
 #    + 0037 DIGIT SEVEN
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33F0}" => "17\x{65E5}",

 # 33F1 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHTEEN
 # ->   0031 DIGIT ONE
 #    + 0038 DIGIT EIGHT
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33F1}" => "18\x{65E5}",

 # 33F2 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINETEEN
 # ->   0031 DIGIT ONE
 #    + 0039 DIGIT NINE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33F2}" => "19\x{65E5}",

 # 33F3 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY
 # ->   0032 DIGIT TWO
 #    + 0030 DIGIT ZERO
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33F3}" => "20\x{65E5}",

 # 33F4 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-ONE
 # ->   0032 DIGIT TWO
 #    + 0031 DIGIT ONE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33F4}" => "21\x{65E5}",

 # 33F5 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-TWO
 # ->   0032 DIGIT TWO
 #    + 0032 DIGIT TWO
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33F5}" => "22\x{65E5}",

 # 33F6 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-THREE
 # ->   0032 DIGIT TWO
 #    + 0033 DIGIT THREE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33F6}" => "23\x{65E5}",

 # 33F7 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FOUR
 # ->   0032 DIGIT TWO
 #    + 0034 DIGIT FOUR
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33F7}" => "24\x{65E5}",

 # 33F8 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FIVE
 # ->   0032 DIGIT TWO
 #    + 0035 DIGIT FIVE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33F8}" => "25\x{65E5}",

 # 33F9 IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SIX
 # ->   0032 DIGIT TWO
 #    + 0036 DIGIT SIX
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33F9}" => "26\x{65E5}",

 # 33FA IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SEVEN
 # ->   0032 DIGIT TWO
 #    + 0037 DIGIT SEVEN
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33FA}" => "27\x{65E5}",

 # 33FB IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-EIGHT
 # ->   0032 DIGIT TWO
 #    + 0038 DIGIT EIGHT
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33FB}" => "28\x{65E5}",

 # 33FC IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-NINE
 # ->   0032 DIGIT TWO
 #    + 0039 DIGIT NINE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33FC}" => "29\x{65E5}",

 # 33FD IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY
 # ->   0033 DIGIT THREE
 #    + 0030 DIGIT ZERO
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33FD}" => "30\x{65E5}",

 # 33FE IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY-ONE
 # ->   0033 DIGIT THREE
 #    + 0031 DIGIT ONE
 #    + 65E5 CJK UNIFIED IDEOGRAPH 65E5
 "\x{33FE}" => "31\x{65E5}",

 # 33FF SQUARE GAL
 # ->   0067 LATIN SMALL LETTER G
 #    + 0061 LATIN SMALL LETTER A
 #    + 006C LATIN SMALL LETTER L
 "\x{33FF}" => "gal",

 # F900 CJK UNIFIED IDEOGRAPH F900
 # ->   8C48 CJK UNIFIED IDEOGRAPH 8C48
 "\x{F900}" => "\x{8C48}",

 # F901 CJK UNIFIED IDEOGRAPH F901
 # ->   66F4 CJK UNIFIED IDEOGRAPH 66F4
 "\x{F901}" => "\x{66F4}",

 # F902 CJK UNIFIED IDEOGRAPH F902
 # ->   8ECA CJK UNIFIED IDEOGRAPH 8ECA
 "\x{F902}" => "\x{8ECA}",

 # F903 CJK UNIFIED IDEOGRAPH F903
 # ->   8CC8 CJK UNIFIED IDEOGRAPH 8CC8
 "\x{F903}" => "\x{8CC8}",

 # F904 CJK UNIFIED IDEOGRAPH F904
 # ->   6ED1 CJK UNIFIED IDEOGRAPH 6ED1
 "\x{F904}" => "\x{6ED1}",

 # F905 CJK UNIFIED IDEOGRAPH F905
 # ->   4E32 CJK UNIFIED IDEOGRAPH 4E32
 "\x{F905}" => "\x{4E32}",

 # F906 CJK UNIFIED IDEOGRAPH F906
 # ->   53E5 CJK UNIFIED IDEOGRAPH 53E5
 "\x{F906}" => "\x{53E5}",

 # F907 CJK UNIFIED IDEOGRAPH F907
 # ->   9F9C CJK UNIFIED IDEOGRAPH 9F9C
 "\x{F907}" => "\x{9F9C}",

 # F908 CJK UNIFIED IDEOGRAPH F908
 # ->   9F9C CJK UNIFIED IDEOGRAPH 9F9C
 "\x{F908}" => "\x{9F9C}",

 # F909 CJK UNIFIED IDEOGRAPH F909
 # ->   5951 CJK UNIFIED IDEOGRAPH 5951
 "\x{F909}" => "\x{5951}",

 # F90A CJK UNIFIED IDEOGRAPH F90A
 # ->   91D1 CJK UNIFIED IDEOGRAPH 91D1
 "\x{F90A}" => "\x{91D1}",

 # F90B CJK UNIFIED IDEOGRAPH F90B
 # ->   5587 CJK UNIFIED IDEOGRAPH 5587
 "\x{F90B}" => "\x{5587}",

 # F90C CJK UNIFIED IDEOGRAPH F90C
 # ->   5948 CJK UNIFIED IDEOGRAPH 5948
 "\x{F90C}" => "\x{5948}",

 # F90D CJK UNIFIED IDEOGRAPH F90D
 # ->   61F6 CJK UNIFIED IDEOGRAPH 61F6
 "\x{F90D}" => "\x{61F6}",

 # F90E CJK UNIFIED IDEOGRAPH F90E
 # ->   7669 CJK UNIFIED IDEOGRAPH 7669
 "\x{F90E}" => "\x{7669}",

 # F90F CJK UNIFIED IDEOGRAPH F90F
 # ->   7F85 CJK UNIFIED IDEOGRAPH 7F85
 "\x{F90F}" => "\x{7F85}",

 # F910 CJK UNIFIED IDEOGRAPH F910
 # ->   863F CJK UNIFIED IDEOGRAPH 863F
 "\x{F910}" => "\x{863F}",

 # F911 CJK UNIFIED IDEOGRAPH F911
 # ->   87BA CJK UNIFIED IDEOGRAPH 87BA
 "\x{F911}" => "\x{87BA}",

 # F912 CJK UNIFIED IDEOGRAPH F912
 # ->   88F8 CJK UNIFIED IDEOGRAPH 88F8
 "\x{F912}" => "\x{88F8}",

 # F913 CJK UNIFIED IDEOGRAPH F913
 # ->   908F CJK UNIFIED IDEOGRAPH 908F
 "\x{F913}" => "\x{908F}",

 # F914 CJK UNIFIED IDEOGRAPH F914
 # ->   6A02 CJK UNIFIED IDEOGRAPH 6A02
 "\x{F914}" => "\x{6A02}",

 # F915 CJK UNIFIED IDEOGRAPH F915
 # ->   6D1B CJK UNIFIED IDEOGRAPH 6D1B
 "\x{F915}" => "\x{6D1B}",

 # F916 CJK UNIFIED IDEOGRAPH F916
 # ->   70D9 CJK UNIFIED IDEOGRAPH 70D9
 "\x{F916}" => "\x{70D9}",

 # F917 CJK UNIFIED IDEOGRAPH F917
 # ->   73DE CJK UNIFIED IDEOGRAPH 73DE
 "\x{F917}" => "\x{73DE}",

 # F918 CJK UNIFIED IDEOGRAPH F918
 # ->   843D CJK UNIFIED IDEOGRAPH 843D
 "\x{F918}" => "\x{843D}",

 # F919 CJK UNIFIED IDEOGRAPH F919
 # ->   916A CJK UNIFIED IDEOGRAPH 916A
 "\x{F919}" => "\x{916A}",

 # F91A CJK UNIFIED IDEOGRAPH F91A
 # ->   99F1 CJK UNIFIED IDEOGRAPH 99F1
 "\x{F91A}" => "\x{99F1}",

 # F91B CJK UNIFIED IDEOGRAPH F91B
 # ->   4E82 CJK UNIFIED IDEOGRAPH 4E82
 "\x{F91B}" => "\x{4E82}",

 # F91C CJK UNIFIED IDEOGRAPH F91C
 # ->   5375 CJK UNIFIED IDEOGRAPH 5375
 "\x{F91C}" => "\x{5375}",

 # F91D CJK UNIFIED IDEOGRAPH F91D
 # ->   6B04 CJK UNIFIED IDEOGRAPH 6B04
 "\x{F91D}" => "\x{6B04}",

 # F91E CJK UNIFIED IDEOGRAPH F91E
 # ->   721B CJK UNIFIED IDEOGRAPH 721B
 "\x{F91E}" => "\x{721B}",

 # F91F CJK UNIFIED IDEOGRAPH F91F
 # ->   862D CJK UNIFIED IDEOGRAPH 862D
 "\x{F91F}" => "\x{862D}",

 # F920 CJK UNIFIED IDEOGRAPH F920
 # ->   9E1E CJK UNIFIED IDEOGRAPH 9E1E
 "\x{F920}" => "\x{9E1E}",

 # F921 CJK UNIFIED IDEOGRAPH F921
 # ->   5D50 CJK UNIFIED IDEOGRAPH 5D50
 "\x{F921}" => "\x{5D50}",

 # F922 CJK UNIFIED IDEOGRAPH F922
 # ->   6FEB CJK UNIFIED IDEOGRAPH 6FEB
 "\x{F922}" => "\x{6FEB}",

 # F923 CJK UNIFIED IDEOGRAPH F923
 # ->   85CD CJK UNIFIED IDEOGRAPH 85CD
 "\x{F923}" => "\x{85CD}",

 # F924 CJK UNIFIED IDEOGRAPH F924
 # ->   8964 CJK UNIFIED IDEOGRAPH 8964
 "\x{F924}" => "\x{8964}",

 # F925 CJK UNIFIED IDEOGRAPH F925
 # ->   62C9 CJK UNIFIED IDEOGRAPH 62C9
 "\x{F925}" => "\x{62C9}",

 # F926 CJK UNIFIED IDEOGRAPH F926
 # ->   81D8 CJK UNIFIED IDEOGRAPH 81D8
 "\x{F926}" => "\x{81D8}",

 # F927 CJK UNIFIED IDEOGRAPH F927
 # ->   881F CJK UNIFIED IDEOGRAPH 881F
 "\x{F927}" => "\x{881F}",

 # F928 CJK UNIFIED IDEOGRAPH F928
 # ->   5ECA CJK UNIFIED IDEOGRAPH 5ECA
 "\x{F928}" => "\x{5ECA}",

 # F929 CJK UNIFIED IDEOGRAPH F929
 # ->   6717 CJK UNIFIED IDEOGRAPH 6717
 "\x{F929}" => "\x{6717}",

 # F92A CJK UNIFIED IDEOGRAPH F92A
 # ->   6D6A CJK UNIFIED IDEOGRAPH 6D6A
 "\x{F92A}" => "\x{6D6A}",

 # F92B CJK UNIFIED IDEOGRAPH F92B
 # ->   72FC CJK UNIFIED IDEOGRAPH 72FC
 "\x{F92B}" => "\x{72FC}",

 # F92C CJK UNIFIED IDEOGRAPH F92C
 # ->   90CE CJK UNIFIED IDEOGRAPH 90CE
 "\x{F92C}" => "\x{90CE}",

 # F92D CJK UNIFIED IDEOGRAPH F92D
 # ->   4F86 CJK UNIFIED IDEOGRAPH 4F86
 "\x{F92D}" => "\x{4F86}",

 # F92E CJK UNIFIED IDEOGRAPH F92E
 # ->   51B7 CJK UNIFIED IDEOGRAPH 51B7
 "\x{F92E}" => "\x{51B7}",

 # F92F CJK UNIFIED IDEOGRAPH F92F
 # ->   52DE CJK UNIFIED IDEOGRAPH 52DE
 "\x{F92F}" => "\x{52DE}",

 # F930 CJK UNIFIED IDEOGRAPH F930
 # ->   64C4 CJK UNIFIED IDEOGRAPH 64C4
 "\x{F930}" => "\x{64C4}",

 # F931 CJK UNIFIED IDEOGRAPH F931
 # ->   6AD3 CJK UNIFIED IDEOGRAPH 6AD3
 "\x{F931}" => "\x{6AD3}",

 # F932 CJK UNIFIED IDEOGRAPH F932
 # ->   7210 CJK UNIFIED IDEOGRAPH 7210
 "\x{F932}" => "\x{7210}",

 # F933 CJK UNIFIED IDEOGRAPH F933
 # ->   76E7 CJK UNIFIED IDEOGRAPH 76E7
 "\x{F933}" => "\x{76E7}",

 # F934 CJK UNIFIED IDEOGRAPH F934
 # ->   8001 CJK UNIFIED IDEOGRAPH 8001
 "\x{F934}" => "\x{8001}",

 # F935 CJK UNIFIED IDEOGRAPH F935
 # ->   8606 CJK UNIFIED IDEOGRAPH 8606
 "\x{F935}" => "\x{8606}",

 # F936 CJK UNIFIED IDEOGRAPH F936
 # ->   865C CJK UNIFIED IDEOGRAPH 865C
 "\x{F936}" => "\x{865C}",

 # F937 CJK UNIFIED IDEOGRAPH F937
 # ->   8DEF CJK UNIFIED IDEOGRAPH 8DEF
 "\x{F937}" => "\x{8DEF}",

 # F938 CJK UNIFIED IDEOGRAPH F938
 # ->   9732 CJK UNIFIED IDEOGRAPH 9732
 "\x{F938}" => "\x{9732}",

 # F939 CJK UNIFIED IDEOGRAPH F939
 # ->   9B6F CJK UNIFIED IDEOGRAPH 9B6F
 "\x{F939}" => "\x{9B6F}",

 # F93A CJK UNIFIED IDEOGRAPH F93A
 # ->   9DFA CJK UNIFIED IDEOGRAPH 9DFA
 "\x{F93A}" => "\x{9DFA}",

 # F93B CJK UNIFIED IDEOGRAPH F93B
 # ->   788C CJK UNIFIED IDEOGRAPH 788C
 "\x{F93B}" => "\x{788C}",

 # F93C CJK UNIFIED IDEOGRAPH F93C
 # ->   797F CJK UNIFIED IDEOGRAPH 797F
 "\x{F93C}" => "\x{797F}",

 # F93D CJK UNIFIED IDEOGRAPH F93D
 # ->   7DA0 CJK UNIFIED IDEOGRAPH 7DA0
 "\x{F93D}" => "\x{7DA0}",

 # F93E CJK UNIFIED IDEOGRAPH F93E
 # ->   83C9 CJK UNIFIED IDEOGRAPH 83C9
 "\x{F93E}" => "\x{83C9}",

 # F93F CJK UNIFIED IDEOGRAPH F93F
 # ->   9304 CJK UNIFIED IDEOGRAPH 9304
 "\x{F93F}" => "\x{9304}",

 # F940 CJK UNIFIED IDEOGRAPH F940
 # ->   9E7F CJK UNIFIED IDEOGRAPH 9E7F
 "\x{F940}" => "\x{9E7F}",

 # F941 CJK UNIFIED IDEOGRAPH F941
 # ->   8AD6 CJK UNIFIED IDEOGRAPH 8AD6
 "\x{F941}" => "\x{8AD6}",

 # F942 CJK UNIFIED IDEOGRAPH F942
 # ->   58DF CJK UNIFIED IDEOGRAPH 58DF
 "\x{F942}" => "\x{58DF}",

 # F943 CJK UNIFIED IDEOGRAPH F943
 # ->   5F04 CJK UNIFIED IDEOGRAPH 5F04
 "\x{F943}" => "\x{5F04}",

 # F944 CJK UNIFIED IDEOGRAPH F944
 # ->   7C60 CJK UNIFIED IDEOGRAPH 7C60
 "\x{F944}" => "\x{7C60}",

 # F945 CJK UNIFIED IDEOGRAPH F945
 # ->   807E CJK UNIFIED IDEOGRAPH 807E
 "\x{F945}" => "\x{807E}",

 # F946 CJK UNIFIED IDEOGRAPH F946
 # ->   7262 CJK UNIFIED IDEOGRAPH 7262
 "\x{F946}" => "\x{7262}",

 # F947 CJK UNIFIED IDEOGRAPH F947
 # ->   78CA CJK UNIFIED IDEOGRAPH 78CA
 "\x{F947}" => "\x{78CA}",

 # F948 CJK UNIFIED IDEOGRAPH F948
 # ->   8CC2 CJK UNIFIED IDEOGRAPH 8CC2
 "\x{F948}" => "\x{8CC2}",

 # F949 CJK UNIFIED IDEOGRAPH F949
 # ->   96F7 CJK UNIFIED IDEOGRAPH 96F7
 "\x{F949}" => "\x{96F7}",

 # F94A CJK UNIFIED IDEOGRAPH F94A
 # ->   58D8 CJK UNIFIED IDEOGRAPH 58D8
 "\x{F94A}" => "\x{58D8}",

 # F94B CJK UNIFIED IDEOGRAPH F94B
 # ->   5C62 CJK UNIFIED IDEOGRAPH 5C62
 "\x{F94B}" => "\x{5C62}",

 # F94C CJK UNIFIED IDEOGRAPH F94C
 # ->   6A13 CJK UNIFIED IDEOGRAPH 6A13
 "\x{F94C}" => "\x{6A13}",

 # F94D CJK UNIFIED IDEOGRAPH F94D
 # ->   6DDA CJK UNIFIED IDEOGRAPH 6DDA
 "\x{F94D}" => "\x{6DDA}",

 # F94E CJK UNIFIED IDEOGRAPH F94E
 # ->   6F0F CJK UNIFIED IDEOGRAPH 6F0F
 "\x{F94E}" => "\x{6F0F}",

 # F94F CJK UNIFIED IDEOGRAPH F94F
 # ->   7D2F CJK UNIFIED IDEOGRAPH 7D2F
 "\x{F94F}" => "\x{7D2F}",

 # F950 CJK UNIFIED IDEOGRAPH F950
 # ->   7E37 CJK UNIFIED IDEOGRAPH 7E37
 "\x{F950}" => "\x{7E37}",

 # F951 CJK UNIFIED IDEOGRAPH F951
 # ->   964B CJK UNIFIED IDEOGRAPH 964B
 "\x{F951}" => "\x{964B}",

 # F952 CJK UNIFIED IDEOGRAPH F952
 # ->   52D2 CJK UNIFIED IDEOGRAPH 52D2
 "\x{F952}" => "\x{52D2}",

 # F953 CJK UNIFIED IDEOGRAPH F953
 # ->   808B CJK UNIFIED IDEOGRAPH 808B
 "\x{F953}" => "\x{808B}",

 # F954 CJK UNIFIED IDEOGRAPH F954
 # ->   51DC CJK UNIFIED IDEOGRAPH 51DC
 "\x{F954}" => "\x{51DC}",

 # F955 CJK UNIFIED IDEOGRAPH F955
 # ->   51CC CJK UNIFIED IDEOGRAPH 51CC
 "\x{F955}" => "\x{51CC}",

 # F956 CJK UNIFIED IDEOGRAPH F956
 # ->   7A1C CJK UNIFIED IDEOGRAPH 7A1C
 "\x{F956}" => "\x{7A1C}",

 # F957 CJK UNIFIED IDEOGRAPH F957
 # ->   7DBE CJK UNIFIED IDEOGRAPH 7DBE
 "\x{F957}" => "\x{7DBE}",

 # F958 CJK UNIFIED IDEOGRAPH F958
 # ->   83F1 CJK UNIFIED IDEOGRAPH 83F1
 "\x{F958}" => "\x{83F1}",

 # F959 CJK UNIFIED IDEOGRAPH F959
 # ->   9675 CJK UNIFIED IDEOGRAPH 9675
 "\x{F959}" => "\x{9675}",

 # F95A CJK UNIFIED IDEOGRAPH F95A
 # ->   8B80 CJK UNIFIED IDEOGRAPH 8B80
 "\x{F95A}" => "\x{8B80}",

 # F95B CJK UNIFIED IDEOGRAPH F95B
 # ->   62CF CJK UNIFIED IDEOGRAPH 62CF
 "\x{F95B}" => "\x{62CF}",

 # F95C CJK UNIFIED IDEOGRAPH F95C
 # ->   6A02 CJK UNIFIED IDEOGRAPH 6A02
 "\x{F95C}" => "\x{6A02}",

 # F95D CJK UNIFIED IDEOGRAPH F95D
 # ->   8AFE CJK UNIFIED IDEOGRAPH 8AFE
 "\x{F95D}" => "\x{8AFE}",

 # F95E CJK UNIFIED IDEOGRAPH F95E
 # ->   4E39 CJK UNIFIED IDEOGRAPH 4E39
 "\x{F95E}" => "\x{4E39}",

 # F95F CJK UNIFIED IDEOGRAPH F95F
 # ->   5BE7 CJK UNIFIED IDEOGRAPH 5BE7
 "\x{F95F}" => "\x{5BE7}",

 # F960 CJK UNIFIED IDEOGRAPH F960
 # ->   6012 CJK UNIFIED IDEOGRAPH 6012
 "\x{F960}" => "\x{6012}",

 # F961 CJK UNIFIED IDEOGRAPH F961
 # ->   7387 CJK UNIFIED IDEOGRAPH 7387
 "\x{F961}" => "\x{7387}",

 # F962 CJK UNIFIED IDEOGRAPH F962
 # ->   7570 CJK UNIFIED IDEOGRAPH 7570
 "\x{F962}" => "\x{7570}",

 # F963 CJK UNIFIED IDEOGRAPH F963
 # ->   5317 CJK UNIFIED IDEOGRAPH 5317
 "\x{F963}" => "\x{5317}",

 # F964 CJK UNIFIED IDEOGRAPH F964
 # ->   78FB CJK UNIFIED IDEOGRAPH 78FB
 "\x{F964}" => "\x{78FB}",

 # F965 CJK UNIFIED IDEOGRAPH F965
 # ->   4FBF CJK UNIFIED IDEOGRAPH 4FBF
 "\x{F965}" => "\x{4FBF}",

 # F966 CJK UNIFIED IDEOGRAPH F966
 # ->   5FA9 CJK UNIFIED IDEOGRAPH 5FA9
 "\x{F966}" => "\x{5FA9}",

 # F967 CJK UNIFIED IDEOGRAPH F967
 # ->   4E0D CJK UNIFIED IDEOGRAPH 4E0D
 "\x{F967}" => "\x{4E0D}",

 # F968 CJK UNIFIED IDEOGRAPH F968
 # ->   6CCC CJK UNIFIED IDEOGRAPH 6CCC
 "\x{F968}" => "\x{6CCC}",

 # F969 CJK UNIFIED IDEOGRAPH F969
 # ->   6578 CJK UNIFIED IDEOGRAPH 6578
 "\x{F969}" => "\x{6578}",

 # F96A CJK UNIFIED IDEOGRAPH F96A
 # ->   7D22 CJK UNIFIED IDEOGRAPH 7D22
 "\x{F96A}" => "\x{7D22}",

 # F96B CJK UNIFIED IDEOGRAPH F96B
 # ->   53C3 CJK UNIFIED IDEOGRAPH 53C3
 "\x{F96B}" => "\x{53C3}",

 # F96C CJK UNIFIED IDEOGRAPH F96C
 # ->   585E CJK UNIFIED IDEOGRAPH 585E
 "\x{F96C}" => "\x{585E}",

 # F96D CJK UNIFIED IDEOGRAPH F96D
 # ->   7701 CJK UNIFIED IDEOGRAPH 7701
 "\x{F96D}" => "\x{7701}",

 # F96E CJK UNIFIED IDEOGRAPH F96E
 # ->   8449 CJK UNIFIED IDEOGRAPH 8449
 "\x{F96E}" => "\x{8449}",

 # F96F CJK UNIFIED IDEOGRAPH F96F
 # ->   8AAA CJK UNIFIED IDEOGRAPH 8AAA
 "\x{F96F}" => "\x{8AAA}",

 # F970 CJK UNIFIED IDEOGRAPH F970
 # ->   6BBA CJK UNIFIED IDEOGRAPH 6BBA
 "\x{F970}" => "\x{6BBA}",

 # F971 CJK UNIFIED IDEOGRAPH F971
 # ->   8FB0 CJK UNIFIED IDEOGRAPH 8FB0
 "\x{F971}" => "\x{8FB0}",

 # F972 CJK UNIFIED IDEOGRAPH F972
 # ->   6C88 CJK UNIFIED IDEOGRAPH 6C88
 "\x{F972}" => "\x{6C88}",

 # F973 CJK UNIFIED IDEOGRAPH F973
 # ->   62FE CJK UNIFIED IDEOGRAPH 62FE
 "\x{F973}" => "\x{62FE}",

 # F974 CJK UNIFIED IDEOGRAPH F974
 # ->   82E5 CJK UNIFIED IDEOGRAPH 82E5
 "\x{F974}" => "\x{82E5}",

 # F975 CJK UNIFIED IDEOGRAPH F975
 # ->   63A0 CJK UNIFIED IDEOGRAPH 63A0
 "\x{F975}" => "\x{63A0}",

 # F976 CJK UNIFIED IDEOGRAPH F976
 # ->   7565 CJK UNIFIED IDEOGRAPH 7565
 "\x{F976}" => "\x{7565}",

 # F977 CJK UNIFIED IDEOGRAPH F977
 # ->   4EAE CJK UNIFIED IDEOGRAPH 4EAE
 "\x{F977}" => "\x{4EAE}",

 # F978 CJK UNIFIED IDEOGRAPH F978
 # ->   5169 CJK UNIFIED IDEOGRAPH 5169
 "\x{F978}" => "\x{5169}",

 # F979 CJK UNIFIED IDEOGRAPH F979
 # ->   51C9 CJK UNIFIED IDEOGRAPH 51C9
 "\x{F979}" => "\x{51C9}",

 # F97A CJK UNIFIED IDEOGRAPH F97A
 # ->   6881 CJK UNIFIED IDEOGRAPH 6881
 "\x{F97A}" => "\x{6881}",

 # F97B CJK UNIFIED IDEOGRAPH F97B
 # ->   7CE7 CJK UNIFIED IDEOGRAPH 7CE7
 "\x{F97B}" => "\x{7CE7}",

 # F97C CJK UNIFIED IDEOGRAPH F97C
 # ->   826F CJK UNIFIED IDEOGRAPH 826F
 "\x{F97C}" => "\x{826F}",

 # F97D CJK UNIFIED IDEOGRAPH F97D
 # ->   8AD2 CJK UNIFIED IDEOGRAPH 8AD2
 "\x{F97D}" => "\x{8AD2}",

 # F97E CJK UNIFIED IDEOGRAPH F97E
 # ->   91CF CJK UNIFIED IDEOGRAPH 91CF
 "\x{F97E}" => "\x{91CF}",

 # F97F CJK UNIFIED IDEOGRAPH F97F
 # ->   52F5 CJK UNIFIED IDEOGRAPH 52F5
 "\x{F97F}" => "\x{52F5}",

 # F980 CJK UNIFIED IDEOGRAPH F980
 # ->   5442 CJK UNIFIED IDEOGRAPH 5442
 "\x{F980}" => "\x{5442}",

 # F981 CJK UNIFIED IDEOGRAPH F981
 # ->   5973 CJK UNIFIED IDEOGRAPH 5973
 "\x{F981}" => "\x{5973}",

 # F982 CJK UNIFIED IDEOGRAPH F982
 # ->   5EEC CJK UNIFIED IDEOGRAPH 5EEC
 "\x{F982}" => "\x{5EEC}",

 # F983 CJK UNIFIED IDEOGRAPH F983
 # ->   65C5 CJK UNIFIED IDEOGRAPH 65C5
 "\x{F983}" => "\x{65C5}",

 # F984 CJK UNIFIED IDEOGRAPH F984
 # ->   6FFE CJK UNIFIED IDEOGRAPH 6FFE
 "\x{F984}" => "\x{6FFE}",

 # F985 CJK UNIFIED IDEOGRAPH F985
 # ->   792A CJK UNIFIED IDEOGRAPH 792A
 "\x{F985}" => "\x{792A}",

 # F986 CJK UNIFIED IDEOGRAPH F986
 # ->   95AD CJK UNIFIED IDEOGRAPH 95AD
 "\x{F986}" => "\x{95AD}",

 # F987 CJK UNIFIED IDEOGRAPH F987
 # ->   9A6A CJK UNIFIED IDEOGRAPH 9A6A
 "\x{F987}" => "\x{9A6A}",

 # F988 CJK UNIFIED IDEOGRAPH F988
 # ->   9E97 CJK UNIFIED IDEOGRAPH 9E97
 "\x{F988}" => "\x{9E97}",

 # F989 CJK UNIFIED IDEOGRAPH F989
 # ->   9ECE CJK UNIFIED IDEOGRAPH 9ECE
 "\x{F989}" => "\x{9ECE}",

 # F98A CJK UNIFIED IDEOGRAPH F98A
 # ->   529B CJK UNIFIED IDEOGRAPH 529B
 "\x{F98A}" => "\x{529B}",

 # F98B CJK UNIFIED IDEOGRAPH F98B
 # ->   66C6 CJK UNIFIED IDEOGRAPH 66C6
 "\x{F98B}" => "\x{66C6}",

 # F98C CJK UNIFIED IDEOGRAPH F98C
 # ->   6B77 CJK UNIFIED IDEOGRAPH 6B77
 "\x{F98C}" => "\x{6B77}",

 # F98D CJK UNIFIED IDEOGRAPH F98D
 # ->   8F62 CJK UNIFIED IDEOGRAPH 8F62
 "\x{F98D}" => "\x{8F62}",

 # F98E CJK UNIFIED IDEOGRAPH F98E
 # ->   5E74 CJK UNIFIED IDEOGRAPH 5E74
 "\x{F98E}" => "\x{5E74}",

 # F98F CJK UNIFIED IDEOGRAPH F98F
 # ->   6190 CJK UNIFIED IDEOGRAPH 6190
 "\x{F98F}" => "\x{6190}",

 # F990 CJK UNIFIED IDEOGRAPH F990
 # ->   6200 CJK UNIFIED IDEOGRAPH 6200
 "\x{F990}" => "\x{6200}",

 # F991 CJK UNIFIED IDEOGRAPH F991
 # ->   649A CJK UNIFIED IDEOGRAPH 649A
 "\x{F991}" => "\x{649A}",

 # F992 CJK UNIFIED IDEOGRAPH F992
 # ->   6F23 CJK UNIFIED IDEOGRAPH 6F23
 "\x{F992}" => "\x{6F23}",

 # F993 CJK UNIFIED IDEOGRAPH F993
 # ->   7149 CJK UNIFIED IDEOGRAPH 7149
 "\x{F993}" => "\x{7149}",

 # F994 CJK UNIFIED IDEOGRAPH F994
 # ->   7489 CJK UNIFIED IDEOGRAPH 7489
 "\x{F994}" => "\x{7489}",

 # F995 CJK UNIFIED IDEOGRAPH F995
 # ->   79CA CJK UNIFIED IDEOGRAPH 79CA
 "\x{F995}" => "\x{79CA}",

 # F996 CJK UNIFIED IDEOGRAPH F996
 # ->   7DF4 CJK UNIFIED IDEOGRAPH 7DF4
 "\x{F996}" => "\x{7DF4}",

 # F997 CJK UNIFIED IDEOGRAPH F997
 # ->   806F CJK UNIFIED IDEOGRAPH 806F
 "\x{F997}" => "\x{806F}",

 # F998 CJK UNIFIED IDEOGRAPH F998
 # ->   8F26 CJK UNIFIED IDEOGRAPH 8F26
 "\x{F998}" => "\x{8F26}",

 # F999 CJK UNIFIED IDEOGRAPH F999
 # ->   84EE CJK UNIFIED IDEOGRAPH 84EE
 "\x{F999}" => "\x{84EE}",

 # F99A CJK UNIFIED IDEOGRAPH F99A
 # ->   9023 CJK UNIFIED IDEOGRAPH 9023
 "\x{F99A}" => "\x{9023}",

 # F99B CJK UNIFIED IDEOGRAPH F99B
 # ->   934A CJK UNIFIED IDEOGRAPH 934A
 "\x{F99B}" => "\x{934A}",

 # F99C CJK UNIFIED IDEOGRAPH F99C
 # ->   5217 CJK UNIFIED IDEOGRAPH 5217
 "\x{F99C}" => "\x{5217}",

 # F99D CJK UNIFIED IDEOGRAPH F99D
 # ->   52A3 CJK UNIFIED IDEOGRAPH 52A3
 "\x{F99D}" => "\x{52A3}",

 # F99E CJK UNIFIED IDEOGRAPH F99E
 # ->   54BD CJK UNIFIED IDEOGRAPH 54BD
 "\x{F99E}" => "\x{54BD}",

 # F99F CJK UNIFIED IDEOGRAPH F99F
 # ->   70C8 CJK UNIFIED IDEOGRAPH 70C8
 "\x{F99F}" => "\x{70C8}",

 # F9A0 CJK UNIFIED IDEOGRAPH F9A0
 # ->   88C2 CJK UNIFIED IDEOGRAPH 88C2
 "\x{F9A0}" => "\x{88C2}",

 # F9A1 CJK UNIFIED IDEOGRAPH F9A1
 # ->   8AAA CJK UNIFIED IDEOGRAPH 8AAA
 "\x{F9A1}" => "\x{8AAA}",

 # F9A2 CJK UNIFIED IDEOGRAPH F9A2
 # ->   5EC9 CJK UNIFIED IDEOGRAPH 5EC9
 "\x{F9A2}" => "\x{5EC9}",

 # F9A3 CJK UNIFIED IDEOGRAPH F9A3
 # ->   5FF5 CJK UNIFIED IDEOGRAPH 5FF5
 "\x{F9A3}" => "\x{5FF5}",

 # F9A4 CJK UNIFIED IDEOGRAPH F9A4
 # ->   637B CJK UNIFIED IDEOGRAPH 637B
 "\x{F9A4}" => "\x{637B}",

 # F9A5 CJK UNIFIED IDEOGRAPH F9A5
 # ->   6BAE CJK UNIFIED IDEOGRAPH 6BAE
 "\x{F9A5}" => "\x{6BAE}",

 # F9A6 CJK UNIFIED IDEOGRAPH F9A6
 # ->   7C3E CJK UNIFIED IDEOGRAPH 7C3E
 "\x{F9A6}" => "\x{7C3E}",

 # F9A7 CJK UNIFIED IDEOGRAPH F9A7
 # ->   7375 CJK UNIFIED IDEOGRAPH 7375
 "\x{F9A7}" => "\x{7375}",

 # F9A8 CJK UNIFIED IDEOGRAPH F9A8
 # ->   4EE4 CJK UNIFIED IDEOGRAPH 4EE4
 "\x{F9A8}" => "\x{4EE4}",

 # F9A9 CJK UNIFIED IDEOGRAPH F9A9
 # ->   56F9 CJK UNIFIED IDEOGRAPH 56F9
 "\x{F9A9}" => "\x{56F9}",

 # F9AA CJK UNIFIED IDEOGRAPH F9AA
 # ->   5BE7 CJK UNIFIED IDEOGRAPH 5BE7
 "\x{F9AA}" => "\x{5BE7}",

 # F9AB CJK UNIFIED IDEOGRAPH F9AB
 # ->   5DBA CJK UNIFIED IDEOGRAPH 5DBA
 "\x{F9AB}" => "\x{5DBA}",

 # F9AC CJK UNIFIED IDEOGRAPH F9AC
 # ->   601C CJK UNIFIED IDEOGRAPH 601C
 "\x{F9AC}" => "\x{601C}",

 # F9AD CJK UNIFIED IDEOGRAPH F9AD
 # ->   73B2 CJK UNIFIED IDEOGRAPH 73B2
 "\x{F9AD}" => "\x{73B2}",

 # F9AE CJK UNIFIED IDEOGRAPH F9AE
 # ->   7469 CJK UNIFIED IDEOGRAPH 7469
 "\x{F9AE}" => "\x{7469}",

 # F9AF CJK UNIFIED IDEOGRAPH F9AF
 # ->   7F9A CJK UNIFIED IDEOGRAPH 7F9A
 "\x{F9AF}" => "\x{7F9A}",

 # F9B0 CJK UNIFIED IDEOGRAPH F9B0
 # ->   8046 CJK UNIFIED IDEOGRAPH 8046
 "\x{F9B0}" => "\x{8046}",

 # F9B1 CJK UNIFIED IDEOGRAPH F9B1
 # ->   9234 CJK UNIFIED IDEOGRAPH 9234
 "\x{F9B1}" => "\x{9234}",

 # F9B2 CJK UNIFIED IDEOGRAPH F9B2
 # ->   96F6 CJK UNIFIED IDEOGRAPH 96F6
 "\x{F9B2}" => "\x{96F6}",

 # F9B3 CJK UNIFIED IDEOGRAPH F9B3
 # ->   9748 CJK UNIFIED IDEOGRAPH 9748
 "\x{F9B3}" => "\x{9748}",

 # F9B4 CJK UNIFIED IDEOGRAPH F9B4
 # ->   9818 CJK UNIFIED IDEOGRAPH 9818
 "\x{F9B4}" => "\x{9818}",

 # F9B5 CJK UNIFIED IDEOGRAPH F9B5
 # ->   4F8B CJK UNIFIED IDEOGRAPH 4F8B
 "\x{F9B5}" => "\x{4F8B}",

 # F9B6 CJK UNIFIED IDEOGRAPH F9B6
 # ->   79AE CJK UNIFIED IDEOGRAPH 79AE
 "\x{F9B6}" => "\x{79AE}",

 # F9B7 CJK UNIFIED IDEOGRAPH F9B7
 # ->   91B4 CJK UNIFIED IDEOGRAPH 91B4
 "\x{F9B7}" => "\x{91B4}",

 # F9B8 CJK UNIFIED IDEOGRAPH F9B8
 # ->   96B8 CJK UNIFIED IDEOGRAPH 96B8
 "\x{F9B8}" => "\x{96B8}",

 # F9B9 CJK UNIFIED IDEOGRAPH F9B9
 # ->   60E1 CJK UNIFIED IDEOGRAPH 60E1
 "\x{F9B9}" => "\x{60E1}",

 # F9BA CJK UNIFIED IDEOGRAPH F9BA
 # ->   4E86 CJK UNIFIED IDEOGRAPH 4E86
 "\x{F9BA}" => "\x{4E86}",

 # F9BB CJK UNIFIED IDEOGRAPH F9BB
 # ->   50DA CJK UNIFIED IDEOGRAPH 50DA
 "\x{F9BB}" => "\x{50DA}",

 # F9BC CJK UNIFIED IDEOGRAPH F9BC
 # ->   5BEE CJK UNIFIED IDEOGRAPH 5BEE
 "\x{F9BC}" => "\x{5BEE}",

 # F9BD CJK UNIFIED IDEOGRAPH F9BD
 # ->   5C3F CJK UNIFIED IDEOGRAPH 5C3F
 "\x{F9BD}" => "\x{5C3F}",

 # F9BE CJK UNIFIED IDEOGRAPH F9BE
 # ->   6599 CJK UNIFIED IDEOGRAPH 6599
 "\x{F9BE}" => "\x{6599}",

 # F9BF CJK UNIFIED IDEOGRAPH F9BF
 # ->   6A02 CJK UNIFIED IDEOGRAPH 6A02
 "\x{F9BF}" => "\x{6A02}",

 # F9C0 CJK UNIFIED IDEOGRAPH F9C0
 # ->   71CE CJK UNIFIED IDEOGRAPH 71CE
 "\x{F9C0}" => "\x{71CE}",

 # F9C1 CJK UNIFIED IDEOGRAPH F9C1
 # ->   7642 CJK UNIFIED IDEOGRAPH 7642
 "\x{F9C1}" => "\x{7642}",

 # F9C2 CJK UNIFIED IDEOGRAPH F9C2
 # ->   84FC CJK UNIFIED IDEOGRAPH 84FC
 "\x{F9C2}" => "\x{84FC}",

 # F9C3 CJK UNIFIED IDEOGRAPH F9C3
 # ->   907C CJK UNIFIED IDEOGRAPH 907C
 "\x{F9C3}" => "\x{907C}",

 # F9C4 CJK UNIFIED IDEOGRAPH F9C4
 # ->   9F8D CJK UNIFIED IDEOGRAPH 9F8D
 "\x{F9C4}" => "\x{9F8D}",

 # F9C5 CJK UNIFIED IDEOGRAPH F9C5
 # ->   6688 CJK UNIFIED IDEOGRAPH 6688
 "\x{F9C5}" => "\x{6688}",

 # F9C6 CJK UNIFIED IDEOGRAPH F9C6
 # ->   962E CJK UNIFIED IDEOGRAPH 962E
 "\x{F9C6}" => "\x{962E}",

 # F9C7 CJK UNIFIED IDEOGRAPH F9C7
 # ->   5289 CJK UNIFIED IDEOGRAPH 5289
 "\x{F9C7}" => "\x{5289}",

 # F9C8 CJK UNIFIED IDEOGRAPH F9C8
 # ->   677B CJK UNIFIED IDEOGRAPH 677B
 "\x{F9C8}" => "\x{677B}",

 # F9C9 CJK UNIFIED IDEOGRAPH F9C9
 # ->   67F3 CJK UNIFIED IDEOGRAPH 67F3
 "\x{F9C9}" => "\x{67F3}",

 # F9CA CJK UNIFIED IDEOGRAPH F9CA
 # ->   6D41 CJK UNIFIED IDEOGRAPH 6D41
 "\x{F9CA}" => "\x{6D41}",

 # F9CB CJK UNIFIED IDEOGRAPH F9CB
 # ->   6E9C CJK UNIFIED IDEOGRAPH 6E9C
 "\x{F9CB}" => "\x{6E9C}",

 # F9CC CJK UNIFIED IDEOGRAPH F9CC
 # ->   7409 CJK UNIFIED IDEOGRAPH 7409
 "\x{F9CC}" => "\x{7409}",

 # F9CD CJK UNIFIED IDEOGRAPH F9CD
 # ->   7559 CJK UNIFIED IDEOGRAPH 7559
 "\x{F9CD}" => "\x{7559}",

 # F9CE CJK UNIFIED IDEOGRAPH F9CE
 # ->   786B CJK UNIFIED IDEOGRAPH 786B
 "\x{F9CE}" => "\x{786B}",

 # F9CF CJK UNIFIED IDEOGRAPH F9CF
 # ->   7D10 CJK UNIFIED IDEOGRAPH 7D10
 "\x{F9CF}" => "\x{7D10}",

 # F9D0 CJK UNIFIED IDEOGRAPH F9D0
 # ->   985E CJK UNIFIED IDEOGRAPH 985E
 "\x{F9D0}" => "\x{985E}",

 # F9D1 CJK UNIFIED IDEOGRAPH F9D1
 # ->   516D CJK UNIFIED IDEOGRAPH 516D
 "\x{F9D1}" => "\x{516D}",

 # F9D2 CJK UNIFIED IDEOGRAPH F9D2
 # ->   622E CJK UNIFIED IDEOGRAPH 622E
 "\x{F9D2}" => "\x{622E}",

 # F9D3 CJK UNIFIED IDEOGRAPH F9D3
 # ->   9678 CJK UNIFIED IDEOGRAPH 9678
 "\x{F9D3}" => "\x{9678}",

 # F9D4 CJK UNIFIED IDEOGRAPH F9D4
 # ->   502B CJK UNIFIED IDEOGRAPH 502B
 "\x{F9D4}" => "\x{502B}",

 # F9D5 CJK UNIFIED IDEOGRAPH F9D5
 # ->   5D19 CJK UNIFIED IDEOGRAPH 5D19
 "\x{F9D5}" => "\x{5D19}",

 # F9D6 CJK UNIFIED IDEOGRAPH F9D6
 # ->   6DEA CJK UNIFIED IDEOGRAPH 6DEA
 "\x{F9D6}" => "\x{6DEA}",

 # F9D7 CJK UNIFIED IDEOGRAPH F9D7
 # ->   8F2A CJK UNIFIED IDEOGRAPH 8F2A
 "\x{F9D7}" => "\x{8F2A}",

 # F9D8 CJK UNIFIED IDEOGRAPH F9D8
 # ->   5F8B CJK UNIFIED IDEOGRAPH 5F8B
 "\x{F9D8}" => "\x{5F8B}",

 # F9D9 CJK UNIFIED IDEOGRAPH F9D9
 # ->   6144 CJK UNIFIED IDEOGRAPH 6144
 "\x{F9D9}" => "\x{6144}",

 # F9DA CJK UNIFIED IDEOGRAPH F9DA
 # ->   6817 CJK UNIFIED IDEOGRAPH 6817
 "\x{F9DA}" => "\x{6817}",

 # F9DB CJK UNIFIED IDEOGRAPH F9DB
 # ->   7387 CJK UNIFIED IDEOGRAPH 7387
 "\x{F9DB}" => "\x{7387}",

 # F9DC CJK UNIFIED IDEOGRAPH F9DC
 # ->   9686 CJK UNIFIED IDEOGRAPH 9686
 "\x{F9DC}" => "\x{9686}",

 # F9DD CJK UNIFIED IDEOGRAPH F9DD
 # ->   5229 CJK UNIFIED IDEOGRAPH 5229
 "\x{F9DD}" => "\x{5229}",

 # F9DE CJK UNIFIED IDEOGRAPH F9DE
 # ->   540F CJK UNIFIED IDEOGRAPH 540F
 "\x{F9DE}" => "\x{540F}",

 # F9DF CJK UNIFIED IDEOGRAPH F9DF
 # ->   5C65 CJK UNIFIED IDEOGRAPH 5C65
 "\x{F9DF}" => "\x{5C65}",

 # F9E0 CJK UNIFIED IDEOGRAPH F9E0
 # ->   6613 CJK UNIFIED IDEOGRAPH 6613
 "\x{F9E0}" => "\x{6613}",

 # F9E1 CJK UNIFIED IDEOGRAPH F9E1
 # ->   674E CJK UNIFIED IDEOGRAPH 674E
 "\x{F9E1}" => "\x{674E}",

 # F9E2 CJK UNIFIED IDEOGRAPH F9E2
 # ->   68A8 CJK UNIFIED IDEOGRAPH 68A8
 "\x{F9E2}" => "\x{68A8}",

 # F9E3 CJK UNIFIED IDEOGRAPH F9E3
 # ->   6CE5 CJK UNIFIED IDEOGRAPH 6CE5
 "\x{F9E3}" => "\x{6CE5}",

 # F9E4 CJK UNIFIED IDEOGRAPH F9E4
 # ->   7406 CJK UNIFIED IDEOGRAPH 7406
 "\x{F9E4}" => "\x{7406}",

 # F9E5 CJK UNIFIED IDEOGRAPH F9E5
 # ->   75E2 CJK UNIFIED IDEOGRAPH 75E2
 "\x{F9E5}" => "\x{75E2}",

 # F9E6 CJK UNIFIED IDEOGRAPH F9E6
 # ->   7F79 CJK UNIFIED IDEOGRAPH 7F79
 "\x{F9E6}" => "\x{7F79}",

 # F9E7 CJK UNIFIED IDEOGRAPH F9E7
 # ->   88CF CJK UNIFIED IDEOGRAPH 88CF
 "\x{F9E7}" => "\x{88CF}",

 # F9E8 CJK UNIFIED IDEOGRAPH F9E8
 # ->   88E1 CJK UNIFIED IDEOGRAPH 88E1
 "\x{F9E8}" => "\x{88E1}",

 # F9E9 CJK UNIFIED IDEOGRAPH F9E9
 # ->   91CC CJK UNIFIED IDEOGRAPH 91CC
 "\x{F9E9}" => "\x{91CC}",

 # F9EA CJK UNIFIED IDEOGRAPH F9EA
 # ->   96E2 CJK UNIFIED IDEOGRAPH 96E2
 "\x{F9EA}" => "\x{96E2}",

 # F9EB CJK UNIFIED IDEOGRAPH F9EB
 # ->   533F CJK UNIFIED IDEOGRAPH 533F
 "\x{F9EB}" => "\x{533F}",

 # F9EC CJK UNIFIED IDEOGRAPH F9EC
 # ->   6EBA CJK UNIFIED IDEOGRAPH 6EBA
 "\x{F9EC}" => "\x{6EBA}",

 # F9ED CJK UNIFIED IDEOGRAPH F9ED
 # ->   541D CJK UNIFIED IDEOGRAPH 541D
 "\x{F9ED}" => "\x{541D}",

 # F9EE CJK UNIFIED IDEOGRAPH F9EE
 # ->   71D0 CJK UNIFIED IDEOGRAPH 71D0
 "\x{F9EE}" => "\x{71D0}",

 # F9EF CJK UNIFIED IDEOGRAPH F9EF
 # ->   7498 CJK UNIFIED IDEOGRAPH 7498
 "\x{F9EF}" => "\x{7498}",

 # F9F0 CJK UNIFIED IDEOGRAPH F9F0
 # ->   85FA CJK UNIFIED IDEOGRAPH 85FA
 "\x{F9F0}" => "\x{85FA}",

 # F9F1 CJK UNIFIED IDEOGRAPH F9F1
 # ->   96A3 CJK UNIFIED IDEOGRAPH 96A3
 "\x{F9F1}" => "\x{96A3}",

 # F9F2 CJK UNIFIED IDEOGRAPH F9F2
 # ->   9C57 CJK UNIFIED IDEOGRAPH 9C57
 "\x{F9F2}" => "\x{9C57}",

 # F9F3 CJK UNIFIED IDEOGRAPH F9F3
 # ->   9E9F CJK UNIFIED IDEOGRAPH 9E9F
 "\x{F9F3}" => "\x{9E9F}",

 # F9F4 CJK UNIFIED IDEOGRAPH F9F4
 # ->   6797 CJK UNIFIED IDEOGRAPH 6797
 "\x{F9F4}" => "\x{6797}",

 # F9F5 CJK UNIFIED IDEOGRAPH F9F5
 # ->   6DCB CJK UNIFIED IDEOGRAPH 6DCB
 "\x{F9F5}" => "\x{6DCB}",

 # F9F6 CJK UNIFIED IDEOGRAPH F9F6
 # ->   81E8 CJK UNIFIED IDEOGRAPH 81E8
 "\x{F9F6}" => "\x{81E8}",

 # F9F7 CJK UNIFIED IDEOGRAPH F9F7
 # ->   7ACB CJK UNIFIED IDEOGRAPH 7ACB
 "\x{F9F7}" => "\x{7ACB}",

 # F9F8 CJK UNIFIED IDEOGRAPH F9F8
 # ->   7B20 CJK UNIFIED IDEOGRAPH 7B20
 "\x{F9F8}" => "\x{7B20}",

 # F9F9 CJK UNIFIED IDEOGRAPH F9F9
 # ->   7C92 CJK UNIFIED IDEOGRAPH 7C92
 "\x{F9F9}" => "\x{7C92}",

 # F9FA CJK UNIFIED IDEOGRAPH F9FA
 # ->   72C0 CJK UNIFIED IDEOGRAPH 72C0
 "\x{F9FA}" => "\x{72C0}",

 # F9FB CJK UNIFIED IDEOGRAPH F9FB
 # ->   7099 CJK UNIFIED IDEOGRAPH 7099
 "\x{F9FB}" => "\x{7099}",

 # F9FC CJK UNIFIED IDEOGRAPH F9FC
 # ->   8B58 CJK UNIFIED IDEOGRAPH 8B58
 "\x{F9FC}" => "\x{8B58}",

 # F9FD CJK UNIFIED IDEOGRAPH F9FD
 # ->   4EC0 CJK UNIFIED IDEOGRAPH 4EC0
 "\x{F9FD}" => "\x{4EC0}",

 # F9FE CJK UNIFIED IDEOGRAPH F9FE
 # ->   8336 CJK UNIFIED IDEOGRAPH 8336
 "\x{F9FE}" => "\x{8336}",

 # F9FF CJK UNIFIED IDEOGRAPH F9FF
 # ->   523A CJK UNIFIED IDEOGRAPH 523A
 "\x{F9FF}" => "\x{523A}",

 # FA00 CJK UNIFIED IDEOGRAPH FA00
 # ->   5207 CJK UNIFIED IDEOGRAPH 5207
 "\x{FA00}" => "\x{5207}",

 # FA01 CJK UNIFIED IDEOGRAPH FA01
 # ->   5EA6 CJK UNIFIED IDEOGRAPH 5EA6
 "\x{FA01}" => "\x{5EA6}",

 # FA02 CJK UNIFIED IDEOGRAPH FA02
 # ->   62D3 CJK UNIFIED IDEOGRAPH 62D3
 "\x{FA02}" => "\x{62D3}",

 # FA03 CJK UNIFIED IDEOGRAPH FA03
 # ->   7CD6 CJK UNIFIED IDEOGRAPH 7CD6
 "\x{FA03}" => "\x{7CD6}",

 # FA04 CJK UNIFIED IDEOGRAPH FA04
 # ->   5B85 CJK UNIFIED IDEOGRAPH 5B85
 "\x{FA04}" => "\x{5B85}",

 # FA05 CJK UNIFIED IDEOGRAPH FA05
 # ->   6D1E CJK UNIFIED IDEOGRAPH 6D1E
 "\x{FA05}" => "\x{6D1E}",

 # FA06 CJK UNIFIED IDEOGRAPH FA06
 # ->   66B4 CJK UNIFIED IDEOGRAPH 66B4
 "\x{FA06}" => "\x{66B4}",

 # FA07 CJK UNIFIED IDEOGRAPH FA07
 # ->   8F3B CJK UNIFIED IDEOGRAPH 8F3B
 "\x{FA07}" => "\x{8F3B}",

 # FA08 CJK UNIFIED IDEOGRAPH FA08
 # ->   884C CJK UNIFIED IDEOGRAPH 884C
 "\x{FA08}" => "\x{884C}",

 # FA09 CJK UNIFIED IDEOGRAPH FA09
 # ->   964D CJK UNIFIED IDEOGRAPH 964D
 "\x{FA09}" => "\x{964D}",

 # FA0A CJK UNIFIED IDEOGRAPH FA0A
 # ->   898B CJK UNIFIED IDEOGRAPH 898B
 "\x{FA0A}" => "\x{898B}",

 # FA0B CJK UNIFIED IDEOGRAPH FA0B
 # ->   5ED3 CJK UNIFIED IDEOGRAPH 5ED3
 "\x{FA0B}" => "\x{5ED3}",

 # FA0C CJK UNIFIED IDEOGRAPH FA0C
 # ->   5140 CJK UNIFIED IDEOGRAPH 5140
 "\x{FA0C}" => "\x{5140}",

 # FA0D CJK UNIFIED IDEOGRAPH FA0D
 # ->   55C0 CJK UNIFIED IDEOGRAPH 55C0
 "\x{FA0D}" => "\x{55C0}",

 # FA10 CJK UNIFIED IDEOGRAPH FA10
 # ->   585A CJK UNIFIED IDEOGRAPH 585A
 "\x{FA10}" => "\x{585A}",

 # FA12 CJK UNIFIED IDEOGRAPH FA12
 # ->   6674 CJK UNIFIED IDEOGRAPH 6674
 "\x{FA12}" => "\x{6674}",

 # FA15 CJK UNIFIED IDEOGRAPH FA15
 # ->   51DE CJK UNIFIED IDEOGRAPH 51DE
 "\x{FA15}" => "\x{51DE}",

 # FA16 CJK UNIFIED IDEOGRAPH FA16
 # ->   732A CJK UNIFIED IDEOGRAPH 732A
 "\x{FA16}" => "\x{732A}",

 # FA17 CJK UNIFIED IDEOGRAPH FA17
 # ->   76CA CJK UNIFIED IDEOGRAPH 76CA
 "\x{FA17}" => "\x{76CA}",

 # FA18 CJK UNIFIED IDEOGRAPH FA18
 # ->   793C CJK UNIFIED IDEOGRAPH 793C
 "\x{FA18}" => "\x{793C}",

 # FA19 CJK UNIFIED IDEOGRAPH FA19
 # ->   795E CJK UNIFIED IDEOGRAPH 795E
 "\x{FA19}" => "\x{795E}",

 # FA1A CJK UNIFIED IDEOGRAPH FA1A
 # ->   7965 CJK UNIFIED IDEOGRAPH 7965
 "\x{FA1A}" => "\x{7965}",

 # FA1B CJK UNIFIED IDEOGRAPH FA1B
 # ->   798F CJK UNIFIED IDEOGRAPH 798F
 "\x{FA1B}" => "\x{798F}",

 # FA1C CJK UNIFIED IDEOGRAPH FA1C
 # ->   9756 CJK UNIFIED IDEOGRAPH 9756
 "\x{FA1C}" => "\x{9756}",

 # FA1D CJK UNIFIED IDEOGRAPH FA1D
 # ->   7CBE CJK UNIFIED IDEOGRAPH 7CBE
 "\x{FA1D}" => "\x{7CBE}",

 # FA1E CJK UNIFIED IDEOGRAPH FA1E
 # ->   7FBD CJK UNIFIED IDEOGRAPH 7FBD
 "\x{FA1E}" => "\x{7FBD}",

 # FA20 CJK UNIFIED IDEOGRAPH FA20
 # ->   8612 CJK UNIFIED IDEOGRAPH 8612
 "\x{FA20}" => "\x{8612}",

 # FA22 CJK UNIFIED IDEOGRAPH FA22
 # ->   8AF8 CJK UNIFIED IDEOGRAPH 8AF8
 "\x{FA22}" => "\x{8AF8}",

 # FA25 CJK UNIFIED IDEOGRAPH FA25
 # ->   9038 CJK UNIFIED IDEOGRAPH 9038
 "\x{FA25}" => "\x{9038}",

 # FA26 CJK UNIFIED IDEOGRAPH FA26
 # ->   90FD CJK UNIFIED IDEOGRAPH 90FD
 "\x{FA26}" => "\x{90FD}",

 # FA2A CJK UNIFIED IDEOGRAPH FA2A
 # ->   98EF CJK UNIFIED IDEOGRAPH 98EF
 "\x{FA2A}" => "\x{98EF}",

 # FA2B CJK UNIFIED IDEOGRAPH FA2B
 # ->   98FC CJK UNIFIED IDEOGRAPH 98FC
 "\x{FA2B}" => "\x{98FC}",

 # FA2C CJK UNIFIED IDEOGRAPH FA2C
 # ->   9928 CJK UNIFIED IDEOGRAPH 9928
 "\x{FA2C}" => "\x{9928}",

 # FA2D CJK UNIFIED IDEOGRAPH FA2D
 # ->   9DB4 CJK UNIFIED IDEOGRAPH 9DB4
 "\x{FA2D}" => "\x{9DB4}",

 # FA30 CJK UNIFIED IDEOGRAPH FA30
 # ->   4FAE CJK UNIFIED IDEOGRAPH 4FAE
 "\x{FA30}" => "\x{4FAE}",

 # FA31 CJK UNIFIED IDEOGRAPH FA31
 # ->   50E7 CJK UNIFIED IDEOGRAPH 50E7
 "\x{FA31}" => "\x{50E7}",

 # FA32 CJK UNIFIED IDEOGRAPH FA32
 # ->   514D CJK UNIFIED IDEOGRAPH 514D
 "\x{FA32}" => "\x{514D}",

 # FA33 CJK UNIFIED IDEOGRAPH FA33
 # ->   52C9 CJK UNIFIED IDEOGRAPH 52C9
 "\x{FA33}" => "\x{52C9}",

 # FA34 CJK UNIFIED IDEOGRAPH FA34
 # ->   52E4 CJK UNIFIED IDEOGRAPH 52E4
 "\x{FA34}" => "\x{52E4}",

 # FA35 CJK UNIFIED IDEOGRAPH FA35
 # ->   5351 CJK UNIFIED IDEOGRAPH 5351
 "\x{FA35}" => "\x{5351}",

 # FA36 CJK UNIFIED IDEOGRAPH FA36
 # ->   559D CJK UNIFIED IDEOGRAPH 559D
 "\x{FA36}" => "\x{559D}",

 # FA37 CJK UNIFIED IDEOGRAPH FA37
 # ->   5606 CJK UNIFIED IDEOGRAPH 5606
 "\x{FA37}" => "\x{5606}",

 # FA38 CJK UNIFIED IDEOGRAPH FA38
 # ->   5668 CJK UNIFIED IDEOGRAPH 5668
 "\x{FA38}" => "\x{5668}",

 # FA39 CJK UNIFIED IDEOGRAPH FA39
 # ->   5840 CJK UNIFIED IDEOGRAPH 5840
 "\x{FA39}" => "\x{5840}",

 # FA3A CJK UNIFIED IDEOGRAPH FA3A
 # ->   58A8 CJK UNIFIED IDEOGRAPH 58A8
 "\x{FA3A}" => "\x{58A8}",

 # FA3B CJK UNIFIED IDEOGRAPH FA3B
 # ->   5C64 CJK UNIFIED IDEOGRAPH 5C64
 "\x{FA3B}" => "\x{5C64}",

 # FA3C CJK UNIFIED IDEOGRAPH FA3C
 # ->   5C6E CJK UNIFIED IDEOGRAPH 5C6E
 "\x{FA3C}" => "\x{5C6E}",

 # FA3D CJK UNIFIED IDEOGRAPH FA3D
 # ->   6094 CJK UNIFIED IDEOGRAPH 6094
 "\x{FA3D}" => "\x{6094}",

 # FA3E CJK UNIFIED IDEOGRAPH FA3E
 # ->   6168 CJK UNIFIED IDEOGRAPH 6168
 "\x{FA3E}" => "\x{6168}",

 # FA3F CJK UNIFIED IDEOGRAPH FA3F
 # ->   618E CJK UNIFIED IDEOGRAPH 618E
 "\x{FA3F}" => "\x{618E}",

 # FA40 CJK UNIFIED IDEOGRAPH FA40
 # ->   61F2 CJK UNIFIED IDEOGRAPH 61F2
 "\x{FA40}" => "\x{61F2}",

 # FA41 CJK UNIFIED IDEOGRAPH FA41
 # ->   654F CJK UNIFIED IDEOGRAPH 654F
 "\x{FA41}" => "\x{654F}",

 # FA42 CJK UNIFIED IDEOGRAPH FA42
 # ->   65E2 CJK UNIFIED IDEOGRAPH 65E2
 "\x{FA42}" => "\x{65E2}",

 # FA43 CJK UNIFIED IDEOGRAPH FA43
 # ->   6691 CJK UNIFIED IDEOGRAPH 6691
 "\x{FA43}" => "\x{6691}",

 # FA44 CJK UNIFIED IDEOGRAPH FA44
 # ->   6885 CJK UNIFIED IDEOGRAPH 6885
 "\x{FA44}" => "\x{6885}",

 # FA45 CJK UNIFIED IDEOGRAPH FA45
 # ->   6D77 CJK UNIFIED IDEOGRAPH 6D77
 "\x{FA45}" => "\x{6D77}",

 # FA46 CJK UNIFIED IDEOGRAPH FA46
 # ->   6E1A CJK UNIFIED IDEOGRAPH 6E1A
 "\x{FA46}" => "\x{6E1A}",

 # FA47 CJK UNIFIED IDEOGRAPH FA47
 # ->   6F22 CJK UNIFIED IDEOGRAPH 6F22
 "\x{FA47}" => "\x{6F22}",

 # FA48 CJK UNIFIED IDEOGRAPH FA48
 # ->   716E CJK UNIFIED IDEOGRAPH 716E
 "\x{FA48}" => "\x{716E}",

 # FA49 CJK UNIFIED IDEOGRAPH FA49
 # ->   722B CJK UNIFIED IDEOGRAPH 722B
 "\x{FA49}" => "\x{722B}",

 # FA4A CJK UNIFIED IDEOGRAPH FA4A
 # ->   7422 CJK UNIFIED IDEOGRAPH 7422
 "\x{FA4A}" => "\x{7422}",

 # FA4B CJK UNIFIED IDEOGRAPH FA4B
 # ->   7891 CJK UNIFIED IDEOGRAPH 7891
 "\x{FA4B}" => "\x{7891}",

 # FA4C CJK UNIFIED IDEOGRAPH FA4C
 # ->   793E CJK UNIFIED IDEOGRAPH 793E
 "\x{FA4C}" => "\x{793E}",

 # FA4D CJK UNIFIED IDEOGRAPH FA4D
 # ->   7949 CJK UNIFIED IDEOGRAPH 7949
 "\x{FA4D}" => "\x{7949}",

 # FA4E CJK UNIFIED IDEOGRAPH FA4E
 # ->   7948 CJK UNIFIED IDEOGRAPH 7948
 "\x{FA4E}" => "\x{7948}",

 # FA4F CJK UNIFIED IDEOGRAPH FA4F
 # ->   7950 CJK UNIFIED IDEOGRAPH 7950
 "\x{FA4F}" => "\x{7950}",

 # FA50 CJK UNIFIED IDEOGRAPH FA50
 # ->   7956 CJK UNIFIED IDEOGRAPH 7956
 "\x{FA50}" => "\x{7956}",

 # FA51 CJK UNIFIED IDEOGRAPH FA51
 # ->   795D CJK UNIFIED IDEOGRAPH 795D
 "\x{FA51}" => "\x{795D}",

 # FA52 CJK UNIFIED IDEOGRAPH FA52
 # ->   798D CJK UNIFIED IDEOGRAPH 798D
 "\x{FA52}" => "\x{798D}",

 # FA53 CJK UNIFIED IDEOGRAPH FA53
 # ->   798E CJK UNIFIED IDEOGRAPH 798E
 "\x{FA53}" => "\x{798E}",

 # FA54 CJK UNIFIED IDEOGRAPH FA54
 # ->   7A40 CJK UNIFIED IDEOGRAPH 7A40
 "\x{FA54}" => "\x{7A40}",

 # FA55 CJK UNIFIED IDEOGRAPH FA55
 # ->   7A81 CJK UNIFIED IDEOGRAPH 7A81
 "\x{FA55}" => "\x{7A81}",

 # FA56 CJK UNIFIED IDEOGRAPH FA56
 # ->   7BC0 CJK UNIFIED IDEOGRAPH 7BC0
 "\x{FA56}" => "\x{7BC0}",

 # FA57 CJK UNIFIED IDEOGRAPH FA57
 # ->   7DF4 CJK UNIFIED IDEOGRAPH 7DF4
 "\x{FA57}" => "\x{7DF4}",

 # FA58 CJK UNIFIED IDEOGRAPH FA58
 # ->   7E09 CJK UNIFIED IDEOGRAPH 7E09
 "\x{FA58}" => "\x{7E09}",

 # FA59 CJK UNIFIED IDEOGRAPH FA59
 # ->   7E41 CJK UNIFIED IDEOGRAPH 7E41
 "\x{FA59}" => "\x{7E41}",

 # FA5A CJK UNIFIED IDEOGRAPH FA5A
 # ->   7F72 CJK UNIFIED IDEOGRAPH 7F72
 "\x{FA5A}" => "\x{7F72}",

 # FA5B CJK UNIFIED IDEOGRAPH FA5B
 # ->   8005 CJK UNIFIED IDEOGRAPH 8005
 "\x{FA5B}" => "\x{8005}",

 # FA5C CJK UNIFIED IDEOGRAPH FA5C
 # ->   81ED CJK UNIFIED IDEOGRAPH 81ED
 "\x{FA5C}" => "\x{81ED}",

 # FA5D CJK UNIFIED IDEOGRAPH FA5D
 # ->   8279 CJK UNIFIED IDEOGRAPH 8279
 "\x{FA5D}" => "\x{8279}",

 # FA5E CJK UNIFIED IDEOGRAPH FA5E
 # ->   8279 CJK UNIFIED IDEOGRAPH 8279
 "\x{FA5E}" => "\x{8279}",

 # FA5F CJK UNIFIED IDEOGRAPH FA5F
 # ->   8457 CJK UNIFIED IDEOGRAPH 8457
 "\x{FA5F}" => "\x{8457}",

 # FA60 CJK UNIFIED IDEOGRAPH FA60
 # ->   8910 CJK UNIFIED IDEOGRAPH 8910
 "\x{FA60}" => "\x{8910}",

 # FA61 CJK UNIFIED IDEOGRAPH FA61
 # ->   8996 CJK UNIFIED IDEOGRAPH 8996
 "\x{FA61}" => "\x{8996}",

 # FA62 CJK UNIFIED IDEOGRAPH FA62
 # ->   8B01 CJK UNIFIED IDEOGRAPH 8B01
 "\x{FA62}" => "\x{8B01}",

 # FA63 CJK UNIFIED IDEOGRAPH FA63
 # ->   8B39 CJK UNIFIED IDEOGRAPH 8B39
 "\x{FA63}" => "\x{8B39}",

 # FA64 CJK UNIFIED IDEOGRAPH FA64
 # ->   8CD3 CJK UNIFIED IDEOGRAPH 8CD3
 "\x{FA64}" => "\x{8CD3}",

 # FA65 CJK UNIFIED IDEOGRAPH FA65
 # ->   8D08 CJK UNIFIED IDEOGRAPH 8D08
 "\x{FA65}" => "\x{8D08}",

 # FA66 CJK UNIFIED IDEOGRAPH FA66
 # ->   8FB6 CJK UNIFIED IDEOGRAPH 8FB6
 "\x{FA66}" => "\x{8FB6}",

 # FA67 CJK UNIFIED IDEOGRAPH FA67
 # ->   9038 CJK UNIFIED IDEOGRAPH 9038
 "\x{FA67}" => "\x{9038}",

 # FA68 CJK UNIFIED IDEOGRAPH FA68
 # ->   96E3 CJK UNIFIED IDEOGRAPH 96E3
 "\x{FA68}" => "\x{96E3}",

 # FA69 CJK UNIFIED IDEOGRAPH FA69
 # ->   97FF CJK UNIFIED IDEOGRAPH 97FF
 "\x{FA69}" => "\x{97FF}",

 # FA6A CJK UNIFIED IDEOGRAPH FA6A
 # ->   983B CJK UNIFIED IDEOGRAPH 983B
 "\x{FA6A}" => "\x{983B}",

 # FB00 LATIN SMALL LIGATURE FF
 # ->   0066 LATIN SMALL LETTER F
 #    + 0066 LATIN SMALL LETTER F
 "\x{FB00}" => "ff",

 # FB01 LATIN SMALL LIGATURE FI
 # ->   0066 LATIN SMALL LETTER F
 #    + 0069 LATIN SMALL LETTER I
 "\x{FB01}" => "fi",

 # FB02 LATIN SMALL LIGATURE FL
 # ->   0066 LATIN SMALL LETTER F
 #    + 006C LATIN SMALL LETTER L
 "\x{FB02}" => "fl",

 # FB03 LATIN SMALL LIGATURE FFI
 # ->   0066 LATIN SMALL LETTER F
 #    + 0066 LATIN SMALL LETTER F
 #    + 0069 LATIN SMALL LETTER I
 "\x{FB03}" => "ffi",

 # FB04 LATIN SMALL LIGATURE FFL
 # ->   0066 LATIN SMALL LETTER F
 #    + 0066 LATIN SMALL LETTER F
 #    + 006C LATIN SMALL LETTER L
 "\x{FB04}" => "ffl",

 # FB05 LATIN SMALL LIGATURE LONG S T
 # ->   0074 LATIN SMALL LETTER T
 #    + 0073 LATIN SMALL LETTER S
 "\x{FB05}" => "ts",

 # FB06 LATIN SMALL LIGATURE ST
 # ->   0073 LATIN SMALL LETTER S
 #    + 0074 LATIN SMALL LETTER T
 "\x{FB06}" => "st",

 # FB13 ARMENIAN SMALL LIGATURE MEN NOW
 # ->   0574 ARMENIAN SMALL LETTER MEN
 #    + 0576 ARMENIAN SMALL LETTER NOW
 "\x{FB13}" => "\x{0574}\x{0576}",

 # FB14 ARMENIAN SMALL LIGATURE MEN ECH
 # ->   0574 ARMENIAN SMALL LETTER MEN
 #    + 0565 ARMENIAN SMALL LETTER ECH
 "\x{FB14}" => "\x{0574}\x{0565}",

 # FB15 ARMENIAN SMALL LIGATURE MEN INI
 # ->   0574 ARMENIAN SMALL LETTER MEN
 #    + 056B ARMENIAN SMALL LETTER INI
 "\x{FB15}" => "\x{0574}\x{056B}",

 # FB16 ARMENIAN SMALL LIGATURE VEW NOW
 # ->   057E ARMENIAN SMALL LETTER VEW
 #    + 0576 ARMENIAN SMALL LETTER NOW
 "\x{FB16}" => "\x{057E}\x{0576}",

 # FB17 ARMENIAN SMALL LIGATURE MEN XEH
 # ->   0574 ARMENIAN SMALL LETTER MEN
 #    + 056D ARMENIAN SMALL LETTER XEH
 "\x{FB17}" => "\x{0574}\x{056D}",

 # FB1D HEBREW LETTER YOD WITH HIRIQ
 # ->   05D9 HEBREW LETTER YOD
 "\x{FB1D}" => "\x{05D9}",

 # FB1F HEBREW LIGATURE YIDDISH YOD YOD PATAH
 # ->   05F2 HEBREW LIGATURE YIDDISH DOUBLE YOD
 "\x{FB1F}" => "\x{05F2}",

 # FB20 HEBREW LETTER ALTERNATIVE AYIN
 # ->   05E2 HEBREW LETTER AYIN
 "\x{FB20}" => "\x{05E2}",

 # FB21 HEBREW LETTER WIDE ALEF
 # ->   05D0 HEBREW LETTER ALEF
 "\x{FB21}" => "\x{05D0}",

 # FB22 HEBREW LETTER WIDE DALET
 # ->   05D3 HEBREW LETTER DALET
 "\x{FB22}" => "\x{05D3}",

 # FB23 HEBREW LETTER WIDE HE
 # ->   05D4 HEBREW LETTER HE
 "\x{FB23}" => "\x{05D4}",

 # FB24 HEBREW LETTER WIDE KAF
 # ->   05DB HEBREW LETTER KAF
 "\x{FB24}" => "\x{05DB}",

 # FB25 HEBREW LETTER WIDE LAMED
 # ->   05DC HEBREW LETTER LAMED
 "\x{FB25}" => "\x{05DC}",

 # FB26 HEBREW LETTER WIDE FINAL MEM
 # ->   05DD HEBREW LETTER FINAL MEM
 "\x{FB26}" => "\x{05DD}",

 # FB27 HEBREW LETTER WIDE RESH
 # ->   05E8 HEBREW LETTER RESH
 "\x{FB27}" => "\x{05E8}",

 # FB28 HEBREW LETTER WIDE TAV
 # ->   05EA HEBREW LETTER TAV
 "\x{FB28}" => "\x{05EA}",

 # FB29 HEBREW LETTER ALTERNATIVE PLUS SIGN
 # ->   002B PLUS SIGN
 "\x{FB29}" => "+",

 # FB2A HEBREW LETTER SHIN WITH SHIN DOT
 # ->   05E9 HEBREW LETTER SHIN
 "\x{FB2A}" => "\x{05E9}",

 # FB2B HEBREW LETTER SHIN WITH SIN DOT
 # ->   05E9 HEBREW LETTER SHIN
 "\x{FB2B}" => "\x{05E9}",

 # FB2C HEBREW LETTER SHIN WITH DAGESH AND SHIN DOT
 # ->   05E9 HEBREW LETTER SHIN
 "\x{FB2C}" => "\x{05E9}",

 # FB2D HEBREW LETTER SHIN WITH DAGESH AND SIN DOT
 # ->   05E9 HEBREW LETTER SHIN
 "\x{FB2D}" => "\x{05E9}",

 # FB2E HEBREW LETTER ALEF WITH PATAH
 # ->   05D0 HEBREW LETTER ALEF
 "\x{FB2E}" => "\x{05D0}",

 # FB2F HEBREW LETTER ALEF WITH QAMATS
 # ->   05D0 HEBREW LETTER ALEF
 "\x{FB2F}" => "\x{05D0}",

 # FB30 HEBREW LETTER ALEF WITH MAPIQ
 # ->   05D0 HEBREW LETTER ALEF
 "\x{FB30}" => "\x{05D0}",

 # FB31 HEBREW LETTER BET WITH DAGESH
 # ->   05D1 HEBREW LETTER BET
 "\x{FB31}" => "\x{05D1}",

 # FB32 HEBREW LETTER GIMEL WITH DAGESH
 # ->   05D2 HEBREW LETTER GIMEL
 "\x{FB32}" => "\x{05D2}",

 # FB33 HEBREW LETTER DALET WITH DAGESH
 # ->   05D3 HEBREW LETTER DALET
 "\x{FB33}" => "\x{05D3}",

 # FB34 HEBREW LETTER HE WITH MAPIQ
 # ->   05D4 HEBREW LETTER HE
 "\x{FB34}" => "\x{05D4}",

 # FB35 HEBREW LETTER VAV WITH DAGESH
 # ->   05D5 HEBREW LETTER VAV
 "\x{FB35}" => "\x{05D5}",

 # FB36 HEBREW LETTER ZAYIN WITH DAGESH
 # ->   05D6 HEBREW LETTER ZAYIN
 "\x{FB36}" => "\x{05D6}",

 # FB38 HEBREW LETTER TET WITH DAGESH
 # ->   05D8 HEBREW LETTER TET
 "\x{FB38}" => "\x{05D8}",

 # FB39 HEBREW LETTER YOD WITH DAGESH
 # ->   05D9 HEBREW LETTER YOD
 "\x{FB39}" => "\x{05D9}",

 # FB3A HEBREW LETTER FINAL KAF WITH DAGESH
 # ->   05DA HEBREW LETTER FINAL KAF
 "\x{FB3A}" => "\x{05DA}",

 # FB3B HEBREW LETTER KAF WITH DAGESH
 # ->   05DB HEBREW LETTER KAF
 "\x{FB3B}" => "\x{05DB}",

 # FB3C HEBREW LETTER LAMED WITH DAGESH
 # ->   05DC HEBREW LETTER LAMED
 "\x{FB3C}" => "\x{05DC}",

 # FB3E HEBREW LETTER MEM WITH DAGESH
 # ->   05DE HEBREW LETTER MEM
 "\x{FB3E}" => "\x{05DE}",

 # FB40 HEBREW LETTER NUN WITH DAGESH
 # ->   05E0 HEBREW LETTER NUN
 "\x{FB40}" => "\x{05E0}",

 # FB41 HEBREW LETTER SAMEKH WITH DAGESH
 # ->   05E1 HEBREW LETTER SAMEKH
 "\x{FB41}" => "\x{05E1}",

 # FB43 HEBREW LETTER FINAL PE WITH DAGESH
 # ->   05E3 HEBREW LETTER FINAL PE
 "\x{FB43}" => "\x{05E3}",

 # FB44 HEBREW LETTER PE WITH DAGESH
 # ->   05E4 HEBREW LETTER PE
 "\x{FB44}" => "\x{05E4}",

 # FB46 HEBREW LETTER TSADI WITH DAGESH
 # ->   05E6 HEBREW LETTER TSADI
 "\x{FB46}" => "\x{05E6}",

 # FB47 HEBREW LETTER QOF WITH DAGESH
 # ->   05E7 HEBREW LETTER QOF
 "\x{FB47}" => "\x{05E7}",

 # FB48 HEBREW LETTER RESH WITH DAGESH
 # ->   05E8 HEBREW LETTER RESH
 "\x{FB48}" => "\x{05E8}",

 # FB49 HEBREW LETTER SHIN WITH DAGESH
 # ->   05E9 HEBREW LETTER SHIN
 "\x{FB49}" => "\x{05E9}",

 # FB4A HEBREW LETTER TAV WITH DAGESH
 # ->   05EA HEBREW LETTER TAV
 "\x{FB4A}" => "\x{05EA}",

 # FB4B HEBREW LETTER VAV WITH HOLAM
 # ->   05D5 HEBREW LETTER VAV
 "\x{FB4B}" => "\x{05D5}",

 # FB4C HEBREW LETTER BET WITH RAFE
 # ->   05D1 HEBREW LETTER BET
 "\x{FB4C}" => "\x{05D1}",

 # FB4D HEBREW LETTER KAF WITH RAFE
 # ->   05DB HEBREW LETTER KAF
 "\x{FB4D}" => "\x{05DB}",

 # FB4E HEBREW LETTER PE WITH RAFE
 # ->   05E4 HEBREW LETTER PE
 "\x{FB4E}" => "\x{05E4}",

 # FB4F HEBREW LIGATURE ALEF LAMED
 # ->   05D0 HEBREW LETTER ALEF
 #    + 05DC HEBREW LETTER LAMED
 "\x{FB4F}" => "\x{05D0}\x{05DC}",

 # FB50 ARABIC LETTER ALEF WASLA ISOLATED FORM
 # ->   0671 ARABIC LETTER ALEF WASLA
 "\x{FB50}" => "\x{0671}",

 # FB51 ARABIC LETTER ALEF WASLA FINAL FORM
 # ->   0671 ARABIC LETTER ALEF WASLA
 "\x{FB51}" => "\x{0671}",

 # FB52 ARABIC LETTER BEEH ISOLATED FORM
 # ->   067B ARABIC LETTER BEEH
 "\x{FB52}" => "\x{067B}",

 # FB53 ARABIC LETTER BEEH FINAL FORM
 # ->   067B ARABIC LETTER BEEH
 "\x{FB53}" => "\x{067B}",

 # FB54 ARABIC LETTER BEEH INITIAL FORM
 # ->   067B ARABIC LETTER BEEH
 "\x{FB54}" => "\x{067B}",

 # FB55 ARABIC LETTER BEEH MEDIAL FORM
 # ->   067B ARABIC LETTER BEEH
 "\x{FB55}" => "\x{067B}",

 # FB56 ARABIC LETTER PEH ISOLATED FORM
 # ->   067E ARABIC LETTER PEH
 "\x{FB56}" => "\x{067E}",

 # FB57 ARABIC LETTER PEH FINAL FORM
 # ->   067E ARABIC LETTER PEH
 "\x{FB57}" => "\x{067E}",

 # FB58 ARABIC LETTER PEH INITIAL FORM
 # ->   067E ARABIC LETTER PEH
 "\x{FB58}" => "\x{067E}",

 # FB59 ARABIC LETTER PEH MEDIAL FORM
 # ->   067E ARABIC LETTER PEH
 "\x{FB59}" => "\x{067E}",

 # FB5A ARABIC LETTER BEHEH ISOLATED FORM
 # ->   0680 ARABIC LETTER BEHEH
 "\x{FB5A}" => "\x{0680}",

 # FB5B ARABIC LETTER BEHEH FINAL FORM
 # ->   0680 ARABIC LETTER BEHEH
 "\x{FB5B}" => "\x{0680}",

 # FB5C ARABIC LETTER BEHEH INITIAL FORM
 # ->   0680 ARABIC LETTER BEHEH
 "\x{FB5C}" => "\x{0680}",

 # FB5D ARABIC LETTER BEHEH MEDIAL FORM
 # ->   0680 ARABIC LETTER BEHEH
 "\x{FB5D}" => "\x{0680}",

 # FB5E ARABIC LETTER TTEHEH ISOLATED FORM
 # ->   067A ARABIC LETTER TTEHEH
 "\x{FB5E}" => "\x{067A}",

 # FB5F ARABIC LETTER TTEHEH FINAL FORM
 # ->   067A ARABIC LETTER TTEHEH
 "\x{FB5F}" => "\x{067A}",

 # FB60 ARABIC LETTER TTEHEH INITIAL FORM
 # ->   067A ARABIC LETTER TTEHEH
 "\x{FB60}" => "\x{067A}",

 # FB61 ARABIC LETTER TTEHEH MEDIAL FORM
 # ->   067A ARABIC LETTER TTEHEH
 "\x{FB61}" => "\x{067A}",

 # FB62 ARABIC LETTER TEHEH ISOLATED FORM
 # ->   067F ARABIC LETTER TEHEH
 "\x{FB62}" => "\x{067F}",

 # FB63 ARABIC LETTER TEHEH FINAL FORM
 # ->   067F ARABIC LETTER TEHEH
 "\x{FB63}" => "\x{067F}",

 # FB64 ARABIC LETTER TEHEH INITIAL FORM
 # ->   067F ARABIC LETTER TEHEH
 "\x{FB64}" => "\x{067F}",

 # FB65 ARABIC LETTER TEHEH MEDIAL FORM
 # ->   067F ARABIC LETTER TEHEH
 "\x{FB65}" => "\x{067F}",

 # FB66 ARABIC LETTER TTEH ISOLATED FORM
 # ->   0679 ARABIC LETTER TTEH
 "\x{FB66}" => "\x{0679}",

 # FB67 ARABIC LETTER TTEH FINAL FORM
 # ->   0679 ARABIC LETTER TTEH
 "\x{FB67}" => "\x{0679}",

 # FB68 ARABIC LETTER TTEH INITIAL FORM
 # ->   0679 ARABIC LETTER TTEH
 "\x{FB68}" => "\x{0679}",

 # FB69 ARABIC LETTER TTEH MEDIAL FORM
 # ->   0679 ARABIC LETTER TTEH
 "\x{FB69}" => "\x{0679}",

 # FB6A ARABIC LETTER VEH ISOLATED FORM
 # ->   06A4 ARABIC LETTER VEH
 "\x{FB6A}" => "\x{06A4}",

 # FB6B ARABIC LETTER VEH FINAL FORM
 # ->   06A4 ARABIC LETTER VEH
 "\x{FB6B}" => "\x{06A4}",

 # FB6C ARABIC LETTER VEH INITIAL FORM
 # ->   06A4 ARABIC LETTER VEH
 "\x{FB6C}" => "\x{06A4}",

 # FB6D ARABIC LETTER VEH MEDIAL FORM
 # ->   06A4 ARABIC LETTER VEH
 "\x{FB6D}" => "\x{06A4}",

 # FB6E ARABIC LETTER PEHEH ISOLATED FORM
 # ->   06A6 ARABIC LETTER PEHEH
 "\x{FB6E}" => "\x{06A6}",

 # FB6F ARABIC LETTER PEHEH FINAL FORM
 # ->   06A6 ARABIC LETTER PEHEH
 "\x{FB6F}" => "\x{06A6}",

 # FB70 ARABIC LETTER PEHEH INITIAL FORM
 # ->   06A6 ARABIC LETTER PEHEH
 "\x{FB70}" => "\x{06A6}",

 # FB71 ARABIC LETTER PEHEH MEDIAL FORM
 # ->   06A6 ARABIC LETTER PEHEH
 "\x{FB71}" => "\x{06A6}",

 # FB72 ARABIC LETTER DYEH ISOLATED FORM
 # ->   0684 ARABIC LETTER DYEH
 "\x{FB72}" => "\x{0684}",

 # FB73 ARABIC LETTER DYEH FINAL FORM
 # ->   0684 ARABIC LETTER DYEH
 "\x{FB73}" => "\x{0684}",

 # FB74 ARABIC LETTER DYEH INITIAL FORM
 # ->   0684 ARABIC LETTER DYEH
 "\x{FB74}" => "\x{0684}",

 # FB75 ARABIC LETTER DYEH MEDIAL FORM
 # ->   0684 ARABIC LETTER DYEH
 "\x{FB75}" => "\x{0684}",

 # FB76 ARABIC LETTER NYEH ISOLATED FORM
 # ->   0683 ARABIC LETTER NYEH
 "\x{FB76}" => "\x{0683}",

 # FB77 ARABIC LETTER NYEH FINAL FORM
 # ->   0683 ARABIC LETTER NYEH
 "\x{FB77}" => "\x{0683}",

 # FB78 ARABIC LETTER NYEH INITIAL FORM
 # ->   0683 ARABIC LETTER NYEH
 "\x{FB78}" => "\x{0683}",

 # FB79 ARABIC LETTER NYEH MEDIAL FORM
 # ->   0683 ARABIC LETTER NYEH
 "\x{FB79}" => "\x{0683}",

 # FB7A ARABIC LETTER TCHEH ISOLATED FORM
 # ->   0686 ARABIC LETTER TCHEH
 "\x{FB7A}" => "\x{0686}",

 # FB7B ARABIC LETTER TCHEH FINAL FORM
 # ->   0686 ARABIC LETTER TCHEH
 "\x{FB7B}" => "\x{0686}",

 # FB7C ARABIC LETTER TCHEH INITIAL FORM
 # ->   0686 ARABIC LETTER TCHEH
 "\x{FB7C}" => "\x{0686}",

 # FB7D ARABIC LETTER TCHEH MEDIAL FORM
 # ->   0686 ARABIC LETTER TCHEH
 "\x{FB7D}" => "\x{0686}",

 # FB7E ARABIC LETTER TCHEHEH ISOLATED FORM
 # ->   0687 ARABIC LETTER TCHEHEH
 "\x{FB7E}" => "\x{0687}",

 # FB7F ARABIC LETTER TCHEHEH FINAL FORM
 # ->   0687 ARABIC LETTER TCHEHEH
 "\x{FB7F}" => "\x{0687}",

 # FB80 ARABIC LETTER TCHEHEH INITIAL FORM
 # ->   0687 ARABIC LETTER TCHEHEH
 "\x{FB80}" => "\x{0687}",

 # FB81 ARABIC LETTER TCHEHEH MEDIAL FORM
 # ->   0687 ARABIC LETTER TCHEHEH
 "\x{FB81}" => "\x{0687}",

 # FB82 ARABIC LETTER DDAHAL ISOLATED FORM
 # ->   068D ARABIC LETTER DDAHAL
 "\x{FB82}" => "\x{068D}",

 # FB83 ARABIC LETTER DDAHAL FINAL FORM
 # ->   068D ARABIC LETTER DDAHAL
 "\x{FB83}" => "\x{068D}",

 # FB84 ARABIC LETTER DAHAL ISOLATED FORM
 # ->   068C ARABIC LETTER DAHAL
 "\x{FB84}" => "\x{068C}",

 # FB85 ARABIC LETTER DAHAL FINAL FORM
 # ->   068C ARABIC LETTER DAHAL
 "\x{FB85}" => "\x{068C}",

 # FB86 ARABIC LETTER DUL ISOLATED FORM
 # ->   068E ARABIC LETTER DUL
 "\x{FB86}" => "\x{068E}",

 # FB87 ARABIC LETTER DUL FINAL FORM
 # ->   068E ARABIC LETTER DUL
 "\x{FB87}" => "\x{068E}",

 # FB88 ARABIC LETTER DDAL ISOLATED FORM
 # ->   0688 ARABIC LETTER DDAL
 "\x{FB88}" => "\x{0688}",

 # FB89 ARABIC LETTER DDAL FINAL FORM
 # ->   0688 ARABIC LETTER DDAL
 "\x{FB89}" => "\x{0688}",

 # FB8A ARABIC LETTER JEH ISOLATED FORM
 # ->   0698 ARABIC LETTER JEH
 "\x{FB8A}" => "\x{0698}",

 # FB8B ARABIC LETTER JEH FINAL FORM
 # ->   0698 ARABIC LETTER JEH
 "\x{FB8B}" => "\x{0698}",

 # FB8C ARABIC LETTER RREH ISOLATED FORM
 # ->   0691 ARABIC LETTER RREH
 "\x{FB8C}" => "\x{0691}",

 # FB8D ARABIC LETTER RREH FINAL FORM
 # ->   0691 ARABIC LETTER RREH
 "\x{FB8D}" => "\x{0691}",

 # FB8E ARABIC LETTER KEHEH ISOLATED FORM
 # ->   06A9 ARABIC LETTER KEHEH
 "\x{FB8E}" => "\x{06A9}",

 # FB8F ARABIC LETTER KEHEH FINAL FORM
 # ->   06A9 ARABIC LETTER KEHEH
 "\x{FB8F}" => "\x{06A9}",

 # FB90 ARABIC LETTER KEHEH INITIAL FORM
 # ->   06A9 ARABIC LETTER KEHEH
 "\x{FB90}" => "\x{06A9}",

 # FB91 ARABIC LETTER KEHEH MEDIAL FORM
 # ->   06A9 ARABIC LETTER KEHEH
 "\x{FB91}" => "\x{06A9}",

 # FB92 ARABIC LETTER GAF ISOLATED FORM
 # ->   06AF ARABIC LETTER GAF
 "\x{FB92}" => "\x{06AF}",

 # FB93 ARABIC LETTER GAF FINAL FORM
 # ->   06AF ARABIC LETTER GAF
 "\x{FB93}" => "\x{06AF}",

 # FB94 ARABIC LETTER GAF INITIAL FORM
 # ->   06AF ARABIC LETTER GAF
 "\x{FB94}" => "\x{06AF}",

 # FB95 ARABIC LETTER GAF MEDIAL FORM
 # ->   06AF ARABIC LETTER GAF
 "\x{FB95}" => "\x{06AF}",

 # FB96 ARABIC LETTER GUEH ISOLATED FORM
 # ->   06B3 ARABIC LETTER GUEH
 "\x{FB96}" => "\x{06B3}",

 # FB97 ARABIC LETTER GUEH FINAL FORM
 # ->   06B3 ARABIC LETTER GUEH
 "\x{FB97}" => "\x{06B3}",

 # FB98 ARABIC LETTER GUEH INITIAL FORM
 # ->   06B3 ARABIC LETTER GUEH
 "\x{FB98}" => "\x{06B3}",

 # FB99 ARABIC LETTER GUEH MEDIAL FORM
 # ->   06B3 ARABIC LETTER GUEH
 "\x{FB99}" => "\x{06B3}",

 # FB9A ARABIC LETTER NGOEH ISOLATED FORM
 # ->   06B1 ARABIC LETTER NGOEH
 "\x{FB9A}" => "\x{06B1}",

 # FB9B ARABIC LETTER NGOEH FINAL FORM
 # ->   06B1 ARABIC LETTER NGOEH
 "\x{FB9B}" => "\x{06B1}",

 # FB9C ARABIC LETTER NGOEH INITIAL FORM
 # ->   06B1 ARABIC LETTER NGOEH
 "\x{FB9C}" => "\x{06B1}",

 # FB9D ARABIC LETTER NGOEH MEDIAL FORM
 # ->   06B1 ARABIC LETTER NGOEH
 "\x{FB9D}" => "\x{06B1}",

 # FB9E ARABIC LETTER NOON GHUNNA ISOLATED FORM
 # ->   06BA ARABIC LETTER NOON GHUNNA
 "\x{FB9E}" => "\x{06BA}",

 # FB9F ARABIC LETTER NOON GHUNNA FINAL FORM
 # ->   06BA ARABIC LETTER NOON GHUNNA
 "\x{FB9F}" => "\x{06BA}",

 # FBA0 ARABIC LETTER RNOON ISOLATED FORM
 # ->   06BB ARABIC LETTER RNOON
 "\x{FBA0}" => "\x{06BB}",

 # FBA1 ARABIC LETTER RNOON FINAL FORM
 # ->   06BB ARABIC LETTER RNOON
 "\x{FBA1}" => "\x{06BB}",

 # FBA2 ARABIC LETTER RNOON INITIAL FORM
 # ->   06BB ARABIC LETTER RNOON
 "\x{FBA2}" => "\x{06BB}",

 # FBA3 ARABIC LETTER RNOON MEDIAL FORM
 # ->   06BB ARABIC LETTER RNOON
 "\x{FBA3}" => "\x{06BB}",

 # FBA4 ARABIC LETTER HEH WITH YEH ABOVE ISOLATED FORM
 # ->   06D5 ARABIC LETTER AE
 "\x{FBA4}" => "\x{06D5}",

 # FBA5 ARABIC LETTER HEH WITH YEH ABOVE FINAL FORM
 # ->   06D5 ARABIC LETTER AE
 "\x{FBA5}" => "\x{06D5}",

 # FBA6 ARABIC LETTER HEH GOAL ISOLATED FORM
 # ->   06C1 ARABIC LETTER HEH GOAL
 "\x{FBA6}" => "\x{06C1}",

 # FBA7 ARABIC LETTER HEH GOAL FINAL FORM
 # ->   06C1 ARABIC LETTER HEH GOAL
 "\x{FBA7}" => "\x{06C1}",

 # FBA8 ARABIC LETTER HEH GOAL INITIAL FORM
 # ->   06C1 ARABIC LETTER HEH GOAL
 "\x{FBA8}" => "\x{06C1}",

 # FBA9 ARABIC LETTER HEH GOAL MEDIAL FORM
 # ->   06C1 ARABIC LETTER HEH GOAL
 "\x{FBA9}" => "\x{06C1}",

 # FBAA ARABIC LETTER HEH DOACHASHMEE ISOLATED FORM
 # ->   06BE ARABIC LETTER HEH DOACHASHMEE
 "\x{FBAA}" => "\x{06BE}",

 # FBAB ARABIC LETTER HEH DOACHASHMEE FINAL FORM
 # ->   06BE ARABIC LETTER HEH DOACHASHMEE
 "\x{FBAB}" => "\x{06BE}",

 # FBAC ARABIC LETTER HEH DOACHASHMEE INITIAL FORM
 # ->   06BE ARABIC LETTER HEH DOACHASHMEE
 "\x{FBAC}" => "\x{06BE}",

 # FBAD ARABIC LETTER HEH DOACHASHMEE MEDIAL FORM
 # ->   06BE ARABIC LETTER HEH DOACHASHMEE
 "\x{FBAD}" => "\x{06BE}",

 # FBAE ARABIC LETTER YEH BARREE ISOLATED FORM
 # ->   06D2 ARABIC LETTER YEH BARREE
 "\x{FBAE}" => "\x{06D2}",

 # FBAF ARABIC LETTER YEH BARREE FINAL FORM
 # ->   06D2 ARABIC LETTER YEH BARREE
 "\x{FBAF}" => "\x{06D2}",

 # FBB0 ARABIC LETTER YEH BARREE WITH HAMZA ABOVE ISOLATED FORM
 # ->   06D2 ARABIC LETTER YEH BARREE
 "\x{FBB0}" => "\x{06D2}",

 # FBB1 ARABIC LETTER YEH BARREE WITH HAMZA ABOVE FINAL FORM
 # ->   06D2 ARABIC LETTER YEH BARREE
 "\x{FBB1}" => "\x{06D2}",

 # FBD3 ARABIC LETTER NG ISOLATED FORM
 # ->   06AD ARABIC LETTER NG
 "\x{FBD3}" => "\x{06AD}",

 # FBD4 ARABIC LETTER NG FINAL FORM
 # ->   06AD ARABIC LETTER NG
 "\x{FBD4}" => "\x{06AD}",

 # FBD5 ARABIC LETTER NG INITIAL FORM
 # ->   06AD ARABIC LETTER NG
 "\x{FBD5}" => "\x{06AD}",

 # FBD6 ARABIC LETTER NG MEDIAL FORM
 # ->   06AD ARABIC LETTER NG
 "\x{FBD6}" => "\x{06AD}",

 # FBD7 ARABIC LETTER U ISOLATED FORM
 # ->   06C7 ARABIC LETTER U
 "\x{FBD7}" => "\x{06C7}",

 # FBD8 ARABIC LETTER U FINAL FORM
 # ->   06C7 ARABIC LETTER U
 "\x{FBD8}" => "\x{06C7}",

 # FBD9 ARABIC LETTER OE ISOLATED FORM
 # ->   06C6 ARABIC LETTER OE
 "\x{FBD9}" => "\x{06C6}",

 # FBDA ARABIC LETTER OE FINAL FORM
 # ->   06C6 ARABIC LETTER OE
 "\x{FBDA}" => "\x{06C6}",

 # FBDB ARABIC LETTER YU ISOLATED FORM
 # ->   06C8 ARABIC LETTER YU
 "\x{FBDB}" => "\x{06C8}",

 # FBDC ARABIC LETTER YU FINAL FORM
 # ->   06C8 ARABIC LETTER YU
 "\x{FBDC}" => "\x{06C8}",

 # FBDD ARABIC LETTER U WITH HAMZA ABOVE ISOLATED FORM
 # ->   06C7 ARABIC LETTER U
 #    + 0674 ARABIC LETTER HIGH HAMZA
 "\x{FBDD}" => "\x{06C7}\x{0674}",

 # FBDE ARABIC LETTER VE ISOLATED FORM
 # ->   06CB ARABIC LETTER VE
 "\x{FBDE}" => "\x{06CB}",

 # FBDF ARABIC LETTER VE FINAL FORM
 # ->   06CB ARABIC LETTER VE
 "\x{FBDF}" => "\x{06CB}",

 # FBE0 ARABIC LETTER KIRGHIZ OE ISOLATED FORM
 # ->   06C5 ARABIC LETTER KIRGHIZ OE
 "\x{FBE0}" => "\x{06C5}",

 # FBE1 ARABIC LETTER KIRGHIZ OE FINAL FORM
 # ->   06C5 ARABIC LETTER KIRGHIZ OE
 "\x{FBE1}" => "\x{06C5}",

 # FBE2 ARABIC LETTER KIRGHIZ YU ISOLATED FORM
 # ->   06C9 ARABIC LETTER KIRGHIZ YU
 "\x{FBE2}" => "\x{06C9}",

 # FBE3 ARABIC LETTER KIRGHIZ YU FINAL FORM
 # ->   06C9 ARABIC LETTER KIRGHIZ YU
 "\x{FBE3}" => "\x{06C9}",

 # FBE4 ARABIC LETTER E ISOLATED FORM
 # ->   06D0 ARABIC LETTER E
 "\x{FBE4}" => "\x{06D0}",

 # FBE5 ARABIC LETTER E FINAL FORM
 # ->   06D0 ARABIC LETTER E
 "\x{FBE5}" => "\x{06D0}",

 # FBE6 ARABIC LETTER E INITIAL FORM
 # ->   06D0 ARABIC LETTER E
 "\x{FBE6}" => "\x{06D0}",

 # FBE7 ARABIC LETTER E MEDIAL FORM
 # ->   06D0 ARABIC LETTER E
 "\x{FBE7}" => "\x{06D0}",

 # FBE8 ARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA INITIAL FORM
 # ->   0649 ARABIC LETTER ALEF MAKSURA
 "\x{FBE8}" => "\x{0649}",

 # FBE9 ARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA MEDIAL FORM
 # ->   0649 ARABIC LETTER ALEF MAKSURA
 "\x{FBE9}" => "\x{0649}",

 # FBEA ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF ISOLATED FORM
 # ->   0627 ARABIC LETTER ALEF
 #    + 064A ARABIC LETTER YEH
 "\x{FBEA}" => "\x{0627}\x{064A}",

 # FBEB ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF FINAL FORM
 # ->   0627 ARABIC LETTER ALEF
 #    + 064A ARABIC LETTER YEH
 "\x{FBEB}" => "\x{0627}\x{064A}",

 # FBEC ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE ISOLATED FORM
 # ->   06D5 ARABIC LETTER AE
 #    + 064A ARABIC LETTER YEH
 "\x{FBEC}" => "\x{06D5}\x{064A}",

 # FBED ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE FINAL FORM
 # ->   06D5 ARABIC LETTER AE
 #    + 064A ARABIC LETTER YEH
 "\x{FBED}" => "\x{06D5}\x{064A}",

 # FBEE ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH WAW ISOLATED FORM
 # ->   0648 ARABIC LETTER WAW
 #    + 064A ARABIC LETTER YEH
 "\x{FBEE}" => "\x{0648}\x{064A}",

 # FBEF ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH WAW FINAL FORM
 # ->   0648 ARABIC LETTER WAW
 #    + 064A ARABIC LETTER YEH
 "\x{FBEF}" => "\x{0648}\x{064A}",

 # FBF0 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U ISOLATED FORM
 # ->   06C7 ARABIC LETTER U
 #    + 064A ARABIC LETTER YEH
 "\x{FBF0}" => "\x{06C7}\x{064A}",

 # FBF1 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U FINAL FORM
 # ->   06C7 ARABIC LETTER U
 #    + 064A ARABIC LETTER YEH
 "\x{FBF1}" => "\x{06C7}\x{064A}",

 # FBF2 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH OE ISOLATED FORM
 # ->   06C6 ARABIC LETTER OE
 #    + 064A ARABIC LETTER YEH
 "\x{FBF2}" => "\x{06C6}\x{064A}",

 # FBF3 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH OE FINAL FORM
 # ->   06C6 ARABIC LETTER OE
 #    + 064A ARABIC LETTER YEH
 "\x{FBF3}" => "\x{06C6}\x{064A}",

 # FBF4 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU ISOLATED FORM
 # ->   06C8 ARABIC LETTER YU
 #    + 064A ARABIC LETTER YEH
 "\x{FBF4}" => "\x{06C8}\x{064A}",

 # FBF5 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU FINAL FORM
 # ->   06C8 ARABIC LETTER YU
 #    + 064A ARABIC LETTER YEH
 "\x{FBF5}" => "\x{06C8}\x{064A}",

 # FBF6 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E ISOLATED FORM
 # ->   06D0 ARABIC LETTER E
 #    + 064A ARABIC LETTER YEH
 "\x{FBF6}" => "\x{06D0}\x{064A}",

 # FBF7 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E FINAL FORM
 # ->   06D0 ARABIC LETTER E
 #    + 064A ARABIC LETTER YEH
 "\x{FBF7}" => "\x{06D0}\x{064A}",

 # FBF8 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E INITIAL FORM
 # ->   06D0 ARABIC LETTER E
 #    + 064A ARABIC LETTER YEH
 "\x{FBF8}" => "\x{06D0}\x{064A}",

 # FBF9 ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORM
 # ->   0649 ARABIC LETTER ALEF MAKSURA
 #    + 064A ARABIC LETTER YEH
 "\x{FBF9}" => "\x{0649}\x{064A}",

 # FBFA ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA FINAL FORM
 # ->   0649 ARABIC LETTER ALEF MAKSURA
 #    + 064A ARABIC LETTER YEH
 "\x{FBFA}" => "\x{0649}\x{064A}",

 # FBFB ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA INITIAL FORM
 # ->   0649 ARABIC LETTER ALEF MAKSURA
 #    + 064A ARABIC LETTER YEH
 "\x{FBFB}" => "\x{0649}\x{064A}",

 # FBFC ARABIC LETTER FARSI YEH ISOLATED FORM
 # ->   06CC ARABIC LETTER FARSI YEH
 "\x{FBFC}" => "\x{06CC}",

 # FBFD ARABIC LETTER FARSI YEH FINAL FORM
 # ->   06CC ARABIC LETTER FARSI YEH
 "\x{FBFD}" => "\x{06CC}",

 # FBFE ARABIC LETTER FARSI YEH INITIAL FORM
 # ->   06CC ARABIC LETTER FARSI YEH
 "\x{FBFE}" => "\x{06CC}",

 # FBFF ARABIC LETTER FARSI YEH MEDIAL FORM
 # ->   06CC ARABIC LETTER FARSI YEH
 "\x{FBFF}" => "\x{06CC}",

 # FC00 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH JEEM ISOLATED FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FC00}" => "\x{062C}\x{064A}",

 # FC01 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HAH ISOLATED FORM
 # ->   062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FC01}" => "\x{062D}\x{064A}",

 # FC02 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM ISOLATED FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FC02}" => "\x{0645}\x{064A}",

 # FC03 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORM
 # ->   0649 ARABIC LETTER ALEF MAKSURA
 #    + 064A ARABIC LETTER YEH
 "\x{FC03}" => "\x{0649}\x{064A}",

 # FC04 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YEH ISOLATED FORM
 # ->   064A ARABIC LETTER YEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC04}" => "\x{064A}\x{064A}",

 # FC05 ARABIC LIGATURE BEH WITH JEEM ISOLATED FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 062C ARABIC LETTER JEEM
 "\x{FC05}" => "\x{0628}\x{062C}",

 # FC06 ARABIC LIGATURE BEH WITH HAH ISOLATED FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 062D ARABIC LETTER HAH
 "\x{FC06}" => "\x{0628}\x{062D}",

 # FC07 ARABIC LIGATURE BEH WITH KHAH ISOLATED FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 062E ARABIC LETTER KHAH
 "\x{FC07}" => "\x{0628}\x{062E}",

 # FC08 ARABIC LIGATURE BEH WITH MEEM ISOLATED FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC08}" => "\x{0628}\x{0645}",

 # FC09 ARABIC LIGATURE BEH WITH ALEF MAKSURA ISOLATED FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC09}" => "\x{0628}\x{0649}",

 # FC0A ARABIC LIGATURE BEH WITH YEH ISOLATED FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC0A}" => "\x{0628}\x{064A}",

 # FC0B ARABIC LIGATURE TEH WITH JEEM ISOLATED FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062C ARABIC LETTER JEEM
 "\x{FC0B}" => "\x{062A}\x{062C}",

 # FC0C ARABIC LIGATURE TEH WITH HAH ISOLATED FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062D ARABIC LETTER HAH
 "\x{FC0C}" => "\x{062A}\x{062D}",

 # FC0D ARABIC LIGATURE TEH WITH KHAH ISOLATED FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062E ARABIC LETTER KHAH
 "\x{FC0D}" => "\x{062A}\x{062E}",

 # FC0E ARABIC LIGATURE TEH WITH MEEM ISOLATED FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC0E}" => "\x{062A}\x{0645}",

 # FC0F ARABIC LIGATURE TEH WITH ALEF MAKSURA ISOLATED FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC0F}" => "\x{062A}\x{0649}",

 # FC10 ARABIC LIGATURE TEH WITH YEH ISOLATED FORM
 # ->   062A ARABIC LETTER TEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC10}" => "\x{062A}\x{064A}",

 # FC11 ARABIC LIGATURE THEH WITH JEEM ISOLATED FORM
 # ->   062B ARABIC LETTER THEH
 #    + 062C ARABIC LETTER JEEM
 "\x{FC11}" => "\x{062B}\x{062C}",

 # FC12 ARABIC LIGATURE THEH WITH MEEM ISOLATED FORM
 # ->   062B ARABIC LETTER THEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC12}" => "\x{062B}\x{0645}",

 # FC13 ARABIC LIGATURE THEH WITH ALEF MAKSURA ISOLATED FORM
 # ->   062B ARABIC LETTER THEH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC13}" => "\x{062B}\x{0649}",

 # FC14 ARABIC LIGATURE THEH WITH YEH ISOLATED FORM
 # ->   062B ARABIC LETTER THEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC14}" => "\x{062B}\x{064A}",

 # FC15 ARABIC LIGATURE JEEM WITH HAH ISOLATED FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FC15}" => "\x{062C}\x{062D}",

 # FC16 ARABIC LIGATURE JEEM WITH MEEM ISOLATED FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC16}" => "\x{062C}\x{0645}",

 # FC17 ARABIC LIGATURE HAH WITH JEEM ISOLATED FORM
 # ->   062D ARABIC LETTER HAH
 #    + 062C ARABIC LETTER JEEM
 "\x{FC17}" => "\x{062D}\x{062C}",

 # FC18 ARABIC LIGATURE HAH WITH MEEM ISOLATED FORM
 # ->   062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC18}" => "\x{062D}\x{0645}",

 # FC19 ARABIC LIGATURE KHAH WITH JEEM ISOLATED FORM
 # ->   062E ARABIC LETTER KHAH
 #    + 062C ARABIC LETTER JEEM
 "\x{FC19}" => "\x{062E}\x{062C}",

 # FC1A ARABIC LIGATURE KHAH WITH HAH ISOLATED FORM
 # ->   062E ARABIC LETTER KHAH
 #    + 062D ARABIC LETTER HAH
 "\x{FC1A}" => "\x{062E}\x{062D}",

 # FC1B ARABIC LIGATURE KHAH WITH MEEM ISOLATED FORM
 # ->   062E ARABIC LETTER KHAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC1B}" => "\x{062E}\x{0645}",

 # FC1C ARABIC LIGATURE SEEN WITH JEEM ISOLATED FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062C ARABIC LETTER JEEM
 "\x{FC1C}" => "\x{0633}\x{062C}",

 # FC1D ARABIC LIGATURE SEEN WITH HAH ISOLATED FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062D ARABIC LETTER HAH
 "\x{FC1D}" => "\x{0633}\x{062D}",

 # FC1E ARABIC LIGATURE SEEN WITH KHAH ISOLATED FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062E ARABIC LETTER KHAH
 "\x{FC1E}" => "\x{0633}\x{062E}",

 # FC1F ARABIC LIGATURE SEEN WITH MEEM ISOLATED FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC1F}" => "\x{0633}\x{0645}",

 # FC20 ARABIC LIGATURE SAD WITH HAH ISOLATED FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 062D ARABIC LETTER HAH
 "\x{FC20}" => "\x{0635}\x{062D}",

 # FC21 ARABIC LIGATURE SAD WITH MEEM ISOLATED FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC21}" => "\x{0635}\x{0645}",

 # FC22 ARABIC LIGATURE DAD WITH JEEM ISOLATED FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 062C ARABIC LETTER JEEM
 "\x{FC22}" => "\x{0636}\x{062C}",

 # FC23 ARABIC LIGATURE DAD WITH HAH ISOLATED FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 062D ARABIC LETTER HAH
 "\x{FC23}" => "\x{0636}\x{062D}",

 # FC24 ARABIC LIGATURE DAD WITH KHAH ISOLATED FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 062E ARABIC LETTER KHAH
 "\x{FC24}" => "\x{0636}\x{062E}",

 # FC25 ARABIC LIGATURE DAD WITH MEEM ISOLATED FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC25}" => "\x{0636}\x{0645}",

 # FC26 ARABIC LIGATURE TAH WITH HAH ISOLATED FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 062D ARABIC LETTER HAH
 "\x{FC26}" => "\x{0637}\x{062D}",

 # FC27 ARABIC LIGATURE TAH WITH MEEM ISOLATED FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC27}" => "\x{0637}\x{0645}",

 # FC28 ARABIC LIGATURE ZAH WITH MEEM ISOLATED FORM
 # ->   0638 ARABIC LETTER ZAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC28}" => "\x{0638}\x{0645}",

 # FC29 ARABIC LIGATURE AIN WITH JEEM ISOLATED FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 062C ARABIC LETTER JEEM
 "\x{FC29}" => "\x{0639}\x{062C}",

 # FC2A ARABIC LIGATURE AIN WITH MEEM ISOLATED FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC2A}" => "\x{0639}\x{0645}",

 # FC2B ARABIC LIGATURE GHAIN WITH JEEM ISOLATED FORM
 # ->   063A ARABIC LETTER GHAIN
 #    + 062C ARABIC LETTER JEEM
 "\x{FC2B}" => "\x{063A}\x{062C}",

 # FC2C ARABIC LIGATURE GHAIN WITH MEEM ISOLATED FORM
 # ->   063A ARABIC LETTER GHAIN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC2C}" => "\x{063A}\x{0645}",

 # FC2D ARABIC LIGATURE FEH WITH JEEM ISOLATED FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 062C ARABIC LETTER JEEM
 "\x{FC2D}" => "\x{0641}\x{062C}",

 # FC2E ARABIC LIGATURE FEH WITH HAH ISOLATED FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 062D ARABIC LETTER HAH
 "\x{FC2E}" => "\x{0641}\x{062D}",

 # FC2F ARABIC LIGATURE FEH WITH KHAH ISOLATED FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 062E ARABIC LETTER KHAH
 "\x{FC2F}" => "\x{0641}\x{062E}",

 # FC30 ARABIC LIGATURE FEH WITH MEEM ISOLATED FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC30}" => "\x{0641}\x{0645}",

 # FC31 ARABIC LIGATURE FEH WITH ALEF MAKSURA ISOLATED FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC31}" => "\x{0641}\x{0649}",

 # FC32 ARABIC LIGATURE FEH WITH YEH ISOLATED FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC32}" => "\x{0641}\x{064A}",

 # FC33 ARABIC LIGATURE QAF WITH HAH ISOLATED FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 062D ARABIC LETTER HAH
 "\x{FC33}" => "\x{0642}\x{062D}",

 # FC34 ARABIC LIGATURE QAF WITH MEEM ISOLATED FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC34}" => "\x{0642}\x{0645}",

 # FC35 ARABIC LIGATURE QAF WITH ALEF MAKSURA ISOLATED FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC35}" => "\x{0642}\x{0649}",

 # FC36 ARABIC LIGATURE QAF WITH YEH ISOLATED FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 064A ARABIC LETTER YEH
 "\x{FC36}" => "\x{0642}\x{064A}",

 # FC37 ARABIC LIGATURE KAF WITH ALEF ISOLATED FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0627 ARABIC LETTER ALEF
 "\x{FC37}" => "\x{0643}\x{0627}",

 # FC38 ARABIC LIGATURE KAF WITH JEEM ISOLATED FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 062C ARABIC LETTER JEEM
 "\x{FC38}" => "\x{0643}\x{062C}",

 # FC39 ARABIC LIGATURE KAF WITH HAH ISOLATED FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 062D ARABIC LETTER HAH
 "\x{FC39}" => "\x{0643}\x{062D}",

 # FC3A ARABIC LIGATURE KAF WITH KHAH ISOLATED FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 062E ARABIC LETTER KHAH
 "\x{FC3A}" => "\x{0643}\x{062E}",

 # FC3B ARABIC LIGATURE KAF WITH LAM ISOLATED FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0644 ARABIC LETTER LAM
 "\x{FC3B}" => "\x{0643}\x{0644}",

 # FC3C ARABIC LIGATURE KAF WITH MEEM ISOLATED FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC3C}" => "\x{0643}\x{0645}",

 # FC3D ARABIC LIGATURE KAF WITH ALEF MAKSURA ISOLATED FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC3D}" => "\x{0643}\x{0649}",

 # FC3E ARABIC LIGATURE KAF WITH YEH ISOLATED FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 064A ARABIC LETTER YEH
 "\x{FC3E}" => "\x{0643}\x{064A}",

 # FC3F ARABIC LIGATURE LAM WITH JEEM ISOLATED FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062C ARABIC LETTER JEEM
 "\x{FC3F}" => "\x{0644}\x{062C}",

 # FC40 ARABIC LIGATURE LAM WITH HAH ISOLATED FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062D ARABIC LETTER HAH
 "\x{FC40}" => "\x{0644}\x{062D}",

 # FC41 ARABIC LIGATURE LAM WITH KHAH ISOLATED FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062E ARABIC LETTER KHAH
 "\x{FC41}" => "\x{0644}\x{062E}",

 # FC42 ARABIC LIGATURE LAM WITH MEEM ISOLATED FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC42}" => "\x{0644}\x{0645}",

 # FC43 ARABIC LIGATURE LAM WITH ALEF MAKSURA ISOLATED FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC43}" => "\x{0644}\x{0649}",

 # FC44 ARABIC LIGATURE LAM WITH YEH ISOLATED FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 064A ARABIC LETTER YEH
 "\x{FC44}" => "\x{0644}\x{064A}",

 # FC45 ARABIC LIGATURE MEEM WITH JEEM ISOLATED FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062C ARABIC LETTER JEEM
 "\x{FC45}" => "\x{0645}\x{062C}",

 # FC46 ARABIC LIGATURE MEEM WITH HAH ISOLATED FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FC46}" => "\x{0645}\x{062D}",

 # FC47 ARABIC LIGATURE MEEM WITH KHAH ISOLATED FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062E ARABIC LETTER KHAH
 "\x{FC47}" => "\x{0645}\x{062E}",

 # FC48 ARABIC LIGATURE MEEM WITH MEEM ISOLATED FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC48}" => "\x{0645}\x{0645}",

 # FC49 ARABIC LIGATURE MEEM WITH ALEF MAKSURA ISOLATED FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC49}" => "\x{0645}\x{0649}",

 # FC4A ARABIC LIGATURE MEEM WITH YEH ISOLATED FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FC4A}" => "\x{0645}\x{064A}",

 # FC4B ARABIC LIGATURE NOON WITH JEEM ISOLATED FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062C ARABIC LETTER JEEM
 "\x{FC4B}" => "\x{0646}\x{062C}",

 # FC4C ARABIC LIGATURE NOON WITH HAH ISOLATED FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062D ARABIC LETTER HAH
 "\x{FC4C}" => "\x{0646}\x{062D}",

 # FC4D ARABIC LIGATURE NOON WITH KHAH ISOLATED FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062E ARABIC LETTER KHAH
 "\x{FC4D}" => "\x{0646}\x{062E}",

 # FC4E ARABIC LIGATURE NOON WITH MEEM ISOLATED FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC4E}" => "\x{0646}\x{0645}",

 # FC4F ARABIC LIGATURE NOON WITH ALEF MAKSURA ISOLATED FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC4F}" => "\x{0646}\x{0649}",

 # FC50 ARABIC LIGATURE NOON WITH YEH ISOLATED FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 064A ARABIC LETTER YEH
 "\x{FC50}" => "\x{0646}\x{064A}",

 # FC51 ARABIC LIGATURE HEH WITH JEEM ISOLATED FORM
 # ->   0647 ARABIC LETTER HEH
 #    + 062C ARABIC LETTER JEEM
 "\x{FC51}" => "\x{0647}\x{062C}",

 # FC52 ARABIC LIGATURE HEH WITH MEEM ISOLATED FORM
 # ->   0647 ARABIC LETTER HEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC52}" => "\x{0647}\x{0645}",

 # FC53 ARABIC LIGATURE HEH WITH ALEF MAKSURA ISOLATED FORM
 # ->   0647 ARABIC LETTER HEH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC53}" => "\x{0647}\x{0649}",

 # FC54 ARABIC LIGATURE HEH WITH YEH ISOLATED FORM
 # ->   0647 ARABIC LETTER HEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC54}" => "\x{0647}\x{064A}",

 # FC55 ARABIC LIGATURE YEH WITH JEEM ISOLATED FORM
 # ->   064A ARABIC LETTER YEH
 #    + 062C ARABIC LETTER JEEM
 "\x{FC55}" => "\x{064A}\x{062C}",

 # FC56 ARABIC LIGATURE YEH WITH HAH ISOLATED FORM
 # ->   064A ARABIC LETTER YEH
 #    + 062D ARABIC LETTER HAH
 "\x{FC56}" => "\x{064A}\x{062D}",

 # FC57 ARABIC LIGATURE YEH WITH KHAH ISOLATED FORM
 # ->   064A ARABIC LETTER YEH
 #    + 062E ARABIC LETTER KHAH
 "\x{FC57}" => "\x{064A}\x{062E}",

 # FC58 ARABIC LIGATURE YEH WITH MEEM ISOLATED FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC58}" => "\x{064A}\x{0645}",

 # FC59 ARABIC LIGATURE YEH WITH ALEF MAKSURA ISOLATED FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC59}" => "\x{064A}\x{0649}",

 # FC5A ARABIC LIGATURE YEH WITH YEH ISOLATED FORM
 # ->   064A ARABIC LETTER YEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC5A}" => "\x{064A}\x{064A}",

 # FC5B ARABIC LIGATURE THAL WITH SUPERSCRIPT ALEF ISOLATED FORM
 # ->   0630 ARABIC LETTER THAL
 "\x{FC5B}" => "\x{0630}",

 # FC5C ARABIC LIGATURE REH WITH SUPERSCRIPT ALEF ISOLATED FORM
 # ->   0631 ARABIC LETTER REH
 "\x{FC5C}" => "\x{0631}",

 # FC5D ARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF ISOLATED FORM
 # ->   0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC5D}" => "\x{0649}",

 # FC5E ARABIC LIGATURE SHADDA WITH DAMMATAN ISOLATED FORM
 # ->   0020 SPACE
 "\x{FC5E}" => " ",

 # FC5F ARABIC LIGATURE SHADDA WITH KASRATAN ISOLATED FORM
 # ->   0020 SPACE
 "\x{FC5F}" => " ",

 # FC60 ARABIC LIGATURE SHADDA WITH FATHA ISOLATED FORM
 # ->   0020 SPACE
 "\x{FC60}" => " ",

 # FC61 ARABIC LIGATURE SHADDA WITH DAMMA ISOLATED FORM
 # ->   0020 SPACE
 "\x{FC61}" => " ",

 # FC62 ARABIC LIGATURE SHADDA WITH KASRA ISOLATED FORM
 # ->   0020 SPACE
 "\x{FC62}" => " ",

 # FC63 ARABIC LIGATURE SHADDA WITH SUPERSCRIPT ALEF ISOLATED FORM
 # ->   0020 SPACE
 "\x{FC63}" => " ",

 # FC64 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH REH FINAL FORM
 # ->   0631 ARABIC LETTER REH
 #    + 064A ARABIC LETTER YEH
 "\x{FC64}" => "\x{0631}\x{064A}",

 # FC65 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ZAIN FINAL FORM
 # ->   0632 ARABIC LETTER ZAIN
 #    + 064A ARABIC LETTER YEH
 "\x{FC65}" => "\x{0632}\x{064A}",

 # FC66 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM FINAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FC66}" => "\x{0645}\x{064A}",

 # FC67 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH NOON FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 064A ARABIC LETTER YEH
 "\x{FC67}" => "\x{0646}\x{064A}",

 # FC68 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA FINAL FORM
 # ->   0649 ARABIC LETTER ALEF MAKSURA
 #    + 064A ARABIC LETTER YEH
 "\x{FC68}" => "\x{0649}\x{064A}",

 # FC69 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YEH FINAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC69}" => "\x{064A}\x{064A}",

 # FC6A ARABIC LIGATURE BEH WITH REH FINAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 0631 ARABIC LETTER REH
 "\x{FC6A}" => "\x{0628}\x{0631}",

 # FC6B ARABIC LIGATURE BEH WITH ZAIN FINAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 0632 ARABIC LETTER ZAIN
 "\x{FC6B}" => "\x{0628}\x{0632}",

 # FC6C ARABIC LIGATURE BEH WITH MEEM FINAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC6C}" => "\x{0628}\x{0645}",

 # FC6D ARABIC LIGATURE BEH WITH NOON FINAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 0646 ARABIC LETTER NOON
 "\x{FC6D}" => "\x{0628}\x{0646}",

 # FC6E ARABIC LIGATURE BEH WITH ALEF MAKSURA FINAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC6E}" => "\x{0628}\x{0649}",

 # FC6F ARABIC LIGATURE BEH WITH YEH FINAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC6F}" => "\x{0628}\x{064A}",

 # FC70 ARABIC LIGATURE TEH WITH REH FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0631 ARABIC LETTER REH
 "\x{FC70}" => "\x{062A}\x{0631}",

 # FC71 ARABIC LIGATURE TEH WITH ZAIN FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0632 ARABIC LETTER ZAIN
 "\x{FC71}" => "\x{062A}\x{0632}",

 # FC72 ARABIC LIGATURE TEH WITH MEEM FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC72}" => "\x{062A}\x{0645}",

 # FC73 ARABIC LIGATURE TEH WITH NOON FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0646 ARABIC LETTER NOON
 "\x{FC73}" => "\x{062A}\x{0646}",

 # FC74 ARABIC LIGATURE TEH WITH ALEF MAKSURA FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC74}" => "\x{062A}\x{0649}",

 # FC75 ARABIC LIGATURE TEH WITH YEH FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC75}" => "\x{062A}\x{064A}",

 # FC76 ARABIC LIGATURE THEH WITH REH FINAL FORM
 # ->   062B ARABIC LETTER THEH
 #    + 0631 ARABIC LETTER REH
 "\x{FC76}" => "\x{062B}\x{0631}",

 # FC77 ARABIC LIGATURE THEH WITH ZAIN FINAL FORM
 # ->   062B ARABIC LETTER THEH
 #    + 0632 ARABIC LETTER ZAIN
 "\x{FC77}" => "\x{062B}\x{0632}",

 # FC78 ARABIC LIGATURE THEH WITH MEEM FINAL FORM
 # ->   062B ARABIC LETTER THEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC78}" => "\x{062B}\x{0645}",

 # FC79 ARABIC LIGATURE THEH WITH NOON FINAL FORM
 # ->   062B ARABIC LETTER THEH
 #    + 0646 ARABIC LETTER NOON
 "\x{FC79}" => "\x{062B}\x{0646}",

 # FC7A ARABIC LIGATURE THEH WITH ALEF MAKSURA FINAL FORM
 # ->   062B ARABIC LETTER THEH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC7A}" => "\x{062B}\x{0649}",

 # FC7B ARABIC LIGATURE THEH WITH YEH FINAL FORM
 # ->   062B ARABIC LETTER THEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC7B}" => "\x{062B}\x{064A}",

 # FC7C ARABIC LIGATURE FEH WITH ALEF MAKSURA FINAL FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC7C}" => "\x{0641}\x{0649}",

 # FC7D ARABIC LIGATURE FEH WITH YEH FINAL FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC7D}" => "\x{0641}\x{064A}",

 # FC7E ARABIC LIGATURE QAF WITH ALEF MAKSURA FINAL FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC7E}" => "\x{0642}\x{0649}",

 # FC7F ARABIC LIGATURE QAF WITH YEH FINAL FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 064A ARABIC LETTER YEH
 "\x{FC7F}" => "\x{0642}\x{064A}",

 # FC80 ARABIC LIGATURE KAF WITH ALEF FINAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0627 ARABIC LETTER ALEF
 "\x{FC80}" => "\x{0643}\x{0627}",

 # FC81 ARABIC LIGATURE KAF WITH LAM FINAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0644 ARABIC LETTER LAM
 "\x{FC81}" => "\x{0643}\x{0644}",

 # FC82 ARABIC LIGATURE KAF WITH MEEM FINAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC82}" => "\x{0643}\x{0645}",

 # FC83 ARABIC LIGATURE KAF WITH ALEF MAKSURA FINAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC83}" => "\x{0643}\x{0649}",

 # FC84 ARABIC LIGATURE KAF WITH YEH FINAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 064A ARABIC LETTER YEH
 "\x{FC84}" => "\x{0643}\x{064A}",

 # FC85 ARABIC LIGATURE LAM WITH MEEM FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC85}" => "\x{0644}\x{0645}",

 # FC86 ARABIC LIGATURE LAM WITH ALEF MAKSURA FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC86}" => "\x{0644}\x{0649}",

 # FC87 ARABIC LIGATURE LAM WITH YEH FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 064A ARABIC LETTER YEH
 "\x{FC87}" => "\x{0644}\x{064A}",

 # FC88 ARABIC LIGATURE MEEM WITH ALEF FINAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 0627 ARABIC LETTER ALEF
 "\x{FC88}" => "\x{0645}\x{0627}",

 # FC89 ARABIC LIGATURE MEEM WITH MEEM FINAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC89}" => "\x{0645}\x{0645}",

 # FC8A ARABIC LIGATURE NOON WITH REH FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0631 ARABIC LETTER REH
 "\x{FC8A}" => "\x{0646}\x{0631}",

 # FC8B ARABIC LIGATURE NOON WITH ZAIN FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0632 ARABIC LETTER ZAIN
 "\x{FC8B}" => "\x{0646}\x{0632}",

 # FC8C ARABIC LIGATURE NOON WITH MEEM FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC8C}" => "\x{0646}\x{0645}",

 # FC8D ARABIC LIGATURE NOON WITH NOON FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0646 ARABIC LETTER NOON
 "\x{FC8D}" => "\x{0646}\x{0646}",

 # FC8E ARABIC LIGATURE NOON WITH ALEF MAKSURA FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC8E}" => "\x{0646}\x{0649}",

 # FC8F ARABIC LIGATURE NOON WITH YEH FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 064A ARABIC LETTER YEH
 "\x{FC8F}" => "\x{0646}\x{064A}",

 # FC90 ARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF FINAL FORM
 # ->   0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC90}" => "\x{0649}",

 # FC91 ARABIC LIGATURE YEH WITH REH FINAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0631 ARABIC LETTER REH
 "\x{FC91}" => "\x{064A}\x{0631}",

 # FC92 ARABIC LIGATURE YEH WITH ZAIN FINAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0632 ARABIC LETTER ZAIN
 "\x{FC92}" => "\x{064A}\x{0632}",

 # FC93 ARABIC LIGATURE YEH WITH MEEM FINAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC93}" => "\x{064A}\x{0645}",

 # FC94 ARABIC LIGATURE YEH WITH NOON FINAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0646 ARABIC LETTER NOON
 "\x{FC94}" => "\x{064A}\x{0646}",

 # FC95 ARABIC LIGATURE YEH WITH ALEF MAKSURA FINAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FC95}" => "\x{064A}\x{0649}",

 # FC96 ARABIC LIGATURE YEH WITH YEH FINAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC96}" => "\x{064A}\x{064A}",

 # FC97 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH JEEM INITIAL FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FC97}" => "\x{062C}\x{064A}",

 # FC98 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HAH INITIAL FORM
 # ->   062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FC98}" => "\x{062D}\x{064A}",

 # FC99 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH KHAH INITIAL FORM
 # ->   062E ARABIC LETTER KHAH
 #    + 064A ARABIC LETTER YEH
 "\x{FC99}" => "\x{062E}\x{064A}",

 # FC9A ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FC9A}" => "\x{0645}\x{064A}",

 # FC9B ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH INITIAL FORM
 # ->   0647 ARABIC LETTER HEH
 #    + 064A ARABIC LETTER YEH
 "\x{FC9B}" => "\x{0647}\x{064A}",

 # FC9C ARABIC LIGATURE BEH WITH JEEM INITIAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 062C ARABIC LETTER JEEM
 "\x{FC9C}" => "\x{0628}\x{062C}",

 # FC9D ARABIC LIGATURE BEH WITH HAH INITIAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 062D ARABIC LETTER HAH
 "\x{FC9D}" => "\x{0628}\x{062D}",

 # FC9E ARABIC LIGATURE BEH WITH KHAH INITIAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 062E ARABIC LETTER KHAH
 "\x{FC9E}" => "\x{0628}\x{062E}",

 # FC9F ARABIC LIGATURE BEH WITH MEEM INITIAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FC9F}" => "\x{0628}\x{0645}",

 # FCA0 ARABIC LIGATURE BEH WITH HEH INITIAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 0647 ARABIC LETTER HEH
 "\x{FCA0}" => "\x{0628}\x{0647}",

 # FCA1 ARABIC LIGATURE TEH WITH JEEM INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062C ARABIC LETTER JEEM
 "\x{FCA1}" => "\x{062A}\x{062C}",

 # FCA2 ARABIC LIGATURE TEH WITH HAH INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062D ARABIC LETTER HAH
 "\x{FCA2}" => "\x{062A}\x{062D}",

 # FCA3 ARABIC LIGATURE TEH WITH KHAH INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062E ARABIC LETTER KHAH
 "\x{FCA3}" => "\x{062A}\x{062E}",

 # FCA4 ARABIC LIGATURE TEH WITH MEEM INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCA4}" => "\x{062A}\x{0645}",

 # FCA5 ARABIC LIGATURE TEH WITH HEH INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0647 ARABIC LETTER HEH
 "\x{FCA5}" => "\x{062A}\x{0647}",

 # FCA6 ARABIC LIGATURE THEH WITH MEEM INITIAL FORM
 # ->   062B ARABIC LETTER THEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCA6}" => "\x{062B}\x{0645}",

 # FCA7 ARABIC LIGATURE JEEM WITH HAH INITIAL FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FCA7}" => "\x{062C}\x{062D}",

 # FCA8 ARABIC LIGATURE JEEM WITH MEEM INITIAL FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCA8}" => "\x{062C}\x{0645}",

 # FCA9 ARABIC LIGATURE HAH WITH JEEM INITIAL FORM
 # ->   062D ARABIC LETTER HAH
 #    + 062C ARABIC LETTER JEEM
 "\x{FCA9}" => "\x{062D}\x{062C}",

 # FCAA ARABIC LIGATURE HAH WITH MEEM INITIAL FORM
 # ->   062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCAA}" => "\x{062D}\x{0645}",

 # FCAB ARABIC LIGATURE KHAH WITH JEEM INITIAL FORM
 # ->   062E ARABIC LETTER KHAH
 #    + 062C ARABIC LETTER JEEM
 "\x{FCAB}" => "\x{062E}\x{062C}",

 # FCAC ARABIC LIGATURE KHAH WITH MEEM INITIAL FORM
 # ->   062E ARABIC LETTER KHAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCAC}" => "\x{062E}\x{0645}",

 # FCAD ARABIC LIGATURE SEEN WITH JEEM INITIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062C ARABIC LETTER JEEM
 "\x{FCAD}" => "\x{0633}\x{062C}",

 # FCAE ARABIC LIGATURE SEEN WITH HAH INITIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062D ARABIC LETTER HAH
 "\x{FCAE}" => "\x{0633}\x{062D}",

 # FCAF ARABIC LIGATURE SEEN WITH KHAH INITIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062E ARABIC LETTER KHAH
 "\x{FCAF}" => "\x{0633}\x{062E}",

 # FCB0 ARABIC LIGATURE SEEN WITH MEEM INITIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCB0}" => "\x{0633}\x{0645}",

 # FCB1 ARABIC LIGATURE SAD WITH HAH INITIAL FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 062D ARABIC LETTER HAH
 "\x{FCB1}" => "\x{0635}\x{062D}",

 # FCB2 ARABIC LIGATURE SAD WITH KHAH INITIAL FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 062E ARABIC LETTER KHAH
 "\x{FCB2}" => "\x{0635}\x{062E}",

 # FCB3 ARABIC LIGATURE SAD WITH MEEM INITIAL FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCB3}" => "\x{0635}\x{0645}",

 # FCB4 ARABIC LIGATURE DAD WITH JEEM INITIAL FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 062C ARABIC LETTER JEEM
 "\x{FCB4}" => "\x{0636}\x{062C}",

 # FCB5 ARABIC LIGATURE DAD WITH HAH INITIAL FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 062D ARABIC LETTER HAH
 "\x{FCB5}" => "\x{0636}\x{062D}",

 # FCB6 ARABIC LIGATURE DAD WITH KHAH INITIAL FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 062E ARABIC LETTER KHAH
 "\x{FCB6}" => "\x{0636}\x{062E}",

 # FCB7 ARABIC LIGATURE DAD WITH MEEM INITIAL FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCB7}" => "\x{0636}\x{0645}",

 # FCB8 ARABIC LIGATURE TAH WITH HAH INITIAL FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 062D ARABIC LETTER HAH
 "\x{FCB8}" => "\x{0637}\x{062D}",

 # FCB9 ARABIC LIGATURE ZAH WITH MEEM INITIAL FORM
 # ->   0638 ARABIC LETTER ZAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCB9}" => "\x{0638}\x{0645}",

 # FCBA ARABIC LIGATURE AIN WITH JEEM INITIAL FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 062C ARABIC LETTER JEEM
 "\x{FCBA}" => "\x{0639}\x{062C}",

 # FCBB ARABIC LIGATURE AIN WITH MEEM INITIAL FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCBB}" => "\x{0639}\x{0645}",

 # FCBC ARABIC LIGATURE GHAIN WITH JEEM INITIAL FORM
 # ->   063A ARABIC LETTER GHAIN
 #    + 062C ARABIC LETTER JEEM
 "\x{FCBC}" => "\x{063A}\x{062C}",

 # FCBD ARABIC LIGATURE GHAIN WITH MEEM INITIAL FORM
 # ->   063A ARABIC LETTER GHAIN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCBD}" => "\x{063A}\x{0645}",

 # FCBE ARABIC LIGATURE FEH WITH JEEM INITIAL FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 062C ARABIC LETTER JEEM
 "\x{FCBE}" => "\x{0641}\x{062C}",

 # FCBF ARABIC LIGATURE FEH WITH HAH INITIAL FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 062D ARABIC LETTER HAH
 "\x{FCBF}" => "\x{0641}\x{062D}",

 # FCC0 ARABIC LIGATURE FEH WITH KHAH INITIAL FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 062E ARABIC LETTER KHAH
 "\x{FCC0}" => "\x{0641}\x{062E}",

 # FCC1 ARABIC LIGATURE FEH WITH MEEM INITIAL FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCC1}" => "\x{0641}\x{0645}",

 # FCC2 ARABIC LIGATURE QAF WITH HAH INITIAL FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 062D ARABIC LETTER HAH
 "\x{FCC2}" => "\x{0642}\x{062D}",

 # FCC3 ARABIC LIGATURE QAF WITH MEEM INITIAL FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCC3}" => "\x{0642}\x{0645}",

 # FCC4 ARABIC LIGATURE KAF WITH JEEM INITIAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 062C ARABIC LETTER JEEM
 "\x{FCC4}" => "\x{0643}\x{062C}",

 # FCC5 ARABIC LIGATURE KAF WITH HAH INITIAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 062D ARABIC LETTER HAH
 "\x{FCC5}" => "\x{0643}\x{062D}",

 # FCC6 ARABIC LIGATURE KAF WITH KHAH INITIAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 062E ARABIC LETTER KHAH
 "\x{FCC6}" => "\x{0643}\x{062E}",

 # FCC7 ARABIC LIGATURE KAF WITH LAM INITIAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0644 ARABIC LETTER LAM
 "\x{FCC7}" => "\x{0643}\x{0644}",

 # FCC8 ARABIC LIGATURE KAF WITH MEEM INITIAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCC8}" => "\x{0643}\x{0645}",

 # FCC9 ARABIC LIGATURE LAM WITH JEEM INITIAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062C ARABIC LETTER JEEM
 "\x{FCC9}" => "\x{0644}\x{062C}",

 # FCCA ARABIC LIGATURE LAM WITH HAH INITIAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062D ARABIC LETTER HAH
 "\x{FCCA}" => "\x{0644}\x{062D}",

 # FCCB ARABIC LIGATURE LAM WITH KHAH INITIAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062E ARABIC LETTER KHAH
 "\x{FCCB}" => "\x{0644}\x{062E}",

 # FCCC ARABIC LIGATURE LAM WITH MEEM INITIAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCCC}" => "\x{0644}\x{0645}",

 # FCCD ARABIC LIGATURE LAM WITH HEH INITIAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0647 ARABIC LETTER HEH
 "\x{FCCD}" => "\x{0644}\x{0647}",

 # FCCE ARABIC LIGATURE MEEM WITH JEEM INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062C ARABIC LETTER JEEM
 "\x{FCCE}" => "\x{0645}\x{062C}",

 # FCCF ARABIC LIGATURE MEEM WITH HAH INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FCCF}" => "\x{0645}\x{062D}",

 # FCD0 ARABIC LIGATURE MEEM WITH KHAH INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062E ARABIC LETTER KHAH
 "\x{FCD0}" => "\x{0645}\x{062E}",

 # FCD1 ARABIC LIGATURE MEEM WITH MEEM INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCD1}" => "\x{0645}\x{0645}",

 # FCD2 ARABIC LIGATURE NOON WITH JEEM INITIAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062C ARABIC LETTER JEEM
 "\x{FCD2}" => "\x{0646}\x{062C}",

 # FCD3 ARABIC LIGATURE NOON WITH HAH INITIAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062D ARABIC LETTER HAH
 "\x{FCD3}" => "\x{0646}\x{062D}",

 # FCD4 ARABIC LIGATURE NOON WITH KHAH INITIAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062E ARABIC LETTER KHAH
 "\x{FCD4}" => "\x{0646}\x{062E}",

 # FCD5 ARABIC LIGATURE NOON WITH MEEM INITIAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCD5}" => "\x{0646}\x{0645}",

 # FCD6 ARABIC LIGATURE NOON WITH HEH INITIAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0647 ARABIC LETTER HEH
 "\x{FCD6}" => "\x{0646}\x{0647}",

 # FCD7 ARABIC LIGATURE HEH WITH JEEM INITIAL FORM
 # ->   0647 ARABIC LETTER HEH
 #    + 062C ARABIC LETTER JEEM
 "\x{FCD7}" => "\x{0647}\x{062C}",

 # FCD8 ARABIC LIGATURE HEH WITH MEEM INITIAL FORM
 # ->   0647 ARABIC LETTER HEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCD8}" => "\x{0647}\x{0645}",

 # FCD9 ARABIC LIGATURE HEH WITH SUPERSCRIPT ALEF INITIAL FORM
 # ->   0647 ARABIC LETTER HEH
 "\x{FCD9}" => "\x{0647}",

 # FCDA ARABIC LIGATURE YEH WITH JEEM INITIAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 062C ARABIC LETTER JEEM
 "\x{FCDA}" => "\x{064A}\x{062C}",

 # FCDB ARABIC LIGATURE YEH WITH HAH INITIAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 062D ARABIC LETTER HAH
 "\x{FCDB}" => "\x{064A}\x{062D}",

 # FCDC ARABIC LIGATURE YEH WITH KHAH INITIAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 062E ARABIC LETTER KHAH
 "\x{FCDC}" => "\x{064A}\x{062E}",

 # FCDD ARABIC LIGATURE YEH WITH MEEM INITIAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCDD}" => "\x{064A}\x{0645}",

 # FCDE ARABIC LIGATURE YEH WITH HEH INITIAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0647 ARABIC LETTER HEH
 "\x{FCDE}" => "\x{064A}\x{0647}",

 # FCDF ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM MEDIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FCDF}" => "\x{0645}\x{064A}",

 # FCE0 ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH MEDIAL FORM
 # ->   0647 ARABIC LETTER HEH
 #    + 064A ARABIC LETTER YEH
 "\x{FCE0}" => "\x{0647}\x{064A}",

 # FCE1 ARABIC LIGATURE BEH WITH MEEM MEDIAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCE1}" => "\x{0628}\x{0645}",

 # FCE2 ARABIC LIGATURE BEH WITH HEH MEDIAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 0647 ARABIC LETTER HEH
 "\x{FCE2}" => "\x{0628}\x{0647}",

 # FCE3 ARABIC LIGATURE TEH WITH MEEM MEDIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCE3}" => "\x{062A}\x{0645}",

 # FCE4 ARABIC LIGATURE TEH WITH HEH MEDIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0647 ARABIC LETTER HEH
 "\x{FCE4}" => "\x{062A}\x{0647}",

 # FCE5 ARABIC LIGATURE THEH WITH MEEM MEDIAL FORM
 # ->   062B ARABIC LETTER THEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCE5}" => "\x{062B}\x{0645}",

 # FCE6 ARABIC LIGATURE THEH WITH HEH MEDIAL FORM
 # ->   062B ARABIC LETTER THEH
 #    + 0647 ARABIC LETTER HEH
 "\x{FCE6}" => "\x{062B}\x{0647}",

 # FCE7 ARABIC LIGATURE SEEN WITH MEEM MEDIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCE7}" => "\x{0633}\x{0645}",

 # FCE8 ARABIC LIGATURE SEEN WITH HEH MEDIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0647 ARABIC LETTER HEH
 "\x{FCE8}" => "\x{0633}\x{0647}",

 # FCE9 ARABIC LIGATURE SHEEN WITH MEEM MEDIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCE9}" => "\x{0634}\x{0645}",

 # FCEA ARABIC LIGATURE SHEEN WITH HEH MEDIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0647 ARABIC LETTER HEH
 "\x{FCEA}" => "\x{0634}\x{0647}",

 # FCEB ARABIC LIGATURE KAF WITH LAM MEDIAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0644 ARABIC LETTER LAM
 "\x{FCEB}" => "\x{0643}\x{0644}",

 # FCEC ARABIC LIGATURE KAF WITH MEEM MEDIAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCEC}" => "\x{0643}\x{0645}",

 # FCED ARABIC LIGATURE LAM WITH MEEM MEDIAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCED}" => "\x{0644}\x{0645}",

 # FCEE ARABIC LIGATURE NOON WITH MEEM MEDIAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCEE}" => "\x{0646}\x{0645}",

 # FCEF ARABIC LIGATURE NOON WITH HEH MEDIAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0647 ARABIC LETTER HEH
 "\x{FCEF}" => "\x{0646}\x{0647}",

 # FCF0 ARABIC LIGATURE YEH WITH MEEM MEDIAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FCF0}" => "\x{064A}\x{0645}",

 # FCF1 ARABIC LIGATURE YEH WITH HEH MEDIAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0647 ARABIC LETTER HEH
 "\x{FCF1}" => "\x{064A}\x{0647}",

 # FCF2 ARABIC LIGATURE SHADDA WITH FATHA MEDIAL FORM
 # ->   0640 ARABIC TATWEEL
 "\x{FCF2}" => "\x{0640}",

 # FCF3 ARABIC LIGATURE SHADDA WITH DAMMA MEDIAL FORM
 # ->   0640 ARABIC TATWEEL
 "\x{FCF3}" => "\x{0640}",

 # FCF4 ARABIC LIGATURE SHADDA WITH KASRA MEDIAL FORM
 # ->   0640 ARABIC TATWEEL
 "\x{FCF4}" => "\x{0640}",

 # FCF5 ARABIC LIGATURE TAH WITH ALEF MAKSURA ISOLATED FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FCF5}" => "\x{0637}\x{0649}",

 # FCF6 ARABIC LIGATURE TAH WITH YEH ISOLATED FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 064A ARABIC LETTER YEH
 "\x{FCF6}" => "\x{0637}\x{064A}",

 # FCF7 ARABIC LIGATURE AIN WITH ALEF MAKSURA ISOLATED FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FCF7}" => "\x{0639}\x{0649}",

 # FCF8 ARABIC LIGATURE AIN WITH YEH ISOLATED FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 064A ARABIC LETTER YEH
 "\x{FCF8}" => "\x{0639}\x{064A}",

 # FCF9 ARABIC LIGATURE GHAIN WITH ALEF MAKSURA ISOLATED FORM
 # ->   063A ARABIC LETTER GHAIN
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FCF9}" => "\x{063A}\x{0649}",

 # FCFA ARABIC LIGATURE GHAIN WITH YEH ISOLATED FORM
 # ->   063A ARABIC LETTER GHAIN
 #    + 064A ARABIC LETTER YEH
 "\x{FCFA}" => "\x{063A}\x{064A}",

 # FCFB ARABIC LIGATURE SEEN WITH ALEF MAKSURA ISOLATED FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FCFB}" => "\x{0633}\x{0649}",

 # FCFC ARABIC LIGATURE SEEN WITH YEH ISOLATED FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 064A ARABIC LETTER YEH
 "\x{FCFC}" => "\x{0633}\x{064A}",

 # FCFD ARABIC LIGATURE SHEEN WITH ALEF MAKSURA ISOLATED FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FCFD}" => "\x{0634}\x{0649}",

 # FCFE ARABIC LIGATURE SHEEN WITH YEH ISOLATED FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 064A ARABIC LETTER YEH
 "\x{FCFE}" => "\x{0634}\x{064A}",

 # FCFF ARABIC LIGATURE HAH WITH ALEF MAKSURA ISOLATED FORM
 # ->   062D ARABIC LETTER HAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FCFF}" => "\x{062D}\x{0649}",

 # FD00 ARABIC LIGATURE HAH WITH YEH ISOLATED FORM
 # ->   062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FD00}" => "\x{062D}\x{064A}",

 # FD01 ARABIC LIGATURE JEEM WITH ALEF MAKSURA ISOLATED FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD01}" => "\x{062C}\x{0649}",

 # FD02 ARABIC LIGATURE JEEM WITH YEH ISOLATED FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FD02}" => "\x{062C}\x{064A}",

 # FD03 ARABIC LIGATURE KHAH WITH ALEF MAKSURA ISOLATED FORM
 # ->   062E ARABIC LETTER KHAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD03}" => "\x{062E}\x{0649}",

 # FD04 ARABIC LIGATURE KHAH WITH YEH ISOLATED FORM
 # ->   062E ARABIC LETTER KHAH
 #    + 064A ARABIC LETTER YEH
 "\x{FD04}" => "\x{062E}\x{064A}",

 # FD05 ARABIC LIGATURE SAD WITH ALEF MAKSURA ISOLATED FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD05}" => "\x{0635}\x{0649}",

 # FD06 ARABIC LIGATURE SAD WITH YEH ISOLATED FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 064A ARABIC LETTER YEH
 "\x{FD06}" => "\x{0635}\x{064A}",

 # FD07 ARABIC LIGATURE DAD WITH ALEF MAKSURA ISOLATED FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD07}" => "\x{0636}\x{0649}",

 # FD08 ARABIC LIGATURE DAD WITH YEH ISOLATED FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 064A ARABIC LETTER YEH
 "\x{FD08}" => "\x{0636}\x{064A}",

 # FD09 ARABIC LIGATURE SHEEN WITH JEEM ISOLATED FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062C ARABIC LETTER JEEM
 "\x{FD09}" => "\x{0634}\x{062C}",

 # FD0A ARABIC LIGATURE SHEEN WITH HAH ISOLATED FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062D ARABIC LETTER HAH
 "\x{FD0A}" => "\x{0634}\x{062D}",

 # FD0B ARABIC LIGATURE SHEEN WITH KHAH ISOLATED FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062E ARABIC LETTER KHAH
 "\x{FD0B}" => "\x{0634}\x{062E}",

 # FD0C ARABIC LIGATURE SHEEN WITH MEEM ISOLATED FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD0C}" => "\x{0634}\x{0645}",

 # FD0D ARABIC LIGATURE SHEEN WITH REH ISOLATED FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0631 ARABIC LETTER REH
 "\x{FD0D}" => "\x{0634}\x{0631}",

 # FD0E ARABIC LIGATURE SEEN WITH REH ISOLATED FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0631 ARABIC LETTER REH
 "\x{FD0E}" => "\x{0633}\x{0631}",

 # FD0F ARABIC LIGATURE SAD WITH REH ISOLATED FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 0631 ARABIC LETTER REH
 "\x{FD0F}" => "\x{0635}\x{0631}",

 # FD10 ARABIC LIGATURE DAD WITH REH ISOLATED FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 0631 ARABIC LETTER REH
 "\x{FD10}" => "\x{0636}\x{0631}",

 # FD11 ARABIC LIGATURE TAH WITH ALEF MAKSURA FINAL FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD11}" => "\x{0637}\x{0649}",

 # FD12 ARABIC LIGATURE TAH WITH YEH FINAL FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 064A ARABIC LETTER YEH
 "\x{FD12}" => "\x{0637}\x{064A}",

 # FD13 ARABIC LIGATURE AIN WITH ALEF MAKSURA FINAL FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD13}" => "\x{0639}\x{0649}",

 # FD14 ARABIC LIGATURE AIN WITH YEH FINAL FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 064A ARABIC LETTER YEH
 "\x{FD14}" => "\x{0639}\x{064A}",

 # FD15 ARABIC LIGATURE GHAIN WITH ALEF MAKSURA FINAL FORM
 # ->   063A ARABIC LETTER GHAIN
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD15}" => "\x{063A}\x{0649}",

 # FD16 ARABIC LIGATURE GHAIN WITH YEH FINAL FORM
 # ->   063A ARABIC LETTER GHAIN
 #    + 064A ARABIC LETTER YEH
 "\x{FD16}" => "\x{063A}\x{064A}",

 # FD17 ARABIC LIGATURE SEEN WITH ALEF MAKSURA FINAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD17}" => "\x{0633}\x{0649}",

 # FD18 ARABIC LIGATURE SEEN WITH YEH FINAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 064A ARABIC LETTER YEH
 "\x{FD18}" => "\x{0633}\x{064A}",

 # FD19 ARABIC LIGATURE SHEEN WITH ALEF MAKSURA FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD19}" => "\x{0634}\x{0649}",

 # FD1A ARABIC LIGATURE SHEEN WITH YEH FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 064A ARABIC LETTER YEH
 "\x{FD1A}" => "\x{0634}\x{064A}",

 # FD1B ARABIC LIGATURE HAH WITH ALEF MAKSURA FINAL FORM
 # ->   062D ARABIC LETTER HAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD1B}" => "\x{062D}\x{0649}",

 # FD1C ARABIC LIGATURE HAH WITH YEH FINAL FORM
 # ->   062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FD1C}" => "\x{062D}\x{064A}",

 # FD1D ARABIC LIGATURE JEEM WITH ALEF MAKSURA FINAL FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD1D}" => "\x{062C}\x{0649}",

 # FD1E ARABIC LIGATURE JEEM WITH YEH FINAL FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FD1E}" => "\x{062C}\x{064A}",

 # FD1F ARABIC LIGATURE KHAH WITH ALEF MAKSURA FINAL FORM
 # ->   062E ARABIC LETTER KHAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD1F}" => "\x{062E}\x{0649}",

 # FD20 ARABIC LIGATURE KHAH WITH YEH FINAL FORM
 # ->   062E ARABIC LETTER KHAH
 #    + 064A ARABIC LETTER YEH
 "\x{FD20}" => "\x{062E}\x{064A}",

 # FD21 ARABIC LIGATURE SAD WITH ALEF MAKSURA FINAL FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD21}" => "\x{0635}\x{0649}",

 # FD22 ARABIC LIGATURE SAD WITH YEH FINAL FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 064A ARABIC LETTER YEH
 "\x{FD22}" => "\x{0635}\x{064A}",

 # FD23 ARABIC LIGATURE DAD WITH ALEF MAKSURA FINAL FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD23}" => "\x{0636}\x{0649}",

 # FD24 ARABIC LIGATURE DAD WITH YEH FINAL FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 064A ARABIC LETTER YEH
 "\x{FD24}" => "\x{0636}\x{064A}",

 # FD25 ARABIC LIGATURE SHEEN WITH JEEM FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062C ARABIC LETTER JEEM
 "\x{FD25}" => "\x{0634}\x{062C}",

 # FD26 ARABIC LIGATURE SHEEN WITH HAH FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062D ARABIC LETTER HAH
 "\x{FD26}" => "\x{0634}\x{062D}",

 # FD27 ARABIC LIGATURE SHEEN WITH KHAH FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062E ARABIC LETTER KHAH
 "\x{FD27}" => "\x{0634}\x{062E}",

 # FD28 ARABIC LIGATURE SHEEN WITH MEEM FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD28}" => "\x{0634}\x{0645}",

 # FD29 ARABIC LIGATURE SHEEN WITH REH FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0631 ARABIC LETTER REH
 "\x{FD29}" => "\x{0634}\x{0631}",

 # FD2A ARABIC LIGATURE SEEN WITH REH FINAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0631 ARABIC LETTER REH
 "\x{FD2A}" => "\x{0633}\x{0631}",

 # FD2B ARABIC LIGATURE SAD WITH REH FINAL FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 0631 ARABIC LETTER REH
 "\x{FD2B}" => "\x{0635}\x{0631}",

 # FD2C ARABIC LIGATURE DAD WITH REH FINAL FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 0631 ARABIC LETTER REH
 "\x{FD2C}" => "\x{0636}\x{0631}",

 # FD2D ARABIC LIGATURE SHEEN WITH JEEM INITIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062C ARABIC LETTER JEEM
 "\x{FD2D}" => "\x{0634}\x{062C}",

 # FD2E ARABIC LIGATURE SHEEN WITH HAH INITIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062D ARABIC LETTER HAH
 "\x{FD2E}" => "\x{0634}\x{062D}",

 # FD2F ARABIC LIGATURE SHEEN WITH KHAH INITIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062E ARABIC LETTER KHAH
 "\x{FD2F}" => "\x{0634}\x{062E}",

 # FD30 ARABIC LIGATURE SHEEN WITH MEEM INITIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD30}" => "\x{0634}\x{0645}",

 # FD31 ARABIC LIGATURE SEEN WITH HEH INITIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0647 ARABIC LETTER HEH
 "\x{FD31}" => "\x{0633}\x{0647}",

 # FD32 ARABIC LIGATURE SHEEN WITH HEH INITIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0647 ARABIC LETTER HEH
 "\x{FD32}" => "\x{0634}\x{0647}",

 # FD33 ARABIC LIGATURE TAH WITH MEEM INITIAL FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD33}" => "\x{0637}\x{0645}",

 # FD34 ARABIC LIGATURE SEEN WITH JEEM MEDIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062C ARABIC LETTER JEEM
 "\x{FD34}" => "\x{0633}\x{062C}",

 # FD35 ARABIC LIGATURE SEEN WITH HAH MEDIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062D ARABIC LETTER HAH
 "\x{FD35}" => "\x{0633}\x{062D}",

 # FD36 ARABIC LIGATURE SEEN WITH KHAH MEDIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062E ARABIC LETTER KHAH
 "\x{FD36}" => "\x{0633}\x{062E}",

 # FD37 ARABIC LIGATURE SHEEN WITH JEEM MEDIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062C ARABIC LETTER JEEM
 "\x{FD37}" => "\x{0634}\x{062C}",

 # FD38 ARABIC LIGATURE SHEEN WITH HAH MEDIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062D ARABIC LETTER HAH
 "\x{FD38}" => "\x{0634}\x{062D}",

 # FD39 ARABIC LIGATURE SHEEN WITH KHAH MEDIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062E ARABIC LETTER KHAH
 "\x{FD39}" => "\x{0634}\x{062E}",

 # FD3A ARABIC LIGATURE TAH WITH MEEM MEDIAL FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD3A}" => "\x{0637}\x{0645}",

 # FD3B ARABIC LIGATURE ZAH WITH MEEM MEDIAL FORM
 # ->   0638 ARABIC LETTER ZAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD3B}" => "\x{0638}\x{0645}",

 # FD3C ARABIC LIGATURE ALEF WITH FATHATAN FINAL FORM
 # ->   0627 ARABIC LETTER ALEF
 "\x{FD3C}" => "\x{0627}",

 # FD3D ARABIC LIGATURE ALEF WITH FATHATAN ISOLATED FORM
 # ->   0627 ARABIC LETTER ALEF
 "\x{FD3D}" => "\x{0627}",

 # FD50 ARABIC LIGATURE TEH WITH JEEM WITH MEEM INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD50}" => "\x{062A}\x{062C}\x{0645}",

 # FD51 ARABIC LIGATURE TEH WITH HAH WITH JEEM FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062D ARABIC LETTER HAH
 #    + 062C ARABIC LETTER JEEM
 "\x{FD51}" => "\x{062A}\x{062D}\x{062C}",

 # FD52 ARABIC LIGATURE TEH WITH HAH WITH JEEM INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062D ARABIC LETTER HAH
 #    + 062C ARABIC LETTER JEEM
 "\x{FD52}" => "\x{062A}\x{062D}\x{062C}",

 # FD53 ARABIC LIGATURE TEH WITH HAH WITH MEEM INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD53}" => "\x{062A}\x{062D}\x{0645}",

 # FD54 ARABIC LIGATURE TEH WITH KHAH WITH MEEM INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062E ARABIC LETTER KHAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD54}" => "\x{062A}\x{062E}\x{0645}",

 # FD55 ARABIC LIGATURE TEH WITH MEEM WITH JEEM INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0645 ARABIC LETTER MEEM
 #    + 062C ARABIC LETTER JEEM
 "\x{FD55}" => "\x{062A}\x{0645}\x{062C}",

 # FD56 ARABIC LIGATURE TEH WITH MEEM WITH HAH INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD56}" => "\x{062A}\x{0645}\x{062D}",

 # FD57 ARABIC LIGATURE TEH WITH MEEM WITH KHAH INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0645 ARABIC LETTER MEEM
 #    + 062E ARABIC LETTER KHAH
 "\x{FD57}" => "\x{062A}\x{0645}\x{062E}",

 # FD58 ARABIC LIGATURE JEEM WITH MEEM WITH HAH FINAL FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD58}" => "\x{062C}\x{0645}\x{062D}",

 # FD59 ARABIC LIGATURE JEEM WITH MEEM WITH HAH INITIAL FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD59}" => "\x{062C}\x{0645}\x{062D}",

 # FD5A ARABIC LIGATURE HAH WITH MEEM WITH YEH FINAL FORM
 # ->   062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FD5A}" => "\x{062D}\x{0645}\x{064A}",

 # FD5B ARABIC LIGATURE HAH WITH MEEM WITH ALEF MAKSURA FINAL FORM
 # ->   062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD5B}" => "\x{062D}\x{0645}\x{0649}",

 # FD5C ARABIC LIGATURE SEEN WITH HAH WITH JEEM INITIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062D ARABIC LETTER HAH
 #    + 062C ARABIC LETTER JEEM
 "\x{FD5C}" => "\x{0633}\x{062D}\x{062C}",

 # FD5D ARABIC LIGATURE SEEN WITH JEEM WITH HAH INITIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062C ARABIC LETTER JEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD5D}" => "\x{0633}\x{062C}\x{062D}",

 # FD5E ARABIC LIGATURE SEEN WITH JEEM WITH ALEF MAKSURA FINAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062C ARABIC LETTER JEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD5E}" => "\x{0633}\x{062C}\x{0649}",

 # FD5F ARABIC LIGATURE SEEN WITH MEEM WITH HAH FINAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD5F}" => "\x{0633}\x{0645}\x{062D}",

 # FD60 ARABIC LIGATURE SEEN WITH MEEM WITH HAH INITIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD60}" => "\x{0633}\x{0645}\x{062D}",

 # FD61 ARABIC LIGATURE SEEN WITH MEEM WITH JEEM INITIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0645 ARABIC LETTER MEEM
 #    + 062C ARABIC LETTER JEEM
 "\x{FD61}" => "\x{0633}\x{0645}\x{062C}",

 # FD62 ARABIC LIGATURE SEEN WITH MEEM WITH MEEM FINAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD62}" => "\x{0633}\x{0645}\x{0645}",

 # FD63 ARABIC LIGATURE SEEN WITH MEEM WITH MEEM INITIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD63}" => "\x{0633}\x{0645}\x{0645}",

 # FD64 ARABIC LIGATURE SAD WITH HAH WITH HAH FINAL FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 062D ARABIC LETTER HAH
 #    + 062D ARABIC LETTER HAH
 "\x{FD64}" => "\x{0635}\x{062D}\x{062D}",

 # FD65 ARABIC LIGATURE SAD WITH HAH WITH HAH INITIAL FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 062D ARABIC LETTER HAH
 #    + 062D ARABIC LETTER HAH
 "\x{FD65}" => "\x{0635}\x{062D}\x{062D}",

 # FD66 ARABIC LIGATURE SAD WITH MEEM WITH MEEM FINAL FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD66}" => "\x{0635}\x{0645}\x{0645}",

 # FD67 ARABIC LIGATURE SHEEN WITH HAH WITH MEEM FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD67}" => "\x{0634}\x{062D}\x{0645}",

 # FD68 ARABIC LIGATURE SHEEN WITH HAH WITH MEEM INITIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD68}" => "\x{0634}\x{062D}\x{0645}",

 # FD69 ARABIC LIGATURE SHEEN WITH JEEM WITH YEH FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062C ARABIC LETTER JEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FD69}" => "\x{0634}\x{062C}\x{064A}",

 # FD6A ARABIC LIGATURE SHEEN WITH MEEM WITH KHAH FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0645 ARABIC LETTER MEEM
 #    + 062E ARABIC LETTER KHAH
 "\x{FD6A}" => "\x{0634}\x{0645}\x{062E}",

 # FD6B ARABIC LIGATURE SHEEN WITH MEEM WITH KHAH INITIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0645 ARABIC LETTER MEEM
 #    + 062E ARABIC LETTER KHAH
 "\x{FD6B}" => "\x{0634}\x{0645}\x{062E}",

 # FD6C ARABIC LIGATURE SHEEN WITH MEEM WITH MEEM FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD6C}" => "\x{0634}\x{0645}\x{0645}",

 # FD6D ARABIC LIGATURE SHEEN WITH MEEM WITH MEEM INITIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD6D}" => "\x{0634}\x{0645}\x{0645}",

 # FD6E ARABIC LIGATURE DAD WITH HAH WITH ALEF MAKSURA FINAL FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 062D ARABIC LETTER HAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD6E}" => "\x{0636}\x{062D}\x{0649}",

 # FD6F ARABIC LIGATURE DAD WITH KHAH WITH MEEM FINAL FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 062E ARABIC LETTER KHAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD6F}" => "\x{0636}\x{062E}\x{0645}",

 # FD70 ARABIC LIGATURE DAD WITH KHAH WITH MEEM INITIAL FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 062E ARABIC LETTER KHAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD70}" => "\x{0636}\x{062E}\x{0645}",

 # FD71 ARABIC LIGATURE TAH WITH MEEM WITH HAH FINAL FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD71}" => "\x{0637}\x{0645}\x{062D}",

 # FD72 ARABIC LIGATURE TAH WITH MEEM WITH HAH INITIAL FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD72}" => "\x{0637}\x{0645}\x{062D}",

 # FD73 ARABIC LIGATURE TAH WITH MEEM WITH MEEM INITIAL FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD73}" => "\x{0637}\x{0645}\x{0645}",

 # FD74 ARABIC LIGATURE TAH WITH MEEM WITH YEH FINAL FORM
 # ->   0637 ARABIC LETTER TAH
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FD74}" => "\x{0637}\x{0645}\x{064A}",

 # FD75 ARABIC LIGATURE AIN WITH JEEM WITH MEEM FINAL FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD75}" => "\x{0639}\x{062C}\x{0645}",

 # FD76 ARABIC LIGATURE AIN WITH MEEM WITH MEEM FINAL FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD76}" => "\x{0639}\x{0645}\x{0645}",

 # FD77 ARABIC LIGATURE AIN WITH MEEM WITH MEEM INITIAL FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD77}" => "\x{0639}\x{0645}\x{0645}",

 # FD78 ARABIC LIGATURE AIN WITH MEEM WITH ALEF MAKSURA FINAL FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 0645 ARABIC LETTER MEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD78}" => "\x{0639}\x{0645}\x{0649}",

 # FD79 ARABIC LIGATURE GHAIN WITH MEEM WITH MEEM FINAL FORM
 # ->   063A ARABIC LETTER GHAIN
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD79}" => "\x{063A}\x{0645}\x{0645}",

 # FD7A ARABIC LIGATURE GHAIN WITH MEEM WITH YEH FINAL FORM
 # ->   063A ARABIC LETTER GHAIN
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FD7A}" => "\x{063A}\x{0645}\x{064A}",

 # FD7B ARABIC LIGATURE GHAIN WITH MEEM WITH ALEF MAKSURA FINAL FORM
 # ->   063A ARABIC LETTER GHAIN
 #    + 0645 ARABIC LETTER MEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD7B}" => "\x{063A}\x{0645}\x{0649}",

 # FD7C ARABIC LIGATURE FEH WITH KHAH WITH MEEM FINAL FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 062E ARABIC LETTER KHAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD7C}" => "\x{0641}\x{062E}\x{0645}",

 # FD7D ARABIC LIGATURE FEH WITH KHAH WITH MEEM INITIAL FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 062E ARABIC LETTER KHAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD7D}" => "\x{0641}\x{062E}\x{0645}",

 # FD7E ARABIC LIGATURE QAF WITH MEEM WITH HAH FINAL FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD7E}" => "\x{0642}\x{0645}\x{062D}",

 # FD7F ARABIC LIGATURE QAF WITH MEEM WITH MEEM FINAL FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD7F}" => "\x{0642}\x{0645}\x{0645}",

 # FD80 ARABIC LIGATURE LAM WITH HAH WITH MEEM FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD80}" => "\x{0644}\x{062D}\x{0645}",

 # FD81 ARABIC LIGATURE LAM WITH HAH WITH YEH FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FD81}" => "\x{0644}\x{062D}\x{064A}",

 # FD82 ARABIC LIGATURE LAM WITH HAH WITH ALEF MAKSURA FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062D ARABIC LETTER HAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD82}" => "\x{0644}\x{062D}\x{0649}",

 # FD83 ARABIC LIGATURE LAM WITH JEEM WITH JEEM INITIAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062C ARABIC LETTER JEEM
 #    + 062C ARABIC LETTER JEEM
 "\x{FD83}" => "\x{0644}\x{062C}\x{062C}",

 # FD84 ARABIC LIGATURE LAM WITH JEEM WITH JEEM FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062C ARABIC LETTER JEEM
 #    + 062C ARABIC LETTER JEEM
 "\x{FD84}" => "\x{0644}\x{062C}\x{062C}",

 # FD85 ARABIC LIGATURE LAM WITH KHAH WITH MEEM FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062E ARABIC LETTER KHAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD85}" => "\x{0644}\x{062E}\x{0645}",

 # FD86 ARABIC LIGATURE LAM WITH KHAH WITH MEEM INITIAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062E ARABIC LETTER KHAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD86}" => "\x{0644}\x{062E}\x{0645}",

 # FD87 ARABIC LIGATURE LAM WITH MEEM WITH HAH FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD87}" => "\x{0644}\x{0645}\x{062D}",

 # FD88 ARABIC LIGATURE LAM WITH MEEM WITH HAH INITIAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD88}" => "\x{0644}\x{0645}\x{062D}",

 # FD89 ARABIC LIGATURE MEEM WITH HAH WITH JEEM INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 #    + 062C ARABIC LETTER JEEM
 "\x{FD89}" => "\x{0645}\x{062D}\x{062C}",

 # FD8A ARABIC LIGATURE MEEM WITH HAH WITH MEEM INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD8A}" => "\x{0645}\x{062D}\x{0645}",

 # FD8B ARABIC LIGATURE MEEM WITH HAH WITH YEH FINAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FD8B}" => "\x{0645}\x{062D}\x{064A}",

 # FD8C ARABIC LIGATURE MEEM WITH JEEM WITH HAH INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062C ARABIC LETTER JEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FD8C}" => "\x{0645}\x{062C}\x{062D}",

 # FD8D ARABIC LIGATURE MEEM WITH JEEM WITH MEEM INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD8D}" => "\x{0645}\x{062C}\x{0645}",

 # FD8E ARABIC LIGATURE MEEM WITH KHAH WITH JEEM INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062E ARABIC LETTER KHAH
 #    + 062C ARABIC LETTER JEEM
 "\x{FD8E}" => "\x{0645}\x{062E}\x{062C}",

 # FD8F ARABIC LIGATURE MEEM WITH KHAH WITH MEEM INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062E ARABIC LETTER KHAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD8F}" => "\x{0645}\x{062E}\x{0645}",

 # FD92 ARABIC LIGATURE MEEM WITH JEEM WITH KHAH INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062C ARABIC LETTER JEEM
 #    + 062E ARABIC LETTER KHAH
 "\x{FD92}" => "\x{0645}\x{062C}\x{062E}",

 # FD93 ARABIC LIGATURE HEH WITH MEEM WITH JEEM INITIAL FORM
 # ->   0647 ARABIC LETTER HEH
 #    + 0645 ARABIC LETTER MEEM
 #    + 062C ARABIC LETTER JEEM
 "\x{FD93}" => "\x{0647}\x{0645}\x{062C}",

 # FD94 ARABIC LIGATURE HEH WITH MEEM WITH MEEM INITIAL FORM
 # ->   0647 ARABIC LETTER HEH
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD94}" => "\x{0647}\x{0645}\x{0645}",

 # FD95 ARABIC LIGATURE NOON WITH HAH WITH MEEM INITIAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD95}" => "\x{0646}\x{062D}\x{0645}",

 # FD96 ARABIC LIGATURE NOON WITH HAH WITH ALEF MAKSURA FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062D ARABIC LETTER HAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD96}" => "\x{0646}\x{062D}\x{0649}",

 # FD97 ARABIC LIGATURE NOON WITH JEEM WITH MEEM FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD97}" => "\x{0646}\x{062C}\x{0645}",

 # FD98 ARABIC LIGATURE NOON WITH JEEM WITH MEEM INITIAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD98}" => "\x{0646}\x{062C}\x{0645}",

 # FD99 ARABIC LIGATURE NOON WITH JEEM WITH ALEF MAKSURA FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062C ARABIC LETTER JEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD99}" => "\x{0646}\x{062C}\x{0649}",

 # FD9A ARABIC LIGATURE NOON WITH MEEM WITH YEH FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FD9A}" => "\x{0646}\x{0645}\x{064A}",

 # FD9B ARABIC LIGATURE NOON WITH MEEM WITH ALEF MAKSURA FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 0645 ARABIC LETTER MEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FD9B}" => "\x{0646}\x{0645}\x{0649}",

 # FD9C ARABIC LIGATURE YEH WITH MEEM WITH MEEM FINAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD9C}" => "\x{064A}\x{0645}\x{0645}",

 # FD9D ARABIC LIGATURE YEH WITH MEEM WITH MEEM INITIAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FD9D}" => "\x{064A}\x{0645}\x{0645}",

 # FD9E ARABIC LIGATURE BEH WITH KHAH WITH YEH FINAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 062E ARABIC LETTER KHAH
 #    + 064A ARABIC LETTER YEH
 "\x{FD9E}" => "\x{0628}\x{062E}\x{064A}",

 # FD9F ARABIC LIGATURE TEH WITH JEEM WITH YEH FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062C ARABIC LETTER JEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FD9F}" => "\x{062A}\x{062C}\x{064A}",

 # FDA0 ARABIC LIGATURE TEH WITH JEEM WITH ALEF MAKSURA FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062C ARABIC LETTER JEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FDA0}" => "\x{062A}\x{062C}\x{0649}",

 # FDA1 ARABIC LIGATURE TEH WITH KHAH WITH YEH FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062E ARABIC LETTER KHAH
 #    + 064A ARABIC LETTER YEH
 "\x{FDA1}" => "\x{062A}\x{062E}\x{064A}",

 # FDA2 ARABIC LIGATURE TEH WITH KHAH WITH ALEF MAKSURA FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 062E ARABIC LETTER KHAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FDA2}" => "\x{062A}\x{062E}\x{0649}",

 # FDA3 ARABIC LIGATURE TEH WITH MEEM WITH YEH FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDA3}" => "\x{062A}\x{0645}\x{064A}",

 # FDA4 ARABIC LIGATURE TEH WITH MEEM WITH ALEF MAKSURA FINAL FORM
 # ->   062A ARABIC LETTER TEH
 #    + 0645 ARABIC LETTER MEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FDA4}" => "\x{062A}\x{0645}\x{0649}",

 # FDA5 ARABIC LIGATURE JEEM WITH MEEM WITH YEH FINAL FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDA5}" => "\x{062C}\x{0645}\x{064A}",

 # FDA6 ARABIC LIGATURE JEEM WITH HAH WITH ALEF MAKSURA FINAL FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 062D ARABIC LETTER HAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FDA6}" => "\x{062C}\x{062D}\x{0649}",

 # FDA7 ARABIC LIGATURE JEEM WITH MEEM WITH ALEF MAKSURA FINAL FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FDA7}" => "\x{062C}\x{0645}\x{0649}",

 # FDA8 ARABIC LIGATURE SEEN WITH KHAH WITH ALEF MAKSURA FINAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062E ARABIC LETTER KHAH
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FDA8}" => "\x{0633}\x{062E}\x{0649}",

 # FDA9 ARABIC LIGATURE SAD WITH HAH WITH YEH FINAL FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FDA9}" => "\x{0635}\x{062D}\x{064A}",

 # FDAA ARABIC LIGATURE SHEEN WITH HAH WITH YEH FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 #    + 062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FDAA}" => "\x{0634}\x{062D}\x{064A}",

 # FDAB ARABIC LIGATURE DAD WITH HAH WITH YEH FINAL FORM
 # ->   0636 ARABIC LETTER DAD
 #    + 062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FDAB}" => "\x{0636}\x{062D}\x{064A}",

 # FDAC ARABIC LIGATURE LAM WITH JEEM WITH YEH FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062C ARABIC LETTER JEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDAC}" => "\x{0644}\x{062C}\x{064A}",

 # FDAD ARABIC LIGATURE LAM WITH MEEM WITH YEH FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDAD}" => "\x{0644}\x{0645}\x{064A}",

 # FDAE ARABIC LIGATURE YEH WITH HAH WITH YEH FINAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FDAE}" => "\x{064A}\x{062D}\x{064A}",

 # FDAF ARABIC LIGATURE YEH WITH JEEM WITH YEH FINAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 062C ARABIC LETTER JEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDAF}" => "\x{064A}\x{062C}\x{064A}",

 # FDB0 ARABIC LIGATURE YEH WITH MEEM WITH YEH FINAL FORM
 # ->   064A ARABIC LETTER YEH
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDB0}" => "\x{064A}\x{0645}\x{064A}",

 # FDB1 ARABIC LIGATURE MEEM WITH MEEM WITH YEH FINAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDB1}" => "\x{0645}\x{0645}\x{064A}",

 # FDB2 ARABIC LIGATURE QAF WITH MEEM WITH YEH FINAL FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDB2}" => "\x{0642}\x{0645}\x{064A}",

 # FDB3 ARABIC LIGATURE NOON WITH HAH WITH YEH FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FDB3}" => "\x{0646}\x{062D}\x{064A}",

 # FDB4 ARABIC LIGATURE QAF WITH MEEM WITH HAH INITIAL FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FDB4}" => "\x{0642}\x{0645}\x{062D}",

 # FDB5 ARABIC LIGATURE LAM WITH HAH WITH MEEM INITIAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 "\x{FDB5}" => "\x{0644}\x{062D}\x{0645}",

 # FDB6 ARABIC LIGATURE AIN WITH MEEM WITH YEH FINAL FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDB6}" => "\x{0639}\x{0645}\x{064A}",

 # FDB7 ARABIC LIGATURE KAF WITH MEEM WITH YEH FINAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDB7}" => "\x{0643}\x{0645}\x{064A}",

 # FDB8 ARABIC LIGATURE NOON WITH JEEM WITH HAH INITIAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062C ARABIC LETTER JEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FDB8}" => "\x{0646}\x{062C}\x{062D}",

 # FDB9 ARABIC LIGATURE MEEM WITH KHAH WITH YEH FINAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062E ARABIC LETTER KHAH
 #    + 064A ARABIC LETTER YEH
 "\x{FDB9}" => "\x{0645}\x{062E}\x{064A}",

 # FDBA ARABIC LIGATURE LAM WITH JEEM WITH MEEM INITIAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FDBA}" => "\x{0644}\x{062C}\x{0645}",

 # FDBB ARABIC LIGATURE KAF WITH MEEM WITH MEEM FINAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FDBB}" => "\x{0643}\x{0645}\x{0645}",

 # FDBC ARABIC LIGATURE LAM WITH JEEM WITH MEEM FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FDBC}" => "\x{0644}\x{062C}\x{0645}",

 # FDBD ARABIC LIGATURE NOON WITH JEEM WITH HAH FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062C ARABIC LETTER JEEM
 #    + 062D ARABIC LETTER HAH
 "\x{FDBD}" => "\x{0646}\x{062C}\x{062D}",

 # FDBE ARABIC LIGATURE JEEM WITH HAH WITH YEH FINAL FORM
 # ->   062C ARABIC LETTER JEEM
 #    + 062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FDBE}" => "\x{062C}\x{062D}\x{064A}",

 # FDBF ARABIC LIGATURE HAH WITH JEEM WITH YEH FINAL FORM
 # ->   062D ARABIC LETTER HAH
 #    + 062C ARABIC LETTER JEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDBF}" => "\x{062D}\x{062C}\x{064A}",

 # FDC0 ARABIC LIGATURE MEEM WITH JEEM WITH YEH FINAL FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062C ARABIC LETTER JEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDC0}" => "\x{0645}\x{062C}\x{064A}",

 # FDC1 ARABIC LIGATURE FEH WITH MEEM WITH YEH FINAL FORM
 # ->   0641 ARABIC LETTER FEH
 #    + 0645 ARABIC LETTER MEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDC1}" => "\x{0641}\x{0645}\x{064A}",

 # FDC2 ARABIC LIGATURE BEH WITH HAH WITH YEH FINAL FORM
 # ->   0628 ARABIC LETTER BEH
 #    + 062D ARABIC LETTER HAH
 #    + 064A ARABIC LETTER YEH
 "\x{FDC2}" => "\x{0628}\x{062D}\x{064A}",

 # FDC3 ARABIC LIGATURE KAF WITH MEEM WITH MEEM INITIAL FORM
 # ->   0643 ARABIC LETTER KAF
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FDC3}" => "\x{0643}\x{0645}\x{0645}",

 # FDC4 ARABIC LIGATURE AIN WITH JEEM WITH MEEM INITIAL FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 062C ARABIC LETTER JEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FDC4}" => "\x{0639}\x{062C}\x{0645}",

 # FDC5 ARABIC LIGATURE SAD WITH MEEM WITH MEEM INITIAL FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 0645 ARABIC LETTER MEEM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FDC5}" => "\x{0635}\x{0645}\x{0645}",

 # FDC6 ARABIC LIGATURE SEEN WITH KHAH WITH YEH FINAL FORM
 # ->   0633 ARABIC LETTER SEEN
 #    + 062E ARABIC LETTER KHAH
 #    + 064A ARABIC LETTER YEH
 "\x{FDC6}" => "\x{0633}\x{062E}\x{064A}",

 # FDC7 ARABIC LIGATURE NOON WITH JEEM WITH YEH FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 #    + 062C ARABIC LETTER JEEM
 #    + 064A ARABIC LETTER YEH
 "\x{FDC7}" => "\x{0646}\x{062C}\x{064A}",

 # FDF0 ARABIC LIGATURE SALLA USED AS KORANIC STOP SIGN ISOLATED FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 0644 ARABIC LETTER LAM
 #    + 06D2 ARABIC LETTER YEH BARREE
 "\x{FDF0}" => "\x{0635}\x{0644}\x{06D2}",

 # FDF1 ARABIC LIGATURE QALA USED AS KORANIC STOP SIGN ISOLATED FORM
 # ->   0642 ARABIC LETTER QAF
 #    + 0644 ARABIC LETTER LAM
 #    + 06D2 ARABIC LETTER YEH BARREE
 "\x{FDF1}" => "\x{0642}\x{0644}\x{06D2}",

 # FDF2 ARABIC LIGATURE ALLAH ISOLATED FORM
 # ->   0627 ARABIC LETTER ALEF
 #    + 0644 ARABIC LETTER LAM
 #    + 0644 ARABIC LETTER LAM
 #    + 0647 ARABIC LETTER HEH
 "\x{FDF2}" => "\x{0627}\x{0644}\x{0644}\x{0647}",

 # FDF3 ARABIC LIGATURE AKBAR ISOLATED FORM
 # ->   0627 ARABIC LETTER ALEF
 #    + 0643 ARABIC LETTER KAF
 #    + 0628 ARABIC LETTER BEH
 #    + 0631 ARABIC LETTER REH
 "\x{FDF3}" => "\x{0627}\x{0643}\x{0628}\x{0631}",

 # FDF4 ARABIC LIGATURE MOHAMMAD ISOLATED FORM
 # ->   0645 ARABIC LETTER MEEM
 #    + 062D ARABIC LETTER HAH
 #    + 0645 ARABIC LETTER MEEM
 #    + 062F ARABIC LETTER DAL
 "\x{FDF4}" => "\x{0645}\x{062D}\x{0645}\x{062F}",

 # FDF5 ARABIC LIGATURE SALAM ISOLATED FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 0644 ARABIC LETTER LAM
 #    + 0639 ARABIC LETTER AIN
 #    + 0645 ARABIC LETTER MEEM
 "\x{FDF5}" => "\x{0635}\x{0644}\x{0639}\x{0645}",

 # FDF6 ARABIC LIGATURE RASOUL ISOLATED FORM
 # ->   0631 ARABIC LETTER REH
 #    + 0633 ARABIC LETTER SEEN
 #    + 0648 ARABIC LETTER WAW
 #    + 0644 ARABIC LETTER LAM
 "\x{FDF6}" => "\x{0631}\x{0633}\x{0648}\x{0644}",

 # FDF7 ARABIC LIGATURE ALAYHE ISOLATED FORM
 # ->   0639 ARABIC LETTER AIN
 #    + 0644 ARABIC LETTER LAM
 #    + 064A ARABIC LETTER YEH
 #    + 0647 ARABIC LETTER HEH
 "\x{FDF7}" => "\x{0639}\x{0644}\x{064A}\x{0647}",

 # FDF8 ARABIC LIGATURE WASALLAM ISOLATED FORM
 # ->   0648 ARABIC LETTER WAW
 #    + 0633 ARABIC LETTER SEEN
 #    + 0644 ARABIC LETTER LAM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FDF8}" => "\x{0648}\x{0633}\x{0644}\x{0645}",

 # FDF9 ARABIC LIGATURE SALLA ISOLATED FORM
 # ->   0635 ARABIC LETTER SAD
 #    + 0644 ARABIC LETTER LAM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 "\x{FDF9}" => "\x{0635}\x{0644}\x{0649}",

 # FDFA ARABIC LIGATURE SALLALLAHOU ALAYHE WASALLAM
 # ->   0635 ARABIC LETTER SAD
 #    + 0644 ARABIC LETTER LAM
 #    + 0649 ARABIC LETTER ALEF MAKSURA
 #    + 0020 SPACE
 #    + 0627 ARABIC LETTER ALEF
 #    + 0644 ARABIC LETTER LAM
 #    + 0644 ARABIC LETTER LAM
 #    + 0647 ARABIC LETTER HEH
 #    + 0020 SPACE
 #    + 0639 ARABIC LETTER AIN
 #    + 0644 ARABIC LETTER LAM
 #    + 064A ARABIC LETTER YEH
 #    + 0647 ARABIC LETTER HEH
 #    + 0020 SPACE
 #    + 0648 ARABIC LETTER WAW
 #    + 0633 ARABIC LETTER SEEN
 #    + 0644 ARABIC LETTER LAM
 #    + 0645 ARABIC LETTER MEEM
 "\x{FDFA}" => "\x{0635}\x{0644}\x{0649} \x{0627}\x{0644}\x{0644}\x{0647} \x{0639}\x{0644}\x{064A}\x{0647} \x{0648}\x{0633}\x{0644}\x{0645}",

 # FDFB ARABIC LIGATURE JALLAJALALOUHOU
 # ->   062C ARABIC LETTER JEEM
 #    + 0644 ARABIC LETTER LAM
 #    + 0020 SPACE
 #    + 062C ARABIC LETTER JEEM
 #    + 0644 ARABIC LETTER LAM
 #    + 0627 ARABIC LETTER ALEF
 #    + 0644 ARABIC LETTER LAM
 #    + 0647 ARABIC LETTER HEH
 "\x{FDFB}" => "\x{062C}\x{0644} \x{062C}\x{0644}\x{0627}\x{0644}\x{0647}",

 # FDFC RIAL SIGN
 # ->   0631 ARABIC LETTER REH
 #    + 06CC ARABIC LETTER FARSI YEH
 #    + 0627 ARABIC LETTER ALEF
 #    + 0644 ARABIC LETTER LAM
 "\x{FDFC}" => "\x{0631}\x{06CC}\x{0627}\x{0644}",

 # FE30 PRESENTATION FORM FOR VERTICAL TWO DOT LEADER
 # ->   002E FULL STOP
 #    + 002E FULL STOP
 "\x{FE30}" => "..",

 # FE31 PRESENTATION FORM FOR VERTICAL EM DASH
 # ->   2014 EM DASH
 "\x{FE31}" => "\x{2014}",

 # FE32 PRESENTATION FORM FOR VERTICAL EN DASH
 # ->   2013 EN DASH
 "\x{FE32}" => "\x{2013}",

 # FE33 PRESENTATION FORM FOR VERTICAL LOW LINE
 # ->   005F LOW LINE
 "\x{FE33}" => "_",

 # FE34 PRESENTATION FORM FOR VERTICAL WAVY LOW LINE
 # ->   005F LOW LINE
 "\x{FE34}" => "_",

 # FE35 PRESENTATION FORM FOR VERTICAL LEFT PARENTHESIS
 # ->   0028 LEFT PARENTHESIS
 "\x{FE35}" => "(",

 # FE36 PRESENTATION FORM FOR VERTICAL RIGHT PARENTHESIS
 # ->   0029 RIGHT PARENTHESIS
 "\x{FE36}" => ")",

 # FE37 PRESENTATION FORM FOR VERTICAL LEFT CURLY BRACKET
 # ->   007B LEFT CURLY BRACKET
 "\x{FE37}" => "{",

 # FE38 PRESENTATION FORM FOR VERTICAL RIGHT CURLY BRACKET
 # ->   007D RIGHT CURLY BRACKET
 "\x{FE38}" => "}",

 # FE39 PRESENTATION FORM FOR VERTICAL LEFT TORTOISE SHELL BRACKET
 # ->   3014 LEFT TORTOISE SHELL BRACKET
 "\x{FE39}" => "\x{3014}",

 # FE3A PRESENTATION FORM FOR VERTICAL RIGHT TORTOISE SHELL BRACKET
 # ->   3015 RIGHT TORTOISE SHELL BRACKET
 "\x{FE3A}" => "\x{3015}",

 # FE3B PRESENTATION FORM FOR VERTICAL LEFT BLACK LENTICULAR BRACKET
 # ->   3010 LEFT BLACK LENTICULAR BRACKET
 "\x{FE3B}" => "\x{3010}",

 # FE3C PRESENTATION FORM FOR VERTICAL RIGHT BLACK LENTICULAR BRACKET
 # ->   3011 RIGHT BLACK LENTICULAR BRACKET
 "\x{FE3C}" => "\x{3011}",

 # FE3D PRESENTATION FORM FOR VERTICAL LEFT DOUBLE ANGLE BRACKET
 # ->   300A LEFT DOUBLE ANGLE BRACKET
 "\x{FE3D}" => "\x{300A}",

 # FE3E PRESENTATION FORM FOR VERTICAL RIGHT DOUBLE ANGLE BRACKET
 # ->   300B RIGHT DOUBLE ANGLE BRACKET
 "\x{FE3E}" => "\x{300B}",

 # FE3F PRESENTATION FORM FOR VERTICAL LEFT ANGLE BRACKET
 # ->   3008 LEFT ANGLE BRACKET
 "\x{FE3F}" => "\x{3008}",

 # FE40 PRESENTATION FORM FOR VERTICAL RIGHT ANGLE BRACKET
 # ->   3009 RIGHT ANGLE BRACKET
 "\x{FE40}" => "\x{3009}",

 # FE41 PRESENTATION FORM FOR VERTICAL LEFT CORNER BRACKET
 # ->   300C LEFT CORNER BRACKET
 "\x{FE41}" => "\x{300C}",

 # FE42 PRESENTATION FORM FOR VERTICAL RIGHT CORNER BRACKET
 # ->   300D RIGHT CORNER BRACKET
 "\x{FE42}" => "\x{300D}",

 # FE43 PRESENTATION FORM FOR VERTICAL LEFT WHITE CORNER BRACKET
 # ->   300E LEFT WHITE CORNER BRACKET
 "\x{FE43}" => "\x{300E}",

 # FE44 PRESENTATION FORM FOR VERTICAL RIGHT WHITE CORNER BRACKET
 # ->   300F RIGHT WHITE CORNER BRACKET
 "\x{FE44}" => "\x{300F}",

 # FE47 PRESENTATION FORM FOR VERTICAL LEFT SQUARE BRACKET
 # ->   005B LEFT SQUARE BRACKET
 "\x{FE47}" => "[",

 # FE48 PRESENTATION FORM FOR VERTICAL RIGHT SQUARE BRACKET
 # ->   005D RIGHT SQUARE BRACKET
 "\x{FE48}" => "]",

 # FE49 DASHED OVERLINE
 # ->   0020 SPACE
 "\x{FE49}" => " ",

 # FE4A CENTRELINE OVERLINE
 # ->   0020 SPACE
 "\x{FE4A}" => " ",

 # FE4B WAVY OVERLINE
 # ->   0020 SPACE
 "\x{FE4B}" => " ",

 # FE4C DOUBLE WAVY OVERLINE
 # ->   0020 SPACE
 "\x{FE4C}" => " ",

 # FE4D DASHED LOW LINE
 # ->   005F LOW LINE
 "\x{FE4D}" => "_",

 # FE4E CENTRELINE LOW LINE
 # ->   005F LOW LINE
 "\x{FE4E}" => "_",

 # FE4F WAVY LOW LINE
 # ->   005F LOW LINE
 "\x{FE4F}" => "_",

 # FE50 SMALL COMMA
 # ->   002C COMMA
 "\x{FE50}" => ",",

 # FE51 SMALL IDEOGRAPHIC COMMA
 # ->   3001 IDEOGRAPHIC COMMA
 "\x{FE51}" => "\x{3001}",

 # FE52 SMALL FULL STOP
 # ->   002E FULL STOP
 "\x{FE52}" => ".",

 # FE54 SMALL SEMICOLON
 # ->   003B SEMICOLON
 "\x{FE54}" => ";",

 # FE55 SMALL COLON
 # ->   003A COLON
 "\x{FE55}" => ":",

 # FE56 SMALL QUESTION MARK
 # ->   003F QUESTION MARK
 "\x{FE56}" => "?",

 # FE57 SMALL EXCLAMATION MARK
 # ->   0021 EXCLAMATION MARK
 "\x{FE57}" => "!",

 # FE58 SMALL EM DASH
 # ->   2014 EM DASH
 "\x{FE58}" => "\x{2014}",

 # FE59 SMALL LEFT PARENTHESIS
 # ->   0028 LEFT PARENTHESIS
 "\x{FE59}" => "(",

 # FE5A SMALL RIGHT PARENTHESIS
 # ->   0029 RIGHT PARENTHESIS
 "\x{FE5A}" => ")",

 # FE5B SMALL LEFT CURLY BRACKET
 # ->   007B LEFT CURLY BRACKET
 "\x{FE5B}" => "{",

 # FE5C SMALL RIGHT CURLY BRACKET
 # ->   007D RIGHT CURLY BRACKET
 "\x{FE5C}" => "}",

 # FE5D SMALL LEFT TORTOISE SHELL BRACKET
 # ->   3014 LEFT TORTOISE SHELL BRACKET
 "\x{FE5D}" => "\x{3014}",

 # FE5E SMALL RIGHT TORTOISE SHELL BRACKET
 # ->   3015 RIGHT TORTOISE SHELL BRACKET
 "\x{FE5E}" => "\x{3015}",

 # FE5F SMALL NUMBER SIGN
 # ->   0023 NUMBER SIGN
 "\x{FE5F}" => "#",

 # FE60 SMALL AMPERSAND
 # ->   0026 AMPERSAND
 "\x{FE60}" => "&",

 # FE61 SMALL ASTERISK
 # ->   002A ASTERISK
 "\x{FE61}" => "*",

 # FE62 SMALL PLUS SIGN
 # ->   002B PLUS SIGN
 "\x{FE62}" => "+",

 # FE63 SMALL HYPHEN-MINUS
 # ->   002D HYPHEN-MINUS
 "\x{FE63}" => "-",

 # FE64 SMALL LESS-THAN SIGN
 # ->   003C LESS-THAN SIGN
 "\x{FE64}" => "<",

 # FE65 SMALL GREATER-THAN SIGN
 # ->   003E GREATER-THAN SIGN
 "\x{FE65}" => ">",

 # FE66 SMALL EQUALS SIGN
 # ->   003D EQUALS SIGN
 "\x{FE66}" => "=",

 # FE68 SMALL REVERSE SOLIDUS
 # ->   005C REVERSE SOLIDUS
 "\x{FE68}" => "\\",

 # FE69 SMALL DOLLAR SIGN
 # ->   0024 DOLLAR SIGN
 "\x{FE69}" => "\$",

 # FE6A SMALL PERCENT SIGN
 # ->   0025 PERCENT SIGN
 "\x{FE6A}" => "%",

 # FE6B SMALL COMMERCIAL AT
 # ->   0040 COMMERCIAL AT
 "\x{FE6B}" => "@",

 # FE70 ARABIC FATHATAN ISOLATED FORM
 # ->   0020 SPACE
 "\x{FE70}" => " ",

 # FE71 ARABIC TATWEEL WITH FATHATAN ABOVE
 # ->   0640 ARABIC TATWEEL
 "\x{FE71}" => "\x{0640}",

 # FE72 ARABIC DAMMATAN ISOLATED FORM
 # ->   0020 SPACE
 "\x{FE72}" => " ",

 # FE74 ARABIC KASRATAN ISOLATED FORM
 # ->   0020 SPACE
 "\x{FE74}" => " ",

 # FE76 ARABIC FATHA ISOLATED FORM
 # ->   0020 SPACE
 "\x{FE76}" => " ",

 # FE77 ARABIC FATHA MEDIAL FORM
 # ->   0640 ARABIC TATWEEL
 "\x{FE77}" => "\x{0640}",

 # FE78 ARABIC DAMMA ISOLATED FORM
 # ->   0020 SPACE
 "\x{FE78}" => " ",

 # FE79 ARABIC DAMMA MEDIAL FORM
 # ->   0640 ARABIC TATWEEL
 "\x{FE79}" => "\x{0640}",

 # FE7A ARABIC KASRA ISOLATED FORM
 # ->   0020 SPACE
 "\x{FE7A}" => " ",

 # FE7B ARABIC KASRA MEDIAL FORM
 # ->   0640 ARABIC TATWEEL
 "\x{FE7B}" => "\x{0640}",

 # FE7C ARABIC SHADDA ISOLATED FORM
 # ->   0020 SPACE
 "\x{FE7C}" => " ",

 # FE7D ARABIC SHADDA MEDIAL FORM
 # ->   0640 ARABIC TATWEEL
 "\x{FE7D}" => "\x{0640}",

 # FE7E ARABIC SUKUN ISOLATED FORM
 # ->   0020 SPACE
 "\x{FE7E}" => " ",

 # FE7F ARABIC SUKUN MEDIAL FORM
 # ->   0640 ARABIC TATWEEL
 "\x{FE7F}" => "\x{0640}",

 # FE80 ARABIC LETTER HAMZA ISOLATED FORM
 # ->   0621 ARABIC LETTER HAMZA
 "\x{FE80}" => "\x{0621}",

 # FE81 ARABIC LETTER ALEF WITH MADDA ABOVE ISOLATED FORM
 # ->   0627 ARABIC LETTER ALEF
 "\x{FE81}" => "\x{0627}",

 # FE82 ARABIC LETTER ALEF WITH MADDA ABOVE FINAL FORM
 # ->   0627 ARABIC LETTER ALEF
 "\x{FE82}" => "\x{0627}",

 # FE83 ARABIC LETTER ALEF WITH HAMZA ABOVE ISOLATED FORM
 # ->   0627 ARABIC LETTER ALEF
 "\x{FE83}" => "\x{0627}",

 # FE84 ARABIC LETTER ALEF WITH HAMZA ABOVE FINAL FORM
 # ->   0627 ARABIC LETTER ALEF
 "\x{FE84}" => "\x{0627}",

 # FE85 ARABIC LETTER WAW WITH HAMZA ABOVE ISOLATED FORM
 # ->   0648 ARABIC LETTER WAW
 "\x{FE85}" => "\x{0648}",

 # FE86 ARABIC LETTER WAW WITH HAMZA ABOVE FINAL FORM
 # ->   0648 ARABIC LETTER WAW
 "\x{FE86}" => "\x{0648}",

 # FE87 ARABIC LETTER ALEF WITH HAMZA BELOW ISOLATED FORM
 # ->   0627 ARABIC LETTER ALEF
 "\x{FE87}" => "\x{0627}",

 # FE88 ARABIC LETTER ALEF WITH HAMZA BELOW FINAL FORM
 # ->   0627 ARABIC LETTER ALEF
 "\x{FE88}" => "\x{0627}",

 # FE89 ARABIC LETTER YEH WITH HAMZA ABOVE ISOLATED FORM
 # ->   064A ARABIC LETTER YEH
 "\x{FE89}" => "\x{064A}",

 # FE8A ARABIC LETTER YEH WITH HAMZA ABOVE FINAL FORM
 # ->   064A ARABIC LETTER YEH
 "\x{FE8A}" => "\x{064A}",

 # FE8B ARABIC LETTER YEH WITH HAMZA ABOVE INITIAL FORM
 # ->   064A ARABIC LETTER YEH
 "\x{FE8B}" => "\x{064A}",

 # FE8C ARABIC LETTER YEH WITH HAMZA ABOVE MEDIAL FORM
 # ->   064A ARABIC LETTER YEH
 "\x{FE8C}" => "\x{064A}",

 # FE8D ARABIC LETTER ALEF ISOLATED FORM
 # ->   0627 ARABIC LETTER ALEF
 "\x{FE8D}" => "\x{0627}",

 # FE8E ARABIC LETTER ALEF FINAL FORM
 # ->   0627 ARABIC LETTER ALEF
 "\x{FE8E}" => "\x{0627}",

 # FE8F ARABIC LETTER BEH ISOLATED FORM
 # ->   0628 ARABIC LETTER BEH
 "\x{FE8F}" => "\x{0628}",

 # FE90 ARABIC LETTER BEH FINAL FORM
 # ->   0628 ARABIC LETTER BEH
 "\x{FE90}" => "\x{0628}",

 # FE91 ARABIC LETTER BEH INITIAL FORM
 # ->   0628 ARABIC LETTER BEH
 "\x{FE91}" => "\x{0628}",

 # FE92 ARABIC LETTER BEH MEDIAL FORM
 # ->   0628 ARABIC LETTER BEH
 "\x{FE92}" => "\x{0628}",

 # FE93 ARABIC LETTER TEH MARBUTA ISOLATED FORM
 # ->   0629 ARABIC LETTER TEH MARBUTA
 "\x{FE93}" => "\x{0629}",

 # FE94 ARABIC LETTER TEH MARBUTA FINAL FORM
 # ->   0629 ARABIC LETTER TEH MARBUTA
 "\x{FE94}" => "\x{0629}",

 # FE95 ARABIC LETTER TEH ISOLATED FORM
 # ->   062A ARABIC LETTER TEH
 "\x{FE95}" => "\x{062A}",

 # FE96 ARABIC LETTER TEH FINAL FORM
 # ->   062A ARABIC LETTER TEH
 "\x{FE96}" => "\x{062A}",

 # FE97 ARABIC LETTER TEH INITIAL FORM
 # ->   062A ARABIC LETTER TEH
 "\x{FE97}" => "\x{062A}",

 # FE98 ARABIC LETTER TEH MEDIAL FORM
 # ->   062A ARABIC LETTER TEH
 "\x{FE98}" => "\x{062A}",

 # FE99 ARABIC LETTER THEH ISOLATED FORM
 # ->   062B ARABIC LETTER THEH
 "\x{FE99}" => "\x{062B}",

 # FE9A ARABIC LETTER THEH FINAL FORM
 # ->   062B ARABIC LETTER THEH
 "\x{FE9A}" => "\x{062B}",

 # FE9B ARABIC LETTER THEH INITIAL FORM
 # ->   062B ARABIC LETTER THEH
 "\x{FE9B}" => "\x{062B}",

 # FE9C ARABIC LETTER THEH MEDIAL FORM
 # ->   062B ARABIC LETTER THEH
 "\x{FE9C}" => "\x{062B}",

 # FE9D ARABIC LETTER JEEM ISOLATED FORM
 # ->   062C ARABIC LETTER JEEM
 "\x{FE9D}" => "\x{062C}",

 # FE9E ARABIC LETTER JEEM FINAL FORM
 # ->   062C ARABIC LETTER JEEM
 "\x{FE9E}" => "\x{062C}",

 # FE9F ARABIC LETTER JEEM INITIAL FORM
 # ->   062C ARABIC LETTER JEEM
 "\x{FE9F}" => "\x{062C}",

 # FEA0 ARABIC LETTER JEEM MEDIAL FORM
 # ->   062C ARABIC LETTER JEEM
 "\x{FEA0}" => "\x{062C}",

 # FEA1 ARABIC LETTER HAH ISOLATED FORM
 # ->   062D ARABIC LETTER HAH
 "\x{FEA1}" => "\x{062D}",

 # FEA2 ARABIC LETTER HAH FINAL FORM
 # ->   062D ARABIC LETTER HAH
 "\x{FEA2}" => "\x{062D}",

 # FEA3 ARABIC LETTER HAH INITIAL FORM
 # ->   062D ARABIC LETTER HAH
 "\x{FEA3}" => "\x{062D}",

 # FEA4 ARABIC LETTER HAH MEDIAL FORM
 # ->   062D ARABIC LETTER HAH
 "\x{FEA4}" => "\x{062D}",

 # FEA5 ARABIC LETTER KHAH ISOLATED FORM
 # ->   062E ARABIC LETTER KHAH
 "\x{FEA5}" => "\x{062E}",

 # FEA6 ARABIC LETTER KHAH FINAL FORM
 # ->   062E ARABIC LETTER KHAH
 "\x{FEA6}" => "\x{062E}",

 # FEA7 ARABIC LETTER KHAH INITIAL FORM
 # ->   062E ARABIC LETTER KHAH
 "\x{FEA7}" => "\x{062E}",

 # FEA8 ARABIC LETTER KHAH MEDIAL FORM
 # ->   062E ARABIC LETTER KHAH
 "\x{FEA8}" => "\x{062E}",

 # FEA9 ARABIC LETTER DAL ISOLATED FORM
 # ->   062F ARABIC LETTER DAL
 "\x{FEA9}" => "\x{062F}",

 # FEAA ARABIC LETTER DAL FINAL FORM
 # ->   062F ARABIC LETTER DAL
 "\x{FEAA}" => "\x{062F}",

 # FEAB ARABIC LETTER THAL ISOLATED FORM
 # ->   0630 ARABIC LETTER THAL
 "\x{FEAB}" => "\x{0630}",

 # FEAC ARABIC LETTER THAL FINAL FORM
 # ->   0630 ARABIC LETTER THAL
 "\x{FEAC}" => "\x{0630}",

 # FEAD ARABIC LETTER REH ISOLATED FORM
 # ->   0631 ARABIC LETTER REH
 "\x{FEAD}" => "\x{0631}",

 # FEAE ARABIC LETTER REH FINAL FORM
 # ->   0631 ARABIC LETTER REH
 "\x{FEAE}" => "\x{0631}",

 # FEAF ARABIC LETTER ZAIN ISOLATED FORM
 # ->   0632 ARABIC LETTER ZAIN
 "\x{FEAF}" => "\x{0632}",

 # FEB0 ARABIC LETTER ZAIN FINAL FORM
 # ->   0632 ARABIC LETTER ZAIN
 "\x{FEB0}" => "\x{0632}",

 # FEB1 ARABIC LETTER SEEN ISOLATED FORM
 # ->   0633 ARABIC LETTER SEEN
 "\x{FEB1}" => "\x{0633}",

 # FEB2 ARABIC LETTER SEEN FINAL FORM
 # ->   0633 ARABIC LETTER SEEN
 "\x{FEB2}" => "\x{0633}",

 # FEB3 ARABIC LETTER SEEN INITIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 "\x{FEB3}" => "\x{0633}",

 # FEB4 ARABIC LETTER SEEN MEDIAL FORM
 # ->   0633 ARABIC LETTER SEEN
 "\x{FEB4}" => "\x{0633}",

 # FEB5 ARABIC LETTER SHEEN ISOLATED FORM
 # ->   0634 ARABIC LETTER SHEEN
 "\x{FEB5}" => "\x{0634}",

 # FEB6 ARABIC LETTER SHEEN FINAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 "\x{FEB6}" => "\x{0634}",

 # FEB7 ARABIC LETTER SHEEN INITIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 "\x{FEB7}" => "\x{0634}",

 # FEB8 ARABIC LETTER SHEEN MEDIAL FORM
 # ->   0634 ARABIC LETTER SHEEN
 "\x{FEB8}" => "\x{0634}",

 # FEB9 ARABIC LETTER SAD ISOLATED FORM
 # ->   0635 ARABIC LETTER SAD
 "\x{FEB9}" => "\x{0635}",

 # FEBA ARABIC LETTER SAD FINAL FORM
 # ->   0635 ARABIC LETTER SAD
 "\x{FEBA}" => "\x{0635}",

 # FEBB ARABIC LETTER SAD INITIAL FORM
 # ->   0635 ARABIC LETTER SAD
 "\x{FEBB}" => "\x{0635}",

 # FEBC ARABIC LETTER SAD MEDIAL FORM
 # ->   0635 ARABIC LETTER SAD
 "\x{FEBC}" => "\x{0635}",

 # FEBD ARABIC LETTER DAD ISOLATED FORM
 # ->   0636 ARABIC LETTER DAD
 "\x{FEBD}" => "\x{0636}",

 # FEBE ARABIC LETTER DAD FINAL FORM
 # ->   0636 ARABIC LETTER DAD
 "\x{FEBE}" => "\x{0636}",

 # FEBF ARABIC LETTER DAD INITIAL FORM
 # ->   0636 ARABIC LETTER DAD
 "\x{FEBF}" => "\x{0636}",

 # FEC0 ARABIC LETTER DAD MEDIAL FORM
 # ->   0636 ARABIC LETTER DAD
 "\x{FEC0}" => "\x{0636}",

 # FEC1 ARABIC LETTER TAH ISOLATED FORM
 # ->   0637 ARABIC LETTER TAH
 "\x{FEC1}" => "\x{0637}",

 # FEC2 ARABIC LETTER TAH FINAL FORM
 # ->   0637 ARABIC LETTER TAH
 "\x{FEC2}" => "\x{0637}",

 # FEC3 ARABIC LETTER TAH INITIAL FORM
 # ->   0637 ARABIC LETTER TAH
 "\x{FEC3}" => "\x{0637}",

 # FEC4 ARABIC LETTER TAH MEDIAL FORM
 # ->   0637 ARABIC LETTER TAH
 "\x{FEC4}" => "\x{0637}",

 # FEC5 ARABIC LETTER ZAH ISOLATED FORM
 # ->   0638 ARABIC LETTER ZAH
 "\x{FEC5}" => "\x{0638}",

 # FEC6 ARABIC LETTER ZAH FINAL FORM
 # ->   0638 ARABIC LETTER ZAH
 "\x{FEC6}" => "\x{0638}",

 # FEC7 ARABIC LETTER ZAH INITIAL FORM
 # ->   0638 ARABIC LETTER ZAH
 "\x{FEC7}" => "\x{0638}",

 # FEC8 ARABIC LETTER ZAH MEDIAL FORM
 # ->   0638 ARABIC LETTER ZAH
 "\x{FEC8}" => "\x{0638}",

 # FEC9 ARABIC LETTER AIN ISOLATED FORM
 # ->   0639 ARABIC LETTER AIN
 "\x{FEC9}" => "\x{0639}",

 # FECA ARABIC LETTER AIN FINAL FORM
 # ->   0639 ARABIC LETTER AIN
 "\x{FECA}" => "\x{0639}",

 # FECB ARABIC LETTER AIN INITIAL FORM
 # ->   0639 ARABIC LETTER AIN
 "\x{FECB}" => "\x{0639}",

 # FECC ARABIC LETTER AIN MEDIAL FORM
 # ->   0639 ARABIC LETTER AIN
 "\x{FECC}" => "\x{0639}",

 # FECD ARABIC LETTER GHAIN ISOLATED FORM
 # ->   063A ARABIC LETTER GHAIN
 "\x{FECD}" => "\x{063A}",

 # FECE ARABIC LETTER GHAIN FINAL FORM
 # ->   063A ARABIC LETTER GHAIN
 "\x{FECE}" => "\x{063A}",

 # FECF ARABIC LETTER GHAIN INITIAL FORM
 # ->   063A ARABIC LETTER GHAIN
 "\x{FECF}" => "\x{063A}",

 # FED0 ARABIC LETTER GHAIN MEDIAL FORM
 # ->   063A ARABIC LETTER GHAIN
 "\x{FED0}" => "\x{063A}",

 # FED1 ARABIC LETTER FEH ISOLATED FORM
 # ->   0641 ARABIC LETTER FEH
 "\x{FED1}" => "\x{0641}",

 # FED2 ARABIC LETTER FEH FINAL FORM
 # ->   0641 ARABIC LETTER FEH
 "\x{FED2}" => "\x{0641}",

 # FED3 ARABIC LETTER FEH INITIAL FORM
 # ->   0641 ARABIC LETTER FEH
 "\x{FED3}" => "\x{0641}",

 # FED4 ARABIC LETTER FEH MEDIAL FORM
 # ->   0641 ARABIC LETTER FEH
 "\x{FED4}" => "\x{0641}",

 # FED5 ARABIC LETTER QAF ISOLATED FORM
 # ->   0642 ARABIC LETTER QAF
 "\x{FED5}" => "\x{0642}",

 # FED6 ARABIC LETTER QAF FINAL FORM
 # ->   0642 ARABIC LETTER QAF
 "\x{FED6}" => "\x{0642}",

 # FED7 ARABIC LETTER QAF INITIAL FORM
 # ->   0642 ARABIC LETTER QAF
 "\x{FED7}" => "\x{0642}",

 # FED8 ARABIC LETTER QAF MEDIAL FORM
 # ->   0642 ARABIC LETTER QAF
 "\x{FED8}" => "\x{0642}",

 # FED9 ARABIC LETTER KAF ISOLATED FORM
 # ->   0643 ARABIC LETTER KAF
 "\x{FED9}" => "\x{0643}",

 # FEDA ARABIC LETTER KAF FINAL FORM
 # ->   0643 ARABIC LETTER KAF
 "\x{FEDA}" => "\x{0643}",

 # FEDB ARABIC LETTER KAF INITIAL FORM
 # ->   0643 ARABIC LETTER KAF
 "\x{FEDB}" => "\x{0643}",

 # FEDC ARABIC LETTER KAF MEDIAL FORM
 # ->   0643 ARABIC LETTER KAF
 "\x{FEDC}" => "\x{0643}",

 # FEDD ARABIC LETTER LAM ISOLATED FORM
 # ->   0644 ARABIC LETTER LAM
 "\x{FEDD}" => "\x{0644}",

 # FEDE ARABIC LETTER LAM FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 "\x{FEDE}" => "\x{0644}",

 # FEDF ARABIC LETTER LAM INITIAL FORM
 # ->   0644 ARABIC LETTER LAM
 "\x{FEDF}" => "\x{0644}",

 # FEE0 ARABIC LETTER LAM MEDIAL FORM
 # ->   0644 ARABIC LETTER LAM
 "\x{FEE0}" => "\x{0644}",

 # FEE1 ARABIC LETTER MEEM ISOLATED FORM
 # ->   0645 ARABIC LETTER MEEM
 "\x{FEE1}" => "\x{0645}",

 # FEE2 ARABIC LETTER MEEM FINAL FORM
 # ->   0645 ARABIC LETTER MEEM
 "\x{FEE2}" => "\x{0645}",

 # FEE3 ARABIC LETTER MEEM INITIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 "\x{FEE3}" => "\x{0645}",

 # FEE4 ARABIC LETTER MEEM MEDIAL FORM
 # ->   0645 ARABIC LETTER MEEM
 "\x{FEE4}" => "\x{0645}",

 # FEE5 ARABIC LETTER NOON ISOLATED FORM
 # ->   0646 ARABIC LETTER NOON
 "\x{FEE5}" => "\x{0646}",

 # FEE6 ARABIC LETTER NOON FINAL FORM
 # ->   0646 ARABIC LETTER NOON
 "\x{FEE6}" => "\x{0646}",

 # FEE7 ARABIC LETTER NOON INITIAL FORM
 # ->   0646 ARABIC LETTER NOON
 "\x{FEE7}" => "\x{0646}",

 # FEE8 ARABIC LETTER NOON MEDIAL FORM
 # ->   0646 ARABIC LETTER NOON
 "\x{FEE8}" => "\x{0646}",

 # FEE9 ARABIC LETTER HEH ISOLATED FORM
 # ->   0647 ARABIC LETTER HEH
 "\x{FEE9}" => "\x{0647}",

 # FEEA ARABIC LETTER HEH FINAL FORM
 # ->   0647 ARABIC LETTER HEH
 "\x{FEEA}" => "\x{0647}",

 # FEEB ARABIC LETTER HEH INITIAL FORM
 # ->   0647 ARABIC LETTER HEH
 "\x{FEEB}" => "\x{0647}",

 # FEEC ARABIC LETTER HEH MEDIAL FORM
 # ->   0647 ARABIC LETTER HEH
 "\x{FEEC}" => "\x{0647}",

 # FEED ARABIC LETTER WAW ISOLATED FORM
 # ->   0648 ARABIC LETTER WAW
 "\x{FEED}" => "\x{0648}",

 # FEEE ARABIC LETTER WAW FINAL FORM
 # ->   0648 ARABIC LETTER WAW
 "\x{FEEE}" => "\x{0648}",

 # FEEF ARABIC LETTER ALEF MAKSURA ISOLATED FORM
 # ->   0649 ARABIC LETTER ALEF MAKSURA
 "\x{FEEF}" => "\x{0649}",

 # FEF0 ARABIC LETTER ALEF MAKSURA FINAL FORM
 # ->   0649 ARABIC LETTER ALEF MAKSURA
 "\x{FEF0}" => "\x{0649}",

 # FEF1 ARABIC LETTER YEH ISOLATED FORM
 # ->   064A ARABIC LETTER YEH
 "\x{FEF1}" => "\x{064A}",

 # FEF2 ARABIC LETTER YEH FINAL FORM
 # ->   064A ARABIC LETTER YEH
 "\x{FEF2}" => "\x{064A}",

 # FEF3 ARABIC LETTER YEH INITIAL FORM
 # ->   064A ARABIC LETTER YEH
 "\x{FEF3}" => "\x{064A}",

 # FEF4 ARABIC LETTER YEH MEDIAL FORM
 # ->   064A ARABIC LETTER YEH
 "\x{FEF4}" => "\x{064A}",

 # FEF5 ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE ISOLATED FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0627 ARABIC LETTER ALEF
 "\x{FEF5}" => "\x{0644}\x{0627}",

 # FEF6 ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0627 ARABIC LETTER ALEF
 "\x{FEF6}" => "\x{0644}\x{0627}",

 # FEF7 ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE ISOLATED FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0627 ARABIC LETTER ALEF
 "\x{FEF7}" => "\x{0644}\x{0627}",

 # FEF8 ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0627 ARABIC LETTER ALEF
 "\x{FEF8}" => "\x{0644}\x{0627}",

 # FEF9 ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW ISOLATED FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0627 ARABIC LETTER ALEF
 "\x{FEF9}" => "\x{0644}\x{0627}",

 # FEFA ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0627 ARABIC LETTER ALEF
 "\x{FEFA}" => "\x{0644}\x{0627}",

 # FEFB ARABIC LIGATURE LAM WITH ALEF ISOLATED FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0627 ARABIC LETTER ALEF
 "\x{FEFB}" => "\x{0644}\x{0627}",

 # FEFC ARABIC LIGATURE LAM WITH ALEF FINAL FORM
 # ->   0644 ARABIC LETTER LAM
 #    + 0627 ARABIC LETTER ALEF
 "\x{FEFC}" => "\x{0644}\x{0627}",

 # FF01 FULLWIDTH EXCLAMATION MARK
 # ->   0021 EXCLAMATION MARK
 "\x{FF01}" => "!",

 # FF02 FULLWIDTH QUOTATION MARK
 # ->   0022 QUOTATION MARK
 "\x{FF02}" => "\"",

 # FF03 FULLWIDTH NUMBER SIGN
 # ->   0023 NUMBER SIGN
 "\x{FF03}" => "#",

 # FF04 FULLWIDTH DOLLAR SIGN
 # ->   0024 DOLLAR SIGN
 "\x{FF04}" => "\$",

 # FF05 FULLWIDTH PERCENT SIGN
 # ->   0025 PERCENT SIGN
 "\x{FF05}" => "%",

 # FF06 FULLWIDTH AMPERSAND
 # ->   0026 AMPERSAND
 "\x{FF06}" => "&",

 # FF07 FULLWIDTH APOSTROPHE
 # ->   0027 APOSTROPHE
 "\x{FF07}" => "'",

 # FF08 FULLWIDTH LEFT PARENTHESIS
 # ->   0028 LEFT PARENTHESIS
 "\x{FF08}" => "(",

 # FF09 FULLWIDTH RIGHT PARENTHESIS
 # ->   0029 RIGHT PARENTHESIS
 "\x{FF09}" => ")",

 # FF0A FULLWIDTH ASTERISK
 # ->   002A ASTERISK
 "\x{FF0A}" => "*",

 # FF0B FULLWIDTH PLUS SIGN
 # ->   002B PLUS SIGN
 "\x{FF0B}" => "+",

 # FF0C FULLWIDTH COMMA
 # ->   002C COMMA
 "\x{FF0C}" => ",",

 # FF0D FULLWIDTH HYPHEN-MINUS
 # ->   002D HYPHEN-MINUS
 "\x{FF0D}" => "-",

 # FF0E FULLWIDTH FULL STOP
 # ->   002E FULL STOP
 "\x{FF0E}" => ".",

 # FF0F FULLWIDTH SOLIDUS
 # ->   002F SOLIDUS
 "\x{FF0F}" => "/",

 # FF10 FULLWIDTH DIGIT ZERO
 # ->   0030 DIGIT ZERO
 "\x{FF10}" => "0",

 # FF11 FULLWIDTH DIGIT ONE
 # ->   0031 DIGIT ONE
 "\x{FF11}" => "1",

 # FF12 FULLWIDTH DIGIT TWO
 # ->   0032 DIGIT TWO
 "\x{FF12}" => "2",

 # FF13 FULLWIDTH DIGIT THREE
 # ->   0033 DIGIT THREE
 "\x{FF13}" => "3",

 # FF14 FULLWIDTH DIGIT FOUR
 # ->   0034 DIGIT FOUR
 "\x{FF14}" => "4",

 # FF15 FULLWIDTH DIGIT FIVE
 # ->   0035 DIGIT FIVE
 "\x{FF15}" => "5",

 # FF16 FULLWIDTH DIGIT SIX
 # ->   0036 DIGIT SIX
 "\x{FF16}" => "6",

 # FF17 FULLWIDTH DIGIT SEVEN
 # ->   0037 DIGIT SEVEN
 "\x{FF17}" => "7",

 # FF18 FULLWIDTH DIGIT EIGHT
 # ->   0038 DIGIT EIGHT
 "\x{FF18}" => "8",

 # FF19 FULLWIDTH DIGIT NINE
 # ->   0039 DIGIT NINE
 "\x{FF19}" => "9",

 # FF1A FULLWIDTH COLON
 # ->   003A COLON
 "\x{FF1A}" => ":",

 # FF1B FULLWIDTH SEMICOLON
 # ->   003B SEMICOLON
 "\x{FF1B}" => ";",

 # FF1C FULLWIDTH LESS-THAN SIGN
 # ->   003C LESS-THAN SIGN
 "\x{FF1C}" => "<",

 # FF1D FULLWIDTH EQUALS SIGN
 # ->   003D EQUALS SIGN
 "\x{FF1D}" => "=",

 # FF1E FULLWIDTH GREATER-THAN SIGN
 # ->   003E GREATER-THAN SIGN
 "\x{FF1E}" => ">",

 # FF1F FULLWIDTH QUESTION MARK
 # ->   003F QUESTION MARK
 "\x{FF1F}" => "?",

 # FF20 FULLWIDTH COMMERCIAL AT
 # ->   0040 COMMERCIAL AT
 "\x{FF20}" => "@",

 # FF21 FULLWIDTH LATIN CAPITAL LETTER A
 # ->   0041 LATIN CAPITAL LETTER A
 "\x{FF21}" => "A",

 # FF22 FULLWIDTH LATIN CAPITAL LETTER B
 # ->   0042 LATIN CAPITAL LETTER B
 "\x{FF22}" => "B",

 # FF23 FULLWIDTH LATIN CAPITAL LETTER C
 # ->   0043 LATIN CAPITAL LETTER C
 "\x{FF23}" => "C",

 # FF24 FULLWIDTH LATIN CAPITAL LETTER D
 # ->   0044 LATIN CAPITAL LETTER D
 "\x{FF24}" => "D",

 # FF25 FULLWIDTH LATIN CAPITAL LETTER E
 # ->   0045 LATIN CAPITAL LETTER E
 "\x{FF25}" => "E",

 # FF26 FULLWIDTH LATIN CAPITAL LETTER F
 # ->   0046 LATIN CAPITAL LETTER F
 "\x{FF26}" => "F",

 # FF27 FULLWIDTH LATIN CAPITAL LETTER G
 # ->   0047 LATIN CAPITAL LETTER G
 "\x{FF27}" => "G",

 # FF28 FULLWIDTH LATIN CAPITAL LETTER H
 # ->   0048 LATIN CAPITAL LETTER H
 "\x{FF28}" => "H",

 # FF29 FULLWIDTH LATIN CAPITAL LETTER I
 # ->   0049 LATIN CAPITAL LETTER I
 "\x{FF29}" => "I",

 # FF2A FULLWIDTH LATIN CAPITAL LETTER J
 # ->   004A LATIN CAPITAL LETTER J
 "\x{FF2A}" => "J",

 # FF2B FULLWIDTH LATIN CAPITAL LETTER K
 # ->   004B LATIN CAPITAL LETTER K
 "\x{FF2B}" => "K",

 # FF2C FULLWIDTH LATIN CAPITAL LETTER L
 # ->   004C LATIN CAPITAL LETTER L
 "\x{FF2C}" => "L",

 # FF2D FULLWIDTH LATIN CAPITAL LETTER M
 # ->   004D LATIN CAPITAL LETTER M
 "\x{FF2D}" => "M",

 # FF2E FULLWIDTH LATIN CAPITAL LETTER N
 # ->   004E LATIN CAPITAL LETTER N
 "\x{FF2E}" => "N",

 # FF2F FULLWIDTH LATIN CAPITAL LETTER O
 # ->   004F LATIN CAPITAL LETTER O
 "\x{FF2F}" => "O",

 # FF30 FULLWIDTH LATIN CAPITAL LETTER P
 # ->   0050 LATIN CAPITAL LETTER P
 "\x{FF30}" => "P",

 # FF31 FULLWIDTH LATIN CAPITAL LETTER Q
 # ->   0051 LATIN CAPITAL LETTER Q
 "\x{FF31}" => "Q",

 # FF32 FULLWIDTH LATIN CAPITAL LETTER R
 # ->   0052 LATIN CAPITAL LETTER R
 "\x{FF32}" => "R",

 # FF33 FULLWIDTH LATIN CAPITAL LETTER S
 # ->   0053 LATIN CAPITAL LETTER S
 "\x{FF33}" => "S",

 # FF34 FULLWIDTH LATIN CAPITAL LETTER T
 # ->   0054 LATIN CAPITAL LETTER T
 "\x{FF34}" => "T",

 # FF35 FULLWIDTH LATIN CAPITAL LETTER U
 # ->   0055 LATIN CAPITAL LETTER U
 "\x{FF35}" => "U",

 # FF36 FULLWIDTH LATIN CAPITAL LETTER V
 # ->   0056 LATIN CAPITAL LETTER V
 "\x{FF36}" => "V",

 # FF37 FULLWIDTH LATIN CAPITAL LETTER W
 # ->   0057 LATIN CAPITAL LETTER W
 "\x{FF37}" => "W",

 # FF38 FULLWIDTH LATIN CAPITAL LETTER X
 # ->   0058 LATIN CAPITAL LETTER X
 "\x{FF38}" => "X",

 # FF39 FULLWIDTH LATIN CAPITAL LETTER Y
 # ->   0059 LATIN CAPITAL LETTER Y
 "\x{FF39}" => "Y",

 # FF3A FULLWIDTH LATIN CAPITAL LETTER Z
 # ->   005A LATIN CAPITAL LETTER Z
 "\x{FF3A}" => "Z",

 # FF3B FULLWIDTH LEFT SQUARE BRACKET
 # ->   005B LEFT SQUARE BRACKET
 "\x{FF3B}" => "[",

 # FF3C FULLWIDTH REVERSE SOLIDUS
 # ->   005C REVERSE SOLIDUS
 "\x{FF3C}" => "\\",

 # FF3D FULLWIDTH RIGHT SQUARE BRACKET
 # ->   005D RIGHT SQUARE BRACKET
 "\x{FF3D}" => "]",

 # FF3E FULLWIDTH CIRCUMFLEX ACCENT
 # ->   005E CIRCUMFLEX ACCENT
 "\x{FF3E}" => "^",

 # FF3F FULLWIDTH LOW LINE
 # ->   005F LOW LINE
 "\x{FF3F}" => "_",

 # FF40 FULLWIDTH GRAVE ACCENT
 # ->   0060 GRAVE ACCENT
 "\x{FF40}" => "`",

 # FF41 FULLWIDTH LATIN SMALL LETTER A
 # ->   0061 LATIN SMALL LETTER A
 "\x{FF41}" => "a",

 # FF42 FULLWIDTH LATIN SMALL LETTER B
 # ->   0062 LATIN SMALL LETTER B
 "\x{FF42}" => "b",

 # FF43 FULLWIDTH LATIN SMALL LETTER C
 # ->   0063 LATIN SMALL LETTER C
 "\x{FF43}" => "c",

 # FF44 FULLWIDTH LATIN SMALL LETTER D
 # ->   0064 LATIN SMALL LETTER D
 "\x{FF44}" => "d",

 # FF45 FULLWIDTH LATIN SMALL LETTER E
 # ->   0065 LATIN SMALL LETTER E
 "\x{FF45}" => "e",

 # FF46 FULLWIDTH LATIN SMALL LETTER F
 # ->   0066 LATIN SMALL LETTER F
 "\x{FF46}" => "f",

 # FF47 FULLWIDTH LATIN SMALL LETTER G
 # ->   0067 LATIN SMALL LETTER G
 "\x{FF47}" => "g",

 # FF48 FULLWIDTH LATIN SMALL LETTER H
 # ->   0068 LATIN SMALL LETTER H
 "\x{FF48}" => "h",

 # FF49 FULLWIDTH LATIN SMALL LETTER I
 # ->   0069 LATIN SMALL LETTER I
 "\x{FF49}" => "i",

 # FF4A FULLWIDTH LATIN SMALL LETTER J
 # ->   006A LATIN SMALL LETTER J
 "\x{FF4A}" => "j",

 # FF4B FULLWIDTH LATIN SMALL LETTER K
 # ->   006B LATIN SMALL LETTER K
 "\x{FF4B}" => "k",

 # FF4C FULLWIDTH LATIN SMALL LETTER L
 # ->   006C LATIN SMALL LETTER L
 "\x{FF4C}" => "l",

 # FF4D FULLWIDTH LATIN SMALL LETTER M
 # ->   006D LATIN SMALL LETTER M
 "\x{FF4D}" => "m",

 # FF4E FULLWIDTH LATIN SMALL LETTER N
 # ->   006E LATIN SMALL LETTER N
 "\x{FF4E}" => "n",

 # FF4F FULLWIDTH LATIN SMALL LETTER O
 # ->   006F LATIN SMALL LETTER O
 "\x{FF4F}" => "o",

 # FF50 FULLWIDTH LATIN SMALL LETTER P
 # ->   0070 LATIN SMALL LETTER P
 "\x{FF50}" => "p",

 # FF51 FULLWIDTH LATIN SMALL LETTER Q
 # ->   0071 LATIN SMALL LETTER Q
 "\x{FF51}" => "q",

 # FF52 FULLWIDTH LATIN SMALL LETTER R
 # ->   0072 LATIN SMALL LETTER R
 "\x{FF52}" => "r",

 # FF53 FULLWIDTH LATIN SMALL LETTER S
 # ->   0073 LATIN SMALL LETTER S
 "\x{FF53}" => "s",

 # FF54 FULLWIDTH LATIN SMALL LETTER T
 # ->   0074 LATIN SMALL LETTER T
 "\x{FF54}" => "t",

 # FF55 FULLWIDTH LATIN SMALL LETTER U
 # ->   0075 LATIN SMALL LETTER U
 "\x{FF55}" => "u",

 # FF56 FULLWIDTH LATIN SMALL LETTER V
 # ->   0076 LATIN SMALL LETTER V
 "\x{FF56}" => "v",

 # FF57 FULLWIDTH LATIN SMALL LETTER W
 # ->   0077 LATIN SMALL LETTER W
 "\x{FF57}" => "w",

 # FF58 FULLWIDTH LATIN SMALL LETTER X
 # ->   0078 LATIN SMALL LETTER X
 "\x{FF58}" => "x",

 # FF59 FULLWIDTH LATIN SMALL LETTER Y
 # ->   0079 LATIN SMALL LETTER Y
 "\x{FF59}" => "y",

 # FF5A FULLWIDTH LATIN SMALL LETTER Z
 # ->   007A LATIN SMALL LETTER Z
 "\x{FF5A}" => "z",

 # FF5B FULLWIDTH LEFT CURLY BRACKET
 # ->   007B LEFT CURLY BRACKET
 "\x{FF5B}" => "{",

 # FF5C FULLWIDTH VERTICAL LINE
 # ->   007C VERTICAL LINE
 "\x{FF5C}" => "|",

 # FF5D FULLWIDTH RIGHT CURLY BRACKET
 # ->   007D RIGHT CURLY BRACKET
 "\x{FF5D}" => "}",

 # FF5E FULLWIDTH TILDE
 # ->   007E TILDE
 "\x{FF5E}" => "~",

 # FF5F FULLWIDTH LEFT WHITE PARENTHESIS
 # ->   2985 LEFT WHITE PARENTHESIS
 "\x{FF5F}" => "\x{2985}",

 # FF60 FULLWIDTH RIGHT WHITE PARENTHESIS
 # ->   2986 RIGHT WHITE PARENTHESIS
 "\x{FF60}" => "\x{2986}",

 # FF61 HALFWIDTH IDEOGRAPHIC FULL STOP
 # ->   3002 IDEOGRAPHIC FULL STOP
 "\x{FF61}" => "\x{3002}",

 # FF62 HALFWIDTH LEFT CORNER BRACKET
 # ->   300C LEFT CORNER BRACKET
 "\x{FF62}" => "\x{300C}",

 # FF63 HALFWIDTH RIGHT CORNER BRACKET
 # ->   300D RIGHT CORNER BRACKET
 "\x{FF63}" => "\x{300D}",

 # FF64 HALFWIDTH IDEOGRAPHIC COMMA
 # ->   3001 IDEOGRAPHIC COMMA
 "\x{FF64}" => "\x{3001}",

 # FF65 HALFWIDTH KATAKANA MIDDLE DOT
 # ->   30FB KATAKANA MIDDLE DOT
 "\x{FF65}" => "\x{30FB}",

 # FF66 HALFWIDTH KATAKANA LETTER WO
 # ->   30F2 KATAKANA LETTER WO
 "\x{FF66}" => "\x{30F2}",

 # FF67 HALFWIDTH KATAKANA LETTER SMALL A
 # ->   30A1 KATAKANA LETTER SMALL A
 "\x{FF67}" => "\x{30A1}",

 # FF68 HALFWIDTH KATAKANA LETTER SMALL I
 # ->   30A3 KATAKANA LETTER SMALL I
 "\x{FF68}" => "\x{30A3}",

 # FF69 HALFWIDTH KATAKANA LETTER SMALL U
 # ->   30A5 KATAKANA LETTER SMALL U
 "\x{FF69}" => "\x{30A5}",

 # FF6A HALFWIDTH KATAKANA LETTER SMALL E
 # ->   30A7 KATAKANA LETTER SMALL E
 "\x{FF6A}" => "\x{30A7}",

 # FF6B HALFWIDTH KATAKANA LETTER SMALL O
 # ->   30A9 KATAKANA LETTER SMALL O
 "\x{FF6B}" => "\x{30A9}",

 # FF6C HALFWIDTH KATAKANA LETTER SMALL YA
 # ->   30E3 KATAKANA LETTER SMALL YA
 "\x{FF6C}" => "\x{30E3}",

 # FF6D HALFWIDTH KATAKANA LETTER SMALL YU
 # ->   30E5 KATAKANA LETTER SMALL YU
 "\x{FF6D}" => "\x{30E5}",

 # FF6E HALFWIDTH KATAKANA LETTER SMALL YO
 # ->   30E7 KATAKANA LETTER SMALL YO
 "\x{FF6E}" => "\x{30E7}",

 # FF6F HALFWIDTH KATAKANA LETTER SMALL TU
 # ->   30C3 KATAKANA LETTER SMALL TU
 "\x{FF6F}" => "\x{30C3}",

 # FF70 HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
 # ->   30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK
 "\x{FF70}" => "\x{30FC}",

 # FF71 HALFWIDTH KATAKANA LETTER A
 # ->   30A2 KATAKANA LETTER A
 "\x{FF71}" => "\x{30A2}",

 # FF72 HALFWIDTH KATAKANA LETTER I
 # ->   30A4 KATAKANA LETTER I
 "\x{FF72}" => "\x{30A4}",

 # FF73 HALFWIDTH KATAKANA LETTER U
 # ->   30A6 KATAKANA LETTER U
 "\x{FF73}" => "\x{30A6}",

 # FF74 HALFWIDTH KATAKANA LETTER E
 # ->   30A8 KATAKANA LETTER E
 "\x{FF74}" => "\x{30A8}",

 # FF75 HALFWIDTH KATAKANA LETTER O
 # ->   30AA KATAKANA LETTER O
 "\x{FF75}" => "\x{30AA}",

 # FF76 HALFWIDTH KATAKANA LETTER KA
 # ->   30AB KATAKANA LETTER KA
 "\x{FF76}" => "\x{30AB}",

 # FF77 HALFWIDTH KATAKANA LETTER KI
 # ->   30AD KATAKANA LETTER KI
 "\x{FF77}" => "\x{30AD}",

 # FF78 HALFWIDTH KATAKANA LETTER KU
 # ->   30AF KATAKANA LETTER KU
 "\x{FF78}" => "\x{30AF}",

 # FF79 HALFWIDTH KATAKANA LETTER KE
 # ->   30B1 KATAKANA LETTER KE
 "\x{FF79}" => "\x{30B1}",

 # FF7A HALFWIDTH KATAKANA LETTER KO
 # ->   30B3 KATAKANA LETTER KO
 "\x{FF7A}" => "\x{30B3}",

 # FF7B HALFWIDTH KATAKANA LETTER SA
 # ->   30B5 KATAKANA LETTER SA
 "\x{FF7B}" => "\x{30B5}",

 # FF7C HALFWIDTH KATAKANA LETTER SI
 # ->   30B7 KATAKANA LETTER SI
 "\x{FF7C}" => "\x{30B7}",

 # FF7D HALFWIDTH KATAKANA LETTER SU
 # ->   30B9 KATAKANA LETTER SU
 "\x{FF7D}" => "\x{30B9}",

 # FF7E HALFWIDTH KATAKANA LETTER SE
 # ->   30BB KATAKANA LETTER SE
 "\x{FF7E}" => "\x{30BB}",

 # FF7F HALFWIDTH KATAKANA LETTER SO
 # ->   30BD KATAKANA LETTER SO
 "\x{FF7F}" => "\x{30BD}",

 # FF80 HALFWIDTH KATAKANA LETTER TA
 # ->   30BF KATAKANA LETTER TA
 "\x{FF80}" => "\x{30BF}",

 # FF81 HALFWIDTH KATAKANA LETTER TI
 # ->   30C1 KATAKANA LETTER TI
 "\x{FF81}" => "\x{30C1}",

 # FF82 HALFWIDTH KATAKANA LETTER TU
 # ->   30C4 KATAKANA LETTER TU
 "\x{FF82}" => "\x{30C4}",

 # FF83 HALFWIDTH KATAKANA LETTER TE
 # ->   30C6 KATAKANA LETTER TE
 "\x{FF83}" => "\x{30C6}",

 # FF84 HALFWIDTH KATAKANA LETTER TO
 # ->   30C8 KATAKANA LETTER TO
 "\x{FF84}" => "\x{30C8}",

 # FF85 HALFWIDTH KATAKANA LETTER NA
 # ->   30CA KATAKANA LETTER NA
 "\x{FF85}" => "\x{30CA}",

 # FF86 HALFWIDTH KATAKANA LETTER NI
 # ->   30CB KATAKANA LETTER NI
 "\x{FF86}" => "\x{30CB}",

 # FF87 HALFWIDTH KATAKANA LETTER NU
 # ->   30CC KATAKANA LETTER NU
 "\x{FF87}" => "\x{30CC}",

 # FF88 HALFWIDTH KATAKANA LETTER NE
 # ->   30CD KATAKANA LETTER NE
 "\x{FF88}" => "\x{30CD}",

 # FF89 HALFWIDTH KATAKANA LETTER NO
 # ->   30CE KATAKANA LETTER NO
 "\x{FF89}" => "\x{30CE}",

 # FF8A HALFWIDTH KATAKANA LETTER HA
 # ->   30CF KATAKANA LETTER HA
 "\x{FF8A}" => "\x{30CF}",

 # FF8B HALFWIDTH KATAKANA LETTER HI
 # ->   30D2 KATAKANA LETTER HI
 "\x{FF8B}" => "\x{30D2}",

 # FF8C HALFWIDTH KATAKANA LETTER HU
 # ->   30D5 KATAKANA LETTER HU
 "\x{FF8C}" => "\x{30D5}",

 # FF8D HALFWIDTH KATAKANA LETTER HE
 # ->   30D8 KATAKANA LETTER HE
 "\x{FF8D}" => "\x{30D8}",

 # FF8E HALFWIDTH KATAKANA LETTER HO
 # ->   30DB KATAKANA LETTER HO
 "\x{FF8E}" => "\x{30DB}",

 # FF8F HALFWIDTH KATAKANA LETTER MA
 # ->   30DE KATAKANA LETTER MA
 "\x{FF8F}" => "\x{30DE}",

 # FF90 HALFWIDTH KATAKANA LETTER MI
 # ->   30DF KATAKANA LETTER MI
 "\x{FF90}" => "\x{30DF}",

 # FF91 HALFWIDTH KATAKANA LETTER MU
 # ->   30E0 KATAKANA LETTER MU
 "\x{FF91}" => "\x{30E0}",

 # FF92 HALFWIDTH KATAKANA LETTER ME
 # ->   30E1 KATAKANA LETTER ME
 "\x{FF92}" => "\x{30E1}",

 # FF93 HALFWIDTH KATAKANA LETTER MO
 # ->   30E2 KATAKANA LETTER MO
 "\x{FF93}" => "\x{30E2}",

 # FF94 HALFWIDTH KATAKANA LETTER YA
 # ->   30E4 KATAKANA LETTER YA
 "\x{FF94}" => "\x{30E4}",

 # FF95 HALFWIDTH KATAKANA LETTER YU
 # ->   30E6 KATAKANA LETTER YU
 "\x{FF95}" => "\x{30E6}",

 # FF96 HALFWIDTH KATAKANA LETTER YO
 # ->   30E8 KATAKANA LETTER YO
 "\x{FF96}" => "\x{30E8}",

 # FF97 HALFWIDTH KATAKANA LETTER RA
 # ->   30E9 KATAKANA LETTER RA
 "\x{FF97}" => "\x{30E9}",

 # FF98 HALFWIDTH KATAKANA LETTER RI
 # ->   30EA KATAKANA LETTER RI
 "\x{FF98}" => "\x{30EA}",

 # FF99 HALFWIDTH KATAKANA LETTER RU
 # ->   30EB KATAKANA LETTER RU
 "\x{FF99}" => "\x{30EB}",

 # FF9A HALFWIDTH KATAKANA LETTER RE
 # ->   30EC KATAKANA LETTER RE
 "\x{FF9A}" => "\x{30EC}",

 # FF9B HALFWIDTH KATAKANA LETTER RO
 # ->   30ED KATAKANA LETTER RO
 "\x{FF9B}" => "\x{30ED}",

 # FF9C HALFWIDTH KATAKANA LETTER WA
 # ->   30EF KATAKANA LETTER WA
 "\x{FF9C}" => "\x{30EF}",

 # FF9D HALFWIDTH KATAKANA LETTER N
 # ->   30F3 KATAKANA LETTER N
 "\x{FF9D}" => "\x{30F3}",

 # FFA0 HALFWIDTH HANGUL FILLER
 # ->   1160 HANGUL JUNGSEONG FILLER
 "\x{FFA0}" => "\x{1160}",

 # FFA1 HALFWIDTH HANGUL LETTER KIYEOK
 # ->   1100 HANGUL CHOSEONG KIYEOK
 "\x{FFA1}" => "\x{1100}",

 # FFA2 HALFWIDTH HANGUL LETTER SSANGKIYEOK
 # ->   1101 HANGUL CHOSEONG SSANGKIYEOK
 "\x{FFA2}" => "\x{1101}",

 # FFA3 HALFWIDTH HANGUL LETTER KIYEOK-SIOS
 # ->   11AA HANGUL JONGSEONG KIYEOK-SIOS
 "\x{FFA3}" => "\x{11AA}",

 # FFA4 HALFWIDTH HANGUL LETTER NIEUN
 # ->   1102 HANGUL CHOSEONG NIEUN
 "\x{FFA4}" => "\x{1102}",

 # FFA5 HALFWIDTH HANGUL LETTER NIEUN-CIEUC
 # ->   11AC HANGUL JONGSEONG NIEUN-CIEUC
 "\x{FFA5}" => "\x{11AC}",

 # FFA6 HALFWIDTH HANGUL LETTER NIEUN-HIEUH
 # ->   11AD HANGUL JONGSEONG NIEUN-HIEUH
 "\x{FFA6}" => "\x{11AD}",

 # FFA7 HALFWIDTH HANGUL LETTER TIKEUT
 # ->   1103 HANGUL CHOSEONG TIKEUT
 "\x{FFA7}" => "\x{1103}",

 # FFA8 HALFWIDTH HANGUL LETTER SSANGTIKEUT
 # ->   1104 HANGUL CHOSEONG SSANGTIKEUT
 "\x{FFA8}" => "\x{1104}",

 # FFA9 HALFWIDTH HANGUL LETTER RIEUL
 # ->   1105 HANGUL CHOSEONG RIEUL
 "\x{FFA9}" => "\x{1105}",

 # FFAA HALFWIDTH HANGUL LETTER RIEUL-KIYEOK
 # ->   11B0 HANGUL JONGSEONG RIEUL-KIYEOK
 "\x{FFAA}" => "\x{11B0}",

 # FFAB HALFWIDTH HANGUL LETTER RIEUL-MIEUM
 # ->   11B1 HANGUL JONGSEONG RIEUL-MIEUM
 "\x{FFAB}" => "\x{11B1}",

 # FFAC HALFWIDTH HANGUL LETTER RIEUL-PIEUP
 # ->   11B2 HANGUL JONGSEONG RIEUL-PIEUP
 "\x{FFAC}" => "\x{11B2}",

 # FFAD HALFWIDTH HANGUL LETTER RIEUL-SIOS
 # ->   11B3 HANGUL JONGSEONG RIEUL-SIOS
 "\x{FFAD}" => "\x{11B3}",

 # FFAE HALFWIDTH HANGUL LETTER RIEUL-THIEUTH
 # ->   11B4 HANGUL JONGSEONG RIEUL-THIEUTH
 "\x{FFAE}" => "\x{11B4}",

 # FFAF HALFWIDTH HANGUL LETTER RIEUL-PHIEUPH
 # ->   11B5 HANGUL JONGSEONG RIEUL-PHIEUPH
 "\x{FFAF}" => "\x{11B5}",

 # FFB0 HALFWIDTH HANGUL LETTER RIEUL-HIEUH
 # ->   111A HANGUL CHOSEONG RIEUL-HIEUH
 "\x{FFB0}" => "\x{111A}",

 # FFB1 HALFWIDTH HANGUL LETTER MIEUM
 # ->   1106 HANGUL CHOSEONG MIEUM
 "\x{FFB1}" => "\x{1106}",

 # FFB2 HALFWIDTH HANGUL LETTER PIEUP
 # ->   1107 HANGUL CHOSEONG PIEUP
 "\x{FFB2}" => "\x{1107}",

 # FFB3 HALFWIDTH HANGUL LETTER SSANGPIEUP
 # ->   1108 HANGUL CHOSEONG SSANGPIEUP
 "\x{FFB3}" => "\x{1108}",

 # FFB4 HALFWIDTH HANGUL LETTER PIEUP-SIOS
 # ->   1121 HANGUL CHOSEONG PIEUP-SIOS
 "\x{FFB4}" => "\x{1121}",

 # FFB5 HALFWIDTH HANGUL LETTER SIOS
 # ->   1109 HANGUL CHOSEONG SIOS
 "\x{FFB5}" => "\x{1109}",

 # FFB6 HALFWIDTH HANGUL LETTER SSANGSIOS
 # ->   110A HANGUL CHOSEONG SSANGSIOS
 "\x{FFB6}" => "\x{110A}",

 # FFB7 HALFWIDTH HANGUL LETTER IEUNG
 # ->   110B HANGUL CHOSEONG IEUNG
 "\x{FFB7}" => "\x{110B}",

 # FFB8 HALFWIDTH HANGUL LETTER CIEUC
 # ->   110C HANGUL CHOSEONG CIEUC
 "\x{FFB8}" => "\x{110C}",

 # FFB9 HALFWIDTH HANGUL LETTER SSANGCIEUC
 # ->   110D HANGUL CHOSEONG SSANGCIEUC
 "\x{FFB9}" => "\x{110D}",

 # FFBA HALFWIDTH HANGUL LETTER CHIEUCH
 # ->   110E HANGUL CHOSEONG CHIEUCH
 "\x{FFBA}" => "\x{110E}",

 # FFBB HALFWIDTH HANGUL LETTER KHIEUKH
 # ->   110F HANGUL CHOSEONG KHIEUKH
 "\x{FFBB}" => "\x{110F}",

 # FFBC HALFWIDTH HANGUL LETTER THIEUTH
 # ->   1110 HANGUL CHOSEONG THIEUTH
 "\x{FFBC}" => "\x{1110}",

 # FFBD HALFWIDTH HANGUL LETTER PHIEUPH
 # ->   1111 HANGUL CHOSEONG PHIEUPH
 "\x{FFBD}" => "\x{1111}",

 # FFBE HALFWIDTH HANGUL LETTER HIEUH
 # ->   1112 HANGUL CHOSEONG HIEUH
 "\x{FFBE}" => "\x{1112}",

 # FFC2 HALFWIDTH HANGUL LETTER A
 # ->   1161 HANGUL JUNGSEONG A
 "\x{FFC2}" => "\x{1161}",

 # FFC3 HALFWIDTH HANGUL LETTER AE
 # ->   1162 HANGUL JUNGSEONG AE
 "\x{FFC3}" => "\x{1162}",

 # FFC4 HALFWIDTH HANGUL LETTER YA
 # ->   1163 HANGUL JUNGSEONG YA
 "\x{FFC4}" => "\x{1163}",

 # FFC5 HALFWIDTH HANGUL LETTER YAE
 # ->   1164 HANGUL JUNGSEONG YAE
 "\x{FFC5}" => "\x{1164}",

 # FFC6 HALFWIDTH HANGUL LETTER EO
 # ->   1165 HANGUL JUNGSEONG EO
 "\x{FFC6}" => "\x{1165}",

 # FFC7 HALFWIDTH HANGUL LETTER E
 # ->   1166 HANGUL JUNGSEONG E
 "\x{FFC7}" => "\x{1166}",

 # FFCA HALFWIDTH HANGUL LETTER YEO
 # ->   1167 HANGUL JUNGSEONG YEO
 "\x{FFCA}" => "\x{1167}",

 # FFCB HALFWIDTH HANGUL LETTER YE
 # ->   1168 HANGUL JUNGSEONG YE
 "\x{FFCB}" => "\x{1168}",

 # FFCC HALFWIDTH HANGUL LETTER O
 # ->   1169 HANGUL JUNGSEONG O
 "\x{FFCC}" => "\x{1169}",

 # FFCD HALFWIDTH HANGUL LETTER WA
 # ->   116A HANGUL JUNGSEONG WA
 "\x{FFCD}" => "\x{116A}",

 # FFCE HALFWIDTH HANGUL LETTER WAE
 # ->   116B HANGUL JUNGSEONG WAE
 "\x{FFCE}" => "\x{116B}",

 # FFCF HALFWIDTH HANGUL LETTER OE
 # ->   116C HANGUL JUNGSEONG OE
 "\x{FFCF}" => "\x{116C}",

 # FFD2 HALFWIDTH HANGUL LETTER YO
 # ->   116D HANGUL JUNGSEONG YO
 "\x{FFD2}" => "\x{116D}",

 # FFD3 HALFWIDTH HANGUL LETTER U
 # ->   116E HANGUL JUNGSEONG U
 "\x{FFD3}" => "\x{116E}",

 # FFD4 HALFWIDTH HANGUL LETTER WEO
 # ->   116F HANGUL JUNGSEONG WEO
 "\x{FFD4}" => "\x{116F}",

 # FFD5 HALFWIDTH HANGUL LETTER WE
 # ->   1170 HANGUL JUNGSEONG WE
 "\x{FFD5}" => "\x{1170}",

 # FFD6 HALFWIDTH HANGUL LETTER WI
 # ->   1171 HANGUL JUNGSEONG WI
 "\x{FFD6}" => "\x{1171}",

 # FFD7 HALFWIDTH HANGUL LETTER YU
 # ->   1172 HANGUL JUNGSEONG YU
 "\x{FFD7}" => "\x{1172}",

 # FFDA HALFWIDTH HANGUL LETTER EU
 # ->   1173 HANGUL JUNGSEONG EU
 "\x{FFDA}" => "\x{1173}",

 # FFDB HALFWIDTH HANGUL LETTER YI
 # ->   1174 HANGUL JUNGSEONG YI
 "\x{FFDB}" => "\x{1174}",

 # FFDC HALFWIDTH HANGUL LETTER I
 # ->   1175 HANGUL JUNGSEONG I
 "\x{FFDC}" => "\x{1175}",

 # FFE0 FULLWIDTH CENT SIGN
 # ->   00A2 CENT SIGN
 "\x{FFE0}" => "\xA2",

 # FFE1 FULLWIDTH POUND SIGN
 # ->   00A3 POUND SIGN
 "\x{FFE1}" => "\xA3",

 # FFE2 FULLWIDTH NOT SIGN
 # ->   00AC NOT SIGN
 "\x{FFE2}" => "\xAC",

 # FFE3 FULLWIDTH MACRON
 # ->   0020 SPACE
 "\x{FFE3}" => " ",

 # FFE4 FULLWIDTH BROKEN BAR
 # ->   00A6 BROKEN BAR
 "\x{FFE4}" => "\xA6",

 # FFE5 FULLWIDTH YEN SIGN
 # ->   00A5 YEN SIGN
 "\x{FFE5}" => "\xA5",

 # FFE6 FULLWIDTH WON SIGN
 # ->   20A9 WON SIGN
 "\x{FFE6}" => "\x{20A9}",

 # FFE8 HALFWIDTH FORMS LIGHT VERTICAL
 # ->   2502 BOX DRAWINGS LIGHT VERTICAL
 "\x{FFE8}" => "\x{2502}",

 # FFE9 HALFWIDTH LEFTWARDS ARROW
 # ->   2190 LEFTWARDS ARROW
 "\x{FFE9}" => "\x{2190}",

 # FFEA HALFWIDTH UPWARDS ARROW
 # ->   2191 UPWARDS ARROW
 "\x{FFEA}" => "\x{2191}",

 # FFEB HALFWIDTH RIGHTWARDS ARROW
 # ->   2192 RIGHTWARDS ARROW
 "\x{FFEB}" => "\x{2192}",

 # FFEC HALFWIDTH DOWNWARDS ARROW
 # ->   2193 DOWNWARDS ARROW
 "\x{FFEC}" => "\x{2193}",

 # FFED HALFWIDTH BLACK SQUARE
 # ->   25A0 BLACK SQUARE
 "\x{FFED}" => "\x{25A0}",

 # FFEE HALFWIDTH WHITE CIRCLE
 # ->   25CB WHITE CIRCLE
 "\x{FFEE}" => "\x{25CB}",

};

#for my $chr_in (sort keys %$map) {
#    my $str_out = $map->{$chr_in};
#    printf "$chr_in -> $str_out\n";
#    print utf8::is_utf8($chr_in) ? 1 : 0, "\n";
#    print utf8::is_utf8($str_out) ? 1 : 0, "\n";
#}

=pod

=head1 FUNCTIONS

=over 4

=item unac_string CHARACTER_STRING

=item unac_string ENCODING, OCTET_STRING

Return the unaccented equivalent to the input string. The one-argument version
assumes the input is a Perl string, i.e., a sequence of characters. (A
character is in the range 0...(2**32-1), or more).

The two-argument version assumes the input is a sequence of octets, i.e., raw,
encoded data. (An octet is eight bits of data with ordinal value in the range
0...255.) It is essentially equivalent to the following unaccent() function

  use Text::Unaccent;
  use Encode;

  sub unaccent {
      ($enc, $oct) = @_;
      encode($enc, unac_string(decode($enc, $oct)));
  }

except that this module's unac_string() doesn't require the Encode module nor
the C compiler required to compile Text::Unaccent.

=cut

sub unac_string ($;$) {
    my $subname = 'unac_string';

    # These two lines aren't necessary with the function prototype.
    #croak "$subname: Not enough input arguments\n" if @_ < 1;
    #croak "$subname: Too many input arguments\n"   if @_ > 2;

    carp "$subname: Useless use of function in void context\n"
      unless defined wantarray;

    my $str_in;

    my $do_recode = @_ == 2;
    my $encoding;

    if ($do_recode) {

        $encoding = shift;
        croak "$subname: Input encoding is undefined\n"
          unless defined $encoding;

        $str_in = shift;
        croak "$subname: Input octet string is undefined\n"
          unless defined $str_in;

        # Decode octets to characters.

        require Encode;
        $str_in = Encode::decode($encoding, $str_in);

    } else {

        $str_in = shift;
        croak "$subname: Input character string is undefined\n"
          unless defined $str_in;
    }

    my $length = length($str_in);
    my $str_out = '';

    # Iterate over each character in the input string. If the character exists
    # in the map, replace the current character according to the map, otherwise
    # keep the character as it is.

    for (my $offset = 0 ; $offset < $length ; ++ $offset) {
        my $chr = substr($str_in, $offset, 1);
        $str_out .= exists $map->{$chr} ? $map->{$chr} : $chr;
        #print "# <$chr> -> <$map->{$chr}>\n" if exists $map->{$chr};
    }

    # Encode characters to octets.

    if ($do_recode) {
        $str_out = Encode::encode($encoding, $str_out);
    }

    return $str_out;
}

=pod

=item unac_string_utf16 OCTET_STRING

This function is mainly provided for compatibility with Text::Unaccent. It is
equivalent to

    unac_string("UTF-16BE", OCTET_STRING);

=cut

sub unac_string_utf16 ($) {
    my $subname = 'unac_string_utf16';

    # These two lines aren't necessary with the function prototype.
    #croak "$subname: Not enough input arguments\n" if @_ < 0;
    #croak "$subname: Too many input arguments\n"   if @_ > 0;

    unac_string('UTF-16BE', $_[0]);
}

=pod

=item unac_version

This function is provided only for compatibility with Text::Unaccent. It
returns the version of this module.

=cut

sub unac_version {
    my $subname = 'unac_version';
    #croak "$subname: Not enough input arguments\n" if @_ < 0;
    croak "$subname: Too many input arguments\n"   if @_ > 0;

    carp "$subname: Useless use of function in void context\n"
      unless defined wantarray;

    return $VERSION;
}

=pod

=item unac_debug LEVEL

This function is provided only for compatibility with Text::Unaccent. It has no
effect on the behaviour of this module.

=cut

sub unac_debug ($) {
    my $subname = 'unac_debug';

    # These two lines aren't necessary with the function prototype.
    #croak "$subname: Not enough input arguments\n" if @_ < 1;
    #croak "$subname: Too many input arguments\n"   if @_ > 1;

    return 1;
}

=pod

=back

=head1 EXAMPLES

=head2 French

  $str1 = "dj vu";
  $str2 = unac_string($str1);
  #     = "deja vu";

=head2 Greek

The following example uses one of the Greek words for the English indefinite
article I<a>, as in I<a house>. (Greek has genders, so there are three words
for the English I<a>.)  It is a three-letter word made up by epsilon with
tonos, nu, and alpha.

  $str1 = "\x{03AD}\x{03BD}\x{03B1}";
  $str2 = unac_string($str1);
  #     = "\x{03B5}\x{03BD}\x{03B1}"

The unaccented string $str2 is made up by the three letters epsilon (without
the tonos), nu, and alpha.

In contrast, the version of unac_string() in the Text::Unaccent module gives

  $oct2 = unac_string("UTF-8", $str1);
  #     = "\xCE\xB5\xCE\xBD\xCE\xB1"

These octets are the UTF-8 encoded equivalent of C<"\x{03B5}\x{03BD}\x{03B1}">.

=head1 BUGS

There are currently no known bugs.

Please report any bugs or feature requests to
C<bug-text-unaccent-pureperl at rt.cpan.org>, or through the web interface at
L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Text-Unaccent-PurePerl>.
I will be notified, and then you'll automatically be notified of progress on
your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Text::Unaccent::PurePerl

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Text-Unaccent-PurePerl>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Text-Unaccent-PurePerl>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Text-Unaccent-PurePerl>

=item * Search CPAN

L<http://search.cpan.org/dist/Text-Unaccent-PurePerl>

=back

=head1 SEE ALSO

Text::Unaccent(3).

=head1 AUTHOR

Peter J. Acklam, E<lt>pjacklam@cpan.orgE<gt>

=head1 COPYRIGHT & LICENSE

Copyright 2008 Peter J. Acklam, all rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut

1;
