package Log::Any::Adapter::Journal;

# ABSTRACT: Adapter for Log::Any that outputs with a priority prefix that systemd's journal can parse

use 5.010;
use strict;
use warnings;

use Log::Any::Adapter::Util qw(logging_methods numeric_level);
use parent 'Log::Any::Adapter::Screen';
use Class::Method::Modifiers;

our $VERSION = '0.10';

# sub init {
#     my ($self) = @_;
# }

# Journal doesn't recognize trace, use debug instead
sub trace    { shift->debug(@_) }
sub is_trace { shift->is_debug(@_) }

# For each of the logging methods exposed by Log::Any, add the level
# prefix in angled brackets.
for my $method ( grep { !/trace/ } logging_methods ) {
    my $level = numeric_level( $method );

    before $method => sub {
        $_[1] = "<$level>$_[1]" unless $_[0]->{use_color};
    };
}

# Log::Any levels       Journal levels
# 0 emergency           0 emerg
# 1 alert               1 alert
# 2 critical            2 crit
# 3 error               3 err
# 4 warning             4 warning
# 5 notice              5 notice
# 6 info                6 info
# 7 debug               7 debug
# 8 trace

1;

__END__

=pod

=head1 NAME

Log::Any::Adapter::Journal - Adapter for Log::Any that outputs with a priority prefix that systemd's journal can parse

=head1 VERSION

version 0.10

=head1 STATUS

=for html <a href="https://travis-ci.org/mvgrimes/Log-Any-Adapter-Journal"><img src="https://travis-ci.org/mvgrimes/Log-Any-Adapter-Journal.svg?branch=master" alt="Build Status"></a>
<a href="https://metacpan.org/pod/Log::Any::Adapter::Journal"><img alt="CPAN version" src="https://badge.fury.io/pl/Log-Any-Adapter-Journal.svg" /></a>

=head1 SYNOPSIS

    use Log::Any::Adapter;
    Log::Any::Adapter->set( 'Journal', 
        # min_level => 'debug', # default is 'warning'
        # colors    => { trace => 'bold yellow on_gray', ... }, # customize colors
        # use_color => 1, # force color even when not interactive
        # stderr    => 0, # print to STDOUT instead of the default STDERR
        # formatter => sub { "LOG: $_[1]" }, # default none
    );

=head1 DESCRIPTION

Desciption to come...

=head1 SEE ALSO

L<Log::Any>, L<Log::Any::Adapter::Journal>

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<http://github.com/mvgrimes/Log-Any-Adapter-Journal/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

Mark Grimes E<lt>mgrimes@cpan.orgE<gt>

=head1 SOURCE

Source repository is at L<https://github.com/mvgrimes/Log-Any-Adapter-Journal>.

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by Mark Grimes E<lt>mgrimes@cpan.orgE<gt>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
