#!perl6

use v6;

unit module Tomtit:ver<0.0.17>;
use File::Directory::Tree;

my %profiles is Map = (
  'perl' => ( 'set-pause', 'make-dist', 'install', 'test', 'release' ),
  'perl6' =>  ( 'set-pause', 'test', 'install', 'release' ),
  'git' => ( 'set-git', 'commit', 'push', 'pull', 'status' ),
  'ruby' => ( 'rvm' ),
  'ado' => ( 'build', 'list', 'az-resources', 'az-account-set' ),
  'hello' => ( 'world' )
);

# tom cli initializer
  
sub init () is export {

  mkdir ".tom/.cache";

}

sub tomtit-usage () is export  {
  say 'usage: tom $action|$options $thing'
}

sub tomtit-help () is export  {
  say q:to/DOC/;
  usage:
    tom $action|$options $thing

  run scenario:
    tom $scenario

  remove scenario:
    tom --remove $scenario

  print out scenario:
    tom --cat $scenario

  install profile:
    tom --profile $profile

  actions:
    tom --bootstrap         # bootstrap system
    tom --list              # list available scenarios
    tom --profile           # list available profiles
    tom --profile $profile  # list profile scenarios
    tom --last              # what is the last run?
    tom --completion        # install Bash completion

  options:
    --verbose   # run scenarios in verbose mode
    --quiet,-q  # run scenrios in less verbose mode
  DOC
}

# clean tomtit internal data
# is useful as with time it might grow

sub tomtit-clean ($dir) is export { 

  say "cleaning $dir/.cache ...";

  if "$dir/.cache/".IO ~~ :e {
    empty-directory "$dir/.cache"
  }

}

sub scenario-last ($dir) is export {
  if "$dir/.cache/history".IO ~~ :e {
    my @history =  "$dir/.cache/history".IO.lines;
    say @history[*-1];
  }
}

sub tomtit-bootstrap () is export {

  my $cmd = "sparrowdo --local_mode --bootstrap";

  shell($cmd);

}

sub scenario-run ($dir,$scenario,%args?) is export {

  die "scenario $scenario not found" unless "$dir/$scenario.pl6".IO ~~ :e;

  my $fh = open "$dir/.cache/history", :a;
  $fh.print($scenario,"\n");
  $fh.close;

  my $cmd = "sparrowdo --local_mode --no_sudo --sparrow_root=$dir/.cache --sparrowfile=$dir/$scenario.pl6";

  $cmd ~= " -q" if %args<quiet>;

  if %args<verbose> {
    $cmd ~= " --verbose";
    say "run $cmd ...";
  }

  shell($cmd);

}

sub scenario-remove ($dir,$scenario) is export {

  if "$dir/$scenario.pl6".IO ~~ :e {
    unlink "$dir/$scenario.pl6";
    say "scenario $scenario removed"
  } else {
    say "scenario $scenario not found"
  }

}

sub scenario-cat ($dir,$scenario,%args?) is export {

  if "$dir/$scenario.pl6".IO ~~ :e {
    say "[scenario $scenario]";
    my $i=0;
    
    for "$dir/$scenario.pl6".IO.lines -> $l {
      $i++;
      say %args<lines> ?? "[$i] $l" !! $l;
    }
  } else {
    say "scenario $scenario not found"
  }

}


# this is one is implemented, but not presented in public API, as we
# have issues with compiling Sparrowdo scenarios
  
sub scenario-doc ($dir,$scenario) is export {

  die "scenario $scenario not found" unless "$dir/$scenario.pl6".IO ~~ :e;

  run $*EXECUTABLE, '--doc', "$dir/$scenario.pl6";

}

sub scenario-list ($dir) is export {

    say "[scenarios list]";
    my @list = Array.new;

    for dir($dir) -> $f {

      next unless "$f".IO ~~ :f;
      next unless $f ~~ /\.pl6$/;
      my $scenario-name = substr($f.basename,0,($f.basename.chars)-4);
      @list.push($scenario-name);

    }

    say join "\n", @list.sort;

}

multi sub profile-list  is export {

  say "[profiles]";

  for %profiles.keys.sort -> $i {
    say "$i"
  }
}

multi sub profile-list($dir,$profile)  is export {

  unless %profiles{$profile}:exists {
    say "profile $profile does not exist";
    return;
  }

  say "[profile scenarios]";

  for %profiles{$profile}.sort -> $s {

    my $installed = "$dir/$s.pl6".IO ~~ :f;
    say "$profile\@$s\tinstalled: $installed";

  }

}

sub profile-install ($dir, $profile is copy,%args?) is export {

  my @list;

  if $profile ~~ s/ '@' (.*) // {
    my $s1 = $0;
    @list = sort &[cmp], grep { $_ eq $s1 }, %profiles{$profile}
  } else {
    @list = %profiles{$profile}.sort
  }

  unless %profiles{$profile}:exists {
    say "profile $profile does not exist";
    return;
  }

  if @list.elems == 0 {
    say "no scenarios found ...";
    return;
  }

  for @list -> $s {
    if %?RESOURCES{"profiles/$profile/$s.pl6"}.Str.IO ~~ :f {
      say "install $profile\@$s ...";
      my $fh = open "$dir/$s.pl6", :w;
      $fh.print(slurp %?RESOURCES{"profiles/$profile/$s.pl6"}.Str);
      $fh.close;
    } else {
      say "no perl6 resource found for  $profile\@$s scenario ... skipping it";
    }
  }
    
}


sub completion-install () is export {

  say "install completion.sh ...";

  my $fh = open '/home/' ~ %*ENV<USER> ~ '/.tom_completion.sh' , :w;

  $fh.print(slurp %?RESOURCES{"completion.sh"}.Str);

  $fh.close;

  say "to activate completion say: source " ~ '/home/' ~ %*ENV<USER> ~ '/.tom_completion.sh';  
    
}
