package Ubic::Service::Toadfarm;

=head1 NAME

Ubic::Service::Toadfarm - Ubic Toadfarm service class

=head1 DESCRIPTION

This class require L<Ubic> which is not automatically installed by
L<Toadfarm>.

=head1 SYNOPSIS

Put the code below in a C<ubic> service file:

  use Ubic::Service::Toadfarm;
  Ubic::Service::Toadfarm->new(
    log => {
      file => '/path/to/log/file' # required
      combined => 1,
    },

    # toadfarm config args
    secret => 'super secret',
    apps => [...],
    plugins => [...],
  );

=head2 Details

=over 4

=item * pid_file

This file is created automatically. It will be stored in the "tmp" directory
in your "ubic" data directory. This means that you do not have to specify the
pid_file in the "hypnotoad" config section.

=item * MOJO_CONFIG

The Toadfarm application will be started with a config file generated by this
service class. The config file will be stored in the "tmp" directory in your
"ubic" data directory.

=back

=head2 Hypnotoad starter

It is possible to use this module as a generic C<hypnotoad> starter, instead
of L<Ubic::Service::Hypnotoad>, by setting the "HYPNOTOAD_APP" environment
variable:

  $ENV{HYPNOTOAD_APP} = '/path/to/my-mojo-app';
  use Ubic::Service::Toadfarm;
  Ubic::Service::Toadfarm->new(...);

=cut

use strict;
use warnings;
use Data::Dumper ();
use File::Basename;
use File::Spec::Functions qw(catfile);
use File::Which qw(which);
use Ubic::Result qw(result);
use Ubic::Settings;
use constant DEBUG => $ENV{UBIC_TOADFARM_DEBUG} || 0;

use parent 'Ubic::Service::Skeleton';

$ENV{HYPNOTOAD_APP} ||= which 'toadfarm';

=head1 METHODS

=head2 new

See L</SYNOPSIS>.

=cut

sub new {
  my $class = shift;
  my $args = @_ ? @_ > 1 ? {@_} : {%{$_[0]}} : {};
  my $self = bless $args, $class;

  ref $self->{hypnotoad}{listen} eq 'ARRAY' or die 'Invalid/missing hypnotoad => listen';
  $self->{stderr} ||= $self->{log}{file} or die "Missing log => file";
  $self->{stdout} ||= $self->{log}{file};
  $self->{ubic_log} ||= $self->{log}{file};

  warn Data::Dumper::Dumper($self) if DEBUG == 2;
  return $self;
}

=head2 start_impl

This is called when you run C<ubic start>. It will start L<toadfarm|Toadfarm>
using L<hypnotoad|Mojo::Server::Hypnotoad> after writing the toadfarm config
and settings L<MOJO_CONFIG>.

The config will be written to the "tmp" directory in ubic's data directory.

=cut

sub start_impl {
  my $self = shift;
  my $hypnotoad = which 'hypnotoad';

  $self->_write_mojo_config;
  warn "MOJO_CONFIG=$ENV{MOJO_CONFIG} system $hypnotoad $ENV{HYPNOTOAD_APP}\n" if DEBUG;
  system $hypnotoad => $ENV{HYPNOTOAD_APP};
}

=head2 status_impl

This method will issue a HTTP "HEAD /ubic-status" request to the server. The
response is not important, the important thing is that the server responds.

=cut

sub status_impl {
  my $self = shift;
  my $listen = $self->{hypnotoad}{listen}[0];
  my %args = ( connect_timeout => 5, request_timeout => 20 );
  my $tx;

  require Mojo::UserAgent;

  $listen =~ s!\*!localhost!;
  $tx = Mojo::UserAgent->new(%args)->head("$listen/ubic-status");
  warn $tx->res->code // 'No HTTP code', "\n" if DEBUG;

  if($tx->res->code) {
    return result 'running';
  }
  else {
    return result 'not running';
  }
}

=head2 stop_impl

This method will kill the server pid found in "pid_file" with "TERM".

=cut

sub stop_impl {
  my $self = shift;
  my $pid = $self->_read_pid;

  warn "pid=$pid\n" if DEBUG;
  return result 'not running' unless $pid;
  warn "kill TERM $pid\n" if DEBUG;
  return result 'not running' unless kill 'TERM', $pid;
  return result 'stopped';
}

=head2 reload

This method will reload the server pid found in "pid_file" with "USR2".

=cut

sub reload {
  my $self = shift;
  my $pid = $self->_read_pid;

  warn "pid=$pid\n" if DEBUG;
  return result 'not running' unless $pid;
  warn "kill USR2 $pid\n" if DEBUG;
  return result 'not running' unless kill 'USR2', $pid;
  return result 'reloaded';
}

sub _path_to_mojo_config {
  $ENV{MOJO_CONFIG} = catfile(
    Ubic::Settings->data_dir,
    'tmp',
    'toadfarm-' .$_[0]->name .'.conf'
  );
}

sub _path_to_pid_file {
  catfile(Ubic::Settings->data_dir, 'tmp', 'toadfarm-' .$_[0]->name .'.pid');
}

sub _read_pid {
  my $self = shift;
  my $pid_file = $self->{hypnotoad}{pid_file} || $self->_path_to_pid_file;

  return 0 unless -e $pid_file;
  return eval {
    open my $PID, $pid_file or die "Could not read $pid_file: $!";
    scalar(<$PID>) =~ /(\d+)/g ? $1 : 0;
  };
}

sub _write_mojo_config {
  my($self, $args) = @_;
  my $data_dir = Ubic::Settings->data_dir;
  my $file = $self->_path_to_mojo_config;
  my %config = %$self;
  my $dumper = Data::Dumper->new([\%config]);

  delete $config{$_} for qw( stderr stdout ubic_log );
  $config{hypnotoad}{pid_file} ||= $self->_path_to_pid_file;

  open my $CONFIG, '>', $file or die "Could not write $file: $!";
  print $CONFIG $dumper->Indent(1)->Sortkeys(1)->Terse(1)->Deepcopy(1)->Dump;
  close $CONFIG or die "Could not write $file: $!";
}

=head1 AUTHOR

Jan Henning Thorsen - C<jhthorsen@cpan.org>

=cut

1;
