use strict;
use warnings;

eval { require Devel::CheckLib; Devel::CheckLib->import; };
use 5.010_000;
use ExtUtils::MakeMaker;
use ExtUtils::MakeMaker::Config;
use File::Spec;

# file globals(!)
my ($pdl_f77conf, $forcebuild);
my ($seen_f77conf);

sub checkbuggysetup {
    # detect buggy Perl setups
    if (!$forcebuild &&
	$Config{osname} eq 'solaris' &&
	($Config{cc} =~ /gcc/ || $Config{gccversion} ne '') &&
	$Config{usemymalloc} eq 'y'
	) {
	die <<'EOM';

FATAL BUG IN YOUR PERL SETUP DETECTED. BUILD TERMINATED.

On this platform the combination of gcc and the Perl malloc
are buggy. The optimizations lead to random coredumps
and make PDL essentially unusable.

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
WORKAROUND: YOU MUST RECOMPILE PERL WITH 'usemymalloc=n' !!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

You may override and force the build by including the
'FORCE' switch:

  perl Makefile.PL FORCE

However, you will most likely end up with an unusable
PDL build unless *all* optimizations are disabled!
YOU HAVE BEEN WARNED!!

EOM
  }

    my($dot);
    for(@INC) {$dot += $_ eq '.'}
    $dot-- if($INC[-1] eq '.');
    if($dot) {
	$INC = join(":",@INC);
	warn << "EOM"

**********************************************************************
Your Perl \@INC path is:

$INC

That seems to include the directory '.' before the last entry in the
path.  Configurations like that might be broken, because they may
prevent strict division of hierarchy in the module name space.
In particular, adding a module Foo::Bar::Baz can cause trouble
if Baz exists in the top level elsewhere, and if you run from the
Foo/Bar directory.

This happens with the PDL::Config and some other modules.  You may
not be able to build PDL with this \@INC setup.

If you strike trouble, you may need to fix your \@INC.
**********************************************************************


EOM
unless $ENV{PDL_INC_OK};
}

}

# The user specifies configuration options using the PDL_CONFIG
# array in perldl.conf - or ~/.perldl.conf or via the PDLCONF command-line
# argument.
#
# We need to use this information during the 'perl Makefile.PL' process,
# and historically this has been done by directly accessing %PDL_CONFIG
# (or %::PDL_CONFIG) variable. However, there are times that this information
# is also needed during the actual build (ie 'make' cycle), at which point
# the variable is not available. However Basic/Core/Config.pm is available,
# but this uses the %PDL::Config variable to access the data.
#
# So, we try to avoid this dichotomy by creating a %PDL::Config during
# the 'perl Makefile.PL' stage and then making sure that PDL_CONFIG
# is destroyed (so it can't be used by accident).
#
# Do we need to worry about Makefile.PL's hacking PDL_CONFIG?
# Probably...
#
%PDL::Config = ();

sub getpdl_config {
  # read in distribution config file
  our %PDL_CONFIG;
  require './perldl.conf';

  # set up the default MALLOCDBG information (if not specified)
  #
  if ( exists $PDL_CONFIG{MALLOCDBG} and defined $PDL_CONFIG{MALLOCDBG} ) {
      my $val = $PDL_CONFIG{MALLOCDBG};
      my $rval = ref($val);
      die "ERROR: MALLOCDBG must be set to an associative array, not to a " .
	( $rval ? $rval : "SCALAR" ) unless $rval eq "HASH";
  } else {
      $PDL_CONFIG{MALLOCDBG} = undef; # exists but undef
  }
  foreach my $item ( qw( libs include ) ) {
      $PDL_CONFIG{MALLOCDBG}->{$item} = ""
	unless exists $PDL_CONFIG{MALLOCDBG}->{$item};
  }

  # create a PDL::Config variable matching the PDL_CONFIG structure
  # and convert yes/no answers to 1/0
  #
  for my $key ( keys %PDL_CONFIG ) {
      my $val = $PDL_CONFIG{$key};
      $val =~ s/^y.*/1/i if defined $val;
      $val =~ s/^n.*/0/i if defined $val;
      $PDL::Config{$key} = $val;
  }

  # destroy PDL_CONFIG so that we can catch any accesses
  # to them in other Makefile.PL's
  #
  %PDL_CONFIG = ();
}

sub check_f77conf {
  my ($seenyet) = @_;
  return 0 unless $seenyet;

  my $pdl_f77conf = File::Spec->rel2abs($pdl_f77conf)
    unless File::Spec->file_name_is_absolute($pdl_f77conf);
  $PDL::Config{F77CONF} = $pdl_f77conf;
  return 1;
}

sub make_PDL_Config_pm {
  print "Writing Basic/Core/Config.pm\n";
  my $versionFile = "Basic/Core/Config.pm";
  require Data::Dumper;
  no warnings 'once';
  local $Data::Dumper::Indent = 1;
  local $Data::Dumper::Terse = 1;
  local $Data::Dumper::Sortkeys = 1;
  open my $fh, ">", $versionFile
    or die "Couldn't open $versionFile for writing";
  print $fh "
# AUTOMATICALLY GENERATED BY THE PDL TOPLEVEL Makefile.PL.
# DO NOT HAND-EDIT - CHANGES WILL BE LOST UPON YOUR NEXT
#  'perl Makefile.PL'!!!
package PDL::Config;
use strict;
use warnings;
%PDL::Config = %{\n";
  print $fh Data::Dumper::Dumper(\%PDL::Config), "};\n1;\n";
}

# Scan ARGV for config file argument
@ARGV = map {
	if (/^F77CONF=(.*)$/) {
		$seen_f77conf=1;
		$pdl_f77conf=$1; ();
	} elsif (/^FORCE$/i) {
		$forcebuild=1;
		();
        } else {
		$_
	}
} @ARGV;

warn "WARNING: forcing build...\n" if $forcebuild;

checkbuggysetup(); # check for buggy Perl setups

# needs to be called before any of the make_XX routines
getpdl_config();
$seen_f77conf = check_f77conf($seen_f77conf);

# Check environment for SKIP_KNOWN_PROBLEMS
if (!defined($PDL::Config{SKIP_KNOWN_PROBLEMS}) and defined($ENV{SKIP_KNOWN_PROBLEMS})) {
   $PDL::Config{SKIP_KNOWN_PROBLEMS} = $ENV{SKIP_KNOWN_PROBLEMS};
   warn "Makefile.PL: setting SKIP_KNOWN_PROBLEMS Config from environment value: $ENV{SKIP_KNOWN_PROBLEMS}";
}
# Add check for POGL if USE_POGL is enabled
if (!defined($PDL::Config{USE_POGL}) or +$PDL::Config{USE_POGL}) {
   eval "use OpenGL $PDL::Config{POGL_VERSION} qw();";
   if ($@) {
      if (defined($PDL::Config{USE_POGL})) {
         warn "Makefile.PL: DEPENDENCY ERROR: USE_POGL requires at least OpenGL version $PDL::Config{POGL_VERSION}!\n";
         exit 0;
      } else {
         warn "Makefile.PL: OpenGL-$PDL::Config{POGL_VERSION} not found, setting \$PDL::Config{USE_POGL} => 0\n";
         $PDL::Config{USE_POGL} = 0;
      }
   } else {
      # Have OpenGL so set USE_POGL option if needed
      $PDL::Config{USE_POGL} ||= 1;
   }
} else {
   warn "Makefile.PL: Have \$PDL::Config{USE_POGL} => 0 so skipping TriD build with POGL\n";
}

# this will make PDL::Core::Dev's functions available
# in Makefile.PL files in all subdirectories
require "./Basic/Core/Dev.pm"; PDL::Core::Dev->import();

my @podpms = map { $_.".pod", '$(INST_LIBDIR)/PDL/' . $_ .".pod"}
  qw/Bugs/;

my @prereq = (
           'Convert::UU'         => 0,         # for PDL::IO::Dumper
           'Data::Dumper'        => 2.121,     # for PDL::IO::Dumper
	   'File::Map'           => 0.57,      # test new mmap implementation
	   'Filter::Util::Call'  => 0,         # for PDL::NiceSlice
	   'Filter::Simple'      => 0.88,      # for new PDL::NiceSlice
	   'Inline'              => 0.83,      # fixed ILSM-finding
	   'Inline::C'           => 0.62,
           'List::Util'          => '0.33',    # for PDL::IO::GD
	   'Pod::Parser'         => 0,         # version TBD for PDL::Doc
	   'Pod::Select'         => 0,         # version TBD for PDL::Doc
           'Scalar::Util'        => 0,
           'Storable'            => 1.03,      # for PDL::IO::Storable
           'Term::ReadKey'       => 2.34,      #for perldl shell
           'Text::Balanced'      => 1.89,      # for PDL::NiceSlice
           'autodie'             => 0,         # for PDL::Graphics::LUT, Centos 7 doesn't have
           'Math::Complex'       => 0,
	  );

# add OpenGL version dependency for CPAN to follow
push @prereq, ('OpenGL' => $PDL::Config{POGL_VERSION}) if $PDL::Config{USE_POGL};

# push @prereq, ('ExtUtils::F77' => 1.10) unless $seen_f77conf;

my @exe_files = qw(perldl pdldoc);
push @exe_files, "pdl$Config{exe_ext}" if $^O !~ /MSWin/;
my @cleanup = ("pdl$Config{exe_ext}", qw(
    pdldoc.db pdldoc.pod perldl.pod Changes.new
));

##############################
# Hack to include fPIC on x86_64 systems -
# use similar mods to affect CCFLAGS on other systems as needed...

my $ccflags =  $Config{ccflags};
if($Config{archname}=~m/x86_64/) {
    $ccflags .= " -fPIC";
}

# create GENERATED subdir with *.pm files during 'make dist' (to make metacpan.org happy)
my $preop = '$(PERL) -I$(INST_ARCHLIB) -I$(INST_LIB) -MPDL::Core::Dev -e pdlpp_mkgen $(DISTVNAME)';

my @optimize = (OPTIMIZE => $PDL::Config{OPTIMIZE}) if $PDL::Config{OPTIMIZE};

my %makefile_hash = (
  PREREQ_PM => { @prereq },
  LICENSE => 'perl',
  CONFIGURE_REQUIRES => {
    'Devel::CheckLib' => 1.01,
    'Carp'            => 1.20,    # EU::MM seems to need this not to crash
    'ExtUtils::MakeMaker' => '7.12', # working .g.c
    'File::Path'          => 0,
    'ExtUtils::ParseXS'   => 3.01, # avoids 2.21, known broken
    'ExtUtils::Depends' => '0.402',
  },
  TEST_REQUIRES => {
    'CPAN::Meta' => '2.120900',
    'IPC::Cmd' => 0.72,
    'Test::Exception' => 0,
    'Test::Warn' => 0,  # for t/pptest.t
    'Test::Deep' => 0,
  },
  BUILD_REQUIRES => {
    'ExtUtils::MakeMaker' => 0,
    'File::Path'          => 0,
  },
  MIN_PERL_VERSION => '5.010',
  NAME => 'PDL',
  VERSION_FROM => 'Basic/PDL.pm',
  EXE_FILES => \@exe_files,
  PM => { @podpms }, #so that the script docs are picked up
  META_MERGE => {
     "meta-spec" => { version => 2 },
     resources => {
        homepage => 'http://pdl.perl.org/',
        bugtracker  => {web=>'https://github.com/PDLPorters/pdl/issues'},
        repository  => {
           url => 'git://github.com/PDLPorters/pdl.git',
           type => 'git',
           web => 'https://github.com/PDLPorters/pdl',
        },
        x_IRC => 'irc://irc.perl.org/#pdl',
     },
     no_index => { file => ['Doc/scantree.pl'] },
     prereqs => {
        configure => {
           recommends => {
              'ExtUtils::F77' => '1.10',
              'PGPLOT' => 0,
           },
        },
        runtime => {
           suggests => {
              'Astro::FITS::Header' => 0,
           },
        },
     },
  },
  @optimize,
  CCFLAGS => $ccflags,
  dist => { COMPRESS => 'gzip', SUFFIX => 'gz', PREOP => $preop },
  clean => {
    FILES => join ' ', @cleanup, qw(
      tbyte.tif tmp0 tmp0.hdr tushort.tif
      MANIFEST.bak tmp1* tmpraw* t/tmpraw* t/tmp1*
      _Inline/ .inlinepdlpp/ .inline-ct/ .inlinewith/ *.xfig
    ),
  },
  realclean => {'FILES' => 'Basic/Core/Config.pm'},
  AUTHOR => 'PerlDL Developers <pdl-general@lists.sourceforge.net>',
  ABSTRACT => 'Perl Data Language',
  BINARY_LOCATION => 'PDL.tar.gz',
);

WriteMakefile(%makefile_hash);

# do *after* WriteMakefile since some options
# are set by the recursively called Makefile.PLs
make_PDL_Config_pm(); # write out config to PDL::Config

# Extra build target to build the doc database
sub MY::postamble {
  my ($self) = @_;
  package MY;
  my  $text =
'
doctest ::
	$(NOECHO) $(ECHO) "doctest: Building PDL documentation database in blib ..."
	$(NOECHO) $(PERLRUNINST) Doc$(DFSEP)scantree.pl
%HTML%	$(NOECHO) $(ECHO) "doctest: Building PDL documentation web pages in blib ..."
%HTML%	$(NOECHO) $(PERLRUNINST) Doc$(DFSEP)mkhtmldoc.pl

doc_site_install ::
	$(NOECHO) $(ECHO) "doc_site_install: Building PDL documentation database ..."
	$(NOECHO) $(PERLRUNINST) Doc$(DFSEP)scantree.pl "$(DESTDIR)$(INSTALLSITEARCH)"
%HTML%	$(NOECHO) $(ECHO) "doc_site_install: Building PDL documentation web pages ..."
%HTML%	$(NOECHO) $(PERL) Doc$(DFSEP)mkhtmldoc.pl "$(DESTDIR)$(INSTALLSITEARCH)$(DFSEP)PDL"

doc_vendor_install ::
	$(NOECHO) $(ECHO) "doc_vendor_install: Building PDL documentation database ..."
	$(NOECHO) $(PERLRUNINST) Doc$(DFSEP)scantree.pl "$(DESTDIR)$(INSTALLVENDORARCH)"
%HTML%	$(NOECHO) $(ECHO) "doc_vendor_install: Building PDL documentation web pages ..."
%HTML%	$(NOECHO) $(PERL) Doc$(DFSEP)mkhtmldoc.pl "$(DESTDIR)$(INSTALLVENDORARCH)$(DFSEP)PDL"

doc_perl_install ::
	$(NOECHO) $(ECHO) "doc_perl_install: Building PDL documentation database ..."
	$(NOECHO) $(PERLRUNINST) Doc$(DFSEP)scantree.pl "$(DESTDIR)$(INSTALLARCHLIB)"
%HTML%	$(NOECHO) $(ECHO) "doc_perl_install: Building PDL documentation web pages ..."
%HTML%	$(NOECHO) $(PERL) Doc$(DFSEP)mkhtmldoc.pl "$(DESTDIR)$(INSTALLARCHLIB)$(DFSEP)PDL"
';

  if(defined $PDL::Config{HTML_DOCS} && !$PDL::Config{HTML_DOCS}){
    $text=~ s/\%HTML\%[^\n]*\n//og; # Remove %HTML% lines
  } else {
    $text=~ s/\%HTML\%//og; # Remove just %HTML% markers
  }

# Here, `pdl.c` is used directly because some makes (e.g., on Solaris) do not
# support the `$<` variable in explicit rules
$text .= <<EOT if $^O !~ /MSWin/;
pdl$::Config{exe_ext} : pdl.c
	\$(CC) \$(CFLAGS) \$(LDFLAGS) \$(CCCDLFLAGS) pdl.c -o \$\@
EOT

$text .= << 'EOT' if $^O =~ /MSWin/;

DISTWIN32NAME=$(DISTVNAME)-win32

ppm : doctest ppd
	$(MV) blib$(DFSEP)lib$(DFSEP)PDL$(DFSEP)HtmlDocs$(DFSEP)PDL blib$(DFSEP)html$(DFSEP)lib$(DFSEP)PDL
	$(COMPRESS) -dc win32$(DFSEP)pbmwin32.tar.gz | $(TAR) xf -
	$(MKPATH) $(DISTWIN32NAME)
	$(CP) win32$(DFSEP)Readme $(DISTWIN32NAME)
	$(CP) win32$(DFSEP)install.ppm .
	$(PERL) -pe "s|<\x{2f}IMPLEMENTATION>|<INSTALL EXEC=\"perl\">install.ppm<\x{2f}INSTALL><\x{2f}IMPLEMENTATION>|" PDL.ppd > PDL.ppd.new
	$(RM) PDL.ppd
	$(MV) PDL.ppd.new PDL.ppd
	$(CP) PDL.ppd $(DISTWIN32NAME)
	$(TAR) cf $(DISTWIN32NAME)$(DFSEP)PDL.tar blib install.ppm
	cd $(DISTWIN32NAME)
	$(COMPRESS) PDL.tar
	$(ZIP) $(DISTWIN32NAME).zip *
	$(MV) $(DISTWIN32NAME).zip ..
	cd ..
	$(RM_RF) $(DISTWIN32NAME)
EOT

$text .= "\n" . ::coretarget($self);
my $coretest = join ' ', map File::Spec->catfile('t', $_.'.t'), qw(
  01-pptest autoload bad basic bool clump config constructor core croak lvalue
  math matrix matrixops nat_complex ops-bitwise ops pdl_from_string
  pp_croaking pp_line_numbers primitive pthread reduce
  slice subclass thread thread_def ufunc
);
$text .= <<EOF;

coretest : core
	prove -b $coretest
EOF

return $text

}

sub coretarget {
    my ($self) = @_;
    # remember the fundamental ones end up far to right as much deps on them
    # a "right" is either scalar (named target) or tuple of
    #     [ \@dir, \@targets, \@prereqs ]
    # @dir is dir parts for use by File::Spec
    # @targets is make targets within that dir
    # @prereqs are named targets - undef=[]
    # all a left's rights are made concurrently, no sequence - list ALL prereqs
    my @left2rights = (
        [
            basics => [
                [ [ qw(Basic) ], [ qw(pm_to_blib) ], ],
                [ [ qw(Basic Core) ], [ qw(pm_to_blib) ], ],
                [ [ qw(Basic Gen) ], [ qw(all) ], ],
            ]
        ],
        [
            core => [
                [ [ qw(Basic SourceFilter) ], [ qw(pm_to_blib) ], ],
                map { [
                    [ 'Basic', $_ ],
                    [ 'pm_to_blib', 'dynamic' ],
                    [ 'basics' ],
                # the modules in PDL::LiteF, used in t/core.t
                ] } qw(Core Ops Primitive Ufunc Slices Bad Math MatrixOps),
            ]
        ],
    );
    join "\n", map flatten_parallel_target($self, $_), @left2rights;
}

sub format_chunk {
    my ($self, $left, $deps, $dir, $targets) = @_;
    my @m = join ' ', $left, ':', @{$deps||[]};
    my $fsdir = File::Spec->catdir(@$dir);
    push @m, "\t" . $self->oneliner(
        "die \$! unless chdir q($fsdir); exec q(\$(MAKE) @$targets)"
    );
    join '', map "$_\n", @m;
}

# output: list of make chunks with target, deps, recipes
sub flatten_parallel_target {
    my ($self, $left2rights) = @_;
    my ($left, $rights) = @$left2rights;
    my (@deps, @recipes, @otherchunks);
    for my $right (@$rights) {
        if (ref $right) {
            #     [ \@dir, \@targets, \@prereqs ]
            # @dir is dir parts for use by File::Spec
            # @targets is make targets within that dir
            # @prereqs are named targets - undef=[]
            my ($dir, $targets, $prereqs) = @$right;
            my $target_name = parallel_target_mangle($self, $dir, $targets);
            push @deps, $target_name;
            push @otherchunks, format_chunk(
                $self, $target_name, $prereqs, $dir, $targets
            );
        } else {
            push @deps, $right;
        }
    }
    (
        join(' : ', $left, join ' ', @deps) . "\n",
        @otherchunks,
    );
}

sub parallel_target_mangle {
    my ($self, $dir, $targets) = @_;
    my $target = join '_', @$dir, @$targets;
    $target =~ s#[/\\]#_#g; # avoid ambiguity with filenames
    $target;
}

# remove pdl.c from making EUMM think this dir has XS in it
sub MY::init_dirscan {
    package MY;
    my ($self) = @_;
    $self->SUPER::init_dirscan;
    $self->{C} = [];
    $self->{O_FILES} = [];
}
