# ABSTRACT: A Net::Object::Peer Subscription
package Net::Object::Peer::Subscription;

use 5.10.0;

use Scalar::Util qw[ weaken refaddr ];
use Types::Standard qw[ ConsumerOf Str CodeRef InstanceOf ];

use Moo;
use strictures 2;
use namespace::clean;

our $VERSION = '0.04'; # TRIAL


#pod =attr peer
#pod
#pod A weak reference to the peer object subscribed to.
#pod
#pod =cut

has peer => (
    is       => 'ro',
    weak_ref => 1,
    required => 1,
    isa      => ConsumerOf ['Net::Object::Peer'],
);

#pod =attr addr
#pod
#pod The address returned by L<Scalar::Util::refaddr|Scalar::Util/refaddr> for
#pod the L</peer> attribute.
#pod
#pod =cut

has addr => (
    is       => 'rwp',
    isa      => InstanceOf['Net::Object::Peer::RefAddr'],
    init_arg => undef,
);

#pod =attr name
#pod
#pod The name of the event listened for.
#pod
#pod =cut

has name => (
    is       => 'ro',
    isa      => Str,
    required => 1,
);

has _unsubscribe => (
    is       => 'ro',
    init_arg => 'unsubscribe',
    isa      => CodeRef,
    required => 1,
);


#pod =begin pod_coverage
#pod
#pod =head3 BUILD
#pod
#pod =end pod_coverage
#pod
#pod =cut

sub BUILD {

    my $self = shift;

    $self->_set_addr( Net::Object::Peer::RefAddr->new( $self->peer ) );
}

#pod =method unsubscribe
#pod
#pod =cut

sub unsubscribe { $_[0]->_unsubscribe->() }

#pod =method as_hashref
#pod
#pod   $hashref = $sub->as_hashref;
#pod
#pod Return non-code attributes as a hash
#pod
#pod =cut

sub as_hashref {

    my $self = shift;

    my %hash = map { $_ => $self->$_ } qw[ peer name addr ];

    weaken $hash{peer};

    return \%hash;
}

1;

#
# This file is part of Net-Object-Peer
#
# This software is Copyright (c) 2016 by Smithsonian Astrophysical Observatory.
#
# This is free software, licensed under:
#
#   The GNU General Public License, Version 3, June 2007
#

=pod

=head1 NAME

Net::Object::Peer::Subscription - A Net::Object::Peer Subscription

=head1 VERSION

version 0.04

=head1 DESCRIPTION

A B<Net::Object::Peer::Subscription> object manages a node's
subscription to an emitter.

It keeps a weak reference to the emitter so that the emitter will
disappear after it's defining scope has been destroyed.

=head1 ATTRIBUTES

=head2 peer

A weak reference to the peer object subscribed to.

=head2 addr

The address returned by L<Scalar::Util::refaddr|Scalar::Util/refaddr> for
the L</peer> attribute.

=head2 name

The name of the event listened for.

=head1 METHODS

=head2 unsubscribe

=head2 as_hashref

  $hashref = $sub->as_hashref;

Return non-code attributes as a hash

=begin pod_coverage

=head3 BUILD

=end pod_coverage

=head1 AUTHOR

Diab Jerius <djerius@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2016 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut

__END__

#pod =head1 DESCRIPTION
#pod
#pod A B<Net::Object::Peer::Subscription> object manages a node's
#pod subscription to an emitter.
#pod
#pod It keeps a weak reference to the emitter so that the emitter will
#pod disappear after it's defining scope has been destroyed.
