package OpenInteract2::Action::SiteSearch;

# $Id: SiteSearch.pm,v 1.6 2003/06/25 14:10:50 lachoy Exp $

use strict;
use base qw( OpenInteract2::Action );
use Log::Log4perl            qw( get_logger );
use OpenInteract2::Constants qw( :log );
use OpenInteract2::Context   qw( CTX );
use OpenInteract2::ResultsManage;

BEGIN {
    eval {
        require OpenInteract2::FullText;
    };
}

$OpenInteract2::Action::SiteSearch::VERSION = sprintf("%d.%02d", q$Revision: 1.6 $ =~ /(\d+)\.(\d+)/);

sub search {
    my ( $self ) = @_;
    my $log = get_logger( LOG_APP );

    my $request = CTX->request;

    my %params = ( search_id => $request->param( 'search_id' ),
                   keywords  => $request->param( 'keywords' ) );

    unless ( $params{keywords} or $params{search_id} ) {
        my $nk_desc = 'You did not give me any keywords to search, ' .
                      'so I did not run a search. Please try again.';
        my %nk_params = ( title       => 'No Keywords',
                          description =>  $nk_desc );
        return $self->generate_content(
                    \%nk_params, { name => 'full_text::tiny_search' } );
    }

    $params{current_page} = $request->param( 'page' ) || 1;
    my $hits_per_page = $request->param( 'per_page' )
                        || $self->param( 'default_per_page' );
    my ( $min, $max ) = OpenInteract2::ResultsManage->find_page_boundaries(
                              $params{current_page}, $hits_per_page );

    # Run the search for the first time and save the results

    unless ( $params{search_id} ) {
        my @terms = split /\s+/, $params{keywords};
        $params{search_type} = $request->param( 'search_type' )
                               || $self->param( 'default_search_type' );
        my @include_classes = $request->param( 'include_class' );
        $log->is_info &&
            $log->info( "Searching for $params{search_type} of ",
                        "[$params{keywords}]" );
        my $search_results = OpenInteract2::FullText->search_fulltext_index({
                                      search_terms => \@terms,
                                      search_type  => $params{search_type},
                                      return       => 'raw' }) || [];

        # Persist the raw results and get the ID so we can use them

        if ( scalar @{ $search_results } > 0 ) {
            my $results = OpenInteract2::ResultsManage->new();
            $params{search_id} = eval {
                $results->save( [ map { $_->[0] } @{ $search_results } ],
                                { extra      => [ map { $_->[1] } @{ $search_results } ],
                                  extra_name => [ 'fulltext_score' ] } )
            };
        }
    }

    # Retrieve the persisted results and pass off to the OpenInteract::FullTextIterator

    if ( $params{search_id} ) {
        my $results = OpenInteract2::ResultsManage->new(
                              { search_id => $params{search_id},
                                min       => $min,
                                max       => $max });
        $results->retrieve;
        $params{search_iterator} =
            OpenInteract2::FullTextIterator->new({
                              saved_results => $results->{result_list} });
        $params{total_hits}  = $results->{num_records};
        $params{total_pages} =
            $results->find_total_page_count( $params{total_hits} );
    }
    else {
        $params{total_hits} = 0;
        $params{total_pages} = 0;
    }
    return $self->generate_content(
                    \%params, { name => 'full_text::search_results' } );
}

1;

__END__

=head1 NAME

OpenInteract2::Action::SiteSearch - Perform searches using the FullText module.

=head1 SYNOPSIS

 http://www.myoisite.com/SiteSearch/?keywords=this+and+that

=head1 DESCRIPTION

Implement a full-text search of all objects on a website -- or in a
group, or whatever. Most of the real work is done in
L<OpenInteract2::FullText|OpenInteract2::FullText> -- you might want
to check it out.

=head1 METHODS

B<search>

Runs the search!

Input parameters (GET or POST, does not matter):

=over 4

=item *

B<keywords> ($)

Space-separated list of words to search for in the index.

=item *

B<search_type> ($) (optional -- defaults to 'all', set in action
config)

Type of search to run. Choices are 'any' (OR all the keywords) or
'all' (AND all the keywords).

=item *

B<include_class> (@) (optional -- defaults to all classes)

NOT IMPLEMENTED RIGHT NOW

Restrict your search results to the following classes. This is
typically implemented by a webform with checkboxes or a
multiple-select list where the value the user sees is the object type
(e.g., 'News') and the value passed to the backend is the class for
that object (e.g., 'MySite::News').

=back

=head1 TO DO

Nothing known.

=head1 BUGS

None known.

=head1 COPYRIGHT

Copyright (c) 2001-2003 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters E<lt>chris@cwinters.comE<gt>
