package OpenInteract::SiteTemplate;

# $Id: SiteTemplate.pm,v 1.9 2002/08/25 17:04:36 lachoy Exp $

use strict;
use base qw( Exporter Class::Accessor );
use File::Basename   qw();
use File::Path       qw();
use File::Spec       qw();

$OpenInteract::SiteTemplate::VERSION = substr(q$Revision: 1.9 $, 10);

my $NAME_SEPARATOR = '::';
sub NAME_SEPARATOR { return $NAME_SEPARATOR }

@OpenInteract::SiteTemplate::EXPORT_OK = qw( NAME_SEPARATOR );

my @FIELDS = qw( package name filename directory modified_on is_package is_global );
OpenInteract::SiteTemplate->mk_accessors( @FIELDS );

my @TEMPLATE_EXTENSIONS = ( '', '.tmpl', '.tt' );

########################################
# CLASS METHODS
########################################

# Parse a combined 'package::name' label into the package and
# name. If label is simply 'name' we return ( undef, $label )

sub parse_name {
    my ( $class, $full_name ) = @_;
    if ( $full_name =~ /$NAME_SEPARATOR/ ) {
        return split /$NAME_SEPARATOR/, $full_name, 2;
    }
    return ( undef, $full_name );
}


# Create a combined 'package::name' label. If no package exists,
# return 'name' by itself

sub create_name {
    my ( $item, $package, $name ) = @_;
    if ( ref $item eq __PACKAGE__ ) {
        $package = $item->{package};
        $name    = $item->{name};
    }
    if ( $package and $name ) {
        return join $NAME_SEPARATOR, $package, $name;
    }
    return $name;
}


sub name_from_file {
    my ( $class, $filename ) = @_;
    my $base = File::Basename::basename( $filename );
    $base =~ s/\.\w+$//;
    return $base;
}

########################################
# CONSTRUCTOR

sub new {
    my ( $class, $params ) = @_;
    my $self = bless( {}, $class );
    for ( @FIELDS ) {
        $self->$_( $params->{ $_ } ) if ( $params->{ $_ } );
    }
    if ( $params->{full_name} ) {
        my ( $package, $name ) = $class->parse_name( $params->{full_name} );
        $self->package( $package );
        $self->name( $name );
    }
    if ( $params->{contents} ) {
        $self->set_contents( $params->{contents} )
    }
    return $self;
}


########################################
# FIND OBJECTS

# Available parameters:

sub fetch {
    my ( $class, $given_name, $params ) = @_;
    my ( $package, $name ) = $class->parse_name( $given_name );
    $params ||= {};

    if ( my $filename = $class->_find_in_global( $package, $name ) ) {
        $params->{is_global}  = 1;
        $params->{is_package} = 0;
        return $class->_create_from_file( $filename, $package, $params );
    }
    elsif ( my $pkg_filename = $class->_find_in_package( $package, $name ) ) {
        $params->{is_global}  = 0;
        $params->{is_package} = 1;
        return $class->_create_from_file( $pkg_filename, $package, $params );
    }
    return undef;
}


sub fetch_by_package {
    my ( $class, $package, $params ) = @_;
    my $R = OpenInteract::Request->instance;
    my @template_dir = $class->_lookup_site_template_dir( $package );
    if ( $package ) {
        push @template_dir, $class->_lookup_package_template_dir( $package );
    }
    my %by_name = ();
    foreach my $dir ( @template_dir ) {
        foreach my $file ( $class->_get_all_templates_in_directory( $dir ) ) {
            $by_name{ $class->name_from_file( $file ) }++;
        }
    }
    my @templates = ();
    foreach my $name ( sort keys %by_name ) {
        my $fq_name = $class->create_name( $package, $name );
        push @templates, $class->fetch( $fq_name );
    }
    return \@templates;
}


# Finds the file corresponding to the package/name given in the global
# template directory. Return undef if not found.

sub _find_in_global {
    my ( $class, $package, $name ) = @_;
    my $R = OpenInteract::Request->instance;
    my $template_dir = $R->CONFIG->get_dir( 'template' );
    my $dir = ( $package )
                ? File::Spec->catfile( $template_dir, $package ) : $template_dir;
    return $class->_find_template_in_directory( $dir, $name );
}


# Finds the file corresponding to the package/name given in the
# package template directory. Return undef if not found.

sub _find_in_package {
    my ( $class, $package, $name ) = @_;
    my $R = OpenInteract::Request->instance;
    my $website_dir = $R->CONFIG->get_dir( 'base' );
    my $repository  = $R->repository->fetch(
                                       undef, { directory => $website_dir } );
    my $package = $repository->fetch_package_by_name({ name => $package });
    my $template_dir = File::Spec->catfile( $website_dir, $package->{package_dir}, 'template' );
    return $class->_find_template_in_directory( $template_dir, $name );
}

# Finds a template file in a particular directory -- cycles through
# the different template extensions to see if one exists.

sub _find_template_in_directory {
    my ( $class, $dir, $name ) = @_;
    for ( @TEMPLATE_EXTENSIONS ) {
        my $try_file = File::Spec->catfile( $dir, "$name$_" );
        return $try_file if ( -f $try_file );
    }
    return undef;
}


sub _create_from_file {
    my ( $class, $full_filename, $package, $params ) = @_;
    my ( $filename, $directory ) = File::Basename::fileparse( $full_filename );
    return $class->new({ package     => $package,
                         name        => $class->name_from_file( $filename ),
                         directory   => $directory,
                         filename    => $filename,
                         modified_on => (stat( $full_filename ))[9],
                         is_package  => $params->{is_package},
                         is_global   => $params->{is_global} });
}


sub _get_all_templates_in_directory {
    my ( $class, $dir ) = @_;

    # No penalties for checking a directory that might not yet exist
    return () unless ( -d $dir );

    opendir( TMPL, $dir )
                    || die "Error scanning [$dir] for templates: $!\n";
    my @files = grep { -f "$dir/$_" } readdir( TMPL );
    closedir( TMPL );

    my @good_files = ();
    foreach my $file ( sort @files ) {
        next if ( $file =~ /(~|\.bak|\.meta)$/ or $file =~ /^(\.|tmp)/ );
        push @good_files, File::Spec->catfile( $dir, $file );
    }
    return @good_files
}


########################################
# SAVE OBJECT

sub save {
    my ( $self, $params ) = @_;
    unless ( $self->name ) {
        die "Cannot save template - property 'name' must be defined\n";
    }
    my $R = OpenInteract::Request->instance;
    my $template_dir = $R->CONFIG->get_dir( 'template' );
    unless ( $self->filename ) {
        $self->filename( $self->name );
    }

    # Always ensure that we're saving to the global directory

    if ( $self->package ) {
        $self->directory( File::Spec->catfile( $template_dir, $self->package ) );
        $self->is_global(1);
        $self->is_package(0);
    }
    else {
        $self->directory( $template_dir );
        $self->is_global(1);
        $self->is_package(0);
    }
    $self->_save_contents();
    $self->modified_on( (stat( $self->full_filename ))[9] );
    return $self;
}


########################################
# REMOVE OBJECT

sub remove {
    my ( $self, $params ) = @_;
    my $full_filename = $self->full_filename;
    unlink( $full_filename )
                    || die "Cannot remove [$full_filename]: $!";
}


########################################
# ACCESSORS/PROPERTY METHODS

sub full_filename {
    my ( $self ) = @_;
    return File::Spec->catfile( $self->directory, $self->filename );
}

sub contents {
    my ( $self ) = @_;
    $self->_load_contents() unless ( $self->{_contents} );
    return $self->{_contents} ;
}


sub set_contents {
    my ( $self, $contents ) = @_;
    return $self->{_contents} = $contents;
}


sub _load_contents {
    my ( $self ) = @_;
    my $full_filename = $self->full_filename;
    open( IN, "< $full_filename" )
                    || die "Cannot open template contents [$full_filename]: $!\n";
    local $/ = undef;
    $self->{_contents} = <IN>;
    close( IN );
}


# Don't fall into the trap of writing to the same file and leaving it
# inconsistent, even though the open/write/close is only three lines.
# (Thanks to merlyn for the friendly reminder!)
#
# If you call this from anywhere else, be sure that 'directory' and
# 'name' properties are defined

sub _save_contents {
    my ( $self ) = @_;
    my $full_filename = $self->full_filename;

    # First make sure the write path exists

    File::Path::mkpath( $self->directory );

    # Then open the relevant file

    my $open_filename = ( -f $full_filename )
                          ? $full_filename . '_tmp' : $full_filename;
    open( OUT, "> $open_filename" )
                    || die "Cannot open template for writing [$open_filename]: $!\n";
    print OUT $self->{_contents};
    close( OUT );
    if ( $full_filename ne $open_filename ) {
        rename( $open_filename, $full_filename )
                    || die "Cannot rename [$open_filename] -> [$full_filename]: $!\n";
    }
}


########################################
# DIRECTORY

sub _lookup_site_template_dir {
    my ( $class, $package ) = @_;
    my $template_dir = OpenInteract::Request->instance->CONFIG->get_dir( 'template' );
    if ( $package ) {
        return File::Spec->catfile( $template_dir, $package );
    }
    return $template_dir;
}


sub _lookup_package_template_dir {
    my ( $class, $package ) = @_;
    return undef unless ( $package );
    my $R = OpenInteract::Request->instance;
    my $website_dir = $R->CONFIG->get_dir( 'base' );
    my $pkg = $R->repository->fetch( undef, { directory => $website_dir })
                            ->fetch_package_by_name({ name => $package });
    return File::Spec->catfile( $website_dir, $pkg->{package_dir}, 'template' );
}

1;

__END__

=head1 NAME

OpenInteract::SiteTemplate - Object to represent templates

=head1 SYNOPSIS

 my $R = OpenInteract::Request->instance;

 # Retreive a single template based on name

 my $tmpl = eval { $R->site_template->fetch( 'base_box::user_info_box' ) };
 die "Cannot retrieve box: $@" if ( $@ );
 print "Template contents: ", $tmpl->contents, "\n";

 # Retrieve multiple templates from a package

 my $tmpl_list = eval { $R->site_template->fetch_by_package( 'base_box' ) };
 die "Cannot retrieve templates from base_box: $@" if ( $@ );
 foreach my $tmpl ( @{ $tmpl_list } ) {
    print "Template contents: ", $tmpl->contents, "\n";
 }

 # Parse the common 'package::name' format

 my $full_name = 'base_box::main_box_shell';
 my ( $pkg, $name ) = $R->site_template->parse_name( $full_name );

 # Template Toolkit usage

 # Include a template from a separate package. (See TT docs for the
 # difference between 'PROCESS' and 'INCLUDE')

 [% PROCESS mypkg::user_info( user = this_user ) %]
 [% INCLUDE mypkg::user_info( user = this_user ) %]


 ***** WARNING *****     ***** WARNING *****     ***** WARNING *****

As of version 2.00+ of this package (included with version 1.50 of
OpenInteract) templates will no longer be fetched from the
database. They are only stored in the filesystem. A migration script
is included with this package in C<script/migrate_to_filesystem.pl>.

 ***** WARNING *****     ***** WARNING *****     ***** WARNING *****


=head1 DESCRIPTION

SiteTemplate objects are used throughout OpenInteract -- in fact, on
every single request multiple template objects will be used.

Each object represents a template which can be interpreted by the
template processing engine (normally the L<Template|Template Toolkit>)
and replaced with information from the OpenInteract environment along
with data that you decide to display on a page.

However, most of the time you will not deal directly with template
objects. The core OpenInteract modules
L<OpenInteract::Template::Process|OpenInteract::Template::Process> and
the custom provider for the Template Toolkit
L<OpenInteract::Template::Provider|OpenInteract::Template::Provider>
will retrieve templates for you based on the name and package
specified.

=head1 METHODS

This module exports the constant C<NAME_SEPARATOR>. This will probably
never change from '::', but I have developed an allergy to hardcoding
such things.

=head2 Class Methods

B<parse_name( $full_template_name )>

Parse a full template name (in 'package::name' format) into the
package and name comprising it.

Returns a two-item list: C<( $package, $name )>. If there is no
package in C<$full_template_name>, the first item in the list is
C<undef>.

B<create_name( $package, $name )>

Create a fully-qualified template name.

Returns a string with the full template name using C<$package> and
C<$name>. If C<$package> is not defined, the fully-qualified name is
just C<$name>.

B<name_from_file( $filename )>

Return the template named by C<$filename>. This just strips off the
directory and the extension.

Returns: template name.

=head2 Constructor and Methods Returning Objects

B<new( \%params )>

Create a new object. The C<\%params> can be any of the properties,
plus C<full_name> which will be parsed into the proper C<package> and
C<name> object properties according to C<parse_name()>

B<fetch( $fully_qualified_template, \%params )>

B<fetch_by_package( $package_name, \%params )>

=head2 Object Methods

B<save( \%params )>

B<remove( \%params )>

B<contents()>

Loads contents of this template into the object. This is a
lazy-loading method -- we only read in the contents on demand.

Returns: contents of template

B<set_contents( $contents )>

Sets the contents of the template to C<$contents>. Note that you still
need to call C<save()> to serialize the contents to the filesystem.

B<full_filename()>

Returns a full path to this template.

=head1 PROPERTIES

All properties except 'contents' can be accessed and set by a method
of their property name, e.g.:

 # Get directory
 my $dir = $template->directory;

 # Set directory
 $template->directory( $dir );

B<contents>

This property is accessed through the method C<contents()> and
modified through the method C<set_contents()>; see L<Class Methods>.

B<package>

Package of this template.

B<name>

Name of this template. Usually this is the same as the filename, but
if your template has a filename 'foo.tmpl' the name will still be
'foo'.

B<directory>

Directory from where this template was loaded. After a C<save()> this
may change.

B<filename>

Filename from where this template was loaded. After a C<save()> this
may change.

B<modified_on>

Epoch time of last modification. (We get this from L<stat>.)

B<is_global>

Boolean defining whether this template came from the global template
directory, either from a package or not.

B<is_package>

Boolean defining whether this template came from a package template
directory.

=head1 TO DO

=head1 BUGS

None known.

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>
