package OpenInteract::Page;

# $Id: Page.pm,v 1.8 2001/11/29 04:14:06 lachoy Exp $

use strict;
use Data::Dumper   qw( Dumper );
use File::Basename qw();

$OpenInteract::Page::VERSION = sprintf("%d.%02d", q$Revision: 1.8 $ =~ /(\d+)\.(\d+)/);

my %STORAGE_CLASS = (
     database => 'OpenInteract::Page::Database',
     file     => 'OpenInteract::Page::File',
     http     => 'OpenInteract::Page::Http',
);


sub _class_initialize {
    my ( $class ) = @_;
    eval "require $_" for ( values %STORAGE_CLASS );
}

########################################
# CLASS METHODS
########################################

# Retrieve unique directory names and counts of member files from the
# system.
#
# Returns: arrayref of arrayrefs, first member is the directory name
# and the second is the number of files in the directory.

sub list_directories {
    my ( $class ) = @_;
    my $directory_list = eval { $class->db_select({
                                   from            => $class->CONFIG->{table_name},
                                   select          => [ 'directory, count(*)' ],
                                   select_modifier => 'DISTINCT',
                                   order           => 'directory',
                                   group           => 'directory' }) };
    if ( $@ ) {
        my $R = OpenInteract::Request->instance;
        $R->scrib( 0, "Error trying to retrieve distinct directory names: $@",
                      Dumper( SPOPS::Error->get ) );
        $directory_list = [];
    }
    return $directory_list;
}


my ( $MAGIC );
sub init_magic {
    my ( $class ) = @_;
    return if ( $MAGIC );
    require File::MMagic;
    $MAGIC = File::MMagic->new;
}


sub mime_type_file {
    my ( $class, $filename ) = @_;
    return undef unless ( -f $filename );
    $class->init_magic;
    return $MAGIC->checktype_filename( $filename );
}


sub mime_type_content {
    my ( $class, $content ) = @_;
    return undef unless ( $content );
    $class->init_magic;
    return $MAGIC->checktype_contents( $content );
}


########################################
# OBJECT METHODS
########################################

# Just replace the generated 'url' method with one that just uses the
# location. This won't work if you do not have the Page handler
# answering all 'unknown' requests.

sub object_description {
    my ( $self ) = @_;
    my $info = $self->SUPER::object_description;
    $info->{url} = $self->{location};
    return $info;
}


# Fetch the content from either the filesystem or database, depending.

sub fetch_content {
    my ( $self ) = @_;
    my $storage_class = $STORAGE_CLASS{ $self->{storage} };
    unless ( $storage_class ) {
        return "Cannot retrieve content -- no storage type specified";
    }
    return $self->{content} = $storage_class->load( $self );
}


########################################
# RULES
########################################

# Here we add a ruleset so we can set the value for 'directory'
# whenever we save the object

sub ruleset_factory {
    my ( $class, $rs_table ) = @_;
    push @{ $rs_table->{pre_save_action} },    \&find_directory;
    push @{ $rs_table->{post_save_action} },   \&save_content;
    push @{ $rs_table->{post_remove_action} }, \&remove_content;
    return __PACKAGE__;
}


sub find_directory {
    my ( $self, $p ) = @_;
    $self->{directory} = File::Basename::dirname( $self->{location} );
    return 1;
}


sub save_content {
    my ( $self, $p ) = @_;
    return 1 unless ( $self->{content} );
    my $storage_class = $STORAGE_CLASS{ $self->{storage} };
    return $self->{content} = $storage_class->save( $self, $self->{content} );
}


sub remove_content {
    my ( $self, $p ) = @_;
    my $storage_class = $STORAGE_CLASS{ $self->{storage} };
    return $self->{content} = $storage_class->remove( $self );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Page - Additional methods for the SPOPS Page object

=head1 SYNOPSIS

 # Create a new basic page object

 my $page = OpenInteract::Page->new();
 $page->{location} = '/mysite/home';
 $page->{is_file}  = 'no';
 $page->{content}  = "<h2>Headline</h2><p>This is the text for my page</p>";
 $page->{author}   = "Scooby Doo";
 $page->save;

 print "Directory for location $page->{location} is $page->{directory}\n";

 # Get the list of directories

 my $directory_info = OpenInteract::Page->list_directories();
 foreach my $info ( @{ $directory_info } ) {
     print "Directory $info->[0] has $info->[1] entries\n";
 }

=head1 DESCRIPTION

This class adds methods to the SPOPS class that represents static
pages. Depending on a property in the object, it will save the content
to the filesystem or in the database.

=head1 METHODS

B<find_directories()>

Class method that finds all unique directories currently used in the
system along with the number of files in each.

Returns an arrayref of arrayrefs. The first element in each arrayref
in the directory name, the second is the number of files in the
directory.

B<fetch_content()>

Retrieve content for this page. Note that we do not make this a
B<post_fetch_action> since there are likely times when you want to
just deal with the page metadata, not the content.

B<object_description()>

(Overrides method from SPOPS.)

Modify the C<url> value of the returned hashref of object information
to simply be the location of the basic page.

=head1 RULES

B<pre_save_action>

Set the C<directory> property from the C<location> property.

B<post_save_action>

If there is content, save it to either the db or filesystem.

B<post_remove_action>

Remove content from either the database or filesystem.

=head1 BUGS

None known.

=head1 TO DO

Nothing known.

=head1 SEE ALSO

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
