package OpenInteract::Handler::SiteSearch;

# $Id: SiteSearch.pm,v 1.5 2001/06/09 19:07:29 lachoy Exp $

use strict;
use Data::Dumper  qw( Dumper );
use OpenInteract::FullText;

@OpenInteract::Handler::SiteSearch::ISA               = qw( OpenInteract::Handler::GenericDispatcher );
$OpenInteract::Handler::SiteSearch::VERSION           = sprintf("%d.%02d", q$Revision: 1.5 $ =~ /(\d+)\.(\d+)/);
$OpenInteract::Handler::SiteSearch::author            = 'chris@cwinters.com';
$OpenInteract::Handler::SiteSearch::default_method    = 'search';
@OpenInteract::Handler::SiteSearch::forbidden_methods = ();

# Pretty straightforward...

sub search {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $keywords = $R->apache->param( 'keywords' );
  unless ( $keywords ) {
    return '<h2>Cannot Search</h2>',
           '<p>Search called without keywords and cannot run. ' .
           'Please <a href="/search/home">try again</a>.';
  }

  my @terms = split /\s+/, $keywords;
  my $search_type = $R->apache->param( 'search_type' ) || 'all';
  my @include_classes = $R->apache->param( 'include_class' );
  $R->DEBUG && $R->scrib( 1, "Searching for (($keywords)) with ($search_type)" );
  my $object_list = OpenInteract::FullText->search_ft_index({ 
                                      search_terms    => \@terms,
                                      search_type     => $search_type,
                                      include_classes => \@include_classes });
  $R->DEBUG && $R->scrib( 2, "List of found objects:", Dumper( $object_list ) );
  $R->{page}->{title} = 'Search Results';
  return $R->template->handler( {}, 
                                { keywords    => $keywords,
                                  object_list => $object_list }, 
                                { db      => 'search_results',
                                  package => 'full_text' } );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::SiteSearch - Perform searches using the FullText module.

=head1 SYNOPSIS

 http://www.myoisite.com/SiteSearch/?keywords=this+and+that

=head1 DESCRIPTION

Implement a full-text search of all objects on a website -- or in a
group, or whatever. Most of the real work is done in
L<OpenInteract::FullText> -- you might want to check it out.

=head1 METHODS

B<search>

Runs the search!

Input parameters (GET or POST, does not matter):

=over 4

=item *

B<keywords> ($)

Space-separated list of words to search for in the index.

=item *

B<search_type> ($) (optional -- defaults to 'all')

Type of search to run. Choices are 'any' (OR all the keywords) or
'all' (AND all the keywords).

=item *

B<include_class> (@) (optional -- defaults to all classes)

Restrict your search results to the following classes. This is
typically implemented by a webform with checkboxes or a
multiple-select list where the value the user sees is the object type
(e.g., 'News') and the value passed to the backend is the class for
that object (e.g., 'MySite::News').

=back

=head1 TO DO

B<Limit>

We should be able to limit search results, save them to a separate
file and page through them n results at a time.

=head1 BUGS

None known.

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
