#!/usr/bin/perl

# $Id: search_index.pl,v 1.3 2000/11/18 23:16:35 cwinters Exp $

# search_index.pl - Search the index for objects matching one or more words.

use strict;
use OpenInteract::Startup;
use OpenInteract::DBI;
use Data::Dumper  qw( Dumper );
use Getopt::Long  qw( GetOptions );

{
 my $usage = "Usage: $0 --application_dir=/path/to/my_app term [ term term ... ]";
 my ( $OPT_application_dir );
 GetOptions( 'application_dir=s' => \$OPT_application_dir );
 unless ( -d $OPT_application_dir ) {
   die "$usage\n Parameter 'application_dir' must refer to an OpenInteract application directory!\n";
 }
 unshift @INC, $OPT_application_dir;
 my $base_conf = join( '/', $OPT_application_dir, 'conf', 'base.conf' );
 unless ( -f $base_conf ) {
   die "$usage\n Cannot find base configuration file in $OPT_application_dir!\n";
 }

 my @terms = @ARGV;
 unless ( scalar @terms ) {
   die "$usage\n";
 }

 my ( $init, $C ) = OpenInteract::Startup->main_initialize( { base_config_file => $base_conf,
                                                              alias_init => 1, spops_init => 1 } );
 my $REQUEST_CLASS = $C->{request_class};
 my $R = $REQUEST_CLASS->instance;
 $R->{stash_class} = $C->{stash_class};
 my $db = OpenInteract::DBI->connect( $C->{db_info} );
 $R->stash( 'db', $db );

 my $object_list = OpenInteract::FullText->search_ft_index( { search_terms => \@terms,
                                                              skip_security => 1 } );
 foreach my $obj ( @{ $object_list } ) {
   printf( "Score %03d for object: %-20s (%-s)\n", $obj->{tmp_ft_score}, ref $obj, $obj->id );
 }
 $db->disconnect; 
}

__END__

=pod

=head1 NAME

search_index.pl - Simple script to search the full-text index for objects

=head1 SYNOPSIS

 search_index.pl /path/to/my_app/conf/base.conf term1 term2

=head1 DESCRIPTION

Searches the full-text index in your application for objects that have
one or more terms in them. This is partly a demonstration of how to
search the full-text index and how to use the search results, and
partly for use as a command-line check to the index which is normally
only searchable via the web.

=head1 BUGS 

None yet!

=head1 TO DO

=head1 SEE ALSO

L<OpenInteract::FullText>

=head1 COPYRIGHT

Copyright (c) 2000 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

 Chris Winters (chris@cwinters.com)

=cut
