package Finance::Asset;
use 5.010;
use strict;
use warnings;

our $VERSION = '1.11';

=head1 NAME

Finance::Asset - Object representation of a financial asset

=head1 VERSION

1.06

=head1 SYNOPSYS

    use Finance::Asset;

    my $asset = Finance::Asset->new('frxAUDUSD');
    print $asset->pip_size;

=head1 DESCRIPTION

Object representation of a financial asset

=cut

use Moose;
use YAML::CacheLoader qw(LoadFile);
use File::ShareDir ();
use Carp;

=head2 new($symbol)

Returns a Finance::Asset object for given I<$symbol>. new() can also be called
using Finance::Asset->new({symbol => $symbol})

=cut

sub new {
    my ($self, $args) = @_;

    $args = {symbol => $args} if (not ref $args);
    my $symbol = $args->{symbol};

    croak 'No symbol provided to constructor.' if (not $symbol);

    my $obj = $self->_new($args);
    return $obj;
}

# We use BUILDARGS since the new() constructor is not called as a normal Moose
# class would call it with the params

sub BUILDARGS {
    my $class = shift;
    my $params_ref = shift;

    # Load YAML file
    my $file = File::ShareDir::dist_file('Finance-Asset', 'underlyings.yml');
    my $assets_ref = LoadFile($file);
    my $params = $assets_ref->{$params_ref->{symbol}};
    $params->{symbol} = $params_ref->{symbol};
    @$params_ref{keys %$params} = @$params{keys %$params};

    if ($params_ref->{exchange_name}) {
        $params_ref->{_exchange_name} = $params_ref->{exchange_name};
        delete $params_ref->{exchange_name};
    }

    return $params_ref;
}

has symbol => (
    is  => 'ro',
    isa => 'Str',
);

=head2 pip_size

The pip value for a forex asset

=cut

has pip_size => (
    is => 'ro',
    default => 0.0001,
);

=head2 market

The financial market to which this asset belongs

=cut

has market => (
    is => 'ro',
    isa => 'Str',
);

=head2 display_name

User friendly name for the asset

=cut

has display_name => (
    is => 'ro',
    lazy_build => 1,
);

sub _build_display_name {
    my ($self) = @_;
    return uc $self->symbol;
}

=head2 exchange_name

Which exchange does the asset belong to. example: FOREX, NASDAQ, etc.

=cut

has _exchange_name => (
    is => 'ro',
);

has exchange_name => (
    is => 'ro',
    lazy_build => 1,
);

sub _build_exchange_name {
    my $self = shift;
    my $exchange_name = $self->_exchange_name || 'FOREX';

    if ($self->symbol =~ /^FUTE(B|C)/i) {

        # International Petroleum Exchange (now called ICE)
        $exchange_name = 'IPE';
    } elsif ($self->symbol =~ /^FUTLZ/i) {

        # Euronext LIFFE FTSE-100 Futures
        $exchange_name = 'EURONEXT';
    }

    return $exchange_name;
}

=head2 market_convention

Returns a hashref. Keys and possible values are:

=over 4

=item * atm_setting

Value can be one of:
    - atm_delta_neutral_straddle
    - atm_forward
    - atm_spot

=item * delta_premium_adjusted

Value can be one of:
    - 1
    - 0

=item * delta_style

Value can be one of:
    - spot_delta
    - forward_delta

=item * rr (Risk Reversal)

Value can be one of:
    - call-put
    - put-call

=item * bf (Butterfly)

Value can be one of:
    - 2_vol

=back

=cut

has market_convention => (
    is      => 'ro',
    isa     => 'HashRef',
    default => sub {
        return {
            delta_style            => 'spot_delta',
            delta_premium_adjusted => 0,
        };
    },
);

=head2 divisor

Divisor

=cut

has divisor => (
    is      => 'ro',
    isa     => 'Num',
    default => 1,
);

=head2 display_decimals

How many decimals to display for this asset

=cut

has display_decimals => (
    is         => 'ro',
    isa        => 'Num',
    lazy_build => 1,
);

sub _build_display_decimals {
    my $self = shift;

    return log(1 / $self->pip_size) / log(10);
}

no Moose;

__PACKAGE__->meta->make_immutable(
    constructor_name    => '_new',
    replace_constructor => 1
);

__END__

=head1 DEPENDENCIES

=over 4

=item L<Moose>

=item L<YAML::CacheLoader>

=item L<File::ShareDir>

=back

=head1 SOURCE CODE

L<GitHub|https://github.com/binary-com/perl-Finance-Asset>

=head1 AUTHOR

binary.com, C<< <perl at binary.com> >>

=head1 BUGS

Please report any bugs or feature requests to
C<bug-moosex-role-registry at rt.cpan.org>, or through the web
interface at
L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Finance-Asset>.
We will be notified, and then you'll automatically be notified of progress on
your bug as we make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Finance::Asset

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Finance-Asset>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Finance-Asset>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Finance-Asset>

=item * Search CPAN

L<http://search.cpan.org/dist/Finance-Asset/>

=back

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2015 binary.com

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=cut

1;
