# Copyright 2019, Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# This module provides utility methods to other services.

package Google::Ads::GoogleAds::Utilities::GoogleAdsHelper;

use strict;
use warnings;

# The following needs to be on one line because CPAN uses a particularly hacky
# eval() to determine module versions.
use Google::Ads::GoogleAds::Constants; our $VERSION = ${Google::Ads::GoogleAds::Constants::VERSION};

use Exporter 'import';
our @EXPORT = qw(remove_unassigned_fields check_params trim);

# Deletes the unassigned fields in the hash reference.
sub remove_unassigned_fields {
  my ($hash_ref, %args) = @_;
  delete @{$hash_ref}{grep { not exists $args{$_} } keys %$hash_ref};
}

# Checks whether the scripts parameters are not the "INSERT_..._HERE" values.
sub check_params {
  my (@params) = @_;
  foreach my $param (@params) {
    if ($param =~ /INSERT_.*_HERE/) {
      return 0;
    }
  }
  return 1;
}

# Remove the leading and trailing white spaces from a string.
sub trim {
  my $str = shift;
  $str =~ s/^\s*(.*?)\s*$/$1/;
  return $str;
}

# TODO
sub field_mask {
  my ($original, $modified) = @_;
  return [];
}

1;

=pod

=head1 NAME

Google::Ads::GoogleAds::Utilities::GoogleAdsHelper

=head1 DESCRIPTION

This module provides utility methods to other services.

=head1 METHODS

=head2 remove_unassigned_fields

Removes the fields not presenting in the constructor arguments from a newly created
Google Ads API entity. These fields will be excluded when encoding the JSON HTTP
request payload.

=head3 Parameters

=over

=item *

I<hash_ref>: a hash reference to the newly created Google Ads API entity.

=item *

I<args>: the arguments for the constructor of a Google Ads API entity.

=back

=head2 check_params

Checks whether the parameters in the code sample are correctly specified. The values
can either be set in the source code or passed in from the command line.

=head3 Parameters

=over

=item *

I<params>: an array of parameters in the code sample to verify.

=back

=head3 Returns

True, if all the parameters are correctly specified. False, otherwise.

=head2 trim

Remove the leading and trailing white spaces from a string.

=head3 Parameters

=over

=item *

The original input string.

=back

=head3 Returns

The trimmed string without leading and trailing white spaces.

=head1 LICENSE AND COPYRIGHT

Copyright 2019 Google LLC

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

=head1 REPOSITORY INFORMATION

 $Rev: $
 $LastChangedBy: $
 $Id: $

=cut
