package Net::DNS::SEC::ECDSA;

#
# $Id: ECDSA.pm 1669 2018-04-27 10:17:13Z willem $
#
our $VERSION = (qw$LastChangedRevision: 1669 $)[1];


=head1 NAME

Net::DNS::SEC::ECDSA - DNSSEC ECDSA digital signature algorithm


=head1 SYNOPSIS

    require Net::DNS::SEC::ECDSA;

    $signature = Net::DNS::SEC::ECDSA->sign( $sigdata, $private );

    $validated = Net::DNS::SEC::ECDSA->verify( $sigdata, $keyrr, $sigbin );


=head1 DESCRIPTION

Implementation of ECDSA elliptic curve digital signature
generation and verification procedures.

=head2 sign

    $signature = Net::DNS::SEC::ECDSA->sign( $sigdata, $private );

Generates the wire-format signature from the sigdata octet string
and the appropriate private key object.

=head2 verify

    $validated = Net::DNS::SEC::ECDSA->verify( $sigdata, $keyrr, $sigbin );

Verifies the signature over the sigdata octet string using the specified
public key resource record.

=cut

use strict;
use integer;
use warnings;
use MIME::Base64;


my %ECDSA = (
	13 => [415, 32, sub { Net::DNS::SEC::libcrypto::EVP_sha256() }],
	14 => [715, 48, sub { Net::DNS::SEC::libcrypto::EVP_sha384() }],
	);


sub sign {
	my ( $class, $sigdata, $private ) = @_;

	my $algorithm = $private->algorithm;
	my ( $nid, $keylen, $evpmd ) = @{$ECDSA{$algorithm} || []};
	die 'private key not ECDSA' unless $nid;

	my $rawkey = pack "a$keylen", decode_base64( $private->PrivateKey );

	my $eckey = Net::DNS::SEC::libcrypto::EC_KEY_new_by_curve_name($nid);
	Net::DNS::SEC::libcrypto::EC_KEY_set_private_key( $eckey, $rawkey );

	my $evpkey = Net::DNS::SEC::libcrypto::EVP_PKEY_new();
	Net::DNS::SEC::libcrypto::EVP_PKEY_assign_EC_KEY( $evpkey, $eckey );

	my $ASN1 = Net::DNS::SEC::libcrypto::EVP_sign( $sigdata, $evpkey, &$evpmd );
	my $l = 2 + unpack 'x3 C', $ASN1;
	pack 'a* a*', map _BERunwrap( $_, $keylen ), unpack "x2 a$l a*", $ASN1;
}


sub verify {
	my ( $class, $sigdata, $keyrr, $sigbin ) = @_;

	my $algorithm = $keyrr->algorithm;
	my ( $nid, $keylen, $evpmd ) = @{$ECDSA{$algorithm} || []};
	die 'private key not ECDSA' unless $nid;

	return unless $sigbin;

	my $eckey = Net::DNS::SEC::libcrypto::EC_KEY_new_by_curve_name($nid);
	my ( $x, $y ) = unpack "a$keylen a$keylen", $keyrr->keybin;
	Net::DNS::SEC::libcrypto::EC_KEY_set_public_key_affine_coordinates( $eckey, $x, $y );

	my $evpkey = Net::DNS::SEC::libcrypto::EVP_PKEY_new();
	Net::DNS::SEC::libcrypto::EVP_PKEY_assign_EC_KEY( $evpkey, $eckey );

	my ( $r, $s ) = map _BERwrap( $_, $keylen ), unpack "a$keylen a$keylen", $sigbin;
	my $ASN1 = pack 'n a* a*', 0x3000 + length($r) + length($s), $r, $s;

	Net::DNS::SEC::libcrypto::EVP_verify( $sigdata, $ASN1, $evpkey, &$evpmd );
}


sub _BERwrap {
	my $s = shift;
	$s =~ s/^\000+(?=.)//;
	$s =~ s/^(?=[\200-\377])/\000/;
	pack 'C2 a*', 2, length($s), $s;
}

sub _BERunwrap {
	my ( $l, $s ) = unpack 'x C a*', shift;
	substr pack( "x56 a$l", $s ), -shift;
}


1;

__END__

########################################

=head1 ACKNOWLEDGMENT

Thanks are due to Eric Young and the many developers and
contributors to the OpenSSL cryptographic library.


=head1 COPYRIGHT

Copyright (c)2014,2018 Dick Franks.

All rights reserved.


=head1 LICENSE

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted, provided
that the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation, and that the name of the author not be used in advertising
or publicity pertaining to distribution of the software without specific
prior written permission.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.


=head1 SEE ALSO

L<Net::DNS>, L<Net::DNS::SEC>,
RFC6090, RFC6605,
L<OpenSSL|http://www.openssl.org/docs>

=cut

