package Bif::Role::Sync::Project;
use strict;
use warnings;
use Coro;
use DBIx::ThinSQL qw/qv/;
use Log::Any '$log';
use Role::Basic;

our $VERSION = '0.1.0_27';

my %import_functions = (
    NEW => {
        topic                 => 'func_import_topic',
        entity                => 'func_import_entity',
        entity_contact_method => 'func_import_entity_contact_method',
        identity              => 'func_import_identity',
        issue                 => 'func_import_issue',
        issue_status          => 'func_import_issue_status',
        project               => 'func_import_project',
        project_status        => 'func_import_project_status',
        task                  => 'func_import_task',
        task_status           => 'func_import_task_status',
        update                => 'func_import_update',
    },
    UPDATE => {
        entity                => 'func_import_entity_delta',
        entity_contact_method => 'func_import_entity_contact_method_delta',
        identity              => 'func_import_identity_delta',
        issue                 => 'func_import_issue_delta',
        issue_status          => 'func_import_issue_status_delta',
        project               => 'func_import_project_delta',
        project_status        => 'func_import_project_status_delta',
        task                  => 'func_import_task_delta',
        task_status           => 'func_import_task_status_delta',
        update                => 'func_import_update_delta',
    },
    QUIT   => {},
    CANCEL => {},
);

sub recv_project_deltas {
    my $self = shift;
    my $db   = $self->db;

    my ( $action, $total ) = $self->read;
    $total //= '*undef*';

    if ( $action ne 'TOTAL' or $total !~ m/^\d+$/ ) {
        return "expected TOTAL <int> (not $action $total)";
    }

    my $ucount;
    my $i   = $total;
    my $got = 0;

    $self->updates_torecv( $self->updates_torecv + $total );
    $self->trigger_on_update;

    while ( $got < $total ) {
        my ( $action, $type, $ref ) = $self->read;

        if ( !exists $import_functions{$action} ) {
            return "not implemented: $action";
        }

        if ( !exists $import_functions{$action}->{$type} ) {
            return "not implemented: $action $type";
        }

        my $func = $import_functions{$action}->{$type};

        my $id;
        if ( $action eq 'NEW' and $type eq 'update' ) {
            $ucount = delete $ref->{ucount};

            $id = $db->xval(
                select => 'u.id',
                from   => 'updates u',
                where  => { 'u.uuid' => $ref->{uuid} },
            );
        }

        $got++;
        $ucount--;
        $self->updates_recv( $self->updates_recv + 1 );

        # If we already have this update then skip the rest of the
        # deltas
        if ($id) {
            while ($ucount) {
                $self->read;
                $got++;
                $ucount--;
                $self->updates_recv( $self->updates_recv + 1 );
            }
        }
        else {

            # This should be a savepoint?
            my $res = $db->xdo(
                insert_into => $func,
                values      => $ref,
            );

            if ( 0 == $ucount ) {
                $db->xdo(
                    insert_into => 'func_merge_updates',
                    values      => { merge => 1 },
                );
            }
        }

        $self->trigger_on_update;
    }

    return $total;
}

sub real_import_project {
    my $self = shift;
    my $uuid = shift;

    my $result = $self->recv_project_deltas;

    if ( $result =~ m/^\d+$/ ) {
        my $id = $self->db->xval(
            select => 't.id',
            from   => 'topics t',
            where  => {
                't.uuid' => $uuid,
            },
        );

        $self->db->xdo(
            update => 'projects',
            set    => 'local = 1',
            where  => { id => $id },
        );

        $self->write( 'Recv', $result );
        return 'ProjectImported';
    }

    $self->write( 'ProtocolError', $result );
    return $result;
}

sub real_sync_project {
    my $self   = shift;
    my $id     = shift || die caller;
    my $ids    = shift || die caller;
    my $prefix = shift // '';

    my @ids       = grep { $_ != $id } @$ids;
    my $tmp       = $self->temp_table;
    my $prefix2   = $prefix . '_';
    my $db        = $self->db;
    my $on_update = $self->on_update;
    my $hub_id    = $self->hub_id;

    $on_update->( 'matching: ' . $prefix2 ) if $on_update;

    my @refs = $db->xarrayrefs(
        select => [qw/pm.prefix pm.hash/],
        from   => 'project_related_updates_merkle pm',
        where  => [
            'pm.project_id = ',     qv($id),
            ' AND pm.hub_id = ',    qv($hub_id),
            ' AND pm.prefix LIKE ', qv($prefix2)
        ],
    );

    my $here = { map { $_->[0] => $_->[1] } @refs };
    $self->write( 'MATCH', $prefix2, $here );
    my ( $action, $mprefix, $there ) = $self->read;

    return "expected MATCH $prefix2 {} (not $action $mprefix ...)"
      unless $action eq 'MATCH'
      and $mprefix eq $prefix2
      and ref $there eq 'HASH';

    my @next;
    my @missing;

    while ( my ( $k, $v ) = each %$here ) {
        if ( !exists $there->{$k} ) {
            push( @missing, $k );
        }
        elsif ( $there->{$k} ne $v ) {
            push( @next, $k );
        }
    }

    if (@missing) {
        my @where;
        foreach my $miss (@missing) {
            push( @where, ' OR ' ) if @where;
            push( @where, "u.uuid LIKE ", qv( $miss . '%' ) ),;
        }

        $self->db->xdo(
            insert_into => "$tmp(id,ucount)",
            select      => [ 'u.id', 'u.ucount' ],
            from        => 'updates u',
            inner_join  => 'project_related_updates pru',
            on          => {
                'pru.update_id'           => \'u.id',
                'pru.project_id'          => $id,
                'NOT pru.real_project_id' => \@ids,
            },
            inner_join => 'projects_tree pt',
            on         => {
                'pt.child'  => \'pru.project_id',
                'pt.parent' => $id,
            },
            where => \@where,
        );
    }

    if (@next) {
        foreach my $next ( sort @next ) {
            $self->real_sync_project( $id, $ids, $next, $tmp );
        }
    }

    return unless $prefix eq '';

    return 'ProjectSync';
}

sub real_transfer_project_related_updates {
    my $self = shift;
    my $tmp  = $self->temp_table;

    my $send = async {
        my $total = $self->db->xval(
            select => 'COALESCE(sum(t.ucount), 0)',
            from   => "$tmp t",
        );

        $self->updates_tosend( $self->updates_tosend + $total );
        $self->write( 'TOTAL', $total );

        my $update_list = $self->db->xprepare(
            select => [
                'u.id',                  'u.uuid',
                'p.uuid AS parent_uuid', 't.uuid AS identity_uuid',
                'u.mtime',               'u.mtimetz',
                'u.author',              'u.email',
                'u.lang',                'u.message',
                'u.action',              'u.ucount',
            ],
            from       => "$tmp tmp",
            inner_join => 'updates u',
            on         => 'u.id = tmp.id',
            left_join  => 'topics t',

            # Don't fetch the identity_uuid for the first identity
            # update
            on        => 't.id = u.identity_id AND t.first_update_id != u.id',
            left_join => 'updates p',
            on        => 'p.id = u.parent_id',
            order_by  => 'u.id ASC',
        );

        $update_list->execute;
        return $self->send_updates( $update_list, $total );
    };

    my $r1 = $self->recv_project_deltas;
    my $r2 = $send->join;

    $self->db->xdo( delete_from => $tmp );

    if ( $r1 =~ m/^\d+$/ ) {
        $self->write( 'Recv', $r1 );
        my ( $recv, $count ) = $self->read;
        return 'TransferProjectRelatedUpdates'
          if $recv eq 'Recv' and $count == $r2;
        $log->debug("MEH: $count $r2");
        return $recv;
    }

    $self->write( 'ProtocolError', $r1 );
    return $r1;
}

sub real_export_project {
    my $self = shift;
    my $id   = shift;

    my $total = $self->db->xval(
        select     => 'sum(u.ucount)',
        from       => 'project_related_updates pru',
        inner_join => 'updates u',
        on         => 'u.id = pru.update_id',
        where      => { 'pru.project_id' => $id },
    );

    $self->updates_tosend( $self->updates_tosend + $total );
    $self->write( 'TOTAL', $total );

    my $sth = $self->db->xprepare(
        select => [
            'updates.id',                  'updates.uuid',
            'parents.uuid AS parent_uuid', 't.uuid AS identity_uuid',
            'updates.mtime',               'updates.mtimetz',
            'updates.author',              'updates.email',
            'updates.lang',                'updates.message',
            'updates.action',              'updates.ucount',
        ],
        from       => 'project_related_updates AS pru',
        inner_join => 'updates',
        on         => 'updates.id = pru.update_id',
        left_join  => 'topics t',

        # Don't fetch the identity_uuid for the first identity
        # update
        on        => 't.id = u.identity_id AND t.first_update_id != u.id',
        left_join => 'updates AS parents',
        on        => 'parents.id = updates.parent_id',
        where     => { 'pru.project_id' => $id },
        order_by  => 'updates.id ASC',
    );

    $sth->execute;
    $self->send_updates( $sth, $total );

    my ( $recv, $count ) = $self->read;
    return 'ProjectExported' if $recv eq 'Recv' and $count == $total;
    return $recv;
}

1;
