# Copyrights 2007-2017 by [Mark Overmeer].
#  For other contributors see ChangeLog.
# See the manual pages for details on the licensing terms.
# Pod stripped from pm file by OODoc 2.02.
package Dancer2::Logger::LogReport;
use vars '$VERSION';
$VERSION = '1.23';

# ABSTRACT: Dancer2 logger engine for Log::Report

use strict;
use warnings;

use Moo;
use Dancer2::Core::Types;
use Scalar::Util qw/blessed/;
use Log::Report  'log-report', syntax => 'REPORT';

our $AUTHORITY = 'cpan:MARKOV';

my %level_dancer2lr =
  ( core  => 'TRACE'
  , debug => 'TRACE'
  );

with 'Dancer2::Core::Role::Logger';

# Set by calling function
has dispatchers =>
  ( is     => 'ro'
  , isa    => Maybe[HashRef]
  );

sub BUILD
{   my $self     = shift;
    my $configs  = $self->dispatchers || {default => undef};
    $self->{use} = [keys %$configs];

    dispatcher 'do-not-reopen';

    foreach my $name (keys %$configs)
    {   my $config = $configs->{$name} || {};
        if(keys %$config)
        {   my $type = delete $config->{type}
                or die "dispatcher configuration $name without type";

            dispatcher $type, $name, %$config;
        }
    }
}

around 'error' => sub {
    my ($orig, $self) = (shift, shift);

    # If it's a route exception (generated by Dancer) and we're also using the
    # Plugin, then the plugin will handle the exception using its own hook into
    # the error system. This should be able to removed in the future with
    # https://github.com/PerlDancer/Dancer2/pull/1287
    return if $_[0] =~ /^Route exception/
           && $INC{'Dancer2/Plugin/LogReport.pm'};

    $self->log(error => @_);
};


sub log   # no protoypes in Dancer2
{   my ($self, $level, $msg) = @_;

    my %options;
    # If only using the logger on its own (without the associated plugin), make
    # it behave like a normal Dancer logger
    unless ($INC{'Dancer2/Plugin/LogReport.pm'})
    {   $msg = $self->format_message($level, $msg);
        $options{is_fatal} = 0;
    }

    # the levels are nearly the same.
    my $reason = $level_dancer2lr{$level} || uc $level;

    report \%options, $reason => $msg;
    undef;
}
 
1;
