package OpenInteract2::Brick::BaseBox;

use strict;
use base qw( OpenInteract2::Brick );
use OpenInteract2::Exception;

my %INLINED_SUBS = (
    'base_box-2.21.zip' => 'BASE_BOX221ZIP',
);

sub get_name {
    return 'base_box';
}

sub get_resources {
    return (
        'base_box-2.21.zip' => [ 'pkg base_box-2.21.zip', 'no' ],
    );
}

sub load {
    my ( $self, $resource_name ) = @_;
    my $inline_sub_name = $INLINED_SUBS{ $resource_name };
    unless ( $inline_sub_name ) {
        OpenInteract2::Exception->throw(
            "Resource name '$resource_name' not found ",
            "in ", ref( $self ), "; cannot load content." );
    }
    return $self->$inline_sub_name();
}

OpenInteract2::Brick->register_factory_type( get_name() => __PACKAGE__ );

=pod

=head1 NAME

OpenInteract2::Brick::BaseBox - Base-64 encoded OI2 package 'base_box-2.21.zip' shipped with distribution

=head1 SYNOPSIS

  oi2_manage create_website --website_dir=/path/to/site

=head1 DESCRIPTION

Are you sure you even need to be reading this? If you are just looking
to install a package just follow the instructions from the SYNOPSIS.

Still here? This class holds the Base64-encoded versions of package
file "base_box-2.21.zip" shipped with OpenInteract2. Once you decode them you
should store them as a ZIP file and then read them in with
Archive::Zip or some other utility.

A typical means to do this is:

 my $brick = OpenInteract2::Brick->new( 'base_box' );

 # there's only one resource in this brick...
 my ( $pkg_name ) = $brick->list_resources;
 my $pkg_info = $brick->load_resource( $pkg_name );
 my $pkg_file = OpenInteract2::Util->decode_base64_and_store(
     \$pkg_info->{content}
 );

 # $pkg_file now references a file on the filesystem that you can
 # manipulate as normal

These resources are associated with OpenInteract2 version 1.99_06.


=head2 Resources

You can grab resources individually using the names below and
C<load_resource()> and C<copy_resources_to()>, or you can copy all the
resources at once using C<copy_all_resources_to()> -- see
L<OpenInteract2::Brick> for details.

=over 4


=item B<base_box-2.21.zip>


=back

=head1 COPYRIGHT

Copyright (c) 2005 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS


Chris Winters E<lt>chris@cwinters.comE<gt>


=cut


sub BASE_BOX221ZIP {
    return <<'SOMELONGSTRING';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SOMELONGSTRING
}

