package OpenInteract2::Action::Security;

# $Id: Security.pm,v 1.5 2003/06/07 17:59:21 lachoy Exp $

use strict;
use base qw( OpenInteract2::Action );
use Data::Dumper             qw( Dumper );
use OpenInteract2::Constants qw( :log );
use OpenInteract2::Context   qw( CTX DEBUG LOG );
use SPOPS::Secure            qw( :level :scope );
use SPOPS::Secure::Hierarchy qw( $ROOT_OBJECT_NAME );

$OpenInteract2::Action::Security::VERSION = sprintf("%d.%02d", q$Revision: 1.5 $ =~ /(\d+)\.(\d+)/);

my $SECURE_CLASS = 'SPOPS::Secure';

# Display the object classes and handler classes currently used by
# this website and track those that are using security

sub list {
    my ( $self ) = @_;
    my ( @object_class, @action_class );
    my $spops_config = CTX->spops_config;
    my $action_table = CTX->action_table;
    my ( %classes_visited );
    my %params = (
      object_list => $self->_process_config( $spops_config, \%classes_visited ),
      action_list => $self->_process_config( $action_table, \%classes_visited ),
    );
    return $self->generate_content(
               \%params, { name => 'base_security::object_class_list' });
}


sub _process_config {
    my ( $self, $config, $visited ) = @_;
    my @object_class = ();
    foreach my $key ( sort keys %{ $config } ) {
        next if ( $key =~ /^_/ );
        my $item_class = $config->{ $key }{class};
        next unless ( $item_class );
        next if ( $visited->{ $item_class } );
        DEBUG && LOG( LDEBUG, "Processing [$key: $item_class]" );
        my $is_secure    = $item_class->isa( 'SPOPS::Secure' );
        my $is_hierarchy = $item_class->isa( 'SPOPS::Secure::Hierarchy' );
        push @object_class, { name             => $key,
                              class            => $item_class,
                              secure           => $is_secure,
                              hierarchy_secure => $is_hierarchy  };
        $visited->{ $item_class }++;
    }
    return \@object_class;
}


sub display {
    my ( $self ) = @_;
    my ( $object_class, $object_id ) = $self->_find_object_info;
    DEBUG && LOG( LDEBUG, "Display security for [$object_class: $object_id]" );
    unless ( $object_class ) {
        DEBUG && LOG( LINFO, "No security class specified, error" );
        return $self->generate_content(
                    {}, { name => 'base_security::error_no_class' });
    }

    unless ( $object_class->isa( 'SPOPS::Secure' )  ) {
        DEBUG && LOG( LINFO, "Specified class is not secured, error" );
        return $self->generate_content(
                    { object_class => $object_class },
                    { name => 'base_security::error_not_secured' } );
    }

    my $request = CTX->request;
    if ( $object_class->isa( 'SPOPS::Secure::Hierarchy' ) ) {
        my $drilldown = $request->param( 'drilldown' );
        unless ( $drilldown ) {
            DEBUG && LOG( LINFO, "Hierarchical class, no drilldown, show summary" );
            return $self->execute({ task => 'hierarchy_display' });
        }
    }
    my ( $type, $desc, $url ) =
        $self->_fetch_description({ object       => $self->param( 'object' ),
                                    object_id    => $object_id,
                                    object_class => $object_class });
    my %params = ( object_class       => $object_class,
                   object_id          => $object_id,
                   object_description => $desc,
                   object_type        => $type,
                   object_url         => $url );

    # Now fetch the security info -- we want to see who already has
    # security set so we can display that information next to the name
    # of the group/user or world in the listing

    my $security = eval {
        CTX->lookup_object( 'security' )
           ->fetch_by_object( undef, { class     => $object_class,
                                       object_id => $object_id,
                                       group     => 'all' } )
    };
    if ( $@ ) {
        LOG( LERROR, "Error fetching security for [$object_class: ",
                      "$object_id]: $@" );
        $self->param_add( error_msg => "Cannot fetch security: $@" );
        return $self->generate_content(
                    {},
                    { name => 'base_security::error_security_fetch' } );
    }

    # First item in the scope is the WORLD setting

    my $world_level = $security->{ SEC_SCOPE_WORLD() };
    my @scopes = ({ scope => SEC_SCOPE_WORLD,
                    name  => 'World',
                    level => $world_level });

    push @scopes, $self->_get_group_scopes( $security );

    # NOTE: We do not fetch user-level security unless specifically
    # requested

    if ( $request->param( 'include_user' ) ) {
        push @scopes, $self->_get_user_scopes( $security );
    }

    $params{scope_list} = \@scopes;
    return $self->generate_content(
                    \%params,
                    { name => 'base_security::assign_object_security' } );
}


sub hierarchy_display {
    my ( $self ) = @_;
    my $object = $self->param( 'object' );
    my ( $object_class, $object_id ) = $self->_find_object_info;
    DEBUG && LOG( LDEBUG, "Display hierarchy security for [$object_class: ",
                          "$object_id]" );

    my $request = CTX->request;

    # Retrieve the security levels so we can display them -- 'user'
    # and 'group' aren't really necessary here and are just passed to
    # keep SPOPS::Secure from doing lots of work...

    my ( $track, $first, $check_list ) =
            SPOPS::Secure::Hierarchy->get_hierarchy_levels(
              { class                 => $object_class,
                object_id             => $object_id,
                security_object_class => CTX->lookup_object( 'security' ),
		        user                  => $request->auth_user,
                group                 => $request->auth_group });

    my @check_list_items =
         map { { object_id        => $_,
                 security_defined => $track->{ $_ } } } @{ $check_list };

    my ( $type, $desc, $url ) =
        $self->_fetch_description({ object       => $object,
                                    object_id    => $object_id,
                                    object_class => $object_class });
    my %params = ( object_class       => $object_class,
                   object_id          => $object_id,
                   check_list         => \@check_list_items,
                   ROOT_OBJECT_NAME   => $ROOT_OBJECT_NAME,
                   object_description => $desc,
                   object_type        => $type,
                   object_url         => $url );
    return $self->generate_content(
                    \%params,
                    { name => 'base_security::hierarchy_security' } );
}



# Edit security for a particular object or class -- note that the
# widget currently only supports setting one level for many scopes at
# one time.

sub update {
    my ( $self ) = @_;
    my $request = CTX->request;
    my ( $level ) = $request->param( 'level' );
    my ( $object_class, $object_id ) = $self->_find_object_info;
    DEBUG && LOG( LDEBUG, "Setting security for [$object_class: ",
                          "$object_id]" );
    my @raw_scope = $request->param( 'scope' );

    # A link with this information exists on the hierarchical security
    # editing screen and clears out all security for a given class and
    # ID so that ID will inherit from its parents

    # TODO: Pass the 'clear' constant to the editing form so it can be
    # consistent

    if ( $raw_scope[0] eq 'all' and $level eq 'clear' ) {
        $self->_clear_all_security({ object_class => $object_class,
                                     object_id    => $object_id });
        $self->param( object_class => $object_class );
        $self->param( object_id    => $object_id );
        return $self->execute({ task => 'display' });;
    }
    my @scope = map { [ split /\s*;\s*/ ] } @raw_scope;

    # Cycle through each scope specification (scope + scope_id) and
    # set its security for the given object class and ID

    my ( $total, $success ) = ( 0, 0 );
    my ( @ok, @error );
    my $security_class = CTX->lookup_object( 'security' );
    foreach my $info ( @scope ) {
        $total++;
        DEBUG && LOG( LDEBUG, "Trying [Level: $level] for [$object_class: ",
                      "$object_id] scope [$info->[0]: $info->[1]]" );
        my %security_params = ( security_object_class => $security_class,
                                class          => $object_class,
                                object_id      => $object_id,
                                scope          => $info->[0],
                                scope_id       => $info->[1],
                                security_level => $level );
        my $action = ( $level eq 'clear' )
                       ? 'remove' : 'set';
        my $method = "${action}_item_security";
        eval { $SECURE_CLASS->$method( \%security_params ) };
        if ( $@ ) {
            $self->param_add(
                error_msg => "Failed to $action security for ",
                             "[$object_class: $object_id] $@" );
        }
        else {
            $self->param_add(
                status_msg => "Successful $action security for ",
                              "[$object_class: $object_id]" );
        }
    }
    $self->param( object_class => $object_class );
    $self->param( object_id    => $object_id );
    return $self->execute({ task => 'display' });

}


# Clear all security for a particular object class and ID

sub _clear_all_security {
    my ( $self, $p ) = @_;
    DEBUG && LOG( LDEBUG, "Clearing all security for [$p->{object_class}] ",
                          "[$p->{object_id}]" );
    my $security_list = eval {
        CTX->lookup_object( 'security' )
           ->fetch_group({ object_id => $p->{object_id},
                           class     => $p->{object_class} })
    };
    foreach my $security_obj ( @{ $security_list } ) {
        my $id = "[$p->{object_class}: $p->{object_id}] " .
                 "[$p->{scope} $p->{scope_id}]";
        eval { $security_obj->remove };
        if ( $@ ) {
            $self->param_add(
                error_msg => "Failed to remove security for $id : $@ " );
        }
        else {
            $self->param_add(
                status_msg => "Removed security for $id ok" );
        }
    }
}


sub _get_group_scopes {
    my ( $self, $security ) = @_;

    # Retrieve groups and match with security level

    my $group_list = eval { CTX->lookup_object( 'group' )
                               ->fetch_group({ order => 'name' }) };
    if ( $@ ) {
        LOG( LERROR, "Failed to fetch groups: $@" );
        $self->param_add( error_msg => "Failed to fetch all groups: $@" );
        $group_list = [];
    }

    my @s = ();
    foreach my $group ( @{ $group_list } ) {
        my $gid = $group->{group_id};
        my $level = $security->{ SEC_SCOPE_GROUP() }->{ $gid };
        push @s, { scope    => SEC_SCOPE_GROUP,
                   scope_id => $gid,
                   name     => $group->{name},
                   level    => $level };
    }
    return @s;
}


sub _get_user_scopes {
    my ( $self, $security ) = @_;
    my $user_list = eval { CTX->lookup_object( 'user' )
                              ->fetch_group({ order => 'login_name' }) };
    if ( $@ ) {
        LOG( LERROR, "Failed to fetch users: $@" );
        $self->param_add( error_msg => "Failed to fetch all users: $@" );
        $user_list = [];
    }

    my ( @s );
    foreach my $user ( @{ $user_list } ) {
        my $uid = $user->{user_id};
        my $level = $security->{ SEC_SCOPE_USER() }->{ $uid };
        push @s, { scope    => SEC_SCOPE_USER,
                   scope_id => $uid,
                   name     => $user->{name},
                   level    => $level };
    }
    return @s;
}


# Get the object class and object ID either from the subroutine
# parameters, object passed in, or GET/POST parameters

sub _find_object_info {
    my ( $self ) = @_;
    my ( $object_class, $object_id );
    if ( ref $self->param( 'object' ) ) {
        $object_class = ref $self->param( 'object' );
        $object_id    = $self->param( 'object' )->id;
    }
    else {
        my $request = CTX->request;
        $object_class = $self->param( 'object_class' ) ||
                        $request->param( 'object_class' ) ||
                        $self->param( 'handler_class' ) ||
                        $request->param( 'handler_class' );
        $object_id    = $self->param( 'object_id' ) ||
                        $request->param( 'object_id' ) ||
                        $request->param( 'oid' ) ||
                        '0';
    }
    return ( $object_class, $object_id );
}


# Get the title of an object given an object or an object class and ID

sub _fetch_description {
    my ( $self, $params ) = @_;
    my $object = $params->{object};
    if ( ! $object and ! $params->{object_class} and ! $params->{object_id} ) {
        return ( 'n/a', 'n/a', undef );
    }
    my ( $name );
    unless ( $object ) {
        unless ( $params->{object_class}->isa( 'SPOPS' ) ) {
            return ( 'Handler', undef, undef );
        }
        $name = $params->{object_class}->CONFIG->{object_name}
                || 'unknown';
        $object = eval {
            $params->{object_class}->fetch( $params->{object_id} )
        };
        return ( $name, undef, undef ) if ( $@ or ! $object );
    }
    my $oi = $object->object_description;
    return ( $oi->{name}, $oi->{title}, $oi->{url} );
}


1;

__END__

=head1 NAME

OpenInteract2::Action::Security - Process changes to security made by users

=head1 SYNOPSIS

 # List the object and handler classes
 /Security/listing/

 # Display security settings for a particular object
 /Security/display/?object_id=13;object_class=MySite::Contact

=head1 DESCRIPTION

Handler to display and process the results of object-level security
setting.

=head1 METHODS

B<display>

Feeds the widget that allows users to edit security on a single object
or item.

B<hierarchy_display>

Feeds the widget that displays the parents of a particular object and
whether each one has security currently defined or not.

B<update>

Processes the results of the 'display' and 'hierarchy_display' tasks.

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters E<lt>chris@cwinters.comE<gt>
