package OpenInteract::Handler::LookupEdit;

# $Id: LookupEdit.pm,v 1.7 2001/11/09 14:08:38 lachoy Exp $

# See 'doc/lookup.pod' for description of the fields in the action
# table we use.

use strict;
use OpenInteract::Handler::GenericDispatcher;
use SPOPS::Secure qw( SEC_LEVEL_WRITE );

@OpenInteract::Handler::LookupEdit::ISA     = qw( OpenInteract::Handler::GenericDispatcher );
$OpenInteract::Handler::LookupEdit::VERSION = sprintf("%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/);

use constant DEFAULT_METHOD  => 'list_lookups';
use constant DEFAULT_DISPLAY => 'column';

my $BLANK_COUNT = 5;
my $NEW_KEY     = '_new_';
my $REMOVE_KEY  = '_remove_';


# Our handler acts as the normal redirector, but instead of passing in
# the security to the individual task we just assume that any access
# == all access

sub handler {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $task = lc shift @{ $R->{path}->{current} } || DEFAULT_METHOD;
    my $level = $R->user->check_security({ class     => $class,
                                           object_id => '0' });
    if ( $level < SEC_LEVEL_WRITE ) {
        $R->throw({ code  => 305,
                    type  => 'security',
                    extra => { user_level     => $level,
                               required_level => SEC_LEVEL_WRITE,
                               class          => $class,
                               task           => $task } });
    }
    return $class->$task();
}


# Just find all the lookup actions

sub list_lookups {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->{page}{title} = 'List Available Lookups';
    return $R->template->handler({}, { lookup_list => $class->_find_all_lookups,
                                       status_msg  => $p->{status_msg},
                                       error_msg   => $p->{error_msg} },
                                 { name => 'lookup::lookup_classes' });
}


# If data partitioning is specified, the view when accessing the
# lookup table is of a dropdown of the available values by which to
# partition the data

sub partition_listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $lookup_info = $p->{lookup_info};
    unless ( $lookup_info ) {
        my ( $error_msg );
        ( $lookup_info, $error_msg ) = $class->_find_lookup_info;
        unless ( $lookup_info ) {
            return $class->list_lookups({ error_msg => $error_msg });
        }
    }
    unless ( $lookup_info->{partition_field} ) {
        my $error_part_msg = "Cannot use ($lookup_info->{lookup_type}) as partitioned -- " .
                             "no value for 'partition_field' specified in action config.";
        return $class->list_lookups({ error_msg => $error_part_msg });
    }
    my $partition_values = eval { $class->_find_distinct_values(
                                               $lookup_info->{object_key},
                                               $lookup_info->{partition_field} ) };
    if ( $@ ) {
        $p->{error_msg} = "Could not retrieve values for " .
                          "($lookup_info->{partition_field}): " .
                          $SPOPS::Error::system_msg;
    }
    $R->{page}{title} = 'Lookup Partitioning Values';
    return $R->template->handler({}, { value_list  => $partition_values,
                                       lookup_type => $lookup_info->{lookup_type},
                                       error_msg   => $p->{error_msg} },
                                 { name => 'lookup::lookup_partitions' });
}


# List relevant entries in a particular lookup table

sub listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my ( $lookup_info, $error_msg ) = $class->_find_lookup_info;
    unless ( $lookup_info ) {
        return $class->list_lookups({ error_msg => $error_msg });
    }

    my @lookup_keys = qw( field_list label_list size_list title
                          lookup_type partition_field );
    my %params = map { $_ => $lookup_info->{ $_ } } @lookup_keys;

    $params{blank_count} = $BLANK_COUNT;
    $params{remove_key}  = $REMOVE_KEY;
    $params{new_key}     = $NEW_KEY;

    if ( $params{partition_field} ) {
        $params{partition_value} = $R->apache->param( 'partition_value' );
        unless ( $params{partition_value} ) {
            return $class->partition_listing({ lookup_info => $lookup_info });
        }
        $params{label_list} ||= [];
        my %lbl = map { $params{field_list}->[ $_ ] => $params{label_list}->[ $_ ] }
                      ( 0 .. ( scalar @{ $params{field_list} } - 1 ) );
        $params{partition_label} = $lbl{ $params{partition_field} } ||
                                   $params{partition_field};
    }

    $params{lookup_list} = $class->_lookup_entries( $lookup_info,
                                                    $params{partition_value} );

    # Check to see if the lookup action has defined a set of related
    # objects -- that is, the user when editing the lookup values
    # should choose one from many values.

    $lookup_info->{relate} ||= {};
    foreach my $field_name ( keys %{ $lookup_info->{relate} } ) {
        my $relate_info = $lookup_info->{relate}{ $field_name };
        next if ( $params{related}->{ $field_name } );
        $params{related}->{ $field_name } = $relate_info;
        $params{related}->{ $field_name }{list} = $class->_lookup_related_objects(
                                                              $relate_info->{object},
                                                              $relate_info );
    }

    my $display_type = $R->apache->param( 'display_type' ) || DEFAULT_DISPLAY;
    my $tmpl_name    = ( $display_type eq 'column' )
                         ? 'lookup_listing_columns' : 'lookup_listing';
    $R->{page}{title} = 'Edit Lookup Entries';
    return $R->template->handler({}, \%params,
                                 { name => "lookup::$tmpl_name" });
}


sub edit {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $apr = $R->apache;
    my $lookup_type  = $apr->param( 'lookup_type' );
    my ( $lookup_info, $error_msg ) = $class->_find_lookup_info( $lookup_type );
    unless ( $lookup_info ) {
        return $class->list_lookups({ error_msg => $error_msg });
    }
    my $object_key   = $lookup_info->{object_key};
    my $lookup_class = $R->$object_key();
    my @id_list      = $class->_retrieve_id_list( $apr, $lookup_info->{field_list}->[0] );
    my @new_id_list  = map { "$NEW_KEY$_" } ( 1 .. $BLANK_COUNT );
    my @save_params  = ( $apr, $lookup_class, $lookup_info->{field_list} );
    foreach my $id ( @id_list, @new_id_list ) {
        $R->DEBUG && $R->scrib( 1, "Trying to find values for ID ($id)" );
        eval { $class->_fetch_and_save( @save_params, $id ) };
        if ( $@ ) {
            $R->scrib( 0, "Cannot save ID ($id): ($@) $SPOPS::Error::system_msg" );
        }
    }
    return $class->list_lookups({
              status_msg => "Entries for $lookup_type entered successfully" });
}


# $field is just a sample field used to get IDs

sub _retrieve_id_list {
    my ( $class, $apr, $field ) = @_;
    my @fields = grep ! /^$field\-$NEW_KEY/, grep /^$field/, $apr->param;
    my ( @id_list );
    foreach my $this_field ( @fields ) {
        $this_field =~ /^$field\-(.*)$/;
        push @id_list, $1;
    }
    return @id_list;
}


sub _find_all_lookups {
    my ( $class ) = @_;
    my $R = OpenInteract::Request->instance;
    my $CONFIG = $R->CONFIG;
    my ( @lookup_list );
    foreach my $key ( keys %{ $CONFIG->{action} } ) {
        next unless ( $key );
        my ( $lookup_info, $error_msg ) = $class->_find_lookup_info( $key );
        if ( $lookup_info ) {
            $R->DEBUG && $R->scrib( 1, "Found lookup item ($key)" );
            push @lookup_list, $lookup_info;
        }
    }
    return \@lookup_list;
}


sub _find_distinct_values {
    my ( $class, $object_type, $field ) = @_;
    my $R = OpenInteract::Request->instance;
    my $object_class = $R->$object_type();
    return $object_class->db_select({ select_modifier => 'DISTINCT',
                                      select          => [ $field ],
                                      from            => [ $object_class->table_name ],
                                      order           => $field,
                                      return          => 'single-list' });
}


sub _find_lookup_info {
    my ( $class, $lookup_type ) = @_;
    my $R = OpenInteract::Request->instance;
    $lookup_type ||= $R->apache->param( 'lookup_type' );
    unless ( $lookup_type ) {
        return ( undef,  'Cannot list lookup entries without a lookup type.' );
    }
    my $lookup_info = $R->CONFIG->{action}{ $lookup_type };
    unless ( ref $lookup_info and $lookup_info->{is_lookup} ) {
        return ( undef, "Cannot list lookup entries: ($lookup_type) is not a valid lookup." );
    }
    $lookup_info->{lookup_type} = $lookup_type;
    return $lookup_info;

}


sub _lookup_entries {
    my ( $class, $lookup_info, $partition_value ) = @_;
    my $R = OpenInteract::Request->instance;
    my $lookup_object_key = $lookup_info->{object_key};
    my $lookup_class = $R->$lookup_object_key();
    $R->DEBUG && $R->scrib( 1, "Trying to find all entries in ($lookup_class);",
                               "in the order ($lookup_info->{order})" );
    my %args = ( order => $lookup_info->{order} );
    if ( $partition_value ) {
        $R->DEBUG && $R->scrib( 1, "Filtering entries by ",
                                   "($lookup_info->{partition_field}) = ($partition_value)" );
        $args{where} = "$lookup_info->{partition_field} = ?";
        $args{value} = [ $partition_value ];
    }
    return $lookup_class->fetch_group( \%args );
}


sub _lookup_related_objects {
    my ( $class, $object_type, $params ) = @_;
    my $R = OpenInteract::Request->instance;
    return $R->$object_type()->fetch_group({ order => $params->{order} })

}


sub _fetch_and_save {
    my ( $class, $apr, $lookup_class, $field_list, $id ) = @_;
    my $R = OpenInteract::Request->instance;
    my $is_new = ( $id =~ /^$NEW_KEY/ );
    my $object =  ( $is_new )
                    ? $lookup_class->new : $lookup_class->fetch( $id );
    my $not_blank = 0;
    my $do_remove = $apr->param( "$REMOVE_KEY-$id" );
    if ( $do_remove ) {
        return if ( $is_new );
        $R->DEBUG && $R->scrib( 1, "Trying to remove entry for ID ($id)" );
        return $class->_remove( $object );
    }
    foreach my $field ( @{ $field_list } ) {
        my $value = $apr->param( "$field-$id" );
        $R->DEBUG && $R->scrib( 1, "Found value: ($id) ($field): ($value)" );
        $object->{ $field } = $value;
        $not_blank++ if ( $value );
    }
    $object->save if ( $not_blank );
    return $object;
}


# We might want to add more stuff here...

sub _remove {
    my ( $class, $object ) = @_;
    return unless ( ref $object and $object->isa( 'SPOPS' ) );
    return $object->remove;
}


1;
