package OpenInteract::Handler::User;

# $Id: User.pm,v 1.2 2001/05/30 03:13:03 lachoy Exp $

use strict;
use SPOPS::Secure qw( :level :scope );

@OpenInteract::Handler::User::ISA     = qw( OpenInteract::Handler::GenericDispatcher  SPOPS::Secure );
$OpenInteract::Handler::User::VERSION = sprintf("%d.%02d", q$Revision: 1.2 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::User::author            = 'chris@cwinters.com';
$OpenInteract::Handler::User::default_method    = 'listing';
@OpenInteract::Handler::User::forbidden_methods = ();
%OpenInteract::Handler::User::security          = ( 
 listing => SEC_LEVEL_READ, edit   => SEC_LEVEL_WRITE,
 show    => SEC_LEVEL_READ, remove => SEC_LEVEL_WRITE,  
);

my $MAIN_SCRIPT = '/User';


# List available users

sub listing {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $params = { main_script => $MAIN_SCRIPT, error_msg => $p->{error_msg} };
  $params->{user_list} = eval { $R->user->fetch_group( { order => 'login_name' } ) };
  if ( $@ ) {
    OpenInteract::Error->set( SPOPS::Error->get );
    $R->throw( { code => 404 } );
    $params->{user_list} = [];
 }
 $R->{page}->{title} = 'User Listing';
 return $R->template->handler( {}, $params, { db => 'user_list', 
                                              package => 'base_user' } );
}



# Display a single user record, either in static or editable format.

sub show {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;

  # Add this to the hashref below if you want users to be able to choose themes
  # theme_list => eval { $R->theme->fetch_group( { order => 'title' } ) } 

  my $params = { main_script => $MAIN_SCRIPT, error_msg => $p->{error_msg} };
  $params->{user} = $p->{user};
  unless ( $params->{user} ) {
    my $user_id = $R->apache->param( 'user_id' ) || $R->apache->param( 'uid' ); 
    $params->{user}  = eval { $R->user->fetch( $user_id ) };
    if ( $@ ) {
      OpenInteract::Error->set( SPOPS::Error->get );
      $R->throw( { code => 404 } );
    }
  }
  
  # If the user has only READ access to module and there is no UID or
  # user object, there's no reason to continue
  
  if ( ! $params->{user} and $p->{level} < SEC_LEVEL_WRITE ) {
    my $msg = 'Sorry, you do not have access to create a new user object. Returning to listing.';
    return $class->listing( { error_msg => $msg } );
  }
  
  my $obj_level = ( $params->{user} ) 
                   ? $params->{user}->{tmp_security_level} 
                   : SEC_LEVEL_WRITE;

  $params->{user} ||= $R->user->new( { theme_id => $R->{theme}->{theme_id} } );
  $R->DEBUG && $R->scrib( 1, "Levels  $params->{user}->{user_id}: $obj_level/$p->{level}" );
  $params->{group_list} = eval { $params->{user}->group };

  my $tmpl_name = 'user_detail';

  if ( $p->{level} == SEC_LEVEL_WRITE and $obj_level == SEC_LEVEL_WRITE ) {
    $tmpl_name = 'user_form';
  }
  
  $R->{page}->{title} = 'View User Detail';
  return $R->template->handler( {}, $params, 
                                { db => $tmpl_name, package => 'base_user' } );
}

sub edit {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  $R->{page}->{return_url} = '/User/';
  my $apr = $R->apache;

  my $user_id = $apr->param( 'user_id' );
  $R->DEBUG && $R->scrib( 1, "Trying to modify ID <<$user_id>>" );
  
  my $is_new = 0;
  my $user = $p->{user} ||
             eval { $R->user->fetch( $user_id ) };
  if ( $@ ) {
    OpenInteract::Error->set( SPOPS::Error->get );
    $R->throw( { code => 404 } );
  }
  
  my $obj_level = ( $user ) ? $user->{tmp_security_level} : SEC_LEVEL_WRITE;
  if ( $obj_level < SEC_LEVEL_WRITE ) {
    my $user_msg = 'Sorry, you do not have access to modify this user object. Returning to listing.';
    return $class->listing( { error_msg => $user_msg } );
  }
  
  # Note if this user is new so we can do some security fun after the
  # information is saved

  unless ( $user ) {
    $user ||= $R->user->new;
    $is_new++;
  }

  # Uncomment the code if you are allowing users to change their own themes
  # Track this so we can change the theme if the user changed it.
  # my $pre_theme_id = $user->{theme_id} || $R->{theme}->{theme_id};
 
  foreach my $field ( @{ $R->user->field_list } ) {
    next if ( $field eq 'password' or $field eq 'user_id' );
    $user->{ $field } = $apr->param( $field );
  }
  my $password = $apr->param( 'password' );
  if ( $password ) {
    $R->DEBUG && $R->scrib( 1, "User asked to change password. Changing." );
    $password = $R->user->crypt_it( $password ) if ( $R->CONFIG->{login}->{crypt_password} );;
    $user->{password} = $password;
  }
  
  eval { $user->save };
  if ( $@ ) {
    OpenInteract::Error->set( SPOPS::Error->get );
    $OpenInteract::Error::user_msg = 'Error saving user object.';
    $R->throw( { code => 407 } );
    return $class->show( { user => $user, 
                           error_msg => 'Cannot save user object! See error log for details...' } );
  }

  # If this is a new user, allow the user to edit his/her own record;
  # other security settings (WORLD, GROUP for site group) should be
  # done in the normal way (via conf/spops.perl->{creation_security};
  # also make the user a member of group 'public'
  
  if ( $is_new ) {
    my $site_admin_id = $R->CONFIG->{default_objects}->{site_admin_group};
    eval { 
      $user->set_security( { scope => SEC_SCOPE_USER,
                             level => SEC_LEVEL_WRITE,
                             scope_id => $user->{user_id} } );
      $user->make_public();
    };
  }
  

  # Uncomment the code if you are allowing users to change their own themes
  # If the user changed his/her theme, show it immediately
# if ( $user->{user_id} == $R->{auth}->{user}->{user_id} and 
#      $user->{theme_id} != $pre_theme_id ) {
#   $R->{theme} = eval { $R->theme->fetch( $user->{theme_id} ) };
# }

  $p->{user} = $user;
  return $class->show( $p ); 
}

sub remove {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  if ( my $user_id = $R->apache->param( 'user_id' ) ) {
    my $user = eval { $R->user->fetch( $user_id ) };
    if ( $@ or ! $user ) {     
      return $class->listing( { error_msg => 'Cannot remove user -- object not created properly.' } );
    }
    if ( $user->{tmp_security_level} < SEC_LEVEL_WRITE ) {
      my $user_msg = 'Sorry, you do not have access to remove this user object. Returning to listing.';
      return $class->listing( { error_msg => $user_msg } );   
    }
    
    eval { $user->remove };
    if ( $@ ) {
      OpenInteract::Error->set( SPOPS::Error->get );
      $R->throw( { code => 405 } );
      $p->{error_msg} = 'Cannot remove user object! See error log for details...';
    }
  }
  return $class->listing( $p );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::User - Functions to manipulate user records

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 METHODS

=head1 NOTES

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
