package OpenInteract::Cookies;

# $Id: Cookies.pm,v 1.2 2001/02/21 13:10:03 lachoy Exp $

use strict;
use Data::Dumper qw( Dumper );

@OpenInteract::Cookies::ISA     = ();
$OpenInteract::Cookies::VERSION = sprintf("%d.%02d", q$Revision: 1.2 $ =~ /(\d+)\.(\d+)/);

sub parse {
  my ( $class ) = @_;
  my $R = OpenInteract::Request->instance;
  my $cookies = $class->get_cookies( $R->{use_cgi} );
  $R->{cookie}->{in} = $cookies;
  return 1
}

sub bake {
  my ( $class ) = @_;
  my $R = OpenInteract::Request->instance;

  # If we're using CGI, then we just put an arrayref of cookies
  # into $R for use later

  if ( $R->{use_cgi} ) {   
    $R->{cookie}->{baked} = [ values %{ $R->{cookie}->{out} } ];
  }

  # Cycle through the Apache::Cookie objects and 
  # call the bake method, which puts the appropriate header
  # into the outgoing headers table.

  else {   
    foreach my $name ( keys %{ $R->{cookie}->{out} } ) {
      $R->scrib( 2, "Setting $name to value ", $R->{cookie}->{out}->{ $name }->value );
      $R->{cookie}->{out}->{ $name }->bake;
    }
  }
  return 1;
}


# Retrieve the cookies whether we're using CGI or mod_perl

sub get_cookies {
  my ( $class, $is_cgi ) = @_;
  my $R = OpenInteract::Request->instance;
  my $cookies = {};
  if ( $is_cgi ) {
    my $q = $R->cgi;
    foreach my $name ( $q->cookie ) {
      my $value = $q->cookie( name => $name );
      $R->scrib( 2, "Getting cookie $name to $value" );
      $cookies->{ $name } = $value;
    }
  }
  else {
    my $cookie_info = Apache::Cookie->fetch;
    foreach my $name ( keys %{ $cookie_info } ) {
      my $value = $cookie_info->{ $name }->value;
      $R->scrib( 2, "Getting cookie $name to $value" );
      $cookies->{ $name } = $value;
    }
  }
  $R->scrib( 2, "Incoming cookie values: ", Dumper( $cookies ) );
  return $cookies;
}


# Create a new cookie whether we're using CGI or mod_perl

sub create_cookie {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $c = undef;
  my $is_cgi = $R->{use_cgi};
  if ( $is_cgi ) {
    $c = $R->cgi->cookie( -name => $p->{name}, -value => $p->{value},
                          -path => $p->{path}, -expires => $p->{expires} );
  }
  else {
    $c = Apache::Cookie->new( $R->apache, -name => $p->{name}, -value => $p->{value},
                              -path => $p->{path}, -expires => $p->{expires} );
  }
  return $R->{cookie}->{out}->{ $p->{name} } = $c;
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Cookies - handler to parse/output cookies from/to the client

=head1 DESCRIPTION

Simple to use.

=head1 METHODS

Methods for this clss.

B<create_cookie( \%params  )>

This function is probably the only one you will ever use from this
module. Pass in normal parameters (see below) and the function will
create a cookie and put it into $R for you.

Parameters:

 name: name of cookie
 path: path it responds to
 value: value of cookie
 expires: when it expires ( '+3d', etc.)

B<parse()>

Read in the cookies passed to this request and file them into the
hashref:

 $R->{cookie}->{in}

with the key as the cookie name.

B<bake()>

Puts the cookies into the headers_out if we are using mod_perl,
otherwise we put them into the arrayref:

 $R->{cookie}->{baked}

So they can be put into the CGI.pm header() call:

 $cgi->header( -cookie => $R->{cookie}->{baked} );

=head1 TO DO

=head1 BUGS 

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
