package OpenInteract::ErrorObject;

# $Id: ErrorObject.pm,v 1.10 2001/02/01 05:24:39 cwinters Exp $

use strict;

use Carp           qw( carp );
use Text::Wrap     ();
use Data::Dumper   qw( Dumper );
use Date::Format   qw( time2str );

$OpenInteract::ErrorObject::VERSION = sprintf("%d.%02d", q$Revision: 1.10 $ =~ /(\d+)\.(\d+)/);

use constant DEBUG  => 0;

my @LIST = (); # keeps errors during request
my $DEFAULT_ACTION = 'n/a';
my $DEFAULT_TYPE   = 'n/a';

# 
# Routines to manipulate the list of errors during a request
#

sub clear_listing { @LIST = () }
sub listing       { return \@LIST }
sub report        { return \@LIST }

# Class method
#

sub throw {
  my ( $class, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  
  unless ( $p->{package} and $p->{filename} and $p->{line} ) {
    my ( $cpkg, $cfile, $cline ) = caller;
    $p->{package}  = $cpkg;
    $p->{filename} = $cfile;
    $p->{line}     = $cline;
  }
  my $err = $class->new( $p );
  $R->scrib( 2, "Object after first create: ", Dumper( $err ) );
  push @LIST, $err; # save it for later...
  $R->error_handler->catch( $err );
}


# Called when we do: OpenInteract::ErrorObject->new() 
# (see SPOPS->new() for more info and how ->initialize() 
# is called)

sub initialize {
  my ( $self, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  $R->scrib( 2, "Initializing new error: ", Dumper( $p ) );

  # Set pkg (etc) context if not already set
  
  unless ( $p->{package} and $p->{filename} and $p->{line} ) {
    my ( $cpkg, $cfile, $cline ) = caller;
    $p->{package}  = $cpkg;
    $p->{filename} = $cfile;
    $p->{line}     = $cline;   
  }
  
  # Set any extra items

  my $ei = OpenInteract::Error->get;
  $p->{extra} ||= $ei->{extra} || {};
  my $extra_info = undef;
  foreach my $key ( keys %{ $p->{extra} } ) {
    $R->scrib( 1, "Set (tmp_$key) to $p->{extra}->{ $key }" );
    $self->{ "tmp_$key" } = $p->{extra}->{ $key };
    $extra_info .= $p->{extra}->{ $key } . "<br>\n";
  }
  
  # If there was nothing passed in except for the code,
  # then get the error information from the package
  # error holding variables.

  foreach my $field ( qw( type user_msg system_msg notes ) ) {
    $p->{ $field } ||= $ei->{ $field };
  }
  $p->{notes} .= $extra_info;
  
  # In any case, clear out any old messages; they're not 
  # needed anymore

  OpenInteract::Error->clear;
  
  # Get the action from $R if not already set

  $p->{action} ||= $R->{current_context}->{action} || $DEFAULT_ACTION;

  # Give a default type if not set

  $p->{type} = $DEFAULT_TYPE;

  # First call SPOPS::DBI->initialize() to set all the
  # information passed in matching up with the object
  # parameters

  $self->SUPER::initialize( $p );

  # We need to abstract out this header information
  # so we can set a switch to use CGI/Apache::Request

  my $head_in = undef;
  if ( $R->{use_cgi} ) {
    $head_in->{'User-Agent'} = $R->cgi->user_agent;
    $head_in->{'Referer'}    = $R->cgi->referer;
  }
  else {
    $head_in = $R->apache->headers_in;
  }

  # Set meta information

  $self->{action}    ||= $R->{current_context}->{action};
  $self->{user_id}     = $R->{auth}->{user}->id       if ( ref $R->{auth}->{user} );
  $self->{session_id}  = $R->{session}->{_session_id} if ( ref $R->{session}   );
  $self->{browser}     = $head_in->{'User-Agent'};
  $self->{error_time}  = $self->now;
  $self->{referer}     = $head_in->{'Referer'};
  return $self;
}


sub fail_save {
  my ( $self, $p ) = @_;
  my $R = OpenInteract::Request->instance;
  my $error_dir = $R->CONFIG->get_dir( 'error' );
  
  # Note that even though it's not saved, the ID should still
  # have been defined in the save() process since we
  # are using random codes. Also, do not do a ->throw with any 
  # errors generated by trying to save the file, otherwise you'll
  # get in an infinite loop, which would be bad.
  eval { open( ERROR, "> $error_dir/$self->{error_id}" ) || die $! };
  if ( $@ ) {
    $R->scrib( 0, "LAST DITCH ERROR: Cannot save error even to filesystem!\n",
                  "Save error message: $@\nError:", Dumper( $self ) );
    return undef;
  }

  $self->{error_time} = scalar localtime;
  print ERROR Dumper( $self );
  close( ERROR );
  $R->scrib( 0, "Error written out to <<$error_dir/$self->{error_id}>>" );
  return 1;
}

sub as_string {
  my ( $self, $opt ) = @_;
  local $Text::Wrap::columns = $opt->{columns} || 65; 
  my $user_msg = Text::Wrap::wrap( '', '', $self->{user_msg} );
  my $sys_msg  = Text::Wrap::wrap( '', '', $self->{system_msg } );
  my $string = <<STRING;
   Error: $self->{error_id}
    Date: $self->{error_time}
    Code: $self->{code}
    Type: $self->{type}
 Browser: $self->{browser}
 User ID: $self->{user_id}
 Session: $self->{session_id}

Messages

========
    User
========
$user_msg

========
  System
========
$sys_msg

STRING
  if ( $opt->{html} ) {
    $string =~ s/\n/<br>\n/g;
  }
  return $string;
}

1;

__END__

=pod

=head1 NAME

OpenInteract::ErrorObject - Use errors as objects and enable persistence

=head1 SYNOPSIS

 my $err = OpenInteract::ErrorObject->new( { code => 405, type => 'db' } );
 $err->save;

=head1 DESCRIPTION

The fact that each error is now an object that gets saved means that
we can create a simple (or complicated) web-based error browsing
application that shows you all errors in a certain time period plus
details, etc.

=head1 METHODS

B<throw( \%params )>

Throws an error from anywhere in the system. We create an error object
with the parameters (or with the error information already stored, see
below) and send the object onto the error handling framework within
OpenInteract. 

Parameters:

 code ($)
   Mandatory. See L<OpenInteract::Error::Main> for what the different
   codes signify. (Basically, the higher the code the less severe the
   error.)

If you call throw() with only the I<code> parameter, the method will
snatch the error information from the package variables already
set. Otherwise, see L<OpenInteract::Error> for the different
parameters that can be set.

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
