#! perl
# Copyright (C) 2007-2008, The Perl Foundation.
# $Id: /mirror/trunk/t/steps/gen_platform-01.t 33351 2008-11-28T22:41:42.455437Z coke  $
# gen_platform-01.t

use strict;
use warnings;
use Test::More tests => 19;
use Carp;
use Cwd;
use File::Copy;
use File::Path qw( mkpath );
use File::Temp qw( tempdir );
use File::Spec;
use lib qw( lib );
use_ok('config::gen::platform');
use Parrot::Configure;
use Parrot::Configure::Options qw( process_options );
use Parrot::Configure::Test qw(
    test_step_thru_runstep
    test_step_constructor_and_description
);
use Parrot::Configure::Utils qw( _slurp );
use IO::CaptureOutput qw( capture );

########## regular ##########

my ($args, $step_list_ref) = process_options(
    {
        argv => [ ],
        mode => q{configure},
    }
);

my $conf = Parrot::Configure->new;
my $pkg = q{gen::platform};
$conf->add_steps($pkg);
$conf->options->set( %{$args} );
my $step = test_step_constructor_and_description($conf);

ok(-f $step->{platform_interface},
    "Located required platform interface header");

my $platform_orig = $conf->data->get_p5('OSNAME');
my $archname_orig = $conf->data->get_p5('archname');
$conf->data->set_p5( archname => 'foo-bar' );
my $verbose = 0;

########## _get_platform() ##########

$conf->data->set_p5( OSNAME => 'msys' );
is( $step->_get_platform( $conf, $verbose ), q{win32},
    "Got expected platform for msys");

$conf->data->set_p5( OSNAME => 'mingw' );
is( $step->_get_platform( $conf, $verbose ), q{win32},
    "Got expected platform for mingw");

$conf->data->set_p5( OSNAME => 'MSWin32' );
is( $step->_get_platform( $conf, $verbose ), q{win32},
    "Got expected platform for MSWin32");

# re-set to original values
$conf->data->set_p5( OSNAME => $platform_orig );
$conf->data->set_p5( archname => $archname_orig );

$conf->data->set_p5( archname => 'ia64-bar' );
is( $step->_get_platform( $conf, $verbose ), q{ia64},
    "Got expected platform for ia64");

$conf->data->set_p5( archname => 'foo-bar' );
$conf->data->set_p5( OSNAME => 'foo' );
{
    $verbose = 1;
    my ($stdout, $stderr, $rv);
    my $expected = q{generic};
    capture(
        sub { $rv = $step->_get_platform( $conf, $verbose ) },
        \$stdout,
        \$stderr,
    );
    is( $rv, $expected, "Got expected platform for foo");
    like( $stdout, qr/platform='$expected'/, "Got expected verbose output");
}

# re-set to original values
$conf->data->set_p5( archname => $archname_orig );
$conf->data->set_p5( OSNAME => $platform_orig );

########## _get_generated() ##########

my $TEMP_generated_orig = $conf->data->get('TEMP_generated');
{
    $verbose = 1;
    my ($stdout, $stderr, $rv);
    my $expected = q{foo};
    $conf->data->set( TEMP_generated => $expected );
    capture(
        sub { $rv = $step->_get_generated( $conf, $verbose ) },
        \$stdout,
        \$stderr,
    );
    is( $rv, $expected, "Got expected generated");
    like( $stdout, qr/\($expected\)/, "Got expected verbose output");
}
$conf->data->set( TEMP_generated => undef );
$verbose = 0;
is( $step->_get_generated( $conf, $verbose ), q{},
    "Got expected generated");

# re-set to original values
$conf->data->set( TEMP_generated => $TEMP_generated_orig );

########## _handle_asm() ##########

my $platform_asm_orig = $conf->data->get('platform_asm');
my $cwd = cwd();
{
    my $tdir = tempdir( CLEANUP => 1 );
    chdir $tdir or croak "Unable to change to temporary directory";
    $conf->data->set( platform_asm => 1 );
    my $platform = 'aix';
    mkpath( 'src', 0, 755 ) or croak "Unable to make testing directory";
    my $asmfile = File::Spec->catfile( 'src', 'platform_asm.s' );
    open my $FH, '>', $asmfile or croak "Unable to open handle for writing";
    print $FH "Hello asm\n";
    close $FH or croak "Unable to close handle after writing";
    $step->_handle_asm($conf, $platform);
    my $text = _slurp( $asmfile );
    like($text, qr/Hello asm/s, "File unchanged, as expected");

    chdir $cwd or croak "Unable to change back to starting directory";
}
# re-set to original values
$conf->data->set( platform_asm => $platform_asm_orig );

{
    my $tdir = tempdir( CLEANUP => 1 );
    chdir $tdir or croak "Unable to change to temporary directory";
    $conf->data->set( platform_asm => 1 );
    my $platform = 'aix';

    mkpath( 'src', 0, 755 ) or croak "Unable to make testing directory";

    my $asmfile = File::Spec->catfile( 'src', 'platform_asm.s' );
    open my $FH, '>', $asmfile or croak "Unable to open handle for writing";
    print $FH "Hello asm\n";
    close $FH or croak "Unable to close handle after writing";

    my $path = File::Spec->catdir( 'config', 'gen', 'platform', $platform );
    mkpath( $path, 0, 755 ) or croak "Unable to make testing directory";

    my $configfile = File::Spec->catfile( $path, 'asm.s' );
    open my $FH2, '>', $configfile or croak "Unable to open handle for writing";
    print $FH2 "Goodbye world\n";
    close $FH2 or croak "Unable to close handle after writing";

    $step->_handle_asm($conf, $platform);

    my $text = _slurp( $asmfile );
    like($text, qr/Goodbye world/s, "File changed, as expected");

    chdir $cwd or croak "Unable to change back to starting directory";
}
# re-set to original values
$conf->data->set( platform_asm => $platform_asm_orig );

########## _handle_begin_c() ##########
{
    my $tdir = tempdir( CLEANUP => 1 );
    chdir $tdir or croak "Unable to change to temporary directory";
    my $platform = 'darwin';

    my $path = File::Spec->catdir( 'config', 'gen', 'platform', $platform );
    mkpath( $path, 0, 755 ) or croak "Unable to make testing directory";
    copy qq{$cwd/config/gen/platform/$platform/begin.c},
        qq{$path/begin.c}
            or croak "Unable to copy file for testing";

    mkpath( 'src', 0, 755 ) or croak "Unable to make testing directory";
    my $plat_c = q{src/platform.c};
    open my $PLATFORM_C, '>', $plat_c
        or croak "Unable to open handle for writing";
    $step->_handle_begin_c($platform, $PLATFORM_C);
    close $PLATFORM_C or croak "Unable to close handle after writing";

    my $text = _slurp( $plat_c );
    like($text, qr/#undef environ.*#undef bool/s,
        "Got expected text in header file");
    unlike($text, qr/Local variables/s, "Coda stripped, as desired");

    chdir $cwd or croak "Unable to change back to starting directory";
}

pass("Completed all tests in $0");

################### DOCUMENTATION ###################

=head1 NAME

gen_platform-01.t - test gen::platform

=head1 SYNOPSIS

    % prove t/steps/gen_platform-01.t

=head1 DESCRIPTION

The files in this directory test functionality used by F<Configure.pl>.

The tests in this file test gen::platform.

=head1 AUTHOR

James E Keenan

=head1 SEE ALSO

config::gen::platform, F<Configure.pl>.

=cut

# Local Variables:
#   mode: cperl
#   cperl-indent-level: 4
#   fill-column: 100
# End:
# vim: expandtab shiftwidth=4:
