package Bencher::Scenario::StringFunctions::Trim;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2021-07-31'; # DATE
our $DIST = 'Bencher-Scenarios-StringFunctions'; # DIST
our $VERSION = '0.005'; # VERSION

use strict;
use warnings;

our $scenario = {
    summary => "Benchmark string trimming (removing whitespace at the start and end of string)",
    participants => [
        #{fcall_template=>'String::Trim::trim(<str>)'}, # currently disabled, see https://github.com/doherty/String-Trim/issues/7
        {fcall_template=>'String::Trim::More::trim(<str>)'},
        {fcall_template=>'String::Trim::NonRegex::trim(<str>)'},
        {fcall_template=>'String::Trim::Regex::trim(<str>)'},
        {fcall_template=>'String::Util::trim(<str>)'},
        {fcall_template=>'Text::Minify::XS::minify(<str>)'},
    ],
    datasets => [
        {name=>'empty'        , args=>{str=>''}},
        {name=>'len10ws1'     , args=>{str=>' '.('x' x   10).' '}},
        {name=>'len100ws1'    , args=>{str=>' '.('x' x  100).' '}},
        {name=>'len100ws10'   , args=>{str=>(' ' x   10).('x' x  100).(' ' x 10)}},
        {name=>'len100ws100'  , args=>{str=>(' ' x  100).('x' x  100).(' ' x 100)}},
        {name=>'len1000ws1'   , args=>{str=>' '.('x' x 1000).' '}},
        {name=>'len1000ws10'  , args=>{str=>(' ' x   10).('x' x 1000).(' ' x 10)}},
        {name=>'len1000ws100' , args=>{str=>(' ' x  100).('x' x 1000).(' ' x 100)}},
        {name=>'len1000ws1000', args=>{str=>(' ' x 1000).('x' x 1000).(' ' x 1000)}},
    ],
};

1;
# ABSTRACT: Benchmark string trimming (removing whitespace at the start and end of string)

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::StringFunctions::Trim - Benchmark string trimming (removing whitespace at the start and end of string)

=head1 VERSION

This document describes version 0.005 of Bencher::Scenario::StringFunctions::Trim (from Perl distribution Bencher-Scenarios-StringFunctions), released on 2021-07-31.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m StringFunctions::Trim

To run module startup overhead benchmark:

 % bencher --module-startup -m StringFunctions::Trim

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<String::Trim::More> 0.03

L<String::Trim::NonRegex> 0.002

L<String::Trim::Regex> 20210604

L<String::Util> 1.32

L<Text::Minify::XS> v0.4.2

=head1 BENCHMARK PARTICIPANTS

=over

=item * String::Trim::More::trim (perl_code)

Function call template:

 String::Trim::More::trim(<str>)



=item * String::Trim::NonRegex::trim (perl_code)

Function call template:

 String::Trim::NonRegex::trim(<str>)



=item * String::Trim::Regex::trim (perl_code)

Function call template:

 String::Trim::Regex::trim(<str>)



=item * String::Util::trim (perl_code)

Function call template:

 String::Util::trim(<str>)



=item * Text::Minify::XS::minify (perl_code)

Function call template:

 Text::Minify::XS::minify(<str>)



=back

=head1 BENCHMARK DATASETS

=over

=item * empty

=item * len10ws1

=item * len100ws1

=item * len100ws10

=item * len100ws100

=item * len1000ws1

=item * len1000ws10

=item * len1000ws100

=item * len1000ws1000

=back

=head1 BENCHMARK SAMPLE RESULTS

Run on: perl: I<< v5.34.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux LinuxMint version 19 >>, OS kernel: I<< Linux version 5.3.0-68-generic >>.

Benchmark command (default options):

 % bencher -m StringFunctions::Trim

Result formatted as table (split, part 1 of 9):

 #table1#
 {dataset=>"empty"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Util::trim           |   1770000 |     565   |                 0.00% |               432.57% | 1.7e-10 |      31 |
 | String::Trim::NonRegex::trim |   1950000 |     514   |                 9.87% |               384.71% |   2e-10 |      21 |
 | String::Trim::More::trim     |   7260000 |     138   |               309.61% |                30.02% | 4.6e-11 |      26 |
 | String::Trim::Regex::trim    |   8282000 |     120.7 |               367.56% |                13.90% | 5.8e-12 |      20 |
 | Text::Minify::XS::minify     |   9434000 |     106   |               432.57% |                 0.00% | 5.8e-12 |      20 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

              Rate  SU:t  STN:t  STM:t  STR:t  TMX:m 
  SU:t   1770000/s    --    -9%   -75%   -78%   -81% 
  STN:t  1950000/s    9%     --   -73%   -76%   -79% 
  STM:t  7260000/s  309%   272%     --   -12%   -23% 
  STR:t  8282000/s  368%   325%    14%     --   -12% 
  TMX:m  9434000/s  433%   384%    30%    13%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify

The above result presented as chart:

=begin html

<img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAtAAAAH4CAMAAABUnipoAAAJJmlDQ1BpY2MAAEiJlZVnUJNZF8fv8zzphUASQodQQ5EqJYCUEFoo0quoQOidUEVsiLgCK4qINEWQRQEXXJUia0UUC4uCAhZ0gywCyrpxFVFBWXDfGZ33HT+8/5l7z2/+c+bec8/5cAEgiINlwct7YlK6wNvJjhkYFMwE3yiMn5bC8fR0A9/VuxEArcR7ut/P+a4IEZFp/OW4uLxy+SmCdACg7GXWzEpPWeGjy0wPj//CZ1dYsFzgMt9Y4eh/eexLzr8s+pLj681dfhUKABwp+hsO/4b/c++KVDiC9NioyGymT3JUelaYIJKZttIJHpfL9BQkR8UmRH5T8P+V/B2lR2anr0RucsomQWx0TDrzfw41MjA0BF9n8cbrS48hRv9/z2dFX73kegDYcwAg+7564ZUAdO4CQPrRV09tua+UfAA67vAzBJn/eqiVDQ0IgALoQAYoAlWgCXSBETADlsAWOAAX4AF8QRDYAPggBiQCAcgCuWAHKABFYB84CKpALWgATaAVnAad4Dy4Aq6D2+AuGAaPgRBMgpdABN6BBQiCsBAZokEykBKkDulARhAbsoYcIDfIGwqCQqFoKAnKgHKhnVARVApVQXVQE/QLdA66At2EBqGH0Dg0A/0NfYQRmATTYQVYA9aH2TAHdoV94fVwNJwK58D58F64Aq6HT8Id8BX4NjwMC+GX8BwCECLCQJQRXYSNcBEPJBiJQgTIVqQQKUfqkVakG+lD7iFCZBb5gMKgaCgmShdliXJG+aH4qFTUVlQxqgp1AtWB6kXdQ42jRKjPaDJaHq2DtkDz0IHoaHQWugBdjm5Et6OvoYfRk+h3GAyGgWFhzDDOmCBMHGYzphhzGNOGuYwZxExg5rBYrAxWB2uF9cCGYdOxBdhK7EnsJewQdhL7HkfEKeGMcI64YFwSLg9XjmvGXcQN4aZwC3hxvDreAu+Bj8BvwpfgG/Dd+Dv4SfwCQYLAIlgRfAlxhB2ECkIr4RphjPCGSCSqEM2JXsRY4nZiBfEU8QZxnPiBRCVpk7ikEFIGaS/pOOky6SHpDZlM1iDbkoPJ6eS95CbyVfJT8nsxmpieGE8sQmybWLVYh9iQ2CsKnqJO4VA2UHIo5ZQzlDuUWXG8uIY4VzxMfKt4tfg58VHxOQmahKGEh0SiRLFEs8RNiWkqlqpBdaBGUPOpx6hXqRM0hKZK49L4tJ20Bto12iQdQ2fRefQ4ehH9Z/oAXSRJlTSW9JfMlqyWvCApZCAMDQaPkcAoYZxmjDA+SilIcaQipfZItUoNSc1Ly0nbSkdKF0q3SQ9Lf5RhyjjIxMvsl+mUeSKLktWW9ZLNkj0ie012Vo4uZynHlyuUOy33SB6W15b3lt8sf0y+X35OQVHBSSFFoVLhqsKsIkPRVjFOsUzxouKMEk3JWilWqUzpktILpiSTw0xgVjB7mSJleWVn5QzlOuUB5QUVloqfSp5Km8oTVYIqWzVKtUy1R1WkpqTmrpar1qL2SB2vzlaPUT+k3qc+r8HSCNDYrdGpMc2SZvFYOawW1pgmWdNGM1WzXvO+FkaLrRWvdVjrrjasbaIdo12tfUcH1jHVidU5rDO4Cr3KfFXSqvpVo7okXY5upm6L7rgeQ89NL0+vU++Vvpp+sP5+/T79zwYmBgkGDQaPDamGLoZ5ht2GfxtpG/GNqo3uryavdly9bXXX6tfGOsaRxkeMH5jQTNxNdpv0mHwyNTMVmLaazpipmYWa1ZiNsulsT3Yx+4Y52tzOfJv5efMPFqYW6RanLf6y1LWMt2y2nF7DWhO5pmHNhJWKVZhVnZXQmmkdan3UWmijbBNmU2/zzFbVNsK20XaKo8WJ45zkvLIzsBPYtdvNcy24W7iX7RF7J/tC+wEHqoOfQ5XDU0cVx2jHFkeRk4nTZqfLzmhnV+f9zqM8BR6f18QTuZi5bHHpdSW5+rhWuT5z03YTuHW7w+4u7gfcx9aqr01a2+kBPHgeBzyeeLI8Uz1/9cJ4eXpVez33NvTO9e7zofls9Gn2eedr51vi+9hP0y/Dr8ef4h/i3+Q/H2AfUBogDNQP3BJ4O0g2KDaoKxgb7B/cGDy3zmHdwXWTISYhBSEj61nrs9ff3CC7IWHDhY2UjWEbz4SiQwNCm0MXwzzC6sPmwnnhNeEiPpd/iP8ywjaiLGIm0iqyNHIqyiqqNGo62ir6QPRMjE1MecxsLDe2KvZ1nHNcbdx8vEf88filhICEtkRcYmjiuSRqUnxSb7JicnbyYIpOSkGKMNUi9WCqSOAqaEyD0tandaXTlz/F/gzNjF0Z45nWmdWZ77P8s85kS2QnZfdv0t60Z9NUjmPOT5tRm/mbe3KVc3fkjm/hbKnbCm0N39qzTXVb/rbJ7U7bT+wg7Ijf8VueQV5p3tudATu78xXyt+dP7HLa1VIgViAoGN1tubv2B9QPsT8M7Fm9p3LP58KIwltFBkXlRYvF/OJbPxr+WPHj0t6ovQMlpiVH9mH2Je0b2W+z/0SpRGlO6cQB9wMdZcyywrK3BzcevFluXF57iHAo45Cwwq2iq1Ktcl/lYlVM1XC1XXVbjXzNnpr5wxGHh47YHmmtVagtqv14NPbogzqnuo56jfryY5hjmceeN/g39P3E/qmpUbaxqPHT8aTjwhPeJ3qbzJqamuWbS1rgloyWmZMhJ+/+bP9zV6tua10bo63oFDiVcerFL6G/jJx2Pd1zhn2m9az62Zp2WnthB9SxqUPUGdMp7ArqGjzncq6n27K7/Ve9X4+fVz5ffUHyQslFwsX8i0uXci7NXU65PHsl+spEz8aex1cDr97v9eoduOZ67cZ1x+tX+zh9l25Y3Th/0+LmuVvsW523TW939Jv0t/9m8lv7gOlAxx2zO113ze92D64ZvDhkM3Tlnv296/d5928Prx0eHPEbeTAaMip8EPFg+mHCw9ePMh8tPN4+hh4rfCL+pPyp/NP637V+bxOaCi+M24/3P/N59niCP/Hyj7Q/Fifzn5Ofl08pTTVNG02fn3Gcufti3YvJlykvF2YL/pT4s+aV5quzf9n+1S8KFE2+Frxe+rv4jcyb42+N3/bMec49fZf4bmG+8L3M+xMf2B/6PgZ8nFrIWsQuVnzS+tT92fXz2FLi0tI/QiyQvpNzTVQAAAAgY0hSTQAAeiYAAICEAAD6AAAAgOgAAHUwAADqYAAAOpgAABdwnLpRPAAAALRQTFRF////AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAlQDVlQDVAAAAlADVlADUAAAAlADUlADUlADUlQDWlADUlQDVlADVlADUeQCtgwC7ZQCRdACnJwA3PgBZLwBEOQBSMQBHCwAQFgAfEQAYAAAAAAAAAAAAlADURQBj////G6yY+wAAADh0Uk5TABFEZiK7Vcwzd4jdme6qjqPVzsfSP+z89vH0dVCn8Hrf5CJEx1xm7LeIp77H1fL49fLr8fb0WzA03be2AAAAAWJLR0QAiAUdSAAAAAlwSFlzAAAASAAAAEgARslrPgAAAAd0SU1FB+UHHwc4MAQOIZoAABjkSURBVHja7d0Le6s4foBxAQKBhGhnZ5rubHdmL73fL9sL7ff/YEXCTsCTnTh/WY6M39/z5ExwMlF8zhsswI6UAgAAAAAAAAAAAAAAAAAAAJBfVZ/eqavtzbXgSwFfpdHn9+r59M68bVjPn/t6wJdqX+t9L2jdGYLGA+nNsOyiG2vrJWgd/rMGXVvbL+82LUHjkTRurFXrrPVNPftxnG0MevB2nJvwCTVB45GEKUe77KTtVM+DUsOsl6D1vOyeGx8+TtB4KOscuu9Mu6a77J7nunH1IlRN0HgsIWg7t2N7CtqHoK1vA4LGw1mC7nyYcixBV0pVcQ/dOXU+L03QeChttxwYLvWGKce4dO3CrKPyzfouQePBTK6pjGvd6P/EGOd8H6fRjTcmvEvQeDBVvcw36rpS5/+eb95fAAcAAAAAAADk7Lxov/q7AG7INl/9HQC307ODxpG0Ov1rAPdUv7vZx2u19fjV3x3wORcvRF43tZnDs8XUxMvu8VAuXoh83mynSjurKv/V3x/wKacXItfhXMZQnzfjq+AGo7rpq78/4JPiU3Qr36nOV+fNtz82/vSb6BffApl8FxP77s+Sg1ad0/7tRULNGvT+Obwvf/7L4PuXvLIPwDDlDvOrmNj8F+lBq8nbt81hDXp/yu7l23wPExv2PsehDFPuMDcJull/E8rPTDkImmEeJ+jKW1+9buqwc27c/hMJmmEeJ+jWqmnabca3LYJmmIcJenCV0r47b6reG2cuXtdJ0AzzCEG/r6p/8r3fKej6Ps8eYZhyh8kR9DvuFDSeHkHjUAgah0LQOBSCxqEQNA6FoHEoBI1DIWgcCkHjUAgah0LQOBSCxqEQNA6FoHEoBI1DIWgcCkHjUAgah0LQuI9f//ChX99gGILGffzwvx/64QbDEDTug6BxKASNQyFoHApB41AIGodC0DgUgsahEDQOhaBxKASNQ3n4oDtn2tff40/QT+/Rg65cpabhvEXQT+/Rg27G7RZBP71HD9pObh6ZcuDsIYLuLxa7qt9utmHK8bq4G0E/vQcIunPzPG2T1nFZN23meVR2ibluzx8h6KdXftCVH1Rl3lbY1J1ZF96cKu1sv8Rs2UPjrPygh7D8cedVHRb6HmrVtGZdGrlfNo0azWSYQ+Os/KCtUXHx2Mp3S9eVuly8vt8sv/nyYxs0X/t3ii+UPWgbE0sIug+74nGul7m09jHe2HKzBr0/XGQP/fTK30Or0bemDVVPfp0sr0t/r0HvF8Il6Kf3AEGr3nannfI6l9hPObYI+umVH3QVphmNCac7bJxCrxnrsHNu3P5zCfrpPUDQc6cqt0w2WqumKdyy7pfb9aYdgn565Qe9zDRaP4bTd5XSvlPnoHtvnLm4hEjQT+8Bgla67t+7uarry5sI+uk9QtDXI+inR9A4FILGoRA0DoWgcSgEjUMhaBwKQeNQCBqHQtA4FILGoRA0DoWgcSgEjUMhaBwKQeNQCBqHQtA4FILGoRA0DoWgcSgEjUMhaBwKQeNQCBqHQtA4FILGffzmtx/6zQ2GIWjcx51KI2jcB0ELEHS5CFqAoMtF0NfpnGlZp/ABEPRVqrDW93DeIuhyEfRVmnG7RdDlIuir2MnNI1OOB0DQb/r98prqtLZKv4Rsw5SDxesfAEGf9W6e2+1yVzqugqXNPI/KLjHX7fkjBF0ugj5zVlXmbaqsO7OuUzhV2tl+idmyh34ABP36/y4zDNuqOqyLPNSqac26kmy/bBo1mskwh34ABH3mB6WmUVW+U11cG3m/1ne/Wa3w5Zs60MKRkNFRgu5jYilB19755dBPdU77GO9pJfv43n4p2Zff2aAWjILMjhJ0ExNLWevbTHUX59CTXyfLMehhDXq/M2bKUa6jBL1KCLpxyx99KLeZm3jLfsqxRdDlIugTa5Y/quXIsPI2TqHXjHVM3O0/l6DLRdAnfTidYb1SrVXTFG5Z98vLZnzbIuhyEfRZMxvnezUsB4bad+ocdO+NM/tjQoIuGEG/0vW7py2qn95M0OUiaAGCLhdBCxB0uQhagKDLRdACBF0ughYg6HIRtABBl4ugBQi6XAQtQNDlImgBgi4XQQsQdLkIWoCgy0XQAgRdLoIWIOhyEbQAQZeLoAUIulwELUDQ5SJoAYIuF0ELEHS5CFqAoMtF0AIEXS6CFiDochG0AEGXi6AFCLpcBC1A0OUiaAGCLhdBCxB0uQhagKDLRdACBC3w+7/80O9vMAxBCxC0wBUJ/NV9hiHoSwQtQNACBF0ughbIF3TnTMs6hSkIWiBb0FVY63s4bxG0AEELZAu6GbdbBC1A0ALZgraTm0emHCkIWuC2QZ/WVumXkG2YcrB4fQqCFkgIup6jzQJBOq6Cpc08j8ouMdft+SMELUDQAilLI4e1wgf/Oq3QnVnXKZwq7Wy/xGzZQ6cgaIHUKYfpVB3WRR5q1bRmXUm2XzaNGs1kmEOnIGiBxKCHadlT+051cUe9X+u730xGCFqAoAXSgq68VuESivYx3thyswa9X0r25cc2aG7wHT8Pgv4UGxNLC9quJ5snv06WY9DDGrTefSJ7aAGCFkgKulp3zMtOed317qccWwQtQNACSUE3Lv6n8nY911GvB4X69SOvCFqAoAWSgp7WGUdr1TSFd9b98rIZ37YIWoCgBZKC9nGmMbhKad+pc9C9N87sjwkJWoKgBXI8l6Oq68ubCFqAoAV4gn+5CFqAoMtF0AIEXS6CFiDochG0AEGXi6AFCLpcBC1A0OUiaAGCLhdBCxB0uQhagKDLRdACBF0ughYg6HIRtABBl4ugBQi6XAQtQNDlImgBgi4XQQsQdLkIWoCgy0XQAgRdLoIWIOhyEbQAQZeLoAUIulwELUDQ5SJoAYIuF0ELEHS5CFqAoMtF0AIEXS6CFiDochG0AEGXi6AFCLpcBC1A0OUiaAGCLhdBC+QLunOmZeHNFAQtkC3oKixeP5y3CFqAoAWyBd2M2y2CFiBogWxB28nNI1OOFAQtkLg0cr/fPi0W1C8h2zDleF3cjaAFCFogJehqmmezXQJZx2XdtJnnUdkl5ro9f4SgBQhaICXo0VTVuuJmpDuzLrw5VdrZfonZsodOQdACCUFX8zLh0FbVYfnNoVZNa9alkZebB7PkPhnm0CkIWiAh6HpWfV2Fpb471cXFvveL1/eb5TdffmeDWjjUcyLoT2liYglBd3PrnO/DJRTtY6qx5WYNer828ss3daAFwzwvgv6UPiaWELSdlymy9ct7k18nyzHoYQ163y5TDgGCFkibcoSJdB12ys3bLW9/bBC0AEELJATdr0H3yyTaxin0mrEOO+fG7T+XoAUIWiDltJ0blBqXclur1rN363552YxvWwQtQNACKUH33oSDwsFVSvtOnYMON5v9MSFBSxC0QNKl76qur7yZoAUIWoAn+JeLoAUIulwELUDQ5SJoAYIuF0ELXBl031/1aX8UQQsQtMBVQTd+bmuX0jRBCxC0wDVB93NTt9XpaqAMQQsQtMA1QdsxvvbEJDz5k6AFCFrgqqAtQX8Fgha4Juja90vQDVOOOyNogasOCofZeeebhGEIWoCgBa47bacb2yXsnwlahKAFrglar5PnJuEFVAQtQNACHwet62EML9bqHAeF90XQAh8H3bTGtcHEQeF9EbTAVRdWUg4HVwQtQNACn3hyEnPoOyNogeueyzGFKYdnDn1fBC1w3YUVa1prxo8/848iaAGCFrjy0nc3xl/JL0bQAgQtcGXQ4XeJtkw57ougBa4JunFazVpxHvrOCFrgqoPCtlXWO5MwDEELELTA1aftuiblyRwELUDQAled5eDCypcgaIFrgh5SJhsrghYgaIGrphyjjb9LOmEYghYgaIGrphzzKmEYghYgaAF+0Uy5CFqAoMtF0AIEXS6CFsgXdOdMyzqFKQhaIFvQ4blM03DeImgBghbIFnSze7YpQQsQtEC2oO3k5pEpRwqCFrht0KdrL/0Ssg1TDhavT0HQAilB23C1pd3coOO1F23meQxPoo6/EW9F0AIELZAS9BR+X8fbb43WnVnXKZwq7Wx4SYBlD52CoAVSgm5PCyKH/wx1+AUe60qyfXw+02gmwxw6BUELpAQ9N9Yus+bKd6qLv5p0v9Z3v3k2E0ELELRAUtDOjmHZ+s7p9VccxJabNej9ywFefoy/fCn9idXPhKA/xcbEEoLWdol28Mt7k18nyzHoYQ16/1tp2EMLELRA6mm7aq7DTvk0m95NObYIWoCgBRKCjgeD4RCw8qcFher1oFCHF4rvP5egBQhaICXocDpjNEq1Vk3Tekv4c9mMb1sELUDQAmkXVlrnejW4SmnfqXPQvTfOXLxEnKAFCFogaQ6t33+dYfXTmwlagKAFeIJ/uQhagKDLRdACBF0ughYg6HIRtABBl4ugBQi6XAQtQNDlImgBgi4XQQsQdLkIWoCgy0XQAgRdLoIWIOhyEbQAQZeLoAUIulwELUDQ5SJoAYIuF0ELEHS5CFqAoMtF0AIEXS6CFiDochG0AEGXi6AFCLpcBC1A0OUiaAGCLhdBCxB0uQhagKDLRdACBF0ughYg6HIRtABBl4ugBQi6XAQtQNDlImgBgi4XQQvkC7pzpmXhzRQELZAt6CosXj+ctwhagKAFsgXdjNstghYgaIFsQdvJzSNTjhQELZAadL/bOi0W1C8h2zDleF3cjaAFCFogMWjbbrd0XNZNm3kelV1irl8/StACBC2QFnQ9b4LWnVkX3pwq7Wy/fMSyh05B0AJJQVd+ak9LJA+1alqzLo28TEMGo0YzGebQKQhaICnoyYYpR+U71cXFvveL1/eb5TdffmyD5gbf8fMg6E+xMbGUoBuzzqE7p32MN7bcrEHv10ZmDy1A0AIJQWunTweFk18nyzHoYQ1a7z6XoAUIWiAhaGuWGYezOuyU17nEfsqxRdACBC2QEHRtT0FX3sYp9JqxDjvnxu0/l6AFCFrgFuehW6umKWyt++VlM75tEbQAQQvcIOjBVUr7Tp2D7r1xZn9MSNASBC2Q47kcVV1f3kTQAgQtwBP8y0XQAgRdLoIWIOhyEbQAQZeLoAUIulwELUDQ5SJoAYIuF0ELEHS5CFqAoMtF0AIEXS6CFiDochG0AEGXi6AFCLpcBC1A0OUiaAGCLhdBCxB0uQhagKDLRdACBF0ughYg6HIRtABBl4ugBQi6XAQtQNDlImgBgi4XQQsQdLkIWoCgy0XQAgRdLoIWIOhyEbQAQZeLoAUIulwELUDQ5SJoAYIuF0EL5Au6c6ZlncIUBC2QLegqrPU9nLcIWoCgBbIF3YzbLYIWIGiBbEHbyc0jU44UBC2QuHi9vthe/9MvIdsw5WDx+hQELZC0NLKf53a73JWOq2BpM8+jskvMdXv+CEELELRAQtCVb1Rl3qbKujPrOoVTpZ3tl5gte+gUBC2QspJsyNe2qg7rIg+1alqzriTbL5tGjWYyzKFTELRA6kHhNC176k51cW3k/Vrf/Wa1wpdv6kALR3lOBP0pfUwsLejWLYd+qnPax3hjy80a9H4p2Zff2aAWjPG8CPpTmphY4lmOxoVp8uTXyXIMeliD3u+MmXIIELRA6pSjW3fKTdzYTzm2CFqAoAUSgg4L18dyK2/jFHrNWIedc+P2n0vQAgQtkHSWo1dqXMptbTg2VOf98rIZ37YIWoCgBVKmHOPcOt+rYTkw1L5T56B7b5zZHxMStARBCyTNoXX97mmL6qc3E7QAQQvwBP9yEbQAQZeLoAUIulwELUDQ5SJoAYIuF0ELEHS5CFqAoMtF0AIEXS6CFiDochG0AEGXi6AFCLpcBC1A0OUiaAGCLhdBCxB0uQhagKDLRdACBF0ughYg6HIRtABBl4ugBQi6XAQtQNDlImgBgi4XQQsQdLkIWoCgy0XQAgRdLoIWIOhyEbQAQZeLoAUIulwELUDQ5SJoAYIuF0ELEHS5CFqAoMtF0AIEXS6CFsgXdOdMy8KbKQhaIFvQVVi8fjhvEbQAQQtkC7oZt1sELUDQAtmCtpObR6YcKQhaIC3o/mLx7tNiQf0Ssg1TjtfF3QhagKAFUoLu3Ty7fnODjsu6aTPPo7JLzHV7/ghBCxC0QErQflTV+LZkrO7MuvDmVGln+yVmyx46BUELJK0kW4Wdcl+Hhb6HWjWtWZdGXnbag1GjmQxz6BQELZAQdBVmzPWsK9+pLi72vV+8vt8sv/nyYxs0N/iOnwdBf4qNiSWe5dBmDJdQtI/xxpabNej92sjsoQUIWiAp6MrOcZY8+XWyHIMe1qD35z+OFfSvf/jQX99gGIIWSDrLYdr1HEczr3OJ/ZRj61hBl1MaQV9KCdqdLgZW3sYp9JqxDjvnxu0/laAfepjnCLqb60Cp1qppCres++VlM75tEfRDD/McQds5UoOrlPadOgfde+PM/piQoB97mOcI+o+q6vryJoJ+6GGePOh3EPRDD0PQlwj6oYch6EsE/dDDEPSlOwX9N3/7ob+7wTDllEbQl44V9N9//Jf22xsMU05pBH2JoAXKKY2gLxG0QDmlEfQlghYopzSCvkTQAuWURtCXCFqgnNII+hJBC5RTGkFfImiBckoj6EsELVBOaQR9iaAFyimNoC8RtEA5pRH0JYIWKKc0gr5E0ALllEbQlwhaoJzSCPoSQQuUUxpBXyJogXJKI+hLBC1QTmkEfYmgBcopjaAvEbRAOaUR9CWCFiinNIK+RNAC5ZRG0JcIWqCc0gj6EkELlFMaQV8iaIFySiPoSwQtUE5pBH2JoAXKKY2gLxG0QDmlEfSlfEF3zrT3XqeQoPMMQ9BKVWGt7+G8RdAPPQxBK9WM2607Bf0P9wn6H+9TWjnD3KK0Ow2TLWg7uXm895Tjn+4T9D/fp7RyhrlFaXcaJjHo+t3NfgnZhinHvRevJ+g8wzxN0Hp+Z1ObeR6VXWKu2/NHCPqhh3mSoHVn5nc226nSzvZLzJY99DGGeZKgmzYWXId1kYf6vKnnftk0ajSTYQ59jGGeJOjTSpuV71QX10ber/XdbybYL99/E/zi27z+5f8+9K83GObfPh7m3w81zH88wjDfxcRuELTqnPb162azBr1fSvbbX0bfv+T1h//80H/dZ5g/MMy9h/lVTOxX/50etJq8fdsc1qB1ytcFvsQp6GZu3jbf/gAezNpt5W2cQq+bOuycG/fV3xrweWvQrVXTtNuMb/d2p0kOwxzYOml2ldK+O2+q3htnqrQvLDC06V+DYfCOqq7Tv8jnR/XtmP5VGCYLff8d3ANbHzZr7/M+ft5pGHW8Yap29v0d7s1BxFOE1fg/2mSduPdzv4yiq7zD9GZQdximarp7DHMyWj0yt7le24bXE3S5T35PbRwl7zDd7Kvsw1TWz/M97k3QeO/D3KbJO8yR9HPY3SzvrOdactHLMHEqmHMYPZtRZR6mc6Ze/tKa/PdmuSe96YZ4Itczjf5Yu/7Yj6ez3trXWaZqbRuPcsNTvDMOs35xP8x93nsz+fjKuPOZ1VzDrHdnDI8C8cDzHnObh3dKrDpdeld2djn2A8sxTUi6chmHqdr1uKltXqebme7NEI8D43wj4zDhqcPLJF2HqYae63D8yS76Q+eSh/CXVdlK3f58oZ66MNmw85J0s6QQRrn9MMso7RyfQGBtbCDPMPHeKLfsMGs3zybbMGp9AAg/NDY8etrwM8o1nJ+jRx9iHk6HNGHfGfcDN7YcPk06PBIsAy5Jt1O+UfQ8+PVVPuFxJ8cw53tTz014OpkObzmGieLzL6vTw1rlu+QveGy9H+su/C2Z9ZAm7gJu/5DWORf/weMjwZK0W/79s41iXb8cEFbz+uOZ4fH5fG/UtL58uclyb1Q/hd3+609KfKo8Z6F/XhWPa9bXFMS/uCzPHdGtH0//Lusx+pL0kPYlf26U5U5Vpq1cf3q5RL57o0/PjhzHHNPa5XHA2+pt13+fa5GPbfPs1PA6r2rMcsAxLnua9nSEZtZ/lQzDvI2yHAlUo4lXVjLfG5vrAC1eF1LV4PzpXM0ySe+ZPH8o7GJqa+bl8Kkyvp3bW//7nKbobzuzOttkczNKOLM1zrd/sPnJvXk9LXRr8bpQYOd51CrPgc0RjfNykG47G/YDzXDzGdrrFP3t5HOOa7evoZ1GqcPduf10851708x5dtHrdaFwV1zdzmOf58HmiPo6/k3lOVv/NkV/25np2z9wbkI7jTLleMbDe/cmwzm0/XWh8LzUJWmud3+Sy/I3tnsBWZPryQ7b0E6jZPipudO9ubwuFE87cer5E2x4OJvyvJBgM0VXr2cGb27/ustco9zl3vzkutApaFxv9OPob34weP7imyl6tmu2+9AyXhnOfW/euS6U5ZDj4JrJ5juAzjpFP9uFllPee/PudSHLbKNEeabor+7yY5P53tzruhDS5JyiX8r8Y5P33tzruhDSZJ2iv7rXj03me5P5uhBuIesU/ew+Pzb5703W60J4IHf5sckv53Uh4P4yXhcCvkC+60LAF+AVgwAAsf8HSzixVj9O1c0AAAAldEVYdGRhdGU6Y3JlYXRlADIwMjEtMDctMzFUMDc6NTY6NDgrMDc6MDCHMOEVAAAAJXRFWHRkYXRlOm1vZGlmeQAyMDIxLTA3LTMxVDA3OjU2OjQ4KzA3OjAw9m1ZqQAAACF0RVh0cHM6SGlSZXNCb3VuZGluZ0JveAA1MDR4NzIwKzUwKzUw1uIiwwAAABN0RVh0cHM6TGV2ZWwAQWRvYmUtMi4wCjmTdA0AAAAASUVORK5CYII=" />

=end html


Result formatted as table (split, part 2 of 9):

 #table2#
 {dataset=>"len1000ws1"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::Regex::trim    |     19000 |   52      |                 0.00% |              6429.62% | 2.1e-07 |      21 |
 | String::Util::trim           |     26100 |   38.3    |                35.13% |              4731.99% | 1.1e-08 |      28 |
 | String::Trim::More::trim     |    479850 |    2.084  |              2384.07% |               162.86% | 5.6e-12 |      20 |
 | Text::Minify::XS::minify     |    514610 |    1.9432 |              2564.06% |               145.10% | 5.8e-12 |      20 |
 | String::Trim::NonRegex::trim |   1261000 |    0.7928 |              6429.62% |                 0.00% | 2.9e-11 |      20 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

              Rate  STR:t   SU:t  STM:t  TMX:m  STN:t 
  STR:t    19000/s     --   -26%   -95%   -96%   -98% 
  SU:t     26100/s    35%     --   -94%   -94%   -97% 
  STM:t   479850/s  2395%  1737%     --    -6%   -61% 
  TMX:m   514610/s  2575%  1870%     7%     --   -59% 
  STN:t  1261000/s  6459%  4730%   162%   145%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 3 of 9):

 #table3#
 {dataset=>"len1000ws10"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::Regex::trim    |     19700 |   50.7    |                 0.00% |              2452.07% | 2.3e-08 |      28 |
 | String::Util::trim           |     26100 |   38.3    |                32.38% |              1827.90% | 1.1e-08 |      31 |
 | String::Trim::NonRegex::trim |    367230 |    2.7231 |              1763.21% |                36.97% | 5.8e-12 |      20 |
 | String::Trim::More::trim     |    479870 |    2.0839 |              2334.70% |                 4.82% | 1.7e-11 |      20 |
 | Text::Minify::XS::minify     |    503010 |    1.9881 |              2452.07% |                 0.00% | 5.7e-12 |      20 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

             Rate  STR:t   SU:t  STN:t  STM:t  TMX:m 
  STR:t   19700/s     --   -24%   -94%   -95%   -96% 
  SU:t    26100/s    32%     --   -92%   -94%   -94% 
  STN:t  367230/s  1761%  1306%     --   -23%   -26% 
  STM:t  479870/s  2332%  1737%    30%     --    -4% 
  TMX:m  503010/s  2450%  1826%    36%     4%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 4 of 9):

 #table4#
 {dataset=>"len1000ws100"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::Regex::trim    |   19627.3 |   50.9494 |                 0.00% |              2230.43% | 5.8e-12 |      20 |
 | String::Util::trim           |   26000   |   38.4    |                32.70% |              1656.15% | 1.2e-08 |      23 |
 | String::Trim::NonRegex::trim |   46312.2 |   21.5926 |               135.96% |               887.65% | 5.8e-12 |      20 |
 | Text::Minify::XS::minify     |  442000   |    2.26   |              2152.59% |                 3.46% | 8.3e-10 |      20 |
 | String::Trim::More::trim     |  457000   |    2.19   |              2230.43% |                 0.00% |   7e-10 |      28 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

              Rate  STR:t   SU:t  STN:t  TMX:m  STM:t 
  STR:t  19627.3/s     --   -24%   -57%   -95%   -95% 
  SU:t     26000/s    32%     --   -43%   -94%   -94% 
  STN:t  46312.2/s   135%    77%     --   -89%   -89% 
  TMX:m   442000/s  2154%  1599%   855%     --    -3% 
  STM:t   457000/s  2226%  1653%   885%     3%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 5 of 9):

 #table5#
 {dataset=>"len1000ws1000"}
 +------------------------------+------------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s)  | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+------------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::NonRegex::trim |   4713.885 |  212.1392 |                 0.00% |              7207.02% | 5.7e-12 |      22 |
 | String::Trim::Regex::trim    |  19434.1   |   51.456  |               312.27% |              1672.37% | 5.8e-12 |      20 |
 | String::Util::trim           |  25200     |   39.6    |               435.21% |              1265.25% | 1.3e-08 |      20 |
 | Text::Minify::XS::minify     | 206960     |    4.8318 |              4290.51% |                66.43% | 5.6e-12 |      20 |
 | String::Trim::More::trim     | 344000     |    2.9    |              7207.02% |                 0.00% | 7.9e-10 |      22 |
 +------------------------------+------------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

               Rate  STN:t  STR:t   SU:t  TMX:m  STM:t 
  STN:t  4713.885/s     --   -75%   -81%   -97%   -98% 
  STR:t   19434.1/s   312%     --   -23%   -90%   -94% 
  SU:t      25200/s   435%    29%     --   -87%   -92% 
  TMX:m    206960/s  4290%   964%   719%     --   -39% 
  STM:t    344000/s  7215%  1674%  1265%    66%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 6 of 9):

 #table6#
 {dataset=>"len100ws1"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::Regex::trim    |    179120 |    5.5828 |                 0.00% |              1752.41% | 5.6e-12 |      20 |
 | String::Util::trim           |    223280 |    4.4786 |                24.65% |              1386.04% | 5.8e-12 |      30 |
 | String::Trim::More::trim     |   1220000 |    0.82   |               580.87% |               172.07% | 3.5e-10 |      29 |
 | String::Trim::NonRegex::trim |   1300000 |    0.79   |               607.26% |               161.91% | 1.1e-09 |      24 |
 | Text::Minify::XS::minify     |   3318000 |    0.3014 |              1752.41% |                 0.00% | 5.8e-12 |      21 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

              Rate  STR:t   SU:t  STM:t  STN:t  TMX:m 
  STR:t   179120/s     --   -19%   -85%   -85%   -94% 
  SU:t    223280/s    24%     --   -81%   -82%   -93% 
  STM:t  1220000/s   580%   446%     --    -3%   -63% 
  STN:t  1300000/s   606%   466%     3%     --   -61% 
  TMX:m  3318000/s  1752%  1385%   172%   162%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 7 of 9):

 #table7#
 {dataset=>"len100ws10"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::Regex::trim    |    179120 |    5.5827 |                 0.00% |              1561.52% | 5.7e-12 |      20 |
 | String::Util::trim           |    200000 |    5      |                 1.51% |              1536.81% |   1e-07 |      20 |
 | String::Trim::NonRegex::trim |    367050 |    2.7244 |               104.92% |               710.83% | 5.7e-12 |      20 |
 | String::Trim::More::trim     |   1000000 |    0.9    |               532.42% |               162.73% | 1.2e-08 |      20 |
 | Text::Minify::XS::minify     |   2980000 |    0.336  |              1561.52% |                 0.00% | 3.1e-10 |      20 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

              Rate  STR:t   SU:t  STN:t  STM:t  TMX:m 
  STR:t   179120/s     --   -10%   -51%   -83%   -93% 
  SU:t    200000/s    11%     --   -45%   -82%   -93% 
  STN:t   367050/s   104%    83%     --   -66%   -87% 
  STM:t  1000000/s   520%   455%   202%     --   -62% 
  TMX:m  2980000/s  1561%  1388%   710%   167%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 8 of 9):

 #table8#
 {dataset=>"len100ws100"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::NonRegex::trim |     46334 |   21.5824 |                 0.00% |              3524.32% | 5.6e-12 |      23 |
 | String::Trim::Regex::trim    |    174000 |    5.75   |               275.55% |               865.08% | 5.2e-09 |      33 |
 | String::Util::trim           |    215000 |    4.64   |               365.00% |               679.43% | 1.7e-09 |      20 |
 | String::Trim::More::trim     |   1100000 |    0.94   |              2189.43% |                58.31% | 1.2e-09 |      23 |
 | Text::Minify::XS::minify     |   1680000 |    0.595  |              3524.32% |                 0.00% | 2.1e-10 |      20 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

              Rate  STN:t  STR:t  SU:t  STM:t  TMX:m 
  STN:t    46334/s     --   -73%  -78%   -95%   -97% 
  STR:t   174000/s   275%     --  -19%   -83%   -89% 
  SU:t    215000/s   365%    23%    --   -79%   -87% 
  STM:t  1100000/s  2196%   511%  393%     --   -36% 
  TMX:m  1680000/s  3527%   866%  679%    57%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 9 of 9):

 #table9#
 {dataset=>"len10ws1"}
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                  | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::Trim::Regex::trim    |    958300 |   1043    |                 0.00% |               718.11% | 1.7e-11 |      20 |
 | String::Util::trim           |    975000 |   1030    |                 1.73% |               704.18% | 4.1e-10 |      21 |
 | String::Trim::NonRegex::trim |   1271500 |    786.48 |                32.68% |               516.62% | 5.6e-12 |      20 |
 | String::Trim::More::trim     |   1600000 |    626    |                66.70% |               390.78% | 2.1e-10 |      20 |
 | Text::Minify::XS::minify     |   7840000 |    128    |               718.11% |                 0.00% | 1.7e-11 |      21 |
 +------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

              Rate  STR:t  SU:t  STN:t  STM:t  TMX:m 
  STR:t   958300/s     --   -1%   -24%   -39%   -87% 
  SU:t    975000/s     1%    --   -23%   -39%   -87% 
  STN:t  1271500/s    32%   30%     --   -20%   -83% 
  STM:t  1600000/s    66%   64%    25%     --   -79% 
  TMX:m  7840000/s   714%  704%   514%   389%     -- 
 
 Legends:
   STM:t: participant=String::Trim::More::trim
   STN:t: participant=String::Trim::NonRegex::trim
   STR:t: participant=String::Trim::Regex::trim
   SU:t: participant=String::Util::trim
   TMX:m: participant=Text::Minify::XS::minify

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Benchmark module startup overhead (C<< bencher -m StringFunctions::Trim --module-startup >>):

Result formatted as table:

 #table10#
 +------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | participant            | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | String::Util           |      10   |               5.1 |                 0.00% |               136.80% |   0.00019 |      20 |
 | String::Trim::Regex    |      10   |               5.1 |                13.78% |               108.12% |   0.00025 |      20 |
 | Text::Minify::XS       |       8   |               3.1 |                44.40% |                63.99% |   0.00013 |      22 |
 | String::Trim::NonRegex |       8   |               3.1 |                46.20% |                61.97% |   0.00021 |      20 |
 | String::Trim::More     |       8   |               3.1 |                51.23% |                56.58% |   0.00019 |      20 |
 | perl -e1 (baseline)    |       4.9 |               0   |               136.80% |                 0.00% | 9.9e-06   |      20 |
 +------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+


The above result formatted in L<Benchmark.pm|Benchmark> style:

                Rate   S:U  ST:R  TM:X  ST:N  ST:M  :perl -e1 ( 
  S:U          0.1/s    --    0%  -19%  -19%  -19%         -51% 
  ST:R         0.1/s    0%    --  -19%  -19%  -19%         -51% 
  TM:X         0.1/s   25%   25%    --    0%    0%         -38% 
  ST:N         0.1/s   25%   25%    0%    --    0%         -38% 
  ST:M         0.1/s   25%   25%    0%    0%    --         -38% 
  :perl -e1 (  0.2/s  104%  104%   63%   63%   63%           -- 
 
 Legends:
   :perl -e1 (: mod_overhead_time=0 participant=perl -e1 (baseline)
   S:U: mod_overhead_time=5.1 participant=String::Util
   ST:M: mod_overhead_time=3.1 participant=String::Trim::More
   ST:N: mod_overhead_time=3.1 participant=String::Trim::NonRegex
   ST:R: mod_overhead_time=5.1 participant=String::Trim::Regex
   TM:X: mod_overhead_time=3.1 participant=Text::Minify::XS

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-StringFunctions>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-StringFunctions>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-StringFunctions>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2021, 2018 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
