package Bencher::Scenario::Perl::Startup;

our $DATE = '2016-01-07'; # DATE
our $VERSION = '0.02'; # VERSION

use 5.010001;
use strict;
use warnings;

use File::Which;

my $participants = [];

if (my $perlbrew_path = which("perlbrew")) {
    my $perlbrew_root = $perlbrew_path;
    $perlbrew_root =~ s!/bin/perlbrew\z!!;
    my $out = `perlbrew list`;

    my @perls;
    while ($out =~ /^\s*\*?\s*(.+)/gm) {
        push @perls, $1;
    }

    for my $perl (@perls) {
        my ($perl_ver) = $perl =~ /perl-(.+)/;

        push @$participants, {
            name => "$perl -e1",
            cmdline => ["$perlbrew_root/perls/$perl/bin/perl", "-e1"],
        };

        if (version->parse($perl_ver) >= version->parse("5.10.0")) {
            push @$participants, {
                name => "$perl -E1",
                cmdline => ["$perlbrew_root/perls/$perl/bin/perl", "-E1"],
            };
        }
    }
} else {
    warn "Please install perlbrew so I can easily locate the different perls";
}

our $scenario = {
    summary => 'Benchmark startup time of perls',
    default_precision => 0.001,
    participants => $participants,
};

1;
# ABSTRACT: Benchmark startup time of perls

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::Perl::Startup - Benchmark startup time of perls

=head1 VERSION

This document describes version 0.02 of Bencher::Scenario::Perl::Startup (from Perl distribution Bencher-Scenarios-Perl), released on 2016-01-07.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m Perl::Startup

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARK PARTICIPANTS

=over

=item * perl-5.10.1 -e1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.10.1/bin/perl -e1



=item * perl-5.10.1 -E1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.10.1/bin/perl -E1



=item * perl-5.12.5 -e1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.12.5/bin/perl -e1



=item * perl-5.12.5 -E1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.12.5/bin/perl -E1



=item * perl-5.14.4 -e1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.14.4/bin/perl -e1



=item * perl-5.14.4 -E1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.14.4/bin/perl -E1



=item * perl-5.16.3 -e1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.16.3/bin/perl -e1



=item * perl-5.16.3 -E1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.16.3/bin/perl -E1



=item * perl-5.18.4 -e1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.18.4/bin/perl -e1



=item * perl-5.18.4 -E1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.18.4/bin/perl -E1



=item * perl-5.20.3 -e1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.20.3/bin/perl -e1



=item * perl-5.20.3 -E1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.20.3/bin/perl -E1



=item * perl-5.22.0 -e1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.22.0/bin/perl -e1



=item * perl-5.22.0 -E1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.22.0/bin/perl -E1



=item * perl-5.22.1 -e1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.22.1/bin/perl -e1



=item * perl-5.22.1 -E1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.22.1/bin/perl -E1



=item * perl-5.6.2 -e1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.6.2/bin/perl -e1



=item * perl-5.8.9 -e1 (command)

Command line:

 /home/s1/perl5/perlbrew/perls/perl-5.8.9/bin/perl -e1



=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default option:

 +-----+-----------------+------+--------+---------+---------+
 | seq | name            | rate | time   | errors  | samples |
 +-----+-----------------+------+--------+---------+---------+
 | 15  | perl-5.22.1 -E1 | 183  | 5.48ms | 5.4e-06 | 915     |
 | 13  | perl-5.22.0 -E1 | 186  | 5.39ms | 5.3e-06 | 679     |
 | 11  | perl-5.20.3 -E1 | 186  | 5.37ms | 5.3e-06 | 1180    |
 | 9   | perl-5.18.4 -E1 | 191  | 5.22ms | 5.2e-06 | 905     |
 | 7   | perl-5.16.3 -E1 | 193  | 5.19ms | 5.2e-06 | 1060    |
 | 5   | perl-5.14.4 -E1 | 196  | 5.1ms  | 5.1e-06 | 1158    |
 | 3   | perl-5.12.5 -E1 | 198  | 5.06ms | 5e-06   | 1040    |
 | 1   | perl-5.10.1 -E1 | 203  | 4.93ms | 4.8e-06 | 1301    |
 | 14  | perl-5.22.1 -e1 | 220  | 4.55ms | 4.4e-06 | 1542    |
 | 12  | perl-5.22.0 -e1 | 220  | 4.54ms | 4.4e-06 | 1680    |
 | 10  | perl-5.20.3 -e1 | 222  | 4.5ms  | 4.4e-06 | 1402    |
 | 6   | perl-5.16.3 -e1 | 225  | 4.44ms | 4.4e-06 | 1541    |
 | 8   | perl-5.18.4 -e1 | 226  | 4.43ms | 4.4e-06 | 1332    |
 | 4   | perl-5.14.4 -e1 | 227  | 4.41ms | 4.3e-06 | 1335    |
 | 2   | perl-5.12.5 -e1 | 228  | 4.38ms | 4.4e-06 | 1219    |
 | 0   | perl-5.10.1 -e1 | 232  | 4.31ms | 4.2e-06 | 1207    |
 | 17  | perl-5.8.9 -e1  | 232  | 4.3ms  | 4.3e-06 | 948     |
 | 16  | perl-5.6.2 -e1  | 247  | 4.04ms | 3.9e-06 | 1184    |
 +-----+-----------------+------+--------+---------+---------+

=head1 DESCRIPTION

Conclusion: in general newer versions of perl has larger startup overhead than
previous ones. If startup overhead is important to you, use C<-e> instead of
C<-E> unless necessary.

=head1 SEE ALSO

L<Bencher::Scenario::Interpreters>

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-Perl>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-Perl>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-Perl>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
