#!/usr/bin/perl

=pod

This benchmark uses different methods to check a number of IP
addresses against a list of IP ranges. Two of them use C<Net::Netmask>
and are predictably slow. One even uses a helper function to call
C<Net::Netmask>'s C<match()> methods. One method manually defines
the required bit shifts and logical OR operations. While this one
is the fastest, finding the required numbers can be a pain. The
method using C<Net::IP::Match> is only a bit slower than the manual
method and much easier to write.

On my Powerbook, using the manual method is only 18% faster than
using C<Net::IP::Raw>, but C<Net::IP::Raw> is almost 20 times faster
than using C<Net::Netmask>.

=cut

use warnings;
use strict;
use Net::Netmask 'quad2int';
use Net::IP::Match;
use Benchmark;

my @n = map { Net::Netmask->new($_) } qw{
    10.0.0.0/8
    87.134.66.128
    87.134.87.0/24
    145.97.0.0/16
    192.168.0.0/16
};

my @t;
for my $n (@n) {
	my $bits = 32 - $n->bits;
	my $mask = quad2int($n->base) >> $bits;
	push @t => [ $mask, $bits ];
}

chomp(my @data = <DATA>);

sub netmask_loop_match {
	for (@n) { return 1 if $_->match($_[0]) }
}

timethese(20, {
    netmask_loop_sub    => sub { netmask_loop_match($_) for @data },
    netmask_loop_inline => sub {
	for (@data) {
		for my $n (@n) { next if $n->match($_) }
	}
    },
    manual_inline => sub {
	for (@data) {
		my $q = unpack("N", pack("C4", split(/\./, $_)));
		next if
		   4093015 == $q >> 8  ||
		     44048 == $q >> 16 ||
		     49320 == $q >> 16 ||
		1047806592 == $q       ||
			10 == $q >> 24;
	}
    },
    net_ip_match => sub {
    	for (@data) {
		next if __MATCH_IP($_, qw{ 10.0.0.0/8 87.134.66.128
		    87.134.87.0/24 145.97.0.0/16 192.168.0.0/16 });
	}
    },
});

# this sample list is from:
# perl -lane'$h{$F[2]}++;$h{$F[4]}++;END{print for sort keys %h}' net-acct-small

__DATA__
10.0.1.38
12.220.206.11
12.225.186.217
12.234.72.155
12.250.104.146
12.250.208.202
12.33.247.6
12.47.217.11
128.101.91.70
128.105.2.10
128.105.6.12
128.11.40.194
128.11.45.101
128.112.129.15
128.121.10.146
128.121.12.60
128.121.12.66
128.121.12.67
128.121.12.70
128.121.12.80
128.121.12.81
128.121.12.85
128.130.182.77
128.130.38.28
128.242.244.242
128.242.252.129
128.242.252.21
128.250.1.21
128.250.22.2
128.252.120.1
128.252.133.13
128.252.19.2
128.63.2.53
128.8.10.90
128.86.1.20
128.9.128.127
129.11.128.119
129.11.64.1
129.16.1.3
129.20.82.33
129.237.110.134
129.250.244.10
129.250.31.190
129.27.218.66
129.88.30.1
130.119.248.112
130.119.248.67
130.119.248.98
130.225.102.182
130.228.230.161
130.235.20.3
130.238.164.6
130.244.143.77
130.244.77.7
130.59.1.30
130.59.1.80
130.59.10.30
131.130.1.11
131.130.1.12
131.169.200.2
131.231.16.16
131.231.16.7
131.251.0.10
131.251.0.4
131.251.42.18
132.185.132.11
132.185.132.21
133.27.228.206
134.109.132.113
134.109.132.51
134.109.132.55
134.222.229.233
134.58.40.4
137.158.128.1
137.195.52.12
137.208.10.10
137.208.20.10
137.208.3.20
137.39.1.3
137.99.174.121
138.96.249.65
139.67.73.112
140.239.140.239
141.1.1.1
141.202.215.12
141.202.248.248
141.202.248.32
141.203.250.194
141.203.254.12
141.203.254.23
143.130.16.8
143.130.50.122
143.130.50.66
143.252.77.105
143.252.78.22
143.252.80.205
143.252.80.3
144.92.104.37
146.231.128.1
147.28.0.34
147.28.0.39
149.174.213.7
149.174.54.3
150.100.2.3
151.1.2.1
151.164.1.1
151.164.1.7
151.189.12.20
152.163.159.232
152.163.159.239
152.163.226.121
152.163.226.153
152.163.226.185
152.163.226.25
152.163.226.57
152.163.226.89
152.78.68.1
152.78.68.140
152.81.1.10
152.81.144.16
154.32.105.30
154.32.105.34
154.32.105.90
158.125.1.100
158.125.96.47
158.169.131.22
158.169.50.70
158.169.50.71
158.38.0.181
158.43.128.26
158.43.128.74
158.43.128.8
158.43.129.80
158.43.192.7
158.43.193.80
158.64.229.2
160.39.195.187
161.114.19.234
161.114.64.24
161.58.158.45
161.58.201.67
161.58.9.10
164.109.51.10
164.109.51.97
165.76.0.98
166.70.10.23
166.90.15.234
166.90.15.235
167.206.1.103
167.216.128.41
167.216.128.42
167.216.136.11
167.216.136.41
167.216.193.232
167.216.210.50
167.216.227.181
167.216.227.182
167.216.227.184
167.216.248.60
167.216.250.42
168.143.173.182
168.143.179.9
168.144.1.33
168.144.137.81
168.144.68.1
168.191.113.65
17.254.0.27
17.254.0.91
17.254.3.196
172.154.253.152
145.97.1.10
145.97.1.101
145.97.1.160
145.97.1.20
145.97.1.3
145.97.1.6
145.97.11.1
145.97.11.2
145.97.11.3
145.97.11.4
145.97.12.1
145.97.12.2
145.97.12.3
145.97.13.1
145.97.2.1
145.97.200.62
145.97.200.63
145.97.200.64
145.97.3.1
145.97.3.2
145.97.6.2
145.97.66.63
145.97.66.70
145.97.7.1
145.97.8.2
145.97.8.3
145.97.8.4
145.97.9.1
145.97.9.2
145.974.216.202
172.191.129.185
18.29.0.200
18.29.1.34
18.29.1.35
18.29.1.50
18.72.0.3
192.0.32.18
192.0.32.19
192.0.34.126
192.100.59.100
192.100.59.110
192.108.21.1
192.116.202.99
192.117.175.54
192.12.94.32
192.153.156.22
192.153.156.3
192.16.202.11
192.168.0.1
192.168.0.99
192.168.1.1
192.168.137.1
192.168.200.1
192.168.200.100
192.168.247.1
192.168.253.1
192.168.26.1
192.168.43.1
192.188.72.18
192.203.178.2
192.203.230.10
192.26.92.32
192.31.80.32
192.33.14.32
192.33.4.12
192.35.241.71
192.35.244.50
192.35.51.31
192.36.133.107
192.36.144.107
192.36.144.116
192.41.162.32
192.42.93.30
192.5.6.32
192.52.71.4
192.54.112.30
192.67.198.5
192.67.198.51
192.67.198.54
192.76.144.16
192.85.16.15
192.85.241.135
192.85.241.141
192.85.241.143
192.87.106.122
192.88.193.144
192.92.138.35
192.93.0.1
192.93.0.4
192.94.163.152
193.0.0.193
193.0.0.237
193.0.14.129
193.10.252.19
193.102.192.16
193.102.192.34
193.102.192.41
193.102.192.49
193.108.91.102
193.108.91.137
193.11.224.1
193.11.224.20
193.11.241.5
193.124.22.65
193.124.224.35
193.124.225.35
193.124.23.3
193.124.83.69
193.136.2.226
193.141.147.117
193.141.40.1
193.141.40.42
193.149.44.49
193.154.160.110
193.154.164.57
193.154.165.44
193.154.172.130
193.154.172.65
193.154.243.18
193.155.33.100
193.158.254.16
193.170.251.71
193.171.255.2
193.171.255.34
193.171.255.66
193.171.255.77
193.175.239.30
193.176.144.130
193.178.158.1
193.178.158.106
193.189.160.11
193.189.160.12
193.189.173.100
193.189.173.99
193.193.190.1
193.197.167.2
193.2.1.91
193.205.245.5
193.232.173.111
193.232.8.20
193.242.91.135
193.41.132.1
193.41.132.2
193.51.208.13
193.51.208.66
193.51.24.1
193.53.80.77
193.60.145.2
193.60.146.2
193.62.157.66
193.63.105.17
193.63.106.103
193.63.55.1
193.63.94.20
193.67.79.134
193.69.116.10
193.80.200.132
193.80.200.133
193.80.200.134
193.80.200.135
193.80.200.136
193.80.200.137
193.80.200.164
193.80.200.168
193.80.200.169
193.81.246.11
193.81.246.12
193.81.246.37
193.81.83.2
193.83.146.12
193.88.44.42
194.107.60.1
194.107.60.10
194.109.218.36
194.112.98.217
194.113.40.45
194.118.11.2
194.118.44.239
194.119.128.65
194.119.128.66
194.129.64.163
194.131.104.13
194.145.150.1
194.151.19.41
194.152.160.15
194.152.160.154
194.152.160.4
194.152.166.68
194.152.178.1
194.152.178.10
194.152.178.14
194.158.133.1
194.158.133.21
194.158.136.121
194.158.136.40
194.158.136.43
194.158.136.44
194.158.136.45
194.158.136.91
194.159.245.16
194.159.73.5
194.163.15.87
194.177.128.113
194.177.151.10
194.183.128.251
194.183.128.35
194.192.186.225
194.2.0.30
194.2.0.60
194.2.159.100
194.2.159.101
194.2.159.103
194.205.125.26
194.208.240.150
194.21.33.10
194.213.64.150
194.22.190.10
194.22.190.12
194.22.194.18
194.221.250.106
194.237.107.19
194.237.107.6
194.24.128.102
194.24.128.92
194.24.128.93
194.244.65.10
194.245.101.58
194.246.96.192
194.246.96.49
194.246.96.79
194.25.0.125
194.25.134.80
194.42.48.120
194.42.96.10
194.48.124.200
194.48.124.202
194.48.124.50
194.48.124.51
194.48.124.65
194.67.2.109
194.67.23.231
194.67.23.232
194.67.23.251
194.67.35.196
194.67.35.250
194.67.35.252
194.67.45.122
194.67.57.104
194.67.57.4
194.72.6.51
194.72.6.52
194.77.219.24
194.81.227.226
194.83.57.11
194.83.57.15
194.83.57.3
194.85.119.1
194.85.32.18
194.98.0.1
194.98.0.2
194.98.19.1
195.102.241.10
195.122.143.130
195.122.226.1
195.122.226.28
195.127.143.17
195.127.143.5
195.129.12.74
195.130.224.18
195.130.225.129
195.130.225.73
195.14.50.21
195.141.10.136
195.141.10.170
195.141.13.136
195.141.13.170
195.145.110.1
195.145.110.3
195.146.128.54
195.154.195.26
195.158.245.141
195.161.0.135
195.161.3.22
195.170.66.149
195.170.70.72
195.19.27.2
195.2.82.114
195.2.83.38
195.20.224.105
195.20.224.113
195.20.224.187
195.20.224.95
195.20.224.97
195.20.225.34
195.20.225.35
195.20.225.36
195.20.225.40
195.20.253.167
195.209.0.6
195.212.102.24
195.216.64.16
195.226.68.155
195.228.242.222
195.228.242.223
195.23.2.4
195.230.39.2
195.239.38.14
195.239.40.130
195.248.63.107
195.27.208.131
195.27.221.2
195.27.221.3
195.27.221.35
195.27.221.41
195.3.96.126
195.3.96.68
195.3.96.69
195.3.96.72
195.3.96.85
195.3.96.86
195.3.96.94
195.3.96.99
195.30.0.1
195.34.133.10
195.34.133.11
195.34.133.50
195.42.198.80
195.46.160.1
195.46.179.3
195.48.204.60
195.54.192.86
195.54.223.4
195.58.160.2
195.58.163.11
195.58.163.12
195.58.163.2
195.58.164.12
195.58.164.2
195.58.165.152
195.58.165.153
195.58.165.156
195.58.165.167
195.58.165.170
195.58.165.171
195.58.176.181
195.58.180.116
195.66.240.130
195.68.156.66
195.68.21.199
195.70.224.61
195.70.248.1
195.70.248.2
195.70.248.6
195.75.254.100
195.75.254.5
195.93.80.120
195.94.80.11
195.94.80.12
195.94.83.15
195.94.90.10
195.96.0.2
196.25.1.1
196.25.111.97
196.4.160.17
198.133.199.100
198.133.199.110
198.186.202.135
198.186.202.136
198.186.203.32
198.186.203.85
198.202.74.90
198.246.0.28
198.246.0.4
198.31.3.18
198.32.4.13
198.32.64.12
198.4.75.100
198.41.0.10
198.41.0.4
198.6.1.161
198.6.1.181
198.6.1.202
198.6.1.65
198.6.1.82
198.6.1.83
198.6.100.21
198.6.100.37
198.6.49.5
198.60.22.2
198.60.22.22
198.81.129.100
198.81.129.193
198.95.251.10
199.175.6.244
199.196.144.3
199.201.232.88
199.201.233.10
199.217.253.11
199.232.41.10
199.60.48.11
199.60.48.12
200.16.97.77
202.12.27.33
202.12.30.131
202.139.101.107
202.139.133.129
202.139.243.153
202.174.44.2
202.232.149.105
202.232.2.34
202.232.89.57
202.65.97.65
203.100.231.238
203.164.97.248
203.178.136.63
203.194.166.182
203.37.255.97
203.81.45.254
204.144.142.1
204.152.169.199
204.152.186.58
204.152.63.221
204.168.28.9
204.176.177.10
204.176.177.20
204.176.177.30
204.176.88.5
204.202.129.250
204.202.131.230
204.202.132.16
204.202.132.19
204.202.132.51
204.202.133.16
204.216.129.2
204.216.190.30
204.248.36.130
204.248.36.131
204.248.36.138
204.253.104.10
204.253.104.11
204.253.104.30
204.70.25.234
204.71.154.5
204.71.200.33
204.74.101.1
204.91.99.140
204.97.17.6
205.138.3.20
205.138.3.243
205.138.3.82
205.181.112.121
205.181.112.65
205.181.112.72
205.181.112.74
205.181.112.76
205.181.112.81
205.181.112.84
205.188.132.235
205.188.132.67
205.188.157.232
205.188.157.239
205.188.165.121
205.188.241.137
205.188.245.120
205.191.194.13
205.191.194.14
205.219.198.34
206.13.28.11
206.13.29.11
206.14.214.156
206.14.240.246
206.141.251.2
206.142.53.201
206.142.53.202
206.142.53.27
206.142.53.37
206.167.33.70
206.167.33.71
206.167.33.75
206.173.119.72
206.183.198.240
206.183.198.241
206.184.59.10
206.191.0.140
206.196.44.241
206.20.56.199
206.204.52.11
206.244.69.15
206.244.69.2
206.252.128.5
206.252.131.195
206.253.194.65
206.253.194.97
206.253.214.11
206.54.224.1
206.65.183.140
206.65.183.155
206.65.183.21
206.65.183.25
206.65.183.40
206.65.183.70
206.83.160.5
207.126.105.146
207.126.96.162
207.136.98.110
207.150.202.250
207.150.209.116
207.155.127.155
207.155.183.72
207.155.183.73
207.155.184.72
207.171.167.7
207.171.168.16
207.171.169.16
207.171.169.7
207.171.189.20
207.175.235.145
207.188.24.156
207.188.7.131
207.200.73.80
207.200.90.3
207.202.245.109
207.202.64.2
207.202.65.146
207.227.54.136
207.231.134.27
207.252.96.3
207.46.106.88
207.46.130.149
207.46.138.11
207.46.197.100
207.46.197.102
207.46.197.113
207.46.230.218
207.46.230.219
207.46.230.220
207.68.131.27
207.68.171.253
207.68.171.254
207.68.172.253
207.68.172.254
207.68.173.253
207.68.173.254
207.68.176.189
207.68.176.190
207.68.176.254
207.68.178.110
207.68.178.123
207.68.178.237
207.68.185.57
207.68.185.58
207.7.32.3
207.7.32.5
207.7.40.227
207.71.44.121
208.10.137.10
208.10.137.15
208.10.137.9
208.148.96.31
208.150.70.201
208.152.90.10
208.184.139.82
208.184.224.94
208.184.29.250
208.185.54.14
208.201.239.31
208.211.225.10
208.216.183.15
208.216.183.23
208.246.241.192
208.249.211.50
208.25.68.10
208.45.140.254
208.48.67.17
208.58.239.198
209.1.235.120
209.10.58.124
209.115.72.62
209.130.30.130
209.133.83.21
209.133.83.22
209.150.128.30
209.150.129.3
209.151.233.12
209.157.68.18
209.16.211.42
209.185.188.14
209.185.253.230
209.192.217.105
209.196.32.34
209.196.32.38
209.197.64.1
209.20.130.34
209.20.130.35
209.202.192.91
209.202.193.252
209.202.196.70
209.202.220.8
209.202.224.253
209.202.228.10
209.207.221.1
209.21.0.72
209.225.26.100
209.225.26.101
209.225.26.108
209.225.26.98
209.227.54.93
209.235.192.3
209.235.192.4
209.235.192.91
209.235.198.211
209.235.219.153
209.235.31.12
209.235.31.3
209.239.19.42
209.240.130.48
209.242.42.233
209.244.0.1
209.244.0.2
209.246.126.109
209.247.40.10
209.247.41.60
209.249.97.8
209.27.251.231
209.61.191.160
209.66.74.10
209.66.98.16
209.67.231.204
209.67.3.80
209.67.50.85
209.67.50.86
209.68.1.11
209.68.1.156
209.68.1.187
209.71.218.73
209.71.218.74
209.71.218.84
209.71.218.86
209.71.218.88
209.73.225.7
209.73.26.165
209.75.20.2
209.75.20.3
209.75.20.41
209.75.22.238
209.75.4.2
209.75.4.7
209.85.157.220
209.87.238.41
209.87.239.18
210.224.192.5
210.224.199.101
211.10.96.12
211.10.96.51
211.133.137.2
211.133.137.66
211.133.137.67
211.183.209.11
212.100.224.113
212.100.224.114
212.111.32.38
212.121.128.1
212.121.128.2
212.121.130.5
212.126.64.1
212.14.160.5
212.14.160.98
212.14.162.35
212.152.252.249
212.152.252.250
212.16.32.1
212.16.37.139
212.168.0.61
212.169.211.164
212.172.17.128
212.172.60.10
212.172.60.17
212.172.60.39
212.172.60.43
212.18.0.8
212.18.18.101
212.18.18.99
212.18.3.5
212.181.166.71
212.183.14.5
212.183.31.26
212.184.208.33
212.184.212.33
212.185.239.115
212.187.205.146
212.187.205.148
212.187.205.150
212.192.0.20
212.192.20.33
212.192.20.40
212.197.128.115
212.197.128.126
212.197.128.68
212.20.191.2
212.218.130.30
212.219.56.157
212.227.118.76
212.227.118.86
212.227.118.98
212.227.119.81
212.227.119.83
212.227.123.12
212.227.123.2
212.227.134.55
212.227.174.241
212.227.180.144
212.227.194.132
212.227.97.115
212.236.21.164
212.236.21.171
212.236.21.172
212.236.212.14
212.236.73.36
212.239.17.220
212.239.17.27
212.27.32.132
212.33.32.160
212.33.59.103
212.38.64.6
212.4.174.101
212.4.174.102
212.4.174.84
212.4.201.17
212.42.242.2
212.42.249.13
212.42.249.21
212.52.195.202
212.52.224.31
212.52.224.6
212.58.175.240
212.58.224.20
212.58.226.40
212.62.17.141
212.62.17.145
212.69.109.130
212.72.38.21
212.72.38.71
212.74.64.34
212.77.165.140
212.77.165.22
212.78.160.237
212.95.31.11
212.95.31.26
212.95.31.35
213.140.1.92
213.141.139.70
213.141.139.72
213.150.1.5
213.150.1.6
213.150.2.91
213.160.193.116
213.161.66.165
213.164.0.2
213.164.12.102
213.164.12.105
213.164.12.20
213.164.12.22
213.164.12.28
213.164.12.44
213.164.25.236
213.164.26.27
213.164.27.100
213.164.29.15
213.164.31.140
213.164.31.150
213.164.31.170
213.164.6.170
213.164.6.171
213.164.6.172
213.164.6.173
213.164.6.174
213.164.6.179
213.164.6.181
213.180.193.1
213.180.193.2
213.180.194.129
213.186.33.99
213.186.34.76
213.196.2.29
213.198.12.202
213.199.144.151
213.199.144.152
213.212.70.59
213.221.172.238
213.221.173.2
213.229.14.251
213.229.60.101
213.229.60.11
213.24.48.2
213.24.56.130
213.24.56.2
213.24.56.202
213.24.56.203
213.24.60.4
213.244.183.194
213.244.183.195
213.244.183.206
213.248.112.35
213.248.70.234
213.33.99.9
213.46.255.17
213.46.255.2
213.47.11.156
213.59.0.47
213.61.120.236
213.61.184.163
213.61.6.2
213.61.6.5
213.8.241.175
213.86.246.20
213.86.246.60
213.86.246.80
213.86.34.162
213.90.0.194
213.90.0.195
213.90.0.226
213.90.0.227
213.90.73.19
213.90.73.22
213.90.73.23
213.90.74.142
213.90.74.20
213.90.74.21
213.90.74.22
213.90.74.23
213.90.74.69
213.96.130.132
213.99.111.215
216.10.100.126
216.10.17.94
216.10.17.95
216.114.1.178
216.115.107.12
216.127.33.6
216.127.33.7
216.127.33.92
216.136.171.196
216.136.171.197
216.136.171.200
216.136.171.201
216.136.171.204
216.136.227.6
216.143.170.159
216.144.70.227
216.144.70.228
216.148.213.143
216.148.213.145
216.148.218.160
216.148.218.165
216.148.218.167
216.148.218.195
216.148.218.197
216.148.221.144
216.152.230.50
216.154.243.11
216.154.243.211
216.154.243.213
216.158.128.25
216.158.128.26
216.158.130.45
216.158.143.52
216.158.143.58
216.158.143.63
216.158.143.77
216.158.143.78
216.168.224.206
216.18.119.241
216.200.14.118
216.200.14.134
216.21.226.88
216.21.234.88
216.218.132.2
216.218.196.178
216.218.196.181
216.221.200.205
216.221.200.206
216.227.91.85
216.239.32.10
216.239.34.10
216.239.35.101
216.239.35.124
216.239.35.246
216.239.35.247
216.239.35.248
216.239.36.10
216.239.37.100
216.239.38.10
216.239.39.100
216.239.51.100
216.239.51.101
216.240.128.4
216.240.128.5
216.240.159.119
216.240.159.121
216.254.0.38
216.255.129.249
216.255.130.249
216.32.126.150
216.32.212.86
216.32.65.105
216.32.65.14
216.32.74.10
216.33.35.214
216.33.46.134
216.33.55.9
216.34.209.13
216.34.209.27
216.34.68.2
216.34.88.20
216.34.88.210
216.34.88.211
216.34.88.212
216.35.16.7
216.35.16.83
216.35.167.58
216.40.32.140
216.40.32.30
216.40.34.65
216.40.37.11
216.40.37.12
216.46.197.4
216.46.200.172
216.49.81.129
216.49.81.144
216.5.163.42
216.5.163.44
216.5.163.46
216.5.163.47
216.5.163.48
216.52.1.1
216.52.1.33
216.87.221.138
216.91.66.3
216.92.54.1
216.92.60.60
217.11.192.1
217.11.192.136
217.11.192.137
217.11.192.141
217.11.192.155
217.11.192.43
217.11.196.197
217.110.206.54
217.115.141.244
217.115.143.37
217.118.98.26
217.12.4.71
217.162.133.89
217.19.35.10
217.19.35.36
217.19.35.37
217.215.1.120
217.215.78.102
217.228.144.202
217.3.5.123
217.36.4.162
217.5.100.185
217.5.95.38
217.6.49.194
217.67.100.20
217.69.237.162
217.69.237.163
217.69.237.166
217.73.192.19
217.73.192.49
217.73.193.23
217.85.138.244
217.96.84.170
217.97.51.116
218.45.208.10
224.0.0.1
224.0.1.1
24.0.0.27
24.100.27.104
24.153.22.141
24.164.177.83
24.165.128.163
24.166.202.169
24.185.94.208
24.2.0.27
24.205.104.180
24.217.175.169
24.219.166.146
24.222.148.87
24.26.124.101
24.52.71.145
24.69.231.229
24.69.68.136
24.77.231.1
38.160.150.21
4.2.49.2
4.2.49.3
4.2.49.4
4.43.229.237
4.62.216.234
61.115.192.17
61.115.192.18
61.120.151.100
62.101.73.108
87.134.1.10
87.134.11.1
87.134.11.2
87.134.11.3
87.134.12.1
87.134.2.1
87.134.2.6
87.134.2.8
87.134.200.100
87.134.31.119
87.134.31.126
87.134.33.8
87.134.44.1
87.134.44.15
87.134.44.2
87.134.44.5
87.134.44.70
87.134.44.74
87.134.44.82
87.134.58.250
87.134.64.105
87.134.66.134
87.134.66.136
87.134.66.137
87.134.66.138
87.134.66.226
87.134.66.228
87.134.8.2
87.134.8.3
87.134.8.4
87.134.87.1
87.134.87.100
87.134.87.101
87.134.87.102
87.134.87.103
87.134.87.105
87.134.87.107
87.134.87.108
87.134.87.109
87.134.87.11
87.134.87.113
87.134.87.116
87.134.87.119
87.134.87.122
87.134.87.125
87.134.87.140
87.134.87.2
87.134.87.255
87.134.9.1
87.134.9.40
87.134.98.39
62.128.1.42
62.128.28.195
62.128.28.197
62.129.133.12
62.13.10.141
62.132.142.2
62.151.2.100
62.151.2.8
62.153.59.66
62.161.94.240
62.163.138.117
62.189.244.232
62.189.30.1
62.189.30.129
62.2.247.10
62.2.247.131
62.2.32.250
62.207.130.15
62.208.140.10
62.208.140.13
62.208.140.52
62.208.141.112
62.210.148.2
62.212.118.165
62.212.119.166
62.215.0.182
62.234.51.161
62.245.129.67
62.26.117.243
62.26.119.34
62.26.127.165
62.26.211.64
62.26.5.168
62.4.16.70
62.4.68.246
62.4.81.10
62.7.228.102
62.7.228.120
62.79.38.228
62.99.146.174
63.100.190.61
63.102.48.3
63.102.49.29
63.102.49.33
63.149.6.93
63.167.205.130
63.208.48.42
63.209.29.137
63.209.29.138
63.209.29.145
63.210.142.26
63.210.161.5
63.210.163.178
63.210.31.70
63.210.31.71
63.211.175.130
63.211.175.131
63.211.175.135
63.211.175.136
63.211.175.137
63.211.210.14
63.211.210.15
63.211.210.20
63.215.198.78
63.215.198.79
63.231.106.226
63.236.72.133
63.236.73.251
63.240.14.66
63.241.16.9
63.70.44.44
63.70.47.44
63.88.212.142
64.0.96.12
64.0.99.133
64.0.99.165
64.0.99.166
64.0.99.229
64.108.177.125
64.108.177.126
64.108.194.221
64.12.174.153
64.12.174.185
64.12.180.21
64.12.184.121
64.12.184.57
64.12.184.89
64.124.186.66
64.124.237.128
64.124.237.129
64.124.237.130
64.124.237.131
64.124.237.132
64.124.237.133
64.124.237.148
64.124.237.156
64.124.237.71
64.125.133.20
64.132.76.244
64.132.76.245
64.14.117.10
64.14.200.154
64.15.251.198
64.15.251.199
64.15.251.200
64.152.73.197
64.157.224.3
64.157.224.7
64.170.115.29
64.192.144.89
64.200.241.28
64.200.68.125
64.21.104.130
64.210.241.99
64.210.248.163
64.220.205.140
64.230.2.73
64.230.90.208
64.231.68.53
64.240.174.5
64.242.248.53
64.245.43.14
64.26.0.23
64.28.67.114
64.28.67.150
64.28.67.21
64.28.67.251
64.28.67.35
64.28.67.55
64.28.67.57
64.28.67.58
64.28.67.80
64.37.200.46
64.37.205.165
64.37.205.5
64.39.31.103
64.39.45.136
64.4.12.206
64.4.13.32
64.41.192.103
64.45.62.2
64.56.174.186
64.58.77.87
64.61.61.86
64.61.61.93
64.61.61.94
64.63.125.18
64.63.125.202
64.63.125.203
64.65.1.66
64.69.180.182
64.69.67.228
64.69.71.11
64.69.76.100
64.69.86.6
64.85.75.9
64.94.123.4
64.95.61.36
64.95.61.4
65.107.16.34
65.107.16.38
65.107.16.44
65.205.179.4
65.211.1.194
65.45.35.243
65.90.98.5
65.90.98.6
65.90.98.69
65.94.241.115
66.113.201.166
66.122.221.5
66.187.233.210
66.189.220.62
66.197.0.128
66.205.216.25
66.207.130.71
66.207.130.72
66.207.130.77
66.35.216.85
66.38.151.10
66.38.151.18
66.51.203.16
66.51.203.17
66.51.205.100
66.54.3.38
66.54.3.6
66.60.188.19
66.60.188.39
66.71.60.164
66.77.24.3
66.77.24.42
66.78.4.194
66.78.4.195
66.78.4.196
66.87.118.77
67.33.179.229
67.82.171.106
68.7.116.167
68.7.162.170
68.82.80.56
80.130.68.125
80.193.226.175
80.25.235.254
80.71.0.129
80.78.225.73
80.78.233.163
80.78.233.164
80.78.233.166
