package Data::Timeline::IScrobbler;

use strict;
use warnings;
use DateTime::Format::DateParse;



our $VERSION = '0.01';


use base qw(Data::Timeline::Builder);


__PACKAGE__->mk_scalar_accessors(qw(log_filename));


use constant DEFAULTS => (
    log_filename => "$ENV{HOME}/Library/Logs/iScrobbler.log",
);


sub create {
    my $self = shift;

    my $timeline = $self->make_timeline;

    my $filename = $self->log_filename;
    open my $fh, '<', $filename or die "can't open $filename: $!\n";

    while (<$fh>) {
        next unless /^\[(.*?)\]-\[VERB\] Added '(.*)'/;
        $timeline->entries_push($self->make_entry(
            timestamp   => DateTime::Format::DateParse->parse_datetime($1),
            description => $2,
            type        => 'iscrobbler',
        ));
    }

    close $fh or die "can't close $filename: $!\n";

    $timeline;
}


1;


__END__



=head1 NAME

Data::Timeline::IScrobbler - Build a timeline from tracks recorded by iScrobbler

=head1 SYNOPSIS

    my $timeline = Data::Timeline::IScrobbler->new->create;

=head1 DESCRIPTION

This class parses the logs created by the Mac OS X application iScrobbler and
creates a timeline of the recently played tracks. iScrobbler is an
application for Mac OS X that creates a simple menu extra that submits your
"currently playing" info from iTunes to AudioScrobbler. The advantage over
using the AudioScrobbler web service is that iScrobbler keeps a lot more
recently played tracks history than the web service. See its page at
http://iscrobbler.sourceforge.net/.

The timeline entries generated by this builder have the type C<iscrobbler>.

See the C<eg/hackmusic-text.pl> and C<eg/hackmusic-html.pl> programs in this
distribution for examples. The idea for those programs was inspired by Greg
McCarroll's blog post "Optimal Hacking Music" at
http://drinkbroken.typepad.com/drink_broken/2007/11/set-the-hack--1.html. The
programs show timelines of your recently played iTunes tracks history
alongside your svk commit history and thus give an impression of which music
you like to hack to.

Data::Timeline::IScrobbler inherits from L<Data::Timeline::Builder>.

=head1 METHODS

=over 4

=item clear_log_filename

    $obj->clear_log_filename;

Clears the value.

=item log_filename

    my $value = $obj->log_filename;
    $obj->log_filename($value);

A basic getter/setter method. If called without an argument, it returns the
value. If called with a single argument, it sets the value.

=item log_filename_clear

    $obj->log_filename_clear;

Clears the value.

=item log_filename

The log filename from which the builder tries to parse the recently played
track information. Defaults to C<~/Library/Logs/iScrobbler.log>.

=item create

Start parsing information from the logfile indicated by the C<log_filename()>.
Returns a L<Data::Timeline> object with the gleaned information.

=back

=head1 TAGS

If you talk about this module in blogs, on del.icio.us or anywhere else,
please use the C<datatimelineiscrobbler> tag.

=head1 BUGS AND LIMITATIONS

No bugs have been reported.

Please report any bugs or feature requests to
C<<bug-data-timeline-iscrobbler@rt.cpan.org>>, or through the web interface at
L<http://rt.cpan.org>.

=head1 INSTALLATION

See perlmodinstall for information and options on installing Perl modules.

=head1 AVAILABILITY

The latest version of this module is available from the Comprehensive Perl
Archive Network (CPAN). Visit <http://www.perl.com/CPAN/> to find a CPAN
site near you. Or see <http://www.perl.com/CPAN/authors/id/M/MA/MARCEL/>.

=head1 AUTHOR

Marcel GrE<uuml>nauer, C<< <marcel@cpan.org> >>

=head1 COPYRIGHT AND LICENSE

Copyright 2007 by Marcel GrE<uuml>nauer

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.


=cut

