package Tk::TabbedForm::CBaseClass;

=head1 NAME

Tk::TabbedForm::CBaseClass - Base class for items in Tk::TabbedForm.

=cut

use strict;
use warnings;
use base qw(Tk::Derived Tk::Frame);
use Tie::Watch;
Construct Tk::Widget 'CBaseClass';

=head1 SYNOPSIS

 package MyFormItem;
 
 use base qw(Tk::TabedForm::CBaseClass);
 Construct Tk::Widget 'MyFormItem';

=head1 DESCRIPTION

Inherits L<Tk::Frame>

Provides a base class for you to inherit. Helps making items for L<Tk::TabbedForm>.

You should never create an instance directly like above. This should
be handled by L<Tk::TabbedForm>.

=head1 B<CONFIG VARIABLES>

=over 4

=item Switch: B<-regex>

By default '.*'. Set a regular expression used for validation.

=item Switch: B<-validatecall>

Callback, called after validation with the result as parameter.

=back

=head1 METHODS

=over 4

=cut

sub Populate {
	my ($self,$args) = @_;

	$self->SUPER::Populate($args);
	my $var = '';
	Tie::Watch->new(
		-variable => \$var,
		-store => sub {
			my ($watch, $value) = @_;
			$watch->Store($value);
			$self->Callback('-validatecall');
		},
	);
	$self->createHandler(\$var);
	$self->{VARIABLE} = \$var;

	$self->ConfigSpecs(
		-regex => ['PASSIVE', undef, undef, '.*'],
		-validatecall => ['CALLBACK', undef, undef, sub {}],
		DEFAULT => ['SELF'],
	);
	$self->after(1, ['validate', $self]);
}

sub createHandler {
}

=item B<get>

Returns the value.

=cut

sub get {
	my $self = shift;
	my $var = $self->variable;
	return $$var;
}

=item B<put>I<($value)>

Sets the value.

=cut

sub put {
	my ($self, $value) = @_;
	my $var = $self->variable;
	$$var = $value;
}

=item B<validate>I<(?$value?)>

Validates the value against the regex in the B<-regex> option.

=cut

sub validate {
	my ($self, $val) = @_;
	my $var = $self->variable;
	return 1 unless defined $var;
	$val = $$var unless defined $val;
	my $reg = $self->cget('-regex');
	my $flag = $val =~ /$reg/;
	$self->validUpdate($flag, $val);
	return $flag;
}

=item B<validUpdate>I<($flag, $value)>

For you to overwrite. Does nothing. Is called to update the 
widget to reflect the outcome of validate.

=cut

sub validUpdate {
}

=item B<variable>

Returns a reference to the internal variable.

=cut

sub variable {
	return $_[0]->{VARIABLE};
}
=back

=head1 LICENSE

Same as Perl.

=head1 AUTHOR

Hans Jeuken (hanje at cpan dot org)

=head1 BUGS

Unknown. If you find any, please contact the author.

=head1 SEE ALSO

=over 4

=item L<Tk::ColorEntry>

=back

=cut

1;

__END__
