package OpenInteract2::Brick::Base;

use strict;
use base qw( OpenInteract2::Brick );
use OpenInteract2::Exception;

my %INLINED_SUBS = (
    'base-2.15.zip' => 'BASE215ZIP',
);

sub get_name {
    return 'base';
}

sub get_resources {
    return (
        'base-2.15.zip' => [ 'pkg base-2.15.zip', 'no' ],
    );
}

sub load {
    my ( $self, $resource_name ) = @_;
    my $inline_sub_name = $INLINED_SUBS{ $resource_name };
    unless ( $inline_sub_name ) {
        OpenInteract2::Exception->throw(
            "Resource name '$resource_name' not found ",
            "in ", ref( $self ), "; cannot load content." );
    }
    return $self->$inline_sub_name();
}

OpenInteract2::Brick->register_factory_type( get_name() => __PACKAGE__ );

=pod

=head1 NAME

OpenInteract2::Brick::Base - Base-64 encoded OI2 package 'base-2.15.zip' shipped with distribution

=head1 SYNOPSIS

  oi2_manage create_website --website_dir=/path/to/site

=head1 DESCRIPTION

Are you sure you even need to be reading this? If you are just looking
to install a package just follow the instructions from the SYNOPSIS.

Still here? This class holds the Base64-encoded versions of package
file "base-2.15.zip" shipped with OpenInteract2. Once you decode them you
should store them as a ZIP file and then read them in with
Archive::Zip or some other utility.

A typical means to do this is:

 my $brick = OpenInteract2::Brick->new( 'base' );

 # there's only one resource in this brick...
 my ( $pkg_name ) = $brick->list_resources;
 my $pkg_info = $brick->load_resource( $pkg_name );
 my $pkg_file = OpenInteract2::Util->decode_base64_and_store(
     \$pkg_info->{content}
 );

 # $pkg_file now references a file on the filesystem that you can
 # manipulate as normal

These resources are associated with OpenInteract2 version 1.99_06.


=head2 Resources

You can grab resources individually using the names below and
C<load_resource()> and C<copy_resources_to()>, or you can copy all the
resources at once using C<copy_all_resources_to()> -- see
L<OpenInteract2::Brick> for details.

=over 4


=item B<base-2.15.zip>


=back

=head1 COPYRIGHT

Copyright (c) 2005 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS


Chris Winters E<lt>chris@cwinters.comE<gt>


=cut


sub BASE215ZIP {
    return <<'SOMELONGSTRING';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SOMELONGSTRING
}

