# $Id$

use strict;
use warnings;

use Test::More;

require DateTime::Calendar::Fiscal5253;
my $class = 'DateTime::Calendar::Fiscal5253';

# This script only tests the basic functionality and not the accuracy
# of the values generated by the object. Another script will perform
# those tests.

# Get an object for testing with. Use values different from defaults
# to ensure the tests are fetching real information.
# 2012 is known to have only 52 weeks.
my %params = (
    year        => 2012,
    end_month   => 1,
    end_dow     => 1,
    end_type    => 'closest',
    leap_period => 'first'
);

my $fc = $class->new(%params);

# Make a second one with a known 53 week year.
my %params53 = (
    year        => 2014,
    end_month   => 1,
    end_dow     => 1,
    end_type    => 'closest',
    leap_period => 'first'
);

my $fc53 = $class->new(%params53);

ok(!$fc->has_leap_week, '2012 does not have a leap week');
ok($fc53->has_leap_week, '2014 does have a leap week');

my $c52 = $fc->calendar();
isa_ok($c52, 'ARRAY', 'scalar context returns array reference');
my @c52 = $fc->calendar();
ok(!ref(@c52), 'list context does not return a reference');
is_deeply($c52, \@c52, 'array and reference are the same');
for (qw( Fiscal Restated Truncated)) {
    my $c = $fc->calendar(style => $_);
    isa_ok($c, 'ARRAY', "parameter style $_");
    is_deeply($c52, $c, "style $_ is same as default in normal year");
}

# Verify that calendars for 53 week years vary for each style
my $cmeta53 = $fc53->calendar()->[0];
ok($cmeta53->{style} eq 'fiscal', 'default style set to "fiscal"');
ok($cmeta53->{weeks} == 53, 'default has 53 weeks');
$cmeta53 = $fc53->calendar(style => 'Fiscal')->[0];
ok($cmeta53->{style} eq 'fiscal', 'style set to "fiscal"');
ok($cmeta53->{weeks} == 53, 'Fiscal has 53 weeks');
$cmeta53 = $fc53->calendar(style => 'Restated')->[0];
ok($cmeta53->{style} eq 'restated', 'style set to "restated"');
ok($cmeta53->{weeks} == 52, 'Restated has 52 weeks');
$cmeta53 = $fc53->calendar(style => 'Truncated')->[0];
ok($cmeta53->{style} eq 'truncated', 'style set to "truncated"');
ok($cmeta53->{weeks} == 52, 'Truncated has 52 weeks');

# Test that "contains" function accepts valid calendar values
ok($fc53->contains(date => $fc53->start) == 1, 'default has start date');
ok($fc53->contains(date => $fc53->end) == 12,  'default has end date');
ok($fc53->contains(date => $fc53->start, calendar => 'Fiscal') == 1,
    'default has start date');
ok($fc53->contains(date => $fc53->end, calendar => 'Fiscal') == 12,
    'default has end date');
ok($fc53->contains(date => $fc53->start, calendar => 'Truncated') == 1,
    'Truncated has start date');
ok($fc53->contains(date => $fc53->end, calendar => 'Restated') == 12,
    'Restated has end date');
ok(!$fc53->contains(date => $fc53->end, calendar => 'Truncated'),
    'Truncated does not have default end date');
ok(!$fc53->contains(date => $fc53->start, calendar => 'Restated'),
    'Restated does not have default start date');

# verify that the period methods are working
my $pref = $fc->period(period => 1);
isa_ok($pref, 'HASH', 'scalar context returns a hash reference');
my %p = $fc->period(period => 1);
ok(!ref(%p), 'list context does not return a reference');
is_deeply($pref, \%p, 'hash and reference are the same');
for (qw( Fiscal Restated Truncated)) {
    my $p = $fc->period(period => 1, calendar => $_);
    isa_ok($p, 'HASH', "parameter calendar $_");
}
for (qw( month start end weeks )) {
    my $pmethod = "period_$_";
    ok($fc->$pmethod(period => 1) eq $pref->{$_}, "$pmethod matches data");
}

# Test the fail conditions now.
# A lexical block is used so the change to STDERR isn't global if we
# add more tests after this.
{
    # Capture any STDERR output
    my $stderr = '';
    local *STDERR;
    open STDERR, '>', \$stderr;

    ok(!$fc->calendar(style => 'foobar'), 'calendars rejects style "foobar"');
    ok(!$fc->contains(calendar => 'foobar'), 'contains rejects style "foobar"');
    ok(!$fc->period(period => 1, calendar => 'foobar'),
        'period rejects style "foobar"');
    for (qw( month start end weeks )) {
        my $pmethod = "period_$_";
        ok(
            !$fc->$pmethod(period => 1, calendar => 'foobar'),
            "$pmethod rejects calendar style 'foobar'"
        );
    }
}

done_testing();

exit;

# package for empty package tests
package Empty::Fiscal5253;
use base qw(DateTime::Calendar::Fiscal5253);

__END__
