#
#
#  Copyright (c) 2003 Andrew W. Speer <andrew.speer@isolutions.com.au>. All rights 
#  reserved.
#
#  This file is part of WebMod::Constant.
#
#  WebMod::Constant is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#  $Id: Constant.pm,v 1.10 2005/09/28 15:12:55 aspeer Exp $

#
#  Constant.pm - Read in constants from file, import them into calling module
#  as package global or sub constant (eg $FOO or just FOO).
#
#  Constants are read from the file CallingModule/Constant.pm
#
package  WebMod::Constant;


#  Compiler Pragma
#
sub BEGIN	{ $^W=0 };
use strict qw(vars);
use vars qw(@EXPORT @EXPORT_OK %EXPORT_TAGS $VERSION $RELEASE $PACKAGE);
use warnings;
no  warnings	qw(uninitialized);


#  Exteral Modules
#
use File::Spec();
require Exporter;


#  Version information in a format suitable for CPAN etc. Must be
#  all on one line
#
$VERSION = eval { require WebMod::Constant::VERSION; do $INC{'WebMod/Constant/VERSION.pm'}};


#  File Release information
#
$RELEASE = (split(/\s+/,'$Revision: 1.10 $'))[1];


#  Pakcage
#
$PACKAGE = __PACKAGE__;


#  Set up an anonymous sub which will be used as an error handler
#  This function calls die unless the error was a "file not found"
#  type error, which is not considered fatal
#
my $error_cr=sub {
    unless ( ($!+0)==2 ) {
	die sprintf("%s\n", shift() || 'undefined error');
    }
    undef;
};


#  Var for constants we have already seen
#
my %Constant;


#  Location of local Constant file
#
my $FILE_CONSTANT_LOCAL={

    #  Make a linux key pair value, and copy this to
    #  the current OS val, so we at least the linux
    #  settings as a default, no matter what the OS.
    #  OS specfic values (eg MSWin32) can overwrite
    #  this anyway
    #
    map ({ $_=> '/etc/Config.pm'} ('linux', $^O)),

    MSWin32 =>	File::Spec->canonpath(
	join('/', $ENV{'windir'} || 'C:', 'Config.pm'))

}->{$^O};


#  Get it
#
my $Constant_Local_hr=eval {do $FILE_CONSTANT_LOCAL} ||
    $error_cr->("error on do $FILE_CONSTANT_LOCAL, $@");
my %Constant_Local=%{ $Constant_Local_hr };


#  Return OK
#
return 1;


#===================================================================================================


sub import {


    #  Get the name of the package that is calling us
    #
    my $caller=caller();


    #  Read in package list of constants we want to import, remove
    #  from @_ once done and expanded
    #
    my @package_list=grep { !/^:/ } @_[1..$#_];
    @package_list=map { ref($_) ? @{$_} : $_ } $caller, @package_list;
    @_=($_[0], grep { /^:/ } @_);


    #  The EXPORT_TAGS hash will hold four tags - const var hash and all,
    #  which means import vars as a package global var (var tag) or
    #  a package constant (const), a Config hash (__PACKAGE__->{'Config'}{$_}
    #  or all three (all)
    #
    #  Start by filling with four empty array refs
    #
    @EXPORT_TAGS{qw(const var hash all)}=(map {[]} (1..4));


    #  The const hash will hold all the key/value pairs for this package
    #  read in. We define a couple of common OK/FAIL type constants
    #  by default
    #
    my %constant=(

	OK	=>	1,
	TRUE	=>	1,
	FAIL	=>	undef,
	FALSE	=>	undef

       );


    #  Go through each package that user wants contast from, build up hash
    #
    for my $package (@package_list) {


	#  OK we have the package name from which we wish to import
	#  the Constants, start processing. Debug
	#
	#_debug("importing caller $caller vars from package $package");


	#  Var to hold constant hash ref
	#
	my $constant_hr;
	my $constant_package="${package}::Constant";


	#  Have we already seen ?
	#
	if (exists($Constant{$constant_package})) {


	    #  Yes, set to that
	    #
	    $constant_hr=$Constant{$constant_package};


	}
	else {


	    #  Set local die handler
	    #
	    local $SIG{'__DIE__'}='IGNORE';


	    #  Must load and set
	    #
	    (my $constant_package_fn=$constant_package)=~s/::/\//g;
	    $constant_package_fn.='.pm';
	    if (my $cn=$INC{$constant_package_fn}) {
		$constant_hr=eval { do($cn) } ||
		    $error_cr->("$@");
	    }
	    else {
		$constant_hr=eval("require $constant_package") ||
		    $error_cr->("$@");
	    }
	    $Constant{$constant_package}=$constant_hr;



	    #  Result must have been a hash ref
	    #
	    unless ( ref($constant_hr) eq 'HASH') {
	    	my $reftype=ref($constant_hr) || 'undefined';
		$error_cr->(
		    	"Result of constant import from $constant_package was $reftype, ".
				"which is not of the required HASH reference type.\n"
			);

	    }

	}
	#_debug("importing caller $caller vars from package $package *complete*");


	#  Now transfer the key/value pairs from the hash ref we just received
	#  into the const hash
	#
	while (my($key, $value)=each %{$constant_hr}) { $constant{$key}=$value }


	#  Debug
	#_debug("package $package constant %s", Data::Dumper::Dumper($constant_hr));


	#  Now transfer the key/value pairs from the hash ref we just received
	#  into the const hash
	#
	#while (my($key, $value)=each %{$Constant_Local_hr->{$package}}) {
	while (my($key, $value)=each %{$Constant_Local{$package}}) {


	    #  Skip debug constant, unless for caller
	    #
	    next if (($key eq 'DEBUG') && ($package ne $caller));
	    #_debug("updating caller $caller const from package $package, $key => $value");
	    $constant{$key}=$value;

	}


    }
    #_debug("caller $caller constant_final %s", Data::Dumper::Dumper($const{'Constant'}));


    #  Clean the EXPORT globs of any unwanted residual vals
    #
    undef *EXPORT; undef *EXPORT_OK; undef *EXPORT_TAGS;


    #  Go through all constants and put them into appropriate Exporter holders
    #
    while (my($constant, $value)=each %constant) {


	#  Make a new glob that refers to the NAME closure function
	#  for this var. If there is already an existing glob defined,
	#  then make sure we delete it so we do not get
	#
	#  "XYZ constant redefined at .." type warnings from perl
	#  if the -w flag is enabled.
	#
	if (UNIVERSAL::can($caller, $constant) || UNIVERSAL::can('WebMod::Constant',$constant)) {
	    undef *{"${caller}::${constant}"};
	    undef *{"${caller}::${caller}"};
	    undef *{$constant}
	}
	*{$constant}=sub () { $value };


	#  Make a new global scalar NAME=var for this var
	#
	${$constant}=$value;


	#  Now get the approriate var and const names ($xyz and &xyz);
	#
	my ($var, $constant_export)=("\$${constant}", "\&${constant}");


	#  And push them onto the appropriate Exporter vars
	#
	push @EXPORT, $var;
	push @EXPORT_OK, $constant_export, $var;
	push @{$EXPORT_TAGS{'var'}}, $var;
	push @{$EXPORT_TAGS{'const'}}, $constant_export;
	push @{$EXPORT_TAGS{'all'}}, $constant_export, $var;

    }


    #  Push the hash ref into the caller package space. This lets the user
    #  do a query on __PACKAGE__->{'Config'}{$_} in their prog.
    #
    *{"::${caller}"}={ Config=>\%constant };


    #  Now pass control to the Exporter::import routine. We use a goto
    #  make it appear that the call comes from the original calling
    #  module
    #
    goto &Exporter::import;


}

