use strict;
use Test::More tests => 35;

#-----------------------------------------------------------------------
# Make sure that we can set up columns properly
#-----------------------------------------------------------------------
package State;

use base 'Class::DBI';

State->table('State');
State->columns('Primary',   'Name');
State->columns('Essential', qw/Abbreviation/);
State->columns('Weather',   qw/Rain Snowfall/);
State->columns('Other',     qw/Capital Population/);

sub accessor_name {
	my ($class, $column) = @_;
	my $return = $column eq "Rain" ? "Rainfall" : $column;
	return $return;
}

sub mutator_name {
	my ($class, $column) = @_;
	my $return = $column eq "Rain" ? "set_Rainfall" : "set_$column";
	return "set_$column";
}

sub Snowfall { 1 }

#-------------------------------------------------------------------------
package CD;
use base 'Class::DBI';

CD->table('CD');
CD->columns('All' => qw/artist title length/);

#-------------------------------------------------------------------------

package main;

is(State->table,          'State', 'State table()');
is(State->primary_column, 'name',  'State primary()');
is_deeply [ State->columns('Primary') ] => [qw/name/],
	'State Primary:' . join ", ", State->columns('Primary');
is_deeply [ sort State->columns('Essential') ] => [qw/abbreviation name/],
	'State Essential:' . join ", ", State->columns('Essential');
is_deeply [ sort State->columns('All') ] =>
	[ sort qw/name abbreviation rain snowfall capital population/ ],
	'State All:' . join ", ", State->columns('All');

is(CD->primary_column, 'artist', 'CD primary()');
is_deeply [ CD->columns('Primary') ] => [qw/artist/],
	'CD primary:' . join ", ", CD->columns('Primary');
is_deeply [ sort CD->columns('All') ] => [qw/artist length title/],
	'CD all:' . join ", ", CD->columns('All');
is_deeply [ sort CD->columns('Essential') ] => [qw/artist/],
	'CD essential:' . join ", ", CD->columns('Essential');

{
	local $SIG{__WARN__} = sub { ok 1, "Error thrown" };
	ok(!State->columns('Nonsense'), "No Nonsense group");
	ok(State->is_column('capital'), 'is_column deprecated');
}
ok(State->has_column('Rain'), 'has_column Rain');
ok(State->has_column('rain'), 'has_column rain');
ok(!State->has_column('HGLAGAGlAG'), '!has_column HGLAGAGlAG');

ok(!State->can('Rain'), 'No Rain accessor set up');
ok(State->can('Rainfall'),           'Rainfall accessor set up');
ok(State->can('_Rainfall_accessor'), ' with correct alias');
ok(!State->can('_Rain_accessor'), ' (not by colname)');
ok(!State->can('rain'),           ' (not normalized)');
ok(State->can('set_Rain'),           'overriden mutator');
ok(State->can('_set_Rain_accessor'), ' with alias');

ok(State->can('Snowfall'),           'overridden accessor set up');
ok(State->can('_Snowfall_accessor'), ' with alias');
ok(!State->can('snowfall'), ' (not normalized)');
ok(State->can('set_Snowfall'),           'overriden mutator');
ok(State->can('_set_Snowfall_accessor'), ' with alias');

{
	eval { my @grps = State->_cols2groups("Huh"); };
	like $@, qr/not in any groups/, "Huh not in groups";

	my @grps = sort State->_cols2groups(qw/rain capital/);
	is @grps, 2, "Rain and Capital = 2 groups";
	is $grps[0], 'Other',   " - Other";
	is $grps[1], 'Weather', " - Weather";
}

{
	local $SIG{__WARN__} = sub { };
	eval { Class::DBI->retrieve(1) };
	like $@, qr/No database connection/, "Need a connection for queries";
}

#-----------------------------------------------------------------------
# Make sure that columns inherit properly
#-----------------------------------------------------------------------
package State;

package A;
@A::ISA = qw(Class::DBI);
__PACKAGE__->columns(Primary => 'id');

package A::B;
@A::B::ISA = 'A';
__PACKAGE__->columns(All => qw(id b1));

package A::C;
@A::C::ISA = 'A';
__PACKAGE__->columns(All => qw(id c1 c2 c3));

package main;
is join (' ', sort A->columns),    'id',          "A columns";
is join (' ', sort A::B->columns), 'b1 id',       "A::B columns";
is join (' ', sort A::C->columns), 'c1 c2 c3 id', "A::C columns";
