package agent

import (
	"context"
	"sync"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/tool/testing/mock_modagent"
	data "gitlab.com/gitlab-org/security-products/analyzers/trivy-k8s-wrapper/data/kas"
	"go.uber.org/mock/gomock"
	"go.uber.org/zap/zaptest"
	corev1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/watch"
	"k8s.io/client-go/kubernetes/fake"
	clienttesting "k8s.io/client-go/testing"
)

const (
	namespace1                = "namespace1"
	namespace2                = "namespace2"
	gitlabAgentNamespace      = "gitlab-agent"
	gitlabAgentServiceAccount = "gitlab-agent"
	podName                   = "Pod name"
)

func TestScanPodFailed(t *testing.T) {
	ctrl := gomock.NewController(t)
	ctx, cancel := context.WithTimeout(context.Background(), 300*time.Millisecond)
	defer cancel()

	kubeClientset := fake.NewSimpleClientset()

	api := mock_modagent.NewMockAPI(ctrl)
	s := scanJob{
		log:                       zaptest.NewLogger(t),
		api:                       api,
		kubeClientset:             kubeClientset,
		gitlabAgentNamespace:      gitlabAgentNamespace,
		gitlabAgentServiceAccount: gitlabAgentServiceAccount,
		agentID:                   5000000000,
		targetNamespaces:          []string{namespace1},
		resourceRequirements:      &defaultResourceRequirements,
	}

	api.EXPECT().MakeGitLabRequest(gomock.Any(), "/",
		gomock.Any(),
		gomock.Any(),
	).Times(0)

	api.EXPECT().MakeGitLabRequest(gomock.Any(), "/scan_result",
		gomock.Any(),
		gomock.Any(),
	).Times(0)

	wg := startCustomWatcher(t, ctx, kubeClientset, 1, corev1.PodFailed)

	s.Run(ctx)
	podList, err := kubeClientset.CoreV1().Pods(gitlabAgentNamespace).List(ctx, metav1.ListOptions{})
	assert.NoError(t, err)
	// Failed Trivy Pods should be deleted
	assert.Empty(t, podList.Items)

	wg.Wait()
}

func TestScanEmptyNamespace(t *testing.T) {
	ctrl := gomock.NewController(t)
	ctx, cancel := context.WithTimeout(context.Background(), 300*time.Millisecond)
	defer cancel()

	kubeClientset := fake.NewSimpleClientset()
	// Ensure that no Trivy Pods are created when there are no namespaces to be scanned
	kubeClientset.PrependReactor("create", "*", func(action clienttesting.Action) (bool, runtime.Object, error) {
		t.Errorf("Create should not be called")
		return true, nil, nil
	})

	api := mock_modagent.NewMockAPI(ctrl)
	s := scanJob{
		log:                       zaptest.NewLogger(t),
		api:                       api,
		kubeClientset:             kubeClientset,
		gitlabAgentNamespace:      gitlabAgentNamespace,
		gitlabAgentServiceAccount: gitlabAgentServiceAccount,
		agentID:                   5000000000,
		targetNamespaces:          []string{}, //No namespace targeted
		resourceRequirements:      &defaultResourceRequirements,
	}

	api.EXPECT().MakeGitLabRequest(gomock.Any(), "/",
		gomock.Any(),
		gomock.Any(),
	).Times(0)

	api.EXPECT().MakeGitLabRequest(gomock.Any(), "/scan_result",
		gomock.Any(),
		gomock.Any(),
	).Times(0)

	wg := startCustomWatcher(t, ctx, kubeClientset, 1, corev1.PodSucceeded)

	s.Run(ctx)

	wg.Wait()
}

var publishedDate = time.Date(2020, time.December, 10, 0, 0, 0, 0, time.UTC)
var lastModifiedDate = time.Date(2022, time.October, 29, 0, 0, 0, 0, time.UTC)
var sampleResult = []data.Result{
	{
		Target: "nginx:1.16 (debian 10.3)",
		Class:  "os-pkgs",
		Type:   "alpine",
		Vulnerabilities: []data.DetectedVulnerability{
			{
				VulnerabilityID:  "CVE-2020-27350",
				PkgName:          "apt",
				InstalledVersion: "1.8.2",
				FixedVersion:     "1.8.2.2",
				PrimaryURL:       "https://avd.aquasec.com/nvd/cve-2020-27350",
				Vulnerability: data.Vulnerability{
					Title:       "apt: integer overflows and underflows while parsing .deb packages",
					Description: "APT had several integer overflows and underflows while parsing .deb packages, aka GHSL-2020-168 GHSL-2020-169, in files apt-pkg/contrib/extracttar.cc, apt-pkg/deb/debfile.cc, and apt-pkg/contrib/arfile.cc. This issue affects: apt 1.2.32ubuntu0 versions prior to 1.2.32ubuntu0.2; 1.6.12ubuntu0 versions prior to 1.6.12ubuntu0.2; 2.0.2ubuntu0 versions prior to 2.0.2ubuntu0.2; 2.1.10ubuntu0 versions prior to 2.1.10ubuntu0.1;",
					Severity:    "MEDIUM",
					References: []string{
						"https://access.redhat.com/security/cve/CVE-2020-27350",
						"https://bugs.launchpad.net/bugs/1899193",
					},
					PublishedDate:    &publishedDate,
					LastModifiedDate: &lastModifiedDate,
				},
			},
		},
	},
}

type CustomLogParser struct {
	err error
}

// ParsePodLogsToReport stubs the report that is parsed from a Trivy Pod's logs
func (m *CustomLogParser) ParsePodLogsToReport(logs []byte) (data.ConsolidatedReport, error) {
	report := data.ConsolidatedReport{
		Findings: []data.Resource{
			{
				Namespace: "some-namespace",
				Kind:      "Deployment",
				Name:      "nginx",
				Results:   sampleResult,
			},
		},
	}
	return report, m.err
}

// customWatcher is used to stub the Pod's watch interface
type customWatcher struct {
	eventChan chan watch.Event
}

func (cw *customWatcher) Stop() {}

func (cw *customWatcher) ResultChan() <-chan watch.Event {
	return cw.eventChan

}

// startCustomWatcher creates a custom watcher and waits for the specified number of Pods to be created before changing the Pod status to the desired status.
// This will ensure that the watcher in scanJob.startPodScanForNamespace does not wait forever and can continue processing.
// As we are using kubernetes fake.Clientset, Status.Phase would be "" instead of PodPending.
// Status.Phase would not change even after the Pod status has been updated since the customWatcher doesn't update the kubernetes object.
// As such, the moment the status has been updated, this function will exit.
func startCustomWatcher(t *testing.T, ctx context.Context, kubeClientset *fake.Clientset, podsToCreate int, desiredPodStatus corev1.PodPhase) *sync.WaitGroup {
	customWatcher := &customWatcher{
		eventChan: make(chan watch.Event),
	}
	kubeClientset.PrependWatchReactor("pods", func(action clienttesting.Action) (handled bool, ret watch.Interface, err error) {
		return true, customWatcher, nil
	})

	// Use WaitGroup to wait for Pod status to change before exiting the test function
	var wg sync.WaitGroup
	wg.Add(1)
	go func(wg *sync.WaitGroup) {
		defer wg.Done()

		for {
			select {
			case <-time.After(10 * time.Millisecond):
				podList, err := kubeClientset.CoreV1().Pods(gitlabAgentNamespace).List(ctx, metav1.ListOptions{})
				assert.NoError(t, err)

				// Only process if number of expected Pods have been created
				if len(podList.Items) == podsToCreate {
					for i := range podList.Items {
						createdPod := &podList.Items[i]
						updatedPod := createdPod.DeepCopy()
						updatedPod.Status.Phase = desiredPodStatus
						customWatcher.eventChan <- watch.Event{Type: watch.Modified, Object: updatedPod}
					}
					close(customWatcher.eventChan)
					return
				}
			case <-ctx.Done():
				return
			}
		}
	}(&wg)

	return &wg
}
