;**************************************************************************
;** ps-util.l                                                            **
;**                                                                      **
;**  Cette programme contient des fonctions auxilieres dans le develop-  **
;** pement de PSYCHOSES.                                                 **
;**                                                                      **
;** Claudia Coiteux-Rosu                                    Juillet 1989 **
;**************************************************************************

(eval-when (compile) (load "varenv"))

;**************************************************************************
;** Manipulation d'atomes et des listes                                  **
;**************************************************************************
;** Substitue un element d'une liste par une sequence d'elements         **
(defun subst@(elem seq liste)
   (cond ((null liste) nil)
         ((eq (car liste) elem) 
              (append seq (subst@ elem seq (cdr liste))))
         (t   (cons (car liste) (subst@ elem seq (cdr liste))))))


;** Cette fonction retourne l'atome oppose a `right' ou a `left' .       **
(defun other-side (part)
   (cond ((eq part 'right) 'left)
         ((eq part 'left) 'right)
         (t part)))


;** Dequotifie les atomes de type variable (precedes par `-') de la lis-**
;** te `liste' presents dans la liste vars                              **
(defun dequotify-vars (liste vars)
   (cond ((null liste) nil)
         (t (cons (cond ((and (listp (car liste))
                              (eq 'quote (caar liste))
                              (varp (cadar liste))
                              (member (cadar liste) vars))
                                             (cadar liste))
                        ((listp (car liste)) 
                              (dequotify-vars (car liste) vars))
                        (t (car liste)))
                  (dequotify-vars (cdr liste) vars)))))
                                             


;** En partant d'une liste retourne une liste composee par de sous-listes**
;** contenant les fonctions d'acces aux elements de la premiere liste et **
;** les elements eux memes. Example (get-paths '(a b)) retournera la lis-**
;** te ((car a) (cadr b)).                                               **
(defun get-paths (x)
 (let ((*psy-paths* (ncons nil)))
   (get-paths2 x "")
   (car *psy-paths*)))


;** Fonction auxiliere a get-paths                                       **
(defun get-paths2 (x sofar)
 (cond ((null x))
       ((varp x))
       ((atom x)
	  (tconc *psy-paths* (list (concat 'c sofar 'r) x)))
       (t (get-paths2 (car x) (concat 'a sofar))
	  (get-paths2 (cdr x) (concat 'd sofar)))))


;** Verifie si l'atome `x' a comme premier symbole `-',c'est-a-dire s'il **
;** represente une variable de psychoses                                 **
(defun varp (x)
  (and (symbolp x) (eq (getchar x 1) #\-)))


;** Retourne la liste de variables contenues dans la liste `x'           **
(defun get-vars (x)
  (let (vars)
   (declare (special vars))
   (get-vars2 x)
    vars))
 

;** Fonction auxiliere de `get-vars'                                     **
(defun get-vars2 (x)
 (declare (special vars))
 (cond ((eq x '-))
       ((varp x)
	  (or (memq x vars)
	      (push1   vars x)))
       ((atom x))
       (t (get-vars2 (car x))
	  (get-vars2 (cdr x)))))


;** En partant d'un modele (pattern), d'une liste (fait) et d'une liste  **
;** de variables verifie l'unification du modele avec la liste, si elle  **
;** reussi retoune une liste des valeurs pris par les variables de la    **
;** liste de variables pendant le processus d'unification, sinon elle re-**
;** tourne la valeur $$FAIL. Si une variable n'est pas presente dans le  **
;** modele elle prendra la valeur $$NONE$$                               **
(defun unify (pat fact vars)
 (let ((values (car (for var from 1 to (length vars)
                           tcollect '$$NONE$$))))
  (cond ((eq (car pat) '|?|) (unify-fn (cadr pat) fact))
        ((eq (attempt (unify2 pat fact vars values)) '$$FAIL)
	   '$$FAIL)
	(t values))))


;** Fonction auxiliere de `unify'                                        **
(defun unify2 (pat fact vars values)
 (let (bnd)
  (declare (ignore bnd))
  (cond	((eq pat '-))
	((varp pat) (for var in vars
			 value on values
			 when (eq var pat)
			 quit (or (and (eq (car value) '$$NONE$$)
				       (rplaca value fact))
				  (equal (car value) fact)
				  (fail))))
	((eq pat fact))
	((atom pat)(fail))
        ((atom fact)(fail))
	(t (unify2 (car pat) (car fact) vars values)
	   (unify2 (cdr pat) (cdr fact) vars values)))))


;** Verifie si tous les elements de la liste lst1 sont elements de lst2  **
(defun allmemq (lst1 lst2)
 (for x in lst1 always (memq x lst2)))


;** Retourne une liste correspondant a l'union dans le sense ensembliste **
;** de lst1 et lste                                                      **
(progn (proclaim '(inline psy-merge))
       (defun psy-merge (lst1 lst2)
         (car (lconc (for x in lst2
	   	      unless (memq x lst1)
		      tcollect x)
	              lst1))))


;** Si x = (<at1>...<atn>) elle retourne (liste <qat1>...<qatm>), avec   **
;** <qati> = '<atj> si <atj>!=`^' et <qati>=<atj+1> si atj=`^'.          **
;** Si x = <atome> elle retourne '<atome> et si x est une variable elle  **
;** la retourne tel quelle                                               **
(defun quotify (x)
 (cond ((varp x) x)
       ((atom x)    `',x)
       ((eq (car x) #\^)(cdr x))
       (t (for y on x
	       bind tail
	       tcollect (cond ((atom y)
				(setq tail y)
				(setq y nil)
				(if (varp tail)
				    tail
				    `',tail))
			      (t (quotify (car y))))
	       finally (if tail
			   `(psy-list* ,@(car $$val))
			   `(list ,@(car $$val)))))))


;** Fonction auxiliere de `quotify'                                      **
(defun psy-list* (&rest lst)
  (cond ((null lst) nil)
	((null (cdr lst)) (car lst))
	(t (for x on lst
		do (if (null (cddr x))
		    (progn
                       (rplacd x (cadr x))
		       (return lst)))))))


;** Insere la valeur x dans la liste lst dans l'ordre etabli par la fonc-**
;** tion fcn.                                                            **
(progn (proclaim '(inline insert))
       (defun insert (x lst fcn)
         (cond ((null lst) (ncons x))
	       ((funcall fcn x (car lst)) (cons x lst))
	       (t (for	y in (cdr lst)
		        back on lst
		        when (funcall fcn x y)
		        quit (rplacd back (cons x (cdr back)))
		        finally ; x belongs at the end
			(rplacd back (ncons x)))
	                lst))))


;** Teste pour inegal (not equal)                                        **
(defmacro <> (&rest x) `(not (equal ,@x)))


;** Retourne la combinaison voulue de cdr et car, path est la combinaison**
;** recherche et value c'est la liste                                    **
(defun carcdrs (path value)
   (let* ((strpath (string path))
          (lenpath (1- (length strpath))))
      (and (eq (char strpath 0) #\C)
           (eq (char strpath lenpath) #\R)
           (for i fromd (1- lenpath) to 1
                bind (result value)
                do (cond ((eq (char strpath i) #\D)
                             (cond ((listp result) (setq result (cdr result)))
                                   (t (setq result nil))))
                         ((eq (char strpath i) #\A)
                             (cond ((listp result) (setq result (car result)))
                                   (t (setq result nil))))
                         (t (setq result nil)))
                finally result))))
(defun caaddar(l) (car (caddar l)))
(defun cadaddar(l) (cadr (caddar l)))


;**************************************************************************
;** Regles de production                                                 **
;**************************************************************************
;** Cette fonction retourne le nom simple d'une regle de production etant**
;** donne que normalement il est prefixe par le nom de la base suivi du  **
;** symbole `-'.                                                         **
(defun take-rule-name(r)
   (let ((str-name (string r)))
      (read-from-string (subseq str-name 
                                (1+ (search "-" str-name))))))


;**************************************************************************
;** Manipulation de fonctions                                            **
;**************************************************************************
;** Cette fonction evalue la fonction `fn' en reemplacant ses variables  **
;** (prefixees par `-') par les valeurs qu'elles prennent. Pour trouver  **
;** la valeur d'une variable, il suffir de chercher sa position dans     **
;** `vars' et chercher la valeur qui se trouve dans la meme position dans**
;** `vals'.                                                              **
(defun eval-with-vars (vars vals fn)
  (eval (for e in fn
             collect (cond ((listp e) 
                                 (cons-fn-a-eval vars vals
                                                (dequotify-vars e vars)))
                           ((and (varp e)
                                 (member e vars))
                                 `',(nth (position e vars) vals))
                           (t e)))))


;** Fonction auxiliere de `eval-with-vars'                               **
(defun cons-fn-a-eval (vars vals fn)
 (cond ((null fn) nil)
       (t (cons (cond ((and (varp (car fn))
                            (member (car fn) vars))
                                      (nth (position (car fn) vars) vals))
                      ((listp (car fn))
                          (cons-fn-a-eval vars vals (car fn)))
                      (t (car fn)))
                (cons-fn-a-eval vars vals (cdr fn))))))
    

;** Unification d'un modele fonction avec un fait                        **      
(defun unify-fn (fn fact)
   (cond ((and (null (get-vars fn))
               (equal (eval fn) fact)) nil)
         (t '$$FN)))



